/*
 * AUTHOR: Pedro Lineu Orso                         pedro.orso@gmail.com
 *                                                            1998, 2009
 * SARG Squid Analysis Report Generator      http://sarg.sourceforge.net
 *
 * SARG donations:
 *      please look at http://sarg.sourceforge.net/donations.php
 * ---------------------------------------------------------------------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 */

#include "include/conf.h"
#include "include/defs.h"

extern numlist hours, weekdays;

static int getparam_string(const char *param,char *buf,char *value,int value_size)
{
   int plen;

   plen=strlen(param);
   if (strncmp(buf,param,plen) != 0) return(0);
   buf+=plen;
   if ((unsigned char)*buf>' ') return(0);
   while (*buf && (unsigned char)*buf<=' ') buf++;

   if (strlen(buf)>=value_size) {
      printf("SARG: Maybe you have a broken record or garbage in %s parameter.\n",param);
      exit(1);
   }
   strcpy(value,buf);
   fixnone(value);
   return(1);
}

static int getparam_quoted(const char *param,char *buf,char *value,int value_size)
{
   int plen;
   int i;

   plen=strlen(param);
   if (strncmp(buf,param,plen) != 0) return(0);
   buf+=plen;
   if ((unsigned char)*buf>' ') return(0);
   while (*buf && (unsigned char)*buf<=' ') buf++;

   if (*buf != '\"') {
      printf("SARG: %s %s.\n",text[139],param);
      exit(1);
   }
   buf++;

   value_size--;
   for (i=0 ; i<value_size && *buf && *buf!='\"' ; i++) {
      value[i]=*buf++;
   }
   value[i]='\0';

   if (*buf != '\"') {
      printf("SARG: Missing double quote after parameter %s or value is more than %d bytes long.\n",param,value_size);
      exit(1);
   }
   fixnone(value);
   return(1);
}

static int getparam_2words(const char *param,char *buf,char *word1,int word1_size,char *word2,int word2_size)
{
   int plen;
   int i;

   plen=strlen(param);
   if (strncmp(buf,param,plen) != 0) return(0);
   buf+=plen;
   if ((unsigned char)*buf>' ') return(0);
   while (*buf && (unsigned char)*buf<=' ') buf++;

   for (i=0 ; i<word1_size && *buf && (unsigned char)*buf>' ' ; i++)
      word1[i]=*buf++;
   if (i>=word1_size) {
      printf("SARG: The first word of parameter %s is more than %d bytes long.\n",param,word1_size-1);
      exit(1);
   }
   if (*buf!=' ') {
      printf("SARG: Missing second word for parameter %s.\n",param);
      exit(1);
   }
   word1[i]=0;

   while (*buf && (unsigned char)*buf<=' ') buf++;

   for (i=0 ; i<word2_size && *buf && (unsigned char)*buf>' ' ; i++)
      word2[i]=*buf++;
   if (i>=word2_size) {
      printf("SARG: The second word of parameter %s is more than %d bytes long.\n",param,word2_size-1);
      exit(1);
   }
   word2[i]=0;

   fixnone(word1);
   fixnone(word2);
   return(1);
}

static int getparam_int(const char *param,char *buf,int *value)
{
   int plen;
   int next;

   plen=strlen(param);
   if (strncmp(buf,param,plen) != 0) return(0);
   buf+=plen;
   if ((unsigned char)*buf>' ') return(0);
   while (*buf && (unsigned char)*buf<=' ') buf++;

   next=0;
   if (sscanf(buf,"%d%n",value,&next) != 1 || (unsigned char)buf[next] > ' ') {
      printf("SARG: Maybe you have a broken record or garbage in %s parameter.\n",param);
      exit(1);
   }
   return(1);
}

static void parmtest(char *buf)
{
   char wbuf[50];

   while (*buf && (unsigned char)*buf<=' ') buf++;

   if(*buf == '#' || *buf == '\0')
      return;

   if(debugz)
      printf("SARG: TAG: %s\n",buf);

   if (getparam_string("background_color",buf,BgColor,sizeof(BgColor))>0) return;

   if (getparam_string("text_color",buf,TxColor,sizeof(TxColor))>0) return;

   if (getparam_string("text_bgcolor",buf,TxBgColor,sizeof(TxBgColor))>0) return;

   if (getparam_string("title_color",buf,TiColor,sizeof(TiColor))>0) return;

   if (getparam_string("logo_image",buf,LogoImage,sizeof(LogoImage))>0) return;

   if (getparam_quoted("logo_text",buf,LogoText,sizeof(LogoText))>0) return;

   if (getparam_string("logo_text_color",buf,LogoTextColor,sizeof(LogoTextColor))>0) return;

   if (getparam_string("background_image",buf,BgImage,sizeof(BgImage))>0) return;

   if (getparam_string("show_sarg_info",buf,ShowSargInfo,sizeof(ShowSargInfo))>0) return;

   if (getparam_string("show_sarg_logo",buf,ShowSargLogo,sizeof(ShowSargLogo))>0) return;

   if (getparam_string("font_face",buf,FontFace,sizeof(FontFace))>0) return;

   if (getparam_string("header_color",buf,HeaderColor,sizeof(HeaderColor))>0) return;

   if (getparam_string("header_bgcolor",buf,HeaderBgColor,sizeof(HeaderBgColor))>0) return;

   if (getparam_string("font_size",buf,FontSize,sizeof(FontSize))>0) return;

   if (getparam_string("header_font_size",buf,HeaderFontSize,sizeof(HeaderFontSize))>0) return;

   if (getparam_string("title_font_size",buf,TitleFontSize,sizeof(TitleFontSize))>0) return;

   if (getparam_2words("image_size",buf,Width,sizeof(Width),Height,sizeof(Height))>0) return;

   if (getparam_quoted("title",buf,Title,sizeof(Title))>0) return;

   if (getparam_string("resolve_ip",buf,Ip2Name,sizeof(Ip2Name))>0) return;

   if (getparam_string("user_ip",buf,UserIp,sizeof(UserIp))>0) return;

   if (getparam_string("max_elapsed",buf,MaxElapsed,sizeof(MaxElapsed))>0) return;

   if(strstr(buf,"date_format") != 0) {
      if (getword_multisep(wbuf,sizeof(wbuf),buf,' ')<0) {
         printf("SARG: Maybe you have a broken record or garbage in date_format parameter.\n");
         exit(1);
      }
      strncpy(DateFormat,buf,1);
      fixnone(DateFormat);
      return;
   }

   if( strstr( buf, "hours" ) != 0 ) {
      if( getnumlist( buf, &hours, 24, 24 ) ) {
         fprintf( stderr, "Error: Invalid syntax in hours tag!\n" );
         exit( 1 );
      }
   }

   if( strstr( buf, "weekdays" ) != 0 ) {
      if( getnumlist( buf, &weekdays, 7, 7 ) ) {
         fprintf( stderr, "Error: Invalid syntax in weekdays tag!\n" );
         exit( 1 );
      }
   }

   if (getparam_2words("topuser_sort_field",buf,TopuserSortField,sizeof(TopuserSortField),TopuserSortOrder,sizeof(TopuserSortOrder))>0) return;

   if (getparam_2words("user_sort_field",buf,UserSortField,sizeof(UserSortField),UserSortOrder,sizeof(UserSortOrder))>0) return;

   if (getparam_string("access_log",buf,AccessLog,sizeof(AccessLog))>0) return;

   if (getparam_string("useragent_log",buf,UserAgentLog,sizeof(UserAgentLog))>0) return;

   if (getparam_string("exclude_hosts",buf,ExcludeHosts,sizeof(ExcludeHosts))>0) return;

   if (getparam_string("exclude_codes",buf,ExcludeCodes,sizeof(ExcludeCodes))>0) return;

   if (getparam_string("exclude_users",buf,ExcludeUsers,sizeof(ExcludeUsers))>0) return;

   if (getparam_string("password",buf,PasswdFile,sizeof(PasswdFile))>0) return;

   if (getparam_string("temporary_dir",buf,TempDir,sizeof(TempDir))>0) return;

   if (getparam_string("report_type",buf,ReportType,sizeof(ReportType))>0) return;

   if (getparam_string("output_dir",buf,OutputDir,sizeof(OutputDir))>0) return;

   if (getparam_string("output_email",buf,OutputEmail,sizeof(OutputEmail))>0) return;

   if (getparam_2words("per_user_limit",buf,PerUserLimitFile,sizeof(PerUserLimitFile),PerUserLimit,sizeof(PerUserLimit))>0) return;

   if (getparam_string("lastlog",buf,LastLog,sizeof(LastLog))>0) return;

   if (getparam_string("remove_temp_files",buf,RemoveTempFiles,sizeof(RemoveTempFiles))>0) return;

   if (getparam_string("replace_index",buf,ReplaceIndex,sizeof(ReplaceIndex))>0) return;

   if (getparam_string("index_tree",buf,IndexTree,sizeof(IndexTree))>0) return;

   if (getparam_string("index",buf,Index,sizeof(Index))>0) return;

   if (getparam_string("overwrite_report",buf,OverwriteReport,sizeof(OverwriteReport))>0) return;

   if (getparam_string("records_without_userid",buf,RecordsWithoutUser,sizeof(RecordsWithoutUser))>0) return;

   if (getparam_string("use_comma",buf,UseComma,sizeof(UseComma))>0) return;

   if (getparam_string("mail_utility",buf,MailUtility,sizeof(MailUtility))>0) return;

   if (getparam_string("topsites_num",buf,TopSitesNum,sizeof(TopSitesNum))>0) return;

   if (getparam_int("topuser_num",buf,&TopUsersNum)>0) return;

   if (getparam_string("usertab",buf,UserTabFile,sizeof(UserTabFile))>0) return;

   if (getparam_string("index_sort_order",buf,IndexSortOrder,sizeof(IndexSortOrder))>0) return;

   if (getparam_2words("topsites_sort_order",buf,TopsitesSortField,sizeof(TopsitesSortField),TopsitesSortType,sizeof(TopsitesSortType))>0) return;

   if (getparam_string("long_url",buf,LongUrl,sizeof(LongUrl))>0) return;

   if (getparam_string("language",buf,language,sizeof(language))>0) return;

   if (getparam_string("dansguardian_conf",buf,DansGuardianConf,sizeof(DansGuardianConf))>0) return;

   if (getparam_string("squidguard_conf",buf,SquidGuardConf,sizeof(SquidGuardConf))>0) return;

   if (getparam_string("date_time_by",buf,datetimeby,sizeof(datetimeby))>0) return;

   if (getparam_string("charset",buf,CharSet,sizeof(CharSet))>0) {
      ccharset(CharSet);
      return;
   }

   if (getparam_quoted("user_invalid_char",buf,UserInvalidChar,sizeof(UserInvalidChar))>0) return;

   if (getparam_quoted("include_users",buf,IncludeUsers+1,sizeof(IncludeUsers)-2)>0) {
      IncludeUsers[0]=':';
      strcat(IncludeUsers,":");
      return;
   }

   if (getparam_quoted("exclude_string",buf,ExcludeString,sizeof(ExcludeString))>0) return;

   if (getparam_string("privacy",buf,Privacy,sizeof(Privacy))>0) return;

   if (getparam_quoted("privacy_string",buf,ExcludeString,sizeof(ExcludeString))>0) return;

   if (getparam_string("privacy_string_color",buf,PrivacyStringColor,sizeof(PrivacyStringColor))>0) return;

   if (getparam_string("show_successful_message",buf,SuccessfulMsg,sizeof(SuccessfulMsg))>0) return;

   if (getparam_string("show_read_statistics",buf,ShowReadStatistics,sizeof(ShowReadStatistics))>0) return;

   if (getparam_string("topuser_fields",buf,TopUserFields,sizeof(TopUserFields))>0) return;

   if (getparam_string("bytes_in_sites_users_report",buf,BytesInSitesUsersReport,sizeof(BytesInSitesUsersReport))>0) return;

   if (getparam_string("user_report_fields",buf,UserReportFields,sizeof(UserReportFields))>0) return;

   if (getparam_string("bytes_in_sites_users_report",buf,BytesInSitesUsersReport,sizeof(BytesInSitesUsersReport))>0) return;

   if (getparam_string("datafile",buf,DataFile,sizeof(DataFile))>0) return;

   if (getparam_quoted("datafile_delimiter",buf,DataFileDelimiter,sizeof(DataFileDelimiter))>0) return;

   if (getparam_string("datafile_fields",buf,DataFileFields,sizeof(DataFileFields))>0) return;

   if (getparam_string("datafile_url",buf,DataFileUrl,sizeof(DataFileUrl))>0) return;

   if (getparam_string("parsed_output_log",buf,ParsedOutputLog,sizeof(ParsedOutputLog))>0) return;

   if (getparam_string("parsed_output_log_compress",buf,ParsedOutputLogCompress,sizeof(ParsedOutputLogCompress))>0) return;

   if (getparam_string("displayed_values",buf,DisplayedValues,sizeof(DisplayedValues))>0) return;

   if (getparam_int("authfail_report_limit",buf,&AuthfailReportLimit)>0) return;

   if (getparam_int("denied_report_limit",buf,&DeniedReportLimit)>0) return;

   if (getparam_int("siteusers_report_limit",buf,&SiteUsersReportLimit)>0) return;

   if (getparam_int("dansguardian_report_limit",buf,&DansGuardianReportLimit)>0) return;

   if (getparam_int("squidguard_report_limit",buf,&SquidGuardReportLimit)>0) return;

   if (getparam_int("user_report_limit",buf,&UserReportLimit)>0) return;

   if (getparam_int("download_report_limit",buf,&DownloadReportLimit)>0) return;

   if (getparam_string("www_document_root",buf,wwwDocumentRoot,sizeof(wwwDocumentRoot))>0) return;

   if (getparam_string("block_it",buf,BlockIt,sizeof(BlockIt))>0) return;

   if (getparam_string("external_css_file",buf,ExternalCSSFile,sizeof(ExternalCSSFile))>0) return;

   if (getparam_string("user_authentication",buf,UserAuthentication,sizeof(UserAuthentication))>0) return;

   if (getparam_string("AuthUserFile",buf,AuthUserFile,sizeof(AuthUserFile))>0) return;

   if (getparam_string("AuthName",buf,AuthName,sizeof(AuthName))>0) return;

   if (getparam_string("AuthType",buf,AuthType,sizeof(AuthType))>0) return;

   if (getparam_string("Require",buf,Require,sizeof(Require))>0) return;

   if (getparam_quoted("download_suffix",buf,DownloadSuffix,sizeof(DownloadSuffix))>0) return;

   if (getparam_string("graphs",buf,Graphs,sizeof(Graphs))>0) return;

   if (getparam_string("graph_days_bytes_bar_color",buf,GraphDaysBytesBarColor,sizeof(GraphDaysBytesBarColor))>0) return;

   if (getparam_string("squidguard_log_format",buf,SquidGuardLogFormat,sizeof(SquidGuardLogFormat))>0) return;

   if (getparam_string("squidguard_ignore_date",buf,SquidguardIgnoreDate,sizeof(SquidguardIgnoreDate))>0) return;

   if (getparam_string("dansguardian_ignore_date",buf,DansguardianIgnoreDate,sizeof(DansguardianIgnoreDate))>0) return;

   if (getparam_string("ulimit",buf,Ulimit,sizeof(Ulimit))>0) return;

   if (getparam_string("ntlm_user_format",buf,NtlmUserFormat,sizeof(NtlmUserFormat))>0) return;

   if (getparam_string("realtime_types",buf,RealtimeTypes,sizeof(RealtimeTypes))>0) return;

   if (getparam_string("realtime_unauthenticated_records",buf,RealtimeUnauthRec,sizeof(RealtimeUnauthRec))>0) return;

   if (getparam_int("realtime_refresh_time",buf,&realtime_refresh)>0) return;

   if (getparam_int("realtime_access_log_lines",buf,&realtime_access_log_lines)>0) return;

   if(strstr(buf,"squid24") != 0) {
      squid24++;
      return;
   }

   if(strstr(buf,"byte_cost") != 0) {
      if (getword_multisep(wbuf,sizeof(wbuf),buf,' ')<0) {
         printf("SARG: Maybe you have a broken record or garbage in byte_cost parameter.\n");
         exit(1);
      }
      cost=atol(buf);
      if (getword_multisep(wbuf,sizeof(wbuf),buf,' ')<0) {
         printf("SARG: Maybe you have a broken record or garbage in byte_cost parameter.\n");
         exit(1);
      }
      nocost=my_atoll(buf);
      return;
   }

   printf("SARG: %s %s\n",text[140],buf);
}

void getconf(void)
{

   FILE *fp_in;
   char buf[MAXLEN];

   if(debug)
      debuga("Loading configuration from: %s",ConfigFile);

   if ((fp_in = fopen(ConfigFile, "r")) == NULL) {
      fprintf(stderr, "SARG: (getconf) Cannot open file: %s\n",ConfigFile);
      exit(1);
   }

   while (fgets(buf, sizeof(buf), fp_in) != NULL) {
      fixendofline(buf);

      if(debugm)
         printf("SYSCONFDIR %s\n",buf);

      parmtest(buf);

   }

   fclose(fp_in);
   language_load(language);
   return;
}
