# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# See the COPYING file for license information.
#
# Copyright (c) 2006, 2007 Guillaume Chazarain <guichaz@yahoo.fr>

import re
import time

UNITS = ['B', 'KiB', 'MiB', 'GiB', 'TiB', 'PiB', 'EiB']

def human_unit(size):
    """Return a string of the form '12.34 MiB' given a size in bytes."""
    for i in xrange(len(UNITS) - 1, 0, -1):
        base = 1 << (10 * i)
        if 2 * base < size:
            return '%.2f %s' % ((float(size) / base), UNITS[i])
    return str(size) + ' ' + UNITS[0]

def short_string(string, length):
    """Returned a cut copy of the string, replace the middle with '~'."""
    if len(string) > length:
        middle = length / 2
        return string[:middle] + '~' + string[middle - length:]
    return string

def sanitize_string(string):
    def clean_blank(c):
        if c != ' ' and c.isspace():
            return '?'
        return c
    def robust_char_decode(c):
        try:
            return c.decode('UTF-8')
        except UnicodeDecodeError:
            return '?'
    blank_cleaned = ''.join(map(clean_blank, string))
    try:
        return blank_cleaned.decode('UTF-8')
    except UnicodeDecodeError:
        return ''.join(map(robust_char_decode, blank_cleaned))

MIN_SIZE_REGEXP = re.compile('^([0-9]+)(.?)(.*)$')

def min_size_to_consider(min_size='auto', height=0):
    if min_size == 'auto':
        return 1.0 / height
    match = MIN_SIZE_REGEXP.match(min_size)
    if not match:
        raise Exception, 'Cannot parse: ' + min_size
    min_size, letter, suffix = match.groups()
    units_letters = [u[0] for u in UNITS]
    if not letter:
        letter = 'B'
    letter = letter.upper()
    if not (letter in units_letters and suffix in ('iB', '')):
        raise Exception, 'Cannot parse unit in: ' + min_size
    return int(min_size) * 1024 ** units_letters.index(letter)

PROGRESS_CHARS = ['/', '-', '\\', '|']
last_progress_char = 0
last_progress_time = time.time()

def update_progress():
    global last_progress_char, last_progress_time
    now = time.time()
    if now - last_progress_time > 0.04:
            last_progress_char = (last_progress_char + 1) % len(PROGRESS_CHARS)
            last_progress_time = now
            return PROGRESS_CHARS[last_progress_char]

class UINotAvailableException(Exception):
    pass
