# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# See the COPYING file for license information.
#
# Copyright (c) 2006, 2007 Guillaume Chazarain <guichaz@yahoo.fr>

import os
import stat

from pysize.core import chdir_browsing
from pysize.core import compute_size
from pysize.core.pysize_global_fs_cache import get_dev_ino, cache_add_dir
from pysize.core.deletion import get_subtracted

def _extract_prefix_suffixes(paths):
    """['/prefix/suffix1', '/prefix/suffix2', '/prefix/suffix3'] =>
        ('/prefix', ['suffix1', 'suffix2', 'suffix3'])"""
    prefix_len = os.path.commonprefix(paths).rfind('/') + 1
    prefix = paths[0][:prefix_len - 1]
    suffixes = [p[prefix_len:] for p in paths]
    return prefix, suffixes

def _join_prefix_suffixes(prefix, suffixes):
    if len(suffixes) == 1:
        suffix = suffixes[0]
    else:
        suffix = '{' + ','.join(suffixes) + '}'
    return os.path.join(prefix, suffix)


def _sort_nodes(nodes):
    def cmp_fn(n1, n2):
        return cmp(n2.size, n1.size) or cmp(n1.get_name(), n2.get_name())
    nodes.sort(cmp=cmp_fn)

class _pysize_node(object):
    """The parent class of all displayed nodes, as these nodes are displayed on
    screen, there should be few instances."""
    def __init__(self, parent, basename, options):
        self.rectangle = None
        self.parent = parent
        self.children = []
        self.basename = basename
        if basename:
            self.size = compute_size.slow(basename, options.cross_device)
        else:
            self.size = 0

    def compute_height(self):
        if self.children:
            children_height = max([c.compute_height() for c in self.children])
            return children_height + 1
        return 0

    def compute_depth(self):
        depth = 0
        node = self
        while node:
            node = node.parent
            depth += 1
        return depth

    def minimum_node_size(self):
        res = self.size
        for child in self.children:
            child_min_node_size = child.minimum_node_size()
            res = min(res, child_min_node_size)
        return res

    def get_dirname(self):
        return os.path.dirname(self.get_fullpaths()[0])

    def is_dir(self):
        return False

    def is_real(self):
        """Does the node correspond to a path that actually exists?"""
        return True

    def get_fullname(self):
        fullpaths = self.get_fullpaths()
        prefix, suffixes = _extract_prefix_suffixes(fullpaths)
        fullname = _join_prefix_suffixes(prefix, suffixes)
        return fullname

    def get_fullpaths(self):
        parent = self.parent
        fullpath = self.basename
        while parent:
            fullpath = os.path.join(parent.basename, fullpath)
            parent = parent.parent
        return [fullpath]

    def get_name(self):
        """Return the name that should be displayed for this node."""
        return self.basename

    def __iter__(self):
        """The iterator is a depth first traversal."""
        yield self
        for child in self.children:
            for node in child:
                yield node

    def contains_point(self, p):
        """Is the point p in the graphical representation of this node?"""
        if not self.rectangle:
            return False
        x0, x1, y0, y1 = self.rectangle
        return x0 < p.x and p.x < x1 and y0 < p.y and p.y < y1

class _pysize_node_collection(_pysize_node):
    """A directory"""
    def __init__(self, parent, prefix, children, max_depth, min_size, options):
        super(_pysize_node_collection, self).__init__(parent, None, options)
        self.basename = prefix
        fullpaths = [os.path.join(prefix, p) for p in children]
        if max_depth == 0:
            self.size = compute_size.slow_sum(fullpaths, options.cross_device)
        else:
            children_size = 0
            remaining_size = 0
            remaining_nodes = []
            cookie = chdir_browsing.init(prefix)
            try:
                for child in children:
                    try:
                        node = create_node(self, child, max_depth - 1, min_size,
                                           options)
                    except OSError:
                        print child, 'disappeared'
                        continue
                    if node.is_real():
                        self.children.append(node)
                        children_size += node.size
                    else:
                        node.__name = child
                        remaining_nodes.append(node)
                        remaining_size += node.size

                _sort_nodes(self.children)
                if remaining_size > min_size:
                    _sort_nodes(remaining_nodes)
                    names = [n.__name for n in remaining_nodes]
                    rem = _pysize_node_remaining(self, names, options)
                    self.children.append(rem)
            finally:
                chdir_browsing.finalize(cookie)
            self.size = children_size + remaining_size

    def is_real(self):
        """This node does not actually exists, it is an aggregate."""
        return False

class _pysize_node_forest(_pysize_node_collection):
    def __init__(self, parent, children, max_depth, min_size, options):
        prefix, suffixes = _extract_prefix_suffixes(children)
        super(_pysize_node_forest, self).__init__(parent, prefix, suffixes,
                                                  max_depth, min_size, options)
        self.basename = prefix
        self.forest_paths = suffixes
        self.forest_name = _join_prefix_suffixes(prefix, suffixes)

    def get_name(self):
        return self.forest_name

    def get_dirname(self):
        return self.basename

    def get_fullpaths(self):
        fullpaths = self.forest_paths
        parent = self
        while parent:
            fullpaths = [os.path.join(parent.basename, fp) for fp in fullpaths]
            parent = parent.parent
        return fullpaths

class _pysize_node_dir(_pysize_node_collection):
    """A directory"""
    def __init__(self, parent, basename, max_depth, min_size, options):
        listing = chdir_browsing.listdir(basename, options.cross_device)
        super(_pysize_node_dir, self).__init__(parent, basename,
                                               listing, max_depth,
                                               min_size, options)
        self.basename = basename
        self.size += os.lstat(basename).st_blocks * 512
        # update size in cache, in case files were added/removed
        dev_ino = get_dev_ino(basename)
        cache_add_dir(dev_ino, self.size + get_subtracted(basename))

    def is_dir(self):
        return True

    def is_real(self):
        return True

    def get_name(self):
        name = self.basename
        if name != '/':
            name += '/'
        return name

class _pysize_node_remaining(_pysize_node_collection):
    """The sum of a directory's children that are too small to be drawn."""
    def __init__(self, parent, elements, options):
        _pysize_node.__init__(self, parent, None, options)
        # The parent constructor would visit the files
        self.size = compute_size.slow_sum(elements, options.cross_device)
        self.remaining_elements = elements

    def get_name(self):
        return '{' + ','.join(self.remaining_elements) + '}'

    def get_fullname(self):
        if self.remaining_elements:
            return _pysize_node_collection.get_fullname(self)
        return '' # This is the initial node

    def get_fullpaths(self):
        fullpaths = self.remaining_elements
        parent = self.parent
        while parent:
            fullpaths = [os.path.join(parent.basename, fp) for fp in fullpaths]
            parent = parent.parent
        return fullpaths

class _pysize_node_file(_pysize_node):
    """A file"""
    def __init__(self, parent, basename, options):
        super(_pysize_node_file, self).__init__(parent, basename, options)

class _pysize_node_hardlink(_pysize_node_file):
    """A hardlink, the canonical one, or a link"""
    def __init__(self, parent, basename, options):
        super(_pysize_node_hardlink, self).__init__(parent, basename, options)

def create_node(parent, what, max_depth, min_size, options):
    """Return a pysize_node for parent/basename traversing up to max_depth
    levels and only taking into account elements bigger than min_size."""
    if not what:
        return _pysize_node_remaining(parent, [], options)

    if isinstance(what, list):
        size = compute_size.slow_sum(what, options.cross_device)
        if len(what) == 1:
            what = what[0]
    else:
        size = compute_size.slow(what, options.cross_device)

    if size < min_size:
        if isinstance(what, str):
            what = [what]
        node = _pysize_node_remaining(parent, what, options)
    elif isinstance(what, list):
        node = _pysize_node_forest(parent, what, max_depth, min_size, options)
    else:
        st = os.lstat(what)
        if stat.S_ISDIR(st.st_mode):
            node = _pysize_node_dir(parent, what, max_depth, min_size, options)
        elif st.st_nlink > 1:
            node = _pysize_node_hardlink(parent, what, options)
        else:
            node = _pysize_node_file(parent, what, options)
    return node
