
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: tvlinks.c 2673 2007-08-23 08:35:21Z mschwerin $
 */
#include "config.h"

#include <regex.h>
#include <string.h>
#include <sys/types.h>

#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "download.h"
#include "logger.h"
#include "oxine.h"
#include "utils.h"
#include "tvlinks.h"
#include "xmlparser.h"


#ifdef HAVE_TVLINKS

extern oxine_t *oxine;


static char *
get_playback_mrl_guba (const char *mrl)
{
    char *mrl_guba = NULL;
    const char *pat0 = "http://free.guba.com/uploaditem/.+/flash.flv";

    regex_t reg0;
    regmatch_t pm;

    int size = 0;
    char *buffer = read_entire_file (mrl, &size);
    if (!buffer) {
        goto buffer_free;
    }

    if (regcomp (&reg0, pat0, REG_EXTENDED | REG_ICASE | REG_NEWLINE) != 0) {
        error (_("Failed to compile regular expression."));
        goto buffer_free;
    }

    if (regexec (&reg0, buffer, 1, &pm, 0) == 0) {
        int size = 0;
        char *dst = NULL;
        const char *src = NULL;

        size = pm.rm_eo - pm.rm_so + 1;
        mrl_guba = ho_malloc (size);
        memset (mrl_guba, 0, size);

        dst = mrl_guba;
        src = buffer + pm.rm_so;
        size = pm.rm_eo - pm.rm_so;
        memcpy (dst, src, size);
    }

    regfree (&reg0);
  buffer_free:
    ho_free (buffer);

    return mrl_guba;
}


static char *
get_playback_mrl (filelist_t * filelist)
{
    const char *pat0 = "src=\"http://.+/go.divx\"";
    const char *pat1 = "flashvars=\"config=.videoFile: "
        "'http://.+', initialScale";
    const char *pat2 = "src=\"http://.+\"";

    char *mrl = NULL;

    regex_t reg0;
    regex_t reg1;
    regex_t reg2;
    regmatch_t pm;

    int size = 0;
    char *buffer = read_entire_file (filelist->mrl, &size);
    if (!buffer) {
        goto buffer_free;
    }

    if (regcomp (&reg0, pat0, REG_EXTENDED | REG_ICASE | REG_NEWLINE) != 0) {
        error (_("Failed to compile regular expression."));
        goto buffer_free;
    }

    if (regcomp (&reg1, pat1, REG_EXTENDED | REG_ICASE | REG_NEWLINE) != 0) {
        error (_("Failed to compile regular expression."));
        goto reg0_free;
    }

    if (regcomp (&reg2, pat2, REG_EXTENDED | REG_ICASE | REG_NEWLINE) != 0) {
        error (_("Failed to compile regular expression."));
        goto reg1_free;
    }

    if (regexec (&reg0, buffer, 1, &pm, 0) == 0) {
        int size = 0;
        char *dst = NULL;
        const char *src = NULL;

        size = pm.rm_eo - pm.rm_so;
        mrl = ho_malloc (size);
        memset (mrl, 0, size);

        dst = mrl;
        src = buffer + pm.rm_so + 5;
        size = pm.rm_eo - pm.rm_so - 6;
        memcpy (dst, src, size);
    }

    else if (regexec (&reg1, buffer, 1, &pm, 0) == 0) {
        int size = 0;
        char *dst = NULL;
        const char *src = NULL;

        size = pm.rm_eo - pm.rm_so;
        mrl = ho_malloc (size);
        memset (mrl, 0, size);

        dst = mrl;
        src = buffer + pm.rm_so + 31;
        size = pm.rm_eo - pm.rm_so - 46;
        memcpy (dst, src, size);
    }

    else if (regexec (&reg2, buffer, 1, &pm, 0) == 0) {
        int size = 0;
        char *dst = NULL;
        const char *src = NULL;

        size = pm.rm_eo - pm.rm_so;
        mrl = ho_malloc (size);
        memset (mrl, 0, size);

        dst = mrl;
        src = buffer + pm.rm_so + 5;
        size = pm.rm_eo - pm.rm_so - 5;
        memcpy (dst, src, size);

        char *f = strchr (mrl, '"');
        if (f) {
            f[0] = '\0';
        }
    }

    else {
        error (_("Failed to retrieve %s playback MRL for '%s'!"),
               _("TV Links"), filelist->mrl);
    }

    if (mrl) {
        debug ("found TV Links playback MRL '%s'", mrl);
        char *redirect_mrl = NULL;
        do {
            redirect_mrl = get_redirect_url (mrl);
            if (redirect_mrl) {
                ho_free (mrl);
                mrl = redirect_mrl;
            }
        } while (redirect_mrl);
    }

    if (mrl && starts_with (mrl, "http://www.guba.com/flash/")) {
        debug ("found TV Links playback MRL '%s'", mrl);
        char *guba_mrl = get_playback_mrl_guba (mrl);
        if (guba_mrl) {
            ho_free (mrl);
            mrl = guba_mrl;
        }
    }

    if (mrl) {
        debug ("found TV Links playback MRL '%s'", mrl);
    }

    regfree (&reg2);
  reg1_free:
    regfree (&reg1);
  reg0_free:
    regfree (&reg0);
  buffer_free:
    ho_free (buffer);

    return mrl;
}


static bool
tvlinks_video_read (filelist_t * filelist)
{
    char *mrl = get_playback_mrl (filelist);

    if (mrl) {
        if (config_get_bool ("streaming.tvlinks.download_to_cache")) {
            char *file = ho_strdup_printf ("%s.flv", filelist->title);
            char *cache = download_to_cache (mrl, file, false);
            if (cache) {
                ho_free (mrl);
                mrl = cache;
            }
            ho_free (file);
        }

        fileitem_t *item = filelist_add (filelist, filelist->title, mrl,
                                         FILE_TYPE_REGULAR);
        if (item && filelist->thumbnail_mrl) {
            item->thumbnail_mrl = ho_strdup (filelist->thumbnail_mrl);
        }

        ho_free (mrl);
        return true;
    }

    return false;
}


static bool
tvlinks_show_do (filelist_t * filelist)
{
    bool success = false;
    const char *prefix = FILELIST_TVLINKS_MRL;

    const char *pat0 = "href=\"/link.do/.+\" ";
    const char *pat1 = ">.+</a>";

    regex_t reg0;
    regex_t reg1;
    regmatch_t pm;

    int size = 0;
    char *buffer = read_entire_file (filelist->mrl, &size);
    if (!buffer) {
        goto buffer_free;
    }

    if (regcomp (&reg0, pat0, REG_EXTENDED | REG_ICASE | REG_NEWLINE) != 0) {
        error (_("Failed to compile regular expression."));
        goto buffer_free;
    }

    if (regcomp (&reg1, pat1, REG_EXTENDED | REG_ICASE | REG_NEWLINE) != 0) {
        error (_("Failed to compile regular expression."));
        goto buffer_free;
    }

    char *buf = buffer;
    while (regexec (&reg0, buf, 1, &pm, 0) == 0) {
        int size = 0;
        char *dst = NULL;
        const char *src = NULL;
        char *mrl = NULL;
        char *title = NULL;

        size = strlen (prefix) + pm.rm_eo - pm.rm_so;
        mrl = ho_malloc (size);
        memset (mrl, 0, size);

        dst = mrl;
        src = prefix;
        size = strlen (prefix);
        memcpy (dst, src, size);

        dst = mrl + strlen (prefix);
        src = buf + pm.rm_so + 6;
        size = pm.rm_eo - pm.rm_so - 8;
        memcpy (dst, src, size);

        buf = buf + pm.rm_eo;

        if (regexec (&reg1, buf, 1, &pm, 0) == 0) {
            size = pm.rm_eo - pm.rm_so - 4;
            char *t = ho_malloc (size);
            dst = t;
            src = buf + pm.rm_so + 1;
            memcpy (dst, src, size);
            t[size - 1] = '\0';
            title = ho_strdup (trim_whitespace (t));
            ho_free (t);
        }
        else {
            title = ho_strdup (_("Unknown Title"));
        }

        filelist_add (filelist, title, mrl, FILE_TYPE_TVLINKS_VIDEO);

        ho_free (mrl);
        ho_free (title);
    }

    success = true;
    regfree (&reg0);
    regfree (&reg1);
  buffer_free:
    ho_free (buffer);

    return success;
}


static bool
tvlinks_index_do (filelist_t * filelist)
{
    bool success = false;
    const char *prefix = FILELIST_TVLINKS_MRL;

    const char *pat0 = "href=\"/show.do/.+/.+\"";
    const char *pat1 = ">.+</a>";

    regex_t reg0;
    regex_t reg1;
    regmatch_t pm;

    int size = 0;
    char *buffer = read_entire_file (filelist->mrl, &size);
    if (!buffer) {
        goto buffer_free;
    }

    if (regcomp (&reg0, pat0, REG_EXTENDED | REG_ICASE | REG_NEWLINE) != 0) {
        error (_("Failed to compile regular expression."));
        goto buffer_free;
    }

    if (regcomp (&reg1, pat1, REG_EXTENDED | REG_ICASE | REG_NEWLINE) != 0) {
        error (_("Failed to compile regular expression."));
        goto buffer_free;
    }

    char *buf = buffer;
    while (regexec (&reg0, buf, 1, &pm, 0) == 0) {
        int size = 0;
        char *dst = NULL;
        const char *src = NULL;
        char *mrl = NULL;
        char *title = NULL;

        size = strlen (prefix) + pm.rm_eo - pm.rm_so;
        mrl = ho_malloc (size);
        memset (mrl, 0, size);

        dst = mrl;
        src = prefix;
        size = strlen (prefix);
        memcpy (dst, src, size);

        dst = mrl + strlen (prefix);
        src = buf + pm.rm_so + 6;
        size = pm.rm_eo - pm.rm_so - 7;
        memcpy (dst, src, size);

        buf = buf + pm.rm_eo;

        if (regexec (&reg1, buf, 1, &pm, 0) == 0) {
            size = pm.rm_eo - pm.rm_so - 4;
            char *t = ho_malloc (size);
            dst = t;
            src = buf + pm.rm_so + 1;
            memcpy (dst, src, size);
            t[size - 1] = '\0';
            title = ho_strdup_printf ("[%s]", trim_whitespace (t));
            ho_free (t);
        }
        else {
            title = ho_strdup_printf ("[%s]", _("Unknown Title"));
        }

        filelist_add (filelist, title, mrl, FILE_TYPE_TVLINKS_VFOLDER);

        ho_free (mrl);
        ho_free (title);
    }

    success = true;
    regfree (&reg0);
    regfree (&reg1);
  buffer_free:
    ho_free (buffer);

    return success;
}


static bool
tvlinks_list_read (filelist_t * filelist)
{
    struct {
        int id;
        char *name;
    } categories[] = {
        { 1, _("Shows")             },
        { 4, _("Movies")            },
        { 2, _("Cartoons")          },
        { 3, _("Anime")             },
        { 5, _("Music Videos")      },
        { 9, _("Documentaries")     },
        { 0, NULL                   },
    };

    int i = 0;
    for (i = 0; categories[i].id != 0; i++) {
        char *title = ho_strdup_printf ("[%s]", categories[i].name);
        char *mrl = ho_strdup_printf ("%s/index.do/%d", filelist->mrl,
                                      categories[i].id);
        fileitem_t *item = filelist_add (filelist, title, mrl,
                                         FILE_TYPE_TVLINKS_VFOLDER);
        item->thumbnail_mrl =
            ho_strdup_printf (OXINE_VISUALDIR "/logo_tvlinks_%d.png",
                              categories[i].id);
        ho_free (mrl);
        ho_free (title);
    }

    return true;
}


void
tvlinks_read (filelist_t * filelist)
{
    bool success = false;

    switch (filelist->parent_item->type) {
    case FILE_TYPE_TVLINKS_VFOLDER:
        if (strstr (filelist->mrl, "index.do")) {
            success = tvlinks_index_do (filelist);
        }
        else if (strstr (filelist->mrl, "show.do")) {
            success = tvlinks_show_do (filelist);
        }
        else {
            success = tvlinks_list_read (filelist);
        }
        break;
    case FILE_TYPE_TVLINKS_VIDEO:
        success = tvlinks_video_read (filelist);
        break;
    default:
        break;
    }

    if (!success) {
        ho_free (filelist->error);
        filelist->error = ho_strdup (_("Connection failure..."));
    }
}


#endif /* HAVE_TVLINKS */
