
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: playlist.h 2471 2007-07-09 11:04:18Z mschwerin $
 *
 */

#ifndef HAVE_PLAYLIST_H
#define HAVE_PLAYLIST_H

#include <assert.h>

#include "list.h"
#include "filelist.h"
#include "mutex.h"
#include "types.h"

/// This is the playback mode.
typedef enum {
    /// Normal mode.
    PLAYBACK_MODE_NORMAL,
    /// Repeat mode.
    PLAYBACK_MODE_REPEAT,
    /// Random mode.
    PLAYBACK_MODE_RANDOM
} playback_mode_t;

typedef struct playlist_s playlist_t;
typedef struct playitem_s playitem_t;


#include "playlist_m3u.h"
#include "playlist_pls.h"
#include "playlist_xml.h"


/**
 * A callback that is called every time the content of the playlist changes.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
typedef void (*playlist_change_cb_t) (playlist_t * playlist);


/**
 * A callback that is used to retrieve the playback length of a playitem.
 *
 * @param mrl                   The MRL of the item to retrieve the playback
 *                              length of.
 */
typedef int (*playitem_length_cb_t) (const char *mrl);


/// This represents one entry of a playlist.
struct playitem_s {
    /// The title of this playitem.
    char *title;

    /// The MRL of this playitem.
    char *mrl;

    /// The MRL of the subtitle belonging to this playitem.
    char *subtitle_mrl;

    /// The MRL of the thumbnail belonging to this playitem.
    char *thumbnail_mrl;

    /**
     * This is set to TRUE once a playitem has been played. This is only used
     * in random mode.
     */
    bool played;

    /**
     * This is used to save the current position in the stream. The current
     * position in the stream is saved in seconds from the start of the stream.
     */
    int start_time;

    /// The playback length of this item in seconds.
    int playback_length;
};


/// This represents a playlist.
struct playlist_s {
    /// The list.
    l_list_t *list;

    /// A mutex to protect the list.
    pthread_mutex_t mutex;
    pthread_mutexattr_t mutex_attr;

    /// The current playitem.
    playitem_t *current;

    /// The current playmode.
    playback_mode_t playmode;

    /// The MRL of this playlist (may be NULL for nonpersistent lists).
    char *mrl;

    /// This callback will be called every time the content of this playlist changes.
    playlist_change_cb_t change_cb;

    /// If this is set to FALSE change_cb will not be called.
    bool call_change_cb;

    /// This is used to save the first position that is visible in a GUI representation of this playlist.
    int top_position;

    /// This is used to save the current position in a GUI representation of this playlist.
    int cur_position;

    /// The playback length of this list in seconds.
    int playback_length;
};


/**
 * Creates a new playlist.
 *
 * @param mrl                   The MRL of the persistent version of this
 *                              playlist.
 * @param cb                    A callback that is called every time the
 *                              content of this playlist changes.
 *
 * @return                      The new playlist object.
 */
playlist_t *playlist_new (const char *mrl, playlist_change_cb_t cb);


/**
 * Removes all playitems and frees the list.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
void playlist_free (playlist_t * playlist);


/**
 * Removes all playitems.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
void playlist_clear (playlist_t * playlist);


/**
 * Locks the playlist mutex.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
#define playlist_lock(playlist)     mutex_lock(&playlist->mutex)


/**
 * Unlocks the playlist mutex.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
#define playlist_unlock(playlist)   mutex_unlock(&playlist->mutex)


/**
 * Inserts a new item to the list at <code>position</code>.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param position              The position at which the new item is inserted.
 * @param title                 The title of the new item.
 * @param mrl                   The MRL of the new item.
 */
playitem_t *playlist_insert (playlist_t * playlist, int position,
                             const char *title, const char *mrl);


/**
 * Adds a new item to the end of the list.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param title                 The title of the new item.
 * @param mrl                   The MRL of the new item.
 */
playitem_t *playlist_add (playlist_t * playlist,
                          const char *title, const char *mrl);


/**
 * Adds a fileitem to the end of the list.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param fileitem              The fileitem that is to be added.
 */
playitem_t *playlist_add_fileitem (playlist_t * playlist,
                                   fileitem_t * fileitem);


/**
 * Removes the item from the list.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param playitem              The item that is to be removed.
 */
void playlist_remove (playlist_t * playlist, playitem_t * playitem);


/**
 * Returns the first entry to play. If we're currently in random mode the
 * returned item must not necessarily be the first entry in the list.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
playitem_t *playlist_get_first (playlist_t * playlist);


/**
 * Get previous entry to play. In random mode this may not be the previous
 * entry in the list.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
playitem_t *playlist_get_prev (playlist_t * playlist);


/**
 * Get next entry to play.  In random mode this may not be the next entry in
 * the list.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
playitem_t *playlist_get_next (playlist_t * playlist);


/**
 * Sets the current pointer.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param current               The new current pointer.
 */
playitem_t *playlist_set_current (playlist_t * playlist,
                                  playitem_t * current);


/**
 * Returns the current pointer.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
playitem_t *playlist_get_current (playlist_t * playlist);


/**
 * Returns TRUE if the current pointer is the last item. Meaning to say, that
 * a call to <code>playlist_get_next</code> will return NULL.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
bool playlist_current_is_last (playlist_t * playlist);


/**
 * Sets the current pointer.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param pos                   The position of the new current pointer in the
 *                              list.
 */
playitem_t *playlist_set_current_pos (playlist_t * playlist, int pos);


/**
 * Returns the position of the currently playing entry.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
int playlist_get_current_pos (playlist_t * playlist);


/**
 * Returns the real first entry.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
playitem_t *playlist_first (playlist_t * playlist);


/**
 * Returns the real next entry.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param playitem              The previous playitem.
 */
playitem_t *playlist_next (playlist_t * playlist, playitem_t * playitem);


/**
 * Moves the item one position towards the top of list.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param playitem              The playitem to move.
 */
void playlist_move_up (playlist_t * playlist, playitem_t * playitem);


/**
 * Moves the item one position towards the end of list.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param playitem              The playitem to move.
 */
void playlist_move_down (playlist_t * playlist, playitem_t * playitem);


/**
 * Returns the number of items in the list.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
int playlist_length (playlist_t * playlist);


/**
 * Returns the current playmode of this playlist.
 *
 * @param playlist              This is a pointer to the playlist structure.
 */
playback_mode_t playlist_get_playmode (playlist_t * playlist);


/**
 * Sets the current playmode of this playlist.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param playmode              The new playmode.
 */
void playlist_set_playmode (playlist_t * playlist, playback_mode_t playmode);


/**
 * Loads a playlist from a file. If <code>mrl</code> is NULL the MRL that
 * was passed when creating the playlist is used.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param mrl                   This may be used to override global playlist MRL.
 */
bool playlist_load (playlist_t * playlist, const char *mrl);


/**
 * This method saves a playlist to a file. If <code>mrl</code> is NULL the MRL
 * that was passed when creating the playlist is used.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @param mrl                   This may be used to override global playlist MRL.
 */
bool playlist_save (playlist_t * playlist, const char *mrl);


/**
 * Set the callback used to retrieve the playback length in seconds of a
 * playitem.
 */
void playlist_set_length_cb (playitem_length_cb_t cb);


/**
 * Returns the playback length of this playlist.
 *
 * @param playlist              This is a pointer to the playlist structure.
 * @return                      The playback length of this playlist in
 *                              seconds.
 */
int playlist_get_playback_length (playlist_t * playlist);


/**
 * Returns the MRL of a thumbnail image of a playitem.
 *
 * @param playitem              This is a pointer to the playitem structure.
 */
const char *playitem_get_thumbnail (playitem_t * playitem);


#endif /* HAVE_PLAYLIST_H */
