
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk_label.c 2412 2007-07-03 08:00:19Z mschwerin $
 */
#include "config.h"

#include "heap.h"
#include "i18n.h"
#include "odk.h"
#include "otk.h"


typedef struct {
    otk_widget_t widget;

    char *text;

    /// Does the text scroll.

    /**
     * If there is not enough space to display the complete text, it scrolls
     * across the screen.
     */
    bool scrolling;

    /// The number of calls to update_label since we last scrolled.
    int scrolling_counter;

    /// The first letter showing when scrolling.
    int scrolling_index;

    /// Upcall to get new text.

    /** 
     * If this label is updated by the update job this is the upcall to the
     * application to get new text.
     */
    otk_update_cb_t update_cb;

    /// Data to pass to upcall.
    void *update_cb_data;
} otk_label_t;


void
otk_label_set_update_cb (otk_widget_t * this,
                         otk_update_cb_t update_cb, void *update_cb_data)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LABEL))
        return;

    label->update_cb = update_cb;
    label->update_cb_data = update_cb_data;

    this->need_update = false;
    this->need_update |= label->scrolling;
    this->need_update |= (label->update_cb != NULL);
}


void
otk_label_set_scrolling (otk_widget_t * this, bool scrolling)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LABEL))
        return;

    label->scrolling = scrolling;
    label->scrolling_index = -5;
    label->scrolling_counter = 0;

    this->need_update = false;
    this->need_update |= label->scrolling;
    this->need_update |= (label->update_cb != NULL);
}


void
otk_label_set_text (otk_widget_t * this, const char *text)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LABEL))
        return;

    if (!label->text && !text)
        return;

    if (label->text && text && (strcmp (label->text, text) == 0))
        return;

    ho_free (label->text);
    label->text = text ? ho_strdup (text) : NULL;
    label->scrolling_index = -5;
    label->scrolling_counter = 0;
    this->need_repaint = true;
}


static void
label_draw (otk_widget_t * this)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LABEL))
        return;

    if (!label->text)
        return;

    odk_osd_set_font (this->odk, this->font, this->fontsize);

    int palette = otk_get_palette (this->otk, OTK_PALETTE_LABEL);

    int tw;
    int th;
    odk_get_text_size (this->odk, label->text, &tw, &th);

    char *txt = NULL;
    if (tw > this->w) {
        if (label->scrolling && (label->scrolling_index > 0)) {
            char *str = ho_strdup_printf ("%s     %s", label->text,
                                          label->text);
            char *ind = str + label->scrolling_index;
            txt = otk_trunc_text_to_width (this->otk, ind, this->w);
            ho_free (str);
        }
        else {
            txt = otk_trunc_text_to_width (this->otk, label->text, this->w);
        }
    }

    odk_draw_text (this->odk, this->x, this->y,
                   txt ? txt : label->text, this->alignment, palette);

    ho_free (txt);
    this->need_repaint = false;
}


static void
label_destroy (otk_widget_t * this)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LABEL))
        return;

    ho_free (label->text);
    ho_free (label);
}


static void
label_update (otk_widget_t * this)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LABEL))
        return;

    if (label->update_cb)
        label->update_cb (label->update_cb_data, this);

    /* If scrolling is enabled we increment the counter on every update. When
     * the counter is 0 we scroll the label. As the update job is called every
     * 250 ms we scroll the label every 1.5 seconds. */
    if (label->scrolling) {
        label->scrolling_counter = (label->scrolling_counter + 1) % 6;
        this->need_repaint = (label->scrolling_counter == 0);
    }

    /* If scrolling is enabled and a repaint is necessary, we increment the
     * index at which to start showing the text. When we reach the end of the
     * text we set the index to -5 so that the label shows the beginning of
     * the text for 7.5 seconds. */
    if (label->scrolling && this->need_repaint) {
        bool reset = (label->scrolling_index == strlen (label->text));
        label->scrolling_index = reset ? -5 : (label->scrolling_index + 1);
    }
}


otk_widget_t *
otk_label_new (otk_t * otk, int x, int y, int w,
               int alignment, const char *text)
{
    otk_label_t *label = ho_new (otk_label_t);

    otk_widget_constructor ((otk_widget_t *) label, otk,
                            OTK_WIDGET_LABEL, x, y, w, 0);

    label->widget.selectable = OTK_SELECTABLE_NONE;

    label->widget.draw = label_draw;
    label->widget.destroy = label_destroy;
    label->widget.update = label_update;

    label->widget.fontsize = 30;
    label->widget.alignment = alignment;

    label->text = text ? ho_strdup (text) : NULL;

    label->scrolling = false;
    label->scrolling_index = -5;
    label->scrolling_counter = 0;

    label->update_cb = NULL;
    label->update_cb_data = NULL;

    return (otk_widget_t *) label;
}
