
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_plugin.h 2657 2007-08-22 10:58:14Z mschwerin $
 *
 */

#ifndef HAVE_ODK_PLUGIN_H
#define HAVE_ODK_PLUGIN_H

#include "event.h"
#include "types.h"

typedef struct odk_window_s odk_window_t;

typedef void (*odk_cb) (void *data, oxine_event_t * ev);

/// This represents one window plugin.
typedef struct {
    /// The name of the plugin.
    const char *name;
    /// A description of the plugin.
    const char *desc;
    /// The xine-lib video driver.
    const char *driver;

    /**
     * The constructor function for this plugin.
     *
     * @param xine              An instance of the xine engine.
     * @param driver            The video driver to use.
     */
    odk_window_t *(*constructor) (xine_t * xine, const char *driver);
} window_plugin_desc_t;


/**
 * This is the base class of the ODK output window.
 */
struct odk_window_s {
    /**
     * The one and only xine instance. The window class may not free this
     * instance!
     */
    xine_t *xine;

    /**
     * The xine video port used by this window. The window class is
     * responsible for opening and closing the video port.
     */
    xine_video_port_t *video_port;

    /// The destructor function of this window class.
    bool (*free) (odk_window_t * this);
    /// Shows the window.
    bool (*show) (odk_window_t * this);
    /// Hides the window.
    bool (*hide) (odk_window_t * this);
    /// Moves the window.
    void (*move) (odk_window_t * this, int x, int y);
    /// Resizes the window.
    void (*resize) (odk_window_t * this, int width, int height);
    /// Moves and resizes the window.
    void (*move_resize) (odk_window_t * this, int x, int y, int width, int height);
    /// Enters the event loop.
    void (*enter_event_loop) (odk_window_t * this);
    /// Leaves the event loop.
    void (*leave_event_loop) (odk_window_t * this);
    /// Sets the event handler.
    void (*set_event_handler) (odk_window_t * this, odk_cb cb, void *data);
    /// Sets the fullscreen mode.
    void (*set_fullscreen) (odk_window_t * this, bool fullscreen);
    /// Sets the window to stay on top.
    void (*set_stay_on_top) (odk_window_t * this, bool stay_on_top);
    /// Enables or disables window borders
    void (*set_with_border) (odk_window_t * this, bool with_border);
    /// Enables or disables video visibility.
    void (*show_video) (odk_window_t * this, bool show);
    /// Returns the ID of the window.
    int (*get_window_id) (odk_window_t * this);
    /// Returns the current width of the window.
    int (*get_width) (odk_window_t * this);
    /// Returns the current height of the window.
    int (*get_height) (odk_window_t * this);
    /// Returns TRUE if the window is in fullscreen mode.
    bool (*is_fullscreen) (odk_window_t * this);
};


#endif /* HAVE_ODK_PLUGIN_H */
