
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk.c 2661 2007-08-22 15:08:37Z mschwerin $
 *
 */

#include "config.h"

#include <assert.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <xine.h>
#include <unistd.h>

#ifdef HAVE_IMAGEMAGICK
#include <sys/types.h>
#include <magick/api.h>
#include <magick/ImageMagick.h>

/* These undefs are necessary, as ImageMagick has these defined in the
 * public headers thus colliding with our own defines of the same name. */
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#endif

#include "environment.h"
#include "i18n.h"
#include "heap.h"
#include "lang.h"
#include "logger.h"
#include "oxine.h"
#include "utils.h"
#include "scheduler.h"

#include "odk.h"
#include "odk_private.h"

/* 
 * ***************************************************************************
 * Description:     This is where window plugins are to be added. The plugin
 *                  descriptions have to be defined in the plugin's header
 *                  file which may only be included in odk.c.
 * ***************************************************************************
 */
#include "odk_plugin.h"
#include "odk_fb.h"
#include "odk_x11.h"

const window_plugin_desc_t *WINDOW_PLUGINS[] = {
#ifdef HAVE_LIBX11
    x11_plugin_desc,
#endif
    fb_plugin_desc,
    NULL
};


/* Thank's to glib project */
#define CLAMP(x, low, high)  (((x) > (high)) ? (high) : (((x) < (low)) ? (low) : (x)))


/// Data needed by the seek-thread.
typedef struct {
    odk_t *odk;
    int how;
} seek_arg_t;


#ifdef HAVE_IMAGE_PLAYBACK
static void image_slideshow_job (void *odk_p);
#endif


bool
odk_get_pos_length (odk_t * odk, int *pos, int *time, int *length)
{
    xine_stream_t *stream = odk->main_stream;

    if (!stream || (xine_get_status (stream) != XINE_STATUS_PLAY)) {
        return false;
    }
#ifdef HAVE_IMAGE_PLAYBACK
    if (odk_current_is_image (odk)) {
        if (config_get_bool ("image.slideshow.enable")) {
            if (length) {
                *length = odk->image_slideshow_length * 1000;
            }
            if (time) {
                *time = odk->image_slideshow_time * 1000;
            }
        }
        else {
            if (length) {
                *length = 1;
            }
            if (time) {
                *time = 1;
            }
        }
        return true;
    }
    else
#endif /* HAVE_IMAGE_PLAYBACK */
    {
        int t = 0;
        int ret = 0;
        while (((ret = xine_get_pos_length (stream, pos, time, length)) == 0)
               && (++t < 10)) {
            /* wait before trying again */
            usleep (100000);
        }
        return (ret == 1) ? true : false;
    }
}


/// Thread to seek in the stream relative to the current position.
static void *
seek_relative_thread (void *seek_arg_p)
{
    seek_arg_t *seek_arg = (seek_arg_t *) seek_arg_p;

    odk_t *odk = seek_arg->odk;
    int how = (int) seek_arg->how;

    pthread_detach (pthread_self ());

    int mute = xine_get_param (odk->main_stream, ODK_PARAM_AO_VOLUME_MUTE);
    xine_set_param (odk->main_stream, ODK_PARAM_AO_VOLUME_MUTE, true);

    int cur_time;
    int length;
    if (odk_get_pos_length (odk, NULL, &cur_time, &length)) {
        int new_time = cur_time + (how * 1000);
        int set_time = CLAMP (new_time, 0, (length - 2000));

        if (!xine_play (odk->main_stream, 0, set_time)) {
            odk_log_stream_error (odk->main_stream, __FILE__, __LINE__,
                                  _("Could not seek '%s': %s!"),
                                  odk->current_mrl);
        }
    }

    xine_set_param (odk->main_stream, ODK_PARAM_AO_VOLUME_MUTE, mute);

    ho_free (seek_arg);
    odk->is_seeking_in_stream = false;

    return NULL;
}


/**
 * Seeks in the currently playing stream relative to the current
 * position.
 *
 * @param odk                   The ODK data object.
 * @param how                   The amount (in ms) to seek.
 */
static void
odk_seek (odk_t * odk, int how)
{
    if (odk->current_mode == ODK_MODE_NULL)
        return;
    if (odk->current_mode == ODK_MODE_LOGO)
        return;
    if (odk->is_seeking_in_stream)
        return;
    if (!xine_get_stream_info (odk->main_stream, XINE_STREAM_INFO_SEEKABLE))
        return;

    seek_arg_t *seek_arg = ho_new (seek_arg_t);
    seek_arg->odk = odk;
    seek_arg->how = how;

    odk->is_seeking_in_stream = true;

    static pthread_t seek_thread;
    if (pthread_create (&seek_thread, NULL,
                        seek_relative_thread, seek_arg) != 0) {
        error (_("Could not create seek thread: %s!"), strerror (errno));
        odk->is_seeking_in_stream = false;
        ho_free (seek_arg);
    }
}


inline bool
odk_get_audio_lang (odk_t * odk, int channel, char *lang)
{
    return xine_get_audio_lang (odk->main_stream, channel, lang);
}


inline bool
odk_get_spu_lang (odk_t * odk, int channel, char *lang)
{
    return xine_get_spu_lang (odk->main_stream, channel, lang);
}


inline const char *
odk_current_get_title (odk_t * odk)
{
    return odk->current_title;
}


inline const char *
odk_current_get_mrl (odk_t * odk)
{
    return odk->current_mrl;
}


inline bool
odk_current_is_logo_mode (odk_t * odk)
{
    return (odk->current_mode == ODK_MODE_LOGO);
}


inline bool
odk_current_is_playback_mode (odk_t * odk)
{
    return (odk->current_mode == ODK_MODE_NORMAL);
}


inline bool
odk_current_is_vcd (odk_t * odk)
{
    const char *input = xine_get_meta_info (odk->main_stream,
                                            XINE_META_INFO_INPUT_PLUGIN);

    return (input && (strcasecmp (input, "vcd") == 0));
}


inline bool
odk_current_is_cdda (odk_t * odk)
{
    const char *input = xine_get_meta_info (odk->main_stream,
                                            XINE_META_INFO_INPUT_PLUGIN);

    return (input && (strcasecmp (input, "cdda") == 0));
}


inline bool
odk_current_is_dvd (odk_t * odk)
{
    const char *input = xine_get_meta_info (odk->main_stream,
                                            XINE_META_INFO_INPUT_PLUGIN);

    return (input && (strcasecmp (input, "dvd") == 0));
}


inline bool
odk_current_is_image (odk_t * odk)
{
#ifdef HAVE_IMAGE_PLAYBACK
    return is_file_image (odk->current_mrl);
#else
    return false;
#endif
}


inline bool
odk_current_is_v4l (odk_t * odk)
{
    const char *input = xine_get_meta_info (odk->main_stream,
                                            XINE_META_INFO_INPUT_PLUGIN);

    return (input && (strncasecmp (input, "v4l", 3) == 0));
}


inline bool
odk_current_is_dvb (odk_t * odk)
{
    const char *input = xine_get_meta_info (odk->main_stream,
                                            XINE_META_INFO_INPUT_PLUGIN);

    return (input && (strcasecmp (input, "dvb") == 0));
}


inline bool
odk_current_is_vdr (odk_t * odk)
{
    const char *input = xine_get_meta_info (odk->main_stream,
                                            XINE_META_INFO_INPUT_PLUGIN);

    return (input && (strcasecmp (input, "vdr") == 0));
}


inline bool
odk_current_is_television (odk_t * odk)
{
    return (odk_current_is_v4l (odk)
            || odk_current_is_dvb (odk)
            || odk_current_is_vdr (odk));
}


inline bool
odk_current_is_audio (odk_t * odk)
{
    return (!odk_current_is_image (odk)
            && !odk_current_has_video (odk));
}


inline bool
odk_current_has_audio (odk_t * odk)
{
    return xine_get_stream_info (odk->main_stream,
                                 XINE_STREAM_INFO_HAS_AUDIO);
}


inline bool
odk_current_has_video (odk_t * odk)
{
    int has_video = (odk->novideo_post != NULL);

    has_video |= xine_get_stream_info (odk->main_stream,
                                       XINE_STREAM_INFO_HAS_VIDEO);
    has_video |= xine_get_stream_info (odk->animation_stream,
                                       XINE_STREAM_INFO_HAS_VIDEO);

    return has_video;
}


inline bool
odk_current_has_chapters (odk_t * odk)
{
    return xine_get_stream_info (odk->main_stream,
                                 XINE_STREAM_INFO_HAS_CHAPTERS);
}


bool
odk_is_available_audio_driver (odk_t * odk, const char *audio_driver)
{
    const char *const *driver_ids;
    if ((driver_ids = xine_list_audio_output_plugins (odk->xine))) {
        const char *driver_id = *driver_ids++;
        while (driver_id) {
            if (strcasecmp (audio_driver, driver_id) == 0) {
                return true;
            }
            driver_id = *driver_ids++;
        }
    }

    return false;
}


bool
odk_is_supported_video_driver (const char *video_driver)
{
    int i = 0;
    const window_plugin_desc_t *plugin_p = WINDOW_PLUGINS[i];

    while (WINDOW_PLUGINS[i]) {
        while (plugin_p->driver) {
            if (strcasecmp (plugin_p->driver, video_driver) == 0) {
                return true;
            }
            plugin_p++;
        }
        plugin_p = WINDOW_PLUGINS[++i];
    }

    return false;
}


char *
odk_get_meta_info (odk_t * odk, meta_info_t type)
{
    return meta_info_get_from_stream (type, odk->main_stream);
}


uint32_t
odk_get_stream_info (odk_t * odk, odk_stream_info_t info)
{
    return xine_get_stream_info (odk->main_stream, info);
}


char *
odk_get_stream_param_name (odk_t * odk, odk_stream_param_t param)
{
    char *param_name = NULL;

    switch (param) {
    case ODK_PARAM_AUDIO_CHANNEL:
        param_name = ho_strdup (_("Audio Channel"));
        break;
    case ODK_PARAM_AUDIO_OFFSET:
        param_name = ho_strdup (_("Audio-Video Offset"));
        break;
    case ODK_PARAM_SPU_CHANNEL:
        param_name = ho_strdup (_("Subtitle Channel"));
        break;
    case ODK_PARAM_SPU_OFFSET:
        param_name = ho_strdup (_("Subtitle Offset"));
        break;
    case ODK_PARAM_VO_DEINTERLACE:
        param_name = ho_strdup (_("Deinterlace"));
        break;
    case ODK_PARAM_VO_ASPECT_RATIO:
        param_name = ho_strdup (_("Aspect Ratio"));
        break;
    case ODK_PARAM_VO_HUE:
        param_name = ho_strdup (_("Hue"));
        break;
    case ODK_PARAM_VO_SATURATION:
        param_name = ho_strdup (_("Saturation"));
        break;
    case ODK_PARAM_VO_CONTRAST:
        param_name = ho_strdup (_("Contrast"));
        break;
    case ODK_PARAM_VO_BRIGHTNESS:
        param_name = ho_strdup (_("Brightness"));
        break;
    case ODK_PARAM_VO_ZOOM_Y:
    case ODK_PARAM_VO_ZOOM_X:
        param_name = ho_strdup (_("Zoom"));
        break;
    case ODK_PARAM_AO_VOLUME:
        param_name = ho_strdup (_("Audio volume"));
        break;
    case ODK_PARAM_AO_VOLUME_MUTE:
        param_name = ho_strdup (_("Audio state"));
        break;
    default:
        break;
    }

    if (!param_name) {
        fatal (_("Unknown parameter %d!"), param);
        abort ();
    }

    return param_name;
}


char *
odk_get_stream_param_value_as_string (odk_t * odk, odk_stream_param_t param,
                                      int param_value)
{
    char *param_string = NULL;

    switch (param) {
    case ODK_PARAM_SPEED:
        switch (param_value) {
        case ODK_SPEED_NORMAL:
            param_string = ho_strdup (_("Playing"));
            break;
        case ODK_SPEED_PAUSE:
            param_string = ho_strdup (_("Paused"));
            break;
        case ODK_SPEED_SLOW_4:
            param_string = ho_strdup (_("Extra Slow Motion"));
            break;
        case ODK_SPEED_SLOW_2:
            param_string = ho_strdup (_("Slow Motion"));
            break;
        case ODK_SPEED_FAST_2:
            param_string = ho_strdup (_("Fast Forward"));
            break;
        case ODK_SPEED_FAST_4:
            param_string = ho_strdup (_("Extra Fast Forward"));
            break;
        }
        break;
    case ODK_PARAM_AUDIO_CHANNEL:
        if (param_value == AUDIO_CHANNEL_OFF) {
            param_string = ho_strdup (_("none"));
        }
        else if (param_value == AUDIO_CHANNEL_AUTO) {
            param_string = ho_strdup (_("auto"));
        }
        else {
            char lang[XINE_LANG_MAX];
            if (odk_get_audio_lang (odk, param_value, lang)) {
                const char *isolang = get_language_from_iso639_1 (lang);
                param_string = ho_strdup_printf (isolang);
            }
            else {
                param_string = ho_strdup_printf (_("unknown language (%d)"),
                                                 param_value);
            }
        }
        break;
    case ODK_PARAM_SPU_CHANNEL:
        if (param_value == SPU_CHANNEL_OFF) {
            param_string = ho_strdup (_("none"));
        }
        else if (param_value == SPU_CHANNEL_AUTO) {
            param_string = ho_strdup (_("auto"));
        }
        else {
            char lang[XINE_LANG_MAX];
            if (odk_get_spu_lang (odk, param_value, lang)) {
                const char *isolang = get_language_from_iso639_1 (lang);
                param_string = ho_strdup_printf (isolang);
            }
            else {
                param_string = ho_strdup_printf (_("unknown language (%d)"),
                                                 param_value);
            }
        }
        break;
    case ODK_PARAM_VO_DEINTERLACE:
        if (param_value) {
            param_string = ho_strdup (_("enabled"));
        }
        else {
            param_string = ho_strdup (_("disabled"));
        }
        break;
    case ODK_PARAM_VO_ASPECT_RATIO:
        switch (param_value) {
        case XINE_VO_ASPECT_AUTO:
            param_string = ho_strdup (_("auto"));
            break;
        case XINE_VO_ASPECT_SQUARE:
            param_string = ho_strdup (_("square"));
            break;
        case XINE_VO_ASPECT_4_3:
            param_string = ho_strdup (_("4:3"));
            break;
        case XINE_VO_ASPECT_ANAMORPHIC:
            param_string = ho_strdup (_("anamorphic"));
            break;
        case XINE_VO_ASPECT_DVB:
            param_string = ho_strdup (_("DVB"));
            break;
        default:
            break;
        }
        break;
    case ODK_PARAM_VO_ZOOM_X:
    case ODK_PARAM_VO_ZOOM_Y:
        {
            int zoom_x = odk_get_stream_param (odk, ODK_PARAM_VO_ZOOM_X);
            int zoom_y = odk_get_stream_param (odk, ODK_PARAM_VO_ZOOM_Y);

            if (zoom_x != zoom_y) {
                param_string = ho_strdup_printf ("x=%d %% y=%d %%",
                                                 zoom_x, zoom_y);
            }
            else {
                param_string = ho_strdup_printf ("%d %%", zoom_x);
            }
        }
        break;
    case ODK_PARAM_VO_HUE:
    case ODK_PARAM_VO_SATURATION:
    case ODK_PARAM_VO_CONTRAST:
    case ODK_PARAM_VO_BRIGHTNESS:
        {
            param_string = ho_strdup_printf ("%d %%", param_value / 655);
        }
        break;
    case ODK_PARAM_SPU_OFFSET:
    case ODK_PARAM_AUDIO_OFFSET:
        {
            double val = (double) param_value / 90.0;
            param_string = ho_strdup_printf ("%4.2f ms", val);
        }
        break;
    case ODK_PARAM_AO_VOLUME:
        {
            int mute = odk_get_stream_param (odk, ODK_PARAM_AO_VOLUME_MUTE);
            if (mute) {
                param_string = ho_strdup_printf ("%d %% (%s)",
                                                 param_value, _("muted"));
            }
            else {
                param_string = ho_strdup_printf ("%d %%", param_value);
            }
        }
        break;
    case ODK_PARAM_AO_VOLUME_MUTE:
        {
            if (param_value) {
                param_string = ho_strdup_printf (_("muted"));
            }
            else {
                param_string = ho_strdup_printf (_("unmuted"));
            }
        }
        break;
    default:
        break;
    }

    if (!param_string) {
#ifdef DEBUG
        fatal ("Unknown parameter 0x%X!", param);
        abort ();
#else
        param_string = ho_strdup ("");
#endif
    }

    return param_string;
}


char *
odk_get_stream_param_as_string (odk_t * odk, odk_stream_param_t param)
{
    int param_value = odk_get_stream_param (odk, param);

    return odk_get_stream_param_value_as_string (odk, param, param_value);
}


int
odk_get_stream_param (odk_t * odk, odk_stream_param_t param)
{
    if (param == ODK_PARAM_POSITION) {
        int time;
        int length;
        int pos = 1;
        if (!odk_get_pos_length (odk, NULL, &time, &length)) {
            return -1;
        }
        if (length) {
            pos = round ((double) time * 100.0 / (double) length);
        }
        return pos;
    }

    return xine_get_param (odk->main_stream, param);
}


int
odk_set_stream_param (odk_t * odk, odk_stream_param_t param, int value)
{
    switch (param) {
    case ODK_PARAM_POSITION:
#ifdef HAVE_IMAGE_PLAYBACK
        if (odk_current_is_image (odk)) {
            int length = odk->image_slideshow_length;
            cancel_job (odk->image_slideshow_job);
            odk->image_slideshow_time =
                round ((double) value * (double) length / 100.0);
            odk->image_slideshow_job =
                schedule_job (1000, image_slideshow_job, odk);
            return value;
        }
        else
#endif /* HAVE_IMAGE_PLAYBACK */
        {
            int cur_time;
            int length_time;

            if (!odk_get_pos_length (odk, NULL, &cur_time, &length_time)) {
                return 0;
            }

            int how = ((length_time * value / 100) - cur_time) / 1000;

            odk_seek (odk, how);

            return value;
        }
        break;
    case ODK_PARAM_SPEED:
        /* If its an audio-only stream we ignore any speed but normal and
         * pause. */
        if (odk_current_is_audio (odk)) {
            if (value == ODK_SPEED_NORMAL) {
                /* Do nothing. */
            }
            else if (value == ODK_SPEED_PAUSE) {
                /* Do nothing. */
            }
            else {
                return xine_get_param (odk->main_stream, param);
            }
        }
#ifdef HAVE_IMAGE_PLAYBACK
        /* If it's an image we ignore any speed but normal and pause. We
         * cancel the slideshow job on pause and restart it on normal. */
        else if (odk_current_is_image (odk)) {
            if (!config_get_bool ("image.slideshow.enable")) {
                /* Do nothing. */
            }
            else if (value == ODK_SPEED_NORMAL) {
                odk->image_slideshow_job =
                    schedule_job (1000, image_slideshow_job, odk);
            }
            else if (value == ODK_SPEED_PAUSE) {
                cancel_job (odk->image_slideshow_job);
                odk->image_slideshow_job = 0;
            }
        }
#endif /* HAVE_IMAGE_PLAYBACK */
        break;
    case ODK_PARAM_SPU_CHANNEL:
        /* We never want to turn subtitles off, as this affects our OSD. */
        if (value == SPU_CHANNEL_OFF) {
            value = SPU_CHANNEL_AUTO;
        }
        break;
    default:
        break;
    }

    xine_set_param (odk->main_stream, param, value);
    value = xine_get_param (odk->main_stream, param);

    switch (param) {
    case ODK_PARAM_VO_ASPECT_RATIO:
        /* We save this so we can restore this value in non-logo mode. */
        if (odk->current_mode != ODK_MODE_LOGO) {
            odk->aspect_ratio = value;
        }
        break;
    case ODK_PARAM_VO_BRIGHTNESS:
        {
            config_set_number ("video.filter.brightness", value / 655);
        }
        break;
    case ODK_PARAM_VO_CONTRAST:
        {
            config_set_number ("video.filter.contrast", value / 655);
        }
        break;
    case ODK_PARAM_VO_HUE:
        {
            config_set_number ("video.filter.hue", value / 655);
        }
        break;
    case ODK_PARAM_VO_SATURATION:
        {
            config_set_number ("video.filter.saturation", value / 655);
        }
        break;
    default:
        break;
    }

    return value;
}


int
odk_change_stream_param (odk_t * odk, odk_stream_param_t param,
                         int how, int min, int max)
{
    if (param == ODK_PARAM_POSITION) {
        odk_seek (odk, how);
        return 0;
    }

    int old_value = xine_get_param (odk->main_stream, param);
    int new_value = old_value + how;

    switch (param) {
    case ODK_PARAM_SPU_CHANNEL:
    case ODK_PARAM_AUDIO_CHANNEL:
    case ODK_PARAM_AO_VOLUME_MUTE:
    case ODK_PARAM_VO_ASPECT_RATIO:
    case ODK_PARAM_VO_DEINTERLACE:
        if (new_value > max) {
            new_value = min;
        }
        if (new_value < min) {
            new_value = max;
        }
        break;
    default:
        if (new_value > max) {
            new_value = max;
        }
        if (new_value < min) {
            new_value = min;
        }
        break;
    }

    return odk_set_stream_param (odk, param, new_value);
}


/**
 * Stops the current stream. Disposes the novideo background stream.
 * Releases and destructors the novideo post plugin. Releases strings.
 */
static void
stream_stop (odk_t * odk)
{
#ifdef HAVE_IMAGE_PLAYBACK
    cancel_job (odk->image_slideshow_job);
    odk->image_slideshow_job = 0;
#endif

    if (odk->novideo_post) {
        odk_post_audio_unwire (odk);
        odk->novideo_post = NULL;
    }

    ho_free (odk->current_mrl);
    ho_free (odk->current_subtitle_mrl);
    ho_free (odk->current_title);

    playlist_clear (odk->current_alternatives);

    odk->is_seeking_in_stream = false;
    odk->current_mode = ODK_MODE_NULL;

    xine_close (odk->subtitle_stream);
    xine_close (odk->animation_stream);
    xine_close (odk->background_stream);
    xine_close (odk->main_stream);
}


bool
odk_stop_stream (odk_t * odk)
{
    stream_stop (odk);

    oxine_event_t ev;
    ev.type = OXINE_EVENT_PLAYBACK_STOPPED;
    odk_oxine_event_send (odk, &ev);

    return true;
}


/// Logs xine errors.
void
odk_log_stream_error (xine_stream_t * stream, const char *file, int line,
                      const char *err_msg, const char *mrl)
{
    int ret = xine_get_error (stream);
    char *xine_msg = _("Unknown error");
    switch (ret) {
    case XINE_ERROR_NO_INPUT_PLUGIN:
        xine_msg = _("No input plugin");
        break;
    case XINE_ERROR_NO_DEMUX_PLUGIN:
        xine_msg = _("Could not find demuxer plugin");
        break;
    case XINE_ERROR_DEMUX_FAILED:
        xine_msg = _("Demuxer plugin failed");
        break;
    case XINE_ERROR_MALFORMED_MRL:
        xine_msg = _("Malformed MRL");
        break;
    default:
        break;
    }
    _log (LEVEL_ERROR, file, line, err_msg, mrl, xine_msg);
}


static bool
xine_open_and_play (xine_stream_t * stream, const char *mrl,
                    const char *file, int line)
{
    if (!xine_open (stream, mrl)) {
        odk_log_stream_error (stream, file, line,
                              _("Could not open '%s': %s!"), mrl);
        return false;
    }

    if (!xine_play (stream, 0, 0)) {
        odk_log_stream_error (stream, file, line,
                              _("Could not play '%s': %s!"), mrl);
        return false;
    }

    return true;
}


static char *
get_animation_stream_mrl (void)
{
    char *mrl = ho_strdup (config_get_string ("visual_anim."
                                              "animation_stream_mrl"));

    if (!file_exists (mrl)) {
        ho_free (mrl);
        mrl = ho_strdup (OXINE_VISUALDIR "/animation.mpg");
    }

    return mrl;
}


static char *
get_background_stream_mrl (void)
{
    char *mrl = ho_strdup_printf ("%s/backgrounds/%s",
                                  get_dir_oxine_skin (),
                                  config_get_string ("visual_anim."
                                                     "background_stream_mrl"));

    if (!file_exists (mrl)) {
        ho_free (mrl);
        mrl = ho_strdup_printf ("%s/backgrounds/menu_main.png",
                                get_dir_oxine_skin ());
    }

    return mrl;
}


static bool
mutexed_play (odk_t * odk, const char *mrl, const char *sub_mrl,
              int start_time, odk_mode_t mode)
{
    bool result = false;

    assert (odk);
    assert (mrl);
    assert (mode != ODK_MODE_NULL);

    info (_("    main MRL: '%s'"), mrl);
    info (_("subtitle MRL: '%s'"), sub_mrl ? sub_mrl : _("no subtitle"));

    /* Because the character '#' has a special meaning to xine-lib MRL's
     * that contain this characters without it having any special meaning
     * may not be played correctly. To prevent this we encode the MRL before
     * passing it to xine-lib. If the MRL is a file on a local volume we can
     * always encode it. If this is not the case we put it up to the user to
     * decide if he wants to encode all MRL's. This is not really a good
     * idea, as it will prevent oxine from playing streams like VDR where
     * the '#' is needed. */
    char *encoded_mrl = NULL;
    if (file_exists (mrl)) {
        encoded_mrl = filename_escape_to_uri (mrl);
        debug ("   encoded MRL: '%s'", encoded_mrl);
    }
    else if (config_get_bool ("misc.encode_mrls")) {
        encoded_mrl = filename_escape_to_uri (mrl);
        debug ("   encoded MRL: '%s'", encoded_mrl);
    }
    else {
        encoded_mrl = ho_strdup (mrl);
    }

    /* We can always encode the subtitle MRL, as this is always generated by
     * oxine and will thus not contain a '#' with a special meaning. */
    char *encoded_sub_mrl = NULL;
    encoded_sub_mrl = filename_escape_to_uri (sub_mrl);

    /* Stop any stream that might be running. */
    stream_stop (odk);
    assert (odk->current_mode == ODK_MODE_NULL);

    /* We try to open the main stream. */
    if (!xine_open (odk->main_stream, encoded_mrl)) {
        odk_log_stream_error (odk->main_stream, __FILE__, __LINE__,
                              _("Could not open '%s': %s!"), mrl);
        goto out_free;
    }

    int has_video = odk_get_stream_info (odk, ODK_STREAM_INFO_HAS_VIDEO);

    /* We always set the aspect ratio to square in logo mode. This is
     * necessary so the backgrounds are displayed correctly. */
    if (mode == ODK_MODE_LOGO) {
        xine_set_param (odk->main_stream, ODK_PARAM_VO_ASPECT_RATIO,
                        ODK_VO_ASPECT_RATIO_SQUARE);
    }
    else {
        xine_set_param (odk->main_stream, ODK_PARAM_VO_ASPECT_RATIO,
                        odk->aspect_ratio);
    }

    /* Reset the zoom factor. */
    int zoom = config_get_number ("video.zoom");
    xine_set_param (odk->main_stream, ODK_PARAM_VO_ZOOM_X, zoom);
    xine_set_param (odk->main_stream, ODK_PARAM_VO_ZOOM_Y, zoom);

    /* Reset audio-video and subtitle offset. */
    xine_set_param (odk->main_stream, ODK_PARAM_AUDIO_OFFSET, 0);
    xine_set_param (odk->main_stream, ODK_PARAM_SPU_OFFSET, 0);

    /* Reset audio channel to auto if selected channel does not exist, or
     * the stream has no video. */
    int audio_channel_cur = xine_get_param (odk->main_stream,
                                            ODK_PARAM_AUDIO_CHANNEL);
    int audio_channel_max = xine_get_param (odk->main_stream,
                                            ODK_STREAM_INFO_MAX_AUDIO_CHANNEL);
    if (audio_channel_cur > audio_channel_max) {
        xine_set_param (odk->main_stream, ODK_PARAM_AUDIO_CHANNEL,
                        AUDIO_CHANNEL_AUTO);
    }
    if (!has_video && (audio_channel_cur == AUDIO_CHANNEL_OFF)) {
        xine_set_param (odk->main_stream, ODK_PARAM_AUDIO_CHANNEL,
                        AUDIO_CHANNEL_AUTO);
    }

    /* Reset subtitle channel to auto if selected channel does not exist. */
    int spu_channel_cur = xine_get_param (odk->main_stream,
                                          ODK_PARAM_SPU_CHANNEL);
    int spu_channel_max = xine_get_param (odk->main_stream,
                                          ODK_STREAM_INFO_MAX_SPU_CHANNEL);
    if (spu_channel_cur > spu_channel_max) {
        xine_set_param (odk->main_stream, ODK_PARAM_SPU_CHANNEL,
                        SPU_CHANNEL_AUTO);
    }

    /* Set background for streams without video. */
    if (!has_video) {
        switch (config_get_number ("visual_anim.anim_type")) {
        case 0:
            /* Show a background animation. */
            {
                char *mrl = get_animation_stream_mrl ();
                if (!xine_open_and_play (odk->animation_stream, mrl,
                                         __FILE__, __LINE__)) {
                    abort ();
                }
                else {
                    ho_free (mrl);
                }
            }
            break;
        case 1:
            /* Show standard background */
            {
                char *mrl = get_background_stream_mrl ();
                if (!xine_open_and_play (odk->background_stream, mrl,
                                         __FILE__, __LINE__)) {
                    abort ();
                }
                else {
                    odk->current_background_mrl = mrl;
                }
            }
            break;
        case 2:
            /* Show a post plugin. */
            {
                odk_post_audio_rewire (odk);
            }
            break;
        }
    }

    /* We try to play the subtitle stream. */
    if (encoded_sub_mrl) {
        if (xine_open (odk->subtitle_stream, encoded_sub_mrl)) {
            xine_stream_master_slave (odk->main_stream,
                                      odk->subtitle_stream,
                                      XINE_MASTER_SLAVE_PLAY |
                                      XINE_MASTER_SLAVE_STOP |
                                      XINE_MASTER_SLAVE_SPEED);
        }
        else {
            odk_log_stream_error (odk->subtitle_stream, __FILE__, __LINE__,
                                  _("Could not open '%s': %s!"), sub_mrl);
        }
    }

    /* We try to play the main stream. */
    if (!xine_play (odk->main_stream, 0, start_time)) {
        odk_log_stream_error (odk->main_stream, __FILE__, __LINE__,
                              _("Could not play '%s': %s!"), mrl);
        goto out_free;
    }

    result = true;
  out_free:
    ho_free (encoded_mrl);
    ho_free (encoded_sub_mrl);

#ifdef XINE_PARAM_GAPLESS_SWITCH
    if (!result) {
        if (xine_check_version (1, 1, 1)) {
            xine_set_param (odk->main_stream, XINE_PARAM_GAPLESS_SWITCH, 0);
        }
    }
#endif

    return result;
}


#ifdef HAVE_IMAGE_PLAYBACK
static void
image_slideshow_job (void *odk_p)
{
    odk_t *odk = (odk_t *) odk_p;

    int time = odk->image_slideshow_time;
    int length = odk->image_slideshow_length;

    if (time == length) {
        odk->current_mode = ODK_MODE_NULL;

        oxine_event_t ev;
        ev.type = OXINE_EVENT_PLAYBACK_FINISHED;
        odk_oxine_event_send (odk, &ev);
    }
    else {
        odk->image_slideshow_time++;
        odk->image_slideshow_job =
            schedule_job (1000, image_slideshow_job, odk);
    }
}
#endif /* HAVE_IMAGE_PLAYBACK */


#ifdef HAVE_IMAGE_ROTATION
void
odk_rotate_current_image (odk_t * odk, int angle)
{
    if (!odk_current_is_playback_mode (odk)) {
        return;
    }
    if (!is_file_image (odk->current_mrl)) {
        return;
    }
    if (!((-360 < angle) && (angle < 360))) {
        warn ("Invalid angle -360° < %d < 360°!", angle);
        return;
    }
    if (angle == 0) {
        return;
    }

    cancel_job (odk->image_slideshow_job);
    odk->image_slideshow_job = 0;

    InitializeMagick (NULL);

    ExceptionInfo exception;
    GetExceptionInfo (&exception);

    ImageInfo *image_info = CloneImageInfo ((ImageInfo *) NULL);
    if (!image_info) {
        error (_("Could not read image '%s'!"), odk->current_mrl);

        DestroyExceptionInfo (&exception);
        DestroyMagick ();

        return;
    }
    strncpy (image_info->filename, odk->current_mrl, MaxTextExtent);

    Image *image = ReadImage (image_info, &exception);
    CatchException (&exception);
    if (!image) {
        error (_("Could not read image '%s'!"), odk->current_mrl);

        DestroyImageInfo (image_info);
        DestroyExceptionInfo (&exception);
        DestroyMagick ();

        return;
    }

    Image *rotated = RotateImage (image, angle, &exception);
    CatchException (&exception);
    if (!rotated) {
        error (_("Could not rotate image '%s'!"), odk->current_mrl);

        DestroyImage (image);
        DestroyImageInfo (image_info);
        DestroyExceptionInfo (&exception);
        DestroyMagick ();

        return;
    }

    WriteImage (image_info, rotated);
    CatchException (&exception);

    DestroyImage (rotated);
    DestroyImage (image);
    DestroyImageInfo (image_info);
    DestroyExceptionInfo (&exception);
    DestroyMagick ();

    char *title = ho_strdup (odk->current_title);
    char *mrl = ho_strdup (odk->current_mrl);
    odk_play_stream (odk, title, mrl, NULL, 0, odk->current_mode);
    ho_free (mrl);
    ho_free (title);
}
#endif /* HAVE_IMAGE_ROTATION */


bool
odk_play_stream (odk_t * odk, const char *title,
                 const char *mrl, const char *sub_mrl,
                 int start_time, odk_mode_t mode)
{
    assert (odk);
    assert (mrl);
    assert (title);
    assert (mode != ODK_MODE_NULL);

    bool result = mutexed_play (odk, mrl, sub_mrl, start_time, mode);

    if (result) {
        /* Set the title. */
        odk->current_title = ho_strdup (title);

        /* Set the mode and MRL. */
        odk->current_mrl = ho_strdup (mrl);
        odk->current_subtitle_mrl = sub_mrl ? ho_strdup (sub_mrl) : NULL;
        odk->current_mode = mode;

#ifdef HAVE_IMAGE_PLAYBACK
        /* Start the slideshow job if we're showing an image. */
        if ((mode != ODK_MODE_LOGO) && is_file_image (mrl)
            && config_get_bool ("image.slideshow.enable")) {

            odk->image_slideshow_time = 0;
            odk->image_slideshow_length =
                config_get_number ("image.slideshow.delay");
            odk->image_slideshow_job =
                schedule_job (1000, image_slideshow_job, odk);
        }
#endif /* HAVE_IMAGE_PLAYBACK */

        /* Adapt the size of the OSD to the new stream. */
        odk_osd_adapt_size (odk, NULL);

        /* Send an event to the frontend telling it that 
         * playback has started. */
        oxine_event_t ev;
        ev.type = OXINE_EVENT_PLAYBACK_STARTED;
        odk_oxine_event_send (odk, &ev);
    }

    else {
        ho_free (odk->current_title);
        ho_free (odk->current_mrl);
        ho_free (odk->current_subtitle_mrl);
        odk->current_mode = ODK_MODE_NULL;
    }

    /* This is necessary as xine-lib sometimes resets the locale. I guess,
     * that this must be a bug in xine-lib. */
#ifdef HAVE_SETLOCALE
    if (!setlocale (LC_ALL, "")) {
        error ("The current locale is not supported by the C library!");
    }
#endif

    return result;
}


static void
odk_unmute_job (void *odk_p)
{
    odk_t *odk = (odk_t *) odk_p;

    xine_set_param (odk->main_stream, ODK_PARAM_AO_VOLUME_MUTE, false);
}


void
odk_play_background_stream (odk_t * odk, const char *mrl)
{
    if ((config_get_number ("visual_anim.anim_type") != 1)
        || odk_current_is_logo_mode (odk)
        || odk_current_has_video (odk)) {
        return;
    }

    if (!mrl) {
        char *mrl = get_background_stream_mrl ();
        odk_play_background_stream (odk, mrl);
        ho_free (mrl);
        return;
    }

    if (strcmp (mrl, odk->current_background_mrl) == 0) {
        return;
    }

    /* OK, now it gets a bit tricky. Because xine-lib resets the stream-speed
     * for all streams to running when xine_play is called we have to reset
     * the speed after we've called xine_play. */
    int mute = xine_get_param (odk->main_stream, ODK_PARAM_AO_VOLUME_MUTE);
    int speed = xine_get_param (odk->main_stream, ODK_PARAM_SPEED);
    if (speed == ODK_SPEED_PAUSE) {
        xine_set_param (odk->main_stream, ODK_PARAM_AO_VOLUME_MUTE, true);
    }

    xine_close (odk->background_stream);
    if (!xine_open_and_play (odk->background_stream, mrl, __FILE__, __LINE__)) {
        abort ();
    }

    ho_free (odk->current_background_mrl);
    odk->current_background_mrl = ho_strdup (mrl);

    /* We have to wait a few seconds before unmuting the stream so that the
     * short playback periods are supressed. Because we don't want to block
     * the GUI, we start a job to unmute the main stream. */
    if (speed == ODK_SPEED_PAUSE) {
        xine_set_param (odk->main_stream, ODK_PARAM_SPEED, ODK_SPEED_PAUSE);
        if (!mute) {
            schedule_job (5000, odk_unmute_job, odk);
        }
    }
}


void
odk_show_window (odk_t * odk, bool fullscreen, bool show_border,
                 bool stay_on_top, int x, int y, int w, int h)
{
    odk->win->show_video (odk->win, false);
    odk->win->set_with_border (odk->win, show_border);
    odk->win->show (odk->win);
    odk->win->set_stay_on_top (odk->win, stay_on_top);
    odk->win->move_resize (odk->win, x, y, w, h);
    odk->win->set_fullscreen (odk->win, fullscreen);
    odk_osd_adapt_size (odk, NULL);
    odk->win->show_video (odk->win, true);
}


inline void
odk_hide_window (odk_t * odk)
{
    odk->win->hide (odk->win);
}


void
odk_window_set_geometry (odk_t * odk, int x, int y, int w, int h)
{
    odk->win->move_resize (odk->win, x, y, w, h);
}


void
odk_window_set_fullscreen (odk_t * odk, bool fullscreen)
{
    odk->win->set_fullscreen (odk->win, fullscreen);
    odk_osd_adapt_size (odk, NULL);
}


inline bool
odk_window_is_fullscreen (odk_t * odk)
{
    return odk->win->is_fullscreen (odk->win);
}


inline void
odk_run (odk_t * odk)
{
    debug (" event loop thread: 0x%X", (int) pthread_self ());
    odk->win->enter_event_loop (odk->win);
}


inline void
odk_exit (odk_t * odk)
{
    odk->win->leave_event_loop (odk->win);
}


/// Creates the video output window.
static odk_window_t *
odk_create_window (xine_t * xine, const char *video_driver)
{
    int i = 0;
    odk_window_t *window = NULL;
    const window_plugin_desc_t *plugin_p = WINDOW_PLUGINS[i];

    while (!window && WINDOW_PLUGINS[i]) {
        /* Try to find our preferred video driver */
        while (plugin_p->driver
               && strcasecmp (plugin_p->driver, video_driver)) {
            plugin_p++;
        }

        /* Try to init window plugin. */
        if (plugin_p->driver) {
            debug ("Driver '%s' found in odk_plugin '%s'.", video_driver,
                   WINDOW_PLUGINS[i]->name);
            window = plugin_p->constructor (xine, plugin_p->driver);
        }
        else {
            debug ("Driver '%s' not found in odk_plugin '%s'.",
                   video_driver, WINDOW_PLUGINS[i]->name);
        }

        plugin_p = WINDOW_PLUGINS[++i];
    }

    /* Some sanity checks. */
#ifdef DEBUG
    if (window) {
        assert (window->show);
        assert (window->hide);
        assert (window->move);
        assert (window->free);
        assert (window->enter_event_loop);
        assert (window->leave_event_loop);
        assert (window->set_fullscreen);
        assert (window->is_fullscreen);
        assert (window->set_event_handler);
        assert (window->get_width);
        assert (window->get_height);
        assert (window->get_window_id);
    }
#endif

    return window;
}


static void
odk_config_changed_handler (void *odk_p, oxine_event_t * event)
{
    odk_t *odk = (odk_t *) odk_p;
    xine_cfg_entry_t *entry = event->data.config.entry;

    if (event->type != OXINE_EVENT_CONFIG_CHANGED) {
        return;
    }
    else if (strcasecmp (entry->key, "video.filter.brightness") == 0) {
        int value = entry->num_value * 655;
        xine_set_param (odk->main_stream, ODK_PARAM_VO_BRIGHTNESS, value);
    }
    else if (strcasecmp (entry->key, "video.filter.contrast") == 0) {
        int value = entry->num_value * 655;
        xine_set_param (odk->main_stream, ODK_PARAM_VO_CONTRAST, value);
    }
    else if (strcasecmp (entry->key, "video.filter.hue") == 0) {
        int value = entry->num_value * 655;
        xine_set_param (odk->main_stream, ODK_PARAM_VO_HUE, value);
    }
    else if (strcasecmp (entry->key, "video.filter.saturation") == 0) {
        int value = entry->num_value * 655;
        xine_set_param (odk->main_stream, ODK_PARAM_VO_SATURATION, value);
    }
    else if (strcasecmp (entry->key, "video.zoom") == 0) {
        int value = entry->num_value;
        xine_set_param (odk->main_stream, ODK_PARAM_VO_ZOOM_X, value);
        xine_set_param (odk->main_stream, ODK_PARAM_VO_ZOOM_Y, value);
    }
}


static void
odk_register_configuration (odk_t * odk)
{
    char *visual_anims[] = {
        "background animation",
        "background image",
        "post plugin",
        NULL
    };

#ifdef HAVE_IMAGE_PLAYBACK
    config_register_number ("image.slideshow.delay", 10,
                            _("slideshow delay (in s)"));

    config_register_bool ("image.slideshow.enable", true,
                          _("enable slideshow support"));

#ifdef HAVE_LIBEXIF
    config_register_bool ("image.sort.by_date", true,
                          _("sort images using EXIF "
                            "date information (slow)"));
#endif
#endif

    config_register_bool ("misc.encode_mrls", false,
                          _("encode special characters in MRLs"));

    config_register_bool ("video.osd.use_unscaled", true,
                          _("use unscaled OSD if available"));

    config_register_enum ("visual_anim.anim_type", 1, visual_anims,
                          _("audio visualization type"));

    config_register_string ("visual_anim.background_stream_mrl",
                            "menu_playback.png",
                            _("MRL of the background image"));

    config_register_string ("visual_anim.animation_stream_mrl",
                            OXINE_VISUALDIR "/animation.mpg",
                            _("MRL of the background animation"));

    config_register_range ("video.filter.brightness", 50, 0, 100,
                           _("Brightness"));

    config_register_range ("video.filter.contrast", 50, 0, 100,
                           _("Contrast"));

    config_register_range ("video.filter.hue", 50, 0, 100, _("Hue"));

    config_register_range ("video.filter.saturation", 50, 0, 100,
                           _("Saturation"));

    config_register_range ("video.zoom", 100, 0, 100, _("Zoom"));
}


static bool
odk_open_audio_port (odk_t * odk, const char *driver, void *data)
{
    xine_audio_port_t *audio_port;

    debug ("Opening audio port using driver '%s'!", driver);

    audio_port = xine_open_audio_driver (odk->xine, driver, data);
    if (!audio_port) {
        error (_("Failed to open audio port "
                 "using audio driver '%s'!"), driver);
        return false;
    }

    odk->audio_ports.count++;
    odk->audio_ports.ports = ho_realloc (odk->audio_ports.ports,
                                         odk->audio_ports.count *
                                         sizeof (xine_audio_port_t *));

    if (!odk->audio_ports.ports) {
        fatal (_("Failed to allocate memory!"));
        abort ();
    }

    odk->audio_ports.ports[odk->audio_ports.count - 1] = audio_port;

    return true;
}


static bool
odk_open_video_port (odk_t * odk, const char *driver)
{
    debug ("Opening video port using driver '%s'!", driver);

    odk->win = odk_create_window (odk->xine, driver);
    if (!odk->win || !odk->win->video_port) {
        error (_("Failed to create window using "
                 "video driver '%s'!"), driver);
        return false;
    }

    return true;
}


static void
odk_open_video (odk_t * odk)
{
    const char *driver = config_get_enum ("video.driver");

    if (!odk_open_video_port (odk, driver)) {
        if (!odk_open_video_port (odk, "auto")) {
            fatal (_("Failed to create output window!"));
            abort ();
        }
    }
}


static void
odk_open_audio (odk_t * odk)
{
#ifdef HAVE_AEX
    if (config_get_bool ("audio.driver.use_aex")) {
        if (!odk_open_audio_port (odk, "aex", NULL)) {
            error (_("Failed to open airport express!"));
        }
        else {
            return;
        }
    }
#endif

    const char *driver = config_get_enum ("audio.driver");
    if (!odk_open_audio_port (odk, driver, NULL)) {
        if (!odk_open_audio_port (odk, "auto", NULL)) {
            fatal (_("Failed to initialize audio output!"));
            abort ();
        }
    }
}


static void
odk_create_streams (odk_t * odk)
{
    xine_audio_port_t *audio_port = odk->audio_ports.ports[0];
    xine_video_port_t *video_port = odk->win->video_port;

    assert (audio_port);
    assert (video_port);

    odk->main_stream = xine_stream_new (odk->xine, audio_port, video_port);

    odk->background_stream = xine_stream_new (odk->xine, NULL, video_port);
    xine_set_param (odk->background_stream, ODK_PARAM_AUDIO_CHANNEL,
                    AUDIO_CHANNEL_OFF);
    odk->animation_stream = xine_stream_new (odk->xine, NULL, video_port);
    xine_set_param (odk->animation_stream, ODK_PARAM_AUDIO_CHANNEL,
                    AUDIO_CHANNEL_OFF);
    odk->subtitle_stream = xine_stream_new (odk->xine, NULL, video_port);
    xine_set_param (odk->subtitle_stream, ODK_PARAM_AUDIO_CHANNEL,
                    AUDIO_CHANNEL_OFF);

    /* Restore old video values. */
    xine_set_param (odk->main_stream, ODK_PARAM_VO_BRIGHTNESS,
                    config_get_number ("video.filter.brightness") * 655);

    xine_set_param (odk->main_stream, ODK_PARAM_VO_CONTRAST,
                    config_get_number ("video.filter.contrast") * 655);

    xine_set_param (odk->main_stream, ODK_PARAM_VO_HUE,
                    config_get_number ("video.filter.hue") * 655);

    xine_set_param (odk->main_stream, ODK_PARAM_VO_SATURATION,
                    config_get_number ("video.filter.saturation") * 655);

    int zoom = config_get_number ("video.zoom");
    xine_set_param (odk->main_stream, ODK_PARAM_VO_ZOOM_X, zoom);
    xine_set_param (odk->main_stream, ODK_PARAM_VO_ZOOM_Y, zoom);
}


odk_t *
odk_init (xine_t * xine)
{
    odk_t *odk = ho_new (odk_t);

    odk->xine = xine;
    odk->audio_ports.count = 0;
    odk->audio_ports.ports = NULL;

    odk->current_mrl = NULL;
    odk->current_subtitle_mrl = NULL;
    odk->current_background_mrl = NULL;
    odk->current_title = NULL;
    odk->current_alternatives = playlist_new (NULL, NULL);
    odk->aspect_ratio = ODK_VO_ASPECT_RATIO_AUTO;

    /* Register configuration values. */
    odk_register_configuration (odk);

    /* Open video port. */
    odk_open_video (odk);

    /* Open audio port. */
    odk_open_audio (odk);

    /* Create streams. */
    odk_create_streams (odk);

    /* Initialize event handling. */
    odk_event_init (odk);

    /* Initialize OSD stuff. */
    odk_osd_init (odk);

    /* Initialize post plugin stuff. */
    odk_post_init (odk);

    /* Register event handler for configuration changed events. */
    odk_add_event_handler (odk, odk_config_changed_handler, odk,
                           EVENT_HANDLER_PRIORITY_HIGH);

    return odk;
}


void
odk_free (odk_t * odk)
{
    /* Free event handling. */
    odk_event_free (odk);

    /* Free OSD stuff. */
    odk_osd_free (odk);

    /* Dispose streams */
    xine_dispose (odk->main_stream);
    xine_dispose (odk->subtitle_stream);
    xine_dispose (odk->background_stream);
    xine_dispose (odk->animation_stream);

    int i = 0;
    for (i = 0; i < odk->audio_ports.count; i++) {
        xine_close_audio_driver (odk->xine, odk->audio_ports.ports[i]);
    }
    ho_free (odk->audio_ports.ports);

    if (odk->win) {
        odk->win->free (odk->win);
    }
    if (odk->current_alternatives) {
        playlist_free (odk->current_alternatives);
    }

    ho_free (odk->current_mrl);
    ho_free (odk->current_subtitle_mrl);
    ho_free (odk->current_background_mrl);
    ho_free (odk->current_title);
    ho_free (odk);
}
