
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: environment.c 2519 2007-07-17 08:39:07Z mschwerin $
 *
 */
#include "config.h"

#include <unistd.h>

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "download.h"
#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "oxine.h"
#include "utils.h"

extern oxine_t *oxine;

static char *dir_home = NULL;
static char *dir_oxine = NULL;
static char *dir_oxine_cache = NULL;
static char *dir_oxine_tmp = NULL;
static char *dir_oxine_playlists = NULL;
static char *dir_oxine_skin = NULL;

static char *file_config = NULL;
static char *file_mediamarks_music = NULL;
static char *file_mediamarks_video = NULL;
static char *file_mediamarks_image = NULL;
static char *file_mediamarks_media = NULL;
static char *file_lirc_config = NULL;
static char *file_jsw_config = NULL;
static char *file_dvb_channels = NULL;
static char *file_favorites = NULL;
static char *file_playlist_rw = NULL;


const char *
get_dir_home (void)
{
    if (!dir_home) {
        char *tmp = getenv ("HOME");
        if (tmp && (strlen (tmp) > 0)
            && (strcmp (tmp, "/") != 0)
            && (access (tmp, X_OK | W_OK | R_OK) == 0)) {
            dir_home = ho_strdup (tmp);
            if (dir_home[strlen (tmp) - 1] == '/')
                dir_home[strlen (tmp) - 1] = '\0';
        }
        else {
            error (_("Unable to find usable home directory, "
                     "using '/tmp'."));
            dir_home = ho_strdup ("/tmp");
        }
    }

    return (const char *) dir_home;
}


const char *
get_dir_oxine (void)
{
    if (!dir_oxine) {
        dir_oxine = ho_strdup_printf ("%s/.oxine", get_dir_home ());

        mkdir (dir_oxine, 0700);
    }

    return (const char *) dir_oxine;
}


const char *
get_dir_oxine_skin (void)
{
    if (!dir_oxine_skin) {
        const char *skin = config_get_enum ("gui.skin");
        dir_oxine_skin = ho_strdup_printf ("%s/%s", OXINE_SKINDIR, skin);
    }

    return (const char *) dir_oxine_skin;
}


const char *
get_dir_oxine_playlists (void)
{
    if (!dir_oxine_playlists) {
        dir_oxine_playlists =
            ho_strdup_printf ("%s/playlists", get_dir_oxine ());

        mkdir (dir_oxine_playlists, 0700);
    }

    return (const char *) dir_oxine_playlists;
}


const char *
get_dir_oxine_cache (void)
{
    if (!dir_oxine_cache) {
        dir_oxine_cache = ho_strdup_printf ("%s/cache", get_dir_oxine ());

        mkdir (dir_oxine_cache, 0700);
    }

    return (const char *) dir_oxine_cache;
}


const char *
get_dir_oxine_tmp (void)
{
    if (!dir_oxine_tmp) {
        dir_oxine_tmp = ho_strdup_printf ("/tmp/oxine-%d", getpid ());

        mkdir (dir_oxine_tmp, 0700);
    }

    return (const char *) dir_oxine_tmp;
}


void
set_file_config (const char *filename)
{
    if (file_config) {
        ho_free (file_config);
    }

    file_config = ho_strdup (filename);
}


const char *
get_file_config (void)
{
    if (!file_config) {
        file_config = ho_strdup_printf ("%s/config", get_dir_oxine ());
    }

    return (const char *) file_config;
}


const char *
get_file_mediamarks_music (void)
{
    if (!file_mediamarks_music) {
        file_mediamarks_music =
            ho_strdup_printf ("%s/mediamarks_music.xml", get_dir_oxine ());
    }

    return (const char *) file_mediamarks_music;
}


const char *
get_file_mediamarks_video (void)
{
    if (!file_mediamarks_video) {
        file_mediamarks_video =
            ho_strdup_printf ("%s/mediamarks_video.xml", get_dir_oxine ());
    }

    return (const char *) file_mediamarks_video;
}


const char *
get_file_mediamarks_image (void)
{
    if (!file_mediamarks_image) {
        file_mediamarks_image =
            ho_strdup_printf ("%s/mediamarks_image.xml", get_dir_oxine ());
    }

    return (const char *) file_mediamarks_image;
}


const char *
get_file_mediamarks_media (void)
{
    if (!file_mediamarks_media) {
        file_mediamarks_media =
            ho_strdup_printf ("%s/mediamarks_media.xml", get_dir_oxine ());
    }

    return (const char *) file_mediamarks_media;
}


const char *
get_file_jsw_config (void)
{
    if (!file_jsw_config) {
        file_jsw_config = ho_strdup_printf ("%s/joystick", get_dir_oxine ());
    }

    return (const char *) file_jsw_config;
}


const char *
get_file_lirc_config (void)
{
    if (!file_lirc_config) {
        file_lirc_config = ho_strdup_printf ("%s/lircrc", get_dir_oxine ());
    }

    return (const char *) file_lirc_config;
}


const char *
get_file_dvb_channels (void)
{
    if (!file_dvb_channels) {
        file_dvb_channels =
            ho_strdup_printf ("%s/.xine/channels.conf", get_dir_home ());
    }

    return (const char *) file_dvb_channels;
}


const char *
get_file_favorites (void)
{
    if (!file_favorites) {
        file_favorites =
            ho_strdup_printf ("%s/favorites.xml", get_dir_oxine ());
    }

    return (const char *) file_favorites;
}


const char *
get_file_playlist_rw (void)
{
    if (!file_playlist_rw) {
        file_playlist_rw =
            ho_strdup_printf ("%s/playlist_rw.oxp", get_dir_oxine ());
    }

    return (const char *) file_playlist_rw;
}


void
environment_free (void)
{
    /* Remove the temporary files. */
    if (dir_oxine_tmp) {
        char *command = ho_strdup_printf ("rm -rf %s", dir_oxine_tmp);
        execute_shell (command, 0);
        ho_free (command);
    }

    ho_free (dir_home);
    ho_free (dir_oxine);
    ho_free (dir_oxine_cache);
    ho_free (dir_oxine_playlists);
    ho_free (dir_oxine_skin);
    ho_free (dir_oxine_tmp);

    ho_free (file_config);
    ho_free (file_mediamarks_music);
    ho_free (file_mediamarks_video);
    ho_free (file_mediamarks_image);
    ho_free (file_mediamarks_media);
    ho_free (file_jsw_config);
    ho_free (file_lirc_config);
    ho_free (file_dvb_channels);
    ho_free (file_favorites);
    ho_free (file_playlist_rw);
}
