/* OpenVAS Client
 * Copyright (C) 1998 Renaud Deraison
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * @file 
 * Build up target selection panel in GUI.
 * 
 * The arglist to manage the target selection looks like:\n
 *  - FRAME > The gtk frame with label, text-entry, button and option in it.
 *  - READ_FROM_FILE > The button to open the file selection dialog.\n
 *  - TARGET > The text entry.\n
 *  - DNS_EXPAND > The Checkbox linked to the DNS expand option.\n
 */

#include <includes.h>

#ifdef USE_GTK
#include <gtk/gtk.h>
#include "read_target_file.h"
#include "openvas_i18n.h"
#include "nvt_pref_sshlogin.h"

/**
 * Access to the currently active Scope- GUI- arglist.
 */
extern struct arglist* MainDialog;

/**
 * When the text in the target entry changed, these changes should be reflected
 * in the gui part (nvt_pref_sshlogin) where the user might select 
 * ssh-"accounts" on a per-target basis.
 * Therefore, this callback function triggers rebuilding these gui parts.
 * 
 * @param ignored  Ignored (callback).
 * @param event    Event that triggered this callback.
 * @param ignored2 Ignored (callback).
 * 
 * @return FALSE (so that other event handlers might do something).
 */
gboolean
targets_changed (GtkWidget* ignored, GdkEventFocus *event,
                          gpointer ignored2)
{
  // Retrieve list of targets.
  struct arglist* targets_gui = arg_get_value (MainDialog, "TARGET");
  const char* targets_text = gtk_entry_get_text (GTK_ENTRY(arg_get_value(targets_gui, "TARGET")));
  
  // Add a "Default" entry.
  const char* targets = g_strconcat ("Default,", target_translate (targets_text), NULL);

  // Force necessary rebuilding of gui elements.
  nvt_pref_sshlogin_rebuild_gui (targets, NULL);
  return FALSE;
}

/**
 * Builds the target selection frame and hooks it up in an arglist.
 * Returns the arglist.
 * 
 * @return Arglist, in which gtk widgets of the Target Selection Frame are
 *         hooked.
 */
struct arglist *
prefs_dialog_target()
{
  GtkWidget * frame;
  GtkWidget * vbox;
  GtkWidget * label;
  GtkWidget * check_dns;
  GtkWidget * entry;
  GtkWidget * hbox;
  GtkWidget * button;
  struct arglist *  ctrls = emalloc(sizeof(struct arglist));

  /* The overall frame */
  frame = gtk_frame_new(_("Target selection"));
  gtk_container_border_width(GTK_CONTAINER(frame), 10);
  gtk_widget_show(frame);
  arg_add_value(ctrls, "FRAME", ARG_PTR, -1, frame);

  /* the overall vbox that will contain all widgets */
  vbox = gtk_vbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_widget_show(vbox);

  /* The hbox that contains the target entry and file selection button */
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  /* The label for target entry */
  label = gtk_label_new(_("Target(s):"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  /* The target entry */
  entry = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 0);
  gtk_widget_show(entry);
  arg_add_value(ctrls, "TARGET", ARG_PTR, -1, entry);

  // Install callback to trigger gui changes when target list chagned.
  // Other possible event might be "move focus".
  gtk_signal_connect(GTK_OBJECT(entry), "focus-out-event", (GtkSignalFunc)targets_changed, NULL);

  /* The button for reading target from a file */
  button = gtk_button_new_with_label(_("Read file..."));
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
  g_signal_connect(GTK_OBJECT(button), "clicked",
                   (GtkSignalFunc)target_file_select, NULL);
  gtk_widget_show(button);
  arg_add_value(ctrls, "READ_FROM_FILE", ARG_PTR, -1, button);

  /* Checkbox for Zone transfer */
  check_dns = gtk_check_button_new_with_label(_("Perform a DNS zone transfer"));
  gtk_box_pack_start(GTK_BOX(vbox), check_dns, FALSE, FALSE, 0);
  gtk_widget_show(check_dns);
  arg_add_value(ctrls, "DNS_EXPAND", ARG_PTR, -1, check_dns);

  return(ctrls);
}
#endif
