/* $Id: prefs_dialog_auth.c,v 1.3 2005-09-14 13:22:16 renaud Exp $
 *
 * Copyright (C) 2004 by Greenbone Networks GmbH
 * Author(s):
 * Thomas Arendsen Hein <thomas@intevation.de>
 *
 * This program is free software under the GNU GPL (>=v2)
 * Read the file COPYING coming with the software for details.
 */

/* for close_stream_connection() */
#include <openvas/network.h>

/* for emalloc, efree, estrdup */
#include <openvas/system.h>

#include "openvas_i18n.h"

#ifdef USE_GTK
#include <gtk/gtk.h>

#include "openvas-client.h"
#include "error_dlg.h"
#include "context.h"
#include "preferences.h"
#include "prefs_context.h"
#include "prefs_scope_tree.h"
#include "prefs_help.h"
#include "globals.h"

#ifdef USE_OMP
gboolean prefs_dialog_auth_writable_settings = FALSE;
#endif

struct auth_fileselect {
  struct auth_dialog *auth;
  GtkWidget *box;
  GtkWidget *entry;
};

struct auth_dialog {
  struct context *context;
  GtkWindow *parent;
  GtkWidget *dialog;
  GtkWidget *hostname;
  GtkWidget *port;
  GtkWidget *username;
  GtkWidget *password;
  GtkWidget *use_client_cert;
  struct auth_fileselect *trusted_ca;
  struct auth_fileselect *cert_file;
  struct auth_fileselect *key_file;
};

/**
 * @brief Returns the hostname specified by the user or NULL if none entered.
 *
 * If the corresponding entry widget is empty, an error message will be shown.
 *
 * @return Hostname entered by the user or NULL if empty.
 */
const gchar *
prefs_dialog_auth_hostname (struct auth_dialog *auth)
{
  const gchar *text;

  text = gtk_entry_get_text(GTK_ENTRY(auth->hostname));
  if((!text) || (!strlen(text)))
  {
    gtk_widget_grab_focus(auth->hostname);
    show_warning(_("You must enter a valid hostname or IP"));
    return NULL;
  }
  return text;
}


/**
 * @brief Returns the user-selected port or -1 if out of range.
 *
 * If the selected port is <0 || > 65536 an error message will be shown.
 *
 * @return Port number selected by user or -1 if out of range.
 */
int
prefs_dialog_auth_port (struct auth_dialog *auth)
{
  int port = gtk_spin_button_get_value(GTK_SPIN_BUTTON(auth->port));
  if((port < 0) || (port > 65536))
  {
    gtk_widget_grab_focus(auth->port);
    show_warning(_("The port number is out of range"));
    return -1;
  }
  return port;
}

void
prefs_dialog_auth_defaultport_clicked (GtkButton *button, struct auth_dialog *auth)
{
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(auth->port),
      GPOINTER_TO_SIZE(prefs_get_default(auth->context, "nessusd_port")));
}

void
prefs_dialog_auth_fileselect_popup (GtkButton *button,
                                    struct auth_fileselect *auth_fileselect)
{
  GtkWidget *dialog = gtk_file_selection_new(_("Select File"));

  gtk_window_set_transient_for(GTK_WINDOW(dialog),
      GTK_WINDOW(auth_fileselect->auth->dialog));
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(dialog),
    gtk_entry_get_text(GTK_ENTRY(auth_fileselect->entry)));
  if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
  {
    gchar *filename = g_strdup(
	gtk_file_selection_get_filename(GTK_FILE_SELECTION(dialog)));

    gtk_widget_hide(dialog);
    if(check_is_dir(filename))
      show_error(_("Please choose a filename."));
    else if(!check_is_file(filename))
      show_error(_("File \"%s\" doesn't exist."), filename);
    else
      gtk_entry_set_text(GTK_ENTRY(auth_fileselect->entry), filename);
    g_free(filename);
  }
  gtk_widget_destroy(dialog);
}

const gchar *
prefs_dialog_auth_username (struct auth_dialog *auth)
{
  const gchar *text;
  text = gtk_entry_get_text(GTK_ENTRY(auth->username));
  if((!text) || (!strlen(text)))
  {
    gtk_widget_grab_focus(auth->username);
    show_warning(_("You must enter a valid username"));
    return NULL;
  }
  return text;
}

const gchar *
prefs_dialog_auth_password (struct auth_dialog *auth)
{
  const gchar *text;
  text = gtk_entry_get_text(GTK_ENTRY(auth->password));
  if(!(text && (strlen(text)
      || gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(auth->use_client_cert)))
      ))
  {
    gtk_widget_grab_focus(auth->password);
    show_warning(_("You must enter a valid password"));
    return NULL;
  }
  return text;
}

const gchar *
prefs_dialog_auth_trusted_ca (struct auth_dialog *auth, struct context *context)
{
  const gchar *text;
  int paranoia_level = prefs_get_int(context, "paranoia_level");

  text = gtk_entry_get_text(GTK_ENTRY(auth->trusted_ca->entry));
  if(!(text && strlen(text)) && (paranoia_level == 2 || paranoia_level == 3))
  {
    gtk_widget_grab_focus(auth->trusted_ca->entry);
    show_warning(_("You must enter a filename for Trusted CA"));
    return NULL;
  }
  return text;
}

const gchar *
prefs_dialog_auth_cert_file (struct auth_dialog *auth)
{
  const gchar *text;
  text = gtk_entry_get_text(GTK_ENTRY(auth->cert_file->entry));
  if(!(text && strlen(text)))
  {
    gtk_widget_grab_focus(auth->cert_file->entry);
    show_warning(_("You must enter a filename for the SSL Certificate"));
    return NULL;
  }
  return text;
}

const gchar *
prefs_dialog_auth_key_file (struct auth_dialog *auth)
{
  const gchar *text;
  text = gtk_entry_get_text(GTK_ENTRY(auth->key_file->entry));
  if(!(text && strlen(text)))
  {
    gtk_widget_grab_focus(auth->key_file->entry);
    show_warning(_("You must enter a filename for the SSL Key"));
    return NULL;
  }
  return text;
}


const char *
prefs_dialog_auth_do_connect (struct context *context, gpointer ctrls)
{
  void *context_window = arg_get_value(MainDialog, "CONTEXT");
  const char *hostname = prefs_get_string(context, "nessusd_host");
  GtkWindow *window = NULL;
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *label;
  gchar *text;
  const char *err;
  int i;

  if(context_window)
    window = GTK_WINDOW(context_window);

  dialog = gtk_dialog_new_with_buttons(_("Connecting ..."),
      window,
      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT
      | GTK_DIALOG_NO_SEPARATOR,
      NULL);
  /* prevent the dialog from being closed */
  g_signal_connect(G_OBJECT(dialog), "delete_event",
      G_CALLBACK(gtk_true), NULL);

  vbox = gtk_vbox_new(FALSE, 4);
  gtk_widget_show(vbox);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox);

  text = g_strdup_printf(_("Connecting to OpenVAS server \"%s\" ..."), hostname);
  label = gtk_label_new(text);
  gtk_widget_show(label);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), label);

  context->pbar = gtk_progress_bar_new();
  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(context->pbar),
      _("Initiating connection ..."));
  gtk_widget_show(context->pbar);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), context->pbar);

  gtk_widget_show(dialog);
  arg_set_value(ctrls, "CONTEXT", -1, dialog);

  for ( i = 0 ; i < 32 ; i ++ )
  {
   if (gtk_events_pending() )
  	gtk_main_iteration();
   else
	break;
  }

  err = connect_to_scanner(context);

  arg_set_value(ctrls, "CONTEXT", -1, window);
  gtk_widget_destroy(dialog);
  g_free(text);

  if(err)
    return err;
  else
  {
    /* FIXME plugin list is only filled if gtk_notebook_set_page is called */
    gtk_notebook_set_page(
	GTK_NOTEBOOK(arg_get_value(MainDialog, "NOTEBOOK")), 1);
    prefs_context_update(context);
    scopetreeview_connected_update(context);
    return NULL;
  }
}

GtkWidget *
prefs_dialog_auth_vbox (const gchar *text, GtkWidget *content)
{
  GtkWidget *label;
  GtkWidget *vbox;

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);

  label = gtk_label_new(text);
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, TRUE, 0);
  gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

  gtk_box_pack_start(GTK_BOX(vbox), content, FALSE, FALSE, 0);

  return vbox;
}

struct auth_fileselect *
prefs_dialog_auth_fileselect (struct auth_dialog *auth, const gchar *text,
                              const gchar *pref_name, GtkWidget *box)
{
  struct auth_fileselect *auth_fileselect;
  GtkWidget *hbox;
  GtkWidget *button;
  const gchar *pref = prefs_get_string(auth->context, pref_name);

  auth_fileselect = emalloc(sizeof(struct auth_fileselect));
  auth_fileselect->auth = auth;

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox);

  auth_fileselect->entry = gtk_entry_new();
  gtk_widget_show(auth_fileselect->entry);
  if(pref)
    gtk_entry_set_text(GTK_ENTRY(auth_fileselect->entry), pref);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), auth_fileselect->entry);

  button = gtk_button_new_with_mnemonic(_("Select ..."));
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
  g_signal_connect((gpointer)button, "clicked",
                   G_CALLBACK(prefs_dialog_auth_fileselect_popup),
                   auth_fileselect);

  auth_fileselect->box = prefs_dialog_auth_vbox(text, hbox);
  gtk_box_pack_start_defaults(GTK_BOX(box), auth_fileselect->box);

  return auth_fileselect;
}


void
prefs_dialog_auth_update (GtkWidget *widget, struct auth_dialog *auth)
{
  gboolean use_ssl = TRUE;
  gboolean use_client_cert = gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON(auth->use_client_cert));

  gtk_widget_set_sensitive(auth->use_client_cert, use_ssl);
  gtk_widget_set_sensitive(auth->trusted_ca->box, use_ssl);
  gtk_widget_set_sensitive(auth->cert_file->box, use_ssl && use_client_cert);
  gtk_widget_set_sensitive(auth->key_file->box, use_ssl && use_client_cert);
}


/**
 * @brief Creates the dialog with details with which to connect.
 */
void
prefs_dialog_auth_create_dialog (struct context *context, struct auth_dialog *auth)
{
  GtkTooltips *tooltips = gtk_tooltips_new();
  GtkWidget *dialog;
  GtkWidget *dialog_vbox;
  GtkWidget *frame;
  GtkWidget *hbox;
  GtkWidget *hbox2;
  GtkWidget *button;
  GtkObject *port_adj;
  GtkWidget *vbox;
#ifdef USE_OMP
  /* Use the values in the manager context, because the same settings are
   * used for everything under a manager. */
  if (context->protocol == PROTOCOL_OMP)
    context = context_by_type (context, CONTEXT_SERVER); // FIX check return?
#endif

  auth->context = context;

  auth->dialog = dialog = gtk_dialog_new_with_buttons(
      _("Connect to OpenVAS Server"), auth->parent,
      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
      GTK_STOCK_OK, GTK_RESPONSE_OK,
      NULL);
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
  gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
  gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);

  dialog_vbox = gtk_vbox_new(FALSE, 4);
  gtk_widget_show(dialog_vbox);
  gtk_container_set_border_width(GTK_CONTAINER(dialog_vbox), 4);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), dialog_vbox);

  frame = gtk_frame_new(_("OpenVAS Server"));
  gtk_widget_show(frame);
  gtk_box_pack_start_defaults(GTK_BOX(dialog_vbox), frame);

#ifdef USE_OMP
  if (context->protocol == PROTOCOL_OMP
      && prefs_dialog_auth_writable_settings == FALSE)
    gtk_widget_set_sensitive (frame, 0);
#endif

  hbox = gtk_hbox_new(TRUE, 8);
  gtk_widget_show(hbox);
  gtk_container_add(GTK_CONTAINER(frame), hbox);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);

  /*
   * scanner_host
   */
  auth->hostname = gtk_entry_new();
  gtk_widget_show(auth->hostname);
  gtk_entry_set_text(GTK_ENTRY(auth->hostname),
      prefs_get_string(context, "nessusd_host"));
  gtk_tooltips_set_tip(tooltips, auth->hostname, HLP_AUTH_SERVER, "");
  gtk_box_pack_start_defaults(GTK_BOX(hbox),
      prefs_dialog_auth_vbox(_("Hostname:"), auth->hostname));

  /*
   * scanner_port
   */
  hbox2 = gtk_hbox_new(FALSE, 4);
  gtk_widget_show(hbox2);

  auth->port = gtk_entry_new();
  port_adj = gtk_adjustment_new(
      prefs_get_int(context, "nessusd_port"), 1, 65535, 1, 100, 100);
  auth->port = gtk_spin_button_new(GTK_ADJUSTMENT(port_adj), 1, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(auth->port), TRUE);
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(auth->port), TRUE);
  gtk_tooltips_set_tip(tooltips, auth->port, HLP_AUTH_PORT, "");
  gtk_widget_show(auth->port);
  gtk_box_pack_start_defaults(GTK_BOX(hbox2), auth->port);

  button = gtk_button_new_with_mnemonic(_("_Default"));
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
  gtk_tooltips_set_tip(tooltips, button, HLP_AUTH_PORT_DEFAULT, "");
  g_signal_connect((gpointer)button, "clicked",
                   G_CALLBACK(prefs_dialog_auth_defaultport_clicked),
                   auth);

  gtk_box_pack_start_defaults(GTK_BOX(hbox),
      prefs_dialog_auth_vbox(_("Port:"), hbox2));



  frame = gtk_frame_new(_("Authentication"));
  gtk_widget_show(frame);
  gtk_box_pack_start_defaults(GTK_BOX(dialog_vbox), frame);

  vbox = gtk_vbox_new(TRUE, 8);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_widget_show(vbox);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);

  /*
   * scanner_user
   */
  auth->username = gtk_entry_new();
  gtk_widget_show(auth->username);
#ifdef USE_OMP
  if (context->protocol == PROTOCOL_OMP)
    {
      if (prefs_dialog_auth_writable_settings == FALSE)
        gtk_widget_set_sensitive (auth->username, 0);

      /* Look in the nessusd_login preference first.  This works around
       * special casing of the saving of nessusd_user. */
      const char *user = prefs_get_string(context, "nessusd_login");
      if (user == NULL)
        user = prefs_get_string(context, "nessusd_user");
      gtk_entry_set_text(GTK_ENTRY(auth->username), user);
    }
  else
#endif /* USE_OMP */
  gtk_entry_set_text(GTK_ENTRY(auth->username),
      prefs_get_string(context, "nessusd_user"));
  gtk_tooltips_set_tip(tooltips, auth->username, HLP_LOGIN_USER, "");
  gtk_box_pack_start_defaults(GTK_BOX(vbox),
      prefs_dialog_auth_vbox(_("Login:"), auth->username));

  /*
   * password
   */
  auth->password = gtk_entry_new();
  gtk_widget_show(auth->password);
  gtk_entry_set_visibility(GTK_ENTRY(auth->password), FALSE);
  /* gtk_tooltips_set_tip(tooltips, auth->password, HLP_LOGIN_PASSWORD, ""); */
  gtk_box_pack_start_defaults(GTK_BOX(vbox),
      prefs_dialog_auth_vbox(_("Password:"), auth->password));
  gtk_widget_grab_focus(auth->password);
  gtk_entry_set_activates_default(GTK_ENTRY(auth->password), TRUE);


  /*
   * use_client_cert
   */
  auth->use_client_cert = gtk_check_button_new_with_mnemonic(
      _("Authentication by certi_ficate"));
  gtk_widget_show(auth->use_client_cert);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), auth->use_client_cert);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(auth->use_client_cert),
      prefs_get_int(context, "use_client_cert"));
  g_signal_connect(G_OBJECT(auth->use_client_cert), "toggled",
      G_CALLBACK(prefs_dialog_auth_update), auth);

  /*
   * trusted_ca
   */
  auth->trusted_ca = prefs_dialog_auth_fileselect(
      auth, _("Trusted CA:"), "trusted_ca", vbox);

  /*
   * cert_file
   */
  auth->cert_file = prefs_dialog_auth_fileselect(
      auth, _("User Certificate File:"), "cert_file", vbox);

  /*
   * key_file
   */
  auth->key_file = prefs_dialog_auth_fileselect(
      auth, _("User Key File:"), "key_file", vbox);

  prefs_dialog_auth_update(NULL, auth);

#ifdef USE_OMP
  if (context->protocol == PROTOCOL_OMP
      && prefs_dialog_auth_writable_settings == FALSE)
    {
      gtk_widget_set_sensitive (auth->use_client_cert, 0);
      gtk_widget_set_sensitive (auth->trusted_ca->box, 0);
      gtk_widget_set_sensitive (auth->cert_file->box, 0);
      gtk_widget_set_sensitive (auth->key_file->box, 0);
    }
#endif /* USE_OMP */
}

gboolean
prefs_dialog_auth_connect_dialog (struct context *context, gpointer ctrls)
{
  void *context_window = arg_get_value(ctrls, "CONTEXT");
  struct auth_dialog *auth = emalloc(sizeof(struct auth_dialog));

  auth->parent = context_window?GTK_WINDOW(context_window):NULL;
  prefs_dialog_auth_create_dialog(context, auth);

  arg_set_value(ctrls, "CONTEXT", -1, auth->dialog);
  for(;;)
  {
    if(gtk_dialog_run(GTK_DIALOG(auth->dialog)) == GTK_RESPONSE_OK)
    {
#ifdef USE_OMP
      struct context *original_context = NULL;
      struct context *manager = NULL;
#endif
      const char *hostname;
      int port;
      const char *username;
      const char *password;
      const char *err;
      const char *trusted_ca;
      const char *cert_file;
      const char *key_file;
      int use_client_cert = 0;

      efree(&context->passwd);
      if(!(hostname = prefs_dialog_auth_hostname(auth)) ||
	 (port = prefs_dialog_auth_port(auth)) < 0)
	continue;
      if(!(username = prefs_dialog_auth_username(auth)) ||
	 !(password = prefs_dialog_auth_password(auth)))
	continue;

      if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(auth->use_client_cert)))
	use_client_cert = 1;

#ifdef USE_OMP
      if (context->protocol == PROTOCOL_OMP)
        {
          /* The same settings are used for everything under a manager, so
           * switch to the manager context to work with the settings in
           * it. */
          original_context = context;
          context = context_by_type (context, CONTEXT_SERVER); // FIX check ret?
          /* Ensuring that nessusd_user is always in the RC is hard because it is
           * special cased, so just use another preference.  */
          prefs_set_string (context, "nessusd_login", username);
        }
#endif /* USE_OMP */

      if(!(trusted_ca = prefs_dialog_auth_trusted_ca(auth, context)))
        continue;

      if(use_client_cert)
      {
        if(!(cert_file = prefs_dialog_auth_cert_file(auth)) ||
            !(key_file = prefs_dialog_auth_key_file(auth)))
          continue;
        prefs_set_string(context, "cert_file", cert_file);
        prefs_set_string(context, "key_file", key_file);
      }

      prefs_set_string(context, "trusted_ca", trusted_ca);
      prefs_set_int(context, "use_client_cert", use_client_cert);

      prefs_set_string(context, "nessusd_host", hostname);
      prefs_set_int(context, "nessusd_port", port);
      prefs_set_string(context, "nessusd_user", username);
      if(password[0])
	context->passwd = estrdup(password);
      else
	context->passwd = estrdup("*"); /* XXX this is ugly */

#ifdef USE_OMP
      if (context->protocol == PROTOCOL_OMP)
        {
          // FIX explain why needed
          context_save_recurse (context);
          manager = context;
          /* Switch back, so the session and socket end up in the given
           * context. */
          context = original_context;
        }
#endif
      err = prefs_dialog_auth_do_connect(context, ctrls);
      if(err)
      {
#ifdef USE_OMP
        /* Clear the stored password, so the user can enter another. */
        if (context->protocol == PROTOCOL_OMP && manager->passwd)
          efree (&manager->passwd);
#endif
	show_error("%s", err);
	continue;
      }
    }
    break;
  }
  arg_set_value(ctrls, "CONTEXT", -1, auth->parent);
  gtk_widget_destroy(auth->dialog);
  efree(&auth->trusted_ca);
  efree(&auth->cert_file);
  efree(&auth->key_file);
  efree(&auth);

  return (context->socket >= 0);
}

void
prefs_dialog_auth_connect (GtkMenuItem *menuitem, gpointer ctrls)
{
  prefs_dialog_auth_connect_dialog(Context, ctrls);
}

/**
 * @brief Try connecting if not already connected and context->passwd is
 * @brief available.
 *
 * passwd may be empty (but != NULL) for user certificates.
 * If this fails, raise the login dialog.
 * Returns TRUE if connected.
 */
gboolean
prefs_dialog_auth_connection (struct context *context)
{
  if (context->socket >= 0)
    return TRUE;

  if (!context->passwd ||
     !prefs_get_int(Global, "nessusd_autoconnect") ||
      prefs_dialog_auth_do_connect(context, MainDialog))
    return prefs_dialog_auth_connect_dialog(context, MainDialog);

  return (context->socket >= 0);
}

#ifdef USE_OMP
gboolean
prefs_dialog_auth_omp_connection (struct context *context)
{
  struct context *manager;

  if (context->socket > 0)
    /* Already connected. */
    return TRUE;

  manager = context_by_type (context, CONTEXT_SERVER);
  // FIX NULL return?

  if (manager->passwd)
    {
      if (prefs_dialog_auth_do_connect (context, MainDialog))
        /* Connection failed, free the password so the user can change it. */
        efree (&manager->passwd);
      else
        return TRUE;
    }
  return prefs_dialog_auth_connect_dialog (context, MainDialog)
         && context->socket > 0;
}
#endif /* USE_OMP */

void
prefs_dialog_auth_disconnect (GtkMenuItem *menuitem, gpointer ctrls)
{
  close_stream_connection(Context->socket);
  Context->socket = -1;
  prefs_context_update(Context);
  scopetreeview_connected_update(Context);
}

#ifdef USE_OMP
void
prefs_dialog_auth_refresh (GtkMenuItem *menuitem, gpointer ctrls)
{
  server_menu_refresh (menuitem, ctrls);
}
#endif /* USE_OMP */

#endif /* USE_GTK */
