








///////////////////////////////////////////////////////////////////////////////
// Name:        wx/univ/window.h
// Purpose:     wxWindow class which is the base class for all
//              wxUniv port controls, it supports the customization of the
//              window drawing and input processing.
// Author:      Vadim Zeitlin
// Modified by:
// Created:     06.08.00
// RCS-ID:      $Id: window.h 39633 2006-06-08 11:25:30Z ABX $
// Copyright:   (c) 2000 SciTech Software, Inc. (www.scitechsoft.com)
// Licence:     wxWindows licence
///////////////////////////////////////////////////////////////////////////////

#ifndef _UNIPARTS_H_
#define _UNIPARTS_H_

#include "wx/bitmap.h"      // for m_bitmapBg

class WXDLLEXPORT wxControlRenderer;
class WXDLLEXPORT wxEventLoop;

#if wxUSE_MENUS
    class WXDLLEXPORT wxMenu;
    class WXDLLEXPORT wxMenuBar;
#endif // wxUSE_MENUS

class WXDLLEXPORT wxRenderer;

#if wxUSE_SCROLLBAR
    class WXDLLEXPORT wxScrollBar;
#endif // wxUSE_SCROLLBAR

#ifdef __WXX11__
#define wxUSE_TWO_WINDOWS 1
#else
#define wxUSE_TWO_WINDOWS 0
#endif

// ----------------------------------------------------------------------------
// wxWindow
// ----------------------------------------------------------------------------

#if defined(__WXMSW__)
#define wxWindowNative wxWindowMSW
#elif defined(__WXGTK__)
#define wxWindowNative wxWindowGTK
#elif defined(__WXMGL__)
#define wxWindowNative wxWindowMGL
#elif defined(__WXX11__)
#define wxWindowNative wxWindowX11
#elif defined(__WXMAC__)
#define wxWindowNative wxWindowMac
#endif

class WXDLLEXPORT uniWindow : public wxWindowNative
{
public:
    // ctors and create functions
    // ---------------------------

    uniWindow() { Init(); }

    uniWindow(wxWindow *parent,
             wxWindowID id,
             const wxPoint& pos = wxDefaultPosition,
             const wxSize& size = wxDefaultSize,
             long style = 0,
             const wxString& name = wxPanelNameStr)
        : wxWindowNative(parent, id, pos, size, style | wxCLIP_CHILDREN, name)
        { Init(); }

    bool Create(wxWindow *parent,
                wxWindowID id,
                const wxPoint& pos = wxDefaultPosition,
                const wxSize& size = wxDefaultSize,
                long style = 0,
                const wxString& name = wxPanelNameStr);

    virtual ~uniWindow();

    // background pixmap support
    // -------------------------

    virtual void SetBackground(const wxBitmap& bitmap,
                               int alignment = wxALIGN_CENTRE,
                               wxStretch stretch = wxSTRETCH_NOT);

    const wxBitmap& GetBackgroundBitmap(int *alignment = NULL,
                                        wxStretch *stretch = NULL) const;

    // scrollbars: we (re)implement it ourselves using our own scrollbars
    // instead of the native ones
    // ------------------------------------------------------------------

    virtual void SetScrollbar(int orient,
                              int pos,
                              int page,
                              int range,
                              bool refresh = true );
    virtual void SetScrollPos(int orient, int pos, bool refresh = true);
    virtual int GetScrollPos(int orient) const;
    virtual int GetScrollThumb(int orient) const;
    virtual int GetScrollRange(int orient) const;
    virtual void ScrollWindow(int dx, int dy,
                              const wxRect* rect = (wxRect *) NULL);

    // take into account the borders here
    virtual wxPoint GetClientAreaOrigin() const;

    // popup menu support
    // ------------------

    // NB: all menu related functions are implemented in menu.cpp

#if wxUSE_MENUS
    // this is wxUniv-specific private method to be used only by wxMenu
    void DismissPopupMenu();
#endif // wxUSE_MENUS

    // miscellaneous other methods
    // ---------------------------

    // get the state information
    virtual bool IsFocused() const;
    virtual bool IsCurrent() const;
    virtual bool IsPressed() const;
    virtual bool IsDefault() const;

    // return all state flags at once (combination of wxCONTROL_XXX values)
    int GetStateFlags() const;

    // set the "highlighted" flag and return true if it changed
    virtual bool SetCurrent(bool doit = true);

#if wxUSE_SCROLLBAR
    // get the scrollbar (may be NULL) for the given orientation
    wxScrollBar *GetScrollbar(int orient) const
    {
        return orient & wxVERTICAL ? m_scrollbarVert : m_scrollbarHorz;
    }
#endif // wxUSE_SCROLLBAR

    // methods used by wxColourScheme to choose the colours for this window
    // --------------------------------------------------------------------

    // return true if this is a panel/canvas window which contains other
    // controls only
    virtual bool IsCanvasWindow() const { return false; }

    // return true if this control can be highlighted when the mouse is over
    // it (the theme decides itself whether it is really highlighted or not)
    virtual bool CanBeHighlighted() const { return false; }

    // return true if we should use the colours/fonts returned by the
    // corresponding GetXXX() methods instead of the default ones
    bool UseFgCol() const { return m_hasFgCol; }
    bool UseFont() const { return m_hasFont; }

    // return true if this window serves as a container for the other windows
    // only and doesn't get any input itself
    virtual bool IsStaticBox() const { return false; }

    // returns the (low level) renderer to use for drawing the control by
    // querying the current theme
    wxRenderer *GetRenderer() const { return m_renderer; }

    // scrolling helper: like ScrollWindow() except that it doesn't refresh the
    // uncovered window areas but returns the rectangle to update (don't call
    // this with both dx and dy non zero)
    wxRect ScrollNoRefresh(int dx, int dy, const wxRect *rect = NULL);

    // after scrollbars are added or removed they must be refreshed by calling
    // this function
    void RefreshScrollbars();

    // erase part of the control
    virtual void EraseBackground(wxDC& dc, const wxRect& rect);

    // overridden base class methods
    // -----------------------------

    // the rect coordinates are, for us, in client coords, but if no rect is
    // specified, the entire window is refreshed
    virtual void Refresh(bool eraseBackground = true,
                         const wxRect *rect = (const wxRect *) NULL);

    // we refresh the window when it is dis/enabled
    virtual bool Enable(bool enable = true);

    // should we use the standard control colours or not?
    virtual bool ShouldInheritColours() const { return false; }

protected:
    // common part of all ctors
    void Init();

#if wxUSE_MENUS
    virtual bool DoPopupMenu(wxMenu *menu, int x, int y);
#endif // wxUSE_MENUS

    // we deal with the scrollbars in these functions
    virtual void DoSetClientSize(int width, int height);
    virtual void DoGetClientSize(int *width, int *height) const;
    virtual wxHitTest DoHitTest(wxCoord x, wxCoord y) const;

    // event handlers
    void OnSize(wxSizeEvent& event);
    void OnNcPaint(wxNcPaintEvent& event);
    void OnPaint(wxPaintEvent& event);
    void OnErase(wxEraseEvent& event);

#if wxUSE_ACCEL || wxUSE_MENUS
    void OnKeyDown(wxKeyEvent& event);
#endif // wxUSE_ACCEL

#if wxUSE_MENUS
    void OnChar(wxKeyEvent& event);
    void OnKeyUp(wxKeyEvent& event);
#endif // wxUSE_MENUS

    // draw the control background, return true if done
    virtual bool DoDrawBackground(wxDC& dc);

    // draw the controls border
    virtual void DoDrawBorder(wxDC& dc, const wxRect& rect);

    // draw the controls contents
    virtual void DoDraw(wxControlRenderer *renderer);

    // calculate the best size for the client area of the window: default
    // implementation of DoGetBestSize() uses this method and adds the border
    // width to the result
    virtual wxSize DoGetBestClientSize() const;
    virtual wxSize DoGetBestSize() const;

    // adjust the size of the window to take into account its borders
    wxSize AdjustSize(const wxSize& size) const;

    // put the scrollbars along the edges of the window
    void PositionScrollbars();

#if wxUSE_MENUS
    // return the menubar of the parent frame or NULL
    wxMenuBar *GetParentFrameMenuBar() const;
#endif // wxUSE_MENUS

    // the renderer we use
    wxRenderer *m_renderer;

    // background bitmap info
    wxBitmap  m_bitmapBg;
    int       m_alignBgBitmap;
    wxStretch m_stretchBgBitmap;

    // old size
    wxSize m_oldSize;

    // is the mouse currently inside the window?
    bool m_isCurrent:1;

#ifdef __WXMSW__
public:
    // override MSWWindowProc() to process WM_NCHITTEST
    WXLRESULT MSWWindowProc(WXUINT message, WXWPARAM wParam, WXLPARAM lParam);
#endif // __WXMSW__

private:

#if wxUSE_SCROLLBAR
    // the window scrollbars
    wxScrollBar *m_scrollbarHorz,
                *m_scrollbarVert;
#endif // wxUSE_SCROLLBAR

#if wxUSE_MENUS
    // the current modal event loop for the popup menu we show or NULL
    static wxEventLoop *ms_evtLoopPopup;

    // the last window over which Alt was pressed (used by OnKeyUp)
    static wxWindow *ms_winLastAltPress;
#endif // wxUSE_MENUS

    DECLARE_DYNAMIC_CLASS(uniWindow)
    DECLARE_EVENT_TABLE()
};

/////////////////////////////////////////////////////////////////////////////
// Name:        wx/toplevel.h
// Purpose:     Top level window, abstraction of wxFrame and wxDialog
// Author:      Vaclav Slavik
// Id:          $Id: toplevel.h 42664 2006-10-29 20:39:31Z VZ $
// Copyright:   (c) 2001-2002 SciTech Software, Inc. (www.scitechsoft.com)
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////


//#ifndef __WX_UNIV_TOPLEVEL_H__
//#define __WX_UNIV_TOPLEVEL_H__

#include "inpcons.h"
#include "inphand.h"
#include "wx/icon.h"

// ----------------------------------------------------------------------------
// constants
// ----------------------------------------------------------------------------

// frame decorations type flags used in wxRenderer and wxColourScheme
enum
{
    wxTOPLEVEL_ACTIVE          = 0x00000001,
    wxTOPLEVEL_MAXIMIZED       = 0x00000002,
    wxTOPLEVEL_TITLEBAR        = 0x00000004,
    wxTOPLEVEL_ICON            = 0x00000008,
    wxTOPLEVEL_RESIZEABLE      = 0x00000010,
    wxTOPLEVEL_BORDER          = 0x00000020,
    wxTOPLEVEL_BUTTON_CLOSE    = 0x01000000,
    wxTOPLEVEL_BUTTON_MAXIMIZE = 0x02000000,
    wxTOPLEVEL_BUTTON_ICONIZE =  0x04000000,
    wxTOPLEVEL_BUTTON_RESTORE  = 0x08000000,
    wxTOPLEVEL_BUTTON_HELP     = 0x10000000
};

// frame hit test return values:
enum
{
    wxHT_TOPLEVEL_NOWHERE         = 0x00000000,
    wxHT_TOPLEVEL_CLIENT_AREA     = 0x00000001,
    wxHT_TOPLEVEL_ICON            = 0x00000002,
    wxHT_TOPLEVEL_TITLEBAR        = 0x00000004,

    wxHT_TOPLEVEL_BORDER_N        = 0x00000010,
    wxHT_TOPLEVEL_BORDER_S        = 0x00000020,
    wxHT_TOPLEVEL_BORDER_E        = 0x00000040,
    wxHT_TOPLEVEL_BORDER_W        = 0x00000080,
    wxHT_TOPLEVEL_BORDER_NE       = wxHT_TOPLEVEL_BORDER_N | wxHT_TOPLEVEL_BORDER_E,
    wxHT_TOPLEVEL_BORDER_SE       = wxHT_TOPLEVEL_BORDER_S | wxHT_TOPLEVEL_BORDER_E,
    wxHT_TOPLEVEL_BORDER_NW       = wxHT_TOPLEVEL_BORDER_N | wxHT_TOPLEVEL_BORDER_W,
    wxHT_TOPLEVEL_BORDER_SW       = wxHT_TOPLEVEL_BORDER_S | wxHT_TOPLEVEL_BORDER_W,
    wxHT_TOPLEVEL_ANY_BORDER      = 0x000000F0,

    wxHT_TOPLEVEL_BUTTON_CLOSE    = /*0x01000000*/ wxTOPLEVEL_BUTTON_CLOSE,
    wxHT_TOPLEVEL_BUTTON_MAXIMIZE = /*0x02000000*/ wxTOPLEVEL_BUTTON_MAXIMIZE,
    wxHT_TOPLEVEL_BUTTON_ICONIZE =  /*0x04000000*/ wxTOPLEVEL_BUTTON_ICONIZE,
    wxHT_TOPLEVEL_BUTTON_RESTORE  = /*0x08000000*/ wxTOPLEVEL_BUTTON_RESTORE,
    wxHT_TOPLEVEL_BUTTON_HELP     = /*0x10000000*/ wxTOPLEVEL_BUTTON_HELP,
    wxHT_TOPLEVEL_ANY_BUTTON      =   0x1F000000
};

// Flags for interactive frame manipulation functions (only in wxUniversal):
enum
{
    wxINTERACTIVE_MOVE           = 0x00000001,
    wxINTERACTIVE_RESIZE         = 0x00000002,
    wxINTERACTIVE_RESIZE_S       = 0x00000010,
    wxINTERACTIVE_RESIZE_N       = 0x00000020,
    wxINTERACTIVE_RESIZE_W       = 0x00000040,
    wxINTERACTIVE_RESIZE_E       = 0x00000080,
    wxINTERACTIVE_WAIT_FOR_INPUT = 0x10000000
};

// ----------------------------------------------------------------------------
// the actions supported by this control
// ----------------------------------------------------------------------------

#define wxACTION_TOPLEVEL_ACTIVATE       _T("activate")   // (de)activate the frame
#define wxACTION_TOPLEVEL_BUTTON_PRESS   _T("pressbtn")   // press titlebar btn
#define wxACTION_TOPLEVEL_BUTTON_RELEASE _T("releasebtn") // press titlebar btn
#define wxACTION_TOPLEVEL_BUTTON_CLICK   _T("clickbtn")   // press titlebar btn
#define wxACTION_TOPLEVEL_MOVE           _T("move")       // move the frame
#define wxACTION_TOPLEVEL_RESIZE         _T("resize")     // resize the frame

//-----------------------------------------------------------------------------
// wxTopLevelWindow
//-----------------------------------------------------------------------------

class WXDLLEXPORT uniTopLevelWindow :  public uniWindow,
                                     public wxInputConsumer
{
public:
    // construction
    uniTopLevelWindow() { Init(); }
    uniTopLevelWindow(wxWindow *parent,
                     wxWindowID id,
                     const wxString& title,
                     const wxPoint& pos = wxDefaultPosition,
                     const wxSize& size = wxDefaultSize,
                     long style = wxDEFAULT_FRAME_STYLE,
                     const wxString& name = wxFrameNameStr)
    {
        Init();

        Create(parent, id, title, pos, size, style, name);
    }

    bool Create(wxWindow *parent,
                wxWindowID id,
                const wxString& title,
                const wxPoint& pos = wxDefaultPosition,
                const wxSize& size = wxDefaultSize,
                long style = wxDEFAULT_FRAME_STYLE,
                const wxString& name = wxFrameNameStr);

    // wxUniv-specific methods: do [not] use native decorations for this (or
    // all) window(s)
    //
    // notice that this has no effect if the system doesn't support any native
    // decorations anyhow and that by default native decorations are used
    //
    // if UseNativeDecorations() is used, it must be called before Create()
    static void UseNativeDecorationsByDefault(bool native = true);
    void UseNativeDecorations(bool native = true);
    bool IsUsingNativeDecorations() const;


    // implement base class pure virtuals
    virtual bool ShowFullScreen(bool show, long style = wxFULLSCREEN_ALL);
    virtual wxPoint GetClientAreaOrigin() const;
    virtual void SetIcon(const wxIcon& icon) { SetIcons( wxIconBundle( icon ) ); }
    virtual void SetIcons(const wxIconBundle& icons);

    // implementation from now on
    // --------------------------

    // tests for frame's part at given point
    long HitTest(const wxPoint& pt) const;

    virtual bool PerformAction(const wxControlAction& action,
                               long numArg = -1,
                               const wxString& strArg = wxEmptyString);

    static wxInputHandler *GetStdInputHandler(wxInputHandler *handlerDef);
    virtual wxInputHandler *DoGetStdInputHandler(wxInputHandler *handlerDef)
    {
        return GetStdInputHandler(handlerDef);
    }

    // move/resize the frame interactively, i.e. let the user do it
    virtual void InteractiveMove(int flags = wxINTERACTIVE_MOVE);

    virtual wxSize GetMinSize() const;

    virtual uniTopLevelWindow *GetInputWindow() const { return wx_const_cast(uniTopLevelWindow*, this); }

protected:
    virtual void DoGetClientSize(int *width, int *height) const;
    virtual void DoSetClientSize(int width, int height);

    // handle titlebar button click event
    virtual void ClickTitleBarButton(long button);

    // return wxTOPLEVEL_xxx combination based on current state of the frame
    long GetDecorationsStyle() const;

    // common part of all ctors
    void Init();

    void RefreshTitleBar();
    void OnNcPaint(wxNcPaintEvent& event);
    void OnSystemMenu(wxCommandEvent& event);

    // true if wxTLW should render decorations (aka titlebar) itself
    static int ms_drawDecorations;
    // true if wxTLW can be iconized
    static int ms_canIconize;

    // true if we're using native decorations
    bool m_usingNativeDecorations;
    // true for currently active frame
    bool m_isActive;
    // version of icon for titlebar (16x16)
    wxIcon m_titlebarIcon;
    // saved window style in fullscreen mdoe
    long m_fsSavedStyle;
    // currently pressed titlebar button
    long m_pressedButton;

        // the renderer we use
    wxRenderer *m_renderer;


    DECLARE_DYNAMIC_CLASS(uniTopLevelWindow)
    DECLARE_EVENT_TABLE()
    WX_DECLARE_INPUT_CONSUMER()
};

//#endif // __WX_UNIV_TOPLEVEL_H__

/////////////////////////////////////////////////////////////////////////////
// Name:        wx/dialog.h
// Purpose:     wxDialogBase class
// Author:      Vadim Zeitlin
// Modified by:
// Created:     29.06.99
// RCS-ID:      $Id: dialog.h 49804 2007-11-10 01:09:42Z VZ $
// Copyright:   (c) Vadim Zeitlin
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

//#ifndef _WX_DIALOG_H_BASE_
//#define _WX_DIALOG_H_BASE_

#include "wx/defs.h"
#include "wx/containr.h"
#include "wx/toplevel.h"
#include "wx/dlimpexp.h"
#include "wx/sizer.h"

         class /*WXDLLIMPEXP_FWD_CORE*/ wxSizer;
         class /*WXDLLIMPEXP_FWD_CORE*/ wxStdDialogButtonSizer;

#define wxDIALOG_NO_PARENT      0x0001  // Don't make owned by apps top window

#ifdef __WXWINCE__
#define wxDEFAULT_DIALOG_STYLE  (wxCAPTION | wxMAXIMIZE | wxCLOSE_BOX | wxNO_BORDER)
#else
#define wxDEFAULT_DIALOG_STYLE  (wxCAPTION | wxSYSTEM_MENU | wxCLOSE_BOX)
#endif

         extern WXDLLEXPORT_DATA(const wxChar) wxDialogNameStr[];

class WXDLLEXPORT uniDialogBase : public uniTopLevelWindow
{
      public:
            enum
            {
        // all flags allowed in wxDialogBase::CreateButtonSizer()
                  ButtonSizerFlags = wxOK|wxCANCEL|wxYES|wxNO|wxHELP|wxNO_DEFAULT
            };

            uniDialogBase() { Init(); }
            virtual ~uniDialogBase() { }

    // define public wxDialog methods to be implemented by the derived classes
            virtual int ShowModal() = 0;
            virtual void EndModal(int retCode) = 0;
            virtual bool IsModal() const = 0;


    // Modal dialogs have a return code - usually the id of the last
    // pressed button
            void SetReturnCode(int returnCode) { m_returnCode = returnCode; }
            int GetReturnCode() const { return m_returnCode; }

    // Set the identifier for the affirmative button: this button will close
    // the dialog after validating data and calling TransferDataFromWindow()
            void SetAffirmativeId(int affirmativeId);
            int GetAffirmativeId() const { return m_affirmativeId; }

    // Set identifier for Esc key translation: the button with this id will
    // close the dialog without doing anything else; special value wxID_NONE
    // means to not handle Esc at all while wxID_ANY means to map Esc to
    // wxID_CANCEL if present and GetAffirmativeId() otherwise
            void SetEscapeId(int escapeId);
            int GetEscapeId() const { return m_escapeId; }

#if wxUSE_STATTEXT // && wxUSE_TEXTCTRL
    // splits text up at newlines and places the
    // lines into a vertical wxBoxSizer
            wxSizer *CreateTextSizer( const wxString &message );
#endif // wxUSE_STATTEXT // && wxUSE_TEXTCTRL

    // returns a horizontal wxBoxSizer containing the given buttons
            //
    // notice that the returned sizer can be NULL if no buttons are put in the
    // sizer (this mostly happens under smart phones and other atypical
    // platforms which have hardware buttons replacing OK/Cancel and such)
            wxSizer *CreateButtonSizer(long flags);

    // returns the sizer containing CreateButtonSizer() below a separating
    // static line for the platforms which use static lines for items
    // separation (i.e. not Mac)
            wxSizer *CreateSeparatedButtonSizer(long flags);

#if wxUSE_BUTTON
            wxStdDialogButtonSizer *CreateStdDialogButtonSizer( long flags );
#endif // wxUSE_BUTTON

      protected:
    // emulate click of a button with the given id if it's present in the dialog
            //
    // return true if button was "clicked" or false if we don't have it
            bool EmulateButtonClickIfPresent(int id);

    // this function is used by OnCharHook() to decide whether the given key
    // should close the dialog
            //
    // for most platforms the default implementation (which just checks for
    // Esc) is sufficient, but Mac port also adds Cmd-. here and other ports
    // could do something different if needed
            virtual bool IsEscapeKey(const wxKeyEvent& event);

    // end either modal or modeless dialog, for the modal dialog rc is used as
    // the dialog return code
            void EndDialog(int rc);

    // call Validate() and TransferDataFromWindow() and close dialog with
    // wxID_OK return code
            void AcceptAndClose();


    // The return code from modal dialog
            int m_returnCode;

    // The identifier for the affirmative button (usually wxID_OK)
            int m_affirmativeId;

    // The identifier for cancel button (usually wxID_CANCEL)
            int m_escapeId;

      private:
    // common part of all ctors
            void Init();

    // handle Esc key presses
            void OnCharHook(wxKeyEvent& event);

    // handle closing the dialog window
            void OnCloseWindow(wxCloseEvent& event);

    // handle the standard buttons
            void OnButton(wxCommandEvent& event);

    // update the background colour
            void OnSysColourChanged(wxSysColourChangedEvent& event);


            DECLARE_NO_COPY_CLASS(uniDialogBase)
                        DECLARE_EVENT_TABLE()
                        WX_DECLARE_CONTROL_CONTAINER();
};

/*
#if defined(__WXUNIVERSAL__) && !defined(__WXMICROWIN__)
#include "wx/univ/dialog.h"
#else
#if defined(__WXPALMOS__)
#include "wx/palmos/dialog.h"
#elif defined(__WXMSW__)
#include "wx/msw/dialog.h"
#elif defined(__WXMOTIF__)
#include "wx/motif/dialog.h"
#elif defined(__WXGTK20__)
#include "wx/gtk/dialog.h"
#elif defined(__WXGTK__)
#include "wx/gtk1/dialog.h"
#elif defined(__WXMAC__)
#include "wx/mac/dialog.h"
#elif defined(__WXCOCOA__)
#include "wx/cocoa/dialog.h"
#elif defined(__WXPM__)
#include "wx/os2/dialog.h"
#endif
#endif
*/

//#endif    // _WX_DIALOG_H_BASE_

/////////////////////////////////////////////////////////////////////////////
// Name:        dialog.h
// Purpose:     wxDialog class
// Author:      Vaclav Slavik
// Created:     2001/09/16
// RCS-ID:      $Id: dialog.h 36891 2006-01-16 14:59:55Z MR $
// Copyright:   (c) 2001 SciTech Software, Inc. (www.scitechsoft.com)
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

//#ifndef _WX_UNIV_DIALOG_H_
//#define _WX_UNIV_DIALOG_H_

extern WXDLLEXPORT_DATA(const wxChar) wxDialogNameStr[];
class WXDLLEXPORT wxWindowDisabler;
class WXDLLEXPORT wxEventLoop;

// Dialog boxes
class WXDLLEXPORT uniDialog : public uniDialogBase
{
      public:
            uniDialog() { Init(); }

    // Constructor with no modal flag - the new convention.
            uniDialog(wxWindow *parent, wxWindowID id,
                      const wxString& title,
                      const wxPoint& pos = wxDefaultPosition,
                      const wxSize& size = wxDefaultSize,
                      long style = wxDEFAULT_DIALOG_STYLE,
                      const wxString& name = wxDialogNameStr)
            {
                  Init();
                  Create(parent, id, title, pos, size, style, name);
            }

            bool Create(wxWindow *parent, wxWindowID id,
                        const wxString& title,
                        const wxPoint& pos = wxDefaultPosition,
                        const wxSize& size = wxDefaultSize,
                        long style = wxDEFAULT_DIALOG_STYLE,
                        const wxString& name = wxDialogNameStr);

            virtual ~uniDialog();

    // is the dialog in modal state right now?
            virtual bool IsModal() const;

    // For now, same as Show(true) but returns return code
            virtual int ShowModal();

    // may be called to terminate the dialog with the given return code
            virtual void EndModal(int retCode);

    // returns true if we're in a modal loop
            bool IsModalShowing() const;

            virtual bool Show(bool show = true);

    // implementation only from now on
    // -------------------------------

    // event handlers
            void OnCloseWindow(wxCloseEvent& event);
            void OnOK(wxCommandEvent& event);
            void OnApply(wxCommandEvent& event);
            void OnCancel(wxCommandEvent& event);

      protected:
    // common part of all ctors
            void Init();

      private:
    // while we are showing a modal dialog we disable the other windows using
    // this object
            wxWindowDisabler *m_windowDisabler;

    // modal dialog runs its own event loop
            wxEventLoop *m_eventLoop;

    // is modal right now?
            bool m_isShowingModal;

            DECLARE_DYNAMIC_CLASS(uniDialog)
                        DECLARE_EVENT_TABLE()
};

//#endif
    // _WX_UNIV_DIALOG_H_


///////////////////////////////////////////////////////////////////////////////
// Name:        wx/univ/theme.h
// Purpose:     wxTheme class manages all configurable aspects of the
//              application including the look (wxRenderer), feel
//              (wxInputHandler) and the colours (wxColourScheme)
// Author:      Vadim Zeitlin
// Modified by:
// Created:     06.08.00
// RCS-ID:      $Id: theme.h 42455 2006-10-26 15:33:10Z VS $
// Copyright:   (c) 2000 SciTech Software, Inc. (www.scitechsoft.com)
// Licence:     wxWindows licence
///////////////////////////////////////////////////////////////////////////////

//#ifndef _WX_UNIV_THEME_H_
//#define _WX_UNIV_THEME_H_

#include "wx/string.h"

// ----------------------------------------------------------------------------
// wxTheme
// ----------------------------------------------------------------------------

         class WXDLLEXPORT wxArtProvider;
         class WXDLLEXPORT wxColourScheme;
         class WXDLLEXPORT wxInputConsumer;
         class WXDLLEXPORT wxInputHandler;
         class WXDLLEXPORT wxRenderer;
         struct WXDLLEXPORT wxThemeInfo;

class WXDLLEXPORT wxTheme
{
      public:
    // static methods
    // --------------

    // create the default theme
            static bool CreateDefault();

    // create the theme by name (will return NULL if not found)
            static wxTheme *Create(const wxString& name);

    // change the current scheme
            static wxTheme *Set(wxTheme *theme);

    // get the current theme (never NULL)
            static wxTheme *Get() { return ms_theme; }

    // the theme methods
    // -----------------

    // get the renderer implementing all the control-drawing operations in
    // this theme
            virtual wxRenderer *GetRenderer() = 0;

    // get the art provider to be used together with this theme
            virtual wxArtProvider *GetArtProvider() = 0;

    // get the input handler of the given type, forward to the standard one
            virtual wxInputHandler *GetInputHandler(const wxString& handlerType,
                        wxInputConsumer *consumer) = 0;

    // get the colour scheme for the control with this name
            virtual wxColourScheme *GetColourScheme() = 0;

    // implementation only from now on
    // -------------------------------

            virtual ~wxTheme();

      private:
    // the list of descriptions of all known themes
            static wxThemeInfo *ms_allThemes;

    // the current theme
            static wxTheme *ms_theme;
            friend struct WXDLLEXPORT wxThemeInfo;
};

// ----------------------------------------------------------------------------
// wxDelegateTheme: it is impossible to inherit from any of standard
// themes as their declarations are in private code, but you can use this
// class to override only some of their functions - all the other ones
// will be left to the original theme
// ----------------------------------------------------------------------------

class wxDelegateTheme : public wxTheme
{
      public:
            wxDelegateTheme(const wxChar *theme);
            virtual ~wxDelegateTheme();

            virtual wxRenderer *GetRenderer();
            virtual wxArtProvider *GetArtProvider();
            virtual wxInputHandler *GetInputHandler(const wxString& control,
                        wxInputConsumer *consumer);
            virtual wxColourScheme *GetColourScheme();

      protected:
    // gets or creates theme and sets m_theme to point to it,
    // returns true on success
            bool GetOrCreateTheme();

            wxString    m_themeName;
            wxTheme    *m_theme;
};

// ----------------------------------------------------------------------------
// dynamic theme creation helpers
// ----------------------------------------------------------------------------

struct WXDLLEXPORT wxThemeInfo
{
      typedef wxTheme *(*Constructor)();

    // theme name and (user readable) description
      wxString name, desc;

    // the function to create a theme object
      Constructor ctor;

    // next node in the linked list or NULL
      wxThemeInfo *next;

    // constructor for the struct itself
      wxThemeInfo(Constructor ctor, const wxChar *name, const wxChar *desc);
};

// ----------------------------------------------------------------------------
// macros
// ----------------------------------------------------------------------------

// to use a standard theme insert this macro into one of the application files:
// without it, an over optimizing linker may discard the object module
// containing the theme implementation entirely
#define WX_USE_THEME(themename)                                             \
    /* this indirection makes it possible to pass macro as the argument */  \
    WX_USE_THEME_IMPL(themename)

#define WX_USE_THEME_IMPL(themename)                                        \
    extern WXDLLEXPORT_DATA(bool) wxThemeUse##themename;                    \
    static struct wxThemeUserFor##themename                                 \
{                                                                       \
        wxThemeUserFor##themename() { wxThemeUse##themename = true; }       \
} wxThemeDoUse##themename

// to declare a new theme, this macro must be used in the class declaration
#define WX_DECLARE_THEME(themename)                                         \
    private:                                                                \
        static wxThemeInfo ms_info##themename;                              \
    public:                                                                 \
        const wxThemeInfo *GetThemeInfo() const                             \
{ return &ms_info##themename; }

// and this one must be inserted in the source file
#define WX_IMPLEMENT_THEME(classname, themename, themedesc)                 \
    WXDLLEXPORT_DATA(bool) wxThemeUse##themename = true;                    \
    wxTheme *wxCtorFor##themename() { return new classname; }               \
    wxThemeInfo classname::ms_info##themename(wxCtorFor##themename,         \
                                              wxT( #themename ), themedesc)

// ----------------------------------------------------------------------------
// determine default theme
// ----------------------------------------------------------------------------

#if 1 //wxUSE_ALL_THEMES
#undef  wxUSE_THEME_WIN32
#define wxUSE_THEME_WIN32  1
#undef  wxUSE_THEME_GTK
#define wxUSE_THEME_GTK    1
#undef  wxUSE_THEME_MONO
#define wxUSE_THEME_MONO   1
#undef  wxUSE_THEME_METAL
#define wxUSE_THEME_METAL  1
#endif // wxUSE_ALL_THEMES

// determine the default theme to use:
#if defined(__WXGTK__) && wxUSE_THEME_GTK
#define wxUNIV_DEFAULT_THEME gtk
#elif defined(__WXDFB__) && wxUSE_THEME_MONO
    // use mono theme for DirectFB port because it cannot correctly
    // render neither win32 nor gtk themes yet:
#define wxUNIV_DEFAULT_THEME mono
#endif

// if no theme was picked, get any theme compiled in (sorted by
// quality/completeness of the theme):
#ifndef wxUNIV_DEFAULT_THEME
#if wxUSE_THEME_WIN32
#define wxUNIV_DEFAULT_THEME win32
#elif wxUSE_THEME_GTK
#define wxUNIV_DEFAULT_THEME gtk
#elif wxUSE_THEME_MONO
#define wxUNIV_DEFAULT_THEME mono
#endif
    // If nothing matches, no themes are compiled and the app must provide
    // some theme itself
    // (note that wxUSE_THEME_METAL depends on win32 theme, so we don't have to
    // try it)
    //
#endif // !wxUNIV_DEFAULT_THEME

//#endif // _WX_UNIV_THEME_H_

///////////////////////////////////////////////////////////////////////////////
// Name:        wx/univ/colschem.h
// Purpose:     wxColourScheme class provides the colours to use for drawing
// Author:      Vadim Zeitlin
// Modified by:
// Created:     19.08.00
// RCS-ID:      $Id: colschem.h 44058 2006-12-24 19:06:39Z VS $
// Copyright:   (c) 2000 SciTech Software, Inc. (www.scitechsoft.com)
// Licence:     wxWindows licence
///////////////////////////////////////////////////////////////////////////////

//#ifndef _WX_UNIV_COLSCHEM_H_
//#define _WX_UNIV_COLSCHEM_H_

class WXDLLEXPORT wxWindow;

#include "wx/colour.h"
#include "wx/checkbox.h"

// ----------------------------------------------------------------------------
// wxColourScheme
// ----------------------------------------------------------------------------

class WXDLLEXPORT wxColourScheme
{
      public:
    // the standard colours
            enum StdColour
            {
        // the background colour for a window
                  WINDOW,

        // the different background and text colours for the control
                  CONTROL,
                  CONTROL_PRESSED,
                  CONTROL_CURRENT,

        // the label text for the normal and the disabled state
                  CONTROL_TEXT,
                  CONTROL_TEXT_DISABLED,
                  CONTROL_TEXT_DISABLED_SHADOW,

        // the scrollbar background colour for the normal and pressed states
                  SCROLLBAR,
                  SCROLLBAR_PRESSED,

        // the background and text colour for the highlighted item
                  HIGHLIGHT,
                  HIGHLIGHT_TEXT,

        // these colours are used for drawing the shadows of 3D objects
                  SHADOW_DARK,
                  SHADOW_HIGHLIGHT,
                  SHADOW_IN,
                  SHADOW_OUT,

        // the titlebar background colours for the normal and focused states
                  TITLEBAR,
                  TITLEBAR_ACTIVE,

        // the titlebar text colours
                  TITLEBAR_TEXT,
                  TITLEBAR_ACTIVE_TEXT,

        // the default gauge fill colour
                  GAUGE,

        // desktop background colour (only used by framebuffer ports)
                  DESKTOP,

        // wxFrame's background colour
                  FRAME,

                  MAX
            };

    // get a standard colour
            virtual wxColour Get(StdColour col) const = 0;

    // get the background colour for the given window
            virtual wxColour GetBackground(wxWindow *win) const = 0;

    // virtual dtor for any base class
            virtual ~wxColourScheme();
};

// some people just can't spell it correctly :-)
typedef wxColourScheme wxColorScheme;

// ----------------------------------------------------------------------------
// macros
// ----------------------------------------------------------------------------

// retrieve the default colour from the theme or the given scheme
#define wxSCHEME_COLOUR(scheme, what) scheme->Get(wxColorScheme::what)
#define wxTHEME_COLOUR(what) \
    wxSCHEME_COLOUR(wxTheme::Get()->GetColourScheme(), what)

// get the background colour for the window in the current theme
#define wxTHEME_BG_COLOUR(win) \
    wxTheme::Get()->GetColourScheme()->GetBackground(win)

//#endif // _WX_UNIV_COLSCHEM_H_



///////////////////////////////////////////////////////////////////////////////
// Name:        wx/univ/renderer.h
// Purpose:     wxRenderer class declaration
// Author:      Vadim Zeitlin
// Modified by:
// Created:     06.08.00
// RCS-ID:      $Id: renderer.h 43726 2006-11-30 23:44:55Z RD $
// Copyright:   (c) 2000 SciTech Software, Inc. (www.scitechsoft.com)
// Licence:     wxWindows licence
///////////////////////////////////////////////////////////////////////////////

//#ifndef _WX_UNIV_RENDERER_H_
//#define _WX_UNIV_RENDERER_H_

/*
   wxRenderer class is used to draw all wxWidgets controls. This is an ABC and
   the look of the application is determined by the concrete derivation of
   wxRenderer used in the program.

   It also contains a few static methods which may be used by the concrete
   renderers and provide the functionality which is often similar or identical
   in all renderers (using inheritance here would be more restrictive as the
   given concrete renderer may need an arbitrary subset of the base class
   methods).

   Finally note that wxRenderer supersedes wxRendererNative in wxUniv build and
   includes the latters functionality (which it may delegate to the generic
   implementation of the latter or reimplement itself).
 */

#include "wx/renderer.h"

                     //dsr fake it out
#undef wxHAS_NATIVE_RENDERER

         class WXDLLEXPORT wxWindow;
         class WXDLLEXPORT wxDC;
         class WXDLLEXPORT wxCheckListBox;

#if wxUSE_LISTBOX
         class WXDLLEXPORT wxListBox;
#endif // wxUSE_LISTBOX

#if wxUSE_MENUS
         class WXDLLEXPORT wxMenu;
         class WXDLLEXPORT wxMenuGeometryInfo;
#endif // wxUSE_MENUS

         class WXDLLEXPORT wxScrollBar;

#if wxUSE_TEXTCTRL
         class WXDLLEXPORT wxTextCtrl;
#endif

#if wxUSE_GAUGE
         class WXDLLEXPORT wxGauge;
#endif // wxUSE_GAUGE

#include "wx/string.h"
#include "wx/gdicmn.h"
#include "wx/icon.h"

// helper class used by wxMenu-related functions
                     class WXDLLEXPORT wxMenuGeometryInfo
{
      public:
    // get the total size of the menu
            virtual wxSize GetSize() const = 0;

            virtual ~wxMenuGeometryInfo();
};

// ----------------------------------------------------------------------------
// wxRenderer: abstract renderers interface
// ----------------------------------------------------------------------------

class WXDLLEXPORT wxRenderer : public wxDelegateRendererNative
{
      public:
    // drawing functions
    // -----------------

    // draw the controls background
            virtual void DrawBackground(wxDC& dc,
                                        const wxColour& col,
                                        const wxRect& rect,
                                        int flags,
                                        wxWindow *window = NULL) = 0;

    // draw the button surface
            virtual void DrawButtonSurface(wxDC& dc,
                                           const wxColour& col,
                                           const wxRect& rect,
                                           int flags) = 0;


    // draw the focus rectangle around the label contained in the given rect
            //
    // only wxCONTROL_SELECTED makes sense in flags here
            virtual void DrawFocusRect(wxDC& dc, const wxRect& rect, int flags = 0) = 0;

    // draw the label inside the given rectangle with the specified alignment
    // and optionally emphasize the character with the given index
            virtual void DrawLabel(wxDC& dc,
                                   const wxString& label,
                                   const wxRect& rect,
                                   int flags = 0,
                                   int alignment = wxALIGN_LEFT | wxALIGN_TOP,
                                   int indexAccel = -1,
                                   wxRect *rectBounds = NULL) = 0;

    // same but also draw a bitmap if it is valid
            virtual void DrawButtonLabel(wxDC& dc,
                                         const wxString& label,
                                         const wxBitmap& image,
                                         const wxRect& rect,
                                         int flags = 0,
                                         int alignment = wxALIGN_LEFT | wxALIGN_TOP,
                                         int indexAccel = -1,
                                         wxRect *rectBounds = NULL) = 0;


    // draw the border and optionally return the rectangle containing the
    // region inside the border
            virtual void DrawBorder(wxDC& dc,
                                    wxBorder border,
                                    const wxRect& rect,
                                    int flags = 0,
                                    wxRect *rectIn = (wxRect *)NULL) = 0;

    // draw text control border (I hate to have a separate method for this but
    // it is needed to accommodate GTK+)
            virtual void DrawTextBorder(wxDC& dc,
                                        wxBorder border,
                                        const wxRect& rect,
                                        int flags = 0,
                                        wxRect *rectIn = (wxRect *)NULL) = 0;

    // draw push button border and return the rectangle left for the label
            virtual void DrawButtonBorder(wxDC& dc,
                                          const wxRect& rect,
                                          int flags = 0,
                                          wxRect *rectIn = (wxRect *)NULL) = 0;

    // draw a horizontal line
            virtual void DrawHorizontalLine(wxDC& dc,
                                            wxCoord y, wxCoord x1, wxCoord x2) = 0;

    // draw a vertical line
            virtual void DrawVerticalLine(wxDC& dc,
                                          wxCoord x, wxCoord y1, wxCoord y2) = 0;

    // draw a frame with the label (horizontal alignment can be specified)
            virtual void DrawFrame(wxDC& dc,
                                   const wxString& label,
                                   const wxRect& rect,
                                   int flags = 0,
                                   int alignment = wxALIGN_LEFT,
                                   int indexAccel = -1) = 0;

    // draw an arrow in the given direction
            virtual void DrawArrow(wxDC& dc,
                                   wxDirection dir,
                                   const wxRect& rect,
                                   int flags = 0) = 0;

    // draw a scrollbar arrow (may be the same as arrow but may be not)
            virtual void DrawScrollbarArrow(wxDC& dc,
                                            wxDirection dir,
                                            const wxRect& rect,
                                            int flags = 0) = 0;

    // draw the scrollbar thumb
            virtual void DrawScrollbarThumb(wxDC& dc,
                                            wxOrientation orient,
                                            const wxRect& rect,
                                            int flags = 0) = 0;

    // draw a (part of) scrollbar shaft
            virtual void DrawScrollbarShaft(wxDC& dc,
                                            wxOrientation orient,
                                            const wxRect& rect,
                                            int flags = 0) = 0;

    // draw the rectangle in the corner between two scrollbars
            virtual void DrawScrollCorner(wxDC& dc,
                                          const wxRect& rect) = 0;

    // draw an item of a wxListBox
            virtual void DrawItem(wxDC& dc,
                                  const wxString& label,
                                  const wxRect& rect,
                                  int flags = 0) = 0;

    // draw an item of a wxCheckListBox
            virtual void DrawCheckItem(wxDC& dc,
                                       const wxString& label,
                                       const wxBitmap& bitmap,
                                       const wxRect& rect,
                                       int flags = 0) = 0;

    // draw a checkbutton (bitmap may be invalid to use default one)
            virtual void DrawCheckButton(wxDC& dc,
                                         const wxString& label,
                                         const wxBitmap& bitmap,
                                         const wxRect& rect,
                                         int flags = 0,
                                         wxAlignment align = wxALIGN_LEFT,
                                         int indexAccel = -1) = 0;

    // draw a radio button
            virtual void DrawRadioButton(wxDC& dc,
                                         const wxString& label,
                                         const wxBitmap& bitmap,
                                         const wxRect& rect,
                                         int flags = 0,
                                         wxAlignment align = wxALIGN_LEFT,
                                         int indexAccel = -1) = 0;

#if wxUSE_TOOLBAR
    // draw a toolbar button (label may be empty, bitmap may be invalid, if
    // both conditions are true this function draws a separator)
            virtual void DrawToolBarButton(wxDC& dc,
                                           const wxString& label,
                                           const wxBitmap& bitmap,
                                           const wxRect& rect,
                                           int flags = 0,
                                           long style = 0,
                                           int tbarStyle = 0) = 0;
#endif // wxUSE_TOOLBAR

#if wxUSE_TEXTCTRL
    // draw a (part of) line in the text control
            virtual void DrawTextLine(wxDC& dc,
                                      const wxString& text,
                                      const wxRect& rect,
                                      int selStart = -1,
                                      int selEnd = -1,
                                      int flags = 0) = 0;

    // draw a line wrap indicator
            virtual void DrawLineWrapMark(wxDC& dc, const wxRect& rect) = 0;
#endif // wxUSE_TEXTCTRL

#if wxUSE_NOTEBOOK
    // draw a notebook tab
            virtual void DrawTab(wxDC& dc,
                                 const wxRect& rect,
                                 wxDirection dir,
                                 const wxString& label,
                                 const wxBitmap& bitmap = wxNullBitmap,
                                 int flags = 0,
                                 int indexAccel = -1) = 0;
#endif // wxUSE_NOTEBOOK

#if wxUSE_SLIDER

    // draw the slider shaft
            virtual void DrawSliderShaft(wxDC& dc,
                                         const wxRect& rect,
                                         int lenThumb,
                                         wxOrientation orient,
                                         int flags = 0,
                                         long style = 0,
                                         wxRect *rectShaft = NULL) = 0;

    // draw the slider thumb
            virtual void DrawSliderThumb(wxDC& dc,
                                         const wxRect& rect,
                                         wxOrientation orient,
                                         int flags = 0,
                                         long style = 0) = 0;

    // draw the slider ticks
            virtual void DrawSliderTicks(wxDC& dc,
                                         const wxRect& rect,
                                         int lenThumb,
                                         wxOrientation orient,
                                         int start,
                                         int end,
                                         int step = 1,
                                         int flags = 0,
                                         long style = 0) = 0;
#endif // wxUSE_SLIDER

#if wxUSE_MENUS
    // draw a menu bar item
            virtual void DrawMenuBarItem(wxDC& dc,
                                         const wxRect& rect,
                                         const wxString& label,
                                         int flags = 0,
                                         int indexAccel = -1) = 0;

    // draw a menu item (also used for submenus if flags has ISSUBMENU flag)
            //
    // the geometryInfo is calculated by GetMenuGeometry() function from below
            virtual void DrawMenuItem(wxDC& dc,
                                      wxCoord y,
                                      const wxMenuGeometryInfo& geometryInfo,
                                      const wxString& label,
                                      const wxString& accel,
                                      const wxBitmap& bitmap = wxNullBitmap,
                                      int flags = 0,
                                      int indexAccel = -1) = 0;

    // draw a menu bar separator
            virtual void DrawMenuSeparator(wxDC& dc,
                                           wxCoord y,
                                           const wxMenuGeometryInfo& geomInfo) = 0;
#endif // wxUSE_MENUS

#if wxUSE_STATUSBAR
    // draw a status bar field: wxCONTROL_ISDEFAULT bit in the flags is
    // interpreted specially and means "draw the status bar grip" here
            virtual void DrawStatusField(wxDC& dc,
                                         const wxRect& rect,
                                         const wxString& label,
                                         int flags = 0, int style = 0) = 0;
#endif // wxUSE_STATUSBAR

    // draw complete frame/dialog titlebar
            virtual void DrawFrameTitleBar(wxDC& dc,
                                           const wxRect& rect,
                                           const wxString& title,
                                           const wxIcon& icon,
                                           int flags,
                                           int specialButton = 0,
                                           int specialButtonFlags = 0) = 0;

    // draw frame borders
            virtual void DrawFrameBorder(wxDC& dc,
                                         const wxRect& rect,
                                         int flags) = 0;

    // draw frame titlebar background
            virtual void DrawFrameBackground(wxDC& dc,
                                             const wxRect& rect,
                                             int flags) = 0;

    // draw frame title
            virtual void DrawFrameTitle(wxDC& dc,
                                        const wxRect& rect,
                                        const wxString& title,
                                        int flags) = 0;

    // draw frame icon
            virtual void DrawFrameIcon(wxDC& dc,
                                       const wxRect& rect,
                                       const wxIcon& icon,
                                       int flags) = 0;

    // draw frame buttons
            virtual void DrawFrameButton(wxDC& dc,
                                         wxCoord x, wxCoord y,
                                         int button,
                                         int flags = 0) = 0;

    // misc functions
    // --------------

#if wxUSE_COMBOBOX
    // return the bitmaps to use for combobox button
            virtual void GetComboBitmaps(wxBitmap *bmpNormal,
                                         wxBitmap *bmpFocus,
                                         wxBitmap *bmpPressed,
                                         wxBitmap *bmpDisabled) = 0;
#endif // wxUSE_COMBOBOX

    // geometry functions
    // ------------------

    // get the dimensions of the border: rect.x/y contain the width/height of
    // the left/top side, width/heigh - of the right/bottom one
            virtual wxRect GetBorderDimensions(wxBorder border) const = 0;

    // the scrollbars may be drawn either inside the window border or outside
    // it - this function is used to decide how to draw them
            virtual bool AreScrollbarsInsideBorder() const = 0;

    // adjust the size of the control of the given class: for most controls,
    // this just takes into account the border, but for some (buttons, for
    // example) it is more complicated - the result being, in any case, that
    // the control looks "nice" if it uses the adjusted rectangle
            virtual void AdjustSize(wxSize *size, const wxWindow *window) = 0;

#if wxUSE_SCROLLBAR
    // get the size of a scrollbar arrow
            virtual wxSize GetScrollbarArrowSize() const = 0;
#endif // wxUSE_SCROLLBAR

    // get the height of a listbox item from the base font height
            virtual wxCoord GetListboxItemHeight(wxCoord fontHeight) = 0;

    // get the size of a checkbox/radio button bitmap
            virtual wxSize GetCheckBitmapSize() const = 0;
            virtual wxSize GetRadioBitmapSize() const = 0;
            virtual wxCoord GetCheckItemMargin() const = 0;

#if wxUSE_TOOLBAR
    // get the standard size of a toolbar button and also return the size of
    // a toolbar separator in the provided pointer
            virtual wxSize GetToolBarButtonSize(wxCoord *separator) const = 0;

    // get the margins between/around the toolbar buttons
            virtual wxSize GetToolBarMargin() const = 0;
#endif // wxUSE_TOOLBAR

#if wxUSE_TEXTCTRL
    // convert between text rectangle and client rectangle for text controls:
    // the former is typicall smaller to leave margins around text
            virtual wxRect GetTextTotalArea(const wxTextCtrl *text,
                                            const wxRect& rectText) const = 0;

    // extra space is for line indicators
            virtual wxRect GetTextClientArea(const wxTextCtrl *text,
                                             const wxRect& rectTotal,
                                             wxCoord *extraSpaceBeyond) const = 0;
#endif // wxUSE_TEXTCTRL

#if wxUSE_NOTEBOOK
    // get the overhang of a selected tab
            virtual wxSize GetTabIndent() const = 0;

    // get the padding around the text in a tab
            virtual wxSize GetTabPadding() const = 0;
#endif // wxUSE_NOTEBOOK

#if wxUSE_SLIDER
    // get the default size of the slider in lesser dimension (i.e. height of a
    // horizontal slider or width of a vertical one)
            virtual wxCoord GetSliderDim() const = 0;

    // get the length of the slider ticks displayed along side slider
            virtual wxCoord GetSliderTickLen() const = 0;

    // get the slider shaft rect from the total slider rect
            virtual wxRect GetSliderShaftRect(const wxRect& rect,
                                              int lenThumb,
                                              wxOrientation orient,
                                              long style = 0) const = 0;

    // get the size of the slider thumb for the given total slider rect
            virtual wxSize GetSliderThumbSize(const wxRect& rect,
                                              int lenThumb,
                                              wxOrientation orient) const = 0;
#endif // wxUSE_SLIDER

    // get the size of one progress bar step (in horz and vertical directions)
            virtual wxSize GetProgressBarStep() const = 0;

#if wxUSE_MENUS
    // get the size of rectangle to use in the menubar for the given text rect
            virtual wxSize GetMenuBarItemSize(const wxSize& sizeText) const = 0;

    // get the struct storing all layout info needed to draw all menu items
    // (this can't be calculated for each item separately as they should be
    // aligned)
            //
    // the returned pointer must be deleted by the caller
            virtual wxMenuGeometryInfo *GetMenuGeometry(wxWindow *win,
                        const wxMenu& menu) const = 0;
#endif // wxUSE_MENUS

#if wxUSE_STATUSBAR
    // get the borders around the status bar fields (x and y fields of the
    // return value)
            virtual wxSize GetStatusBarBorders() const = 0;

    // get the border between the status bar fields
            virtual wxCoord GetStatusBarBorderBetweenFields() const = 0;

    // get the mergin between a field and its border
            virtual wxSize GetStatusBarFieldMargins() const = 0;
#endif // wxUSE_STATUSBAR

    // get client area rectangle of top level window (i.e. subtract
    // decorations from given rectangle)
            virtual wxRect GetFrameClientArea(const wxRect& rect, int flags) const = 0;

    // get size of whole top level window, given size of its client area size
            virtual wxSize GetFrameTotalSize(const wxSize& clientSize, int flags) const = 0;

    // get the minimal size of top level window
            virtual wxSize GetFrameMinSize(int flags) const = 0;

    // get titlebar icon size
            virtual wxSize GetFrameIconSize() const = 0;

    // returns one of wxHT_TOPLEVEL_XXX constants
            virtual int HitTestFrame(const wxRect& rect,
                                     const wxPoint& pt,
                                     int flags = 0) const = 0;

    // virtual dtor for any base class
            virtual ~wxRenderer();

};

// ----------------------------------------------------------------------------
// wxDelegateRenderer: it is impossible to inherit from any of standard
// renderers as their declarations are in private code, but you can use this
// class to override only some of the Draw() functions - all the other ones
// will be left to the original renderer
// ----------------------------------------------------------------------------

class WXDLLEXPORT wxDelegateRenderer : public wxRenderer
{
      public:
            wxDelegateRenderer(wxRenderer *renderer) : m_renderer(renderer) { }

            virtual void DrawBackground(wxDC& dc,
                                        const wxColour& col,
                                        const wxRect& rect,
                                        int flags,
                                        wxWindow *window = NULL )
            { m_renderer->DrawBackground(dc, col, rect, flags, window ); }
            virtual void DrawButtonSurface(wxDC& dc,
                                           const wxColour& col,
                                           const wxRect& rect,
                                           int flags)
            { m_renderer->DrawButtonSurface(dc, col, rect, flags); }
            virtual void DrawFocusRect(wxDC& dc, const wxRect& rect, int flags = 0)
            { m_renderer->DrawFocusRect(dc, rect, flags); }
            virtual void DrawLabel(wxDC& dc,
                                   const wxString& label,
                                   const wxRect& rect,
                                   int flags = 0,
                                   int align = wxALIGN_LEFT | wxALIGN_TOP,
                                   int indexAccel = -1,
                                   wxRect *rectBounds = NULL)
            { m_renderer->DrawLabel(dc, label, rect,
                                    flags, align, indexAccel, rectBounds); }
                                    virtual void DrawButtonLabel(wxDC& dc,
                                                const wxString& label,
                                                            const wxBitmap& image,
                                                                        const wxRect& rect,
                                                                                int flags = 0,
                                                                                int align = wxALIGN_LEFT | wxALIGN_TOP,
                                                                                int indexAccel = -1,
                                                                                wxRect *rectBounds = NULL)
                                    { m_renderer->DrawButtonLabel(dc, label, image, rect,
                                                flags, align, indexAccel, rectBounds); }
                                        virtual void DrawBorder ( wxDC& dc,
                                                                  wxBorder border,
                                                                  const wxRect& rect,
                                                                  int flags = 0,
                                                                  wxRect *rectIn = ( wxRect * ) NULL )
                                        { m_renderer->DrawBorder ( dc, border, rect, flags, rectIn ); }
                                                virtual void DrawTextBorder ( wxDC& dc,
                                                                              wxBorder border,
                                                                              const wxRect& rect,
                                                                              int flags = 0,
                                                                              wxRect *rectIn = ( wxRect * ) NULL )
                                                { m_renderer->DrawTextBorder ( dc, border, rect, flags, rectIn ); }
                                                virtual void DrawButtonBorder(wxDC& dc,
                                                            const wxRect& rect,
                                                                        int flags = 0,
                                                                                wxRect *rectIn = (wxRect *)NULL)
                                                { m_renderer->DrawButtonBorder(dc, rect, flags, rectIn); }
                                                virtual void DrawFrame(wxDC& dc,
                                                            const wxString& label,
                                                                        const wxRect& rect,
                                                                                int flags = 0,
                                                                                int align = wxALIGN_LEFT,
                                                                                int indexAccel = -1)
                                                { m_renderer->DrawFrame(dc, label, rect, flags, align, indexAccel); }
                                                virtual void DrawHorizontalLine(wxDC& dc,
                                                            wxCoord y, wxCoord x1, wxCoord x2)
                                                { m_renderer->DrawHorizontalLine(dc, y, x1, x2); }
                                                virtual void DrawVerticalLine(wxDC& dc,
                                                            wxCoord x, wxCoord y1, wxCoord y2)
                                                { m_renderer->DrawVerticalLine(dc, x, y1, y2); }
                                                virtual void DrawArrow(wxDC& dc,
                                                            wxDirection dir,
                                                                        const wxRect& rect,
                                                                                int flags = 0)
                                                { m_renderer->DrawArrow(dc, dir, rect, flags); }
                                                virtual void DrawScrollbarArrow(wxDC& dc,
                                                            wxDirection dir,
                                                                        const wxRect& rect,
                                                                                int flags = 0)
                                                { m_renderer->DrawScrollbarArrow(dc, dir, rect, flags); }
                                                virtual void DrawScrollbarThumb(wxDC& dc,
                                                            wxOrientation orient,
                                                                        const wxRect& rect,
                                                                                int flags = 0)
                                                { m_renderer->DrawScrollbarThumb(dc, orient, rect, flags); }
                                                virtual void DrawScrollbarShaft(wxDC& dc,
                                                            wxOrientation orient,
                                                                        const wxRect& rect,
                                                                                int flags = 0)
                                                { m_renderer->DrawScrollbarShaft(dc, orient, rect, flags); }
                                                virtual void DrawScrollCorner(wxDC& dc,
                                                            const wxRect& rect)
                                                { m_renderer->DrawScrollCorner(dc, rect); }
                                                virtual void DrawItem(wxDC& dc,
                                                            const wxString& label,
                                                                        const wxRect& rect,
                                                                                int flags = 0)
                                                { m_renderer->DrawItem(dc, label, rect, flags); }
                                                virtual void DrawCheckItem(wxDC& dc,
                                                            const wxString& label,
                                                                        const wxBitmap& bitmap,
                                                                                const wxRect& rect,
                                                                                int flags = 0)
                                                { m_renderer->DrawCheckItem(dc, label, bitmap, rect, flags); }
                                                virtual void DrawCheckButton(wxDC& dc,
                                                            const wxString& label,
                                                                        const wxBitmap& bitmap,
                                                                                const wxRect& rect,
                                                                                int flags = 0,
                                                                                wxAlignment align = wxALIGN_LEFT,
                                                                                int indexAccel = -1)
                                                { m_renderer->DrawCheckButton(dc, label, bitmap, rect,
                                                            flags, align, indexAccel); }
                                                            virtual void DrawRadioButton(wxDC& dc,
                                                                        const wxString& label,
                                                                                const wxBitmap& bitmap,
                                                                                const wxRect& rect,
                                                                                int flags = 0,
                                                                                wxAlignment align = wxALIGN_LEFT,
                                                                                int indexAccel = -1)
                                                            { m_renderer->DrawRadioButton(dc, label, bitmap, rect,
                                                                        flags, align, indexAccel); }
#if wxUSE_TOOLBAR
                                                                        virtual void DrawToolBarButton(wxDC& dc,
                                                                                const wxString& label,
                                                                                const wxBitmap& bitmap,
                                                                                const wxRect& rect,
                                                                                int flags = 0,
                                                                                long style = 0,
                                                                                int tbarStyle = 0)
                                                                        { m_renderer->DrawToolBarButton(dc, label, bitmap, rect, flags, style, tbarStyle); }
#endif // wxUSE_TOOLBAR

#if wxUSE_TEXTCTRL
                                                                        virtual void DrawTextLine(wxDC& dc,
                                                                                const wxString& text,
                                                                                const wxRect& rect,
                                                                                int selStart = -1,
                                                                                int selEnd = -1,
                                                                                int flags = 0)
                                                                        { m_renderer->DrawTextLine(dc, text, rect, selStart, selEnd, flags); }
                                                                        virtual void DrawLineWrapMark(wxDC& dc, const wxRect& rect)
                                                                        { m_renderer->DrawLineWrapMark(dc, rect); }
#endif // wxUSE_TEXTCTRL

#if wxUSE_NOTEBOOK
                                                                        virtual void DrawTab(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                wxDirection dir,
                                                                                const wxString& label,
                                                                                const wxBitmap& bitmap = wxNullBitmap,
                                                                                int flags = 0,
                                                                                int accel = -1)
                                                                        { m_renderer->DrawTab(dc, rect, dir, label, bitmap, flags, accel); }
#endif // wxUSE_NOTEBOOK

#if wxUSE_SLIDER

                                                                        virtual void DrawSliderShaft(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                int lenThumb,
                                                                                wxOrientation orient,
                                                                                int flags = 0,
                                                                                long style = 0,
                                                                                wxRect *rectShaft = NULL)
                                                                        { m_renderer->DrawSliderShaft(dc, rect, lenThumb, orient, flags, style, rectShaft); }
                                                                        virtual void DrawSliderThumb(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                wxOrientation orient,
                                                                                int flags = 0,
                                                                                long style = 0)
                                                                        { m_renderer->DrawSliderThumb(dc, rect, orient, flags, style); }
                                                                        virtual void DrawSliderTicks(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                int lenThumb,
                                                                                wxOrientation orient,
                                                                                int start,
                                                                                int end,
                                                                                int WXUNUSED(step) = 1,
                                                                                int flags = 0,
                                                                                long style = 0)
                                                                        { m_renderer->DrawSliderTicks(dc, rect, lenThumb, orient,
                                                                                start, end, start, flags, style); }
#endif // wxUSE_SLIDER

#if wxUSE_MENUS
                                                                                virtual void DrawMenuBarItem(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                const wxString& label,
                                                                                int flags = 0,
                                                                                int indexAccel = -1)
                                                                                { m_renderer->DrawMenuBarItem(dc, rect, label, flags, indexAccel); }
                                                                                virtual void DrawMenuItem(wxDC& dc,
                                                                                wxCoord y,
                                                                                const wxMenuGeometryInfo& gi,
                                                                                const wxString& label,
                                                                                const wxString& accel,
                                                                                const wxBitmap& bitmap = wxNullBitmap,
                                                                                int flags = 0,
                                                                                int indexAccel = -1)
                                                                                { m_renderer->DrawMenuItem(dc, y, gi, label, accel,
                                                                                bitmap, flags, indexAccel); }
                                                                                virtual void DrawMenuSeparator(wxDC& dc,
                                                                                wxCoord y,
                                                                                const wxMenuGeometryInfo& geomInfo)
                                                                                { m_renderer->DrawMenuSeparator(dc, y, geomInfo); }
#endif // wxUSE_MENUS

#if wxUSE_STATUSBAR
                                                                                virtual void DrawStatusField(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                const wxString& label,
                                                                                int flags = 0, int style = 0)
                                                                                { m_renderer->DrawStatusField(dc, rect, label, flags, style); }
#endif // wxUSE_STATUSBAR

                                                                                virtual void DrawFrameTitleBar(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                const wxString& title,
                                                                                const wxIcon& icon,
                                                                                int flags,
                                                                                int specialButton = 0,
                                                                                int specialButtonFlag = 0)
                                                                                { m_renderer->DrawFrameTitleBar(dc, rect, title, icon, flags,
                                                                                specialButton, specialButtonFlag); }
                                                                                virtual void DrawFrameBorder(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                int flags)
                                                                                { m_renderer->DrawFrameBorder(dc, rect, flags); }
                                                                                virtual void DrawFrameBackground(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                int flags)
                                                                                { m_renderer->DrawFrameBackground(dc, rect, flags); }
                                                                                virtual void DrawFrameTitle(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                const wxString& title,
                                                                                int flags)
                                                                                { m_renderer->DrawFrameTitle(dc, rect, title, flags); }
                                                                                virtual void DrawFrameIcon(wxDC& dc,
                                                                                const wxRect& rect,
                                                                                const wxIcon& icon,
                                                                                int flags)
                                                                                { m_renderer->DrawFrameIcon(dc, rect, icon, flags); }
                                                                                virtual void DrawFrameButton(wxDC& dc,
                                                                                wxCoord x, wxCoord y,
                                                                                int button,
                                                                                int flags = 0)
                                                                                { m_renderer->DrawFrameButton(dc, x, y, button, flags); }

#if wxUSE_COMBOBOX
                                                                                virtual void GetComboBitmaps(wxBitmap *bmpNormal,
                                                                                wxBitmap *bmpFocus,
                                                                                wxBitmap *bmpPressed,
                                                                                wxBitmap *bmpDisabled)
                                                                                { m_renderer->GetComboBitmaps(bmpNormal, bmpFocus,
                                                                                bmpPressed, bmpDisabled); }
#endif // wxUSE_COMBOBOX

                                                                                virtual void AdjustSize(wxSize *size, const wxWindow *window)
                                                                                { m_renderer->AdjustSize(size, window); }
                                                                                virtual wxRect GetBorderDimensions(wxBorder border) const
                                                                                { return m_renderer->GetBorderDimensions(border); }
                                                                                virtual bool AreScrollbarsInsideBorder() const
                                                                                { return m_renderer->AreScrollbarsInsideBorder(); }

#if wxUSE_SCROLLBAR
                                                                                virtual wxSize GetScrollbarArrowSize() const
                                                                                { return m_renderer->GetScrollbarArrowSize(); }
#endif // wxUSE_SCROLLBAR

                                                                                virtual wxCoord GetListboxItemHeight(wxCoord fontHeight)
                                                                                { return m_renderer->GetListboxItemHeight(fontHeight); }
                                                                                virtual wxSize GetCheckBitmapSize() const
                                                                                { return m_renderer->GetCheckBitmapSize(); }
                                                                                virtual wxSize GetRadioBitmapSize() const
                                                                                { return m_renderer->GetRadioBitmapSize(); }
                                                                                virtual wxCoord GetCheckItemMargin() const
                                                                                { return m_renderer->GetCheckItemMargin(); }

#if wxUSE_TOOLBAR
                                                                                virtual wxSize GetToolBarButtonSize(wxCoord *separator) const
                                                                                { return m_renderer->GetToolBarButtonSize(separator); }
                                                                                virtual wxSize GetToolBarMargin() const
                                                                                { return m_renderer->GetToolBarMargin(); }
#endif // wxUSE_TOOLBAR

#if wxUSE_TEXTCTRL
                                                                                virtual wxRect GetTextTotalArea(const wxTextCtrl *text,
                                                                                const wxRect& rect) const
                                                                                { return m_renderer->GetTextTotalArea(text, rect); }
                                                                                virtual wxRect GetTextClientArea(const wxTextCtrl *text,
                                                                                const wxRect& rect,
                                                                                wxCoord *extraSpaceBeyond) const
                                                                                { return m_renderer->GetTextClientArea(text, rect, extraSpaceBeyond); }
#endif // wxUSE_TEXTCTRL

#if wxUSE_NOTEBOOK
                                                                                virtual wxSize GetTabIndent() const { return m_renderer->GetTabIndent(); }
                                                                                virtual wxSize GetTabPadding() const { return m_renderer->GetTabPadding(); }
#endif // wxUSE_NOTEBOOK

#if wxUSE_SLIDER
                                                                                virtual wxCoord GetSliderDim() const
                                                                                { return m_renderer->GetSliderDim(); }
                                                                                virtual wxCoord GetSliderTickLen() const
                                                                                { return m_renderer->GetSliderTickLen(); }

                                                                                virtual wxRect GetSliderShaftRect(const wxRect& rect,
                                                                                int lenThumb,
                                                                                wxOrientation orient,
                                                                                long style = 0) const
                                                                                { return m_renderer->GetSliderShaftRect(rect, lenThumb, orient, style); }
                                                                                virtual wxSize GetSliderThumbSize(const wxRect& rect,
                                                                                int lenThumb,
                                                                                wxOrientation orient) const
                                                                                { return m_renderer->GetSliderThumbSize(rect, lenThumb, orient); }
#endif // wxUSE_SLIDER

                                                                                virtual wxSize GetProgressBarStep() const
                                                                                { return m_renderer->GetProgressBarStep(); }

#if wxUSE_MENUS
                                                                                virtual wxSize GetMenuBarItemSize(const wxSize& sizeText) const
                                                                                { return m_renderer->GetMenuBarItemSize(sizeText); }
                                                                                virtual wxMenuGeometryInfo *GetMenuGeometry(wxWindow *win,
                                                                                const wxMenu& menu) const
                                                                                { return m_renderer->GetMenuGeometry(win, menu); }
#endif // wxUSE_MENUS

#if wxUSE_STATUSBAR
                                                                                virtual wxSize GetStatusBarBorders() const
                                                                                { return m_renderer->GetStatusBarBorders(); }
                                                                                virtual wxCoord GetStatusBarBorderBetweenFields() const
                                                                                { return m_renderer->GetStatusBarBorderBetweenFields(); }
                                                                                virtual wxSize GetStatusBarFieldMargins() const
                                                                                { return m_renderer->GetStatusBarFieldMargins(); }
#endif // wxUSE_STATUSBAR

                                                                                virtual wxRect GetFrameClientArea(const wxRect& rect, int flags) const
                                                                                { return m_renderer->GetFrameClientArea(rect, flags); }
                                                                                virtual wxSize GetFrameTotalSize(const wxSize& clientSize, int flags) const
                                                                                { return m_renderer->GetFrameTotalSize(clientSize, flags); }
                                                                                virtual wxSize GetFrameMinSize(int flags) const
                                                                                { return m_renderer->GetFrameMinSize(flags); }
                                                                                virtual wxSize GetFrameIconSize() const
                                                                                { return m_renderer->GetFrameIconSize(); }
                                                                                virtual int HitTestFrame(const wxRect& rect,
                                                                                const wxPoint& pt,
                                                                                int flags) const
                                                                                { return m_renderer->HitTestFrame(rect, pt, flags); }

                                                                                virtual int  DrawHeaderButton(wxWindow *win,
                                                                                wxDC& dc,
                                                                                const wxRect& rect,
                                                                                int flags = 0,
                                                                                wxHeaderSortIconType sortIcon = wxHDR_SORT_ICON_NONE,
                                                                                wxHeaderButtonParams* params = NULL)
                                                                                { return m_renderer->DrawHeaderButton(win, dc, rect, flags, sortIcon, params); }
                                                                                virtual void DrawTreeItemButton(wxWindow *win,
                                                                                wxDC& dc,
                                                                                const wxRect& rect,
                                                                                int flags = 0)
                                                                                { m_renderer->DrawTreeItemButton(win, dc, rect, flags); }

      protected:
            wxRenderer *m_renderer;
};

// ----------------------------------------------------------------------------
// wxControlRenderer: wraps the wxRenderer functions in a form easy to use from
// OnPaint()
// ----------------------------------------------------------------------------

class WXDLLEXPORT wxControlRenderer
{
      public:
    // create a renderer for this dc with this "fundamental" renderer
            wxControlRenderer(wxWindow *control, wxDC& dc, wxRenderer *renderer);

    // operations
            void DrawLabel(const wxBitmap& bitmap = wxNullBitmap,
                           wxCoord marginX = 0, wxCoord marginY = 0);
#if wxUSE_LISTBOX
            void DrawItems(const wxListBox *listbox,
                           size_t itemFirst, size_t itemLast);
#endif // wxUSE_LISTBOX
#if wxUSE_CHECKLISTBOX
            void DrawCheckItems(const wxCheckListBox *listbox,
                                size_t itemFirst, size_t itemLast);
#endif // wxUSE_CHECKLISTBOX
            void DrawButtonBorder();
    // the line must be either horizontal or vertical
            void DrawLine(wxCoord x1, wxCoord y1, wxCoord x2, wxCoord y2);
            void DrawFrame();
            void DrawBitmap(const wxBitmap& bitmap);
            void DrawBackgroundBitmap();
            void DrawScrollbar(const wxScrollBar *scrollbar, int thumbPosOld);
#if wxUSE_GAUGE
            void DrawProgressBar(const wxGauge *gauge);
#endif // wxUSE_GAUGE

    // accessors
            wxWindow *GetWindow() const { return m_window; }
            wxRenderer *GetRenderer() const { return m_renderer; }

            wxDC& GetDC() { return m_dc; }

            const wxRect& GetRect() const { return m_rect; }
            wxRect& GetRect() { return m_rect; }

    // static helpers
            static void DrawBitmap(wxDC &dc,
                                   const wxBitmap& bitmap,
                                   const wxRect& rect,
                                   int alignment = wxALIGN_CENTRE |
                                               wxALIGN_CENTRE_VERTICAL,
                                               wxStretch stretch = wxSTRETCH_NOT);

      private:

#if wxUSE_LISTBOX
    // common part of DrawItems() and DrawCheckItems()
            void DoDrawItems(const wxListBox *listbox,
                             size_t itemFirst, size_t itemLast,
                             bool isCheckLbox = false);
#endif // wxUSE_LISTBOX

            wxWindow *m_window;
            wxRenderer *m_renderer;
            wxDC& m_dc;
            wxRect m_rect;
};

//#endif // _WX_UNIV_RENDERER_H_




///////////////////////////////////////////////////////////////////////////////
// Name:        wx/univ/stdrend.h
// Purpose:     wxStdRenderer class declaration
// Author:      Vadim Zeitlin
// Created:     2006-09-18
// RCS-ID:      $Id: stdrend.h 42716 2006-10-30 12:33:25Z VS $
// Copyright:   (c) 2006 Vadim Zeitlin <vadim@wxwindows.org>
// Licence:     wxWindows licence
///////////////////////////////////////////////////////////////////////////////

//#ifndef _WX_UNIV_STDREND_H_
//#define _WX_UNIV_STDREND_H_

//#include "wx/univ/renderer.h"
//#include "wx/pen.h"

                     class WXDLLEXPORT wxColourScheme;

// ----------------------------------------------------------------------------
// wxStdRenderer: implements as much of wxRenderer API as possible generically
// ----------------------------------------------------------------------------

class wxStdRenderer : public wxRenderer
{
      public:
    // the renderer will use the given scheme, whose lifetime must be at least
    // as long as of this object itself, to choose the colours for drawing
            wxStdRenderer(const wxColourScheme *scheme);

            virtual void DrawBackground(wxDC& dc,
                                        const wxColour& col,
                                        const wxRect& rect,
                                        int flags = 0,
                                        wxWindow *window = NULL);
            virtual void DrawButtonSurface(wxDC& dc,
                                           const wxColour& col,
                                           const wxRect& rect,
                                           int flags);


            virtual void DrawFocusRect(wxDC& dc, const wxRect& rect, int flags = 0);
            virtual void DrawLabel(wxDC& dc,
                                   const wxString& label,
                                   const wxRect& rect,
                                   int flags = 0,
                                   int alignment = wxALIGN_LEFT | wxALIGN_TOP,
                                   int indexAccel = -1,
                                   wxRect *rectBounds = NULL);
            virtual void DrawButtonLabel(wxDC& dc,
                                         const wxString& label,
                                         const wxBitmap& image,
                                         const wxRect& rect,
                                         int flags = 0,
                                         int alignment = wxALIGN_LEFT | wxALIGN_TOP,
                                         int indexAccel = -1,
                                         wxRect *rectBounds = NULL);


            virtual void DrawBorder(wxDC& dc,
                                    wxBorder border,
                                    const wxRect& rect,
                                    int flags = 0,
                                    wxRect *rectIn = NULL);
            virtual void DrawTextBorder(wxDC& dc,
                                        wxBorder border,
                                        const wxRect& rect,
                                        int flags = 0,
                                        wxRect *rectIn = NULL);

            virtual void DrawHorizontalLine(wxDC& dc,
                                            wxCoord y, wxCoord x1, wxCoord x2);
            virtual void DrawVerticalLine(wxDC& dc,
                                          wxCoord x, wxCoord y1, wxCoord y2);
            virtual void DrawFrame(wxDC& dc,
                                   const wxString& label,
                                   const wxRect& rect,
                                   int flags = 0,
                                   int alignment = wxALIGN_LEFT,
                                   int indexAccel = -1);


            virtual void DrawItem(wxDC& dc,
                                  const wxString& label,
                                  const wxRect& rect,
                                  int flags = 0);
            virtual void DrawCheckItem(wxDC& dc,
                                       const wxString& label,
                                       const wxBitmap& bitmap,
                                       const wxRect& rect,
                                       int flags = 0);

            virtual void DrawCheckButton(wxDC& dc,
                                         const wxString& label,
                                         const wxBitmap& bitmap,
                                         const wxRect& rect,
                                         int flags = 0,
                                         wxAlignment align = wxALIGN_LEFT,
                                         int indexAccel = -1);
            virtual void DrawRadioButton(wxDC& dc,
                                         const wxString& label,
                                         const wxBitmap& bitmap,
                                         const wxRect& rect,
                                         int flags = 0,
                                         wxAlignment align = wxALIGN_LEFT,
                                         int indexAccel = -1);

            virtual void DrawScrollbarArrow(wxDC& dc,
                                            wxDirection dir,
                                            const wxRect& rect,
                                            int flags = 0);
            virtual void DrawScrollCorner(wxDC& dc,
                                          const wxRect& rect);

#if wxUSE_TEXTCTRL
            virtual void DrawTextLine(wxDC& dc,
                                      const wxString& text,
                                      const wxRect& rect,
                                      int selStart = -1,
                                      int selEnd = -1,
                                      int flags = 0);

            virtual void DrawLineWrapMark(wxDC& dc, const wxRect& rect);

            virtual wxRect GetTextTotalArea(const wxTextCtrl *text,
                                            const wxRect& rect) const;
            virtual wxRect GetTextClientArea(const wxTextCtrl *text,
                                             const wxRect& rect,
                                             wxCoord *extraSpaceBeyond) const;
#endif // wxUSE_TEXTCTRL

            virtual wxRect GetBorderDimensions(wxBorder border) const;

            virtual bool AreScrollbarsInsideBorder() const;

            virtual void AdjustSize(wxSize *size, const wxWindow *window);

            virtual wxCoord GetListboxItemHeight(wxCoord fontHeight);

#if wxUSE_STATUSBAR
            virtual void DrawStatusField(wxDC& dc,
                                         const wxRect& rect,
                                         const wxString& label,
                                         int flags = 0, int style = 0);

            virtual wxSize GetStatusBarBorders() const;

            virtual wxCoord GetStatusBarBorderBetweenFields() const;

            virtual wxSize GetStatusBarFieldMargins() const;
#endif // wxUSE_STATUSBAR

            virtual wxCoord GetCheckItemMargin() const { return 0; }


            virtual void DrawFrameTitleBar(wxDC& dc,
                                           const wxRect& rect,
                                           const wxString& title,
                                           const wxIcon& icon,
                                           int flags,
                                           int specialButton = 0,
                                           int specialButtonFlag = 0);
            virtual void DrawFrameBorder(wxDC& dc,
                                         const wxRect& rect,
                                         int flags);
            virtual void DrawFrameBackground(wxDC& dc,
                                             const wxRect& rect,
                                             int flags);
            virtual void DrawFrameTitle(wxDC& dc,
                                        const wxRect& rect,
                                        const wxString& title,
                                        int flags);
            virtual void DrawFrameIcon(wxDC& dc,
                                       const wxRect& rect,
                                       const wxIcon& icon,
                                       int flags);
            virtual void DrawFrameButton(wxDC& dc,
                                         wxCoord x, wxCoord y,
                                         int button,
                                         int flags = 0);

            virtual wxRect GetFrameClientArea(const wxRect& rect, int flags) const;

            virtual wxSize GetFrameTotalSize(const wxSize& clientSize, int flags) const;

            virtual wxSize GetFrameMinSize(int flags) const;

            virtual wxSize GetFrameIconSize() const;

            virtual int HitTestFrame(const wxRect& rect,
                                     const wxPoint& pt,
                                     int flags = 0) const;
      protected:
    // various constants
            enum ArrowDirection
            {
                  Arrow_Left,
                  Arrow_Right,
                  Arrow_Up,
                  Arrow_Down,
                  Arrow_Max
            };

            enum ArrowStyle
            {
                  Arrow_Normal,
                  Arrow_Disabled,
                  Arrow_Pressed,
                  Arrow_Inverted,
                  Arrow_InvertedDisabled,
                  Arrow_StateMax
            };

            enum FrameButtonType
            {
                  FrameButton_Close,
                  FrameButton_Minimize,
                  FrameButton_Maximize,
                  FrameButton_Restore,
                  FrameButton_Help,
                  FrameButton_Max
            };

            enum IndicatorType
            {
                  IndicatorType_Check,
                  IndicatorType_Radio,
                  IndicatorType_MaxCtrl,
                  IndicatorType_Menu = IndicatorType_MaxCtrl,
                  IndicatorType_Max
            };

            enum IndicatorState
            {
                  IndicatorState_Normal,
                  IndicatorState_Pressed, // this one is for check/radioboxes
                  IndicatorState_Disabled,
                  IndicatorState_MaxCtrl,

        // the rest of the states are valid for menu items only
                  IndicatorState_Selected = IndicatorState_Pressed,
                  IndicatorState_SelectedDisabled = IndicatorState_MaxCtrl,
                  IndicatorState_MaxMenu
            };

            enum IndicatorStatus
            {
                  IndicatorStatus_Checked,
                  IndicatorStatus_Unchecked,
                  IndicatorStatus_Undetermined,
                  IndicatorStatus_Max
            };

    // translate the appropriate bits in flags to the above enum elements
            static void GetIndicatorsFromFlags(int flags,
                                               IndicatorState& state,
                                               IndicatorStatus& status);

    // translate wxDirection to ArrowDirection
            static ArrowDirection GetArrowDirection(wxDirection dir);


    // fill the rectangle with a brush of given colour (must be valid)
            void DrawSolidRect(wxDC& dc, const wxColour& col, const wxRect& rect);


    // all the functions in this section adjust the rect parameter to
    // correspond to the interiour of the drawn area

        // draw complete rectangle
            void DrawRect(wxDC& dc, wxRect *rect, const wxPen& pen);

        // draw the rectange using the first pen for the left and top sides
        // and the second one for the bottom and right ones
            void DrawShadedRect(wxDC& dc, wxRect *rect,
                                const wxPen& pen1, const wxPen& pen2);

        // border drawing routines, may be overridden in the derived class
            virtual void DrawRaisedBorder(wxDC& dc, wxRect *rect);
            virtual void DrawSunkenBorder(wxDC& dc, wxRect *rect);
            virtual void DrawAntiSunkenBorder(wxDC& dc, wxRect *rect);
            virtual void DrawBoxBorder(wxDC& dc, wxRect *rect);
            virtual void DrawStaticBorder(wxDC& dc, wxRect *rect);
            virtual void DrawExtraBorder(wxDC& dc, wxRect *rect);


    // draw the frame with non-empty label inside the given rectText
            virtual void DrawFrameWithLabel(wxDC& dc,
                                            const wxString& label,
                                            const wxRect& rectFrame,
                                            const wxRect& rectText,
                                            int flags,
                                            int alignment,
                                            int indexAccel);

    // draw the (static box) frame without the part corresponding to rectLabel
            void DrawFrameWithoutLabel(wxDC& dc,
                                       const wxRect& rectFrame,
                                       const wxRect& rectLabel);


    // draw the bitmap for a check item (which is by default the same as check
    // box one but may be different)
            virtual void DrawCheckItemBitmap(wxDC& dc,
                                             const wxBitmap& bitmap,
                                             const wxRect& rect,
                                             int flags);

    // common routine for drawing check and radio buttons
            void DrawCheckOrRadioButton(wxDC& dc,
                                        const wxString& label,
                                        const wxBitmap& bitmap,
                                        const wxRect& rect,
                                        int flags,
                                        wxAlignment align,
                                        int indexAccel);

    // return the check/radio bitmap for the given flags
            virtual wxBitmap GetRadioBitmap(int flags) = 0;
            virtual wxBitmap GetCheckBitmap(int flags) = 0;

    // return the frame icon bitmap
            virtual wxBitmap GetFrameButtonBitmap(FrameButtonType type) = 0;

    // get the width of either normal or resizeable frame border depending on
    // whether flags contains wxTOPLEVEL_RESIZEABLE bit
            //
    // notice that these methods only make sense with standard border drawing
    // code which uses the borders of the same width on all sides, this is why
    // they are only present here and not in wxRenderer itself
            virtual int GetFrameBorderWidth(int flags) const;

#if wxUSE_TEXTCTRL
    // return the width of the border around the text area in the text control
            virtual int GetTextBorderWidth(const wxTextCtrl *text) const;
#endif // wxUSE_TEXTCTRL

    // GDI objects we often use
            wxPen m_penBlack,
            m_penDarkGrey,
            m_penLightGrey,
            m_penHighlight;

            wxFont m_titlebarFont;

    // the colours we use, they never change currently so we don't have to ever
    // update m_penXXX objects above
            const wxColourScheme * const m_scheme;

            DECLARE_NO_COPY_CLASS(wxStdRenderer)
};

//#endif // _WX_UNIV_STDREND_H_


///////////////////////////////////////////////////////////////////////////////
// Name:        wx/univ/scrolbar.h
// Purpose:     wxScrollBar for wxUniversal
// Author:      Vadim Zeitlin
// Modified by:
// Created:     20.08.00
// RCS-ID:      $Id: scrolbar.h 42716 2006-10-30 12:33:25Z VS $
// Copyright:   (c) 2000 SciTech Software, Inc. (www.scitechsoft.com)
// Licence:     wxWindows licence
///////////////////////////////////////////////////////////////////////////////

//#ifndef _WX_UNIV_SCROLBAR_H_
//#define _WX_UNIV_SCROLBAR_H_

class WXDLLEXPORT wxScrollTimer;

//#include "wx/univ/scrarrow.h"
//#include "wx/renderer.h"

// ----------------------------------------------------------------------------
// the actions supported by this control
// ----------------------------------------------------------------------------

// scroll the bar
#define wxACTION_SCROLL_START       _T("start")     // to the beginning
#define wxACTION_SCROLL_END         _T("end")       // to the end
#define wxACTION_SCROLL_LINE_UP     _T("lineup")    // one line up/left
#define wxACTION_SCROLL_PAGE_UP     _T("pageup")    // one page up/left
#define wxACTION_SCROLL_LINE_DOWN   _T("linedown")  // one line down/right
#define wxACTION_SCROLL_PAGE_DOWN   _T("pagedown")  // one page down/right

// the scrollbar thumb may be dragged
#define wxACTION_SCROLL_THUMB_DRAG      _T("thumbdrag")
#define wxACTION_SCROLL_THUMB_MOVE      _T("thumbmove")
#define wxACTION_SCROLL_THUMB_RELEASE   _T("thumbrelease")


// ----------------------------------------------------------------------------
// Standard scrollbar input handler which can be used as a base class
// ----------------------------------------------------------------------------

class WXDLLEXPORT wxStdScrollBarInputHandler : public wxStdInputHandler
{
      public:
    // constructor takes a renderer (used for scrollbar hit testing) and the
    // base handler to which all unhandled events are forwarded
            wxStdScrollBarInputHandler(wxRenderer *renderer,
                                       wxInputHandler *inphand);

            virtual bool HandleKey(wxInputConsumer *consumer,
                                   const wxKeyEvent& event,
                                   bool pressed);
            virtual bool HandleMouse(wxInputConsumer *consumer,
                                     const wxMouseEvent& event);
            virtual bool HandleMouseMove(wxInputConsumer *consumer, const wxMouseEvent& event);

            virtual ~wxStdScrollBarInputHandler();

    // this method is called by wxScrollBarTimer only and may be overridden
            //
    // return true to continue scrolling, false to stop the timer
            virtual bool OnScrollTimer(wxScrollBar *scrollbar,
                                       const wxControlAction& action);

      protected:
    // return true if the mouse button can be used to activate scrollbar, false
    // if not (any button under GTK+ unlike left button only which is default)
            virtual bool IsAllowedButton(int button) const
            { return button == wxMOUSE_BTN_LEFT; }

    // set or clear the specified flag on the scrollbar element corresponding
    // to m_htLast
            void SetElementState(wxScrollBar *scrollbar, int flag, bool doIt);

    // [un]highlight the scrollbar element corresponding to m_htLast
            virtual void Highlight(wxScrollBar *scrollbar, bool doIt)
            { SetElementState(scrollbar, wxCONTROL_CURRENT, doIt); }

    // [un]press the scrollbar element corresponding to m_htLast
            virtual void Press(wxScrollBar *scrollbar, bool doIt)
            { SetElementState(scrollbar, wxCONTROL_PRESSED, doIt); }

    // stop scrolling because we reached the end point
            void StopScrolling(wxScrollBar *scrollbar);

    // get the mouse coordinates in the scrollbar direction from the event
            wxCoord GetMouseCoord(const wxScrollBar *scrollbar,
                                  const wxMouseEvent& event) const;

    // generate a "thumb move" action for this mouse event
            void HandleThumbMove(wxScrollBar *scrollbar, const wxMouseEvent& event);


    // the window (scrollbar) which has capture or NULL and the flag telling if
    // the mouse is inside the element which captured it or not
            wxWindow *m_winCapture;
            bool      m_winHasMouse;
            int       m_btnCapture;  // the mouse button which has captured mouse

    // the position where we started scrolling by page
            wxPoint m_ptStartScrolling;

    // one of wxHT_SCROLLBAR_XXX value: where has the mouse been last time?
            wxHitTest m_htLast;

    // the renderer (we use it only for hit testing)
            wxRenderer *m_renderer;

    // the offset of the top/left of the scrollbar relative to the mouse to
    // keep during the thumb drag
            int m_ofsMouse;

    // the timer for generating scroll events when the mouse stays pressed on
    // a scrollbar
            wxScrollTimer *m_timerScroll;
};

//#endif // _WX_UNIV_SCROLBAR_H_













#endif // _UNIPARTS_H_
