/*
Copyright (C) 2011, 2012, 2013 Craciun Dan <craciun.dan@tuxarena.com>

This file is part of NeonView.

NeonView is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 2 of the License, or
(at your option) any later version.

NeonView is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with NeonView.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "header.h"

void init_keys (TM *tm) {
  strcpy(tm->cfg->last_dir, ".");
  strcpy(tm->cfg->start_dir, ".");
  tm->cfg->window_width = 840;
  tm->cfg->window_height = 480;
  tm->cfg->window_min_width = 520;
  tm->cfg->window_min_height = 320;
  strcpy(tm->cfg->start_maximized, "no");
  strcpy(tm->cfg->show_toolbar, "yes");
  strcpy(tm->cfg->show_status, "yes");
  strcpy(tm->cfg->show_left_panel, "yes");
  strcpy(tm->cfg->show_right_panel, "yes");
  strcpy(tm->cfg->remember_fit, "no");
  strcpy(tm->cfg->lock_fit, "no");
  strcpy(tm->cfg->remember_zoom, "no");
  strcpy(tm->cfg->remember_window_size, "yes");
  strcpy(tm->cfg->window_title, "NeonView");
  strcpy(tm->cfg->gtk_win_pos, "GTK_WIN_POS_CENTER");
}

void read_config (TM *tm) {
  int rv;

  rv = access(tm->env->config_dir, F_OK);
  if (rv) {
    fprintf(stdout, "[NeonView] Directory %s not found.\n", tm->env->config_dir);
    mk_config_dir(tm);
    mk_config_file(tm);
    return;
  }

  rv = access(tm->env->config_file, F_OK);
  if (rv) {
    fprintf(stdout, "[NeonView] Config file %s not found.\n", tm->env->config_file);
    mk_config_file(tm);
    return;
  }

  fprintf(stdout, "[NeonView] Config file %s found.\n", tm->env->config_file);

  FILE *file;

  file = fopen(tm->env->config_file, "rt");
  if (!file) {
    fprintf(stdout, "[NeonView] WARNING: Could not open config file %s for reading. Check permissions and disk space.\n", tm->env->config_file);
    fprintf(stdout, "[NeonView] WARNING: NeonView will start using the default values, but it will probably not be able to save any configuration changes that you make.\n");
    return;
  }

  char c;
  int i, j;
  int iskey, isval;
  char key[65], val[PATH_LENGTH + 1];
  int skip;

  i = 0;
  j = 0;
  iskey = 1;
  isval = 0;
  skip = 0;

  while ((c = fgetc(file)) != EOF) {
    if (skip == 0) {
      if (c == '#') {
	skip = 1;
      }
      if (c == '\n') {
	key[i] = '\0';
	val[j] = '\0';
	if (strlen(key) > 0 && strlen(val) > 0) {
	  strip(val);
	  assign_keys(tm, key, val);
	}
	if (strlen(key) > 0 && j == 0) {
	  fprintf(stdout, "[NeonView] WARNING: Invalid config line: %s\n", key);
	  fprintf(stdout, "[NeonView] WARNING: Ignore.\n");
	}
	if (strlen(key) > 64) {
	  fprintf(stdout, "[NeonView] WARNING: Invalid config key (maximum 64 characters): %s\n", key);
	  fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
	}
	i = 0;
	j = 0;
	iskey = 1;
	isval = 0;
      }
      if (c != '#' && c != '\n') {
	if (c != '=') {
	  if (iskey == 1) {
	    key[i] = c;
	    i++;
	  }
	  if (isval == 1) {
	    val[j] = c;
	    j++;
	  }
	} else {
	  iskey = 0;
	  isval = 1;
	}
      }
    } else {
      if (c == '\n') {
	skip = 0;
	key[i] = '\0';
	val[j] = '\0';
	i = 0;
	j = 0;
	iskey = 1;
	isval = 0;
	if (strlen(key) > 0 && strlen(val) > 0) {
	  strip(val);
	  assign_keys(tm, key, val);
	}
      }
    }
  }

  fclose(file);

  fprintf(stdout, "[NeonView] Finished reading the %s config file.\n", tm->env->config_file);
}

int mk_config_dir (TM *tm) {
  int rv;

  rv = mkdir(tm->env->config_dir, 0777);
  if (rv) {
    fprintf(stdout, "[NeonView] WARNING: Could not create directory %s. Check permissions and disk space.\n", tm->env->config_dir);
    fprintf(stdout, "[NeonView] WARNING: NeonView will start using the default values, but it will probably not be able to save any configuration changes that you make.\n");
    return 1;
  } else {
    fprintf(stdout, "[NeonView] Directory %s created.\n", tm->env->config_dir);
    return 0;
  }
  return -1;
}

int mk_config_file (TM *tm) {
  FILE *file;

  file = fopen(tm->env->config_file, "wt");
  if (!file) {
    fprintf(stdout, "[NeonView] WARNING: Could not open config file %s for writing. Check permissions and disk space.\n", tm->env->config_file);
    fprintf(stdout, "[NeonView] WARNING: NeonView will start using the default values, but it will probably not be able to save any configuration changes that you make.\n");
    return 1;
  }

  fprintf(file, "################################################################################\n");
  fprintf(file, "# NeonView Config File\n");
  fprintf(file, "# Changes done in the program are reflected here. You can modify these values,\n");
  fprintf(file, "# however make sure to read the manual for instructions. Everything after a '#'\n");
  fprintf(file, "# is a comment, double-quotes are not necessary. NeonView checks this file at\n");
  fprintf(file, "# start-up, if something is wrong it will just load the default values.\n");
  fprintf(file, "################################################################################\n");
  fprintf(file, "last_dir=\".\"\n");
  fprintf(file, "start_dir=\".\"\n");
  fprintf(file, "window_width=\"840\"\n");
  fprintf(file, "window_height=\"480\"\n");
  fprintf(file, "window_min_width=\"520\"\n");
  fprintf(file, "window_min_height=\"320\"\n");
  fprintf(file, "start_maximized=\"no\"\n");
  fprintf(file, "show_toolbar=\"yes\"\n");
  fprintf(file, "show_status=\"yes\"\n");
  fprintf(file, "show_left_panel=\"yes\"\n");
  fprintf(file, "show_right_panel=\"yes\"\n");
  fprintf(file, "remember_fit=\"no\"\n");
  fprintf(file, "lock_fit=\"yes\"\n");
  fprintf(file, "remember_zoom=\"no\"\n");
  fprintf(file, "remember_window_size=\"yes\"\n");
  fprintf(file, "window_title=\"NeonView\"\n");
  fprintf(file, "gtk_win_pos=\"GTK_WIN_POS_CENTER\"\n");

  fclose(file);

  fprintf(stdout, "[NeonView] Default config file %s written successfully.\n", tm->env->config_file);

  return 0;
}

void print_config (TM *tm) {
  fprintf(stdout, "--------------------------------------------------------------------------------\n");
  fprintf(stdout, "last_dir=%s\n", tm->cfg->last_dir);
  fprintf(stdout, "start_dir=%s\n", tm->cfg->start_dir);
  fprintf(stdout, "window_width=%d\n", tm->cfg->window_width);
  fprintf(stdout, "window_height=%d\n", tm->cfg->window_height);
  fprintf(stdout, "window_min_width=%d\n", tm->cfg->window_min_width);
  fprintf(stdout, "window_min_height=%d\n", tm->cfg->window_min_height);
  fprintf(stdout, "start_maximized=%s\n", tm->cfg->start_maximized);
  fprintf(stdout, "show_toolbar=%s\n", tm->cfg->show_toolbar);
  fprintf(stdout, "show_status=%s\n", tm->cfg->show_status);
  fprintf(stdout, "show_left_panel=%s\n", tm->cfg->show_left_panel);
  fprintf(stdout, "show_right_panel=%s\n", tm->cfg->show_right_panel);
  fprintf(stdout, "remember_fit=%s\n", tm->cfg->remember_fit);
  fprintf(stdout, "lock_fit=%s\n", tm->cfg->lock_fit);
  fprintf(stdout, "remember_zoom=%s\n", tm->cfg->remember_zoom);
  fprintf(stdout, "remember_window_size=%s\n", tm->cfg->remember_window_size);
  fprintf(stdout, "window_title=%s\n", tm->cfg->window_title);
  fprintf(stdout, "gtk_win_pos=%s\n", tm->cfg->gtk_win_pos);
  fprintf(stdout, "--------------------------------------------------------------------------------\n");
}

void assign_keys (TM *tm, char *key, char *val) {
  int n;

  if (strcmp(key, "last_dir") == 0) {
    if (strlen(val) > PATH_LENGTH + 1) {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (exceeds maximum path length): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "start_dir") == 0) {
    if (strlen(val) > PATH_LENGTH + 1) {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (exceeds maximum path length): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "window_width") == 0) {
    if (all_digits(val) == 0) {
      sscanf(val, "%d", &n);
      if (n >= 0 && n <= 2560 * 2) {
	tm->cfg->window_width = n;
      } else {
	fprintf(stdout, "[NeonView] WARNING: Invalid config value (out of range, min 0, max 5120): %s=%s\n", key, val);
	fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
      }
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (not an integer): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "window_height") == 0) {
    if (all_digits(val) == 0) {
      sscanf(val, "%d", &n);
      if (n >= 0 && n <= 1440 * 2) {
	tm->cfg->window_height = n;
      } else {
	fprintf(stdout, "[NeonView] WARNING: Invalid config value (out of range, min 0, max 2880): %s=%s\n", key, val);
	fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
      }
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (not an integer): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "window_min_width") == 0) {
    if (all_digits(val) == 0) {
      sscanf(val, "%d", &n);
      if (n >= 0 && n <= 2560 * 2) {
	tm->cfg->window_min_width = n;
      } else {
	fprintf(stdout, "[NeonView] WARNING: Invalid config value (out of range, min 0, max 5120): %s=%s\n", key, val);
	fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
      }
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (not an integer): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "window_min_height") == 0) {
    if (all_digits(val) == 0) {
      sscanf(val, "%d", &n);
      if (n >= 0 && n <= 1440 * 2) {
	tm->cfg->window_min_height = n;
      } else {
	fprintf(stdout, "[NeonView] WARNING: Invalid config value (out of range, min 0, max 2880): %s=%s\n", key, val);
	fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
      }
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (not an integer): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "start_maximized") == 0) {
    if (strcmp(val, "yes") == 0 || strcmp(val, "no") == 0) {
      strcpy(tm->cfg->start_maximized, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (should be \"yes\" or \"no\"): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "show_toolbar") == 0) {
    if (strcmp(val, "yes") == 0 || strcmp(val, "no") == 0) {
      strcpy(tm->cfg->show_toolbar, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (should be \"yes\" or \"no\"): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "show_status") == 0) {
    if (strcmp(val, "yes") == 0 || strcmp(val, "no") == 0) {
      strcpy(tm->cfg->show_status, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (should be \"yes\" or \"no\"): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "show_left_panel") == 0) {
    if (strcmp(val, "yes") == 0 || strcmp(val, "no") == 0) {
      strcpy(tm->cfg->show_left_panel, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (should be \"yes\" or \"no\"): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "show_right_panel") == 0) {
    if (strcmp(val, "yes") == 0 || strcmp(val, "no") == 0) {
      strcpy(tm->cfg->show_right_panel, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (should be \"yes\" or \"no\"): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "remember_fit") == 0) {
    if (strcmp(val, "yes") == 0 || strcmp(val, "no") == 0) {
      strcpy(tm->cfg->remember_fit, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (should be \"yes\" or \"no\"): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "lock_fit") == 0) {
    if (strcmp(val, "yes") == 0 || strcmp(val, "no") == 0) {
      strcpy(tm->cfg->lock_fit, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (should be \"yes\" or \"no\"): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "remember_zoom") == 0) {
    if (strcmp(val, "yes") == 0 || strcmp(val, "no") == 0) {
      strcpy(tm->cfg->remember_zoom, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (should be \"yes\" or \"no\"): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "remember_window_size") == 0) {
    if (strcmp(val, "yes") == 0 || strcmp(val, "no") == 0) {
      strcpy(tm->cfg->remember_window_size, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (should be \"yes\" or \"no\"): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "window_title") == 0) {
    if (strlen(val) >= 0 && strlen(val) <= 80) {
      strcpy(tm->cfg->window_title, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value (maximum 80 characters): %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "gtk_win_pos") == 0) {
    if (strcmp(val, "GTK_WIN_POS_NONE") == 0 || strcmp(val, "GTK_WIN_POS_CENTER") == 0 ||
	strcmp(val, "GTK_WIN_POS_MOUSE") == 0 || strcmp(val, "GTK_WIN_POS_CENTER_ALWAYS") == 0 ||
	strcmp(val, "GTK_WIN_POS_CENTER_ON_PARENT") == 0) {
      strcpy(tm->cfg->gtk_win_pos, val);
    } else {
      fprintf(stdout, "[NeonView] WARNING: Invalid config value: %s=%s\n", key, val);
      fprintf(stdout, "[NeonView] WARNING: Using default value.\n");
    }
  }
  if (strcmp(key, "last_dir") != 0 && strcmp(key, "start_dir") != 0 && strcmp(key, "window_width") != 0 &&
      strcmp(key, "window_height") != 0 && strcmp(key, "start_maximized") != 0 && strcmp(key, "show_toolbar") != 0 &&
      strcmp(key, "show_status") != 0 && strcmp(key, "show_left_panel") != 0 && strcmp(key, "remember_fit") != 0 &&
      strcmp(key, "lock_fit") != 0 && strcmp(key, "remember_zoom") != 0 && strcmp(key, "remember_window_size") != 0 &&
      strcmp(key, "window_title") != 0 && strcmp(key, "gtk_win_pos") != 0 && strcmp(key, "show_right_panel") != 0 &&
      strcmp(key, "window_min_width") != 0 && strcmp(key, "window_min_height") != 0) {
    fprintf(stdout, "[NeonView] WARNING: Invalid config key: %s=%s\n", key, val);
    fprintf(stdout, "[NeonView] WARNING: Ignored.\n");
  }
}

void write_config (TM *tm) {
  FILE *file;

  file = fopen(tm->env->config_file, "wt");
  if (!file) {
    fprintf(stdout, "[NeonView] WARNING: Could not open config file %s for writing. Check permissions and disk space.\n", tm->env->config_file);
    fprintf(stdout, "[NeonView] WARNING: NeonView could not save any configuration changes that you made.\n");
    return;
  }

  fprintf(file, "################################################################################\n");
  fprintf(file, "# NeonView Config File\n");
  fprintf(file, "# Changes done in the program are reflected here. You can modify these values,\n");
  fprintf(file, "# however make sure to read the manual for instructions. Everything after a '#'\n");
  fprintf(file, "# is a comment, double-quotes are not necessary. NeonView checks this file at\n");
  fprintf(file, "# start-up, if something is wrong it will just load the default values.\n");
  fprintf(file, "################################################################################\n");
  fprintf(file, "last_dir=\"%s\"\n", tm->cfg->last_dir);
  fprintf(file, "start_dir=\"%s\"\n", tm->cfg->start_dir);
  fprintf(file, "window_width=\"%d\"\n", tm->cfg->window_width);
  fprintf(file, "window_height=\"%d\"\n", tm->cfg->window_height);
  fprintf(file, "window_min_width=\"%d\"\n", tm->cfg->window_min_width);
  fprintf(file, "window_min_height=\"%d\"\n", tm->cfg->window_min_height);
  fprintf(file, "start_maximized=\"%s\"\n", tm->cfg->start_maximized);
  fprintf(file, "show_toolbar=\"%s\"\n", tm->cfg->show_toolbar);
  fprintf(file, "show_status=\"%s\"\n", tm->cfg->show_status);
  fprintf(file, "show_left_panel=\"%s\"\n", tm->cfg->show_left_panel);
  fprintf(file, "show_right_panel=\"%s\"\n", tm->cfg->show_right_panel);
  fprintf(file, "remember_fit=\"%s\"\n", tm->cfg->remember_fit);
  fprintf(file, "lock_fit=\"%s\"\n", tm->cfg->lock_fit);
  fprintf(file, "remember_zoom=\"%s\"\n", tm->cfg->remember_zoom);
  fprintf(file, "remember_window_size=\"%s\"\n", tm->cfg->remember_window_size);
  fprintf(file, "window_title=\"%s\"\n", tm->cfg->window_title);
  fprintf(file, "gtk_win_pos=\"%s\"\n", tm->cfg->gtk_win_pos);

  fclose(file);

  fprintf(stdout, "[NeonView] Saved settings to %s config file.\n", tm->env->config_file);
}
