# -*- coding: utf-8 -*-

#   This file is part of Moonshiner.
#   Copyright 2008, 2009 Benjamin Dumke
#
#   Moonshiner is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   Moonshiner is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with Moonshiner (see the file LICENSE).  If not, see
#   <http://www.gnu.org/licenses/>.

from moonshiner_config import get_settings


# Unlike what we do here, the ps2pdf documentation gives a /Blend value of 1 for the highest quality
jpeg_qualities = { "0.15" : { "QFactor" : 0.15 , "Blend" : 0 , "HSamples" : [1,1,1,1], "VSamples" : [1,1,1,1] } ,
                   "0.25" : { "QFactor" : 0.25 , "Blend" : 0 , "HSamples" : [1,1,1,1], "VSamples" : [1,1,1,1] } ,
                   "0.4" : { "QFactor" : 0.4 , "Blend" : 1 , "HSamples" : [1,1,1,1], "VSamples" : [1,1,1,1] } ,
                   "0.76" : { "QFactor" : 0.76 , "Blend" : 1 , "HSamples" : [2,1,1,2], "VSamples" : [2,1,1,2] } ,
                   "0.9" : { "QFactor" : 0.9 , "Blend" : 1 , "HSamples" : [2,1,1,2], "VSamples" : [2,1,1,2] } ,
                   "1.2" : { "QFactor" : 1.2 , "Blend" : 1 , "HSamples" : [2,1,1,2], "VSamples" : [2,1,1,2] } ,
                   "1.5" : { "QFactor" : 1.5 , "Blend" : 1 , "HSamples" : [2,1,1,2], "VSamples" : [2,1,1,2] } }



def preset(param,default,screen=None,ebook=None,printer=None,prepress=None):
    """Sets the preset dictionaries."""

    def dif(x):
        """Default If None"""
        return x if x is not None else default
        
    global param_presets

    param_presets["default"][param]=default
    param_presets["screen"][param]=dif(screen)
    param_presets["ebook"][param]=dif(ebook)
    param_presets["printer"][param]=dif(printer)
    param_presets["prepress"][param]=dif(prepress)
    
param_presets={"default":{}, "screen":{}, "ebook":{}, "printer":{}, "prepress":{}}

preset("_BasedOn","default","screen","ebook","printer","prepress")
preset("_ColorImageJpegQuality","0.9","0.76","0.76","0.4","0.15")
preset("_CompressFonts",True)
preset("_Encrypt",False)
preset("_GrayImageJpegQuality","0.9","0.9","0.76","0.4","0.15")
preset("_OwnerPassword","password")
preset("_Restrictions",4+8+16+32+256+512+1024+2048)
preset("_UserPassword","")
preset("AutoFilterColorImages",True)
preset("AutoFilterGrayImages",True)
preset("AutoRotatePages","PageByPage","PageByPage","All","None","None")
preset("ASCII85EncodePages",False)
preset("ColorConversionStrategy","LeaveColorUnchanged","sRGB","sRGB","UseDeviceIndependentColor","LeaveColorUnchanged")
preset("ColorImageDownsampleThreshold",1.5)
preset("ColorImageFilter","Auto") # for obvious reasons, this is different from GS's setting
preset("ColorImageResolution",72,72,150,300,300)
preset("CompatibilityLevel",14,13,14,14,14)
preset("CompressPages",True)
preset("DefaultRenderingIntent","Default")
preset("DownsampleColorImages",False,True,True,False,False)
preset("DownsampleGrayImages",False,True,True,False,False)
preset("DownsampleMonoImages",False,True,True,False,False)
preset("EmbedAllFonts",True,False,True,True,True)
preset("EncodeColorImages",True)
preset("EncodeGrayImages",True)
preset("EncodeMonoImages",True)
preset("GrayImageDownsampleThreshold",1.5)
preset("GrayImageFilter","Auto") # for obvious reasons, this is different from GS's setting
preset("GrayImageResolution",72,72,150,300,300)
preset("LockDistillerParams",False)
preset("MaxSubsetPCT",100)
preset("MonoImageDownsampleThreshold",1.5)
preset("MonoImageFilter","CCITTFaxEncode")
preset("MonoImageResolution",300,300,300,1200,1200)
preset("PreserveHalftoneInfo",False)
preset("PreserveOverprintSettings",False,False,False,True,True)
preset("SubsetFonts",True)
preset("UCRandBGInfo","Remove","Remove","Remove","Preserve","Preserve")

def params_to_ui(params,notebook):
    """Takes a dict of Moonshiner parameters and sets the ui notebook accordingly."""
    
    notebook.tab_misc.opts_row_basedon[1].set(params["_BasedOn"])
    notebook.tab_images.opt__ColorImageJpegQuality[1].set(params["_ColorImageJpegQuality"])
    notebook.tab_fonts.opt__CompressFonts.set_active(params["_CompressFonts"])
    notebook.tab_encryption.opt__Encrypt.set_active(params["_Encrypt"])
    notebook.tab_images.opt__GrayImageJpegQuality[1].set(params["_GrayImageJpegQuality"])
    notebook.tab_encryption.opt_Passwords[0][1].set_text(params["_OwnerPassword"])
    
    notebook.tab_encryption.opt_Restrictions13[0].set_active(params["_Restrictions"] & 2048+4 > 4)
    notebook.tab_encryption.opt_Restrictions13[1][1].set(params["_Restrictions"] & 8+32)
    notebook.tab_encryption.opt_Restrictions13[2].set_active(params["_Restrictions"] & 16)

    notebook.tab_encryption.opt_Restrictions14[0][1].set(params["_Restrictions"] & 4+2048)
    notebook.tab_encryption.opt_Restrictions14[1][1].set(params["_Restrictions"] & 8+32+256)
    notebook.tab_encryption.opt_Restrictions14[2][1].set(params["_Restrictions"] & 16+512)
    notebook.tab_encryption.opt_Restrictions14[3].set_active(params["_Restrictions"] & 1024)

    notebook.tab_encryption.opt_Passwords[1][1].set_text(params["_UserPassword"])
    if params["AutoFilterColorImages"]:
        notebook.tab_images.opt_ColorImageCompression[1].set("Auto")
    else:
        notebook.tab_images.opt_ColorImageCompression[1].set(params["ColorImageFilter"])
    if params["AutoFilterGrayImages"]:
        notebook.tab_images.opt_GrayImageCompression[1].set("Auto")
    else:
        notebook.tab_images.opt_GrayImageCompression[1].set(params["GrayImageFilter"])
    notebook.tab_misc.opt_ASCII85EncodePages.set_active(params["ASCII85EncodePages"])
    notebook.tab_misc.opt_AutoRotatePages[1].set(params["AutoRotatePages"])
    notebook.tab_color.opt_ColorConversionStrategy[1].set(params["ColorConversionStrategy"])
    notebook.tab_images.opt_ColorImageDownsampling[4].set(int(params["ColorImageDownsampleThreshold"]*100))
    notebook.tab_images.opt_ColorImageDownsampling[2].set(params["ColorImageResolution"])
    notebook.tab_misc.opt_CompatibilityLevel[1].set(params["CompatibilityLevel"])
    notebook.tab_misc.opt_CompressPages.set_active(params["CompressPages"])
    notebook.tab_color.opt_DefaultRenderingIntent[1].set(params["DefaultRenderingIntent"])
    notebook.tab_images.opt_ColorImageDownsampling[0].set_active(params["DownsampleColorImages"])
    notebook.tab_images.opt_GrayImageDownsampling[0].set_active(params["DownsampleGrayImages"])
    notebook.tab_images.opt_MonoImageDownsampling[0].set_active(params["DownsampleMonoImages"])
    notebook.tab_fonts.opt_EmbedAllFonts.set_active(params["EmbedAllFonts"])
    if not params["EncodeColorImages"]:
        notebook.tab_images.opt_ColorImageCompression[1].set("None")
    if not params["EncodeGrayImages"]:
        notebook.tab_images.opt_GrayImageCompression[1].set("None")
    notebook.tab_images.opt_GrayImageDownsampling[4].set(int(params["GrayImageDownsampleThreshold"]*100))
    notebook.tab_images.opt_GrayImageDownsampling[2].set(params["GrayImageResolution"])
    notebook.tab_misc.opt_LockDistillerParams.set_active(params["LockDistillerParams"])
    notebook.tab_fonts.opt_FontSubsetting[2].set(params["MaxSubsetPCT"])
    notebook.tab_images.opt_MonoImageDownsampling[4].set(int(params["MonoImageDownsampleThreshold"]*100))
    notebook.tab_images.opt_MonoImageCompression[1].set(params["MonoImageFilter"])
    if not params["EncodeMonoImages"]: #has to be after /MonoImageFilter
        notebook.tab_images.opt_MonoImageCompression[1].set("None")
    notebook.tab_images.opt_MonoImageDownsampling[2].set(params["MonoImageResolution"])
    notebook.tab_color.opt_PreserveHalftoneInfo.set_active(params["PreserveHalftoneInfo"])
    notebook.tab_color.opt_PreserveOverprintSettings.set_active(params["PreserveOverprintSettings"])
    notebook.tab_fonts.opt_FontSubsetting[0].set_active(params["SubsetFonts"])
    notebook.tab_color.opt_UCRandBGInfo.set_active(params["UCRandBGInfo"]=="Preserve")
    
    
def ui_to_params(notebook,restrictions_version=None):
    """Returns a dict of Moonshiner parameters with the settings as in the ui notebook."""
    
    result=dict(param_presets[notebook.tab_misc.opt__BasedOn])

    result["_ColorImageJpegQuality"] = notebook.tab_images.opt__ColorImageJpegQuality[1].get()
    result["_CompressFonts"] = notebook.tab_fonts.opt__CompressFonts.get_active()
    result["_Encrypt"] = notebook.tab_encryption.opt__Encrypt.get_active()
    result["_GrayImageJpegQuality"] = notebook.tab_images.opt__GrayImageJpegQuality[1].get()
    result["_OwnerPassword"] = notebook.tab_encryption.opt_Passwords[0][1].get_text()
    result["_UserPassword"] = notebook.tab_encryption.opt_Passwords[1][1].get_text()
    result["ASCII85EncodePages"] = notebook.tab_misc.opt_ASCII85EncodePages.get_active()
    result["AutoFilterColorImages"] = ( notebook.tab_images.opt_ColorImageCompression[1].get() == "Auto" )
    result["AutoFilterGrayImages"] = ( notebook.tab_images.opt_GrayImageCompression[1].get() == "Auto" )
    result["AutoRotatePages"] = notebook.tab_misc.opt_AutoRotatePages[1].get()
    result["ColorConversionStrategy"] = notebook.tab_color.opt_ColorConversionStrategy[1].get()
    result["ColorImageDownsampleThreshold"] = notebook.tab_images.opt_ColorImageDownsampling[4].get()/100.0
    result["ColorImageFilter"] = notebook.tab_images.opt_ColorImageCompression[1].get()
    result["ColorImageResolution"] = notebook.tab_images.opt_ColorImageDownsampling[2].get()
    result["CompatibilityLevel"]=notebook.tab_misc.opt_CompatibilityLevel[1].get()
    result["CompressPages"] = notebook.tab_misc.opt_CompressPages.get_active()
    result["DefaultRenderingIntent"] = notebook.tab_color.opt_DefaultRenderingIntent[1].get()
    result["DownsampleColorImages"] = notebook.tab_images.opt_ColorImageDownsampling[0].get_active()
    result["DownsampleGrayImages"] = notebook.tab_images.opt_GrayImageDownsampling[0].get_active()
    result["DownsampleMonoImages"] = notebook.tab_images.opt_MonoImageDownsampling[0].get_active()
    result["EmbedAllFonts"] = notebook.tab_fonts.opt_EmbedAllFonts.get_active()
    result["EncodeColorImages"] = ( notebook.tab_images.opt_ColorImageCompression[1].get() != "None" )
    result["EncodeGrayImages"] = ( notebook.tab_images.opt_GrayImageCompression[1].get() != "None" )
    result["EncodeMonoImages"] = ( notebook.tab_images.opt_MonoImageCompression[1].get() != "None" )
    result["GrayImageDownsampleThreshold"] = notebook.tab_images.opt_GrayImageDownsampling[4].get()/100.0
    result["GrayImageFilter"] = notebook.tab_images.opt_GrayImageCompression[1].get()
    result["GrayImageResolution"] = notebook.tab_images.opt_GrayImageDownsampling[2].get()
    result["LockDistillerParams"] = notebook.tab_misc.opt_LockDistillerParams.get_active()
    result["MaxSubsetPCT"] = notebook.tab_fonts.opt_FontSubsetting[2].get()
    result["MonoImageDownsampleThreshold"] = notebook.tab_images.opt_MonoImageDownsampling[4].get()/100.0
    result["MonoImageFilter"] = notebook.tab_images.opt_MonoImageCompression[1].get()
    result["MonoImageResolution"] = notebook.tab_images.opt_MonoImageDownsampling[2].get()
    result["PreserveHalftoneInfo"] = notebook.tab_color.opt_PreserveHalftoneInfo.get_active()
    result["PreserveOverprintSettings"] = notebook.tab_color.opt_PreserveOverprintSettings.get_active()
    result["SubsetFonts"] = notebook.tab_fonts.opt_FontSubsetting[0].get_active()
    result["UCRandBGInfo"] = "Preserve" if notebook.tab_color.opt_PreserveOverprintSettings.get_active() else "Remove"

    restr=0
    if restrictions_version is None: restrictions_version=result["CompatibilityLevel"]
    
    if restrictions_version==13:
        if notebook.tab_encryption.opt_Restrictions13[0].get_active(): restr += 4+2048
        restr += notebook.tab_encryption.opt_Restrictions13[1][1].get()
        if restr & 8+32: restr+= 256
        if notebook.tab_encryption.opt_Restrictions13[2].get_active(): restr += 16+512
        if restr & 8: restr+=1024
    else:
        restr += notebook.tab_encryption.opt_Restrictions14[0][1].get()
        restr += notebook.tab_encryption.opt_Restrictions14[1][1].get()
        restr += notebook.tab_encryption.opt_Restrictions14[2][1].get()
        if notebook.tab_encryption.opt_Restrictions14[3].get_active() or (restr & 8+32+256 > 40) : restr += 1024
    result["_Restrictions"]=restr
        
    
    return result

def compare_params(p1,p2):
    """Returns an empty dict if p1 and p2 are equal"""
    result={}
    keyset=set(p1.keys()).union(p2.keys())
    for key in keyset:
        try:
            v1=p1[key]
        except KeyError:
            v1=None
        try:
            v2=p2[key]
        except KeyError:
            v2=None
        if v1!=v2:
            result[key]=(v1,v2)
    return result

def params_to_command(curparams):
    params=dict(curparams)
    params["ColorImageDict"]=jpeg_qualities[params["_ColorImageJpegQuality"]]
    params["ColorACSImageDict"]=jpeg_qualities[params["_ColorImageJpegQuality"]]
    params["GrayImageDict"]=jpeg_qualities[params["_GrayImageJpegQuality"]]
    params["GrayACSImageDict"]=jpeg_qualities[params["_GrayImageJpegQuality"]]
    devpar={ "UseDeviceIndependentColor" : ["-dUseCIEColor=true"],
             "CMYK" : ["-sProcessColorModel=DeviceCMYK"],
             "sRGB" : ["-sProcessColorModel=DeviceRGB"],
             "Gray" : ["-sProcessColorModel=DeviceGray"],
             "LeaveColorUnchanged" : [] }

    if params["ColorImageFilter"] in ["Auto","None"]: del(params["ColorImageFilter"])
    if params["GrayImageFilter"] in ["Auto","None"]: del(params["GrayImageFilter"])
    if params["MonoImageFilter"] == "None": del(params["MonoImageFilter"])

    enc=[]
    if params["_Encrypt"]:
        if params["CompatibilityLevel"]==13:
            perm = -64 + (params["_Restrictions"] & (4+8+16+32))
            enc=["-dEncryptionR=2","-dKeyLength=40"]
        else:
            perm = -3904 + params["_Restrictions"]
            enc=["-dEncryptionR=3","-dKeyLength=128"]
        enc += ["-sOwnerPassword="+params["_OwnerPassword"],
                "-sUserPassword="+params["_UserPassword"],
                "-dPermissions="+str(perm)]                
    params["CompatibilityLevel"]=params["CompatibilityLevel"]/10.0

    result=[get_settings()["ghostscript_exec"],"-dSAFER","-dNOPAUSE","-dBATCH","-dPDFSETTING=/"+params["_BasedOn"],
            "-sDEVICE=pdfwrite"]+devpar[params["ColorConversionStrategy"]]+enc+\
            ["-dCompressFonts="+py_to_ps(params["_CompressFonts"]),
            "-c",".setpdfwrite "+py_to_ps(params)+" setdistillerparams","-f"]
    return result

def py_to_ps(o):
    """Convert Python object to PostScript object"""
    if isinstance(o,dict):
        res = "<< "
        for k in o:
            if k[0]!="_":
                res += py_to_ps(k) + " "+py_to_ps(o[k])+" "
        return res+">>"
    elif isinstance(o,list):
        res = "[ "
        for k in o:
            res += py_to_ps(k) + " "
        return res+"]"
    elif isinstance(o,bool):
        return "true" if o else "false"
    elif isinstance(o,str):
            return "/"+o
    else:
        return str(o)
        
