#include <stdlib.h>
#include <stdio.h>
#include "ccls.h"
#include "ccl_private.h"

/* Static functions */
static void _tarif_rebuild(gint tarif_id, GSList ** tarif);
static void _tarif_clear(gint tarif_id);
static time_t _weekstart(time_t t);
static guint _CCL_tarif_realcalc(GSList * slist, time_t s, time_t e,
				 gboolean permin);

extern CCL *ccl;

/* Public interface */
gint
CCL_tarif_new(guint hr, guint min, guint days, guint hourprice)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gint tarif_id;
  gint i;

  g_return_val_if_fail(days <= 127 && days, -1);
  g_return_val_if_fail(hr <= 23 && min <= 59, -1);

  for (i = 0; -1 != CCL_tarif_get_nth(i); i++)
    ;
  
  tarif_id = i ? 1 + CCL_tarif_get_nth(i - 1) : 1;
  cmd = sqlite3_mprintf("INSERT INTO TARIFS\n"
			"(TARIF, ID, DAYS, STIME, HOURPRICE)\n"
			"VALUES(%d, %d, %u, '%.2u:%.2u', %u);",
			tarif_id, 1, days, hr, min, hourprice);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  sqlite3_step(stmt);
  sqlite3_finalize(stmt);

  return tarif_id;
}

void
CCL_tarif_delete(gint tarif)
{
  gchar *cmd = NULL;

  g_return_if_fail(tarif != ccl->tarif_id && tarif > 0);
  
  cmd = sqlite3_mprintf("DELETE FROM TARIFS WHERE TARIF = %d;\n"
			"DELETE FROM PRICES WHERE TARIF = %d;", tarif, tarif);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

gint
CCL_tarif_get_nth(guint nth)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gint i;
  gint tarif = -1;

  cmd = sqlite3_mprintf("SELECT DISTINCT TARIF FROM TARIFS\n"
			"ORDER BY TARIF ASC;");
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  for (i = nth; i > 0; i--)
    sqlite3_step(stmt);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    tarif = sqlite3_column_int(stmt, 0);

  sqlite3_finalize(stmt);

  return tarif;
}

void
CCL_tarif_rebuild(void)
{
  g_return_if_fail(ccl->tarif_id >= 1);
  
  _tarif_clear(ccl->tarif_id);
  ccl->tarif = g_datalist_id_get_data(&(ccl->tarifs), ccl->tarif_id);
  _tarif_rebuild(ccl->tarif_id, &(ccl->tarif));
}

void
CCL_tarif_rebuild_all(void)
{
  GSList ** tarif = NULL;
  gint tarif_id;
  gint i;

  for (i = 0; -1 != (tarif_id = CCL_tarif_get_nth(i)); i++)
    _tarif_clear(tarif_id);

  g_datalist_clear(&ccl->tarifs);
  g_datalist_init(&ccl->tarifs);
  
  for (i = 0; -1 != (tarif_id = CCL_tarif_get_nth(i)); i++)
    {
      tarif = g_new0(GSList *, 1);
      *tarif = NULL;
      _tarif_rebuild(tarif_id, tarif);
      g_datalist_id_set_data(&ccl->tarifs, tarif_id, *tarif);
      g_free(tarif);
    }
}

gboolean
CCL_tarif_set(gint tarif)
{
  g_return_val_if_fail(CCL_tarif_exists(tarif), FALSE);
  g_assert(g_datalist_id_get_data(&ccl->tarifs, tarif));

  ccl->tarif_id = tarif;
  ccl->tarif = g_datalist_id_get_data(&ccl->tarifs, tarif);

  return TRUE;
}

gint
CCL_tarif_get(void)
{
  return ccl->tarif_id;
}

gboolean
CCL_tarif_exists(gint tarif)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gboolean found = FALSE;

  cmd = sqlite3_mprintf("SELECT TARIF FROM TARIFS WHERE TARIF = %d;", tarif);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    found = TRUE;

  sqlite3_finalize(stmt);

  return found;
}

gint
CCL_tarif_get_part_at(guint hour, guint min, guint day)
{
  GSList *link = ccl->tarif;
  guint stime = hour * 60 + min;
  gint i;

  g_return_val_if_fail((hour <= 23 && min <= 59 &&
			(day == SUN || day == MON || day == TUE ||
			 day == WED || day == THU || day == FRI ||
			 day == SAT)), -1);

  for (i = 0; (1 << i) != day; i++)
    ;

  stime += i * 24 * 60;

  while (link)
    {
      CCL_tarifpart *curr = (CCL_tarifpart *) link->data;
      CCL_tarifpart *next = NULL;
      if (g_slist_next(link))
	next = (CCL_tarifpart *) g_slist_next(link)->data;
      else
	next = (CCL_tarifpart *) ccl->tarif->data;
	  
      if ((curr->stime <= stime && next->stime > stime)
	  || (curr->stime > next->stime))
	return curr->id;
      
      link = g_slist_next(link);
    }

  g_assert_not_reached();
  
  return -1;
}

guint
CCL_tarif_calc(time_t start, time_t end, gboolean permin)
{
  return _CCL_tarif_realcalc(ccl->tarif, start, end, permin);
}

guint
CCL_tarif_calc_with_tarifpart(gint id, guint mins, gboolean permin)
{
  guint hprice = CCL_tarifpart_hourprice_get(id);

  g_return_val_if_fail(CCL_tarifpart_exists(id), 0);

  if (permin)
    return (hprice * mins) / 60;
  else
    return CCL_tarifpart_price_get(id, mins, FALSE);
}

/**********************************************************/
gint
CCL_tarifpart_new(guint hour, guint min, guint days, guint hourprice)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gint part_id;
  gint i;

  g_return_val_if_fail(-1 == CCL_tarifpart_id_get(hour, min, days), -1);
  g_return_val_if_fail(!CCL_tarifpart_conflicts(hour, min, days, NULL), -1);
  g_return_val_if_fail(days <= 127 && days, -1);
  g_return_val_if_fail(1 <= ccl->tarif_id, -1);

  for (i = 0; -1 != CCL_tarifpart_get_nth(i); i++)
    ;
  
  part_id = i ? 1 + CCL_tarifpart_get_nth(i - 1) : 1;
  cmd = sqlite3_mprintf("INSERT INTO TARIFS\n"
			"(TARIF, ID, DAYS, STIME, HOURPRICE)\n"
			"VALUES(%d, %d, %u, '%.2u:%.2u', %u);",
			ccl->tarif_id, part_id, days, hour, min, hourprice);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  sqlite3_step(stmt);
  sqlite3_finalize(stmt);

  return CCL_tarifpart_id_get(hour, min, days);
}

void
CCL_tarifpart_delete(gint id)
{
  gchar *cmd = NULL;

  cmd = sqlite3_mprintf("DELETE FROM TARIFS WHERE ID = %d AND TARIF = %d;\n"
			"DELETE FROM PRICES WHERE ID = %d AND TARIF = %d;",
			id, ccl->tarif_id, id, ccl->tarif_id);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}


gint
CCL_tarifpart_get_nth(guint nth)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gint i;
  gint id = -1;

  cmd = sqlite3_mprintf("SELECT DISTINCT ID FROM TARIFS WHERE TARIF = %d\n"
			"ORDER BY ID ASC;", ccl->tarif_id);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  for (i = nth; i > 0; i--)
    sqlite3_step(stmt);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    id = sqlite3_column_int(stmt, 0);

  sqlite3_finalize(stmt);

  return id;
}

gint
CCL_tarifpart_id_get(guint hour, guint min, guint days)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gint id = -1;

  g_return_val_if_fail(hour <= 23 && min <= 59, -1);

  cmd = sqlite3_mprintf("SELECT ID FROM TARIFS WHERE\n"
			"TARIF = %d AND DAYS = %u AND STIME = '%.2u:%.2u';",
			ccl->tarif_id, days, hour, min);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  if (sqlite3_step(stmt) == SQLITE_ROW)
    id = sqlite3_column_int(stmt, 0);

  sqlite3_finalize(stmt);

  return id;
}

gint
CCL_tarifpart_conflicts(guint hour, guint min, guint days, guint * conflicts)
{
  guint nth = 0;
  gint id;
  
  while (-1 != (id = CCL_tarifpart_get_nth(nth)))
    {
      guint h;
      guint m;
      guint d;

      CCL_tarifpart_info_get(id, &h, &m, &d, NULL);
      if (h == hour && m == min && days & d)
	{
	  if (conflicts)
	    *conflicts = days & d;
	      
	  return id;
	}

      nth++;
    }
  
  return 0;
}

gboolean
CCL_tarifpart_info_get(gint id, guint * hour, guint * min,
		       guint * days, guint * hourprice)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gboolean found = FALSE;

  cmd = sqlite3_mprintf("SELECT STIME, DAYS, HOURPRICE FROM TARIFS\n"
			"WHERE ID = %d AND TARIF = %d;", id, ccl->tarif_id);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  if (sqlite3_step(stmt) == SQLITE_ROW)
    {
      guint _hour = 0;
      guint _min = 0;

      sscanf((gchar *) sqlite3_column_text(stmt, 0), "%u:%u", &_hour, &_min);
      if (hour)
	*hour = _hour;
      if (min)
	*min = _min;
      if (days)
	*days = sqlite3_column_int(stmt, 1);
      if (hourprice)
	*hourprice = sqlite3_column_double(stmt, 2);
      found = TRUE;
    }
  sqlite3_finalize(stmt);

  return found;
}

gboolean
CCL_tarifpart_exists(gint id)
{
  return CCL_tarifpart_info_get(id, NULL, NULL, NULL, NULL);
}

void
CCL_tarifpart_stime_set(gint id, guint hour, guint min)
{
  gchar *cmd = NULL;

  g_return_if_fail(hour <= 23 && min <= 59);

  cmd = sqlite3_mprintf("UPDATE TARIFS\n"
			"SET STIME = '%.2u:%.2u'\n"
			"WHERE ID = %d AND TARIF = %d;",
			hour, min, id, ccl->tarif_id);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

void
CCL_tarifpart_stime_get(gint id, guint * hour, guint * min)
{
  CCL_tarifpart_info_get(id, hour, min, NULL, NULL);
}

void
CCL_tarifpart_days_set(gint id, guint days)
{
  gchar *cmd = NULL;

  g_return_if_fail(days <= 127);

  cmd = sqlite3_mprintf("UPDATE TARIFS\n"
			"SET DAYS = %u WHERE ID = %d AND TARIF = %d;",
			days, id, ccl->tarif_id);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

guint
CCL_tarifpart_days_get(gint id)
{
  guint days = 0;

  CCL_tarifpart_info_get(id, NULL, NULL, &days, NULL);

  return days;
}

void
CCL_tarifpart_hourprice_set(gint id, guint price)
{
  gchar *cmd = NULL;

  cmd = sqlite3_mprintf("UPDATE TARIFS\n"
			"SET HOURPRICE = %u WHERE ID = %d AND TARIF = %d;",
			price, id, ccl->tarif_id);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

guint
CCL_tarifpart_hourprice_get(gint id)
{
  guint hprice = 0;

  CCL_tarifpart_info_get(id, NULL, NULL, NULL, &hprice);

  return hprice;
}

void
CCL_tarifpart_price_add(gint id, guint mins, guint price)
{
  gchar *cmd = NULL;

  g_return_if_fail((CCL_tarifpart_exists(id) &&
		    !CCL_tarifpart_price_exists(id, mins)));

  cmd = sqlite3_mprintf("INSERT INTO PRICES (ID, MINS, PRICE, TARIF)\n"
			"VALUES(%d, %u, %u, %d);",
			id, mins, price, ccl->tarif_id);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

void
CCL_tarifpart_price_del(gint id, guint mins)
{
  gchar *cmd = NULL;
  
  g_return_if_fail(CCL_tarifpart_exists(id));

  cmd = sqlite3_mprintf("DELETE FROM PRICES\n"
			"WHERE ID = %d AND TARIF = %d AND MINS = %u;",
			id, ccl->tarif_id, mins);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

void
CCL_tarifpart_price_clear(gint id)
{
  gchar *cmd = NULL;
  
  g_return_if_fail(CCL_tarifpart_exists(id));

  cmd = sqlite3_mprintf("DELETE FROM PRICES\n"
			"WHERE ID = %d AND TARIF = %d;", id, ccl->tarif_id);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

guint
CCL_tarifpart_price_get(gint id, guint mins, gboolean strict)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;

  g_return_val_if_fail(CCL_tarifpart_exists(id), 0);

  cmd = sqlite3_mprintf("SELECT PRICE FROM PRICES\n"
			"WHERE ID = %d AND TARIF = %d AND MINS = %u;",
			id, ccl->tarif_id, mins);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  if (sqlite3_step(stmt) == SQLITE_ROW)
    {
      guint price;

      price = sqlite3_column_double(stmt, 0);
      sqlite3_finalize(stmt);

      return price;
    }
  else
    sqlite3_finalize(stmt);

  if (strict)
    return 0;
  else
    {
      guint lprice = CCL_tarifpart_price_get_last(id);
      
      if (mins < lprice)
	{
	  gint i;
	  guint _mins = 0;
	  guint _price = 0;

	  for (i = 0; CCL_tarifpart_price_get_nth(id, i, &_mins, &_price); i++)
	    {
	      if (mins < _mins)
		return _price;
	    }
	}
      else if (60 >= mins)
	return CCL_tarifpart_hourprice_get(id);
      else
	{
	  if (60 > lprice) lprice = 60;
	  return CCL_tarifpart_price_get(id, mins - lprice, FALSE) +
		 CCL_tarifpart_price_get(id, lprice, FALSE);
	}
    }

  return 0;
}

gboolean
CCL_tarifpart_price_get_nth(gint id, guint nth, guint * mins, guint * price)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gint i;
  gboolean found = FALSE;

  g_return_val_if_fail(CCL_tarifpart_exists(id), FALSE);

  cmd = sqlite3_mprintf("SELECT MINS, PRICE FROM PRICES\n"
			"WHERE ID = %d AND TARIF = %d\n"
			"ORDER BY MINS ASC;", id, ccl->tarif_id);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  for (i = nth; i > 0; i--)
    sqlite3_step(stmt);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    {
      found = TRUE;
      if (mins)
	*mins = sqlite3_column_double(stmt, 0);
      if (price)
	*price = sqlite3_column_double(stmt, 1);
    }
  sqlite3_finalize(stmt);

  return found;
}

guint
CCL_tarifpart_price_get_nearest(gint id, guint mins)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  guint nextmins = 0;

  g_return_val_if_fail(CCL_tarifpart_exists(id), 0);

  if (mins >= ccl->perminafter)
    return mins;

  cmd = sqlite3_mprintf("SELECT MINS FROM PRICES WHERE MINS >= %d\n"
			"ORDER BY MINS ASC;", mins);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  if (sqlite3_step(stmt) == SQLITE_ROW)
    nextmins = sqlite3_column_double(stmt, 0);
  
  sqlite3_finalize(stmt);

  if (!nextmins)
    nextmins = ccl->perminafter;

  return nextmins;
}

gboolean
CCL_tarifpart_price_exists(gint id, guint mins)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;

  g_return_val_if_fail(CCL_tarifpart_exists(id), FALSE);

  cmd = sqlite3_mprintf("SELECT PRICE FROM PRICES\n"
			"WHERE ID = %d AND TARIF = %d AND MINS = %u;",
			id, ccl->tarif_id, mins);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  if (sqlite3_step(stmt) == SQLITE_ROW)
    {
      sqlite3_finalize(stmt);
      return TRUE;
    }
  else
    {
      sqlite3_finalize(stmt);
      return FALSE;
    }
}

guint
CCL_tarifpart_price_get_last(gint id)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  guint mins = 0;

  g_return_val_if_fail(CCL_tarifpart_exists(id), 0);

  cmd = sqlite3_mprintf("SELECT MAX(MINS) FROM PRICES\n"
			"WHERE ID = %d AND TARIF = %d;", id, ccl->tarif_id);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  if (sqlite3_step(stmt) == SQLITE_ROW)
    mins = sqlite3_column_double(stmt, 0);
  
  sqlite3_finalize(stmt);

  return mins;
}

/**********************************************************/

void
_tarif_rebuild(gint tarif_id, GSList ** tarif)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  GSList *link = NULL;

  g_assert(tarif_id > 0 && tarif != NULL);
  
  _tarif_clear(tarif_id);
  cmd = sqlite3_mprintf("SELECT ID, DAYS, STIME, HOURPRICE FROM TARIFS\n"
			"WHERE TARIF = %d;", tarif_id);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  while (sqlite3_step(stmt) == SQLITE_ROW)
    {
      CCL_tarifpart *part = NULL;
      guint days = sqlite3_column_double(stmt, 1);
      guint hour = 0;
      guint min = 0;
      guint stime = 0;
      gint i;
      gchar *cmd2 = NULL;
      sqlite3_stmt *stmt2 = NULL;
      GData ** prices = g_malloc0(sizeof(GData *));
      gint id = 0;

      id = sqlite3_column_int(stmt, 0);
      /* Load prices */
      g_datalist_init(prices);
      cmd2 = sqlite3_mprintf("SELECT MINS, PRICE FROM PRICES\n"
			     "WHERE ID = %d AND TARIF = %d;",
			     id, ccl->tarif_id);
      sqlite3_prepare(ccl->db, cmd2, -1, &stmt2, NULL);
      sqlite3_free(cmd2);
      while (sqlite3_step(stmt2) == SQLITE_ROW)
	{
	  gint mins;
	  guint price;

	  mins = sqlite3_column_int(stmt2, 0);
	  price = sqlite3_column_double(stmt2, 1);
	  g_datalist_id_set_data(prices, mins, GINT_TO_POINTER(price));
	}
      sqlite3_finalize(stmt2);

      /* Add tarifpart */
      sscanf((gchar *) sqlite3_column_text(stmt, 2), "%u:%u", &hour, &min);
      stime = hour * 60 + min;

      for (i = 0; i < 7; i++)
	{
	  if ((1 << i) & days)
	    {
	      part = g_new0(CCL_tarifpart, 1);
	      part->stime = stime + i * 24 * 60;
	      part->id = sqlite3_column_int(stmt, 0);
	      part->hprice = sqlite3_column_double(stmt, 3);
	      part->prices = prices;

	      *tarif = g_slist_insert_sorted(*tarif, part, _TarifCompareFunc);
	    }
	}
    }
  sqlite3_finalize(stmt);
  /* Now i am going to remove unnessesary links to clean things */
  link = *tarif;
  while (link)
    {
      GSList *nextlink = g_slist_next(link);
      CCL_tarifpart *part = (CCL_tarifpart *) link->data;

      if (!nextlink)
	link = NULL;
      else if (((CCL_tarifpart *) nextlink->data)->id == part->id)
	{
	  *tarif = g_slist_remove_link(*tarif, nextlink);
	  g_free(nextlink->data);
	  nextlink->data = NULL;
	  g_slist_free_1(nextlink);
	}
      else
	link = nextlink;
    }
  /* If the first link in the list does not start on sunday at 00:00  *
   * insert the last one there, so things don't broke on calculation  */
  if (*tarif && 0 != ((CCL_tarifpart *) (*tarif)->data)->stime)
    {
      CCL_tarifpart *newpart = g_new0(CCL_tarifpart, 1);
      CCL_tarifpart *lastpart = NULL;
      GSList *link = *tarif;
      
      while (link->next)
	link = link->next;
      
      lastpart = (CCL_tarifpart *) link->data;

      newpart->id = lastpart->id;
      newpart->hprice = lastpart->hprice;
      newpart->stime = 0;
      newpart->prices = lastpart->prices;

      *tarif = g_slist_prepend(*tarif, newpart);
    }
  /* add it to the list */
  g_datalist_id_set_data(&ccl->tarifs, tarif_id, *tarif);
}

static void
_tarif_clear(gint tarif_id)
{
  GSList **tarif = NULL;
  GSList *link = NULL;
  CCL_tarifpart *tp = NULL;
  GData *freed = NULL;

  tarif = g_new0(GSList *, 1);
  *tarif = g_datalist_id_get_data(&ccl->tarifs, tarif_id);

  g_assert(NULL != tarif);
  
  g_datalist_init(&freed);

  while ((link = g_slist_last(*tarif)) && *tarif)
    {
      *tarif = g_slist_remove_link(*tarif, link);
      tp = (CCL_tarifpart *) link->data;
      if (!g_datalist_id_get_data(&freed, GPOINTER_TO_INT(tp->prices)))
	{
	  g_datalist_clear(tp->prices);
	  g_datalist_id_set_data(&freed, GPOINTER_TO_INT(tp->prices),
				 (void *)tp->prices);
	  g_free(tp->prices);
	}
      g_slist_free_1(link);
      g_free(tp);
    }
  g_datalist_clear(&freed);
  *tarif = NULL;
  g_datalist_id_remove_data(&ccl->tarifs, tarif_id);
  g_free(tarif);
}

gint
_TarifCompareFunc(gconstpointer a, gconstpointer b)
{
  CCL_tarifpart *t1 = (CCL_tarifpart *) a;
  CCL_tarifpart *t2 = (CCL_tarifpart *) b;

  g_assert(t1 && t2);

  if (t1->stime > t2->stime) return 1;
  else if (t1->stime < t2->stime) return -1;
  else return 0;
}

/* Static */
static time_t
_weekstart(time_t t)
{
  struct tm *ws = NULL;
  time_t weekst = 0;
  time_t weekstart = 0;

  ws = localtime(&t);
  ws->tm_sec = ws->tm_min = ws->tm_hour = 0;
  weekst = mktime(ws);
  weekst -= 24 * 60 * 60 * (ws->tm_wday);
  ws = localtime(&weekst);
  weekstart = mktime(ws);

  return weekstart;
}

static guint
_CCL_tarif_realcalc(GSList * slist, time_t s, time_t e, gboolean permin)
{
  time_t now = time(NULL);	  /* current time */
  time_t et = e - _weekstart(s);  /* interval end time */
  time_t st = s - _weekstart(s);  /* interval start time */
  time_t etime;			  /* this part end time */
  time_t stime;			  /* this part start time */
  CCL_tarifpart *next = NULL;
  CCL_tarifpart *this = NULL;
  GSList *nextlink = g_slist_next(slist);
  
  if (slist)
    this = slist->data;
  else
    return 0;

  if (nextlink)
    next = (CCL_tarifpart *) nextlink->data;

  if (et == st)
    et++;			/* To make total time > 0 */

  if (s > now || e > now)
    g_warning("[!]:The given times are in the future!");

  stime = this->stime;

  if (!next)
    etime = 60 * 60 * 24 * 7;
  else
    etime = 60 * next->stime;	/* If et > etime, it belongs
				 * to the next */
  /* All inside */
  if (st >= stime && st < etime && et >= stime && et <= etime)
    {
      if (!permin)
	return CCL_tarifpart_price_get(this->id, (et - st) / 60, FALSE);
      else
	return ((this->hprice * (et - st)) / 3600);
    }
  else if (next && st >= etime)	/* All outside */
    return _CCL_tarif_realcalc(nextlink, s, e, permin);
  else if (et > etime)
    {				/* A part inside */
      if (!next)		/* On next week */
	nextlink = ccl->tarif;	/* then make next, the start of the week */
      else
	nextlink = g_slist_next(slist);

      next = (CCL_tarifpart *) nextlink->data;

      if (permin)		/* calculate it for each min? */
	{
	  time_t end = _weekstart(s) + etime; 

	  return _CCL_tarif_realcalc(slist, s, end, permin) +
		 _CCL_tarif_realcalc(nextlink, end, e, permin);
	}
      else
	{
	  gdouble frac1;
	  gdouble frac2;
	  guint minutes = CCL_tarifpart_price_get_nearest(this->id,
							  (et - st) / 60);
	  time_t et = st + minutes * 60;

	  frac1 = (etime - st) / (gdouble) (et - st);
	  frac2 = (et - etime) / (gdouble) (et - st);

	  return (guint) (CCL_tarifpart_price_get
			  (this->id, (et - st) / 60, FALSE) * frac1 +
			  CCL_tarifpart_price_get
			  (next->id, (et - st) / 60, FALSE) * frac2);
	}
    }
  else
    {				/* Something is wrong */
      g_message("[E]:There is something wrong with the tarif\n");
      g_message
	("stime:%ld,st:%ld,etime:%ld,et:%ld,s:%ld,e:%ld,now:%ld ws:%ld\n",
	 stime, st, etime, et, s, e, now, _weekstart(s));
      g_assert_not_reached();

      return 0;
    }
}
