#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "ccls.h"
#include "ccl_private.h"

extern CCL *ccl;

/* Static functions */
static gint _highest_logged_session_id();

/* Public interface */

void
CCL_log_expense(const gchar description[128], guint cash)
{
  gchar *cmd = NULL;

  cmd = sqlite3_mprintf("INSERT INTO EXPENSESLOG (DESCRIPTION, TIME, CASH)\n"
			"VALUES(%Q, %ld, %u);", description, time(NULL), cash);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

gint
CCL_log_expenses_get(CCL_log_search_rules * sr, CCL_log_expense_entry ** ee)
{
  CCL_log_expense_entry *entryarray = NULL;
  gint nrow = 0;
  gint ncol = 0;
  gchar *cmd = NULL;
  gchar **argv = NULL;
  gchar *srtxt[6] = { 0, 0, 0, 0, 0, 0 };
  gint i;
  gint tdiff;
  struct tm t;
  
  memset(&t, 0, sizeof(struct tm));
  t.tm_year = 70;
  t.tm_mday = 1;
  tdiff = (gint)mktime(&t);

  if (sr)
    {
      if (sr->rulemask & SR_TIMEMIN)
	srtxt[0] = sqlite3_mprintf("AND TIME >= %ld", sr->time_min);
      if (sr->rulemask & SR_TIMEMAX)
	srtxt[1] = sqlite3_mprintf("AND TIME <= %ld", sr->time_max);
      if (sr->rulemask & SR_PRICEMIN)
	srtxt[2] = sqlite3_mprintf("AND PRICE >= %u", sr->price_min);
      if (sr->rulemask & SR_PRICEMAX)
	srtxt[3] = sqlite3_mprintf("AND PRICE <= %u", sr->price_max);
      if (sr->rulemask & SR_DAYTIME_RANGE)
	{
	  if (sr->daytime_min == 0 && sr->daytime_max == 86399)
	    ; /* Ignore this rule, because all the day is covered */
	  else if (sr->daytime_min <= sr->daytime_max)
	    srtxt[4] = sqlite3_mprintf("AND (STIME - %d) %% 86400 >= %d "
				       "AND (STIME - %d) %% 86400 <= %d",
				       tdiff, sr->daytime_min,
				       tdiff, sr->daytime_max);
	  else
	    srtxt[4] = sqlite3_mprintf("AND ((STIME - %d) %% 86400 >= %d "
				       "OR (STIME - %d) %% 86400 <= %d)",
				       tdiff, sr->daytime_min,
				       tdiff, sr->daytime_max);
	}
      if (sr->rulemask & SR_DAYS && sr->days < 127) /* 127 is all days */
	srtxt[5] = sqlite3_mprintf("AND (((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u)",
				   tdiff, (sr->days & SUN ? 3 : 7 ),
				   tdiff, (sr->days & MON ? 4 : 7 ),
				   tdiff, (sr->days & TUE ? 5 : 7 ),
				   tdiff, (sr->days & WED ? 6 : 7 ),
				   tdiff, (sr->days & THU ? 0 : 7 ),
				   tdiff, (sr->days & FRI ? 1 : 7 ),
				   tdiff, (sr->days & SAT ? 2 : 7 ));
	/* it has an offset of - 5 days because 01/01/1970 is thursday
	 * (day 5 of the week) */
    }

  cmd = sqlite3_mprintf("SELECT DESCRIPTION,TIME,CASH\n"
			"FROM EXPENSESLOG WHERE CASH != -1\n"
			"%s %s %s %s %s %s\n"
			"ORDER BY TIME ASC;", srtxt[0], srtxt[1], srtxt[2],
			srtxt[3], srtxt[4], srtxt[5]);
  /* free what we have in srtxt[] */
  for (i = 0; i < 6; i++)
    sqlite3_free(srtxt[i]);

  if (sqlite3_get_table(ccl->db, cmd, &argv, &nrow, &ncol, NULL) == SQLITE_OK
      && argv && nrow > 0 && ncol)
    {
      gint i;
      
      if (!ee)	      /* If !ee, return the number of found entries */
	return nrow;
		      /* Otherwise, lets fill the array with found entries */
      entryarray = g_new0(CCL_log_expense_entry, nrow);
      
      for (i = 0; i < nrow; i++)
	{
	  gint os = (i+1) * ncol; /* offset */
	 
	  g_snprintf((gchar *) entryarray[i].description, 128, "%s", argv[os]);
	  sscanf(argv[os+1], "%ld", &(entryarray[i].time));
	  sscanf(argv[os+2], "%u", &(entryarray[i].cash));
	}
    }

  sqlite3_free(cmd);
  sqlite3_free_table(argv);
  *ee = entryarray;
  
  return nrow;	/* nrow is the number of found entries */

}

gint
CCL_log_session(gint client, gint paid, guint price)
{
  CCL_client *c = g_datalist_id_get_data(&(ccl->clients), client);
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gint i;
  gint id;
  guint amount;
  guint nintervals;
  gint ibytes;
  gint sessionid;
  time_t stime;
  time_t etime;
  CCL_interval *interval = NULL;
  CCL_interval *intervals = NULL;

  g_return_val_if_fail((c && c->status == INACTIVE), 0);
  
  nintervals = c->intervals->len;
  ibytes = nintervals * sizeof(CCL_interval);
  sessionid = 1 + _highest_logged_session_id();
  stime = CCL_client_stime_get(client);
  etime = CCL_client_etime_get(client);
  interval = NULL;
  intervals = g_new0(CCL_interval, nintervals);

  for (i = 0; i < nintervals; i++)
    {
      interval = (CCL_interval *) g_ptr_array_index(c->intervals, i);
      intervals[i].stime = interval->stime;
      intervals[i].etime = interval->etime;
    }

  cmd = sqlite3_mprintf("INSERT INTO SESSIONSLOG (SESSION, CLIENT, MEMBER,"
			"STIME, ETIME, TIME, PRICE, PAID, INTERVALS)\n"
			"VALUES(%d, %d, %d, %d, %d, %d, %u, %d, ?1);",
			sessionid, client, c->member, stime, etime,
			CCL_client_time_used(client), price, paid);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  sqlite3_bind_blob(stmt, 1, intervals, ibytes, g_free);
  sqlite3_step(stmt);
  sqlite3_finalize(stmt);

  for (i = 0; CCL_client_product_get_nth(client, i, &id, &amount); i++)
    {
      guint price = 0;
      
      CCL_product_info_get(id, NULL, NULL, &price);
      cmd = sqlite3_mprintf("INSERT INTO PRODUCTSLOG (SESSION, CLIENT,"
			    "MEMBER, PRODUCT, AMOUNT, PRICE, TIME)\n"
			    "VALUES(%d, %d, %d, %d, %u, %u, %ld);",
			    sessionid, client, c->member, id, amount,
			    price * amount, etime);
      sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
      sqlite3_free(cmd);
    }
  
  return sessionid;
}

gint
CCL_log_session_find(gint client, time_t stime, time_t etime)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gint session = -1;

  g_return_val_if_fail(stime <= etime, -1);

  cmd = sqlite3_mprintf("SELECT SESSION FROM SESSIONSLOG\n"
			"WHERE CLIENT = %d AND STIME = %ld AND ETIME = %ld;",
			client, stime, etime);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  if (sqlite3_step(stmt) == SQLITE_ROW)
    session = sqlite3_column_int(stmt, 0);

  sqlite3_finalize(stmt);

  return session;
}

void
CCL_log_session_clear(gint session)
{
  gchar *cmd = NULL;

  cmd = sqlite3_mprintf("DELETE FROM SESSIONSLOG WHERE SESSION = %d", session);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
  cmd = sqlite3_mprintf("DELETE FROM PRODUCTSLOG WHERE SESSION = %d", session);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

void
CCL_log_session_set_paid(gint session, gint paid, guint price)
{
  gchar *cmd = NULL;

  cmd = sqlite3_mprintf("UPDATE SESSIONSLOG SET PAID = %d, PRICE = %u\n"
			"WHERE SESSION = %d;", paid, price, session);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

void
CCL_log_session_set_member(gint session, gint member)
{
  gchar *cmd = NULL;
  
  g_return_if_fail(member == 0 || CCL_member_exists(member));

  cmd = sqlite3_mprintf("UPDATE SESSIONSLOG SET MEMBER = %d\n"
			"WHERE SESSION = %d;", member, session);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

gint
CCL_log_sessions_get(CCL_log_search_rules * sr, CCL_log_session_entry ** se)
{
  CCL_log_session_entry *entryarray = NULL;
  gint i;
  gint nrow = 0;
  gint ncol = 0;
  gchar *cmd = NULL;
  gchar **argv = NULL;
  gchar *srtxt[14] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
  gint tdiff;
  struct tm t;
  
  memset(&t, 0, sizeof(struct tm));
  t.tm_year = 70;
  t.tm_mday = 1;
  tdiff = (gint)mktime(&t);

  if (sr)
    {
      if (sr->rulemask & SR_SESSION)
	srtxt[0] = sqlite3_mprintf("AND SESSION = %d", sr->session);
      if (sr->rulemask & SR_CLIENT)
	srtxt[1] = sqlite3_mprintf("AND CLIENT = %d", sr->client);
      if (sr->rulemask & SR_MEMBER)
	srtxt[2] = sqlite3_mprintf("AND MEMBER = %d", sr->member);
      if (sr->rulemask & SR_STIMEMIN)
	srtxt[3] = sqlite3_mprintf("AND STIME >= %ld", sr->stime_min);
      if (sr->rulemask & SR_STIMEMAX)
	srtxt[4] = sqlite3_mprintf("AND STIME <= %ld", sr->stime_max);
      if (sr->rulemask & SR_ETIMEMIN)
	srtxt[5] = sqlite3_mprintf("AND ETIME >= %ld", sr->etime_min);
      if (sr->rulemask & SR_ETIMEMAX)
	srtxt[6] = sqlite3_mprintf("AND ETIME <= %ld", sr->etime_max);
      if (sr->rulemask & SR_TIMEMIN)
	srtxt[7] = sqlite3_mprintf("AND TIME >= %ld", sr->time_min);
      if (sr->rulemask & SR_TIMEMAX)
	srtxt[8] = sqlite3_mprintf("AND TIME <= %ld", sr->time_max);
      if (sr->rulemask & SR_PRICEMIN)
	srtxt[9] = sqlite3_mprintf("AND PRICE >= %u", sr->price_min);
      if (sr->rulemask & SR_PRICEMAX)
	srtxt[10] = sqlite3_mprintf("AND PRICE <= %u", sr->price_max);
      if (sr->rulemask & SR_PAIDMODE)
	srtxt[11] = sqlite3_mprintf("AND PAID = %d", sr->paidmode);
      if (sr->rulemask & SR_DAYTIME_RANGE)
	{
	  if (sr->daytime_min == 0 && sr->daytime_max == 86399)
	    ; /* Ignore this rule, because all the day is covered */
	  else if (sr->daytime_min <= sr->daytime_max)
	    srtxt[12] = sqlite3_mprintf("AND (STIME - %d) %% 86400 >= %d "
					"AND (STIME - %d) %% 86400 <= %d",
					tdiff, sr->daytime_min,
					tdiff, sr->daytime_max);
	  else
	    srtxt[12] = sqlite3_mprintf("AND ((STIME - %d) %% 86400 >= %d "
					"OR (STIME - %d) %% 86400 <= %d)",
					tdiff, sr->daytime_min,
					tdiff, sr->daytime_max);
	}
      if (sr->rulemask & SR_DAYS && sr->days < 127) /* 127 is all days */
	srtxt[13] = sqlite3_mprintf("AND (((STIME - %d) / 86400) %% 7 = %u "
				    "OR ((STIME - %d) / 86400) %% 7 = %u "
				    "OR ((STIME - %d) / 86400) %% 7 = %u "
				    "OR ((STIME - %d) / 86400) %% 7 = %u "
				    "OR ((STIME - %d) / 86400) %% 7 = %u "
				    "OR ((STIME - %d) / 86400) %% 7 = %u "
				    "OR ((STIME - %d) / 86400) %% 7 = %u)",
				    tdiff, (sr->days & SUN ? 3 : 7 ),
				    tdiff, (sr->days & MON ? 4 : 7 ),
				    tdiff, (sr->days & TUE ? 5 : 7 ),
				    tdiff, (sr->days & WED ? 6 : 7 ),
				    tdiff, (sr->days & THU ? 0 : 7 ),
				    tdiff, (sr->days & FRI ? 1 : 7 ),
				    tdiff, (sr->days & SAT ? 2 : 7 ));
	/* it has an offset of - 5 days because 01/01/1970 is thursday
	 * (day 5 of the week) */
    }

  cmd = sqlite3_mprintf("SELECT SESSION, CLIENT, MEMBER,\n"
			"STIME, ETIME, TIME, PRICE, PAID\n"
			"FROM SESSIONSLOG WHERE SESSION != -1\n"
			"%s %s %s %s %s %s %s %s %s %s %s %s %s %s\n"
			"ORDER BY STIME ASC;", srtxt[0], srtxt[1], srtxt[2],
			srtxt[3], srtxt[4], srtxt[5], srtxt[6], srtxt[7],
			srtxt[8], srtxt[9], srtxt[10], srtxt[11], srtxt[12],
			srtxt[13]);
  /* free what we have in srtxt[] */
  for (i = 0; i < 14; i++)
    sqlite3_free(srtxt[i]);

  if (sqlite3_get_table(ccl->db, cmd, &argv, &nrow, &ncol, NULL) == SQLITE_OK
      && argv && nrow > 0 && ncol)
    {
      gint i;
      
      if (!se)	      /* If !se, return the number of found entries */
	return nrow;
		      /* Otherwise, lets fill the array with found entries */
      entryarray = g_new0(CCL_log_session_entry, nrow);
      
      for (i = 0; i < nrow; i++)
	{
	  gint os = (i+1) * ncol; /* offset */
	  
	  sscanf(argv[os], "%d", &(entryarray[i].session));
	  sscanf(argv[os+1], "%d", &(entryarray[i].client));
	  sscanf(argv[os+2], "%d", &(entryarray[i].member));
	  sscanf(argv[os+3], "%ld", &(entryarray[i].stime));
	  sscanf(argv[os+4], "%ld", &(entryarray[i].etime));
	  sscanf(argv[os+5], "%d", &(entryarray[i].time));
	  sscanf(argv[os+6], "%u", &(entryarray[i].price));
	  sscanf(argv[os+7], "%d", &(entryarray[i].paid));
	}
    }
 
  sqlite3_free(cmd);
  sqlite3_free_table(argv);
  *se = entryarray;
  
  return nrow;	/* nrow is the number of found entries */
}

gint
CCL_log_session_intervals_get(gint session, time_t ** intervals)
{
  gchar *cmd;
  sqlite3_stmt *stmt = NULL;
  gint inum = 0;

  cmd = sqlite3_mprintf("SELECT INTERVALS FROM SESSIONSLOG\n"
			"WHERE SESSION = %d;", session);

  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  if (sqlite3_step(stmt) == SQLITE_ROW)
    {
      gint bytes = sqlite3_column_bytes(stmt, 0);
      
      inum = bytes / sizeof(CCL_interval);
      
      if (inum)
	{
	  *intervals = g_malloc0(bytes);
	  memcpy(*intervals, sqlite3_column_blob(stmt, 0), bytes);
	}
    }
  sqlite3_finalize(stmt);

  return inum;
}

gint
CCL_log_products_get(CCL_log_search_rules * sr, CCL_log_product_entry ** pe )
{
  CCL_log_product_entry *entryarray = NULL;
  gint nrow = 0;
  gint ncol = 0;
  gint i;
  gchar *cmd = NULL;
  gchar **argv = NULL;
  gchar *srtxt[7] = { 0, 0, 0, 0, 0, 0, 0 };
  gint tdiff;
  struct tm t;
  
  memset(&t, 0, sizeof(struct tm));
  t.tm_year = 70;
  t.tm_mday = 1;
  tdiff = (gint)mktime(&t);

  if (sr)
    {
      if (sr->rulemask & SR_SESSION)
	srtxt[0] = sqlite3_mprintf("AND SESSION = %d", sr->session);
      if (sr->rulemask & SR_CLIENT)
	srtxt[1] = sqlite3_mprintf("AND CLIENT = %d", sr->client);
      if (sr->rulemask & SR_MEMBER)
	srtxt[2] = sqlite3_mprintf("AND MEMBER = %d", sr->member);
      if (sr->rulemask & SR_TIMEMIN)
	srtxt[3] = sqlite3_mprintf("AND TIME >= %ld", sr->time_min);
      if (sr->rulemask & SR_TIMEMAX)
	srtxt[4] = sqlite3_mprintf("AND TIME <= %ld", sr->time_max);
      if (sr->rulemask & SR_DAYTIME_RANGE)
	{
	  if (sr->daytime_min == 0 && sr->daytime_max == 86399)
	    ; /* Ignore this rule, because all the day is covered */
	  else if (sr->daytime_min <= sr->daytime_max)
	    srtxt[5] = sqlite3_mprintf("AND (TIME - %d) %% 86400 >= %d "
				       "AND (TIME - %d) %% 86400 <= %d",
				       tdiff, sr->daytime_min,
				       tdiff, sr->daytime_max);
	  else
	    srtxt[5] = sqlite3_mprintf("AND ((TIME - %d) %% 86400 >= %d "
				      "OR (TIME - %d) %% 86400 <= %d)",
				      tdiff, sr->daytime_min,
				      tdiff, sr->daytime_max);
	}
      if (sr->rulemask & SR_DAYS && sr->days < 127) /* 127 is all days */
	srtxt[6] = sqlite3_mprintf("AND (((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u "
				   "OR ((TIME - %d) / 86400) %% 7 = %u)",
				   tdiff, (sr->days & SUN ? 3 : 7 ),
				   tdiff, (sr->days & MON ? 4 : 7 ),
				   tdiff, (sr->days & TUE ? 5 : 7 ),
				   tdiff, (sr->days & WED ? 6 : 7 ),
				   tdiff, (sr->days & THU ? 0 : 7 ),
				   tdiff, (sr->days & FRI ? 1 : 7 ),
				   tdiff, (sr->days & SAT ? 2 : 7 ));
	/* it has an offset of - 5 days because 01/01/1970 is thursday
	 * (day 5 of the week) */
    }

  cmd = sqlite3_mprintf("SELECT SESSION, CLIENT, MEMBER,\n"
			"PRODUCT, AMOUNT, PRICE, TIME\n"
			"FROM PRODUCTSLOG WHERE SESSION != -1\n"
			"%s %s %s %s %s %s\n"
			"ORDER BY TIME ASC;", srtxt[0], srtxt[1], srtxt[2],
			srtxt[3], srtxt[4], srtxt[5], srtxt[6]);
  /* free what we have in srtxt[] */
  for (i = 0; i < 7; i++)
    sqlite3_free(srtxt[i]);

  if (sqlite3_get_table(ccl->db, cmd, &argv, &nrow, &ncol, NULL) == SQLITE_OK
      && argv && nrow > 0 && ncol)
    {
      gint i;
     
      if (!pe)		/* If !pe, return the number of found entries */
	return nrow;

      entryarray = g_new0(CCL_log_product_entry, nrow);
      
      for (i = 0; i < nrow; i++)
	{
	  gint os = (i+1) * ncol; /* offset */
	  
	  sscanf(argv[os], "%d", &(entryarray[i].session));
	  sscanf(argv[os+1], "%d", &(entryarray[i].client));
	  sscanf(argv[os+2], "%d", &(entryarray[i].member));
	  sscanf(argv[os+3], "%d", &(entryarray[i].product));
	  sscanf(argv[os+4], "%u", &(entryarray[i].amount));
	  sscanf(argv[os+5], "%u", &(entryarray[i].price));
	  sscanf(argv[os+6], "%ld", &(entryarray[i].time));
	}
    }

  sqlite3_free(cmd);
  sqlite3_free_table(argv);
  *pe = entryarray;
  
  return nrow;
}

/* Static */
static gint
_highest_logged_session_id()
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gint id = 0;

  cmd = sqlite3_mprintf("SELECT MAX(SESSION) FROM SESSIONSLOG;");
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  if (sqlite3_step(stmt) == SQLITE_ROW &&
      sqlite3_column_type(stmt, 0) != SQLITE_NULL)
    id = sqlite3_column_int(stmt, 0);
  
  sqlite3_finalize(stmt);

  return id;
}
