#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "ccls.h"
#include "ccl_private.h"

extern CCL *ccl;

/* Private functions */

enum { CLIENTSDATA = 0, MEMBERSDATA, _LAST };

static const gchar * tables[] = { "CLIENTSDATA", "MEMBERSDATA" };

typedef gboolean (*thing_exists_func) (gint id);
static thing_exists_func _thing_exists[] = { CCL_client_exists,
					     CCL_member_exists };

static void _thingdata_set_string(gint thing, gint id, const gchar * key,
				  const gchar * value);
static gchar * _thingdata_get_string(gint thing, gint id, const char * key,
				     const char * defvalue);
static void _thingdata_set_int(gint thing, gint id, const gchar * key,
			       gint value);
static gint _thingdata_get_int(gint thing, gint id, const gchar * key,
			       gint defval);
static void _thingdata_set_blob(gint thing, gint id, const gchar * key,
				void * value, gint size);
static void * _thingdata_get_blob(gint thing, gint id, const gchar * key,
				  int * size);
static gboolean _thingdata_key_exists(gint thing, gint id, const gchar * key);
static void _thingdata_key_delete(gint thing, gint id, const gchar * key);

/* Public interface */

/*******
* DATA *
*******/

void
CCL_data_set_string(const gchar * key, const gchar * value)
{
  gchar *cmd = NULL;

  g_return_if_fail(key);
  
  if (CCL_data_key_exists(key))
    cmd = sqlite3_mprintf("UPDATE DATA\n"
			  "SET VALUE = %Q\n"
			  "WHERE KEY = %Q;", value, key);
  else
    cmd = sqlite3_mprintf("INSERT INTO DATA\n"
			  "(KEY, VALUE)\n"
			  "VALUES(%Q, %Q);", key, value);

  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

gchar *
CCL_data_get_string(const char * key, const char * defvalue)
{
  gchar *cmd = NULL;
  gchar *ret = NULL;
  sqlite3_stmt *stmt = NULL;

  g_return_val_if_fail(key, NULL);

  cmd = sqlite3_mprintf("SELECT VALUE FROM DATA\n"
			"WHERE KEY = %Q;", key);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    ret = g_strdup((gchar *) sqlite3_column_text(stmt, 0));

  sqlite3_finalize(stmt);

  if (!ret && defvalue)
    return g_strdup(defvalue);
  else
    return ret;
}

void
CCL_data_set_int(const gchar * key, gint value)
{
  gchar *cmd = NULL;
  
  g_return_if_fail(key);

  if (CCL_data_key_exists(key))
    cmd = sqlite3_mprintf("UPDATE DATA\n"
			  "SET VALUE = %d\n"
			  "WHERE KEY = %Q;", value, key);
  else
    cmd = sqlite3_mprintf("INSERT INTO DATA\n"
			  "(KEY, VALUE)\n"
			  "VALUES(%Q, %d);", key, value);

  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

gint
CCL_data_get_int(const gchar * key, gint defval)
{
  gchar *cmd = NULL;
  gint ret = defval;
  sqlite3_stmt *stmt = NULL;

  g_return_val_if_fail(key, -1);

  cmd = sqlite3_mprintf("SELECT VALUE FROM DATA\n"
			"WHERE KEY = %Q;", key);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    ret = sqlite3_column_int(stmt, 0);

  sqlite3_finalize(stmt);

  return ret;
}

void
CCL_data_set_blob(const gchar * key, void * value, gint size)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  
  g_return_if_fail(key);

  if (CCL_data_key_exists(key))
    cmd = sqlite3_mprintf("UPDATE DATA\n"
			  "SET VALUE = ?1\n"
			  "WHERE KEY = %Q;", key);
  else
    cmd = sqlite3_mprintf("INSERT INTO DATA\n"
			  "(KEY, VALUE)\n"
			  "VALUES(%Q, ?1);", key);

  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  sqlite3_bind_blob(stmt, 1, value, size, SQLITE_TRANSIENT);
  sqlite3_step(stmt);
  sqlite3_finalize(stmt);
}

void *
CCL_data_get_blob(const gchar * key, int * size)
{
  gchar *cmd = NULL;
  void *ret = NULL;
  sqlite3_stmt *stmt = NULL;

  g_return_val_if_fail(key, NULL);
  g_return_val_if_fail(size, NULL);

  cmd = sqlite3_mprintf("SELECT VALUE FROM DATA\n"
			"WHERE KEY = %Q;", key);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    {
      *size = sqlite3_column_bytes(stmt, 0);
      ret = g_malloc0(*size);
      memcpy(ret, sqlite3_column_blob(stmt, 0), *size);
    }

  sqlite3_finalize(stmt);

  return ret;
}

gboolean
CCL_data_key_exists(const gchar * key)
{
  gchar *cmd = NULL;
  gboolean ret = FALSE;
  sqlite3_stmt *stmt = NULL;

  g_return_val_if_fail(key, FALSE);

  cmd = sqlite3_mprintf("SELECT VALUE FROM DATA\n"
			"WHERE KEY = %Q;", key);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    ret = TRUE;

  sqlite3_finalize(stmt);

  return ret;
}

void
CCL_data_key_delete(const gchar * key)
{
  gchar *cmd = NULL;

  g_return_if_fail(key);

  cmd = sqlite3_mprintf("DELETE FROM DATA\n"
			"WHERE KEY = %Q;", key);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

/**************
* CLIENTSDATA *
**************/

void
CCL_clientdata_set_string(gint client, const gchar * key, const gchar * value)
{
  _thingdata_set_string(CLIENTSDATA, client, key, value);
}

gchar *
CCL_clientdata_get_string(gint client, const char * key, const char * defvalue)
{
  return _thingdata_get_string(CLIENTSDATA, client, key, defvalue);
}

void
CCL_clientdata_set_int(gint client, const gchar * key, gint value)
{
  _thingdata_set_int(CLIENTSDATA, client, key, value);
}

gint
CCL_clientdata_get_int(gint client, const gchar * key, gint defval)
{
  return _thingdata_get_int(CLIENTSDATA, client, key, defval);
}

void
CCL_clientdata_set_blob(gint client, const gchar * key, void * value, gint size)
{
  _thingdata_set_blob(CLIENTSDATA, client, key, value, size);
}

void *
CCL_clientdata_get_blob(gint client, const gchar * key, int * size)
{
  return _thingdata_get_blob(CLIENTSDATA, client, key, size);
}

gboolean
CCL_clientdata_key_exists(gint client, const gchar * key)
{
  return _thingdata_key_exists(CLIENTSDATA, client, key);
}

void
CCL_clientdata_key_delete(gint client, const gchar * key)
{
  _thingdata_key_delete(CLIENTSDATA, client, key);
}

/**************
* MEMBERSDATA *
**************/

void
CCL_memberdata_set_string(gint member, const gchar * key, const gchar * value)
{
  _thingdata_set_string(MEMBERSDATA, member, key, value);
}

gchar *
CCL_memberdata_get_string(gint member, const char * key, const char * defvalue)
{
  return _thingdata_get_string(MEMBERSDATA, member, key, defvalue);
}

void
CCL_memberdata_set_int(gint member, const gchar * key, gint value)
{
  _thingdata_set_int(MEMBERSDATA, member, key, value);
}

gint
CCL_memberdata_get_int(gint member, const gchar * key, gint defval)
{
  return _thingdata_get_int(MEMBERSDATA, member, key, defval);
}

void
CCL_memberdata_set_blob(gint member, const gchar * key, void * value, gint size)
{
  _thingdata_set_blob(MEMBERSDATA, member, key, value, size);
}

void *
CCL_memberdata_get_blob(gint member, const gchar * key, int * size)
{
  return _thingdata_get_blob(MEMBERSDATA, member, key, size);
}

gboolean
CCL_memberdata_key_exists(gint member, const gchar * key)
{
  return _thingdata_key_exists(MEMBERSDATA, member, key);
}

void
CCL_memberdata_key_delete(gint member, const gchar * key)
{
  _thingdata_key_delete(MEMBERSDATA, member, key);
}

/*********************************************************/

static void
_thingdata_set_string(gint thing, gint id, const gchar * key,
		      const gchar * value)
{
  gchar *cmd = NULL;

  g_assert(thing >= 0 && thing < _LAST);
  g_return_if_fail(_thing_exists[thing](id));
  g_return_if_fail(key);
  
  if (_thingdata_key_exists(thing, id, key))
    cmd = sqlite3_mprintf("UPDATE %s\n"
			  "SET VALUE = %Q\n"
			  "WHERE ID = %d AND KEY = %Q;",
			  tables[thing], value, id, key);
  else
    cmd = sqlite3_mprintf("INSERT INTO %s\n"
			  "(ID, KEY, VALUE)\n"
			  "VALUES(%d, %Q, %Q);",
			  tables[thing], id, key, value);

  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

static gchar *
_thingdata_get_string(gint thing, gint id, const char * key,
		      const char * defvalue)
{
  gchar *cmd = NULL;
  gchar *ret = NULL;
  sqlite3_stmt *stmt = NULL;

  g_return_val_if_fail(_thing_exists[thing](id), NULL);
  g_return_val_if_fail(key, NULL);

  cmd = sqlite3_mprintf("SELECT VALUE FROM %s\n"
			"WHERE ID = %d AND KEY = %Q;",
			tables[thing], id, key);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    ret = g_strdup((gchar *) sqlite3_column_text(stmt, 0));

  sqlite3_finalize(stmt);

  if (!ret && defvalue)
    return g_strdup(defvalue);
  else
    return ret;
}

static void
_thingdata_set_int(gint thing, gint id, const gchar * key, gint value)
{
  gchar *cmd = NULL;
  
  g_return_if_fail(_thing_exists[thing](id));
  g_return_if_fail(key);

  if (_thingdata_key_exists(thing, id, key))
    cmd = sqlite3_mprintf("UPDATE %s\n"
			  "SET VALUE = %d\n"
			  "WHERE ID = %d, KEY = %Q;",
			  tables[thing], value, id, key);
  else
    cmd = sqlite3_mprintf("INSERT INTO %s\n"
			  "(ID, KEY, VALUE)\n"
			  "VALUES(%d, %Q, %d);",
			  tables[thing], id, key, value);

  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

static gint
_thingdata_get_int(gint thing, gint id, const gchar * key, gint defval)
{
  gchar *cmd = NULL;
  gint ret = defval;
  sqlite3_stmt *stmt = NULL;

  g_return_val_if_fail(_thing_exists[thing](id), -1);
  g_return_val_if_fail(key, -1);

  cmd = sqlite3_mprintf("SELECT VALUE FROM %s\n"
			"WHERE ID = %d AND KEY = %Q;",
			tables[thing], id, key);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    ret = sqlite3_column_int(stmt, 0);

  sqlite3_finalize(stmt);

  return ret;
}

static void
_thingdata_set_blob(gint thing, gint id, const gchar * key, void * value,
		    gint size)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  
  g_return_if_fail(_thing_exists[thing](id));
  g_return_if_fail(key);

  if (_thingdata_key_exists(thing, id, key))
    cmd = sqlite3_mprintf("UPDATE %s\n"
			  "SET VALUE = ?1\n"
			  "WHERE ID = %d AND KEY = %Q;",
			  tables[thing], id, key);
  else
    cmd = sqlite3_mprintf("INSERT INTO %s\n"
			  "(ID, KEY, VALUE)\n"
			  "VALUES(%d, %Q, ?1);",
			  tables[thing], id, key);

  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  sqlite3_bind_blob(stmt, 1, value, size, SQLITE_TRANSIENT);
  sqlite3_step(stmt);
  sqlite3_finalize(stmt);
}

static void *
_thingdata_get_blob(gint thing, gint id, const gchar * key, int * size)
{
  gchar *cmd = NULL;
  void *ret = NULL;
  sqlite3_stmt *stmt = NULL;

  g_return_val_if_fail(_thing_exists[thing](id), NULL);
  g_return_val_if_fail(key, NULL);
  g_return_val_if_fail(size, NULL);

  cmd = sqlite3_mprintf("SELECT VALUE FROM %s\n"
			"WHERE ID = %d AND KEY = %Q;",
			tables[thing], id, key);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    {
      *size = sqlite3_column_bytes(stmt, 0);
      ret = g_malloc0(*size);
      memcpy(ret, sqlite3_column_blob(stmt, 0), *size);
    }

  sqlite3_finalize(stmt);

  return ret;
}

static gboolean
_thingdata_key_exists(gint thing, gint id, const gchar * key)
{
  gchar *cmd = NULL;
  gboolean ret = FALSE;
  sqlite3_stmt *stmt = NULL;

  g_return_val_if_fail(_thing_exists[thing](id), FALSE);
  g_return_val_if_fail(key, FALSE);

  cmd = sqlite3_mprintf("SELECT VALUE FROM %s\n"
			"WHERE ID = %d AND KEY = %Q;",
			tables[thing], id, key);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    ret = TRUE;

  sqlite3_finalize(stmt);

  return ret;
}

static void
_thingdata_key_delete(gint thing, gint id, const gchar * key)
{
  gchar *cmd = NULL;

  g_return_if_fail(_thing_exists[thing](id));
  g_return_if_fail(key);

  cmd = sqlite3_mprintf("DELETE FROM %s\n"
			"WHERE ID = %d AND KEY = %Q;",
			tables[thing], id, key);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}
