#ifndef CCLS_H
#define CCLS_H

#include <time.h>

#define SUN		(1<<0)
#define MON		(1<<1)
#define TUE		(1<<2)
#define WED		(1<<3)
#define THU		(1<<4)
#define FRI		(1<<5)
#define SAT		(1<<6)

#define DELETEDPRODUCT	(0x80000000)

enum
{ INACTIVE = 0, ACTIVE, PAUSED };
enum
{ NOTPAID = 0, PAID, CANCELED };

/* Rules for searching the sessions log */
#define SR_SESSION	  (1<<0)
#define SR_CLIENT	  (1<<1)
#define SR_MEMBER	  (1<<2)
#define SR_STIMEMIN	  (1<<3)
#define SR_STIMEMAX	  (1<<4)
#define SR_ETIMEMIN	  (1<<5)
#define SR_ETIMEMAX	  (1<<6)
#define SR_TIMEMIN	  (1<<7)
#define SR_TIMEMAX	  (1<<8)
#define SR_PRICEMIN	  (1<<9)
#define SR_PRICEMAX	  (1<<10)
#define SR_PAIDMODE	  (1<<11)
#define SR_DAYTIME_RANGE  (1<<12)
#define SR_DAYS		  (1<<13)
#define SR_STIME_RANGE	  (SR_STIMEMIN|SR_STIMEMAX)
#define SR_ETIME_RANGE	  (SR_ETIMEMIN|SR_ETIMEMAX)
#define SR_TIME_RANGE	  (SR_TIMEMIN|SR_TIMEMAX)
#define SR_PRICE_RANGE	  (SR_PRICEMIN|SR_PRICEMAX)

/********************* Error codes ************************/
#define CCL_ERROR_NO_ERROR		0
#define CCL_ERROR_BAD_PASSWORD		1
#define CCL_ERROR_COULD_NOT_LOAD_VL	2

/******************* MD5 Digest Length ********************/
#define CCL_MD5_DIGEST_LENGTH		16

/******************** Callback Types **********************/
typedef void (*on_event_cb) (int client, unsigned cmd, void * data,
			     unsigned datasize, void * userdata);
typedef void (*on_connect_cb) (int client, void * userdata);
typedef void (*on_disconnect_cb) (int client, void * userdata);

struct _CCL_log_search_rules /* Log search rules struct */
{
  unsigned rulemask;
  int session;
  int client;
  int member;
  int product;
  time_t stime_min, stime_max;
  time_t etime_min, etime_max;
  time_t time_min, time_max;
  unsigned daytime_min, daytime_max;
  unsigned days;
  unsigned price_min, price_max;
  int paidmode;
};
typedef struct _CCL_log_search_rules CCL_log_search_rules;

struct _CCL_log_session_entry  /* Logged clients session struct */
{
  int session;
  int client;
  int member;
  time_t stime;
  time_t etime;
  int time;
  unsigned price;
  int paid;
};
typedef struct _CCL_log_session_entry CCL_log_session_entry;

struct _CCL_log_product_entry /* Logged sold product entry */
{
  int session;
  int client;
  int member;
  int product;
  unsigned amount;
  unsigned price;
  time_t time;
};
typedef struct _CCL_log_product_entry CCL_log_product_entry;

struct _CCL_log_expense_entry /* Logged expense entry */
{
  const char description[128];
  time_t time;
  unsigned cash;
};
typedef struct _CCL_log_expense_entry CCL_log_expense_entry;

/**********************************************************/
#ifdef __cplusplus
extern "C"
{
#endif
int	      CCL_init(const char * dbfile);
int	      CCL_shutdown(void);
void	      CCL_set_on_event_callback(on_event_cb callback, void * userdata);
void	      CCL_set_on_connect_callback(on_connect_cb callback,
					  void * userdata);
void	      CCL_set_on_disconnect_callback(on_disconnect_cb callback,
					     void * userdata);
int	      CCL_SSL_init(const char * cafile, const char * certfile,
			   const char * certpass, int * error);
int	      CCL_networking_init(unsigned short port, int * error);
int	      CCL_networking_shutdown(void);
int	      CCL_check_events(void);
int	      CCL_product_new(const char * category, const char * name,
			      unsigned price);
void	      CCL_product_delete(int id);
int	      CCL_product_price_set(int id, unsigned price);
int	      CCL_product_get_nth(unsigned nth);
int	      CCL_product_info_get(int id, char **category,
				   char **name, unsigned * price);
void	      CCL_product_sell(int id, unsigned amount);
void	      CCL_product_stock_set(int id, int amount);
int	      CCL_product_stock_get(int id);
void	      CCL_perminafter_set(int mins);
int	      CCL_perminafter_get(void);
/**********************************************************/
void	      CCL_data_set_string(const char * key, const char * value);
char *	      CCL_data_get_string(const char * key, const char * defval);
void	      CCL_data_set_int(const char * key, int value);
int	      CCL_data_get_int(const char * key, int defval);
void	      CCL_data_set_blob(const char * key, void * value, int size);
void *	      CCL_data_get_blob(const char * key, int * size);
int	      CCL_data_key_exists(const char * key);
void	      CCL_data_key_delete(const char * key);
/**********************************************************/
void	      CCL_clientdata_set_string(int client, const char * key,
					const char * value);
char *	      CCL_clientdata_get_string(int client, const char * key,
					const char * defval);
void	      CCL_clientdata_set_int(int client, const char * key,
				     int value);
int	      CCL_clientdata_get_int(int client, const char * key,
				     int defval);
void	      CCL_clientdata_set_blob(int client, const char * key,
				      void * value, int size);
void *	      CCL_clientdata_get_blob(int client, const char * key,
				      int * size);
int	      CCL_clientdata_key_exists(int client, const char * key);
void	      CCL_clientdata_key_delete(int client, const char * key);
/**********************************************************/
void	      CCL_memberdata_set_string(int member, const char * key,
					const char * value);
char *	      CCL_memberdata_get_string(int member, const char * key,
					const char * defval);
void	      CCL_memberdata_set_int(int member, const char * key,
				     int value);
int	      CCL_memberdata_get_int(int member, const char * key,
				     int defval);
void	      CCL_memberdata_set_blob(int member, const char * key,
				      void * value, int size);
void *	      CCL_memberdata_get_blob(int member, const char * key,
				      int * size);
int	      CCL_memberdata_key_exists(int member, const char * key);
void	      CCL_memberdata_key_delete(int member, const char * key);
/**********************************************************/
void	      CCL_free(void * mem);
unsigned char *
	      CCL_MD5(const unsigned char * d, unsigned long n,
		      unsigned char * md);
/**********************************************************/
int	      CCL_client_new(const char * name);
int	      CCL_client_exists(int client);
int	      CCL_client_find(const char * name);
int	      CCL_client_get_nth(unsigned nth);
void	      CCL_client_reset(int client);
const char *  CCL_client_name_get(int client);
int	      CCL_client_status_get(int client);
void	      CCL_client_start(int client);
void	      CCL_client_stop(int client);
void	      CCL_client_unstop(int client);
void	      CCL_client_pause(int client);
void	      CCL_client_unpause(int client);
void	      CCL_client_swap(int lclient, int rclient);
void	      CCL_client_product_add(int client, int product, unsigned num);
void	      CCL_client_product_sub(int client, int product, unsigned num);
unsigned      CCL_client_product_amount_get(int client, int product);
void	      CCL_client_product_amount_set(int client, int product,
					    unsigned num);
int	      CCL_client_product_get_nth(int client, unsigned nth,
					 int * product, unsigned * num);
void	      CCL_client_timeout_set(int client, time_t timeout);
time_t	      CCL_client_timeout_get(int client);
time_t	      CCL_client_time_used(int client);
time_t	      CCL_client_time_left(int client);
time_t	      CCL_client_stime_get(int client);
time_t	      CCL_client_etime_get(int client);
int	      CCL_client_intervals_get_num(int client);
int	      CCL_client_interval_get_nth(int client, unsigned nth,
					  time_t * stime, time_t * etime);
void	      CCL_client_flags_set(int client, unsigned flags);
unsigned      CCL_client_flags_get(int client);
void	      CCL_client_flags_toggle(int client, unsigned flags, int on);
int	      CCL_client_flags_areset(int client, unsigned flags);
void *	      CCL_client_data_set(int client, void * data);
void *	      CCL_client_data_get(int client);
unsigned      CCL_client_owed_terminal(int client);
unsigned      CCL_client_owed_products(int client);
void	      CCL_client_member_set(int client, int member);
int	      CCL_client_member_get(int client);
void	      CCL_client_send_cmd(int client, unsigned cmd,
				  const void * data, unsigned datasize);
/**********************************************************/
int	      CCL_member_new(const char * name);
int	      CCL_member_get_nth(unsigned nth);
int	      CCL_member_exists(int member);
int	      CCL_member_find(const char * name);
const char *  CCL_member_name_get(int member);
void	      CCL_member_name_set(int member, const char * name);
int	      CCL_member_tarif_get(int member);
void	      CCL_member_tarif_set(int member, int tarif);
const char *  CCL_member_phone_get(int member);
void	      CCL_member_phone_set(int member, const char * phone);
const char *  CCL_member_email_get(int member);
void	      CCL_member_email_set(int member, const char * email);
void *	      CCL_member_data_set(int member, void * data);
void *	      CCL_member_data_get(int member);
void	      CCL_member_flags_set(int member, unsigned flags);
unsigned      CCL_member_flags_get(int member);
void	      CCL_member_flags_toggle(int member, unsigned flags, int on);
int	      CCL_member_flags_areset(int member, unsigned flags);

/**********************************************************/
void	      CCL_log_expense(const char description[128], unsigned cash);
int	      CCL_log_expenses_get(CCL_log_search_rules * sr,
				   CCL_log_expense_entry ** ee);
int	      CCL_log_session(int client, int paid, unsigned price);
int	      CCL_log_session_find(int client, time_t stime, time_t etime);
void	      CCL_log_session_clear(int session);
void	      CCL_log_session_set_paid(int session, int paid, unsigned price);
void	      CCL_log_session_set_member(int session, int member);
int	      CCL_log_sessions_get(CCL_log_search_rules * sr,
				   CCL_log_session_entry ** se);
int	      CCL_log_session_intervals_get(int session, time_t ** intervals);
int	      CCL_log_products_get(CCL_log_search_rules * sr,
				   CCL_log_product_entry ** pe);

/**********************************************************/
int	      CCL_tarif_new(unsigned hr, unsigned min, unsigned days,
			    unsigned hourprice);
void	      CCL_tarif_delete(int tarif);
int	      CCL_tarif_get_nth(unsigned nth);
void	      CCL_tarif_rebuild(void);
void	      CCL_tarif_rebuild_all(void);
int	      CCL_tarif_set(int tarif);
int	      CCL_tarif_get(void);
int	      CCL_tarif_exists(int tarif);
unsigned      CCL_tarif_calc(time_t start, time_t end, int permin);
unsigned      CCL_tarif_calc_with_tarifpart(int id, unsigned mins,
					    int permin);

/**********************************************************/
int	      CCL_tarifpart_new(unsigned hr, unsigned min, unsigned days,
				unsigned hourprice);
void	      CCL_tarifpart_delete(int id);
int	      CCL_tarifpart_get_nth(unsigned nth);
int	      CCL_tarifpart_exists(int id);
int	      CCL_tarifpart_id_get(unsigned hr, unsigned min, unsigned days);
int	      CCL_tarifpart_conflicts(unsigned hr, unsigned min,
				      unsigned days, unsigned * conflicts); 
int	      CCL_tarifpart_info_get(int id, unsigned * hr, unsigned * min,
				     unsigned * days, unsigned * hourprice);
void	      CCL_tarifpart_stime_set(int id, unsigned hr, unsigned min);
void	      CCL_tarifpart_stime_get(int id, unsigned * hr, unsigned * min);
void	      CCL_tarifpart_days_set(int id, unsigned days);
unsigned      CCL_tarifpart_days_get(int id);
void	      CCL_tarifpart_hourprice_set(int id, unsigned price);
unsigned      CCL_tarifpart_hourprice_get(int id);
void	      CCL_tarifpart_price_add(int id, unsigned mins, unsigned price);
void	      CCL_tarifpart_price_del(int id, unsigned mins);
void	      CCL_tarifpart_price_clear(int id);
unsigned      CCL_tarifpart_price_get(int id, unsigned mins, int strict);
int	      CCL_tarifpart_price_exists(int id, unsigned mins);
unsigned      CCL_tarifpart_price_get_nearest(int id, unsigned mins);
unsigned      CCL_tarifpart_price_get_last(int id);
int	      CCL_tarifpart_price_get_nth(int id, unsigned nth,
					  unsigned * mins, unsigned * price);
#ifdef __cplusplus
}
#endif

#endif
