#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "ccls.h"
#include "ccl_private.h"

/* Static functions */
static void _init_db(sqlite3 * db);
static gint _loadMembersCB(gpointer ptr, gint argc,
			   gchar ** argv, gchar ** colnames);
static gint _loadClientsCB(gpointer ptr, gint argc,
			   gchar ** argv, gchar ** colnames);
static void _shutdown_client_connection(gint clientid);
static void _FindClientByFdFunc(GQuark key_id, gpointer data,
				gpointer user_data);
static void _ShutdownClientConnectionFunc(GQuark key_id, gpointer data,
					  gpointer user_data);
static SSL_CTX * _initialize_ssl_ctx(const gchar * cafile,
				     const gchar * keyfile,
				     const gchar * password, gint * err);

CCL *ccl = NULL;

/* Public interface */

gboolean
CCL_init(const gchar * dbfile)
{
  if (ccl)
    return TRUE;
  if (NULL == (ccl = g_new0(CCL, 1)))
    return FALSE;
  if (SQLITE_OK != sqlite3_open(dbfile, &(ccl->db)))
    return FALSE;
  _init_db(ccl->db);
  ccl->events.listenfd = INVALID_SOCKET;
  ccl->perminafter = 60;
  ccl->tarif = NULL;
  g_datalist_init(&(ccl->members));
  g_datalist_init(&(ccl->clients));

  SSL_library_init();
  SSL_load_error_strings();

  sqlite3_exec(ccl->db,
	       "SELECT ID, NAME FROM MEMBERS;",
	       _loadMembersCB, NULL, NULL);
  sqlite3_exec(ccl->db,
	       "SELECT ID, NAME FROM CLIENTS;",
	       _loadClientsCB, NULL, NULL);
 
  return TRUE;
}

gboolean
CCL_shutdown(void)
{
  g_datalist_clear(&(ccl->clients));
  sqlite3_close(ccl->db);
  CCL_networking_shutdown();
  g_free(ccl->events.certpass);
  if (ccl->events.ssl_ctx) SSL_CTX_free(ccl->events.ssl_ctx);
  g_free(ccl);

  return FALSE;
}

void
CCL_set_on_event_callback(on_event_cb callback, void * userdata)
{
  ccl->events.on_event = callback;
  ccl->events.on_event_data = userdata;
}

void
CCL_set_on_connect_callback(on_connect_cb callback, void * userdata)
{
  ccl->events.on_connect = callback;
  ccl->events.on_connect_data = userdata;
}

void
CCL_set_on_disconnect_callback(on_disconnect_cb callback, void * userdata)
{
  ccl->events.on_disconnect = callback;
  ccl->events.on_disconnect_data = userdata;
}

gboolean
CCL_SSL_init(const gchar * cafile, const gchar * certfile,
	     const gchar * certpass, gint * error)
{
  if (!ccl->events.ssl_ctx)
    if (!(ccl->events.ssl_ctx = _initialize_ssl_ctx(cafile, certfile, certpass,
						    error)))
      return FALSE;

  return TRUE;
}

gboolean
CCL_networking_init(gushort port, gint * error)
{
  gchar portstr[256];

  if (ccl->events.listenbio) /* Already initialized */
    return TRUE;

  g_snprintf(portstr, sizeof(portstr)/sizeof(gchar), "%u", port);

  ccl->events.listenbio = BIO_new_accept(portstr);
  BIO_set_bind_mode(ccl->events.listenbio, BIO_BIND_REUSEADDR_IF_UNUSED);
  BIO_do_accept(ccl->events.listenbio);
  ccl->events.listenfd = BIO_get_fd(ccl->events.listenbio, NULL);

  if (ccl->events.maxfd < ccl->events.listenfd)
    ccl->events.maxfd = ccl->events.listenfd;
  
  FD_SET(ccl->events.listenfd, &(ccl->events.readfds));

  return TRUE;
}

gboolean
CCL_networking_shutdown(void)
{
  g_datalist_foreach(&(ccl->clients), _ShutdownClientConnectionFunc, NULL);

  if (ccl->events.listenbio)
    {
      BIO_free(ccl->events.listenbio);
      ccl->events.listenbio = NULL;
      ccl->events.listenfd = INVALID_SOCKET;
    }

  return TRUE;
}

void
CCL_perminafter_set(gint mins)
{
  ccl->perminafter = mins;
}

gint
CCL_perminafter_get(void)
{
  return ccl->perminafter;
}

gint
CCL_product_new(const gchar * category, const gchar * name, guint price)
{
  gchar *cmd = NULL;
  gint id;
  gint i;
  
  for (i = 0; -1 != (id = CCL_product_get_nth(i)); i++)
    ;
  id = 1 + CCL_product_get_nth(i - 1);
  
  if (0 >= id) id = 1; /* Make the first one have id 1 instead of 0 */
  
  cmd = sqlite3_mprintf("INSERT INTO PRODUCTS\n"
			"(ID, NAME, CATEGORY, PRICE, STOCK)\n"
			"VALUES(%d, %Q, %Q, %u.%.2u, %u);",
			id, name, category, price / 100, price % 100, 0);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);

  return id;
}

void
CCL_product_delete(gint id)
{
  gchar *cmd = NULL;
  
  cmd = sqlite3_mprintf("UPDATE PRODUCTS SET STOCK = %d\n"
			"WHERE ID = %d;", DELETEDPRODUCT, id);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

gint
CCL_product_price_set(gint id, guint price)
{
  gchar *cmd = NULL;

  if (!CCL_product_info_get(id, NULL, NULL, NULL))
    return FALSE;
  else
    {
      cmd = sqlite3_mprintf("UPDATE PRODUCTS\n"
			    "SET PRICE = %u.%.2u WHERE ID = %d;",
			    price / 100, price % 100, id);
      sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
      sqlite3_free(cmd);

      return TRUE;
    }
}

gint
CCL_product_get_nth(guint nth)
{
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;
  gint i;
  gint id = -1;

  cmd = sqlite3_mprintf("SELECT ID FROM PRODUCTS WHERE STOCK != %d\n"
			"ORDER BY ID ASC;", DELETEDPRODUCT);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);
  for (i = nth; i > 0; i--)
    sqlite3_step(stmt);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    id = sqlite3_column_int(stmt, 0);

  sqlite3_finalize(stmt);

  return id;
}

gboolean
CCL_product_info_get(gint id, char **category, char **name, guint * price)
{
  gboolean retval = FALSE;
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;

  cmd = sqlite3_mprintf("SELECT CATEGORY, NAME, PRICE FROM PRODUCTS\n"
			"WHERE ID = %d;", id);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    {
      if (category)
	*category = g_strdup((gchar *) sqlite3_column_text(stmt, 0));
      if (name)
	*name = g_strdup((gchar *) sqlite3_column_text(stmt, 1));
      if (price)
	*price = (guint) (100 * sqlite3_column_double(stmt, 2));
      retval = TRUE;
    }
  sqlite3_finalize(stmt);

  return retval;
}

void
CCL_product_sell(gint id, guint amount)
{
  gchar *cmd = NULL;
  guint price = 0;

  if (CCL_product_info_get(id, NULL, NULL, &price))
    {
      cmd = sqlite3_mprintf("INSERT INTO PRODUCTSLOG\n"
			    "(SESSION,CLIENT,MEMBER,PRODUCT,AMOUNT,PRICE,TIME)"
			    "\nVALUES(%d, %d, %d, %d, %u, %u, %ld);", 0, 0, 0,
			    id, amount, price * amount, time(NULL));
      sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
      sqlite3_free(cmd);
      if (CCL_product_stock_get(id) != DELETEDPRODUCT)
	CCL_product_stock_set(id, CCL_product_stock_get(id) - amount);
    }
}

void
CCL_product_stock_set(gint id, gint amount)
{
  gchar *cmd = NULL;

  cmd = sqlite3_mprintf("UPDATE PRODUCTS SET STOCK = %d\n"
			"WHERE ID = %d;", amount, id);
  sqlite3_exec(ccl->db, cmd, NULL, NULL, NULL);
  sqlite3_free(cmd);
}

gint
CCL_product_stock_get(gint id)
{
  gint amount = 0;
  gchar *cmd = NULL;
  sqlite3_stmt *stmt = NULL;

  cmd = sqlite3_mprintf("SELECT STOCK FROM PRODUCTS WHERE ID = %d;", id);
  sqlite3_prepare(ccl->db, cmd, -1, &stmt, NULL);
  sqlite3_free(cmd);

  if (sqlite3_step(stmt) == SQLITE_ROW)
    amount = sqlite3_column_int(stmt, 0);

  sqlite3_finalize(stmt);

  return amount;
}

gboolean
CCL_check_events(void)
{
  struct timeval delta;
  gint maxfd = ccl->events.maxfd;
  gint nfds;
  gint fd;
  fd_set readfds = ccl->events.readfds;

  if (!maxfd || !(ccl->events.listenbio))
    return FALSE;

  delta.tv_usec = 0;
  delta.tv_sec = 0;

  nfds = select(maxfd + 1, &readfds, NULL, NULL, &delta);

  if (nfds <= 0)
    return FALSE;
  
  for (fd = 0; fd <= maxfd; fd++)
    {
      if (FD_ISSET(fd, &readfds) && fd != ccl->events.listenfd)
	{
	  gint fd_cid[2];
	  gint client = 0;
	  gint sockfd = 0;

	  fd_cid[0] = fd;
	  fd_cid[1] = 0;

	  g_datalist_foreach(&(ccl->clients), _FindClientByFdFunc,
			     (void *)fd_cid);
	  sockfd = fd_cid[0];
	  client = fd_cid[1];

	  if (client)
	    {
	      gboolean connection_closed = FALSE;
	      guint cmd = 0;
	      guint size = 0;
	      void *data = NULL;
	      gint bytes = 0;
	      CCL_client *c = g_datalist_id_get_data(&(ccl->clients), client);
	      BIO *bio = c->bio;
	      
	      bytes = _recvall(bio, &cmd, sizeof(cmd));
	      if (0 >= bytes || bytes != sizeof(cmd))
		connection_closed = TRUE;
	     
	      if (!connection_closed)
		{
		  bytes = _recvall(bio, &size, sizeof(size));
		  if (0 >= bytes || bytes != sizeof(size))
		    connection_closed = TRUE;
		}
	      
	      if (!connection_closed && 0 < cmd && 0 < size)
		{
		  data = g_malloc0(size);
		  bytes = _recvall(bio, data, size);
		  if (0 >= bytes || bytes != size)
		    connection_closed = TRUE;
		}
	      if (ccl->events.on_event)
		ccl->events.on_event(client, cmd, data, size,
				     ccl->events.on_event_data);
	      if (data)
		g_free(data);
	      /* If the connection broke, or was closed */
	      if (connection_closed)
		{
		  _shutdown_client_connection(client);
		  if (ccl->events.on_disconnect)
		    ccl->events.on_disconnect(client,
					      ccl->events.on_disconnect_data);
		}
	    }
	}
    }

  /* Now i am going to check for new connections */
  fd = ccl->events.listenfd;

  if (FD_ISSET(fd, &readfds))
    {
      SSL *ssl;
      BIO *newbio;
      fd_set rfd;

      FD_ZERO(&rfd);

      if (1 == BIO_do_accept(ccl->events.listenbio))
	{
	  newbio = BIO_pop(ccl->events.listenbio);
	  if (ccl->events.ssl_ctx)
	    {
	      ssl = SSL_new(ccl->events.ssl_ctx);
	      SSL_set_fd(ssl, BIO_get_fd(newbio, NULL));
	      BIO_set(newbio, BIO_f_ssl());
	      BIO_set_ssl(newbio, ssl, 0);
	      BIO_set_ssl_mode(newbio, FALSE);
	    }
	  FD_SET(BIO_get_fd(newbio, NULL), &rfd);

	  /* Only perform the handshake if this is a genuine connection, 
	   * I dont want to block forever, it the connection is not valid */
	  if (!select(BIO_get_fd(newbio, NULL) + 1, &rfd, NULL, NULL, &delta))
	    BIO_free(newbio);
	  else if (!ccl->events.ssl_ctx || 1 == BIO_do_handshake(newbio))
	    {
	      gchar *name = NULL;
	      gint size;
	      CCL_client *client = NULL;
	      gint id;

	      BIO_read(newbio, &size, sizeof(size));
	      name = g_malloc0(size);
	      BIO_read(newbio, name, size);
	  
	      id = CCL_client_new(name);
	      g_free(name);

	      client = g_datalist_id_get_data(&(ccl->clients), id);

	      /* If a connection for this client already exists, lets
	       * make sure, that our old connection still exists.
	       * If not, then free it, and set it as disconnected */
	      if (INVALID_SOCKET != client->sockfd)
		{
		  fd_set wfd;

		  FD_ZERO(&wfd);
		  FD_SET(client->sockfd, &wfd);
		  
		  if (!select(client->sockfd + 1,
			      NULL, &wfd, NULL, &delta))
		    {
		      BIO_free(client->bio);
		      client->sockfd = INVALID_SOCKET;
		    }
		}

	      if (INVALID_SOCKET == client->sockfd)
		{
		  client->bio = newbio;
		  client->sockfd = BIO_get_fd(newbio, NULL);
		  FD_SET(client->sockfd, &(ccl->events.readfds));
		  
		  if (ccl->events.maxfd < client->sockfd)
		    ccl->events.maxfd = client->sockfd;
		  if (ccl->events.on_connect)
		    ccl->events.on_connect(id,
					   ccl->events.on_connect_data);
		}
	      else
		BIO_free(newbio);
	    }
	}
    }
  
  return TRUE;
}

/**********************************************************/

static void
_init_db(sqlite3 * db)
{
  sqlite3_stmt *stmt = NULL;
  gboolean DATA = FALSE;
  gboolean CLIENTSDATA = FALSE;
  gboolean MEMBERSDATA = FALSE;
  gboolean CLIENTS = FALSE;
  gboolean MEMBERS = FALSE;
  gboolean PRICES = FALSE;
  gboolean PRODUCTS = FALSE;
  gboolean TARIFS = FALSE;
  gboolean SESSIONSLOG = FALSE;
  gboolean PRODUCTSLOG = FALSE;
  gboolean EXPENSESLOG = FALSE;

  sqlite3_prepare(db,
		  "SELECT name FROM"
		  "  (SELECT * FROM sqlite_master UNION ALL"
		  "   SELECT * FROM sqlite_temp_master)"
		  "WHERE type='table' ORDER BY name;", -1, &stmt, NULL);

  while (sqlite3_step(stmt) == SQLITE_ROW)
    {
      if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
			      "DATA"))
	DATA = TRUE;
      else if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
				   "CLIENTSDATA"))
	CLIENTSDATA = TRUE;
      else if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
				   "MEMBERSDATA"))
	MEMBERSDATA = TRUE;
      else if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
				   "CLIENTS"))
	CLIENTS = TRUE;
      else if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
				   "MEMBERS"))
	MEMBERS = TRUE;
      else if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
				   "PRICES"))
	PRICES = TRUE;
      else if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
				   "PRODUCTS"))
	PRODUCTS = TRUE;
      else if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
				   "TARIFS"))
	TARIFS = TRUE;
      else if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
				   "SESSIONSLOG"))
	SESSIONSLOG = TRUE;
      else if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
				   "PRODUCTSLOG"))
	PRODUCTSLOG = TRUE;
      else if (!g_ascii_strcasecmp((gchar *) sqlite3_column_text(stmt, 0),
				   "EXPENSESLOG"))
	EXPENSESLOG = TRUE;
    }
  
  sqlite3_finalize(stmt);
 
  if (!DATA)
    sqlite3_exec(db,
		 "CREATE TABLE DATA (\n"
		 "    KEY VARCHAR(256) NOT NULL UNIQUE,\n"
		 "    VALUE BLOB);", NULL, NULL, NULL);
  if (!CLIENTSDATA)
    sqlite3_exec(db,
		 "CREATE TABLE CLIENTSDATA (\n"
		 "    ID INTEGER NOT NULL,\n"
		 "    KEY VARCHAR(256) NOT NULL,\n"
		 "    VALUE BLOB);", NULL, NULL, NULL);
  if (!MEMBERSDATA)
    sqlite3_exec(db,
		 "CREATE TABLE MEMBERSDATA (\n"
		 "    ID INTEGER NOT NULL,\n"
		 "    KEY VARCHAR(256) NOT NULL,\n"
		 "    VALUE BLOB);", NULL, NULL, NULL);
  if (!CLIENTS)
    sqlite3_exec(db,
		 "CREATE TABLE CLIENTS (\n"
		 "    ID INTEGER PRIMARY KEY,\n"
		 "    NAME VARCHAR(128) NOT NULL UNIQUE,\n"
		 "    STATUS INTEGER NOT NULL,\n"
		 "    INTERVALS BLOB,\n"
		 "    PRODUCTS BLOB,\n"
		 "    TIMEOUT INTEGER NOT NULL,\n"
		 "    MEMBER INTEGER NOT NULL,\n"
		 "    FLAGS INTEGER NOT NULL);", NULL, NULL, NULL);
  if (!MEMBERS)
    sqlite3_exec(db,
		 "CREATE TABLE MEMBERS (\n"
		 "    ID INTEGER PRIMARY KEY,\n"
		 "    NAME VARCHAR(128) NOT NULL UNIQUE,\n"
		 "    SDATE INTEGER NOT NULL,\n"
		 "    TARIF INTEGER DEFAULT 0 NOT NULL,\n"
		 "    EMAIL VARCHAR(128),\n"
		 "    PHONE VARCHAR(128),\n"
		 "    FLAGS INTEGER NOT NULL);", NULL, NULL, NULL);
  if (!PRICES)
    sqlite3_exec(db,
		 "CREATE TABLE PRICES (\n"
		 "    TARIF INTEGER NOT NULL,\n"
		 "    ID INTEGER NOT NULL,\n"
		 "    MINS INTEGER NOT NULL,\n"
		 "    PRICE NUMBER NOT NULL);", NULL, NULL, NULL);
  if (!PRODUCTS)
    sqlite3_exec(db,
		 "CREATE TABLE PRODUCTS (\n"
		 "    ID INTEGER PRIMARY KEY,\n"
		 "    NAME VARCHAR(128) NOT NULL,\n"
		 "    CATEGORY VARCHAR(128) NOT NULL,\n"
		 "    PRICE NUMBER NOT NULL,\n"
		 "    STOCK INTEGER);", NULL, NULL, NULL);
  if (!TARIFS)
    sqlite3_exec(db,
		 "CREATE TABLE TARIFS (\n"
		 "    TARIF INTEGER NOT NULL,\n"
		 "    ID INTEGER NOT NULL,\n"
		 "    DAYS INTEGER NOT NULL,\n"
		 "    STIME TIME NOT NULL,\n"
		 "    HOURPRICE NUMBER NOT NULL);", NULL, NULL, NULL);
  if (!SESSIONSLOG)
    sqlite3_exec(db,
		 "CREATE TABLE SESSIONSLOG (\n"
		 "    SESSION INTEGER NOT NULL UNIQUE,\n"
		 "    CLIENT INTEGER NOT NULL,\n"
		 "    MEMBER INTEGER NOT NULL,\n"
		 "    STIME INTEGER NOT NULL,\n"
		 "    ETIME INTEGER NOT NULL,\n"
		 "    TIME INTEGER NOT NULL,\n"
		 "    PRICE INTEGER NOT NULL,\n"
		 "    PAID INTEGER NOT NULL,\n"
		 "    INTERVALS BLOB NOT NULL);", NULL, NULL, NULL);
  if (!PRODUCTSLOG)
    sqlite3_exec(db,
		 "CREATE TABLE PRODUCTSLOG (\n"
		 "    SESSION INTEGER NOT NULL,\n"
		 "    CLIENT INTEGER NOT NULL,\n"
		 "    MEMBER INTEGER NOT NULL,\n"
		 "    PRODUCT INTEGER NOT NULL,\n"
		 "    AMOUNT INTEGER NOT NULL, \n"
		 "    PRICE INTEGER NOT NULL, \n"
		 "    TIME INTEGER NOT NULL);", NULL, NULL, NULL);
  if (!EXPENSESLOG)
    sqlite3_exec(db,
		 "CREATE TABLE EXPENSESLOG (\n"
		 "    DESCRIPTION VARCHAR(127) NOT NULL,\n"
		 "    TIME INTEGER NOT NULL,\n"
		 "    CASH INTEGER NOT NULL);", NULL, NULL, NULL);
}

static gint
_loadMembersCB(void *ptr, int argc, char **argv, char **colnames)
{
  gint id = atoi(argv[0]);
  CCL_member *member = g_new0(CCL_member, 1);

  _CCL_member_init(member, argv[1]);
  g_datalist_id_set_data_full(&(ccl->members), id, member, _destroy_member);
  _CCL_member_restore(id);

  return 0;
}

static gint
_loadClientsCB(void *ptr, int argc, char **argv, char **colnames)
{
  gint id = atoi(argv[0]);
  CCL_client *client = g_new0(CCL_client, 1);

  _CCL_client_init(client, argv[1]);
  g_datalist_id_set_data_full(&(ccl->clients), id, client, _destroy_client);
  _CCL_client_restore(id);

  return 0;
}

static void
_shutdown_client_connection(gint client)
{
  CCL_client *c = (CCL_client *) g_datalist_id_get_data(&ccl->clients,
							client);

  g_assert(c);

  _shutdown_connection(c);
}

static void
_FindClientByFdFunc(GQuark key_id, gpointer data, gpointer user_data)
{
  CCL_client *client = (CCL_client *) data;
  gint *fd_cid = (gint *) user_data;

  if (fd_cid[0] == client->sockfd)
    fd_cid[1] = key_id;
}

static void
_ShutdownClientConnectionFunc(GQuark key_id, gpointer data, gpointer user_data)
{
  CCL_client *client = (CCL_client *) data;

  _shutdown_connection(client);
}

static gint
_cert_verify(gint ok, X509_STORE_CTX * x509_ctx)
{
  gint error = X509_STORE_CTX_get_error(x509_ctx); 

  ERR_load_X509_strings();
  if (!ok)
    printf("error: %d\n%s\n", error, X509_verify_cert_error_string(error));

  return 1;
}

static gint
_passwd_cb(gchar * buf, gint num, gint rw, void * password)
{
  if (!password) password = "_nopassword";

  strncpy(buf, (gchar *) password, num);
  buf[num - 1] = '\0';

  return strlen(buf);
}

static SSL_CTX *
_initialize_ssl_ctx(const char * cafile, const gchar * keyfile,
		    const gchar * certpass, gint * err)
{
  SSL_CTX *ctx;

  if (err) *err = CCL_ERROR_NO_ERROR;

  if (ccl->events.certpass) g_free(ccl->events.certpass);
  ccl->events.certpass = certpass ? g_strdup(certpass) : NULL;

  /* Create our context*/
  ctx = SSL_CTX_new(SSLv23_server_method());
  SSL_CTX_set_default_passwd_cb(ctx, _passwd_cb);
  SSL_CTX_set_default_passwd_cb_userdata(ctx, ccl->events.certpass);

  /* Load our keys and certificates*/
  SSL_CTX_use_certificate_file(ctx, keyfile, SSL_FILETYPE_PEM);
  
  if (1 != SSL_CTX_use_PrivateKey_file(ctx, keyfile, SSL_FILETYPE_PEM))
    {
      if (err) *err = CCL_ERROR_BAD_PASSWORD;
      goto error;
    }
 
  if (1 != SSL_CTX_check_private_key(ctx))
    { 
      if (err) *err = CCL_ERROR_BAD_PASSWORD;
      goto error;
    }
  
  if (1 != SSL_CTX_load_verify_locations(ctx, cafile, NULL))
    {
      if (err) *err = CCL_ERROR_COULD_NOT_LOAD_VL;
      goto error;
    }

  SSL_CTX_set_verify(ctx,
		     SSL_VERIFY_PEER|SSL_VERIFY_CLIENT_ONCE
		     |SSL_VERIFY_FAIL_IF_NO_PEER_CERT, _cert_verify);
  
  return ctx;

error:
  SSL_CTX_free(ctx);

  return NULL;
}
