/*
//  Source code for generic Win32 JPEG DLL
//  This DLL encapsulates the libJPEG functions
//
//  Copyright 1998 M. Scott Heiman
//  All Rights Reserved
//
// You may use the software for any purpose you see fit. You may modify
// it, incorporate it in a commercial application, use it for school,
// even turn it in as homework. You must keep the Copyright in the
// header and source files. This software is not in the "Public Domain".
//You may use this software at your own risk. I have made a reasonable
// effort to verify that this software works in the manner I expect it to;
// however,...
//
// THE MATERIAL EMBODIED ON THIS SOFTWARE IS PROVIDED TO YOU "AS-IS" AND
// WITHOUT WARRANTY OF ANY KIND, EXPRESS, IMPLIED OR OTHERWISE, INCLUDING
// WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR FITNESS FOR A
// PARTICULAR PURPOSE. IN NO EVENT SHALL MICHAEL S. HEIMAN BE LIABLE TO
// YOU OR ANYONE ELSE FOR ANY DIRECT, SPECIAL, INCIDENTAL, INDIRECT OR
// CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER, INCLUDING
// WITHOUT LIMITATION, LOSS OF PROFIT, LOSS OF USE, SAVINGS OR REVENUE,
// OR THE CLAIMS OF THIRD PARTIES, WHETHER OR NOT MICHAEL S. HEIMAN HAS
// BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS, HOWEVER CAUSED AND ON
// ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE
// POSSESSION, USE OR PERFORMANCE OF THIS SOFTWARE.
//
// Portions of this code were copied from the tiff2dib.c written by
// Philippe Tenenhaus.  The tiff2dib file was bundled with the tifflib
// package
//
// Other portions were blatantly stolen from example.c that was
// bundled with the libjpeg source.
*/

#include "myqt.h"

extern MyQtApp *myapp;

#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>

#include <qwindowdefs.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "jpeglib.h"

/* The following array represents the pixel values for each shade
 * of the primary color components.
 * If 'p' is a pointer to a source image rgb-byte-triplet, we can
 * construct the output pixel value simply by 'oring' together
 * the corresponding components:
 *
 *	unsigned char *p;
 *	unsigned long pixval;
 *
 *	pixval  = screen_rgb[0][*p++];
 *	pixval |= screen_rgb[1][*p++];
 *	pixval |= screen_rgb[2][*p++];
 *
 * This is both efficient and generic, since the only assumption
 * is that the primary color components have separate bits.
 * The order and distribution of bits does not matter, and we
 * don't need additional variables and shifting/masking code.
 * The array size is 3 KBytes total and thus very reasonable.
 */

static unsigned long screen_rgb[3][256];

/* The following array holds the exact color representations
 * reported by the system.
 * This is useful for less than 24 bit deep displays as a base
 * for additional dithering to get smoother output.
 */

static unsigned char screen_set[3][256];

/* The following routine initializes the screen_rgb and screen_set
 * arrays.
 * Since it is executed only once per program run, it does not need
 * to be super-efficient.
 *
 * The method is to draw points in a pixmap with the specified shades
 * of primary colors and then get the corresponding XImage pixel
 * representation.
 * Thus we can get away with any Bit-order/Byte-order dependencies.
 *
 * The routine uses some global X variables:
 * theDisp, theScreen, dispDEEP, and theCmap.
 * Adapt these to your application as necessary.
 * I've not passed them in as parameters, since for other platforms
 * than X these may be different (see vfixpix.c), and so the
 * screen_init() interface is unique.
 */

static Display *theDisp;
static int theScreen;
static Colormap theCmap;
static int dispDEEP;
static Visual *theVisual;
  
static void screen_init()
{
  static int init_flag; /* assume auto-init as 0 */
  Pixmap check_map;
  GC check_gc;
  XColor check_col;
  XImage *check_image;
  int ci, i;
  
  if (init_flag) return;
  init_flag = 1;

  theDisp = qt_xdisplay();
  theScreen = qt_xscreen();
  theCmap = DefaultColormap(theDisp,theScreen);
  dispDEEP = DefaultDepth(theDisp,theScreen);
  theVisual = DefaultVisual(theDisp,theScreen);
  check_map = XCreatePixmap(theDisp, RootWindow(theDisp,theScreen),
			    1, 1, dispDEEP);
  check_gc = XCreateGC(theDisp, check_map, 0, NULL);
  for (ci = 0; ci < 3; ci++) {
    for (i = 0; i < 256; i++) {
      check_col.red = 0;
      check_col.green = 0;
      check_col.blue = 0;
      /* Do proper upscaling from unsigned 8 bit (image data values)
	 to unsigned 16 bit (X color representation). */
      ((unsigned short *)&check_col.red)[ci] = (unsigned short)((i << 8) | i);
//      if (theVisual->class == TrueColor)
	XAllocColor(theDisp, theCmap, &check_col);
//      else
//	xvAllocColor(theDisp, theCmap, &check_col);
      screen_set[ci][i] =
	(((unsigned short *)&check_col.red)[ci] >> 8) & 0xff;
      XSetForeground(theDisp, check_gc, check_col.pixel);
      XDrawPoint(theDisp, check_map, check_gc, 0, 0);
      check_image = XGetImage(theDisp, check_map, 0, 0, 1, 1,
			      AllPlanes, ZPixmap);
      if (check_image) {
	switch (check_image->bits_per_pixel) {
	case 8:
	  screen_rgb[ci][i] = *(unsigned char *)check_image->data;
	  break;
	case 16:
	  screen_rgb[ci][i] = *(unsigned short *)check_image->data;
	  break;
	case 24:
	  screen_rgb[ci][i] =
	    ((unsigned long)*(unsigned char *)check_image->data << 16) |
	    ((unsigned long)*(unsigned char *)(check_image->data + 1) << 8) |
	    (unsigned long)*(unsigned char *)(check_image->data + 2);
	  break;
	case 32:
	  screen_rgb[ci][i] = *(unsigned long *)check_image->data;
	  break;
	}
	XDestroyImage(check_image);
      }
    }
  }
  XFreeGC(theDisp, check_gc);
  XFreePixmap(theDisp, check_map);
}


/* The following switch should better be provided at runtime for
 * comparison purposes.
 * At the moment it's only compile time, unfortunately.
 * Who can make adaptions for use as a runtime switch by a menu option?
 */

/* The following code is based in part on:
 *
 * jquant1.c
 *
 * Copyright (C) 1991-1996, Thomas G. Lane.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains 1-pass color quantization (color mapping) routines.
 * These routines provide mapping to a fixed color map using equally spaced
 * color values.  Optional Floyd-Steinberg or ordered dithering is available.
 */

/* Declarations for Floyd-Steinberg dithering.
 *
 * Errors are accumulated into the array fserrors[], at a resolution of
 * 1/16th of a pixel count.  The error at a given pixel is propagated
 * to its not-yet-processed neighbors using the standard F-S fractions,
 *		...	(here)	7/16
 *		3/16	5/16	1/16
 * We work left-to-right on even rows, right-to-left on odd rows.
 *
 * We can get away with a single array (holding one row's worth of errors)
 * by using it to store the current row's errors at pixel columns not yet
 * processed, but the next row's errors at columns already processed.  We
 * need only a few extra variables to hold the errors immediately around the
 * current column.  (If we are lucky, those variables are in registers, but
 * even if not, they're probably cheaper to access than array elements are.)
 *
 * We provide (#columns + 2) entries per component; the extra entry at each
 * end saves us from special-casing the first and last pixels.
 */

typedef INT16 FSERROR;		/* 16 bits should be enough */
typedef int LOCFSERROR;		/* use 'int' for calculation temps */

typedef struct { unsigned char *colorset;
		 FSERROR *fserrors;
	       } FSBUF;

/* Floyd-Steinberg initialization function.
 *
 * It is called 'fs2_init' since it's specialized for our purpose and
 * could be embedded in a more general FS-package.
 *
 * Returns a malloced FSBUF pointer which has to be passed as first
 * parameter to subsequent 'fs2_dither' calls.
 * The FSBUF structure does not need to be referenced by the calling
 * application, it can be treated from the app like a void pointer.
 *
 * The current implementation does only require to free() this returned
 * pointer after processing.
 *
 * Returns NULL if malloc fails.
 *
 * NOTE: The FSBUF structure is designed to allow the 'fs2_dither'
 * function to work with an *arbitrary* number of color components
 * at runtime! This is an enhancement over the IJG code base :-).
 * Only fs2_init() specifies the (maximum) number of components.
 */

FSBUF *fs2_init(int width, int nc)
{
  FSBUF *fs;
  FSERROR *p;
  int i;

  fs = (FSBUF *)
    malloc(sizeof(FSBUF) * nc + ((size_t)width + 2) * sizeof(FSERROR) * nc);
  if (fs == 0) return fs;

  for (i = 0; i < nc; i++) fs[i].colorset = screen_set[i];

  p = (FSERROR *)(fs + nc);
  memset(p, 0, ((size_t)width + 2) * sizeof(FSERROR) * nc);

  for (i = 0; i < nc; i++) fs[i].fserrors = p++;

  return fs;
}

/* Floyd-Steinberg dithering function.
 *
 * NOTE:
 * (1) The image data referenced by 'ptr' is *overwritten* (input *and*
 *     output) to allow more efficient implementation.
 * (2) Alternate FS dithering is provided by the sign of 'nc'. Pass in
 *     a negative value for right-to-left processing. The return value
 *     provides the right-signed value for subsequent calls!
 */

int fs2_dither(FSBUF *fs, unsigned char *ptr, int nc, int num_rows, int num_cols, int num_scan)
{
  int abs_nc, ci, row, col;
  LOCFSERROR delta, cur, belowerr, bpreverr;
  unsigned char *dataptr, *colsetptr;
  FSERROR *errorptr;

  if ((abs_nc = nc) < 0) abs_nc = -abs_nc;

  num_scan -= abs_nc;
  for (row = 0; row < num_rows; row++) {
    for (ci = 0; ci < abs_nc; ci++, ptr++) {
      dataptr = ptr;
      colsetptr = fs[ci].colorset;
      errorptr = fs[ci].fserrors;
      if (nc < 0) {
	dataptr += (num_cols - 1) * abs_nc;
	errorptr += (num_cols + 1) * abs_nc;
      }
      cur = belowerr = bpreverr = 0;
      for (col = 0; col < num_cols; col++) {
	cur += errorptr[nc];
	cur += 8; cur >>= 4;
	if ((cur += *dataptr) < 0) cur = 0;
	else if (cur > 255) cur = 255;
	*dataptr = colsetptr[cur];
	cur -= colsetptr[cur];
	delta = cur << 1; cur += delta;
	bpreverr += cur; cur += delta;
	belowerr += cur; cur += delta;
	errorptr[0] = (FSERROR)bpreverr;
	bpreverr = belowerr;
	belowerr = delta >> 1;
	dataptr += nc;
	errorptr += nc;
      }
      errorptr[0] = (FSERROR)bpreverr;
    }
    ptr += num_scan;
    nc = -nc;
  }
  return nc;
}


/* error handling structures  */

struct win_jpeg_error_mgr {
  struct jpeg_error_mgr pub;	/* public fields */
  jmp_buf setjmp_buffer;		/* for return to caller */
};

typedef struct win_jpeg_error_mgr *error_ptr;

void win_jpeg_error_exit( j_common_ptr cinfo )
{
  error_ptr myerr = (error_ptr)cinfo->err;
  (*cinfo->err->output_message)(cinfo);
  longjmp( myerr->setjmp_buffer, 1);
}

/* Read_JPEG_File
// 1) reads the data from a JPEG file
// 2) creates an XImage and stores the data in it
// 3) returns the XImage
// portions of this subroutine copied from example.c in the libjpeg source
// and from the libtiff examples */
XImage *Read_JPEG_File( const char *filename, int scale, int grayscale,
                        int *image_width, int *image_height,
                        int *MCUwidth, int *MCUheight,
                        int *scale_denom )
{
  struct jpeg_decompress_struct cinfo;
  struct win_jpeg_error_mgr jerr;
  XImage *ximage = 0;
  unsigned long xcol;
  int bperpix, bperline;
  unsigned char *imagedata, *ip;
  JSAMPARRAY buffer;  /* Output row buffer */
  JDIMENSION i;
  JSAMPROW jp;
  int nc;
  FSBUF *fs;
  FILE *infile;

  infile = fopen( filename, "rb" );
  if ( !infile )
    return 0;

/* overide the error_exit handler */
  cinfo.err = jpeg_std_error( &jerr.pub );
  jerr.pub.error_exit = win_jpeg_error_exit;
/* if this is true some weird error has already occured */
  if ( setjmp(jerr.setjmp_buffer) ) {
    if ( ximage ) {
      if ( cinfo.output_scanline < cinfo.output_height ) {
        XDestroyImage( ximage );
        ximage = 0;
      }
    }
    jpeg_destroy_decompress( &cinfo );
    fclose( infile );
    return ximage;
  }

/* initialize the jpeg decompression object */
  jpeg_create_decompress( &cinfo );

/* specify the data source */
  jpeg_stdio_src( &cinfo, infile );

/* read file parameters */
  jpeg_read_header( &cinfo, TRUE );

/* set parameters for decompression
   add here if needed */
  if (scale)
    cinfo.scale_num = scale;

  if (grayscale) /* force monochrome output */
    if (cinfo.jpeg_color_space == JCS_YCCK)
      cinfo.out_color_space = JCS_YCCK;
    else
      cinfo.out_color_space = JCS_GRAYSCALE;

/* start decompressor */
  jpeg_start_decompress( &cinfo );

  *image_width = (int) cinfo.image_width;
  *image_height = (int) cinfo.image_height;
  switch (cinfo.out_color_space) {
  case JCS_GRAYSCALE:
  case JCS_YCCK:
    *MCUwidth = cinfo.scale_denom;
    *MCUheight = cinfo.scale_denom;
    break;
  case JCS_RGB:
  case JCS_CMYK:
    *MCUwidth = cinfo.max_h_samp_factor * cinfo.scale_denom;
    *MCUheight = cinfo.max_v_samp_factor * cinfo.scale_denom;
    break;
  default: /* Unsupported output color space */
    jpeg_destroy_decompress( &cinfo );
    fclose( infile );
    return 0;
  }
  *scale_denom = cinfo.scale_denom;

/* Initialize screen data */
  screen_init();

/* we now have the correct dimensions in the cinfo structure.
   Let's create an XImage to hold the data */
  ximage = XCreateImage(theDisp, theVisual, dispDEEP, ZPixmap, 0, NULL,
                        cinfo.output_width, cinfo.output_height, 32, 0);
  if (!ximage) {
//    FatalError("couldn't create X image!");
    jpeg_destroy_decompress( &cinfo );
    fclose( infile );
    return 0;
  }

  bperline = ximage->bytes_per_line;
  bperpix  = ximage->bits_per_pixel;

  if (bperpix != 8 && bperpix != 16 && bperpix != 24 && bperpix != 32) {
//    char buf[128];
//    sprintf(buf,"Sorry, no code written to handle %d-bit %s",
//      bperpix, "TrueColor/DirectColor displays!");
//    FatalError(buf);
    XDestroyImage( ximage );
    jpeg_destroy_decompress( &cinfo );
    fclose( infile );
    return 0;
  }

  imagedata = (unsigned char *) malloc((size_t) (cinfo.output_height * bperline));
  if (!imagedata) {
//    FatalError("couldn't malloc imagedata");
    XDestroyImage( ximage );
    jpeg_destroy_decompress( &cinfo );
    fclose( infile );
    return 0;
  }

  ximage->data = (char *) imagedata;

  /* Make a one-row-high sample array that will go away when done with image */
  buffer = (*cinfo.mem->alloc_sarray)
                ((j_common_ptr) &cinfo, JPOOL_IMAGE,
                 cinfo.output_width * cinfo.output_components, 1);

  fs = 0;
  if (myapp->m_dither && bperpix < 24) {
    nc = cinfo.output_components;
    if (nc < 4) fs = fs2_init(cinfo.output_width, nc);
  }

/* loop through the decompression object and store the data in the image */

  while ( cinfo.output_scanline < cinfo.output_height ) {
    jpeg_read_scanlines( &cinfo, buffer, 1 );
    if (fs) {
      nc = fs2_dither(fs, *buffer, nc, 1, cinfo.output_width, bperline);
    }
    ip = imagedata; jp = *buffer; i = cinfo.output_width;
    switch (cinfo.out_color_space) {
    case JCS_GRAYSCALE:
      switch (bperpix) {
      case 8:
        do {
          int j = *jp++;
          xcol = screen_rgb[0][j] | screen_rgb[1][j] | screen_rgb[2][j];
          *ip++ = xcol & 0xff;
        } while (--i);
        break;
      case 16:
        do {
          int j = *jp++;
          xcol = screen_rgb[0][j] | screen_rgb[1][j] | screen_rgb[2][j];
          *((unsigned short *)ip)++ = (unsigned short)xcol;
        } while (--i);
        break;
      case 24:
        do {
          int j = *jp++;
          xcol = screen_rgb[0][j] | screen_rgb[1][j] | screen_rgb[2][j];
          *ip++ = (xcol >> 16) & 0xff;
          *ip++ = (xcol >> 8)  & 0xff;
          *ip++ =  xcol        & 0xff;
        } while (--i);
        break;
      case 32:
        do {
          int j = *jp++;
          xcol = screen_rgb[0][j] | screen_rgb[1][j] | screen_rgb[2][j];
          *((unsigned long *)ip)++ = xcol;
        } while (--i);
        break;
      }
      break;
    case JCS_RGB:
      switch (bperpix) {
      case 8:
        do {
          xcol  = screen_rgb[0][*jp++];
          xcol |= screen_rgb[1][*jp++];
          xcol |= screen_rgb[2][*jp++];
          *ip++ = xcol & 0xff;
        } while (--i);
        break;
      case 16:
        do {
          xcol  = screen_rgb[0][*jp++];
          xcol |= screen_rgb[1][*jp++];
          xcol |= screen_rgb[2][*jp++];
          *((unsigned short *)ip)++ = (unsigned short)xcol;
        } while (--i);
        break;
      case 24:
        do {
          xcol  = screen_rgb[0][*jp++];
          xcol |= screen_rgb[1][*jp++];
          xcol |= screen_rgb[2][*jp++];
          *ip++ = (xcol >> 16) & 0xff;
          *ip++ = (xcol >> 8)  & 0xff;
          *ip++ =  xcol        & 0xff;
        } while (--i);
        break;
      case 32:
        do {
          xcol  = screen_rgb[0][*jp++];
          xcol |= screen_rgb[1][*jp++];
          xcol |= screen_rgb[2][*jp++];
          *((unsigned long *)ip)++ = xcol;
        } while (--i);
        break;
      }
      break;
    case JCS_CMYK:
      switch (bperpix) {
      case 8:
        do {
          int k = GETJSAMPLE(jp[3]);
          xcol  = screen_rgb[0][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          xcol |= screen_rgb[1][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          xcol |= screen_rgb[2][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          *ip++ = xcol & 0xff;
          jp++;
        } while (--i);
        break;
      case 16:
        do {
          int k = GETJSAMPLE(jp[3]);
          xcol  = screen_rgb[0][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          xcol |= screen_rgb[1][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          xcol |= screen_rgb[2][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          *((unsigned short *)ip)++ = (unsigned short)xcol;
          jp++;
        } while (--i);
        break;
      case 24:
        do {
          int k = GETJSAMPLE(jp[3]);
          xcol  = screen_rgb[0][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          xcol |= screen_rgb[1][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          xcol |= screen_rgb[2][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          *ip++ = (xcol >> 16) & 0xff;
          *ip++ = (xcol >> 8)  & 0xff;
          *ip++ =  xcol        & 0xff;
          jp++;
        } while (--i);
        break;
      case 32:
        do {
          int k = GETJSAMPLE(jp[3]);
          xcol  = screen_rgb[0][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          xcol |= screen_rgb[1][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          xcol |= screen_rgb[2][(GETJSAMPLE(*jp++) * k) / MAXJSAMPLE];
          *((unsigned long *)ip)++ = xcol;
          jp++;
        } while (--i);
        break;
      }
      break;
    default: /* YCCK */
      switch (bperpix) {
      case 8:
        do {
          int j = ((MAXJSAMPLE - GETJSAMPLE(jp[0])) * GETJSAMPLE(jp[3]))
                  / MAXJSAMPLE;
          xcol = screen_rgb[0][j] | screen_rgb[1][j] | screen_rgb[2][j];
          *ip++ = xcol & 0xff;
          jp += 4;
        } while (--i);
        break;
      case 16:
        do {
          int j = ((MAXJSAMPLE - GETJSAMPLE(jp[0])) * GETJSAMPLE(jp[3]))
                  / MAXJSAMPLE;
          xcol = screen_rgb[0][j] | screen_rgb[1][j] | screen_rgb[2][j];
          *((unsigned short *)ip)++ = (unsigned short)xcol;
          jp += 4;
        } while (--i);
        break;
      case 24:
        do {
          int j = ((MAXJSAMPLE - GETJSAMPLE(jp[0])) * GETJSAMPLE(jp[3]))
                  / MAXJSAMPLE;
          xcol = screen_rgb[0][j] | screen_rgb[1][j] | screen_rgb[2][j];
          *ip++ = (xcol >> 16) & 0xff;
          *ip++ = (xcol >> 8)  & 0xff;
          *ip++ =  xcol        & 0xff;
          jp += 4;
        } while (--i);
        break;
      case 32:
        do {
          int j = ((MAXJSAMPLE - GETJSAMPLE(jp[0])) * GETJSAMPLE(jp[3]))
                  / MAXJSAMPLE;
          xcol = screen_rgb[0][j] | screen_rgb[1][j] | screen_rgb[2][j];
          *((unsigned long *)ip)++ = xcol;
          jp += 4;
        } while (--i);
        break;
      }
      break;
    }
    imagedata += bperline;
  }

  if (fs) free(fs);

/* finish decompression */
  jpeg_finish_decompress( &cinfo );

/* release decompression object */
  jpeg_destroy_decompress( &cinfo );

  fclose( infile );
  return ximage;
}
