/* xms.h */
#ifndef XMS_H
#define XMS_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

typedef struct {
    unsigned	    size; /* in bytes, must be even */
    unsigned short  src_handle; /* if 0, src_offset is physical addr */
    unsigned        src_offset;
    unsigned short  dst_handle; /* if 0, dst_offset is physical addr */
    unsigned        dst_offset;
    } __attribute__ ((packed)) XMS_EMM;

enum XMS_error_enum {
    XMS_error_successful = 0x00,
    XMS_error_function_not_implemented = 0x80,
    XMS_error_Vdisk_was_detected = 0x81,
    XMS_error_an_A20_error_occurred = 0x82,
    XMS_error_a_general_driver_error = 0x8E,
    XMS_error_unrecoverable_driver_error = 0x8F,
    XMS_error_HMA_does_not_exist_or_is_not_managed_by_XMS_provider = 0x90,
    XMS_error_HMA_is_already_in_use = 0x91,
    XMS_error_DX_is_less_than_the_HMAMIN_parameter = 0x92,
    XMS_error_HMA_is_not_allocated = 0x93,
    XMS_error_A20_line_still_enabled = 0x94,
    XMS_error_all_extended_memory_is_allocated = 0xA0,
    XMS_error_all_available_extended_memory_handles_are_allocated = 0xA1,
    XMS_error_invalid_handle = 0xA2,
    XMS_error_source_handle_is_invalid = 0xA3,
    XMS_error_source_offset_is_invalid = 0xA4,
    XMS_error_destination_handle_is_invalid = 0xA5,
    XMS_error_destination_offset_is_invalid = 0xA6,
    XMS_error_length_is_invalid = 0xA7,
    XMS_error_move_has_an_invalid_overlap = 0xA8,
    XMS_error_parity_error_occurred = 0xA9,
    XMS_error_block_is_not_locked = 0xAA,
    XMS_error_block_is_locked = 0xAB,
    XMS_error_block_lock_count_overflowed = 0xAC,
    XMS_error_lock_failed = 0xAD,
    XMS_error_only_a_smaller_UMB_is_available = 0xB0,
    XMS_error_no_UMB_s_are_available = 0xB1,
    XMS_error_UMB_segment_number_is_invalid = 0xB2
    };

extern inline farptr
_XMS_get_entry_point (void)
  {
  unsigned short command = 0x4310;
  farptr returned;

  asm ASMREAD (
"	pushl	%%es							\n"
"	movw	%%bx,%%es						\n"
"	int	$0x2F	# _XMS_get_entry_point				\n"
"	pushw	%%es							\n"
"	pushw	%%bx							\n"
"	popl	%%ebx	# register, never 14(%%esp) because %%es and pop!\n"
"	popl	%%es							\n"
	: "=b" (returned)
	: "a" (command), "b" (0)
	: "cc"
	);
  return returned;
  }

struct XMS_handle_table {
    unsigned char	signature_1;
    unsigned char	handle_descriptor_size;
    unsigned short	nbof_handle;
    farptr		array;
    } __attribute__ ((packed));

struct XMS_handle {
    enum { XMS_free = 1, XMS_used = 2, XMS_inpool = 4 } flags : 8;
    unsigned char	lock_count;
    unsigned		addr_shr10;
    unsigned		size;
    } __attribute__ ((packed));

extern inline farptr
_XMS_get_XMS_handle_table (void)
  {
  unsigned short command = 0x4309;
  unsigned char supported;
  farptr returned;

  asm ASMREAD (
"	pushl	%%es							\n"
"	movw	%%bx,%%es						\n"
"	int	$0x2F	# _XMS_get_XMS_handle_table			\n"
"	pushw	%%es							\n"
"	pushw	%%bx							\n"
"	popl	%%ebx	# register, never 14(%%esp) because %%es and pop!\n"
"	popl	%%es							\n"
	: "=a" (supported), "=b" (returned)
	: "a" (command), "b" (0)
	: "cc"
	);
  if (supported != 0x43)
      return 0;
  return returned;
  }

/* flag bit 0: 1=HMA exists, 0=no HMA */
extern inline unsigned short
_XMS_get_version (farptr entrypoint, unsigned short *revision,
			unsigned short *flags)
  {
  unsigned short version_bcd;

  asm ASMREAD (" lcallw *%a4	# XMS entry point, _XMS_get_version "
	: "=a" (version_bcd), "=b" (*revision), "=d" (*flags)
	: "a" (0), "pr" (&entrypoint), "m" (entrypoint)
	);
  return version_bcd;
  }

/* To allocate the HMA: */
extern inline unsigned char
_XMS_allocate_high_memory (farptr entrypoint, unsigned short nbbytes)
  {
  unsigned short status;
  unsigned char error;

  asm volatile (" lcallw *%a4	# XMS entry point, _XMS_allocate_high_memory "
	: "=a" (status), "=b" (error)
	: "a" (0x0100), "d" (nbbytes), "pr" (&entrypoint), "m" (entrypoint)
	);
  if (status == 0x0001)
      return 0; /* OK */
    else
      return error;
  }

extern inline unsigned char
_XMS_free_high_memory (farptr entrypoint)
  {
  unsigned short status;
  unsigned char error;

  asm volatile (" lcallw *%a3	# XMS entry point, _XMS_free_high_memory "
	: "=a" (status), "=b" (error)
	: "a" (0x0200), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001)
      return 0; /* OK */
    else
      return error;
  }

/* To use the HMA: */
extern inline unsigned short
_XMS_global_enable_A20 (farptr entrypoint, unsigned char *error)
  {
  unsigned short result;

  asm volatile (" lcallw *%a3	# XMS entry point, _XMS_global_enable_A20 "
	: "=a" (result), "=b" (*error)
	: "a" (0x0300), "pr" (&entrypoint), "m" (entrypoint)
	);
  if (result == 1)
      return 0; /* success */
    else
      return 1;
  }

extern inline unsigned short
_XMS_global_disable_A20 (farptr entrypoint, unsigned char *error)
  {
  unsigned short result;

  asm volatile (" lcallw *%a3	# XMS entry point, _XMS_global_disable_A20 "
	: "=a" (result), "=b" (*error)
	: "a" (0x0400), "pr" (&entrypoint), "m" (entrypoint)
	);
  if (result == 1)
      return 0; /* success */
    else
      return 1;
  }

/* To use the extended memory: */
extern inline unsigned short
_XMS_local_enable_A20 (farptr entrypoint, unsigned char *error)
  {
  unsigned short result;

  asm volatile (" lcallw *%a3	# XMS entry point, _XMS_local_enable_A20 "
	: "=a" (result), "=b" (*error)
	: "a" (0x0500), "pr" (&entrypoint), "m" (entrypoint)
	);
  if (result == 1)
      return 0; /* success */
    else
      return 1;
  }

extern inline unsigned short
_XMS_local_disable_A20 (farptr entrypoint, unsigned char *error)
  {
  unsigned short result;

  asm volatile (" lcallw *%a3	# XMS entry point, _XMS_local_disable_A20 "
	: "=a" (result), "=b" (*error)
	: "a" (0x0600), "pr" (&entrypoint), "m" (entrypoint)
	);
  if (result == 1)
      return 0;	/* success */
    else
      return 1;
  }

extern inline unsigned
_XMS_query_A20 (farptr entrypoint, unsigned char *error)
  {
  unsigned short result;

  asm volatile (" lcallw *%a3	# XMS entry point, _XMS_query_A20 "
	: "=a" (result), "=b" (*error)
	: "a" (0x0700), "pr" (&entrypoint), "m" (entrypoint)
	);
  return result; /* 1 if enabled, 0 if disabled */
  }

/* not including HMA: */
extern inline unsigned short
_XMS_query_free_memory (farptr entrypoint, unsigned short *largest,
			unsigned char *error)
  {
  unsigned short result;

  asm volatile (
"	xorb	%%bl,%%bl	# some implementation leave %%bl unchanged\n"
"	lcallw	*%a4	# XMS entry point, _XMS_query_free_memory	\n"
	: "=a" (*largest), "=b" (*error), "=d" (result)
	: "a" (0x0800), "pr" (&entrypoint), "m" (entrypoint)
	);
  return result; /* in Kb */
  }

extern inline unsigned short
_XMSv2_allocate_memory (farptr entrypoint, unsigned short nbKbytes,
			unsigned char *error)
  {
  unsigned short status, handle;

  asm volatile (" lcallw *%a5	# XMS entry point, _XMSv2_allocate_memory "
	: "=a" (status), "=b" (*error), "=d" (handle)
	: "a" (0x0900), "d" (nbKbytes), "pr" (&entrypoint), "m" (entrypoint)
	);
  if (status == 0x0001)
      return handle;
    else
      return 0;
  }

extern inline unsigned short
_XMS_free_memory (farptr entrypoint, unsigned short handle,
			unsigned char *error)
  {
  unsigned short status;

  asm volatile (" lcallw *%a4	# XMS entry point, _XMS_free_memory "
	: "=a" (status), "=b" (*error)
	: "a" (0x0A00), "d" (handle), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001)
      return 0; /* OK */
    else
      return 1; /* failed */
  }

/* ONLY forward moves work if overlap */
extern inline unsigned short
_XMS_move_memory (farptr entrypoint, XMS_EMM *emm_struct,
			unsigned char *error)
  {
  unsigned short status;

  asm volatile (" lcallw *%a4	# XMS entry point, _XMS_move_memory "
	: "=a" (status), "=b" (*error)
	: "a" (0x0B00), "S" (emm_struct) /* %ds:%si */,
	  "pr" (&entrypoint), "m" (*emm_struct), "m" (entrypoint)
	);

  if (status == 0x0001)
      return 0; /* OK */
    else
      return 1; /* failed */
  }

/* Locks are rejected after windows 3.x start */
extern inline unsigned
_XMS_lock (farptr entrypoint, unsigned short handle,
			unsigned char *error)
  {
  unsigned short status;
  unsigned address;

  asm volatile (
"	lcallw	*%a5	# XMS entry point, _XMS_lock_handle	\n"
"	pushw	%%dx						\n"
"	pushw	%%bx						\n"
"	popl	%%edx						\n"
	: "=a" (status), "=b" (*error), "=d" (address)
	: "a" (0x0C00), "d" (handle), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001)
      return address; /* OK */
    else
      return 0; /* failed */
  }

extern inline unsigned
_XMS_unlock (farptr entrypoint, unsigned short handle,
			unsigned char *error)
  {
  unsigned short status;

  asm volatile (" lcallw *%a4	# XMS entry point, _XMS_unlock_handle "
	: "=a" (status), "=b" (*error)
	: "a" (0x0D00), "d" (handle), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001)
      return 0; /* OK */
    else
      return 1; /* failed */
  }

/* window has problem with that: */
extern inline unsigned
_XMS_get_handle_info (farptr entrypoint, unsigned short handle,
			unsigned short *blocksize,
			unsigned char *block_lock_count,
			unsigned char *nb_free_handle,
			unsigned char *error)
  {
  unsigned short status, tmp;

  asm volatile (" lcallw *%a5	# XMS entry point, _XMS_get_handle_info "
	: "=a" (status), "=b" (tmp), "=d" (*blocksize)
	: "a" (0x0E00), "d" (handle), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001) {
      *block_lock_count = tmp >> 8;
      *nb_free_handle = tmp & 0xFF;
      *error = 0;
      return 0; /* OK */
      }
    else {
      *error = tmp & 0xFF;
      return 1; /* failed */
      }
  }

/* Cannot reallocate a locked handle. */
extern inline unsigned
_XMSv2_realloc_memory (farptr entrypoint, unsigned short handle,
			unsigned short newsize,
			unsigned char *error)
  {
  unsigned short status;

  asm volatile (" lcallw *%a5	# XMS entry point, _XMSv2_realloc_memory "
	: "=a" (status), "=b" (*error)
	: "a" (0x0F00), "d" (handle), "b" (newsize), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001)
      return 0; /* OK */
    else
      return 1; /* failed */
  }

/*
 * Here "*size" is the actual size of the block if success,
 * if failure it is the largest available block.
 * UMB are not allocatable if "DOS=UMB" in config.sys because
 * DOS get them all.
 */
extern inline unsigned short
_XMSv3_allocate_UMB (farptr entrypoint, unsigned short nbParagraph,
			unsigned short *size, unsigned char *error)
  {
  unsigned short status, tmp;

  asm volatile (" lcallw *%a5	# XMS entry point, _XMSv3_allocate_UMB "
	: "=a" (status), "=b" (tmp), "=d" (*size)
	: "a" (0x1000), "d" (nbParagraph), "pr" (&entrypoint), "m" (entrypoint)
	);
  if (status == 0x0001) {
      *error = 0;
      return tmp; /* the segment allocated */
      }
    else {
      *error = tmp & 0xFF;
      return 0;
      }
  }

extern inline unsigned short
_XMSv3_free_UMB (farptr entrypoint, unsigned short segment,
			unsigned char *error)
  {
  unsigned short status;

  asm volatile (" lcallw *%a4	# XMS entry point, _XMSv3_free_UMB "
	: "=a" (status), "=b" (*error)
	: "a" (0x1100), "d" (segment), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001)
      return 0; /* OK */
    else
      return 1; /* failed */
  }

extern inline unsigned
_XMSv3_realloc_UMB (farptr entrypoint, unsigned short segment,
			unsigned short newsize,
			unsigned char *error)
  {
  unsigned short status;
  unsigned short max_available; /* filled if failure */

  asm volatile (" lcallw *%a6	# XMS entry point, _XMSv3_realloc_UMB "
	: "=a" (status), "=b" (*error), "=d" (max_available)
	: "a" (0x1200), "d" (segment), "b" (newsize), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001)
      return 0; /* OK */
    else
      return 1; /* failed */
  }

extern inline unsigned
_XMSv3_query_free_memory (farptr entrypoint,
			  unsigned *largest_KB,
			  unsigned *highest_memory_address,
			  unsigned char *error)
  {
  unsigned result;

  asm ASMREAD (" lcallw	*%a5	# XMS entry point, _XMSv3_query_free_memory "
	: "=a" (*largest_KB), "=b" (*error), "=d" (result),
	  "=c" (*highest_memory_address)
	: "a" (0x8800), "pr" (&entrypoint), "m" (entrypoint)
	);
  return result; /* in Kb */
  }

/* HIMEM memory (> 128 Mb) malloced here can be freed with _XMS_free_memory() */
extern inline unsigned short
_XMSv3_allocate_memory (farptr entrypoint, unsigned nbKbytes,
			unsigned char *error)
  {
  unsigned short status, handle;

  asm volatile (" lcallw *%a5	# XMS entry point, _XMSv3_allocate_memory "
	: "=a" (status), "=b" (*error), "=d" (handle)
	: "a" (0x8900), "d" (nbKbytes), "pr" (&entrypoint), "m" (entrypoint)
	);
  if (status == 0x0001)
      return handle;
    else
      return 0;
  }

extern inline unsigned
_XMSv3_get_handle_info (farptr entrypoint, unsigned short handle,
			unsigned *blocksize,
			unsigned char *block_lock_count,
			unsigned char *error)
  {
  unsigned short status, tmp, nb_free_handle;

  asm volatile (
"	xorw	%%cx,%%cx	# MSDOS 6.0 HIMEM.SYS bug: cx unchanged	\n"
"	lcallw	*%a6	# XMS entry point, _XMSv3_get_handle_info	\n"
	: "=a" (status), "=b" (tmp), "=d" (*blocksize), "=c" (nb_free_handle)
	: "a" (0x8E00), "d" (handle), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001) {
      *block_lock_count = tmp >> 8;
      *error = 0;
      return 0; /* OK */
      }
    else {
      *error = tmp & 0xFF;
      return 1; /* failed */
      }
  }

/* Cannot reallocate a locked handle. */
extern inline unsigned
_XMSv3_realloc_memory (farptr entrypoint, unsigned short handle,
			unsigned newsize,
			unsigned char *error)
  {
  unsigned short status;

  asm volatile (" lcallw *%a5	# XMS entry point, _XMSv3_realloc_memory "
	: "=a" (status), "=b" (*error)
	: "a" (0x8F00), "d" (handle), "b" (newsize), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001)
      return 0; /* OK */
    else
      return 1; /* failed */
  }

/* with version autodetection - do not insert two asm volatile blocks */
extern inline unsigned short
_XMS_allocate_memory (farptr entrypoint, unsigned nbKbytes,
			unsigned char *error, unsigned short XMS_version)
  {
  unsigned short status, handle, command = 0x8900;

  if ((XMS_version >> 8) < 3) /* to manage initrd > 128 Mbytes */
      command = 0x0900;

  asm volatile (" lcallw *%a5	# XMS entry point, _XMS_allocate_memory "
	: "=a" (status), "=b" (*error), "=d" (handle)
	: "a" (command), "d" (nbKbytes), "pr" (&entrypoint), "m" (entrypoint)
	);
  if (status == 0x0001)
      return handle;
    else
      return 0;
  }

extern inline unsigned
_XMS_realloc_memory (farptr entrypoint, unsigned short handle,
			unsigned newsize, unsigned char *error,
			unsigned short XMS_version)
  {
  unsigned short status, command = 0x8F00;

  if ((XMS_version >> 8) < 3) /* to manage initrd > 128 Mbytes */
      command = 0x0F00;

  asm volatile (" lcallw *%a5	# XMS entry point, _XMS_realloc_memory "
	: "=a" (status), "=b" (*error)
	: "a" (command), "d" (handle), "b" (newsize), "pr" (&entrypoint), "m" (entrypoint)
	);

  if (status == 0x0001)
      return 0; /* OK */
    else
      return 1; /* failed */
  }

#endif /* XMS_H */
