/* vmlinuz.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"
#if SETUP & XCODE_SEGMENT	/* before other includes */
#define CALLING_FROM_XCODESEG
#endif

#include "instboot.h"
#include "library.h"
#include "boot.h"
#include "debug.h"
#include "user.h"
#include "bios.h"	/* vmlinuz_EGA() */
#include "disk.h"	/* before ide.h */
#include "ide.h"	/* _IDE_guess_altadr() */
#include "mouse.h"	/* togpl.MOUSE */
#include "dos.h"	/* SMARTDRV_control() */
#include "util.h"	/* UTIL.{v86_info,v86_callback,emm_type} */
#include "xms.h"
#include "vesabios.h"	/* _VESA_GetPMinterface() */
#include "vgabios.h"	/* _VGA_get_display_combination_code() */
#include "vmlinuz.h"
#include "fs.h"		/* struct desc_str */
#include "messages.h"
#include "gzlib.h"	/* only for GZLIB_TEST_MEASURE*() */
#include "font.h"
#include "kbd.h"

/* Get the smallest code size - we do not really care at all
   if it takes two or three cycles more... */
extern inline unsigned
is_space (char c)
  {
#if 1
/* 0x0001e300  __sizeof_all_code */
  static const char spacing[] = { ' ', '\t', '\n', '\r' };
  int cpt = sizeof (spacing);

  while (cpt --)
      if (c == spacing[cpt])
	  return 1;
#elif 1
/* 0x0001e340  __sizeof_all_code */
  struct {
      char ch[4];
      unsigned all;
      } spacing = {{ ' ', '\t', '\n', '\r' }};
  if (c == (char)spacing.all)
      return 1;
  spacing.all >>= 8;
  if (c == (char)spacing.all)
      return 1;
  spacing.all >>= 8;
  if (c == (char)spacing.all)
      return 1;
  spacing.all >>= 8;
  if (c == (char)spacing.all)
      return 1;
#elif 1
/*  0x0001e340  __sizeof_all_code */
  static const char spacing[] = { ' ', '\t', '\n', '\r' };
  const char *returned = _strnchr (spacing, c, sizeof(spacing));

  if (*returned == c)
      return 1;
#elif 1
/* 0x0001e380  __sizeof_all_code */
  static const char spacing[] = { ' ', '\t', '\n', '\r' };

  if (c == spacing[0] || c == spacing[1] || c == spacing[2] || c == spacing[3])
      return 1;
#endif
  return 0;
  }

/**
 ** GZIP related stuff:
 **/
LOAD_FCT_PREFIX(treat_gzip_comment) void
treat_gzip_comment (const char *ptr)
  {bound_stack();{
       /* This allow spacing/newline between:
	*	"This software is distributed under the GPL license."
	*     and (one or more lines like):
	*	"Its source code is at: " + the http/ftp address
	* and in between:
	*	"This software is not used outside the copyright owner company."
	*     and (one or more lines like):
	*	"Copyright owner: " + the company official name
	* These two identification fields have to finish with '\n' and/or '\r',
	* they cannot finnish with '\0' without stoping the GZIP comment zone.
	* Lines shall not be longer than 255 chars and contain only ASCII 7bit chars.
	* Take care of the needed space after the semicolon. No tabs.
	*/
  static const char
      comment1[] = "This software is distributed under the GPL license.",
      extra1[] = "Its source code is at: ", /* http/ftp address, max 128 characters, ends at tab or \n */
      comment2[] = "This software is not used outside the copyright owner company.",
      extra2[] = "Copyright owner: ", /* official name, max 128 characters, ends at tab or \n */
      comment3[] = "This software has been authorised in writing by Etienne LORRAIN.";
  unsigned max_extra1 = 0, max_extra2 = 0, delay_something_printed = 0;

  if (LOADER.curfileload != LOADER.fileload) {
      ZDBG (("Just displaying initrd GZIP comment field '%s'\r\n", ptr));
      puts (ptr);
      return;
      }

  ZDBG (("\r\n%s parsing string, initial LOADER.cmdline_extraparam is '%s'\r\n",
		__FUNCTION__, LOADER.cmdline_extraparam));

  while (*ptr == ' ') ptr++;

#if USER_SUPPORT != 0
  {
  unsigned char row, col;
  unsigned get_cursor_returned = UI.function.getcursor (&row, &col);
#endif

  if (memeql (ptr, comment1, sizeof (comment1) - 1)) {
      LOADER.license_compliant = 1;
      puts ("\010,");
      puts (comment1);
      ZDBG (("Found '%s'\r\n", comment1));
      ptr += sizeof (comment1) - 1;
      delay_something_printed = 5 * 100 * 1000;
      }
    else if (memeql (ptr, comment2, sizeof (comment2) - 1)) {
      LOADER.license_compliant = 2;
      puts ("\010,");
      puts (comment2);
      ZDBG (("Found '%s'\r\n", comment2));
      ptr += sizeof (comment2) - 1;
      delay_something_printed = 15 * 100 * 1000;
      }
    else if (memeql (ptr, comment3, sizeof (comment3) - 1)) {
      LOADER.license_compliant = 3;
      ZDBG (("Found '%s'\r\n", comment3));
      ptr += sizeof (comment3) - 1;
      }
    else
      LOADER.license_compliant = 0;

  for (;;) {
      char tmpstr[256];
      unsigned cpt = 0, *maxptr;

      while (is_space (*ptr))
	  ptr++;

      if (memeql (ptr, extra2, sizeof (extra2) - 1)) {
	  /* I do accept one or more Copyright owner: line anyways: */
	  ZDBG (("Found '%s'\r\n", extra2));
	  maxptr = &max_extra2;
	  }
	else if (memeql (ptr, extra1, sizeof (extra1) - 1)) {
	  /* I do accept one or more source code line if it is present: */
	  ZDBG (("Found '%s'\r\n", extra1));
	  maxptr = &max_extra1;
	  }
	else
	  break;
      while (*ptr >= ' ' && cpt < sizeof (tmpstr) - 1)
	  tmpstr[cpt++] = *ptr++;
      while (cpt && tmpstr[--cpt] == ' ')
	  /* nothing (remove spaces at end) */;
      tmpstr[++cpt] = '\0';
      if (cpt > *maxptr)
	  *maxptr = cpt;
      puts (tmpstr);
      }

  /* I do not accept "f**k" as an answer... */
  if (LOADER.license_compliant == 1 && max_extra1 <= 5) {
      ZDBG (("Downgrade license_compliant: no source location of GPL\r\n"));
      LOADER.license_compliant = 0;
      }
  if (LOADER.license_compliant == 2 && max_extra2 <= 5) {
      ZDBG (("Downgrade license_compliant: no copyright owner of non redistributed\r\n"));
      LOADER.license_compliant = 0; /* I want it if not redistributed */
      }

  unsigned cpt;
  for (cpt = 0; cpt < sizeof (LOADER.cmdline_extraparam) - 1; cpt++)
      if ((LOADER.cmdline_extraparam[cpt] = ptr[cpt]) == '\0')
	  break;
  LOADER.cmdline_extraparam[cpt] = '\0';

#if USER_SUPPORT != 0
  if (UI.function.getcursor (&LOADER.post_comment_row, &LOADER.post_comment_col) != 0)
	LOADER.post_comment_row = LOADER.post_comment_col = 0xFF;
  if (get_cursor_returned == 0)
      UI.function.setcursor (row, col);
  }
#else
  if (LOADER.license_compliant == 1 || LOADER.license_compliant == 2)
      print (MSG_LOADING);  /* after display license string */
#endif

  for (;;) {
      static const char *const kern_param[] = {
/* Just for non-gpl code writers: */
	"NONGPL_productID=0x",
	"NONGPL_productNB=0x",
/* min_gujin_version=0x100 : refuse to start if below (gujin bugs) */
	"min_gujin_version=0x",
/* loadadr=0x00100000 : the load address (late_relocation_address) */
	"loadadr=0x",
/* runadr=0x00100000 : the run address */
	"runadr=0x",
/* paramadr=0x00080000 : the parameter address (fit in real mode!) */
	"paramadr=0x",
/* realfct_size=0x8000 : max 32 Kb */
	"realfct_size=0x",
/* minram=0x0001000 : the minimum amount of memory (in KB) to have */
	"minram=0x",
/* option=0x00000000 : able to handle "dangerous" things like IDE password
   unfrozen: see disk.h/vmlinuz.h */
	"option=0x",
/* if bit (1 << [34567]) set, 80[34567]86 excluded (see vmlinuz.h) */
/* All (i386 compatible) processors: 0x000000F8 */
/* http://www.paradicesoftware.com/specs/cpuid/?the_id=5 */
	"maskcpu=0x",
/* maskDflags=0x...  : request those bits set, cpuid 0x00000001, in edx */
/* example TSC required: maskDflags=0x00000010 */
	"maskDflags=0x",
/* maskCflags=0x...  : request those bits set, cpuid 0x00000001, in ecx */
	"maskCflags=0x",
/* maskBflags=0x...  : unused on PCs */
	"maskBflags=0x",
/* maskAflags=0x... : request those bits set, cpuid 0x80000001, AMD */
	"maskAflags=0x",
/* Which video mode is managed by the kernel (see vmlinuz.h) */
	"maskvesa=0x",
/* Which video resolution is refused by the kernel (see vmlinuz.h) */
	"maskresolution=0x"
/* TODO? minfreq=100MHz, buildroot=/dev/hda5 */
	};
      unsigned index;

      while (is_space (*ptr))
	  ptr++;
      if (*ptr == '\0')
	  break;

      for (index = 0; index < nbof (kern_param); index++) {
	  unsigned len = strlen (kern_param[index]);
	  if (memeql (ptr, kern_param[index], len)) {
	      ptr += len;
	      break;
	      }
	  }
      if (index < nbof (kern_param)) {
	  unsigned val = 0;

	  for (;;) {
	      int tmp = hexval(*ptr);
	      if (tmp == -1)
		  break;
	      val = (val << 4) + tmp;
	      ptr++;
	      }
	  LOADER.comment.array[index] = val;
	  ZDBG (("\r\nSetting from GZIP comment parameter %s%x",
		kern_param[index], LOADER.comment.array[index]));
	  }
	else {
	  /* Ignore that */
	  while (!is_space (*ptr) && *ptr != '\0')
	      ptr++;
	  }
      }
  /* note that treat_gzip_comment() may not be executed if there is no GZIP comment */
  /* if this has been reseted to zero, execute in place relocatable kernel,
     at MSDOS HIMEM alloced memory: */
  if (LOADER.comment.elem.late_relocation_address == 0)
      LOADER.comment.elem.late_relocation_address = LOADER.fileload[0].load_address;

  if (   LOADER.license_compliant == 3
      && (   LOADER.comment.elem.NONGPL_productID == 0
	  || LOADER.comment.elem.NONGPL_productNB == 0)) {
      ZDBG (("Downgrade license_compliant: productID or productNB null\r\n"));
      LOADER.license_compliant = 0;
      LOADER.comment.elem.NONGPL_productID = LOADER.comment.elem.NONGPL_productNB = 0;
      }
  if (LOADER.comment.elem.NONGPL_productID || LOADER.comment.elem.NONGPL_productNB) {
      printf (PRODUCT_X_SERIAL_X, LOADER.comment.elem.NONGPL_productID, LOADER.comment.elem.NONGPL_productNB);
      delay_something_printed = 5 * 100 * 1000;
      }

  if (copy_gujin_param.attrib.ignore_kernel_option)
      LOADER.comment.elem.option = 0;

  if (delay_something_printed) {
      _BIOS_wait (delay_something_printed);
      while (_BIOS_get_shift_flags().control_key_pressed)
	  continue;
      }

  ZDBG (("\r\nLOADER.cmdline_extraparam is now '%s'\r\n", LOADER.cmdline_extraparam));
  }}

LOAD_FCT_PREFIX(check_comment_restriction) static inline unsigned
check_comment_restriction (unsigned nbKbRam)
  {
  if (LOADER.comment.elem.minram > nbKbRam) {
      ZDBG (("%s: minram = %u Kb, currently %u Kb of RAM -> cannot run\r\n",
		__FUNCTION__, LOADER.comment.elem.minram, nbKbRam));
      return 0x5;
      }
#define GUJIN_HEX_VERSION	0x0208	/*VERSION*/
  if (LOADER.comment.elem.min_gujin_version > GUJIN_HEX_VERSION) {
      ZDBG (("%s: min_gujin_version = 0x%X, currently 0x%X -> cannot run\r\n",
		__FUNCTION__, LOADER.comment.elem.min_gujin_version, GUJIN_HEX_VERSION));
      return 0x6;
      }
  if (LOADER.comment.elem.maskcpu && !(LOADER.comment.elem.maskcpu & MASKCPU_BIOSIA32)) {
      ZDBG (("%s: kernel not for PC BIOS IA32 systems, cannot run\r\n", __FUNCTION__));
      return 0x10;
      }
  if (LOADER.comment.elem.maskcpu & (1 << UTIL.processor.family)) {
      ZDBG (("%s: current processor family: %u, mask: 0x%X -> cannot run\r\n",
		__FUNCTION__, UTIL.processor.family, LOADER.comment.elem.maskcpu));
      return 0x10;
      }
  if ((LOADER.comment.elem.maskcpu & MASKCPU_BIOSONLY) && BOOT1_DOS_running()) {
      ZDBG (("%s: currently under DOS and kernel refuses -> cannot run\r\n",
		__FUNCTION__));
      return 0x11;
      }
  if ((LOADER.comment.elem.maskcpu & MASKCPU_REALONLY) && (getsw() & 1)) {
      ZDBG (("%s: currently virtual 8086 -> cannot run\r\n", __FUNCTION__));
      return 0x12;
      }
  if ((LOADER.comment.elem.maskDflags & UTIL.processor.Dflags) != LOADER.comment.elem.maskDflags) {
      ZDBG (("%s: current standard edx flags: 0x%X, mask: 0x%X -> cannot run\r\n",
		__FUNCTION__, UTIL.processor.Dflags, LOADER.comment.elem.maskDflags));
      return 0x13;
      }
  if ((LOADER.comment.elem.maskCflags & UTIL.processor.Cflags) != LOADER.comment.elem.maskCflags) {
      ZDBG (("%s: current standard ecx flags: 0x%X, mask: 0x%X -> cannot run\r\n",
		__FUNCTION__, UTIL.processor.Cflags, LOADER.comment.elem.maskCflags));
      return 0x14;
      }
  if ((LOADER.comment.elem.maskAflags & UTIL.processor.Aflags) != LOADER.comment.elem.maskAflags) {
      ZDBG (("%s: current AMD extra flags: 0x%X, mask: 0x%X -> cannot run\r\n",
		__FUNCTION__, UTIL.processor.Aflags, LOADER.comment.elem.maskAflags));
      return 0x15;
      }

#if USER_SUPPORT != 0
  if (   (LOADER.comment.elem.maskvesa & MASKVESA_NOGRAPHICVGA) != 0
      && VGA_ACTIVE()
      && UI.parameter.attr.isgraphic ) {
      ZDBG (("%s: Do not accept VGA graphic -> cannot run\r\n",
		__FUNCTION__));
      return 0x24;
      }
  if ((LOADER.comment.elem.maskvesa & MASKVESA_TYPE) != 0) {
      unsigned bpp;
      if (   (LOADER.comment.elem.maskvesa & MASKVESA_TYPE) == MASKVESA_LIN
#if USER_SUPPORT & VESA_SUPPORT
	  && (UI.parameter.winsize == VESA2_MARKER)
#endif
	  ) {
	  ZDBG (("%s: accept only VESA linear and not active -> cannot run\r\n",
		__FUNCTION__));
	  return 0x20;
	  }
      if (   UI.parameter.attr.isgraphic == 0
	  && !(LOADER.comment.elem.maskvesa & MASKVESA_TEXT)) {
	  ZDBG (("%s: in TEXT mode and does not accept it -> cannot run\r\n",
		__FUNCTION__));
	  return 0x21;
	  }
      if (   !(LOADER.comment.elem.maskvesa & MASKVESA_WIN)
#if USER_SUPPORT & VESA_SUPPORT
	  && (UI.parameter.winsize != VESA2_MARKER && UI.parameter.winsize != 0)
#endif
	  ) {
	  ZDBG (("%s: in VESA_WINDOW mode and does not accept it -> cannot run\r\n",
		__FUNCTION__));
	  return 0x22;
	  }
      if (   LOADER.comment.elem.maskvesa & MASKVESA_LIN2WIN
#if USER_SUPPORT & VESA_SUPPORT
	  && UI.parameter.winsize == VESA2_MARKER
#endif
	  ) {
	  unsigned gcc41_tmp;
	  ZDBG (("%s: switching back from VESA2 to VESA1: ", __FUNCTION__));
	  gcc41_tmp = UI.function.getmode();
	  UI.function.setmode ((gcc41_tmp & ~0x4000) | 0x8000);
	  ZDBG (("done, now winsize = 0x%X\r\n", UI.parameter.winsize));
	  }
      bpp = get_bpp (&UI.parameter);
      if (( (1 << bpp) & LOADER.comment.elem.maskvesa) != 0) {
	  ZDBG (("%s: do not accept this number of BPP (%u) -> cannot run\r\n",
		__FUNCTION__, bpp));
	  return 0x23;
	  }
      }

  if (LOADER.comment.elem.maskresolution != 0) {
      static const unsigned char txtwidth[] = { 40, 80, 100, 132, 160 };
      static const unsigned char txtheight[] = { 24, 25, 30, 36, 43, 44, 48, 50, 60, 64 };
      static const struct {
	  unsigned short width, height;
	  } gfx[] = {
	  {  320,  200 },  {  320,  400 },  {  640,  200 },  {  640,  350 },
	  {  640,  400 },  {  640,  480 },  {  720,  400 },  {  768, 1024 },
	  {  800,  600 },  {  832,  624 },  { 1024,  768 },  { 1152,  864 },
	  { 1280,  960 },  { 1280, 1024 },  { 1600, 1200 },  { 2048, 1540 }
	  };
      unsigned char basebit, cpt;

      if (nbof (txtwidth) + nbof (txtheight) + nbof (gfx) + 1 != 32) {
	  /* do not forget MASKRESOLUTION_UNLISTED */
	  __ERROR();
	  }

      if (UI.parameter.attr.isgraphic == 0) {
	  basebit = 0;
	  for (cpt = 0; cpt < nbof (txtwidth); cpt ++) {
	      if (UI.parameter.nbcol == txtwidth[cpt])
		  break;
	      }
	  if (cpt < nbof (txtwidth)) {
	      if (LOADER.comment.elem.maskresolution & (1 << (cpt + basebit))) {
		  ZDBG (("%s: cannot accept %u columns -> cannot run\r\n",
			__FUNCTION__, UI.parameter.nbcol));
		  return 0x33;
		  }
	      }
	    else if (LOADER.comment.elem.maskresolution & MASKRESOLUTION_UNLISTED) {
	      ZDBG (("%s: cannot accept %u cols * %u lines (unlisted col) -> cannot run\r\n",
			__FUNCTION__, UI.parameter.nbcol, UI.parameter.nbrow));
	      return 0x30;
	      }
	  basebit = nbof(txtwidth);
	  for (cpt = 0; cpt < nbof (txtheight); cpt ++) {
	      if (UI.parameter.nbrow == txtheight[cpt])
		  break;
	      }
	  if (cpt < nbof (txtheight)) {
	      if (LOADER.comment.elem.maskresolution & (1 << (cpt + basebit))) {
		  ZDBG (("%s: cannot accept %u lines -> cannot run\r\n",
			__FUNCTION__, UI.parameter.nbrow));
		  return 0x34;
		  }
	      }
	    else if (LOADER.comment.elem.maskresolution & MASKRESOLUTION_UNLISTED) {
	      ZDBG (("%s: cannot accept %u cols * %u lines (unlisted lines) -> cannot run\r\n",
			__FUNCTION__, UI.parameter.nbcol, UI.parameter.nbrow));
	      return 0x31;
	      }
	  }
	else { /* graphic modes: */
	  basebit = nbof(txtwidth) + nbof (txtheight);
	  for (cpt = 0; cpt < nbof (gfx); cpt ++) {
	      if (   UI.parameter.width == gfx[cpt].width
		  && UI.parameter.height == gfx[cpt].height)
		  break;
	      }
	  if (cpt < nbof (gfx)) {
	      if (LOADER.comment.elem.maskresolution & (1 << (cpt + basebit))) {
		  ZDBG (("%s: cannot accept %ux%u -> cannot run\r\n",
			__FUNCTION__, UI.parameter.width, UI.parameter.height));
		  return 0x35;
		  }
	      }
	    else if (LOADER.comment.elem.maskresolution & MASKRESOLUTION_UNLISTED) {
	      ZDBG (("%s: cannot accept %ux%u (unlisted gfx) -> cannot run\r\n",
			__FUNCTION__, UI.parameter.width, UI.parameter.height));
	      return 0x32;
	      }
	  }
      }
#endif

  return 0;
  }

LOAD_FCT_PREFIX(treat_gzip_name) void
treat_gzip_name (const char *ptr)
  {bound_stack();{
  if (LOADER.curfileload == LOADER.fileload) {
      unsigned cpt;
      for (cpt = 0; cpt < sizeof (LOADER.cmdline_name) - 1; cpt++)
	  if ((LOADER.cmdline_name[cpt] = ptr[cpt]) == '\0')
	      break;
      LOADER.cmdline_name[cpt] = '\0';
      ZDBG (("LOADER.cmdline_name inited to kernel GZIP name field '%s'\r\n", LOADER.cmdline_name));
      }
    else {
      ZDBG (("Ignoring initrd GZIP name field '%s'\r\n", ptr));
      }
  }}

#if defined (LINUZNAME1) || defined (LINUZNAME2) || defined (LINUZNAME3)
/**
 ** Get few parameters from the vmlinuz header:
 **/
LOAD_FCT_PREFIX(vmlinuz_header_treat) unsigned
vmlinuz_header_treat (const struct linux_param *param, unsigned gzlib_avail_in)
  {bound_stack();{
  const union header_signature_u header_sig = { { 'H', 'd', 'r', 'S' } };

  /* the loaded file is at least bigger than gzlib->avail_in: */
  if (gzlib_avail_in <= offsetof(struct linux_param, command_line)) {
      ZDBG ((" [vmlinuz filesize too small: gzlib_avail_in %u, min %u] ", gzlib_avail_in, offsetof(struct linux_param, command_line)));
      return 1;
      }

  /* the loaded file shall have the 0xAA55 magic number, two bytes at param->in_out.AUX_DEVICE_INFO: */
  if (*CAST (unsigned short *, &param->in_out.AUX_DEVICE_INFO) != 0xAA55) {
      ZDBG ((" [vmlinuz file without 0xAA55 signature but 0x%X] ", *CAST (unsigned short *, &param->in_out.AUX_DEVICE_INFO)));
//printf ("no AA55 but 0x%X at %u\r\n", *CAST (unsigned short *, &param->in_out.AUX_DEVICE_INFO, (unsigned)(&param->in_out.AUX_DEVICE_INFO) - (unsigned)&param));
      return 1;
      }

  if (param->in_out.header_signature.longword != header_sig.longword) {
      ZDBG ((" [%s: bad vmlinuz signature 0x%X] ", __FUNCTION__, param->in_out.header_signature.longword));
      /* Could still be a "old" protocol version: zImage w/o initrd, cannot test anyway */
      return 2;
      }
  ZDBG ((" [setup_sects=%u, root_flags 0x%X i.e. %s, kernel_compressed_size %u, ramdisk 0x%X, vid_mode 0x%X] \r\n",
	param->in_out.setup_sects, *CAST(unsigned short *, &param->in_out.root_flags), param->in_out.root_flags.read_only ? "ro" : "rw",
	param->in_out.kernel_compressed_size, *CAST(unsigned short *, &param->in_out.ramdisk), param->in_out.vid_mode));
  ZDBG ((" [major_root=0x%X, minor_root=0x%X, kernel version ptr=0x%X]",
	param->in_out.root.fields.major_root, param->in_out.root.fields.minor_root, param->in_out.kernel_version));

  if (param->in_out.code32_start != 0x100000) {	/* No more support zImage */
      ZDBG ((" [code32_start != 0x100000, I did not plan that entry value: 0x%X:0x%X] ", param->in_out.start_sys_seg, param->in_out.code32_start));
      return 3;
      }

  LOADER.minor_major_root = param->in_out.root.minor_major;
  LOADER.setup_sects = param->in_out.setup_sects;
  if (param->in_out.header_version_number >= 0x203) {
      ZDBG ((" [header_version_number 0x%X, ramdisk_max 0x%X] ", param->in_out.header_version_number, param->in_out.ramdisk_max));
      LOADER.ramdisk_max = param->in_out.ramdisk_max;
      }
    else
      LOADER.ramdisk_max = 0x37FFFFFF;

  /* We always load in full LINUX compatibility mode from Gujin-2.6 */
  if (LOADER.accept_uncompressed_filesize == 0) {
      ZDBG (("LOADER.accept_uncompressed_filesize == 0 loading bzImage!\r\n"));
      return 0xD;
      }
  if (param->in_out.header_version_number < 0x0200 || !param->in_out.loadflags.LOADED_HIGH) {
      ZDBG (("Cannot load conventional zImage, need at least bzImage!\r\n"));
      return 0xB;
      }
  ZDBG ((" loading kernel with maximum compatibility\r\n"));

  return 0;
  }}
#endif /* LINUZNAME* */

/**
 ** These (up to elf_start()) are the (only) DOS incompatible
 ** function if you are running "gujin.exe" (and segment doors closed,
 ** i.e. without BIG_MALLOC).
 ** The operating system (if there is one) will stop the execution of
 ** the process at least at the "protected_mode()" function call.
 **/
//#define LOAD_TASKREG

#if !(SETUP & BIOS_ONLY)
#define VCPI_SWITCH
//#define VCPI_COPY	/* not needed */
#endif

#ifdef VCPI_SWITCH

#ifdef VCPI_COPY
LOAD_FCT_PREFIX(VCPI_free_contiguous) static inline void
VCPI_free_contiguous (unsigned addr, unsigned last)
  {
  while (addr <= last) {
      _VCPI_free_4Kpage (addr);
      addr += 4096;
      }
  }

LOAD_FCT_PREFIX(VCPI_malloc_contiguous) static unsigned
VCPI_malloc_contiguous (unsigned size, unsigned *last)
  {
  struct { unsigned first, last; } blocks[32];
  unsigned returned = 0;
  int index = 0;

//  ZDBG (("%s dummy invalid alloc: 0x%X\r\n", __FUNCTION__, _VCPI_alloc_4Kpage()));

  blocks[index].first = blocks[index].last = _VCPI_alloc_4Kpage();
  ZDBG (("%s searching size %u starting at 0x%X.\r\n",
		__FUNCTION__, size, blocks[index].first));

  for (;;) {
      unsigned tmp = _VCPI_alloc_4Kpage();

      if (tmp == blocks[index].first - 4096) {
	  blocks[index].first = tmp;
	  ZDBG ((" [0x%X as start] ", tmp));
	  }
	else if (tmp == blocks[index].last + 4096) {
	  blocks[index].last = tmp;
	  ZDBG ((" [0x%X as last] ", tmp));
	  }
	else {
	  unsigned lookahead[4], cpt, nb;

	  cpt = lastof (lookahead);
	  lookahead[cpt] = tmp;
	  while (cpt --)
	      lookahead[cpt] = _VCPI_alloc_4Kpage();

//	  ZDBG ((" [lookahead 0x%X, 0x%X, 0x%X, 0x%X ",
//		lookahead[0], lookahead[1],
//		lookahead[2], lookahead[3]));

	  nb = nbof (lookahead);
	  while (nb != 0) {
	      unsigned oldnb = nb;

	      for (cpt = 0; cpt < nbof (lookahead); cpt++) {
		  if (lookahead[cpt] == blocks[index].last + 4096) {
		      blocks[index].last = lookahead[cpt];
		      lookahead[cpt] = 0;
		      nb--;
//		      ZDBG (("l"));
		      }
		    else if (lookahead[cpt] == blocks[index].first - 4096) {
		      blocks[index].first = lookahead[cpt];
		      lookahead[cpt] = 0;
		      nb--;
//		      ZDBG (("f"));
		      }
		    else {
//		      ZDBG (("?"));
		      }
		  }
	      if (oldnb == nb)
		  break;
	      }

//	  ZDBG ((" => 0x%X, 0x%X, 0x%X, 0x%X] ",
//			lookahead[0], lookahead[1],
//			lookahead[2], lookahead[3]));
	  if (nb != 0) {
	      unsigned nb_free_pages;

	      ZDBG (("bad continuity, first 0x%X, last 0x%X; ",
			blocks[index].first, blocks[index].last));
	      cpt = nbof (lookahead);
	      while (cpt--) {
		  if (lookahead[cpt]) {
		      ZDBG (("freeing 0x%X, ", lookahead[cpt]));
		      _VCPI_free_4Kpage (lookahead[cpt]);
		      }
		  }
	      nb_free_pages = _VCPI_get_nb_free_pages();
	      ZDBG (("Free VCPI memory: %u Kb, index: %u ",
			nb_free_pages * 4, index));
	      if (index == lastof (blocks)) {
		  ZDBG (("not enought blocks.\r\n"));
		  break;
		  }
	      if (nb_free_pages * 4096 < size) {
		  ZDBG (("not enought memory.\r\n"));
		  break;
		  }
	      ZDBG (("try next block "));
	      index++;
	      blocks[index].first = blocks[index].last = _VCPI_alloc_4Kpage();
	      ZDBG (("starting at 0x%X:\r\n", blocks[index].first));
	      }
	  }
      if (blocks[index].last + 4096 - blocks[index].first < size)
	  continue;
      ZDBG (("\r\n%s got memory block 0x%X..0x%X for requested size 0x%X.\r\n",
		__FUNCTION__, blocks[index].first, blocks[index].last, size));
      returned = blocks[index].first;
      if (last)
	  *last = blocks[index].last;
      index --;
      break;
      }

  /* Free the too small blocks: */
  while (index >= 0) {
      ZDBG (("%s free small memory block 0x%X..0x%X.\r\n", __FUNCTION__,
		blocks[index].first, blocks[index].last));
      VCPI_free_contiguous (blocks[index].first, blocks[index].last);
      index --;
      }
  return returned;
  }

LOAD_FCT_PREFIX(VCPI_copy) static unsigned
VCPI_copy (struct _VCPI_switch_str *VCPI_switch, unsigned entrypoint,
		farptr pagetable, unsigned short first_unused,
		struct LOADER_str *loader, farptr buffer16K)
  {
  ZDBG (("using buffer16K 0x%X\r\n", buffer16K));
  loader->curfileload = &loader->fileload[0];

  while (loader->curfileload->handle) {
      ZDBG (("VCPI handle address 0x%X\r\n", loader->curfileload->load_address));
      if (loader->curfileload->load_address & 0xFFF) {
	  unsigned cpt, last;
	  unsigned char error;
	  unsigned save_load_address = loader->curfileload->load_address;

	  ZDBG (("Contiguous VCPI memory alloc of %u bytes: ", loader->curfileload->uncompressed_size));
	  loader->curfileload->load_address = VCPI_malloc_contiguous (loader->curfileload->uncompressed_size, &last);
	  ZDBG (("at 0x%X\r\n", loader->curfileload->load_address));
	  if (loader->curfileload->load_address == 0) {
	      loader->curfileload->load_address = save_load_address; /* Will be unlocked */
	      return 2;
	      }
#define NBPAGE	4
	  for (cpt = 0; cpt < DIVROUNDUP (loader->curfileload->uncompressed_size, NBPAGE * 4096); cpt ++) {
	      unsigned baseaddr = cpt * NBPAGE * 4096;
	      XMS_EMM emm_struct = {
		  .size		= NBPAGE * 4096,	/* even number of bytes */
		  .src_handle	= loader->curfileload->handle,
		  .src_offset	= baseaddr,
		  .dst_handle	= 0,		/* real (virtual) mode address */
		  .dst_offset	= buffer16K
		  };
	      if (_XMS_move_memory (UTIL.HIMEM_entrypoint, &emm_struct, &error) != 0) {
		  ZDBG (("_XMS_move_memory error 0x%X, at %u * 16K!\r\n", error, cpt));
		  VCPI_free_contiguous (loader->curfileload->load_address, last);
		  loader->curfileload->load_address = save_load_address; /* Will be unlocked */
		  return 3;
		  }
	      /* Following uses hardcoded seg 0x18, 0x20 and 0x28 : */
	      _VCPI_copypages (VCPI_switch, entrypoint, pagetable,
				(pagetable & 0xFFFF0000) | first_unused,
				loader->curfileload->load_address + baseaddr,
				farptr2linear(buffer16K),
				NBPAGE);
	      }
#undef NBPAGE
	  ZDBG (("%s passed, base address 0x%X.\r\n", __FUNCTION__,
			loader->curfileload->load_address));
	  }
      loader->curfileload++;
      }
  return 0;
  }
#endif /* VCPI_COPY */


LOAD_FCT_PREFIX(VCPI_virtual2real) static unsigned __attribute__ ((noinline))
VCPI_virtual2real (void)
  {bound_stack();{
  /* There is no more unzip activity, we use the gzlib buffer as
     pointer for the pagetable; pagetable should be aligned to 4K */
  /* FIXME for BIG_MALLOC, but anyways we will not reach here... */
#if defined (GZLIB_USE_INTERMEDIATE_WINDOW) && !defined (GZLIB_BIG_MALLOC)
  extern unsigned char gzlib_window[];
  unsigned baseaddr = data_adr(gzlib_window);
#else
  unsigned baseaddr = (getss() + 0x1000) << 16;
#endif
  unsigned alignedaddr = (farptr2linear (baseaddr) + 0xFFF) & ~0xFFF;
// biggest offset:
//  farptr pagedir = linear2farptr(alignedaddr);
//  farptr pagetable = pagedir + 4096;
//  farptr buffer16K = pagedir + 2*4096; // overflow possible
  farptr pagedir = smallestoffset (linear2farptr (alignedaddr));
  farptr pagetable = pagedir + (0x100 << 16); /* 4096 >> 4 = 0x100 */

  TSS_t taskstate = {};
  seg_t VCPI_PMseg[] __attribute__ (( aligned(8) )) = {
      { .data = {}}, /* segment 0 is the invalid one */
      { .system = {	/* segment 0x08 */
	  .limit	= sizeof (taskstate) - 1,
	  .base		= farptr2linear (stack_adr (&taskstate)),
	  .type		= TSS_16bit_avail,
	  .cste_0	= 0,
	  .dpl		= 0,
	  .present	= 1,
	  .limit_msb	= 0,
	  .reserved	= 0,
	  .granularity	= 0,
	  .base_msb	= 0x00
	  }},
      { .code = {    /* segment 0x10 */
	  .limit	= 0xFFFF,
	  .base		= ((unsigned)getcs()) << 4,
	  .accessed	= 1,
	  .readable	= 1,
	  .cste_11	= 3,
	  .present	= 1,
	  .limit_msb	= 0x0,
	  .deflt	= 0,
	  .granularity	= 0
	  }},
      { .data = {	/* segment 0x18, doors opened */
	  .limit	= 0xFFFF,
	  .base		= 0,
	  .accessed	= 1,
	  .writable	= 1,
	  .cste_10	= 2,
	  .present	= 1,
	  .limit_msb	= 0xF,
	  .big		= 0,
	  .granularity	= 1
	  }},
      { .data = {	/* segment 0x20, local stack for copy */
	  .limit	= 0xFFFF,
	  .base		= ((unsigned)getss()) << 4,	/* only %sp used, not %esp */
	  .accessed	= 1,
	  .writable	= 1,
	  .cste_10	= 2,
	  .present	= 1,
	  .limit_msb	= 0x0,
	  .big		= 0,
	  .granularity	= 0
	  }},
    /* segment 0x28, 0x30, 0x38 */
      { .code = {}}, { .data = {}}, { .data = {}}
      };
  gdt_idt_param_t VCPI_gdt __attribute__ (( aligned(16) )) = {
      .limit = sizeof (VCPI_PMseg) - 1, /* -1 : see Intel doc */
      .base  = farptr2linear (stack_adr (VCPI_PMseg))
      };
  gdt_idt_param_t VCPI_idt __attribute__ (( aligned(16) )) = {
      .limit = 256 * 4, /* 0x400 given by sidt() in real mode */
      .base  = 0
      };
  struct _VCPI_switch_str VCPI_switch = {
      .cr3 = { .bit = { .pagedir_base = farptr2linear (pagedir) >> 12} },
      .gdtr_linadr = farptr2linear (stack_adr (&VCPI_gdt)),
      .idtr_linadr = farptr2linear (stack_adr (&VCPI_idt)),
      .ldtr = 0,
      .tr = 0x08,
      .entry_adr = 0, /* either switch2realmode or VCPI_copy16K */
      .entry_seg = 0x10
      };
  pagedir4K_t apagedir = { .bit = {
      .present		= 1,
      .read_write	= 1,
      .user_super	= 1,
      .write_through	= 1,
      .cache_disabled	= 0,
      .accessed		= 0,
      .reserved_0	= 0,
      .page_size_0	= 0,
      .global_page	= 0,
      .available	= 0,
      .base_address	= farptr2linear (pagetable) >> 12
      }};
  unsigned short first_unused;
  unsigned entrypoint;
  unsigned char result;

  ZDBG (("_VCPI_get_physical_address(getcs() = 0x%X >> 8) = 0x%X, ",
	getcs(), _VCPI_get_physical_address(getcs() >> 8)));
  ZDBG (("using pagedir 0x%X, pagetable 0x%X\r\n", pagedir, pagetable));

  fmemset (pagedir, 0, 2 * 4086);
  pokel (pagedir, apagedir.all);
  result = _VCPI_get_PM_interface (pagetable, &first_unused, &VCPI_PMseg[5], &entrypoint);
  if (result) {
      ZDBG (("_VCPI_get_PM_interface() returns 0x%X\r\n", result));
      return 1;
      }
  ZDBG (("_VCPI_get_PM_interface(): entry 0x%X, first_unused 0x%X\r\n",
		entrypoint, first_unused));

#ifdef VCPI_COPY
  {
  unsigned tmp = VCPI_copy (&VCPI_switch, entrypoint, pagetable, first_unused,
				&LOADER, pagedir + (0x200 << 16));
  if (tmp)
      return tmp;
  }
#endif

#ifdef TREAT_EXCEPTION
  /* We will not be able to call DOS to reset vectors later,
     so do it now (it is protected against double call) */
  restore_vectors();
#endif
#if (DEBUG & DEBUG_OUTPUT) == DOSFILE
  /* If debug to a DOS file, it will crash soon
   * because BIOS is partly working but DOS is in a
   * really bad state after _VCPI_switch_return_real_mode()...
   */
  DBG_END();
#endif

  disable_interrupt();
  /* Following uses hardcoded seg 0x18: */
  _VCPI_switch_virtual_to_real (&VCPI_switch);

  set_idt (&VCPI_idt);
  /* Now that the maximum improbability drive engine is energized,
     we just need just press the right randomized button: */
  clear_a_bit();
  /* Improbability now 1:10^2056 decreasing ...
     .....
     Probability back to 1:1 - reality restored,
     everything happening is back to normal - bugs are bugs again. */

  return 0;
  }}
#endif /* VCPI_SWITCH */

#if DISK_SUPPORT == DOS_SUPPORT	/* only for mingujin.exe */
LOAD_FCT_PREFIX(DOS_tiny_execute_extra_command) static
void DOS_tiny_execute_extra_command (void)
  {
  extern char extraexecute[80];
  if (extraexecute[0] != '\0') {
      char command[80], *dptr = command, *sptr = extraexecute + 1;

      while (*sptr != '\0' && *sptr != '\r' && *sptr != ' ' && *sptr != '\t')
	  *dptr++ = *sptr++;
      *dptr = '\0';
      dptr -= 4;
      if (*(unsigned *)dptr == *(unsigned *)".exe" || *(unsigned *)dptr == *(unsigned *)".com") {
	  /* We call DOS_exec_pgm() because it is faster and needs less memory
		but I do not know how to pass arguments (doesn't work) */
	  farptr psp = (unsigned)DOS_GetCurrentPSPSegment () << 16;
	  struct DOS_exec_str env_to_set = {
//	      .current_environment_segment = 0,	// i.e. copy current environment
	      .current_environment_segment = peekw (psp + 0x2C),
	      .command_tail_pointer = data_adr(extraexecute), // doesn't work as documented
	      .first_fcb = psp + 0x5C,
	      .second_fcb = psp + 0x6C,
	      };
	  unsigned short error = 0;
	  if (DOS_exec_pgm (command, &env_to_set, &error)) /* extension not used but shall be there */
	      printf ("%s\n (%s) error 0x%X\r\n", extraexecute+1, command, error);
	  }
	else {
	  /* We have a complete command line with arguments, pass to interpreter;
		may not work in .bat files */
	  unsigned short ret = DOS_exec_cmd (extraexecute);
	  if (ret) {	/* This return is not reliable */
	      ret = DOS_get_return_code();
	      if (ret)
		  printf ("%s\n error 0x%X\r\n", extraexecute, ret);
	      }
	  }
      }
  }
#else
#define DOS_tiny_execute_extra_command() /* nothing if not mingujin.exe */
#endif

/**
 ** System loading & starting:
 **/
struct elf_reloc_s {
    unsigned p_offset;
    unsigned p_paddr;
    unsigned p_filesz;
    unsigned p_memsz;
    };

#define MAXNBCPY 10 /* also change in assembler after the far jump */
struct latereloc_s {
      unsigned address;
      unsigned short segment;
      unsigned short nbcpy;
      struct { unsigned nb, src, dst, nbclear; } cpy[MAXNBCPY];
      } __attribute__ ((packed));

LOAD_FCT_PREFIX(adjust_addresses) static
unsigned adjust_addresses (unsigned nb_elf_reloc, const struct elf_reloc_s *elf_reloc,
			struct latereloc_s *cs_latereloc, struct linux_param *LnxParam)
  {
  unsigned cpt, relocate_offset = 0;

  if (LOADER.comment.elem.runadr == 0)
      LOADER.comment.elem.runadr = LOADER.comment.elem.late_relocation_address;
  cs_latereloc->nbcpy = 0;

 if (nb_elf_reloc == 0) {
      /* just trying to find "the secret of the Transfinite Drive" */
      if (LOADER.fileload[0].load_address != LOADER.comment.elem.late_relocation_address) {
	  cs_latereloc->cpy[cs_latereloc->nbcpy].nb = LOADER.fileload[0].uncompressed_size;
	  cs_latereloc->cpy[cs_latereloc->nbcpy].src = LOADER.fileload[0].load_address;
	  cs_latereloc->cpy[cs_latereloc->nbcpy].dst = LOADER.comment.elem.late_relocation_address;
	  cs_latereloc->cpy[cs_latereloc->nbcpy].nbclear = 0;
	  cs_latereloc->nbcpy ++;
	  }
      }
    else {
#define default_late_relocation_address	0x100000
      if (LOADER.comment.elem.late_relocation_address != default_late_relocation_address) {
	  /* Modified inside the kernel real-mode function, or given as a kernel parameter with min_gujin_version */
	  relocate_offset = LOADER.comment.elem.late_relocation_address - elf_reloc[0].p_paddr;
	  ZDBG (("\r\nOFFSETING ELF by 0x%X, hopes code compiled with '-PIC' or auto-relocate code is working and ld had '--emit-relocs' option and .rel.* section are not empty!\r\n", relocate_offset));
	  }
      for (cpt = 0; cpt < nb_elf_reloc; cpt++) {
	  cs_latereloc->cpy[cs_latereloc->nbcpy].nb = elf_reloc[cpt].p_filesz;
	  cs_latereloc->cpy[cs_latereloc->nbcpy].src = LOADER.fileload[0].load_address + elf_reloc[cpt].p_offset;
	  cs_latereloc->cpy[cs_latereloc->nbcpy].dst = elf_reloc[cpt].p_paddr + relocate_offset;
	  cs_latereloc->cpy[cs_latereloc->nbcpy].nbclear = elf_reloc[cpt].p_memsz - elf_reloc[cpt].p_filesz;
	  cs_latereloc->nbcpy ++;
	  }
      }

  if (LOADER.ramdisk_max > 1024 * LnxParam->ALT_MEM_K
      || LOADER.ramdisk_max <= LOADER.comment.elem.late_relocation_address + LOADER.fileload[0].uncompressed_size)
	LOADER.ramdisk_max = 1024 * LnxParam->ALT_MEM_K;
  if (LnxParam->in_out.ramdisk_size
	&& LnxParam->in_out.ramdisk_image != ((LOADER.ramdisk_max - LnxParam->in_out.ramdisk_size) & ~0x1FFFFF)) {
      /* move initrd at end of RAM */
      cs_latereloc->cpy[cs_latereloc->nbcpy].nb = LnxParam->in_out.ramdisk_size;
      cs_latereloc->cpy[cs_latereloc->nbcpy].src = LnxParam->in_out.ramdisk_image;
      cs_latereloc->cpy[cs_latereloc->nbcpy].dst = (LOADER.ramdisk_max - LnxParam->in_out.ramdisk_size) & ~0x1FFFFF; /* 2 Mb page aligned */
      cs_latereloc->cpy[cs_latereloc->nbcpy].nbclear = 0;
      LnxParam->in_out.ramdisk_image = cs_latereloc->cpy[cs_latereloc->nbcpy].dst;
      ZDBG (("\r\nLate relocation of initrd at 0x%X", LnxParam->in_out.ramdisk_image));
      cs_latereloc->nbcpy ++;
      }

  if (LOADER.comment.elem.realfct_size != 0 && LOADER.comment.elem.paramadr != 0 && LOADER.comment.elem.paramadr < 0x100000) {
      /* let's bring Claribel the canary in our next trip! */
      cs_latereloc->cpy[cs_latereloc->nbcpy].nb = sizeof(*LnxParam);
      cs_latereloc->cpy[cs_latereloc->nbcpy].src = farptr2linear(data_adr(LnxParam));
      cs_latereloc->cpy[cs_latereloc->nbcpy].dst = LOADER.comment.elem.paramadr;
      cs_latereloc->cpy[cs_latereloc->nbcpy].nbclear = 0;
      cs_latereloc->nbcpy ++;
      }

  cs_latereloc->segment = 0x0010;	/* __KERNEL_CS */
  if (LOADER.adjust_runadr && relocate_offset) {
      cs_latereloc->address = LOADER.comment.elem.runadr + relocate_offset;
      ZDBG (("\r\nAdjusting runadr after relocation: +0x%X", relocate_offset));
      }
    else
      cs_latereloc->address = LOADER.comment.elem.runadr;

  ZDBG (("\r\nnumber of late relocations: %u (%u in ELF file), run address 0x%X\r\n", cs_latereloc->nbcpy, nb_elf_reloc, cs_latereloc->address));
  for (cpt = 0; cpt < cs_latereloc->nbcpy; cpt++)
      ZDBG (("  0x%X bytes from 0x%X to 0x%X, then clear 0x%X bytes\r\n",
		cs_latereloc->cpy[cpt].nb, cs_latereloc->cpy[cpt].src,
		cs_latereloc->cpy[cpt].dst, cs_latereloc->cpy[cpt].nbclear));
  return relocate_offset;
  }

LOAD_FCT_PREFIX(copy_from_first_file) static int
copy_from_first_file (unsigned offset, farptr buff, unsigned size)
  {
  if (HIMEM_ACTIVE()) {
      unsigned char error;
      XMS_EMM emm_struct = {
	  .size = (size + 1) & ~1,	/* even number of bytes */
	  .src_handle = LOADER.fileload[0].handle,
	  /* If HIMEM is active,the first file (kernel) has been loaded at offset
		zero in the HIMEM handle (maybe also the second (initrd) file if
		it was not concat'ed), and LOADER.fileload[0].load_address is the
		LOCKED address in memory; here we want the offset into the handle,
		so adjust: */
	  .src_offset = offset,
	  .dst_handle = 0,          /* real mode address */
	  .dst_offset = buff,
	  };
      if (buff == data_adr(fourKbuffer))
	  ZDBG (("_XMS_move_memory (offset 0x%X) to fourKbuffer: ", offset));
      else
	  ZDBG (("_XMS_move_memory (offset 0x%X) to buff at 0x%X: ", offset, buff));
      if (_XMS_move_memory (UTIL.HIMEM_entrypoint, &emm_struct, &error) != 0) {
	  return 1;
	  ZDBG (("error 0x%X!\r\n", error));
	  }
	else
	  ZDBG (("OK\r\n"));
      }
    else {
#if SETUP & USE_INT1587
      unsigned short res = 0xFFFF;
#if USER_SUPPORT & VESA_SUPPORT
      if (UI.parameter.winsize != VESA2_MARKER)
#endif
	{
	ZDBG (("_BIOS_copy_extended_memory (offset 0x%X, %u halfwords) to buff: ", offset, (size + 1)/2));
	res = _BIOS_copy_extended_memory (farptr2linear(buff), LOADER.fileload[0].load_address + offset, (size + 1)/2);
	}
      if (res == 0)
	  ZDBG (("OK\r\n"));
	else {
	  ZDBG (("error 0x%X, try memcpy_door_open() ", res));
#else
	  {
	  ZDBG (("memcpy_door_open(0x%X, 0x%X, %u): ", farptr2dsrelative(buff), linear2dsrelative(LOADER.fileload[0].load_address + offset), size));
#endif
	  memcpy_door_open ((void *)farptr2dsrelative(buff), (void *)linear2dsrelative(LOADER.fileload[0].load_address + offset), size);
	  ZDBG (("done\r\n"));
	  }
      }
  return 0;
  }

LOAD_FCT_PREFIX(copy_to_first_file) static int
copy_to_first_file (unsigned offset, farptr buff, unsigned size)
  {
  if (HIMEM_ACTIVE()) {
      unsigned char error;
      XMS_EMM emm_struct = {
	  .size = (size + 1) & ~1,	/* even number of bytes */
	  .src_handle = 0,          /* real mode address */
	  .src_offset = buff,
	  .dst_handle = LOADER.fileload[0].handle,
	  /* If HIMEM is active,the first file (kernel) has been loaded at offset
		zero in the HIMEM handle (maybe also the second (initrd) file if
		it was not concat'ed), and LOADER.fileload[0].load_address is the
		LOCKED address in memory; here we want the offset into the handle,
		so adjust: */
	  .dst_offset = offset,
	  };
      if (buff == data_adr(fourKbuffer))
	  ZDBG (("_XMS_move_memory (offset 0x%X) from fourKbuffer: ", offset));
      else
	  ZDBG (("_XMS_move_memory (offset 0x%X) from buff at 0x%X: ", offset, buff));
      if (_XMS_move_memory (UTIL.HIMEM_entrypoint, &emm_struct, &error) != 0) {
	  return 1;
	  ZDBG (("error 0x%X!\r\n", error));
	  }
	else
	  ZDBG (("OK\r\n"));
      }
    else {
#if SETUP & USE_INT1587
      unsigned short res = 0xFFFF;
#if USER_SUPPORT & VESA_SUPPORT
      if (UI.parameter.winsize != VESA2_MARKER)
#endif
	{
	ZDBG (("_BIOS_copy_extended_memory (offset 0x%X, %u halfwords) from buff: ", offset, (size + 1)/2));
	res = _BIOS_copy_extended_memory (LOADER.fileload[0].load_address + offset, farptr2linear(buff), (size + 1)/2);
	}
      if (res == 0)
	  ZDBG (("OK\r\n"));
	else {
	  ZDBG (("error 0x%X, try memcpy_door_open() ", res));
#else
	  {
	  ZDBG (("memcpy_door_open(): "));
#endif
	  memcpy_door_open ((void *)linear2dsrelative(LOADER.fileload[0].load_address + offset), (void *)farptr2dsrelative(buff), size);
	  ZDBG (("done\r\n"));
	  }
      }
  return 0;
  }

/* Some linker documentation at http://www.iecc.com/linker/linker07.html */
LOAD_FCT_PREFIX(adjust_ELF_relocation) static void
adjust_ELF_relocation (int delta)
  {bound_stack();{
  union { struct ELF32_header elf32; struct ELF64_header elf64; } header;
  unsigned char buffer_reloc[1024]; /* increase to be quicker, but take care of stack size. */

  ZDBG (("\r\n%s(0x%X): ", __FUNCTION__, delta));
  if (LOADER.uncompressed_signature != *(unsigned*)"\x7F""ELF") {
      ZDBG (("not ELF file.\r\n"));
      return;
      }
  copy_from_first_file (0, stack_adr(&header), sizeof(header));

  unsigned long long rdtsc_start = (UTIL.processor.calibrate_rdtsc)? rdtsc() : 0;
  unsigned minreloced = 0xFFFFFFFFU, maxreloced = 0, nb_relocation_treated = 0;
  unsigned currentblock_treated = 0xFFFFFFFFU; /* copy_{form|to}_first_file() offset currently in fourKbuffer */

  if (header.elf32.EI_CLASS == ELFCLASS32 && header.elf32.e_phnum != 0 && header.elf32.e_shnum != 0) {
      union {
	    struct ELF32_pgm_header header;
	    unsigned char sized[header.elf32.e_phentsize];
	  } pgm[header.elf32.e_phnum];
      const unsigned e_phnum = header.elf32.e_phnum, e_shnum = header.elf32.e_shnum,
				e_shoff = header.elf32.e_shoff, e_shentsize = header.elf32.e_shentsize;
      unsigned phnum, shnum;
      unsigned allowed_range_low = 0xFFFFFFFFU, allowed_range_high = 0;

      ZDBG (("\r\nTreating ELF32 (entry 0x%X, %u progheader of %u bytes at %u, %u sectheader of %u bytes at %u)\r\n",
	header.elf32.e_entry, header.elf32.e_phnum, header.elf32.e_phentsize, header.elf32.e_phoff, header.elf32.e_shnum, header.elf32.e_shentsize, header.elf32.e_shoff));

      if (delta == 0)
	  return;

      print (MSG_RELOCATING);
      copy_from_first_file (header.elf32.e_phoff, stack_adr(&pgm), sizeof(pgm));

      for (phnum = 0; phnum < e_phnum; phnum++) {
	  if (pgm[phnum].header.p_vaddr == 0)
	      continue; /* .realmode */
	  if (pgm[phnum].header.p_type != PT_LOAD)
	      continue;
	  if (!pgm[phnum].header.p_flags.PF_X && !pgm[phnum].header.p_flags.PF_W && !pgm[phnum].header.p_flags.PF_R)
	      continue;
	  if (pgm[phnum].header.p_vaddr < allowed_range_low)
	      allowed_range_low = pgm[phnum].header.p_vaddr;
	  if (pgm[phnum].header.p_vaddr + pgm[phnum].header.p_memsz > allowed_range_high)
	      allowed_range_high = pgm[phnum].header.p_vaddr + pgm[phnum].header.p_memsz;
	  }
      ZDBG (("Range of program loaded: 0x%X..0x%X\r\n", allowed_range_low, allowed_range_high));

      for (phnum = shnum = 0; shnum < e_shnum; shnum++) {
	  union {
	      struct ELF32_sect_header header;
	      unsigned char sized[e_shentsize];
	      } sect, linked_sect;
	  copy_from_first_file (e_shoff + shnum * e_shentsize, stack_adr(&sect), e_shentsize);
	  unsigned sh_offset = sect.header.sh_offset, sh_size = sect.header.sh_size;

	  ZDBG (("section %u sh_type %u sh_flags 0x%X sh_addr 0x%X sh_offset 0x%X sh_size 0x%X sh_link %u sh_info %u sh_addralign %u sh_entsize %u\r\n",
		shnum, sect.header.sh_type, *CAST (unsigned *, &sect.header.sh_flags),
		sect.header.sh_addr, sect.header.sh_offset, sect.header.sh_size,
		sect.header.sh_link, sect.header.sh_info, sect.header.sh_addralign, sect.header.sh_entsize));
	  if (sect.header.sh_type != SHT_REL && sect.header.sh_type != SHT_RELA)
	      continue;
	  if (sect.header.sh_flags.SHF_WRITE || sect.header.sh_flags.SHF_ALLOC || sect.header.sh_flags.SHF_EXECINSTR)
	      continue;
	  if (sect.header.sh_entsize == 0 || sect.header.sh_entsize > 1024)
	      continue;
	  if ((sect.header.sh_size % sect.header.sh_entsize) != 0)
	      continue;
	  if (sect.header.sh_info < e_shnum)
	      copy_from_first_file (e_shoff + sect.header.sh_info * e_shentsize, stack_adr(&linked_sect), e_shentsize);
	  while (sh_size > 0) {
	      unsigned nbreloc = ((sh_size > sizeof(buffer_reloc)) ? sizeof(buffer_reloc) : sh_size) / sect.header.sh_entsize;
	      struct ELF32_reloca *reloc = (struct ELF32_reloca *)(buffer_reloc - sect.header.sh_entsize);

	      copy_from_first_file (sh_offset, stack_adr(buffer_reloc), nbreloc * sect.header.sh_entsize);
	      sh_size -= nbreloc * sect.header.sh_entsize;
	      sh_offset += nbreloc * sect.header.sh_entsize;

	      while (nbreloc--) {
		  reloc = (struct ELF32_reloca *)((unsigned long)reloc + sect.header.sh_entsize);

#if DEBUG & DEBUG_LOADER
		  char dbgmsg[100];
		  if (sect.header.sh_type == SHT_RELA)
		      sprintf (dbgmsg, "Relocation to addr 0x%X symbol 0x%X type 0x%X addend 0x%X", reloc->addr, reloc->symbol, reloc->type, reloc->addend);
		    else
		      sprintf (dbgmsg, "Relocation to addr 0x%X symbol 0x%X type 0x%X", reloc->addr, reloc->symbol, reloc->type);
#endif
		  unsigned finite_loop = 2;
		  while (reloc->addr < pgm[phnum].header.p_vaddr || reloc->addr >= pgm[phnum].header.p_vaddr + pgm[phnum].header.p_memsz)
		      if (++phnum >= e_phnum) {
			  phnum = 0;
			  if (--finite_loop == 0) {
			      ZDBG (("%s not in a loaded segment, IGNORED\r\n", dbgmsg));
			      break;
			      }
			  }
		  if (finite_loop == 0)
		      continue;
		  if (sect.header.sh_info < e_shnum && (reloc->addr < linked_sect.header.sh_addr || reloc->addr >= linked_sect.header.sh_addr + linked_sect.header.sh_size)) {
		      ZDBG (("%s not in linked section %u loaded part (0x%X..0x%X), IGNORED\r\n", dbgmsg, sect.header.sh_info, linked_sect.header.sh_addr, linked_sect.header.sh_addr + linked_sect.header.sh_size));
		      continue;
		      }
		  if (pgm[phnum].header.p_vaddr == 0) {
		      ZDBG (("%s in .realmode, ignore it\r\n", dbgmsg));
		      continue;
		      }
		  if (reloc->type != R_386_32)
		      continue;

		  unsigned addr = reloc->addr - pgm[phnum].header.p_vaddr, *contentptr;
		  if (addr >= pgm[phnum].header.p_filesz) {
		      ZDBG (("%s in BSS, ignore it - is it really possible?\r\n", dbgmsg));
		      continue;
		      }
		  addr += pgm[phnum].header.p_offset;
		  if (currentblock_treated != (addr & ~(sizeof(fourKbuffer)-1))) {
		      if ((currentblock_treated & (sizeof(fourKbuffer)-1)) == 0)
			  copy_to_first_file (currentblock_treated, data_adr(fourKbuffer), sizeof(fourKbuffer));
		      currentblock_treated = addr & ~(sizeof(fourKbuffer)-1);
		      copy_from_first_file (currentblock_treated, data_adr(fourKbuffer), sizeof(fourKbuffer));
		      contentptr = (unsigned *)(fourKbuffer + (addr & (sizeof(fourKbuffer)-1) ));
		      }
		    else if ((addr & (sizeof(fourKbuffer)-1)) > sizeof(fourKbuffer) - 4) { /* for 32 bits reloc, 64 bits may still cross boundary here */
		      if ((currentblock_treated & (sizeof(fourKbuffer)-1)) == 0)
			  copy_to_first_file (currentblock_treated, data_adr(fourKbuffer), sizeof(fourKbuffer));
		      currentblock_treated = addr;
		      copy_from_first_file (currentblock_treated, data_adr(fourKbuffer), 8); /* enought for 64 bits relocs */
		      contentptr = (unsigned *)fourKbuffer;
		      }
		    else
		      contentptr = (unsigned *)(fourKbuffer + (addr & (sizeof(fourKbuffer)-1) ));
		  if (*contentptr < allowed_range_low || *contentptr > allowed_range_high) {
//		  if ((*contentptr & 0x3FFFFFFF) < (allowed_range_low & 0x3FFFFFFF) || (*contentptr & 0x3FFFFFFF) > (allowed_range_high & 0x3FFFFFFF)) {
			ZDBG (("%s in section index %u, so at 0x%X content 0x%X out of range, IGNORED\r\n", dbgmsg, phnum, addr, *contentptr));
			/* Here are the whole lot in this kernel:
$ readelf -r tstrelo1.elf | grep "R_386_32          f"
c0202a2a  006c9b01 R_386_32          ffffe440   __kernel_rt_sigreturn
c0202b79  0078ab01 R_386_32          ffffe420   __kernel_sigreturn
c0203007  007c4301 R_386_32          ffffe410   SYSENTER_RETURN
c02757ec  00607b01 R_386_32          ffffe400   __kernel_vsyscall
			*/
			continue;
			}
		  if (reloc - (struct ELF32_reloca *)buffer_reloc <= 10) {
			ZDBG (("%s in section index %u, so at 0x%X content 0x%X adds 0x%X\r\n", dbgmsg, phnum, addr, *contentptr, delta));
			if (reloc - (struct ELF32_reloca *)buffer_reloc == 10)
			    ZDBG (("...more following...\r\n"));
			}
		  if (*contentptr < minreloced)
		      minreloced = *contentptr;
		    else if (*contentptr > maxreloced)
		      maxreloced = *contentptr;
		  *contentptr += delta;
		  nb_relocation_treated++;
		  if (currentblock_treated & (sizeof(fourKbuffer)-1)) { /* We had a crossing relocation */
		      copy_to_first_file (currentblock_treated, data_adr(fourKbuffer), 8); /* enought for 64 bits relocs */
		      currentblock_treated = 0xFFFFFFFFU;
		      }
		  }
	      }
	  }
      }
    else if (header.elf64.EI_CLASS == ELFCLASS64 && header.elf64.e_phnum != 0 && header.elf64.e_shnum != 0) {
      union {
	    struct ELF64_pgm_header header;
	    unsigned char sized[header.elf64.e_phentsize];
	  } pgm[header.elf64.e_phnum];
      const unsigned e_phnum = header.elf64.e_phnum, e_shnum = header.elf64.e_shnum,
				e_shoff = header.elf64.e_shoff, e_shentsize = header.elf64.e_shentsize; /* no need for 64 bits for e_shentsize */
      unsigned phnum, shnum;
      unsigned long long allowed_range_low = 0xFFFFFFFFFFFFFFFFLLU, allowed_range_high = 0;

      ZDBG (("\r\nTreating ELF64 (entry 0x%llX, %u progheader of %u bytes at 0x%llX, %u sectheader of %u bytes at 0x%llX)\r\n",
	header.elf64.e_entry, header.elf64.e_phnum, header.elf64.e_phentsize, header.elf64.e_phoff, header.elf64.e_shnum, header.elf64.e_shentsize, header.elf64.e_shoff));

      if (delta == 0)
	  return;

      print (MSG_RELOCATING);
      copy_from_first_file (header.elf64.e_phoff, stack_adr(&pgm), sizeof(pgm));

      for (phnum = 0; phnum < e_phnum; phnum++) {
	  if (pgm[phnum].header.p_vaddr == 0)
	      continue; /* .realmode */
	  if (pgm[phnum].header.p_type != PT_LOAD)
	      continue;
	  if (!pgm[phnum].header.p_flags.PF_X && !pgm[phnum].header.p_flags.PF_W && !pgm[phnum].header.p_flags.PF_R)
	      continue;
	  if (pgm[phnum].header.p_vaddr < allowed_range_low)
	      allowed_range_low = pgm[phnum].header.p_vaddr;
	  if (pgm[phnum].header.p_vaddr + pgm[phnum].header.p_memsz > allowed_range_high)
	      allowed_range_high = pgm[phnum].header.p_vaddr + pgm[phnum].header.p_memsz;
	  }
      ZDBG (("Range of program loaded: 0x%llX..0x%llX\r\n", allowed_range_low, allowed_range_high));

      for (phnum = shnum = 0; shnum < e_shnum; shnum++) {
	  union {
	      struct ELF64_sect_header header;
	      unsigned char sized[header.elf64.e_shentsize];
	      } sect, linked_sect;
	  copy_from_first_file (e_shoff + shnum * e_shentsize, stack_adr(&sect), e_shentsize);
	  const unsigned sh_entsize = sect.header.sh_entsize; /* no need to deal with 64 bits number for sh_entsize */
	  unsigned sh_offset = sect.header.sh_offset, sh_size = sect.header.sh_size;

	  ZDBG (("section %u sh_type %u sh_flags 0x%X sh_addr 0x%llX sh_offset 0x%llX sh_size 0x%llX sh_link %u sh_info %u sh_addralign 0x%llX sh_entsize 0x%llX\r\n",
		shnum, sect.header.sh_type, *CAST (unsigned *, &sect.header.sh_flags),
		sect.header.sh_addr, sect.header.sh_offset, sect.header.sh_size,
		sect.header.sh_link, sect.header.sh_info, sect.header.sh_addralign, sect.header.sh_entsize));
	  if (sect.header.sh_type != SHT_REL && sect.header.sh_type != SHT_RELA)
	      continue;
	  if (sect.header.sh_flags.SHF_WRITE || sect.header.sh_flags.SHF_ALLOC || sect.header.sh_flags.SHF_EXECINSTR)
	      continue;
	  if (sect.header.sh_entsize == 0 || sect.header.sh_entsize > 1024)
	      continue;
	  if (((unsigned)sect.header.sh_size % sh_entsize) != 0)
	      continue;
	  if (sect.header.sh_info < e_shnum)
	      copy_from_first_file (e_shoff + sect.header.sh_info * e_shentsize, stack_adr(&linked_sect), e_shentsize);
	  while (sh_size > 0) {
	      unsigned nbreloc = ((sh_size > sizeof(buffer_reloc)) ? sizeof(buffer_reloc) : sh_size) / sh_entsize;
	      struct ELF64_reloca *reloc = (struct ELF64_reloca *)(buffer_reloc - sh_entsize);

	      copy_from_first_file (sh_offset, stack_adr(buffer_reloc), nbreloc * sh_entsize);
	      sh_size -= nbreloc * sh_entsize;
	      sh_offset += nbreloc * sh_entsize;

	      while (nbreloc--) {
		  reloc = (struct ELF64_reloca *)((unsigned long)reloc + sh_entsize);

#if DEBUG & DEBUG_LOADER
		  char dbgmsg[100];
		  if (sect.header.sh_type == SHT_RELA)
		      sprintf (dbgmsg, "Relocation to addr 0x%llX symbol 0x%X type 0x%X addend 0x%llX", reloc->addr, reloc->symbol, reloc->type, reloc->addend);
		    else
		      sprintf (dbgmsg, "Relocation to addr 0x%llX symbol 0x%X type 0x%X", reloc->addr, reloc->symbol, reloc->type);
#endif
		  unsigned finite_loop = 2;
		  while (reloc->addr < pgm[phnum].header.p_vaddr || reloc->addr >= pgm[phnum].header.p_vaddr + pgm[phnum].header.p_memsz)
		      if (++phnum >= e_phnum) {
			  phnum = 0;
			  if (--finite_loop == 0) {
			      ZDBG (("%s not in a loaded segment, IGNORED\r\n", dbgmsg));
			      break;
			      }
			  }
		  if (finite_loop == 0)
		      continue;
		  if (sect.header.sh_info < e_shnum && (reloc->addr < linked_sect.header.sh_addr || reloc->addr >= linked_sect.header.sh_addr + linked_sect.header.sh_size)) {
		      ZDBG (("%s not in linked section %u loaded part (0x%llX..0x%llX), IGNORED\r\n", dbgmsg, sect.header.sh_info, linked_sect.header.sh_addr, linked_sect.header.sh_addr + linked_sect.header.sh_size));
		      continue;
		      }
		  if (pgm[phnum].header.p_vaddr == 0) {
		      ZDBG (("%s in .realmode, ignore it\r\n", dbgmsg));
		      continue;
		      }
		  if (reloc->type != R_X86_64_32 && reloc->type != R_X86_64_64)
		      continue;

		  unsigned long long addr = reloc->addr - pgm[phnum].header.p_vaddr, *contentptr, content;
		  if (addr >= pgm[phnum].header.p_filesz) {
		      ZDBG (("%s in BSS, ignore it - is it really possible?\r\n", dbgmsg));
		      continue;
		      }
		  addr += pgm[phnum].header.p_offset;
		  if (currentblock_treated != (addr & ~(sizeof(fourKbuffer)-1))) {
		      if ((currentblock_treated & (sizeof(fourKbuffer)-1)) == 0)
			  copy_to_first_file (currentblock_treated, data_adr(fourKbuffer), sizeof(fourKbuffer));
		      currentblock_treated = addr & ~(sizeof(fourKbuffer)-1);
		      copy_from_first_file (currentblock_treated, data_adr(fourKbuffer), sizeof(fourKbuffer));
		      contentptr = (unsigned long long *)(fourKbuffer + (addr & (sizeof(fourKbuffer)-1) ));
		      }
		    else if ((addr & (sizeof(fourKbuffer)-1)) > sizeof(fourKbuffer) - 8) {
		      if ((currentblock_treated & (sizeof(fourKbuffer)-1)) == 0)
			  copy_to_first_file (currentblock_treated, data_adr(fourKbuffer), sizeof(fourKbuffer));
		      currentblock_treated = addr;
		      copy_from_first_file (currentblock_treated, data_adr(fourKbuffer), 8); /* enought for 64 bits relocs */
		      contentptr = (unsigned long long *)fourKbuffer;
		      }
		    else
		      contentptr = (unsigned long long *)(fourKbuffer + (addr & (sizeof(fourKbuffer)-1) ));
		  content = *contentptr;
		  if (reloc->type == R_X86_64_32)
		      content &= 0xFFFFFFFF;
		  if (content < allowed_range_low || content > allowed_range_high) {
			ZDBG (("%s in section index %u, so at 0x%llX content 0x%llX out of range, IGNORED\r\n", dbgmsg, phnum, addr, content));
			continue;
			}
		  if (reloc - (struct ELF64_reloca *)buffer_reloc <= 10) {
			ZDBG (("%s in section index %u, so at 0x%llX content 0x%llX adds 0x%X\r\n", dbgmsg, phnum, addr, content, delta));
			if (reloc - (struct ELF64_reloca *)buffer_reloc == 10)
			    ZDBG (("...more following...\r\n"));
			}
		  if (content < minreloced)
		      minreloced = content;
		    else if (content > maxreloced)
		      maxreloced = content;
		  *contentptr += delta;	/* little endian */
		  nb_relocation_treated++;
		  if (currentblock_treated & (sizeof(fourKbuffer)-1)) { /* We had a crossing relocation */
		      copy_to_first_file (currentblock_treated, data_adr(fourKbuffer), 8); /* enought for 64 bits relocs */
		      currentblock_treated = 0xFFFFFFFFU;
		      }
		  }
	      }
	  }
      }

  if ((currentblock_treated & (sizeof(fourKbuffer)-1)) == 0)
      copy_to_first_file (currentblock_treated, data_adr(fourKbuffer), sizeof(fourKbuffer));
  if (UTIL.processor.calibrate_rdtsc > 10) {
      rdtsc_start -= rdtsc();
      ZDBG (("%u relocations computed in %u us, before-reloc ranges: 0x%X..0x%X\r\n",
		nb_relocation_treated, (unsigned)rdtsc_start / (UTIL.processor.calibrate_rdtsc / 10), minreloced, maxreloced));
      }
  }}


/* Someone has documentation for this one? eax and ebx modified? */
extern inline void
_BIOS_we_go_to_long_mode (unsigned longmode2_else1)
  {
  asm volatile (" int	$0x15	"
		: : "a" (0xec00),	// declare target operating mode
		    "b" (longmode2_else1));	// long mode is surely 2
  }

LOAD_FCT_PREFIX(prepare_exit) static unsigned
prepare_exit (unsigned set_kbd_and_open_A20)
{
  if (_BIOS_control_break_pressed()) {
      if (_BIOS_checkkey())
	  _BIOS_getkey();
      ZDBG (("Abort loading because ControlBreak pressed.\r\n"));
      return 0x18; /* ERROR_CONTROL_BREAK */
      }

  if (set_kbd_and_open_A20)
      _BIOS_set_repeat_rate_delay (0, 0); /* max repeat rate, min delay */

#if 0
  /* Network may use EMM */
  if (_BIOS_NETWORK_installed())
      ZDBG (("Will have to stop network soon, after closing DBG?\r\n"));

  if (BOOT1_DOS_running()) {
      unsigned short protman_handle = 0;

      if (DOS_OpenFile ("PROTMAN$", DOS_open_read_only, &protman_handle) == 0) {
	  struct NDIS_s GetProtocolManagerInfo = { .command = 1 };
	  unsigned short error_nbbytes = 0;
	  if (_BIOS_NDIS_execute (protman_handle, &GetProtocolManagerInfo, &error_nbbytes))
	      ZDBG (("NDIS seems abscent, error 0x%X\r\n", error_nbbytes));
	    else
	      ZDBG (("NDIS seems present, no error 0x%X, status 0x%X, ptr 0x%X, version 0x%X\r\n",
			error_nbbytes, GetProtocolManagerInfo.status, GetProtocolManagerInfo.param1, GetProtocolManagerInfo.extra));
	  }
	else {
	  ZDBG (("NDIS: Failed opening 'PROTMAN$', error 0x%X\r\n", protman_handle));
	  protman_handle = 0;
	  }
/* How to stop network???
--------N-6124-------------------------------
INT 61 - PC/TCP kernel v2.05+ - "net_abortall" - RESET ALL NETWORK CONNECTIONS
	AH = 24h
Return: always successful
Note:   performs "net_abort" (AH=19h) on all open non-global descriptors
SeeAlso: INT 61"FTP Software",INT 61/AH=00h"PC/TCP",INT 61/AH=19h
--------b-6124-------------------------------
--------N-6316-------------------------------
INT 63 - BW-TCP - TCPIP.SYS - RESET NETWORK CONNECTION
	AH = 16h
	DS:DI -> ???
Return: ???
Note:   calls AH=17h after preprocessing
SeeAlso: AH=17h,INT 61/AH=19h"PC/TCP"
--------N-6317-------------------------------
*/

      /* move after DBG_END() later on to be able to have debug file DBG on a network drive ? */
      if (protman_handle) {
	  struct NDIS_s UnbindAndStop = { .command = 8 };
	  unsigned short error_nbbytes = 0, errorcode;
	  if (_BIOS_NDIS_execute (protman_handle, &UnbindAndStop, &error_nbbytes))
	      ZDBG (("NDIS UnbindAndStop error 0x%X\r\n", error_nbbytes));
	    else
	      ZDBG (("NDIS UnbindAndStop no error 0x%X, status 0x%X, failing modules 0x%X, param2 0x%X\r\n",
			error_nbbytes, UnbindAndStop.status, UnbindAndStop.param1, UnbindAndStop.param2));
	  if (DOS_CloseFile (protman_handle, &errorcode))
	      ZDBG (("NDIS: Cannot close handle %u, errorcode 0x%X\r\n", protman_handle, errorcode));
	  }
      }
#endif

#if !(SETUP & BIOS_ONLY)
  if (UTIL.emm_type == EMM_emmx || UTIL.emm_type == EMM_emmq) {
      unsigned short err;
      ZDBG ((" [EMM*X present: "));
      DOS_ResetDisk();
#ifdef VCPI_SWITCH
      if (UTIL.VCPI_version != 0) {
	  ZDBG (("_VCPI_get_physical_address(paramadr = 0x%X >> 12) = 0x%X, ",
		LOADER.comment.elem.paramadr,
		_VCPI_get_physical_address(LOADER.comment.elem.paramadr >> 12)));

	  err = VCPI_virtual2real ();

	  if (err != 0)
	      return 5;
	  UTIL.emm_type = EMM_VCPI_suspended; /* for set_A20() */
	  ZDBG (("_VCPI_switch_to_protected_mode (realmode) OK\r\n"));
	  }
	else
#endif
	     if ((err = init_broadcast (0x30A,	/* Window 3.10, always */
			&UTIL.v86_info,
			&UTIL.v86_callback)) == 0) {
	  if (   UTIL.v86_callback == 0
	      || virtual86 (UTIL.v86_callback, 0) != 0) {
	      ZDBG (("init_broadcast switch realmode failed] "));
	      exit_broadcast();
	      return 3;
	      }
	    else {
	      ZDBG (("init_broadcast switch realmode OK] "));
	      }
	  }
	else {
	  ZDBG (("init_broadcast failed, error 0x%X] ", err));
	  /* to read EMMQ message: did not find C:\dos\emm386.exe */
	  _BIOS_wait (3 * 1000 * 1000);
	  return 4;
	  }
      /* Now we are in real mode, interrupts disabled */
      UTIL.emm_type |= 0x10; /* suspend it for set_A20() */
      ZDBG (("\r\n"));
      }
    else {
      ZDBG ((" [EMM*X abscent] "));
      UTIL.v86_callback = 0;
      }
#endif /* SETUP & BIOS_ONLY */

#if !(SETUP & USE_INT1587)
  if (HIMEM_ACTIVE() && set_kbd_and_open_A20)
#endif
      {
      /* HIMEM and INT1587 may enable it only with I/O 0x92 */
      /* Maybe has already been enabled by VESA2, but redo it: */
      if (set_A20 (1) != 0) {
#if !(SETUP & BIOS_ONLY)
	  UTIL.emm_type &= ~0x10;	/* un-suspend it */
	  if (UTIL.v86_callback != 0)
	      exit_broadcast();
#endif
	  return 1;
	  }
      }

#if DISK_SUPPORT & IDE_SUPPORT
  if (STATE.has_just_been_uninstalled == 1367) {
/*      puts ("IDE freeze request ignored after uninstall."); _BIOS_wait (10000000); */
      }
    else {
      ZDBG ((" [freeze_IDE: "));
      freeze_IDE (*CAST (const struct freeze_IDE_s *, &LOADER.comment.elem.option));
      ZDBG (("\r\nOK] "));
      }
#endif

#ifdef TREAT_EXCEPTION
  restore_vectors();
#endif

  DBG_END();
  return 0;
}

LOAD_FCT_PREFIX(elf_start) static unsigned __attribute__ ((noinline))
elf_start (unsigned linadr_togpl, struct linux_param *LnxParam,
		unsigned nb_elf_reloc, const struct elf_reloc_s *elf_reloc)
  {bound_stack();{
  unsigned nbdelay;
#ifdef LOAD_TASKREG
  static TSS_t taskstate; /* inited at zero because in BSS */
  static seg_t PMSeg[] __attribute__ (( aligned(8) )) = {
#else
  static const seg_t PMSeg[] asm ("PMseg") __attribute__ (( aligned(8) )) = {
#endif
    { .data = {}}, /* segment 0 is the invalid one */
    { .system = {	/* segment 0x08 */
#ifdef LOAD_TASKREG
    .limit	= sizeof (taskstate) - 1,
//    .base	= farptr2linear (data_adr (&taskstate)),
    .type	= TSS_32bit_avail,
    .cste_0	= 0,
    .dpl	= 0,
    .present	= 1,
    .limit_msb	= 0,
    .reserved	= 0,
    .granularity	= 0,
    .base_msb	= 0x00
#endif
    }},
    { .code = {	/* segment 0x10 : __KERNEL_CS , pre Linux-2.5, post Linux-2.6.5 */
    .limit	= 0xFFFF,
    .base	= 0,
    .accessed	= 1,	/* 486+: do not update it if it is already set */
    .readable	= 1,
    .cste_11	= 3,
    .present	= 1,
    .limit_msb	= 0xF,
    .deflt	= 1,
    .granularity	= 1	/* limit = 0xFFFFF * 4 Kb = 4 Gb */
    }},
    { .data = {	/* segment 0x18 : __KERNEL_DS , pre Linux-2.5, post Linux-2.6.5 */
    .limit	= 0xFFFF,
    .base	= 0,
    .accessed	= 1,	/* 486+: do not update it if it is already set */
    .writable	= 1,
    .cste_10	= 2,
    .present	= 1,
    .limit_msb	= 0xF,
//    .big	= 1,
    .granularity	= 1
    }},
    };
  /* NOTE: On some CPUs, the GDT _must_ be 8 byte aligned. */
  gdt_idt_param_t gdt __attribute__ (( aligned(16) )) = {
      .limit = sizeof (PMSeg) - 1, /* -1 : see Intel doc */
      .base  = farptr2linear (data_adr (PMSeg))
      };
  unsigned relocate_offset;
#ifdef LOAD_TASKREG
  PMSeg[1].system.base  = farptr2linear (data_adr (&taskstate));
#endif

  if (_BIOS_get_shift_flags().control_key_pressed)
	nbdelay = 1500; /* * 1/100 s */
      else
	nbdelay = 0;

  {
  extern char patchdata[];
  struct latereloc_s cs_latereloc;

  relocate_offset = adjust_addresses (nb_elf_reloc, elf_reloc, &cs_latereloc, LnxParam);

#ifdef CALLING_FROM_XCODESEG
  smemcpy (loadcode_adr(patchdata), &cs_latereloc, sizeof(cs_latereloc));
#else
  smemcpy (code_adr(patchdata), &cs_latereloc, sizeof(cs_latereloc));
#endif
//  asm ("wbinvd");
  }

  /* Probably the last time we can check corruption of what has been loaded: */
  GZLIB_TEST_MEASURE_do();
  adjust_ELF_relocation (relocate_offset);

  {
  unsigned tmp = prepare_exit (1);
  if (tmp)
      return tmp;
  }

 DOS_tiny_execute_extra_command();

#if 0
  /* switch OFF floppy drive if restarted in DOS_tiny_execute_extra_command() ? */
  if (peekb (0x0040003F) & 0x03) { /* motor disk 0 or 1 active */
      unsigned nbticks = _BIOS_nbtick() + 1;
      pokeb (0x00400040, 1); /* MOTOR TURN-OFF TIMEOUT COUNT */
      while (nbticks <= _BIOS_nbtick())
	  continue; /* wait at least a tick */
      }
#endif

  const char *startfilemsg;
  if (LOADER.cmdline_name[0] != '\0')
      startfilemsg = LOADER.cmdline_name;
    else
      startfilemsg = BOOTWAY.desc[LOADER.fileload[0].BOOTWAY_desc_index].filename;
  printf (STARTING_KERNEL, startfilemsg);	/* Faites chier la vache! */
  disable_interrupt();
  if (LOADER.kernel_64bits)
      _BIOS_we_go_to_long_mode (2);
  disable_nmi();
  reset_387();

  if (protected_mode() != 0) {
#if !(SETUP & BIOS_ONLY)
      if (UTIL.v86_callback != 0)
	  exit_broadcast();
#endif
      if (LOADER.kernel_64bits)
	  _BIOS_we_go_to_long_mode (1);
      enable_interrupt();
      return 2;
      }

  set_gdt (&gdt);
#ifdef LOAD_TASKREG
  set_task (0x08);
#endif

  while (nbdelay--)
      short_delay (UTIL.processor.calibrate_loopl); /* 1/100 s */

  /* As input also: GDT segment __KERNEL_DS = 0x18 (loaded in %ds since 2.6.5)
     is an unlimited R/W data segment starting at zero */
  asm volatile (
"	movl	%%ebx,%%ebp	# GCC features				\n"
"	movw	$0x18,%%ax						\n"
"	movw	%%ax,%%es						\n"
"									\n"
"	# If DOS=HIGH or HIMEM, the following crash their own memory:	\n"
"	mov	$patchdata+6,%%ebx					\n"
"	mov	%%cs:(%%ebx),%%ax					\n"
"	add	$2,%%ebx						\n"
"	test	%%ax,%%ax						\n"
"	jz	4f							\n"
"	3:								\n"
"	mov	%%cs:(%%ebx),%%ecx					\n"
"	add	$3,%%ecx						\n"
"	shr	$2,%%ecx						\n"
"	add	$4,%%ebx						\n"
"	mov	%%cs:(%%ebx),%%esi					\n"
"	add	$4,%%ebx						\n"
"	mov	%%cs:(%%ebx),%%edi					\n"
"	add	$4,%%ebx						\n"
"	cld								\n"
"	cmp	%%esi,%%edi						\n"
"	jl	1f							\n"
"	leal	-4(%%edi,%%ecx,4),%%edi					\n"
"	leal	-4(%%esi,%%ecx,4),%%esi					\n"
"	std								\n"
"	1:								\n"
"	rep movsl %%es:(%%esi),%%es:(%%edi)				\n"
"	xchg	%%eax,%%esi						\n"
"	mov	%%cs:-4(%%ebx),%%edi					\n"
"	add	%%cs:-12(%%ebx),%%edi					\n"
"	mov	%%cs:(%%ebx),%%ecx					\n"
"	add	$4,%%ebx						\n"
"	add	$3,%%ecx						\n"
"	shr	$2,%%ecx						\n"
"	xor	%%eax,%%eax						\n"
"	cld								\n"
"	rep stosl %%eax,%%es:(%%edi)					\n"
"	xchg	%%eax,%%esi						\n"
"	dec	%%ax							\n"
"	jnz	3b							\n"
"	4:								\n"
"	mov	%%cr0,%%eax						\n"
"	and	$0x00000001,%%eax					\n"
"	mov	%%eax,%%cr0						\n"
"	jmp 5f	\n 5:							\n"
"	xorl	%%eax,%%eax						\n"
"	mov	%%eax,%%cr3						\n"
"	mov	%%eax,%%cr4						\n"
"	pushl	%%eax							\n"
"	popfl				# so cld, cli			\n"
"	mov	$0x00000400,%%eax					\n"
"	mov	%%eax,%%dr7						\n"
"	mov	$0xFFFF0FF0,%%eax					\n"
"	mov	%%eax,%%dr6						\n"
"	xorl	%%eax,%%eax						\n"
//"	mov	%%eax,%%dr5						\n"
//"	mov	%%eax,%%dr4						\n"
"	mov	%%eax,%%dr3						\n"
"	mov	%%eax,%%dr2						\n"
"	mov	%%eax,%%dr1						\n"
"	mov	%%eax,%%dr0						\n"
"									\n"
"	movl	%%edx,%%esi	# moved to %%esi			\n"
"	xorl	%%ebx,%%ebx	# Flag to indicate an initial boot	\n"
"	cmpl	%%ebp,%%ebx						\n"
"	jne	1f							\n"
"	xorl	%%eax,%%eax						\n"
"	xorl	%%edi,%%edi						\n"
"	movw	%%cs,%%di						\n"
#ifdef CALLING_FROM_XCODESEG
"	sub	$loadcodeseg,%%di					\n"
#endif
"	shll	$4,%%edi						\n"
"	.align 4							\n"
"	start_protected:						\n"
#ifdef CALLING_FROM_XCODESEG
"	mov	$loadcodeseg,%%ecx					\n"
"	shll	$4,%%ecx						\n"
"	addl	$start_protected,%%ecx					\n"
#else
"	movl	$start_protected,%%ecx					\n"
#endif
"	shrl	$2,%%ecx						\n"
"	rep stosl %%eax,%%es:(%%edi)					\n"
"	addl	$(end_protected - start_protected),%%edi		\n"
"	movw	%%ds,%%cx	# ecx is null				\n"
"	shll	$4,%%ecx						\n"
"	addl	%0,%%ecx						\n"
"	subl	%%edi,%%ecx						\n"
"	shrl	$2,%%ecx						\n"
"	rep stosl %%eax,%%es:(%%edi)					\n"
"	movw	%%ss,%%cx	# ecx is null				\n"
"	movzwl	%%cx,%%edi						\n"
"	addw	$0x1000,%%cx						\n"
"	shll	$4,%%ecx						\n"
"	shll	$4,%%edi						\n"
"	addl	%1+8,%%edi						\n"
"	subl	%%edi,%%ecx						\n"
"	shrl	$2,%%ecx						\n"
"	rep stosl %%eax,%%es:(%%edi)					\n"
"	1:								\n"
"	movw	%%es,%%di						\n"
"	mov	%%di,%%ds						\n"
"	mov	%%ds:0,%%al						\n"
"	mov	%%di,%%ss	# BTW I do not smoke crack, I properly cook it.	\n"
"	mov	%%ss:0,%%al							\n"
"	mov	%%di,%%fs	# And I keep %%gs pointing to graphic mem.	\n"
"	mov	%%fs:0,%%al							\n"
"	xorl	%%edi,%%edi							\n"
".equ patchdata, . + 2								\n"
"	# Ready for the transdimensional jump.					\n"
"	ljmpl	$0x10, $0x100000 # self-modifying code: NOT IN ROM!		\n"
"	.hword	10 # i.e. MAXNBCPY						\n"
"	.int   0, 0, 0, 0	# reserve MAXNBCPY times 4 32bits words here.	\n"
"	.int   0, 0, 0, 0	# reserve MAXNBCPY times 4 32bits words here.	\n"
"	.int   0, 0, 0, 0	# reserve MAXNBCPY times 4 32bits words here.	\n"
"	.int   0, 0, 0, 0	# reserve MAXNBCPY times 4 32bits words here.	\n"
"	.int   0, 0, 0, 0	# reserve MAXNBCPY times 4 32bits words here.	\n"
"	.int   0, 0, 0, 0	# reserve MAXNBCPY times 4 32bits words here.	\n"
"	.int   0, 0, 0, 0	# reserve MAXNBCPY times 4 32bits words here.	\n"
"	.int   0, 0, 0, 0	# reserve MAXNBCPY times 4 32bits words here.	\n"
"	.int   0, 0, 0, 0	# reserve MAXNBCPY times 4 32bits words here.	\n"
"	.int   0, 0, 0, 0	# reserve MAXNBCPY times 4 32bits words here.	\n"
"	.align 4								\n"
"	end_protected:								\n"
	: /* no output -:) at least in the current world */
	: "i" (PMSeg), "i" (&PMSeg[nbof (PMSeg)]),
	  "b" (linadr_togpl), "d" (LOADER.comment.elem.paramadr) // moved to %%esi
	);
  return 0;	/* will not happen often */
  }}

LOAD_FCT_PREFIX(get_base_real_fct) static unsigned
get_base_real_fct (unsigned *nb_elf_reloc, struct elf_reloc_s *elf_reloc)
  {
  unsigned base_real_fct = LOADER.fileload[0].uncompressed_size;
  const unsigned max_elf_reloc = *nb_elf_reloc;
  *nb_elf_reloc = 0;

  if (LOADER.comment.elem.realfct_size != 0) {
      /* If given in the GZIP comment line, we do trust the value: */
      base_real_fct -= LOADER.comment.elem.realfct_size;
      ZDBG ((" Use GZIP comment line provided realfct_size: 0x%X\r\n", LOADER.comment.elem.realfct_size));
      }
    else if (LOADER.uncompressed_signature == *(unsigned*)"\x7F""ELF") {
      /* If not given in the GZIP comment line, look for the ELF Section Headers */
      union { struct ELF32_header elf32; struct ELF64_header elf64; } header;
      copy_from_first_file (0, stack_adr(&header), sizeof(header));
      if (   header.elf32.EI_CLASS == ELFCLASS32 && header.elf32.EI_DATA == ELFDATA_LITTLEENDIAN
	  && header.elf32.EI_VERSION == EV_CURRENT && header.elf32.e_type == ET_EXEC
	  && header.elf32.e_machine == EM_I386 && header.elf32.e_version == EV_CURRENT
	  && header.elf32.e_ehsize <= header.elf32.e_phoff) {
	  unsigned phnum;
	  union {
	      struct ELF32_pgm_header header;
	      unsigned char sized[header.elf32.e_phentsize];
	      } pgm[header.elf32.e_phnum];

	  ZDBG ((" Valid ELF32 (entry 0x%X, %u progheader of %u bytes at %u, %u sectheader of %u bytes at %u, string section %u)\r\n",
		header.elf32.e_entry, header.elf32.e_phnum, header.elf32.e_phentsize, header.elf32.e_phoff,
		header.elf32.e_shnum, header.elf32.e_shentsize, header.elf32.e_shoff, header.elf32.e_shstrndx));
	  /* LOADER.comment.elem.realfct_size = 0; / * obviously */
	  copy_from_first_file (header.elf32.e_phoff, stack_adr(&pgm), sizeof(pgm));
	  for (phnum = 0; phnum < header.elf32.e_phnum; phnum++) {
	      if (pgm[phnum].header.p_type != PT_LOAD)
		  continue;
	      if (pgm[phnum].header.p_vaddr == 0 && pgm[phnum].header.p_filesz != 0 && pgm[phnum].header.p_filesz < 640*1024) {
		  ZDBG (("Found the '.realmode' program header by its intrinsic properties, offset 0x%X, size 0x%X; ",
			pgm[phnum].header.p_offset, pgm[phnum].header.p_filesz));
		  base_real_fct = pgm[phnum].header.p_offset;
		  LOADER.comment.elem.realfct_size = pgm[phnum].header.p_filesz;
		  /* readelf -lS vmlinux: (Why PhysAddr == 0x00549000 ?)
			Program Headers:
			  Type           Offset   VirtAddr   PhysAddr   FileSiz MemSiz  Flg Align
			  LOAD           0x001000 0xc0100000 0x00100000 0x39965c 0x39965c R E 0x1000
			  LOAD           0x39b000 0xc049a000 0x0049a000 0x75085 0xaf000 RWE 0x1000
			  NOTE           0x000000 0x00000000 0x00000000 0x00000 0x00000 R   0x4
			  LOAD           0x411000 0x00000000 0x00549000 0x01830 0x01830 R E 0x1000
		   */
		  continue;
		  }
	      if (*nb_elf_reloc < max_elf_reloc) {
		  elf_reloc[*nb_elf_reloc].p_offset = pgm[phnum].header.p_offset;
		  elf_reloc[*nb_elf_reloc].p_paddr = pgm[phnum].header.p_paddr;
		  elf_reloc[*nb_elf_reloc].p_filesz = pgm[phnum].header.p_filesz;
		  elf_reloc[*nb_elf_reloc].p_memsz = pgm[phnum].header.p_memsz;

		  /* Only with buggy link - shall no more happen */
		  if (*nb_elf_reloc >= 1 && elf_reloc[*nb_elf_reloc-1].p_paddr + elf_reloc[*nb_elf_reloc-1].p_filesz > elf_reloc[*nb_elf_reloc].p_paddr) {
		      ZDBG (("WARNING: limiting ELF section %u memcpy from 0x%X bytes to 0x%X bytes (stripped ELF?).\r\n", *nb_elf_reloc-1,
				elf_reloc[*nb_elf_reloc-1].p_filesz, elf_reloc[*nb_elf_reloc].p_paddr - elf_reloc[*nb_elf_reloc-1].p_paddr));
		      elf_reloc[*nb_elf_reloc-1].p_filesz = elf_reloc[*nb_elf_reloc].p_paddr - elf_reloc[*nb_elf_reloc-1].p_paddr;
		      }
		  if (*nb_elf_reloc >= 1 && elf_reloc[*nb_elf_reloc-1].p_paddr + elf_reloc[*nb_elf_reloc-1].p_memsz > elf_reloc[*nb_elf_reloc].p_paddr) {
		      ZDBG (("WARNING: limiting ELF section %u memset from 0x%X bytes to 0x%X bytes (stripped ELF?).\r\n", *nb_elf_reloc-1,
				elf_reloc[*nb_elf_reloc-1].p_memsz, elf_reloc[*nb_elf_reloc].p_paddr - elf_reloc[*nb_elf_reloc-1].p_paddr));
		      elf_reloc[*nb_elf_reloc-1].p_memsz = elf_reloc[*nb_elf_reloc].p_paddr - elf_reloc[*nb_elf_reloc-1].p_paddr;
		      }
		  *nb_elf_reloc += 1;
		  }
	      ZDBG (("memcpy (0x%X, 0x%X, 0x%X); ", pgm[phnum].header.p_paddr, pgm[phnum].header.p_offset, pgm[phnum].header.p_filesz));
	      if (pgm[phnum].header.p_memsz > pgm[phnum].header.p_filesz)
		  ZDBG (("memset (0x%X, 0, 0x%X); ", pgm[phnum].header.p_paddr + pgm[phnum].header.p_filesz,
						     pgm[phnum].header.p_memsz - pgm[phnum].header.p_filesz));
	      if (pgm[phnum].header.p_paddr != pgm[phnum].header.p_vaddr)
		  ZDBG (("p_vaddr 0x%X, ", pgm[phnum].header.p_vaddr));
	      ZDBG (("align %u\r\n", pgm[phnum].header.p_align));
	      /* FIXME: it seems pgm.p_filesz may not be a multiple of 4 in vmlinux, unaligned memset will follow...
			Valid ELF (entry 0x100000, 3 progheader of 32 bytes at 52)
			memcpy (0x100000, 0x1000, 0x44A830); p_vaddr 0xC0100000, align 4096
			memcpy (0x49A000, 0x44C000, 0x75085); memset (0x50F085, 0, 0x39F7B); p_vaddr 0xC049A000, align 4096
			Valid ELF file, set runadr to entry point: 0x100000
	      */
	      }
	  LOADER.adjust_runadr = 0;
	  if (*nb_elf_reloc >= 1) {
	      if (LOADER.comment.elem.runadr == 0) {
		  ZDBG (("Valid ELF32 file with %u relocs, set runadr to entry point: 0x%X\r\n", *nb_elf_reloc, header.elf32.e_entry));
		  LOADER.comment.elem.runadr = header.elf32.e_entry;
		  LOADER.adjust_runadr = 1;
		  }
		else
		  ZDBG (("Valid ELF32 file, but keep runadr to: 0x%X\r\n", LOADER.comment.elem.runadr));
	      }
	  if (LOADER.comment.elem.realfct_size == 0)
	      return 0xFFFFFFFF;
	  }
	else if (header.elf64.EI_CLASS == ELFCLASS64 && header.elf64.EI_DATA == ELFDATA_LITTLEENDIAN
	  && header.elf64.EI_VERSION == EV_CURRENT && header.elf64.e_type == ET_EXEC
	  && header.elf64.e_machine == EM_X86_64 && header.elf64.e_version == EV_CURRENT
	  && header.elf64.e_ehsize <= header.elf64.e_phoff) {
	  unsigned phnum;
	  union {
	      struct ELF64_pgm_header header;
	      unsigned char sized[header.elf64.e_phentsize];
	      } pgm[header.elf64.e_phnum];

	  LOADER.kernel_64bits = 1; /* BUG? only set if realfct_size not given on the command line */
	  ZDBG ((" Valid ELF64 (entry 0x%llX, %u progheader of %u bytes at 0x%llX, %u sectheader of %u bytes at 0x%llX, string section %u)\r\n",
		header.elf64.e_entry, header.elf64.e_phnum, header.elf64.e_phentsize, header.elf64.e_phoff,
		header.elf64.e_shnum, header.elf64.e_shentsize, header.elf64.e_shoff, header.elf64.e_shstrndx));
	  /* LOADER.comment.elem.realfct_size = 0; / * obviously */
	  copy_from_first_file (header.elf64.e_phoff, stack_adr(&pgm), sizeof(pgm));
	  for (phnum = 0; phnum < header.elf64.e_phnum; phnum++) {
	      if (pgm[phnum].header.p_type != PT_LOAD)
		  continue;
	      if (pgm[phnum].header.p_vaddr == 0 && pgm[phnum].header.p_filesz != 0 && pgm[phnum].header.p_filesz < 640*1024) {
		  ZDBG (("Found the '.realmode' program header by its intrinsic properties, offset 0x%llX, size 0x%llX; ",
			pgm[phnum].header.p_offset, pgm[phnum].header.p_filesz));
		  base_real_fct = pgm[phnum].header.p_offset;
		  LOADER.comment.elem.realfct_size = pgm[phnum].header.p_filesz;
		  continue;
		  }
	      if (*nb_elf_reloc < max_elf_reloc) { /* convert all address to 32 bits, anyway we cannot do more in real mode */
		  elf_reloc[*nb_elf_reloc].p_offset = pgm[phnum].header.p_offset;
		  elf_reloc[*nb_elf_reloc].p_paddr = pgm[phnum].header.p_paddr;
		  elf_reloc[*nb_elf_reloc].p_filesz = pgm[phnum].header.p_filesz;
		  elf_reloc[*nb_elf_reloc].p_memsz = pgm[phnum].header.p_memsz;

		  if (*nb_elf_reloc >= 1 && elf_reloc[*nb_elf_reloc-1].p_paddr + elf_reloc[*nb_elf_reloc-1].p_filesz > elf_reloc[*nb_elf_reloc].p_paddr) {
		      ZDBG (("WARNING: limiting ELF section %u memcpy from 0x%llX bytes to 0x%llX bytes (stripped ELF?).\r\n", *nb_elf_reloc-1,
				elf_reloc[*nb_elf_reloc-1].p_filesz, elf_reloc[*nb_elf_reloc].p_paddr - elf_reloc[*nb_elf_reloc-1].p_paddr));
		      elf_reloc[*nb_elf_reloc-1].p_filesz = elf_reloc[*nb_elf_reloc].p_paddr - elf_reloc[*nb_elf_reloc-1].p_paddr;
		      }
		  if (*nb_elf_reloc >= 1 && elf_reloc[*nb_elf_reloc-1].p_paddr + elf_reloc[*nb_elf_reloc-1].p_memsz > elf_reloc[*nb_elf_reloc].p_paddr) {
		      ZDBG (("WARNING: limiting ELF section %u memset from 0x%llX bytes to 0x%llX bytes (stripped ELF?).\r\n", *nb_elf_reloc-1,
				elf_reloc[*nb_elf_reloc-1].p_memsz, elf_reloc[*nb_elf_reloc].p_paddr - elf_reloc[*nb_elf_reloc-1].p_paddr));
		      elf_reloc[*nb_elf_reloc-1].p_memsz = elf_reloc[*nb_elf_reloc].p_paddr - elf_reloc[*nb_elf_reloc-1].p_paddr;
		      }
		  *nb_elf_reloc += 1;
		  }
	      ZDBG (("memcpy (0x%llX, 0x%llX, 0x%llX); ", pgm[phnum].header.p_paddr, pgm[phnum].header.p_offset, pgm[phnum].header.p_filesz));
	      if (pgm[phnum].header.p_memsz > pgm[phnum].header.p_filesz)
		  ZDBG (("memset (0x%llX, 0, 0x%llX); ", pgm[phnum].header.p_paddr + pgm[phnum].header.p_filesz,
				pgm[phnum].header.p_memsz - pgm[phnum].header.p_filesz));
	      if (pgm[phnum].header.p_paddr != pgm[phnum].header.p_vaddr)
		  ZDBG (("p_vaddr 0x%llX, ", pgm[phnum].header.p_vaddr));
	      ZDBG (("align 0x%llX\r\n", pgm[phnum].header.p_align));
	      /* Why is p_vaddr starting with 0xFFFFFFFF ?? The last memset() start address is not aligned to 8 bytes.
_XMS_move_memory (offset 0x110000) to buff at 0x2F96E4D8: OK
 Valid ELF64 (entry 0x200100, 5 progheader of 56 bytes at 0x40, 14592 sectheader of 64 bytes at 0x9E5818, string section 14589)
_XMS_move_memory (offset 0x110040) to buff at 0x2F96E380: OK
memcpy (0x200000, 0x200000, 0x469A28); p_vaddr 0xFFFFFFFF80200000, align 0x200000
memcpy (0x66A000, 0x66A000, 0xD57A0); p_vaddr 0xFFFFFFFF8066A000, align 0x200000
memcpy (0x740000, 0x800000, 0xC08); p_vaddr 0xFFFFFFFFFF600000, align 0x200000
memcpy (0x742000, 0x942000, 0x3F004); memset (0x781004, 0, 0x6A3D4); p_vaddr 0xFFFFFFFF80742000, align 0x200000
Valid ELF file, set runadr to entry point: 0x200100 & 0xFFFFFFFF		*/
	      }
	  LOADER.adjust_runadr = 0;
	  if (*nb_elf_reloc >= 1) {
	      if (LOADER.comment.elem.runadr == 0) {
		  ZDBG (("Valid ELF64 file with %u relocs, set runadr to entry point: 0x%llX & 0xFFFFFFFF (to overwrite with 32bits entry point!)\r\n", *nb_elf_reloc, header.elf64.e_entry));
		  LOADER.comment.elem.runadr = header.elf64.e_entry;
		  LOADER.adjust_runadr = 1;
		  }
		else
		  ZDBG (("Valid ELF64 file, but keep runadr to: 0x%X\r\n", LOADER.comment.elem.runadr));
	      }
	  if (LOADER.comment.elem.realfct_size == 0)
	      return 0xFFFFFFFF;
	  }
	else
	  LOADER.uncompressed_signature = 0; /* Remember invalid ELF */
      }
    else {
      /* If not given in the GZIP comment line, look for end of file signature */
      struct {
	  unsigned null, size_realmode, addr_realmode, signature_BIOS;
	  } trailer;
      copy_from_first_file (base_real_fct - 16, stack_adr(&trailer), sizeof(trailer));
      ZDBG (("trailer: null 0x%X, size_realmode 0x%X, addr_realmode 0x%X, signature_BIOS 0x%X (==0x%X?)\r\n",
		trailer.null, trailer.size_realmode, trailer.addr_realmode, trailer.signature_BIOS, *(unsigned *)"BIOS"));
      if (trailer.signature_BIOS == *(unsigned *)"BIOS" && trailer.addr_realmode == 0) {
	  ZDBG (("Set size of real mode function to %u\r\n", trailer.size_realmode));
	  LOADER.comment.elem.realfct_size = trailer.size_realmode;
	  base_real_fct -= LOADER.comment.elem.realfct_size;
	  }
      }
  return base_real_fct;
  }

LOAD_FCT_PREFIX(linux_disk_letter_of_disk) static char
linux_disk_letter_of_disk (struct diskparam_str *dp)
  {
  char ret = 'a'; /* the 'a' of /dev/sda */
  unsigned cpt;

  /* That will work for most configurations with extended EBIOS... if USB controllers are taken is the same order */
  /* First find the EBIOS interface if we have booted from the IDE interface */
//printf ("Search disk letter of %s: ", dp->diskname);
  if (dp->access >= hardide_chs && dp->access <= hardide_lba48) {
      for (cpt = 0; cpt < DI.nbdisk; cpt++)
	  if (DI.param[cpt].access != dp->access && DI.param[cpt].ideIOadr == dp->ideIOadr && DI.param[cpt].lba_slave_mask == dp->lba_slave_mask) {
	      dp = &DI.param[cpt];
//printf ("(equivalent to %s) ", dp->diskname);
	      break;
	      }
      }
#define IS_USB_DISK(dp) ((dp)->ebios_Interface[0] == 'U' && (dp)->ebios_Interface[1] == 'S' && (dp)->ebios_Interface[2] == 'B')

  /* If we have booted from a USB disk BIOS 0x80, that is the first listed but not the first for Linux... */
//  if (dp->access == ebios_lba && IS_USB_DISK(dp)) {
  if (IS_USB_DISK(dp)) {
//printf ("(is USB)  ");
      for (cpt = 0; cpt < DI.nbdisk; cpt++)
	  if (DI.param[cpt].access == ebios_lba && DI.param[cpt].disknb >= 0x80 && !IS_USB_DISK(&DI.param[cpt]))
	      ret++;
      for (cpt = 0; (int)cpt < dp - DI.param; cpt++)
	  if (DI.param[cpt].access == ebios_lba && DI.param[cpt].disknb >= 0x80 && IS_USB_DISK(&DI.param[cpt]))
	      ret++;
      }
    else if (dp->access == ebios_lba) {
//printf ("(isn't USB)  ");
      for (cpt = 0; (int)cpt < dp - DI.param; cpt++)
	  if (DI.param[cpt].access == ebios_lba && DI.param[cpt].disknb >= 0x80 && !IS_USB_DISK(&DI.param[cpt]))
	      ret++;
      }
    else if (dp->access == bios_chs) {
//printf ("(don't know USB)  ");
      for (cpt = 0; (int)cpt < dp - DI.param; cpt++)
	  if ((DI.param[cpt].access == bios_chs || DI.param[cpt].access == ebios_lba) && DI.param[cpt].disknb >= 0x80)
	      ret++;
      }
    else {	/* The IDE interfaces should be in Linux order */
//printf ("(don't know BIOS/EBIOS)  ");
      for (cpt = 0; (int)cpt < dp - DI.param; cpt++)
	  if (DI.param[cpt].access != ebios_lba)
	      ret++;
      }
//printf ("assumes /dev/sd%c\r\n", ret); _BIOS_getkey();
  return ret;
  }

/* This function, if called with param==0xFFFFFFFFU, will disallocate the internal buffer;
   if called with param==0x80000000U|system_desc, will initialise the internal buffer
   to the concatenated string copy_gujin_param.extra_cmdline + commandfile_get(cmdf_getcmdline);
   if called with param==0xFFFFFFFEU will return what is left in the internal buffer;
   if called with an address of a string which is the start of a parameter, will extract this
   parameter at the end of the internal buffer and return the address of this parameter.
   It will also remove all occurence of that parameter from the buffer. */
LOAD_FCT_PREFIX(cmdline_param) static const char *
cmdline_param (const char *param)
  {
  static char *allcommandline = 0;
#define MAX_CMDF_COMMANDLINE_LEN 512
#ifdef FS_USE_MALLOC
  if ((unsigned)param == 0xFFFFFFFFU) {
      if (allcommandline != 0)
	  FREE(allcommandline);
//printf ("freed command line\r\n"); _BIOS_getkey();
      allcommandline = 0;
      return 0;
      }
  if ((unsigned)param == 0xFFFFFFFEU) {
//printf ("return what is left of command line: \"%s\"\r\n", allcommandline); _BIOS_getkey();
      return allcommandline;
      }
  if ((unsigned)param & 0x80000000U) {
      const struct desc_str *system_desc = (const struct desc_str *)((unsigned)param & ~0x80000000U);
      if (allcommandline != 0)
	  FREE(allcommandline);
      allcommandline = MALLOC (MAX_CMDF_COMMANDLINE_LEN, "CURCMDL");
//printf ("malloced command line at 0x%X\r\n", allcommandline); _BIOS_getkey();
      if (allcommandline == 0) {
	  DBG (("Failure malloc allcommandline!\r\n"));
	  return 0;
	  }
#else /* FS_USE_MALLOC */
  static char _allcommandline[MAX_CMDF_COMMANDLINE_LEN];
  if ((unsigned)param == 0xFFFFFFFFU) {
      allcommandline = 0;
      return 0;
      }
  if ((unsigned)param == 0xFFFFFFFEU)
      return allcommandline;
  if ((unsigned)param & 0x80000000U) {
      const struct desc_str *system_desc = (const struct desc_str *)((unsigned)param & ~0x80000000U);
      allcommandline = _allcommandline;
#endif /* FS_USE_MALLOC */
      unsigned cmdline_len;
      for (cmdline_len = 0; cmdline_len < sizeof(copy_gujin_param.extra_cmdline); cmdline_len++) {
	  allcommandline[cmdline_len] = copy_gujin_param.extra_cmdline[cmdline_len];
	  if (copy_gujin_param.extra_cmdline[cmdline_len] == '\0')
	      break;
	  }
//printf ("command line inited to internal \"%s\"\r\n", allcommandline); _BIOS_getkey();
#ifdef COMMANDFILE
      unsigned ret = commandfile_get (system_desc->iso_fsname ?: (char *)DI.param[system_desc->disk].partition[system_desc->partition].name,
		system_desc->filename, BOOTWAY.desc[LOADER.fileload[1].BOOTWAY_desc_index].filename,
		cmdf_getcmdline, system_desc->variant_number,
		&allcommandline[cmdline_len], MAX_CMDF_COMMANDLINE_LEN-cmdline_len);
//printf ("command line extended by \"%s\"\r\n", &allcommandline[cmdline_len]); _BIOS_getkey();
      if (ret & 0x80000000U) {/* Do not modify cmdlline from gujin.cmd */
//printf ("return unmodified command line \"%s\"\r\n", &allcommandline[cmdline_len]); _BIOS_getkey();
	  return &allcommandline[cmdline_len];
	  }
//printf ("total command line \"%s\"\r\n", allcommandline); _BIOS_getkey();
#endif
      /* Add a space to mach "FROMISO=iso-scan " at end of line: */
      char *end = allcommandline;
      while (*end != '\0')
	  end++;
      if (end[-1] != ' ')
	  *end++ = ' ';
      *end = '\0';
      return 0;
      }

//printf ("searching in command line \"%s\"\r\n", param); _BIOS_getkey();
  char *start, *returned = 0;
  while ((start = (char *)strstr (allcommandline, param)) != 0) {
      if (start != allcommandline && start[-1] != ' ' && start[-1] != '\t')
	  continue; /* localroot= vs root= */
      char *end = start;
      while (*end != '\0' && *end != ' ' && *end != '\t')
	end++;
      if (returned == 0) {
	  returned = &allcommandline[MAX_CMDF_COMMANDLINE_LEN - 1];
	  *returned = '\0';
	  char *ptr = end;
	  while (--ptr >= start)
	      *--returned = *ptr;
//printf ("saved past command line \"%s\"\r\n", returned); _BIOS_getkey();
	  }
      while (*end != '\0')
	*start++ = *end++;
      *start = '\0';
//printf ("left command line \"%s\", past command line still \"%s\"\r\n", allcommandline, returned); _BIOS_getkey();
      }
  return returned;
  }

  /* "lang=" and "keyboard=" are used because the are defined in Knoppix, see:
  http://www.knoppix.net/wiki/Language,_Locale_&_Keyboard_settings_FAQ */
  /* you will have the lang=en|fr|... in your Linux environment so you may want to do for Debian:
	 if [ "${lang}" ]; then export LANG=`locale -a | grep ${lang} | head -1` ; fi
       just before comment "# load new map" in /etc/rcS.d/S01glibc.sh
       or in Fedora just before "LOADKEYS=loadkeys" in /etc/rc.d/rc.sysinit
       For other distribution, do a "grep LANG /etc/rcS.d/?*"
    */
LOAD_FCT_PREFIX(build_final_command_line) static void
build_final_command_line (unsigned char *cmdline, char *buffer, unsigned char *minor_root, struct root_flags_str *root_flags, const struct desc_str *system_desc)
  {
  const char *ptr;

  ptr = cmdline_param ((const char *)((unsigned)system_desc | 0x80000000U));
  if (ptr) { /* do not modify the one in gujin.cmd */
      strcpy ((char *)cmdline, ptr);
      cmdline_param ((const char *)0xFFFFFFFFU);
      return;
      }
#if DISK_SUPPORT != (EBIOS_SUPPORT|CDBIOS_SUPPORT|ISOFS_PROBE|PROBE_ONLY_BIOSBOOTDEVICE) /* not minigujin.bcd */
#if DISK_SUPPORT != (BIOS_SUPPORT|EBIOS_SUPPORT|FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|PROBE_ONLY_BIOSBOOTDEVICE) /* not tinyusb */
#if DISK_SUPPORT != (BIOS_SUPPORT|EBIOS_SUPPORT|E2FS_PROBE|PROBE_ONLY_BIOSBOOTDEVICE) /* not tinyext4 */
#if DISK_SUPPORT != (BIOS_SUPPORT|FAT12_PROBE|FAT16_PROBE|PROBE_ONLY_BIOSBOOTDEVICE|NO_PROBE_PARTITIONS) /* not tinystd */
  if (BOOT1_COM_port() < 0) {
#if 0
      ptr = cmdline_param ("video=");
      if (ptr) {
	  while (*ptr && *ptr != ' ' && *ptr != '\t')
	      *cmdline++ = *ptr++;
	  *cmdline++ = ' ';
	  }
#if USER_SUPPORT & (VGA_SUPPORT|VESA_SUPPORT)
	else if (VESA_ACTIVE())
	  cmdline = STRINIT (cmdline, "video=vesa "); /* Documentation/fb/modedb.txt ? */
#endif

      ptr = cmdline_param ("NumLock=");
      if (ptr) {
	  while (*ptr && *ptr != ' ' && *ptr != '\t')
	      *cmdline++ = *ptr++;
	  *cmdline++ = ' ';
	  }
	else {
	  cmdline = STRINIT (cmdline, "NumLock=");
	  if (_BIOS_get_shift_flags().num_lock_active)
	      cmdline = STRINIT (cmdline, "on ");
	    else
	      cmdline = STRINIT (cmdline, "off ");
	  }
#endif

      ptr = cmdline_param ("keyboard=");
      if (ptr) {
	  while (*ptr && *ptr != ' ' && *ptr != '\t')
	      *cmdline++ = *ptr++;
	  *cmdline++ = ' ';
	  }
	else {
	  const char *kbd_name = kbdname (copy_gujin_param.kbdmap);
	  if (kbd_name != 0) {
	      cmdline = STRINIT (cmdline, "keyboard=");
	      strcpy ((char *)cmdline, kbd_name);
	      cmdline += strlen ((char *)cmdline);
	      *cmdline++ = ' ';
	      }
	  }
      }
    else {
      ptr = cmdline_param ("console=");
      if (ptr) {
	  while (*ptr && *ptr != ' ' && *ptr != '\t')
	      *cmdline++ = *ptr++;
	  *cmdline++ = ' ';
	  }
	else {
	  /* prohibited cross reference from .loadcode to `password_serial' in .text, and
		STATE.serial_port does not contains serial configuration... */
	  union { unsigned short all; serialconf_t conf; } currconf = {peekw (code_adr((void *)0x1B6))};
	  /* there can be multiple "console=" given: */
	  static const char *const speed_array[] = {
		  "110", "150", "300", "600", "1200", "2400", "4800", "9600"
		  }, *nbptr;

	  cmdline = STRINIT (cmdline, "console=ttyS");
	  *cmdline++ = '0' + BOOT1_COM_port();
	  *cmdline++ = ',';
	  nbptr = speed_array[currconf.conf.speed];
	  while (*nbptr)
	      *cmdline++ = *nbptr++;
	  *cmdline++ = currconf.conf.parity ? (currconf.conf.even? 'e' : 'o') : 'n';
	  *cmdline++ = currconf.conf.eightbit ? '8' : '7';
	  *cmdline++ = ' ';
	  }

      ptr = cmdline_param ("COLS=");
      if (ptr) {
	  while (*ptr && *ptr != ' ' && *ptr != '\t')
	      *cmdline++ = *ptr++;
	  *cmdline++ = ' ';
	  }
#if USER_SUPPORT != 0
	else {
	  cmdline = STRINIT (cmdline, "COLS=");
	  *cmdline = (char)(UI.parameter.nbcol / 100);
	  if (*cmdline != '\0')
	      *cmdline++ += '0';
	  *cmdline++ = '0' + (UI.parameter.nbcol % 100) / 10;
	  *cmdline++ = '0' + (UI.parameter.nbcol % 10);
	  *cmdline++ = ' ';
	  }
#endif

      ptr = cmdline_param ("LINES=");
      if (ptr) {
	  while (*ptr && *ptr != ' ' && *ptr != '\t')
	      *cmdline++ = *ptr++;
	  *cmdline++ = ' ';
	  }
#if USER_SUPPORT != 0
	else {
	  cmdline = STRINIT (cmdline, "LINES=");
	  *cmdline = (char)(UI.parameter.nbrow / 100);
	  if (*cmdline != '\0')
	      *cmdline++ += '0';
	  *cmdline++ = '0' + (UI.parameter.nbrow % 100) / 10;
	  *cmdline++ = '0' + (UI.parameter.nbrow % 10);
	  *cmdline++ = ' ';
	  }
#endif
      }

  ptr = cmdline_param ("mouse=");
  if (ptr) {
      while (*ptr && *ptr != ' ' && *ptr != '\t')
	  *cmdline++ = *ptr++;
      *cmdline++ = ' ';
      }
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT | SERIAL_MOUSE_SUPPORT)
    else {
      if (MOUSE.type != MOUSE_NONE && MOUSE.type != MOUSE_PS2) {	/* MOUSE_PS2: can be USB because it is BIOS */
	  cmdline = STRINIT (cmdline, "mouse=/dev/ttyS");
	  *cmdline++ = '0' + (MOUSE.type & ~MOUSE_SERIAL_MASK);
	  *cmdline++ = ',';
	  if ((MOUSE.type & MOUSE_SERIAL_MASK) == MOUSE_SERIAL_COM1)
	      *cmdline++ = '2';
	    else if ((MOUSE.type & MOUSE_SERIAL_MASK) == MOUSE_SERIAL3_COM1)
	      *cmdline++ = '3';
	    else
	      *cmdline++ = 'M';
	  *cmdline++ = ' ';
	  }
      }
#endif /* MOUSE_SUPPORT */

  ptr = cmdline_param ("TZ=");
  if (ptr) {
      while (*ptr && *ptr != ' ' && *ptr != '\t')
	  *cmdline++ = *ptr++;
      *cmdline++ = ' ';
      }
#if DISK_SUPPORT != DOS_SUPPORT	/* nothing for mingujin.exe */
    else {
      cmdline = STRINIT (cmdline, "TZ=UTC");
      if (UTIL.time_hour_offset < 0) {
	  *cmdline++ = '-';
	  *cmdline++ = '0' + -UTIL.time_hour_offset;
	  }
	else
	  *cmdline++ = '0' + UTIL.time_hour_offset;
      *cmdline++ = ' ';
      }

  ptr = cmdline_param ("lang=");
  if (ptr) {
      while (*ptr && *ptr != ' ' && *ptr != '\t')
	  *cmdline++ = *ptr++;
      *cmdline++ = ' ';
      }
    else {
#ifdef LANG_CMDLINE
      const char *src = LANG_CMDLINE;
      while (*src)
	  *cmdline++ = *src++;
      *cmdline++ = ' ';
#endif
      }

#endif	/* nothing for mingujin.exe */
#endif /* not tinystd */
#endif /* not tinyext4 */
#endif /* not tinyusb */
#endif /* not minigujin.bcd */

  ptr = cmdline_param ("BOOT_IMAGE=");
  if (ptr) {
      while (*ptr && *ptr != ' ' && *ptr != '\t')
	  *cmdline++ = *ptr++;
      *cmdline++ = ' ';
      }
    else if (system_desc->filename) {
      /* Add BOOT_IMAGE= if it was not present */
      cmdline = STRINIT (cmdline, "BOOT_IMAGE=");
      const char *nptr = system_desc->filename;
      while (*nptr)
	  if (*nptr++ == ':') /* remove "image.iso:" or DOS "C:" */
	      break;
      if (*nptr == '\0')
	  nptr = system_desc->filename;
      while (*nptr)
	  if (*nptr == '\\')
	      *cmdline++ = '/', nptr++; /* mingujin.exe replace '\\' by '/' */
	    else
	      *cmdline++ = *nptr++;
      *cmdline++ = ' ';
      }

  /* Those flags should only set a variable, but are often used in init scripts: */
  if (cmdline_param ("ro ")) {
      if (root_flags)
	  root_flags->read_only = 1;
      *cmdline++ = 'r';
      *cmdline++ = 'o';
      *cmdline++ = ' ';
      }
  if (cmdline_param ("rw ")) {
      if (root_flags)
	  root_flags->read_only = 0;
      *cmdline++ = 'r';
      *cmdline++ = 'w';
      *cmdline++ = ' ';
      }

  /* Here we treat switches which are for Gujin only and are not copied to Linux cmdline
     they contain a space so shall not be followed by a tab: */
  unsigned sda_hda_switch = 0;
  if (cmdline_param ("root=/dev/hd* "))
      sda_hda_switch = 1;
  if (cmdline_param ("root=/dev/sd* "))
      sda_hda_switch = 2;
  if (cmdline_param ("root=no-label "))
      sda_hda_switch = 3;
  if (cmdline_param ("root=no-auto "))
      sda_hda_switch = 4;

  unsigned fromiso_type = 0;
  if (cmdline_param ("FROMISO=iso-scan "))
      fromiso_type = 1; /* iso-scan/filename=... */
    else if (cmdline_param ("FROMISO=from "))
      fromiso_type = 2; /* from= */
    else if (cmdline_param ("FROMISO=fromiso "))
      fromiso_type = 3; /* fromiso= */
    else if (cmdline_param ("FROMISO=root "))
      fromiso_type = 4; /* root=live:LABEL=...:/boot/ or root=live:/dev/sda1:/boot/ */
    else if (cmdline_param ("FROMISO=isofrom "))
      fromiso_type = 5; /* isofrom= */
//    else if (cmdline_param ("FROMISO=find_iso "))
//      fromiso_type = 6; /* find_iso/filename=... */
//    else if (cmdline_param ("FROMISO=findiso "))
//      fromiso_type = 7; /* findiso=... */

  struct diskparam_str *dp = &DI.param[system_desc->disk];
  struct partition_str *dpp = &dp->partition[system_desc->partition];

  /* are we trying to run a kernel in iso file? */
  /* Check that we have a filename with semicolon: (But not the DOS drive letter semicolon for mingujin.exe!) */
  unsigned char has_semicolon = (strchr (&system_desc->filename[2], ':') != 0);
  /* or check that Gujin has been loaded from a simulated disk (not 100% waterproof) */
  farptr int13addr = peekl(4 * 0x13);
  unsigned char simdisk = (dp->access == bios_chs || dp->access == ebios_lba)
	&& peekll(int13addr + 3) == *(unsigned long long *)"$INT13SF"
	&& peekb(int13addr + 27) == dp->disknb
	&& peekb(int13addr + 28) > ' ' && peekb(int13addr + 28) <= '~'
	&& peekll(int13addr + 28) != *(unsigned long long *)"unknown";

  if (   (ptr = cmdline_param ("iso-scan/filename=")) != 0
      || (ptr = cmdline_param ("find_iso/filename=")) != 0
      || (ptr = cmdline_param ("root=live:LABEL=")) != 0
      || (ptr = cmdline_param ("fromiso=")) != 0
      || (ptr = cmdline_param ("from=")) != 0) {
      while (*ptr && *ptr != ' ' && *ptr != '\t')
	  *cmdline++ = *ptr++;
      *cmdline++ = ' ';
      }
    else if (has_semicolon || simdisk) {
      if (fromiso_type == 0) { /* Use fslabel to try to guess */
	  if (!strncmp (system_desc->iso_fsname, "NimbleX", strlen("NimbleX"))) // NimbleX
	      fromiso_type = 2; /* from= */
	    else if (!strncmp (system_desc->iso_fsname, "sidux", strlen("sidux")) // sidux-2009-0*.iso
		|| !strncmp (system_desc->iso_fsname, "Elive", strlen("Elive"))) // elive_2.0_Topaz_new-kernel_up001.iso, "Elive 2010-03-03"
	      fromiso_type = 3; /* fromiso= */
	    else if (!strncmp (system_desc->iso_fsname, "Fedora", strlen("Fedora"))
		|| !strncmp (system_desc->iso_fsname, "F1", strlen("F1")) // F13-Alpha-i686-Live
		|| !strncmp (system_desc->iso_fsname, "rhel", strlen("rhel"))
		|| !strncmp (system_desc->iso_fsname, "CentOS", strlen("CentOS")))
	      fromiso_type = 4; /* root=live:LABEL=...:/boot/ or root=live:/dev/sda1:/boot/ */
	    else if (!strncmp (system_desc->iso_fsname, "XBMC", strlen("XBMC"))) // XBMC_Live
	      fromiso_type = 5; /* isofrom= */
//	    else if (0)  // bt4-pre-final.iso
//	      fromiso_type = 6; /* find_iso/filename=... */
//	    else if (!strncmp (system_desc->iso_fsname, "grml", strlen("grml"))) // grml-small 2009.10
//	      fromiso_type = 7; /* findiso=... */
	    else /* if (!strncmp (system_desc->iso_fsname, "Debian", strlen("Debian")) // Debian lenny 20100306-10:48 in debian-live-504-i386-rescue.iso
		|| !strncmp (system_desc->iso_fsname, "Ubuntu", strlen("Ubuntu")) // Ubuntu 9.10 i386
		|| !strncmp (system_desc->iso_fsname, "BT", strlen("BT"))) // BT4 (BackTrack Live CD)) in bt4-final.iso */
	      fromiso_type = 1; /* iso-scan/filename=... */
	  }
      /* fromiso_type no more null: */
      static const char *const fromisoarray[] = {
	  "iso-scan/filename=",
	  "from=",
	  "fromiso=",
	  "root=live:", /* followed by LABEL=xxx: or /dev/sdb1: */
	  "isofrom=", /* followed by /dev/sdb1 */
//	  "find_iso/filename=",
//	  "findiso="
	  };
      const char *ptr = fromisoarray[fromiso_type-1];
      while (*ptr)
	  *cmdline++ = *ptr++;
      if (fromiso_type == 4 || fromiso_type == 5) {
	  unsigned char *dppname = dpp->name;
	  if (*dppname && *(unsigned *)dppname != *(unsigned *)"part" && fromiso_type != 5) {
	      cmdline = STRINIT (cmdline, "LABEL=");
	      while (*dppname > ' ') // (*dppname != '\0' && *dppname != ' ')
		  *cmdline++ = *dppname++;
	      }
	    else {
	      cmdline = STRINIT (cmdline, "/dev/sda0");
	      if (sda_hda_switch == 1) /* root=/dev/sd?? by default */
		  cmdline[-4] = 'h';
	      cmdline[-2] = linux_disk_letter_of_disk (dp);
	      if (dp->nbpartition != 1)
		  cmdline[-1] += dpp->OSnumber;
		else
		  cmdline--; /* handle superfloppy: "/dev/sdb" */
	      }
	  if (fromiso_type != 5)
	      *cmdline++ = ':';
	  }
      *cmdline++ = '/';
      if (has_semicolon) {
	  ptr = system_desc->filename;
	  while (*ptr != ':') /* strchr (system_desc->filename, ':') != 0 */
	      *cmdline++ = *ptr++;
	  }
	else {
	  extern char int13filename[64];
	  int13addr += 28;
	  unsigned cpt;
	  for (cpt = 0; cpt < sizeof(int13filename); cpt++, cmdline++)
	      if ((*cmdline = peekb(int13addr++)) < ' ')
		  break;
	  }
      *cmdline++ = ' ';
      }

  ptr = cmdline_param ("root=");
  if (ptr) {
      while (*ptr && *ptr != ' ' && *ptr != '\t')
	  *cmdline++ = *ptr++;
      *cmdline++ = ' ';
      }
    else if (sda_hda_switch != 4 && !(has_semicolon || simdisk)) { //&& fromiso_type != 4) {
      if (minor_root)
	  *minor_root = dpp->OSnumber;

      unsigned label_contains_space = 0;
      unsigned char *ptr = dpp->name;
      while (*ptr != '\0')
	  if (*ptr++ <= ' ')
	      label_contains_space++;
      ptr = dpp->name;

      /* param->in_out.root.fields.major_root = ; difficult to initialise, 0x03 for /dev/hda, 0xCA for /dev/sda */
      if (*ptr != '\0' && label_contains_space == 0 && sda_hda_switch != 3 && !dpp->misc.conflicting_fsname) {
	  /* FIXME: Shall we use " root=CDLABEL=" in some case? */
	  cmdline = STRINIT (cmdline, "root=LABEL=");
	  while (*ptr)
	      *cmdline++ = *ptr++;
	  }
	else {
	  cmdline = STRINIT (cmdline, "root=/dev/sda0");
	  if (sda_hda_switch == 1) /* root=/dev/sd?? by default */
	      cmdline[-4] = 'h';
	  cmdline[-2] = linux_disk_letter_of_disk (dp);
	  if (dp->nbpartition != 1)
	      cmdline[-1] += dpp->OSnumber;
	    else
	      cmdline--; /* handle superfloppy: "/dev/sdb" */
	  }
      *cmdline++ = ' ';
      }

  /* If signature "cmdline:" is found in the bzImage bootsector, it is added to
     the command line, after the proposed cmdline, before Gujin-setup cmdline. */
  if (buffer) {
      // FIXME: here param == fourKbuffer = address 0
      // in 64 bits mode, we try to put 0xFFFFFFFFFFFFFFFF in edx...
      //char *sigsearch = (char *)param - 1;
      char *sigsearch = buffer;
      while (++sigsearch < buffer + 0x1F0 - sizeof ("cmdline:"))
	  if (*(unsigned long long *)sigsearch == *(unsigned long long *)"cmdline:")
	      break;
      if (sigsearch < buffer + 0x1F0 - sizeof ("cmdline:")) {
	  char *toerase = sigsearch, *end;

	  sigsearch += sizeof ("cmdline:") - 1;
	  end = sigsearch - 1;
	  while (++end < (char *)buffer + 0x1F0)
	      if (*end == '\0')
		  break;
	  if (end < (char *)buffer + 0x1F0) {
	      /* There is already a space before stringptr added by build_proposed_cmdline() */
	      while (sigsearch < end)
		  *cmdline++ = *sigsearch++;
	      *cmdline++ = ' ';
	      while (toerase < end)
		  *toerase++ = '\0'; /* leaves no trace of it */
	      }
	  }
      }

  /* Copy unrecognised parameter still in buffer and free buffer: */
  const char *unrecognised_part = cmdline_param ((const char *)0xFFFFFFFEU);
  *cmdline = '\0';
  strcpy ((char *)cmdline, unrecognised_part);
  cmdline_param ((const char *)0xFFFFFFFFU);
  }

LOAD_FCT_PREFIX(bzImage_protocol_run) static inline unsigned
bzImage_protocol_run (struct desc_str *system_desc, unsigned totalmem)
  {
//#define KERNEL_SETUP_AT_90000
#ifndef KERNEL_SETUP_AT_90000
  farptr kernel_setup = (getss() + 0x1000) << 16; /* value hardcoded in next asm() block */
#else
  farptr kernel_setup = 0x9000 << 16; /* value hardcoded in next asm() block */
#endif
  unsigned setup_len = 512 * ((LOADER.setup_sects ?: 4) + 1);
  struct linux_param *param = (struct linux_param *)fourKbuffer;

  ZDBG (("In %s, kernel_setup = 0x%X\r\n", __FUNCTION__, kernel_setup));

  ZDBG (("%u files loaded, first one at 0x%X size compressed %u uncompressed %u, filesize %u\r\n",
	LOADER.curfileload - LOADER.fileload, LOADER.fileload[0].load_address, LOADER.fileload[0].compressed_size,
	LOADER.fileload[0].uncompressed_size, system_desc->filesize));
  /* note that LOADER.initrd_index is reseted to zero at this point, and compressed size is rounded to FS block size */
  if (LOADER.curfileload - LOADER.fileload > 1)
      ZDBG (("initrd at 0x%X size compressed %u uncompressed %u, filesize %u\r\n",
	LOADER.fileload[1].load_address, LOADER.fileload[1].compressed_size, LOADER.fileload[1].uncompressed_size,
	BOOTWAY.desc[LOADER.fileload[1].BOOTWAY_desc_index].filesize));
  /* Kernel has passed the tests in vmlinuz_header_treat() so we already know it has HdrS signature,
     it is loaded high (bzImage) and these fields has been set: */
  ZDBG (("LOADER.minor_major_root = 0x%X, LOADER.ramdisk_max 0x%X, LOADER.gzip_byte_treated %u, LOADER.setup_sects %u\r\n",
	LOADER.minor_major_root, LOADER.ramdisk_max, LOADER.gzip_byte_treated, LOADER.setup_sects));

#ifndef KERNEL_SETUP_AT_90000
  fmemset (kernel_setup + setup_len, 0, 0xFFFF - setup_len);
#else
  fmemset (kernel_setup + setup_len, 0, 1024 * UTIL.basemem - 0x90000 - setup_len);
#endif
  copy_from_first_file (0, kernel_setup, setup_len);
  lmemcpy (param, kernel_setup, sizeof(fourKbuffer));

  /*   vga=<mode>
	<mode> here is either an integer (in C notation, either
	decimal, octal, or hexadecimal) or one of the strings
	"normal" (meaning 0xFFFF), "ext" (meaning 0xFFFE) or "ask"
	(meaning 0xFFFD).  This value should be entered into the
	vid_mode field, as it is used by the kernel before the command
	line is parsed. What is that 0x200 ? should be 0x4000 for linear...
     Strange behaviour with Fedora7 - cannot contain mode 3 on at least this VIA/S3/AMD sempron board */

  param->in_out.vid_mode = copy_gujin_param.default_graphic_video_mode & 0xFFF;
  if (param->in_out.vid_mode == 0xFFF)
      param->in_out.vid_mode = copy_gujin_param.default_video_mode & 0xFFF; /* tinycdrom.img with --default_video_mode=0x123 --force_textmode=0 */

  if (param->in_out.vid_mode == 0 || param->in_out.vid_mode == 0xFFF || copy_gujin_param.attrib.force_textmode)
      param->in_out.vid_mode = -1; /* Normal VGA, also for tiny*.img */
    else
      param->in_out.vid_mode += 0x200;

  param->in_out.type_of_loader.type = GUJIN;
  param->in_out.type_of_loader.version = 1;

  build_final_command_line (param->command_line, (char *)param, &param->in_out.root.fields.minor_root, &param->in_out.root_flags, system_desc);
//printf ("Final cmdline: '%s'\r\n", param->command_line); _BIOS_wait (3 * 1000 * 1000);

#if 0
  if (param->in_out.header_version_number >= 0x0201) {
      param->in_out.loadflags.CAN_USE_HEAP = 1;
      /* Keep 0x200 bytes for Linux real-mode stack: */
#ifndef KERNEL_SETUP_AT_90000
      param->in_out.heap_end_ptr = 0x10000 - 0x200;
#else
      param->in_out.heap_end_ptr = 0x9000 - 0x200;
#endif
      }
    else {
      /* param->in_out.setup_move_size = 0x9100; / * check value: "something + command line size", only protocols 2.01 and before */
      param->in_out.setup_move_size = 1024 * UTIL.basemem - 0x90000;
      if (param->in_out.setup_move_size == 0 /* i.e. 0x10000 */ || param->in_out.setup_move_size > 0xA000 /* avoid using memory above 0x9a000 */)
	  param->in_out.setup_move_size = 0xA000;
      }
#else /* I should understand what are those two constants... */
  unsigned short heap_end;
  if (param->in_out.header_version_number >= 0x0202 && param->in_out.loadflags.LOADED_HIGH)
      heap_end = 0xe000;
    else
      heap_end = 0x9800;
  if (param->in_out.header_version_number >= 0x0201) {
      param->in_out.heap_end_ptr = heap_end - 0x200;
      param->in_out.loadflags.CAN_USE_HEAP = 1;
      }
#endif

  param->screen_info.CL_MAGIC  = 0xA33F;
  param->screen_info.CL_OFFSET = offsetof (struct linux_param, command_line);
  if (param->in_out.header_version_number >= 0x0206) {
      param->screen_info.CL_MAGIC = param->screen_info.CL_OFFSET = 0;
      /* Linux command line is and stay in fourKbuffer, not in kernel_setup: */
      param->in_out.cmd_line_ptr = farptr2linear(data_adr(param->command_line));
      }
    else if (param->in_out.header_version_number >= 0x0202)
      param->in_out.cmd_line_ptr = farptr2linear (kernel_setup + param->screen_info.CL_OFFSET);

  if (LOADER.curfileload - LOADER.fileload > 1) {
      param->in_out.ramdisk_size = LOADER.fileload[1].uncompressed_size;
      if (LOADER.ramdisk_max > 1024*totalmem)
	  param->in_out.ramdisk_image = (1024*totalmem - param->in_out.ramdisk_size) & ~4095; /* start is 4 Kb aligned */
	else
	  param->in_out.ramdisk_image = (LOADER.ramdisk_max + 1 - param->in_out.ramdisk_size) & ~4095; /* start is 4 Kb aligned */
      }
     else
      param->in_out.ramdisk_image = param->in_out.ramdisk_size = 0;

  ZDBG (("Initial in_out.header_version_number 0x%X, in_out.vid_mode = 0x%X, screen_info.orig_video_mode = 0x%X, \r\n",
	param->in_out.header_version_number, param->in_out.vid_mode, param->screen_info.orig_video_mode));
  ZDBG (("in_out.type_of_loader.{type,version} = 0x%X,%u, ext_loader_type 0x%X, ext_loader_ver 0x%X, \r\n",
	param->in_out.type_of_loader.type, param->in_out.type_of_loader.version, param->in_out.ext_loader_type, param->in_out.ext_loader_ver));
  if (param->in_out.relocatable_kernel)
	ZDBG (("relocatable_kernel %u, kernel_alignment 0x%X\r\n", param->in_out.relocatable_kernel, param->in_out.kernel_alignment));
  ZDBG (("command line: \""));
  ZDBGVAR (((char *)param->command_line));
  ZDBG (("\" in_out.heap_end_ptr = 0x%X, screen_info.CL_MAGIC 0x%X, screen_info.CL_OFFSET 0x%X, in_out.cmd_line_ptr 0x%X\r\n",
	param->in_out.heap_end_ptr, param->screen_info.CL_MAGIC, param->screen_info.CL_OFFSET, param->in_out.cmd_line_ptr));
  ZDBG (("in_out.ext_loader_ver/ext_loader_type = 0x%X/0x%X, LOADER.ramdisk_max = 0x%X, totalmem = %u Kb, in_out.ramdisk_size 0x%X, in_out.ramdisk_image 0x%X\r\n",
	param->in_out.ext_loader_ver, param->in_out.ext_loader_type, LOADER.ramdisk_max, totalmem, param->in_out.ramdisk_size, param->in_out.ramdisk_image));

  {
  extern unsigned move_kernel_src[], move_kernel_dst[], move_kernel_nblongs[], move_kernel_nblongs2[],
		move_initrd_src[], move_initrd_dst[], move_initrd_nblongs[], /* copy top to bottom */
		move_jump_addr[];
  extern char code32_start_relay[];

  if (param->in_out.header_version_number >= 0x020A && param->in_out.pref_address) {
      param->in_out.code32_start = LOADER.comment.elem.late_relocation_address = param->in_out.pref_address;
      ZDBG (("Adjust code32_start & late_relocation_address to pref_address = 0x%X\r\n", param->in_out.pref_address));
      }
    else {
      LOADER.comment.elem.late_relocation_address = 0x00100000U;
      if (param->in_out.header_version_number >= 0x0205 && param->in_out.relocatable_kernel != 0)
	  if (param->in_out.kernel_alignment > LOADER.comment.elem.late_relocation_address)
	      param->in_out.code32_start = LOADER.comment.elem.late_relocation_address = param->in_out.kernel_alignment;
      }

  if (LOADER.fileload[0].load_address + setup_len > LOADER.comment.elem.late_relocation_address)
      cs_pokel (move_kernel_nblongs, (LOADER.fileload[0].uncompressed_size - setup_len + 3 )/4);
    else {
      cs_pokel (move_kernel_nblongs, 0);
      cs_pokel (move_kernel_nblongs2, (LOADER.fileload[0].uncompressed_size - setup_len + 3 )/4);
      }
  cs_pokel (move_kernel_src, LOADER.fileload[0].load_address + setup_len);
  cs_pokel (move_kernel_dst, LOADER.comment.elem.late_relocation_address);

  cs_pokel (move_initrd_nblongs, (param->in_out.ramdisk_size + 3) / 4);
  cs_pokel (move_initrd_src, LOADER.fileload[1].load_address);
  cs_pokel (move_initrd_dst, param->in_out.ramdisk_image);

  cs_pokel (move_jump_addr, param->in_out.code32_start);
  param->in_out.code32_start = farptr2linear(((unsigned)getcs() << 16) + (unsigned)code32_start_relay);
  ZDBG (("Late relocation: will move %s %u bytes from 0x%X to 0x%X\r\n",
	cs_peekl(move_kernel_nblongs)? "(bottom to top)" : "(top to bottom)",
	4 * (cs_peekl(move_kernel_nblongs)? cs_peekl(move_kernel_nblongs) : cs_peekl(move_kernel_nblongs2)),
	cs_peekl(move_kernel_src), cs_peekl(move_kernel_dst)));
  ZDBG (("Late relocation: will move (top to bottom) %u bytes from 0x%X to 0x%X\r\n",
	4 * cs_peekl(move_initrd_nblongs), cs_peekl(move_initrd_src), cs_peekl(move_initrd_dst)));
  ZDBG (("Late jump: 0x%X, interceptor at 0x%X\r\n", cs_peekl(move_jump_addr), param->in_out.code32_start));

#if USER_SUPPORT != 0
  if (LOADER.edit_cmdline_b4_run) {
      draw_bg_box (1, UI.parameter.nbrow - 1, UI.stdcolor[lightgray]);
      UI.function.setfgcolor (UI.stdcolor[blue]);
      UI.function.setbgcolor (UI.stdcolor[lightgray]);
      UI.function.setcursor (3, 0);
      printf (" loaded kernel '%s' size %u will relocate at 0x%X\r\n", BOOTWAY.desc[LOADER.fileload[0].BOOTWAY_desc_index].filename, system_desc->filesize, LOADER.comment.elem.late_relocation_address);
      if (param->in_out.ramdisk_size)
	  printf (" loaded initrd '%s' size %u will relocate at 0x%X\r\n", BOOTWAY.desc[LOADER.fileload[1].BOOTWAY_desc_index].filename, param->in_out.ramdisk_size, param->in_out.ramdisk_image);
      printf (" relocate interceptor at 0x%X, code32 entry 0x%X\r\n", param->in_out.code32_start, cs_peekl(move_jump_addr));
      printf (" kernel_setup at 0x%X length %u, command line at 0x%X\r\n", kernel_setup, setup_len, param->in_out.cmd_line_ptr);
      puts ("");
      print (MSG_COMMAND_LINE);
      UI.function.setfgcolor (UI.stdcolor[brown]);
      get_line ((char *)param->command_line, sizeof (param->command_line), '\0', ' ', '\377');
      puts ("");
      UI.function.setfgcolor (UI.stdcolor[lightgray]);
      UI.function.setbgcolor (UI.stdcolor[black]);
      puts ("");
      UI.function.getcursor(&param->screen_info.orig_y, &param->screen_info.orig_x);
      }
    else
      param->screen_info.orig_y = param->screen_info.orig_x = 0;
#endif
  }

  smemcpy (kernel_setup, param, (param->in_out.header_version_number >= 0x0206)? (sizeof(fourKbuffer)/2) : sizeof(fourKbuffer));

  if (_BIOS_get_shift_flags().control_key_pressed) {
      print (MSG_COMMAND_LINE);
      print ("\"");
      print ((char *)param->command_line);
      print ("\", ");
      while (_BIOS_get_shift_flags().control_key_pressed)
	  continue;
      }

  {
  unsigned tmp = prepare_exit (0);
  if (tmp)
      return tmp;
  }

  puts (GO_KERNEL);
  asm volatile (
"	pushfl					\n"
"	pushw	%ds				\n"
"	pushw	%es				\n"
"	pushw	%fs				\n"
"	pushw	%gs				\n"
"	pushal					\n"
"	pushl	4*0x18	# INT 0x18		\n"
"	pushl	4*0x19	# INT 0x19		\n"
"	movw	%ss,%cs:unlikely_ret+1		\n"
"	movl	%esp,%cs:unlikely_ret+7		\n"
"	movw	%cs,%ax				\n"
"	shl	$16,%eax			\n"
"	movw	$unlikely_ret,%ax		\n"
"	mov	%eax,4*0x18	# INT 0x18	\n"
"	mov	%eax,4*0x19	# INT 0x19	\n"
"	cli					\n"
#ifndef KERNEL_SETUP_AT_90000
"	movw	%ss,%ax				\n"
"	addw	$0x1000,%ax			\n"
#else
"	movw	$0x9000,%ax			\n"
#endif
"	movw	%ax,%ds				\n"
"	movw	%ax,%es				\n"
"	movw	%ax,%fs				\n"
"	movw	%ax,%gs				\n"
"	movw	%ax,%ss				\n"
#ifndef KERNEL_SETUP_AT_90000
"	xor	%esp,%esp			\n"
#else
"	mov	$0x9000,%esp			\n"
#endif
"	pushw	%cs				\n"
"	pushw	$unlikely_ret			\n"
"	addw	$0x0020,%ax			\n"
"	pushw	%ax				\n"
"	pushw	$0				\n"
"	lretw					\n"
"unlikely_ret:					\n"
"	movw	$1234,%sp			\n"
"	movw	%sp,%ss				\n"
"	movl	$12345678,%esp			\n"
"	popl	4*0x19	# INT 0x19		\n"
"	popl	4*0x18	# INT 0x18		\n"
"	popal					\n"
"	popw	%gs				\n"
"	popw	%fs				\n"
"	popw	%es				\n"
"	popw	%ds				\n"
"	popfl					\n"
"	jmp	bzimage_return			\n"

/* Self modifying code: */
"	.align 8				\n"
"	.code32					\n"
"code32_start_relay:				\n"
"	movw	$0x18,%cx			\n"
"	movw	%cx,%ds				\n"
"	movw	%cx,%es				\n"
"	mov	%esi,%eax			\n"
"	mov	%edi,%edx			\n"

"move_initrd:					\n"
"	movl	$0x11111111,%esi		\n"
"	movl	$0x22222222,%edi		\n"
"	movl	$0x33333333,%ecx		\n"
"move_initrd_src = move_initrd + 1		\n"
"move_initrd_dst = move_initrd + 6		\n"
"move_initrd_nblongs = move_initrd + 11		\n"
//"	jecxz	move_kernel			\n"
"	std		# from top to bottom	\n"
"	leal    -4(%esi,%ecx,4),%esi		\n"
"	leal    -4(%edi,%ecx,4),%edi		\n"
"	rep movsl %ds:(%esi),%es:(%edi)		\n"
"	cld					\n"

"move_kernel:					\n"
"	movl	$0x44444444,%esi		\n"
"	movl	$0x55555555,%edi		\n"
"	movl	$0x66666666,%ecx		\n"
"move_kernel_src = move_kernel + 1		\n"
"move_kernel_dst = move_kernel + 6		\n"
"move_kernel_nblongs = move_kernel + 11		\n"
"	jecxz	1f				\n"
"	cld		# from bottom to top	\n"
"	rep movsl %ds:(%esi),%es:(%edi)		\n"
"	jmp	kernel_moved			\n"
"	1:					\n"
"	movl	$0x77777777,%ecx		\n"
"move_kernel_nblongs2 = 1b + 1			\n"
//"	jecxz	kernel_moved			\n"
"	leal    -4(%esi,%ecx,4),%esi		\n"
"	leal    -4(%edi,%ecx,4),%edi		\n"
"	std		# from top to bottom	\n"
"	rep movsl %ds:(%esi),%es:(%edi)		\n"
"	cld					\n"

"kernel_moved:					\n"
"	mov	%eax, %esi			\n"
"	mov	%edx, %edi			\n"

"move_jump:					\n"
"	movl	$0x88888888,%eax		\n"
"move_jump_addr = move_jump + 1			\n"
"	jmp	*%eax				\n"
"	.code16gcc				\n"
"bzimage_return:				\n"
	);
  return 0;
  }

//#define GRUB_MULTIBOOT
#ifdef GRUB_MULTIBOOT
/* Grub multiboot version 0.6.95, http://www.gnu.org/software/grub/manual/multiboot/multiboot.html */
#define MULTIBOOT_HEADER_MAGIC		0x1BADB002
#define MULTIBOOT_BOOTLOADER_MAGIC	0x2BADB002

struct multiboot_header_s {
    unsigned	magic;
    struct {
	unsigned	align_on_4Kpage		: 1;
	unsigned	memory_available	: 1;
	unsigned	video_available		: 1;
	unsigned	unused_requirement	: 13;
	unsigned	adress_valid		: 1;
	unsigned	unused_features		: 15;
	} __attribute__((packed)) flags;
    unsigned	checksum;
    /* if flags.adress_valid : */
    unsigned	header_addr;
    unsigned	load_addr;
    unsigned	load_end_addr;
    unsigned	bss_end_addr;
    unsigned	entry_addr;
    /* if flags.video_available : */
    unsigned	mode_type;
    unsigned	width;
    unsigned	height;
    unsigned	depth;
    } __attribute__((packed));

struct multiboot_info_s {
    struct {
	unsigned	mem_present			: 1;
	unsigned	boot_device_present		: 1;
	unsigned	cmdline_present			: 1;
	unsigned	mods_present			: 1;
	enum { syms_a_out = 1, syms_elf = 2 } syms_present	: 2;
	unsigned	mmap_present			: 1;
	unsigned	drive_present			: 1;
	unsigned	config_table_present		: 1;
	unsigned	boot_loader_name_present	: 1;
	unsigned	apm_table_present		: 1;
	unsigned	vbe_present			: 1;
	unsigned	unused				: 20;
	} __attribute__((packed)) flags;
    unsigned	mem_lower;
    unsigned	mem_upper;
    unsigned	boot_device;
    unsigned	cmdline;
    unsigned	mods_count;
    unsigned	mods_addr;
    union {
	struct {
	    unsigned	tabsize;
	    unsigned	strsize;
	    unsigned	addr;
	    unsigned	reserved;
	    } aout_sym;
	struct {
	    unsigned	num;
	    unsigned	size;
	    unsigned	addr;
	    unsigned	shndx;
	    } elf_sec;
	} syms;
    unsigned	mmap_length;
    unsigned	mmap_addr;
    unsigned	drive_length;
    unsigned	drive_addr;
    unsigned	config_table;
    unsigned	boot_loader_name;
    unsigned	apm_table;
    unsigned	vbe_control_info;
    unsigned	vbe_mode_info;
    unsigned	vbe_mode;
    unsigned short	vbe_interface_seg;
    unsigned short	vbe_interface_off;
    unsigned short	vbe_interface_len;
    } __attribute__((packed));

struct multiboot_mods_s {
    unsigned	mod_start;
    unsigned	mod_end;
    unsigned	string;	/* linear ptr */
    unsigned	reserved;
    } __attribute__((packed));

/* The memory map. Be careful that the offset 0 is base_addr_low but no size. */
struct memory_map_s {
    unsigned	size;
//    unsigned	base_addr_low;
//    unsigned	base_addr_high;
    unsigned long long	base_addr;
//    unsigned	length_low;
//    unsigned	length_high;
    unsigned long long	length;
    unsigned	type;
    };

struct drive_structure_s {
    unsigned		size;
    unsigned char	drive_number;
    unsigned char	drive_mode;
    unsigned short	drive_cylinders;	/* maximum cylinders or number of cylinders as documented? */
    unsigned char	drive_heads;		/* maximum head number or number of heads as documented? */
    unsigned char	drive_sectors;
// anyway not knowing the master/slave bit won't help, do not provide:
//    unsigned short	drive_port1;
//    unsigned short	drive_port2;
// This structure does not even encode the size of the drive for EBIOS, let me think...
    } __attribute__((packed));

LOAD_FCT_PREFIX(multiboot_protocol_run) static unsigned
multiboot_protocol_run (struct desc_str *system_desc, unsigned totalmem)
  {
  static const seg_t PMSeg2[] asm ("PMseg2") __attribute__ (( aligned(8) )) = {
    { .data = {}}, /* segment 0 is the invalid one */
    { .data = {	/* segment 0x08 */
    .limit	= 0xFFFF,
    .base	= 0,
    .accessed	= 1,	/* 486+: do not update it if it is already set */
    .writable	= 1,
    .cste_10	= 2,
    .present	= 1,
    .limit_msb	= 0xF,
//    .big	= 1,
    .granularity	= 1	/* limit = 0xFFFFF * 4 Kb = 4 Gb */
    }},
    { .code = {	/* segment 0x10 */
    .limit	= 0xFFFF,
    .base	= 0,
    .accessed	= 1,	/* 486+: do not update it if it is already set */
    .readable	= 1,
    .cste_11	= 3,
    .present	= 1,
    .limit_msb	= 0xF,
    .deflt	= 1,
    .granularity	= 1	/* limit = 0xFFFFF * 4 Kb = 4 Gb */
    }}
    };
  /* NOTE: On some CPUs, the GDT _must_ be 8 byte aligned. */
  gdt_idt_param_t gdt __attribute__ (( aligned(16) )) = {
      .limit = sizeof (PMSeg2) - 1, /* -1 : see Intel doc */
      .base  = farptr2linear (data_adr (PMSeg2))
      };
  unsigned multiboot_offset;

  copy_from_first_file (0, data_adr (fourKbuffer), sizeof (fourKbuffer));
  for (multiboot_offset = 0; multiboot_offset < 1024; multiboot_offset++)
      if (((unsigned*)fourKbuffer)[multiboot_offset] == MULTIBOOT_HEADER_MAGIC && (((unsigned*)fourKbuffer)[multiboot_offset] + ((unsigned*)fourKbuffer)[multiboot_offset+1] + ((unsigned*)fourKbuffer)[multiboot_offset+2]) == 0)
	  break;	/* Magic and checksum OK */
  if (multiboot_offset >= 1024) {
      ZDBG (("In %s, did not find back multiboot header!\r\n", __FUNCTION__));
      return 1;
      }
  struct multiboot_header_s *multiboot_header_ptr = (struct multiboot_header_s *)&((unsigned*)fourKbuffer)[multiboot_offset];
  ZDBG (("In %s, multiboot_header at offset 0x%X\r\n", __FUNCTION__, multiboot_header_ptr));
  struct multiboot_header_s multiboot_header = *multiboot_header_ptr;
  if (multiboot_header.flags.adress_valid)
      ZDBG (("header_addr = 0x%X, load_addr = 0x%X, load_end_addr = 0x%X, bss_end_addr = 0x%X, entry_addr = 0x%X\r\n", multiboot_header.header_addr, multiboot_header.load_addr, multiboot_header.load_end_addr, multiboot_header.bss_end_addr, multiboot_header.entry_addr));
    else {
      ZDBG (("multiboot_header.flags.adress_valid is not set\r\n"));
      return 1;
      }
  if (multiboot_header.header_addr != (unsigned)multiboot_header_ptr)
      ZDBG (("Need to apply an offset of %d\r\n", multiboot_header.header_addr - (unsigned)multiboot_header_ptr));
  if (multiboot_header.flags.video_available) {
      unsigned cptmode = UI.nbmode;
      ZDBG (("mode_type = 0x%X, width %u, height %u, depth %u\r\n", multiboot_header.mode_type, multiboot_header.width, multiboot_header.height, multiboot_header.depth));
      if (multiboot_header.mode_type == 1) { /* EGA */
	  for (cptmode = 0; cptmode < UI.nbmode; cptmode++)
	      if (UI.mode[cptmode].text && UI.mode[cptmode].width == multiboot_header.width && UI.mode[cptmode].height == multiboot_header.height)
		  break;
	  if (cptmode >= UI.nbmode) {
	      for (cptmode = 0; cptmode < UI.nbmode; cptmode++)
		  if (UI.mode[cptmode].text && (UI.mode[cptmode].width == multiboot_header.width || UI.mode[cptmode].height == multiboot_header.height))
		      break;
	      }
	  }
	else if (UI.info.vesa.version >= 0x200) {
	  for (cptmode = 0; cptmode < UI.nbmode; cptmode++)
	      if (UI.mode[cptmode].vesa && UI.mode[cptmode].width == multiboot_header.width && UI.mode[cptmode].height == multiboot_header.height
		  &&  (UI.mode[cptmode].bpp == multiboot_header.depth || multiboot_header.depth == 0))
		  break;
	  if (cptmode >= UI.nbmode) {
	      for (cptmode = 0; cptmode < UI.nbmode; cptmode++)
		  if (UI.mode[cptmode].vesa && UI.mode[cptmode].width == multiboot_header.width && UI.mode[cptmode].height == multiboot_header.height)
		      break;
	      }
	  if (cptmode >= UI.nbmode) {
	      for (cptmode = 0; cptmode < UI.nbmode; cptmode++)
		  if (UI.mode[cptmode].vesa && (UI.mode[cptmode].width == multiboot_header.width || UI.mode[cptmode].height == multiboot_header.height))
		      break;
	      }
	  }
      if (cptmode < UI.nbmode)
	  UI.function.setmode(UI.mode[cptmode].number);
      }

  struct multiboot_info_s *multiboot_info = (struct multiboot_info_s *)fourKbuffer;
  unsigned char *buffer_command_line = &fourKbuffer[3*1024];
  *multiboot_info = (struct multiboot_info_s){};

  multiboot_info->mem_lower = _BIOS_getBaseMemory();
  multiboot_info->mem_upper = 0;
  unsigned short sizeinKb;
  if (_BIOS_getExtendedMemory(&sizeinKb) == 0)
      multiboot_info->mem_upper = sizeinKb;
    else
      multiboot_info->mem_upper = totalmem;
  multiboot_info->flags.mem_present = 1;

  multiboot_info->boot_device = 0xFFFF0000 | (256 * DI.param[system_desc->disk].partition[system_desc->partition].OSnumber) | DI.param[system_desc->disk].disknb;
  multiboot_info->flags.boot_device_present = 1;

  build_final_command_line (buffer_command_line, 0, 0, 0, system_desc)
//printf ("Final cmdline: '%s'\r\n", buffer_command_line); _BIOS_wait (3 * 1000 * 1000);

  multiboot_info->cmdline = farptr2linear(stack_adr(buffer_command_line));
  multiboot_info->flags.cmdline_present = 1;

  multiboot_info->mods_count = 0;
  multiboot_info->mods_addr = 0;
  multiboot_info->flags.mods_present = 0;
#if 0
  multiboot_info->syms.aout_sym.tabsize = ;
  multiboot_info->syms.aout_sym.strsize = ;
  multiboot_info->syms.aout_sym.addr = ;
  multiboot_info->syms.aout_sym.reserved = ;
  multiboot_info->flags.syms_present = syms_a_out;
#elif 0
  multiboot_info->syms.elf_sec.num = ;
  multiboot_info->syms.elf_sec.size = ;
  multiboot_info->syms.elf_sec.addr = ;
  multiboot_info->syms.elf_sec.shndx = ;
  multiboot_info->flags.syms_present = syms_elf;
#else
  multiboot_info->flags.syms_present = 0;
#endif


  struct e820map_info e820map_elem;
  unsigned cont_val = 0, nb_e820map_elem = 0, cpt_e820map;
  while (_BIOS_QueryMemoryMap(&cont_val, sizeof(e820map_elem), &e820map_elem) == sizeof(e820map_elem))
      nb_e820map_elem++;
  struct memory_map_s memory_map[nb_e820map_elem];
  for (cont_val = 0, cpt_e820map = 0; cpt_e820map < nb_e820map_elem; cpt_e820map++) {
      _BIOS_QueryMemoryMap(&cont_val, sizeof(e820map_elem), &e820map_elem);
      memory_map[cpt_e820map].size = sizeof (struct memory_map_s);
      memory_map[cpt_e820map].base_addr = e820map_elem.base;
      memory_map[cpt_e820map].length = e820map_elem.length;
      memory_map[cpt_e820map].type = e820map_elem.type;
      };
  multiboot_info->mmap_length = sizeof (memory_map);
  multiboot_info->mmap_addr = farptr2linear(stack_adr(memory_map));
  multiboot_info->flags.mmap_present = 1;

  unsigned i, j = 0, cpt_drive = 0;
  for (i = 0; i < DI.nbdisk; i++)
      if (DI.param[i].access == bios_chs || DI.param[i].access == ebios_lba)
	  cpt_drive++;
  struct drive_structure_s drive_structure[cpt_drive];
  for (i = 0; i < DI.nbdisk; i++)
      if (DI.param[i].access == bios_chs || DI.param[i].access == ebios_lba) {
	  drive_structure[j].size = sizeof (drive_structure[0]);
	  drive_structure[j].drive_number = DI.param[i].disknb;
	  drive_structure[j].drive_mode = (DI.param[i].access == ebios_lba);
	  drive_structure[j].drive_cylinders = DI.param[i].nbcylinder;
	  drive_structure[j].drive_heads = DI.param[i].nbhead - 1;	/* maximum head number, it is a byte */
	  drive_structure[j].drive_sectors = DI.param[i].nbsectorpertrack;
//	  drive_structure[j].drive_port1 = DI.param[i].ideIOadr;
//	  drive_structure[j].drive_port2 = DI.param[i].ideIOctrladr;
	  j++;
	  }
  multiboot_info->drive_length = sizeof (drive_structure);
  multiboot_info->drive_addr = farptr2linear(stack_adr(&drive_structure));
  multiboot_info->flags.drive_present = 1;

  unsigned short XtAt_carry;
  multiboot_info->config_table = farptr2linear(_BIOS_getConfiguration (&XtAt_carry));
  multiboot_info->flags.config_table_present = !!multiboot_info->config_table;

  static const char boot_loader_name[] = "Gujin-v2.8";	/* VERSION */
  multiboot_info->boot_loader_name = farptr2linear(data_adr(&boot_loader_name));
  multiboot_info->flags.boot_loader_name_present = 1;

  struct apm_bios_info apm_bios_info;
  if (_APM_installation_check (&apm_bios_info.version, &apm_bios_info.flags) == 0
	&& (apm_bios_info.flags & 2) == 0
	&& _APM_connect_32bit(&apm_bios_info.cseg, &apm_bios_info.offset,
				    &apm_bios_info.cseg_16, &apm_bios_info.dseg,
				    &apm_bios_info.cseg_len, &apm_bios_info.dseg_len) != 0) {
      apm_bios_info.unused = 0;	/* cseg_16_len */
      multiboot_info->apm_table = farptr2linear(stack_adr(&apm_bios_info));
      multiboot_info->flags.apm_table_present = 1;
      }
    else
      multiboot_info->flags.apm_table_present = 0;

  unsigned short videomode;
  VESA_VbeInfoBlock VESA_info;
  VESA_modeinfo_t VESA_modeinfo;
  if (multiboot_header.flags.video_available && _VESA_getmode (&videomode) == 0
	&& _VESA_getinfo (&VESA_info) == 0 && _VESA_getmodeinfo (&VESA_modeinfo, videomode) == 0) {
      unsigned short len;
      farptr addr;
      multiboot_info->vbe_control_info = farptr2linear(stack_adr(&VESA_info));
      multiboot_info->vbe_mode_info = farptr2linear(stack_adr(&VESA_modeinfo));
      multiboot_info->vbe_mode = videomode;
      if (_VESA_GetPMinterface (&addr, &len) == 0) {
	  multiboot_info->vbe_interface_seg = addr >> 16;
	  multiboot_info->vbe_interface_off = addr;
	  multiboot_info->vbe_interface_len = len;
	  }
      multiboot_info->flags.vbe_present = 1;
      }
    else
      multiboot_info->flags.vbe_present = 0;


  unsigned tmp = prepare_exit (1);
  if (tmp)
      return tmp;

  printf (STARTING_KERNEL, BOOTWAY.desc[LOADER.fileload[0].BOOTWAY_desc_index].filename);
  disable_interrupt();
  disable_nmi();
  reset_387();

  if (protected_mode() != 0) {
#if !(SETUP & BIOS_ONLY)
      if (UTIL.v86_callback != 0)
	  exit_broadcast();
#endif
      enable_interrupt();
      return 2;
      }

  set_gdt (&gdt);

  asm volatile (
"	movl	%%eax,%%cs:patchentry		\n"
"	movl	$0x8,%%eax			\n"
"	movw	%%ax,%%ds			\n"
"	movw	%%ax,%%es			\n"
"	movw	%%ax,%%fs			\n"
"	movw	%%ax,%%gs			\n"
"	movw	%%ax,%%ss			\n"
"	mov	%%fs:0,%%al			\n"
"	mov	%%gs:0,%%al			\n"
"	mov	%%ss:0,%%al			\n"
"	cld					\n"
"	cmp	%%esi,%%edi			\n"
"	jl	1f				\n"
"	leal	-4(%%edi,%%ecx,4),%%edi		\n"
"	leal	-4(%%esi,%%ecx,4),%%esi		\n"
"	std					\n"
"	1:					\n"
"	rep movsl %%ds:(%%esi),%%es:(%%edi)	\n"
"	pushl	$0				\n"
"	popfl			# so cld, cli	\n"
"	mov	$0x2BADB002,%%eax		\n"
"patchentry = . + 2				\n"
"	ljmpl	$0x10, $0x100000 # self-modifying code: NOT IN ROM!	\n"
	: : "a" (multiboot_header.entry_addr),
	"b" (farptr2linear(stack_adr(&multiboot_info))),
	"c" ((multiboot_header.load_end_addr - multiboot_header.load_addr + 3)/4),
	"D" (multiboot_header.load_addr - (multiboot_header.header_addr - (unsigned)multiboot_header_ptr)),
	"S" (LOADER.fileload[0].load_address)
	);

  return 1;
  }
#endif /* GRUB_MULTIBOOT */

#if SETUP & XCODE_SEGMENT
/* Dummy function just to define "fptr_elf_set_params" */
awk_farret (elf_set_params);
LOAD_FCT_PREFIX(elf_set_params) __attribute__ ((cdecl, regparm (0), noinline))
static unsigned elf_set_params (
		unsigned totalmem,
		struct loader_t bootloader_type,
		struct linux_param *LnxParam,
		const struct LOADER_str *loader,
		const struct gpl_compliant_str *togpl,
		const struct desc_str *system_desc,
		int proposed_row,
		unsigned char **stringptr,
		unsigned char *cmdline)
  {
  return 1; /* THAT IS NOT THE VALUE RETURNED */
  }
#else
/* mingujin.exe and tiny_img shall still call the linux_set_param of
   the ELF file if it is present, and they do not have the AWK treatment */
LOAD_FCT_PREFIX(afterstack_set_param) __attribute__ ((cdecl, regparm (0), noinline)) static unsigned
afterstack_set_param (unsigned totalmem,
		struct loader_t bootloader_type,
		struct linux_param *LnxParam,
		const struct LOADER_str *loader,
		const struct gpl_compliant_str *togpl,
		const struct desc_str *system_desc,
		int proposed_row,
		unsigned char **stringptr,
		unsigned char *cmdline)
  {
  asm volatile (
	/* called intermediate-function shall finish by lretw: */
"	popl	%fs		\n"
"	pushw	%cs		\n"
"	pushw	%fs		\n"
	/* called intermediate-function shall be just after the stack: */
"	pushw	%ss		\n"
"	addw	$0x1000,(%esp)	\n"
"	pushw	$0		\n"
"	lretw			\n"
	);
  return 0; /* THAT IS NOT THE VALUE RETURNED */
  }
#endif

LOAD_FCT_PREFIX(elf_getparams_run) static inline unsigned
elf_getparams_run (struct desc_str *system_desc,
			unsigned totalmem, char **message,
			struct registers *regs)
  {
  unsigned returned, elf_set_params_returned = 0;
  struct linux_param LnxParam = {};
  struct gpl_compliant_str togpl = {};
  struct elf_reloc_s elf_reloc[MAXNBCPY - 2]; /* keep 1 for param and 1 for initrd */
  unsigned nb_elf_reloc = sizeof(elf_reloc)/sizeof(elf_reloc[0]); // input: max size
  /* Adjusted later by removing LOADER.comment.elem.realfct_size: */
  unsigned base_real_fct;

  test_linux_param_struct();

  /* Do not call an external function if that is the wrong processor: */
  returned = check_comment_restriction (totalmem);
  base_real_fct = get_base_real_fct (&nb_elf_reloc, elf_reloc);

  if (returned) {
      *message = 0;
      return returned;
      }

//  if (base_real_fct == 0xFFFFFFFF && !LOADER.license_compliant)
//      returned = 0x17; // no more linux_set_param
  if (!LOADER.license_compliant && LOADER.comment.elem.realfct_size != 0
	&& (nb_elf_reloc > 1 || LOADER.comment.elem.runadr != 0))
      returned = 0x16;	/* Still accept single reloc real mode without any license */

  ZDBG (("In %s, LOADER.license_compliant = %u, LOADER.kernel_64bits = %u\r\n", __FUNCTION__, LOADER.license_compliant, LOADER.kernel_64bits));
  /* initialise that before calling the indirect function to set parameters: */
  if (LOADER.license_compliant) {
#ifndef LOAD_TASKREG
      extern const seg_t PMseg[];
#endif
#if USER_SUPPORT == 0
      unsigned final_loadrun (unsigned index, struct registers *regs,
		struct gujin_param_attrib gujin_attr);
#endif
      togpl = (struct gpl_compliant_str) {
	  .signature =	0x01031967,
	  .version =	0x208,	/* VERSION */
	  .feature =	3,
	  .size =	sizeof (struct gpl_compliant_str),

	  .filename_array =	0,
#ifndef LOAD_TASKREG
	  .gdt =		(unsigned)PMseg,
#endif
	  .regs =		(unsigned)regs,
	  .fourKbuffer =	(unsigned)fourKbuffer,

	  .LOADER =	(unsigned)&LOADER,
	  .STATE =	(unsigned)&STATE,
#if USER_SUPPORT != 0
	  .UI =		(unsigned)&UI,
#endif
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT | SERIAL_MOUSE_SUPPORT)
	  .MOUSE =	(unsigned)&MOUSE,
#endif
	  .DI =		(unsigned)&DI,
	  .UTIL =	(unsigned)&UTIL,
	  .BOOTWAY =	(unsigned)&BOOTWAY,

#if USER_SUPPORT != 0
#if (SETUP & XDATA_SEGMENT)
	  .font8x8_farptr = xdata_adr(font8x8),
	  .font8x14_farptr = xdata_adr(font8x14),
	  .font8x16_farptr = xdata_adr(font8x16),
	  .font10x20_farptr = xdata_adr(font10x20),
	  .font12x24_farptr = 0,
#else
	  .font8x8_farptr = data_adr(font8x8),
	  .font8x14_farptr = data_adr(font8x14),
	  .font8x16_farptr = data_adr(font8x16),
	  .font10x20_farptr = data_adr(font10x20),
	  .font12x24_farptr = 0,
#endif

	  .fct = {
	      .putstr = UI.function.putstr,
	      .printf = exported_togpl.printf,
	      .getkey = UI.function.getkey,
	      .get_number = exported_togpl.get_number,
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
	      .reset_field = MOUSE.function.reset_field,
	      .print_field = MOUSE.function.print_field,
#endif
	      .clearscreen = UI.function.clearscreen,
	      .draw_bg_box = exported_togpl.draw_bg_box,
	      .setcursor = UI.function.setcursor,
	      .getcursor = UI.function.getcursor,
	      .setfgcolor = UI.function.setfgcolor,
	      .setbgcolor = UI.function.setbgcolor,
	      .setattribute = UI.function.setattribute,
	      .setpixel = exported_togpl.setpixel,
	      .getpixel = exported_togpl.getpixel,
	      .plotHline = exported_togpl.plotHline,
	      .VESA_color = exported_togpl.VESA_color,
	      .final_loadrun = exported_togpl.final_loadrun,
	      .get_line = exported_togpl.get_line, /* Gujin-1.6+ */
	      },

	  .current_param = (typeof(togpl.current_param)) &UI.parameter,
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
	  .mouse_colors = &MOUSE.all_colors.colors,
#endif
#else
	  .fct = {
	      .putstr = (typeof (togpl.fct.putstr))(((unsigned)getcs() << 16) | (unsigned)print),
	      .printf = (typeof (togpl.fct.printf))(((unsigned)getcs() << 16) | (unsigned)printf),
	      },
#endif /* USER_SUPPORT */
	  };
      }
//    else if (UTIL.processor.family >= 6)
//      disable_serial_cpu();

  /* paramadr is a linear address and should be in low memory,
	else you know what you are doing: */
  if (   LOADER.comment.elem.paramadr != 0
      && (   LOADER.comment.elem.paramadr < 0x100000
	  || LOADER.comment.elem.realfct_size != 0)) {
      int proposed_row = -1;

 #if USER_SUPPORT != 0 /* else UI does not exist */
      /* leave few lines to display Gujin messages: */
      {
      unsigned char row, col;

      UI.function.getcursor (&row, &col);
      if (row >= UI.parameter.nbrow - 4)
	  proposed_row = UI.parameter.nbrow - 1;
	else
	  proposed_row = row + 4;
      ZDBG (("giving cursor position as beginning of line %d.\r\n", proposed_row));
      }
#endif

     if (LOADER.comment.elem.realfct_size != 0) {
	  unsigned char *stringptr = (unsigned char *)(STATE.codeseg_adr + STATE.gujin_param_ptr->extra_cmdline);
	  unsigned char proposed_cmdline[512];
		/* There is no more unzip activity, we use the gzlib buffer,
		or for tiny the area after the stack (non PIC16 code): */
	  farptr buffer = (getss() + 0x1000) << 16;

	  build_final_command_line (proposed_cmdline, 0, 0, 0, system_desc);
	  ZDBG (("Big buffer at 0x%X, proposed_cmdline %s.\r\n", buffer, proposed_cmdline));

	  fmemset (buffer + LOADER.comment.elem.realfct_size, 0, 0xFFFF - LOADER.comment.elem.realfct_size);
	  if (copy_from_first_file (base_real_fct, buffer, LOADER.comment.elem.realfct_size) != 0)
	      return 7;

#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
	  unsigned saved_mouse_colors = MOUSE.all_colors.all;
#endif
#if SETUP & XCODE_SEGMENT
	  asm volatile ("mov %0,fptr_elf_set_params " : : "r" (buffer), "p" (elf_set_params));
//puts ("---------- BEFORE elf_set_params ----------------"); _BIOS_wait(5000000);
	  elf_set_params_returned = elf_set_params (totalmem,
			(struct loader_t) { .type = GUJIN, .version = 1 }, /* VERSION */
			&LnxParam, &LOADER, &togpl, system_desc, proposed_row,
			&stringptr, proposed_cmdline);
//puts ("---------- AFTER elf_set_params -----------------"); _BIOS_wait(5000000);
#else
	  ZDBG (("Calling afterstack_set_param (peekl(0x%X) = 0x%X)\r\n", buffer, peekl(buffer)));
//puts ("---------- BEFORE afterstack_set_param ----------"); _BIOS_wait(5000000);
	  elf_set_params_returned = afterstack_set_param (totalmem,
		(struct loader_t) { .type = GUJIN, .version = 1 }, /* VERSION */
		&LnxParam, &LOADER, &togpl, system_desc, proposed_row,
		&stringptr, proposed_cmdline);
//puts ("---------- AFTER afterstack_set_param -----------"); _BIOS_wait(5000000);
#endif
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
	  MOUSE.all_colors.all = saved_mouse_colors;
#endif

//	  GZLIB_TEST_MEASURE_do();

	  *message = (char *)stringptr;

	  ZDBG (("Final command line: '"));
	  ZDBGVAR (((char *)LnxParam.command_line));
	  ZDBG (("'.\r\n"));

//	  GZLIB_TEST_MEASURE_do();

	  /* if elf_set_params() returns a non zero value following an error,
		it has to be 7 to display the generic message in the right language.
		The 3 MSBits have the following significance:
		- 0x80000000 : that kernel has no protected part (not a linux kernel
			but a simple real mode utility).
		- 0x40000000 : Do not clear the screen after running the real mode
			utility.
		- 0x20000000 : Do not display "press a key to continue"
		- 0x10000000 : Do reprobe all disk & FS
		- 0x01000000 : Do not re-check the kernel CRC32
		- 0x02000000 : Do not re-check the initrd CRC32
		- 0x03000000 : Do not re-check the kernel/initrd CRC32
		*/
	  if ((elf_set_params_returned & 0x80FFFFFF) != 0)
	      return elf_set_params_returned;

	  if (stringptr && (unsigned long)stringptr < 0xFFF8) /* no error, but a message to display? */
	      puts ((char *)stringptr);

	  if (_BIOS_get_shift_flags().control_key_pressed) {
	      print (MSG_COMMAND_LINE);
	      print ("\"");
	      print ((char *)LnxParam.command_line);
	      print ("\", ");
	      while (_BIOS_get_shift_flags().control_key_pressed)
		  continue;
	      }
	  }
	else
	  ZDBG (("realfct_size == 0, NO initialisation of Parameters in real mode.\r\n"));
      }
    else
      ZDBG (("paramadr==0x%X, NO initialisation of Parameters in real mode.\r\n",
		LOADER.comment.elem.paramadr));

  if (BOOT1_DOS_running()) {
      ZDBG ((" [DISK cache "));
      DOS_ResetDisk ();	/* also flush - i.e. write - everything */
      ZDBG (("flushed] "));
      }

  { /* For ease of use, transform all pointers to linear addresses: */
  unsigned dataptr = (unsigned)getds() << 4;
  unsigned *ptr = &togpl.filename_array;

  while (ptr <= &togpl.BOOTWAY)
      *ptr++ += dataptr;
  }

  if ((returned = setjmp (&togpl.jmpbuf)) == 0) {
      if ((returned = crc32check(elf_set_params_returned >> 24)) == 0)
	  returned = elf_start ((LOADER.license_compliant) ? farptr2linear(stack_adr(&togpl)) : 0,
				&LnxParam, nb_elf_reloc, elf_reloc);
      }
    else
      return 0; /* back from longjump to togpl.jmpbuf */

  return returned;
  }

LOAD_FCT_PREFIX(vmlinuz_videomode_getparams_run) static inline unsigned
vmlinuz_videomode_getparams_run (unsigned totalmem, struct registers *regs)
  {
  unsigned returned;
  char *message;

#if DISK_SUPPORT == DOS_SUPPORT
  extern unsigned short selected_video_mode;
  unsigned short initial_video_mode = 0xFFFF;

  ZDBG (("selected_video_mode 0x%X, ", selected_video_mode));
  if (selected_video_mode != 0xFFFF) {
      if (_VESA_getmode(&initial_video_mode) == 0)
	  initial_video_mode &= 0x7FFF;
	else {
	  unsigned char nbcol, page;
	  initial_video_mode = _VGA_getmode (&nbcol, &page) & 0x7F;
	  }
      _VESA_setmode (selected_video_mode | 0x4000);
      }
  ZDBG (("initial_video_mode 0x%X\r\n", initial_video_mode));
#elif USER_SUPPORT != 0
  unsigned short current_mode = current_mode; /* it _is_ inited B4 used */
  unsigned fgcolor = fgcolor, bgcolor = bgcolor;
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
  unsigned char in_vesa = 0;
#endif

  if (BOOT1_COM_port() < 0) {
      current_mode = UI.function.getmode() & ~0x8000;
      fgcolor = UI.fgcolor;
      bgcolor = UI.bgcolor;
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
      in_vesa = VESA_ACTIVE();
#endif
      /*
       * Arggg... I do not know if this kernel has been compiled with
       * framebuffer enabled, and which kind of framebuffer (VESA, ATI...)
       * Starting a kernel in a VESA graphic mode when VESA framebuffer
       * is not compiled-in result in a kernel booting correctly but
       * the display is very strange...
       * I let the user decide on the configuration if it is safe
       * to run the kernel in graphic (VESA framebuffer stuff
       * for the current mode/BPP).
       */
      if (   UI.parameter.attr.isgraphic
	  && copy_gujin_param.attrib.force_textmode) {
	  unsigned short tmp = copy_gujin_param.default_text_video_mode;
	  unsigned char cpt;
	  unsigned err;

	  if (tmp != 0xFFFF) {
	      ZDBG (("Searching default text mode 0x%X\r\n", tmp));
	      for (cpt = 0; cpt < UI.nbmode; cpt++)
		  if (((UI.mode[cpt].number ^ tmp) & 0x3FF) == 0)
		      break;
	      if (cpt >= UI.nbmode)
		  ZDBG (("Default text mode 0x%X is not known!\r\n", tmp));
	      if (!UI.mode[cpt].text) {
		  cpt = UI.nbmode;
		  ZDBG (("Default text mode 0x%X is NOT text!\r\n", tmp));
		  }
	      }
	    else
	      cpt = UI.nbmode;
	  if (cpt >= UI.nbmode) {
	      ZDBG (("Searching a text mode: "));
	      for (cpt = 0; cpt < UI.nbmode; cpt++)
		  if (UI.mode[cpt].text)
		      break;
	      }
	  if (cpt >= UI.nbmode) {
	      ZDBG (("did not find any text mode, setting mode 3.\r\n"));
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
	      if (in_vesa) {
		  err = VGA_init();
		  if (err)
		      ZDBG (("Panic: cannot switch to VGA, error %u!\r\n", err));
		  }
#endif
	      err = UI.function.setmode (3 | 0x8000);
	      if (err)
		  ZDBG (("Panic: cannot switch to text mode 3, error %u!\r\n", err));
	      }
	    else {
	      ZDBG (("found mode 0x%X/%s.\r\n",
			UI.mode[cpt].number,
			UI.mode[cpt].vesa? "VESA" : "VGA"));
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
	      if (UI.mode[cpt].vesa) {
		  if (!in_vesa) {
		      ZDBG (("Switching to VESA.\r\n"));
		      err = VESA_init();
		      if (err)
			  ZDBG (("Panic: cannot switch to VESA, error %u!\r\n", err));
		      }
		  }
		else {
		  if (in_vesa) {
		      ZDBG (("Switching to VGA.\r\n"));
		      err = VGA_init();
		      if (err)
			  ZDBG (("Panic: cannot switch to VGA, error %u!\r\n", err));
		      }
		  }
#endif
	      err = UI.function.setmode (UI.mode[cpt].number | 0x8000);
	      if (err)
		  ZDBG (("Panic: cannot switch to text mode 0x%X, error 0x%X!\r\n",
				UI.mode[cpt].number | 0x8000, err));
	      }
	  UI.function.setfgcolor (UI.stdcolor[lightgray]);
	  UI.function.setbgcolor (UI.stdcolor[black]);
	  UI.function.clearscreen();
	  }
	else {
	  unsigned char row, col, cpt;

	  UI.function.setfgcolor (UI.stdcolor[lightgray]);
	  UI.function.setbgcolor (UI.stdcolor[black]);
	  UI.function.getcursor (&row, &col);
	  for (cpt = row; cpt < UI.parameter.nbrow; cpt++)
	      puts (""); /* scroll once */
	  UI.function.setcursor (row - 1, col);
	  }
      }
#endif /* USER_SUPPORT != 0 */

  if (BOOTWAY.desc[LOADER.fileload[0].BOOTWAY_desc_index].boottype == is_linux) {
      returned = bzImage_protocol_run (&BOOTWAY.desc[LOADER.fileload[0].BOOTWAY_desc_index], totalmem);
      ZDBG (("Back from bzImage_protocol_run with returned=0x%X, Is there really an error?\r\n", returned));
      }
#ifdef GRUB_MULTIBOOT
    else if (BOOTWAY.desc[LOADER.fileload[0].BOOTWAY_desc_index].boottype == is_multiboot) {
      returned = multiboot_protocol_run (&BOOTWAY.desc[LOADER.fileload[0].BOOTWAY_desc_index], totalmem);
      ZDBG (("Back from multiboot_protocol_run with returned=0x%X, Is there really an error?\r\n", returned));
      }
#endif
    else {
      returned = elf_getparams_run (&BOOTWAY.desc[LOADER.fileload[0].BOOTWAY_desc_index], totalmem, &message, regs);
      ZDBG (("Back from elf_getparams_run with returned=0x%X, Is there really an error?\r\n", returned));
      }

#if DISK_SUPPORT == DOS_SUPPORT
  if (initial_video_mode != 0xFFFF)
      if (_VESA_setmode(initial_video_mode) != 0)
	  _VGA_setmode (initial_video_mode);
#elif USER_SUPPORT != 0
  if (BOOT1_COM_port() < 0) {
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
      if (in_vesa && !VESA_ACTIVE())
	  VESA_init();
#endif
      if (current_mode != (UI.function.getmode() & ~0x8000)) {
	  UI.function.setmode (current_mode);
	  UI.function.setfgcolor (fgcolor);
	  UI.function.setbgcolor (bgcolor);
	  UI.function.clearscreen();
	  }
	else if ((returned & 0x40000000) == 0) {
	  UI.function.setfgcolor (fgcolor);
	  UI.function.setbgcolor (bgcolor);
	  UI.function.clearscreen();
	  }
      }
#endif

  if (returned & 0x80000000) {
      if (message) {
	  if ((returned & 0x00FFFFFF) != 0)
	      print (MSG_ERROR);
	  if ((unsigned long)message < 0xFFF8)
	      puts (message);
	  }
      }
    else if (returned != 0) {
      if (message) {	/* error, print that message after clearing the screen */
	  print (MSG_ERROR);
	  if ((unsigned long)message < 0xFFF8)
	      puts (message);
	  }
      printf (ERROR_LOADING_KERNEL, returned);
      const char *puts_arg = 0;
	/* error codes in check_comment_restriction() */
#if SETUP & MULTILINGUAL /* code size optimise, cannot use it if strings not constant */
      if (returned == 1)
	  puts_arg = ERROR_STARTUP_ENABLE_A20;
	else if (returned == 2)
	  puts_arg = ERROR_SWITCH_PROTECTED_MODE;
	else if (returned == 3 || returned == 4)
	  puts_arg = ERROR_EXIT_V86_MODE;
	else if (returned == 5)
	  puts_arg = ERROR_AMOUNT_MEMORY;
	else if (returned == 6)
	  puts_arg = ERROR_GUJIN_VERSION;
	else if (returned == 7)
	  puts_arg = ERROR_EXTERNAL_SET_PARAMS;
	else if (returned == 8)
	  puts_arg = ERROR_RECHECK_KERNEL_CRC32;
	else if (returned == 9)
	  puts_arg = ERROR_RECHECK_INITRD_CRC32;
	else if (returned == 0x10)
	  puts_arg = ERROR_BAD_CPU;
	else if (returned == 0x11)
	  puts_arg = ERROR_NOT_FROM_DOS;
	else if (returned == 0x12)
	  puts_arg = ERROR_NOT_FROM_VIRTUAL;
	else if (returned == 0x13 || returned == 0x14 || returned == 0x15)
	  puts_arg = ERROR_BAD_CPU_FLAGS;
	else if (returned == 0x16)
	  puts_arg = ERROR_BAD_LICENSE;
	else if (returned == 0x17)
	  puts_arg = ERROR_BAD_ELF;
	else if (returned == 0x18)
	  puts_arg = ERROR_CONTROL_BREAK;
#else
      if (returned <= 0x18)
	  puts_arg = (const char *[]) {
		  "(?)",
		  ERROR_STARTUP_ENABLE_A20,
		  ERROR_SWITCH_PROTECTED_MODE,
		  ERROR_EXIT_V86_MODE,
		  ERROR_EXIT_V86_MODE,
		  ERROR_AMOUNT_MEMORY,
		  ERROR_GUJIN_VERSION,
		  ERROR_EXTERNAL_SET_PARAMS,
		  ERROR_RECHECK_KERNEL_CRC32,
		  ERROR_RECHECK_INITRD_CRC32,
		  "(?)", "(?)", "(?)", "(?)", "(?)","(?)",
		  ERROR_BAD_CPU,
		  ERROR_NOT_FROM_DOS,
		  ERROR_NOT_FROM_VIRTUAL,
		  ERROR_BAD_CPU_FLAGS,
		  ERROR_BAD_CPU_FLAGS,
		  ERROR_BAD_CPU_FLAGS,
		  ERROR_BAD_LICENSE,
		  ERROR_BAD_ELF,
		  ERROR_CONTROL_BREAK,
		  }[returned];
#endif
	else if ((returned & 0xF0) == 0x20)
	  puts_arg = ERROR_BAD_SCREEN_ACCESS;
	else if ((returned & 0xF0) == 0x30)
	  puts_arg = ERROR_BAD_SCREEN_RESOLUTION;
      if (puts_arg)
	  puts (puts_arg);
	else
	  puts ("");
      _BIOS_wait (3 * 1000 * 1000);
      }

  return returned;
  }

LOAD_FCT_PREFIX(menu_load_system) unsigned
menu_load_system (unsigned system_index, struct registers *regs, unsigned char edit_cmdline_b4_run)
  {bound_stack();{
  unsigned returned = returned; /* inited b4 used */
  unsigned freemem; /* to load kernel & initrd, freemem in Kb */
  unsigned loadaddr;
  unsigned totalmem;
  unsigned useablemem = 0;

  _BIOS_reset_control_break();
#ifdef NB_ISO
  BOOTWAY.iso.current_used = -1; /* try to map only once the ISO */
#endif

  LOADER = (struct LOADER_str) {}; /* LOADER.license_compliant = LOADER.kernel_64bits = 0; */
  LOADER.edit_cmdline_b4_run = edit_cmdline_b4_run;
  LOADER.sizeof_loader_str = sizeof (struct LOADER_str);
  /* If you are lost in the web, go and visit "http://www.perdu.com" */
  LOADER.sizeof_fileload_str = sizeof (struct fileload_str);
  DBG (("sizeof struct LOADER_str %u, sizeof struct fileload_str %u\r\n",
	LOADER.sizeof_loader_str, LOADER.sizeof_fileload_str));
  /* These values can be overwritten by GZIP comments, init early: */
  LOADER.comment.elem.paramadr = 0x00080000;
  LOADER.comment.elem.late_relocation_address = default_late_relocation_address;
  ZDBG (("default ELF addresses: real mode parameter: 0x%X, late_relocation_address 0x%X.\r\n",
	LOADER.comment.elem.paramadr, LOADER.comment.elem.late_relocation_address));

  if (HIMEM_ACTIVE()) { /* implies (SETUP & BIOS_ONLY) == 0 */
      unsigned char error;

      ZDBG ((" [HIMEM: present, using XMS] "));
#if 0 /* A20 seems to be enabled even before, so no need for that: */
      if (_XMS_query_A20 (UTIL.HIMEM_entrypoint, &error))
	  ZDBG (("[init: A20 enabled, error 0x%X] ", error));
	else
	  ZDBG (("[init: A20 disabled, error 0x%X] ", error));

      if (_XMS_global_enable_A20 (UTIL.HIMEM_entrypoint, &error)) {
	  ZDBG (("cannot disable A20-HMA, error 0x%X!] ", error));
	  print (ERROR_XMS_ENABLE_A20);
	  return;
	  }
#endif
      if (_XMS_local_enable_A20 (UTIL.HIMEM_entrypoint, &error)) {
	  ZDBG (("cannot enable A20, error 0x%X!] ", error));
	  print (ERROR_XMS_ENABLE_A20);
	  return 0x00DDDDDD;
	  }
      freemem = UTIL.XMS_freemem;
      if (UTIL.emm_type != EMM_none) {
	  /* Note that when HIMEM.SYS just report 64 Mb of RAM on a
		machine with more than this, VCPI report the same. */
	  totalmem = UTIL.VCPI_max_address / 1024;
	  totalmem += 4;	/* max beginning address of last page */
	  /* totalmem -= 1024;	/ * Remove first megabyte */
	  ZDBG (("%s: using HIMEM.SYS freemem: %u Kb, VCPI totalmem: %u Kb\r\n",
		__FUNCTION__, freemem, totalmem));
	  }
	else {
	  totalmem = UTIL.extendmem + 1024; /* the first megabyte */
	  ZDBG (("%s: using HIMEM.SYS freemem: %u Kb, BIOS totalmem: %u Kb\r\n",
		__FUNCTION__, freemem, totalmem));
	  }
      loadaddr = 0; /* is constant, start offset in handle */
      }
    else {
      ZDBG ((" [HIMEM: abscent, using direct memory access, "));
#if (ASSEMBLY_TYPE != ASSEMBLY_DSES)
#if !(SETUP & USE_INT1587)
      ZDBG (("open doors, "));
      if (segment_door (1) != 0) {
	  ZDBG (("cannot, aborting] "));
	  print (ERROR_SEGMENT_DOOR);
	  return;
	  }
	else
	  ZDBG (("OK] "));
#else
      ZDBG (("USE_INT1587, no need to manage A20 now] "));
#endif
#else
      ZDBG (("ASSEMBLY_DSES, A20 & doors already open] "));
#endif
      totalmem = freemem = UTIL.extendmem + 1024; /* the first megabyte */
      ZDBG (("%s: using BIOS freemem/totalmem: %u Kb\r\n",
		__FUNCTION__, freemem));
      loadaddr = LOADER.comment.elem.late_relocation_address; /* is variable */
      }

  GZLIB_TEST_MEASURE_init();

  LOADER.curfileload = &LOADER.fileload[0];
  LOADER.curfileload->BOOTWAY_desc_index = system_index;
#if defined (INITRDNAME) || defined (INITRDNAME2)
  LOADER.initrd_index = initrd_file (system_index);
#endif
  /* this will be copied in gzlib.avail_out, and not modified later: */
  LOADER.curfileload->max_size_or_crc32 = freemem * 1024;
  LOADER.gzip_byte_treated = 0;
  if (BOOTWAY.desc[system_index].boottype == is_linux) /* always full LINUX compatibility mode */
      LOADER.accept_uncompressed_filesize = BOOTWAY.desc[system_index].filesize;
    else
      LOADER.accept_uncompressed_filesize = 0;	/* needs ELF to be compressed */
#if USER_SUPPORT != 0
  LOADER.post_comment_row = LOADER.post_comment_col = 0xFF;
#endif
  for (;;) {
      LOADER.curfileload->load_address = loadaddr;
// That is initialised to zero with "LOADER.*":
//      LOADER.curfileload->KBalloced = 0;
//      LOADER.curfileload->handle = 0;	/* allocate at first use */

      ZDBG (("%s: loading %s '%s' at 0x%X, max size %u, ",
		__FUNCTION__,
		((const char *[]){ "kernel", "initrd", "setup" }) [LOADER.curfileload - LOADER.fileload],
		BOOTWAY.desc[LOADER.curfileload->BOOTWAY_desc_index].filename,
		LOADER.curfileload->load_address,
		LOADER.curfileload->max_size_or_crc32));

      const char *loadfilemsg;
      if (LOADER.curfileload == LOADER.fileload)
	  loadfilemsg = LOADING_KERNEL_S;
	else
	  loadfilemsg = LOADING_INITRD_S;
      printf (loadfilemsg, BOOTWAY.desc[LOADER.curfileload->BOOTWAY_desc_index].filename);

      returned = system_file_load (&BOOTWAY.desc[LOADER.curfileload->BOOTWAY_desc_index]);
      ZDBG (("result: 0x%X (size: %u compressed, %u uncompressed bytes).\r\n",
		returned,
		LOADER.curfileload->compressed_size,
		LOADER.curfileload->uncompressed_size));
      if (BOOTWAY.desc[LOADER.curfileload->BOOTWAY_desc_index].filesize != LOADER.gzip_byte_treated)
	  ZDBG (("File is bigger than a single GZIP, contains %u - %u = %d bytes extra\r\n",
		BOOTWAY.desc[LOADER.curfileload->BOOTWAY_desc_index].filesize, LOADER.gzip_byte_treated,
		BOOTWAY.desc[LOADER.curfileload->BOOTWAY_desc_index].filesize - LOADER.gzip_byte_treated));
      if (LOADER.accept_uncompressed_filesize != 0)
	  ZDBG (("Stop accepting uncompressed file of size %u to load\r\n", LOADER.accept_uncompressed_filesize & 0x7FFFFFFF));

      if (returned == 0) {
	  if (_BIOS_control_break_pressed()) {
	      _BIOS_reset_control_break();
	      ZDBG (("Abort loading because ControlBreak pressed.\r\n"));
	      returned = 0x18; /* ERROR_CONTROL_BREAK */
	      break;
	      }
	  if (HIMEM_ACTIVE()) {
#if defined (INITRDNAME) || defined (INITRDNAME2)
	      unsigned char error;

	      if ((LOADER.curfileload->load_address = _XMS_lock (UTIL.HIMEM_entrypoint, LOADER.curfileload->handle, &error)) != 0) {
		  ZDBG (("Locked handle %u at 0x%X, error 0x%X\r\n",
			LOADER.curfileload->handle, LOADER.curfileload->load_address, error));
		  useablemem = 1024 * (freemem - LOADER.curfileload->KBalloced);
		  }
	      if (error) {
		  ZDBG (("FAILED last realloc or Lock handle %u error 0x%X\r\n",
			LOADER.curfileload->handle, error));
#if USER_SUPPORT != 0
		  if ((LOADER.post_comment_row & LOADER.post_comment_col) != 0xFF) {
		      UI.function.setcursor (LOADER.post_comment_row, LOADER.post_comment_col);
		      LOADER.post_comment_row = LOADER.post_comment_col = 0xFF;
		      }
#endif
		  printf (ERROR_XMS_LOCK, error);
		  returned = 0xFFFFFFFF;
		  break;
		  }
#endif
	      }
	    else {
	      /* Load the initrd just after kernel, at a round addr, no more any trick needed,
	       * the new late relocation system will move it at end of memory.
	       * TODO: try to guess now where it will be moved, to have it directly at the
	       *   right position, if it is not compressed.
	       */
	      loadaddr = ROUNDUP(loadaddr + LOADER.curfileload->uncompressed_size, 0x1000);
	      useablemem = 1024 * freemem - loadaddr;
	      ZDBG (("File loaded, next loadaddr 0x%X, maxsize %u\r\n", loadaddr, useablemem));
	      }

	  LOADER.curfileload++;
#if defined (INITRDNAME) || defined (INITRDNAME2)
	  if (LOADER.initrd_index > 0) {
	      LOADER.curfileload->BOOTWAY_desc_index = LOADER.initrd_index;
	      LOADER.gzip_byte_treated = 0;
	      LOADER.accept_uncompressed_filesize = BOOTWAY.desc[LOADER.initrd_index].filesize;
	      if (BOOTWAY.desc[LOADER.initrd_index].boottype == is_initramfs
			|| BOOTWAY.desc[system_index].boottype == is_linux)
		  LOADER.accept_uncompressed_filesize |= 0x80000000;	/* i.e. do not decompress */
	      }
	    else
#endif
		 if (LOADER.initrd_index == -1) {	/* -1 if GZIP concated initrd, see fs.c::gzlib_extract() */
		/* Gentoo fails here - there is an external initramfs treated as initrd and that shall not be
			decompressed because there is too many comcatenated GZIP... Simply compile the kernel
			with the initramfs included? Gujin could also back down and restart the loading
			without decompressing, or the complete initramfs can be gzip'ed again...
		   Since v1.6, just rename /boot/initrd{,.img} to /boot/initramfs{,.img} to solve the problem. */
	      if (LOADER.curfileload >= &LOADER.fileload[sizeof(LOADER.fileload)/sizeof(LOADER.fileload[0])]) {
		  ZDBG (("FAILED Loading/decompress too many concated initrd\r\n"));
		  returned = 0xFFFFFFFE;
		  puts (ERROR_TOO_MANY_INITRD);
		  break;
		  }
	      LOADER.accept_uncompressed_filesize = 0;
	      LOADER.curfileload->BOOTWAY_desc_index = LOADER.fileload[(LOADER.curfileload-LOADER.fileload)-1].BOOTWAY_desc_index;
	      }
	    /* Minimum 4 Kb for initial uncompressed concated initrd: */
	    else if (LOADER.curfileload == &LOADER.fileload[1]
		&& BOOTWAY.desc[system_index].filesize > LOADER.gzip_byte_treated + 4096) {
	      LOADER.accept_uncompressed_filesize = BOOTWAY.desc[system_index].filesize - LOADER.gzip_byte_treated;
	      LOADER.curfileload->BOOTWAY_desc_index = LOADER.fileload[(LOADER.curfileload-LOADER.fileload)-1].BOOTWAY_desc_index;
	      }
	    else {
	      LOADER.accept_uncompressed_filesize = 0;
#if USER_SUPPORT != 0
	      if ((LOADER.post_comment_row & LOADER.post_comment_col) != 0xFF) {
		  UI.function.setcursor (LOADER.post_comment_row, LOADER.post_comment_col);
		  LOADER.post_comment_row = LOADER.post_comment_col = 0xFF;
		  }
#endif
	      break;
	      }

	  if ((LOADER.accept_uncompressed_filesize & 0x80000000) == 0)
	      ZDBG (("Will accept possible uncompressed initrd of size %u\r\n", LOADER.accept_uncompressed_filesize));
	    else
	      ZDBG (("Will load initramfs of size %u without decompression\r\n", LOADER.accept_uncompressed_filesize & 0x7FFFFFFF));
	  LOADER.curfileload->max_size_or_crc32 = useablemem;
	  LOADER.initrd_index = 0;
	  }
	else {
	  ZDBG (("FAILED Loading error 0x%X\r\n", returned));
#if USER_SUPPORT != 0
	  if ((LOADER.post_comment_row & LOADER.post_comment_col) != 0xFF) {
	      UI.function.setcursor (LOADER.post_comment_row, LOADER.post_comment_col);
	      LOADER.post_comment_row = LOADER.post_comment_col = 0xFF;
	      }
#endif
	  printf (ERROR_LOADING_FILE, returned);
	  break;
	  }
      }

  /* switch OFF floppy drive, we no more deal with the floppy (but maybe DBG) */
  if (peekb (0x0040003F) & 0x03) { /* motor disk 0 or 1 active */
      unsigned nbticks = _BIOS_nbtick() + 1;
      pokeb (0x00400040, 1); /* MOTOR TURN-OFF TIMEOUT COUNT */
      while (nbticks <= _BIOS_nbtick())
	  continue; /* wait at least a tick */
      }

//  GZLIB_TEST_MEASURE_do();
  ZDBG (("minor_major_root: 0x%X, reloc adr 0x%X, run adr 0x%X.\r\n",
	 LOADER.minor_major_root,
	 LOADER.comment.elem.late_relocation_address,
	 LOADER.comment.elem.runadr));

  if (returned != 0) {
      ZDBG ((" [Failure, returned = 0x%X] ", returned));
      /* error message already displayed - I hope so */
      }
    else if ((LOADER.comment.elem.late_relocation_address & ~0xC0000000)
		+ LOADER.fileload[0].uncompressed_size > 1024 * totalmem) {
      ZDBG (("Will not be able to relocate %u Kbytes at address 0x%X "
		"because only %u Kbytes memory available!\r\n",
		LOADER.fileload[0].uncompressed_size / 1024,
		LOADER.comment.elem.late_relocation_address,
		totalmem));
      puts (ERROR_RELOCATE_TOO_HIGH);
      returned = 0x00EEEEEE;
      }
    else
      returned = vmlinuz_videomode_getparams_run (totalmem - 1024, regs);

  /* If we come back here, there is an error... */

  if (HIMEM_ACTIVE()) { /* implies (SETUP & BIOS_ONLY) == 0 */
      LOADER.curfileload = &LOADER.fileload[lastof (LOADER.fileload)];

      ZDBG ((" [unlock & free XMS]\r\n"));
      while (LOADER.curfileload >= LOADER.fileload) {
	  unsigned char error;

	  if (LOADER.curfileload->handle != 0) {
	      ZDBG (("[handle %u] ", LOADER.curfileload->handle));
	      if (   LOADER.curfileload->load_address != 0
		  && _XMS_unlock(UTIL.HIMEM_entrypoint,
			LOADER.curfileload->handle, &error))
		  ZDBG (("[FAILED unlock handle 0x%X: 0x%X] ",
			LOADER.curfileload->handle, error));
	      if (_XMS_free_memory(UTIL.HIMEM_entrypoint,
			LOADER.curfileload->handle, &error))
		  ZDBG (("[FAILED free handle 0x%X: 0x%X] ",
			LOADER.curfileload->handle, error));
	      }
	  LOADER.curfileload --;
	  }
      ZDBG (("[unlock & free XMS done]\r\n\r\n"));
      }
  return returned;
  }}

/**
 ** Main variable declaration, init to BSS:
 **/
struct LOADER_str LOADER;
