/* vgabios.h */
#ifndef VGABIOS_H
#define VGABIOS_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

typedef struct {
    unsigned char red, green, blue;
    } __attribute__ ((packed)) VGA_RGB;

typedef struct {
    unsigned char mode_supported[7]; /* bit 0 offset 0 : mode 0; up to 55 */
    unsigned char scanline; /* bit 0: 200, bit 1: 350, bit 2: 400 */
    unsigned char nbcharblock;
    unsigned char nbactivecharblock;
    struct {
	unsigned short all_mode_all_display :1;
	unsigned short gray_summing :1;
	unsigned short charfont_loading :1;
	unsigned short default_palette_loading_disable :1;
	unsigned short cursor_emulation :1;
	unsigned short EGA_palette :1;
	unsigned short color_palette :1;
	unsigned short color_register_paging :1;
	unsigned short light_pen :1;
	unsigned short save_restore_state :1;
	unsigned short intensity_blinking :1;
	unsigned short DCC :1;
	unsigned short unused :4;
	} __attribute__ ((packed)) miscellaneous_function;
    unsigned short reserved;
    struct {
	unsigned char char512set :1;
	unsigned char dynamic_save_area :1;
	unsigned char alpha_font_override :1;
	unsigned char graphic_font_override :1;
	unsigned char palette_override :1;
	unsigned char DCC_extensions :1;
	unsigned char unused :2;
	} __attribute__ ((packed)) saveptrfunctionflag;
    unsigned char reserved1;
    } __attribute__ ((packed)) static_functionality_table_t;

typedef struct {
    farptr static_functionality_table;
    unsigned char current_mode;
    unsigned short nbcolumn;
    unsigned short regenbufferlen;	/* in bytes */
    unsigned short regenbufferstart;
    unsigned short cursorpos_page[8];	/* page 0..7 */
    unsigned short cursortype;
    unsigned char currentpage;
    unsigned short CRTCport;
    unsigned char value_0x3X8;
    unsigned char value_0x3X9;
    unsigned char nbrow;
    unsigned short nbbyteperchar;
    unsigned char activeDCC;
    unsigned char alternateDCC;
    unsigned short nbcolor;	/* 0 for mono, reported bug Trident */
    unsigned char nbpage;	/* reported bug Trident */
    unsigned char nbscanline; /* 0:200,1:350,2:400,3:480,4:512,5:600,6:728 */
    unsigned char primarycharblock;
    unsigned char secondarycharblock;
    struct {
	unsigned char BIOSinfo_valid : 1;     /* ??? */
	unsigned char gray_summing_on : 1;
	unsigned char mono_display_attached : 1;
	unsigned char default_palette_loading_disabled : 1;
	unsigned char cursor_emulation_enabled : 1;
	unsigned char blinking_or_intensity : 1; /* blinking if 1 */
	unsigned char flat_pannel_active : 1;
	unsigned char unused : 1;
	} __attribute__ ((packed)) miscellaneous;
    struct {
	unsigned char   BIOSinfo_valid : 1;	/* ??? */
	unsigned char   adapter_required : 1;
	unsigned char   VGAgraphic16bit : 1;
	unsigned char   MFIenabled : 1;	/* else VGA, see INT10/0x12,0x37 */
	unsigned char	col132support : 1;
	unsigned char	reserved : 3;
	} __attribute__ ((packed)) nonVGAmodes;
    unsigned short reserved;
    unsigned char memory;		/* in 64 K blocks */
    struct {
	unsigned char char512set_active : 1;
	unsigned char dynamic_save_area : 1;
	unsigned char alpha_font_override_active : 1;
	unsigned char graphic_font_override_active : 1;
	unsigned char palette_override_active : 1;
	unsigned char DCC_override_active : 1;
	unsigned char unused : 2;
	} __attribute__ ((packed)) saveinfo;
    struct {
	unsigned char flat_panel_display_attached : 1;
	unsigned char flat_panel_display_active : 1;
	unsigned char color_display : 1;
	unsigned char reserved : 4;
	unsigned char simultaneous_use_640x480_flat_pannel : 1;
	} __attribute__ ((packed)) display;
    unsigned char reserved1[12];
    } __attribute__ ((packed)) functionality_table_t;

/*
 * an unexistant variable to synchronise reads/writes
 * on video memory - and more generally the state of the VGA card.
 * Let's explain a bit more:
 * - an asm() is volatile if, even if the result of the
 *  function is not used, the asm should be inserted.
 *  For instance a set_video_mode() has to be inserted even
 *  if we do not check its result success/failure.
 * - an asm() is not volatile and not synchronised to any
 *  barrier if its result will not change whatever you do
 *  with the video card; if you do not use its result, the
 *  asm() shall not be inserted. For instance, a function
 *  which ask a pointer to the name of the video card to maybe
 *  display it: if it is not displayed, do not even bother
 *  asking for the pointer.
 * - There should be a lot of these barrier variable, each
 *  representing one of the state: the cursor line and row,
 *  the active page, the video mode... It would be good
 *  but the number of argument of asm() is limited, and it is
 *  a bit too complex: the function printing a char on the
 *  video screen depend on the cursor position (any call
 *  which changes the cursor position has to be inserted
 *  before), the active video page, the current writing color,
 *  it modify the cursor position and the video memory. Moreover,
 *  it cannot be inverted with a function changing the video mode.
 *  Some set/get are easier: reading and writing the
 *  refresh rate can be completely independant, same
 *  for setting the base address of video RAM.
 *  To simplify we reduce everything to synchronisation to one
 *  dummy variable, referenced as "=m" to never be referenced,
 *  because its %[0-9] shall never exist in the string.
 *  "=X" is not usefull here, you would get some references in
 *  the assembly produced.
 *
 * Note that independently of that "volatile" feature, if the
 * asm() takes as input a structure, you usually need to give
 * the structure address as parameter - but also need to say
 * that the content of the pointer as to be up-to-date, usually
 * I do that by adding "X" (*my_struct_ptr) as input.
 * If it modify the content of a structure, you usually need
 * to give the structure address as an input like "g" (my_struct_ptr)
 * and also the content as an output, I usually do that like
 * "=X" (*my_struct_ptr). Writing "+g" (my_struct_ptr) will
 * just say that the pointer will be modified, it is another thing.
 *
 * Unfortunately GCC refuse something like "+b" (*my_struct_ptr)
 * because it thinks wrongly that we are trying to put the
 * entire structure in the "b" register, it will accept "+g" .
 *
 * For arrays, it is not so simple, but because the optimiser
 * is thinking with types only (strict aliasing), referencing
 * one of the array content should be sufficient.
 */
extern struct { unsigned dummy : 8; } VGA_barrier;

extern inline void
_VGA_setmode (char mode)
  {
  unsigned short dummy;
  asm volatile (" int $0x10 # setmode "
	: "=a" (dummy), "=m" (VGA_barrier)
	: "a" ((unsigned short)(unsigned char)mode) /* %ah = 0 */
		/* if %%al bit 7 is set (negative): do not clear video */
	);
  }

/*
 * To be used when the video mode is maybe invalid:
 */
#define SAFE_PREFIX_KEEP_ES() \
"	pushl	%%ds	\n"	\
"	pushl	%%es	\n"	\
"	pushl	%%gs	\n"	\
"	pushal		\n"	\
"	pushw	$0xA000	\n"	\
"	popw	%%gs	\n"	\
"	pushw	%%gs	\n"	\
"	popw	%%fs	\n"	\
"	pushw	%%gs	\n"	\
"	popw	%%ds	\n"	\

#define SAFE_PREFIX() \
	SAFE_PREFIX_KEEP_ES()	\
"	pushw	%%ds	\n"	\
"	popw	%%es	\n"	\

#define SAFE_POSTFIX() \
"						\n"	\
"	popal	# take care of popa bug on i386	\n"	\
"	popl	%%gs				\n"	\
"	popl	%%es				\n"	\
"	popl	%%ds				\n"	\

#define SAFE_POSTFIX_KEEP_AX() \
"						\n"	\
"	movw	%%ax,%%fs			\n"	\
"	popal	# take care of popa bug on i386	\n"	\
"	movw	%%fs,%%ax			\n"	\
"	popl	%%gs				\n"	\
"	popl	%%es				\n"	\
"	popl	%%ds				\n"	\

extern inline void
_VGA_safe_setmode (unsigned char mode)
  {
  asm volatile (
	SAFE_PREFIX()
"	xorl %%ebx,%%ebx		\n"
"	xorl %%ecx,%%ecx		\n"
"	xorl %%edx,%%edx		\n"
"	xorl %%esi,%%esi		\n"
"	xorl %%edi,%%edi		\n"
"	xorl %%ebp,%%ebp		\n"
"	int	$0x10	# safe setmode	\n"
	SAFE_POSTFIX()
	: "=m" (VGA_barrier)
	: "a" ((unsigned short)mode) /* %ah = 0 */
	);
  }

extern inline unsigned char
_VGA_getmode (unsigned char *nbcol, unsigned char *page)
  {
  unsigned short nbcol_mode;
  unsigned short tmppage;

  asm (" int $0x10 # getmode "
	: "=b" (tmppage), "=a" (nbcol_mode)
	: "a" ((unsigned short)0x0F00), "m" (VGA_barrier)
	);
  *nbcol = nbcol_mode >> 8;
  *page = tmppage >> 8;

  return (unsigned char) nbcol_mode;	/* bit 7 set if setmode had it */
  }

/*
 * To be used when the video mode is maybe invalid:
 */
extern inline unsigned char
_VGA_safe_getmode (unsigned char *nbcol)
  {
  unsigned short nbcol_mode;

  asm (
	SAFE_PREFIX()
"	xorl %%ebx,%%ebx		\n"
"	xorl %%ecx,%%ecx		\n"
"	xorl %%edx,%%edx		\n"
"	xorl %%esi,%%esi		\n"
"	xorl %%edi,%%edi		\n"
"	xorl %%ebp,%%ebp		\n"
"	int	$0x10	# safe getmode	\n"
	SAFE_POSTFIX_KEEP_AX()
	: "=a" (nbcol_mode)
	: "a" ((unsigned short)0x0F00), "m" (VGA_barrier)
	);
  *nbcol = nbcol_mode >> 8;

  return (unsigned char) nbcol_mode;	/* bit 7 set if setmode had it */
  }

extern inline void
_VGA_scroll (unsigned char ul_row, unsigned char ul_col,
	     unsigned char lr_row, unsigned char lr_col,
	     unsigned char color, char nbline)
  {
  asm volatile (
"	pushl	%%ebp						\n"
"	pushl	%%ds	# Trident in SVGA mode reset it to 0	\n"
"	int	$0x10	# scroll				\n"
"	popl	%%ds						\n"
"	popl	%%ebp						\n"
	: "=m" (VGA_barrier)
	: "a" (0x0600 | ((short)nbline & 0x1FF)), /* sign extended! */
	  "b" ((unsigned short)color << 8),
	  "c" (((unsigned short)ul_row << 8) | ul_col),
	  "d" (((unsigned short)lr_row << 8) | lr_col)
//	: "ebp"	// IBM PC bug
	);
  }

extern inline void
_VGA_getcursor (unsigned char page, unsigned char *row, unsigned char *col,
		unsigned char *cursorstart, unsigned char *cursorstop)
  {
  unsigned short row_col, cstart_cstop;
  unsigned short destroyed_Phoenix_BIOS;

  asm (" int $0x10 # getcursor "
	: "=d" (row_col), "=c" (cstart_cstop), "=a" (destroyed_Phoenix_BIOS)
	: "a" ((unsigned short)0x0300), "b" ((unsigned short)page << 8),
	  "m" (VGA_barrier)
	);
  *col = (unsigned char)row_col;
  *row = row_col >> 8;
  *cursorstart = cstart_cstop >> 8;	/* bug reported with mono displays */
  *cursorstop = (unsigned char)cstart_cstop;
  }

extern inline void
_VGA_setcursor (unsigned char page, unsigned char row, unsigned char col)
  {
  asm volatile (" int $0x10 # setcursor "	/* origin at 0 */
	: "=m" (VGA_barrier)
	: "d" (((unsigned short)row << 8) | col),
	  "a" ((unsigned short)0x0200),
	  "b" ((unsigned short)page << 8)
	);
  }

extern inline void
_VGA_setpage (unsigned char page)
  {
  asm volatile (" int $0x10 # setpage " /* first page: 0 */
	: "=m" (VGA_barrier)
	: "a" ((unsigned short)0x0500 | page)
	);
  }

/*
 * To be used when the video mode is maybe invalid:
 */
extern inline void
_VGA_safe_setcursor (unsigned char page, unsigned char row, unsigned char col)
  {
  asm volatile (
	SAFE_PREFIX()
"	xorl %%ecx,%%ecx			\n"
"	xorl %%esi,%%esi			\n"
"	xorl %%edi,%%edi			\n"
"	xorl %%ebp,%%ebp			\n"
"	int	$0x10	# safe setcursor	\n"
	SAFE_POSTFIX()
	: "=m" (VGA_barrier)
	: "d" (((unsigned short)row << 8) | col),
	  "a" ((unsigned short)0x0200),
	  "b" ((unsigned short)page << 8)
	);
  }

extern inline void
_VGA_readchar (unsigned char *character, unsigned char page, unsigned char *attr)
  {
  unsigned short tmp;

  asm (
"	pushl	%%ebp			\n"
"	int	$0x10	# readchar	\n"
"	popl	%%ebp			\n"
	: "=a" (tmp)
	: "a" ((unsigned short)0x0800),
	  "b" ((unsigned short)page << 8),
	  "m" (VGA_barrier)
//	: "ebp"	// IBM PC
	);
  *character = (char) tmp;
  *attr = tmp >> 8;
  }

/*
 * To be used when the video mode is maybe invalid:
 */
extern inline void
_VGA_safe_readchar (unsigned char *character, unsigned char page, unsigned char *attr)
  {
  unsigned short tmp;

  asm (
	SAFE_PREFIX()
"	xorl %%ecx,%%ecx		\n"
"	xorl %%edx,%%edx		\n"
"	xorl %%esi,%%esi		\n"
"	xorl %%edi,%%edi		\n"
"	xorl %%ebp,%%ebp		\n"
"	int	$0x10	# safe readchar	\n"
	SAFE_POSTFIX_KEEP_AX()
	: "=a" (tmp)
	: "a" ((unsigned short)0x0800),
	  "b" ((unsigned short)page << 8),
	  "m" (VGA_barrier)
	);
  *character = (char) tmp;
  *attr = tmp >> 8;
  }

extern inline void
_VGA_writechar_attr (char character, unsigned char page,
			  unsigned char attr, unsigned short nbtimes)
  {
  /* 256 color modes: page = background color */
  /* attr = color in graphic modes, if bit 7 set and nbcolor < 256, XORed */
  asm volatile (" int $0x10 # writechar+attr "
	: ASM_STRUCT_INOUT (VGA_barrier)
	: "a" ((unsigned short)0x0900 | (character & 0xFF)),
	  "c" (nbtimes),
	  "b" (((unsigned short)page << 8) | attr)
	);
  }

/*
 * To be used when the video mode is maybe invalid:
 */
extern inline void
_VGA_safe_writechar_attr (unsigned char character, unsigned char page,
			  unsigned char attr, unsigned short nbtimes)
  {
  asm volatile (
	SAFE_PREFIX()
"	xorl %%edx,%%edx			\n"
"	xorl %%esi,%%esi			\n"
"	xorl %%edi,%%edi			\n"
"	xorl %%ebp,%%ebp			\n"
"	int	$0x10	# safe writechar+attr	\n"
	SAFE_POSTFIX()
	: ASM_STRUCT_INOUT (VGA_barrier)
	: "a" ((unsigned short)0x0900 | (character & 0xFF)),
	  "c" (nbtimes),
	  "b" (((unsigned short)page << 8) | attr)
	);
  }

extern inline void
_VGA_writechar_color (char character, unsigned char page, unsigned char attr,
			unsigned short nbtimes)
  {
  /* 256 color modes: page = background color */
  /* This doesn't write attribute/colors in text modes */
  asm volatile ( " int $0x10 # writechar+color "
	: ASM_STRUCT_INOUT (VGA_barrier)
	: "a" ((unsigned short)0x0A00 | (character & 0xFF)),
	  "c" (nbtimes),
	  "b" (((unsigned short)page << 8) | attr)
	);
  }

extern inline void
_VGA_setbackground (unsigned char color)
  {
  asm volatile ( " int $0x10 # setbackground "
	: "=m" (VGA_barrier)
	: "a" ((unsigned short)0x0B00),
	  "b" ((unsigned short) color) /* %bh == 0 */
	);
  }

extern inline void
_VGA_writestring (const char *string, unsigned short stringsize,
		  unsigned char page, unsigned char attr,
		  unsigned char update_cursor, /* = 0 or 1 if yes */
		  unsigned char row, unsigned char col)
  {
  /* CR, LF, BS, BEL recognised, update cursor after */
  /* A scroll is generated when writing to the lowest left
     char */
  /* Using 256 colors coded in "page", with or without
     update_cursor, does not display more than 1 or 2 chars
     on the screen - at least on the current page 0 -, YMMV */
  asm volatile (
//	" int	$0x10 # writestring	\n"
"	xchgl	%%ebp,%5		\n"
"	int	$0x10 # writestring	\n"
"	xchgl	%%ebp,%5		\n"
	: "=m" (VGA_barrier)
	: "c" (stringsize),
	  "b" (((unsigned short)page << 8) | attr),
	  "d" (((unsigned short)row << 8) | col),
	  "a" ((unsigned short)0x1300 | !!update_cursor),
	  "r" (string)	/* in fact %es:%bp */
//	  "B" (string)	/* in fact %es:%bp */ // DOES NOT WORK
//	: "ebp" // DOES NOT WORK
	);
  }

extern inline void
_VGA_setpixel (unsigned short x, unsigned short y,
	       unsigned char color, unsigned char page)
  {
  unsigned short tmp = 0x0C00 | color;	/* Cirrus chipset changes ax */

  /* if color bit7 = 1 and mode nbcolor < 256, XOR color and pixel */
  asm volatile ( " int	$0x10 # setpixel "
	: "+a" (tmp), "=m" (VGA_barrier)
	: "c" (x), "d" (y),
	  "b" ((unsigned short)page << 8)
	: "si", "di" /* reported on some PS2 VGA systems */
	);
  }

/*
 * To be used when the video mode is maybe invalid:
 */
extern inline void
_VGA_safe_setpixel (unsigned short x, unsigned short y,
	    unsigned char color, unsigned char page)
  {
  asm volatile (
	SAFE_PREFIX()
"	xorl %%esi,%%esi			\n"
"	xorl %%edi,%%edi			\n"
"	xorl %%ebp,%%ebp			\n"
"	int	$0x10	# safe setpixel		\n"
	SAFE_POSTFIX()
	: "=m" (VGA_barrier)
	: "a" ((unsigned short)0x0C00 | color),
	  "c" (x),
	  "d" (y),
	  "b" ((unsigned short)page << 8)
	);
  }

extern inline unsigned	char
_VGA_getpixel (unsigned short x, unsigned short y, unsigned char page)
  {
  unsigned char bytecolor;

  asm ( " int	$0x10 # getpixel "
	: "=a" (bytecolor)
	: "a" ((unsigned short)0x0D00),
	  "c" (x),
	  "d" (y),
	  "b" ((unsigned short)page << 8),
	  "m" (VGA_barrier)
	);

  return bytecolor;
  }

/*
 * To be used when the video mode is maybe invalid:
 */
extern inline unsigned char
_VGA_safe_getpixel (unsigned short x, unsigned short y, unsigned char page)
  {
  unsigned char bytecolor;

  asm (
	SAFE_PREFIX()
"	xorl %%esi,%%esi			\n"
"	xorl %%edi,%%edi			\n"
"	xorl %%ebp,%%ebp			\n"
"	int	$0x10	# safe getpixel		\n"
	SAFE_POSTFIX_KEEP_AX()
	: "=a" (bytecolor)
	: "a" ((unsigned short)0x0D00),
	  "c" (x),
	  "d" (y),
	  "b" ((unsigned short)page << 8),
	  "m" (VGA_barrier)
	);

  return bytecolor;
  }

extern inline void
_VGA_blinking (unsigned char on_off) /* on if 1 */
  {
  asm volatile ( " int $0x10 # blinking "
	: "=m" (VGA_barrier)
	: "a" ((unsigned short)0x1003),
	  "b" ((unsigned short) on_off) /* %bh == 0 */
	);
  }

extern inline void
_VGA_setPELmask (unsigned char PELmask)
  {
  asm volatile ( " int $0x10 # setPELmask "
	: "=m" (VGA_barrier)
	: "a" ((unsigned short)0x1018),
	  "b" (PELmask)
	);
  }

extern inline unsigned char
_VGA_getPELmask (void)
  {
  unsigned char returned;

  asm (" int $0x10 # getPELmask "
	: "=b" (returned)
	: "a" ((unsigned short)0x1019),
	  "m" (VGA_barrier)
	);
  return returned;
  }

extern inline unsigned short VGA_get_dac_color_page (void)
  {
  unsigned short returned;

  asm (" int $0x10 # get_dac_color_page "
	: "=b" (returned)
	: "a" ((unsigned short)0x101A),
	  "m" (VGA_barrier)
	);
  /* bl = 0 => four pages of 64; bl = 1 => sixteen pages of 16
     bh = current page */
  return returned;
  }

#define VGA_get_save_size_video_hardware 0x01
#define VGA_get_save_size_BIOS_data_area 0x02
#define VGA_get_save_size_color_and_DAC  0x04

extern inline unsigned short VGA_get_save_size (unsigned short mask)
  {
  unsigned short returned;
  unsigned char error;

  asm (" int $0x10 # get_save_size "
	: "=b" (returned), "=a" (error)
	: "a" ((unsigned short)0x1C00),
	  "c" (mask)
	);
  if (error != 0x1C)
      return 0;
  return returned;
  }

extern inline void
_VGA_setpalette (unsigned char index, unsigned color)
  {
  asm volatile (" int $0x10 # setpalette "
	: "=m" (VGA_barrier)
	: "a" ((unsigned short)0x1000),
	  "b" (color << 8 | index)
	);
  }

extern inline unsigned char
_VGA_getpalette (unsigned char index)
  {
  unsigned short returned;
  asm (" int $0x10 # getpalette "
	: "=b" (returned)
	: "a" ((unsigned short)0x1007),
	  "b" (index),
	  "m" (VGA_barrier)
	);
  return returned >> 8;
  }

extern inline void
_VGA_getDAC (unsigned short index,
	     unsigned char *_red, unsigned char *_green, unsigned char *_blue)
  {
  unsigned short regdx, regcx;
  unsigned short destroyed_tseng_ET400_V8_00;

  asm (" int $0x10 # getDAC "
	: "=c" (regcx), "=d" (regdx), "=a" (destroyed_tseng_ET400_V8_00)
	: "a" ((unsigned short)0x1015),
	  "b" (index),
	  "m" (VGA_barrier)
	);
  *_red = regdx >> 8;
  *_blue = (unsigned char)regcx;
  *_green = regcx >> 8;
  }

extern inline void
_VGA_setDAC (unsigned short index,
	     unsigned char _red, unsigned char _green, unsigned char _blue)
  {
  asm volatile (" int $0x10 # setDAC "
	: "=m" (VGA_barrier)
	: "d" ((unsigned short)_red << 8),
	  "c" (((unsigned short)_green << 8) | _blue),
	  "a" ((unsigned short)0x1010), "b" (index)
	);
  }

extern inline void
_VGA_setDACarray (unsigned short startindex, unsigned short nb,
		  const VGA_RGB block[])
  {
  asm volatile (" int $0x10 # setDACarray "
	: "=m" (VGA_barrier)
	: "a" ((unsigned short)0x1012),
	  "b" (startindex),
	  "c" (nb),
	  "d" (block), /* in fact %es:%dx */
	  "m" (block[startindex])
	);
  }

extern inline void
_VGA_getDACarray (unsigned short startindex, unsigned short nb,
		  VGA_RGB block[])
  {
  asm (" int $0x10 # getDACarray "
	: "=m" (block[startindex])
	: "a" ((unsigned short)0x1017),
	  "b" (startindex),
	  "c" (nb),
	  "d" (block), /* in fact %es:%dx */
	  "m" (VGA_barrier)
	);
  }

extern inline farptr
_VGA_getfont (unsigned char font,
		unsigned char *nbrow,
		unsigned short *byteperchar)
  {
  farptr returned;
  /* %%bh: 0: INT 0x1F pointer; 1: INT 0x43 pointer
	   2: ROM 8x14
	   3: ROM 8x8 low128, 4: ROM 8x8 high128
	   5: ROM alpha alternate 9x14 (EGA/VGA), char to be replaced.
	   6: ROM 8x16 (MCGA/VGA)
	   7: ROM alternate 9x16 (VGA), char to be replaced.
   To save memory, newer VIDEO BIOS do no more have the 8x14 font,
   or even some have only _a part_ of it (up to char 'z'!)... Even more,
   some BIOS modes which were using this font may not be disabled!
   */

  asm (
"	pushl	%%ebp				\n"
"	pushl	%%es				\n"
"	xorw	%%bp,%%bp			\n"
"	xorw	%%dx,%%dx			\n"
"	movw	%%bp,%%es			\n"
"	int	$0x10		# getfont	\n"
"	pushw	%%es				\n"
"	pushw	%%bp				\n"
"	popl	%%eax				\n"
"	popl	%%es				\n"
"	popl	%%ebp				\n"
	: "=a" (returned),
	  "=d" (*nbrow), /* minus one, but for some EGA */
	  "=c" (*byteperchar) /* of the on screen font, not requested font */
	: "a" ((unsigned short)0x1130),
	  "b" ((unsigned short)font << 8)
//	: "ebp"
	);

  if ((returned & 0xFFFF) == 0) /* to get at least a font... */
      returned = peekl (43*4);  /* Vector 43, but font of which size ? */
  return returned;
  }

/* _VGAtxt_load_font() should work on EGA also, it is to be used
   when there is no need to recalculate nb of lines, buffer len...
   i.e. when the font _size_ if not changed. */
extern inline void
_VGAtxt_load_font (unsigned char byteperchar, unsigned char block,
		unsigned short count, unsigned short first,
		farptr font)
  {
  asm volatile (
"	pushl	%%es					\n"
"	pushl	%%ebp					\n"
"	pushl	%%esi					\n"
"	popw	%%bp					\n"
"	popw	%%es					\n"
"	int	$0x10		# _VGAtxt_load_font	\n"
"	popl	%%ebp					\n"
"	popl	%%es					\n"
	: /* no output */
	: "a" ((unsigned short)0x1100),
	  "b" (((unsigned short)byteperchar) << 8 | block),
	  "c" (count), "d" (first),
	  "S" (font) /* in fact %es:%bp */
	);
  }

extern inline void
_VGAtxt_load_ROM8x14 (unsigned char block)
  {
  asm volatile ( " int $0x10 # _VGAtxt_load_ROM8x14 "
	: /* no output */
	: "a" ((unsigned short)0x1101),
	  "b" (block)
	);
  }

extern inline void
_VGAtxt_load_ROM8x8 (unsigned char block)
  {
  asm volatile ( " int $0x10 # _VGAtxt_load_ROM8x8 "
	: /* no output */
	: "a" ((unsigned short)0x1102),
	  "b" (block)
	);
  }

extern inline void
_VGAtxt_set_block_specifier (unsigned char block_bit3_cleared,
			     unsigned char block_bit3_set)
  {
  struct {
      unsigned char	lsb_b3clear : 2;
      unsigned char	lsb_b3set   : 2;
      unsigned char	msb_b3clear : 1;	/* VGA only */
      unsigned char	msb_b3set   : 1;	/* VGA only */
      } __attribute__ ((packed)) block_specifier = {
      lsb_b3clear : block_bit3_cleared & 0x3,
      lsb_b3set   : block_bit3_set & 0x03,
      msb_b3clear : block_bit3_cleared >> 2,
      msb_b3set   : block_bit3_set >> 2,
      };
  asm volatile ( " int $0x10 # _VGAtxt_set_block_specifier "
	: "=m" (VGA_barrier)
	: "a" ((unsigned short)0x1103),
	  "b" (block_specifier)
	);
  }

extern inline void
_VGAtxt_load_ROM8x16 (unsigned char block)
  {
  asm volatile ( " int $0x10 # _VGAtxt_load_ROM8x16 "
	: /* no output */
	: "a" ((unsigned short)0x1104),
	  "b" (block)
	);
  }

extern inline void
_VGA_set_user_8x8_high_graphic_char (farptr ptr)
  {
  asm volatile (
"	pushl	%%es					\n"
"	pushl	%%ebp					\n"
"	pushl	%%edx					\n"
"	popw	%%bp					\n"
"	popw	%%es					\n"
"	int	$0x10 # _VGA_set_user_8x8_high_graphic_char	\n"
"	popl	%%ebp					\n"
"	popl	%%es					\n"
	: /* no output */
	: "a" ((unsigned short)0x1120), "d" (ptr)
	);
  }

extern inline void
_VGA_set_user_graphic_char (farptr ptr,
			    unsigned short byteperchar,
			    unsigned char nbrow)
  {
  asm volatile (
"	pushl   %%es							\n"
"	pushl   %%ebp							\n"
"	pushl	%%ebx							\n"
"	popw	%%bp							\n"
"	popw	%%es							\n"
"	mov	$3,%%bl							\n"
"	cmp	$43,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
"	cmp	$25,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
"	cmp	$14,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
" 1:									\n"
"	int     $0x10		# _VGA_set_user_graphic_char		\n"
"	popl    %%ebp							\n"
"	popl    %%es							\n"
	:
	: "a" (0x1121), "b" (ptr), "c" (byteperchar), "d" (nbrow)
	);
  }

extern inline void
_VGA_loadgraphic_ROM8x14 (unsigned char nbrow)
  {
   asm volatile (
"	mov	$3,%%bl							\n"
"	cmp	$43,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
"	cmp	$25,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
"	cmp	$14,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
" 1:									\n"
"	int     $0x10		# _VGA_loadgraphic_ROM8x14		\n"
	:
	: "a" (0x1122), "d" (nbrow)
	: "ebx"
	);
  }

extern inline void
_VGA_loadgraphic_ROM8x8 (unsigned char nbrow)
  {
   asm volatile (
"	mov	$3,%%bl							\n"
"	cmp	$43,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
"	cmp	$25,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
"	cmp	$14,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
" 1:									\n"
"	int     $0x10		# _VGA_loadgraphic_ROM8x8		\n"
	:
	: "a" (0x1123), "d" (nbrow)
	: "ebx"
	);
  }


extern inline void
_VGA_loadgraphic_ROM8x16 (unsigned char nbrow)
  {
  asm volatile (
"	mov	$3,%%bl							\n"
"	cmp	$43,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
"	cmp	$25,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
"	cmp	$14,%%dl						\n"
"	je	1f							\n"
"	dec	%%bl							\n"
" 1:									\n"
"	int     $0x10		# _VGA_loadgraphic_ROM8x16		\n"
	:
	: "a" (0x1124), "d" (nbrow)
	: "ebx"
	);
  }

extern inline unsigned
_VGA_get_functionality_info (functionality_table_t *modetable)
  {
  unsigned char result;

  if (   sizeof (functionality_table_t) != 64
      || sizeof (static_functionality_table_t) != 16)
      __ERROR();

  asm (
	SAFE_PREFIX_KEEP_ES()
"	xorl %%ecx,%%ecx				\n"
"	xorl %%edx,%%edx				\n"
"	xorl %%esi,%%esi				\n"
"	xorl %%ebp,%%ebp				\n"
"	int	$0x10	# safe get_functionality_info	\n"
	SAFE_POSTFIX_KEEP_AX()
	: "=a" (result), "=m" (*modetable)
	: "a" ((unsigned short)0x1B00),
	  "b" ((unsigned short)0),
	  "D" (modetable) /* in fact %es:%di */
	);

  return (result != 0x1B);
  }

extern inline unsigned
_VGA_isgraphic (void)
  {
  if (peekb (0x00400065) & 2)	/* BIOS data area, only for modes < 0x0C */
      return 1;
    else
      return 0;
  }

extern inline unsigned
_VGA_get_display_combination_code (unsigned char *active_dcc,
				   unsigned char *alternate_dcc)
  {
  unsigned char result;
  unsigned short alternate_active;

  asm (" int $0x10 # get_display_combination_code "
	: "=a" (result), "=b" (alternate_active)
	: "a" ((unsigned short)0x1A00),
	  "m" (VGA_barrier)
	);
  *active_dcc = (unsigned char) alternate_active;
  *alternate_dcc = alternate_active >> 8;

  return (result != 0x1A);
  }

extern inline unsigned
_VGA_set_display_combination_code (unsigned char active_dcc,
	   unsigned char alternate_dcc)
  {
  unsigned char result;

  asm volatile (" int $0x10 # set_display_combination_code "
	: "=a" (result), "=m" (VGA_barrier)
	: "a" ((unsigned short)0x1A01),
	  "b" ((((unsigned short)alternate_dcc) << 8) | active_dcc)
	);

  return (result != 0x1A);
  }

/* enable reset of palette at mode changes if paletteswitch = 1 */
extern inline int _VGA_autoload_default_palette (int paletteswitch)
  {
  unsigned char tmp;

  asm volatile (" int $0x10 # autoload_default_palette "
	: "=a" (tmp)
	: "a" ((unsigned short)0x1200 | !paletteswitch),
	  "b" ((unsigned short)0x31)
	);
  if (tmp == 0x12)
      return 1; /* returns true if OK */
    else
      return 0;
  }

extern inline int _VGA_enable_gray_scale_summing (int gray_switch)
  {
  unsigned char tmp;

  asm volatile (" int $0x10 # enable_gray_scale_summing "
	: "=a" (tmp)
	: "a" ((unsigned short)0x1200 | (!gray_switch) << 1),
	  "b" ((unsigned short)0x33)
	);
  if (tmp == 0x12)
      return 1; /* returns true if OK */
    else
      return 0;
  }

extern inline void _VGA_safe_autoload_default_palette (int paletteswitch)
  {
  asm volatile (
	SAFE_PREFIX()
"	xorl %%ecx,%%ecx				\n"
"	xorl %%edx,%%edx				\n"
"	xorl %%esi,%%esi				\n"
"	xorl %%edi,%%edi				\n"
"	xorl %%ebp,%%ebp				\n"
"	int	$0x10 # safe_autoload_default_palette	\n"
	SAFE_POSTFIX()
	: /* no output */
	: "a" ((unsigned short)0x1200 | !paletteswitch),
	  "b" (0x31)
	);
  }

extern inline unsigned short
_XGA_get_SVGA_DMQS_data_length (void)
  {
  unsigned char supported;
  unsigned short length;

  asm (" int $0x10 # _XGA_get_SVGA_DMQS_data_length "
	: "=a" (supported), "=b" (length)
	: "a" ((unsigned short)0x1F02)
	);
  if (supported == 0x1F)
      return length;
    else
      return 0;
  }

struct DMQS_data_str {
    unsigned short	DMQS_next_instance_offset; /* or 0 if no more card */
    unsigned char	reserved;
    unsigned char	SVGA_implementation_functional_level;
    unsigned char	SVGA_implementation_resolution_level;
    unsigned short	vendor_id;
    unsigned short	vendor_specific;
    unsigned char	reserved_array1[7];
    enum { window_64K, window_1M, window_2M, window_4M } winsize : 8;
    unsigned short	winadr_div0x10000;
    unsigned short	monitor_id;
    unsigned char	memory_div256K;
    unsigned char	reserved_array2[11];
    } __attribute__ ((packed));

extern inline unsigned
_XGA_get_SVGA_DMQS_data (struct DMQS_data_str *data)
  {
  unsigned char supported;

  asm (" int $0x10 # _XGA_get_SVGA_DMQS_data "
	: "=a" (supported), "=m" (*data)
	: "a" ((unsigned short)0x1F03),
	  "D" (data) /* in fact %es:%di */
	);
  if (supported == 0x1F)
      return 0;
    else
      return 1; /* failed */
  }



static inline farptr Everex_get_paging_function (void)
  {
  farptr farfct;
  unsigned char error;

  asm (
"	pushl	%%es		\n"
"	int	$0x10		\n"
"	pushw	%%es		\n"
"	pushw	%%di		\n"
"	popl	%%edi		\n"
"	popl	%%es		\n"
	: "=D" (farfct), "=a" (error)
	: "a" ((unsigned short)0x7000),
	  "b" ((unsigned short)4)
	: "cc"
	);
    if (error != 0x70)
	return 0;
    return farfct;
//   printf (" Everex_get_paging_function: 0x%X, len = %u, end = 0x%X\r\n",
//	     fctptr, peekw (fctptr-2), peekb (fctptr + peekw (fctptr-2) - 1));
//   param in dl, pages of 64K, ends with FAR return
    }

extern inline unsigned
_CIRRUS_read_monitor_ID (unsigned char *ID, unsigned char *type)
  {
  unsigned short result;

  asm (" int $0x10 # CIRRUS_read_monitor_ID "
	: "=b" (result)
	: "a" ((unsigned short)0x1200),
	  "b" ((unsigned short)0xA1)
	);
  *ID = result >> 8;
  *type = (unsigned char)result; /* 0: color, 1: grayscale, 2: no display */

  return ((unsigned char)result > 2); /* if true, fct unsupported */
  }

extern inline void
_COMPAQ_setDACbit (unsigned char bit8_bit6) /* 8 bit if non zero */
  {
  asm volatile (" int $0x10 # COMPAQ_setDACbit "
	: "=m" (VGA_barrier)
	: "a" ((unsigned short)0xBF00 | (bit8_bit6? 0x0D : 0x0C))
	);
  /* we could check CF = 0 */
  }

extern inline unsigned
_COMPAQ_getDACbit (unsigned char *bit8_bit6) /* 8 bit if non zero */
  {
  asm (" int $0x10 # COMPAQ_getDACbit "
	: "=a" (*bit8_bit6)
	: "a" ((unsigned short)0xBF0E),
	  "m" (VGA_barrier)
	);
  if (*bit8_bit6 > 1)
      return 1; /* not supported */
  return 0; /* we could check also CF = 0 */
  }

extern inline signed int
_COMPAQ_getvideoramaddr (void)
  {
  unsigned short addr_in_Mbytes; /* zero if disabled */
  asm (" int $0x10 # COMPAQ_getvideoramaddr "
	: "=a" (addr_in_Mbytes)
	: "a" ((unsigned short)0xBF10),
	  "m" (VGA_barrier)
	);
  if (addr_in_Mbytes == 0xBF10) /* we could check also CF = 0 */
      return -1;
    else
      return addr_in_Mbytes;
  }

extern inline signed int
_PARADISE_installation_check (void)
  {
  unsigned short result;
  unsigned rom_serial_number;

  asm (
"	int	$0x10 # _PARADISE_WD90C24_installation_check		\n"
"	roll	$16,%%eax						\n"
"	movzbw	%%dh,%%ax						\n"
"	rorl	$16,%%eax # 00:dh:ah:al last 3 digit of ROM serial number\n"
	: "=b" (result), "=a" (rom_serial_number)
	: "a" ((unsigned short)0x6E00)
	: "cx", "cc"
	);
  if (result == 0x5744)
      return rom_serial_number;	/* present */
    else
      return -1;
  }

extern inline unsigned
_PARADISE_WD90C24_installation_check (void)
  {
  unsigned short result;

  asm (" int $0x10 # _PARADISE_WD90C24_installation_check "
	: "=a" (result)
	: "a" ((unsigned short)0x7F << 8)
	);
  if (result == 0x1234)
      return 1; /* present */
    else
      return 0;
  }

/* The only function to get parameters of a video mode without
   switching to it first: */
struct ATI_mode_table {
    unsigned char	number_of_columns;
    unsigned char	maximum_row; /* number of row - 1 */
    unsigned char	scan_line_per_row;
    unsigned short	video_buffer_size; /* in bytes */
    struct {
	struct {
	    unsigned char asynchronous_reset_and_halt	: 1;
	    unsigned char synchronous_reset_and_halt	: 1;
	    unsigned char reserved			: 6;
	    } __attribute__ ((packed)) reset;
	struct {
	    unsigned char eight_dot_clock_per_char	: 1; /* instead of 9 */
	    unsigned char reserved			: 1;
	    unsigned char halve_clock_rate		: 1;
	    unsigned char divide_master_dot_clock_by_two : 1;
	    unsigned char quarter_clock_rate		: 1;
	    unsigned char display_off			: 1;
	    unsigned char unknown			: 2;
	    } __attribute__ ((packed)) clocking_mode;
	struct {
	    unsigned char enable_memory_plane_0	: 1;
	    unsigned char enable_memory_plane_1	: 1;
	    unsigned char enable_memory_plane_2	: 1;
	    unsigned char enable_memory_plane_3	: 1;
	    unsigned char unknown		: 4;
	    } __attribute__ ((packed)) map_mask;
	struct {
	    unsigned char charmap_select_B	: 2;
	    unsigned char charmap_select_A	: 2;
	    unsigned char charmap_select_high_B	: 1;
	    unsigned char charmap_select_high_A	: 1;
	    unsigned char unknown		: 2;
	    } __attribute__ ((packed)) character_map;
	} __attribute__ ((packed)) sequencer;
    struct {
	unsigned char	CRTC_port_address	: 1; /* 0=3B4h mono 1=3D4h color */
	unsigned char	enable_CPU_RAM_access	: 1;
	unsigned char	pixelclock		: 2;
	unsigned char	disable_video_driver	: 1;
	unsigned char	odd_even_pagebit	: 1;
	unsigned char	horizontal_sync_polarity: 1;
	unsigned char	vertical_sync_polarity	: 1;
	} __attribute__ ((packed)) misc_output;
    struct {
	unsigned char	horizontal_total_size_chars;
	unsigned char	horizontal_displayed_chars;
	unsigned char	horizontal_sync_position_chars;
	unsigned char	horizontal_sync_width_chars;
	unsigned char	vertical_total_size_char_rows;
	unsigned char	vertical_total_adjust_scan_lines;
	unsigned char	vertical_displayed_char_rows;
	struct {
	    unsigned char vertical_total_8		: 1;
	    unsigned char vertical_display_end_8	: 1;
	    unsigned char vertical_sync_start_8		: 1;
	    unsigned char vertical_blank_start_8	: 1;
	    unsigned char line_compare_8		: 1;
	    unsigned char vertical_total_9		: 1;
	    unsigned char vertical_display_end_9	: 1;
	    unsigned char vertical_sync_start_9		: 1;
	    } __attribute__ ((packed)) overflow;
	unsigned char	interlace_mode;
	struct {
	    unsigned char nb_scan_line_per_char_minus1	: 5;
	    unsigned char vertical_blank_start_9	: 1;
	    unsigned char line_compare_9		: 1;
	    unsigned char enable_double_scan		: 1;
	    } __attribute__ ((packed)) maximum_row_address;
	unsigned char	cursor_start_scan_line;
	unsigned char	cursor_end_scan_line;
	unsigned char	screen_memory_start_high;
	unsigned char	screen_memory_start_low;
	unsigned char	cursor_address_high;
	unsigned char	cursor_address_low;
	unsigned char	light_pen_high; /* or start_vertical_retrace */
	unsigned char	light_pen_low; /* or end_vertical_retrace */
	unsigned char	vertical_display_end;
	unsigned char	row_offset;
	unsigned char	underline_location;
	unsigned char	start_vertical_blanking_minus_1;
	unsigned char	end_vertical_blanking;
	struct {
	    unsigned char compatibility_6845	: 1;
	    unsigned char row_scan_counter	: 1;
	    unsigned char horizontal_retrace_clock	: 1;
	    unsigned char linear_address_counter	: 1;
	    unsigned char output_control	: 1;
	    unsigned char address_wrap		: 1;
	    unsigned char byte_mode		: 1;
	    unsigned char reset_and_stop	: 1;
	    } __attribute__ ((packed)) mode_control;
	unsigned char	line_compare;
	} __attribute__ ((packed)) crtc;
    struct {
	struct {
	    unsigned char primary_blue    : 1;
	    unsigned char primary_green   : 1;
	    unsigned char primary_red     : 1;
	    unsigned char secondary_blue  : 1; /* mono */
	    unsigned char secondary_green : 1; /* intensity */
	    unsigned char secondary_red   : 1;
	    unsigned char reserved	  : 2;
	    } __attribute__ ((packed)) palette[16];
	struct {
	    unsigned char graphic		: 1;
	    unsigned char monochrome		: 1;
	    unsigned char line_graphics_enable	: 1;
	    unsigned char background_intensity	: 1;
	    unsigned char reserved		: 1;
	    unsigned char enable_pixel_panning	: 1;
	    unsigned char PELCLK_div_2		: 1;
	    unsigned char SB_SG_select		: 1;
	    } __attribute__ ((packed)) mode;
	struct {
	    unsigned char primary_blue		: 1;
	    unsigned char primary_green		: 1;
	    unsigned char primary_red		: 1;
	    unsigned char intensity_border_color1 : 1;
	    unsigned char secondary_blue	: 1;
	    unsigned char secondary_green	: 1;
	    unsigned char secondary_red		: 1;
	    unsigned char intensity_border_color2 : 1;
	    } __attribute__ ((packed)) vga_overscan;
	struct {
	    unsigned char enable_plane_0	: 1;
	    unsigned char enable_plane_1	: 1;
	    unsigned char enable_plane_2	: 1;
	    unsigned char enable_plane_3	: 1;
	    unsigned char video_status_select	: 2;
	    unsigned char reserved		: 2;
	    } __attribute__ ((packed)) color_enable;
	struct {
	    unsigned char value			: 4;
	    unsigned char reserved		: 4;
	    } __attribute__ ((packed)) horizontal_pixel_panning;
	} __attribute__ ((packed)) attribute_controller;
    struct {
	unsigned char	set_reset;
	unsigned char	enable_set_reset;
	unsigned char	color_compare;
	unsigned char	data_rotate;
	unsigned char	read_map_select;
	struct {
	    unsigned char write_mode		: 2;
	    unsigned char test_condition	: 1;
	    unsigned char read_mode		: 1;
	    unsigned char enable_odd_even_addr_mode : 1;
	    unsigned char shift_register_mode	: 1;
	    unsigned char enable_256_color	: 1;
	    unsigned char reserved		: 1;
	    } __attribute__ ((packed)) mode;
	struct {
	    unsigned char graphic		: 1;
	    unsigned char subst_address_bit0	: 1; /* 0:chain odd maps to even */
	    enum {
		map_A0000_BFFFF, map_A0000_AFFFF,
		map_B0000_B7FFF, map_B8000_BFFFF
		} memory_map			: 2;
	    unsigned char reserved		: 4;
	    } __attribute__ ((packed)) miscellaneous;
	unsigned char	color_dont_care;
	unsigned char	bit_mask;
	} __attribute__ ((packed)) graphic_controller;
    } __attribute__ ((packed));

extern inline farptr
_ATI_alt_funct_select_get_mode_table (unsigned char mode)
  {
  farptr ati_mode_table;

  asm (
"	pushl	%%es		\n"
"	pushl	%%ebp		\n"
"	orw	$0xFFFF, %%bp	\n"
"	int	$0x10	# _ATI_alt_funct_select_get_mode_table	\n"
"	pushw	%%es		\n"
"	pushw	%%bp		\n"
"	popl	%%eax		\n"
"	popl	%%ebp		\n"
"	popl	%%es		\n"
	: "=a" (ati_mode_table)
	: "a" ((unsigned short)0x1200 | mode),
	  "b" ((unsigned short)0x5506)
	: "esi" /* modified by ATI Mach32 */
	);
  return ati_mode_table; /* 0x????FFFF on error */
  }

extern inline unsigned
_CT_set_refresh_rate (unsigned char mode, unsigned char maxHz)
  {
  unsigned short error, index;

  if (maxHz < 56)
      index = 0;
    else if (maxHz < 60)
      index = 1;
    else if (maxHz < 70)
      index = 2;
    else if (maxHz < 72)
      index = 3;
    else if (maxHz < 75)
      index = 4;
    else
      index = 5;

  asm volatile (" int $0x10 # _CT_set_refresh_rate "
	: "=a" (error), "=m" (VGA_barrier)
	: "a" ((unsigned short)0x5F05),
	  "b" (((unsigned short)mode << 8) | (1 << index))
	    /* bx bit 7: 0: program, 1: check */
	);
  if ((error & 0xFF) != 0x5F || (error >> 8) != 1)
      return 1; /* error */
  return 0;
  }

/* GCC-3.0.3 no more remove those translating array
   when the function is not called... oh dear... */
extern inline unsigned
_CT_get_refresh_rate (unsigned char mode)
  {
  unsigned short error, result;

  asm (" int $0x10 # _CT_get_refresh_rate "
	: "=a" (error), "=b" (result)
	: "a" ((unsigned short)0x5F04), "b" (mode),
	  "m" (VGA_barrier)
	);
  if ((error & 0xFF) != 0x5F || (error >> 8) != 1
			     || (result & 0x3F) == 0)
      return 0; /* zero mean error */

  /* 43 means interlaced (87Hz/2): */
  return ((const unsigned char []){ 43, 56, 60, 70, 72, 75 })
			[__builtin_ffs (result) - 1];
  }

extern inline unsigned
_ET4000_get_refresh_rate (unsigned short screenheight)
  {
  unsigned short bx, result;
  unsigned char error;

  if (screenheight == 640)
      bx = 0x00F1;
    else if (screenheight == 800)
      bx = 0x01F1;
    else if (screenheight == 1024)
      bx = 0x02F1;
    else if (screenheight == 1280)
      bx = 0x03F1;
    else
      return 0;

  asm (" int $0x10 # _ET4000_get_refresh_rate "
	: "=a" (error), "=c" (result)
	: "a" ((unsigned short)0x1201), "b" (bx),
	  "m" (VGA_barrier)
	);
  if (error != 0x12)
      return 0; /* error */

  if (screenheight == 640 && result <= 3)
      return ((const unsigned char []){60, 72, 75, 90}) [result];
    else if (screenheight == 800 && result <= 4)
      return ((const unsigned char []){56, 60, 72, 75, 90}) [result];
    else if (result <= 3)
      return ((const unsigned char []){43, 60, 70, 75}) [result];
  return 0;
  }

extern inline unsigned
_ET4000_set_refresh_rate (unsigned short screenheight, unsigned char maxHz)
  {
  unsigned short bx, cx, result;
  unsigned char error;

  if (screenheight == 640) {
      bx = 0x00F1;
      if (maxHz < 72)
	  cx = 0;
	 else if (maxHz < 75)
	  cx = 1;
	 else if (maxHz < 90)
	  cx = 2;
	 else
	   cx = 3;
      }
    else if (screenheight == 800) {
      bx = 0x01F1;
      if (maxHz < 60)
	  cx = 0;
	 else if (maxHz < 72)
	  cx = 1;
	 else if (maxHz < 75)
	  cx = 2;
	 else if (maxHz < 90)
	   cx = 3;
	 else
	   cx = 4;
      }
    else if (screenheight == 1024) {
      bx = 0x02F1;
      if (maxHz < 60)
	  cx = 0;
	 else if (maxHz < 70)
	  cx = 1;
	 else if (maxHz < 75)
	  cx = 2;
	 else
	   cx = 3;
      }
    else if (screenheight == 1280) {
      bx = 0x03F1;
      if (maxHz < 60)
	  cx = 0;
	 else if (maxHz < 70)
	  cx = 1;
	 else if (maxHz < 75)
	  cx = 2;
	 else
	   cx = 3;
      }
    else
      return 1; /* error */

  asm volatile (" int $0x10 # _ET4000_set_refresh_rate "
	: "=a" (error), "=c" (result), "=m" (VGA_barrier)
	: "a" ((unsigned short)0x1200), "b" (bx), "c" (cx)
	);
  if (error != 0x12 || cx != result)
      return 1; /* error */
  return 0;
  }

extern inline void
_CIRRUS_set_monitor_type (unsigned char maxHz640,
			   unsigned char maxHz800,
			   unsigned char maxHz1024,
			   unsigned char maxHz1280,
			   unsigned short maxVertical)
  {
  unsigned short ax, bx, cx, dx;

  ax = 0; /* 480 lines, 60 Hz @ 640x480 */
  if (maxHz640 >= 72)
      ax++;
  ax <<= 4;

  if (maxVertical >= 600)
      ax++;
  if (maxVertical >= 768)
      ax++;
  if (maxVertical >= 1024)
      ax++;

  ax |= 0x1200;

  bx = 0; /* 56 Hz @ 800x600, 87i Hz @ 1024x768 */
  if (maxHz1024 >= 60)
      bx ++;
  if (maxHz1024 >= 70)
      bx ++;
  if (maxHz1024 >= 72)
      bx ++;
  if (maxHz1024 >= 76)
      bx ++;
  bx <<= 4;

  if (maxHz800 >= 60)
      bx++;
  if (maxHz800 >= 72)
      bx++;

  bx = (bx << 8) | 0xA4;

  cx = 0; /* 87i Hz @ 1280x1024 */
  if (maxHz1280 >= 60)
      cx ++;
  if (maxHz1280 >= 70)
      cx ++;
  cx <<= 12;

  dx = 0;

  asm volatile (" int $0x10 # _CIRRUS_set_monitor_type "
	: "+a" (ax), "+b" (bx), "+c" (cx), "+d" (dx)
	);
  }

enum _CIRRUS_set_horizontal_frequency {
    _CIRRUS_standard_VGA = 0,			/* 31.5 kHz */
    _CIRRUS_8514_compatible = 1,		/* 31.5 kHz and 35.5 kHz interlaced */
    _CIRRUS_SuperVGA = 2,			/* 31.5-35.1 kHz */
    _CIRRUS_extended_SuperVGA = 3,		/* 31.5-35.5 kHz */
    _CIRRUS_multi_frequency = 4,		/* 31.5-37.8 kHz */
    _CIRRUS_extended_multi_frequency = 5,	/* 31.5-48.0 kHz */
    _CIRRUS_super_multi_frequency = 6,		/* 31.5-56.0 kHz */
    _CIRRUS_extended_super_multi_frequency = 7	/* 31.5-64.0 kHz */
    };

extern inline void
_CIRRUS_set_horizontal_frequency (enum _CIRRUS_set_horizontal_frequency freq)
  {
  asm volatile (" int $0x10 # _CIRRUS_set_horizontal_frequency "
	:
	: "a" ((unsigned short)0x1200 | freq),
	  "b" ((unsigned short)0xA2)
	);
  }

#endif /* VGABIOS_H */
