/* vesabios.h */
#ifndef VESABIOS_H
#define VESABIOS_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* VESA_RGB is NOT packed to 3 byte here, unlike VGA */
typedef struct {
    unsigned char blue, green, red, align;
    } VESA_RGB;

typedef struct {
    char		VbeSignature[4]; /* "VESA" but NOT zero terminated.*/
    unsigned short	VbeVersion;
    farptr		OemStringPtr;
    struct vesa_capabilities_str {
	unsigned	DAC8bit_present : 1;
	unsigned	nonVGAcontroller : 1;
	unsigned	DAC_needs_programmed_while_blanking : 1;
	unsigned	VBE_AF_extension : 1;
	unsigned	hardware_mouse_cursor : 1;
	unsigned	hardware_clipping : 1;
	unsigned	transparent_bitblt : 1;
	unsigned	reserved : 25;
	} __attribute__ ((packed)) Capabilities;
    farptr		VideoModePtr;
    unsigned short	TotalMemory;	/* in 64K blocks */
    /* if VBE 2.0+ (else structure is 256 bytes) : */
    unsigned short	OemSoftwareRev;
    farptr		OemVendorNamePtr;
    farptr		OemProductNamePtr;
    farptr		OemProductRevPtr;

    unsigned char	Reserved[222];	/* ONLY VESA version 1.2- info */
    unsigned char	OemData[256]; /* will not kill VESA1.2- */
    } __attribute__ ((packed)) VESA_VbeInfoBlock;

typedef struct {
    struct vesa_mode_attribute_str {
	/* VESA: */
	unsigned short mode_supported : 1;	/* e.g. enought RAM for it */
	unsigned short optional_info_available : 1;
	unsigned short BIOS_output_supported : 1;
	unsigned short color_mode : 1;		/* else monochrome */
	unsigned short graphic_mode : 1;	/* else text mode */

	/* VBE 2.0: */
	unsigned short mode_not_VGA_compatible : 1;
	unsigned short no_VGA_compatible_window : 1;
	unsigned short linear_framebuffer_supported : 1;
	unsigned short reserved : 8;
	} __attribute__ ((packed)) ModeAttributes;
    struct {
       unsigned char win_exist    : 1;
       unsigned char win_readable : 1;
       unsigned char win_writable : 1;
	unsigned char padding      : 5;
       } __attribute__ ((packed)) WinAAttributes, WinBAttributes;
    unsigned short WinGranularity;	/* in Kb */
    unsigned short WinSize;		/* in Kb */
    unsigned short WinASegment;
    unsigned short WinBSegment;
    farptr         WinFuncPtr;
    unsigned short BytesPerScanLine;
    /* optional part, see "attribute.optional_info_available": */
    unsigned short XResolution; /* char in text modes, pixel in graphic modes */
    unsigned short YResolution;
    unsigned char  XCharSize;
    unsigned char  YCharSize;
    unsigned char  NumberOfPlanes;
    unsigned char  BitsPerPixel;
    unsigned char  NumberOfBanks;	/* unused here : no multipaged screen */
    enum { mem_text = 0, mem_CGA, mem_HGC, mem_EGA, mem_packedpixel,
	   mem_seq256, mem_HiColor24, mem_YUV } MemoryModel : 8;
	   /* mem_seq256 = non chain 4 */
    unsigned char  BankSize;		/* in Kb, unused here */
    unsigned char  NumberOfImagePages;
    unsigned char  Reserved;
    /* VBE 1.2+: */
	/* see instboot.h, vesabios.h, user.h, vmlinuz.h: */
    struct vesa_color_layout_str layout;
    struct {
	unsigned char color_ramp_programmable : 1;
	unsigned char reserved_space_usable : 1;
	} __attribute__ ((packed)) DirectColorModeInfo;
    /* VBE 2.0: */
    unsigned       PhysBasePtr;
    unsigned       OffScreenMemOffset;
    unsigned short OffScreenMemSize;	/* in Kb */
    unsigned char  Reserved2[206];
    } __attribute__ ((packed)) VESA_modeinfo_t;

/* "BIOS_output_supported" means those are supported:
 *  01	Set Cursor Size
 *  02	Set Cursor Position
 *  06	Scroll TTY window up or Blank Window
 *  07	Scroll TTY window down or Blank Window
 *  09	Write character and attribute at cursor position
 *  0A	Write character only at cursor position
 *  0E	Write character and advance cursor
 */


typedef struct {
    unsigned char padding[8];
    struct {
	unsigned short padding : 1;	/* mapping right ? */
	unsigned short first_letter : 5;
	unsigned short second_letter : 5;
	unsigned short third_letter : 5;
	} __attribute__ ((packed)) manufacturer_id;
    unsigned short monitor_model;
    unsigned serial_number;
    unsigned char manufacture_week;
    unsigned char manufacture_year; /* base: 1990 */
    unsigned char version;
    unsigned char revision;
    unsigned char video_input_type; /* bitfield */
    unsigned char max_horizontal_size_cm;
    unsigned char max_vertical_size_cm;
    unsigned char gamma_factor; /* gamma = 1.0 + gamma_factor/100 */
    struct {
	unsigned char unused : 3;
	unsigned char RBGcolor : 1;
	unsigned char unused2 : 1;
	unsigned char active_off_supported : 1;
	unsigned char suspend_supported : 1;
	unsigned char standby_supported : 1;
	} __attribute__ ((packed)) DPMS;
    unsigned char chroma_green_and_red;
    unsigned char chroma_white_and_blue;
    unsigned char chroma_red_Y;
    unsigned char chroma_red_X;
    unsigned char chroma_green_Y;
    unsigned char chroma_green_X;
    unsigned char chroma_blue_Y;
    unsigned char chroma_blue_X;
    unsigned char chroma_white_Y;
    unsigned char chroma_white_X;
    struct established_timing_s {
	unsigned short _720_400_70Hz :1;
	unsigned short _720_400_88Hz :1;
	unsigned short _640_480_60Hz :1;
	unsigned short _640_480_67Hz :1;
	unsigned short _640_480_72Hz :1;
	unsigned short _640_480_75Hz :1;
	unsigned short _800_600_56Hz :1;
	unsigned short _800_600_60Hz :1;

	unsigned short _800_600_72Hz :1;
	unsigned short _800_600_75Hz :1;
	unsigned short _832_624_75Hz :1;
	unsigned short _1024_768_i87Hz :1;
	unsigned short _1024_768_60Hz :1;
	unsigned short _1024_768_70Hz :1;
	unsigned short _1024_768_75Hz :1;
	unsigned short _1280_1024_75Hz :1;
	} __attribute__ ((packed)) established_timing;
    unsigned char manufacturer_timing;
    struct {
	unsigned char	resolution;	/* X resolution = (field+31)*8 */
	unsigned char	vertical_refresh_freq : 6; /* add 60Hz */
	unsigned char	aspect_ratio : 2; /* 00=16:10, 01=4:3, 10=5:4, 11=16:9 */
	} __attribute__ ((packed)) standard_timing[8];
    union {
	struct {
	    unsigned short	wordmarker_0;
	    unsigned char	bytemarker_0;
	    enum { serial_number = 0xFF,
		   vendor_name = 0xFE,
		   vertical_horizontal_frequency_range = 0xFD,
		   model_name = 0xFC
// RB Interrupt List say that but I read something else...
//		} str_type : 8;
//	    char		string[14];
		} str_type : 16;
	    char		string[13];
	    } __attribute__ ((packed)) strings;
	struct {
	    unsigned char	horizontal_freq_Khz;	/* != 0 */
	    unsigned char	vertical_freq_Hz;	/* != 0 */
	    unsigned char	horizontal_active_time_pix; /* != 0 */
	    unsigned char	horizontal_blanking_time_pix;
	    unsigned char	horizontal_active2_time;
	    unsigned char	vertical_active_time_line;
	    unsigned char	vertical_blanking_time_line;
	    unsigned char	vertical_active_time2;
	    unsigned char	horizontal_sync_offset;
	    unsigned char	horizontal_sync_pulsewidth_pix;
	    unsigned char	vertical_sync_pulsewidth;
	    unsigned char	horizontal_vertical_sync_offset2;
	    unsigned char	horizontal_image_size_mm;
	    unsigned char	vertical_image_size_mm;
	    unsigned char	horizontal_image_size_2;
	    unsigned char	horizontal_border_pix;
	    unsigned char	vertical_border_pix;
	    struct {
		unsigned char unused : 1;
		unsigned char horizontal_sync_pol : 1;
		unsigned char vertical_sync_pol : 1;
		unsigned char sync_type : 2;
		enum { no_sound, stereo_right,
			stereo_left, undefined } sound : 2;
		unsigned char interlaced : 1;
		} __attribute__ ((packed)) display_t;
	    } __attribute__ ((packed)) timings;
	} __attribute__ ((packed)) detailed_timing[4];
    unsigned char nb_extension_blocks;
    unsigned char checksum;
    } __attribute__ ((packed)) VBE_EDID_t;

extern inline unsigned
_VESA_getinfo (VESA_VbeInfoBlock *VESA_info)
  {
  unsigned short cmd_status = 0x4F00;

  if (sizeof (VESA_VbeInfoBlock) != 512)
      __ERROR();

  asm (" int $0x10 # _VESA_getinfo "
	: "+a" (cmd_status), "+m" (*VESA_info) /* VbeSignature inited */
	: "D" (VESA_info)	/* in fact %es:%di */
	);
  return (cmd_status != 0x004F);
  }

extern inline unsigned
_VESA_getmodeinfo (VESA_modeinfo_t *VESA_modeinfo, unsigned short mode)
  {
  unsigned short cmd_status = 0x4F01;

  if (sizeof (VESA_modeinfo_t) != 256)
      __ERROR();

  asm (" int $0x10 # _VESA_getmodeinfo "
	: "+a" (cmd_status), ASM_STRUCT_OUTPUT (*VESA_modeinfo)
	: "c" (mode), "D" (VESA_modeinfo)	/* in fact %es:%di */
	);
  return (cmd_status != 0x004F);
  }

extern inline unsigned
_VESA_setmode (unsigned short mode)
  {
  unsigned short cmd_status = 0x4F02;

  /* %%bx bit 15 set: do not clear video
   *      bit 14 set: linear framebuffer
   *	  bit 13 set: initialise hardware accelerator
   */
  asm volatile (" int $0x10 # _VESA_setmode "
	: "+a" (cmd_status)
	: "b" (mode)
	);
  return (cmd_status != 0x004F);
  }

struct VBE3_crtc_info_block_str {
    unsigned short total_number_of_pixels_horizontally;
    unsigned short horizontal_sync_start_in_pixels;
    unsigned short horizontal_sync_end_in_pixels;
    unsigned short total_number_of_scan_lines;
    unsigned short vertical_sync_start_in_scan_lines;
    unsigned short vertical_sync_end_in_scan_lines;
    struct {
	unsigned char enable_double_scanning	: 1;
	unsigned char enable_interlacing	: 1;
	unsigned char horizontal_sync_polarity	: 1; /* 0: positive */
	unsigned char vertical_sync_polarity	: 1; /* 0: positive */
	unsigned char unknown			: 4;
	} __attribute__((packed)) flags;
    unsigned	   pixel_clock_in_Hz;
    /* MUST be set to pixel_clock / (HTotal * VTotal) : */
    unsigned short refresh_rate_in_0_01_Hz_units;
    unsigned char  reserved[40];
    };

/* mode bit 11 has to be set in list of accelerated modes: */
extern inline unsigned
_VESA3_setmode (unsigned short mode,
		struct VBE3_crtc_info_block_str *infoblock)
  {
  unsigned short cmd_status = 0x4F02;

  /* %%bx bit 15 set: do not clear video
   *      bit 14 set: linear framebuffer
   *	  bit 13 set: initialise hardware accelerator
   */
  asm volatile (" int $0x10 # _VESA_setmode "
	: "+a" (cmd_status)
	: "b" (mode), "D" (infoblock),	/* in fact %es:%di */
		ASM_STRUCT_INPUT (*infoblock)
	);
  return (cmd_status != 0x004F);
  }

extern inline unsigned
_VESA_getmode (unsigned short *mode)
  {
  unsigned short cmd_status = 0x4F03;

  asm (" int $0x10 # _VESA_getmode "
	: "+a" (cmd_status), "=b" (*mode) /* *mode contains bits 13, 14 & 15 */
	);
  return (cmd_status != 0x004F);
  }

extern inline unsigned
_VESA_setwin (unsigned short offsetDIVgranul, unsigned char winBorA)
  {
  unsigned short cmd_status = 0x4F05;

#if USER_SUPPORT & VESA_WINFUNCTION
  if (UI.parameter.addrpositionfarfct != 0) {
      /*
       * For VESA < 2.0, the function may not set the result in %ax at
       * exit. Also %ax and %dx may be destroyed.
       */
      unsigned short dummydx;
      asm volatile ( " lcallw *%a4 # _VESA_setwin "
	    : "+a" (cmd_status), "=d" (dummydx)
	    : "b" ((unsigned short)winBorA), /* %bh = 0 */
	      "d" (offsetDIVgranul),
	      "pr" (&UI.parameter.addrpositionfarfct)
	    );
      return 0;
      }
#endif

  asm volatile ( " int $0x10 # _VESA_setwin "
	: "+a" (cmd_status)
	: "b" ((unsigned short)winBorA), /* %bh = 0 */
	  "d" (offsetDIVgranul)
	);
  return (cmd_status != 0x004F);
  }

extern inline unsigned
_VESA_getwin (unsigned short *offsetDIVgranul, unsigned char winBorA)
  {
  unsigned short cmd_status = 0x4F05;

	/* reported buggy Phoenix S3 Trio64V+ */
  asm (" int $0x10 # _VESA_getwin "
	: "+a" (cmd_status), "=d" (*offsetDIVgranul)
	: "b" ((unsigned short)0x0100 | winBorA)
	);
  return (cmd_status != 0x004F);
  }

/*
 * xstart and ystart are zero based, should be multiplyed
 * by char sizes while in text modes.
 */
extern inline unsigned
_VESA_setDisplayStart (unsigned short xstart, unsigned short ystart)
  {
  unsigned short cmd_status = 0x4F07;
  unsigned short tmp = 0; /* 0x80: "while vertical retrace" */

  asm volatile (" int $0x10 # _VESA_setDisplayStart "
	: "+a" (cmd_status), "+b" (tmp)
	: "c" (xstart), "d" (ystart)
	);
  return (cmd_status != 0x004F);
  }

extern inline unsigned
_VESA_getDisplayStart (unsigned short *xstart, unsigned short *ystart)
  {
  unsigned short cmd_status = 0x4F07;
  unsigned short tmp = 1; /* %bh = 0 for get and set */

  asm (" int $0x10 # _VESA_getDisplayStart "
	: "+a" (cmd_status), "+b" (tmp), "=c" (*xstart), "=d" (*ystart)
	);
  return (cmd_status != 0x004F);
  }

extern inline unsigned
_VESA_setDACwidth (unsigned char depth)
  {
  unsigned short cmd_status = 0x4F08;
  unsigned short result = depth << 8; /* %bl = 0 */

  asm volatile (" int $0x10 # _VESA_setDACwidth "	/* VESA 1.2+ */
	: "+a" (cmd_status), "+b" (result)
	);
  return (cmd_status != 0x004F || (result >> 8) != depth);
  }

extern inline unsigned
_VESA_getDACwidth (unsigned char *depth)
  {
  unsigned short cmd_status = 0x4F08;
  unsigned short tmp = 1;

  asm (" int $0x10 # _VESA_getDACwidth "		/* VESA 1.2+ */
	: "+a" (cmd_status), "+b" (tmp)
	);
  *depth = tmp >> 8;
  return (cmd_status != 0x004F);
  }

extern inline unsigned
_VESA_setDACarray (unsigned short startindex, unsigned short nb,
		   VESA_RGB *palette)
  {
  unsigned short cmd_status = 0x4F09;
  unsigned char getorset = 0;

  asm volatile (" int $0x10 # _VESA_setDACarray "	/* VESA 2.0+ */
	: "+a" (cmd_status)
	: "b" (getorset), "c" (nb), "d" (startindex),
	  "D" (palette), /* in fact %es:%di */
	  ASM_STRUCT_INPUT (*palette)
	);
  return (cmd_status != 0x004F);
  }

extern inline unsigned
_VESA_getDACarray (unsigned short startindex, unsigned short nb,
		   VESA_RGB *palette)
  {
  unsigned short cmd_status = 0x4F09;
  unsigned char getorset = 1;

  asm (" int $0x10 # _VESA_setDACarray "		/* VESA 2.0+ */
	: "+a" (cmd_status), ASM_STRUCT_OUTPUT (*palette)
	: "b" (getorset), "c" (nb), "d" (startindex),
	  "D" (palette) /* in fact %es:%di */
	);
  return (cmd_status != 0x004F);
  }

struct GetPMinterface_str {
    unsigned short offset_Set_Window_Call;
    unsigned short offset_Set_Display_Start;
    unsigned short offset_Set_Primary_Palette_data;
     /* The following field:
      * can be == 0 if standart VGA
      * contains: Port, Port, ... , Port,  (2 bytes, little endian words)
      *     Terminate Port List with FF FF
      * contains: Memory locations (4 bytes), Length (2 bytes)
      *     Terminate Memory List with FF FF
      */
    unsigned short offset_Ports_and_Memory_Locations;
    };

extern inline unsigned	/* VESA 2.0+ */
_VESA_GetPMinterface (farptr *addr, unsigned short *len)
  {
  unsigned short cmd_status = 0x4F0A, null = 0;

  asm (
"	pushl	%%es						\n"
"	mov	%%di,%%es					\n"
"	int	$0x10		# _VESA_GetPMinterface		\n"
"	pushw	%%es						\n"
"	pushw	%%di						\n"
"	popl	%%edi						\n"
"	popl	%%es						\n"
	: "+a" (cmd_status), "=D" (*addr), "=c" (*len)
	: "b" (0), "D" (null), "c" (null)
	);

  return (cmd_status != 0x004F);
  }

extern inline unsigned
_VESA3_GetNearestPixelClock (unsigned short mode, unsigned requested)
  {
  unsigned short cmd_status = 0x4F0B;
  unsigned char subcmd = 0;

  asm (" int $0x10 # _VESA3_GetNearestPixelClock "
	: "+a" (cmd_status), "+c" (requested)
	: "b" (subcmd), "d" (mode)
	);
  if (cmd_status != 0x004F)
      return 0;
  return requested;
  }

/*
 * EDID interface:
 */
extern inline unsigned
_EDID_detect (unsigned short monitor_port, unsigned short *bx)
  {
  unsigned short cmd_status = 0x4F15;
/* %cx modified by NVIDIA RIVA TNT2 Model 64/Model 64 Pro */
/* All other registers may be destroyed */
  unsigned dx, si;
  *bx = 0;

/* output:
	BH= Approx. time in seconds, rounded up, to transfer one EDID block (128 bytes).
	BL = DDC level supported (**)
		bit 0 = 0 DDC1 not supported
			= 1 DDC1 supported
		bit 1 = 0 DDC2 not supported
			= 1 DDC2 supported
		bit 2 = 0 Screen not blanked during data transfer (***)
			= 1 Screen blanked during data transfer
 */
asm (
"	pushw	%%es			\n"
"	movw	%%di,%%es		\n"
"	int	$0x10 # _EDID_detect	\n"
"	popw	%%es			\n"
	: "+a" (cmd_status), "+b" (*bx), "+c" (monitor_port),
	  "=d" (dx), "=S" (si)
	: "D" (0)
	);
  return (cmd_status != 0x004F);
  }

extern inline unsigned
_EDID_read (unsigned short monitor_port, VBE_EDID_t *info, unsigned short blocknr)
  {
  unsigned short cmd_status = 0x4F15, read_cmd = 1, si;

  if (sizeof (VBE_EDID_t) != 0x80)
      __ERROR();

  asm (" int $0x10 # _EDID_read "
	: "+a" (cmd_status), "+b" (read_cmd), "+c" (monitor_port), "+d" (blocknr), "=S" (si), ASM_STRUCT_OUTPUT (*info)
	: "D" (info) /* in fact %es:%di */
	);
  /* returns 0x014F if monitor non EDID */
  return (cmd_status != 0x004F);
  }

#endif /* VESABIOS_H */
