/* util.h */
#ifndef UTIL_H
#define UTIL_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"

/*
 * This configures the memory manager:
 */
#define MALLOC_ARRAY_SIZE	100
#define MALLOC_QUANTUM		32	/* bytes */
#define MALLOC_WASTE		2	/* in quantum */

#if SETUP & BIG_MALLOC
#define MALLOC_SIZE	0x60000U /* 4*64 Kbytes for the pool */
#define MALLOC_BASE	0x10000U /* pool just after the top of the stack */
#endif

/*
 * This is the type of memory managed by malloc:
 */
typedef void allocmem __attribute__((aligned(MALLOC_QUANTUM)));

/*
 * The big variable:
 */
extern struct UTIL_str {
    struct processor_str {
	enum processor_enum {
	    proc_nocpuid = 0,
	    proc_intel,
	    proc_amd,
	    proc_umc,
	    proc_cyrix,
	    proc_nexgen,
	    proc_idt,
	    proc_rise,
	    proc_transmeta,
	    proc_unknown
	    } brand;
	unsigned family;
	unsigned Dflags, Cflags, Aflags; /* D: std, A: AMD extension, C: newest */
	char longname[3 * 4 * 4];
	/* not initialised by init_processor_str() : */
	unsigned calibrate_rdtsc;	/* cycle (rdtsc) per 1/100 second */
	unsigned calibrate_loopl;	/* "loopl ." %ecx value for 1/100 second */
	} processor;
    enum { stdkbd, extkbd, ext122kbd, serialkbd } keyboard;
    const unsigned short *keyptr;	/* non US qwerty keyboard if != 0 */
    unsigned short joystick_present;
    signed char time_hour_offset, time_minute_offset;
    unsigned char boot_device_read_only;
    unsigned char bios_boot_dl;
    unsigned char reserved4[2];
    unsigned swap_BIOSint0x13_irqhandler;
    unsigned reserved1[2];
#if KEEP_STRUCT_MAPPING || !(SETUP & BIOS_ONLY)
    unsigned short DOS_version;	/* 0 if on BIOS, i.e. !STATE.dos_running */
    unsigned short WIN_version;	/* 0x310 or 0x410 */
    unsigned short XMS_version;
    unsigned short EMM_version;
    unsigned short VCPI_version;
    unsigned short reserved;
    /* HIMEM, GEMMIS & VCPI stuff: */
    farptr HIMEM_entrypoint;
    farptr v86_info, v86_callback;
    enum {
	EMM_none = 0,
	EMM_emmx = 0x1,
	EMM_emmq = 0x2,
	EMM_emmx_suspended = 0x11,
	EMM_emmq_suspended = 0x12,
	EMM_VCPI_suspended = 0x13
	} emm_type;
    unsigned VCPI_max_address, VCPI_nb_free_pages, XMS_freemem;
#endif
#ifdef COMMANDFILE
    char *cmdfile_base;
    unsigned cmdfile_len;
#else
    unsigned reserved2[4];
#endif
    unsigned basemem, extendmem;

    CDROMbootspec CDROM_HDFD_emulation_spec; /* size 19 bytes, 0x13 bytes / TODO: move as local variable in main() */
    unsigned char CDROM_HDFD_emulation_state; /* 1:running, 2:stopped */
    unsigned reserved3[10];

    /* Keep at end, malloc variables inited at 0 (BSS) */
    unsigned malloc_failure_size_requested;
    struct memalloc_str {
	unsigned end_malloc;
	unsigned short array_size, quantum;
	struct malloc_array_str {
	    allocmem *adr;	/* strictly increasing */
	    signed   size;	/* if < 0, not in use */
	    } malloc_array[MALLOC_ARRAY_SIZE];
	} memalloc;
//#define DBG_MALLOC
#ifdef DBG_MALLOC
    char memalloc_content[MALLOC_ARRAY_SIZE][8];
#endif
    } UTIL;

#if !(SETUP & BIOS_ONLY)
#define HIMEM_ACTIVE()	(BOOT1_DOS_running() && UTIL.XMS_freemem)
#define EMM_ACTIVE()	(BOOT1_DOS_running() && UTIL.emm_type != EMM_none)
#else
#define HIMEM_ACTIVE()	0
#define EMM_ACTIVE()	0
#endif

/*
 * Dynamic memory management - over-simplified !
 *
 * The fact that it is "over-simplified" does not mean that
 * it is not covered by GPL: if you ever include [a part of]
 * this code, or even [a part of] its design into a private software,
 * the resulting code immediately goes to GPL and you have to
 * distribute the complete source code of your product, or you
 * have to stop using any GPL products, like GCC or Linux.
 *
 * Using "allocmem *" instead of "void *" to show their alignment.
 * How about using FASTCALL ?
 */

#ifdef DBG_MALLOC
#define MALLOC(siz, name) ({ \
	if (siz == 0) printf (" malloc(0) in %s line %d ", __FILE__, __LINE__);		\
	allocmem *adr = malloc ((siz) + 1);							\
	if (adr) {										\
		unsigned i;									\
		for (i = 0; i < MALLOC_ARRAY_SIZE; i++)						\
			if (UTIL.memalloc.malloc_array[i].adr == adr) {				\
				*(long long*)UTIL.memalloc_content[i] = *(long long*)name;	\
				UTIL.memalloc_content[i][7] = '\0';				\
				break;								\
				}								\
		((char *)adr)[UTIL.memalloc.malloc_array[i].size-1] = 0xBE;			\
		}										\
	/* return */ adr;									\
	})

#define FREE(ptr) do { \
	if (ptr) {											\
		unsigned i;										\
		for (i = 0; i < MALLOC_ARRAY_SIZE; i++)							\
			if (UTIL.memalloc.malloc_array[i].adr == ptr) {					\
				if (((char *)ptr)[UTIL.memalloc.malloc_array[i].size-1] != 0xBE)	\
					printf ("free(%s) no marker ", UTIL.memalloc_content[i]);	\
				UTIL.memalloc_content[i][0] = '\0';					\
				break;									\
				}									\
		free (ptr);										\
		}											\
	else printf (" free(0) in %s line %d ", __FILE__, __LINE__);					\
	} while (0)

#define REALLOC(ptr, siz, name) ({ \
	if (siz == 0) printf (" realloc(0) in %s line %d ", __FILE__, __LINE__);			\
	if (ptr) {											\
		unsigned i;										\
		for (i = 0; i < MALLOC_ARRAY_SIZE; i++)							\
			if (UTIL.memalloc.malloc_array[i].adr == ptr) {					\
				if (((char *)ptr)[UTIL.memalloc.malloc_array[i].size-1] != 0xBE)	\
					printf ("realloc(%s) no marker ", UTIL.memalloc_content[i]);	\
				UTIL.memalloc_content[i][0] = '\0';					\
				break;									\
				}									\
		}											\
	allocmem *adr = realloc (ptr, (siz)+1);								\
	if (adr) {											\
		unsigned i;										\
		for (i = 0; i < MALLOC_ARRAY_SIZE; i++)							\
			if (UTIL.memalloc.malloc_array[i].adr == adr) {					\
				*(long long*)UTIL.memalloc_content[i] = *(long long*)name;		\
				UTIL.memalloc_content[i][7] = '\0';					\
				break;									\
				}									\
		((char *)adr)[UTIL.memalloc.malloc_array[i].size-1] = 0xBE;				\
		}											\
	/* return */ adr;										\
	})

#else
#define MALLOC(size, name)		malloc(size)
#define FREE(adr)			free(adr)
#define REALLOC(ptr, size, name)	realloc(ptr, size)
#endif

/**
 ** Some inline functions, in the caller segment:
 **/
extern inline void init_malloc (void)
  {
  UTIL.memalloc.array_size = MALLOC_ARRAY_SIZE;
  UTIL.memalloc.quantum    = MALLOC_QUANTUM;
#if SETUP & BIG_MALLOC
  UTIL.memalloc.end_malloc = MALLOC_BASE;
#else
  UTIL.memalloc.end_malloc = MALLOC_QUANTUM
		* DIVROUNDUP((unsigned)_edata, MALLOC_QUANTUM);
#endif
  }

extern inline unsigned usedstack (void)
  {
#if SETUP & BIG_MALLOC
  unsigned *stkptr = _edata;
#else
  unsigned *stkptr = (unsigned *)UTIL.memalloc.end_malloc;
#endif

  while (*stkptr == 0)
      stkptr++;

  return 0x10000 - ((unsigned)stkptr);
  }

extern inline unsigned totalstack (void)
  {
#if SETUP & BIG_MALLOC
  unsigned *stkptr = _edata;
#else
  unsigned *stkptr = (unsigned *)UTIL.memalloc.end_malloc;
#endif

  return 0x10000 - ((unsigned)stkptr);
  }

extern inline unsigned freemem (void)
  {
#if SETUP & BIG_MALLOC
  return MALLOC_SIZE - ((unsigned)UTIL.memalloc.end_malloc - MALLOC_BASE);
#else
  return getesp() - stack_safety - (unsigned)UTIL.memalloc.end_malloc;
#endif
  }

extern inline void update_limit (unsigned EndMalloc, unsigned StackSafety)
  {
#if SETUP & BIG_MALLOC
#if DEBUG & DEBUG_ADDR
  asm volatile (" mov %0,%1 "
	: : "r" (EndMalloc), "m" (STATE.data_limit.high_limit));
#endif
#else  /* BIG_MALLOC */
#if DEBUG & DEBUG_STACK
  asm volatile (" mov %0,%1 "
	: : "r" (EndMalloc + StackSafety), "m" (STATE.stack_limit.low_limit));
#endif
#endif /* BIG_MALLOC */
  }

/**
 ** The functions:
 **/
allocmem *malloc (unsigned size);
awk_farcall (malloc);
void free (const allocmem *ptr);
awk_farcall (free);
allocmem *realloc (allocmem *ptr, unsigned size);
awk_farcall (realloc);

void detect_environment (void);
awk_farcall (detect_environment);
unsigned hide_unhide_partitions (unsigned index);
awk_farcall (hide_unhide_partitions);

struct diskparam_str;
unsigned long long get_remote_uninstall_lba (struct diskparam_str *dp, const bootbefore_t *bootbefore);
awk_farcall (get_remote_uninstall_lba);
unsigned initrd_file (unsigned index);
awk_farcall (initrd_file);

#ifdef COMMANDFILE
enum commandfile_cmd {cmdf_getnb, cmdf_getinitrd,
	cmdf_gettimeout, cmdf_getmenuname, cmdf_getcmdline};

unsigned commandfile_get (const char *filesystemlabel, const char *kernelname,
	const char *initrdname, enum commandfile_cmd cmd,
	unsigned iteration, char *buffer, unsigned bufsize);
awk_farcall (commandfile_get);
#endif


unsigned set_A20 (unsigned enabled);
awk_farcall (set_A20);
unsigned segment_door (unsigned open);
awk_farcall (segment_door);

#endif /* UTIL_H */
