/* util.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"
#if (SETUP & XCODE_SEGMENT)
#define CALLING_FROM_XCODESEG	/* before other includes */
#endif

#include "instboot.h"
#include "library.h"
#include "boot.h"	/* BOOT1_COM_port() */
#include "debug.h"
#include "bios.h"	/* _BIOS_nbtick(), _BIOS_equipment_flags(), CDROMbootspec */
#include "util.h"
#include "dos.h"	/* get_window_version(), DOS_version() */
#include "disk.h"	/* DI.readsector(), DI.writesector() */
#include "user.h"
#include "vesabios.h"	/* struct vesa_mode_attribute_str */
#include "vmlinuz.h"	/* struct e820map_info */
#include "fs.h"		/* BOOTWAY */
#include "messages.h"
#include "xms.h"

//#define MDBG(X) ZDBG(X)	// uncomment if needed, but size of strings may create stack overflow.
#ifndef MDBG
#define MDBG(X)
#endif

//#define A20DBG(X) ZDBG(X)	// uncomment if needed, but size of strings may create stack overflow.
#ifndef A20DBG
#define A20DBG(X)
#define EXCLUDE_GETA20
#endif

//#define HIDEDBG(X) ZDBG(X)	// uncomment if needed, but size of strings may create stack overflow.
#ifndef HIDEDBG
#define HIDEDBG(X)
#endif

/*
 * Malloc matter: over-simplified !
 * I do not think it is SMP safe -:)
 *
 * The fact that it is "over-simplified" does not mean that
 * it is not covered by GPL: if you ever include [a part of]
 * this code, or even [a part of] its design into a private software,
 * the resulting code immediately goes to GPL and you have to
 * distribute the complete source code of your product.
 */
UTIL_FCT_PREFIX(malloc) __attribute__ ((malloc)) allocmem *
malloc (unsigned size)
  {
  unsigned i;
  unsigned size_quantum = DIVROUNDUP (size, MALLOC_QUANTUM);

  MDBG (("MALLOC %u: ", size));

  if (size == 0) {
      DBG ((" [malloc returns 0 because 0 bytes requested] "));
      MDBG (("repeat size: zero!\r\n"));
      return 0;
      }

  for (i = 0; i < nbof (UTIL.memalloc.malloc_array); i++) {
      if (UTIL.memalloc.malloc_array[i].size >= 0) {
	  if (UTIL.memalloc.malloc_array[i].size == 0)
	      break;	 /* end of used array */
	  }
	else {
	  unsigned nbquantum = -UTIL.memalloc.malloc_array[i].size / MALLOC_QUANTUM;

	  if (   size_quantum <= nbquantum
	      && size_quantum + MALLOC_WASTE >= nbquantum) {
	      UTIL.memalloc.malloc_array[i].size = -UTIL.memalloc.malloc_array[i].size;
	      MDBG (("affecting to block %u of %u (= 0x%X) bytes"
			" at address 0x%X\r\n",
			i,
			UTIL.memalloc.malloc_array[i].size,
			UTIL.memalloc.malloc_array[i].size,
			UTIL.memalloc.malloc_array[i].adr));
	      return UTIL.memalloc.malloc_array[i].adr;
	      }
	  }
      }

  if (i == nbof (UTIL.memalloc.malloc_array)) {
      DBG ((" [malloc returns 0] "));
      MDBG (("no more space in array!\r\n"));
      UTIL.malloc_failure_size_requested = size;
      return 0;
      }

#if SETUP & BIG_MALLOC
  if (UTIL.memalloc.end_malloc + size_quantum * MALLOC_QUANTUM > MALLOC_BASE + MALLOC_SIZE) {
      UTIL.malloc_failure_size_requested = size;
      DBG ((" [malloc returns 0] "));
      MDBG (("no more space in the pool (requested: %u, free: %u) !\r\n",
		size_quantum * MALLOC_QUANTUM,
		MALLOC_BASE + MALLOC_SIZE - UTIL.memalloc.end_malloc));
      return 0;
      }
#else
  if (UTIL.memalloc.end_malloc + size_quantum * MALLOC_QUANTUM > getesp() - stack_safety) {
      UTIL.malloc_failure_size_requested = size;
      DBG ((" [malloc returns 0] "));
      MDBG (("no more space before stack (requested: %u, free: %u) !\r\n",
		size_quantum * MALLOC_QUANTUM,
		getesp() - stack_safety - UTIL.memalloc.end_malloc));
      return 0;
      }
#endif

  UTIL.memalloc.malloc_array[i].size = size_quantum * MALLOC_QUANTUM;
  UTIL.memalloc.malloc_array[i].adr = (unsigned *)UTIL.memalloc.end_malloc;
  UTIL.memalloc.end_malloc += UTIL.memalloc.malloc_array[i].size;

  update_limit (UTIL.memalloc.end_malloc, stack_safety);

  MDBG (("allocated new block No %u of %d (= 0x%X) bytes at 0x%X\r\n",
	i,
	UTIL.memalloc.malloc_array[i].size,
	UTIL.memalloc.malloc_array[i].size,
	UTIL.memalloc.malloc_array[i].adr));
  return UTIL.memalloc.malloc_array[i].adr;
  }

UTIL_FCT_PREFIX(free) void
free (const allocmem *ptr)
  {
  unsigned i, j;

  MDBG (("FREE addr 0x%X: ", ptr));

  for (i = 0; i < nbof (UTIL.memalloc.malloc_array); i++)
      if (   UTIL.memalloc.malloc_array[i].adr == ptr
	  && UTIL.memalloc.malloc_array[i].size > 0)
	  break;
  if (i == nbof (UTIL.memalloc.malloc_array)) {
      MDBG (("invalid address!\r\n"));
      return;
      }

  j = i++;
  while (i < nbof (UTIL.memalloc.malloc_array)) {
      if (UTIL.memalloc.malloc_array[i].size > 0)
	  break;
      i++;
      }

  UTIL.memalloc.malloc_array[j].size = - UTIL.memalloc.malloc_array[j].size;

  if (i != nbof (UTIL.memalloc.malloc_array)) {
      MDBG (("block %u size %u is not the last block.\r\n",
		j, - UTIL.memalloc.malloc_array[j].size));
      return;
      }

#if SETUP & BIG_MALLOC
  MDBG (("last block %u size %u freed to pool",
		j, - UTIL.memalloc.malloc_array[j].size));
#else
  MDBG (("last block %u size %u freed to stack",
		j, - UTIL.memalloc.malloc_array[j].size));
#endif

  while (j != 0 && UTIL.memalloc.malloc_array[j-1].size < 0)
      j --;

  UTIL.memalloc.end_malloc = (unsigned) UTIL.memalloc.malloc_array[j].adr;
  MDBG ((" [new end_malloc 0x%X] ", UTIL.memalloc.end_malloc));

  while (j < nbof (UTIL.memalloc.malloc_array)) {
      if (UTIL.memalloc.malloc_array[j].size < 0) {
	  MDBG ((", block %u size %u",
		j, - UTIL.memalloc.malloc_array[j].size));
#if !(SETUP & BIG_MALLOC) /* clear the pool: unused stack */
	  memset (UTIL.memalloc.malloc_array[j].adr, 0,
			-UTIL.memalloc.malloc_array[j].size);
#endif
	  UTIL.memalloc.malloc_array[j].adr = 0;
	  UTIL.memalloc.malloc_array[j].size = 0;
	  }
      j++;
      }

  MDBG (("\r\n"));

  update_limit (UTIL.memalloc.end_malloc, stack_safety);
  }

/* TODO: integrate malloc() and free() as:
#define malloc(size)	realloc (0,size)
#define free(ptr)		realloc (ptr, 0)
#define alloc_report()	realloc (0, 0)
and array itself allocated
*/
UTIL_FCT_PREFIX(realloc) allocmem *
realloc (allocmem *ptr, unsigned size)
  {
  unsigned i, j;
  unsigned size_quantum;

  MDBG (("REALLOC 0x%X size: %d: ", ptr, size));

  if (size == 0) {
      DBG ((" [realloc returns 0] "));
      MDBG (("repeat size: zero!\r\n"));
      return 0;
      }

  if (ptr == 0) {
      MDBG ((" [realloc called with address 0] "));
      return malloc (size);
      }

  for (i = 0; i < nbof (UTIL.memalloc.malloc_array); i++)
      if (   UTIL.memalloc.malloc_array[i].adr == ptr
	  && UTIL.memalloc.malloc_array[i].size > 0)
	  break;

  if (i == nbof (UTIL.memalloc.malloc_array)) {
      DBG ((" [realloc returns 0] "));
      MDBG (("invalid address!\r\n"));
      return 0;
      }

  if ((signed) size <= UTIL.memalloc.malloc_array[i].size) {
      MDBG (("size preallocated sufficient.\r\n"));
      return ptr;
      }

  j = i++;

  while (i < nbof (UTIL.memalloc.malloc_array)) {
      if (UTIL.memalloc.malloc_array[i].size > 0)
	  break;
      i++;
      }

  if (i != nbof (UTIL.memalloc.malloc_array)) {
      unsigned *tmp;
      MDBG (("is not the last block.\r\n"));

      tmp = malloc (size);
      if (tmp == 0) {
	  MDBG (("REALLOC returns zero\r\n"));
	  return 0;
	  }
      memcpy (tmp, ptr, MIN ((unsigned)UTIL.memalloc.malloc_array[j].size, size));
      free (UTIL.memalloc.malloc_array[j].adr);
      return tmp;
      }

  MDBG (("is the last block.\r\n"));
  size_quantum = ROUNDUP (size, MALLOC_QUANTUM);

#if SETUP & BIG_MALLOC
  if (UTIL.memalloc.end_malloc + size_quantum - UTIL.memalloc.malloc_array[j].size
	> MALLOC_BASE + MALLOC_SIZE) {
      UTIL.malloc_failure_size_requested = size;
      DBG ((" [realloc returns 0] "));
      MDBG (("last block, no more space in the pool!\r\n"));
      return 0;
      }
#else
  if (UTIL.memalloc.end_malloc + size_quantum - UTIL.memalloc.malloc_array[j].size
//	> getesp() - stack_safety) {
	> getesp() - stack_safety - 4096) { /* we will realloc a block very shortly after read_analyse_chain->E2FS_add_sector_chain() */
      UTIL.malloc_failure_size_requested = size;
      DBG ((" [realloc returns 0] "));
      MDBG (("last block, no more space before stack!\r\n"));
      return 0;
      }
#endif

  UTIL.memalloc.end_malloc += size_quantum - UTIL.memalloc.malloc_array[j].size;
  UTIL.memalloc.malloc_array[j].size = size_quantum;

  update_limit (UTIL.memalloc.end_malloc, stack_safety);

  return ptr;
  }

/*
 * reset_8042() is used at init if startup with keyboard locked,
 * and when trying to open A20:
 */

UTIL_FCT_PREFIX(flush_8042) static unsigned
flush_8042 (unsigned timeout)
  {
  unsigned char val;
  unsigned char data __attribute__ ((unused)) = 0;
  unsigned counter = timeout, nb_read = 0;

  A20DBG ((" {%s: ", __FUNCTION__));

  do {
      val = inb (0x64);	/* 1 ms per inb/outb */
      if (val & 1) {
	  short_delay (1000);
	  data = inb (0x60);	/* so 2 ms per loop if val & 1 */
	  nb_read++;
	  }
	else if ((val & 2) == 0)
	  break;
      short_delay (1000);
      } while (--counter);

  if (nb_read)
      A20DBG (("[inb (0x60) read %u times, last 0x%X] ",
		nb_read, data));
  A20DBG (("[inb (0x64) = 0x%X] ", val));

  if (counter == 0) {
      A20DBG (("TIMEOUT with timeout %u, please report!} ", timeout));
      return 1;
      }

  A20DBG (("done in %u loops.} ", timeout - counter));
  return 0;
  }

UTIL_FCT_PREFIX(reset_8042) static void
reset_8042 (unsigned timeout)
  {
  A20DBG ((" [%s with timeout %u:", __FUNCTION__, timeout));
  if (flush_8042 (timeout))
      A20DBG (("[timeout 8042, ignored] "));
  outb (0x64, 0xAD);	/* disable keyboard */
  if (flush_8042 (timeout))
      A20DBG (("[timeout 8042, ignored] "));
  outb (0x64, 0xAE);	/* enable keyboard */
  A20DBG (("done] "));
  }

/**
 ** The processor/environment detection stuff:
 **/

UTIL_FCT_PREFIX(init_processor_str) static inline void
init_processor_str (struct processor_str *proc)
  {
  static const union cpuid_string cpuid_string_array[10] = {
		{ .string = "GenuineIntel" },
		{ .string = "AuthenticAMD" },
		{ .string = "UMC UMC UMC " },
		{ .string = "CyrixInstead" },
		{ .string = "NexGenDriven" },
		{ .string = "CentaurHauls" },
		{ .string = "RiseRiseRise" },
		{ .string = "GenuineTMx86" },
		{ .string = "SiS SiS SiS " },
		{ .string = "Geode by NSC" }
		};
  unsigned char cyrix_enable = 0;

  proc->longname[0] = '\0';
  proc->Dflags = proc->Aflags = proc->Cflags = 0;
  proc->brand = proc_nocpuid;

  if (!is_80486()) {
      proc->family = 3;
      ZDBG (("Processor: 386\r\n"));
      }
    else if (!has_cpuid () && !(cyrix_enable = cpu_is_cyrix())) {
      proc->family = 4;
      ZDBG (("Processor: 486 w/o CPUID\r\n"));
      }
    else {
      unsigned short i;
      union cpuid_string cpuidstr;
      int cpuid_max;

      if (cyrix_enable)
	  cyrix_enable_cpuid(); /* also SLOP bug */

      cpuid_max = cpuid0 (&cpuidstr);
      if ((cpuid_max & 0xFFFFFF00) == 0x00000500) {
	  ZDBG ((" [old CPUID for pre-B0 step Intel P5] "));
	  proc->brand = proc_intel;
	  proc->family = 5;
	  cpuid_max &= 0xFF;
	  }
	else {
	  for (i = 0; i < nbof (cpuid_string_array); i++)
	    if (   cpuidstr.number.str1 == cpuid_string_array[i].number.str1
		&& cpuidstr.number.str2 == cpuid_string_array[i].number.str2
		&& cpuidstr.number.str3 == cpuid_string_array[i].number.str3) {
		proc->brand = proc_intel + i;
		break;
		}
	  if (i == nbof (cpuid_string_array))
	      proc->brand = proc_unknown;
	  }

      ZDBG (("processor: CPUID max 0x%X, brand %u, ",
		cpuid_max, proc->brand));
      if (cpuid_max != 0) {
	  struct cpuid_type_str type;
	  union {
	      unsigned long long data;
	      struct cpuid_flags_str fields;
	      } flags;
	  struct cpuid_brand_str brand;
	  unsigned cpuid_emax, reIdentifyAMD_done = 0;

reIdentifyAMD:
	  type = cpuid1 (&flags.fields, &brand);
	  proc->family = type.family;
	  /* family 0x10: Intel Itanium 2 (IA-64) */
	  if (type.family == 0x0F)
	      proc->family += type.extended_family;
	  proc->Dflags = (unsigned)flags.data;
	  proc->Cflags = (unsigned)(flags.data >> 32);
	  cpuid_emax = cpuid8000 (&cpuidstr);

	  ZDBG (("cpuid_emax 0x%X, ", cpuid_emax));
	  if (cpuid_emax > 0x80000001) {
	      struct cpuid_amd_type_str amd_type __attribute__ ((unused));
	      union {
		  unsigned data;
		  struct cpuid_amd_flags_str fields;
		  } amd_flags;
	      struct cpuid_flags_str Dflags = *(struct cpuid_flags_str *)&proc->Dflags;

	      amd_type = cpuid8001 (&amd_flags.fields);
	      proc->Aflags = amd_flags.data;
	      /* Some AMD processors needs black magic to enable LongMode, i.e. enable SSE */
	      if (   proc->brand == 1 /* AuthenticAMD */
		  && reIdentifyAMD_done++ == 0
		  && Dflags.fpu && Dflags.pse && Dflags.tsc && Dflags.msr
		  && Dflags.pae && Dflags.cx8 && Dflags.pge && Dflags.cmov
		  && Dflags.fxsr && !Dflags.sse && !Dflags.sse2 && !amd_flags.fields.lm ) {
		  wrmsr (0xc0010015, rdmsr (0xc0010015) & ~(1LL << 15)); /* enable SSE */
		  goto reIdentifyAMD;
		  }
	      if (cpuid_emax > 0x80000004) {
		  union cpuid_extended_string withspaces;
		  char *src = withspaces.string, *dst = proc->longname;
		  unsigned trimming_space = 1;

		  cpuid_estring (&withspaces);
		  while ((unsigned)(src - withspaces.string) < sizeof (withspaces.string)-1) {
		      if (trimming_space && *src == ' ')
			  src++;
			else {
			  trimming_space = (*src == ' ');
			  *dst++ = *src++;
			  }
		      }
		  if (trimming_space)
		      dst--;
		  *dst = '\0';
		  }
	      }
	  }
      ZDBG (("family 0x%X, Dflags 0x%X, Aflags 0x%X, Cflags: 0x%X\r\n",
		proc->family, proc->Dflags, proc->Aflags, proc->Cflags));
      }
  }

UTIL_FCT_PREFIX(display_processor) static inline void
display_processor (struct processor_str *processor)
  {
#if !(SETUP & QUITE)
#if 0	/* just space saving stuff... */
  static const char * const brand[] = {
      "NoCpuId",
      "Intel", "AMD", "UMC", "Cyrix", "NexGen",
	  "Centaur", "Rise", "Transmeta", "SiS", "NS",
	  "Unknown"
      };
  static const char *stringptr = brand[processor->brand];
#else
    /* Here we will not have an array of 4 bytes pointers to store
	pointers to string of less than 4 bytes... */
    static const char brand_multistr[] =
	"NoCpuId"	  "\0"
	"Intel"		  "\0"
	"AMD"		  "\0"
	"UMC"		  "\0"
	"Cyrix"		  "\0"
	"NexGen"	  "\0"
	"Centaur"	  "\0"
	"Rise"		  "\0"
	"Transmeta"	  "\0"
	"SiS"		  "\0"
	"NS"		  "\0"
	"Unknown"	  "\0";	/* finishes with two '\0' */
    const char *stringptr = brand_multistr;
    unsigned cpt = processor->brand;
    while (cpt--) { /* zero is first choice */
	while (*stringptr++) /* nothing */;
	if (*stringptr == '\0')
	    break; /* double zero is after last choice */
	}
 #endif

  PRINTF (PROCESSOR_MSG,
	  (processor->longname[0] == '\0')? stringptr : processor->longname,
	  processor->family);
  /* processor.calibrate_rdtsc is per 1/100 seconds */
  if (processor->calibrate_rdtsc != 0)
      PRINTF (AT_FREQUENCY, DIVROUNDUP (processor->calibrate_rdtsc, 10000));
  if ((processor->Dflags & 0x1) == 0)
      PRINT (WITHOUT_FPU);
#endif
  }

UTIL_FCT_PREFIX(detect_processor) static inline void
detect_processor (struct processor_str *processor)
  {bound_stack();{
  init_processor_str (processor);

  if (   processor->family <= 4 /* unknown, 386, 486 */
      && _BIOS_equipment_flags().coprocessor80x87installed)
      processor->Dflags |= 0x1;	/* set the FPU flag */

  /* Microsoft EMM386.EXE v4.49 set this flag but do an invalid
     instruction on the rdtsc assembly instruction... how boring!
     in this config I have the noems parameter, and Dflags=0x1bf,
     and reading CR4 register gives... 0, i.e. TSC enabled!
     so the shit_handler() (on a P120).
     FIXME: processor->Dflags &= ~0x10 ? */
  if (processor->Dflags & 0x10 || processor->Aflags & 0x10) {
      /* Has rdtsc assembly function */
      unsigned nbtick, calibrate_rdtsc;
      unsigned long long rdtsc_start = rdtsc_start; /* inited B4 used */

      calibrate_rdtsc = 0x1000000;
      nbtick = _BIOS_nbtick();
      while (_BIOS_nbtick() == nbtick && --calibrate_rdtsc) ;

#ifdef TREAT_EXCEPTION
      if (BOOT1_DOS_running() && (shit_handler (1, 0) != 0)) {
	  DBG (("Invalid instruction while reading TSC... "
		"(CR4: 0x%X, TSD: %u)\r\n",
			get_CR4().all, get_CR4().bit.TSD));
	  calibrate_rdtsc = 0;
	  }
	else {
#endif
	  rdtsc_start = rdtsc(); /* start it just in case... */
#ifdef TREAT_EXCEPTION
	  if (BOOT1_DOS_running())
	      shit_handler (0, 0);
	  }
#endif

      if (calibrate_rdtsc == 0) {
	  ZDBG (("Ticks are not increased - cannot measure Up freq.\r\n"));
	  processor->calibrate_rdtsc = 0;
	  }
	else if (rdtsc_start == rdtsc()) {
	  ZDBG (("RDTSC is not increased - cannot measure Up freq.\r\n"));
	  processor->calibrate_rdtsc = 0;
	  }
	else {
	  unsigned div = 55;
	  unsigned remainder;

	  nbtick = _BIOS_nbtick();
	  while (_BIOS_nbtick() == nbtick) ;
	  rdtsc_start = rdtsc();
	  nbtick += 10;	/* incremented every 55 ms */
	  while (_BIOS_nbtick() <= nbtick) ; /* 550 ms */
	  /* processor.calibrate_rdtsc is per 1/100 seconds */
#if 1
	  {
	  unsigned rem;
	  calibrate_rdtsc = ull_div_ul (rdtsc() - rdtsc_start, div, &rem);
	  }
#else
	  calibrate_rdtsc = (rdtsc() - rdtsc_start) / div;
#endif
	  processor->calibrate_rdtsc = calibrate_rdtsc;
	  ZDBG (("RDTSC: %u ticks per 1/100 second, ", calibrate_rdtsc));

	  /* disable interrupts to not under-estimate it for disk access: */
	  disable_interrupt();

	  /* I have a problem with a PII 400 MHz - is there a strange
	   * sleeping mode - a performace counter slowing things???
	   * Maybe just alignement - strange anyway: I wait sometimes
	   * approx twice the time requested in short_delay().
	   * The measure is always the same, so that seems right.
	   */

#define NBLOOP	1000000	/* approx 1/10 second */
#if 0
	  asm volatile ("nop");
	  rdtsc_start = rdtsc();
	  short_delay (NBLOOP);
	  rdtsc_start = rdtsc() - rdtsc_start;
#else
	  rdtsc_start = measure_short_delay (NBLOOP);
#endif
	  enable_interrupt();
	  ZDBG (("measure: %u loops in %llu ticks ", NBLOOP, rdtsc_start));
	  /* processor.calibrate_rdtsc is per 1/100 seconds, there was NBLOOP loop */
#if 1
	  processor->calibrate_loopl = ull_div_ul (ul_mul_ul(calibrate_rdtsc, NBLOOP), rdtsc_start, &remainder);
#else
	  {
	  unsigned long long tmp = ul_mul_ul(calibrate_rdtsc, NBLOOP);
	  processor->calibrate_loopl = tmp / (unsigned)rdtsc_start;
	  remainder = tmp % (unsigned)rdtsc_start;
	  }
#endif
	  if (remainder)
	      processor->calibrate_loopl++; /* rounds up */
#undef NBLOOP
	  ZDBG (("short_delay(%u) = 1/100 second ", processor->calibrate_loopl));

#if 0 /* save extra space/volatile asm problem */
	  {
	  extern char addr_aligned[], addr_unaligned[], addr_disabled[];

	  rdtsc_start = rdtsc();
	  aligned_short_delay (processor->calibrate_loopl);
	  rdtsc_start = rdtsc() - rdtsc_start;
	  ZDBG (("i.e. (aligned, addr = 0x%X) %u; ",
			addr_aligned, (unsigned)rdtsc_start));

	  rdtsc_start = rdtsc();
	  unaligned_short_delay (processor->calibrate_loopl);
	  rdtsc_start = rdtsc() - rdtsc_start;
	  ZDBG (("(unaligned, addr = 0x%X): %u; ",
			addr_unaligned, (unsigned)rdtsc_start));

	  disable_interrupt();
	  rdtsc_start = rdtsc();
	  disabled_aligned_short_delay (processor->calibrate_loopl);
	  rdtsc_start = rdtsc() - rdtsc_start;
	  enable_interrupt();
	  ZDBG (("(interrupts disabled, addr = 0x%X): %u.\r\n",
			addr_disabled, (unsigned)rdtsc_start));

	  rdtsc_start = rdtsc();
	  very_short_delay (processor->calibrate_loopl/100);
	  rdtsc_start = rdtsc() - rdtsc_start;
	  ZDBG (("very_short_delay (%u): %u; ",
			processor->calibrate_loopl/100, (unsigned)rdtsc_start));

	  rdtsc_start = rdtsc();
	  aligned_very_short_delay (processor->calibrate_loopl/100);
	  rdtsc_start = rdtsc() - rdtsc_start;
	  ZDBG (("aligned_very_short_delay (%u): %u; ",
			processor->calibrate_loopl/100, (unsigned)rdtsc_start));

	  rdtsc_start = rdtsc();
	  aligned_very_short_delay (processor->calibrate_loopl/100);
	  rdtsc_start = rdtsc() - rdtsc_start;
	  ZDBG (("unaligned_very_short_delay (%u): %u\r\n",
			processor->calibrate_loopl/100, (unsigned)rdtsc_start));
	  }
#else
	  ZDBG (("\r\n"));
#endif
	}
      }
    else
      processor->calibrate_rdtsc = 0;

  display_processor (processor);
  }}

UTIL_FCT_PREFIX(detect_serial_timeout) static inline void
detect_serial_timeout (void)
  {
  unsigned char number, tmp, max;

  max = _BIOS_equipment_flags().nb_serial_port; /* 3 bits */

  for (number = 0; number < max; number++) {
      unsigned short addr = peekw (0x00400000 + (2*number));

      //printf ("serial port %u/%u, addr 0x%X; ", number, max, addr);

      if (addr < 0x80 || addr >= 0xFFF8) {
	  ZDBG (("Warning: %u serial ports by BIOS equipment_flags but %u "
		 "address is 0x%X, max serial set to %u\r\n",
		max, number, addr, number));
	  max = number;
	  }
      }

  PRINTF (U_SERIAL_PORT, max, (max > 1) ? "s" : "", (max > 1) ? "s" : "");

  ZDBG (("Timeout for Serial: "));
  for (number = 0; number < max; number++) {
      tmp = peekb (0x0040007C + number);
      ZDBG (("COM%u at 0x%X: 0x%X", number+1, peekw (0x00400000 + (2*number)), tmp));
      if (tmp == 0 || tmp > 5) {
	  ZDBG ((" set to 1"));
	  pokeb (0x0040007C + number, 1);
	  }
      if (number < max - 1)
	  ZDBG ((", "));
	else
	  ZDBG (("\r\n"));
      }
  }

/**
 ** Memory detection related stuff:
 **/

UTIL_FCT_PREFIX(detect_bios_memory) static inline enum detect_memory_enum {
    mem_E820 = 0, mem_E801, mem_AMI, mem_Phoenix, mem_8800, mem_nvram
    }
detect_bios_memory (unsigned *basemem, unsigned *extendmem)
  {
  unsigned short low16Mb_inKb, high16Mb_in64Kb, tmp;
  unsigned ebdaseg;

  ZDBG (("%s: ", __FUNCTION__));
  *basemem = _BIOS_getBaseMemory ();
  if (*basemem == 0) {
      ZDBG (("[_BIOS_getBaseMemory returns zero, use NVRAM] "));
      *basemem = ((unsigned short)getnvram(0x16) << 8) + getnvram(0x15);
      }
  ZDBG (("basemem %u KB, ", *basemem));
  if (!BOOT1_DOS_running()) {
      ebdaseg = _BIOS_getEBDA_segment();
      ZDBG (("EBDA: at segment 0x%X; ", ebdaseg));
      if (ebdaseg != 0 && *basemem > ebdaseg * 16) {
	  *basemem = ebdaseg * 16;
	  ZDBG ((" [correcting basemem = %u Kb] ", *basemem));
	  }
      }

  *extendmem = 0;
  {
  struct e820map_info info;
  unsigned cont_val = 0;
  unsigned long long start_address = 0x100000;

  ZDBG (("_BIOS_QueryMemoryMap:"));
  while (_BIOS_QueryMemoryMap(&cont_val, sizeof (struct e820map_info), &info) != 0) {
      ZDBG (("\r\n    0x%llX+0x%llX,0x%X", info.base, info.length, (unsigned)info.type));
      if (info.type == MemAvailable) {
	  unsigned infoKBsize = (unsigned)(info.length / 1024);
	  if (info.base == 0 && infoKBsize < *basemem) {
	      ZDBG ((": CORRECTING basemem to %u Kb", infoKBsize));
	      *basemem = infoKBsize;
	      }
	  if (info.base == start_address) {
	      ZDBG ((": extended memory size 0x%llX, i.e. %U Kb", info.length, infoKBsize));
	      *extendmem += infoKBsize; /* concat adjacent sections */
	      start_address = info.base + info.length;
	      }
	  }
      if (cont_val == 0)
	  break;
      }
  }

  if (*extendmem != 0) {
      ZDBG (("\r\n_BIOS_QueryMemoryMap done.\r\n"));
      return mem_E820;
      }

  ZDBG (("\r\n_BIOS_QueryMemoryMap failed, _BIOS_getExtendedE801Memory: "));
  if (   _BIOS_getExtendedE801Memory (&low16Mb_inKb, &high16Mb_in64Kb) == 0
      && low16Mb_inKb != 0) {
      *extendmem = 64U * high16Mb_in64Kb + low16Mb_inKb;
      ZDBG (("OK: low16Mb: %u Kb, high16Mb: %u * 64Kb, total: %u Kb.\r\n",
	low16Mb_inKb, high16Mb_in64Kb, *extendmem));
      return mem_E801;
      }
    else
#if 0
      ZDBG (("failed, try _BIOS_getAMIExtendedMemory: "));
  *extendmem = _BIOS_getAMIExtendedMemory();
  if (*extendmem != 0) {
      ZDBG (("OK: total: %u Kb.\r\n", *extendmem));
      return mem_AMI;
      }
    else
      ZDBG (("failed, try _BIOS_getPhoenixExtendedMemory: "));
  /* _BIOS_getPhoenixExtendedMemory():
   * INT15 AH=0x8A is also defined by the Award Modular BIOS v4.50G
   * reference 02/28/95-UMC-498GP-2C4X6U01-00
   * and not documented: It does just crash here.
   * This BIOS version support E820.
   */
  *extendmem = _BIOS_getPhoenixExtendedMemory();
  if (*extendmem != 0) {
      ZDBG (("OK: total: %u Kb.\r\n", *extendmem));
      return mem_Phoenix;
      }
    else
#endif
      ZDBG (("failed, try _BIOS_getExtendedMemory: "));
  if (_BIOS_getExtendedMemory (&tmp) == 0 && tmp != 0) {
      *extendmem = tmp;
      ZDBG (("OK: total: %u Kb.\r\n", *extendmem));
      return mem_8800;
      }
    else
      ZDBG (("failed, get from NVRAM: "));
  /* also 0x31/0x30:
   * extended memory size found above 1 megabyte during POST
   * Using NVRAM here is far to be compatible...
   */
  tmp = getnvram(0x18);
  *extendmem = (tmp << 8) + getnvram(0x17);
  ZDBG (("OK: total: %u Kb.\r\n", *extendmem));
  return mem_nvram;
  }

UTIL_FCT_PREFIX(check_himem) static inline unsigned
check_himem (struct UTIL_str *util)
  {
#if !(SETUP & BIOS_ONLY)
  ZDBG (("XMS: "));
  util->HIMEM_entrypoint = _XMS_get_entry_point ();
  ZDBG (("(at 0x%X) ", util->HIMEM_entrypoint));
  if ((util->HIMEM_entrypoint >> 16) != 0) { /* segment not null */
      unsigned short revision, flags, largest;
      unsigned short free __attribute__ ((unused));
      unsigned char  error1;

// printf ("HIMEM_entrypoint: 0x%X\r\n", util->HIMEM_entrypoint); // _BIOS_wait (2 * 1000 * 1000);
      util->XMS_version = _XMS_get_version (util->HIMEM_entrypoint, &revision, &flags);
// printf ("HIMEM XMS_version: 0x%X, revision 0x%X, flags 0x%X\r\n", util->XMS_version, revision, flags);
      ZDBG (("v%u.%u, rev 0x%X, HMA %s; ",
		util->XMS_version >> 8, util->XMS_version & 0xFF,
		revision, (flags & 1)? "present" : "abscent"));
      free = _XMS_query_free_memory (util->HIMEM_entrypoint, &largest, &error1);
      ZDBG (("free mem: %u Kb, largest: %u Kb, error: 0x%X; ", free, largest, error1));
// printf ("HIMEM free mem: %u Kb, largest: %u Kb, error: 0x%X; ", free, largest, error1);
      if ((util->XMS_version >> 8) >= 3) {
	  unsigned largest_KB, highest_memory_address;
	  unsigned total_KB_free __attribute__ ((unused));
	  unsigned char error2;

	  total_KB_free = _XMSv3_query_free_memory (util->HIMEM_entrypoint,
			&largest_KB, &highest_memory_address, &error2);
	  ZDBG (("V3: total free: %u Kb, largest: %u Kb, highest address: 0x%X, error: 0x%X, ",
		total_KB_free, largest_KB, highest_memory_address, error2));
	  if (error2 == 0) {
	      ZDBG (("use V3 %u Kb.\r\n", largest_KB));
	      util->XMS_freemem = largest_KB;
	      }
	    else if (error1 == 0) {
	      ZDBG (("V3 error happend, use short value: %u Kb.\r\n",
			largest));
	      util->XMS_freemem = largest;
	      }
	    else {
	      ZDBG (("two error happened, returns HIMEM abscent.\r\n"));
	      util->XMS_freemem = 0;
	      return 1;
	      }
	  }
	else {
	  ZDBG (("old version, use short value: %u Kb.\r\n", largest));
	  util->XMS_freemem = largest;
	  }
      return 0;
      }
    else {
      ZDBG (("invalid address.\r\n"));
      util->XMS_freemem = 0;
      util->HIMEM_entrypoint = 0;
      return 1;
      }
#else
  return 0;
#endif
  }

UTIL_FCT_PREFIX(check_VCPI) static inline void
check_VCPI (struct UTIL_str *util)
  {
#if !(SETUP & BIOS_ONLY)
  unsigned peekl0x67 = peekl (0x67*4);

  ZDBG (("%s: INT 0x67: 0x%X, ", __FUNCTION__, peekl0x67));

  {
  unsigned short handle, errorcode;
  union DOS_device_info devinfo;

  if (DOS_OpenFile ("EMMXXXX0", DOS_open_read_only, &handle) == 0) {
      util->emm_type = EMM_emmx;
      ZDBG (("EMMXXXX0 opened, "));
      }
    else if (DOS_OpenFile ("EMMQXXX0", DOS_open_read_only, &handle) == 0) {
      util->emm_type = EMM_emmq;
      ZDBG (("EMMQXXX0 opened, "));
      }
    else {
      util->emm_type = EMM_none;
      ZDBG (("No EMM file.\r\n"));
      return;
      }
  ZDBG (("get device info: "));
  if (DOS_GetDeviceInfo (handle, &devinfo, &errorcode)) {
      ZDBG (("failed 0x%X, ", errorcode));
      util->emm_type = EMM_none;
      }
    else {
      if (devinfo.character.char_device == 0) {
	  ZDBG (("this is a real file! "));
	  util->emm_type = EMM_none;
	  }
	else {
	  ZDBG (("OK, get output status: "));
	  if (DOS_GetDeviceOutputStatus(handle, &errorcode)) {
	      ZDBG (("failed 0x%X, ", errorcode));
	      util->emm_type = EMM_none;
	      }
	    else if (errorcode != 0xFF) {
	      ZDBG (("is not ready 0%X, ", errorcode));
	      util->emm_type = EMM_none;
	      }
	    else
	      ZDBG (("is ready, "));
	  }
      }
  ZDBG (("closing handle %u: ", handle));
  if (DOS_CloseFile (handle, &errorcode))
      ZDBG (("failed 0x%X, ", errorcode));
    else
      ZDBG (("OK, "));
  if (util->emm_type == EMM_none)
      return;
  }

  if (peekl0x67 == 0) {
      ZDBG (("invalid vector.\r\n"));
      util->emm_type = EMM_none;
      return;
      }

  util->EMM_version = EMM_get_version();
  /* set it like others: */
  util->EMM_version = ((util->EMM_version & 0xF0) << 4)
			 |  (util->EMM_version & 0x0F);
  PRINTF ("EMM-%u.%u, ", util->EMM_version >> 8,
			 util->EMM_version & 0xFF);

  if (util->WIN_version == 0x300) {
      ZDBG (("Window 3.0, do not check VCPI.\r\n"));
      return;
      }
  util->VCPI_version = _VCPI_installation_check();
  if (util->VCPI_version == 0) {
      ZDBG (("invalid version.\r\n"));
      util->emm_type = EMM_none;
      return;
      }
  PRINTF ("VCPI-%u.%u, ", util->VCPI_version >> 8,
			  util->VCPI_version & 0xFF);

  util->VCPI_max_address = _VCPI_get_max_address();
  util->VCPI_nb_free_pages = _VCPI_get_nb_free_pages();
  ZDBG (("VCPI_max_address: 0x%X, _VCPI_get_nb_free_pages: 0x%X\r\n",
	util->VCPI_max_address, util->VCPI_nb_free_pages));
  PRINTF (VCPI_TOTAL_MAXFREE,
		(util->VCPI_max_address + 4096) / 1024,
		util->VCPI_nb_free_pages * 4);
#endif
  }

UTIL_FCT_PREFIX(detect_memory) static inline void
detect_memory (struct UTIL_str *util)
  {
  enum detect_memory_enum returned __attribute__ ((unused));

  returned = detect_bios_memory (&util->basemem, &util->extendmem);

#if !(SETUP & QUITE)
#if 0	/* just space saving stuff... */
  {
  static const char *const array[] = { "E820", "E801", "AMI", "Phoenix",
	  "8800", "nvram" };
  PRINTF (BASE_MEMORY_U_EXTENDED_MEMORY_U_S,
		     util->basemem, util->extendmem, array[returned]);
  }
#else
  {
  /* Here we will not have an array of 4 bytes pointers to store
	pointers to string of less than 4 bytes... */
  static const char method_multistr[] =
      "E820"	    "\0"
      "E801"	    "\0"
      "AMI"	    "\0"
      "Phoenix"	    "\0"
      "8800"	    "\0"
      "nvram"	    "\0";	/* finishes with two '\0' */
  const char *stringptr = method_multistr;
  unsigned cpt = returned;
  while (cpt--) { /* zero is first choice */
      while (*stringptr++) /* nothing */;
      if (*stringptr == '\0')
	  break; /* double zero is after last choice */
      }
  PRINTF (BASE_MEMORY_U_EXTENDED_MEMORY_U_S,
			 util->basemem, util->extendmem, stringptr);
  }
#endif
#endif

#if !(SETUP & BIOS_ONLY)
  if (BOOT1_DOS_running()) {
      if (check_himem (util) == 0)
	  PRINTF (XMS_VERSION_FREE,
			util->XMS_version >> 8,
			util->XMS_version & 0xFF,
			util->XMS_freemem);
      check_VCPI (util);
      }
    else
      util->XMS_freemem = 0;
#endif
  }

UTIL_FCT_PREFIX(detect_environment) void
detect_environment (void)
  {
  detect_processor (&UTIL.processor);

#if !(SETUP & QUITE)
  if (BOOT1_DOS_running()) {
      unsigned short window_version, window_mode;

      if (get_window_version(&window_version, &window_mode)) {
	  UTIL.WIN_version = window_version;
	  PRINTF (RUNNING_ON_WINDOWS_UU, window_version >> 8, window_version & 0xFF);
	  }
      UTIL.DOS_version = DOS_version ();
      PRINTF (RUNNING_ON_DOS_UU, UTIL.DOS_version >> 8, UTIL.DOS_version & 0xFF);
#if 0
      if (_DPMI_installation_check() == 0)
	  PRINTF (" [DPMI available, int0x31: %s] ",
			(_DPMI_int0x31_available() == 0)? "yes" : "no");
#endif
      }
    else {
      /* http://invisiblethings.org/papers/redpill.html */
      gdt_idt_param_t idtptr = {};
      PRINT (RUNNING_ON_BIOS);
      get_idt (&idtptr);
      if (idtptr.base >> 24 == 0xE8)
	  PRINT ("/VirtualPC");
	else if (idtptr.base >> 24 == 0xB0 || idtptr.base >> 24 == 0xC0)
	  PRINT ("/DOSEMU");
	else if (idtptr.base >> 24 == 0xFF)
	  PRINT ("/VMWare");
	else if (idtptr.base != 0)
	  PRINTF ("/IDT@0x%X+0x%X", idtptr.base, idtptr.limit);
//      UTIL.DOS_version = 0;	// inited to BSS
      }

#if 0
  /* No more detection of DOSEMU because the video bug has been corrected,
	note that it seems that interrupt 0xE6 has been used by HP Vectra Vl
	Pentium 166 MHz EISA Phoenix BIOS V4.05.E 4/15/98 (Gujin crashes) */
  {
  unsigned short dosemu_version, dosemu_patchlevel;

  if (_DOSEMU_check (&dosemu_version, &dosemu_patchlevel) == 0)
      PRINTF (", DOSEMU: V%u.%u-%u",
		dosemu_version >> 8,
		dosemu_version & 0xFF, dosemu_patchlevel);
  }
#endif
#endif /* QUITE */

  detect_memory (&UTIL);

#if !(SETUP & QUITE)
  if (BOOT1_COM_port() < 0) {
      if (!BOOT1_DOS_running()) {
	  unsigned char val = inb(0x64);
	  if (val != 0 && val != 0xFF && val & 1) {
	      ZDBG ((" [reset kbd because read data present (inb(0x64) = 0x%X): ", val));
	      reset_8042 (1000000);
	      ZDBG (("now inb(0x64) = 0x%X] ", inb(0x64)));
	      }
	  }
      if (_BIOS_is_extended_keyboard()) {
	  if (_BIOS_is_122key_keyboard()) {
	      UTIL.keyboard = ext122kbd;
	      PRINT (KEYBOARD_122);
	      }
	    else {
	      UTIL.keyboard = extkbd;
	      PRINT (KEYBOARD_EXT);
	      }
	  }
	else {
	  UTIL.keyboard = stdkbd;
	  PRINT (KEYBOARD_STD);
	  }
      {
      const char *kbd_name = kbdname (copy_gujin_param.kbdmap);
      if (kbd_name) {
	  PRINT (" (");
	  PRINT (kbd_name);
	  PRINT (")");
	  }
      }
      }
    else {
      UTIL.keyboard = serialkbd;
      PRINT (KEYBOARD_SERIAL);
      }
  PRINT (", ");
#endif /* QUITE */
  detect_serial_timeout();
  DBG (("sizeof struct UTIL_str %u, sizeof struct memalloc_str %u.\r\n",
	sizeof (struct UTIL_str), sizeof (struct memalloc_str)));

#if 0
  {
  gdt_idt_param_t VCPI_old_idt;
  get_idt (&VCPI_old_idt);
  printf ("\r\nidt.base = 0x%X, idt.limit = 0x%X\r\n",
	  VCPI_old_idt.base, VCPI_old_idt.limit);
  }
  printf ("int 0x8: 0x%X, int 0x10: 0x%X, int 0x13: 0x%X, int 0x21: 0x%X\r\n",
	   peekl (4*0x8), peekl (4*0x10), peekl (4*0x13), peekl (4*0x21));
#endif
  }

/**
 ** Partition hiding stuff, for those who still use O.S.
 ** which need such a s..t:
 **/
#if DISK_SUPPORT & BOOTSECT_PROBE
/*
 * Just a size optimisation here:
 */
#if 0
extern inline unsigned
partition_is_hidden (unsigned char part_type)
  {
  if (   part_type == HiddenDOS_FAT12
      || part_type == HiddenDOS_FAT16
      || part_type == HiddenBIG_DOS_FAT16
      || part_type == HiddenWin95_FAT32
      || part_type == HiddenWin95_FAT32LBA
      || part_type == HiddenWin95_FAT16LBA
//      || part_type == Hidden_NTFS // Am I right here?
      )
      return 1;
    else
      return 0;
  }

extern inline unsigned
partition_can_be_hidden (unsigned char part_type)
  {
  if (   part_type == DOS_FAT12
      || part_type == DOS_FAT16
      || part_type == BIG_DOS_FAT16
      || part_type == Win95_FAT32
      || part_type == Win95_FAT32LBA
      || part_type == Win95_FAT16LBA
//      || part_type == OS2_HPFS_NTFS // Am I right here?
      )
      return 1;
    else
      return 0;
  }
#else
extern inline unsigned
partition_is_hidden (unsigned char part_type)
  {
  static const unsigned char hidden_partition_str[] = {
      HiddenDOS_FAT12,
      HiddenDOS_FAT16, /* < 32 Mb */
      HiddenBIG_DOS_FAT16, HiddenWin95_FAT16LBA,
      HiddenWin95_FAT32, HiddenWin95_FAT32LBA,
      };
  unsigned cpt;

  for (cpt = 0; cpt < nbof (hidden_partition_str); cpt++)
      if (part_type == hidden_partition_str[cpt])
	  return 1;
  return 0;
  }

extern inline unsigned
partition_can_be_hidden (unsigned char part_type)
  {
  if (partition_hide(part_type) == part_type)
      return 0; /* bit already set */
  if (partition_is_hidden (partition_hide(part_type)))
      return 1;
  return 0;
  }
#endif

/*
 * These functions just reads / may write all extended partitions MBRs,
 * and the MBR of each disk, it does not touch data partitions.
 * Those boot sector of extended partitions are in order already
 * because they are in "start" increasing order and each extended
 * partition is self contained.
 * Is it possible to have 2 extended type partition as primaries,
 * and they are inverted... may not work then.
 *
 * This function hides all the partitions before (excluding) the booted
 * partition and unhide all partitions after (including) the
 * booted partition. It also set the "bootable" flags for the
 * booted partition and clear it for all others.
 * It acts on all BIOS hard drives, because DOS only recognise them.
 * It has to treat all drives because the BIOS order is: first
 * all the primary partitions of all drives, then extended partitions;
 * and Win9* can be on an extended partition. Note that MsDos 6.*
 * and before just use the simple INT 0x13 to access the disk, so
 * if it is on an extended partition, it still has to be below
 * the simple BIOS limit.
 *
 * Use BIOS to write so it is a bit safer, unfortunately it may
 * trigger the BIOS "bootsector write protection" for viruses.
 * Note that the DI.param is ordered to have all the BIOS
 * drives in increasing order from BIOS 0x80.
 */
/*
 * <?sgml version="1.0" encoding="strict"?> <paranoid>
 * We really take care here to read/write to disk:
 */

  /*
   * Someone says here that you should never have more
   * than one primary visible partition per disk because
   * of a bug in Microsoft DOS, Win3.*, Win9*, WinMe
   * causing data corruption... Confirm?
   *
   * Defining this, Gujin will not make more than one
   * primary partition visible per disk.
   *
   * The definition of ONLY_ONE_PRIMARY_VISIBLE is perfectly
   * working (no bug never seen) but too complex for me,
   * so I changed the logic of the code.
#define ONLY_ONE_PRIMARY_VISIBLE
   */
#define NEW_ONLY_ONE_PRIMARY_VISIBLE

#if 1
// "memeql (&mbr1, &mbr2, sizeof(mbr1))" should not work because
// mbr1 and mbr2 are in the stack and my "memeql()" is described as
// depending on "extern char memory[]" only, i.e. external memory.
// One day I will _really_ need agregate comparisson:
// struct { int a, b, c; } s1, s2; if (s1 != s2) {...}
union bs_t {
    bootsector_t data;
    unsigned     all[sizeof (bootsector_t) / sizeof (unsigned)];
    };
extern inline unsigned bootsector_equal (const union bs_t *bs1, const union bs_t *bs2)
  {
  unsigned cpt = nbof (bs1->all);
  while (--cpt)
      if (bs1->all[cpt] != bs2->all[cpt])
	  return 0;
  return 1;
  }
#define BOOTSECTOR_T	union bs_t
#define MBR1 mbr1.data
#define BOOTSECTOR_EQUAL(b1,b2)	bootsector_equal (b1, b2)
#else /* test the old behaviour with optimising compiler */
#define BOOTSECTOR_T	bootsector_t
#define MBR1 mbr1
#define BOOTSECTOR_EQUAL(b1,b2)	memeql (b1, b2, sizeof(bootsector_t))
#endif

UTIL_FCT_PREFIX(hide_unhide) static unsigned
hide_unhide (unsigned short *hiding, unsigned index, unsigned just_test,
	     struct diskparam_str *dp, int partition)
  {bound_stack();{
  BOOTSECTOR_T mbr1 = {}, mbr2 = {};
  struct diskparam_str *booted_dp = &DI.param[BOOTWAY.desc[index].disk];
  struct partition_str *booted_part = &booted_dp->partition[BOOTWAY.desc[index].partition];
  unsigned short cpt;
#ifdef ONLY_ONE_PRIMARY_VISIBLE
  signed char part_bootable, part_visible;
#endif

  if (DI.readsector (dp, partition, 0, 1, stack_adr(&mbr1)) != 0) {
      HIDEDBG (("Readsector 1 failed, stop\r\n"));
      return 1;
      }
  if (BOOTSECTOR_EQUAL (&mbr1, &mbr2)) {
      if (!dp->sigAA55) {
	  HIDEDBG (("full blank disk, not even 0x55AA signature, ignore.\r\n"));
	  return 0;
	  }
      HIDEDBG (("Readsector 1 only zeroes, stop\r\n"));
      return 2;
      }
  if (DI.readsector (dp, partition, 0, 1, stack_adr(&mbr2)) != 0) {
      HIDEDBG (("Readsector 2 failed, stop\r\n"));
      return 3;
      }
  if (!BOOTSECTOR_EQUAL (&mbr1, &mbr2)) {
      HIDEDBG (("Readsector 1 different from Readsector 2, stop\r\n"));
      return 4;
      }
  if (MBR1.after.Signature0xAA55 != 0xAA55) {
      /* There is crappy software nearby which adds an unsigned
	 at 0xdc..0xdf before even checking if 0xAA55 is present:
00000d0: 0000 0000 0000 0000 0000 0000 8156 5518  .............VU.
	*/
      if (!dp->sigAA55) {
	  HIDEDBG (("full blank disk, not even 0x55AA signature, ignore.\r\n"));
	  return 0;
	  }
      HIDEDBG (("Sector read not 0xAA55 signature, stop\r\n"));
      return 5;
      }

  HIDEDBG (("Part hiding [%s] disk 0x%X part %d: 0x%X 0x%X, 0x%X, 0x%X -> ",
		just_test? "test" : "write",
		dp->disknb, partition,
		MBR1.after.bootsect_partition[0].system_indicator,
		MBR1.after.bootsect_partition[1].system_indicator,
		MBR1.after.bootsect_partition[2].system_indicator,
		MBR1.after.bootsect_partition[3].system_indicator));

#ifdef ONLY_ONE_PRIMARY_VISIBLE
  part_bootable = part_visible = -1;
#endif
  cpt = nbof (MBR1.after.bootsect_partition);
  while (cpt-- != 0) { /* reverse order */
      struct bootsect_partition_str *bp = &MBR1.after.bootsect_partition[cpt];
      unsigned nb_sector_before;

      if (*hiding) {
	  if (partition_can_be_hidden (bp->system_indicator))
	      bp->system_indicator = partition_hide (bp->system_indicator);
	  }
	else {
	  if (partition_is_hidden (bp->system_indicator))
	      bp->system_indicator = partition_unhide (bp->system_indicator);
#ifdef NEW_ONLY_ONE_PRIMARY_VISIBLE
	  {
	  struct bootsect_partition_str *bp1 = bp;
	  while (++bp1 <= &MBR1.after.bootsect_partition[lastof (MBR1.after.bootsect_partition)])
	      if (partition_can_be_hidden (bp1->system_indicator))
		  bp1->system_indicator = partition_hide (bp1->system_indicator); /* forward, so keep the first one */
	  }
#endif
	  }

#ifdef ONLY_ONE_PRIMARY_VISIBLE
      if (partition_can_be_hidden (bp->system_indicator))
	  part_visible = cpt; /* in reverse, so keep the first one */
#endif

      nb_sector_before = bp->nb_sector_before;
      if (partition >= 0)
	  nb_sector_before += dp->partition[partition].start;

      if (   booted_part->start == nb_sector_before
	  && booted_part->length == bp->nb_sector
	  /* Do also a basic identification of the disk,
	     unfortunately this one is not sufficient...
	     Think: boot MSDOS on disk with IDE interface,
	       C/H/S of the disk may not be the same.
	   */
	  && booted_dp->ideIOadr == dp->ideIOadr
	  && (booted_dp->lba_slave_mask & 0x10) == (dp->lba_slave_mask & 0x10)) {
	  HIDEDBG ((" [booted partition No %u] ", cpt));
	  *hiding += 1;
	  if (bp->indicator == non_bootable) {
	      bp->indicator = bootable;
	      HIDEDBG ((" [set boot flag on %u] ", cpt));
	      }
	    else if (bp->indicator != bootable) {
	      HIDEDBG ((" [bp->indicator = 0x%X!] ", bp->indicator));
	      return 6;
	      }
#ifdef ONLY_ONE_PRIMARY_VISIBLE
	  part_bootable = cpt;
#endif
	  }
	else {
	  if (bp->indicator == bootable) {
	      bp->indicator = non_bootable;
	      HIDEDBG ((" [clear boot flag on %u] ", cpt));
	      }
	    else if (bp->indicator != non_bootable) {
	      HIDEDBG ((" [bp->indicator = 0x%X!] ", bp->indicator));
	      return 7;
	      }
	  }
      }

#ifdef ONLY_ONE_PRIMARY_VISIBLE
  if (partition < 0 && part_visible != -1) {
      if (part_bootable != -1)
	  part_visible = part_bootable; /* keep this one visible */

      cpt = nbof (MBR1.after.bootsect_partition);
      while (cpt-- != 0) { /* reverse order */
	  struct bootsect_partition_str *bp = &MBR1.after.bootsect_partition[cpt];

	  if (partition_can_be_hidden (bp->system_indicator) && cpt != part_visible)
	      bp->system_indicator = partition_hide (bp->system_indicator);
	  }
      }
#endif /* ONLY_ONE_PRIMARY_VISIBLE */

  if (BOOTSECTOR_EQUAL (&mbr1, &mbr2)) {
      HIDEDBG (("unchanged.\r\n"));
      }
    else {
#ifdef WRITE_ENABLE
      BOOTSECTOR_T mbr3 = {};
      unsigned returned;

      HIDEDBG (("0x%X 0x%X, 0x%X, 0x%X.\r\n",
		MBR1.after.bootsect_partition[0].system_indicator,
		MBR1.after.bootsect_partition[1].system_indicator,
		MBR1.after.bootsect_partition[2].system_indicator,
		MBR1.after.bootsect_partition[3].system_indicator));
      if (just_test)
	  return 0;
#if __GNUC__ == 4 && __GNUC_MINOR__ == 4 && __GNUC_PATCHLEVEL__ == 0
//      asm ("" : : : "memory"); /* stops "movl DI.writesector, %edx ; call *%edx" */
      asm ("" : "=m" (DI.writesector)); /* stops "movl DI.writesector, %edx ; call *%edx" */
#endif
      returned = DI.writesector (dp, partition, 0, 1, stack_adr(&mbr1));
      if (returned != 0)
	  HIDEDBG (("Failure writing disk 0x%X for hide/unhide partition, ",
			dp->disknb));
	else
	  HIDEDBG (("writing disk 0x%X for hide/unhide partition OK, ",
			dp->disknb));
      if (DI.readsector (dp, partition, 0, 1, stack_adr(&mbr3)) != 0) {
	  HIDEDBG (("Readsector 3 failed, stop\r\n"));
	  return 10 + returned;
	  }
	else {
	  if (BOOTSECTOR_EQUAL (&mbr1, &mbr3))
	      HIDEDBG (("correct data written, OK.\r\n"));
	    else if (BOOTSECTOR_EQUAL (&mbr2, &mbr3)) {
	      HIDEDBG (("data on disk has not been modified, bad.\r\n"));
	      return 20 + returned;
	      }
	    else {
	      HIDEDBG (("contains unknown data, call the fireman now.\r\n"));
	      return 30 + returned;
	      }
	  }
#else
      HIDEDBG (("0x%X 0x%X, 0x%X, 0x%X.\r\n",
		MBR1.after.bootsect_partition[0].system_indicator,
		MBR1.after.bootsect_partition[1].system_indicator,
		MBR1.after.bootsect_partition[2].system_indicator,
		MBR1.after.bootsect_partition[3].system_indicator));
      HIDEDBG (("WRITE_ENABLE undefined, cannot update disk!\r\n"));
      return 0xEE;
#endif
      }
  return 0;
  }}

UTIL_FCT_PREFIX(hide_unhide_all) static inline unsigned
hide_unhide_all (unsigned index, unsigned just_test)
  {
  unsigned short cptdsk, hiding = 0;
  int nbBiosHd = 0, firstBiosHd = -1;

  for (cptdsk = 0; cptdsk < DI.nbdisk; cptdsk++) {
      if (DI.param[cptdsk].disknb >= 0x80) { /* IDE or ATAPI: it is 0 or 1 for master/slave */
	  if (firstBiosHd == -1)
	      firstBiosHd = cptdsk;
	  nbBiosHd++;
	  }
      }

  /* Treat only extended partitions in each disk: (reverse order) */
  cptdsk = nbBiosHd;
  while (cptdsk-- != 0) {
      unsigned short cptpart = DI.param[firstBiosHd + cptdsk].nbpartition;

      while (cptpart-- > 0) {
	  if (DI.param[firstBiosHd + cptdsk].partition[cptpart].misc.active != part_extended)
	      continue;
	  if (hide_unhide (&hiding, index, just_test, &DI.param[firstBiosHd + cptdsk], cptpart) != 0) {
	      return just_test? 1 : 0x10;
	      }
	  }
      }
  /* Now treat MBRs of each disk: */
  cptdsk = nbBiosHd;
  while (cptdsk-- != 0) {
      if (hide_unhide (&hiding, index, just_test, &DI.param[firstBiosHd + cptdsk], -1) != 0) {
	  if (cptdsk != 0)
	      return just_test? 2 : 0x20;	/* bootsector protection */
	    else
	      return just_test? 3 : 0x30;
	  }
      }
  if (hiding != 1) {
      HIDEDBG (("Problem finding the booted partition (%u): hiding = %u\r\n",
		DI.param[BOOTWAY.desc[index].disk]
			.partition[BOOTWAY.desc[index].partition].start,
		hiding));
      return just_test? 4 : 0x40;
      }
  return 0;
  }

UTIL_FCT_PREFIX(hide_unhide_partitions) unsigned
hide_unhide_partitions (unsigned index)
  {bound_stack();{
  /* First try in read-only mode, if we have an error do not even think
     of modifying any partition descriptor; then try in read/write mode */
  unsigned short cpt = 2;
  unsigned returned;

  while (cpt-- != 0) {
      if ((returned = hide_unhide_all (index, cpt)) != 0) {
	  if (cpt == 1)
	      HIDEDBG (("hide_unhide_all(read-only) -> %u\r\n", returned));
	    else
	      HIDEDBG (("hide_unhide_all(read-write) -> %u\r\n", returned));
	  return returned;
	  }
      }
  HIDEDBG (("successfull end hide/unhide.\r\n"));
  return 0;
  }}
/* </paranoid> */
#endif /* DISK_SUPPORT & BOOTSECT_PROBE */

//#define VERBOSE_LOG
enum access_e { access_ERROR, read_BIOS, write_BIOS, read_EBIOS, write_EBIOS,
    read_IDEchs, write_IDEchs, read_IDElba, write_IDElba, read_IDElba48, write_IDElba48 };
const char *const access_string[] = {
    "access_ERROR", "read_BIOS", "write_BIOS", "read_EBIOS", "write EBIOS",
    "read_IDEchs", "write_IDEchs", "read_IDElba", "write_IDElba", "read_IDElba48", "write_IDElba48"
    };
UTIL_FCT_PREFIX(analyse_bootloader2_cmd) static enum access_e
analyse_bootloader2_cmd (const struct diskparam_str *dp, const union diskcmd *cmd,
		unsigned long long *lba, unsigned *nb, const bootbefore_t *bootbefore)
  {
  if ((cmd->int1302.reg_ax >> 8) == 0x42 || (cmd->int1302.reg_ax >> 8) == 0x43){
      *lba = cmd->int1342.lba;
      *nb = cmd->int1342.nb_block;
//      printf ("read EBIOS %u sectors at 0x%llX\r\n", *nb, *lba);
      return (cmd->int1302.reg_ax >> 8) == 0x42 ? read_EBIOS : write_EBIOS;
      }
    else if ((cmd->int1302.reg_ax >> 8) == 0x02 || (cmd->int1302.reg_ax >> 8) == 0x03) {
      unsigned C = 4 * (cmd->int1302.reg_cx & 0xC0) + (cmd->int1302.reg_cx >> 8),
		H = cmd->int1302.reg_dx >> 8,
		S = cmd->int1302.reg_cx & 0x3F;
      *lba = chs2lba (C, H, S, bootbefore->part1.NbHead, bootbefore->part1.NbSectorpertrack);
      *nb = cmd->int1302.reg_ax & 0xFF;
//      printf ("read BIOS %u sectors at C/H/S %u/%u/%u, H0/C0 %u/%u, i.e. lba %llu\r\n",
//		*nb, C, H, S, bootbefore->part1.NbHead, bootbefore->part1.NbSectorpertrack, *lba);
      return (cmd->int1302.reg_ax >> 8) == 0x02 ? read_BIOS : write_BIOS;
      }
    else if ((cmd->int1302.reg_ax >> 8) != 0xF0
		|| (   cmd->hardide.ide_command != 0x20 && cmd->hardide.ide_command != 0x24
		    && cmd->hardide.ide_command != 0x30 && cmd->hardide.ide_command != 0x34)) {
      *lba = 0;
      *nb = 0;
//      printf ("NOT an IDE command to read (0x%X,0x%X)?\r\n", cmd->int1302.reg_ax >> 8, cmd->hardide.ide_command);
      return access_ERROR;
      }
    else if ((cmd->hardide.ide_command == 0x20 || cmd->hardide.ide_command == 0x30) && (cmd->hardide.lba_head & 0x40) == 0) { /* LBA bit */
      unsigned C = 256 * cmd->hardide.cylinder_high + cmd->hardide.cylinder_low,
		H = cmd->hardide.lba_head & 0x0F,
		S = cmd->hardide.sector;
      *lba = chs2lba (C, H, S, dp->nbhead, dp->nbsectorpertrack);
      *nb = cmd->hardide.nb_sect;
//      printf ("read IDECHS %u sectors at C/H/S %u/%u/%u, H0/C0 %u/%u, i.e. lba %llu\r\n",
//		*nb, C, H, S, dp->nbhead, dp->nbsectorpertrack, *lba);
      return cmd->hardide.ide_command == 0x20 ? read_IDEchs : write_IDEchs;
      }
    else {
      if (cmd->hardide.ide_command == 0x24 || cmd->hardide.ide_command == 0x34) {
	  *nb = 256 * cmd->hardide.lba48_nb_sect + cmd->hardide.sector;
	  *lba |= (unsigned long long)cmd->hardide.lba48_cylinder_high << 40;
	  *lba |= (unsigned long long)cmd->hardide.lba48_cylinder_low << 32;
	  *lba |= (unsigned)cmd->hardide.lba48_sector << 24;
	  *lba |= (unsigned)cmd->hardide.cylinder_high << 16;
	  *lba |= (unsigned)cmd->hardide.cylinder_low << 8;
	  *lba |= (unsigned)cmd->hardide.sector;
//	  printf ("read IDELBA48 %u sectors at lba %llu\r\n", *nb, *lba);
	  return cmd->hardide.ide_command == 0x24 ? read_IDElba48 : write_IDElba48;
	  }
	else {
	  *nb = cmd->hardide.sector;
	  *lba |= (unsigned)(cmd->hardide.lba_head & 0x0F) << 24;
	  *lba |= (unsigned)cmd->hardide.cylinder_high << 16;
	  *lba |= (unsigned)cmd->hardide.cylinder_low << 8;
	  *lba |= (unsigned)cmd->hardide.sector;
//	  printf ("read IDELBA %u sectors at %llu\r\n", *nb, *lba);
	  return cmd->hardide.ide_command != 0x20 ? read_IDElba : write_IDElba;
	  }
      }
  }

UTIL_FCT_PREFIX(load_gujin_sector) static unsigned
load_gujin_sector (struct diskparam_str *dp, unsigned sector, farptr buffer_fptr, unsigned nbsectors,
			const bootloader2_t *bootchain, const boot1param_t *boot1param, const bootbefore_t *bootbefore)
{
  unsigned nb, gujin_sector = 1; /* MBR already there */
  const bootloader2_t *bootchain_ptr = &boot1param->bootloader2;
  unsigned long long lba;
#ifdef VERBOSE_LOG
  unsigned cpt_bootchain = 0;
#endif

  do {
#ifdef VERBOSE_LOG
      unsigned tmp =
#endif
      analyse_bootloader2_cmd (dp, &bootchain_ptr->bootloader2_cmd, &lba, &nb, bootbefore);
#ifdef VERBOSE_LOG
      printf ("%s %u sectors at lba %llu\r\n", access_string[tmp], nb, lba);
#endif
      if (sector >= gujin_sector && sector < gujin_sector + nb) {
	  unsigned long long searched_lba = lba + sector - gujin_sector;
	  unsigned nb_available = gujin_sector + nb - sector,
			nb_loaded = (nbsectors <= nb_available)? nbsectors : nb_available;
#ifdef VERBOSE_LOG
	  printf ("Read %u sectors at %llu:\r\n", nb_loaded, searched_lba);
#endif
	  if (DI.readsector (dp, -1, searched_lba, nb_loaded, buffer_fptr) != 0) {
	      printf ("Read %u sectors at %llu failed!\r\n", nb_loaded, searched_lba);
	      return -1;
	      }
	  nbsectors -= nb_loaded;
	  if (nbsectors == 0)
	      return 0;
	  sector += nb_loaded;
	  buffer_fptr += nb_loaded * bootbefore->part1.Bytepersector;
	  }
      gujin_sector += nb;
      bootchain_ptr = (const bootloader2_t *)(fourKbuffer + bootchain_ptr->header.next);
#ifdef VERBOSE_LOG
      if (bootchain_ptr != &bootchain[cpt_bootchain])
	  printf ("Unusual link: bootchain_ptr = 0x%X, &bootchain[%u] = 0x%X\r\n", bootchain_ptr, cpt_bootchain, &bootchain[cpt_bootchain]);
      cpt_bootchain++;
#endif
      } while (nb != 0);
  return -2;
}

UTIL_FCT_PREFIX(get_remote_uninstall_lba) unsigned long long
get_remote_uninstall_lba (struct diskparam_str *dp, const bootbefore_t *bootbefore)
  {
  unsigned long long lba = 0;
  unsigned nb, uninstall_offset;
  // Misdesign: boot1param address is described in instboot_info, but we do not have ((unsigned long)fourKbuffer + 0x200) here...
  const boot1param_t *boot1param = (boot1param_t *)&fourKbuffer[0x19A];
  const bootloader2_t *bootchain = (bootloader2_t *)(fourKbuffer + boot1param->bootloader2.header.next);

#ifdef VERBOSE_LOG
  printf ("Signaturebyte0x29: 0x%X, NbHead %u, NbSectorpertrack %u, NbHiddensector: %u,\r\n",
	bootbefore->part2.Signaturebyte0x29, bootbefore->part1.NbHead, bootbefore->part1.NbSectorpertrack, bootbefore->part1.NbHiddensector);
  printf ("NbTotalsector: %u, NbTotalsector2: %u, String '%s', MBRsig 0x%X, bootchain at 0x%X\r\n",
	bootbefore->part1.NbTotalsector, bootbefore->part1.NbTotalsector2, bootbefore->part1.String, ((unsigned short *)fourKbuffer)[255], bootchain);

  printf ("boot1_checksum 0x%X, next 0x%X checksum 0x%X nbword %u, reg_dx/ide_dcr_address 0x%X reg_ax 0x%X\r\n",
	boot1param->boot1_checksum,
	boot1param->bootloader2.header.next, boot1param->bootloader2.header.checksum, boot1param->bootloader2.header.nbword,
	boot1param->bootloader2.bootloader2_cmd.int1302.reg_dx, boot1param->bootloader2.bootloader2_cmd.int1302.reg_ax);
#endif

  if (load_gujin_sector (dp, 1, data_adr(fourKbuffer + bootbefore->part1.Bytepersector),
		sizeof(fourKbuffer) / 2 / bootbefore->part1.Bytepersector - 1, bootchain, boot1param, bootbefore)) {
      puts ("Error loading bootchain sectors");
      return 0;
      }

#ifdef VERBOSE_LOG
  {
  unsigned cpt_bootchain = 0;
  for (;;) {
      unsigned tmp = analyse_bootloader2_cmd (dp, &bootchain[cpt_bootchain++].bootloader2_cmd, &lba, &nb, bootbefore);
      if (nb == 0)
	  break;
      printf ("%s %u sectors at lba %llu; ", access_string[tmp], nb, lba);
      }
  puts ("");
  }
#endif

  const instboot_info_t **instboot_info_ptr_ptr = (const instboot_info_t **)(fourKbuffer + 0x200);
  unsigned instboot_info_ptr = *(unsigned *)instboot_info_ptr_ptr;
  unsigned char *const buffer = fourKbuffer + sizeof(fourKbuffer) / 2;
  const unsigned bufsize = sizeof(fourKbuffer) / 2;

  if (load_gujin_sector (dp, instboot_info_ptr / bootbefore->part1.Bytepersector, data_adr(buffer),
		bufsize / bootbefore->part1.Bytepersector, bootchain, boot1param, bootbefore)) {
      puts ("Error loading instboot_info sector");
      return 0;
      }
  {
  instboot_info_t *instboot_info = (instboot_info_t *)(buffer + instboot_info_ptr % bootbefore->part1.Bytepersector);

#ifdef VERBOSE_LOG
  printf ("instboot_info at 0x%X, checksum_start_adr %u, boot1param_adr %u, bootend_adr %u, uninstall_mbr_adr %u\r\n",
	instboot_info, instboot_info->checksum_start_adr, instboot_info->boot1param_adr, instboot_info->bootend_adr, instboot_info->uninstall_mbr_adr);
  printf ("bootchain_adr %u, bootchain_nbblock %u, gujin_param_adr %u, code_start_adr %u\r\n",
	instboot_info->bootchain_adr, instboot_info->bootchain_nbblock, instboot_info->gujin_param_adr, instboot_info->code_start_adr);
  printf ("patch2adr %u, patch3adr %u, patch4adr %u, sbss 0x%X\r\n",
	instboot_info->patch2adr, instboot_info->patch3adr, instboot_info->patch4adr, instboot_info->sbss);
  printf ("edata 0x%X, deltaseg 0x%X, diskcodeseg 0x%X, fscodeseg 0x%X, loadcodeseg 0x%X, usercodeseg 0x%X, xdataseg 0x%X, xstring1seg 0x%X, xstring2seg 0x%X\r\n",
	instboot_info->edata, instboot_info->deltaseg, instboot_info->diskcodeseg, instboot_info->fscodeseg, instboot_info->loadcodeseg, instboot_info->usercodeseg, instboot_info->xdataseg, instboot_info->xstring1seg, instboot_info->xstring2seg);
  printf ("password_serial_adr %u, serialconf_port_plus_1_adr %d, read_disk_adr %u, partition_relative_read_disk_adr %u\r\n",
	instboot_info->password_serial_adr, instboot_info->serialconf_port_plus_1_adr, instboot_info->read_disk_adr, instboot_info->partition_relative_read_disk_adr);
  printf ("read_disk_1st_adr %u, usbhdd_patch1 %u, usbhdd_patch2 %u, singlesectorload_patch1 %u, singlesectorload_patch2 %u\r\n",
	instboot_info->read_disk_1st_adr, instboot_info->usbhdd_patch1, instboot_info->usbhdd_patch2, instboot_info->singlesectorload_patch1, instboot_info->singlesectorload_patch2);
#endif
  uninstall_offset = instboot_info->uninstall_mbr_adr;
  }

  if (load_gujin_sector (dp, uninstall_offset / bootbefore->part1.Bytepersector, data_adr(buffer),
		bufsize / bootbefore->part1.Bytepersector, bootchain, boot1param, bootbefore)) {
      puts ("Error loading uninstall_mbr sector");
      return 0;
      }
  {
  const bootloader2_t *uninstall_mbr = (const bootloader2_t *)(buffer + uninstall_offset % bootbefore->part1.Bytepersector);
#ifdef VERBOSE_LOG
  unsigned long long another_lba;
  unsigned tmp;

  puts ("Uninstall MBR (where to read, where to read current partition table, where to write):");
  tmp = analyse_bootloader2_cmd (dp, &uninstall_mbr[0].bootloader2_cmd, &lba, &nb, bootbefore);
  printf ("%s %u sectors at lba %llu\r\n", access_string[tmp], nb, lba);
  tmp = analyse_bootloader2_cmd (dp, &uninstall_mbr[1].bootloader2_cmd, &another_lba, &nb, bootbefore);
  printf ("%s %u sectors at lba %llu\r\n", access_string[tmp], nb, another_lba);
  tmp = analyse_bootloader2_cmd (dp, &uninstall_mbr[2].bootloader2_cmd, &another_lba, &nb, bootbefore);
  printf ("%s %u sectors at lba %llu\r\n", access_string[tmp], nb, another_lba);
#else
  analyse_bootloader2_cmd (dp, &uninstall_mbr[0].bootloader2_cmd, &lba, &nb, bootbefore);
#endif

  if (uninstall_mbr[0].header.nbword == 0)
      puts (" No uninstall MBR present.");
    else if (DI.readsector (dp, -1, lba, 1, data_adr(fourKbuffer)) != 0)
      printf ("failed reading old MBR lba at 0x%llX!\r\n", lba);
    else
      printf ("old MBR at lba %llX is %s\r\n", lba, recognise_MBR_string[recognise_MBR((const bootsector_t *)fourKbuffer)]);
  }

  return lba;
  }

#ifdef COMMANDFILE
UTIL_FCT_PREFIX(file_match) static const char *
file_match (char separator, const char *ptr, const char *end, const char *label, int len)
  {
  if (*ptr == separator)
      return ++ptr;
  if (len == 0) {
      while (*ptr != separator && ptr < end)
	  if (*ptr++ == '\n')
	      return 0;
      return ++ptr;
      }
  while (len--)
      if (*ptr++ != *label++)
	  break;
  if (len < 0 && *ptr == separator) {
      if (*label == '\0')
	  return ++ptr;
	else
	  return 0;
      }
  if (ptr[-1] == '*' || (len < 0 && *ptr == '*')) {
      while (*ptr != separator && ptr < end)
	  if (*ptr++ == '\n')
	      return 0;
      return ++ptr;
      }
  return 0;
  }

UTIL_FCT_PREFIX(commandfile_get) unsigned
commandfile_get (const char *filesystemlabel, const char *kernelname, const char *initrdname,
	enum commandfile_cmd cmd, unsigned iteration, char *buffer, unsigned bufsize)
  {bound_stack();{
  /* Format of a line (note the first char is the delimiter for that line, one of "!$%&+/:=?@|\t" excluding quotes)(ends with "\r\n" or "\n"):
	:filesystemlabel:kernelname:initrdname::::timeout:menuname:commandline
      There is a set of "::::" in the middle to have an expendable system, only the first, second, last, last-1 and last-2 parameter are used,
	so there should be a minimum of 6 fields.
      The last char of the first 3 fields can be a star to describe any ending.
      where:
	filesystemlabel: label of the filesystem (as displayed by Gujin) where this configuration is active / when empty no restriction
	kernelname: file name of the kernel, without pathname / when empty all kernels on this filesystemlabel
	initrdname: file name of the initial ram disk, without pathname. has to begin with "initrd" or "initra"(mfs) / when empty Gujin default is used
			That initrdname is still unused and should be given as an empty field.
	timeout: the timeout in seconds before running that kernel (minimum 1, smallest timeout found wins)
	menuname: the string displayed in Gujin menu / when empty Gujin default is displayed (UTF-8)
	commandline: the command line when this selection is chosen
      Comments start with '#' and finish at end of the line. Comments and empty lines are not stored so do not reduce memory available.
      Lines with another format (not beginning with "!$%&+/:=?@|") are stored but skiped by commandfile_get() function.
      If the first 3 fields are identical in two different lines, the last two fields should be different (generate two menu lines).

      ADD Gujin-v2.8.2: if the last char of the line is the delimiter (first char == last char), the command line is not to
      be modified at all by Gujin when loading that kernel.
   */
  const char *ptr = UTIL.cmdfile_base, *end = &UTIL.cmdfile_base[UTIL.cmdfile_len], *initrdendptr = 0;
  unsigned counter = 0, separator_count = 0;
  unsigned char skip2eol = 0, treat_that_line = 0;
  char separator = '\0';
  unsigned filesystemlabel_len = 0, kernelname_len = 0, initrdname_len = 0;

  if (filesystemlabel)
      for (filesystemlabel_len = 0; filesystemlabel[filesystemlabel_len] != 0; filesystemlabel_len++)
	  continue;
  if (kernelname)
      for (kernelname_len = 0; kernelname[kernelname_len] != 0; kernelname_len++)
	  if (kernelname[kernelname_len] == '/') {
	      kernelname += kernelname_len + 1;
	      kernelname_len = (unsigned)-1;
	      }
  if (initrdname)
      for (initrdname_len = 0; initrdname[initrdname_len] != 0; initrdname_len++)
	  if (initrdname[initrdname_len] == '/') {
	      initrdname += initrdname_len + 1;
	      initrdname_len = (unsigned)-1;
	      }

  if (cmd == cmdf_getnb)
      iteration = 0xFFFFFFFF; /* will return nb lines matching pattern */

  if (bufsize)
      *buffer = '\0';

  while (ptr < end || (ptr == end && treat_that_line)) {
      if (*ptr == separator)
	  separator_count++;
      if (skip2eol) {
	  if (*ptr != '\n' && ptr < end) {
	      ptr++;
	      continue;
	      }
	  if (!treat_that_line) {
	      separator_count = 0;
	      treat_that_line = separator = skip2eol = 0;
	      ptr++;
	      continue;
	      }
	  if (ptr == end)
	      ptr++;
	  unsigned returned_mask = 0;
	  if (ptr[-1] == separator) {
	      separator_count--; /* ignore possible last separator in the count */
	      returned_mask = 0x80000000U; /* flags "do not modify cmdlline" */
	      ptr--;
	      }
	  switch (cmd) {
	    case cmdf_getnb:
	      /* handled in switch */
	      break;
	    case cmdf_getinitrd:
	      if (bufsize == 0)
		  return 0;
	      if (!initrdendptr) {
		  separator_count = 0;
		  treat_that_line = separator = skip2eol = 0;
		  continue;
		  }
	      returned_mask = 0;
	      ptr = initrdendptr;
	      goto common_string;
	    case cmdf_getcmdline:
	      if (separator_count < 3 || bufsize == 0)
		  return 0;
common_string: {
	      const char *lastchar = ptr;
	      while (*--ptr != separator)
		  continue;
	      while (++ptr < lastchar && --bufsize)
		  *buffer++ = *ptr;
	      }
	      *buffer = '\0';
	      return returned_mask | (unsigned)buffer;
	    case cmdf_getmenuname:
	      if (separator_count < 4 || bufsize == 0)
		  return 0;
	      while (*--ptr != separator)
		  continue;
	      returned_mask = 0;
	      goto common_string;
	    case cmdf_gettimeout:
	      if (separator_count < 5)
		  return 0;
	      while (*--ptr != separator)
		  continue;
	      while (*--ptr != separator)
		  continue;
	      const char *lastdigit = ptr;
	      while (*--ptr != separator)
		  continue;
	      if (ptr[1] == '0' && (ptr[2] == 'x' || ptr[2] == 'X')) {
		  unsigned number = 0;
		  ptr += 3;
		  while (ptr < lastdigit)
		      if (*ptr >= '0' && *ptr <= '9')
			  number = 16 * number + (unsigned)*ptr++ - '0';
			else if (*ptr >= 'a' && *ptr <= 'f')
			  number = 16 * number + (unsigned)*ptr++ - 'a' + 10;
			else if (*ptr >= 'A' && *ptr <= 'F')
			  number = 16 * number + (unsigned)*ptr++ - 'A' + 10;
			else
			  return 0;
		  return number;
		  }
		else {
		  unsigned number = 0;
		  ptr += 1;
		  while (ptr < lastdigit)
		      if (*ptr < '0' || *ptr > '9')
			  return 0;
			else
			  number = 10 * number + (unsigned)*ptr++ - '0';
		  return number;
		  }
	      return 0;
	      }
	  }

      if (separator == '\0') {
//	  if (strchr ("!\"#$%&'()*+,-./:;<=>?@[\\]^_`{|}~", *ptr)) {
//	  if (strchr ("!$%&+/:=?@|", *ptr))
	  const char *xptr;
	  for (xptr = "!$%&+/:=?@|\t"; *xptr && *xptr != *ptr; xptr++)
	      continue;
	  if (*xptr)
	      separator = *ptr++;
	    else
	      skip2eol = 1;
	  continue;
	  }

      skip2eol = 1;
      const char *ptr1 = file_match (separator, ptr, end, filesystemlabel, filesystemlabel_len);
      if (ptr1) {
	  const char *ptr2 = file_match (separator, ptr1, end, kernelname, kernelname_len);
	  if (ptr2) {
	      const char *ptr3 = file_match (separator, ptr2, end, initrdname, initrdname_len);
	      if (ptr3) {
		  counter++;
		  if (iteration-- != 0)
		      continue;
		  initrdendptr = ptr3 - 1;
		  treat_that_line = 1;
		  }
	      }
	  }
      }
  return (cmd == cmdf_getnb)? counter : 0;
  }}
#endif

#if defined (INITRDNAME) || defined (INITRDNAME2)
/*
 * Select the closest match between initrd* and vmlinuz*
 * at the end of filename, returns the index (zero is
 * none because initrd's are at end of list):
 */
UTIL_FCT_PREFIX(initrd_file) unsigned
initrd_file (unsigned index)
  {bound_stack();{
#if (DISK_SUPPORT != DOS_SUPPORT) || (USER_SUPPORT != 0) /* !tiny_exe */
  unsigned short cpt, len, match_len = 0;
  struct desc_str *elem, *vmlin, *match_elem = 0;
  char exact_match[128];
  char match_next = '\0';

  if (index >= BOOTWAY.nb) {
      ZDBG (("ERROR: no initrd of an initrd.\r\n"));
      return 0;
      }

  if (copy_gujin_param.min_nb_initrd >= 9) {
      ZDBG (("%s: copy_gujin_param.min_nb_initrd = %u.\r\n", __FUNCTION__, copy_gujin_param.min_nb_initrd));
      return 0;
      }

  vmlin = &BOOTWAY.desc[index];
  elem = &BOOTWAY.desc[BOOTWAY.nb]; /* first initrd */

//#undef ZDBG
//#define ZDBG(X) printf X
  exact_match[0] = '\0';
// TODO: variant
  commandfile_get (vmlin->iso_fsname ?: (char *)DI.param[vmlin->disk].partition[vmlin->partition].name, vmlin->filename, 0, cmdf_getinitrd, 0, exact_match, sizeof(exact_match));
  unsigned strlen_exact_match = strlen(exact_match);

  if (strlen_exact_match)
      ZDBG (("%s of: '%s' want exact match '%s' of length %u ", __FUNCTION__, vmlin->filename, exact_match, strlen_exact_match));
    else
      ZDBG (("%s of: '%s' vm_ptr: %s compared to ", __FUNCTION__, vmlin->filename, &vmlin->filename[vmlin->name_offset]));

  for (cpt = 0; cpt < BOOTWAY.nb_initrd; cpt++, elem++) {
      if (   elem->disk == vmlin->disk
	  && elem->partition == vmlin->partition
	  && elem->inRoot == vmlin->inRoot
	  && elem->inISO == vmlin->inISO) {
	  if (vmlin->inISO) {
	      /* Check that they are in the same ISO image, compare name up to ':' */
	      unsigned idx = 0;
	      for (;;) {
		  if (vmlin->filename[idx] == '\0' || elem->filename[idx] == '\0')
		      break;
		  if (vmlin->filename[idx] == ':' && elem->filename[idx] == ':')
		      break;
		  if (vmlin->filename[idx] != elem->filename[idx])
		      break;
		  idx++;
		  }
	      if (vmlin->filename[idx] == ':' && elem->filename[idx] == ':')
		  ZDBG (("[same ISO image] "));
		else {
		  ZDBG (("[different ISO image] "));
		  continue;
		  }
	      }
	  if (strlen_exact_match) {
	      if (strlen_exact_match == 1 && exact_match[0] == '*') {
		  ZDBG (("found any match %s for pattern %s.\r\n", elem->filename, exact_match));
		  return elem - BOOTWAY.desc;
		  }
	      char *elem_filename = elem->filename;
	      while (*elem_filename != '\0')
		  elem_filename++;
	      while (*elem_filename != '/')
		  elem_filename--;
	      elem_filename++;
//printf ("exact match comparing '%s' and '%s'\n", elem_filename, exact_match); _BIOS_getkey();
	      if ((exact_match[strlen_exact_match - 1] == '*')
			? !strncmp(exact_match, elem_filename, strlen_exact_match-1)
			: !strcmp(exact_match, elem_filename)) {
		  ZDBG (("found exact match %s for pattern %s.\r\n", elem->filename, exact_match));
//printf ("exact match found '%s' and '%s'\n", elem_filename, exact_match); _BIOS_getkey();
		  return elem - BOOTWAY.desc;
		  }
	      continue;
	      }
	  char *in_ptr = &elem->filename[elem->name_offset],
	       *vm_ptr = &vmlin->filename[vmlin->name_offset];
	  len = 0;
	  if (*(unsigned *)(in_ptr-1) == *(const unsigned *)"amfs" || *(unsigned *)(in_ptr-1) == *(const unsigned *)"AMFS") /* initra -> initramfs */
	      in_ptr += 3;
	  if (*(unsigned *)in_ptr == *(const unsigned *)".img" || *(unsigned *)in_ptr == *(const unsigned *)".IMG")
	      in_ptr += sizeof(".img") - 1;

	  ZDBG (("\"%s\"", in_ptr));
	  for (;;) {
	      if (*vm_ptr == 0 && *in_ptr == 0) {
		  len = sizeof (BOOTWAY.desc->filename);
		  ZDBG ((" exact match, stop searching "));
		  break;
		  }
	      if (*vm_ptr != *in_ptr || *vm_ptr == 0)
		  break;
	      len++;
	      vm_ptr++;
	      in_ptr++;
	      }
	  ZDBG (("=%u, ", len));
	  if (match_elem != 0 && len < match_len)
	      continue;
	  if (len == match_len) {
	      /* We have, for instance: /boot/vmlinuz-2.4.9-21
		and: /boot/initrd-2.4.9-21.img  (len = 9, start at -)
		and: /boot/initrd-2.4.9-21smp.img (len = 9)
		so we better like '.' than 's' */
	      if (match_next == '.')
		  continue;
	      }
	  match_len = len;
	  match_elem = elem;
	  match_next = *in_ptr;
	  }
      }
  if (match_elem) {
      ZDBG ((" => best match %s, len %u, ",
		 match_elem->filename, match_len));
      /* Shall we still accept "vmlinuz" and "initrd.gz",
	match_len will be zero and copy_gujin_param.min_nb_initrd != 9 here ?
	"vmlinuz" matches any present "initrd*" */
      if (   vmlin->filename[vmlin->name_offset] != '\0'
	  && match_len <= copy_gujin_param.min_nb_initrd) {
	  ZDBG (("not sufficient (min_nb_initrd = %u).\r\n", copy_gujin_param.min_nb_initrd));
	  return 0;
	  }
      ZDBG (("found (min_nb_initrd = %u).\r\n", copy_gujin_param.min_nb_initrd));
      return match_elem - BOOTWAY.desc;
      }
    else {
      ZDBG ((" => no initrd found (min_nb_initrd = %u).\r\n", copy_gujin_param.min_nb_initrd));
      return 0;
      }
#else /* tiny_exe */
  if (BOOTWAY.nb_initrd != 0)
      return BOOTWAY.nb; /* first initrd */
    else
      return 0;
#endif /* tiny_exe */
  }}
//#undef ZDBG
//#define ZDBG(X)

#endif /* INITRDNAME | INITRDNAME2 */

/**
 ** A20 related stuff:
 ** Debugging done with A20DBG to keep ZDBG (i.e. dbgload.exe) linking
 **/

/*
 * This is the address we use to check that A20 is enabled or
 * disabled, by reading/writing at 0x0000:A20_TEST_ADDR and
 * reading at 0xFFFF:A20_TEST_ADDR+0x10.
 */
//#define A20_TEST_ADDR 0x007C
#define A20_TEST_ADDR 0x200

#define A20_TEST_LOW_ADDR	A20_TEST_ADDR
#define A20_TEST_HIGH_ADDR	(0xFFFF0000 + A20_TEST_ADDR + 0x10)

/* If open != 0, check that A20 is open, i.e. no mirror */
/* To be called with timeout >= 1 */
UTIL_FCT_PREFIX(check_mirror) static unsigned
check_mirror (unsigned open, unsigned timeout)
  {
  unsigned char saved, tmp, read;

  tmp = saved = peekb_doorclosed (A20_TEST_LOW_ADDR);
  do {
      tmp = ~tmp;
      pokeb_doorclosed (A20_TEST_LOW_ADDR, tmp);
      read = peekb_doorclosed (A20_TEST_HIGH_ADDR);
      if (open) {
	  if (read != tmp)
	      break;
	  }
	else {
	  if (read == tmp)
	      break;
	  }
      } while (--timeout);

  pokeb_doorclosed (A20_TEST_LOW_ADDR, saved);
  return timeout; /* return 0 if ERROR timeout */
  }

/*
 * The BIOS functions:
 */
UTIL_FCT_PREFIX(BIOS_enable_A20) static inline unsigned
BIOS_enable_A20 (unsigned enabled)
  {
#if 0
  return 1;
#else
  unsigned char can_report, ret;
  farptr A20Handler;

  A20DBG (("%s: get A20 handler: ", __FUNCTION__));
  A20Handler = GetExternalA20Handler(&can_report);

  if (A20Handler == 0)  {
      A20DBG (("no handler.\r\n"));
      return 2;
      }

  A20DBG (("at 0x%X, can_report = %u", A20Handler, can_report));
  if (can_report)
      A20DBG ((" (report %u)", GetA20(A20Handler)));
  A20DBG (("; call 'SetA20' with enabled = %u: ", enabled));
  ret = SetA20 (A20Handler, enabled);
  A20DBG (("returns %u", ret));
  if (ret != 1) {
      A20DBG ((", failed, unable to process.\r\n"));
      return 1;
      }
  if (can_report)
      A20DBG ((" (report %u)", GetA20(A20Handler)));
  A20DBG ((", success.\r\n"));
  return 0;
#endif
  }

UTIL_FCT_PREFIX(PS2_enable_A20) static inline unsigned
PS2_enable_A20 (unsigned enabled, struct PS2_A20_str *switch_type)
  {
#if 0
  return 1;
#else
  unsigned short tmp;

#if DEBUG & DEBUG_LOADER
  unsigned char state = state;	/* inited B4 used */
#endif

  A20DBG (("%s: check support: ", __FUNCTION__));
  tmp = PS2_A20_supported (switch_type);
  A20DBG (("returns 0x%X, bitfield 0x%X ", tmp, *(unsigned short *)switch_type));
  if (   tmp != 0
      || (   !switch_type->kbd_ctrl
	  && !switch_type->IO0x92_bit1)) {
      A20DBG (("error.\r\n"));
      return 2;
      }

  A20DBG (("(getA20 returns 0x%X, ", PS2_get_A20 (&state)));
  A20DBG (("enabled = %u) ", state));

  A20DBG (("call PS2_set_A20 with enabled = %u ", enabled));
  tmp = PS2_set_A20 (enabled);
  A20DBG (("returns 0x%X, ", tmp));

  A20DBG (("(getA20 returns 0x%X, ", PS2_get_A20 (&state)));
  A20DBG (("enabled = %u) ", state));

  if (tmp != 0) {
      A20DBG (("failed, unable to process.\r\n"));
      return 1;
      }
  A20DBG (("success.\r\n"));
  return 0;
#endif
  }

/*
 * The hard functions:
 */

#if (DEBUG & DEBUG_LOADER) && !defined (EXCLUDE_GETA20)
UTIL_FCT_PREFIX(get_A20) static unsigned
get_A20 (unsigned timeout)
  {
  unsigned char val;

  A20DBG (("%s: ", __FUNCTION__));
  if (flush_8042 (500000))
      A20DBG (("[timeout 8042, continuing] "));
  outb (0x64, 0xD0);	/* write to port: read I/O port */
  while (((inb (0x64) & 1) == 0) && --timeout) ;
  if (timeout == 0) {
      A20DBG (("timeout while reading I/O port.\r\n"));
      return 0xFFFFFFFFU;
      }
  val = inb(0x60);
  A20DBG (("inb(0x60) = 0x%X ", val));
  return val & 0x02;
  }
#endif

/*
 * THE A20 (address line 20) oppening or closing, too long
 * a subject to describe here...
 *
 * Note that you can open and close using the keyboard
 * controller and the I/O 0x92, the keyboard controller
 * opening method is prioritary (then I/O 0x92 is ignored).
 *
 * We are trying to do it with BIOS as much as possible,
 * in case the A20 logic is removed from the PC one day,
 * so the BIOS will just always answer "yes, done".
 * Note that most PCs have probably already removed
 * this hardware and replaced it with System Management
 * Mode (and real mode paging?). I did not checked.
 * Also the keyboard logic will go when the PC will only
 * support USB keyboards.
 *
 * himem.sys from Windows98 _need_ to be able to close
 * (and open) the A20 with I/O on 0x92, it will not try
 * to open and close with the KBD controller.
 * WinMe does not seem to have a problem here.
 *
 * If we are already in protected mode (virtual mode after
 * loading EMM386), we should not even think about closing
 * A20 with I/O 0x92, the VCPI server would send us to hell.
 * EMM386 of Win98 do not open A20 with the kbd controller.
 *
 * We do not bother to check if A20 is really disabled
 * if that was the request, we answer then "no error".
 *
 * We have to take care here: if A20 is set with I/O 0x92,
 * and can be set with the keyboard controller but is not
 * currently, we have to send the command to the kbd
 * controller anyway (bug report on Linux with APM)
 *
 * Some PC have a nasty BIOS, and start the boot process
 * with the keyboard controller completely f***ed, and assume
 * that when DOS will load it will load EMM386 and so reset
 * the keyboard controller to open A20 - oh dear...
 *
 * Updated a bit from:
 * http://www.win.tue.nl/~aeb/linux/kbd/A20.html
 */
UTIL_FCT_PREFIX(set_fast_A20) static inline void
set_fast_A20 (void)
  {
  unsigned char val = inb(0x92);
  if ((val & 0x02) == 0) {
      short_delay (10);
      outb (0x92, (val | 0x02) & 0xFE);
      }
  }

UTIL_FCT_PREFIX(set_A20) unsigned
set_A20 (unsigned enabled)
  {
  unsigned char val;
  struct PS2_A20_str switch_type;

  A20DBG (("%s (enabled = %u): ", __FUNCTION__, enabled));
  if (
#if !(SETUP & BIOS_ONLY)
  /* This will crash my PC when "UTIL.emm_type == EMM_emmq_suspended", fix the shit: */
      UTIL.emm_type != EMM_emmq_suspended && UTIL.emm_type != EMM_VCPI_suspended &&
#endif
      (PS2_enable_A20 (enabled, &switch_type) == 0 || BIOS_enable_A20 (enabled) == 0)) {
      if (!enabled) {
	  A20DBG (("set_A20(0) BIOS => check_mirror returns %s\r\n",
		check_mirror (enabled, 500)? "success" : "fail" ));
	  return 0;
	  }
	else if (check_mirror (enabled, 10)) {
	  A20DBG (("set_A20(0) BIOS => OK\r\n"));
	  return 0;
	  }
	else
	  switch_type = (struct PS2_A20_str) { .kbd_ctrl = 1, .IO0x92_bit1 = 1};
      }
  else
    switch_type = (struct PS2_A20_str) { .kbd_ctrl = 1, .IO0x92_bit1 = 1};

  A20DBG ((" (note NVRAM@0x2D = 0x%X bit 2 set when Fast gate A20 operation enabled) ",
	getnvram (0x2D)));

  val = inb(0x64);
  A20DBG (("initial inb(0x64) = 0x%X\r\n", val));
  if (val & 1) {
      A20DBG (("reset kbd because read data present: "));
      reset_8042 (1000);
      val = inb(0x64);
      if ((val & 1) == 0) {
	  A20DBG (("(inb(0x64) = 0x%X), reset seems OK\r\n", val));
	  }
	else {
	  A20DBG (("(inb(0x64) = 0x%X), there is no keyboard here!\r\n", val));
	  if (check_mirror (enabled, 1)) {
	      A20DBG (("initial check_mirror (%u, 1) is success, state not changed\r\n", enabled));
	      return 0;
	      }
#if 0 // to enable when tested on i386SL/i486SL and AMD Elan SC400
	  inb (0xEE);
	  if (check_mirror (enabled, 1)) {
	      A20DBG (("check_mirror (%u, 1) is success after inb (0xEE)\r\n", enabled));
	      return 0;
	      }
#endif
	  if (enabled && switch_type.IO0x92_bit1) {
	      A20DBG (("enabling rescue outb(0x92, 2)\r\n\r\n"));
	      set_fast_A20 ();
	      if (check_mirror (enabled, 1)) {
		  A20DBG (("check_mirror (%u, 1) is success after set_fast_A20()\r\n", enabled));
		  return 0;
		  }
	      }
	  A20DBG (("inb(0xe0) = 0x%X, inb(0x329) = 0x%X, inb(0x65) = 0x%X\r\n",
			inb(0xe0), inb(0x329), inb(0x65)));
	  return 1;
	  }
      }

#if 0
  if (val & 0x10 == 0) {
      A20DBG (("keyboard locked... try to unlock:"));
	/*
	 F4h     sngl    enable keyboard
	 F5h    sngl    disable keyboard. set default parameters (no keyboard scanning)
	 */
      outb (0x64, 0xF4);	    /* write to port */
      flush_8042 (500000);
      val = inb(0x64);
      A20DBG (("reread inb(0x64) = 0x%X\r\n", val));
      }
#endif

  A20DBG (("Read hard A20 1: 0x%X\r\n", get_A20 (1000)));

  {
#if 0
  CR0_t cr0 = {0};

  if (VCPI_ACTIVE()) {
      A20DBG (("VCPI server present and active, get CR0: "));
      cr0 = _VCPI_get_CR0();
      A20DBG (("0x%X, ", cr0.all));
      }
    else {
      A20DBG (("no VCPI or VCPI inactive, directly get CR0: "));
      cr0 = get_CR0();
      A20DBG (("0x%X, ", cr0.all));
      }

  if (cr0.bit.protected_mode) {
#else
  /* The assembly instruction get_CR0() is privileged, but getsw() is not: */
  unsigned short sw = getsw();
  A20DBG (("Checking Virtual86 mode, getsw: 0x%X, ", sw));
  if (sw & 1) {
#endif
      A20DBG (("in virtual mode, disable I/O 0x92 treatment.\r\n"));
      switch_type.IO0x92_bit1 = 0;
      }
    else {
      if (switch_type.IO0x92_bit1) {
	  val = inb(0x92);
	  if (val & 2) {
	      short_delay (10);
	      A20DBG (("First disabling fast A20 using outb (0x92, 0x%X & ~3)\r\n", val));
	      outb (0x92, val & ~3);
	      }
	    else
	      A20DBG (("Fast A20 disabled, OK to continue\r\n"));
	  }

      if (check_mirror (enabled, 1)) {
	  A20DBG (("initial check_mirror (%u, 1) is success, state not changed\r\n", enabled));
	  if (enabled && switch_type.IO0x92_bit1) {
	      A20DBG (("enabling also outb(0x92, 2)\r\n\r\n"));
	      set_fast_A20();
	      }
	  return 0;
	  }
      }
  }

  flush_8042 (500);
  A20DBG (("outb (0x64, 0xD1)\r\n"));
  outb (0x64, 0xD1);		/* write to port */
  flush_8042 (500000); /* White magic, shall no more do that */
/*  short_delay (1000); / * works with 800, doesn't with 500 on a PII/450 */
  if (enabled) {
      A20DBG (("outb (0x60, 0xDF)\r\n"));
      outb (0x60, 0xDF);	/* set A20 active  */
      }
    else {
      A20DBG (("outb (0x60, 0xDD)\r\n"));
      outb (0x60, 0xDD);	/* set A20 inactive */
      }
  flush_8042 (500000);
  A20DBG (("outb (0x64, 0xFF)\r\n"));
  outb (0x64, 0xFF);		/* black magic */
  flush_8042 (500000);

  A20DBG (("Read hard A20 2: 0x%X\r\n", get_A20 (1000)));

  val = check_mirror (enabled, 500);
  A20DBG (("set_A20(%u) => check_mirror returns %s\r\n\r\n",
	enabled, check_mirror (enabled, 500)? "success" : "fail" ));

  if (enabled && switch_type.IO0x92_bit1) {
      A20DBG (("enabling also outb(0x92, 2)\r\n\r\n"));
      set_fast_A20();
      }

  if (enabled && val == 0) {
      A20DBG (("mirror ERROR "));
      return 1;
      }
  return 0;
  }
#if 0 /* No such hardware, cannot test. */
    else if (outb (0xf2, 0x00), outb (0xf6, 0x02), check_mirror (enabled, 500)) {
      A20DBG (("PC98 only, please report, mirror OK "));
      if (enabled && switch_type.IO0x92_bit1) {
	  A20DBG (("enabling also outb(0x92, 2)\r\n\r\n"));
	  set_fast_A20();
	  }
      return 0;
      }
#endif
#if 0 /* No such hardware, cannot test. */
      {
      unsigned char tmp;

      tmp = inb(0x65);
      if ((tmp & 0x04) == 0) {
	  outb (0x65, tmp | 0x04)
	  if (check_mirror (enabled, 500)) {
	      A20DBG (("!! FOUND a machine using port 0x65 for A20, please report!!\r\n"));
	      return 0;
	      }
	  }
      tmp = inb(0x1f8);
      if ((tmp & 0x01) == 0) {
	  outb (0x1f8, tmp | 0x01);
	  if (check_mirror (enabled, 500)) {
	      A20DBG (("!! FOUND a machine using port 0x1f8 for A20, please report!!\r\n"));
	      return 0;
	      }
	  }
      outb (0x3f20, 0x90);
      if (check_mirror (enabled, 500)) {
	  A20DBG (("!! FOUND machine AT&T 6300+ for A20, please report!!\r\n"));
	  return 0;
	  }
      }
      /* Shall we try that: outb(0x64, x):
       DDh	sngl	disable address line A20 (HP Vectra only???)
		default in Real Mode
       DFh	sngl	enable address line A20 (HP Vectra only???)

	 Shall we try then:
       outb (0x329, 0xdf or 0xdd)
	 Shall we try then:
       outb (0xe0, 0xdd) and outb (0xe0, 0xdf) instead of 0x60
	 How about:
       outb (0x3f20, 0x80) / outb (0x3f00, 0x10)
	 or:
       outb (0x3f00, 0) / outb (0x3f00, 0x10)
	 While we are here:
       outb (0x65, inb(0x65) & 0xfb) and wait bit 0x04 of inb(0x65)
       */
#endif


/**
 ** Hem ... that's mine, it may even work!
 **
 ** The segment _door_ opening/closing: I hate windows -:)
 ** I better like doors because "Doors will always open politely, and it
 ** will be their pleasure to serve your entrance and exit needs.", at
 ** least if you take the right brand (Sirius Cybernetics Corporation).
 **
 ** This stuff is not new, but this way to use it is GPL.
 **/
UTIL_FCT_PREFIX(segment_door) unsigned
segment_door (unsigned open)
  {
  /* You're only supposed to blow the bloody doors off! */
  static const dataseg_t opendoor[] __attribute__ (( aligned(8) )) = {
    {}, /* segment 0 is the invalid one */
    {
    .cste_10	= 2,
    .present	= 1,
    .writable	= 1,
    .accessed	= 1,        /* 486+: do not update it if it is already set */
    .limit	= 0xFFFF,
    .limit_msb	= 0xF,
    .granularity	= 1	/* limit = 0xFFFFF * 4 Kb = 4 Gb */
    },
    {
    .cste_10	= 2,
    .present	= 1,
    .writable	= 1,
    .accessed	= 1,        /* 486+: do not update it if it is already set */
    .limit	= 0xFFFF,
    .limit_msb	= 0,
    .granularity	= 0
    }
    };
  gdt_idt_param_t gdtptr __attribute__ (( aligned(16) )) = {
    .limit = sizeof (opendoor) - 1, /* -1 : see Intel doc */
    .base  = farptr2linear (stack_adr (opendoor))
    };
  gdt_idt_param_t save_gdtptr;
  register unsigned save_ds_es, save_fs_gs;
  register unsigned short save_ss, selector;

  get_gdt(&save_gdtptr);
  ZDBG ((" [%s: %s doors (gdt: 0x%X+0x%X): ", __FUNCTION__, open? "openning" : "closing", save_gdtptr.base, save_gdtptr.limit));

  if (open && set_A20 (1) != 0) {
      ZDBG (("cannot set A20] "));
      return 1;
      }

  disable_interrupt();
  set_gdt (&gdtptr);

  save_ss = getss();
#if 0
  save_ds_es = getds() | ((unsigned)getes() << 16);
  save_fs_gs = getfs() | ((unsigned)getgs() << 16);
#else
  asm (" movw %%es,%w0 ; shll $16,%0 ; movw %%ds,%w0 " : "=r" (save_ds_es));
  asm (" movw %%gs,%w0 ; shll $16,%0 ; movw %%fs,%w0 " : "=r" (save_fs_gs));
#endif

  selector = open? 0x0008 : 0x0010;

  if (protected_mode() != 0) {
      enable_interrupt();
      set_gdt(&save_gdtptr);
      ZDBG (("cannot switch to protected mode] "));
      return 2;
      }
  /* stay in protected mode for very short time, no need for .code32 / .code16gcc */

  setgs (selector);
  setfs (selector);
  setes (selector);
  setds (selector);
  setss (selector); /* -fomit-frame-pointer, "cmpl $0x7f2bb701,0x0(%ebp)" */

#if defined (__i386__)
  /* update accessed bit on 386: read %%seg:(0) */
  update_ss();
  update_gs();
  update_fs();
  update_es();
  update_ds();
#endif

  while (real_mode () != 0) ;

  setss (save_ss);
#if 0
  setds (save_ds_es);
  setes (save_ds_es >> 16);
  setfs (save_fs_gs);
  setgs (save_fs_gs >> 16);
#else
  asm (" movw %w0,%%ds ; shrl $16,%0 ; movw %w0,%%es " : : "r" (save_ds_es));
  asm (" movw %w0,%%fs ; shrl $16,%0 ; movw %w0,%%gs " : : "r" (save_fs_gs));
#endif

  enable_interrupt();
  set_gdt(&save_gdtptr);
  ZDBG (("done] "));

  return 0;
  }

/**
 ** Main variable declaration, init to BSS:
 **/
struct UTIL_str UTIL;

