/* user.h */
#ifndef USER_H
#define USER_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"

#if USER_SUPPORT != 0

#if USER_SUPPORT & VGA_SUPPORT
/*
 * Define this if you think extended VGA modes (> 0x37)
 * with less than or equal to 2 BPP are valid modes:
 * There is probably some NDA to sign somewhere to get
 * the real description of version 2 of the structure
 * "functionality_table_t" in "vga.h", but there is not
 * enought time in live to sign all the NDA of PC BIOSes.
 */
#undef KEEP_ALL_VGA_MODE
#endif

#if USER_SUPPORT & VESA_SUPPORT
/*
 * This value has been chosen (for winsize) because:
 *  if (offset1 / UI.parameter.winsize
 *	 != offset2 / UI.parameter.winsize)
 *      is always false if VESA2 because 0 == 0
 * And also:
 *  if (offset - UI.parameter.winadr >= UI.parameter.winsize)
 *      is always false if VESA2
 */
#define VESA2_MARKER UINT_MAX

#if (USER_SUPPORT & VESA_4BPP_EGA)
/*
 * Define that to the fastest read_pixel (0 or 1) in 16 color mode:
 */
//#define EGA_DEFAULT_READ_MODE 0
#define EGA_DEFAULT_READ_MODE 1

#endif /* VESA_4BPP_EGA */

#if USER_SUPPORT & VESA_HARDWINDOW
/*
 * Define it if it works, report if it does not work,
 * that saves quite a lot of malloc'ed memory.
 * It is normal to have some mode without compression,
 * but I would like to know if I failed to compress
 * some obvious pattern when it would save few Kbytes:
 */
#define COMPRESS_HARDWIN

#ifdef COMPRESS_HARDWIN
/*
 * ET4000 needs a special treatment because it outputs
 * the read and write page in the same outb() - This
 * define fixes it but is not very clean an algorithm.
 */
#define COMPRESS_EXTENDED
#endif /* COMPRESS_HARDWIN */
#endif /* VESA_HARDWINDOW */
#endif /* VESA_SUPPORT */

/*
 * We cannot send "null" character with our interface (to slow down
 * the commands because even at 9600 bauds a true VT420 cannot
 * process everything) so we use _BIOS_wait () with this parameter
 * for "slow" operations (parameter in microseconds):
 */
#define SERIAL_DELAY 30000  /* 3/100 of a second - or should be... */

/*
 * This is just for testing purposes, it is only
 * using get/setpixel to scroll and display horizontal lines.
 */
//#define VERY_SLOW

/***************************************************************
 * The graphic interface:
 */

enum stdcolor_enum {
    black,    blue,         green,      cyan,
    red,      magenta,      brown,      lightgray,
    darkgray, lightblue,    lightgreen, lightcyan,
    lightred, lightmagenta, yellow,     white
    };

struct DAC_str { /* same as VESA_RGB */
    unsigned char blue, green, red, align;
    };

typedef struct {
    unsigned short x, y;	/* x should be the easyest to read */
    } __attribute__ ((packed)) coord;

struct user_interface_str {
    unsigned cursor_bgcolor;	/* only used if UI.parameter.attr.isgraphic, pre gujin-v1.5:    unsigned (*getkey) (unsigned timeout); */
    unsigned fgcolor, bgcolor;
    unsigned stdcolor[16];	/* content: VESA_RGB, VGA_RGB or 1..16 */
    struct {
	struct attribute_str {
	    unsigned char underline	:1;
	    unsigned char reverse	:1;
	    unsigned char blink		:1;
	    unsigned char brighter	:1;
	    unsigned char darker	:1;
	    unsigned char transparent	:1; /* no background drawn in VESA */
	    unsigned char reserved	:2;
	    } __attribute__ ((packed)) current, valid;
	} attributes;
    unsigned short	monitor;    /* enum unknown, B&W, mono, color or EDID type */
    unsigned short	align1;

    struct {
	unsigned char _VGA_get_display_combination_code_works	: 1;
	unsigned char _EGA_getdisplay_works			: 1; /* only set if !_VGA_get_display_combination_code_works */
	unsigned char _VGA_get_functionality_info_works		: 1;
	unsigned char EDID_detect_and_read_works		: 1;
	unsigned char reserved					: 4;
	} videocard;
    unsigned char	nbmode;
    struct videomode_str {
	unsigned short	number;
	unsigned short	width;
	unsigned short	height;
	unsigned char	bpp;
	unsigned char	text     : 1;
	unsigned char	attr     : 1;
	unsigned char	vesa     : 1;
	unsigned char	reserved : 5;
	} __attribute__ ((packed)) *mode;

    union {
#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_SUPPORT)
	struct {
	    unsigned short	version;
	    unsigned short	memory_64Kb;
	    unsigned char	cardname[40];
	    } vesa;
#endif
#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & SERIAL_SUPPORT)
	struct {
	    unsigned		max_mode;
	    unsigned char	terminalname[40];
	    } serial;
#endif
	} info;

#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_EDID)
    struct edid_str {
	unsigned short	horizontal_resolution;
	unsigned short	vertical_resolution;
	unsigned short	frequency;
	unsigned char	Hsize_cm, Vsize_cm;
	unsigned char	lowHfreq, highHfreq, lowVfreq, highVfreq;
	} edid;
#endif
#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_PMINFO)
    unsigned short ProtectedModeIO[32];
    struct {
	unsigned address;
	unsigned short length;
	unsigned short align;
	} ProtectedModeMem[4];
#endif

    struct video_parameter_str {
	unsigned        identification; /* do not have linear or clear bit */
	unsigned        nbcolor;
	unsigned short  width, height;
	unsigned	base_address;
	unsigned char   row, col;
	unsigned char   nbrow, nbcol;
	unsigned char   charwidth, charheight;
	struct {
	    unsigned char isgraphic	 : 1;
	    /*
	     * Some VGA cards, in VGA mode, can accept 256 colors background,
	     * color given by the "page" parameter, VESA card have always
	     * this flag set when graphic (at least >= 16 colors):
	     */
	    unsigned char graphicbgcolor : 1;
	    /* Else only the PC font is available, always set for serial: */
	    unsigned char ansi_font	 : 1;
	    /* null used for text mode mouse, del used as a ticker for tick-box */
	    unsigned char null_del_available	 : 1;
	    /* Always set when scroll, never unset but by application: */
	    unsigned char has_scrolled	 : 1;
	    /* Always set when one char has been written on screen, never unset but by application: */
	    unsigned char has_outputed	 : 1;
	    } attr;

	/* current (UTF base char / 128) loaded in video card text font memory at '\x80'
		nothing downloaded if still set to 0 */
	unsigned char	currentUTFpage;

	/* used also for VGA 256 colors modes: */
	/* see instboot.h, vesabios.h, user.h, vmlinuz.h: */
	struct vesa_color_layout_str layout;

#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_SUPPORT)
	farptr		font;	  /* 0 if text mode, hardware font used */
	unsigned	winadr;
	unsigned	winsize;
	unsigned short  linelength;
	unsigned char   bitperpixel;
	unsigned char   nbplane;
	unsigned char   memtype;
	unsigned char	EGA_compatible;
	unsigned char	BIOS_compatible;
	unsigned char	align3;

#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_WINDOW)
	unsigned char	NbWin;	/* Number of VESA windows */
	unsigned char	winNo;	/* R/W window,
				   or WO window, then RO is (winNo ^ 1) */
#endif
	unsigned short	fontnbchar;
#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_WINDOW)
#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_HARDWINDOW)
	unsigned short	*offsetarray;
	struct hardwindow_str {
	    unsigned short instruction;
	    unsigned short port;
	    unsigned	   data;
	    } *hardwindow;
#endif
#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_STRANGE_WINDOW)
	unsigned	wingranul;
#else
	unsigned char	align5[3];
	unsigned char	wingranul;
#endif
#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_2WINDOWS)
	unsigned	winreadadr;
	unsigned	readdelta; /* HAS TO BE null if NbWin < 2 */
#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_HARDWINDOW)
	unsigned short	*readoffsetarray;
	struct hardwindow_str *readhardwindow;
#endif
#endif
#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VESA_WINFUNCTION)
	unsigned        addrpositionfarfct;
#endif
#endif /* VESA_WINDOW */
#endif /* VESA_SUPPORT */
	} parameter;

    struct user_interface_fct_str {
	unsigned (*getsize) (unsigned mode, struct video_parameter_str *param);
	unsigned (*setmode) (unsigned mode);
	unsigned (*getmode) (void);
	void     (*clearscreen) (void);

	/* Cursor: home at (0,0) */
	unsigned (*setcursor) (unsigned char row, unsigned char col);
	unsigned (*getcursor) (unsigned char *row, unsigned char *col);

	/* Colors are set by functions, and get directly from structure: */
	unsigned (*setfgcolor) (unsigned color);
	unsigned (*setbgcolor) (unsigned color);

	void     (*setattribute) (struct attribute_str attr);

	void     (*putstr) (const char *str);

	void     (*setpixel) (coord xy, unsigned color) FASTCALL;
	unsigned (*getpixel) (coord xy) FASTCALL;
	void	 (*plotHline) (coord xy, unsigned short xend, unsigned color);

	/* pre gujin-v1.5: unsigned (*setDACblock) (unsigned short startindex, unsigned short nb,
				 const struct DAC_str *palette); */
	unsigned (*getkey) (unsigned timeout); /* timeout in ticks, 18/s, 0 = none */
	} function;
    };

extern struct exported_togpl_s {
    int      (*printf) (const char *format, ...);
    void     (*draw_bg_box) (unsigned baserow, unsigned height, unsigned color);
    unsigned (*final_loadrun) (unsigned index, struct registers *regs, struct gujin_param_attrib gujin_attr);
    unsigned (*get_number) (unsigned deflt, unsigned max);
    void     (*setpixel) (coord xy, unsigned color) FASTCALL;
    unsigned (*getpixel) (coord xy) FASTCALL;
    void     (*plotHline) (coord xy, unsigned short xend, unsigned color);
    unsigned __attribute__ ((const)) (*VESA_color) (unsigned char _red, unsigned char _green, unsigned char _blue);
    unsigned (*get_line) (char *buffer, unsigned maxlen, char display, unsigned char lowlimit, unsigned char highlimit); /* Gujin-1.6+ */
    } exported_togpl;

/* static consts removal (by GCC) does not seems to work, so do that: */
#define all_attribute		((struct attribute_str) {1, 1, 1, 1, 1, 1})
#define no_attribute		((struct attribute_str) {0, 0, 0, 0, 0, 0})
#define egatxt_attribute	((struct attribute_str) {1, 1, 1, 1, 0, 0})
#define egagfx_attribute	((struct attribute_str) {0, 0, 1, 1, 0, 0})
#define graphic_attribute	((struct attribute_str) {1, 1, 0, 0, 0, 1})

#define attr_underline		((struct attribute_str) {1, 0, 0, 0, 0, 0})
#define attr_reverse		((struct attribute_str) {0, 1, 0, 0, 0, 0})
#define attr_blink		((struct attribute_str) {0, 0, 1, 0, 0, 0})
#define attr_brighter		((struct attribute_str) {0, 0, 0, 1, 0, 0})
#define attr_darker		((struct attribute_str) {0, 0, 0, 0, 1, 0})
#define attr_transparent	((struct attribute_str) {0, 0, 0, 0, 0, 1})

#define FCT_KEYCODE(i)	(((i <= 10)? (0x3AU + i) : (0x85U + i - 11)) << 8)
#define SFCT_KEYCODE(i)	(((i <= 10)? (0x53U + i) : (0x87U + i - 11)) << 8)
#define CFCT_KEYCODE(i)	(((i <= 10)? (0x5DU + i) : (0x89U + i - 11)) << 8)
#define AFCT_KEYCODE(i)	(((i <= 10)? (0x67U + i) : (0x8BU + i - 11)) << 8)
extern const unsigned short all_fct_keycode[12], all_sfct_keycode[12];

#define ESC_KEYCODE()		(0x0100 | '\033')
#define SPACE_KEYCODE()		(0x3900 | ' ')
#define BACKSPACE_KEYCODE()	(0x0E00 | '\010')
#define RETURN_KEYCODE()	(0x1C00 | '\r')
#define KP_RETURN_KEYCODE()	(0xE000 | '\r') /* only extkbd */
#define KP_MINUS_KEYCODE()	(0x4A00 | '-')
#define MINUS_KEYCODE()		0x0C2D
#define KP_PLUS_KEYCODE()	(0x4E00 | '+')
#define PLUS_KEYCODE()		0x0D2B
#define DIVIDE_KEYCODE()	(0x3500 | '/')
#define KP_DIVIDE_KEYCODE()	(0xE000 | '/') /* only extkbd */
#define KP_MULTIPLY_KEYCODE()	(0x3700 | '*')
#define MULTIPLY_KEYCODE()	0x092A
#define DOT_KEYCODE()		(0x3400 | '.')
#define KP_DOT_KEYCODE()	(0x5300 | '.')
#define DIGIT_KEYCODE(i) \
    (((i? i+1 : 0x0B)<<8) | ('0' + i))
#define NUMPAD_DIGIT_KEYCODE(i) \
    ((((i == 0)? 0x52 : (0x4E - ((i - 1)/3 * 7) + i)) << 8) | ('0' + i))

#define PAGEUP_KEYCODE()	(0x4900)
#define PAGEDOWN_KEYCODE()	(0x5100)
#define HOME_KEYCODE()		(0x4700)
#define END_KEYCODE()		(0x4F00)
#define DELETE_KEYCODE()	(0x5300)
#define INSERT_KEYCODE()        (0x5200)
#define UP_KEYCODE()		(0x4800)
#define DOWN_KEYCODE()		(0x5000)
#define RIGHT_KEYCODE()		(0x4D00)
#define LEFT_KEYCODE()		(0x4B00)
#define TAB_KEYCODE()		(0x0F09)
#define ON_OFF_KEYCODE()	(0x5F80)

#define CTRL_L_KEYCODE()	(0x260C)
#define CTRL_R_KEYCODE()	(0x1312)
#define CTRL_P_KEYCODE()	(0x1910)
#define CTRL_X_KEYCODE()	(0x2D18)
#define CTRL_T_KEYCODE()	(0x1414)
#define ALT_T_KEYCODE()		(0x1400)

#define CTRLENTER		(0x1C0A)	/* PC keyboard only */
#define CTRLBREAK_KEYCODE()	(0x0000)	/* Hopes it work everywhere */
// Not sure of those:
#define POWER_KEYCODE()		(0x5EE0)
#define SLEEP_KEYCODE()		(0x5FE0)
#define WAKE_KEYCODE()		(0x63E0)

// Fake key for timeouts:
#define TIMEOUT_KEYCODE()	0xFFFE

/*
 * Colors of the fields themselves:
 */
#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
/* Backgrounds in graphic modes >= 256 colors: */
#define FIRST_LINES_BACKGROUND	VESA_color (35, 30, 30)
#define TOP_MENU_BACKGROUND	VESA_color (35, 45, 35)
#define TOP_MENU_BUTTON_BG	VESA_color (35, 40, 35)
#define BOTTOM_MENU_BACKGROUND	VESA_color (35, 35, 35)
#define BOTTOM_MENU_BUTTON_BG	VESA_color (30, 30, 35)
#define ADVERT_BACKGROUND	VESA_color (30, 30, 30)
#define WORK_ADVERT_BACKGROUND	VESA_color (35, 35, 30)
#define COPYRIGHT_BACKGROUND	VESA_color (20, 40, 40)
#endif /* VGA_SUPPORT | VESA_SUPPORT */

#define CHECKBOX_BG		UI.stdcolor[white]
#define CHECKBOX_INACTVE_BG	UI.stdcolor[lightgray]

/*
 * Only if no interresting attribute available (even if no mouse):
 * (but also on VT/serial interface if no attributes)
 */
#define MOUSE_SELECTABLE_CHARS	UI.stdcolor[lightblue]


#define USER_ACTIVE()	(UI.parameter.nbcol != 0)

#if USER_SUPPORT & VGA_SUPPORT
#if USER_SUPPORT & VESA_SUPPORT
#define VGA_ACTIVE()	(USER_ACTIVE() && (UI.parameter.linelength == 0))
#define VESA_ACTIVE()	(USER_ACTIVE() && (UI.parameter.linelength != 0))
#else
#define VGA_ACTIVE()	USER_ACTIVE()
#define VESA_ACTIVE()	0
#endif
#else /* VGA_SUPPORT */
#define VGA_ACTIVE()	0
#define VESA_ACTIVE()	USER_ACTIVE()
#endif /* VGA_SUPPORT */

extern inline unsigned char
get_bpp (struct video_parameter_str *ptr)
  {
#if USER_SUPPORT & VESA_SUPPORT
  if (   ptr->bitperpixel != 16	/* 15 & 16 bpp are same */
      && ptr->bitperpixel != 1	/* 4bpp EGA */
      && ptr->bitperpixel != 0) /* VGA modes */
      return ptr->bitperpixel;
    else
#endif
      {
      if (ptr->nbcolor != 0)
	  return __builtin_ffs (ptr->nbcolor) - 1;
	else
	  return 0;
      }
  }

/* UTF-8:
 ** Two bytes char are encoded as: 110yyyxx 10xxxxxx
 ** Three bytes char are encoded as: 1110yyyy 10yyyyxx 10xxxxxx
 ** Four bytes char are encoded as: 11110zzz 10zzyyyy 10yyyyxx 10xxxxxx
 ** Not recognised and invalid UTF8 (like 1yyyyyyy 0xxxxxxx) displayed as ISO-8859-1 by Gujin.
 **/
#define STR_I_M_S(STRING, INDEX, MASK, SHIFT)	((((unsigned char *)STRING)[INDEX] & MASK) << SHIFT)
#define UTF8VAL2(String)	(STR_I_M_S(String,0,0x1F,6) + STR_I_M_S(String,1,0x3F,0))
#define UTF8VAL3(String)	(STR_I_M_S(String,0,0x0F,12) + STR_I_M_S(String,1,0x3F,6) + STR_I_M_S(String,2,0x3F,0))
#define UTF8VAL4(String)	(STR_I_M_S(String,0,0x07,18) + STR_I_M_S(String,1,0x3F,12) + STR_I_M_S(String,2,0x3F,6) + STR_I_M_S(String,3,0x3F,0))

#if 0
#define UTF8LEN(String)	( ((String[0] & 0xF8) == 0xF0 && (String[1] & 0xC0) == 0x80 && (String[2] & 0xC0) == 0x80 && (String[3] & 0xC0) == 0x80) ? 4 : \
			   ((String[0] & 0xF0) == 0xE0 && (String[1] & 0xC0) == 0x80 && (String[2] & 0xC0) == 0x80) ? 3 : \
			   ((String[0] & 0xE0) == 0xC0 && (String[1] & 0xC0) == 0x80)? 2 : 0)
#define UTF8VAL(String)	( ((String[0] & 0xF8) == 0xF0 && (String[1] & 0xC0) == 0x80 && (String[2] & 0xC0) == 0x80 && (String[3] & 0xC0) == 0x80) ? UTF8VAL4(String) : \
			   ((String[0] & 0xF0) == 0xE0 && (String[1] & 0xC0) == 0x80 && (String[2] & 0xC0) == 0x80) ? UTF8VAL3(String) : \
			   ((String[0] & 0xE0) == 0xC0 && (String[1] & 0xC0) == 0x80)? UTF8VAL2(String) : String[0])
#elif defined (BIGENDIAN)
#define UTF8LEN(String)	( ((*(unsigned *)String & 0xF8C0C0C0) == 0xF0808080) ? 4 : \
			   ((*(unsigned *)String & 0xF0C0C000) == 0xE0808000) ? 3 : \
			   ((*(unsigned short *)String & 0xE0C0) == 0xC080)? 2 : 0)
#define UTF8VAL(String)	( ((*(unsigned *)String & 0xF8C0C0C0) == 0xF0808080) ? UTF8VAL4(String) : \
			   ((*(unsigned *)String & 0xF0C0C000) == 0xE0808000) ? UTF8VAL3(String) : \
			   ((*(unsigned short *)String & 0xE0C0) == 0xC080)? UTF8VAL2(String) : String[0])
#else
#define UTF8LEN(String)	( ((*(unsigned *)String & 0xC0C0C0F8) == 0x808080F0) ? 4 : \
			   ((*(unsigned *)String & 0x00C0C0F0) == 0x008080E0) ? 3 : \
			   ((*(unsigned short *)String & 0xC0E0) == 0x80C0)? 2 : 0)
#define UTF8VAL(String)	( ((*(unsigned *)String & 0xC0C0C0F8) == 0x808080F0) ? UTF8VAL4(String) : \
			   ((*(unsigned *)String & 0x00C0C0F0) == 0x008080E0) ? UTF8VAL3(String) : \
			   ((*(unsigned short *)String & 0xC0E0) == 0x80C0)? UTF8VAL2(String) : String[0])
#endif

extern inline unsigned NbUtf8Char(const char *str) {
  unsigned returned = 0;
  while (*str) {
      unsigned charlen = UTF8LEN(str);
      if (charlen)
	  str += charlen;
	else
	  str++;
      returned ++;
      }
  return returned;
  }

/*
 * Indirect functions:
 */
/* Do NOT call this function: */
extern struct user_interface_str UI;
__attribute__((weak)) void asm_dummy_user (void)
  {
  awk_farfct (UI.function.getkey);
  awk_farfct (UI.function.getsize);
  awk_farfct (UI.function.setmode);
  awk_farfct (UI.function.getmode);
  awk_farfct (UI.function.clearscreen);
  awk_farfct (UI.function.setcursor);
  awk_farfct (UI.function.getcursor);
  awk_farfct (UI.function.setfgcolor);
  awk_farfct (UI.function.setbgcolor);
  awk_farfct (UI.function.setattribute);

  awk_farfct (UI.function.putstr);
  }

#ifdef CALLING_FROM_USERSEG
/* The following functions cannot be called from diskcodeseg/fscodeseg/loadcodeseg,
   because UI.function.* are intra-segment calls: */
extern inline void
UI_setpixel (coord xy, unsigned color) {
    UI.function.setpixel (xy, color);
    }

extern inline unsigned
UI_getpixel (coord xy) {
    return UI.function.getpixel (xy);
    }

extern inline void
UI_plotHline (coord xy, unsigned short xend, unsigned color) {
    UI.function.plotHline (xy, xend, color);
    }
#endif /* !defined (CALLING_FROM_XCODESEG) */

#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
unsigned BIOS_probemonitor (void);
awk_farcall(BIOS_probemonitor);
unsigned __attribute__ ((const))
VESA_color (unsigned char _red, unsigned char _green, unsigned char _blue);
awk_farcall(VESA_color);
#endif

#if (USER_SUPPORT & VGA_SUPPORT) && (SETUP & UPDATE_BOOTPARAM)
void setup_gujin_param_vga_mode (void);
awk_farcall(setup_gujin_param_vga_mode);
void restore_gujin_param_vga_mode (void);
awk_farcall(restore_gujin_param_vga_mode);
#endif

void UI_init (unsigned serial);
awk_farcall (UI_init);

#if USER_SUPPORT & VESA_SUPPORT
unsigned VESA_preinit (void);
awk_farcall (VESA_preinit);
unsigned VESA_init (void);
awk_farcall (VESA_init);
#endif

#if USER_SUPPORT & VGA_SUPPORT
unsigned VGA_init (void);
awk_farcall (VGA_init);
#endif

#if USER_SUPPORT & SERIAL_SUPPORT
unsigned SERIAL_init (void);
awk_farcall (SERIAL_init);
#endif

void reset_low_valid_mode (unsigned up_to);
awk_farcall (reset_low_valid_mode);

#ifdef MOUSE_ON_VT /* Called from mouse.c: */
unsigned SERIAL_setcursor (unsigned char row, unsigned char col);
awk_farcall (SERIAL_setcursor);
unsigned SERIAL_getcursor (unsigned char *row, unsigned char *col);
awk_farcall (SERIAL_getcursor);
#endif

void draw_bg_box (unsigned baserow, unsigned height, unsigned color);
awk_farcall (draw_bg_box);

void show_ladders (unsigned bgcolor);
awk_farcall (show_ladders);

#endif /* USER_SUPPORT == 0 */

#endif /* USER_H */
