/* user.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"
#if (SETUP & XCODE_SEGMENT)
#define CALLING_FROM_XCODESEG   /* before other includes */
#define CALLING_FROM_USERSEG	/* can call some non far video function for efficiency */
#endif

#if USER_SUPPORT != 0

#include "instboot.h"
#include "library.h"
#include "boot.h"
#include "debug.h"
#include "bios.h"	/* CDROMbootspec for util.h */
#include "util.h"
#include "kbd.h"
#include "user.h"
#include "gmem.h"
#include "dos.h"	/* DOS_[GS]etInterruptVector() */
#include "messages.h"	/* VIDEO_CARD_CHANGED */
#include "vgabios.h"	/* _VGA_get_display_combination_code, monitor freq */
#include "font.h"

#if USER_SUPPORT & VESA_SUPPORT
#include "vesabios.h"
#endif

#if USER_SUPPORT & (VESA_4BPP_TEXT | VESA_4BPP_EGA | VESA_4BPP | VESA_8BPP | VESA_RECALC)
#include "ega.h"	/* palette loading */
#endif

#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
#include "mouse.h"
#endif

const unsigned short all_fct_keycode[12] = {
      FCT_KEYCODE(1), FCT_KEYCODE(2), FCT_KEYCODE(3), FCT_KEYCODE(4),
      FCT_KEYCODE(5), FCT_KEYCODE(6), FCT_KEYCODE(7), FCT_KEYCODE(8),
      FCT_KEYCODE(9), FCT_KEYCODE(10), FCT_KEYCODE(11), FCT_KEYCODE(12)
      };
const unsigned short all_sfct_keycode[12] = {
      SFCT_KEYCODE(1), SFCT_KEYCODE(2), SFCT_KEYCODE(3), SFCT_KEYCODE(4),
      SFCT_KEYCODE(5), SFCT_KEYCODE(6), SFCT_KEYCODE(7), SFCT_KEYCODE(8),
      SFCT_KEYCODE(9), SFCT_KEYCODE(10), SFCT_KEYCODE(11), SFCT_KEYCODE(12)
      };

/*
 * Few basic macros:
 */
#define ATTR2BYTE(A)	(*(unsigned char *)&(A))

#if USER_SUPPORT & VESA_4BPP_EGA

#if EGA_DEFAULT_READ_MODE == 0
#define EGA_READ_PLANE(plane)	_EGA_select_read_plane (plane)
#define EGA_READ_INIT()	/* */
#elif EGA_DEFAULT_READ_MODE == 1
#define EGA_READ_PLANE(plane)	_EGA_select_read_ignore (1 << plane)
#define EGA_READ_INIT()	do { \
    _EGA_read_compare (0x0F);						\
    _EGA_select_read_ignore (0);	/* ignore if bit cleared */	\
    } while (0)
#else
#error EGA_DEFAULT_READ_MODE should be defined to 0 or 1.
#endif

#if USER_SUPPORT & VESA_2WINDOWS
#define GXCHGB(xy, color) do { \
    volatile unsigned char dummyread __attribute__ ((unused));	\
    dummyread = gpeekb (VESA_readpixeloffset (xy));		\
    gpokeb (VESA_pixeloffset (xy), color);			\
    } while (0)
#else
#define GXCHGB(xy, color) \
    gxchgb (VESA_pixeloffset (xy), color)
#endif

#endif /* VESA_4BPP_EGA */

#if USER_SUPPORT & VESA_STRANGE_WINDOW
#define DIV_WINDOW_GRANUL(off) \
	((off) / UI.parameter.wingranul)
#define ROUND_WINDOW_GRANUL(off) \
	((off) / UI.parameter.wingranul * UI.parameter.wingranul)
#define DISPLAY_WINDOW_GRANUL(str) \
	((str)->wingranul)
#define INIT_WINDOW_GRANUL(size, param) \
	param->wingranul = (size)
#else
#define DIV_WINDOW_GRANUL(off)  \
	((off) >> UI.parameter.wingranul)
#define ROUND_WINDOW_GRANUL(off) \
	((off) & (0xFFFFFFFFU << UI.parameter.wingranul))
#define DISPLAY_WINDOW_GRANUL(str) \
	(1U << (str)->wingranul)
#define INIT_WINDOW_GRANUL(size, param) \
	param->wingranul = (__builtin_ffs (size) - 1)
#endif /* VESA_STRANGE_WINDOW */

#if SETUP & UNICODE_FONT
// Not sure about which inteface to load font for VGA graphic modes, is it the
// same as the interface for CGA graphic modes??? test it:
// TEST DONE: calling _VGAtxt_load_font() in graphic mode is undesireable
#define TEST_UTF_ACTIVE() (!UI.parameter.attr.isgraphic && !_VGA_isgraphic() \
	&& copy_gujin_param.attrib.use_gujin_embedded_font && UI.parameter.charheight >= 16)
/*
#define TEST_UTF_ACTIVE() (!_VGA_isgraphic() \
	&& copy_gujin_param.attrib.use_gujin_embedded_font && UI.parameter.charheight >= 16)
*/
#endif

/*
 * The default "common library" functions:
 */
awk_farret (COMMON_setfgcolor);
VIDEO_FCT_PREFIX(COMMON_setfgcolor) static unsigned
COMMON_setfgcolor (unsigned color)
  {
  if (   color > UI.parameter.nbcolor
      || UI.parameter.nbcolor == 0) {
      VDBG ((" [%s: color too high: %u] ", __FUNCTION__, color));
      return 1;
      }
  UI.fgcolor = color;
  return 0;
  }

awk_farret (COMMON_setbgcolor);
VIDEO_FCT_PREFIX(COMMON_setbgcolor) static unsigned
COMMON_setbgcolor (unsigned color)
  {
  if (   color > UI.parameter.nbcolor
      || UI.parameter.nbcolor == 0) {
      VDBG ((" [%s: color too high: %u] ", __FUNCTION__, color));
      return 1;
      }
  UI.bgcolor = color;
  return 0;
  }

awk_farret (COMMON_setcursor);
VIDEO_FCT_PREFIX(COMMON_setcursor) static unsigned
COMMON_setcursor (unsigned char row, unsigned char col)
  {
  if (row > UI.parameter.nbrow) {
      VDBG ((" [%s to row %u impossible, max = %u] ",
	    __FUNCTION__, row, UI.parameter.nbrow));
      return 1;
      }

  if (col > UI.parameter.nbcol) {
      VDBG ((" [%s to col %u impossible, max = %u] ",
	    __FUNCTION__, col, UI.parameter.nbcol));
      return 1;
      }

  UI.parameter.row = row;
  UI.parameter.col = col;
  return 0;
  }

VIDEO_FCT_PREFIX(COMMON_error) static unsigned
COMMON_error (void)
  {
  DBG (("ERROR: %s called.\r\n", __FUNCTION__));
  return 1;
  }

#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
#if (USER_SUPPORT & VESA_SUPPORT) || !(USER_SUPPORT & VGA_USEINT1013)
awk_farret (COMMON_getcursor);
VIDEO_FCT_PREFIX(COMMON_getcursor) static unsigned
COMMON_getcursor (unsigned char *row, unsigned char *col)
  {
  checkptr (row, "COMMON_getcursor invalid 1st parameter.");
  checkptr (col, "COMMON_getcursor invalid 2nd parameter.");

  // could use EGA registers, but _we_ are maintaining the cursor
  // position.
  *row = UI.parameter.row;
  *col = UI.parameter.col;
  return 0;
  }
#endif /* VESA_SUPPORT || !VGA_USEINT1013 */
#endif /* VGA_SUPPORT || VESA_SUPPORT */

VIDEO_FCT_PREFIX(add_valid_mode) static void
add_valid_mode (struct videomode_str *mode)
  {bound_stack();{
  unsigned short i;

  if (UI.mode == 0 && UI.nbmode != 0)
      VDBG (("\r\n\tERROR: UI.mode == 0 with UI.nbmode = %u\r\n", UI.nbmode));

  for (i = 0; i < UI.nbmode; i++) {
      if (UI.mode[i].number == mode->number) {
	  if (UI.mode[i].vesa && !mode->vesa)
	      return;
	  VDBG (("\r\n\t%s: replace in UI.mode index %u with mode 0x%X.\r\n",
		 __FUNCTION__, i, mode->number));
	  UI.mode[i] = *mode;
	  return;
	  }
      }

  /* get memory for new mode, realloc can be called with 1st param == 0 */
  {
  struct videomode_str *tmp;

  tmp = REALLOC (UI.mode, (UI.nbmode+1) * sizeof (struct videomode_str), "VIDmodes");
  if (tmp == 0) {
      VDBG ((" [%s failed realloc] ", __FUNCTION__));
      return;
      }
  UI.mode = tmp;
  }

  if (!mode->vesa) {
      /* VGA modes: add at end */
      UI.mode[UI.nbmode] = *mode;
      }
    else {
      unsigned short j;

      /* VESA modes: add at beginning, but keep same order */
      for (i = 0; i < UI.nbmode && UI.mode[i].vesa; i++) ;

      /* copy from high to low memory, no memcpy! */
      for (j = UI.nbmode; j > i; j--)
	  UI.mode[j] = UI.mode[j-1];

      UI.mode[i] = *mode;
      }
  UI.nbmode++;
  }}

VIDEO_FCT_PREFIX(remove_valid_mode) static void
remove_valid_mode (unsigned short number)
  {bound_stack();{
  unsigned short i;

  VDBG ((" [remove_valid_mode mode 0x%X", number));
  for (i = 0; i < UI.nbmode; i++) {
      if (UI.mode[i].number == number) {
	  while (i < UI.nbmode - 1) {
	      UI.mode[i] = UI.mode[i+1];
	      i++;
	      }
	  UI.nbmode--;
	  VDBG ((" done] "));
	  return;
	  }
      }
  VDBG ((" not found] "));
  }}

/* This is called in the menu */
VIDEO_FCT_PREFIX(reset_low_valid_mode) void
reset_low_valid_mode (unsigned up_to)
  {bound_stack();{
  unsigned short i;

  for (i = 0; i < up_to; i++)
      remove_valid_mode (i);
  }}

#if USER_SUPPORT & VESA_HARDWINDOW
VIDEO_FCT_PREFIX(free_hardwin) static void
free_hardwin (void)
  {
  if (UI.parameter.hardwindow != 0) {
      FREE (UI.parameter.hardwindow);
      UI.parameter.hardwindow = 0;
      }
  if (UI.parameter.offsetarray != 0) {
      FREE (UI.parameter.offsetarray);
      UI.parameter.offsetarray = 0;
      }
#if USER_SUPPORT & VESA_2WINDOWS
  if (UI.parameter.readhardwindow != 0) {
      FREE (UI.parameter.readhardwindow);
      UI.parameter.readhardwindow = 0;
      }
  if (UI.parameter.readoffsetarray != 0) {
      FREE (UI.parameter.readoffsetarray);
      UI.parameter.readoffsetarray = 0;
      }
#endif /* VESA_2WINDOWS */
  }

VIDEO_FCT_PREFIX(set_hardwin) static unsigned
set_hardwin (struct hardwindow_str *hardwin,
		unsigned short *offsetarray,
		unsigned offset)
  {
  unsigned short page = DIV_WINDOW_GRANUL(offset);
  unsigned short enabled;

  if (hardwin == 0)
      return 1;

  if (offsetarray != 0)
      hardwin = &hardwin[offsetarray[page]];

  enabled = interrupt_enabled();
  if (enabled)
      disable_interrupt ();

  while (hardwin->instruction != 0) {
      if (hardwin->instruction == 0xEF66)
	  outl (hardwin->port, hardwin->data);
	else {
	  unsigned short data;
	  unsigned char comp = hardwin->instruction >> 8,
			inst = hardwin->instruction & 0xFF;

	  if (inst != 0xEE && inst != 0xEF) {
	      if (enabled)
		  enable_interrupt();
	      VDBG ((" [%s invalid instruction!] ", __FUNCTION__));
	      return 2;
	      }

	  if (comp == 0)
	      data = hardwin->data;
#ifdef COMPRESS_HARDWIN
	    else if (   comp == 1  || comp == 3  || comp == 19
		     || comp == 67 || comp == 20 || comp == 28
		     || comp == 87 || comp == 187
#ifdef COMPRESS_EXTENDED
		     || comp == 37 || comp == 11
#endif /* COMPRESS_EXTENDED */
		     ) {
	      unsigned short pg = page;
	      unsigned tmp = hardwin->data;

#ifdef COMPRESS_EXTENDED
	      if (inst == 0xEE) {
		  if (comp == 37) {
		      tmp ^= inb (hardwin->port) & 0xF0;
		      comp = 1;
		      }
		  if (comp == 11) {
		      tmp ^= inb (hardwin->port) & 0x0F;
		      comp = 3;
		      }
		  }
#endif /* COMPRESS_EXTENDED */

	      if (comp == 1 || comp == 3)
		  pg &= 0x0F;

	      if (comp == 3)
		  pg <<= 4;
		else if (comp == 28)
		  pg <<= 2;
		else if (comp == 67)
		  pg = (pg & 0xF0) >> 2;
		else if (comp == 20)
		  pg >>= 4;
		else if (comp == 87 || comp == 187)
		  pg <<= 1;

	      if (inst == 0xEE || comp == 187)
		  data = tmp ^ pg;
		else
		  data = tmp ^ (pg << 8);
	      }
#endif  /* COMPRESS_HARDWIN */
	    else {
	      if (enabled)
		  enable_interrupt();
	      VDBG ((" [%s invalid byte!] ", __FUNCTION__));
	      return 3;
	      }

	  if (inst == 0xEE)
	      outb (hardwin->port, data);
	    else
	      outw (hardwin->port, data);
	  }
      hardwin++;
      }
  if (enabled)
      enable_interrupt();
  return 0;
  }
#endif /* VESA_HARDWINDOW */

/*
 * The default functions, at startup:
 */
awk_farret (DEFLT_setfgcolor);
VIDEO_FCT_PREFIX(DEFLT_setfgcolor) static unsigned
DEFLT_setfgcolor (unsigned color)
  {
  if (COMMON_setfgcolor (color))
      return 1;
  BOOT1_setcolor (color);	/* ONLY graphic BIOS modes */
  return 0;
  }

#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
/*
 * The "common VESA/VGA BIOS library" functions:
 */
VIDEO_FCT_PREFIX(display_time) static void
display_time (void)
  {
  struct biostime_t biostime;

  if (_BIOS_gettime(&biostime) == 0) {
      char timestr[sizeof(" 00:00:00 ")] = " 00:00:00 ";
      unsigned char minute, new_minute, hour, new_hour;

      minute = 10 * (biostime.minute >> 4) + (biostime.minute & 0x0F);
      new_minute = (minute + UTIL.time_minute_offset + 60) % 60;
      hour = 10 * (biostime.hour >> 4) + (biostime.hour & 0x0F);
      new_hour = hour + UTIL.time_hour_offset + 48;
      if (minute + UTIL.time_minute_offset >= 60)
	  new_hour += 1;
	else if (minute < UTIL.time_minute_offset)
	  new_hour -= 1;
      new_hour %= 24;

      if (new_hour >= 10)
	  timestr[1] += new_hour / 10;
	else
	  timestr[1] = ' ';
      timestr[2] += new_hour % 10;
      timestr[4] += new_minute / 10;
      timestr[5] += new_minute % 10;
      timestr[7] += biostime.second / 16;
      timestr[8] += biostime.second % 16;
      print (timestr);
      }
  }

awk_farret (BIOS_getkey);
VIDEO_FCT_PREFIX(BIOS_getkey) static unsigned
BIOS_getkey (unsigned timeout)
  {bound_stack();{
  unsigned returned, need_translate = !BOOT1_DOS_running();
  unsigned init_tick = _BIOS_nbtick();

  if (timeout) {
      if (timeout != 0xFFFFFFFFU)
	  timeout += init_tick;
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
      if (MOUSE_inited())
	  MOUSE_start();
#endif
      }
  for (;;) {
      unsigned cur_tick = _BIOS_nbtick();
      if (UTIL.keyboard != stdkbd) {
	  if (_BIOS_extkbd_checkkey()) {
	      returned = _BIOS_extkbd_getkey ();
	      break;
	      }
	  }
	else
	  {
	  if (_BIOS_checkkey()) {
	      returned = _BIOS_getkey ();
	      break;
	      }
	  }
      if (timeout == 0 || (cur_tick >= timeout)) {
	  returned = TIMEOUT_KEYCODE();
	  need_translate = 0;
	  break;
	  }
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
      if (MOUSE_inited()) {
	  returned = MOUSE_poll();
	  if (returned != 0xFFFFFFFF) {
	      need_translate = 0;
	      break;
	      }
	  }
#endif
      /* Serial/parallel port or Joystick are not managed using interrupt, do not halt processor then: */
#if ((DEBUG & DEBUG_OUTPUT) < COM1 || (DEBUG & DEBUG_OUTPUT) > LPT4)
      if (BOOT1_DOS_running())
	  DOS_IdleInterrupt();
	else if (1
#if USER_SUPPORT & SERIAL_MOUSE_SUPPORT
	&& MOUSE.type < MOUSE_SERIAL_COM1
#endif
#if USER_SUPPORT & JOYSTICK_SUPPORT
	&& !UTIL.joystick_present
#endif
	)
	  halt_processor ();
	else
#endif
	  cool_processor ();
#if USER_SUPPORT & BIOS_MOUSE_SUPPORT
      if (!MOUSE_inited() || !MOUSE.data.PS2.irq_trigged)
#endif
	  _BIOS_wait (1000);
      if (  timeout == 0xFFFFFFFFU && UI.parameter.width >= 70
	  && cur_tick - init_tick > TICKS_PER_SECOND
	  && UI.function.setcursor (1, UI.parameter.nbcol - 11) == 0) {
	  display_time();
	  init_tick = cur_tick;
	  }
      }
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
  if (MOUSE_inited() && timeout)
      MOUSE_stop();
#endif

//VDBG ((" [%s : 0x%X] ", __FUNCTION__, returned));

  if (need_translate)
      returned = kbd_translate (returned);

  if (   (returned & 0xFF) == 3 /* ^C */
      || (returned & 0xFF) == 4 /* ^D, PuTTY do not transmit ^C */
	) {
      VDBG (("\r\nTrying to stop '.com', stop DOSEMU, generate DOSEMU exception, halt\r\n"));
      BIOS_killme();
      }

  return returned;
  }}

#if USER_SUPPORT & VESA_EDID
VIDEO_FCT_PREFIX(EDID_detect_and_read) extern inline unsigned
EDID_detect_and_read (VBE_EDID_t *info)
  {
  unsigned short bx, monitor_port = 0;

  VDBG ((" [EDID0 "));
  if (_EDID_detect (monitor_port, &bx) != 0) {
      VDBG (("not supported, EDID1 "));
      monitor_port = 1;
      if (_EDID_detect (monitor_port, &bx) != 0) {
	  VDBG (("not supported] "));
	  return 1;
	  }
      }
  VDBG (("present (bx = 0x%X), reading ", bx));
  if (_EDID_read (monitor_port, info, 0) == 0) {
      if (info->version != 0) {
	  VDBG (("OK] "));
	  return 0;
	  }
      }
  VDBG (("failed, retry with blocknr 1 "));
  if (_EDID_read (monitor_port, info, 1) == 0) {
      VDBG (("OK] "));
      return 0;
      }
  monitor_port = !monitor_port;
  VDBG (("failed, retry with monitor_port %u ", monitor_port));
  if (_EDID_read (monitor_port, info, 0) == 0) {
      VDBG (("OK] "));
      return 0;
      }
  VDBG (("failed, give up] "));
  return 2;
  }
#endif

/*
 * The initial probing for VGA/VESA cards:
 */
VIDEO_FCT_PREFIX(BIOS_probemonitor) unsigned
BIOS_probemonitor (void)
  {bound_stack();{
  unsigned char alternate_display, active_display;
  union EGA_bx_union EGA_bx;
  struct EGA_cx_str EGA_cx;

  VDBG (("%s: ", __FUNCTION__));

#if USER_SUPPORT & VESA_EDID
  {
  VBE_EDID_t info = {};
  static const struct {
      unsigned	horizontal_resolution : 12;
      unsigned	vertical_resolution : 12;
      unsigned	frequency : 8;
      } extablished_edid[] = {
	{ 720, 400, 70 },	{ 720, 400, 88 },
	{ 640, 480, 60 },	{ 640, 480, 67 },
	{ 640, 480, 72 },	{ 640, 480, 75 },
	{ 800, 600, 56 },	{ 800, 600, 60 },
	{ 800, 600, 72 },	{ 800, 600, 75 },
	{ 832, 624, 75 },
	{ 1024, 768, 87 },	/* interlaced */
	{ 1024, 768, 60 },	{ 1024, 768, 70 },
	{ 1024, 768, 75 },
	{ 1280, 1024, 75},
	/* manufacturer_timing: */
	{ 1152, 870, 75},
	};

//#define VDBG(X) printf X

  if (EDID_detect_and_read (&info) == 0) {
      int i;
      unsigned short mask;

      VDBG (("%u.%u: screen 0x%X, size %u cm x %u cm, "
	     "established timing 1:0x%X 2:0x%X, manufacturer timing mask: 0x%X, ",
	     info.version, info.revision, info.monitor_model,
	     info.max_horizontal_size_cm, info.max_vertical_size_cm,
	     *(unsigned char *) &info.established_timing,
	     *((unsigned char *) &info.established_timing + 1),
	     info.manufacturer_timing));

      UI.edid.Hsize_cm = info.max_horizontal_size_cm;
      UI.edid.Vsize_cm = info.max_vertical_size_cm;
      UI.videocard.EDID_detect_and_read_works = 1;

      for (mask = 0x8000, i = 15; mask != 0; mask >>= 1, i--)
	  if (mask & *CAST(unsigned short *, &info.established_timing))
	      break;
      /* If there is a major problem and none of the bit are set
	 here, we assume { 800, 600, 56 } is valid, because else... */
      if (i < 0)
	  i = 6;
      /* treat the only bit defined in info.manufacturer_timing,
	 i.e. bit 7: 1152x870 @ 75 Hz (Mac II, Apple) */
      if (i != 15 && info.manufacturer_timing & 0x80)
	  i = 16;
      UI.edid.horizontal_resolution = extablished_edid[i].horizontal_resolution;
      UI.edid.vertical_resolution = extablished_edid[i].vertical_resolution;
      UI.edid.frequency = extablished_edid[i].frequency;
      VDBG (("best established_edid: %ux%u at %u Hz\r\n\t", UI.edid.horizontal_resolution, UI.edid.vertical_resolution, UI.edid.frequency));

      for (i = 0; i < (signed) nbof(info.standard_timing); i++) {
	  unsigned short hres, vres, freq;

	  if (info.standard_timing[i].resolution <= 1) {
	      VDBG (("standart timing mode %u invalid: 0x%X 0x%X\r\n\t", i, ((char *)&info.standard_timing[i])[0], ((char *)&info.standard_timing[i])[1]));
	      continue;
	      }
	  hres = (info.standard_timing[i].resolution + 31) * 8;
	  switch (info.standard_timing[i].aspect_ratio) {
	    case 3: vres = hres * 9 / 16; break;
	    case 2: vres = hres * 4 / 5; break;
	    case 1: vres = hres * 3 / 4; break;
	    default: vres = hres * 10 / 16; break;
	    }
	  freq = info.standard_timing[i].vertical_refresh_freq + 60;
	  VDBG (("mode %u: horizontal=%u, vertical=%u, refresh=%u Hz (0x%X 0x%X)\r\n\t", i, hres, vres, freq, ((char *)&info.standard_timing[i])[0], ((char *)&info.standard_timing[i])[1]));

	  if (hres > UI.edid.horizontal_resolution || (hres == UI.edid.horizontal_resolution && UI.edid.frequency < freq)) {
	      UI.edid.horizontal_resolution = hres;
	      UI.edid.vertical_resolution = vres;
	      UI.edid.frequency = freq;
	      }
	  }

      for (i = 0; i < (signed) nbof(info.detailed_timing); i++) {
	  VDBG (("detailed_timing(%u): ", i));
	  if (   info.detailed_timing[i].strings.wordmarker_0 == 0
	      && info.detailed_timing[i].strings.bytemarker_0 == 0) {
	      switch (info.detailed_timing[i].strings.str_type) {
		  case serial_number:
		      VDBG (("serial number: "));
		      break;
		  case vendor_name:
		      VDBG (("vendor name: "));
		      break;
		  case vertical_horizontal_frequency_range:
		      VDBG (("vertical horizontal frequency range: "));
		      break;
		  case model_name:
		      VDBG (("model name: "));
		      break;
		  default:
		      VDBG (("unknown string (0x%X): ", info.detailed_timing[i].strings.str_type));
		      break;
		  }
	      if (info.detailed_timing[i].strings.str_type
			== vertical_horizontal_frequency_range) {
		  VDBG (("Hfreq %u-%u Hz, Vfreq %u-%u kHz",
		      (unsigned char)info.detailed_timing[i].strings.string[0],
		      (unsigned char)info.detailed_timing[i].strings.string[1],
		      (unsigned char)info.detailed_timing[i].strings.string[2],
		      (unsigned char)info.detailed_timing[i].strings.string[3]
			));
		  UI.edid.lowHfreq = info.detailed_timing[i].strings.string[0];
		  UI.edid.highHfreq = info.detailed_timing[i].strings.string[1];
		  UI.edid.lowVfreq = info.detailed_timing[i].strings.string[2];
		  UI.edid.highVfreq = info.detailed_timing[i].strings.string[3];
		  }
		else {
		  /* text can be also ended by LF... */
		  char *ptr,
			str[nbof (info.detailed_timing[i].strings.string)],
			*pstr = str;

		  for (ptr = info.detailed_timing[i].strings.string;
		       ptr < &info.detailed_timing[i].strings.string[
				nbof (info.detailed_timing[i].strings.string)];
		       ptr ++)
		      if (*ptr != 0x0A)
			  *pstr++ = *ptr;
			else
			  break;
		  *pstr = '\0';
		  VDBG (("%s", str));
		  }
	      VDBG ((" [0x%x 0x%x 0x%x 0x%x 0x%x 0x%x "
		     "0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x]\r\n\t",
		  (unsigned char)info.detailed_timing[i].strings.string[0],
		  (unsigned char)info.detailed_timing[i].strings.string[1],
		  (unsigned char)info.detailed_timing[i].strings.string[2],
		  (unsigned char)info.detailed_timing[i].strings.string[3],
		  (unsigned char)info.detailed_timing[i].strings.string[4],
		  (unsigned char)info.detailed_timing[i].strings.string[5],
		  (unsigned char)info.detailed_timing[i].strings.string[6],
		  (unsigned char)info.detailed_timing[i].strings.string[7],
		  (unsigned char)info.detailed_timing[i].strings.string[8],
		  (unsigned char)info.detailed_timing[i].strings.string[9],
		  (unsigned char)info.detailed_timing[i].strings.string[10],
		  (unsigned char)info.detailed_timing[i].strings.string[11],
		  (unsigned char)info.detailed_timing[i].strings.string[12]
			));
	      }
	    else if (info.detailed_timing[i].timings.horizontal_freq_Khz > 1) {
	      VDBG (("h_freq_Khz: %u, "
		     "v_freq_Hz: %u, "
		     "h_active_time_pix: 0x%X, "
		     "h_blanking_time_pix: 0x%X, "
		     "h_active2_time: 0x%X, "
		     "v_active_time_line: 0x%X, "
		     "v_blanking_time_line: 0x%X, "
		     "v_active_time2: 0x%X, "
		     "h_sync_offset: 0x%X, ",
		     info.detailed_timing[i].timings.horizontal_freq_Khz,
		     info.detailed_timing[i].timings.vertical_freq_Hz,
		     info.detailed_timing[i].timings.horizontal_active_time_pix,
		     info.detailed_timing[i].timings.horizontal_blanking_time_pix,
		     info.detailed_timing[i].timings.horizontal_active2_time,
		     info.detailed_timing[i].timings.vertical_active_time_line,
		     info.detailed_timing[i].timings.vertical_blanking_time_line,
		     info.detailed_timing[i].timings.vertical_active_time2,
		     info.detailed_timing[i].timings.horizontal_sync_offset
		     ));

	      VDBG (("h_sync_pulsewidth_pix: 0x%X, "
		     "v_sync_pulsewidth: 0x%X, "
		     "h_vertical_sync_offset2: 0x%X, "
		     "h_image_size_mm: %u, "
		     "v_image_size_mm: %u, "
		     "h_image_size_2: 0x%X, "
		     "h_border_pix: 0x%X, "
		     "v_border_pix: 0x%X\r\n\t",
		     info.detailed_timing[i].timings.horizontal_sync_pulsewidth_pix,
		     info.detailed_timing[i].timings.vertical_sync_pulsewidth,
		     info.detailed_timing[i].timings.horizontal_vertical_sync_offset2,
		     info.detailed_timing[i].timings.horizontal_image_size_mm,
		     info.detailed_timing[i].timings.vertical_image_size_mm,
		     info.detailed_timing[i].timings.horizontal_image_size_2,
		     info.detailed_timing[i].timings.horizontal_border_pix,
		     info.detailed_timing[i].timings.vertical_border_pix
		     ));
	      unsigned short hres, vres;
	      /* Has a Manipulator of Religions been on Arrakis? */
	      hres = ((info.detailed_timing[i].timings.horizontal_active2_time >> 4) << 8) + info.detailed_timing[i].timings.horizontal_active_time_pix;
	      vres = ((info.detailed_timing[i].timings.vertical_active_time2 >> 4) << 8) + info.detailed_timing[i].timings.vertical_active_time_line;
//	      if (hres >= UI.edid.horizontal_resolution) { /* prefer 1280x768@31Hz to 1280x1024@75 Hz on TV used as monitor */
	      if (hres > UI.edid.horizontal_resolution) {
		  UI.edid.horizontal_resolution = hres;
		  UI.edid.vertical_resolution = vres;
		  UI.edid.frequency = info.detailed_timing[i].timings.vertical_freq_Hz;
		  }
		else
		  VDBG (("\tdetailed_timing gives %ux%u@%uHz not as good as best established_edid\r\n\t", hres, vres, info.detailed_timing[i].timings.vertical_freq_Hz));
	      }
	    else
	      VDBG (("not defined\r\n\t"));
	  }
      VDBG (("nb_extension_blocks: %u, overall best edid: %ux%u at %u Hz]\r\n", info.nb_extension_blocks, UI.edid.horizontal_resolution, UI.edid.vertical_resolution, UI.edid.frequency));

	/* For a two head system, _VGA_get_display_combination_code ()
	   does not work here (1 PCI and 1 AGP) */

      if (info.DPMS.RBGcolor)
	  return info.monitor_model;
	else
	  return 2;
      }
  }
#endif /* VESA_EDID */

//#define VDBG(X) /**/

  if (_VGA_get_display_combination_code (&active_display,
					 &alternate_display) == 0) {
      VDBG (("INT10/1A00 returns 0x%X, i.e. ", active_display));
      UI.videocard._VGA_get_display_combination_code_works = 1;
#if 0 /* assembly too big */
      switch (active_display) {
	case 0: /* no display */
	  VDBG (("no display. "));
	  break;
	default:
	  VDBG (("unknown. "));
	  break;
	case 1: /* mono adapter & mono display */
	  VDBG (("B&W display.\r\n"));
	  return 1;
	case 5: /* EGA mono */
	case 7: /* VGA mono */
	case 0x0B: /* MCGA mono */
	  VDBG (("monochrome display.\r\n"));
	  return 2;
	case 2: /* CGA color */
	case 4: /* EGA color */
	case 6: /* PGA color */
	case 8: /* VGA color */
	case 0x0A: /* MCGA digital color */
	case 0x0C: /* MCGA analog color */
	  VDBG (("color display.\r\n"));
	  return 3;
	}
#else
      if (active_display == 0)
	  VDBG (("no display. "));
	else if (active_display == 3 || active_display == 9 || active_display > 0x0C)
	  VDBG (("unknown. "));
	else if (active_display == 1) {
	  VDBG (("B&W display.\r\n"));
	  return 1;
	  }
	else if (active_display & 1) {
	  VDBG (("monochrome display, forced to color.\r\n"));
	  _VGA_set_display_combination_code (active_display + 1,
						alternate_display);
	  _VGA_enable_gray_scale_summing (0);
	  return 3;
	  }
	else {
	  VDBG (("color display.\r\n"));
	  return 3;
	  }
#endif
      }
  VDBG (("INT10/1A00 not supported, "));

  _EGA_getdisplay (&EGA_bx, &EGA_cx);

#if DEBUG & DEBUG_VIDEO
  {
  union { struct EGA_cx_str bits; unsigned all; } tmp = { EGA_cx };
  VDBG (("INT10/1200 reports bx = 0x%X, cx = 0x%X, i.e. ",
	EGA_bx.IOaddr_memory, tmp.all));
  }
#endif

  if (EGA_bx.enums.IOaddr == EGA_COLOR_3Dx) {
      UI.videocard._EGA_getdisplay_works = 1;
      VDBG (("color display.\r\n"));
      return 3;
      }
    else if (EGA_bx.enums.IOaddr == EGA_MONO_3Bx) {
      VDBG (("B&W display.\r\n"));
      return 1;
      }
    else {
      VDBG (("error.\r\n"));
      return 0;
      }
  }}

static const struct DAC_str VGA_VESA_default_colordac16[16] = {
#define H 63
#define M 2*H/3
#define L H/3
#define X 4*H/9
#define Y H/4
    { .red = 0, .green = 0, .blue = 0 },	/* black */
    { .red = 0, .green = 0, .blue = M },	/* blue */
    { .red = 0, .green = M, .blue = 0 },	/* green */
    { .red = 0, .green = M, .blue = M },	/* cyan */
    { .red = M, .green = 0, .blue = 0 },	/* red */
    { .red = M, .green = 0, .blue = M },	/* magenta */
    { .red = X, .green = Y, .blue = Y },	/* brown */
    { .red = M, .green = M, .blue = M },	/* light gray */
    { .red = L, .green = L, .blue = L },	/* dark gray */
    { .red = 0, .green = 0, .blue = H },	/* light blue */
    { .red = 0, .green = H, .blue = 0 },	/* light green */
    { .red = 0, .green = H, .blue = H },	/* light cyan */
    { .red = H, .green = 0, .blue = 0 },	/* light red */
    { .red = H, .green = 0, .blue = H },	/* light magenta */
    { .red = H, .green = H, .blue = M },	/* yellow */
    { .red = H, .green = H, .blue = H }		/* white */
#undef H
#undef M
#undef L
#undef X
#undef Y
	};

VIDEO_FCT_PREFIX(VESA_color) unsigned __attribute__ ((const))
VESA_color (unsigned char _red, unsigned char _green, unsigned char _blue)
  {
  /* red, green, blue as 6 bit DAC, 0..63 */
  unsigned tmpred =   _red   << 2;
  unsigned tmpgreen = _green << 2;
  unsigned tmpblue =  _blue  << 2;

  tmpred   >>= (8 - UI.parameter.layout.Red.MaskSize);
  tmpgreen >>= (8 - UI.parameter.layout.Green.MaskSize);
  tmpblue  >>= (8 - UI.parameter.layout.Blue.MaskSize);

  return   (tmpred << UI.parameter.layout.Red.FieldPosition)
	 | (tmpgreen << UI.parameter.layout.Green.FieldPosition)
	 | (tmpblue << UI.parameter.layout.Blue.FieldPosition);
  }

VIDEO_FCT_PREFIX(VGA_setDACblock) static unsigned __attribute__ ((noinline))
VGA_setDACblock (unsigned short startindex, unsigned short nb,
		 const struct DAC_str *palette)
  {bound_stack();{
  VGA_RGB vga_palette[3*nb];
  unsigned cpt;

  for (cpt = 0; cpt < nb; cpt++) { /* convert */
      vga_palette[cpt].red = palette[cpt].red;
      vga_palette[cpt].green = palette[cpt].green;
      vga_palette[cpt].blue = palette[cpt].blue;
      }

  _VGA_setDACarray (startindex, nb, vga_palette);
  return 0;
  }}

VIDEO_FCT_PREFIX(VGA_VESA_set_DAC_332) static inline void
VGA_VESA_set_DAC_332 (void) /* 332 palette, 2 bits for blue */
  {
  struct DAC_str array[256];
//  struct DAC_str *array = (struct DAC_str *)fourKbuffer;
  struct DAC_str *ptr = array;
  unsigned char R, G, B;
  /* The linear form wastes bits, because nothing can be seen at bottom */
  /* To have some gray, we need R == G == B */
  const unsigned char ladder4[] = {0,     23,     39,         63};
  const unsigned char ladder8[] = {0, 15, 23, 31, 39, 47, 55, 63};

//  if (sizeof (fourKbuffer) < 256 * sizeof (struct DAC_str))
//      __ERROR();	/* the linker will produce an error here */

  for (B = 0; B < 4; B++) /* blue is MSB, to easy write in octal */
      for (G = 0; G < 8; G++)
	  for (R = 0; R < 8; R++) {
	      ptr->red = ladder8[R];
	      ptr->green = ladder8[G];
	      ptr->blue = ladder4[B];
	      ptr++;
	      }
  if (VESA_ACTIVE()) {
#if USER_SUPPORT & (VESA_4BPP_TEXT | VESA_4BPP_EGA | VESA_4BPP | VESA_8BPP)
      if (_VESA_setDACarray (0, 256, (VESA_RGB *)array) != 0)
	  _EGA_setDACarray (0, 256, array);
#endif /* VESA_4BPP_TEXT | VESA_4BPP_EGA | VESA_4BPP | VESA_8BPP */
      }
    else if (VGA_ACTIVE())
      VGA_setDACblock (0, 256, array);
  }

VIDEO_FCT_PREFIX(VGA_VESA_setup_color) static void __attribute__ ((noinline))
VGA_VESA_setup_color (void)
  {bound_stack();{
  unsigned i;

  /* REALTEK QUADTEL BIOS version 6.C01, seems to overwrite return address */
#ifdef TREAT_EXCEPTION
  if (shit_handler (1, 3) != 0) {
      VDBG ((" EXCEPTION/timeout 3 second RECEIVED! ] "));
      return;
      }
#endif

  switch (UI.parameter.nbcolor) {
    case 0: /* no colors, attributes */
#if 0
	/* would be nice to have the same
	   attributes in text & graphic, but... */
      _VGA_setpalette (0, 0x0);
      for (i = 1; i < 8; i++)
	 _VGA_setpalette (i, 0x8);
      _VGA_setpalette (8, 0x10);
      for (i = 9; i < 16; i++)
	 _VGA_setpalette (i, 0x18);
#endif

      if (UI.parameter.attr.isgraphic) {
	  VDBG ((" [no palette, graphic attributes] "));
	  for (i = black; i <= white; i++)
	      UI.stdcolor[i] = 0x85; /* mouse button pressed */
	  /* 0x80: flash when on active field */
	  /* 0x04: highlight */
	  UI.stdcolor[black] = 0;
	  UI.stdcolor[white] = 1;
	  UI.stdcolor[brown] = 1; /* mouse no button pressed */
	  }
	else
	  {
	  VDBG ((" [no palette, text attributes] "));
	  for (i = black; i <= white; i++)
	     UI.stdcolor[i] = 0x07;
	  /* standart no_attribute: from 0x02 to 0x08 */
	  UI.stdcolor[black] = 0;
	  UI.stdcolor[brown] = 0x06; /* mouse no button pressed */
	  }
      break;
    case 1 ... 2:
      VDBG ((" [no palette <= 2 colors] "));
      UI.stdcolor[0] = 0;
      for (i = 1; i < 16; i++)
	  UI.stdcolor[i] = 1;
      break;
    case 3 ... 15:
      VDBG ((" [no palette < 16 colors] "));
      for (i = 0; i < 16; i++)
	  UI.stdcolor[i] = i % 4; /* ??? */
      break;
    case 16:
      VDBG ((" [setting palette 16 colors"));
      for (i = 0; i < 16; i++) {
	   UI.stdcolor[i] = i;
#if USER_SUPPORT & VGA_SUPPORT
	   _VGA_setpalette (i, i);	/* ATI bios, when probing VGA */
#endif
	   }
      if (VESA_ACTIVE()) {
#if USER_SUPPORT & (VESA_4BPP_TEXT | VESA_4BPP_EGA | VESA_4BPP | VESA_8BPP)
	  if (_VESA_setDACarray (0, 16, (VESA_RGB *)VGA_VESA_default_colordac16) != 0)
	      _EGA_setDACarray (0, 16, VGA_VESA_default_colordac16);
#endif /* VESA_4BPP_TEXT | VESA_4BPP_EGA | VESA_4BPP | VESA_8BPP */
	  }
	else if (VGA_ACTIVE())
	  VGA_setDACblock (0, 16, VGA_VESA_default_colordac16);
      VDBG (("] "));
      break;
    case 17 ... 256:
      VDBG ((" [setting palette 256 colors"));
      VGA_VESA_set_DAC_332 ();
      VDBG (("] "));

      UI.parameter.layout = (struct vesa_color_layout_str) {
		.Red   = { .MaskSize = 3, .FieldPosition = 0 },
		.Green = { .MaskSize = 3, .FieldPosition = 3 },
		.Blue  = { .MaskSize = 2, .FieldPosition = 6 }
		};
#if 0
      /* organised 2 MSB:blue, 3middle:green, 3LSB:red */
			 /* octal! BGR*/
      UI.stdcolor[black] =        0000;
      UI.stdcolor[blue] =         0200;
      UI.stdcolor[green] =        0040;
      UI.stdcolor[cyan] =         0240;
      UI.stdcolor[red] =          0004;
      UI.stdcolor[magenta] =      0204;
      UI.stdcolor[brown] =        0044;
      UI.stdcolor[lightgray] =    0244;
      UI.stdcolor[darkgray] =     0122;
      UI.stdcolor[lightblue] =    0300;
      UI.stdcolor[lightgreen] =   0070;
      UI.stdcolor[lightcyan] =    0370;
      UI.stdcolor[lightred] =     0007;
      UI.stdcolor[lightmagenta] = 0307;
      UI.stdcolor[yellow] =       0277;
      UI.stdcolor[white] =        0377;
      break;
#else
      /* fall through */
#endif
    default: /* more than 256 colors */
      VDBG ((" [init base color, >= 256 color] "));
      for (i = 0; i < 16; i++)
	   UI.stdcolor[i] = VESA_color (VGA_VESA_default_colordac16[i].red,
					VGA_VESA_default_colordac16[i].green,
					VGA_VESA_default_colordac16[i].blue);
      break;
    }

#ifdef TREAT_EXCEPTION
  shit_handler (0, 3);
#endif
  }}

/* Some PC have ISO/IEC 8859-1 (Latin-1) in the BIOS instead of the standard CP437 (mostly PC emulators),
	try to detect that by checking space/non-blank-space (FIXME: any other default font in other countries?) : */
VIDEO_FCT_PREFIX(VGA_VESA_current_font_ansi) static unsigned
VGA_VESA_current_font_ansi (void)
  {
  /* non-blank-space is '' in CP437, we just need to compare the first 8 bytes, whatever font heigh */
  unsigned char nbrow;
  unsigned short byteperchar;
#if !(USER_SUPPORT & VGA_MEMORY)
  farptr INTpointer = _VGA_getfont ((UI.parameter.charheight > 8)? 6 : 0, &nbrow, &byteperchar); /* INT 0x44 */
#else
  farptr INTpointer = peekl(4 * 0x44); /* INT 0x44 */
  byteperchar = peekw (0x00400085);
#endif
  unsigned long long top_bitmap_space = peekll (INTpointer + 0x20 * byteperchar);
  VDBG ((" [check ANSI font%u: (8x16 font @ 0x%X) space bitmap should be zero: 0x%llX, ", byteperchar, INTpointer, top_bitmap_space));
  if (UI.parameter.charheight > 8) {
      unsigned long long top_bitmap_nonblankspace = peekll (INTpointer + 0xA0 * byteperchar);
      VDBG ((" non-blank-space at 0xA0 is 0x%llX] ", top_bitmap_nonblankspace));
      return top_bitmap_space == top_bitmap_nonblankspace;
      }
    else {
#if !(USER_SUPPORT & VGA_MEMORY)
      INTpointer = _VGA_getfont (0, &nbrow, &byteperchar); /* INT 0x1F */
#else
      farptr INTpointer = peekl(4 * 0x1F); /* INT 0x1F */
#endif
      unsigned long long top_bitmap_nonblankspace = peekll (INTpointer + 0x20 * byteperchar);
      VDBG ((" non-blank-space at 0xA0 (INT 0x1F @ 0x%X) is 0x%llX] ", INTpointer, top_bitmap_nonblankspace));
      return top_bitmap_space ==  top_bitmap_nonblankspace;
      }
  }
#endif	/* VGA_SUPPORT | VESA_SUPPORT */

/*
 * The "VGA BIOS library" functions:
 */
#if USER_SUPPORT & VGA_SUPPORT
// not static only to not be removed by compiler:
awk_farret (VGA_getmode);
VIDEO_FCT_PREFIX(VGA_getmode) unsigned
VGA_getmode (void)
//  {bound_stack();{
  {{
  unsigned char nbcol, page, result;

  result = _VGA_getmode (&nbcol, &page);
  if (result & 0x80)
      return 0x8000 | (result & 0x7F);
    else
      return result;
  }}

awk_farret (VGA_setattribute);
VIDEO_FCT_PREFIX(VGA_setattribute) static void
VGA_setattribute (struct attribute_str attr)
  {
  ATTR2BYTE(attr) &= ATTR2BYTE(UI.attributes.valid);

  if (ATTR2BYTE(UI.attributes.valid) != 0) {
      if (UI.parameter.attr.isgraphic) {
	  /* only monochrome graphic (0x0F) mode in VGA */
	  UI.bgcolor = 0;
	  if (attr.blink)
	      UI.fgcolor = 4; /* also highlight... */
	    else if (attr.brighter)
	      UI.fgcolor = 5;
	    else
	      UI.fgcolor = 1;
	  }
	else {
	  /* only monochrome text (0x07) mode in VGA */
	  UI.bgcolor = 0;
	  if (attr.underline)
	      UI.fgcolor = 0x01;
	    else if (attr.reverse)
	      UI.fgcolor = 0x10;
	    else
	      UI.fgcolor = 0x07;	/* see also init of stdcolor array */
	  if (attr.blink)
	      UI.fgcolor |= 0x80;
	  if (attr.brighter)
	      UI.fgcolor |= 0x08;
	  }
      }
  UI.attributes.current = attr;
  }

/* Cursor movement: home at (0,0) */
awk_farret (VGA_setcursor);
VIDEO_FCT_PREFIX(VGA_setcursor) static unsigned
VGA_setcursor (unsigned char row, unsigned char col)
  {
  if (COMMON_setcursor (row, col) == 0)
      _VGA_setcursor (0, row, col);
  return 0;
  }

#if USER_SUPPORT & VGA_USEINT1013
awk_farret (VGA_getcursor);
VIDEO_FCT_PREFIX(VGA_getcursor) static unsigned
VGA_getcursor (unsigned char *row, unsigned char *col)
  {bound_stack();{
  unsigned char cursorstart, cursorend;

  checkptr (row, "VGA_getcursor invalid 1st parameter.");
  checkptr (col, "VGA_getcursor invalid 2nd parameter.");

  /* the BIOS _is_ maintainning the cursor position */
  _VGA_getcursor (0, row, col, &cursorstart, &cursorend);
  return 0;
  }}

VIDEO_FCT_PREFIX(VGA_set_color_and_page) static inline void
VGA_set_color_and_page (unsigned char *color, unsigned char *page)
  {
   if (UI.parameter.attr.isgraphic || UI.parameter.nbcolor == 2) {
#if USER_SUPPORT & VGA_INT1013_BG
      /* 256 colors do not completely work with INT10, %ah=0x13 */
      if (UI.parameter.nbcolor == 256)
	  *page = UI.bgcolor; /* used as background color in 256 color modes */
	else
#endif
	  *page = 0;
      *color = UI.fgcolor;
      }
    else {
      *page = 0;
      *color = (UI.bgcolor << 4) | UI.fgcolor;
      }

  }

VIDEO_FCT_PREFIX(VGA_writechar) static inline void
VGA_writechar (char character, unsigned char page, unsigned char attr,
		unsigned short nbtimes)
  {
  _VGA_writechar_attr (character, page, attr, nbtimes);
  }

VIDEO_FCT_PREFIX(VGA_readchar) static inline void
VGA_readchar (unsigned char *character, unsigned char *attr)
  {
  _VGA_readchar (character, 0, attr);
  }

awk_farret (VGA_putstr);
VIDEO_FCT_PREFIX(VGA_putstr) static void
VGA_putstr (const char *str)
  {bound_stack();{
  unsigned char color, page;

  VGA_set_color_and_page (&color, &page);

  /* str pointer has to be in %es segment ! */
  /* _VGA_writestring do not clear the end of line when printing "\r\n", so do it ourself : */

#if SETUP & UNICODE_FONT
  unsigned char UTF_active = TEST_UTF_ACTIVE();
#endif

  while (*str != 0) {
      UI.parameter.attr.has_outputed = 1;
      char tmpbuff[strlen(str)], *dst = tmpbuff; /* UTF-8 will reduce that string */
      while (*str != '\0' && *str != '\r') {
	  unsigned utfchar, charlen = UTF8LEN(str);	/* handle UTF-8 subset */
	  if (charlen == 2 && (utfchar = UTF8VAL2(str)) < UI.parameter.fontnbchar) {
	      str += charlen;
	      }
	    else if (charlen >= 2) {
	      utfchar = '?';
	      str += charlen;
	      }
	    else
	      utfchar = *str++;

#if SETUP & UNICODE_FONT
#if defined (GAPSTART) && defined (GAPLENGTH)
	  if (UTF_active && utfchar >= GAPSTART && utfchar < GAPSTART + GAPLENGTH)
	      *dst++ = '?';
	    else
#endif
	  if (UTF_active && utfchar >= 0x80) {
	      if ((utfchar >> 7) != UI.parameter.currentUTFpage) {
		  UI.parameter.currentUTFpage = (utfchar >> 7);
		  farptr fontptr = xdata_adr(font8x16);
		  fontptr += UI.parameter.currentUTFpage * 0x80 * UI.parameter.charheight;
#if defined (GAPSTART) && defined (GAPLENGTH)
		  if (utfchar >= GAPSTART)
		      fontptr -= GAPLENGTH * UI.parameter.charheight;
#endif
		  _VGAtxt_load_font (UI.parameter.charheight, 0, 128, 0x80, fontptr);
		  }
	      *dst++ = 0x80 | (utfchar & 0x7F);
	      }
	    else
#endif
	 if (utfchar >= 0x100)
	      *dst++ = '?';
	    else if (utfchar >= 0x7F && !UI.parameter.attr.ansi_font) {
	      const char *ptr = ansi2pc;

	      while (*ptr && *ptr != utfchar)
		  ptr++;
	      if (*ptr != 0)
		  *dst++ = ptr - ansi2pc;
		else
		  *dst++ = '?';
	      }
	    else
	      *dst++ = utfchar;
	  }
      if (dst - tmpbuff != 0) {
	  unsigned char row, col, cursorstart, cursorend;

	  _VGA_getcursor (0, &row, &col, &cursorstart, &cursorend);
	  _VGA_writestring (tmpbuff, dst - tmpbuff, page, color, 1, row, col);
	  }
      if (*str == '\r') {
	  unsigned char row, col, cursorstart, cursorend;

	  _VGA_getcursor (0, &row, &col, &cursorstart, &cursorend);
	  if (col < UI.parameter.nbcol)
	      VGA_writechar (' ', page, color, UI.parameter.nbcol - col);
	  _VGA_writestring ("\r", 1, page, color, 1, row, col);
	  str++;
	  }
      }
  }}

#else  /* VGA_USEINT1013 */

#define VGA_getcursor COMMON_getcursor

#if USER_SUPPORT & VGA_MEMORY
VIDEO_FCT_PREFIX(VGA_writechar) static inline void
VGA_writechar (char character, unsigned char page, unsigned char attr,
			unsigned short nbtimes)
  {
  unsigned offset = (UI.parameter.row * UI.parameter.nbcol
			+ UI.parameter.col) * 2;
  nbtimes ++;
  while (--nbtimes) {
      gpokew (offset, ((unsigned short)attr << 8) | (character & 0xFF));
      offset += 2;
      }
  }

VIDEO_FCT_PREFIX(VGA_readchar) static inline void
VGA_readchar (unsigned char *character, unsigned char *attr)
  {
  unsigned offset = (UI.parameter.row * UI.parameter.nbcol
			+ UI.parameter.col) * 2;
  unsigned short tmp = gpeekw (offset);

  *character = tmp & 0xFF;
  *attr = tmp >> 8;
  }

VIDEO_FCT_PREFIX(VGA_scroll) static inline void
VGA_scroll (int nblines)
  {
  unsigned delta = UI.parameter.nbcol * 2 * nblines, cpt;

  if (nblines > 0)
      for (cpt = delta;
	   cpt < UI.parameter.nbrow * UI.parameter.nbcol * 2U;
	   cpt += 2)
	  gpokew (cpt - delta, gpeekw (cpt));
    else /* untested */
      for (cpt = UI.parameter.nbrow * UI.parameter.nbcol * 2U;
	   cpt >= delta;
	   cpt -= 2)
	  gpokew (cpt, gpeekw (cpt - delta));
  }

VIDEO_FCT_PREFIX(VGA_set_color_and_page) static inline void
VGA_set_color_and_page (unsigned char *color, unsigned char *page)
  {
  *page = 0;
  *color = (UI.bgcolor << 4) | UI.fgcolor;
  }

#else /* VGA_MEMORY */
VIDEO_FCT_PREFIX(VGA_writechar) static inline void
VGA_writechar (char character, unsigned char page, unsigned char attr,
		unsigned short nbtimes)
  {
  /* For the people measuring the quality of software
     by grep-ing the bad words: fuck fuck.
     Sorry, I do not have a very extended vocabulary,
     at least in this language!
   */

#if 1
  _VGA_writechar_attr (character, page, attr, nbtimes);
#elif 0
  _VGA_writechar_color (character, page, attr, nbtimes);
#elif 0
  /* problem of scroll when writing the leftmost lowest char: */
  unsigned char row = UI.parameter.row, col = UI.parameter.col;
  while (nbtimes--)
      _VGA_writestring (&character, 1, page, attr, 0, row, col++);
#endif
  }

VIDEO_FCT_PREFIX(VGA_readchar) static inline void
VGA_readchar (unsigned char *character, unsigned char *attr)
  {
  _VGA_readchar (character, 0, attr);
  }

VIDEO_FCT_PREFIX(VGA_scroll) static inline void
VGA_scroll (int nblines)
  {
  /*
   * Scrolling the screen is not so often working,
   * it is better to do the alternate method...
   */
#if 0 /* to be added again if needed? */
/*
 * Scrolling the screen by writing '\n' on the last line
 * of the screen is not working in DOSEMU over telnet.
 */
  unsigned short version, patchlevel;
  if (_DOSEMU_check(&version, &patchlevel) != 0) {
      while (--nblines >= 0) {
	  _VGA_writestring ("\n", 1, 0, 0, 0, UI.parameter.nbrow-1, 0);
	  }
      }
    else
#endif
      {
      unsigned char filler = 0;
      /* We cannot use the "filler" parameter here because in 256 colors,
	 we do not know if the BIOS is able to manage the "page" parameter
	 as the background color - so then we do not know if the filler
	 should be the background color or zero for graphic 256 color modes. */
      _VGA_scroll (0, 0, UI.parameter.nbrow - 1, UI.parameter.nbcol - 1, filler, nblines);
      }
  }

VIDEO_FCT_PREFIX(VGA_set_color_and_page) static inline void
VGA_set_color_and_page (unsigned char *color, unsigned char *page)
  {
   if (UI.parameter.attr.isgraphic || UI.parameter.nbcolor == 2) {
      if (UI.parameter.nbcolor == 256)
	  *page = UI.bgcolor; /* used as background color in 256 color modes */
	else
	  *page = 0;
      *color = UI.fgcolor;
      }
    else {
      *page = 0;
      *color = (UI.bgcolor << 4) | UI.fgcolor;
      }

  }

#endif /* VGA_MEMORY */

awk_farret (VGA_putstr);
VIDEO_FCT_PREFIX(VGA_putstr) static void
VGA_putstr (const char *str)
  {bound_stack();{
  unsigned char color, page;

  VGA_set_color_and_page (&color, &page);

#if SETUP & UNICODE_FONT
  unsigned char UTF_active = TEST_UTF_ACTIVE();
#endif

  while (*str != '\0') {
      UI.parameter.attr.has_outputed = 1;
      switch (*str++) {
	case '\007': /* ^G */
	  BOOT1_putstr ("\007");
	  break;
	case '\010':
	  if (UI.parameter.col > 0)
	      UI.parameter.col -= 1;
	  break;
	default:
	  str--; /* restore str pointer if not ^G, ^H, \n, \r */
	  unsigned utfchar, charlen = UTF8LEN(str);	/* handle UTF-8 subset */
	  if (charlen == 2 && (utfchar = UTF8VAL2(str)) < UI.parameter.fontnbchar) {
	      str += charlen;
	      }
	    else if (charlen >= 2) {
	      utfchar = '?';
	      str += charlen;
	      }
	    else
	      utfchar = *str++;

#if SETUP & UNICODE_FONT
#if defined (GAPSTART) && defined (GAPLENGTH)
	  if (UTF_active && utfchar >= GAPSTART && utfchar < GAPSTART + GAPLENGTH)
	      utfchar = '?';
	    else
#endif
	  if (UTF_active && utfchar >= 0x80) {
	      if ((utfchar >> 7) != UI.parameter.currentUTFpage) {
		  UI.parameter.currentUTFpage = (utfchar >> 7);
		  farptr fontptr = xdata_adr(font8x16);
		  fontptr += UI.parameter.currentUTFpage * 0x80 * UI.parameter.charheight;
#if defined (GAPSTART) && defined (GAPLENGTH)
		  if (utfchar >= GAPSTART)
		      fontptr -= GAPLENGTH * UI.parameter.charheight;
#endif
		  _VGAtxt_load_font (UI.parameter.charheight, 0, 128, 0x80, fontptr);
		  }
	      utfchar = 0x80 | (utfchar & 0x7F);
	      }
	    else
#endif
	 if (utfchar >= 0x100)
	      utfchar = '?';
	    else if (utfchar >= 0x7F && !UI.parameter.attr.ansi_font) {
	      const char *ptr = ansi2pc;

	      while (*ptr && (unsigned char)*ptr != utfchar)
		  ptr++;
	      if (*ptr != 0)
		  utfchar = ptr - ansi2pc;
		else
		  utfchar = '?';
	      }
	  VGA_writechar (utfchar, page, color, 1);
	  if (++UI.parameter.col >= UI.parameter.nbcol) {
	      UI.parameter.col = 0;	/* automatic CRLF */
	case '\n':  /* it is valid in C */
	      if (++UI.parameter.row >= UI.parameter.nbrow) {
		  UI.parameter.row--;
		  VGA_scroll (1);
		  UI.parameter.attr.has_scrolled = 1;
	case '\r': /* it is valid in C */
		  VGA_writechar (' ', page, color,
			UI.parameter.nbcol - UI.parameter.col);
		  UI.parameter.col = 0;
		  }
	      }
	  break;
	}
      _VGA_setcursor (0, UI.parameter.row, UI.parameter.col);
      }
  }}
#endif /* VGA_USEINT1013 */

VIDEO_FCT_PREFIX(VGA_graphic_setpixel) static void FASTCALL
VGA_graphic_setpixel (coord xy, unsigned color)
  {
  _VGA_setpixel (xy.x, xy.y, color, 0);
  }

VIDEO_FCT_PREFIX(VGA_graphic_getpixel) static unsigned FASTCALL
VGA_graphic_getpixel (coord xy)
  {
  return _VGA_getpixel (xy.x, xy.y, 0);
  }

VIDEO_FCT_PREFIX(VGA_graphic_plotHline) static void
VGA_graphic_plotHline (coord xy, unsigned short xend, unsigned color)
  {
  while (xy.x < xend) {
      _VGA_setpixel (xy.x, xy.y, color, 0);
      xy.x++;
      }
  }

VIDEO_FCT_PREFIX(VGA_text_setpixel) static void FASTCALL
VGA_text_setpixel (coord xy, unsigned color)
  {
  unsigned char row, col;

  VGA_getcursor (&row, &col);
  VGA_setcursor (xy.y, xy.x);
  VGA_writechar (color & 0xFF, 0, (color >> 8) & 0xFF, 1);
  VGA_setcursor (row, col);
  }

VIDEO_FCT_PREFIX(VGA_text_getpixel) static unsigned FASTCALL
VGA_text_getpixel (coord xy)
  {
  unsigned char row, col, letter, attr;

  VGA_getcursor (&row, &col);
  VGA_setcursor (xy.y, xy.x);
  VGA_readchar(&letter, &attr);
  VGA_setcursor (row, col);
  return ((unsigned)attr << 8) | letter;
  }

VIDEO_FCT_PREFIX(VGA_text_plotHline) static void
VGA_text_plotHline (coord xy, unsigned short xend, unsigned color)
  {
  unsigned char row, col;

  VGA_getcursor (&row, &col);
  VGA_setcursor (xy.y, xy.x);
  VGA_writechar (color & 0xFF, 0, (color >> 8) & 0xFF, xend - xy.x);
  VGA_setcursor (row, col);
  }

awk_farret (VGA_clearscreen);
VIDEO_FCT_PREFIX(VGA_clearscreen) static void
VGA_clearscreen (void)
  {
/*
 * The scroll method to clear the screen is only working
 * when the scroll BIOS is working, which is not so usual...
 */
#if 0
  unsigned char color, page;

  VGA_set_color_and_page (&color, &page);

  _VGA_scroll (0, 0, UI.parameter.nbrow-1, UI.parameter.nbcol-1, color, 0);
#else
  signed short cpt;
  unsigned char color, page;

  VGA_set_color_and_page (&color, &page);

  for (cpt = 0; cpt < UI.parameter.nbrow; cpt++) {
      VGA_setcursor (cpt, 0);
      VGA_writechar (' ', page, color, UI.parameter.nbcol);
      }

  if (UI.parameter.height % UI.parameter.charheight != 0) {
      /* 800x600 with background color and 8x16 chars, half
	 a line left. hopes that will not crash anything */
      VGA_setcursor (cpt, 0);
      VGA_writechar (' ', page, color, UI.parameter.nbcol);
      }
#endif
  VGA_setcursor (0, 0);
  }

/*
 * Use out of line versions of these functions because they are
 * a bit long:
 */
VIDEO_FCT_PREFIX(VGA_safe_setmode) static void
VGA_safe_setmode (unsigned char mode)
  {
  _VGA_safe_setmode (mode);
  }

VIDEO_FCT_PREFIX(VGA_safe_writechar_attr) static void
VGA_safe_writechar_attr (unsigned char character, unsigned char page,
			 unsigned char attr)
  {
  _VGA_safe_writechar_attr (character, page, attr, 1);
  }

#if !(USER_SUPPORT & VGA_MEMORY)
VIDEO_FCT_PREFIX(VGA_safe_readchar) static void
VGA_safe_readchar (unsigned char *character, unsigned char *attr)
  {
  _VGA_safe_readchar (character, 0, attr);
  }

/* we do a "_VGA_safe_setcursor (0, 0, 1);", and
 * to correctly detect "graphicbgcolor", we should
 * have coordinate correctly chosen here, whatever
 * the font height:
 */
#define COORD_X		10 /* on the second char of the screen */
#define COORD_Y		6  /* on the first line, font height >= 8 */
#define COORD_PAGE	0
#define TESTCOLOR	3

VIDEO_FCT_PREFIX(VGA_safe_getpixel) static unsigned char
VGA_safe_getpixel (void)
  {
  return _VGA_safe_getpixel (COORD_X, COORD_Y, COORD_PAGE);
  }

VIDEO_FCT_PREFIX(VGA_safe_setpixel) static void
VGA_safe_setpixel (unsigned char color)
  {
  _VGA_safe_setpixel (COORD_X, COORD_Y, color, COORD_PAGE);
  }

#undef COORD_X
#undef COORD_Y
#undef COORD_PAGE

#endif

awk_farret (VGA_getsize);
VIDEO_FCT_PREFIX(VGA_getsize) static unsigned
VGA_getsize (unsigned mode, struct video_parameter_str *param)
  {bound_stack();{
  volatile unsigned char oldmode; /* because setjmp() */
  unsigned char newmode, nbcol, page;
  unsigned error = 0;
  functionality_table_t table = {0};
  unsigned char cursorstart, cursorend;

  VDBG (("%s mode 0x%X: ", __FUNCTION__, mode));

  checkptr (param, "VGA_getsize invalid parameter.");

  memset (param, 0, sizeof(*param));
  param->identification = mode & 0x7F;

  if (!VIDEO_mode_is_valid (param->identification)) {
      VDBG (("refused from validmode bitfield.\r\n"));
      return 0x100;
      }
  VIDEO_mode_invalidate (param->identification); /* may be revalidated later */
#if SETUP & UPDATE_BOOTPARAM
  BOOT1_update_bootparam (); /* if the PC crash, it will not be tried again */
#endif
  _VGA_getcursor (0, &param->row, &param->col, &cursorstart, &cursorend);

  oldmode = _VGA_getmode (&nbcol, &page) & 0x7F;
  if (oldmode != param->identification) {
#ifdef TREAT_EXCEPTION
      if (shit_handler (1, 4) != 0) {
//	  if (_VIDEO_cold_initialise() != 0)
//	      VGA_safe_setmode (oldmode);
	  /* may reset more parameters: */
	  _VGA_safe_autoload_default_palette (1);
	  VGA_safe_setmode (oldmode | 0x80);
	  if ((void *)VGA_VESA_setup_color != (void *)COMMON_error)
	      VGA_VESA_setup_color();
	    else
	      DBG ((" [cannot call VGA_VESA_setup_color] "));
	  _VGA_blinking (0); /* it is re-enabled by INT10/setmode */
	  VDBG (("have hit an exception, like division by 0"
		 " or invalid instruction, or timeout (4 second).\r\n"));
	  return 0x800;
	  }
#endif
       /* do not reset palette in the following setmode (can be long): */
      if (!_VGA_autoload_default_palette (0))
	  VDBG ((" [autoload_default_palette not supported!] "));

      /*
       * Ater next _VGA_setmode, we have to be very carefull,
       * because we are maybe in an invalid mode - where
       * some VGA primitives modify unusual registers or even memory!
       * (old S3 Elsa Winner card) - use the VGA_safe_*() functions.
       */
      VGA_safe_setmode (mode | 0x80); /* try to not clear screen */
      newmode = _VGA_safe_getmode (&nbcol); /* will check nbcol later */

      if (((unsigned)newmode & 0x7F) != param->identification) {
	  VDBG (("_VGA_setmode (0x%X) and _VGA_getmode (0x%X) differ.\r\n",
		param->identification, newmode & 0x7F));
	  error = 0x101;
	  }
      }

#if 0	/* to test that setjmp/longjmp rescue is working */
  asm volatile (
"	cmpl	$2,%0		\n"
"	jne	1f		\n"
"	xor     %%ax,%%ax	\n"
"	mov     %%ax,%%ds	\n"
"	mov     %%ax,%%es	\n"
"	ud2			\n"
"	1:			\n"
"	cmpl	$1,%0		\n"
"	je	1b		\n"
	: : "g" (mode));
#endif

  if (   error == 0
      && (   _VGA_get_functionality_info (&table) != 0
	  || param->identification != table.current_mode
	  || nbcol != table.nbcolumn)) {
      unsigned char BIOS_data_video_mode = peekb (0x00400049);
      /* That can be an EGA video card, just do a basic answer for current mode only: */
      VDBG (("wrong _VGA_get_functionality_info (id: 0x%X vs 0x%X,"
		 " nbcol %u vs %u) and BIOS_data_video_mode = 0x%X.\r\n",
		 param->identification, table.current_mode,
		 nbcol, table.nbcolumn,
		 BIOS_data_video_mode));
      if (   param->identification == BIOS_data_video_mode
	  && param->identification <= 0x13) {
	  static const unsigned short EGA_nbcolor_per_mode[] = {
		/* 0..7: */      16, 16, 16, 16, 4, 4, 1, 0,
		/*invalid: */   255, 255, 255, 255, 255,
		/* 0xD..0x13: */ 16, 16, 0, 16, 1, 16, 256
		};
	  VDBG (("Probably EGA card or less, no autodetect but can get from "
			"BIOS DATA area, assume %u colors for mode %u.\r\n",
			EGA_nbcolor_per_mode[BIOS_data_video_mode], BIOS_data_video_mode));
	  table.current_mode = BIOS_data_video_mode;
	  table.nonVGAmodes.adapter_required = 0;
	  table.nbcolor = EGA_nbcolor_per_mode[BIOS_data_video_mode];
	  table.nbcolumn = peekw (0x0040004A);
	  table.nbrow = peekb (0x00400084) + 1;
	  table.nbbyteperchar = peekw (0x00400085);
	  table.regenbufferlen = peekw (0x0040004C);
	  table.regenbufferstart = peekw (0x0040004E);
	  error = 0;
	  }
	else {
	  VDBG (("Probably EGA card, cannot get from BIOS DATA area.\r\n"));
	  error = 0x102;
	  }
      }

  if (error == 0 && table.nonVGAmodes.adapter_required) {
      VDBG (("needs an adapter.\r\n"));
      error = 0x103;
      }

  if (error == 0) {
#if USER_SUPPORT & VGA_MEMORY
      /* This accepts only TEXT modes: */
      _VGA_safe_setcursor (0, 0, 0);
      VGA_safe_writechar_attr ('@', 0, 7);
      if (peekb (0xB8000000U) != '@')
	  error = 0x104;
      VGA_safe_writechar_attr ('!', 0, 7);
      if (peekb (0xB8000000U) != '!')
	  error = 0x105;
      param->base_address = 0xB8000;
      param->attr.isgraphic = 0;
      param->attr.graphicbgcolor = 0;
#else
      unsigned char charread, charsaved, attrread, attrsaved;

      /* Detect (in a BIOS way) if this is a graphic mode:
       * By writing a char in the background color (which
       * is 0 in graphic modes, at least if page == 0) - If we
       * can read it back, it is a text mode.
       * Take care, with 2 color modes, the writing color
       * is ignored (always set to 1) so check attribute also.
       * This can crash VIDEO BIOS of very old cards.
       */
      _VGA_safe_setcursor (0, 0, 1);
      VDBG (("["));
      VGA_safe_readchar (&charsaved, &attrsaved); /* not change cursor */
      VDBG (("G"));
      VGA_safe_writechar_attr ('@', 0, 0); /* not change/update cursor */
      VDBG (("?"));
      VGA_safe_readchar (&charread, &attrread);
      if (charread != '@') {
	  /* It is a graphic mode, but check that s/getpixel are working,
	     else treat it as a text mode: */
	  unsigned char color = VGA_safe_getpixel ();
	  VGA_safe_setpixel (0);
	  if (   (VGA_safe_setpixel (0), VGA_safe_getpixel () != 0)
	      || (VGA_safe_setpixel (1), VGA_safe_getpixel () != 1)) {
	      VDBG (("y, but no s/getpixel()] "));
	      VGA_safe_setpixel (color);
	      param->base_address = 0xA0000;
	      param->attr.isgraphic = 0;
	      param->attr.graphicbgcolor = 0;
	      }
	    else {
	      VDBG (("y] "));
	      param->base_address = 0xA0000;
	      param->attr.isgraphic = 1;
#if !(USER_SUPPORT & VGA_USEINT1013) || (USER_SUPPORT & VGA_INT1013_BG)
	      /* Is it graphic and can it draw bgcolor? maybe */
	      if (table.nbcolor == 256) {
		  VDBG (("[bgcolor"));
		  /* not change/update cursor, but write in page
		     TESTCOLOR, will it change bgcolor of page 0 ? */
		  VGA_safe_writechar_attr (' ', TESTCOLOR, 0);
		  VDBG (("? "));
		  color = VGA_safe_getpixel ();
		  VDBG (("read %u ", color));
		  if (color == TESTCOLOR) {
		      param->attr.graphicbgcolor = 1;
		      VDBG (("yes] "));
		      }
		    else {
		      param->attr.graphicbgcolor = 0;
		      VDBG (("no] "));
		      }
		  }
		else
#endif
		  param->attr.graphicbgcolor = 0;
	      }
	  }
	else if (attrread != 0) {
	  VDBG (("y - 2 colors] "));
	  table.nbcolor = 2;
	  param->base_address = 0xA0000;
	  param->attr.isgraphic = 1;
	  /* Is it graphic and can it draw bgcolor? no */
	  param->attr.graphicbgcolor = 0;
	  }
	else {
	  VDBG (("n] "));
	  param->base_address = 0xB8000;
	  param->attr.isgraphic = 0;
	  /* Is it graphic and can it draw bgcolor? no */
	  param->attr.graphicbgcolor = 0;
	  if (table.nbcolor > 16) {
	      VDBG (("<correcting textmode table.nbcolor = %u -> 16> ",
			table.nbcolor));
	      table.nbcolor = 16;
	      }
	  }
      VGA_safe_writechar_attr (charsaved, 0, attrsaved);

      if (   table.nbcolor != 0
	  && (   table.nbcolor > 256
	      || table.nbcolor != (1 << (__builtin_ffs (table.nbcolor) - 1)))) {
#if 1
	  /*
	   * I am fed up. The VGA BIOS should be publically described,
	   * even its latest revision... There are hidden fields here.
	   * I will not sign a NDA to know what it is.
	   */
	  error = 0x107;
	  VDBG (("<invalid table.nbcolor: %u>\r\n ", table.nbcolor));
#else
	  /* Another VIDEO BIOS which is lying...
	     and the number of color cannot be safely detected. */
	  VDBG ((" [BIOS say nbcolor = %u, set it to ", table.nbcolor));
	  if (!param->attr.isgraphic) {
	      VDBG (("text16] "));
	      table.nbcolor = 16;
	      }
	    else {
	      unsigned savecolor, color;

	      VDBG ((" <getpixel"));
	      savecolor = VGA_safe_getpixel ();
	      VDBG ((", setpixel"));
	      VGA_safe_setpixel (2);
	      VDBG ((", getpixel"));
	      color = VGA_safe_getpixel ();

	      if (color != 2) {
		  VDBG (("> (2->%u) 2] ", color));
		  table.nbcolor = 2;
		  }
		else {
		  VDBG ((", setpixel"));
		  VGA_safe_setpixel (17);
		  VDBG ((", getpixel"));
		  color = VGA_safe_getpixel ();
		  if (color != 17) {
		      VDBG (("> (17->%u) 16] ", color));
		      table.nbcolor = 16;
		      }
		    else {
		      VDBG (("> 256] "));
		      table.nbcolor = 256;
		      }
		  }
	      VDBG ((" <setpixel"));
	      VGA_safe_setpixel (savecolor);
	      VDBG ((">"));
	      }
#endif
	  }
#endif /* VGA_MEMORY */
      }

  if (oldmode != param->identification)
      VGA_safe_setmode (oldmode | 0x80);

#ifdef TREAT_EXCEPTION
  shit_handler (0, 4);
#endif

  if (oldmode != param->identification) {
      if ((void *)VGA_VESA_setup_color != (void *)COMMON_error)
	  VGA_VESA_setup_color(); /* redo shit_handler() */
	else
	  DBG ((" [cannot call VGA_VESA_setup_color] "));
       /* do reset palette in other setmode: */
      if (!_VGA_autoload_default_palette (1))
	  VDBG ((" [autoload_default_palette not supported!] "));
      _VGA_blinking (0); /* it is re-enabled by INT10/setmode */
//      _VGAtxt_set_block_specifier (0, 0);
      }

  _VGA_setpage (0); /* necessary ? */
  _VGA_setcursor (0, param->row, param->col);	/* force it */

  if (error != 0) {
      return error;
      }

  param->nbcol = table.nbcolumn;
#if 0 /* It has been corrected - thanks */
  {
  unsigned short version, patchlevel;
  if (_DOSEMU_check(&version, &patchlevel) == 0) { /* bug in DOSEMU, see
      http://marc.theaimsgroup.com/?l=linux-msdos&m=98830103525815&w=2
      */
      table.nbrow = peekb (0x00400084) + 1;
      }
  }
#endif
  param->nbrow = table.nbrow;
  param->nbcolor = table.nbcolor;
  VDBG (("%ux%u, %u colors/attributes, regenlen: %u,"
	 " regenstart: %u, nbscanline: %u, %u bytes/char,"
	 " primarycharblock = %u, secondarycharblock = %u, ",
	table.nbcolumn, table.nbrow, table.nbcolor,
	table.regenbufferlen, table.regenbufferstart,
	table.nbscanline, table.nbbyteperchar,
	table.primarycharblock, table.secondarycharblock));

  param->charwidth = 8;	/* nothing else on BIOS */
  param->charheight = table.nbbyteperchar;

  if (param->attr.isgraphic) {
      param->height = table.nbrow * param->charheight;
      param->width = table.nbcolumn * param->charwidth;
      if (param->height == 592 && param->charheight == 16)
	  param->height = 600;
      VDBG (("graphic, width = %u, height = %u", param->width, param->height));
      }
    else {
      param->height = table.nbrow;	/* needed for text mouse */
      param->width = table.nbcolumn;
      VDBG (("text, width = %u, height = %u", param->width, param->height));
      }
  VIDEO_mode_revalidate (param->identification);

#ifndef KEEP_ALL_VGA_MODE
  if (param->identification <= 0x37 || param->nbcolor > 4)
#endif
      {
      struct videomode_str videomode = {
	  .number = param->identification,
	  .width = param->width,
	  .height = param->height,
	  .text = !param->attr.isgraphic,
	  .vesa = 0,
	  .bpp = (param->nbcolor)? __builtin_ffs (param->nbcolor) - 1 : 0,
	  .attr = param->attr.isgraphic,
	  };
      add_valid_mode (&videomode);
      }

#if SETUP & UPDATE_BOOTPARAM
  BOOT1_update_bootparam (); /* no PC crash, OK */
#endif
  VDBG ((", end getsize.\r\n"));
  return 0;
  }}

// not static only to not be removed by compiler:
awk_farret (VGA_setmode);
VIDEO_FCT_PREFIX(VGA_setmode) unsigned
VGA_setmode (unsigned mode)
  {bound_stack();{
  struct video_parameter_str param;
  unsigned saved_mode = mode, tmp;

  mode &= ~0x8080;

  if (mode > 0x7F) {
      VDBG (("%s called with mode 0x%X !\r\n", __FUNCTION__, mode));
      return 1;
      }
  if ((tmp = VGA_getsize (mode, &param)) != 0)
      return tmp;

#if USER_SUPPORT & VESA_HARDWINDOW
  free_hardwin();
#endif

  UI.parameter = param;
  // UI.function.getsize, UI.function.setmode, UI.function.getmode : already initialised
  // init local functions without far-call convention:
  if (UI.parameter.attr.isgraphic) {
      UI.function.setpixel = VGA_graphic_setpixel;
      UI.function.getpixel = VGA_graphic_getpixel;
      UI.function.plotHline = VGA_graphic_plotHline;
      }
  else {
      UI.function.setpixel = VGA_text_setpixel;
      UI.function.getpixel = VGA_text_getpixel;
      UI.function.plotHline = VGA_text_plotHline;
      }
  // init functions with far-call convention:
  extern void (*fptr_VGA_clearscreen) (void);
  asm ("# function %0 used " : : "g" (VGA_clearscreen));
  extern unsigned (*fptr_VGA_setcursor) (unsigned char row, unsigned char col);
  asm ("# function %0 used " : : "g" (VGA_setcursor));
#if USER_SUPPORT & VGA_USEINT1013
  extern unsigned (*fptr_VGA_getcursor) (unsigned char *row, unsigned char *col);
  asm ("# function %0 used " : : "g" (VGA_getcursor));
#else
  extern unsigned (*fptr_COMMON_getcursor) (unsigned char *row, unsigned char *col);
#endif
  extern unsigned (*fptr_COMMON_setfgcolor) (unsigned color);
  asm ("# function %0 used " : : "g" (COMMON_setfgcolor));
  extern unsigned (*fptr_COMMON_setbgcolor) (unsigned color);
  asm ("# function %0 used " : : "g" (COMMON_setbgcolor));
  extern void (*fptr_VGA_setattribute) (struct attribute_str attr);
  asm ("# function %0 used " : : "g" (VGA_setattribute));
  extern void (*fptr_VGA_putstr) (const char *str);
  asm ("# function %0 used " : : "g" (VGA_putstr));
  extern unsigned (*fptr_BIOS_getkey) (unsigned timeout);
  asm ("# function %0 used " : : "g" (BIOS_getkey));

  UI.function.clearscreen = *fptr_VGA_clearscreen;
  UI.function.setcursor = *fptr_VGA_setcursor;
#if USER_SUPPORT & VGA_USEINT1013
  UI.function.getcursor = *fptr_VGA_getcursor;
#else
  UI.function.getcursor = *fptr_COMMON_getcursor;
#endif
  UI.function.setfgcolor = *fptr_COMMON_setfgcolor;
  UI.function.setbgcolor = *fptr_COMMON_setbgcolor;
  UI.function.setattribute = *fptr_VGA_setattribute;
  UI.function.putstr = *fptr_VGA_putstr;
  UI.function.getkey = *fptr_BIOS_getkey;

  {
  unsigned char nbcol, page;

  if ((_VGA_getmode (&nbcol, &page) & 0x7FU) != mode)
      _VGA_setmode (mode | 0x80);	/* do not clearscreen now */
  }

 /*
  * NOTE: some video card have an INCOMPLETE or CORRUPTED 8x14 font!
  * at least one has this font up to the char 'z', no more !
  */

  UI.parameter.attr.ansi_font = VGA_VESA_current_font_ansi ();
#if (USER_SUPPORT & VESA_SUPPORT)
  UI.parameter.fontnbchar = 256;
#endif
  if (copy_gujin_param.attrib.use_gujin_embedded_font) {
  /* downloading font does not seem to work on MDA/CGA cards... */
//  if (   copy_gujin_param.attrib.use_gujin_embedded_font) {
//      && (   UI.videocard._VGA_get_display_combination_code_works
//	  || UI.videocard._EGA_getdisplay_works)) {
      farptr fontptr;

      switch (UI.parameter.charheight) {
	  case 8: fontptr = xdata_adr(font8x8); break;
	  case 14: fontptr = xdata_adr(font8x14); break;
	  case 16: fontptr = xdata_adr(font8x16); break;
	  default: fontptr = 0; break;
	  }
      if (fontptr == 0) {
	  VDBG ((" [after _VGA_setmode, UI.parameter.charheight = %u, "
			"none of 8,14 or 16, cannot load font!] ",
			UI.parameter.charheight));
	  }
	else { /* always (UI.parameter.font == 0) in VGA */
	  VDBG ((" [loading ANSI hard %s font 8x%u ",
		UI.parameter.attr.isgraphic? "graphic" :"text", UI.parameter.charheight));
	  if (   UI.parameter.attr.isgraphic
	     /* On some video cards, VGA modes with 1 Bit Per Pixel are considered by
		Gujin as TEXT modes because their "read pixel" BIOS is not working,
		but we still have to load the graphic font: */
	      || _VGA_isgraphic()
	      ) {
		/* INT 1F - SYSTEM DATA - 8x8 GRAPHICS FONT, upper part:
			   (needed for mode 0x4, 0x5, 0x6) */
	      if (UI.parameter.charheight == 8)
		  _VGA_set_user_8x8_high_graphic_char (fontptr + 128 * 8);
	      /* This table needs to be contiguous for the 256 chars... */
	      _VGA_set_user_graphic_char (fontptr, UI.parameter.charheight, UI.parameter.nbrow);
	      }
	    else {
	      _VGAtxt_load_font (UI.parameter.charheight, 0, 256, 0, fontptr);
#if SETUP & UNICODE_FONT
	      if (UI.parameter.charheight == 16)
		  UI.parameter.fontnbchar = NBCHAR8x16;
	      if (UI.parameter.charheight == 20)
		  UI.parameter.fontnbchar = NBCHAR10x20;
#endif
	      }
	  VDBG (("OK]"));
	  UI.parameter.currentUTFpage = 1;
	  UI.parameter.attr.ansi_font = 1;
	  UI.parameter.attr.null_del_available = 1;
	  }
      }
    else {
      /* We _need_ to load the default font, in case previously we uploaded our own font */
#if 1
      if (UI.parameter.charheight == 16) {
	  if (UI.parameter.attr.isgraphic)
	      _VGA_loadgraphic_ROM8x16 (UI.parameter.nbrow);
	    else
	      _VGAtxt_load_ROM8x16 (0);
	  }
	else if (UI.parameter.charheight == 14) {
	  if (UI.parameter.attr.isgraphic)
	      _VGA_loadgraphic_ROM8x14 (UI.parameter.nbrow);
	    else
	      _VGAtxt_load_ROM8x14 (0);
	  }
	else if (UI.parameter.charheight == 8) {
	  if (UI.parameter.attr.isgraphic)
	      _VGA_loadgraphic_ROM8x8 (UI.parameter.nbrow);
	    else
	      _VGAtxt_load_ROM8x8 (0);
	  }
#else
      farptr fontptr;
      unsigned short byteperchar;
      unsigned char nbrow;
      if (UI.parameter.charheight == 16)
	  fontptr = _VGA_getfont (6, &nbrow, &byteperchar); /* 8x16 */
	else if (UI.parameter.charheight == 14)
	  fontptr = _VGA_getfont (2, &nbrow, &byteperchar); /* 8x14 */
	else /* UI.parameter.charheight == 8 */
	  fontptr = _VGA_getfont (3, &nbrow, &byteperchar); /* 8x8 */
      if (UI.parameter.attr.isgraphic || _VGA_isgraphic()) {
	  _VGA_set_user_graphic_char (fontptr, UI.parameter.charheight, nbrow + 1);
	  if (UI.parameter.charheight == 8) {
	      fontptr = _VGA_getfont (4, &nbrow, &byteperchar); /* 8x8 high */
	      _VGA_set_user_8x8_high_graphic_char (fontptr);
	      }
	  }
	else
	  _VGAtxt_load_font (UI.parameter.charheight, 0, 256, 0, fontptr);
#endif
      }

  _VGA_setcursor (0, UI.parameter.row, UI.parameter.col); /* but redo that */

#if ASSEMBLY_TYPE == ASSEMBLY_DSES
  UI.parameter.base_address = linear2dsrelative (UI.parameter.base_address);
#else
#if USER_SUPPORT & VGA_MEMORY
  setgs (UI.parameter.base_address >> 4);
#endif
#endif

  VGA_VESA_setup_color ();

#if USER_SUPPORT & VGA_MEMORY
  if (UI.parameter.nbcolor != 0) {
      UI.parameter.nbcolor = 16;
      UI.attributes.valid = no_attribute;
      UI.attributes.current = no_attribute;
      UI.function.setfgcolor (UI.stdcolor[blue]);
      UI.function.setbgcolor (UI.stdcolor[lightgray]);
      _VGA_blinking (0); /* disable it */
//      _VGAtxt_set_block_specifier (0, 0);
      }
    else {
      UI.function.setfgcolor (1);
      UI.bgcolor = 0;
      UI.attributes.valid = egatxt_attribute;
      UI.function.setattribute (no_attribute);
      }
#else /* VGA_MEMORY */
  {
  unsigned fgcolor, bgcolor;
  if (UI.parameter.nbcolor > 2) {
      _VGA_blinking (0); /* disable it */
//      _VGAtxt_set_block_specifier (0, 0);
      UI.attributes.valid = no_attribute;
      UI.attributes.current = no_attribute;

      if (UI.parameter.attr.isgraphic && UI.parameter.nbcolor != 256) {
	  /* graphic modes with > 1 page: BIOS do not have background color */
	  fgcolor = UI.stdcolor[cyan];
	  bgcolor = UI.stdcolor[black];
//	  _VGA_setbackground (UI.stdcolor[lightgray]); /* only whole screen */
	  }
	else {
	  fgcolor = UI.stdcolor[blue];
	  bgcolor = UI.stdcolor[lightgray];
	  }
      }
    else {
      if (UI.parameter.nbcolor == 0) {
	  if (UI.parameter.attr.isgraphic)
	      UI.attributes.valid = egagfx_attribute;
	    else
	      UI.attributes.valid = egatxt_attribute;
	  UI.function.setattribute (no_attribute);
	  }
      fgcolor = UI.stdcolor[white];
      bgcolor = UI.stdcolor[black];
      }
  UI.function.setfgcolor (fgcolor);
  UI.function.setbgcolor (bgcolor);
  }
#endif /* VGA_MEMORY */

  if ((saved_mode & 0x8080) == 0)
      UI.function.clearscreen (); /* with the right color */

  VDBG (("success VGA_setmode\r\n"));
  return 0;
  }}

VIDEO_FCT_PREFIX(VGA_init) unsigned
VGA_init (void)
  {bound_stack();{
  functionality_table_t	tmptable;
  unsigned returned;

  VDBG (("%s: ", __FUNCTION__));

  tmptable.static_functionality_table = 0;
  returned = _VGA_get_functionality_info (&tmptable);
  if (returned != 0 || tmptable.static_functionality_table == 0) {
      unsigned char nbcol, page, i;

      if (returned != 0)	/* for instance EGA only card: */
	  VDBG (("FAILED, VGA get functionality info not supported."));
	else			/* for instance DOSEMU w/o console: */
	  VDBG (("FAILED, VGA get functionality gives static_functionality_table at address 0."));

     /* We do not allow probing video mode - it would be possible to
	probe but for the excluded mode numbers and for the number of colors */
     for (i = 0; i <= VIDEO_mode_max(); i++)
	  VIDEO_mode_invalidate (i);
      VIDEO_mode_revalidate (_VGA_getmode (&nbcol, &page));
      }
    else {
      unsigned i, extended_inited;
      static_functionality_table_t	tmpstatictable;

      VDBG (("VGA detected, static table at 0x%X",
		tmptable.static_functionality_table));
      UI.videocard._VGA_get_functionality_info_works = 1;

      lmemcpy (&tmpstatictable, tmptable.static_functionality_table,
		sizeof (static_functionality_table_t));

      /*
       * Should we enable detection of modes 0x13..0x37 & 0x38..0x7F ?
       * Most VIDEO BIOS fill "mode_supported" only for first 0x13 modes.
       * This should be only usefull when there is no VESA BIOS; and VESA
       * modes are now only mapped in the VGA number to get "clearscreen"
       * compatibility working (get video mode number, set the same
       * video mode with bit 7 cleared).
       * BUT there is also some interresting text modes, which are
       * usually not mapped in VESA (132 columns, more than 25 lines...).
       * Clearly, some VIDEO BIOS will crash when setting/using some
       * of those extended modes.
       */
      VDBG ((", modes:"));
      extended_inited = (tmpstatictable.mode_supported[2] & 0xF0) != 0;
      for (i = 0; i < nbof(tmpstatictable.mode_supported); i++) {
	  VDBG ((" 0x%X,", tmpstatictable.mode_supported[i]));
	  if (i > 2)
	      extended_inited |= tmpstatictable.mode_supported[i] != 0;
	  }
      if (extended_inited)
	  extended_inited = 0x7F;
	else
	  extended_inited = 0x13;

      for (i = 0; i <= extended_inited; i++) {
	  if ((tmpstatictable.mode_supported[i / 8] & (1 << (i % 8))) == 0)
	      VIDEO_mode_invalidate (i); /* do not revalidate others */
	  }

      if (_PARADISE_installation_check() != -1) {
	  VDBG ((" [Paradise: disable mode 0x5B] "));
	  /* This mode crashes at VGA_getsize() (lidt done ?) */
	  VIDEO_mode_invalidate (0x5B);
	  }

#if SETUP & UPDATE_BOOTPARAM
      BOOT1_update_bootparam (); /* save all unsupported modes once */
#endif
      } /* DOSEMU */

#if USER_SUPPORT & VESA_SUPPORT
  if (VESA_ACTIVE() && !UI.parameter.BIOS_compatible) {
      if (VGA_setmode (0x8003) != 0) /* for instance VGA_MEMORY */
	  VDBG ((" [Initial VGA_setmode failed!] "));
      }
#endif

  /* EGA seems to work a bit with them - still testing */
  extern unsigned (*fptr_VGA_getsize) (unsigned mode, struct video_parameter_str *param);
  asm ("# function %0 used " : : "g" (VGA_getsize));
  extern unsigned (*fptr_VGA_setmode) (unsigned mode);
  asm ("# function %0 used " : : "g" (VGA_setmode));
  extern unsigned (*fptr_VGA_getmode) (void);
  asm ("# function %0 used " : : "g" (VGA_getmode));
  UI.function.getsize = *fptr_VGA_getsize;
  UI.function.setmode = *fptr_VGA_setmode;
  UI.function.getmode = *fptr_VGA_getmode;

  VDBG (("\r\n"));

  return returned;
  }}

#if SETUP & UPDATE_BOOTPARAM
VIDEO_FCT_PREFIX(setup_gujin_param_vga_mode) void
setup_gujin_param_vga_mode (void)
  {
  __attribute__ ((unused)) gujin_param_t *dummy;
  unsigned i, index = 0, max = nbof (dummy->vga_mode), test;

  for (test = 0; test < 4; test++) {
      for (i = 0; i < UI.nbmode && index < max; i++) {
	  if (UI.mode[i].vesa)
	      continue;
	  if (   (test == 0) ? (UI.mode[i].text && UI.mode[i].bpp == 4)
	      : ((test == 1) ? (UI.mode[i].bpp == 8)
	      : ((test == 2) ? (!UI.mode[i].text && UI.mode[i].bpp == 4)
	      : /*(test == 3) ?*/ (UI.mode[i].bpp < 4)))
	      ) {
	      /* save all text mode first */
	      struct vga_mode_str mode = {
		  .isgraphic  = !UI.mode[i].text,
		  .number     = UI.mode[i].number,
		  .bpp        = UI.mode[i].bpp,
		  .width_div8 = (UI.mode[i].text)? UI.mode[i].width : UI.mode[i].width / 8,
		  .height = UI.mode[i].height,
		  };
	      VGA_set_saved_mode (index++, mode);
	      }
	  }
      }
  while (index < max)
      VGA_set_saved_mode (index++, ((struct vga_mode_str) {0}));
  }

VIDEO_FCT_PREFIX(restore_gujin_param_vga_mode) void
restore_gujin_param_vga_mode (void)
  {
  __attribute__ ((unused)) gujin_param_t *dummy;
  unsigned i;

  for (i = 0; i < nbof (dummy->vga_mode); i++) {
      struct vga_mode_str savedmode;
      struct videomode_str mode;

      VGA_get_saved_mode (i, &savedmode);
      if (savedmode.width_div8 == 0 || savedmode.height == 0) {
	  mode = (struct videomode_str) {};
	  add_valid_mode (&mode);
	  break;
	  }

      mode = (struct videomode_str) {
	  .number = savedmode.number,
	  .height = savedmode.height,
	  .text = !savedmode.isgraphic,
	  .vesa = 0,
	  .width = (savedmode.isgraphic)? savedmode.width_div8 * 8 : savedmode.width_div8,
	  .bpp = savedmode.bpp,
	  .attr = (savedmode.bpp == 0)
	  };
      add_valid_mode (&mode);
      }
  }
#endif /* UPDATE_BOOTPARAM */


#endif /* VGA_SUPPORT */

/*
 * The "VESA BIOS library" functions:
 */
#if USER_SUPPORT & VESA_SUPPORT
// not static only to not be removed by compiler:
awk_farret (VESA_getmode);
VIDEO_FCT_PREFIX(VESA_getmode) unsigned
VESA_getmode (void)
//  {bound_stack();{
  {{
  unsigned short mode;

  if (_VESA_getmode (&mode) != 0)
      VDBG (("Error in %s, mode read 0x%X. ", __FUNCTION__, mode));
    else if ((mode & ~0xC000U) == UI.parameter.identification) {
      if (UI.parameter.winsize == VESA2_MARKER)
	  mode |= 0x4000; /* false VESA2 */
	else
	  mode &= ~0x4000;
      }
  return mode & ~0x8000;
  }}

/* Cursor movment: home at (0,0) */
awk_farret (VESA_setcursor);
VIDEO_FCT_PREFIX(VESA_setcursor) static unsigned
VESA_setcursor (unsigned char row, unsigned char col)
  {
  if (COMMON_setcursor (row, col) != 0)
      return 1;

#if USER_SUPPORT & VESA_4BPP_TEXT
  if (   UI.parameter.memtype == mem_text
      && UI.parameter.EGA_compatible)
      _EGA_setcursor (row * UI.parameter.nbcol + col);
#endif

  return 0;
  }

#if USER_SUPPORT & VESA_WINDOW

VIDEO_FCT_PREFIX(VESA_pixeloffset) static unsigned FASTCALL VESA_pixeloffset (coord xy);

#if USER_SUPPORT & VESA_2WINDOWS
VIDEO_FCT_PREFIX(VESA_changewin) static unsigned FASTCALL
VESA_changewin (unsigned offset)	/* NEVER called if linear */
  {
#if USER_SUPPORT & VESA_HARDWINDOW
  if (set_hardwin (UI.parameter.hardwindow, UI.parameter.offsetarray,
		   offset) != 0)
#endif
      {
      if (_VESA_setwin (DIV_WINDOW_GRANUL(offset), UI.parameter.winNo) != 0) {
	  VDBG (("%s error at 0x%X wingranul = %u (writewin: %u)\r\n",
	  __FUNCTION__, DIV_WINDOW_GRANUL(offset),
	  UI.parameter.wingranul, UI.parameter.winNo));
	  }
      }

  return ROUND_WINDOW_GRANUL(offset);
  }

VIDEO_FCT_PREFIX(VESA_changereadwin) static unsigned FASTCALL
VESA_changereadwin (unsigned offset)
  {
#if USER_SUPPORT & VESA_HARDWINDOW
  if (set_hardwin (UI.parameter.readhardwindow, UI.parameter.readoffsetarray,
		   offset) != 0)
#endif
      {
      if (_VESA_setwin (DIV_WINDOW_GRANUL(offset), UI.parameter.winNo ^ 1) != 0) {
	  VDBG (("%s error at 0x%X wingranul = %u (writewin: %u ^ 1)\r\n",
		__FUNCTION__, DIV_WINDOW_GRANUL(offset),
		UI.parameter.wingranul, UI.parameter.winNo));
	  }
      }

  return ROUND_WINDOW_GRANUL(offset);
  }

VIDEO_FCT_PREFIX(VESA_readpixeloffset) static unsigned FASTCALL
VESA_readpixeloffset (coord xy)
  {
  unsigned offset;

  if (UI.parameter.NbWin <= 1) /* only one read/write window or VESA2 */
      return VESA_pixeloffset (xy);

  offset = xy.y * UI.parameter.linelength
	   + (xy.x * UI.parameter.bitperpixel) / 8;

  /* unsigned calculus, still working when overflow: */
  if (UNUSUAL(offset - UI.parameter.winreadadr >= UI.parameter.winsize))
      UI.parameter.winreadadr = VESA_changereadwin (offset);
  return offset - UI.parameter.winreadadr;
  }

#else /* VESA_2WINDOWS */

VIDEO_FCT_PREFIX(VESA_changewin) static unsigned FASTCALL
VESA_changewin (unsigned offset)
  {
#if USER_SUPPORT & VESA_HARDWINDOW
  if (set_hardwin (UI.parameter.hardwindow, UI.parameter.offsetarray, offset)
		!= 0)
#endif
      {
      unsigned char tmp = UI.parameter.winNo;

      if (_VESA_setwin (DIV_WINDOW_GRANUL(offset), tmp) != 0) {
	  VDBG (("%s error at 0x%X wingranul = %u (NbWin: %u, 1)\r\n",
	 __FUNCTION__, DIV_WINDOW_GRANUL(offset),
	 UI.parameter.wingranul, UI.parameter.NbWin));
	  }

      if (UI.parameter.NbWin == 2) {
	  if (_VESA_setwin (DIV_WINDOW_GRANUL(offset), (tmp ^ 1)) != 0) {
	      VDBG (("%s error at 0x%X wingranul = %u (NbWin: %u, 2)\r\n",
	     __FUNCTION__, DIV_WINDOW_GRANUL(offset),
	     UI.parameter.wingranul, UI.parameter.NbWin));
	      }
	  }
      }

  return ROUND_WINDOW_GRANUL(offset);
  }

#define VESA_readpixeloffset	VESA_pixeloffset

#endif /* VESA_2WINDOWS */

VIDEO_FCT_PREFIX(VESA_pixeloffset) static unsigned FASTCALL
VESA_pixeloffset (coord xy)
  {
  unsigned offset = xy.y * UI.parameter.linelength
		    + (xy.x * UI.parameter.bitperpixel) / 8;

  /* unsigned calculus, still working when overflow: */
  if (UNUSUAL(offset - UI.parameter.winadr >= UI.parameter.winsize))
      UI.parameter.winadr = VESA_changewin (offset);
  return offset - UI.parameter.winadr;
  }

#ifndef VERY_SLOW
VIDEO_FCT_PREFIX(VESA_max_size) static unsigned __attribute__ ((const))
VESA_max_size (unsigned offset_start, unsigned offset_end, unsigned winsize)
  {
  if (winsize <= 0x10000) { /* i.e. != VESA2_MARKER */
      if (winsize == 0x10000) { /* very usual for VESA 1 */
	  if (offset_end / 0x10000 != offset_start / 0x10000)
	      return 0x10000 - (offset_start % 0x10000);
	  }
	else {
	  unsigned short us_winsize = (unsigned short)winsize;

	  if (offset_end / us_winsize != offset_start / us_winsize)
	      return us_winsize - (offset_start % us_winsize);
	  }
      }
  return offset_end - offset_start;
  }
#endif /* VERY_SLOW */

#else /* VESA_WINDOW */

VIDEO_FCT_PREFIX(VESA_changewin) static unsigned inline
VESA_changewin (unsigned offset)
  {
  return UI.parameter.winadr;
  }

#define VESA_readpixeloffset	VESA_pixeloffset

VIDEO_FCT_PREFIX(VESA_pixeloffset) static unsigned inline
VESA_pixeloffset (coord xy)
  {
  return xy.y * UI.parameter.linelength
       + (xy.x * UI.parameter.bitperpixel) / 8
       - UI.parameter.winadr;
  }

#ifndef VERY_SLOW
VIDEO_FCT_PREFIX(VESA_max_size) static unsigned inline __attribute__ ((const))
VESA_max_size (unsigned offset_start, unsigned offset_end, unsigned winsize)
  {
  return offset_end - offset_start;
  }
#endif /* VERY_SLOW */

#endif /* VESA_WINDOW */

#if USER_SUPPORT & VESA_4BPP_EGA
VIDEO_FCT_PREFIX(EGA_setpixel) static void FASTCALL
EGA_setpixel (coord xy, unsigned color)
  {
  _EGA_write_bitmask (0x80 >> (xy.x % 8));
  /* read to load the latch for the 7 other pixels: */
  /* Default: write mode 2 */
  GXCHGB (xy, color);
  }

VIDEO_FCT_PREFIX(EGA_getpixel) static unsigned FASTCALL
EGA_getpixel (coord xy)
//  {bound_stack();{
  {{
  unsigned offset = VESA_readpixeloffset (xy);
  unsigned char mask = 0x80 >> (xy.x % 8), color = 0, plane;

  plane = UI.parameter.nbplane;
  while (plane--) {
      volatile unsigned char tmp;

      EGA_READ_PLANE (plane);
      tmp = gpeekb (offset);
      if (tmp & mask)
	  color |= 1 << plane;
      }

  return color;
  }}
#endif /* VESA_4BPP_EGA */

#if USER_SUPPORT & VESA_1BPP
VIDEO_FCT_PREFIX(VESA_setpixel_color1bit) static void FASTCALL
VESA_setpixel_color1bit (coord xy, unsigned color)
  {
  unsigned pixoffset = VESA_pixeloffset (xy);
  unsigned char bitToSet = 0x80 >> (xy.x % 8);

  if (color)
      gORb (pixoffset, bitToSet);
    else
      gANDb (pixoffset, ~bitToSet);
  }

VIDEO_FCT_PREFIX(VESA_getpixel_color1bit) static unsigned FASTCALL
VESA_getpixel_color1bit (coord xy)
  {
  unsigned char bitToGet = 0x80 >> (xy.x % 8);

  return !!(gpeekb (VESA_readpixeloffset (xy)) & bitToGet);
  }
#endif /* VESA_1BPP */

#if USER_SUPPORT & VESA_4BPP
VIDEO_FCT_PREFIX(VESA_setpixel_color4bit) static void FASTCALL
VESA_setpixel_color4bit (coord xy, unsigned color)
  {
  unsigned pixoffset = VESA_pixeloffset (xy);
  unsigned char twopixel = gpeekb (pixoffset);
  if (xy.x & 1)
      twopixel = (twopixel & 0x0F) | (color << 4);
    else
      twopixel = (twopixel & 0xF0) | color;
  gpokeb (pixoffset, twopixel);
  }

VIDEO_FCT_PREFIX(VESA_getpixel_color4bit) static unsigned FASTCALL
VESA_getpixel_color4bit (coord xy)
  {
  unsigned pixoffset = VESA_pixeloffset (xy);
  unsigned char twopixel = gpeekb (pixoffset);
  if (xy.x & 1)
      return twopixel >> 4;
    else
      return twopixel & 0x0F;
  }
#endif /* VESA_4BPP */

#if USER_SUPPORT & (VESA_8BPP | VESA_4BPP_TEXT)
VIDEO_FCT_PREFIX(VESA_setpixel_color8bit) static void FASTCALL
VESA_setpixel_color8bit (coord xy, unsigned color)
  {
  gpokeb (VESA_pixeloffset (xy), color);
  }

VIDEO_FCT_PREFIX(VESA_getpixel_color8bit) static unsigned FASTCALL
VESA_getpixel_color8bit (coord xy)
  {
  return gpeekb (VESA_readpixeloffset (xy));
  }
#endif /* VESA_8BPP | VESA_4BPP_TEXT */

#if USER_SUPPORT & VESA_16BPP
VIDEO_FCT_PREFIX(VESA_setpixel_color16bit) static void FASTCALL
VESA_setpixel_color16bit (coord xy, unsigned color)
  {
  gpokew (VESA_pixeloffset (xy), color);
  }

VIDEO_FCT_PREFIX(VESA_getpixel_color16bit) static unsigned FASTCALL
VESA_getpixel_color16bit (coord xy)
  {
  return gpeekw (VESA_readpixeloffset (xy));
  }
#endif /* VESA_16BPP */

#if USER_SUPPORT & VESA_24BPP
VIDEO_FCT_PREFIX(VESA_setpixel_color24bit) static void FASTCALL
VESA_setpixel_color24bit (coord xy, unsigned color)
//  {bound_stack();{
  {{
  unsigned pixoffset = VESA_pixeloffset (xy);

  pixoffset &= ~3;

  switch (xy.x & 3) {
    case 0:
      gpokew (pixoffset, color);
      gpokeb (pixoffset+2, color >> 16);
      break;
    case 1:
#if USER_SUPPORT & VESA_WINDOW
      if (UNUSUAL(pixoffset + 4 == UI.parameter.winsize)) {
	  gpokeb (pixoffset+3, color);
	  pixoffset += 4;
	  UI.parameter.winadr = VESA_changewin (pixoffset + UI.parameter.winadr);
	  gpokew (0, color >> 8);
	  }
	else
#endif /* VESA_WINDOW */
	  {
	  gpokeb (pixoffset+3, color);
	  gpokew (pixoffset+4, color >> 8);
	  }
      break;
    case 2:
#if USER_SUPPORT & VESA_WINDOW
      if (UNUSUAL(pixoffset + 4 == UI.parameter.winsize)) {
	  gpokew (pixoffset+2, color);
	  pixoffset += 4;
	  UI.parameter.winadr = VESA_changewin (pixoffset + UI.parameter.winadr);
	  gpokeb (0, color >> 16 );
	  }
	else
#endif /* VESA_WINDOW */
	  {
	  gpokew (pixoffset+2, color);
	  gpokeb (pixoffset+4, color >> 16);
	  }
      break;
    case 3:
      gpokeb (pixoffset+1, color);
      gpokew (pixoffset+2, color>>8);
      break;
    }
  }}

VIDEO_FCT_PREFIX(VESA_getpixel_color24bit) static unsigned FASTCALL
VESA_getpixel_color24bit (coord xy)
//  {bound_stack();{
  {{
  unsigned pixoffset = VESA_readpixeloffset (xy);

  pixoffset &= ~3;
  switch (xy.x & 3) {
    case 0:
      return gpeekl (pixoffset) & 0x00FFFFFFU;
    case 1: {
      unsigned returned;
#if USER_SUPPORT & VESA_WINDOW
      if (UNUSUAL(pixoffset + 4 == UI.parameter.winsize)) {
	  returned = gpeekb (pixoffset + 3);
	  pixoffset += 4;

#if USER_SUPPORT & VESA_2WINDOWS
	  if (UI.parameter.NbWin == 2)
	      UI.parameter.winreadadr = VESA_changereadwin (pixoffset + UI.parameter.winreadadr);
	    else
#endif
	      UI.parameter.winadr = VESA_changewin (pixoffset + UI.parameter.winadr);
	  returned |= gpeekw (0) << 8;
	  }
	else
#endif /* VESA_WINDOW */
	  {
	  returned = gpeekb (pixoffset + 3);
	  returned |= gpeekw (pixoffset + 4) << 8;
	  }
      return returned;
      }
    case 2: {
      unsigned returned;
#if USER_SUPPORT & VESA_WINDOW
      if (UNUSUAL(pixoffset + 4 == UI.parameter.winsize)) {
	  returned = gpeekw (pixoffset + 2);
	  pixoffset += 4;
#if USER_SUPPORT & VESA_2WINDOWS
	  if (UI.parameter.NbWin == 2)
	      UI.parameter.winreadadr = VESA_changereadwin (pixoffset + UI.parameter.winreadadr);
	    else
#endif
	      UI.parameter.winadr = VESA_changewin (pixoffset + UI.parameter.winadr);
	  returned |= gpeekb (0) << 16;
	  }
	else
#endif /* VESA_WINDOW */
	  {
	  returned = gpeekw (pixoffset + 2);
	  returned |= gpeekb (pixoffset + 4) << 16;
	  }
      return returned;
      }
    case 3:
      return gpeekl (pixoffset) >> 8;
    }
  return 0; /* will not happen often ! */
  }}
#endif /* VESA_24BPP */

#if USER_SUPPORT & VESA_32BPP
VIDEO_FCT_PREFIX(VESA_setpixel_color32bit) static void FASTCALL
VESA_setpixel_color32bit (coord xy, unsigned color)
  {
  gpokel (VESA_pixeloffset (xy), color);
  }

VIDEO_FCT_PREFIX(VESA_getpixel_color32bit) static unsigned FASTCALL
VESA_getpixel_color32bit (coord xy)
  {
  return gpeekl (VESA_readpixeloffset (xy));
  }
#endif /* VESA_32BPP */

awk_farret (VESA_setattribute);
VIDEO_FCT_PREFIX(VESA_setattribute) static void
VESA_setattribute (struct attribute_str attr)
  {
  if (~ATTR2BYTE(UI.attributes.valid) & ATTR2BYTE(attr))
      ATTR2BYTE(attr) &= ATTR2BYTE(UI.attributes.valid);
  UI.attributes.current = attr;
  }

VIDEO_FCT_PREFIX(GRAPHIC_drawchar) static inline void
GRAPHIC_drawchar (unsigned utfchar)
  {
  coord xy = {
      .x = UI.parameter.col * UI.parameter.charwidth,
      .y = UI.parameter.row * UI.parameter.charheight
      };
  unsigned short charheight = UI.parameter.charheight + 1;
  farptr font = UI.parameter.font;
  unsigned fgcolor, bgcolor;

  if (utfchar >= 0x100 && !copy_gujin_param.attrib.use_gujin_embedded_font) {
#if SETUP & UNICODE_FONT
      if (UI.parameter.charheight == 16 && utfchar < NBCHAR8x16
#if defined (GAPSTART) && defined (GAPLENGTH)
	&& (utfchar < GAPSTART || utfchar >= GAPSTART + GAPLENGTH)
#endif
	)
	  font = xdata_adr(font8x16); /* Use anyway Gujin font for those (in DOS) */
	else
#endif
	  utfchar = '?';
      }

  if (UI.parameter.charwidth >= 32)
      VDBG (("ERROR: charwidth too high in %s (%u).\r\n",
		__FUNCTION__, UI.parameter.charwidth));

  if (UI.attributes.current.reverse) {
      fgcolor = UI.bgcolor;
      bgcolor = UI.fgcolor;
      }
    else {
      fgcolor = UI.fgcolor;
      bgcolor = UI.bgcolor;
      }

#if defined (GAPSTART) && defined (GAPLENGTH)
  if (utfchar > GAPSTART)
      font += (utfchar - GAPLENGTH) * ((UI.parameter.charwidth + 7) / 8) * UI.parameter.charheight;
    else
#endif
  font += utfchar * ((UI.parameter.charwidth + 7) / 8) * UI.parameter.charheight;

  while	(--charheight != 0) {
      unsigned mask = 1 << (UI.parameter.charwidth - 1), draw = peekl (font);

      if (UI.attributes.current.underline && charheight == 2)
	  draw = 0xFFFFFFFFU;

      while (mask != 0) {
	  if (draw & mask)
	      UI_setpixel (xy, fgcolor);
	    else if (!UI.attributes.current.transparent)
	      UI_setpixel (xy, bgcolor);
	  mask >>= 1;
	  xy.x++;
	  }
      xy.x -= UI.parameter.charwidth;
      font += (UI.parameter.charwidth + 7) / 8;
      xy.y++;
      }
  }

VIDEO_FCT_PREFIX(VESA_drawchar) static void inline
VESA_drawchar (unsigned utfchar)
  {
#if USER_SUPPORT & VESA_4BPP_TEXT
  if (UI.parameter.memtype == mem_text) {
      coord xy = { .x = UI.parameter.col, .y = UI.parameter.row};
      unsigned short color;
//#define VESA_0BPP_TEXT doesn't work, how memory is organised?
#ifdef VESA_0BPP_TEXT
      if (UI.parameter.nbcolor == 0) {
	  struct attribute_str attr = UI.attributes.current;
	  if (attr.underline)
	      color = 0x01;
	    else if (attr.reverse)
	      color = 0x10;
	    else
	      color = 0x07;        /* see also init of stdcolor array */
	  if (attr.blink)
	      color |= 0x80;
	  if (attr.brighter)
	      color |= 0x08;
	  }
	else
#endif
	  color = (UI.bgcolor << 4) | UI.fgcolor;

	  if (utfchar >= UI.parameter.fontnbchar)
	      utfchar = '?';
#if SETUP & UNICODE_FONT
#if defined (GAPSTART) && defined (GAPLENGTH)
	  if (utfchar >= GAPSTART && utfchar < GAPSTART + GAPLENGTH)
	      utfchar = '?';
#endif
	    else if (copy_gujin_param.attrib.use_gujin_embedded_font && utfchar >= 0x80) {
	      if ((utfchar >> 7) != UI.parameter.currentUTFpage) {
		  UI.parameter.currentUTFpage = (utfchar >> 7);
		  farptr fontptr = UI.parameter.font;
		  fontptr += UI.parameter.currentUTFpage * 0x80 * UI.parameter.charheight;
#if defined (GAPSTART) && defined (GAPLENGTH)
		  if (utfchar >= GAPSTART)
		      fontptr -= GAPLENGTH * UI.parameter.charheight;
#endif
		  _VGAtxt_load_font (UI.parameter.charheight, 0, 128, 0x80, fontptr);
		  }
	      utfchar = 0x80 | (utfchar & 0x7F);
	      }
#endif
	    else if (utfchar >= 0x100)
	      utfchar = '?';

      UI_setpixel (xy, (color << 8) | (utfchar & 0xFF));
      }
    else
#endif /* VESA_4BPP_TEXT */
      GRAPHIC_drawchar (utfchar);
  }

#if USER_SUPPORT & VESA_4BPP_EGA
VIDEO_FCT_PREFIX(EGA_plotHline) static void
EGA_plotHline (coord xy, unsigned short xend, unsigned color)
  {bound_stack();{
  if (xend < xy.x) {
      VDBG (("%s called with xy.x = %u and xend = %u",
	__FUNCTION__, xy.x, xend));
      return;
      }
#ifdef VERY_SLOW
  while (xy.x < xend) {
      UI_setpixel (xy, color);
      xy.x++;
      }
#else /* VERY_SLOW */
  {
  const unsigned mask1 = 0x07, /* first align on 8 bits */
		 mask2 = 0x1F; /* then align on 32 bits */
  unsigned offset, offsetend;
//  CR0_t CR0 = ({ force_cache (EGA_plotHline, sizeof(EGA_plotHline)); disable_cache (); });

  if (color > UI.parameter.nbcolor)
      VDBG (("ERROR: %s called with color=%u\r\n", __FUNCTION__, color));

  /* Default: write mode 2 */
  if ((xy.x & ~mask1) == (xend & ~mask1)) {
      _EGA_write_bitmask (   (0xFF >> (xy.x & mask1))
			  & ~(0xFF >> (xend & mask1)  ));
      GXCHGB (xy, color);
      xy.x = xend;
//      enable_cache (CR0);
      return;
      }
    else {
      coord xyend;
      if ((xy.x & mask1) != 0 && xy.x < xend) {
	  _EGA_write_bitmask (0xFF >> (xy.x & mask1));
	  GXCHGB (xy, color);
	  xy.x = (xy.x + mask1) & ~mask1;
	  }
      _EGA_write_bitmask (0xFF);
      while ((xy.x & mask2) != 0 && xy.x < (xend & ~7)) {
	  gpokeb (VESA_pixeloffset (xy), color);
	  xy.x += 8;
	  }

	/*
	The first line was an uninterrupted file of zeros, left to right.
	The second line showed a single numeral one, exactly in the middle,
	with zeros to the borders, left and right.  After a few more lines,
	an unmistakable arc had formed, composed of ones.
	*/
      if ((xend & mask1) != 0 && xy.x < xend) {
	  _EGA_write_bitmask (~(0xFF >> (xend & mask1)));
	  xyend.y = xy.y;
	  xyend.x = xend;
	  GXCHGB (xyend, color);
	  xend = xend & ~mask1;
	  }
      _EGA_write_bitmask (0xFF);
      while ((xend & mask2) != 0 && xy.x < xend) {
	  xend -= 8;
	  xyend.y = xy.y;
	  xyend.x = xend;
	  gpokeb (VESA_pixeloffset (xyend), color);
	  }
      }

  offset = (UI.parameter.linelength * xy.y) + (xy.x >> 3);
  offsetend = (UI.parameter.linelength * xy.y) + (xend >> 3);

  color |= color << 8;
  color |= color << 16;

  while (offset < offsetend) {
      unsigned size = VESA_max_size (offset, offsetend, UI.parameter.winsize);

      if (   offset - UI.parameter.winadr >= UI.parameter.winsize
	  || offset - UI.parameter.winadr + size >= UI.parameter.winsize)
	  UI.parameter.winadr = VESA_changewin (offset); /* wingranul */
      offset = rep_gpokel (size / 4, offset, UI.parameter.winadr, color);
      }
//  enable_cache (CR0);
  }
#endif /* VERY_SLOW */
  }}
#endif /* VESA_4BPP_EGA */

#if USER_SUPPORT & VESA_24BPP
VIDEO_FCT_PREFIX(VESA_24BPP_plotHline) static void
VESA_24BPP_plotHline (coord xy, unsigned short xend, unsigned color)
  {bound_stack();{
  if (xend < xy.x) {
      VDBG (("%s called with xy.x = %u and xend = %u",
	__FUNCTION__, xy.x, xend));
      return;
      }
#ifdef VERY_SLOW
  while (xy.x < xend) {
      UI_setpixel (xy, color);
      xy.x++;
      }
#else /* VERY_SLOW */
  {
  unsigned offset, offsetend;
//  CR0_t CR0 = ({ force_cache (VESA_24BPP_plotHline, sizeof(VESA_24BPP_plotHline)); disable_cache (); });

  if (color > UI.parameter.nbcolor)
      VDBG (("ERROR: %s called with color=%u\r\n", __FUNCTION__, color));

  offset = (UI.parameter.linelength * xy.y) + (xy.x * 3);
  offsetend = (UI.parameter.linelength * xy.y) + (xend * 3);

  while (offset < offsetend) {
      unsigned size = VESA_max_size (offset, offsetend, UI.parameter.winsize);

      if (   offset - UI.parameter.winadr >= UI.parameter.winsize
	  || offset - UI.parameter.winadr + size >= UI.parameter.winsize)
	  UI.parameter.winadr = VESA_changewin (offset); /* wingranul */
      offset = rep_g3pokel (size, offset, UI.parameter.winadr, color);
      }
//  enable_cache (CR0);
  }
#endif /* VERY_SLOW */
  }}
#endif /* VESA_24BPP */

#if USER_SUPPORT & (VESA_1BPP | VESA_4BPP | VESA_8BPP | VESA_16BPP | VESA_32BPP | VESA_4BPP_TEXT)
VIDEO_FCT_PREFIX(VESA_plotHline) static void
VESA_plotHline (coord xy, unsigned short xend, unsigned color)
  {bound_stack();{
  if (xend < xy.x) {
      VDBG (("%s called with xy.x = %u and xend = %u",
	__FUNCTION__, xy.x, xend));
      return;
      }
#ifdef VERY_SLOW
  while (xy.x < xend) {
      UI_setpixel (xy, color);
      xy.x++;
      }
#else /* VERY_SLOW */
  {
  unsigned offset, offsetend;
  unsigned char mask, shift, cpt;
//  CR0_t CR0 = ({ force_cache (VESA_plotHline, sizeof(VESA_plotHline)); disable_cache (); });

  if (   color > UI.parameter.nbcolor
#if USER_SUPPORT & VESA_4BPP_TEXT
      && UI.parameter.memtype != mem_text
#endif
      )
      VDBG (("ERROR: %s called with color=%u\r\n", __FUNCTION__, color));

  /* FIXME: This is too generic and could even handle 1, 2, 4 BPP,
     but we do not need that here: those VESA modes are refused. */
  shift = __builtin_ffs (UI.parameter.bitperpixel) - 1; /* 8BPP => shift = 3 */
  mask = 0x1F >> shift;

  while ((xy.x & mask) != 0 && xy.x < xend) {
      UI_setpixel (xy, color);
      xy.x++;
      }
  while ((xend & mask) != 0 && xy.x < xend) {
      coord xyend = { .x = --xend, .y = xy.y};
      UI_setpixel (xyend, color);
      }

  offset = xy.x << 2;
  offsetend = xend << 2;
  for (cpt = UI.parameter.bitperpixel; cpt < 32; cpt <<= 1) {
      color |= color << cpt;
      offset >>= 1;
      offsetend >>= 1;
      }

  offset += UI.parameter.linelength * xy.y;
  offsetend += UI.parameter.linelength * xy.y;

  while (offset < offsetend) {
      unsigned size = VESA_max_size (offset, offsetend, UI.parameter.winsize);

      if (   offset - UI.parameter.winadr >= UI.parameter.winsize
	  || offset - UI.parameter.winadr + size >= UI.parameter.winsize)
	  UI.parameter.winadr = VESA_changewin (offset); /* wingranul */
      offset = rep_gpokel (size / 4, offset, UI.parameter.winadr, color);
      }
//  enable_cache (CR0);
  }
#endif /* VERY_SLOW */
  }}
#endif /* (VESA_1BPP | VESA_4BPP | VESA_8BPP | VESA_16BPP | VESA_32BPP | VESA_4BPP_TEXT) */

VIDEO_FCT_PREFIX(VESA_scroll) static void inline
VESA_scroll (signed int charheight) /* works now only with charheight > 0 */
  {
#ifdef VERY_SLOW
  coord xy = { .y = 0 }, xydest;
  do {
      xydest.x = xy.x = UI.parameter.width;
      xydest.y = xy.y + charheight;
      do {
	  xy.x--;
	  xydest.x--;
	  UI_setpixel (xy, UI_getpixel (xydest));
	  } while (xy.x != 0);
      } while (++xy.y < UI.parameter.height - charheight);
#else

  unsigned offsetsrc = UI.parameter.linelength * charheight,
	   offsetdst = 0,
	   offsetend = UI.parameter.linelength * UI.parameter.height;

//  CR0_t CR0 = ({ force_cache (VESA_scroll, sizeof(VESA_scroll)); disable_cache (); });

#if USER_SUPPORT & VESA_4BPP_EGA
  unsigned char plane;
  unsigned save_src, save_dst, save_end;
  struct EGA_mode_str EGA_mode = {
      .write_source		= cpuORsetreset,
      .enable_tristate		= 0,
      .read_mode		= EGA_DEFAULT_READ_MODE,
      .enable_odd_even_addr	= 0,
      .CGA_shift_register	= 0,
      .enable_256_color		= 0,
      .reserved			= 0
      };

//  bound_stack();	/* failure treated here in write mode 2 ! */

  if (UI.parameter.memtype == mem_EGA && UI.parameter.nbplane > 1) {
      _EGA_read_write_mode (EGA_mode);      /* write mode 0 */
      _EGA_write_bitmask (0xFF);
      }

  if (   UI.parameter.nbplane != 1
      && UI.parameter.memtype != mem_EGA
      && UI.parameter.memtype != mem_text)
      VDBG (("ERROR %s: nbplane = %u, memtype = %u! ",
	     __FUNCTION__, UI.parameter.nbplane,
	     (unsigned)UI.parameter.memtype));

  /*
   * Limit the copy size of each plane because of the visual effect:
   */
  save_end = offsetend;
  while (offsetsrc < save_end) {
      if (UI.parameter.memtype == mem_EGA && UI.parameter.nbplane > 1)
	  offsetend = MIN (save_end, offsetsrc + 8*1024);

      plane = UI.parameter.nbplane;
      save_src = offsetsrc;
      save_dst = offsetdst;
      while (plane--) {
	  if (UI.parameter.memtype == mem_EGA && UI.parameter.nbplane > 1) {
	      EGA_READ_PLANE (plane);
	      _EGA_enable_plane (1 << plane);
	      offsetsrc = save_src;
	      offsetdst = save_dst;
	      }
#else
//  bound_stack();
#endif

  while (offsetsrc < offsetend) {
      unsigned size, hastochangewin;
      unsigned winadr, winreadadr;

      size = VESA_max_size (offsetsrc, offsetend, UI.parameter.winsize);
      size = VESA_max_size (offsetdst, offsetdst+size, UI.parameter.winsize);

      if (size % 4 != 0 || size == 0 || size > UI.parameter.winsize)
	  VDBG (("ERROR %s: size = %u! ", __FUNCTION__, size));

#if USER_SUPPORT & VESA_2WINDOWS
      if (UI.parameter.NbWin == 2) {
	  if (   offsetsrc - UI.parameter.winreadadr >= UI.parameter.winsize
	      || offsetsrc - UI.parameter.winreadadr + size >= UI.parameter.winsize)
	      UI.parameter.winreadadr = VESA_changereadwin (offsetsrc);
	  winreadadr = UI.parameter.winreadadr;

	  if (   offsetdst - UI.parameter.winadr >= UI.parameter.winsize
	      || offsetdst - UI.parameter.winadr + size >= UI.parameter.winsize)
	      UI.parameter.winadr = VESA_changewin (offsetdst); /* wingranul */
	  winadr = UI.parameter.winadr;

	  hastochangewin = 0;
	  }
	else
#endif
	  {
	  if (   offsetsrc - UI.parameter.winadr >= UI.parameter.winsize
	      || offsetsrc - UI.parameter.winadr + size >= UI.parameter.winsize)
	      UI.parameter.winadr = VESA_changewin (offsetsrc);	/* wingranul */

	  winadr = winreadadr = UI.parameter.winadr;
	  hastochangewin = (   offsetdst - UI.parameter.winadr >= UI.parameter.winsize
			    || offsetdst - UI.parameter.winadr + size >= UI.parameter.winsize);
	  }

#if USER_SUPPORT & VESA_WINDOW
      if (hastochangewin && UI.parameter.NbWin != 0) {
	  /* We have to go through a buffer */
	  unsigned treated = MIN (size, 1024) / 4;
	  unsigned buffer[256]; //treated]; /* get an align of 4 */
//	  unsigned *buffer = (unsigned *)fourKbuffer;
//	  if (sizeof (fourKbuffer) < 256 * 4)
//	      __ERROR();	/* the linker will produce an error here */

	  offsetsrc = winadr + rep_movel_fromg (buffer, offsetsrc - winadr, treated);

//	  VDBG (("#0x%X ", treated*4));

	  UI.parameter.winadr = VESA_changewin (offsetdst); /* wingranul, hastochangewin is true */

	  winadr = UI.parameter.winadr;
	  offsetdst = winadr + rep_movel_tog (offsetdst - winadr, buffer, treated);
	  }
	else
#endif
	  {
	  /* We can do here a direct copy, i.e. without buffer */
//	  VDBG (("=0x%X ", size));

	  rep_movel_gtog (offsetdst - winadr, offsetsrc - winreadadr, size/4);
	  offsetsrc += size;
	  offsetdst += size;
	  }
      }

#if USER_SUPPORT & VESA_4BPP_EGA
	  } /* while (plane--) */
      }  /* while (offsetsrc < save_end) */

  if (UI.parameter.memtype == mem_EGA && UI.parameter.nbplane > 1) {
      _EGA_enable_plane (0xFF >> (8-UI.parameter.nbplane));
      EGA_mode.write_source = cpuASsetreset; /* Default: write mode 2 */
      _EGA_read_write_mode (EGA_mode);
      }
#endif
//  enable_cache (CR0);
#endif /* VERY_SLOW */
  }

awk_farret (VESA_clearscreen);
VIDEO_FCT_PREFIX(VESA_clearscreen) static void
VESA_clearscreen (void)
  {bound_stack();{
  unsigned fillcolor;
  coord xy = { .x = 0, .y = 0} ;

  if (UI.parameter.memtype == mem_text)
      fillcolor = (UI.bgcolor << 12) | (UI.fgcolor << 8) | ' ';
    else
      fillcolor = UI.bgcolor;

  while (xy.y < UI.parameter.height) {
      UI_plotHline (xy, UI.parameter.width, fillcolor);
      xy.y++;
      }

  VESA_setcursor (0, 0);
  }}

awk_farret (VESA_putstr);
VIDEO_FCT_PREFIX(VESA_putstr) static void
VESA_putstr (const char *str)
  {
  while (*str) {
      UI.parameter.attr.has_outputed = 1;
      unsigned nblines = 1; /* scroll only once even when multiple lines are displayed at once */

      switch (*str++) {
	case '\007': /* ^G */
	  BOOT1_putstr ("\007");
	  break;
	case '\010': /* ^H */
	  if (UI.parameter.col > 0)
	      UI.parameter.col -= 1;
//	  VESA_drawchar (' ');	// for attribute "transparent"
	  break;
	default:
	  str--; /* restore str pointer if not ^G, ^H, \n, \r */
	  unsigned utfchar, charlen = UTF8LEN(str);	/* handle UTF-8 subset */

	  if (charlen == 2 && (utfchar = UTF8VAL2(str)) < UI.parameter.fontnbchar) {
	      str += charlen;
	      }
	    else if (charlen >= 2) {
	      utfchar = '?';
	      str += charlen;
	      }
	    else
	      utfchar = *str++;
	  /* Also translate checkbox as square root, but do not try to translate '?': */
	  if (!UI.parameter.attr.ansi_font && utfchar >= 0x7F && utfchar < 256) {
	      const char *ptr = ansi2pc;

	      while (*ptr && (unsigned char)*ptr != utfchar)
		  ptr++;
	      if (*ptr != 0)
		  utfchar = ptr - ansi2pc;
		else
		  utfchar = '?';
	      }
	  VESA_drawchar (utfchar);
	  if (++UI.parameter.col >= UI.parameter.nbcol) {
	      UI.parameter.col = 0;	/* automatic CRLF */
	case '\n': /* it is valid in C */
	      if (++UI.parameter.row >= UI.parameter.nbrow) {
		  /* estimate how many lines to scroll on this string */
		  const char *tmp = str;
		  unsigned height;

		  while (*tmp != 0)
		      if (*tmp++ == '\n')
			  nblines ++;
		  if (UI.parameter.memtype == mem_text)
		      height = nblines;
		    else
		      height = nblines * UI.parameter.charheight;
		  VESA_scroll (height);
		  UI.parameter.attr.has_scrolled = 1;
		  UI.parameter.row = UI.parameter.nbrow - nblines;
//		  VDBG (("has scroll %u lines, i.e. height %u, new cursor %u,%u max %u,%u\r\n",
//			nblines, height, UI.parameter.row, UI.parameter.col, UI.parameter.nbrow, UI.parameter.nbcol));

	case '\r': /* it is valid in C */ {
		  coord xy = { .x = UI.parameter.col, .y = UI.parameter.row};
		  unsigned short end_y = UI.parameter.row + nblines, width;
		  unsigned fillcolor;

		  if (UI.parameter.memtype == mem_text) {
		      fillcolor = (UI.bgcolor << 12) | (UI.fgcolor << 8) | ' ';
		      width = UI.parameter.nbcol;
		      }
		    else {
		      xy.x *= UI.parameter.charwidth;
		      xy.y *= UI.parameter.charheight;
		      end_y *= UI.parameter.charheight;
		      fillcolor = UI.bgcolor;
		      width = UI.parameter.width;
		      }
		  while (xy.y < end_y) {
		      UI_plotHline (xy, width, fillcolor);
		      xy.y++;
		      }
		  }
		  UI.parameter.col = 0;
		  }
	      }
	  break;
	}
      }
#if USER_SUPPORT & VESA_4BPP_TEXT
  VESA_setcursor (UI.parameter.row, UI.parameter.col); // display cursor
#endif
  }

#if USER_SUPPORT & VESA_WINDOW
VIDEO_FCT_PREFIX(VESA_setup_window) static unsigned
VESA_setup_window (struct video_parameter_str *param, VESA_modeinfo_t *modeinfo)
  {bound_stack();{
  if (modeinfo->WinGranularity == 0 && modeinfo->MemoryModel != mem_text) {
      VDBG ((" [modeinfo.WinGranularity == 0, force linear graphic mode] \r\n"));
      return 0x300;
      }

#if USER_SUPPORT & VESA_2WINDOWS
  param->readdelta = 0; /* HAS TO BE null if NbWin < 2 */
#endif

  if (   !modeinfo->WinAAttributes.win_writable
      && !modeinfo->WinAAttributes.win_readable)
      modeinfo->WinAAttributes.win_exist = 0;

  if (   !modeinfo->WinBAttributes.win_writable
      && !modeinfo->WinBAttributes.win_readable)
      modeinfo->WinBAttributes.win_exist = 0;

  if (!modeinfo->WinBAttributes.win_exist) {
      if (!modeinfo->WinAAttributes.win_exist) {
	  VDBG ((" [no window exists] "));
	  return 0x301;
	  }
      if (!modeinfo->WinAAttributes.win_writable) {
	  VDBG ((" [only window A not writeable] "));
	  return 0x302;
	  }
      if (!modeinfo->WinAAttributes.win_readable) {
	  VDBG ((" [only window A not readable] "));
	  return 0x303;
	  }
      param->base_address = modeinfo->WinASegment << 4;
      param->winNo = 0;
      VDBG ((" [using winA] "));
      param->NbWin = 1;
      }
    else if (!modeinfo->WinAAttributes.win_exist) {
      if (!modeinfo->WinBAttributes.win_writable) {
	  VDBG ((" [only window B not writeable] "));
	  return 0x304;
	  }
      if (!modeinfo->WinBAttributes.win_readable) {
	  VDBG ((" [only window B not readable] "));
	  return 0x305;
	  }
      param->base_address = modeinfo->WinBSegment << 4;
      param->winNo = 1;
      VDBG ((" [using winB] "));
      param->NbWin = 1;
      }
    else {
#if USER_SUPPORT & VESA_2WINDOWS
      if (   modeinfo->WinAAttributes.win_writable
	  && modeinfo->WinBAttributes.win_readable) {
	  VDBG ((" [using winA write / winB read] "));
	  param->winNo = 0; /* winA as write only */
	  param->base_address = modeinfo->WinASegment << 4;
	  param->readdelta = ((unsigned)modeinfo->WinBSegment
				- modeinfo->WinASegment) << 4;
	  }
	else if (   modeinfo->WinBAttributes.win_writable
	 && modeinfo->WinAAttributes.win_readable) {
	  VDBG ((" [using winA read / winB write] "));
	  param->winNo = 1; /* winB as write only */
	  param->base_address = modeinfo->WinBSegment << 4;
	  param->readdelta = ((unsigned)modeinfo->WinASegment
				- modeinfo->WinBSegment) << 4;
	  }
	else {
	  VDBG ((" [either winA & winB read only or winA & winB write only] "));
	  return 0x306;
	  }
#else
      if (modeinfo->WinASegment != modeinfo->WinBSegment) {
	  if (   modeinfo->WinAAttributes.win_writable
	      && modeinfo->WinAAttributes.win_readable) {
	      VDBG ((" [using small winA read/write] "));
	      param->base_address = modeinfo->WinASegment << 4;
	      param->winNo = 0;
	      }
	    else if (   modeinfo->WinBAttributes.win_writable
	     && modeinfo->WinBAttributes.win_readable) {
	      VDBG ((" [using small winB read/write] "));
	      param->base_address = modeinfo->WinBSegment << 4;
	      param->winNo = 1;
	      }
	    else {
	      VDBG (("unsupported winA seg != winB seg"));
	      return 0x307;
	      }
	  }
	else {
	  param->base_address = modeinfo->WinASegment << 4;
	  param->winNo = 0;
	  VDBG (("using winA AND winB "));
	  }
#endif
      param->NbWin = 2;
      }
  param->winsize = modeinfo->WinSize * 1024;
  return 0;
  }}
#else  /* VESA_WINDOW */
VIDEO_FCT_PREFIX(VESA_setup_window) static unsigned inline
VESA_setup_window (struct video_parameter_str *param, VESA_modeinfo_t *modeinfo)
  {
  return -1;
  }
#endif /* VESA_WINDOW */

#if USER_SUPPORT & VESA_LINEAR
VIDEO_FCT_PREFIX(VESA_setup_linear) static unsigned
VESA_setup_linear (struct video_parameter_str *param,
		   VESA_modeinfo_t *modeinfo)
  {bound_stack();{

/* FIXME: "S3 Incorporated. 86C775/86C785" report:
   VESA2 call successfull, version: 2.0, capabilities: 0,
	videoram: 32 x 64 Kbytes, name:[S3 Incorporated. 86C775/86C785], modes:
	... mode 0x104: graphic 1024x768, 4 bpp ...
	VESA_getsize mode 0x4104: graphic 1024x768, 4 bpp, local char: 8x16,
	winA = 7, winB = 0, wingranul = 64 Kb, size = 64 Kb winA start = 0xA000,
	winB start = 0xA000 using linear framebuffer at 0xF0000000,
	MemoryModel 0x3, BytesPerScanLine = 128,
	BitsPerPixel 4 / NumberOfPlanes 4, NumberOfImagePages 4,
	red mask/off: 0/0, green mask/off: 0/0, blue mask/off: 0/0
	get VGA font farptr 0xC0006AE0, nbrow = 63.
	[VESA_setmode: EGA_s/getpixel]  [setting palette 16 colors] ,
	success VESA_setmode
	But does not work (linear mode MemoryModel 0x3).
	Works nicely in mode 0x4106: graphic 1280x1024, 4 bpp because
	refuses linear then.
  AllWin: outb (0x3C4, 0x4); outw (0x3D4, 0x4838); outw (0x3D4, 0xA539);
	  outw (0x3D4, 0x6A ^ (page << 2));
  Shall we disable linear mode in MemoryModel 0x3 ?
*/
  if ((unsigned)modeinfo->BytesPerScanLine * modeinfo->YResolution
	 > modeinfo->WinSize * 1024U) {
      if (!modeinfo->ModeAttributes.linear_framebuffer_supported) {
	  VDBG (("no linear buffer supported.\r\n"));
	  return 0x201;
	  }
      if (getsw() & 1) {
	  VDBG (("Cannot use linear framebuffer, "
		  "processor not in real mode.\r\n"));
	  return 0x202; /* Under DOS, maybe do something else... */
	  }
      if (modeinfo->PhysBasePtr <= 0x100000) {
	  VDBG (("graphic linear framebuffer in the first megabyte! (0x%X)\r\n",
		 modeinfo->PhysBasePtr));
	  return 0x203;
	  }
      if (modeinfo->MemoryModel == 3) {
	  VDBG (("Do not know how to handle 4 BPP MemoryModel 3 with linear video memory\r\n"));
	  return 0x204;
	  }
      param->base_address = 0xA8000; /* != A0000 to know linear addressing */
      }
    else {
      /*
       * Support text modes if only VESA_LINEAR is defined, normally
       * they are not maped with (modeinfo->PhysBasePtr == 0xB800), also
       * 800x600x16 can be supported because it fits in one 64 Kb window,
       * normally modeinfo->ModeAttributes.linear_framebuffer_supported
       * is false. Some 320x240*256 mode can be there too.
       * On "3dfx Interactive, Inc." video Bioses, Win[AB]Attributes.win_exist
       * is not set if the window is at 0xA000 or 0xB000/0xB800.
       */
      VDBG ((" [VESA2 mapping linear framebuffer to window (<64K mode)] "));
      param->base_address = modeinfo->PhysBasePtr =
		((unsigned)(modeinfo->WinASegment
				 ? modeinfo->WinASegment
				 : modeinfo->WinBSegment)) << 4;
      }
#if USER_SUPPORT & VESA_WINDOW
  param->NbWin = 0;
  param->winNo = 0;
  param->wingranul = 0;

#if USER_SUPPORT & VESA_2WINDOWS
  param->readdelta = 0; /* HAS TO BE null if NbWin < 2 */
#endif

#endif
  param->winsize = VESA2_MARKER;
  param->winadr = param->base_address - modeinfo->PhysBasePtr;

  VDBG (("using linear framebuffer at 0x%X, ", modeinfo->PhysBasePtr));
  return 0;
  }}
#else  /* VESA_LINEAR */
VIDEO_FCT_PREFIX(VESA_setup_linear) static unsigned
VESA_setup_linear (struct video_parameter_str *param, VESA_modeinfo_t *modeinfo)
  {
  return -1;
  }
#endif /* VESA_LINEAR */

awk_farret (VESA_getsize);
VIDEO_FCT_PREFIX(VESA_getsize) static unsigned
VESA_getsize (unsigned mode, struct video_parameter_str *param)
  {bound_stack();{
  VESA_modeinfo_t modeinfo;
  unsigned saved_mode = mode & ~0x8000, tmp;

  VDBG (("%s mode 0x%X: ", __FUNCTION__, mode));

  checkptr (param, "VESA_getsize invalid parameter.\r\n");

  memset (param, 0, sizeof(*param));
  /* param->attr.ansi_font = 0; */
  mode &= ~0x8000;
  mode &= ~0x4000;
  param->identification = mode;

  /* Alliance Semiconductor PMAT3D 4 Mbytes VESA 1.2 probing mode <= 0x7F,
     division by 0 */
#ifdef TREAT_EXCEPTION
  if (shit_handler (1, 2) != 0) {
      VDBG (("have hit an exception, like division by 0"
	" or invalid instruction, or timeout (2 seconds).\r\n"));
      return 0x700;
      }
#endif
  /* "CHIPS 65520/525/530 Flat Panel VGA" on 486 portable T2130CS always
  answer "no error" and the data of the current mode if mode <= 0x7F !!! */
  tmp = _VESA_getmodeinfo (&modeinfo, mode);
#if 0 /* I have never seen this rescue working */
  if (saved_mode & 0x4000) {
      if (tmp != 0)
	  VDBG ((" [_VESA_getmodeinfo failed (0x%X) for mode 0x%X, trying 0x%X] ",
			tmp, mode, mode | 0x4000));
	else if (modeinfo.ModeAttributes.mode_supported == 0)
	  VDBG ((" [_VESA_getmodeinfo unsupported (0x%X) for mode 0x%X, trying 0x%X] ",
			*(unsigned short*) &modeinfo.ModeAttributes,
			mode, mode | 0x4000));
      if (tmp != 0 || modeinfo.ModeAttributes.mode_supported == 0) {
	  mode |= 0x4000;
	  param->identification |= 0x4000;
	  tmp = _VESA_getmodeinfo (&modeinfo, mode);
	  }
      }
#endif
#ifdef TREAT_EXCEPTION
  shit_handler (0, 2);
#endif

  if (tmp != 0) {
      VDBG (("_VESA_getmodeinfo error.\r\n"));
      return 0x101;
      }

  if (   modeinfo.ModeAttributes.optional_info_available == 0
      || modeinfo.ModeAttributes.mode_supported == 0) {
      VDBG (("_VESA_getmodeinfo: no optional info / unsupported (0x%X)\r\n",
	    *CAST(unsigned short *, &modeinfo.ModeAttributes)));
      return 0x102;
      }

  VDBG (("%s %ux%u, %u bpp, local char: %ux%u, ",
	 (modeinfo.ModeAttributes.graphic_mode)? "graphic" : "text",
	 modeinfo.XResolution, modeinfo.YResolution,
	 modeinfo.BitsPerPixel,
	 modeinfo.XCharSize, modeinfo.YCharSize));

  if (modeinfo.ModeAttributes.graphic_mode) {
      /* Refuse now a video mode if there is clearly not enought
	 memory to map it - it should have the unsupported bit,
	 but... */
      if ((unsigned)modeinfo.XResolution * modeinfo.YResolution
		* modeinfo.BitsPerPixel / (8 * 64 * 1024)
		 > UI.info.vesa.memory_64Kb) {
	  VDBG (("_VESA_getmodeinfo: no enough memory on board!\r\n"));
	  return 0x103;
	  }
      }

  VDBG (("winA = %u, winB = %u, wingranul = %u Kb, size = %u Kb "
	"winA start = 0x%X, winB start = 0x%X ",
	 *(unsigned char *)&modeinfo.WinAAttributes,
	 *(unsigned char *)&modeinfo.WinBAttributes,
	 modeinfo.WinGranularity, modeinfo.WinSize,
	 modeinfo.WinASegment, modeinfo.WinBSegment));

#if USER_SUPPORT & VESA_4BPP_TEXT
#if 0 /* Those are probably been volumtary invalidated, no refresh */
  if (   modeinfo.MemoryModel == mem_text
      && !modeinfo.WinAAttributes.win_exist
      && modeinfo.WinAAttributes.win_readable
      && modeinfo.WinAAttributes.win_writable
      && modeinfo.WinASegment == 0xB800
      && modeinfo.WinSize == 0) {
      VDBG ((" [correcting WinSize on window A for"
		" ATI radeon 7500, \"ATI RV200\"] "));
      modeinfo.WinSize = 32 * 1024;
      }
  /* Also some versions of "ATI MACH64" have modes 0x109. 0x10A, 0x230
     declared as graphic, wrong linelength, no windows...
     keep them disabled. */
#endif

  if (modeinfo.MemoryModel == mem_text && modeinfo.NumberOfPlanes == 4) {
      if (modeinfo.BytesPerScanLine != 2 * modeinfo.XResolution) {
	  VDBG ((" [correcting linelength: old=%u, new=%u] ", modeinfo.BytesPerScanLine, 2 * modeinfo.XResolution));
	  modeinfo.BytesPerScanLine = 2 * modeinfo.XResolution;
	  }
      if (modeinfo.WinASegment != 0xB800) {
	  VDBG ((" [correcting WinASegment: old=0x%X, new=0xB800] ", modeinfo.WinASegment));
	  modeinfo.WinASegment = 0xB800;
	  }
      if (modeinfo.BitsPerPixel != 4) {
	  VDBG ((" [correcting BitsPerPixel: old=%u, new=4] ", modeinfo.BitsPerPixel));
	  modeinfo.BitsPerPixel = 4;
	  }
      }
/* VESA2 call successfull, version: 3.0, capabilities: 1, videoram: 1024 x 64 Kbytes, name:[VIA K8M800], modes:
 * 0x108 VESA_getsize mode 0x108: text 80x60, 4 bpp, local char: 8x8, winA = 7, winB = 0, wingranul = 64 Kb,
 * size = 64 Kb winA start = 0xA000, winB start = 0xA000 MemoryModel 0x0, BytesPerScanLine = 160,
 * BitsPerPixel 4 / NumberOfPlanes 1, NumberOfImagePages 254,  [VESA2 mapping linear framebuffer to window (<64K mode)]
 * using linear framebuffer at 0xA0000, red mask/off: 0/0, green mask/off: 0/0, blue mask/off: 0/0 using hardware font.
 * OK,
 * but does not display anything -> correct winA start = 0xB800 and winB start = 0xB800.
 */
  if (modeinfo.MemoryModel == mem_text && modeinfo.BytesPerScanLine == 160 && !modeinfo.WinBAttributes.win_exist
	&& modeinfo.WinASegment == modeinfo.WinBSegment	&& modeinfo.WinASegment != 0xB800) {
      VDBG ((" [correcting WinASegment: old=0x%X, new=0xB800] ", modeinfo.WinASegment));
      modeinfo.WinASegment = 0xB800;
      }
#endif /* VESA_4BPP_TEXT */

  /* for offsets calculus, our bitperpixel is not necessarily linked to
     the number of colors (in case of different color planes): */
  VDBG (("MemoryModel 0x%X, BytesPerScanLine = %u, BitsPerPixel %u / NumberOfPlanes %u, NumberOfImagePages %u, ",
	  modeinfo.MemoryModel, modeinfo.BytesPerScanLine, modeinfo.BitsPerPixel, modeinfo.NumberOfPlanes, modeinfo.NumberOfImagePages));

  if (mode == saved_mode && modeinfo.MemoryModel != mem_text) {
      unsigned tmp = VESA_setup_window (param, &modeinfo);
      if (tmp != 0) {
	  VDBG ((" [window failed, try linear] "));
	  tmp = VESA_setup_linear (param, &modeinfo);
	  }
      if (tmp != 0) {
	  VDBG ((" cannot acces video memory.\r\n"));
	  return tmp;
	  }
      }
    else {
      /* always treat text modes as linear modes (at 0xB800)
	 to not try to switch windows,
	 and maybe WinGranularity is zero in text modes: */
      unsigned tmp = VESA_setup_linear (param, &modeinfo);
      if (tmp != 0) {
	  VDBG ((" [linear failed, try window] "));
	  tmp = VESA_setup_window (param, &modeinfo);
	  }
      if (tmp != 0) {
	  VDBG ((" cannot acces video memory.\r\n"));
	  return tmp;
	  }
      }

#if USER_SUPPORT & VESA_WINDOW
  if (param->winsize != VESA2_MARKER) {
#if USER_SUPPORT & VESA_WINFUNCTION
      VDBG (("addr positionning fct = 0x%X ", modeinfo.WinFuncPtr));
      param->addrpositionfarfct = modeinfo.WinFuncPtr;
#endif

      INIT_WINDOW_GRANUL (modeinfo.WinGranularity * 1024, param);
      if (DISPLAY_WINDOW_GRANUL(param) != modeinfo.WinGranularity * 1024U) {
	  VDBG ((" [Strange VESA window granularity: %u bytes, "
		 "recompile with VESA_STRANGE_WINDOW defined!]\r\n",
		    modeinfo.WinGranularity * 1024));
	  return 0x104;
	  }

      }
#endif

  param->memtype = modeinfo.MemoryModel;
  param->linelength = modeinfo.BytesPerScanLine;
  /* Note: some cards ([Alliance Semiconductor PMAT3D]) says modeinfo.NumberOfImagePages = 0 */

  if ( 0
#if USER_SUPPORT & VESA_4BPP_TEXT
      || (modeinfo.MemoryModel == mem_text && (modeinfo.BitsPerPixel == 4
#ifdef VESA_0BPP_TEXT
						|| modeinfo.BitsPerPixel == 0
#endif
						))
#endif
#if USER_SUPPORT & VESA_4BPP_EGA
      || (modeinfo.NumberOfPlanes == 4 && modeinfo.BitsPerPixel == 4 )
#endif
#if USER_SUPPORT & VESA_1BPP
	/* This can handle 1 bpp & 1 plane or 1 bpp & 4 planes as long
	   as the active plane is plane 0 and MemoryModel == mem_EGA
	   (equivalent of EGA 16 colors with only first plane active).
		Usually monochrome modes 640x350 and 640x480.
	   It will not work for one mode I have: 600x200
		1 bpp & 1 plane & MemoryModel 0x1 == mem_CGA */
      || (modeinfo.BitsPerPixel == 1 && modeinfo.MemoryModel != mem_CGA)
#endif
//#if USER_SUPPORT & VESA_2BPP
//    || (modeinfo.BitsPerPixel == 2 && modeinfo.MemoryModel == mem_CGA)
//#endif
#if USER_SUPPORT & VESA_4BPP
      || (modeinfo.NumberOfPlanes <= 1 && modeinfo.BitsPerPixel == 4 )
#endif
#if USER_SUPPORT & VESA_8BPP
      || (modeinfo.NumberOfPlanes <= 1 && modeinfo.BitsPerPixel == 8 )
#endif
#if USER_SUPPORT & VESA_16BPP
      || (modeinfo.NumberOfPlanes <= 1 && modeinfo.BitsPerPixel == 16)
      || (modeinfo.NumberOfPlanes <= 1 && modeinfo.BitsPerPixel == 15)
#endif
#if USER_SUPPORT & VESA_24BPP
      || (modeinfo.NumberOfPlanes <= 1 && modeinfo.BitsPerPixel == 24)
#endif
#if USER_SUPPORT & VESA_32BPP
      || (modeinfo.NumberOfPlanes <= 1 && modeinfo.BitsPerPixel == 32)
#endif
     ) {
      param->nbplane = 1;
      if (param->memtype == mem_text)
	  param->bitperpixel = 16;    /* scroll, plotHline (16 instead of 4) */
	else if (modeinfo.BitsPerPixel == 15)
	  param->bitperpixel = 16;
	else if (modeinfo.BitsPerPixel == 4 && modeinfo.NumberOfPlanes > 1) {
	  param->bitperpixel = 1;
	  param->nbplane = 4;
	  }
	else
	  param->bitperpixel = modeinfo.BitsPerPixel;
      }
    else {
      VDBG (("%u bpp & %u planes memory model %u not supported or not compiled-in!\r\n",
		modeinfo.BitsPerPixel, modeinfo.NumberOfPlanes, modeinfo.MemoryModel));
      return 0x105;
      }

  VDBG (("red mask/off: %u/%u, green mask/off: %u/%u, blue mask/off: %u/%u ",
	modeinfo.layout.Red.MaskSize,   modeinfo.layout.Red.FieldPosition,
	modeinfo.layout.Green.MaskSize, modeinfo.layout.Green.FieldPosition,
	modeinfo.layout.Blue.MaskSize,  modeinfo.layout.Blue.FieldPosition));

  param->layout = modeinfo.layout;

  if (   modeinfo.layout.Red.MaskSize != 0
      && modeinfo.layout.Green.MaskSize != 0
      && modeinfo.layout.Blue.MaskSize != 0) {
      param->nbcolor = 1 << (modeinfo.layout.Red.MaskSize
				+ modeinfo.layout.Green.MaskSize
				+ modeinfo.layout.Blue.MaskSize);
      }
#ifdef VESA_0BPP_TEXT
    else if (modeinfo.BitsPerPixel == 0)
      param->nbcolor = 0; /* text with attributes */
#endif
    else
      param->nbcolor = 1 << modeinfo.BitsPerPixel;

  param->attr.isgraphic = modeinfo.ModeAttributes.graphic_mode;
  /*
   * Is it graphic and can it draw bgcolor? Lets say yes
   * if it is graphic and has more than 16 colors
   */
  param->attr.graphicbgcolor = param->attr.isgraphic && param->nbcolor > 2;
  param->width = modeinfo.XResolution;	/* set it also for text/scroll */
  param->height = modeinfo.YResolution;	/* set it also for text/scroll */
  if (modeinfo.ModeAttributes.mode_not_VGA_compatible)
      param->EGA_compatible = 0;
    else
      param->EGA_compatible = 1;
  if (modeinfo.ModeAttributes.BIOS_output_supported)
      param->BIOS_compatible = 1;
    else
      param->BIOS_compatible = 0;

#if USER_SUPPORT & VESA_4BPP_TEXT
  if (param->memtype == mem_text) {
      param->font = 0;
      param->nbcol = param->width;
      param->nbrow = param->height;
      param->charwidth = modeinfo.XCharSize;
//      param->width *= param->charwidth;
      param->charheight = modeinfo.YCharSize;
//      param->height *= modeinfo.YCharSize;
      VDBG (("using hardware font.\r\n"));
      }
    else
#endif
	 if (copy_gujin_param.attrib.use_gujin_embedded_font) {
      if (param->height < 350) {
	  param->font = xdata_adr(font8x8);
	  param->charheight = 8;
	  param->charwidth = 8;
	  }
	else if (param->height == 350) {
	  param->font = xdata_adr(font8x14);
	  param->charheight = 14;
	  param->charwidth = 8;
	  }
	else if (param->height <= 768) {
	  param->font = xdata_adr(font8x16);
	  param->charheight = 16;
	  param->charwidth = 8;
	  }
	else {
	  param->font = xdata_adr(font10x20);
	  param->charheight = 20;
	  param->charwidth = 10;
	  }
      param->nbcol = param->width / param->charwidth;
      param->nbrow = param->height / param->charheight;
      param->attr.ansi_font = 1;
      param->attr.null_del_available = 1;
      VDBG (("do not get VGA font, ANSI soft font %ux%u used; ", param->charwidth, param->charheight));
      VDBG (("farptr 0x%X, nbrow = %u.\r\n", param->font, param->nbrow));
      }
    else {
      unsigned short byteperchar;
      unsigned char nbrow;

      VDBG (("get VGA font "));
      if (param->height >= 350) {
	  param->font = _VGA_getfont (6, &nbrow, &byteperchar); /* 8x16 */
	  param->charheight = 16;
	  if (param->height == 350) {
	      unsigned short byteperchar14;
	      unsigned char nbrow14;
	      farptr font = _VGA_getfont (2, &nbrow14, &byteperchar14); /* 8x14 */

	      VDBG (("(font 8x16 at 0x%X, font 8x14 at 0x%X nbrow %u) ",
			param->font, font, nbrow14));
	      if (   font >= 0xC0000000 /* PC real mode address */
		  && font != param->font
		  && nbrow14 == (350 / 14) - 1) {
		  /* This font does seem to exist: */
		  param->font = font;
		  param->charheight = 14;
		  }
	      }
	  }
	else {
	  param->font = _VGA_getfont (3, &nbrow, &byteperchar); /* 8x8 */
	  param->charheight = 8;
	  }
      param->charwidth = 8;
      param->nbcol = param->width / param->charwidth;
      param->nbrow = param->height / param->charheight;
      VDBG (("farptr 0x%X, nbrow = %u.\r\n", param->font, nbrow));
      }

  if (   STRBEGINEQL (UI.info.vesa.cardname, "\007Copyright 1988-1991 TRIDENT MICROSYSTEM")
      && modeinfo.BitsPerPixel == 16) {
      VDBG ((" [16 BPP for TRIDENT card not working] "));
      return 0x106;
      }

  {
  struct videomode_str mode = {
      .number = param->identification,
      .width = param->width,
      .height = param->height,
      .bpp = get_bpp (param),
      .text = !param->attr.isgraphic,
      .attr = param->attr.isgraphic,
      .vesa = 1,
      };
  add_valid_mode (&mode);
  }
  return 0;
  }}

#if USER_SUPPORT & VESA_HARDWINDOW
/*
 * Before continuing reading that, I hope you have already read the
 * General Public License (GPL) version 2 under which this software
 * is released.
 * You can only use an extract of this code OR an idea used in this
 * code OR an algorithm present here if your software is released
 * under the GPL license. In fact, your software will be immediately
 * placed under GPL (free and unlimited source code available)
 * if you use, even by linking only (statically or dynamically),
 * anything written here - or in any other file of Gujin.
 */
static struct hard_recorder_str {
    unsigned short        nbwrite;
    unsigned short	  maxwrite;
    struct hardwindow_str *array;
    } __attribute__ ((packed)) hard_recorder;

VIDEO_FCT_PREFIX(hard_recorder_register) static void __attribute__ ((cdecl, regparm (0), attr_used))
hard_recorder_register (const unsigned edx, const unsigned eax, unsigned bx)
	/* "const" is written in purpose ! */
  {
  /* NO memcpy/memset here, %es is not initialised */
  /* NO %fs or %gs access, i.e. peekb() or display */
  if (hard_recorder.nbwrite < hard_recorder.maxwrite) {
      hard_recorder.array[hard_recorder.nbwrite].instruction = bx;
      hard_recorder.array[hard_recorder.nbwrite].port = edx;
      hard_recorder.array[hard_recorder.nbwrite].data = eax;
      }
  hard_recorder.nbwrite++;
  }

asm (
#if (SETUP & XCODE_SEGMENT)
"	.section        .usercode, \"ax\",@progbits			\n"
#else
".text									\n"
#endif
"	.align 0x20							\n"
"hard_recorder_handler:							\n"
"	pushl	%ds							\n"
"	pushl	%ebx							\n"
"	movw	%sp,%bx							\n"
"	movw	%ss:10(%bx),%bx						\n"
"	movw	%bx,%ds							\n"
"	movw	%sp,%bx							\n"
"	movw	%ss:8(%bx),%bx						\n"
"	movw	(%bx),%bx	# %bx now contains the opcode		\n"
"	cmpb	$0xEE,%bl						\n"
"	je	hard_recorder_match					\n"
"	cmpb	$0xEF,%bl						\n"
"	je	hard_recorder_match					\n"
"	cmpw	$0xEF66,%bx						\n"
"	je	hard_recorder_match					\n"
"hard_recorder_end:							\n"
"	popl	%ebx							\n"
"	popl	%ds							\n"
"	iretw								\n"
"hard_recorder_match:							\n"
"	pushl	%ecx							\n"
"	pushl	%ebx							\n"
"	pushl	%eax							\n"
"	pushl	%edx							\n"
"	movw	%cs,%bx							\n"
#ifdef CALLING_FROM_USERSEG
"	subw	$usercodeseg,%bx					\n"
#endif
"	addw	$deltaseg,%bx						\n"
"	movw	%bx,%ds							\n"
"	calll	hard_recorder_register					\n"
"	popl	%edx							\n"
"	popl	%eax							\n"
"	popl	%ebx							\n"
"	popl	%ecx							\n"
"	jmp	hard_recorder_end					\n"
	);

#ifdef COMPRESS_HARDWIN

VIDEO_FCT_PREFIX(clean_before_compress) static inline void
clean_before_compress (struct hardwindow_str *array, unsigned nb)
  {
  nb ++;

  while (--nb) {
      if ((array->instruction & 0xFF) == 0xEF) { /* outw */
	  array->instruction &= 0xFF;
	  array->data &= 0xFFFF;
	  }
	else if ((array->instruction & 0xFF) == 0xEE) { /* outb */
	  array->instruction &= 0xFF;
	  array->data &= 0xFF;
	  }
      array ++;
      }
  }

VIDEO_FCT_PREFIX(hardwin_init_pattern) static inline unsigned
hardwin_init_pattern (const struct hardwindow_str *array,
		      struct hardwindow_str *compress,
		      unsigned page_counter)
  {
  unsigned xor_data = array->data ^ compress->data;

  if (compress->instruction == 0xEF66) { /* outl */
      if (xor_data == 0) {
	  VDBG (("l")); /* always same outl */
	  return 0;
	  }
	else {
	  VDBG ((" failed, cannot compress] "));
	  return 1;
	  }
      }
    else if (compress->instruction == 0xEF) { /* outw */
      if (xor_data == 0) {
	  VDBG (("w")); /* always same outw */
	  return 0;
	  }
      if ((xor_data & 0xFF) != 0) {
	  /* direct port access, only one pattern */
	  if (xor_data == (page_counter << 1)) {
	      compress->instruction |= 187 << 8;
	      VDBG (("Z"));
	      return 0;
	      }
	    else {
	      VDBG ((" failed outw, cannot compress] "));
	      return 1;
	      }
	  }
	else {
	  /* std VGA access method */
	  xor_data >>= 8;
	  }
      }
    else if (compress->instruction == 0xEE) { /* outb */
      if (xor_data == 0) {
	  VDBG (("b")); /* always same outb */
	  return 0;
	  }
      }
    else {
      VDBG ((" unknown instruction, cannot compress] "));
      return 1;
      }

  /* Try to find a pattern */
  if (xor_data == page_counter) {
      compress->instruction |= 19 << 8;
      VDBG (("C"));
      }
#if 0 /* start as the previous compression */
    else if (xor_data == (page_counter & 0x0F)) {
      compress->instruction |= 1 << 8;
      VDBG (("D"));
      }
#endif
    else if (xor_data == ((page_counter & 0x0F) << 4)) {
      compress->instruction |= 3 << 8;
      VDBG (("E"));
      }
    else if (xor_data == (page_counter << 1)) {
      compress->instruction |= 87 << 8;
      VDBG (("I"));
      }
    else if (xor_data == (page_counter << 2)) {
      compress->instruction |= 28 << 8;
      VDBG (("G"));
      }
    else if (xor_data == ((page_counter & 0xF0) >> 2)) {
      compress->instruction |= 67 << 8;
      VDBG (("H"));
      }
    else if (xor_data == (page_counter >> 4)) {
      compress->instruction |= 20 << 8;
      VDBG (("F"));
      }
    else {
      VDBG ((" failed, cannot compress] "));
      return 1;
      }
  return 0;
  }

VIDEO_FCT_PREFIX(hardwin_check_pattern) static inline unsigned
hardwin_check_pattern (const struct hardwindow_str *array,
		       struct hardwindow_str *compress,
		       unsigned page_counter,
		       unsigned char method)
  {
  unsigned short page = page_counter;

  if (   method != 1 &&  method != 3 &&  method != 19 && method != 67
      && method != 20 && method != 28 && method != 87 && method != 187) {
      VDBG ((" failed, unknown compression method!] "));
      return 1;
      }

  if (method == 1 || method == 3)
      page &= 0x0F;

  if (method == 3)
      page <<= 4;
    else if (method == 28)
      page <<= 2;
    else if (method == 87 || method == 187)
      page <<= 1;
    else if (method == 20)
      page >>= 4;
    else if (method == 67)
      page = (page & 0xF0) >> 2;

  if (array->instruction == 0xEE) {
      if (array->data != (compress->data ^ page)) {
	  if (   method == 19
	      && (array->data == (compress->data ^ (page & 0x0F)))) {
	      compress->instruction &= 0x1FF;
	      VDBG (("d"));
	      }
	    else {
	      VDBG ((" failed, outb compression1 differ] "));
	      return 1;
	      }
	  }
	else
	  VDBG (("*"));
      }
    else if (array->instruction == 0xEF) {
      if ((method == 187)? array->data != (compress->data ^ page)
			 : array->data != (compress->data ^ (page << 8))) {
	  if (   method == 19
	      && (array->data == (compress->data ^ ((page & 0x0F) << 8)))) {
	      compress->instruction &= 0x1FF;
	      VDBG (("d"));
	      }
	    else {
	      VDBG ((" failed, outw compression1 differ] "));
	      return 1;
	      }
	  }
	else
	  VDBG (("*"));
      }
    else {
      VDBG ((" failed, unknown compressed1 instruction!] "));
      return 1;
      }
  return 0;
  }

VIDEO_FCT_PREFIX(hardwindow_compress) static unsigned inline
hardwindow_compress (struct hardwindow_str *long_array,
		     struct hardwindow_str *end,
		     unsigned secondwin_first)
  {
  struct hardwindow_str compressed_array[secondwin_first],
			*compress = compressed_array,
			*array = &long_array[secondwin_first];
  unsigned short page_counter = 1;

  memcpy (compressed_array, long_array, sizeof (compressed_array));
  VDBG (("%s: ", __FUNCTION__));

  while (array < end) {
      if (array->port != compress->port) {
	  VDBG ((" failed, different ports] "));
	  break;
	  }

      if (compress->instruction == 0xEF66) {
	  if (array->instruction != 0xEF66) { /* outl in ".code16" */
	      VDBG ((" failed, different instructions on outl] "));
	      break;
	      }
	  }
	else {
	  if ((compress->instruction & 0xFF) != array->instruction) {
	      VDBG ((" failed, different instructions on outw or outb] "));
	      break;
	      }
	  }

      if (compress->instruction == 0) {
	  if (array->instruction != 0) {
	      VDBG ((" failed, different number of instructions] "));
	      break;
	      }
	  page_counter ++;
	  compress = compressed_array;
	  VDBG ((","));
	  }
	else {
	  unsigned char method;

	  if (array->instruction == 0xEF66)
	      method = 0;
	    else
	      method = compress->instruction >> 8;

	  if (method == 0) { /* simple */
	      if (hardwin_init_pattern (array, compress, page_counter))
		  break;
	      }
	    else { /* compressed */
	      if (hardwin_check_pattern (array, compress, page_counter, method))
		  break;
	      }
	  compress++;
	  }
      array ++;
      }
  if (array < end)
      return 0; /* compression impossible with this intelligence level
			(sorry, it's mine !) */

  VDBG ((" OK] "));
  memcpy (long_array, compressed_array, sizeof (compressed_array));
  return nbof (compressed_array);
  }
#else /* COMPRESS_HARDWIN */
#define hardwindow_compress(a, b, c) (0) /* failed */
#endif /* COMPRESS_HARDWIN */

VIDEO_FCT_PREFIX(hardwindow_init) static unsigned
hardwindow_init (struct hardwindow_str **hardwindow, unsigned short **offsetarray, unsigned char winNo)
  {bound_stack();{
  /* This array has to be long, nbwindow is a byte: max 256, mostly
     when the VESA1 window granularity is 4 Kbytes... 12 Kbytes */
  struct hardwindow_str long_array[256 * 6];

  unsigned tmp, saved_interrupt_vector;
  unsigned short *offarray, nbwindow, max_array, i, j;

  nbwindow = DIVROUNDUP ((unsigned) UI.parameter.linelength * UI.parameter.height,
			 DISPLAY_WINDOW_GRANUL(&UI.parameter));

  /* to switch to window i, start hardwindow at hardwindow[offsetarray[i]] */
  /* hardwindow[offsetarray[0]] is first valid call, offsetarray[0] == 0  */
  /* hardwindow[offsetarray[nbwindow-1]] is last valid start index */
  /* offsetarray[nbwindow]-1 == last hardwindow[] index used */
  /* offsetarray[nbwindow+1] == (unsigned short)0 */
  /* hardwindow[offsetarray[nbwindow]-1] == {0} */

  VDBG ((" malloc1 %u bytes for %u windows ", 2 + 2 * nbwindow + 2, nbwindow));
  offarray = MALLOC (2 + 2 * nbwindow + 2, "VESAhwin");
  *offsetarray = offarray;
  if (offarray == 0) {
      *hardwindow = 0;
      VDBG (("FAILED!] "));
      return 1;
      }
  VDBG (("OK, "));

  if (nbwindow != 1) { /* remove possible optimisation */
      _VESA_setwin (1, winNo);
#if !(USER_SUPPORT & VESA_2WINDOWS)
      if (UI.parameter.NbWin == 2)
	  _VESA_setwin (1, winNo ^ 1);
#endif
      }

 if (BOOT1_DOS_running()) {
      extern unsigned char hard_recorder_handler[];

      saved_interrupt_vector = DOS_GetInterruptVector (1);
      DOS_SetInterruptVector (1, usercode_adr (hard_recorder_handler));
      }
    else {
      extern unsigned char hard_recorder_handler[];

      saved_interrupt_vector = peekl (4 * 1);
      pokel (4 * 1, usercode_adr (hard_recorder_handler));
      }

  offarray[0] = 0;
  for (i = 0; i < nbwindow; i++) {
      hard_recorder.nbwrite = 0;
      hard_recorder.array = &long_array[offarray[i]];
      if (nbof (long_array) - offarray[i] > 4)
	  hard_recorder.maxwrite = nbof (long_array) - offarray[i] - 3;
	else {
	  VDBG ((" no more space left in long_array!] "));
	  FREE (*offsetarray);
	  *offsetarray = 0;
	  *hardwindow = 0;
	  return 5;
	  }
      VDBG ((" set win %u ", i));
      setTFflag();
      _VESA_setwin (i, winNo);
#if !(USER_SUPPORT & VESA_2WINDOWS)
      if (UI.parameter.NbWin == 2)
	  _VESA_setwin (i, winNo ^ 1); /* concatenate ! */
#endif
      clearTFflag();
      if (hard_recorder.nbwrite == 0) {
	  VDBG ((" did not work, nothing recorded!] "));
	  FREE (*offsetarray);
	  *offsetarray = 0;
	  *hardwindow = 0;
	  return 3;
	  }
	else if (hard_recorder.nbwrite >= hard_recorder.maxwrite) {
	  VDBG ((" OVERFLOW %d out* !] ", hard_recorder.nbwrite));
	  FREE (*offsetarray);
	  *offsetarray = 0;
	  *hardwindow = 0;
	  return 4;
	  }
	else
	  VDBG ((" %d out*, ", hard_recorder.nbwrite));
      /* A zero in between each window: */
      hard_recorder.array[hard_recorder.nbwrite] = ((struct hardwindow_str) {});
      /* Set the end for current window / start for next window: */
      offarray[i+1] = offarray[i] + hard_recorder.nbwrite + 1;
      }
  _VESA_setwin (0, winNo);
#if !(USER_SUPPORT & VESA_2WINDOWS)
  if (UI.parameter.NbWin == 2)
      _VESA_setwin (0, winNo ^ 1);
#endif

  if (BOOT1_DOS_running())
      DOS_SetInterruptVector (1, saved_interrupt_vector);
    else
      pokel (4 * 1, saved_interrupt_vector);

  offarray[nbwindow + 1] = 0;	/* keep end array, write end marker after */

  clean_before_compress (long_array, offarray[nbwindow]);

  if ((tmp = hardwindow_compress (long_array, &long_array[offarray[nbwindow]],
				  offarray[1])) != 0) {
      FREE (*offsetarray);
      *offsetarray = 0; /* uncompressed */
      max_array = tmp;
      }
    else
      max_array = offarray[nbwindow];

  tmp = sizeof (struct hardwindow_str) * max_array;
  VDBG ((" OK, malloc2 %u bytes ", tmp));
  *hardwindow = MALLOC (tmp, "VESAwin2");
  if (*hardwindow == 0) {
       VDBG ((" FAILED!] "));
       if (*offsetarray) {
	   FREE (*offsetarray);
	   *offsetarray = 0;
	   }
       return 2;
       }

  if (*offsetarray)
      VDBG (("OK,\r\nWin0 [%u]: ", (*offsetarray)[0]));
    else
      VDBG (("OK,\r\nAllWin: "));
  for (i = 0, j = 0; i < max_array; i++) {
      (*hardwindow)[i] = long_array[i];

      if ((*hardwindow)[i].instruction == 0xEF66)
	  VDBG (("outl (0x%X, 0x%X); ",
		 (*hardwindow)[i].port, (*hardwindow)[i].data));
#ifdef COMPRESS_HARDWIN
	else if ((tmp = ((*hardwindow)[i].instruction >> 8)) != 0) {
	  if (((*hardwindow)[i].instruction & 0xFF) == 0xEE)
	      VDBG (("outb "));
	    else if (((*hardwindow)[i].instruction & 0xFF) == 0xEF)
	      VDBG (("outw "));
	    else
	      VDBG (("ERROR "));
	  VDBG (("(0x%X, ", (*hardwindow)[i].port));
	  if (tmp == 1)
	      VDBG (("0x%X ^ (page & 0x0F)); ", (*hardwindow)[i].data));
	    else if (tmp == 3)
	      VDBG (("0x%X ^ ((page & 0x0F) << 4)); ", (*hardwindow)[i].data));
	    else if (tmp == 19)
	      VDBG (("0x%X ^ page); ", (*hardwindow)[i].data));
	    else if (tmp == 20)
	      VDBG (("0x%X ^ (page >> 4)); ", (*hardwindow)[i].data));
	    else if (tmp == 87)
	      VDBG (("0x%X ^ (page << 1)); ", (*hardwindow)[i].data));
	    else if (tmp == 187)
	      VDBG (("0x%X ^ [page << 1]); ", (*hardwindow)[i].data));
	    else if (tmp == 28)
	      VDBG (("0x%X ^ (page << 2)); ", (*hardwindow)[i].data));
	    else if (tmp == 67)
	      VDBG (("0x%X ^ ((page & 0xF0) >> 2)); ", (*hardwindow)[i].data));
#ifdef COMPRESS_EXTENDED
	    else if (tmp == 37 && (((*hardwindow)[i].instruction & 0xFF) == 0xEE))
	      VDBG (("0x%X ^ ((inb(0x%X) & 0xF0) ^ (page & 0x0F))); ", (*hardwindow)[i].data, (*hardwindow)[i].port));
	    else if (tmp == 11 && (((*hardwindow)[i].instruction & 0xFF) == 0xEE))
	      VDBG (("0x%X ^ ((inb(0x%X) & 0x0F) ^ ((page & 0x0F) << 4))); ", (*hardwindow)[i].data, (*hardwindow)[i].port));
#endif
	    else
	      VDBG (("ERROR"));
	    }
#endif /* COMPRESS_HARDWIN */
	else if ((*hardwindow)[i].instruction == 0xEE)
	  VDBG (("outb (0x%X, 0x%X); ",
		 (*hardwindow)[i].port, (*hardwindow)[i].data));
	else if ((*hardwindow)[i].instruction == 0xEF)
	  VDBG (("outw (0x%X, 0x%X); ",
		 (*hardwindow)[i].port, (*hardwindow)[i].data));
	else if (   (*hardwindow)[i].instruction == 0
		 && (*hardwindow)[i].data == 0
		 && (*hardwindow)[i].port == 0) {
	  if (i == max_array - 1)
	      VDBG (("\r\nEnd."));
	    else {
	      j++;
	      VDBG (("\r\nWin%u [%u]: ", j, (*offsetarray)[j]));
	      }
	  }
	else
	  VDBG (("ERROR; "));
      }

  VDBG ((" OK"));
  return 0;
  }}
#endif /* VESA_HARDWINDOW */

#if (USER_SUPPORT & VESA_HARDWINDOW) && (USER_SUPPORT & VESA_2WINDOWS)
VIDEO_FCT_PREFIX(compare_hardwin) static inline unsigned
compare_hardwin (struct hardwindow_str *hardwindow1,
		 unsigned short *offsetarray1,
		 struct hardwindow_str *hardwindow2,
		 unsigned short *offsetarray2)
  {
  unsigned short nb, end;

  if ((offsetarray1 == 0) ^ (offsetarray2 == 0)) {
      VDBG ((" [%s compression difference] ", __FUNCTION__));
      return 0xFFFFFFFFU;
      }
  if (offsetarray1) { /* not compressed */
      unsigned short *ptr = offsetarray1 + 1; /* start index of second page */

      while (*ptr) ptr++; /* go to the end */
      end = *--ptr; /* that is the last index of hardwin used */
      VDBG ((" [%s uncompressed total used %u] ", __FUNCTION__, end));
      }
    else { /* compressed */
      struct hardwindow_str *ptr = hardwindow1;

      while (ptr->instruction) ptr++;
      end = ptr - hardwindow1;
      VDBG ((" [%s compressed total used %u] ", __FUNCTION__, end));
      }

    VDBG ((" ["));
    for (nb = 0; nb < end; nb++) {
      VDBG (("%d, ", nb));
      if (   hardwindow1->instruction != hardwindow2->instruction
	  || hardwindow1->port !=        hardwindow2->port
	  || hardwindow1->data !=        hardwindow2->data)
	  break;
      hardwindow1++;
      hardwindow2++;
      }
    VDBG (("] "));

  if (nb < end) {
      VDBG ((" [%s dependancy detected at %u] ", __FUNCTION__, nb));
      return nb + 1; /* can have a dependancy at offset 0 */
      }
    else {
      VDBG ((" [%s no dependancy] ", __FUNCTION__));
      return 0;
      }
  }

VIDEO_FCT_PREFIX(initialise_second_hardwindow) static inline void
initialise_second_hardwindow (unsigned short nbwindow)
  {
  /* _VESA_setwin (0, UI.parameter.winNo); already done */
  if (hardwindow_init(&UI.parameter.readhardwindow,
		      &UI.parameter.readoffsetarray,
		      UI.parameter.winNo ^ 1) != 0)
      VDBG (("FAILED hardwin 2"));
    else if (nbwindow >= 2) {
      struct hardwindow_str *hardwindow2;
      unsigned short *offsetarray2;

      _VESA_setwin (1, UI.parameter.winNo);
      if (hardwindow_init(&hardwindow2, &offsetarray2,
			  UI.parameter.winNo ^ 1) != 0)
	  VDBG (("FAILED 2nd try hardwin 2"));
	else {
	  unsigned short pos;
	  while ((pos = compare_hardwin (UI.parameter.readhardwindow,
					 UI.parameter.readoffsetarray,
					 hardwindow2,
					 offsetarray2)) != 0) {
	      VDBG (("HARDWINDOW of two window inter-dependant "
		     "at %u: inst1=0x%X, data1=0x%X, addr1=0x%X ; "
		     "inst2=0x%X, data2=0x%X, addr2=0x%X! "
		     "readoffsetarray=0x%X, offsetarray=0x%X; "
		     "1st window: instruction=0x%X, port=0x%X ",
		     pos - 1,
		     UI.parameter.readhardwindow[pos-1].instruction,
		     UI.parameter.readhardwindow[pos-1].data,
		     UI.parameter.readhardwindow[pos-1].port,
		     hardwindow2[pos-1].instruction,
		     hardwindow2[pos-1].data,
		     hardwindow2[pos-1].port,
		     (unsigned)UI.parameter.readoffsetarray,
		     (unsigned)UI.parameter.offsetarray,
		     UI.parameter.hardwindow[pos-1].instruction,
		     UI.parameter.hardwindow[pos-1].port));
#ifdef COMPRESS_EXTENDED
	      if (   UI.parameter.readoffsetarray == 0
		  && UI.parameter.offsetarray == 0
		  && UI.parameter.hardwindow[pos-1].port
				== UI.parameter.readhardwindow[pos-1].port) {
		  if (   UI.parameter.hardwindow[pos-1].instruction == 0x3EE
		      && (   UI.parameter.readhardwindow[pos-1].instruction == 0x1EE
			  || UI.parameter.readhardwindow[pos-1].instruction == 0x13EE) /* 0d19 */) {
		      VDBG ((" simple HARDWINDOW inter-dependancy fixed 1"));
		      UI.parameter.readhardwindow[pos-1].instruction = 0x25EE; /* 0d37 */
		      hardwindow2[pos-1].instruction = 0x25EE; /* 0d37 */
		      UI.parameter.hardwindow[pos-1].instruction |= 0x800; /* 0d11 */
		      UI.parameter.readhardwindow[pos-1].data = 0;
		      hardwindow2[pos-1].data = 0;
		      UI.parameter.hardwindow[pos-1].data = 0;
		      continue;
		      }
		  if (    UI.parameter.readhardwindow[pos-1].instruction == 0x3EE
		      && (   UI.parameter.hardwindow[pos-1].instruction == 0x1EE
			  || UI.parameter.hardwindow[pos-1].instruction == 0x13EE) /* 0d19 */) {
		      VDBG ((" simple HARDWINDOW inter-dependancy fixed 2"));
		      UI.parameter.readhardwindow[pos-1].instruction |= 0x800; /* 0d11 */
		      hardwindow2[pos-1].instruction |= 0x800; /* 0d11 */
		      UI.parameter.hardwindow[pos-1].instruction = 0x25EE; /* 0d37 */
		      UI.parameter.readhardwindow[pos-1].data = 0;
		      hardwindow2[pos-1].data = 0;
		      UI.parameter.hardwindow[pos-1].data = 0;
		      continue;
		      }
		  }
#endif
	      free_hardwin (); /* disable read and write hardwindow */
	      break;
	      }
	  if (offsetarray2)
	      FREE (offsetarray2);
	  FREE (hardwindow2);
	  }
      _VESA_setwin (0, UI.parameter.winNo);
      }
  }
#endif /* VESA_HARDWINDOW */

// not static only to not be removed by compiler:
awk_farret (VESA_setmode);
VIDEO_FCT_PREFIX(VESA_setmode) unsigned
VESA_setmode (unsigned mode)
  {bound_stack();{
  struct video_parameter_str param;
  unsigned tmp, clearscreen;

  if (mode & 0x8000)
      clearscreen = 0;
    else
      clearscreen = 1;
  mode &= ~0x8000;

  if ((tmp = VESA_getsize (mode, &param)) != 0)
      return tmp;

  mode &= ~0x4000;

#if USER_SUPPORT & VESA_LINEAR
  if (param.winsize == VESA2_MARKER) {
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
      if (param.winadr != 0 /* false VESA2 */ && segment_door (1) != 0) {
	  VDBG (("GS door or A20 gate does not want to open!\r\n"));
#if !(USER_SUPPORT & VESA_WINDOW)
	  remove_valid_mode (mode);
#endif
	  return 3;
	  }
      unsigned char segment4Gwork = 1;
      unsigned short old_gs = getgs();
      setgs(0);
      unsigned test_opened_door = gs_peekl ((const void *)0x100500);
      if (test_opened_door == gs_peekl ((const void *)0x000500)) {
	  gs_pokel((void *)0x100500, ~test_opened_door);
	  if (gs_peekl ((const void *)0x000500) == ~test_opened_door)
	      segment4Gwork = 0;
	  gs_pokel((void *)0x100500, test_opened_door);
	  }
      setgs(old_gs);
      if (!segment4Gwork) {
	  VDBG (("GS door opened but changing gs:0x100500 modifies gs:0x000500! (VirtualBox?)\r\n"));
	  copy_gujin_param.attrib.VESA2_interface = 0;
	  return 3;
	  }
#endif /* ASSEMBLY_TYPE != ASSEMBLY_DSES */

      if (param.winadr != 0 /* false VESA2 */) {
	  if (_VESA_setmode (mode | 0x4000) != 0) {
#if !(USER_SUPPORT & VESA_WINDOW)
	      remove_valid_mode (mode);
#endif
	      return 4;
	      }
	  }
	else {
	  if (_VESA_setmode (mode & ~0x4000) != 0) { /* VESA2 < 64K mode */
#if !(USER_SUPPORT & VESA_WINDOW)
	      remove_valid_mode (mode);
#endif
	      return 5;
	      }
	  _VESA_setwin (0, 1);
	  _VESA_setwin (0, 0);
	  }
#if USER_SUPPORT & VESA_HARDWINDOW
      free_hardwin();
#endif
      UI.parameter = param;
      }
    else
#endif /* VESA_LINEAR */
      {
#if USER_SUPPORT & VESA_WINDOW
      unsigned short nbwindow;
#if USER_SUPPORT & VESA_WINFUNCTION
      unsigned saved = UI.parameter.addrpositionfarfct;
#endif
      if (_VESA_setmode (mode & ~0x4000) != 0) {
	  VDBG (("VESA1: cannot switch to mode 0x%X!\r\n", mode & ~0x4000));
#if USER_SUPPORT & VESA_LINEAR
	  if ((UI.info.vesa.version >> 8) < 2) /* else can try VESA2 also */
#endif
	      remove_valid_mode (mode);
	  return 6;
	  }
#if USER_SUPPORT & VESA_WINFUNCTION
      UI.parameter.addrpositionfarfct = 0;
#endif
      if (param.NbWin >= 1 && param.memtype != mem_text)
	  if (_VESA_setwin (0, param.winNo) != 0) {
	      VDBG (("VESA1: cannot setup window for "
			"write/readwrite win!\r\n"));
#if USER_SUPPORT & VESA_WINFUNCTION
	      UI.parameter.addrpositionfarfct = saved;
#endif
	      _VESA_setmode (UI.parameter.identification);
#if USER_SUPPORT & VESA_LINEAR
	      if ((UI.info.vesa.version >> 8) < 2) /* else can try VESA2 also */
#endif
		  remove_valid_mode (mode);
	      return 7;
	      }
      if (param.NbWin == 2 && param.memtype != mem_text)
	  if (_VESA_setwin (0, param.winNo ^ 1) != 0) {
	      VDBG (("VESA1: cannot setup window for read win!\r\n"));
#if USER_SUPPORT & VESA_WINFUNCTION
	      UI.parameter.addrpositionfarfct = saved;
#endif
	      _VESA_setmode (UI.parameter.identification);
#if USER_SUPPORT & VESA_LINEAR
	      if ((UI.info.vesa.version >> 8) < 2) /* else can try VESA2 also */
#endif
		  remove_valid_mode (mode);
	      return 8;
	      }
#if USER_SUPPORT & VESA_HARDWINDOW
      free_hardwin();
#endif

      UI.parameter = param;
      UI.parameter.winadr = 0;

      nbwindow = DIVROUNDUP ((unsigned) UI.parameter.linelength	* UI.parameter.height,
				DISPLAY_WINDOW_GRANUL(&UI.parameter));

      VDBG ((" [access: "));
      if (nbwindow == 0)
	  VDBG (("nbwindow null] "));
#if USER_SUPPORT & VESA_WINFUNCTION
	else if (UI.parameter.addrpositionfarfct == 0)
	  VDBG (("function pointer null] "));
#endif
	else {
#if USER_SUPPORT & VESA_HARDWINDOW
	  if (copy_gujin_param.attrib.enable_VESA_hardwin) {
	      if (hardwindow_init(&UI.parameter.hardwindow,
				  &UI.parameter.offsetarray,
				  UI.parameter.winNo) != 0)
		  VDBG (("FAILED hardwin 1"));
#if USER_SUPPORT & VESA_2WINDOWS
		else if (UI.parameter.NbWin == 2)
		  initialise_second_hardwindow (nbwindow);
#endif /* VESA_2WINDOWS */
	      VDBG (("] "));
	      }
	    else
	      VDBG (("enable_VESA_hardwin gujin param attribute not set, "));
#else
	  VDBG (("function pointer not null] "));
#endif /* VESA_HARDWINDOW */
	  }

#else /* VESA_WINDOW */
      return 9;
#endif /* VESA_WINDOW */
      }

#if ASSEMBLY_TYPE == ASSEMBLY_DSES
  UI.parameter.base_address = linear2dsrelative (UI.parameter.base_address);
#else
  setgs (UI.parameter.base_address >> 4);
#endif

// Do we need _VESA_setDisplayStart(0,0); ?

  // UI.function.getsize, UI.function.setmode, UI.function.getmode : already initialised
  // init local functions without far-call convention:
  // will be inited depending of the number of bit per pixel...
  //    UI.function.setpixel = ;
  //    UI.function.getpixel = ;
  //    UI.function.plotHline = ;
  // init functions with far-call convention:
  extern void (*fptr_VESA_clearscreen) (void);
  asm ("# function %0 used " : : "g" (VESA_clearscreen));
  extern unsigned (*fptr_VESA_setcursor) (unsigned char row, unsigned char col);
  asm ("# function %0 used " : : "g" (VESA_setcursor));
  extern unsigned (*fptr_COMMON_getcursor) (unsigned char *row, unsigned char *col);
  asm ("# function %0 used " : : "g" (COMMON_getcursor));
  extern unsigned (*fptr_COMMON_setfgcolor) (unsigned color);
  asm ("# function %0 used " : : "g" (COMMON_setfgcolor));
  extern unsigned (*fptr_COMMON_setbgcolor) (unsigned color);
  asm ("# function %0 used " : : "g" (COMMON_setbgcolor));
  extern void (*fptr_VESA_setattribute) (struct attribute_str attr);
  asm ("# function %0 used " : : "g" (VESA_setattribute));
  extern void (*fptr_VESA_putstr) (const char *str);
  asm ("# function %0 used " : : "g" (VESA_putstr));
  extern unsigned (*fptr_BIOS_getkey) (unsigned timeout);
  asm ("# function %0 used " : : "g" (BIOS_getkey));

  UI.function.clearscreen = *fptr_VESA_clearscreen;
  UI.function.setcursor = *fptr_VESA_setcursor;
  UI.function.getcursor = *fptr_COMMON_getcursor;
  UI.function.setfgcolor = *fptr_COMMON_setfgcolor;
  UI.function.setbgcolor = *fptr_COMMON_setbgcolor;
  UI.function.setattribute = *fptr_VESA_setattribute;
  UI.function.putstr = *fptr_VESA_putstr;
  UI.function.getkey = *fptr_BIOS_getkey;

#if USER_SUPPORT & VESA_4BPP_EGA
// Amasingly working for the two following lines, treating 1BPP as
// EGA 1 plane seems to be a bit faster (VESA_plotHline not so optimised):

  if (UI.parameter.memtype == mem_EGA && UI.parameter.nbplane > 1) {
//  if (UI.parameter.memtype == mem_EGA) {
      const struct EGA_mode_str EGA_mode = {
	  .write_source		= cpuASsetreset,	/* write mode 2 as default */
	  .enable_tristate	= 0,
	  .read_mode		= EGA_DEFAULT_READ_MODE,
	  .enable_odd_even_addr	= 0,
	  .CGA_shift_register	= 0,
	  .enable_256_color	= 0,
	  .reserved		= 0
	  };

      VDBG ((" [%s: EGA_s/getpixel] ", __FUNCTION__));
      UI.function.setpixel = EGA_setpixel;
      UI.function.getpixel = EGA_getpixel;
      UI.function.plotHline = EGA_plotHline;
      _EGA_read_write_mode (EGA_mode);
      _EGA_write_setup (EGA_overwride, 0);
      _EGA_set_reset_enable (0);	/* write (mode 0) comes from CPU */
      EGA_READ_INIT();
      }
    else
#endif
      {
      if (UI.parameter.bitperpixel == 32) {
#if USER_SUPPORT & VESA_32BPP
	  UI.function.setpixel = VESA_setpixel_color32bit;
	  UI.function.getpixel = VESA_getpixel_color32bit;
	  UI.function.plotHline = VESA_plotHline;
#endif
	  }
	else if (UI.parameter.bitperpixel == 24) {
#if USER_SUPPORT & VESA_24BPP
	  UI.function.setpixel = VESA_setpixel_color24bit;
	  UI.function.getpixel = VESA_getpixel_color24bit;
	  UI.function.plotHline = VESA_24BPP_plotHline;
#endif
	  }
	else if (UI.parameter.bitperpixel == 16) {
#if USER_SUPPORT & VESA_16BPP
	  UI.function.setpixel = VESA_setpixel_color16bit;
	  UI.function.getpixel = VESA_getpixel_color16bit;
	  UI.function.plotHline = VESA_plotHline;
#endif
	  }
	else if (UI.parameter.bitperpixel == 4)  {
#if USER_SUPPORT & VESA_4BPP
	  UI.function.setpixel = VESA_setpixel_color4bit;
	  UI.function.getpixel = VESA_getpixel_color4bit;
	  UI.function.plotHline = VESA_plotHline;
#endif
	  }
	else if (UI.parameter.bitperpixel == 1)  {
#if USER_SUPPORT & VESA_1BPP
	  UI.function.setpixel = VESA_setpixel_color1bit;
	  UI.function.getpixel = VESA_getpixel_color1bit;
	  UI.function.plotHline = VESA_plotHline;
#endif
	  }
	else {
#if USER_SUPPORT & (VESA_8BPP | VESA_4BPP_TEXT)
	  UI.function.setpixel = VESA_setpixel_color8bit;
	  UI.function.getpixel = VESA_getpixel_color8bit;
	  UI.function.plotHline = VESA_plotHline;
#endif
	  }
      }

  VGA_VESA_setup_color ();
  UI.parameter.fontnbchar = 256;
#if SETUP & UNICODE_FONT
  if (UI.parameter.charheight == 16)
      UI.parameter.fontnbchar = NBCHAR8x16;
  if (UI.parameter.charheight == 20)
      UI.parameter.fontnbchar = NBCHAR10x20;
#endif

  {
  unsigned bgcolor, fgcolor;

#if USER_SUPPORT & VESA_4BPP_TEXT
  if (UI.parameter.memtype == mem_text) {
#ifdef VESA_0BPP_TEXT
      if (UI.parameter.nbcolor == 0) {
	  bgcolor = UI.stdcolor[black];
	  fgcolor = UI.stdcolor[white];
	  UI.attributes.valid = egatxt_attribute;
	  }
	else
#endif
	  {
	  bgcolor = UI.stdcolor[lightgray];
	  fgcolor = UI.stdcolor[blue];
	  UI.attributes.valid = no_attribute;
	  }

      if (copy_gujin_param.attrib.use_gujin_embedded_font) {
	  farptr fontptr;

	  switch (UI.parameter.charheight) {
	      case 8: fontptr = xdata_adr(font8x8); break;
	      case 14: fontptr = xdata_adr(font8x14); break;
	      case 16: fontptr = xdata_adr(font8x16); break;
	      default: fontptr = 0; break;
	      }
	  if (fontptr == 0)
	      VDBG (("Really bad: text mode with char height %u.\r\n",
			UI.parameter.charheight));
	    else {
	      VDBG (("downloading hardware font 8x%u at 0x%X.\r\n",
			UI.parameter.charheight, fontptr));
	      _VGAtxt_load_font (UI.parameter.charheight, 0, 256, 0, fontptr);
	      UI.parameter.currentUTFpage = 1;
	      UI.parameter.attr.ansi_font = 1;
	      UI.parameter.attr.null_del_available = 1;
	      }
	  }
	else {
	  VDBG (("using default hardware font.\r\n"));
	  UI.parameter.attr.ansi_font = VGA_VESA_current_font_ansi ();
	  }
      if (UI.parameter.EGA_compatible)
	  _EGA_blinking (0); /* No way to do so by VESA interface */
      }
    else
#endif
      {
      if (UI.parameter.nbcolor <= 2) {
	  bgcolor = UI.stdcolor[black];
	  fgcolor = UI.stdcolor[white];
	  }
	else {
	  bgcolor = UI.stdcolor[lightgray];
	  fgcolor = UI.stdcolor[blue];
	  }
      UI.attributes.valid = graphic_attribute;
      }
  UI.function.setbgcolor (bgcolor);
  UI.function.setfgcolor (fgcolor);
  }

  UI.attributes.current = no_attribute;

#if USER_SUPPORT & VESA_RECALC
  /*
   * EGA VERTICAL_DISPLAY_END field is only 10 bits, so max 1024 lines.
   * This will not work for "monochrome" VESA modes, i.e. I/O port
   * at 0x3B4/0x3B5 - I do not have such a hardware.
   * We do not modify the "overflow" register (2 most significant bits of
   * reg 0x07), because then we have to unprotect write access (reg 0x11),
   * and this register is maybe not readable.
   * We just modify the register if the buggy BIOS has programmed an
   * erroneous value not too far away (hope this reg is readable on
   * buggy BIOSes, else no correction).
   */
  if (   UI.parameter.EGA_compatible
      && UI.parameter.nbrow * UI.parameter.charheight <= 1024) {
      unsigned char tmp = _EGA_CRTR_in(0x12);
      unsigned short maxline;
      short error;

      if (UI.parameter.font != 0)
	  maxline = UI.parameter.height - 1; // to be used all the time
	else
	  maxline = UI.parameter.charheight * UI.parameter.nbrow - 1;
      error = (maxline & 0xFF) - tmp;

  if (error != 0 && ABS(error) <= 4) {
      VDBG ((" [VESA_RECALC: old reg 0x12: 0x%X, maxline = %u, error = %d] ",
		tmp, maxline, error));
      _EGA_CRTR_out (0x12, maxline & 0xFF);
      }
  }
#endif

  if (clearscreen)
      UI.function.clearscreen();

  VDBG ((", success %s\r\n\r\n", __FUNCTION__));
  return 0;
  }}

VIDEO_FCT_PREFIX(VESA_preinit) unsigned
VESA_preinit (void)
  {bound_stack();{
  VESA_VbeInfoBlock vesainfo;
  unsigned short mode, first_init;
  unsigned char *ptr;
  const char signature[4] = { 'V', 'E', 'S', 'A' };

#if USER_SUPPORT & VESA_LINEAR
  const char signature_vesa2[4] = { 'V', 'B', 'E', '2' };

  VDBG (("VESA2 call "));
  *CAST (unsigned *, vesainfo.VbeSignature) = *(unsigned *)signature_vesa2;

  if (_VESA_getinfo (&vesainfo) != 0 || *CAST (unsigned *, vesainfo.VbeSignature) != *CAST (unsigned *, signature)) {
      VDBG (("failed, trying VESA1, "));
#else
  if (1) {
      VDBG (("VESA1 call "));
#endif
      *CAST (unsigned *, vesainfo.VbeSignature) = *CAST (unsigned *, signature);
      if (_VESA_getinfo (&vesainfo) != 0 || *CAST (unsigned *, vesainfo.VbeSignature) != *CAST (unsigned *, signature)) {
	  VDBG (("failed\r\n"));
	  UI.info.vesa.version = 0;
	  memset (UI.info.vesa.cardname, 0, sizeof (UI.info.vesa.cardname));
	  return 1;
	  }
      }

  VDBG (("successfull, version: %u.%u, capabilities: %u, videoram: %u x 64 Kbytes, ",
	vesainfo.VbeVersion >> 8, vesainfo.VbeVersion & 0xFF,
	*CAST(unsigned *, &vesainfo.Capabilities), vesainfo.TotalMemory));

  first_init = (UI.info.vesa.version == 0)? 1 : 0;
  UI.info.vesa.version = vesainfo.VbeVersion;
  UI.info.vesa.memory_64Kb = vesainfo.TotalMemory;

  if (peekb (vesainfo.OemStringPtr) == '\007')  /* Trident 8900 */
      vesainfo.OemStringPtr++;
  for (ptr = UI.info.vesa.cardname; ptr < &UI.info.vesa.cardname[sizeof(UI.info.vesa.cardname)-2]; ptr++) {
      *ptr = peekb (vesainfo.OemStringPtr++);
      if (*ptr < ' ' || *ptr > '~')
	  break;
      }
  *ptr = '\0';
  VDBG (("name:[%s], modes:", UI.info.vesa.cardname));
  while ((mode = peekw(vesainfo.VideoModePtr)) != 0xFFFF) {
      if (first_init) {
	  struct video_parameter_str param;

	  VDBG (("\r\n0x%X ", mode));
	  /*
	   * This will call add_valid_mode():
	   * Take care, some boards have two times the same VESA number
	   * in the list "vesainfo.VideoModePtr" !
	   */
	  if (VESA_getsize (mode, &param) == 0)
	      VDBG (("OK, "));
	    else
	      VDBG (("ERROR, "));
	  }
	else {
	  /*
	   * Some VESA video mode may have been disabled after a failure
	   * of VESA_setmode(), do not revalidate them here...
	   */
	  VDBG (("0x%X, ", mode));
	  }
      vesainfo.VideoModePtr += sizeof(short);
      }
  VDBG (("\r\n"));
  return 0;
  }}

VIDEO_FCT_PREFIX(VESA_init) unsigned
VESA_init (void)
  {bound_stack();{
#if USER_SUPPORT & VESA_PMINFO
  farptr PMdata_addr, base;
  unsigned short j, PMdata_len, tmp;

  tmp = _VESA_GetPMinterface (&PMdata_addr, &PMdata_len);
  VDBG (("[get PM data: 0x%X, PMdata_addr = 0x%X, PMdata_len = 0x%X: ",
		tmp, PMdata_addr, PMdata_len));
  if (   tmp
      || (unsigned long)PMdata_addr <= 0xA0000000U
      || (unsigned long)PMdata_addr > 0xFFFFFFFFU - 10
      || PMdata_len <= 10) {
      VDBG (("failed] "));
      UI.ProtectedModeIO[0] = 0xFFFF;
      UI.ProtectedModeMem[0].address = 0xFFFFFFFF;
      UI.ProtectedModeMem[0].length = 0;
      }
    else {
      base = peekw (PMdata_addr + 6);
      VDBG (("Offsets = %u, %u, %u, I/O at %u; ",
		peekw (PMdata_addr),
		peekw (PMdata_addr+2),
		peekw (PMdata_addr+4),
		base ));
      VDBG (("I/O: "));
      if (base < 8) {
	  VDBG (("abscent] "));
	  UI.ProtectedModeIO[0] = 0xFFFF;
	  UI.ProtectedModeMem[0].address = 0xFFFFFFFFU;
	  UI.ProtectedModeMem[0].length = 0;
	  }
	else {
	  base += PMdata_addr;
	  j = 0;
	  do {
	      tmp = peekw (base + 2*j);
	      VDBG (("%x, ", tmp));
#if 0 /* It is too late to enable it here, do it in /etc/dosemu.conf */
 /* In /etc/dosemu.conf, add for Voodoo3 (VESA page switching at 0x142C):
  $_ports = "0x3C8 0x3C9 0x3D4 0x3D5 0x3DA range 0x142C,0x142F" */
	      {
	      unsigned short version, patchlevel;

	      if (_DOSEMU_check (&version, &patchlevel) == 0) {
		  VDBG (("Info: enabling DOSEMU I/O access for 0x%X\r\n",
				tmp));
		  _DOSEMU_enable_ioports (tmp, 1);
		  }
	      }
#endif
	      if (j >= lastof (UI.ProtectedModeIO)) {
		  if (j == lastof (UI.ProtectedModeIO)) {
		      VDBG ((" [UI.ProtectedModeIO array too short] "));
		      UI.ProtectedModeIO[j] = 0xFFFF;
		      }
		  }
		else
		  UI.ProtectedModeIO[j] = tmp;
	      j++;
	      } while (tmp != 0xFFFF && 2*j <= PMdata_len);
	  if (tmp != 0xFFFF && j <= lastof (UI.ProtectedModeIO)) {
	      UI.ProtectedModeIO[j] = 0xFFFF;
	      }
	  VDBG (("MEM: "));
	  if (2*j <= PMdata_len) {
	      base += 2*j;
	      PMdata_len -= 2*j;
	      j = 0;
	      do {
		  UI.ProtectedModeMem[j].address = peekl (base + 6*j);
		  UI.ProtectedModeMem[j].length = peekw (base + 6*j+4);
		  VDBG (("%x+%x, ",
			UI.ProtectedModeMem[j].address,
			UI.ProtectedModeMem[j].length));
		  j++;
		  } while (   (UI.ProtectedModeMem[j-1].address & 0xFFFF) != 0xFFFF
			   && j < nbof (UI.ProtectedModeMem)
			   && 6*j <= PMdata_len);
	      if (j == nbof (UI.ProtectedModeMem))
		  VDBG ((" [UI.ProtectedModeMem array too short] "));
	      UI.ProtectedModeMem[j-1].address = 0xFFFFFFFFU;
	      }
	    else {
	      VDBG (("over PMdata_len limit "));
	      UI.ProtectedModeMem[0].address = 0xFFFFFFFFU;
	      }
	  VDBG (("] \r\n"));
	  }
      }
#endif /* VESA_PMINFO */

#if 0
  {
  unsigned short mode;
  if (_VESA_getmode (&mode) != 0)
      VDBG (("Initial VESA_getmode failed.\r\n"));
    else
      if (VESA_setmode (mode | 0x8000) != 0)	/* Voodoo: clear screen */
	  VDBG (("Initial VESA_setmode failed.\r\n"));
  }
#endif

  /* If VESA_setmode() failed, it does not matter, stay with previous UI */
  extern unsigned (*fptr_VESA_getsize) (unsigned mode, struct video_parameter_str *param);
  asm ("# function %0 used " : : "g" (VESA_getsize));
  extern unsigned (*fptr_VESA_setmode) (unsigned mode);
  asm ("# function %0 used " : : "g" (VESA_setmode));
  extern unsigned (*fptr_VESA_getmode) (void);
  asm ("# function %0 used " : : "g" (VESA_getmode));
  UI.function.getsize = *fptr_VESA_getsize;
  UI.function.setmode = *fptr_VESA_setmode;
  UI.function.getmode = *fptr_VESA_getmode;

  return 0;
  }}
#endif /* VESA_SUPPORT */

#if USER_SUPPORT & SERIAL_SUPPORT
/*
 * The default "serial BIOS library" functions, only for
 * DEC VT100+ compatible terminal.
 */
VIDEO_FCT_PREFIX(SERIAL_getc) static unsigned
SERIAL_getc (unsigned char *C)
  {bound_stack();{
  _BIOS_serialstatus ret;
  unsigned int cpt = 3; /* cpt * BIOS serial timeout */
  unsigned port = BOOT1_COM_port();

  checkptr (C, "SERIAL_getc invalid parameter.");

  do {
      ret = _BIOS_getserial (port);
      } while ((ret.timeout) && (--cpt != 0));

  *C = ret.al.charread;
  if (ret.timeout)
      return 1;
    else
      return 0;
  }}

VIDEO_FCT_PREFIX(CSI_tilda) static inline unsigned
CSI_tilda (unsigned short nb)
  {
  if (nb >= 11 && nb <= 15)
      return FCT_KEYCODE(nb-11+1);	/* F1..F5 */
    else if (nb >= 17 && nb <= 21)
      return FCT_KEYCODE(nb-17+6);	/* F6..F10 */
    /* We want to handle ShiftF1..ShiftF12 even at the cost of losing F13,F14... of real VT */
    /* F13 and next are treated as ShiftF1 and next */
    else if (nb >= 25 && nb <= 28)
      return SFCT_KEYCODE(nb-25+1);	/* SF1..SF4 */
    else if (nb >= 30 && nb <= 33)
      return SFCT_KEYCODE(nb-30+5);	/* SF5..SF8 */
    else if (nb >= 34 && nb <= 37)
      return SFCT_KEYCODE(nb-34+9);	/* SF9..SF12 */
    /* End handling ShiftF1..ShiftF12 */
    else if (nb >= 23 && nb <= 26)
      return FCT_KEYCODE(nb-23+11);	/* F11..F14 */
    else if (nb == 28)
      return FCT_KEYCODE(1);		/* Help == F1 */
    else if (nb == 29)
      return FCT_KEYCODE(5);		/* Do == F5 (PF1..PF4) */
    else if (nb >= 31 && nb <= 34)
      return FCT_KEYCODE(nb-31+17);	/* F15..F18 */
    else if (nb == 6)
      return PAGEDOWN_KEYCODE();
    else if (nb == 5)
      return PAGEUP_KEYCODE();
    else if (nb == 1)
      return HOME_KEYCODE(); /* Find, on a PC */
    else if (nb == 4)
      return END_KEYCODE(); /* Select, on a PC */
    else
      return '~';
  }

awk_farret (SERIAL_getkey);
VIDEO_FCT_PREFIX(SERIAL_getkey) static unsigned
SERIAL_getkey (unsigned timeout)
  {bound_stack();{
  unsigned returned;
  unsigned init_tick = _BIOS_nbtick();

  if (timeout) {
      if (timeout != 0xFFFFFFFFU)
	  timeout += init_tick;
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
      if (MOUSE_inited())
	  MOUSE_start();
#endif
      }
  for (;;) {
      if (_BIOS_getserialstatus(BOOT1_COM_port()).receive_data_ready) {
	  unsigned char tmp;

	  /* Ignore ^S, ^Q, i.e. XOFF XON, sorry: */
	  while (   SERIAL_getc(&tmp) != 0
		 || tmp == '\023' || tmp == '\021') {
	      _BIOS_wait (SERIAL_DELAY);
	      continue;
	      }
	  if (   tmp == (unsigned char)'\233' /* CSI */
	      || tmp == (unsigned char)'\217' /* SS3 */
	      || (   tmp == '\033' && SERIAL_getc(&tmp) == 0
		  && (tmp == '[' || tmp == 'O'))) {
	      if (tmp == (unsigned char)'\233' || tmp == '[') {
		  unsigned short nb = 0;

		  /* TREAT CSI or ESC [ */
		  while (SERIAL_getc(&tmp) == 0 && tmp >= '0' && tmp <= '9')
		      nb = (nb * 10) + tmp - '0';
		  if (tmp == '~')
		      returned = CSI_tilda(nb);
		    else if (tmp == 'A')
		      returned = UP_KEYCODE();
		    else if (tmp == 'B')
		      returned = DOWN_KEYCODE();
		    else if (tmp == 'C')
		      returned = RIGHT_KEYCODE();
		    else if (tmp == 'D')
		      returned = LEFT_KEYCODE();
		    else if (tmp == 'M')
		      returned = FCT_KEYCODE(5);
		    else if (tmp == 'K')
		      returned = SFCT_KEYCODE(5);
		    else if (tmp == 'H')
		      returned = HOME_KEYCODE();
		    else if (tmp == '[') { /* ESC [ [ A..E  (PC equivalent) */
		      if (SERIAL_getc(&tmp) == 0  && (tmp >= 'A' && tmp <= 'E'))
			  returned = FCT_KEYCODE(tmp - 'A' + 1);
			else
			  returned = tmp;
		      }
		    else
		      returned = tmp;
		  }
		else {
		  /* TREAT SS3 or ESC O */
		  if (SERIAL_getc(&tmp) == 0  && (tmp >= 'P' && tmp <= 'S'))
		      returned = FCT_KEYCODE(tmp - 'P' + 1); /* PF1..PF4 */
		    else
		      returned = tmp;
		  }
	      }
	    else
	      returned = tmp;
	  break;
	  }
      if (timeout == 0 || (_BIOS_nbtick() >= timeout)) {
	  returned = TIMEOUT_KEYCODE();
	  break;
	  }
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
      if (MOUSE_inited()) {
	  returned = MOUSE_poll();
	  if (returned != 0xFFFFFFFF)
	      break;
	  }
#endif
      if (BOOT1_DOS_running())
	  DOS_IdleInterrupt();
      cool_processor ();
//      _BIOS_wait (1000);
      _BIOS_wait (10); /* not loosing chars on UART for function keys */
      }
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
  if (MOUSE_inited() && timeout)
      MOUSE_stop();
#endif

//VDBG ((" [%s : 0x%X] ", __FUNCTION__, returned));

  if (returned == 3) { /* ^C */
      SDBG (("\r\nTrying to stop '.com', stop DOSEMU, generate DOSEMU exception, halt\r\n"));
      BIOS_killme();
      }

  return returned;
  }}

VIDEO_FCT_PREFIX(BOOT1_printf) static int
BOOT1_printf (const char *format, ...)
  {bound_stack();{
  char buffer[256];

#if 0 // DEBUG & DEBUG_SERIAL
  char *tmp;
  unsigned short cpt;
  static unsigned report = 0;

  tmp = snprtf (buffer, sizeof(buffer), format,
	  (void **)((unsigned)&format + sizeof(const char *)) );
  for (cpt = 0; tmp[cpt] != 0; cpt++) {
      if ((tmp[cpt] & 0x80) | (tmp[cpt] & 0xFF))
	  report = 10;
      if (report) {
	  if ((tmp[cpt] & 0xFF) < 0x20)
	      SDBG ((" {0x%X} ", tmp[cpt] & 0xFF));
	    else
	      SDBG ((" {%c} ", tmp[cpt] & 0xFF));
	  report --;
	  }
      }
  BOOT1_putstr (tmp);
#else
  BOOT1_putstr (snprtf (buffer, sizeof(buffer), format,
	  (void **)((unsigned)&format + sizeof(const char *)) ));
#endif
  return 0;
  }}

VIDEO_FCT_PREFIX(SERIAL_setup_color) static inline void
SERIAL_setup_color (void)
  {
  static const unsigned char SERIAL_color[16] = {
      /* Serial terminal with linux colors: */
      0,	/* black */
      4,	/* blue  */
      2,	/* green */
      6,	/* cyan  */
      1,	/* red   */
      5,	/* magenta */
      3,	/* brown */
	/* light colors only for foreground, not background. */
      7 + 8,	/* lightgray */
      7,	/* darkgray */
      4 + 8,	/* lightblue */
      2 + 8,	/* lightgreen */
      6 + 8,	/* lightcyan */
      1 + 8,	/* lightred */
      5 + 8,	/* lightmagenta */
      3 + 8,	/* yellow */
      8		/* white */
      };
  static const unsigned char DEC_VT_color[16] = {
      0,	/* black */
      1,	/* blue  */
      2,	/* green */
      3,	/* cyan  */
      4,	/* red   */
      5,	/* magenta */
      6,	/* brown */
      7,	/* lightgray */
      8,	/* darkgray */
      9,	/* lightblue */
      10,	/* lightgreen */
      11,	/* lightcyan */
      12,	/* lightred */
      13,	/* lightmagenta */
      14,	/* yellow */
      15	/* white */
      };
  unsigned i;
  const unsigned char *color;

  if (UI.monitor == 4 || UI.monitor == 3)	/* VT[2345]40 or VT[234]30 */
      color = DEC_VT_color;
    else if (UI.monitor == 0)	/* Linux terminal (color?) */
      color = SERIAL_color;
    else {			/* B&W display, even if it is green! */
      for (i = 0; i < nbof (UI.stdcolor); i++)
	  UI.stdcolor[i] = i & 1;
      return;
      }

  for (i = 0; i < nbof (UI.stdcolor); i++)
      UI.stdcolor[i] = color[i];
  }

VIDEO_FCT_PREFIX(SERIAL_waitchar) static unsigned
SERIAL_waitchar (unsigned char waited)
  {bound_stack();{
  int cpt = 0, status, nb = 0;
  unsigned char C;

  while (1) {
      status = SERIAL_getc(&C);
      if (status == 0 && C == waited)
	  return 0;
      if (status == 0 && C == '\033' && waited >= 0x80 && waited <= 0x9F) {
	  /* special VT 8->7bits conversion */
#define CHAR_CSI	'\233'
#define CHAR_DCS	'\220'
#define CHAR_ST		'\234'
	  waited -= 0x40;
	  continue;
	  }
      SDBG ((" %s, waiting 0x%X", __FUNCTION__, waited));
      if (status != 0 || cpt++ > 10) {
	  SDBG ((", %s received.\r\n", (nb != 0)? "other char" : "nothing"));
	  return 1;
	  }
      if (status == 0) {
	  nb++;
	  SDBG ((", get: 0x%X\r\n", C));
	  }
      }
  }}

VIDEO_FCT_PREFIX(SERIAL_waitreportNUM) static unsigned __attribute__ ((noinline))
SERIAL_waitreportNUM (unsigned int *val, unsigned char separator)
  {bound_stack();{
  unsigned int cpt = 0, tmp = 0;
  unsigned char C;

  while (cpt++ < 10) {
      if (SERIAL_getc(&C) != 0) {
	  SDBG ((" %s wait number, nothing received.\r\n", __FUNCTION__));
	  continue;
	  }
      if (C >= '0' && C <= '9')
	  tmp = (tmp * 10) + (C - '0');
	else if (C == separator)
	  break;
	else if (C == '\033' && separator >= 0x80 && separator <= 0x9F) {
	  separator -= 0x40;
	  continue;
	  }
	else
	  SDBG ((" %s, not a digit nor 0x%X but 0x%X\r\n",
		__FUNCTION__, separator, C));
      }
  if (cpt >= 10) {
      SDBG ((" %s aborted.\r\n", __FUNCTION__));
      return 1;
      }
  if (val)
      *val = tmp;
  return 0;
  }}

VIDEO_FCT_PREFIX(SERIAL_getVTreport) static inline unsigned
SERIAL_getVTreport (unsigned feature, unsigned int *result)
  {
  unsigned tmp;

  BOOT1_printf ("\033[?%u$p", feature);

  if (SERIAL_waitchar (CHAR_CSI) != 0) {
      SDBG ((" %s aborted on CSI ", __FUNCTION__));
      return 1;
      }
  if (SERIAL_waitchar ('?') != 0) {
      SDBG ((" %s aborted on '?' ", __FUNCTION__));
      return 1;
      }
  if (SERIAL_waitreportNUM (&tmp, ';') != 0 || tmp != feature) {
      SDBG ((" %s bad 'feature' response ", __FUNCTION__));
      return 1;
      }
  if (SERIAL_waitreportNUM (&tmp, '$') != 0) {
      SDBG ((" %s bad 'value' response ", __FUNCTION__));
      return 1;
      }
  if (SERIAL_waitchar ('y') != 0) {
      SDBG ((" %s aborted on trailer ", __FUNCTION__));
      return 1;
      }
  if (result)
      *result = tmp;
  return 0;
  }

VIDEO_FCT_PREFIX(SERIAL_getcontrol) static unsigned inline
SERIAL_getcontrol (const char *feature, unsigned int *result)
  {
  unsigned int tmp;

  BOOT1_printf ("\033P$q%s\033\\", feature); /* DCS $ q <for instance "*|"> ST */

  if (SERIAL_waitchar (CHAR_DCS) != 0) {
      SDBG ((" %s aborted on DCS ", __FUNCTION__));
      return 1;
      }
  if (SERIAL_waitreportNUM (&tmp, '$') != 0) {
      SDBG ((" %s bad format ", __FUNCTION__));
      return 1;
      }
  if (tmp != 0) {
      SDBG ((" %s 'invalid request': 0x%X ", __FUNCTION__, tmp));
//      return 1;	// my VT420 always set this bit, why?
      }
  if (SERIAL_waitchar ('r') != 0) {
      SDBG ((" %s aborted on 'r' ", __FUNCTION__));
      return 1;
      }
  if (SERIAL_waitreportNUM (&tmp, feature[0]) != 0) {
      SDBG ((" %s bad 'value/terminator' response ", __FUNCTION__));
      return 1;
      }
  /* Control Fuction are either one or two char wide */
  if (feature[1] != 0 && SERIAL_waitchar (feature[1]) != 0) {
      SDBG ((" %s bad terminator2 response ", __FUNCTION__));
      return 1;
      }
  if (SERIAL_waitchar (CHAR_ST) != 0) {
      SDBG ((" %s bad terminator ", __FUNCTION__));
      return 1;
      }
  if (result)
      *result = tmp;
  return 0;
  }

awk_farret (SERIAL_putstr);
VIDEO_FCT_PREFIX(SERIAL_putstr) static void
SERIAL_putstr (const char *str)
  {bound_stack();{
  /*
   * terminal do not clear the end of line when printing "\r\n", so do it ourself:
   * Cursor is entirely managed by terminal.
   */

  /* UI.parameter.attr.ansi_font is always set here, shall we use it to enable
	UTF-8 transmission on the serial line? */
  while (*str != 0) {
      UI.parameter.attr.has_outputed = 1;
      char tmpbuff[strlen(str)+1], *dst = tmpbuff; /* UTF-8 will reduce that string */
      while (*str != '\0' && *str != '\r') {
	  unsigned utfchar, charlen = UTF8LEN(str);	/* handle UTF-8 subset */
	  if (charlen == 2 && (utfchar = UTF8VAL2(str)) < UI.parameter.fontnbchar) {
	      *dst++ = utfchar;
	      str += charlen;
	      }
	    else if (charlen >= 2) {
	      *dst++ = '?';
	      str += charlen;
	      }
	    else
	      *dst++ = *str++;
	  }
      *dst = '\0';
      if (dst - tmpbuff != 0)
	  BOOT1_putstr (tmpbuff);
      if (*str == '\r') {
//	  if (UI.monitor != 0) { /* minicom is here... */
	      BOOT1_putstr ("\033[K\r"); /* clear end of line */
	      _BIOS_wait (SERIAL_DELAY);
//TODO:   UI.parameter.attr.has_scrolled = 1;
//	      }
	  str++;
	  }
      }
  }}

awk_farret (SERIAL_clearscreen);
VIDEO_FCT_PREFIX(SERIAL_clearscreen) static void
SERIAL_clearscreen (void)
  {
  BOOT1_putstr ("\033[H\033[J");
  _BIOS_wait (SERIAL_DELAY);
  }

awk_farret (SERIAL_getmode);
VIDEO_FCT_PREFIX(SERIAL_getmode) static unsigned __attribute__ ((noinline))
SERIAL_getmode (void)
  {bound_stack();{
  unsigned int tmp;
  unsigned mode;

  if (SERIAL_getVTreport (3, &tmp) != 0) {
      SDBG ((" %s aborted on columns\r\n", __FUNCTION__));
      return 0;	/* return mode 0 if error */
      }
  if (tmp == 1 || tmp == 3) /* set or permanently set */
      mode = 1;
    else
      mode = 0;

  if (UI.monitor <= 1)
      return mode; /* would still work, but too long a timeout */

  if (SERIAL_getcontrol ("*|", &tmp) != 0) {
      SDBG ((" %s aborted on lines\r\n", __FUNCTION__));
      return mode;	/* return 0 or 1 if partial error */
      }
  switch (tmp) {
    case 24: break;
    case 36: mode += 2; break;
    case 48: mode += 4; break;
    default:
      SDBG ((" %s invalid lines\r\n", __FUNCTION__));
      break;
    }
  return mode; /* always return screen cleared on last mode set */
  }}

#ifndef MOUSE_ON_VT
awk_farret (SERIAL_getcursor);
static
#endif /* Also called from mouse.c: */
VIDEO_FCT_PREFIX(SERIAL_getcursor) unsigned
SERIAL_getcursor (unsigned char *row, unsigned char *col)
  {bound_stack();{
  unsigned int tmprow, tmpcol;

  checkptr (row, "SERIAL_getcursor invalid 1st parameter.");
  checkptr (col, "SERIAL_getcursor invalid 2nd parameter.");

  BOOT1_putstr ("\033[6n");
  if (SERIAL_waitchar (CHAR_CSI) != 0) {
      SDBG ((" %s aborted on CSI ", __FUNCTION__));
      return 1;
      }
  if (SERIAL_waitreportNUM (&tmprow, ';') != 0) {
      SDBG ((" %s aborted on number 1 ", __FUNCTION__));
      return 1;
      }
  if (SERIAL_waitreportNUM (&tmpcol, 'R') != 0) {
      SDBG ((" %s aborted on number 2 ", __FUNCTION__));
      return 1;
      }
  *row = (unsigned char)tmprow - 1;	/* HOME on VT at (1,1) */
  *col = (unsigned char)tmpcol - 1;
  return 0;
  }}

#ifndef MOUSE_ON_VT
awk_farret (SERIAL_setcursor);
static
#endif /* Also called from mouse.c: */
VIDEO_FCT_PREFIX(SERIAL_setcursor) unsigned
SERIAL_setcursor (unsigned char row, unsigned char col)
  {
#if 1
  BOOT1_printf ("\033[%u;%uH", row+1, col+1);   /* HOME on VT at (1,1) */
  _BIOS_wait (SERIAL_DELAY);
#else
  char tmpstr[10], *ptr = tmpstr;

  row++; col++; /* HOME on VT at (1,1) */
  *ptr++ = '\033';
  *ptr++ = '[';
  if (row >= 10)
      *ptr++ = '0' + (row / 10);
  *ptr++ = '0' + (row % 10);
  *ptr++ = ';';
  if (col >= 100)
      *ptr++ = '0' + (col / 100);
  if (col >= 10)
      *ptr++ = '0' + ((col / 10) % 10);
  *ptr++ = '0' + (col % 10);
  *ptr++ = 'H';
  *ptr = '\0';
  BOOT1_putstr (tmpstr);
#endif
  if (row <= UI.parameter.nbrow)
      UI.parameter.row = row;
  if (col <= UI.parameter.nbcol)
      UI.parameter.col = col;
  return 0;
  }

awk_farret (SERIAL_setattribute);
VIDEO_FCT_PREFIX(SERIAL_setattribute) static void
SERIAL_setattribute (struct attribute_str attr)
  {bound_stack();{
  unsigned char old = ATTR2BYTE(UI.attributes.current);

  ATTR2BYTE(attr) &= ATTR2BYTE(UI.attributes.valid);

  if (   ATTR2BYTE(attr) == 0
      || ((ATTR2BYTE(attr) & old) != old)) {
      /* force if 0, or if some attr are removed */
      /* "\033[21m" do not work on VT100 */
      BOOT1_putstr ("\033[m");
      old = 0;
      }
  UI.attributes.current = attr;

  ATTR2BYTE(attr) &= ~old;

  if (ATTR2BYTE(attr) != 0) {
#if 0
      unsigned char first = 1;
      BOOT1_putstr ("\033[");
      if (attr.underline) {
	  BOOT1_putstr ("4");
	  first = 0;
	  }
      if (attr.reverse) {
	  if (!first)
	      BOOT1_putstr (";");
	  first = 0;
	  BOOT1_putstr ("7");
	  }
      if (attr.blink) {
	  if (!first)
	      BOOT1_putstr (";");
	  first = 0;
	  BOOT1_putstr ("5");
	  }
      if (attr.brighter) {
	  if (!first)
	      BOOT1_putstr (";");
	  BOOT1_putstr ("1");
	  }
      BOOT1_putstr ("m");
#else
      char tmp[11], *ptr = tmp;

      *ptr++ = '\033'; *ptr++ = '[';
      if (attr.underline)
	  *ptr++ = '4';
      if (attr.reverse) {
	  if (ptr != &tmp[1])
	      *ptr++ = ';';
	  *ptr++ = '7';
	  }
      if (attr.blink) {
	  if (ptr != &tmp[1])
	      *ptr++ = ';';
	  *ptr++ = '5';
	  }
      if (attr.brighter) {
	  if (ptr != &tmp[1])
	      *ptr++ = ';';
	  *ptr++ = '1';
	  }
      *ptr++ = 'm'; *ptr++ = '\0';
      BOOT1_putstr (tmp);
#endif
      }
  _BIOS_wait (SERIAL_DELAY);
  }}

awk_farret (SERIAL_setfgcolor);
VIDEO_FCT_PREFIX(SERIAL_setfgcolor) static unsigned
SERIAL_setfgcolor (unsigned color)
  {
  if (UI.monitor == 1) /* B&W */
      return 0;

  /* How to test validity of the color ? */
  UI.fgcolor = color;
  if (UI.monitor != 0) { /* DEC VT terminal */
      /* "\033[1m" for extra bright foreground */
      if (UI.fgcolor > 8)
	  BOOT1_printf ("\033[1;%um", 30 + UI.fgcolor - 8);
	else
	  BOOT1_printf ("\033[0;%um", 30 + UI.fgcolor);
      }
    else
      BOOT1_printf ("\033[%um", 30 + UI.fgcolor);
  _BIOS_wait (SERIAL_DELAY);

  return 0;
  }

awk_farret (SERIAL_setbgcolor);
VIDEO_FCT_PREFIX(SERIAL_setbgcolor) static unsigned
SERIAL_setbgcolor (unsigned color)
  {
  if (UI.monitor == 1) /* B&W */
      return 0;

  /* How to test validity of the color ? */
  UI.bgcolor = color;
  BOOT1_printf ("\033[%um", 40 + UI.bgcolor);
  _BIOS_wait (SERIAL_DELAY);
  return 0;
  }

VIDEO_FCT_PREFIX(SERIAL_setpixel) static void FASTCALL
SERIAL_setpixel (coord xy, unsigned color)
  {
  SDBG ((" %s called! ", __FUNCTION__));
  }

VIDEO_FCT_PREFIX(SERIAL_getpixel) static unsigned FASTCALL
SERIAL_getpixel (coord xy)
  {
  SDBG ((" %s called! ", __FUNCTION__));
  return 1;
  }

VIDEO_FCT_PREFIX(SERIAL_plotHline) static void
SERIAL_plotHline (coord xy, unsigned short xend, unsigned color)
  {
  SDBG ((" %s called! ", __FUNCTION__));
  }

awk_farret (SERIAL_getsize);
VIDEO_FCT_PREFIX(SERIAL_getsize) static unsigned
SERIAL_getsize (unsigned mode, struct video_parameter_str *param)
  {
  memset (param, 0, sizeof(*param));

  param->identification = mode;
  param->attr.isgraphic = 0;
  param->attr.graphicbgcolor = 0;

#ifdef MOUSE_ON_VT
  param->charwidth = 8;
  param->charheight = 16;
#endif

  if (mode & 1)
      param->nbcol = 132;
    else
      param->nbcol = 80;
  param->width = param->nbcol; /* mouse and change_mode() */

  if (mode >= 2 && UI.monitor == 0)
      return 0x100;	/* not a VT, but may accept 80/132 cols */

  switch (mode >> 1) {
    default:
    case 0: param->nbrow = 24; break;
    case 1: param->nbrow = 36; break;
    case 2: param->nbrow = 48; break;
    }
  param->height = param->nbrow; /* mouse and change_mode() */

  if (UI.monitor <= 1) /* B&W display */
      param->nbcolor = 0; /* with attributes */
    else
      param->nbcolor = 16; /* nbof (SERIAL_color) */
  return 0;
  }

VIDEO_FCT_PREFIX(SERIAL_probe) static unsigned
SERIAL_probe (void)
  {bound_stack();{
  unsigned char C, nb1, nb2, cpt, returned;

  UI.monitor = 0;

  STRINIT (UI.info.serial.terminalname, "VT000");
  UI.parameter.nbcol = 80;	// never to be null, see mouse.c::mouse_print_field()

  if (_BIOS_getserialstatus(BOOT1_COM_port()).receive_data_ready) {
      SDBG (("\r\nFlushing serial line: "));
      cpt = 30; /* max char flushed */
      while (--cpt) {
	  if (SERIAL_getc (&C) != 0)
	      break;
	  SDBG (("0x%X ", C));
	  }
      }

  SDBG (("\r\nSending Primary DA request to terminal, i.e. CSI c: "));
  BOOT1_putstr ("\033[c");

  returned = 0x10;
  if (SERIAL_getc (&C) != 0) {
      SDBG (("no answer.\r\n"));
      return returned;
      }
  returned++;
  if (C != '\033' && C != 0233 /* octal! */) {
      SDBG (("char received: 0x%X, abort.\r\n", C));
      return returned;
      }
  returned++;
  if (C != 0233 /* octal! */) {
      if (SERIAL_getc (&C) != 0) {
	  SDBG (("no answer after ESC.\r\n"));
	  return returned;
	  }
      if (C != '[') {
	  SDBG (("no '[' after ESC.\r\n"));
	  return returned;
	  }
      }

  returned++;
  if (SERIAL_getc (&C) != 0) {
      SDBG (("nothing after 'ESC ['\r\n"));
      return returned;
      }
  returned++;
  if (C != '?') {
      SDBG (("not '?' after 'ESC [': 0x%X\r\n", C));
      return returned;
      }

  returned++;
  if (SERIAL_getc (&C) != 0) {
      SDBG (("nothing after 'ESC [ ?'\r\n"));
      return returned;
      }
  returned++;
  if (C < '0' || C > '9') {
      SDBG (("not a digit after 'ESC [ ?': 0x%X\r\n", C));
      return returned;
      }
  nb1 = C - '0';

  returned++;
  if (SERIAL_getc (&C) != 0) {
      SDBG (("nothing after 'ESC [ ? digit1'\r\n"));
      return returned;
      }
  returned++;
  if ((C < '0' || C > '9') && (C != ';') && (C != 'c')) {
      SDBG (("no digit, nor ';' or 'c' after 'ESC [ ? digit1': 0x%X\r\n", C));
      return returned;
      }
  if (C >= '0' && C <= '9')
      nb1 = 10 * nb1 + C - '0';

  switch (nb1) {
    case 1 :
      SDBG (("VT100.\r\n"));
      UI.info.serial.terminalname[2] = '1';
      break;
    case 6 :
      SDBG (("VT102.\r\n"));
      UI.info.serial.terminalname[2] = '1';
      UI.info.serial.terminalname[4] = '2';
      break;
    case 62:
      SDBG (("VT2*0.\r\n"));
      UI.info.serial.terminalname[2] = '2';
      break;
    case 63:
      SDBG (("VT3*0.\r\n"));
      UI.info.serial.terminalname[2] = '3';
      break;
    case 64:
      SDBG (("VT4*0.\r\n"));
      UI.info.serial.terminalname[2] = '4';
      break;
    case 65:
      SDBG (("VT5*0.\r\n"));
      UI.info.serial.terminalname[2] = '5';
      break;
    default:
      SDBG (("unknown: %u.\r\n", nb1));
      break;
    }

  while (C != 'c' && SERIAL_getc (&C) == 0) {
      SDBG ((" flushing char 0x%X", C));
      }

  SDBG (("\r\nSending Secondary DA request to terminal, i.e. CSI > c: "));
  BOOT1_putstr ("\033[>c");

  returned = 0x20;
  if (SERIAL_getc (&C) != 0) {
      SDBG (("no answer.\r\n"));
      return returned;
      }
  returned++;
  if (C != '\033' && C != 0233 /* octal! */) {
      SDBG (("char received: 0x%X, abort.\r\n", C));
      return returned;
      }
  returned++;
  if (C != 0233 /* octal! */) {
      if (SERIAL_getc (&C) != 0) {
	  SDBG (("no answer after ESC.\r\n"));
	  return returned;
	  }
      if (C != '[') {
	  SDBG (("no '[' after ESC.\r\n"));
	  return returned;
	  }
      }

  returned++;
  if (SERIAL_getc (&C) != 0) {
      SDBG (("nothing after 'ESC ['\r\n"));
      return returned;
      }
  returned++;
  if (C != '>') {
      SDBG (("not '>' after 'ESC [': 0x%X\r\n", C));
      return returned;
      }

  returned++;
  if (SERIAL_getc (&C) != 0) {
      SDBG (("nothing after 'ESC [ >'\r\n"));
      return returned;
      }
  returned++;
  if (C < '0' || C > '9') {
      SDBG (("not a digit after 'ESC [ >': 0x%X\r\n", C));
      return returned;
      }
  nb2 = C - '0';

  returned++;
  if (SERIAL_getc (&C) != 0) {
      SDBG (("nothing after 'ESC [ > digit1'\r\n"));
      return returned;
      }
  returned++;
  if ((C < '0' || C > '9') && (C != ';') && (C != 'c')) {
      SDBG (("no digit, nor ';' or 'c' after 'ESC [ > digit1': 0x%X\r\n", C));
      return returned;
      }
  if (C >= '0' && C <= '9')
      nb2 = 10 * nb2 + C - '0';

  switch (nb2) {
    case 24:
    case 41: /* B & W display */
      SDBG (("VT*20 (%u).\r\n", nb2));
      UI.monitor = 1;
      if (UI.info.serial.terminalname[2] != '1')
	  UI.info.serial.terminalname[3] = '2';
      break;
    case 18: /* grayscale display */
      SDBG (("VT*30.\r\n"));
      UI.monitor = 2;
      UI.info.serial.terminalname[3] = '3';
      break;
    case 19: /* color display */
      SDBG (("VT*40.\r\n"));
      UI.monitor = 3;
      UI.info.serial.terminalname[3] = '4';
      break;
    default:
      SDBG (("unknown: %u.\r\n", nb2));
      break;
    }

  while (C != 'c' && SERIAL_getc (&C) == 0) {
      SDBG ((" flushing char 0x%X", C));
      }
  returned = 0;

  {
  /* Set the two default mode, hope that is 80x24 and 132x24 */
  struct videomode_str mode = {
	.number	= 0, .width	= 80, .height	= 24,
	.bpp	= 0, .text	= 1,  .attr	= 1
	};

  if (UI.mode == 0 && UI.nbmode != 0)
      VDBG (("ERROR: UI.mode == 0 with UI.nbmode = %u\r\n", UI.nbmode));
  while (UI.nbmode != 0)
      remove_valid_mode (UI.mode[0].number);

  if (nb2 == 18 || nb2 == 19)
      mode.bpp = 4;
  add_valid_mode (&mode);
  mode.number = 1;
  mode.width = 132;
  add_valid_mode (&mode);

  /* everything passed, we can enable extended modes: */
  if (nb1 >= 64) {	/* VT400+ family */
      mode.number = 2;
      mode.width = 80;
      mode.height = 36;
      add_valid_mode (&mode);
      mode.number = 3;
      mode.width = 132;
      add_valid_mode (&mode);
      mode.number = 4;
      mode.width = 80;
      mode.height = 48;
      add_valid_mode (&mode);
      mode.number = 5;
      mode.height = 48;
      mode.width = 132;
      add_valid_mode (&mode);
      UI.info.serial.max_mode = 5;
      }
    else
      UI.info.serial.max_mode = 1;
  }

  return returned;
  }}

awk_farret (SERIAL_setmode);
VIDEO_FCT_PREFIX(SERIAL_setmode) static unsigned
SERIAL_setmode (unsigned mode)
  {bound_stack();{
  struct video_parameter_str param;
  unsigned lines;
  unsigned saved_mode = mode, tmp;

  mode &= ~0x8000;

  if (mode > UI.info.serial.max_mode) {
      SDBG (("%s: mode 0x%X cannot be supported by this terminal (max %u)\r\n",
	    __FUNCTION__, mode, UI.info.serial.max_mode));
      return 1;
      }

  if ((tmp = SERIAL_getsize (mode, &param)) != 0)
      return tmp;
  UI.parameter = param;
  UI.parameter.attr.ansi_font = 1;
  SERIAL_setup_color ();
#if (USER_SUPPORT & VESA_SUPPORT)
  UI.parameter.fontnbchar = 256;
#endif

  if (mode & 1)
      BOOT1_putstr ("\033[?3h");	/* 132 column */
    else
      BOOT1_putstr ("\033[?3l");	/* 80 column */
  _BIOS_wait (SERIAL_DELAY);

  switch (mode >> 1) {
    case 0: lines = 24; break;
    case 1: lines = 36; break;
    case 2: lines = 48; break;
      break;
    default:
      return 3;
    }

  BOOT1_printf ("\033[%ut", lines);	/* X lines/page */
  _BIOS_wait (SERIAL_DELAY);
  BOOT1_printf ("\033[%u*|", lines);	/* X lines/screen */
  _BIOS_wait (SERIAL_DELAY);
  BOOT1_printf ("\033[1;%ur", lines);	/* scrolling region 1..X */
  _BIOS_wait (SERIAL_DELAY);

  UI.attributes.valid = egatxt_attribute;
  UI.attributes.current = no_attribute;

  if ((saved_mode & 0x8000) == 0)
      UI.function.clearscreen ();
  return 0;
  }}

VIDEO_FCT_PREFIX(SERIAL_init) unsigned
SERIAL_init (void)
  {
  extern unsigned (*fptr_SERIAL_getsize) (unsigned mode, struct video_parameter_str *param);
  asm ("# function %0 used " : : "g" (SERIAL_getsize));
  extern unsigned (*fptr_SERIAL_setmode) (unsigned mode);
  asm ("# function %0 used " : : "g" (SERIAL_setmode));
  extern unsigned (*fptr_SERIAL_getmode) (void);
  asm ("# function %0 used " : : "g" (SERIAL_getmode));
  UI.function.getsize = *fptr_SERIAL_getsize;
  UI.function.setmode = *fptr_SERIAL_setmode;
  UI.function.getmode = *fptr_SERIAL_getmode;

  extern void (*fptr_SERIAL_clearscreen) (void);
  asm ("# function %0 used " : : "g" (SERIAL_clearscreen));
  extern unsigned (*fptr_SERIAL_setcursor) (unsigned char row, unsigned char col);
  asm ("# function %0 used " : : "g" (SERIAL_setcursor));
  extern unsigned (*fptr_SERIAL_getcursor) (unsigned char *row, unsigned char *col);
  asm ("# function %0 used " : : "g" (SERIAL_getcursor));
  extern unsigned (*fptr_SERIAL_setfgcolor) (unsigned color);
  asm ("# function %0 used " : : "g" (SERIAL_setfgcolor));
  extern unsigned (*fptr_SERIAL_setbgcolor) (unsigned color);
  asm ("# function %0 used " : : "g" (SERIAL_setbgcolor));
  extern void (*fptr_SERIAL_setattribute) (struct attribute_str attr);
  asm ("# function %0 used " : : "g" (SERIAL_setattribute));
  extern void (*fptr_SERIAL_putstr) (const char *str);
  asm ("# function %0 used " : : "g" (SERIAL_putstr));
  extern unsigned (*fptr_SERIAL_getkey) (unsigned timeout);
  asm ("# function %0 used " : : "g" (SERIAL_getkey));

  if (SERIAL_probe() == 0) { /* all requests answered */
      // UI.function.getsize, UI.function.setmode, UI.function.getmode : already initialised
      // init local functions without far-call convention:
      UI.function.setpixel = SERIAL_setpixel;
      UI.function.getpixel = SERIAL_getpixel;
      UI.function.plotHline = SERIAL_plotHline;
      // init functions with far-call convention:
      UI.function.clearscreen = *fptr_SERIAL_clearscreen;
      UI.function.setcursor = *fptr_SERIAL_setcursor;
      UI.function.getcursor = *fptr_SERIAL_getcursor;
      UI.function.setfgcolor = *fptr_SERIAL_setfgcolor;
      UI.function.setbgcolor = *fptr_SERIAL_setbgcolor;
      UI.function.setattribute = *fptr_SERIAL_setattribute;
      UI.function.putstr = *fptr_SERIAL_putstr;
      UI.function.getkey = *fptr_SERIAL_getkey;

      if (SERIAL_getsize (SERIAL_getmode(), &UI.parameter) != 0)
	  SDBG (("Initial SERIAL_getsize failed!\r\n"));
      UI.attributes.valid = egatxt_attribute;
      UI.attributes.current = no_attribute;
      if (UI.monitor == 1)	/* B&W display - i.e. green/black */
	  UI.fgcolor = 1;
      return 0;
      }
    else {
      unsigned char saverow, savecol, row, col;

      if (SERIAL_getcursor (&saverow, &savecol) != 0) {
	  SDBG (("SERIAL_probe and SERIAL_getcursor failed!\r\n"));
	  /* We should still initialise UI.function.getkey() else we will crash... */
	  // UI.function.getsize, UI.function.setmode, UI.function.getmode : already initialised
	  // init local functions without far-call convention:
	  UI.function.setpixel = SERIAL_setpixel;
	  UI.function.getpixel = SERIAL_getpixel;
	  UI.function.plotHline = SERIAL_plotHline;
	  // init functions with far-call convention:
	  UI.function.clearscreen = *fptr_SERIAL_clearscreen;
	  UI.function.setcursor = *fptr_SERIAL_setcursor;
	  UI.function.getcursor = *fptr_SERIAL_getcursor;
	  UI.function.setfgcolor = *fptr_SERIAL_setfgcolor;
	  UI.function.setbgcolor = *fptr_SERIAL_setbgcolor;
	  UI.function.setattribute = *fptr_SERIAL_setattribute;
	  UI.function.putstr = *fptr_SERIAL_putstr;
	  UI.function.getkey = *fptr_SERIAL_getkey;
	  return 1;
	  }
      /* try this... */
      SERIAL_setcursor (99, 255);
      if (SERIAL_getcursor (&row, &col) == 0) {
	  SDBG (("SERIAL_probe failed, using SERIAL_getcursor: %ux%u\r\n", col, row));
	  if (col != savecol)
	      UI.parameter.width = UI.parameter.nbcol = col; /* mouse and change_mode() */
	  if (row != saverow)
	      UI.parameter.height = UI.parameter.nbrow = row; /* mouse and change_mode() */
	  // UI.function.getsize, UI.function.setmode, UI.function.getmode : already initialised
	  // init local functions without far-call convention:
	  UI.function.setpixel = SERIAL_setpixel;
	  UI.function.getpixel = SERIAL_getpixel;
	  UI.function.plotHline = SERIAL_plotHline;
	  // init functions with far-call convention:
	  UI.function.clearscreen = *fptr_SERIAL_clearscreen;
	  UI.function.setcursor = *fptr_SERIAL_setcursor;
	  UI.function.getcursor = *fptr_SERIAL_getcursor;
	  UI.function.setfgcolor = *fptr_SERIAL_setfgcolor;
	  UI.function.setbgcolor = *fptr_SERIAL_setbgcolor;
	  UI.function.setattribute = *fptr_SERIAL_setattribute;
	  UI.function.putstr = *fptr_SERIAL_putstr;
	  UI.function.getkey = *fptr_SERIAL_getkey;

	  UI.attributes.valid = egatxt_attribute;
	  UI.attributes.current = no_attribute;
	  UI.parameter.row = saverow;
	  UI.parameter.col = savecol;
	  UI.parameter.height = UI.parameter.nbrow = row;
	  UI.parameter.width = UI.parameter.nbcol = col;
	  }
      SERIAL_setcursor (saverow, savecol);
      return 0;
      }
  }
#endif /* SERIAL_SUPPORT */

awk_farret (exported_setpixel);
VIDEO_FCT_PREFIX(exported_setpixel) FASTCALL void
exported_setpixel (coord xy, unsigned color)
  {
  UI.function.setpixel (xy, color);
  }

awk_farret (exported_getpixel);
VIDEO_FCT_PREFIX(exported_getpixel) FASTCALL unsigned
exported_getpixel (coord xy)
  {
  return UI.function.getpixel (xy);
  }

awk_farret (exported_plotHline);
VIDEO_FCT_PREFIX(exported_plotHline) void
exported_plotHline (coord xy, unsigned short xend, unsigned color)
  {
  UI.function.plotHline (xy, xend, color);
  }

awk_farret (exported_VESA_color);
VIDEO_FCT_PREFIX(exported_VESA_color) unsigned __attribute__ ((const))
exported_VESA_color (unsigned char _red, unsigned char _green, unsigned char _blue)
  {
#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
  return VESA_color (_red, _green, _blue);
#else
  return 0xFFFFFFFF;
#endif
  }

struct exported_togpl_s exported_togpl;

VIDEO_FCT_PREFIX(fill_exported_togpl) inline void
fill_exported_togpl (unsigned serial)
  {
  extern void     (*fptr_draw_bg_box) (unsigned baserow, unsigned height, unsigned color);
  extern unsigned (*fptr_final_loadrun) (unsigned index, struct registers *regs, struct gujin_param_attrib gujin_attr);
  extern int      (*fptr_printf) (const char *format, ...);
  extern void     (*fptr_draw_bg_box) (unsigned baserow, unsigned height, unsigned color);
  extern unsigned (*fptr_final_loadrun) (unsigned index, struct registers *regs, struct gujin_param_attrib gujin_attr);
  extern unsigned (*fptr_get_number) (unsigned deflt, unsigned max);
  extern void     (*fptr_exported_setpixel) (coord xy, unsigned color) FASTCALL;
  extern unsigned (*fptr_exported_getpixel) (coord xy) FASTCALL;
  extern void     (*fptr_exported_plotHline) (coord xy, unsigned short xend, unsigned color);
  extern unsigned __attribute__ ((const)) (*fptr_exported_VESA_color) (unsigned char _red, unsigned char _green, unsigned char _blue);
  extern unsigned (*fptr_get_line) (char *buffer, unsigned maxlen, char display, unsigned char lowlimit, unsigned char highlimit); /* Gujin-1.6+ */

  exported_togpl = (struct exported_togpl_s) {
    .printf = *fptr_printf,
    .draw_bg_box = *fptr_draw_bg_box,
    .final_loadrun = *fptr_final_loadrun,
    .get_number = *fptr_get_number,
    .get_line = *fptr_get_line,
    .setpixel = *fptr_exported_setpixel,
    .getpixel = *fptr_exported_getpixel,
    .plotHline = *fptr_exported_plotHline,
    .VESA_color = serial? 0 : *fptr_exported_VESA_color,
    };
  }

/* At initialisation, we still did not probe anything... */
awk_farret (BOOT1_pututf8);
VIDEO_FCT_PREFIX(BOOT1_pututf8) void
BOOT1_pututf8 (const char *str)
  {
  char tmpbuff[strlen(str) + 1], *dst = tmpbuff; /* UTF-8 will reduce that string */
  while (*str) {
      UI.parameter.attr.has_outputed = 1;
      unsigned utfchar, charlen = UTF8LEN(str);	/* handle UTF-8 subset */
      if (charlen == 2 && (utfchar = UTF8VAL2(str)) < 256) {
	  *dst++ = utfchar;
	  str += charlen;
	  }
	else if (charlen >= 2) {
	  *dst++ = '?';
	  str += charlen;
	  }
	else
	  *dst++ = *str++;
      }
  *dst = '\0';
  if (FIRST_BOOT1_COM_port() >= 0)
      BOOT1_putstr (tmpbuff);
    else
      BOOT1_ansi_putstr (tmpbuff);
  }

/*
 * Init a basic generic system which should work everywhere:
 */
VIDEO_FCT_PREFIX(UI_init) void
UI_init (unsigned serial)
  {
  unsigned i;

  memset (&UI, 0, sizeof(UI));

  UI.fgcolor = 7;
  for (i = 0; i < nbof (UI.stdcolor); i++)
      UI.stdcolor[i] = i;
  UI.parameter.nbcolor = 16;

  for (i = 0; i < sizeof (UI.function) / sizeof (void *); i++)
/* GCC-4.2: warning: dereferencing type-punned pointer will break strict-aliasing rules
      ((void **)&UI.function) [i] = (void *) COMMON_error; */
      ((unsigned *)(void *)&UI.function) [i] = (unsigned) COMMON_error;

  extern unsigned (*fptr_DEFLT_setfgcolor) (unsigned color);
  asm ("# function %0 used " : : "g" (DEFLT_setfgcolor));
  extern unsigned (*fptr_COMMON_setbgcolor) (unsigned color);
  asm ("# function %0 used " : : "g" (COMMON_setbgcolor));
  extern unsigned (*fptr_COMMON_setcursor) (unsigned char row, unsigned char col);
  asm ("# function %0 used " : : "g" (COMMON_setcursor));
  extern unsigned (*fptr_COMMON_getcursor) (unsigned char *row, unsigned char *col);
  asm ("# function %0 used " : : "g" (COMMON_getcursor));
  UI.function.setfgcolor = *fptr_DEFLT_setfgcolor; /* VGA: only graphic modes */
  UI.function.setbgcolor = *fptr_COMMON_setbgcolor;
  UI.function.setcursor = *fptr_COMMON_setcursor;
//  UI.function.getcursor = *fptr_COMMON_getcursor;
  extern void (*fptr_BOOT1_pututf8) (const char *str);
  UI.function.putstr = fptr_BOOT1_pututf8;

  fill_exported_togpl (serial);

  if (serial) {
#if !(USER_SUPPORT & SERIAL_SUPPORT)
      DBG (("WARNING: %s SUPPORT NOT COMPILED IN!\r\n", "SERIAL"));
      printf ("WARNING: %s SUPPORT NOT COMPILED IN!\r\n", "SERIAL");
#endif
      }
    else {
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
      DBG (("sizeof MOUSE: %u, ", sizeof (MOUSE)));
#endif
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
      {
      VESA_VbeInfoBlock vesainfo;
      const char signature[4] = { 'V', 'E', 'S', 'A' };
      farptr OemStringPtr;

      DBG (("sizeof UI: %u, Initial VESA name check: ", sizeof(UI)));
      *CAST (unsigned *, vesainfo.VbeSignature) = *CAST (unsigned *, signature);
      if (   _VESA_getinfo (&vesainfo) == 0
	  && *CAST (unsigned *, vesainfo.VbeSignature) == *CAST (unsigned *, signature))
	  OemStringPtr = vesainfo.OemStringPtr;
	else {
	  OemStringPtr = stack_adr ("");
	  DBG (("[No VESA signature] "));
	  }

      if (VIDEO_mode_checkname (OemStringPtr) != 0) {
	  VIDEO_mode_initname (OemStringPtr);
	  DBG (("Video card changed, reset VGA parameter.\r\n"));
	  VIDEO_mode_reset();
	  PUTS (VIDEO_CARD_CHANGED);
	  }
	else
	  DBG (("same VESA card.\r\n"));
      }
#endif
#if !(USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT))
      DBG (("WARNING: %s SUPPORT NOT COMPILED IN!\r\n", "VGA/VESA"));
      printf ("WARNING: %s SUPPORT NOT COMPILED IN!\r\n", "VGA/VESA");
#endif
      }
  }

/* Try to keep following functions in the same segments to do short calls: */
VIDEO_FCT_PREFIX(draw_box) static inline void
draw_box (unsigned baserow, unsigned height)
  {
  if (UI.parameter.attr.graphicbgcolor) {
      unsigned short xend = UI.parameter.width - UI.parameter.charheight/2;
      coord xy = {.x = UI.parameter.charheight / 2,
		  .y = baserow * UI.parameter.charheight
			 - UI.parameter.charheight/2};

      if (baserow == 1) {
	  /* The top, the half line over the first line: */
	  coord xytmp = { .x = 0, .y = 0 };
	  while (xytmp.y < UI.parameter.charheight / 2) {
	      UI_plotHline (xytmp, UI.parameter.width,
					UI.stdcolor[black]);
	      xytmp.y++;
	      }
	  }
      do {
	  coord xytmp = { .x = 0, .y = xy.y };
	  UI_plotHline (xytmp, xy.x, UI.stdcolor[black]);
	  UI_plotHline (xy, xend, UI.bgcolor);
	  xytmp.x = xend;
	  UI_plotHline (xytmp, UI.parameter.width,
					UI.stdcolor[black]);
	  xy.x--; xy.y++; xend++;
	  } while (xy.x != 0);
      if (VESA_ACTIVE()) {
	  height *= UI.parameter.charheight;
	  if (UI.parameter.nbcolor == 16)
	      height --; /* one black line as separation */
	  do {
	      UI_plotHline (xy, xend, UI.bgcolor);
	      xy.y++;
	      } while (--height);
	  }
	else {
	  /* A lot faster in VGA graphic, here only VGA 256 colors */
	  xy.y += height * UI.parameter.charheight;
	  UI.function.setcursor (baserow, 0);
	  while (height--)
	      puts ("");
	  }
      do {
	  coord xytmp = { .x = 0, .y = xy.y };
	  UI_plotHline (xytmp, xy.x, UI.stdcolor[black]);
	  UI_plotHline (xy, xend, UI.bgcolor);
	  xytmp.x = xend;
	  UI_plotHline (xytmp, UI.parameter.width,
					UI.stdcolor[black]);
	  xy.x++; xy.y++; xend--;
	  } while (   xy.x != UI.parameter.charheight / 2
		   && xy.y < UI.parameter.height);
      if (UI.parameter.nbcolor == 16 && xy.y < UI.parameter.height) {
	  /* one black line as separation */
	  coord xytmp = { .x = 0, .y = xy.y };
	  UI_plotHline (xytmp, UI.parameter.width,
					UI.stdcolor[black]);
	  }
      }
    else {
      unsigned short i;

      if (baserow == 1) {
	  baserow = 0;
	  height++;
	  }
      if (baserow + height < UI.parameter.height) {
	  i = UI.parameter.width;
	  if (UI.parameter.attr.isgraphic)
	      i /= UI.parameter.charwidth;
	  height++;
	  }
	else
	  i = 0;
      UI.function.setcursor (baserow, 0);
      while (--height)
	  puts ("");
      while (i--)
	  print ("-");
      }
  }

VIDEO_FCT_PREFIX(draw_bg_box) void
draw_bg_box (unsigned baserow, unsigned height, unsigned color)
  {
#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
  if (UI.parameter.attr.graphicbgcolor && UI.parameter.nbcolor >= 256)
      UI.function.setbgcolor (color);
    /* else the grey color (due to the mode initialisation) is used */
#else
#define FIRST_LINES_BACKGROUND	0xFF000001
#define TOP_MENU_BACKGROUND	0xFF000002
#define BOTTOM_MENU_BACKGROUND	0xFF000003
#define ADVERT_BACKGROUND	0xFF000004
#define WORK_ADVERT_BACKGROUND	0xFF000005
#endif
  draw_box (baserow, height);
  }

VIDEO_FCT_PREFIX(show_ladders) void
show_ladders (unsigned bgcolor)
  {
#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
  unsigned short endy = UI.parameter.nbrow - 1;
  coord xy;

  if (UI.parameter.attr.isgraphic) {
      endy *= UI.parameter.charheight;
      xy.y = endy - UI.parameter.charheight / 2;

      for (xy.x = 0; xy.y < endy; xy.y++)
	  UI_plotHline (xy, UI.parameter.width, bgcolor);

      if (UI.parameter.nbcolor > 16) {
	  const unsigned dacsize = 6; /* 6 bit DAC */
	  unsigned char cpt = (1 << dacsize) - 1;
	  unsigned color[3];
	  unsigned endx = UI.parameter.width - 32 * UI.parameter.charwidth;
	  unsigned short increment = (endx / (1 << dacsize))
					? (endx / (1 << dacsize)) : 1;
	  unsigned char i, j = DIVROUNDUP (UI.parameter.charheight, 3);

	  do {
	      color[0] = VESA_color (cpt,   0,   0);
	      color[1] = VESA_color (0,   cpt,   0);
	      color[2] = VESA_color (0,     0, cpt);
	      for (i = UI.parameter.charheight; i != 0; i--, xy.y++)
		  UI_plotHline (xy, xy.x + increment, color[i / j]);
	      xy.x += increment;
	      xy.y = endy;
	      } while (cpt-- != 0);
	  for (i = UI.parameter.charheight; i-- > 0; xy.y++)
	      UI_plotHline (xy, UI.parameter.width, bgcolor);
	  }
	else if (VGA_ACTIVE()) { /* a lot faster then */
	  xy.y += UI.parameter.charheight;
	  UI.function.setcursor (endy/UI.parameter.charheight, 0);
	  print ("\r");
	  }
      }
    else {
      xy.y = endy;
      bgcolor <<= 8;
      bgcolor |= ' ';
      }

  for (xy.x = 0; xy.y < UI.parameter.height; xy.y++)
      UI_plotHline(xy, UI.parameter.width, bgcolor);
#endif /* VGA_SUPPORT | VESA_SUPPORT */
  }

/*
 * This is in ".data" or ".common/.bss" segment.
 *
 * It is NOT malloced, to have a linker defined address
 * and so produce faster call than the indirect access:
 * In the assembly file, a field of UI is accessed by:
 *   mov (UI+10),%%eax
 * If it were malloced, it would be:
 *   mov UI,%%eax
 *   mov 10(%%eax),%%eax
 */
struct user_interface_str UI;

#endif /* USER_SUPPORT != 0 */
