#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/fs.h>
#include <linux/hdreg.h>
#include <linux/cdrom.h>
#include <linux/major.h>
#include <linux/raid/md_u.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>

typedef struct {
    unsigned char	jmpinstruction[2];
    unsigned char	EBIOSaccess;	/* 0x90 = nop if BIOS only; 0x0E = Win95_FAT16LBA if EBIOS accessible */
    unsigned char	String[8];	/* blank padded ! */
    unsigned short	Bytepersector;
    unsigned char	Sectorpercluster;
    unsigned short	Reservedsector;	/* nb sectors before FAT */
    unsigned char	NbFAT;
    unsigned short	NbRootdirentry;	/* = 0 for FAT32 */
    unsigned short	NbTotalsector;	/* = 0 if >= 0x10000 or FAT32 */
    unsigned char	Mediadescriptor; /* Some people use 0xF8 for non removeable, 0xF0 for removeable... */
    unsigned short	NbSectorperFAT;	/* = 0 for FAT32 */
    /* DOS 3+ : */
    unsigned short	NbSectorpertrack;
    unsigned short	NbHead;
    unsigned		NbHiddensector;	/* nb of sectors before this partition */
    /* DOS 4+ : */
    unsigned		NbTotalsector2;
    } __attribute__ ((packed, aligned(1))) FAT_bootsect1_t;

typedef struct {
    unsigned		NbSectorperFAT2;
    struct FAT32_flags_str {
	unsigned short	active_fat	: 4;
	unsigned short	reserved1	: 3;
	unsigned short	one_fat_active	: 1;
	unsigned short	reserved2	: 8;
	} __attribute__ ((packed)) flags;
    unsigned short	version;
    unsigned		root_cluster; /* usually 2 */
    unsigned short	FSinfosector; /* below Reservedsector */
    unsigned short	save_bootrecord; /* usually 6 if != 0 */
    unsigned char	reserved[12];	/* fill with 0 */
    } __attribute__ ((packed, aligned(1))) FAT32_bootsect_t;

typedef struct {
    unsigned char	PhysicaldriveNb;
    unsigned char	dirty_flag	: 1;	/* need checkdisk at next boot */
    unsigned char	surface_scan	: 1;	/* checkdisk shall also do surface scan */
    unsigned char	FATreserved	: 6;	/* was current head, head for the boot sector */
    unsigned char	Signaturebyte0x29;	/* can also be 0x28 for Window NT */
    unsigned		VolumeserialNb;
    unsigned char	Volumelabel[11];	/* blank padded ! */
    unsigned char	FileSysName[8];		/* blank padded ! */
    } __attribute__ ((packed, aligned(1))) FAT_bootsect2_t;

typedef struct {
    FAT_bootsect1_t	part1;
    FAT_bootsect2_t	part2;
    } __attribute__ ((packed, aligned(1))) bootbefore_t;

typedef struct {
    FAT_bootsect1_t	part1;
    FAT32_bootsect_t	partFAT32;
    FAT_bootsect2_t	part2;
    } __attribute__ ((packed, aligned(1))) bootbefore_FAT32_t;

/*
 * The "diskread" command structure for disk interface:
 */
typedef struct {
    struct {
	unsigned short next;
	unsigned short checksum;	/* sum of sector + checksum = 0 */
	unsigned short nbword;		/* size to calculate checksum */
	} __attribute__ ((packed)) header;

    union diskcmd {
	struct diskcmd_bios {	/* 12 bytes, used if reg_ah != 0xF0 */
	    unsigned short	reg_dx;
	    unsigned short	reg_ax;
	    unsigned short	reg_cx;
	    /* no need to initialise the following, just a remainder: */
	    /* but checksum of boot1 is recalculated if first attempt failed! */
	    unsigned short	dummy;
	    unsigned long	will_be_set_to_esbx;
	    } __attribute__ ((packed)) int1302;
	struct diskcmd_ebios {	/* 20 bytes, used (logically) if reg_ah = 0x42, 0x43... */
	    unsigned short	reg_dx;
	    unsigned short	reg_ax;
	    unsigned char	cst_16;
	    unsigned char	cst_0;		/* has to be zero */
	    unsigned char	nb_block;	/* max 127 */
	    unsigned char	cst2_0;		/* has to be zero */
	    unsigned short	offset, segment;
	    unsigned long long	lba;
	    } __attribute__ ((packed)) int1342;
	struct diskcmd_hard {	/* 20 bytes, used if reg_ah == 0xF0 */
	    unsigned short	ide_dcr_address; /* usually base_ide+0x206 */
	    unsigned short	reg_ax;		/* %al has to be 0x0A */
	    unsigned short	base_ide_plus7;	/* added by 7, i.e. 1F7, 3F7 */
	    unsigned char	lba_head;
	    unsigned char	lba48_cylinder_high; /* lba48_* ignored if lba28 */
	    unsigned char	lba48_cylinder_low;
	    unsigned char	lba48_sector;
	    unsigned char	lba48_nb_sect;
	    unsigned char	cylinder_high;
	    unsigned char	cylinder_low;
	    unsigned char	sector;
	    unsigned char	nb_sect;
	    unsigned char       ide_command;    /* to read with retry, 0x20 */
	    unsigned short	nb_sect_read;	/* keep it equal to nb_sect */
	    unsigned short	sector_size_div2;
	    } __attribute__ ((packed)) hardide;
	} bootloader2_cmd;
    } __attribute__ ((packed, aligned(1))) bootloader2_t;

/*
 * The parameter block used by bootloader 1 to get
 * bootloader 2 in memory. Its size can not be modified !
 * Checksum calculus (also in instboot.c) is done on 16 bits,
 * so "boot1_checksum" has to be aligned like symbol "start",
 * and like symbol "bootend".
 */
typedef struct {
    unsigned short	boot1_checksum;
    bootloader2_t	bootloader2;
    } __attribute__ ((packed, aligned(2))) boot1param_t;

/*
 * To use the serial (BIOS) interface, you need hardware flow control,
 * i.e. either a true VT connected or a null-modem cable (DTR-> DCD, RI,
 * CTS, DSR), and fill in the following structure (serialconf) with
 * the configuration you need and the port in serialconf_port_plus_1.
 * Minicom works on another Linux box with a standard twisted cable.
 */
typedef struct {
    unsigned char	eightbit   :1;	/* else seven bits */
    unsigned char	constant_1 :1;
    unsigned char	doublestop :1;	/* else one stop bit */
    unsigned char	parity     :1;	/* else no parity generation/check */
    unsigned char	even       :1;	/* else odd */
    enum { B110 = 0, B150, B300, B600, B1200, B2400, B4800, B9600 }
			speed      :3;
    unsigned char	constant_0;	/* loaded in %ah before INT 0x14 */
    } __attribute__ ((packed, aligned(1))) serialconf_t;

/*
 * The boot sector contains this structure at its end:
 */

/*
 * The list of partition kind:
 */
#define DOS_FAT12_PARTITIONLIST() \
	P (DOS_FAT12,		0x1,	"FAT12")		\
	P (HiddenDOS_FAT12,	0x11,	"Hidden FAT12")

#define DOS_FAT16_PARTITIONLIST() \
	/* below 32Mb partition : */				\
	P (DOS_FAT16,		0x4,	"FAT16")		\
	P (HiddenDOS_FAT16,	0x14,	"Hidden FAT16")		\
	/* over 32Mb partition : */				\
	P (BIG_DOS_FAT16,	0x6,	"Big FAT16")		\
	P (HiddenBIG_DOS_FAT16,	0x16,	"Hidden Big FAT16")	\
	P (Win95_FAT16LBA,	0x0E,	"FAT16 LBA")		\
	P (HiddenWin95_FAT16LBA,0x1E,	"Hidden FAT16 LBA")

#define DOS_FAT32_PARTITIONLIST() \
	P (Win95_FAT32,		0x0B,	"FAT32")		\
	P (Win95_FAT32LBA,	0x0C,	"FAT32 LBA")		\
	P (HiddenWin95_FAT32,	0x1B,	"Hidden FAT32")		\
	P (HiddenWin95_FAT32LBA,0x1C,	"Hidden FAT32 LBA")

#define NTFS_PARTITIONLIST() \
	P (OS2_HPFS_NTFS,	0x7,	"NTFS/HPFS")		\
	P (Hidden_NTFS,		0x17,	"Hidden NTFS")

#define EXTENDED_PARTITIONLIST() \
	P (DOS_extended,	0x5,	"extended")		\
	P (Win95_extendedLBA,	0x0F,	"extended LBA")		\
	P (Linux_extended,	0x85,	"Linux extended")	\

#define PARTITIONLIST() \
	P (EMPTY,		0x0,	"empty")		\
	DOS_FAT12_PARTITIONLIST()				\
	DOS_FAT16_PARTITIONLIST()				\
	DOS_FAT32_PARTITIONLIST()				\
	NTFS_PARTITIONLIST()					\
	EXTENDED_PARTITIONLIST()				\
								\
	P (LDM,			0x42,	"LDM")			\
	P (GNU_HURD,		0x63,	"GNU HURD")		\
	P (MINIX,		0x81,	"MINIX")		\
	P (Linux_Swap,		0x82,	"Linux swap")		\
	P (Linux,		0x83,	"Linux")		\
	P (BSD_386,		0xA5,	"FreeBSD, BSD/386")	\
	P (OpenBSD,		0xA6,	"OpenBSD")		\
	P (NetBSD,		0xA9,	"NetBSD")		\
	P (BSDI_fs,		0xB7,	"BSDI fs")		\
	P (BSDI_swap,		0xB8,	"BSDI swap")		\
	P (Solaris_boot,	0xBE,	"Solaris boot")		\
	P (DOS_R_O,		0xE3,	"DOS R/O")		\
	P (DOS_secondary,	0xF2,	"DOS secondary")	\
	P (LinuxAutoRaid,	0xFD,	"Linux RAID autodetect")

enum system_indicator_enum {
#define P(shortname, value, longname) shortname = value,
	PARTITIONLIST()
#undef P /*(shortname, value, longname)*/
    };

typedef struct {
    unsigned long	WindowsNTmarker;
    unsigned short	Unknown;
    struct bootsect_partition_str {
	enum {non_bootable = 0, bootable = 0x80, invalid}
			indicator			:8;
	unsigned char	begin_head			:8;
	unsigned char	begin_sector			:6;
	unsigned char	begin_cylinder_msb		:2;
	unsigned char	begin_cylinder_lsb		:8;
//	unsigned short	begin_sector				   : 6;
//	unsigned short	begin_cylinder __attribute__((big_endian)) : 10;

	enum system_indicator_enum system_indicator	:8;
	unsigned char	end_head			:8;
	unsigned char	end_sector			:6;
	unsigned char	end_cylinder_msb		:2;
	unsigned char	end_cylinder_lsb		:8;
//	unsigned short	end_sector				 : 6;
//	unsigned short	end_cylinder __attribute__((big_endian)) : 10;

	unsigned long	nb_sector_before;
	unsigned long	nb_sector;
	} __attribute__ ((packed)) bootsect_partition[4];
    unsigned short	Signature0xAA55;
    } __attribute__ ((packed, aligned(1))) bootafter_t;


typedef struct {
    bootbefore_t	before;
    unsigned char	code_part[512 - sizeof(bootbefore_t)
					- sizeof(boot1param_t)
					- sizeof (serialconf_t)
					- sizeof(bootafter_t) ];
    boot1param_t	param;
    serialconf_t        serial_configuration;
    bootafter_t		after;
    } __attribute__ ((packed)) bootsector_t;

typedef struct {
    bootbefore_FAT32_t	before;
    unsigned char	code_part[512 - sizeof(bootbefore_FAT32_t)
					- sizeof(boot1param_t)
					- sizeof (serialconf_t)
					- sizeof(bootafter_t) ];
    boot1param_t	param;
    serialconf_t        serial_configuration;
    bootafter_t		after;
    } __attribute__ ((packed)) bootsector_FAT32_t;

typedef struct {
    unsigned checksum_start_adr;
    unsigned boot1param_adr;
    unsigned bootend_adr;
    unsigned uninstall_mbr_adr;
    unsigned bootchain_adr;
    unsigned bootchain_nbblock;
    unsigned gujin_param_adr;
    unsigned code_start_adr, patch2adr, patch3adr, patch4adr;
    unsigned sbss, edata, deltaseg, diskcodeseg, xdataseg;
    unsigned password_serial_adr, serialconf_port_plus_1_adr;
    unsigned read_disk_adr;
    unsigned partition_relative_read_disk_adr;
    unsigned read_disk_1st_adr;	/* CDROM EL TORITO */
    unsigned fscodeseg, loadcodeseg, usercodeseg;
    unsigned xstringseg;
    unsigned unused;
    } instboot_info_t;

int main (int argc, char **argv)
{
	int file_desc, cpt_partition;
	char *filename;
	union { bootsector_t bootFAT16; bootsector_FAT32_t bootFAT32;
#ifdef SHOW_OFFSET
		unsigned char upto_instboot_info[8196];
#endif
		} bootFAT;
	FAT_bootsect1_t	*part1 = 0;
	FAT_bootsect2_t	*part2 = 0;
	serialconf_t *serialconf = 0;
	boot1param_t *param = 0;

	if (argc != 2) {
		printf ("Utility to display the content of the boot block of a device\n");
		printf ("USAGE: %s <device>\n", argv[0]);
		return -1;
		}
	filename = argv[1];

	if ((file_desc = open (filename, O_RDONLY)) == -1) {
		perror (filename);
		return -2;
		}

	if (read (file_desc, &bootFAT, sizeof (bootFAT)) != sizeof (bootFAT)) {
		printf ("Cannot read %u bytes from the device '%s'!\n", (unsigned)sizeof (bootFAT), filename);
		perror ("fstat");
		if (close (file_desc) != 0)
			perror (filename);
		return -3;
		};

	if (close (file_desc) != 0)
		perror (filename);

	printf ("Boot record of size %u bytes read successfully from '%s'\n", (unsigned)sizeof (bootFAT), filename);
	printf ("First bytes 0x%X 0x%X 0x%X, signature 0x%X, FAT16 sig 0x%X, FAT32 sig 0x%X,\n",
		bootFAT.bootFAT16.before.part1.jmpinstruction[0],
		bootFAT.bootFAT16.before.part1.jmpinstruction[1],
		bootFAT.bootFAT16.before.part1.jmpinstruction[2],
		bootFAT.bootFAT16.after.Signature0xAA55,
		bootFAT.bootFAT16.before.part2.Signaturebyte0x29,
		bootFAT.bootFAT32.before.part2.Signaturebyte0x29);

	if (bootFAT.bootFAT16.before.part2.Signaturebyte0x29 == 0x29 || bootFAT.bootFAT16.before.part2.Signaturebyte0x29 == 0x28) {
		printf ("  MAX code size in boot record: %u bytes, EBIOS access %s.\n",
			(unsigned)sizeof(bootFAT.bootFAT16.code_part),
			bootFAT.bootFAT16.before.part1.EBIOSaccess == 0x0E ? "seems possible" : (bootFAT.bootFAT16.before.part1.EBIOSaccess == 0x90 ? "seems unsupported" : "unknown"));
		part1 = &bootFAT.bootFAT16.before.part1;
		part2 = &bootFAT.bootFAT16.before.part2;
		param = &bootFAT.bootFAT16.param;
		serialconf = (serialconf_t *)((unsigned long)&bootFAT.bootFAT16.param - 2);
	} else if (bootFAT.bootFAT32.before.part2.Signaturebyte0x29 == 0x29 || bootFAT.bootFAT32.before.part2.Signaturebyte0x29 == 0x28) {
		FAT32_bootsect_t *partFAT32 = &bootFAT.bootFAT32.before.partFAT32;
		unsigned cpt;
		printf ("  MAX code size in boot record: %u bytes, EBIOS access %s,\n",
			(unsigned)sizeof(bootFAT.bootFAT32.code_part),
			bootFAT.bootFAT32.before.part1.EBIOSaccess == 0x0E ? "seems possible" : (bootFAT.bootFAT32.before.part1.EBIOSaccess == 0x90 ? "seems unsupported" : "unknown"));
		printf ("  NbSectorperFAT2 %u, flags.active_fat %u, flags.reserved1 %u, flags.one_fat_active %u, flags.reserved2 %u\n",
			partFAT32->NbSectorperFAT2,
			partFAT32->flags.active_fat,
			partFAT32->flags.reserved1,
			partFAT32->flags.one_fat_active,
			partFAT32->flags.reserved2);
		printf ("  version %u, root_cluster %u, FSinfosector %u, save_bootrecord %u, reserved: ",
			partFAT32->version,
			partFAT32->root_cluster,
			partFAT32->FSinfosector,
			partFAT32->save_bootrecord);
		for (cpt = 0; cpt < sizeof(partFAT32->reserved); cpt++)
			printf ("%2.2X ", partFAT32->reserved[cpt]);
		printf (".\n");
		part1 = &bootFAT.bootFAT32.before.part1;
		part2 = &bootFAT.bootFAT32.before.part2;
		param = &bootFAT.bootFAT32.param;
		serialconf = (serialconf_t *)((unsigned long)&bootFAT.bootFAT16.param - 2);
	} else {
		printf ("  No FAT signature recognised, cannot analyse header.\n");
	}

	if (part1 != 0) {
		char String[9];
		unsigned cpt;
		for (cpt = 0; cpt < sizeof(part1->String); cpt++)
			String[cpt] = part1->String[cpt];
		String[8] = '\0';

		printf ("Mediadescriptor 0x%X%s, name '%s', %u bytes/sectors,\n",
			part1->Mediadescriptor,
			(part1->Mediadescriptor == 0xF8) ? " (non removeable?)" : ((part1->Mediadescriptor == 0xF0) ? " (removeable?)" : ""),
			String, part1->Bytepersector);
		printf ("  Reservedsector %u, NbHiddensector %u, NbTotalsector %u, NbTotalsector2 %u,\n",
			part1->Reservedsector, part1->NbHiddensector, part1->NbTotalsector, part1->NbTotalsector2);
		if (part1->NbHead != 0 && part1->NbSectorpertrack != 0)
			printf ("  %u heads, %u sector/track (so %u cylinders, rounded down),\n",
				part1->NbHead, part1->NbSectorpertrack,
				(part1->NbTotalsector? part1->NbTotalsector : part1->NbTotalsector2) / part1->NbHead / part1->NbSectorpertrack);
		else
			printf ("  %u heads, %u sector/track,\n", part1->NbHead, part1->NbSectorpertrack);
		printf ("  Sectorpercluster %u, NbFAT %u, NbRootdirentry %u, NbSectorperFAT %u.\n",
			part1->Sectorpercluster, part1->NbFAT, part1->NbRootdirentry, part1->NbSectorperFAT);
	}

	if (part2 != 0) {
		char Volumelabel[12];
		char FileSysName[9];
		unsigned cpt;
		printf ("PhysicaldriveNb 0x%X, VolumeserialNb 0x%X,\n", part2->PhysicaldriveNb, part2->VolumeserialNb);
		printf ("  dirty_flag %u, surface_scan %u, FATreserved 0x%X, Signaturebyte0x29 0x%X\n",
			part2->dirty_flag, part2->surface_scan, part2->FATreserved, part2->Signaturebyte0x29);
		for (cpt = 0; cpt < sizeof(part2->Volumelabel); cpt++)
			Volumelabel[cpt] = part2->Volumelabel[cpt];
		Volumelabel[11] = '\0';
		for (cpt = 0; cpt < sizeof(part2->FileSysName); cpt++)
			FileSysName[cpt] = part2->FileSysName[cpt];
		FileSysName[8] = '\0';
		printf ("  Volumelabel '%s', FileSysName '%s'\n", Volumelabel, FileSysName);
	}

	if (bootFAT.bootFAT16.before.part1.jmpinstruction[0] == 0xEB && bootFAT.bootFAT16.before.part1.jmpinstruction[1] == 0x6F) {
		printf ("Looks like that is a Gujin boot record, deeper analysis:\n");
		if (param != 0) {
			printf ("boot1_checksum 0x%X, header: (next 0x%X, checksum 0x%X, nbword %u)\n",
				param->boot1_checksum, param->bootloader2.header.next,
				param->bootloader2.header.checksum, param->bootloader2.header.nbword);
			if ((param->bootloader2.bootloader2_cmd.int1302.reg_ax & 0xF000) == 0) {
				printf ("Read chain with BIOS: ax 0x%X, dx 0x%X, cx 0x%X, dummy 0x%X, EsBx 0x%X\n",
					param->bootloader2.bootloader2_cmd.int1302.reg_ax,
					param->bootloader2.bootloader2_cmd.int1302.reg_dx,
					param->bootloader2.bootloader2_cmd.int1302.reg_cx,
					param->bootloader2.bootloader2_cmd.int1302.dummy,
					(unsigned)param->bootloader2.bootloader2_cmd.int1302.will_be_set_to_esbx);
			} else if ((param->bootloader2.bootloader2_cmd.int1302.reg_ax & 0xF000) == 0x4000) {
				printf ("Read chain with EBIOS: ax 0x%X, dx 0x%X, cst_16 %u, cst_0 %u, nb_block %u, cst2_0 %u, offset 0x%X, segment 0x%X, lba %llu\n",
					param->bootloader2.bootloader2_cmd.int1342.reg_ax,
					param->bootloader2.bootloader2_cmd.int1342.reg_dx,
					param->bootloader2.bootloader2_cmd.int1342.cst_16,
					param->bootloader2.bootloader2_cmd.int1342.cst_0,
					param->bootloader2.bootloader2_cmd.int1342.nb_block,
					param->bootloader2.bootloader2_cmd.int1342.cst2_0,
					param->bootloader2.bootloader2_cmd.int1342.offset,
					param->bootloader2.bootloader2_cmd.int1342.segment,
					param->bootloader2.bootloader2_cmd.int1342.lba);
			} else {
				printf ("Read chain with IDE: ax 0x%X, ide_dcr_address 0x%X, base_ide_plus7 0x%X,\n"
					" lba48_cylinder_high %u, lba48_cylinder_low %u, lba48_sector %u, lba48_nb_sect %u,\n"
					" cylinder_high %u, cylinder_low %u, sector %u, nb_sect %u,\n"
					" lba_head 0x%X, ide_command 0x%X nb_sect_read %u, sector_size_div2 %u\n",
					param->bootloader2.bootloader2_cmd.hardide.reg_ax,
					param->bootloader2.bootloader2_cmd.hardide.ide_dcr_address,
					param->bootloader2.bootloader2_cmd.hardide.base_ide_plus7,
					param->bootloader2.bootloader2_cmd.hardide.lba48_cylinder_high,
					param->bootloader2.bootloader2_cmd.hardide.lba48_cylinder_low,
					param->bootloader2.bootloader2_cmd.hardide.lba48_sector,
					param->bootloader2.bootloader2_cmd.hardide.lba48_nb_sect,
					param->bootloader2.bootloader2_cmd.hardide.cylinder_high,
					param->bootloader2.bootloader2_cmd.hardide.cylinder_low,
					param->bootloader2.bootloader2_cmd.hardide.sector,
					param->bootloader2.bootloader2_cmd.hardide.nb_sect,
					param->bootloader2.bootloader2_cmd.hardide.lba_head,
					param->bootloader2.bootloader2_cmd.hardide.ide_command,
					param->bootloader2.bootloader2_cmd.hardide.nb_sect_read,
					param->bootloader2.bootloader2_cmd.hardide.sector_size_div2);
			}
		}

		if (serialconf != 0 && serialconf->speed != 0) {	/* i.e. != 110 bauds */
			/* Variable serialconf_port_plus_1 is not at a fixed position, normally constant_0 should tell if we are on VGA */
			printf ("Text output: constant_0 %u, eightbit %u, constant_1 %u, doublestop %u,\n"
				"  parity %u, even %u, speed %u bauds.\n",
				serialconf->constant_0,
				serialconf->eightbit,
				serialconf->constant_1,
				serialconf->doublestop,
				serialconf->parity,
				serialconf->even,
				((unsigned []){ 110, 150, 300, 600, 1200, 2400, 4800, 9600 })[serialconf->speed]
				);
		} else
			printf ("output probably on screen, serialconf = 0x%4.4X.\n", *(unsigned short *)serialconf);

		if (bootFAT.bootFAT16.code_part[0] == 0xEC && bootFAT.bootFAT16.code_part[1] == 0xA8
			&& bootFAT.bootFAT16.code_part[2] == 0xD0 && bootFAT.bootFAT16.code_part[3] == 0x7E)
			printf ("Standard codebase detected\n"); // inb	%dx,%al ; test	$0xD0,%al ; jle	L_waitdiskready
		else if (bootFAT.bootFAT16.code_part[0] == 0x83 && bootFAT.bootFAT16.code_part[1] == 0xE0
			&& bootFAT.bootFAT16.code_part[2] == 0x0F && bootFAT.bootFAT16.code_part[3] == 0x9E)
			printf ("ONLY_PRINT_GEOMETRY codebase detected\n"); // andw	$0x000F,%ax ; sahf
		else
			printf ("UNKNOWN codebase\n");

#ifdef SHOW_OFFSET
		const instboot_info_t *instboot_info = *(const instboot_info_t **)&bootFAT.upto_instboot_info[512];
		if ((unsigned)instboot_info & 0xFFFF0000) // instboot_info_ptr approx 0x00001490
			printf ("This boot record is not physically followed by Gujin code\n");
		else {
			printf ("instboot_info probably at offset 0x%X (== 0x00001490 ?)\n", (unsigned)instboot_info);
			instboot_info = (const instboot_info_t *)((unsigned)instboot_info + (unsigned)&bootFAT);
			printf ("unsigned checksum_start_adr = 0x%X;\n", instboot_info->checksum_start_adr);
			printf ("unsigned boot1param_adr = 0x%X;\n", instboot_info->boot1param_adr);
			printf ("unsigned bootend_adr = 0x%X;\n", instboot_info->bootend_adr);
			printf ("unsigned uninstall_mbr_adr = 0x%X;\n", instboot_info->uninstall_mbr_adr);
			printf ("unsigned bootchain_adr = 0x%X;\n", instboot_info->bootchain_adr);
			printf ("unsigned bootchain_nbblock = 0x%X;\n", instboot_info->bootchain_nbblock);
			printf ("unsigned gujin_param_adr = 0x%X;\n", instboot_info->gujin_param_adr);
			printf ("unsigned code_start_adr = 0x%X;\n", instboot_info->code_start_adr);
			printf ("unsigned patch2adr = 0x%X;\n", instboot_info->patch2adr);
			printf ("unsigned patch3adr = 0x%X;\n", instboot_info->patch3adr);
			printf ("unsigned patch4adr = 0x%X;\n", instboot_info->patch4adr);
			printf ("unsigned sbss = 0x%X;\n", instboot_info->sbss);
			printf ("unsigned edata = 0x%X;\n", instboot_info->edata);
			printf ("unsigned deltaseg = 0x%X;\n", instboot_info->deltaseg);
			printf ("unsigned diskcodeseg = 0x%X;\n", instboot_info->diskcodeseg);
			printf ("unsigned fscodeseg = 0x%X;\n", instboot_info->fscodeseg);
			printf ("unsigned loadcodeseg = 0x%X;\n", instboot_info->loadcodeseg);
			printf ("unsigned usercodeseg = 0x%X;\n", instboot_info->usercodeseg);
			printf ("unsigned xdataseg = 0x%X;\n", instboot_info->xdataseg);
			printf ("unsigned xstring1seg = 0x%X;\n", instboot_info->xstring1seg);
			printf ("unsigned xstring2seg = 0x%X;\n", instboot_info->xstring2seg);
			printf ("unsigned password_serial_adr = 0x%X;\n", instboot_info->password_serial_adr);
			printf ("unsigned serialconf_port_plus_1_adr = 0x%X;\n", instboot_info->serialconf_port_plus_1_adr);
			printf ("unsigned read_disk_adr = 0x%X;\n", instboot_info->read_disk_adr);
			printf ("unsigned partition_relative_read_disk_adr = 0x%X;\n", instboot_info->partition_relative_read_disk_adr);
			printf ("unsigned read_disk_1st_adr = 0x%X;\n", instboot_info->read_disk_1st_adr);
/*
$ ./showmbr boot.bin
Standard codebase detected, instboot_info probably at offset 0x1490 (== 0x00001490 ?)
unsigned checksum_start_adr = 0x3E;
unsigned boot1param_adr = 0x19A;
unsigned bootend_adr = 0x1B6;
unsigned uninstall_mbr_adr = 0x828;
unsigned bootchain_adr = 0x3C6;
unsigned bootchain_nbblock = 0x23;
unsigned gujin_param_adr = 0x230;
unsigned code_start_adr = 0x71;
unsigned patch2adr = 0x76;
unsigned patch3adr = 0x156;
unsigned patch4adr = 0x14A;
unsigned sbss = 0x51E0;
unsigned edata = 0x5C78;
unsigned deltaseg = 0x24F4;
unsigned xcodeseg = 0xF3C; << no more
unsigned xdataseg = 0x1D6A;
unsigned password_serial_adr = 0x1B6;
unsigned serialconf_port_plus_1_adr = 0xC7;
unsigned read_disk_adr = 0xA4;
unsigned partition_relative_read_disk_adr = 0x98;
unsigned read_disk_1st_adr = 0x362;
 */
		}
#endif
		unsigned patch2adr = 0x76;
		unsigned short *delta_ptr = (unsigned short *)((unsigned long)patch2adr + (unsigned long)&bootFAT);
		printf ("delta for data/stack: 0x%X\n", *delta_ptr);

		unsigned read_disk_adr = 0xA4;
		unsigned short *movaxdx_ptr = (unsigned short *)((unsigned long)read_disk_adr + 1 + (unsigned long)&bootFAT);
		printf ("BIOSdlDISK: 0x%X i.e. patch ", *movaxdx_ptr);
		if (*movaxdx_ptr == 0xE688)
			puts ("present");
		else if (*movaxdx_ptr == 0xC289)
			puts ("abscent");
		else
			puts ("unknown");

		unsigned patch4adr = 0x14A;
		unsigned *exit_ptr = (unsigned *)((unsigned long)patch4adr + (unsigned long)&bootFAT);
		printf ("ExitPatch: 0x%X i.e. patch ", *exit_ptr);
		if (*exit_ptr == 0x21CD4CB4)
			puts ("DOS exit");
		else if (*exit_ptr == 0xCF1F0761)
			puts ("read retry");
		else if (*exit_ptr == 0x18CD19CD)
			puts ("no read retry");
		else
			puts ("unknown");

		unsigned patch3adr = 0x156;
		unsigned short *call_ptr = (unsigned short *)((unsigned long)patch3adr + (unsigned long)&bootFAT);
		unsigned short call_val = *call_ptr + patch3adr + 2; // relative call
		printf ("readdisk call = 0x%X i.e. ", call_val);
		if (call_val == 0xA4)
			puts ("unpatched");
		else if (call_val == 0x98 - 1)
			puts ("noop");
		else if (call_val == 0x362)
			puts ("cdrom first");
		else if (call_val == 0x98)
			puts ("partition relative");
		else
			puts ("unknown");

#define OFFSET_PATTERN_2        0xB2
#define INIT_PATTERN_2  "\x89\x5c\x04\xcd\x13\x73\x05"
#define REPL_PATTERN_2  "\xe8\xb2\xff\xcd\x13\x73\x98"
		if (!memcmp ((char *)&bootFAT+OFFSET_PATTERN_2, INIT_PATTERN_2, sizeof(INIT_PATTERN_2) - 1))
			puts ("Single sector load patch: abscent");
		else if (!memcmp ((char *)&bootFAT+OFFSET_PATTERN_2, REPL_PATTERN_2, sizeof(REPL_PATTERN_2) - 1))
			puts ("Single sector load patch: present");
		else
			puts ("Single sector load patch: unknown");

		if (!memcmp ((char *)&bootFAT + 0xAF, "\x8c\x44\x06", sizeof ("\x8c\x44\x06") - 1))
			puts ("USB-HDD patch: abscent");
		else if (!memcmp ((char *)&bootFAT + 0xAF, "\xe8\xd0\xff", sizeof("\xe8\xd0\xff") - 1))
			puts ("USB-HDD patch: present");
		else
			puts ("USB-HDD patch: unknown");
	}

	/* The FAT is at the same place, even if there isn't any FAT16/32 marker. */
	printf ("Partition table: WindowsNTmarker 0x%X, Unknown 0x%X, signature 0x%X\n",
		(unsigned)bootFAT.bootFAT16.after.WindowsNTmarker,
		bootFAT.bootFAT16.after.Unknown,
		bootFAT.bootFAT16.after.Signature0xAA55);
	for (cpt_partition = 0; cpt_partition < 4; cpt_partition++) {
		struct bootsect_partition_str *ptr = &bootFAT.bootFAT16.after.bootsect_partition[cpt_partition];
		const char *pname = "unknown";

#define P(shortname, value, longname) if (ptr->system_indicator == value) pname = longname;
		PARTITIONLIST()
#undef P /*(shortname, value, longname)*/

		printf ("%u: indicator 0x%X i.e. %s, start %u length %u\n"
			"  (i.e. end at %u); start %u/%u/%u end %u/%u/%u name '%s'\n",
			cpt_partition, ptr->indicator,
			(ptr->indicator == 0x80)? "bootable" : ((ptr->indicator == 0)? "non bootable" : "invalid"),
			(unsigned)ptr->nb_sector_before, (unsigned)ptr->nb_sector, (unsigned)ptr->nb_sector_before + (unsigned)ptr->nb_sector,
			((unsigned)ptr->begin_cylinder_msb << 8) + ptr->begin_cylinder_lsb, ptr->begin_head, ptr->begin_sector,
			((unsigned)ptr->end_cylinder_msb << 8) + ptr->end_cylinder_lsb, ptr->end_head, ptr->end_sector,
			pname);
		}
	return 0;
}

