/* showmap.c is a utility to display the mapping of a file on a device.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#define _FILE_OFFSET_BITS 64
#include <features.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/fs.h>
#include <linux/hdreg.h>
#include <linux/cdrom.h>
#include <linux/major.h>
#include <linux/raid/md_u.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <alloca.h>
#include <stdio.h>
#include <string.h>
#ifdef FS_IOC_FIEMAP
#include <linux/fiemap.h>
#endif

static int show_mapping (int file_desc, unsigned filesize, int device_desc)
{
	unsigned nb_block, block_index, current_block = 0, block_count = 0;
	unsigned bmap_block_size, fragment_cpt = 1;
	unsigned long block;

#ifdef FS_IOC_FIEMAP
	struct fiemap shortfiemap = {
		fm_start: 0,
		fm_length: filesize,
		fm_flags: FIEMAP_FLAG_SYNC
		};
	ioctl (file_desc, FS_IOC_FIEMAP, &shortfiemap); /* errno: FS_IOC_FIEMAP: File too large */
	errno = 0;
	printf ("First FIEMAP says %u extents\n", shortfiemap.fm_mapped_extents);
	struct fiemap *fullfiemap = alloca (sizeof(struct fiemap) + shortfiemap.fm_mapped_extents * sizeof(struct fiemap_extent));
	if (fullfiemap == 0) {
		perror ("alloca");
		return -1;
		}
	fullfiemap->fm_start = 0;
	fullfiemap->fm_length = filesize;
	fullfiemap->fm_flags = fullfiemap->fm_mapped_extents = fullfiemap->fm_reserved = 0;
	fullfiemap->fm_extent_count = shortfiemap.fm_mapped_extents;
	if (ioctl (file_desc, FS_IOC_FIEMAP, fullfiemap))
		perror ("second FIEMAP");
	else {
		printf ("second FIEMAP success, %u extents filled:\n", fullfiemap->fm_mapped_extents);
		for (block_count = 0; block_count < fullfiemap->fm_mapped_extents; block_count++)
			printf ("  %u: logical offset %llu (%llu * 4096), physical offset %llu (%llu * 4096),\n"
				"           length %llu (%llu * 4096) flags 0x%X\n", block_count,
				fullfiemap->fm_extents[block_count].fe_logical, fullfiemap->fm_extents[block_count].fe_logical/4096,
				fullfiemap->fm_extents[block_count].fe_physical, fullfiemap->fm_extents[block_count].fe_physical/4096,
				fullfiemap->fm_extents[block_count].fe_length, fullfiemap->fm_extents[block_count].fe_length/4096,
				fullfiemap->fm_extents[block_count].fe_flags);
		printf ("  flags meaning: FIEMAP_EXTENT_LAST 0x%X, FIEMAP_EXTENT_UNKNOWN 0x%X,\n"
			"    FIEMAP_EXTENT_DELALLOC 0x%X, FIEMAP_EXTENT_ENCODED 0x%X,\n"
			"    FIEMAP_EXTENT_DATA_ENCRYPTED 0x%X, FIEMAP_EXTENT_NOT_ALIGNED 0x%X,\n"
			"    FIEMAP_EXTENT_DATA_INLINE 0x%X, FIEMAP_EXTENT_DATA_TAIL 0x%X,\n"
			"    FIEMAP_EXTENT_UNWRITTEN 0x%X, FIEMAP_EXTENT_MERGED 0x%X.\n",
			FIEMAP_EXTENT_LAST, FIEMAP_EXTENT_UNKNOWN, FIEMAP_EXTENT_DELALLOC, FIEMAP_EXTENT_ENCODED,
			0, //FIEMAP_EXTENT_ENCRYPTED,
			FIEMAP_EXTENT_NOT_ALIGNED, FIEMAP_EXTENT_DATA_INLINE, FIEMAP_EXTENT_DATA_TAIL,
			FIEMAP_EXTENT_UNWRITTEN, FIEMAP_EXTENT_MERGED);
		}
#endif

	if (ioctl (file_desc, FIGETBSZ, &bmap_block_size)) {
		perror ("FIGETBSZ");
		return -1;
		}
	printf ("FIGETBSZ: block size %u bytes\n", bmap_block_size);
	nb_block = (filesize + bmap_block_size - 1) / bmap_block_size;
	block_count = 0;
	for (block_index = 0; block_index < nb_block; block_index++) {
		char device_buffer[bmap_block_size], file_buffer[bmap_block_size];
		unsigned read_len;

		block = block_index;
		if (ioctl (file_desc, FIBMAP, &block)) {
			perror ("FIBMAP");
			return -2;
			}
		if (lseek (file_desc, block_index * bmap_block_size, SEEK_SET) != block_index * bmap_block_size) {
			perror ("lseek");
			return -3;
			}
		read_len = read (file_desc, file_buffer, sizeof (file_buffer));
		if (read_len != sizeof (file_buffer) && block_index != nb_block - 1) {
			perror ("read file");
			return -4;
			}
		if (block == 0)
			memset (device_buffer, 0, sizeof (device_buffer));
		else {
			typedef unsigned long long off64_t;
			loff_t lseek64(int fildes, loff_t offset, int whence);
			/* FIXME: It is officially undefined to access files that way when the FS is mounted... */

			if (lseek64 (device_desc, (off64_t)block*bmap_block_size, SEEK_SET) != (off64_t)block*bmap_block_size) {
				perror ("lseek64");
				return -5;
				}
			if (read (device_desc, device_buffer, read_len) != read_len) {
				perror ("read device");
				return -6;
				}
			}
		if (memcmp (file_buffer, device_buffer, read_len) != 0) {
			unsigned offset;
			unsigned char *ptr;
			for (offset = 0; offset < sizeof (device_buffer); offset++)
				if (file_buffer[offset] != device_buffer[offset])
					break;
			printf ("file/device block compare failed at offset 0x%X (lseek64 fail?):\n", offset);
			if (offset > 8)
				offset -= 8;
			ptr = (unsigned char *)&file_buffer[offset & ~0x7];
			printf ("0x%8.8X: %02X, %02X, %02X, %02X, %02X, %02X, %02X, %02X\n",
				block_index * bmap_block_size + (offset & ~0x7),
				ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
			ptr += 8;
			printf ("0x%8.8X: %02X, %02X, %02X, %02X, %02X, %02X, %02X, %02X\n",
				block_index * bmap_block_size + (offset & ~0x7) + 8,
				ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
			ptr += 8;
			printf ("0x%8.8X: %02X, %02X, %02X, %02X, %02X, %02X, %02X, %02X\n",
				block_index * bmap_block_size + (offset & ~0x7) + 16,
				ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
			ptr += 8;
			printf ("0x%8.8X: %02X, %02X, %02X, %02X, %02X, %02X, %02X, %02X\n",
				block_index * bmap_block_size + (offset & ~0x7) + 24,
				ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
			printf ("\nDevice at offset %lu * %u contains:\n", block, bmap_block_size);
			ptr = (unsigned char *)&device_buffer[offset & ~0x7];
			printf ("0x%8.8X: %02X, %02X, %02X, %02X, %02X, %02X, %02X, %02X\n",
				block_index * bmap_block_size + (offset & ~0x7),
				ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
			ptr += 8;
			printf ("0x%8.8X: %02X, %02X, %02X, %02X, %02X, %02X, %02X, %02X\n",
				block_index * bmap_block_size + (offset & ~0x7) + 8,
				ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
			ptr += 8;
			printf ("0x%8.8X: %02X, %02X, %02X, %02X, %02X, %02X, %02X, %02X\n",
				block_index * bmap_block_size + (offset & ~0x7) + 16,
				ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
			ptr += 8;
			printf ("0x%8.8X: %02X, %02X, %02X, %02X, %02X, %02X, %02X, %02X\n",
				block_index * bmap_block_size + (offset & ~0x7) + 24,
				ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
			return -7;
			}
		if (block_index == nb_block - 1) {
			if (block_index == block_count && current_block == 0)
				printf ("File is a single hole of %u blocks\n", block_count + 1);
			else {
				if (block == current_block + block_count)
					block_count++;
				else
					fragment_cpt++;
				if (block_count != 1)
					printf (" for %u blocks", block_count);
				if (block == 0)
					printf (",\n  end with a hole");	/* E2FS */
				else
					printf (",\n  last block %lu", block);	/* FAT */

				printf (" and file has %u fragments.\n", fragment_cpt);
				}
			}
		else if (block == 0) { /* we are in a hole */
			block_count++;
			}
		else if (block_index == 0) {
			printf ("File (%u blocks of %u) start at block %lu", nb_block, bmap_block_size, block);
			current_block = block;
			block_count = 1;
			}
		else if (block_index == block_count && current_block == 0) {
			printf ("File (%u blocks of %u) begin with a hole of %u block, and start at block %lu",
				nb_block, bmap_block_size, block_count, block);
			current_block = block;
			block_count = 1;
			}
		else if (block == current_block + block_count) {
			block_count++;
			}
		else {
			if (block_count != 1)
				printf (" for %u blocks", block_count);
			if (++fragment_cpt % 2 == 0)
				printf (",\n  then %lu", block);
			else
				printf (", then %lu", block);
			current_block = block;
			block_count = 1;
			}
		}
//	while (ioctl (file_desc, FIBMAP, (block = block_index, &block)) == 0)
	if (ioctl (file_desc, FIBMAP, (block = block_index, &block)) == 0)
		printf ("FIBMAP succeeded after end of file, block index %u give block %lu\n", block_index++, block);
	return 0;
}

static void show_cd_extra (int device_desc, const struct cdrom_tochdr *cdrom_param)
{
#ifdef CDROMREADTOCENTRY
	struct cdrom_tocentry cdrom_toc;
#endif
#ifdef CDROM_GET_MCN
	struct cdrom_mcn cdrom_mcn;
#endif
#ifdef CDROMMULTISESSION
	struct cdrom_multisession cdrom_multisession;
#endif
#ifdef CDROMREADTOCENTRY
	cdrom_toc.cdte_track = cdrom_param->cdth_trk1;
	if (ioctl (device_desc, CDROMREADTOCENTRY, &cdrom_toc) == 0)
		printf ("; track=%u, adr:4=0x%X, ctrl:4=0x%X, format=0x%X,"
			" cdte_addr=%u, datamode=0x%X",
			cdrom_toc.cdte_track, cdrom_toc.cdte_adr, cdrom_toc.cdte_ctrl,
			cdrom_toc.cdte_format, cdrom_toc.cdte_addr.lba,
			cdrom_toc.cdte_datamode);
#endif
#ifdef CDROM_GET_MCN
	if (ioctl (device_desc, CDROM_GET_MCN, &cdrom_mcn) == 0)
		printf ("; cdrom_mcn: '%s'", cdrom_mcn.medium_catalog_number);
#endif
#ifdef CDROMMULTISESSION
	if (ioctl (device_desc, CDROMMULTISESSION, &cdrom_multisession) == 0)
		printf ("; start last session %u, is_xa %u addr format %s",
			cdrom_multisession.addr.lba,
			cdrom_multisession.xa_flag,
			cdrom_multisession.addr_format == CDROM_LBA? "CDROM_LBA": "CDROM_MSF");
#endif
	printf ("\n");
}

int main (int argc, char **argv)
{
	int file_desc, device_desc;
	char *filename, device_name[80];
	struct stat statbuf;
	unsigned device_nb_block = 0, device_block_size = 0, fs_block_size = 0;
#ifdef HDIO_GETGEO_BIG
	struct hd_big_geometry hd_big_param;
#endif
#ifdef HDIO_GETGEO
	struct hd_geometry hd_param;
#endif
#ifdef BLKGETSIZE64
	unsigned long long device_length = 0;
#endif
#ifdef RAID_VERSION
	struct mdu_version_s raid_version;
#endif
#ifdef CDROMREADTOCHDR
	struct cdrom_tochdr cdrom_param;
#endif

	if (argc != 2) {
		printf ("Utility to display the physical mapping of a file on a device\n");
		printf ("USAGE: %s <file>\n", argv[0]);
		return -1;
		}
	filename = argv[1];

	/* FIXME: there was a proposal somewhere to an open() flag
		for just ioctl's and not even read access, would it enable
		non-root to run this software? What's the name of it? */
	if ((file_desc = open (filename, O_RDONLY)) == -1) {
		perror (filename);
		return -2;
		}
	if (fstat (file_desc, &statbuf)) {
		perror ("fstat");
		if (close (file_desc) != 0)
			perror (filename);
		return -3;
		}
	printf ("File \"%s\" of size %ld (%ld blocks512) is on filesystem 0x%llX.\n",
		filename, (long)statbuf.st_size, (long)statbuf.st_blocks, (unsigned long long)statbuf.st_dev);

	FILE *file = fopen ("/proc/diskstats", "r");
	if (file != NULL) {
		int ret, major, minor;
		strcpy (device_name, "/dev/");
		while ((ret = fscanf (file, " %d %d %s %*[^\n]", &major, &minor, &device_name[5])) != EOF)
			if (ret == 3 && 256 * major + minor == statbuf.st_dev)
				break;
		if (fclose(file))
			perror ("/proc/diskstats");
		if (ret != EOF)
			printf ("According to /proc/diskstats, file '%s' is on device '%s'\n", filename, device_name);
		else
			goto create_device;
		}
	else {
create_device:
		strcpy (device_name, "showmap_device.tmp");
		printf ("Did not find the device 0x%llX in /proc/diskstats, need to create local device named '%s'\n", (unsigned long long)statbuf.st_dev, device_name);
		if (mknod (device_name, S_IFBLK | 0600, statbuf.st_dev) != 0) {
			perror("mknod [need to be root]");
			if (close (file_desc))
				perror ("close");
			return -4;
			}
		}

	if ((device_desc = open (device_name, O_RDONLY)) == -1) {
		perror(device_name);
		if (!strcmp (device_name, "showmap_device.tmp") && unlink (device_name))
			perror ("unlink");
		return -5;
		}

	if (ioctl (device_desc, BLKBSZGET, &fs_block_size))
		perror ("BLKBSZGET");
	if (ioctl (device_desc, BLKSSZGET, &device_block_size))
		perror ("ioctl BLKSSZGET");
	if (ioctl (device_desc, BLKGETSIZE, &device_nb_block))
		perror ("ioctl BLKGETSIZE");
	printf ("Device block size: %u, FS block size: %u, device size: %u blocks\n",
		device_block_size, fs_block_size, device_nb_block);

#ifdef BLKGETSIZE64
	if (ioctl (device_desc, BLKGETSIZE64, &device_length))
		perror ("ioctl BLKGETSIZE64");
	printf ("Device length: %llu bytes\n", device_length);
#endif
#ifdef HDIO_GETGEO_BIG
	if (ioctl (device_desc, HDIO_GETGEO_BIG, &hd_big_param) == 0) {
		printf ("The device start at %lu sectors, C/H/S: %u/%u/%u.\n",
			hd_big_param.start, hd_big_param.cylinders, hd_big_param.heads,
			hd_big_param.sectors);
		} else
#endif
#ifdef HDIO_GETGEO
	if (ioctl (device_desc, HDIO_GETGEO, &hd_param) == 0) {
		printf ("The device start at %lu sectors, C/H/S: %u/%u/%u.\n",
			hd_param.start, hd_param.cylinders, hd_param.heads, hd_param.sectors);
		}
#endif

#ifdef RAID_VERSION
	if (ioctl (device_desc, RAID_VERSION, &raid_version) == 0) {
		printf ("The device is RAID version %u.%u.%u\n",
			raid_version.major, raid_version.minor, raid_version.patchlevel);
		}
#endif
#ifdef CDROMREADTOCHDR
	if (ioctl (device_desc, CDROMREADTOCHDR, &cdrom_param) == 0) {
		printf ("The device: start track = %u, end track = %u",
			cdrom_param.cdth_trk0, cdrom_param.cdth_trk1);
		show_cd_extra(device_desc, &cdrom_param);
		}
#endif

	show_mapping(file_desc, statbuf.st_size, device_desc);

	if (close (device_desc))
		perror ("close");
	if (!strcmp (device_name, "showmap_device.tmp") && unlink (device_name))
		perror ("unlink");
	if (close (file_desc) != 0)
		perror (filename);
	return 0;
}




#if 0
#define MAJOR_LVM	0xFD
#ifdef MAJOR_LVM
			if ((statbuf.st_dev >> 8) == MAJOR_LVM) {
				unsigned short version = 0;
				int lvm_desc = open("/dev/lvm", O_RDONLY);

				if (lvm_desc < 0)
					perror ("open /dev/lvm");
				else {
#ifndef LVM_GET_IOP_VERSION
#define LVM_GET_IOP_VERSION		_IOR(0xfe, 0x98, char)
#endif
					if (ioctl(lvm_desc, LVM_GET_IOP_VERSION, &version) != 0)
						printf ("No LVM version info\n");
					else
						printf ("LVM version: 0x%X\n", version);
					if (close (lvm_desc))
						perror ("close /dev/lvm");
					}

				struct lv_bmap {
					__u32 lv_block;
					__u16 lv_dev;
					} lbm;

#ifndef LV_BMAP
#define LV_BMAP				_IOWR(0xfe, 0x30, char)
#endif

				lbm.lv_dev = statbuf.st_dev;
				lbm.lv_block = block;

	lvm_bmap(&lbm);
	if (lbm.lv_dev != geo->base_dev)
	    die("LVM boot LV cannot be on multiple PVs\n");
	block = lbm.lv_block;
#endif

				}
#endif
