/* mouse.h */
#ifndef MOUSE_H
#define MOUSE_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"

#if !(USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT))

#if USER_SUPPORT != 0
void NOMOUSE_print (const char *str, unsigned active, unsigned row, unsigned col);
awk_farcall (NOMOUSE_print);

/* Just to save code space if mouse is not defined: */
#define MOUSE_declare_attribute(varname)	 unsigned char varname = 0
#define MOUSE_attrset_fullscreenfield(varname, val)	/* */
#define MOUSE_attrset_mainfield(varname, val)	 /* */
#define MOUSE_attrset_upperfield(varname, val)	 /* */
#define MOUSE_attrset_pressedfield(varname, val) /* */
#define MOUSE_attrset_chkboxfield(varname, val)	 /* */
#define MOUSE_attrset_activefield(varname, val)	 varname = val
#define MOUSE_attrset_greyfield(varname, val)	 /* */
#define PUT_FIELD(name, active, key, row, col)	NOMOUSE_print (name, active, row, col)
#define MOUSE_has_moved()	0

#endif /* USER_SUPPORT != 0 */

#else  /* USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT) */

/* Just to save code space if mouse is not defined: */
#define MOUSE_declare_attribute(varname) struct mousefieldattr_str varname = {}
#define MOUSE_attrset_fullscreenfield(varname, val)	 varname.fullscreenfield = val
#define MOUSE_attrset_mainfield(varname, val)	 varname.mainfield = val
#define MOUSE_attrset_upperfield(varname, val)	 varname.upperfield = val
#define MOUSE_attrset_pressedfield(varname, val) varname.pressed = val
#define MOUSE_attrset_chkboxfield(varname, val)	 varname.checkbox = val
#define MOUSE_attrset_activefield(varname, val)	 varname.active = val
#define MOUSE_attrset_greyfield(varname, val)	 varname.greyed = val
#define PUT_FIELD(name, attr, key, row, col)	MOUSE_print_field (name, attr, key, row, col)

/*
 * This is the height of the mouse graphic cursor.
 * It has to be changed depending on:
 * static const union mouse_cursor cursor_mask[]
 */
#define SIZEOF_CURSOR_MASK	14

union mouse_cursor {
      struct {
	unsigned char b0 :1, b1 :1, b2 :1, b3 :1, b4 :1, b5 :1, b6 :1, b7 :1;
	} __attribute__ ((packed)) bit;
      unsigned char group;
      };

#if USER_SUPPORT & SERIAL_MOUSE_SUPPORT
/*
 * To use the extended serial BIOS instead of direct I/O access
 * to read the "modem" lines.
 */
//#define EXTENDED_SERIAL_BIOS

#ifdef EXTENDED_SERIAL_BIOS
#define SETMODEM(cpt, ioaddr, modem) \
	_BIOS_extended_set_modem_control (cpt, modem)
#define GETMODEM(cpt, ioaddr, modem) \
	_BIOS_extended_get_modem_control (cpt, &modem)
#else
#define SETMODEM(cpt, ioaddr, modem) \
	outb (ioaddr+4, *(unsigned char *)&modem)
#define GETMODEM(cpt, ioaddr, modem) \
	modem = inb (ioaddr+4)
#endif

#if 0 /* too slow on a 386: */
#define SERIAL_DATA_RECEIVED(Mouse_data_serial) \
    (_BIOS_getserialstatus(Mouse_data_serial.comport).receive_data_ready != 0)
#else
#define SERIAL_DATA_RECEIVED(Mouse_data_serial) \
    ((inb (Mouse_data_serial.comIOaddr+5) & 0x01) != 0)
#endif

/* The only way which do not wait modem lines: */
#define SERIAL_DATA_GET(Mouse_data_serial) \
    (inb (Mouse_data_serial.comIOaddr))

#endif /* SERIAL_MOUSE_SUPPORT */

extern const struct mousefieldattr_str reset_all_fields;

struct mousefield_str {
    coord upperleft, lowerright;
    unsigned short keypress;
    struct mousefieldattr_str {
	unsigned short fullscreenfield	: 1;	/* for "Press a key to continue" */
	unsigned short upperfield	: 1;
	unsigned short mainfield	: 1;
	unsigned short active		: 1;
	unsigned short pressed		: 1;
	unsigned short checkbox		: 1;
	unsigned short greyed		: 1;
	} attr;
    };

/*
 * The main structure:
 */
struct mouse_interface_str {
    coord		position, maximum;
    struct mouse_button_str {
	unsigned char left	: 1;
	unsigned char right	: 1;
	unsigned char middle	: 1;
	unsigned char top	: 1;
	unsigned char bottom	: 1;
	unsigned char reserved	: 1;
	unsigned char notfunny	: 1;
	unsigned char hidden	: 1;
	} __attribute__ ((packed)) button;
    unsigned char curmaskindex;
#ifdef MOUSE_ON_VT
    unsigned char curs_row, curs_col;
#else
    unsigned char reserved[2];
#endif

    enum MOUSE_type {
	MOUSE_NONE= 0,
	MOUSE_PS2 = 0xAA,
	MOUSE_SERIAL_COM1 = 0x10, MOUSE_SERIAL_COM2,   /* 2 buttons */
	MOUSE_SERIAL_COM3, MOUSE_SERIAL_COM4,
	MOUSE_SERIAL3_COM1 = 0x30, MOUSE_SERIAL3_COM2, /* 3 buttons */
	MOUSE_SERIAL3_COM3, MOUSE_SERIAL3_COM4,         /* Logitech */
	MOUSE_SERIALM_COM1 = 0x40, MOUSE_SERIALM_COM2, /* 3 buttons */
	MOUSE_SERIALM_COM3, MOUSE_SERIALM_COM4         /* MouseSystems */
#define MOUSE_SERIAL_MASK 0xF0
	} type;

    unsigned			nb_field;
    struct mousefield_str	*fields;
    union mouse_color_union {
	struct mouse_color_str {
	    unsigned deflt		: 4;
	    unsigned leftbutton		: 4;
	    unsigned rightbutton	: 4;
	    unsigned middlebutton	: 4;
	    unsigned topbutton		: 4;
	    unsigned twobutton		: 4;
	    unsigned invalid		: 4;
	    unsigned infield		: 4;
	    } colors;
	unsigned all;
	} all_colors;

    struct {
	void (*reset_field) (struct mousefieldattr_str attr);
	void (*print_field) (const char *str, struct mousefieldattr_str attr,
		unsigned short key, unsigned short row, unsigned short col);
	} function;

    union {
	struct PS2_struct {
	      signed char z_zero	: 2;
	      signed char z_button5	: 1;
	      signed char z_button4	: 1;
	      signed char z_data	: 4;
	      signed char z_reserved	: 8;
	      signed char y_data	: 8;
	      signed char y_reserved	: 8;
	      signed char x_data	: 8;
	      signed char x_reserved	: 8;
	    unsigned char left_button	: 1;
	    unsigned char right_button	: 1;
	    unsigned char middle_button	: 1;
	    unsigned char reservedbit	: 1;
	    unsigned char x_negative	: 1;
	    unsigned char y_negative	: 1;
	    unsigned char x_overflow	: 1;
	    unsigned char y_overflow	: 1;
	    unsigned char reservedbyte	: 8;
	    /* Not sent by the protocol: */
	    unsigned char irq_trigged;
	    unsigned char device_id;
	    } __attribute__ ((packed)) PS2;
	struct {
	    unsigned comIOaddr			: 16;
	    unsigned comport			: 8;
	    unsigned cpt			: 8;
	    union {
		unsigned char bytes[5]; /* 3, 4 or 5 */
		unsigned char align[8];
		struct {
		    unsigned char xmsb		: 2;
		    unsigned char ymsb		: 2;
		    unsigned char right_button	: 1;
		    unsigned char left_button	: 1;
		    unsigned char cste1		: 1;
		    unsigned char sevenbit1	: 1;
		    unsigned char x		: 6;
		    unsigned char cste0_1	: 1;
		    unsigned char sevenbit2	: 1;
		    unsigned char y		: 6;
		    unsigned char cste0_2	: 1;
		    unsigned char sevenbit3	: 1;
		    /* 32 = 0x20 when pressed, 0 when released: */
		    unsigned logitech		: 8;
		    } __attribute__ ((packed)) stddata;
		struct {
		    unsigned char right_button	: 1;
		    unsigned char middle_button	: 1;
		    unsigned char left_button	: 1;
		    unsigned char synchro_0x10	: 5;
		      signed char x		: 8;
		      signed char y		: 8;
		      signed char deltax	: 8;
		      signed char deltay	: 8;
		    } __attribute__ ((packed)) mmdata;
		} received;
	    } serial;
	} data;

#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & JOYSTICK_SUPPORT)
    struct joystick_str {
	unsigned short Xmin, Xmid, Xmax, Ymin, Ymid, Ymax;
	struct mouse_button_str button;
	unsigned char accelerate;
	unsigned short last_nbtick;
	} joystick;
#endif

    unsigned savearray[8 * SIZEOF_CURSOR_MASK];
    };

extern struct mouse_interface_str MOUSE;

extern inline unsigned
MOUSE_inited (void)
  {
  return (MOUSE.function.print_field != 0);
  }

extern inline int
MOUSE_has_moved (void)
  {
  /* Called after a MOUSE_stop() */
  /* MOUSE is no more in its default position, so cancel the autoload timer */
  return (MOUSE.position.x != 0xFFFFU / 8 || MOUSE.position.y != 0xFFFFU / 16);
  }

extern inline int
MOUSE_button_equal (const struct mouse_button_str a, const struct mouse_button_str b)
//  __attribute__ ((pure))
  {
  const union {
      struct mouse_button_str fields;
      unsigned char all;
      } ba = { a }, bb = { b };
  return ba.all == bb.all;
  }

/*
 * Indirect functions:
 */
/* Do NOT call this function: */
__attribute__((weak)) void asm_dummy_mouse (void)
  {
  awk_farfct (MOUSE.function.reset_field);
  awk_farfct (MOUSE.function.print_field);
  }

extern inline void
MOUSE_print_field (const char *str, struct mousefieldattr_str attr,
	unsigned short key, unsigned short row, unsigned short col)
  {
  MOUSE.function.print_field (str, attr, key, row, col);
  }

extern inline void
MOUSE_reset_field (struct mousefieldattr_str attr)
  {
  MOUSE.function.reset_field (attr);
  }

/*
 * The functions:
 */
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT | SERIAL_MOUSE_SUPPORT)
unsigned MOUSE_init (unsigned nb_serial_port);
awk_farcall (MOUSE_init);
#endif

#if USER_SUPPORT & JOYSTICK_SUPPORT
unsigned MOUSE_joystick_init (unsigned enable_joystick);
awk_farcall (MOUSE_joystick_init);
#endif

void MOUSE_start (void);
awk_farcall (MOUSE_start);

void MOUSE_stop (void);
awk_farcall (MOUSE_stop);

unsigned MOUSE_poll (void);
awk_farcall (MOUSE_poll);

#endif /* USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT) */

#endif /* MOUSE_H */
