/* mouse.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"
#if (SETUP & XCODE_SEGMENT)
#define CALLING_FROM_XCODESEG   /* before other includes */
#define CALLING_FROM_USERSEG	/* can call some non far video function for efficiency */
#endif

#include "instboot.h"
#include "library.h"
#include "boot.h"	/* BOOT1_COM_port() */
#include "debug.h"
#include "bios.h"	/* CDROMbootspec */
#include "util.h"	/* UTIL.joystick_present */
#include "user.h"
#include "mouse.h"

#if !(USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT))

#if USER_SUPPORT != 0
/*
 * Here the mouse is disabled probably to save code space,
 * so we do not do 3D button: no mouse.c::drawbutton()
 */

/* Shall this function also manage the screen scrolling ? */
VIDEO_FCT_PREFIX(NOMOUSE_print) void
NOMOUSE_print (const char *str, unsigned active, unsigned row, unsigned col)
  {bound_stack();{
  unsigned endpos = col;
  char displayed[UI.parameter.nbcol];

  /* We should not have \r, \n, \t, \b ... in str! */
  if (str == 0 || *str == '\0') {
      PDBG ((" [%s: add an empty field!] ", __FUNCTION__));
      return;
      }
    else {
      char *dst = displayed;
      const char *src = str;

      for (;;) {
	  /* 3D buttons needs a end-of-line space to draw right end button,
		but we do not have 3D buttons here */
	  unsigned utf8len;
	  if (endpos >= sizeof (displayed)  /* - 1U */) {
	      dst[-1] = '.';
	      dst[-2] = '.';
	      dst[-3] = '.';
	      *dst = '\0';
	      break;
	      }
	    else if ((utf8len = UTF8LEN(src)) != 0) {
	      while (utf8len--)
		  *dst++ = *src++;
	      endpos++;
	      continue;
	      }
	    else if (!active && *src == '<')
	      *dst = '(';
	    else if (!active && *src == '>')
	      *dst = ')';
	    else
	      *dst = *src;
	  if (*src == '\0')
	      break;
	  src++;
	  dst++;
	  endpos++;
	  }
      }

  {
  unsigned saved_color = UI.fgcolor;
  struct attribute_str saved_attr = UI.attributes.current;

  if (active) {
      if (!UI.attributes.valid.underline && !UI.attributes.valid.brighter)
	  UI.function.setfgcolor (MOUSE_SELECTABLE_CHARS);
	else {
	  /* Write it this way else GCC-4.6.0 uses "movl UI.function.setattribute,%eax ; call *%eax"
		see http://gcc.gnu.org/bugzilla/show_bug.cgi?id=48888 */
	  struct attribute_str newattr = UI.attributes.valid.underline? attr_underline : attr_brighter;
	  UI.function.setattribute (newattr);
	  }
      }

  UI.function.putstr (displayed);

  if (saved_color != UI.fgcolor)
      UI.function.setfgcolor (saved_color);
//  if (saved_attr != UI.attributes.current) sometimes C is boring...
  if (memcmp (&saved_attr, &UI.attributes.current, sizeof (struct attribute_str))) {
//      UI.function.setattribute (saved_attr);
      /* Write it this way else GCC-4.6.0 uses "movl UI.function.setattribute,%eax ; call *%eax" */
      struct attribute_str newattr = saved_attr;
      UI.function.setattribute (newattr);
      }
  }
  }}
#endif /* USER_SUPPORT != 0 */

#else  /* USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT) */

static const union mouse_cursor cursor_mask[2][SIZEOF_CURSOR_MASK] = {
#define _ 0
#define X 1
    {
	{{X,_,_,_,_,_,_,_}},
	{{X,X,_,_,_,_,_,_}},
	{{X,X,X,_,_,_,_,_}},
	{{X,X,X,X,_,_,_,_}},
	{{X,X,X,X,X,_,_,_}},
	{{X,X,X,X,X,X,_,_}},
	{{X,X,X,X,X,X,X,_}},
	{{X,X,X,X,X,X,X,X}},
	{{X,X,X,X,X,_,_,_}},
	{{X,X,_,X,X,_,_,_}},
	{{X,_,_,_,X,X,_,_}},
	{{_,_,_,_,X,X,_,_}},
	{{_,_,_,_,_,X,X,_}},
	{{_,_,_,_,_,X,X,_}}
    }, {
#if 0
	{{_,_,_,X,X,_,_,_}},
	{{_,_,X,X,X,X,_,_}},
	{{_,X,X,X,X,X,X,_}},
	{{X,X,X,X,X,X,X,X}},
	{{X,X,X,X,X,X,X,X}},
	{{X,X,X,X,X,X,X,X}},
	{{_,_,X,X,X,X,_,_}},
	{{_,_,X,X,X,X,_,_}},
	{{_,_,X,X,X,X,_,_}},
	{{_,_,X,X,X,X,_,_}},
	{{_,_,X,X,X,X,_,_}},
	{{_,_,X,X,X,X,_,_}},
	{{_,_,X,X,X,X,_,_}},
	{{_,_,X,X,X,X,_,_}}
#else
	{{_,_,_,X,_,_,_,_}},
	{{_,_,X,X,X,_,_,_}},
	{{_,X,X,X,X,X,_,_}},
	{{X,X,X,X,X,X,X,_}},
	{{X,X,X,X,X,X,X,_}},
	{{X,X,X,X,X,X,X,_}},
	{{_,_,X,X,X,_,_,_}},
	{{_,_,X,X,X,_,_,_}},
	{{_,_,X,X,X,_,_,_}},
	{{_,_,X,X,X,_,_,_}},
	{{_,_,X,X,X,_,_,_}},
	{{_,_,X,X,X,_,_,_}},
	{{_,_,X,X,X,_,_,_}},
	{{_,_,_,_,_,_,_,_}}
#endif
    }
#undef X
#undef _
	};

const union mouse_color_union MOUSE_reset_colors = { .colors = {
    .deflt =		brown,
    .leftbutton =	red,
    .rightbutton =	green,
    .middlebutton =	lightcyan,
    .topbutton =	cyan,
    .twobutton =	magenta,
    .invalid =		black,
    .infield =		magenta,
    }};

const struct mousefieldattr_str reset_all_fields = {
    .fullscreenfield	= 1,
    .upperfield		= 1,
    .mainfield		= 1
    };

/**
 ** The part managing the "clickable fields":
 **/

VIDEO_FCT_PREFIX(drawbutton) static void
drawbutton (coord    upperleft, coord  lowerright,
	    unsigned upperleftcolor, unsigned lowerrightrcolor,
	    unsigned fillcolor, unsigned drawbackground)
  {bound_stack();{
  /* Enlarge the button by few pixels: */
  upperleft.x -= 2;
  lowerright.x += 2;
  lowerright.y -= 1; /* do not overlap two consecutive lines */

  UI_plotHline (upperleft, lowerright.x, upperleftcolor);	/* top line */
  /* do not change VESA1 banks too often, process horizontally,
     left to right, line per line */
  for (;;) {
      upperleft.y += 1;
      if (upperleft.y >= lowerright.y)
	  break;
      UI_setpixel (upperleft, upperleftcolor);
      if (drawbackground)
	  UI_plotHline (((coord) { .x = upperleft.x + 1, .y = upperleft.y }),
			lowerright.x - 1, fillcolor);
      UI_setpixel (((coord) { .x = lowerright.x - 1, .y = upperleft.y }),
			lowerrightrcolor);
      }

  UI_plotHline (upperleft, lowerright.x, lowerrightrcolor);	/* bottom line */
  }}

/* Shall this function also manage the screen scrolling ? */
awk_farret (mouse_print_field);
VIDEO_FCT_PREFIX(mouse_print_field) static void
mouse_print_field (const char *str, struct mousefieldattr_str attr,
		unsigned short key, unsigned short row, unsigned short col)
  {bound_stack();{
  unsigned endpos = col;
  char displayed[UI.parameter.nbcol + 1];	/* including ending zero char */

  if (MOUSE.nb_field != 0 && MOUSE.fields == 0)
      PDBG ((" [%s inconsistent MOUSE structure] ", __FUNCTION__));

  /* We should not have \r, \n, \t, \b ... in str! */
  if (str == 0 || *str == '\0') {
      PDBG ((" [%s: add an empty field!] ", __FUNCTION__));
      return;
      }
    else {
      char *dst = displayed;
      const char *src = str;

      for (;;) {
	  /* 3D buttons needs a end-of-line space to draw right end button */
	  unsigned utf8len;
	  if (endpos >= sizeof (displayed)  - 1U) {
	      dst[-1] = '.';
	      dst[-2] = '.';
	      dst[-3] = '.';
	      *dst = '\0';
	      break;
	      }
	    else if ((utf8len = UTF8LEN(src)) != 0) {
	      while (utf8len--)
		  *dst++ = *src++;
	      endpos++;
	      continue;
	      }
	    else if (!attr.active && *src == '<')
	      *dst = '(';
	    else if (!attr.active && *src == '>')
	      *dst = ')';
	    else
	      *dst = *src;
	  if (*src == '\0')
	      break;
	  src++;
	  dst++;
	  endpos++;
	  }
      }
  /* get memory for new field, realloc 1st param can be 0: */
  {
  unsigned saved_color = UI.fgcolor;
  struct attribute_str saved_attr = UI.attributes.current;
  struct mousefield_str *field = 0;

  if (attr.active)
      field = REALLOC (MOUSE.fields, (MOUSE.nb_field + 1) * sizeof (MOUSE.fields[0]), "MOSfield");

  if (field != 0) {
      MOUSE.fields = field;
      MOUSE.fields[MOUSE.nb_field] = (struct mousefield_str) {
	  .attr = attr,
	  .keypress = key,
	  .upperleft = {
		.x = UI.parameter.charwidth * (attr.fullscreenfield? 0 : col),
		.y = UI.parameter.charheight * (attr.fullscreenfield? 0 : row),
		},
	  .lowerright = { /* FIXME: one pixel too big - do we care? */
		.x = UI.parameter.charwidth * (attr.fullscreenfield? UI.parameter.nbcol  : endpos),
		.y = UI.parameter.charheight * (attr.fullscreenfield? UI.parameter.nbrow : (row+1)),
		}
	  };
      field = &MOUSE.fields[MOUSE.nb_field];
      MOUSE.nb_field += 1;

      PDBG ((" [adding mouse field {%u,%u},{%u,%u} ] ",
		field->upperleft.x, field->upperleft.y,
		field->lowerright.x, field->lowerright.y));

      /* Time travel is increasingly regarded as a menace. History is being polluted.
	 ... time travel was, by its very nature, discovered simultaneously at all
	 periods of history ... */

      if (UI.attributes.valid.transparent && !attr.fullscreenfield) {
	  unsigned bgcolor, ulcolor, lrcolor;

	  /* Write 2 lines this way else GCC-4.6.0 uses "movl UI.function.setattribute,%eax ; call *%eax" */
	  struct attribute_str newattr = attr_transparent;
	  UI.function.setattribute (newattr);

	  if (field->attr.checkbox) {
	      if (UI.parameter.nbcolor <= 2)
		  bgcolor = UI.stdcolor[black];
		else if (field->attr.greyed)
		  bgcolor = CHECKBOX_INACTVE_BG;
		else
		  bgcolor = CHECKBOX_BG;
	      }
#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
	    else if (UI.parameter.nbcolor > 16) {
	      if (field->attr.mainfield)
		  bgcolor = BOTTOM_MENU_BUTTON_BG;
		else
		  bgcolor = TOP_MENU_BUTTON_BG;
	      }
#endif
	    else
	      bgcolor = UI.bgcolor;

	  if (UI.parameter.nbcolor <= 2) {
	      ulcolor = UI.stdcolor[white];
	      lrcolor = UI.stdcolor[white];
	      }
	    else if (field->attr.pressed == 0) {
	      ulcolor = UI.stdcolor[white];
	      lrcolor = UI.stdcolor[black];
//	      ulcolor = UI.stdcolor[lightgray];
//	      lrcolor = UI.stdcolor[darkgray];
	      }
	    else {
	      ulcolor = UI.stdcolor[black];
	      lrcolor = UI.stdcolor[white];
//	      ulcolor = UI.stdcolor[darkgray];
//	      lrcolor = UI.stdcolor[lightgray];
	      }
	  drawbutton (field->upperleft, field->lowerright,
		  ulcolor, lrcolor, bgcolor, 1);
	  }
	else if (!UI.attributes.valid.underline && !UI.attributes.valid.brighter)
	  UI.function.setfgcolor (MOUSE_SELECTABLE_CHARS);
	else {
	  /* Write it this way else GCC-4.6.0 uses "movl UI.function.setattribute,%eax ; call *%eax" */
	  struct attribute_str newattr = UI.attributes.valid.underline? attr_underline : attr_brighter;
	  UI.function.setattribute (newattr);
	  }
      }

  UI.function.putstr (displayed);

  if (saved_color != UI.fgcolor)
      UI.function.setfgcolor (saved_color);
//  if (saved_attr != UI.attributes.current) sometimes C is boring...
  if (memcmp (&saved_attr, &UI.attributes.current, sizeof (struct attribute_str))) {
//      UI.function.setattribute (saved_attr);
      /* Write it this way else GCC-4.6.0 uses "movl UI.function.setattribute,%eax ; call *%eax" */
      struct attribute_str newattr = saved_attr;
      UI.function.setattribute (newattr);
      }
  }
  }}

/* Removes part/all the clickable fields: */
awk_farret (mouse_reset_field);
VIDEO_FCT_PREFIX(mouse_reset_field) static void
mouse_reset_field (struct mousefieldattr_str attr)
  {bound_stack();{
  if (MOUSE.nb_field != 0 && MOUSE.fields == 0)
      PDBG ((" [%s inconsistent MOUSE structure] ", __FUNCTION__));

  if (attr.fullscreenfield)
      PDBG ((" [%s: remove all fullscreenfield] ", __FUNCTION__));
  if (attr.upperfield)
      PDBG ((" [%s: remove all upperfield] ", __FUNCTION__));
  if (attr.mainfield)
      PDBG ((" [%s: remove all mainfield] ", __FUNCTION__));

  if (MOUSE.fields != 0) {
      struct mousefield_str array[MOUSE.nb_field];
      unsigned nbstaying, cpt;

      for (nbstaying = cpt = 0; cpt < MOUSE.nb_field; cpt ++) {
	  if (MOUSE.fields[cpt].attr.fullscreenfield && attr.fullscreenfield)
	      continue;
	  if (MOUSE.fields[cpt].attr.upperfield && attr.upperfield)
	      continue;
	  if (MOUSE.fields[cpt].attr.mainfield && attr.mainfield)
	      continue;
	  array[nbstaying++] = MOUSE.fields[cpt];
	  }
      free (MOUSE.fields);
      MOUSE.fields = 0; /* needed because of the realloc(0, x) */
      MOUSE.nb_field = nbstaying;
      if (nbstaying != 0) {
	  PDBG ((" [%s: kept %u clickable fields] ", __FUNCTION__, nbstaying));
	  MOUSE.fields = MALLOC (nbstaying * sizeof (MOUSE.fields[0]), "MOfields");
	  if (MOUSE.fields == 0) {
	      PDBG ((" [%s failed malloc] ", __FUNCTION__));
	      MOUSE.nb_field = 0;
	      return;
	      }
	  for (cpt = 0; cpt < nbstaying; cpt ++)
	      MOUSE.fields[cpt] = array[cpt];
	  }
      }
  }}

/**
 ** The display mouse cursor function:
 **/
VIDEO_FCT_PREFIX(mouse_move) static unsigned
mouse_move (int deltax, int deltay, struct mouse_button_str newbutton)
  {bound_stack();{
  unsigned field;
  unsigned char draw_infield_mouse = 0, color_index = MOUSE.all_colors.colors.deflt;
  coord newpos;

  deltax += MOUSE.position.x;
  if (deltax < 0)
      newpos.x = 0;
    else if (deltax > MOUSE.maximum.x)
      newpos.x = MOUSE.maximum.x;
    else
      newpos.x = deltax;

  deltay = MOUSE.position.y - deltay;
  if (deltay < 0)
      newpos.y = 0;
    else if (deltay > MOUSE.maximum.y)
      newpos.y = MOUSE.maximum.y;
    else
      newpos.y = deltay;

  for (field = 0; field < MOUSE.nb_field; field++) {
      struct mousefield_str *fieldptr = &MOUSE.fields[field];

      /* This find the _first_ field in the list, if you have a "grab-all-click"
	field (with fullscreenfield bit set), you probably want it as the last,
	so each time you add a field, you first delete all fields having
	fullscreenfield, then add yours, the re-add the background field
	The fullscreenfield bit do not show the cursor as "on a field" */
      if (   newpos.x >= fieldptr->upperleft.x
	  && newpos.x <  fieldptr->lowerright.x
	  && newpos.y >= fieldptr->upperleft.y
	  && newpos.y <  fieldptr->lowerright.y) {
	  if (!fieldptr->attr.fullscreenfield) {
	      draw_infield_mouse = 1;
	      /* Changing the color of the graphic mouse when inside a field
		looks bad, anyway we already change its shape */
	      if (!UI.parameter.attr.isgraphic)
		  color_index = MOUSE.all_colors.colors.infield;
	      }
	  if (!newbutton.left && !newbutton.right && !newbutton.middle)
	      field = 0xFFFFFFFF; /* do not treat it now */
	  break;
	  }
      }

  if (field >= MOUSE.nb_field && !MOUSE.button.notfunny) {
      if (newbutton.middle)
	  color_index = MOUSE.all_colors.colors.middlebutton;
	else if (newbutton.top)
	  color_index = MOUSE.all_colors.colors.topbutton;
	else if (newbutton.left && newbutton.right)
	  color_index = MOUSE.all_colors.colors.twobutton;
	else if (newbutton.left)
	  color_index = MOUSE.all_colors.colors.leftbutton;
	else if (newbutton.right)
	  color_index = MOUSE.all_colors.colors.rightbutton;
      }

  if (UI.parameter.attr.isgraphic) {
      if (   newpos.x != MOUSE.position.x
	  || newpos.y != MOUSE.position.y
	  || !MOUSE_button_equal (MOUSE.button, newbutton)) {
	  unsigned *arrayptr;
	  union mouse_cursor mask;
	  const union mouse_cursor *maskptr;
	  coord xy;

	  if (!MOUSE.button.hidden) {
	      PDBG (("hide graphic mouse at %u,%u; ", MOUSE.position.x, MOUSE.position.y));
	      arrayptr = MOUSE.savearray;
	      for  (maskptr = cursor_mask[MOUSE.curmaskindex], xy.y = MOUSE.position.y;
		    maskptr < &cursor_mask[MOUSE.curmaskindex][SIZEOF_CURSOR_MASK];
		    maskptr++, xy.y++) {
		  for (mask.group = 1, xy.x = MOUSE.position.x;
		       mask.group != 0;
		       mask.group <<= 1, xy.x++) {
		      if (maskptr->group & mask.group)
			  UI_setpixel (xy, *arrayptr++);
		      }
		  }
	      }

	  /* Before re-printing the mouse: */
	  if (   field < MOUSE.nb_field
	      && UI.attributes.valid.transparent) {
	      unsigned ulcolor, lrcolor;

	      if (MOUSE.fields[field].attr.pressed) {
		  ulcolor = UI.stdcolor[white];
		  lrcolor = UI.stdcolor[black];
//		  ulcolor = UI.stdcolor[lightgray];
//		  lrcolor = UI.stdcolor[darkgray];
		  }
		else {
		  ulcolor = UI.stdcolor[black];
		  lrcolor = UI.stdcolor[white];
//		  ulcolor = UI.stdcolor[darkgray];
//		  lrcolor = UI.stdcolor[lightgray];
		  }
	      drawbutton (MOUSE.fields[field].upperleft,
			  MOUSE.fields[field].lowerright,
			  ulcolor, lrcolor, 0, 0);
	      }

	  if (!newbutton.hidden) {
	      MOUSE.curmaskindex = !!draw_infield_mouse;
	      PDBG (("Redraw graphic mouse nb %u at %u,%u\r\n", MOUSE.curmaskindex, newpos.x, newpos.y));
	      arrayptr = MOUSE.savearray;
	      for (maskptr = cursor_mask[MOUSE.curmaskindex], xy.y = newpos.y;
		   maskptr < &cursor_mask[MOUSE.curmaskindex][SIZEOF_CURSOR_MASK];
		   maskptr++, xy.y++) {
		  for (mask.group = 1, xy.x = newpos.x;
		       mask.group != 0;
		       mask.group <<= 1, xy.x++) {
		      if (maskptr->group & mask.group) {
			  *arrayptr++ = UI_getpixel (xy);
			  UI_setpixel (xy, UI.stdcolor[color_index]);
			  }
		      }
		  }
	      }
	  }
      }
    else {
      coord oldM = {
	  .x = MOUSE.position.x / UI.parameter.charwidth,
	  .y = MOUSE.position.y / UI.parameter.charheight
	  }, newM = {
	  .x = newpos.x / UI.parameter.charwidth,
	  .y = newpos.y / UI.parameter.charheight
	  };

      if (   oldM.x != newM.x
	  || oldM.y != newM.y
	  || !MOUSE_button_equal (MOUSE.button, newbutton)) {
	  PDBG (("Redraw text mouse at %u,%u\r\n", newM.x, newM.y));
#ifdef MOUSE_ON_VT
	  if (BOOT1_COM_port() >= 0) {
	      /* nothing to clear the mouse */
	      if (!newbutton.hidden)
		  SERIAL_setcursor (newM.y, newM.x);
		else
		  SERIAL_setcursor (MOUSE.curs_row, MOUSE.curs_col);
	      }
	    else
#endif
	      {
	      unsigned newpixel;

	      if (!MOUSE.button.hidden)
		  UI_setpixel (oldM, *MOUSE.savearray);
	      *MOUSE.savearray = UI_getpixel (newM);
#if USER_SUPPORT & VGA_SUPPORT
	      if (UI.parameter.nbcolor == 2 || UI.parameter.nbcolor == 0) {
		  if (UI.parameter.attr.null_del_available)
		      newpixel = '\0'; /* fully inverted char */
		    else if (!UI.parameter.attr.ansi_font)
		      newpixel = (unsigned char)'\333'; /* '\xDB', background ignored */
		    else
		      newpixel = '@'; /* no good choice anyway */
		  if (UI.parameter.nbcolor == 0)
		      newpixel |= (draw_infield_mouse)? 0x0F00 : 0x0700;
		  }
		else
#endif
		  newpixel = (UI.stdcolor[color_index] << 12) | (*MOUSE.savearray & 0xFFF);
	      if (!newbutton.hidden)
		  UI_setpixel (newM, newpixel);
	      }
	  }
      }

  MOUSE.button = newbutton;
  MOUSE.position = newpos;
  return field;
  }}

/**
 ** The pooling function:
 **/
VIDEO_FCT_PREFIX(MOUSE_poll) unsigned
MOUSE_poll (void)
  {bound_stack();{
  int deltax = deltax, deltay = deltay; /* inited b4 used */
  unsigned field;
  struct mouse_button_str newbutton;
#if USER_SUPPORT & JOYSTICK_SUPPORT
  unsigned tmp, marker = 0;
#endif

  if (MOUSE.maximum.x == 0) {
      PDBG ((" [%s shall not be called when MOUSE.maximum.x == 0] ", __FUNCTION__));
      return 0xFFFFFFFF;
      }

  if (MOUSE.nb_field != 0 && MOUSE.fields == 0)
      PDBG ((" [%s inconsistent MOUSE structure] ", __FUNCTION__));

#if USER_SUPPORT & JOYSTICK_SUPPORT
  if (UTIL.joystick_present && MOUSE.joystick.last_nbtick != (typeof (MOUSE.joystick.last_nbtick))(tmp = _BIOS_nbtick())) {
      unsigned short Ax, Ay, Bx, By;
      struct JoystickButton_str button;

      MOUSE.joystick.last_nbtick = tmp;	/* only one sample per tick */
      PDBG (("%s: treat joystick ", __FUNCTION__));

      if (JoystickPosition (&Ax, &Ay, &Bx, &By) != 0) {
	  PDBG (("JoystickPosition() error, Ax = 0x%X\r\n", Ax));
	  }
	else if ((Ax | Ay | Bx | By) == 0) {
	  PDBG (("Ax, Ay, Bx, By all zero, no joystick there\r\n"));
	  }
	else if (JoystickButton (&button) != 0) {
	  PDBG (("JoystickButton() error\r\n"));
	  }
	else {
	  if (Ax < MOUSE.joystick.Xmin) {
	      MOUSE.joystick.Xmin = Ax;
	      PDBG (("[recalibrate Xmin = %u] ", Ax));
	      }
	  if (Ax > MOUSE.joystick.Xmax) {
	      MOUSE.joystick.Xmax = Ax;
	      PDBG (("[recalibrate Xmax = %u] ", Ax));
	      }
	  if (Ay < MOUSE.joystick.Ymin) {
	      MOUSE.joystick.Ymin = Ay;
	      PDBG (("[recalibrate Ymin = %u] ", Ay));
	      }
	  if (Ay > MOUSE.joystick.Ymax) {
	      MOUSE.joystick.Ymax = Ay;
	      PDBG (("[recalibrate Ymax = %u] ", Ay));
	      }
	  PDBG (("buttons: "));
	  newbutton = (struct mouse_button_str) {
	      .left   = !button.button_C,
	      .right  = !button.button_B,
	      .middle = !button.button_A,
	      .top    = !button.button_D
	      };
	  if (newbutton.left)
	      PDBG (("C=left "));
	  if (newbutton.right)
	      PDBG (("B=right "));
	  if (newbutton.middle)
	      PDBG (("A=middle "));
	  if (newbutton.left)
	      PDBG (("D=unmapped "));
	  if (   MOUSE_button_equal (newbutton, MOUSE.joystick.button)
//	      && JoystickButtonEqual (button, MOUSE.joystick.DefaultButton)
	      && (Ax > MOUSE.joystick.Xmid - 8 && Ax < MOUSE.joystick.Xmid + 8)
	      && (Ay > MOUSE.joystick.Ymid - 8 && Ay < MOUSE.joystick.Ymid + 8)) {
	      PDBG (("; joystick standby, treat mouse\r\n"));
	      MOUSE.joystick.accelerate = 3;
	      }
	    else {
	      /* this is _not_ MOUSE.button, possible sharing joystick/mouse: */
	      MOUSE.joystick.button = newbutton;
	      PDBG (("; accelerate %u: ", MOUSE.joystick.accelerate));

	      deltax = (int)Ax - (int)MOUSE.joystick.Xmid;
	      PDBG (("deltax = %d ", deltax));
	      if (MOUSE.joystick.Xmax - MOUSE.joystick.Xmin != 0) {
		  deltax = (deltax * MOUSE.joystick.accelerate) / (MOUSE.joystick.Xmax - MOUSE.joystick.Xmin);
		  PDBG (("-> %d ", deltax));
		  }
	      deltay = (int)MOUSE.joystick.Ymid - (int)Ay;
	      PDBG (("deltay = %d ", deltay));
	      if (MOUSE.joystick.Ymax - MOUSE.joystick.Ymin != 0) {
		  deltay = (deltay * MOUSE.joystick.accelerate) / (MOUSE.joystick.Ymax - MOUSE.joystick.Ymin);
		  PDBG (("-> %d ", deltay));
		  }
	      MOUSE.joystick.accelerate ++;
	      marker = 1;
	      }
	  }
      }

  if (marker)
      PDBG (("joystick active: "));
    else
#endif
#if USER_SUPPORT & BIOS_MOUSE_SUPPORT
	 if (MOUSE.type == MOUSE_PS2) {

      if (!MOUSE.data.PS2.irq_trigged) {
	  //PDBG (("%s: treat PS2 no report\r\n", __FUNCTION__));
	  return 0xFFFFFFFF;
	  }
/* ctmouse.asm:
		; stack for non-wheel mice: ... - - Y - X - BTN -
		; stack for wheel mice:     ... - - W - Y - BTN X
		; flags: (yext) (xext) ysign xsign 1 btn3 btn1 btn2
		; "ext" flag can be used to trigger "xor value,100h"
PLAIN:
		mov	al,[bp+_ARG_OFFS_+6]	; buttons and flags
		mov	al,[bp+_ARG_OFFS_+4]	; AX=X movement
		mov	cl,[bp+_ARG_OFFS_+2]	; CX=Y movement
PS2WHEEL:	; handler based on public domain code from Konstantin Koll
		; old KoKo code used only ext, not sign, ok on all but Alps
		mov	al,[bp+_ARG_OFFS_+6]	; buttons and flags
		mov	al,[bp+_ARG_OFFS_+7]	; AX=X movement <--
		mov	cl,[bp+_ARG_OFFS_+4]	; CX=Y movement <--
		mov	ah,[bp+_ARG_OFFS_+2]	; AH=Wheel data <--
  If bad USB BIOS emulation, seems cannot reset mouse properly, some
  mouse stay in "weel" mode but X zero'ed out, only ... - - W - Y - BTN -
*/

#if 0
printf ("nbIrq %u, 0x%X, 0x%X, 0x%X, 0x%X\r\n", MOUSE.data.PS2.irq_trigged, ((short *)&MOUSE.data)[0], ((short *)&MOUSE.data)[1], ((short *)&MOUSE.data)[2], ((short *)&MOUSE.data)[3]);
if (MOUSE.data.PS2.device_id != 0) {
  if (MOUSE.data.PS2.z_zero || MOUSE.data.PS2.z_reserved || MOUSE.data.PS2.y_reserved || MOUSE.data.PS2.x_reserved || MOUSE.data.PS2.reservedbyte)
      printf ("z_zero %u, z_reserved 0x%X, y_reserved 0x%X, x_reserved 0x%X, reservedbyte 0x%X\r\n",
	MOUSE.data.PS2.z_zero, MOUSE.data.PS2.z_reserved, MOUSE.data.PS2.y_reserved, MOUSE.data.PS2.x_reserved, MOUSE.data.PS2.reservedbyte);
  printf ("button: left %u, middle %u, right %u, 4 %u, 5 %u, z_data %d, y_data %d, x_data %d\r\n",
	MOUSE.data.PS2.left_button, MOUSE.data.PS2.middle_button, MOUSE.data.PS2.right_button, MOUSE.data.PS2.z_button4, MOUSE.data.PS2.z_button5,
	MOUSE.data.PS2.z_data, MOUSE.data.PS2.y_data, MOUSE.data.PS2.x_data);
  printf ("        reservedbit %u, x_negative %u, y_negative %u, x_overflow %u, y_overflow %u\r\n",
	MOUSE.data.PS2.reservedbit, MOUSE.data.PS2.x_negative, MOUSE.data.PS2.y_negative, MOUSE.data.PS2.x_overflow, MOUSE.data.PS2.y_overflow);
  MOUSE.data.PS2 = (struct PS2_struct) {device_id: MOUSE.data.PS2.device_id}; /* reset MOUSE.data.PS2.irq_trigged */
  return 0xFFFFFFFF;
  }
#endif

      PDBG (("%s: treat PS2 ", __FUNCTION__));

      if (!MOUSE.data.PS2.reservedbit) {
	  if (MOUSE.data.PS2.z_zero || MOUSE.data.PS2.z_reserved || MOUSE.data.PS2.y_reserved || MOUSE.data.PS2.x_reserved || MOUSE.data.PS2.reservedbyte)
	      PDBG (("invalid format: z_zero %u, z_reserved 0x%X, y_reserved 0x%X, x_reserved 0x%X, reservedbyte 0x%X\r\n",
		MOUSE.data.PS2.z_zero, MOUSE.data.PS2.z_reserved, MOUSE.data.PS2.y_reserved, MOUSE.data.PS2.x_reserved, MOUSE.data.PS2.reservedbyte));
	  PDBG (("invalid format: button: left %u, middle %u, right %u, 4 %u, 5 %u, z_data %d, y_data %d, x_data %d\r\n",
		MOUSE.data.PS2.left_button, MOUSE.data.PS2.middle_button, MOUSE.data.PS2.right_button, MOUSE.data.PS2.z_button4, MOUSE.data.PS2.z_button5,
		MOUSE.data.PS2.z_data, MOUSE.data.PS2.y_data, MOUSE.data.PS2.x_data));
	  PDBG (("invalid format:        reservedbit %u, x_negative %u, y_negative %u, x_overflow %u, y_overflow %u\r\n",
		MOUSE.data.PS2.reservedbit, MOUSE.data.PS2.x_negative, MOUSE.data.PS2.y_negative, MOUSE.data.PS2.x_overflow, MOUSE.data.PS2.y_overflow));
	  MOUSE.data.PS2 = (struct PS2_struct) {device_id: MOUSE.data.PS2.device_id}; /* reset MOUSE.data.PS2.irq_trigged */
	  return 0xFFFFFFFF;
	  }

      if (MOUSE.data.PS2.x_overflow || MOUSE.data.PS2.y_overflow) {
	  PDBG (("overflow\r\n"));
	  MOUSE.data.PS2 = (struct PS2_struct) {device_id: MOUSE.data.PS2.device_id}; /* reset MOUSE.data.PS2.irq_trigged */
	  return 0xFFFFFFFF;
	  }

      newbutton = (struct mouse_button_str) {
	  .left   = MOUSE.data.PS2.left_button,
	  .right  = MOUSE.data.PS2.right_button,
	  .middle = MOUSE.data.PS2.middle_button,
	  .top = MOUSE.data.PS2.z_button4,
	  .bottom = MOUSE.data.PS2.z_button5
	  };
      deltax = MOUSE.data.PS2.x_data;
      deltay = MOUSE.data.PS2.y_data;

      MOUSE.data.PS2 = (struct PS2_struct) {device_id: MOUSE.data.PS2.device_id}; /* reset MOUSE.data.PS2.irq_trigged */
      }
    else
#endif /* BIOS_MOUSE_SUPPORT */
#if USER_SUPPORT & SERIAL_MOUSE_SUPPORT
     /* While we are here, do you know why the serial mouse disappeared at some point
	in the PC history, to be replaced by PS/2 mouse?
	Try it, serial mouse are a lot smouthier and linear compared to a PS/2 one,
	whatever advertisement you read	about mouse precision on the packaging.
	The real reason is that serial mouse was not behaving correctly on some PCs,
	nobody found the real problem... in fact a big company produced a video
	card, with its own video BIOS. To debug this new video BIOS, they used
	the PC serial line. They forgot to disable the debugging messages before
	shipping to costumer - and that ISA board did not have a FLASH but ROM,
	so they could not update... The characters sent randomly to the first serial
	line where seriously bothering the mouse on those PCs...
	Modems were also a serious problem on those PCs: the modem driver was not
	working... You could not get the source of the video BIOS neither. */
	 if (MOUSE.type != MOUSE_NONE) {
      if (SERIAL_DATA_RECEIVED (MOUSE.data.serial)) {
	  unsigned char received = SERIAL_DATA_GET (MOUSE.data.serial);
	  PDBG (("%s: treat serial received 0x%X ", __FUNCTION__, received));

	  if ((MOUSE.type & MOUSE_SERIAL_MASK) == MOUSE_SERIALM_COM1) {
	      PDBG (("interpret MMdata: "));
	      /* Then we have to wait a pattern for first byte: */
	      if (MOUSE.data.serial.cpt == 0 && (received & 0xF8) != 0x80) {
		  PDBG (("reject first byte 0x%X: no sync bit 0x80\r\n", received));
		  return 0xFFFFFFFF;
		  }
	      MOUSE.data.serial.received.bytes[MOUSE.data.serial.cpt++] = received;
	      if (MOUSE.data.serial.cpt < 5) {
		  PDBG (("waiting end of frame\r\n"));
		  return 0xFFFFFFFF;
		  }

	      PDBG (("treating frame: "));
	      MOUSE.data.serial.cpt = 0;
	      newbutton = (struct mouse_button_str) {
		  .left   = !MOUSE.data.serial.received.mmdata.left_button,
		  .right  = !MOUSE.data.serial.received.mmdata.right_button,
		  .middle = !MOUSE.data.serial.received.mmdata.middle_button
		  };
	      /* small acceleration: * 2 */
	      deltax = (int)MOUSE.data.serial.received.mmdata.x
			+ 2 * (int)MOUSE.data.serial.received.mmdata.deltax;
	      deltay = (int)MOUSE.data.serial.received.mmdata.y
			+ 2 * (int)MOUSE.data.serial.received.mmdata.deltay;
	      }
	    else {
	      unsigned char middleval = 0, lost_report = 0;
	      PDBG (("interpret STDdata: "));

	      /* Then we have a resync bit: */
	      if (received & 0x40) {
		  PDBG (("[resync"));
		  if (MOUSE.data.serial.cpt != 0 && MOUSE.data.serial.cpt <= 2) {
		      /* maybe added protocol here: */
		      PDBG ((" ignore short frame 0x%X", MOUSE.data.serial.received.bytes[0]));
		      if (MOUSE.data.serial.cpt == 2)
			  PDBG ((", 0x%X", MOUSE.data.serial.received.bytes[1]));
		      }
		  PDBG (("] "));
		  if (MOUSE.data.serial.cpt == 3 && MOUSE.button.middle) {
		      /* we've lost the release-middle event, generate exactly
			the previous report with middle button up: */
		      PDBG (("generating lost frame: "));
		      lost_report = 1;
		      }
		  MOUSE.data.serial.cpt = 0;
		  }
		else if (MOUSE.data.serial.cpt >= 3) {
		  if (MOUSE.data.serial.cpt == 3 && (received & ~0x20) == 0) {
		      if (received == 0)
			  PDBG (("[Logitech extension middle button up] "));
			else {
			  PDBG (("[Logitech extension middle button down] "));
			  middleval = 1;
			  /* We should switch to this protocol as soon as we recognise
				a fourth char, but recognising char zero is not
				special enough... */
			  if ((MOUSE.type & MOUSE_SERIAL_MASK) == MOUSE_SERIAL_COM1) {
			      PDBG (("So switching to serial 3 buttons.\r\n"));
			      MOUSE.type &= ~MOUSE_SERIAL_MASK;
			      MOUSE.type |= MOUSE_SERIAL3_COM1;
			      }
			  }
		      }
		    else {
		      PDBG (("reject serial char No %u received: 0x%X\r\n",
				MOUSE.data.serial.cpt, received));
		      MOUSE.data.serial.cpt = 4; /* stuck there up to resync */
		      return 0xFFFFFFFF;
		      }
		  }
	      MOUSE.data.serial.received.bytes[MOUSE.data.serial.cpt++] = received;

	      if (  !lost_report
		  && MOUSE.data.serial.cpt != 3 + (middleval | MOUSE.button.middle)) {
		  PDBG (("waiting end of frame\r\n"));
		  return 0xFFFFFFFF;
		  }
		else {
		  union {
		      struct {
			  unsigned char lsb : 6;
			  unsigned char msb : 2;
			  } fields;
		      signed char all;
		      } tmp;

		  PDBG (("treating frame: "));
		  tmp.fields.msb = MOUSE.data.serial.received.stddata.xmsb;
		  tmp.fields.lsb = MOUSE.data.serial.received.stddata.x;
		  deltax = tmp.all;
		  tmp.fields.msb = MOUSE.data.serial.received.stddata.ymsb;
		  tmp.fields.lsb = MOUSE.data.serial.received.stddata.y;
		  deltay = - tmp.all;
		  newbutton = (struct mouse_button_str) {
		      .left   = MOUSE.data.serial.received.stddata.left_button,
		      .right  = MOUSE.data.serial.received.stddata.right_button,
		      .middle = middleval
		      };
		  }
	      }
	  }
	else /* No data received: */
	  return 0xFFFFFFFF;
      }
    else
#endif /* SERIAL_MOUSE_SUPPORT */
      return 0xFFFFFFFF;

  PDBG (("deltax = %d, deltay = %d\r\n", deltax, deltay));
  if ((field = mouse_move (deltax, deltay, newbutton)) < MOUSE.nb_field) {
      unsigned returned = MOUSE.fields[field].keypress;
      struct shift_flags_str flags = _BIOS_get_shift_flags();
      if (flags.right_shift_key_pressed || flags.left_shift_key_pressed) {
	  unsigned short cptfctkey = nbof(all_fct_keycode) + 1;
	  while (cptfctkey-- > 0)
	      if (returned == all_fct_keycode[cptfctkey]) {
		  returned = all_sfct_keycode[cptfctkey];
		  break;
		  }
	  }
      return returned;
      }
    else
      return 0xFFFFFFFF;
  }}

/**
 ** The start/stop:
 **/
VIDEO_FCT_PREFIX(MOUSE_start) void
MOUSE_start (void)
  {bound_stack();{
  unsigned char status;

  PDBG (("\r\n%s ", __FUNCTION__));
  if (UI.parameter.charwidth == 0 || UI.parameter.charheight == 0) {
      /* We may have switch to an invalid mode since MOUSE_init() */
      PDBG (("Cannot start mouse, charwidth = %u, charheight = %u\r\n",
		UI.parameter.charwidth, UI.parameter.charheight));
      return;
      }

#ifdef MOUSE_ON_VT
  if (   BOOT1_COM_port() >= 0
      && SERIAL_getcursor (&MOUSE.curs_row, &MOUSE.curs_col) != 0) {
      PDBG (("disabling mouse on VT due to error in getcursor\r\n"));
      return;
      }
#endif

  if (MOUSE.maximum.x != 0) {
      PDBG (("mouse already started!\r\n"));
      return;
      }

  /* Just in case we try to display before the end of the function,
     and because we will switch from hidden to visible: */
  MOUSE.button = (struct mouse_button_str){ .hidden = 1 };

  if (MOUSE.type == MOUSE_PS2 && ((status = _PS_enable(1)) != 0)) {
      PDBG (("PS2 enable error, status = 0x%X, cannot start mouse\r\n", status));
      return;
      }

  if (UI.parameter.attr.isgraphic) {
      MOUSE.maximum.x = UI.parameter.width - (8 * sizeof(cursor_mask[0][0].group));
      MOUSE.maximum.y = UI.parameter.height - SIZEOF_CURSOR_MASK;
      }
    else {
      MOUSE.maximum.x = UI.parameter.width * UI.parameter.charwidth - 1;
      MOUSE.maximum.y = UI.parameter.height * UI.parameter.charheight - 1;
      }

  PDBG (("max.x = %u, max.y = %u, ", MOUSE.maximum.x, MOUSE.maximum.y));
  if (MOUSE.position.x == 0 || MOUSE.position.y == 0) { /* that is just after MOUSE_init() */
      MOUSE.position = MOUSE.maximum;
      }
    else {
      MOUSE.position.x *= UI.parameter.charwidth;
      MOUSE.position.x += UI.parameter.charwidth/2;
      MOUSE.position.y *= UI.parameter.charheight;
      MOUSE.position.y += UI.parameter.charheight/2;
      if (MOUSE.position.x > MOUSE.maximum.x - UI.parameter.charwidth/2)
	  MOUSE.position.x = MOUSE.maximum.x - UI.parameter.charwidth/2;
      if (MOUSE.position.y > MOUSE.maximum.y - UI.parameter.charheight/2)
	  MOUSE.position.y = MOUSE.maximum.y - UI.parameter.charheight/2;
      }

  mouse_move (0, 0, (struct mouse_button_str) {});
  if (MOUSE.type != MOUSE_PS2)
      PDBG (("serial/joystick started\r\n"));
    else
      PDBG (("BIOS started\r\n"));
  }}

VIDEO_FCT_PREFIX(MOUSE_stop) void
MOUSE_stop (void)
  {bound_stack();{
  unsigned char status;

  PDBG (("%s: ", __FUNCTION__));
  if (MOUSE.maximum.x == 0) {
      PDBG (("mouse already stopped!\r\n"));
      return;
      }

  if (MOUSE.type == MOUSE_PS2 && ((status = _PS_enable(0)) != 0))
      PDBG (("!PS2 disable error, status = 0x%X!, continuing ", status));

  mouse_move (0, 0, (struct mouse_button_str) { .hidden = 1 });

  MOUSE.position.x /= UI.parameter.charwidth;
  MOUSE.position.y /= UI.parameter.charheight;
  /* will keep mouse at bottom right if it is there when definition increases */
  if (MOUSE.position.x >= UI.parameter.nbcol - 2)
      MOUSE.position.x = 0xFFFFU / 8; /* over limits */
  if (MOUSE.position.y >= UI.parameter.nbrow - 2)
      MOUSE.position.y = 0xFFFFU / 16; /* over limits */

  MOUSE.maximum = ((coord) {0,0});

  PDBG (("%s: OK\r\n", __FUNCTION__));
  }}

/**
 ** The initialisation:
 **/
#if USER_SUPPORT & BIOS_MOUSE_SUPPORT
VIDEO_FCT_PREFIX(MOUSE_PS2_init) static inline unsigned
MOUSE_PS2_init (void)
  {
  struct PS_resetID mouseconf;
  unsigned char status, protocol_nbbytes = 3; /* protocol_nbbytes max 4 for our assembly */
  extern char PS2_mouse_callback[];

//farptr ebda_seg = _BIOS_getEBDA_segment();
//printf ("b4init ebda 0x%X, farcallptr 0x%X flag1 0x%X flag2 0x%X\r\n", ebda_seg, peekl(ebda_seg+0x22), peekb(ebda_seg+0x26), peekb(ebda_seg+0x27)); _BIOS_getkey();

  PDBG (("%s: MOUSE: ", __FUNCTION__));
  if ((status = _PS_reset(&mouseconf)) != 0) {
      PDBG (("reset error, status = 0x%X\r\n", status));
      return 1;
      }
  PDBG (("device ID: 0x%X, reset value: 0x%X, ", mouseconf.device_id, mouseconf.reset_value));
  // Compatible mouse: device_id=0, reset_value=0xAA
  // one wireless mouse (5 buttons, scrollweel): device_id=0x9F, reset_value=0xE2 (do not work in compatible mode)
  MOUSE.data.PS2.device_id = mouseconf.device_id;

#if 0
//printf ("_PS_reset device ID: 0x%X, reset value: 0x%X, ", mouseconf.device_id, mouseconf.reset_value);
  if (1 || MOUSE.data.PS2.device_id == 0) {
      unsigned char device;
      /* Sequence to enter intellimouse: */
      if ((status = _PS_set_sampling(6 /* 200/s */)) != 0)
	  PDBG (("set sampling 200/s error, status = 0x%X, ", status));
      else if ((status = _PS_set_sampling(5 /* 100/s */)) != 0)
	  PDBG (("set sampling 100/s error, status = 0x%X, ", status));
      else if ((status = _PS_set_sampling(4 /* 80/s */)) != 0)
	  PDBG (("set sampling 80/s error, status = 0x%X, ", status));
      else if ((status = _PS_get_type(&device)) != 0)
	  PDBG (("get type error, status = 0x%X, ", status));
      else if (device == 0x03) {
	  MOUSE.data.PS2.device_id = device;
	  /* Sequence to enter Intellimouse Explorer: */
	  if ((status = _PS_set_sampling(6 /* 200/s */)) != 0)
	      PDBG (("set sampling 200/s error, status = 0x%X, ", status));
	  else if ((status = _PS_set_sampling(6 /* 200/s */)) != 0)
	      PDBG (("set sampling 200/s error, status = 0x%X, ", status));
	  else if ((status = _PS_set_sampling(4 /* 80/s */)) != 0)
	      PDBG (("set sampling 80/s error, status = 0x%X, ", status));
	  else if ((status = _PS_get_type(&device)) != 0)
	      PDBG (("get type error, status = 0x%X, ", status));
	  else if (device == 0x04)
	      MOUSE.data.PS2.device_id = device;
	  }
      }
//printf (" device_id 0x%X\r\n", MOUSE.data.PS2.device_id); _BIOS_getkey();

//  if (MOUSE.data.PS2.device_id != 0)
//      protocol_nbbytes = 4; // doesn't work, locks down kbd & mouse...
#endif

asm (
"	jmp 1f								\n"
"PS2_mouse_callback:							\n"
"	pushl	%ds							\n"
"	pushl	%eax							\n"
"	movw	%cs,%ax							\n"
#ifdef CALLING_FROM_USERSEG
"	subw	$usercodeseg,%ax					\n"
#endif
"	addw	$deltaseg,%ax	# not necessary if no CODE_SEGMENT	\n"
"	movw	%ax,%ds							\n"
#if 1
"	movl	12(%esp),%eax						\n"
"	movl	%eax,dataPS2						\n"
"	movl	16(%esp),%eax						\n"
"	movl	%eax,dataPS2 + 4					\n"
#else
"	pushl	%ebx				\n"
"	pushl	%ecx				\n"
"	pushl	%edx				\n"
"	mov	$0xC209,%eax			\n"
"	int	$0x15	# Pointing device read	\n"
"	mov	%bl,dataPS2+6			\n"
"	mov	%cl,dataPS2+4			\n"
"	mov	%dl,dataPS2+2			\n"
"	popl	%edx				\n"
"	popl	%ecx				\n"
"	popl	%ebx				\n"
#endif
//"	movb	$1,dataPS2 + 8	# irq_trigged				\n"
"	incb	dataPS2 + 8	# irq_trigged				\n"
"	popl	%eax							\n"
"	popl	%ds							\n"
"	lretw								\n"
"1:									\n"
	);

  if ((status = _PS_set_handler(0x10000 * getcs() + (unsigned)PS2_mouse_callback)) != 0) {
      PDBG (("set driver error, status = 0x%X\r\n", status));
      return 3;
      }

  if ((status = _PS_initialise(protocol_nbbytes)) != 0) {
      PDBG (("init error, status = 0x%X\r\n", status));
      return 2;
      }

  if ((status = _PS_set_sampling(5 /* 100/s */)) != 0)
      PDBG (("set sampling error, status = 0x%X, ", status));

  if ((status = _PS_set_resolution(3 /* 8/mm */)) != 0)
      PDBG (("set resolution error, status = 0x%X, ", status));

//unsigned char device;
//if ((status = _PS_get_type(&device)) != 0)
//    PDBG (("get type error, status = 0x%X, ", status));
//printf ("get type done status 0x%X device 0x%X ebda 0x%X, farcallptr 0x%X flag1 0x%X flag2 0x%X\r\n", status, device, ebda_seg, peekl(ebda_seg+0x22), peekb(ebda_seg+0x26), peekb(ebda_seg+0x27)); _BIOS_getkey();

  PDBG (("BIOS mouse ready.\r\n"));

  MOUSE.type = MOUSE_PS2;
  asm volatile (" dataPS2 = %c0 ": : "i" (&MOUSE.data.PS2));
  return 0;
  }

#endif /* BIOS_MOUSE_SUPPORT */

#if USER_SUPPORT & SERIAL_MOUSE_SUPPORT
VIDEO_FCT_PREFIX(MOUSE_SERIAL_init) static inline unsigned
MOUSE_SERIAL_init (unsigned nb_serial_port)
  {
  unsigned short cpt, last;
  /* inline */ const struct serial_setup_str
      mouse_std_setup = {
		.word_length	= serial_7bit,
		.stop_bit	= serial_1stopbit,
		.parity		= serial_none,
		.speed		= B1200b
		},
      mouse_mm_setup = {
		.word_length	= serial_8bit,
		.stop_bit	= serial_1stopbit,
		.parity		= serial_none,
		.speed		= B1200b
		};
  /* inline */ const struct serial_modem_str
		setmodem   = { .dtr = 1, .rts = 1 },
		restmodem  = { .dtr = 1, .rts = 0 },
		clearmodem = { .dtr = 0, .rts = 0 };

  PDBG (("%s: called with nb_serial_port = %u.\r\n", __FUNCTION__, nb_serial_port));

  if (nb_serial_port > 4)
      last = 4; /* due to the inb(), due to BIOS hardware flow control */
    else
      last = nb_serial_port;

  for (cpt = 0; cpt < last; cpt++) {
      unsigned short ioaddr = peekw (0x00400000 + (2*cpt));
      unsigned char  i, j, k, array[5];
      _BIOS_serialstatus status;

      if ((unsigned short)(BOOT1_COM_port()) == cpt) {
	  PDBG (("Do not check COM%u, it is stdout\r\n", cpt));
	  continue;
	  }
#if (DEBUG & DEBUG_OUTPUT) == COM1
      if (cpt == 0) {
	  PDBG (("Do not check COM1, it is stderr\r\n"));
	  continue;
	  }
#elif (DEBUG & DEBUG_OUTPUT) == COM2
      if (cpt == 1) {
	  PDBG (("Do not check COM2, it is stderr\r\n"));
	  continue;
	  }
#elif (DEBUG & DEBUG_OUTPUT) == COM3
      if (cpt == 2) {
	  PDBG (("Do not check COM3, it is stderr\r\n"));
	  continue;
	  }
#elif (DEBUG & DEBUG_OUTPUT) == COM4
      if (cpt == 3) {
	  PDBG (("Do not check COM4, it is stderr\r\n"));
	  continue;
	  }
#endif

      PDBG (("Checking serial mouse on COM%u, I/O address 0x%X: ",
		cpt+1, ioaddr));

#if 0
      {
      /* loopback => no more power supply to mouse */
      const struct serial_modem_str resetmodem = {
	  .dtr = 0, .rts = 0, .loopback = 1
	  };
      SETMODEM (cpt, ioaddr, resetmodem);
      _BIOS_wait (50000); /* 50 ms */
      SETMODEM (cpt, ioaddr, clearmodem);
      _BIOS_wait (50000); /* 50 ms */
      }
#endif
#if 1
      /* Oh, less than 50 ms would be enough, but you know,
	 micro$oft has also made hardware - and on a 16650... */
      /* Maybe have a look at APM->serial or _BIOS_initserial() before ? */
      SETMODEM (cpt, ioaddr, setmodem);
      _BIOS_wait (400000); /* 400 ms */
#endif
      _BIOS_initserial (cpt, mouse_std_setup);
      SETMODEM (cpt, ioaddr, restmodem);
      _BIOS_wait (100000); /* 100 ms */

      status = _BIOS_getserialstatus (cpt);

      if (status.receive_data_ready) {
	  volatile unsigned char tmp __attribute__ ((unused)) = inb (ioaddr);
	  PDBG (("[flush received char 0x%X], ", tmp));
	  }

      PDBG ((" CTS=%u, DSR=%u, RI=%u, CD=%u; ",
		status.al.modem.cts, status.al.modem.dsr,
		status.al.modem.ri, status.al.modem.cd));

      SETMODEM (cpt, ioaddr, setmodem);

      for (i = 30, j = 0; i != 0 && j < sizeof (array); --i) {
	  _BIOS_wait (9000); /* 10 bits at 1200 bauds: 8.3 ms => 9 ms */
	  status = _BIOS_getserialstatus(cpt);
	  if (status.receive_data_ready) {
	      array[j] = inb (ioaddr);
	      PDBG (("0x%X, ", array[j]));
	      if (array[j])	/* some UARTs, when break received */
		j++;
	      }
	  }

      status = _BIOS_getserialstatus (cpt);
      PDBG ((" DTR and RTS now set; CTS=%u, DSR=%u, RI=%u, CD=%u; ",
		status.al.modem.cts, status.al.modem.dsr,
		status.al.modem.ri, status.al.modem.cd));

      k = 0;
      while (j != 0) {
	  if (j >= 2 && array[j-2] == 'M' && array[j-1] == '3') {
	      PDBG (("serial 3 button mouse\r\n"));
	      MOUSE.type = MOUSE_SERIAL3_COM1 + cpt;
	      MOUSE.data.serial.comIOaddr = ioaddr;
	      MOUSE.data.serial.comport = cpt;
	      return 0;
	      }
	  if (array[j-1] == 'M') {
	      PDBG (("serial std mouse\r\n"));
	      MOUSE.type = MOUSE_SERIAL_COM1 + cpt;
	      MOUSE.data.serial.comIOaddr = ioaddr;
	      MOUSE.data.serial.comport = cpt;
	      return 0;
	      }
	  if (array[j-1] == 'H') {
	      _BIOS_initserial (cpt, mouse_mm_setup);
	      PDBG (("serial MM mouse\r\n"));
	      MOUSE.type = MOUSE_SERIALM_COM1 + cpt;
	      MOUSE.data.serial.comIOaddr = ioaddr;
	      MOUSE.data.serial.comport = cpt;
	      return 0;
	      }
	  /* Leave this debug message for some time: */
	  if (k == 0) {
	      PRINTF (" [on COM%u, %u bytes: 0x%X", cpt+1, j, array[k++]);
	      i = j;
	      while (--i)
		  PRINTF (", 0x%X", array[k++]);
	      PRINTF ("] ");
	      }
	  j--;
	  }
      PDBG (("no mouse there\r\n"));
      SETMODEM (cpt, ioaddr, clearmodem);
      }
  return 1;
  }
#endif /* SERIAL_MOUSE_SUPPORT */

#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT | SERIAL_MOUSE_SUPPORT)
VIDEO_FCT_PREFIX(MOUSE_init) unsigned
MOUSE_init (unsigned nb_serial_port)
  {bound_stack();{
  unsigned pointing_device_installed;
  unsigned csmask = 0x10000 * getcs();
  MOUSE.maximum = ((coord) {0,0});

  MOUSE.function.reset_field = (typeof(MOUSE.function.reset_field))((unsigned)mouse_reset_field | csmask);
  MOUSE.function.print_field = (typeof(MOUSE.function.print_field))((unsigned)mouse_print_field | csmask);

  PDBG (("%s: sizeof struct mouse_interface_str %u, ",
	__FUNCTION__, sizeof (struct mouse_interface_str)));

#ifndef MOUSE_ON_VT
  if (BOOT1_COM_port() >= 0) {
      PDBG (("MOUSE on VT not compiled-in\r\n"));
      return 2;
      }
#endif

  pointing_device_installed = _BIOS_equipment_flags().pointing_device_installed;
  PDBG (("int 0x11 pointing_device_installed = %u\r\n", pointing_device_installed));

  if (   0
#if USER_SUPPORT & BIOS_MOUSE_SUPPORT
      || (pointing_device_installed && MOUSE_PS2_init() == 0)
#endif
#if USER_SUPPORT & SERIAL_MOUSE_SUPPORT
      || MOUSE_SERIAL_init(nb_serial_port) == 0
#endif
      ) {
      MOUSE.all_colors = MOUSE_reset_colors;
      PDBG (("Mouse initialised\r\n"));
      return 0;
      }
    else {
      PDBG (("Mouse initialisation failed\r\n"));
      return 1;
      }
  }}
#endif

#if USER_SUPPORT & JOYSTICK_SUPPORT
VIDEO_FCT_PREFIX(MOUSE_joystick_init) unsigned
MOUSE_joystick_init (unsigned enable_joystick)
  {bound_stack();{
#define NBJOYTEST	5
  unsigned short Ax, Ay, Bx, By, nbtest = NBJOYTEST + 1;
  unsigned accumulate_Ax = 0, accumulate_Ay = 0;
  struct JoystickButton_str button;

#if !(USER_SUPPORT & (BIOS_MOUSE_SUPPORT | SERIAL_MOUSE_SUPPORT))
  unsigned csmask = 0x10000 * getcs();

  MOUSE.function.reset_field = (typeof(MOUSE.function.reset_field))((unsigned)mouse_reset_field | csmask);
  MOUSE.function.print_field = (typeof(MOUSE.function.print_field))((unsigned)mouse_print_field | csmask);
#endif

  UTIL.joystick_present = 0;
  if (!enable_joystick) {
      PDBG ((" [Do not enable joystick: enable_joystick false] "));
      return UTIL.joystick_present;
      }
  while (--nbtest) {
      PDBG ((" [test joystick No %u] ", nbtest));
      if (JoystickPosition (&Ax, &Ay, &Bx, &By) != 0) {
	  PDBG ((" [Do not enable joystick: JoystickPosition() failure Ax 0x%X] ", Ax));
	  break;
	  }
	else if ((Ax | Ay | Bx | By) == 0) {
	  PDBG ((" [Do not enable joystick: Ax == Ay == Bx == By == 0] "));
	  break;
	  }
	else if (JoystickButton (&button) != 0) {
	  PDBG ((" [Do not enable joystick: JoystickButton() failure] "));
	  break;
	  }
      accumulate_Ax += Ax;
      accumulate_Ay += Ay;
      }
  if (nbtest == 0) {
      Ax = accumulate_Ax / NBJOYTEST;
      Ay = accumulate_Ay / NBJOYTEST;
#if 0
      MOUSE.joystick = (const struct joystick_str) {
	  .Xmin = Ax, .Xmid = Ax, .Xmax = Ax,
	  .Ymin = Ay, .Ymid = Ay, .Ymax = Ay,
	  };
#else
      memset (&MOUSE.joystick, 0, sizeof(MOUSE.joystick));
      MOUSE.joystick.Xmin = MOUSE.joystick.Xmid = MOUSE.joystick.Xmax = Ax;
      MOUSE.joystick.Ymin = MOUSE.joystick.Ymid = MOUSE.joystick.Ymax = Ay;
#endif
      PDBG ((" [Joystick initialised, AxMid %u, AyMid %u, default button 0x%X] ", Ax, Ay, button));
      MOUSE.all_colors = MOUSE_reset_colors;
      UTIL.joystick_present = 1;
      }

  return UTIL.joystick_present;
  }}
#endif

/**
 ** The global variable inited to zero because in BSS:
 **/
struct mouse_interface_str MOUSE = {};

#endif /* USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT) */

