/* make.h */
#ifndef MAKE_H
#define MAKE_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/**
 ** This file contains only PREPROCESSOR directives, so can be included
 ** in files not written in C, like the linker configuration file.
 ** It is the only file included in ".h" files, and does basic checking
 ** of the configuration.
 **/

/*
 * This is self explainning:
 * It has to compile when it is zero, so those structure fields
 * shall not be accessed, but we try to keep the mapping of
 * structures as much as possible, always distribute set to 1.
 */
//#define KEEP_STRUCT_MAPPING 0
#define KEEP_STRUCT_MAPPING 1

/**
 ** Now we define the relation in between Makefile string-variables
 ** and CPP number-variables - in short do not touch them.
 **/

/*
 * The setup configuration:
 */
#define ASSEMBLY_NONE     0x000000  /* use C equivalent */
#define ASSEMBLY_CISC     0x000001  /* complex addressing mode OK */
#define ASSEMBLY_RISC     0x000002  /* more instructions but simpler ones */
#define ASSEMBLY_DSES     0x000003  /* use only C pointers refering to %ds
					and %es, do not use %fs nor %gs */
#define ASSEMBLY_TYPE     (SETUP & 0x0F)

#define CODE_SEGMENT      0x000010  /* use a different segment for code */
#define XCODE_SEGMENT     0x000020  /* use three extra segment (LOAD, FS and DISK) for code */
#define UNICODE_FONT	0x000040	/* else only 256 chars available */

#define BIG_MALLOC        0x000100  /* have a bigger pool for malloc */
#define XDATA_SEGMENT     0x000200  /* use an extra segment for data (char fonts) */
#define MULTILINGUAL	0x000400	/* else only English available */
#define XSTRING_SEGMENT	0x000800  /* use extra segments for MultiString (MULTILINGUAL) */
#define UPDATE_BOOTPARAM  0x001000  /* update gujin parameters on disk */
#define BIOS_ONLY         0x004000  /* disable gujin.exe/gujin.com */
#define USE_INT1587       0x008000  /* to copy to high memory */
//#define BOOT_NEVER_RETRY  0x010000  /* to read again if failure, just int 0x19, int 0x18 */ see --read_retry

#define QUITE             0x400000  /* Be really quite */


#if (SETUP & XDATA_SEGMENT) && !(SETUP & CODE_SEGMENT)
#error XDATA_SEGMENT without CODE_SEGMENT
#endif

#if (SETUP & XCODE_SEGMENT) && !(SETUP & CODE_SEGMENT)
#error XCODE_SEGMENT without CODE_SEGMENT
#endif

#if (SETUP & BIG_MALLOC) && (SETUP & USE_INT1587)
#error BIG_MALLOC and USE_INT1587 do not like each other in SETUP
#endif

#if !(SETUP & QUITE)
#define PRINTF(format...)	verbose_printf (format)
#define PRINT(msg)		verbose_print (msg)
#define PUTS(msg)		verbose_puts (msg)
#else
#define PRINTF(format...)	/* */
#define PRINT(msg)		/* */
#define PUTS(msg)		/* */
#endif

#if (ASSEMBLY_TYPE == ASSEMBLY_DSES) && (SETUP & USE_INT1587)
#warning USE_INT1587 will not be used with ASSEMBLY_DSES
#endif

/*
 * The user interface configuration:
 */
#define VGA_SUPPORT     (1<< 0)
#define VGA_EXTENDED    (1<< 1) // modes > 0x13 can be probed/used
#define VGA_MEMORY      (1<< 2) // to be implemented: direct memory access
#define VGA_USEINT1013  (1<< 3) // use _VGA_writestring, reduces code size
#define VGA_INT1013_BG  (1<< 4) // use background color for 256 color modes

#define VESA_WINDOW     (1<< 8)  // i.e. VESA 1 compatible
#define VESA_STRANGE_WINDOW (1<< 9)// window without 2^x bytes granularity, slow
#define VESA_WINFUNCTION (1<< 10) // optimise by using fct instead interrupt
#define VESA_HARDWINDOW (1<< 11) // analyse first code to change window(s)
#define VESA_2WINDOWS   (1<< 12) // use separate read & write window if present
#define VESA_LINEAR     (1<< 13) // i.e. VESA 2 only
#define VESA_EDID       (1<< 14) // interogate screen with EDID
#define VESA_PMINFO     (1<< 15) // Get the I/O ports used by the graphic card
#define VESA_RECALC	(1<< 16) // recalc nb of visible lines if EGA compat
#define VESA_1BPP       (1<< 17) // 1 Bit Per Pixel
#define VESA_4BPP_EGA   (1<< 18) // 4 Bits Per Pixel
#define VESA_4BPP_TEXT  (1<< 19) // text modes, 16 colors
#define VESA_4BPP       (1<< 20) // 8 Bits Per Pixel
#define VESA_8BPP       (1<< 21) // 8 Bits Per Pixel
#define VESA_16BPP      (1<< 22) // 15 or 16 Bits Per Pixel
#define VESA_24BPP      (1<< 23) // 24 Bits Per Pixel
#define VESA_32BPP      (1<< 24) // 32 Bits Per Pixel
#define VESA_SUPPORT    (VESA_1BPP|VESA_4BPP_EGA|VESA_4BPP_TEXT|VESA_4BPP \
			 |VESA_8BPP|VESA_16BPP|VESA_24BPP|VESA_32BPP)

#define SERIAL_VT100    (1<< 25)
#define SERIAL_VT420    (1<< 26)
#define SERIAL_SUPPORT  (SERIAL_VT100|SERIAL_VT420)

#define BIOS_MOUSE_SUPPORT   (1<< 28) // mostly PS2 mouses
#define SERIAL_MOUSE_SUPPORT (1<< 29) // serial 2/3 buttons mouse
#define JOYSTICK_SUPPORT     (1<< 30) // the standart BIOS one

#if (USER_SUPPORT & VGA_MEMORY) && (USER_SUPPORT & VGA_USEINT1013)
#error Cannot define VGA_MEMORY and VGA_USEINT1013 at the same time.
#endif

#if (USER_SUPPORT & VESA_HARDWINDOW) && !(USER_SUPPORT & VESA_WINFUNCTION)
#error Cannot define VESA_HARDWINDOW without VESA_WINFUNCTION
#endif

/*
 * The disk subsystem configuration:
 */
// The way to read sectors on permanent storage device:
#define BIOS_SUPPORT    0x0000001
#define EBIOS_SUPPORT   0x0000002
#define IDE_SUPPORT     0x0000004
#define DOS_SUPPORT	0x0000008
#define CDBIOS_SUPPORT	0x0000010
#define ATAPI_SUPPORT	0x0000020

// What is probed for the boot menu:
#define E2FS_PROBE	0x0000100
#define FAT12_PROBE	0x0000200
#define FAT16_PROBE	0x0000400
#define FAT32_PROBE	0x0000800
#define ISOFS_PROBE	0x0001000
#define BOOTSECT_PROBE  0x0002000

// Each of them reduce code size when defined:
#define NO_FLOPPY_SUPPORT		0x0010000
#define PROBE_ONLY_BIOSBOOTDEVICE	0x0020000
#define NO_PROBE_PARTITIONS		0x0040000

#if DISK_SUPPORT & PROBE_ONLY_BIOSBOOTDEVICE
#if DISK_SUPPORT & (IDE_SUPPORT | ATAPI_SUPPORT)
#warning IDE_SUPPORT and ATAPI_SUPPORT not usefull when PROBE_ONLY_BIOSBOOTDEVICE
#endif
#if !(DISK_SUPPORT & (BIOS_SUPPORT | EBIOS_SUPPORT))
#error PROBE_ONLY_BIOSBOOTDEVICE needs either BIOS_SUPPORT or EBIOS_SUPPORT
#endif
#endif
#if (DISK_SUPPORT & ATAPI_SUPPORT) && !(DISK_SUPPORT & IDE_SUPPORT)
#error ATAPI_SUPPORT set in DISK_SUPPORT without IDE_SUPPORT
#endif

/*
 * The debug subsystem configuration:
 */

/* debug options: */
#define DEBUG_STACK     0x000100  /* is a standalone - default output */
#define DEBUG_ADDR      0x000200  /* is a standalone - default output */
#define DEBUG_FS        0x000400
#define DEBUG_LOADER	0x000800
#define DEBUG_DISK      0x001000
#define DEBUG_VIDEO     0x002000
#define DEBUG_SERIAL    0x004000
#define DEBUG_MOUSE     0x008000

#define DEBUG_OPTION    0xFFFC00

/* debug output: */
#define COM1            0x01
#define COM2            0x02
#define COM3            0x03
#define COM4            0x04
#define LPT1            0x80
#define LPT2            0x81
#define LPT3            0x82
#define LPT4            0x83
#define SCREEN          0xF0
#define DOSFILE         0xF8

#define DEBUG_OUTPUT    0x00FF

/**
 ** Now are defines to select the more logical configuration:
 ** They can be modified if you need something special.
 **/

/*
 * For user.c/mouse.c:
 */

#if USER_SUPPORT & SERIAL_SUPPORT
/*
 * This one is probably only for fun: implement the mouse
 * on a VT terminal...
 */
#define MOUSE_ON_VT
#endif /* SERIAL_SUPPORT */

/*
 * For disk.c:
 */

#if !(DISK_SUPPORT & PROBE_ONLY_BIOSBOOTDEVICE) && (DISK_SUPPORT != DOS_SUPPORT)
/*
 * Use malloc for storing informations of disks, partitions, freelist,
 * those structure are quite big so malloc library is usefull.
 */
#define DISK_USE_MALLOC

/*
 * Use malloc for storing where are the file fragments on disk
 * (LBA, nbsector)[] and not an array of FILE_MAX_FRAGMENT size.
 * Without any *_USE_MALLOC, the memory allocation function are not
 * needed: nice for the TINY configuration.
 */
#define FS_USE_MALLOC

#endif

/*
 * For boot.c:
 */

#if (USER_SUPPORT != 0)
#define WRITE_ENABLE
#endif

#if    (DEBUG & (DEBUG_STACK | DEBUG_ADDR)) \
    || (USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)) \
    || (DISK_SUPPORT & EBIOS_SUPPORT)
#define TREAT_EXCEPTION
#define TREAT_TIMER
#else
// We disable it for tiny configuration to save space, just re-enable
// if you have a problem (to treat invalid instructions, crashes...)
//#define TREAT_EXCEPTION
//#define TREAT_TIMER
#endif

/*
 * For gzlib.c:
 */
#if ((DISK_SUPPORT & E2FS_PROBE) || DEBUG != 0) && !(SETUP & BIG_MALLOC)
/* Use the gzlib window trick when we cannot do anything else: */
#define GZLIB_BIG_MALLOC
#endif

/*
 * For vmlinuz.c:
 */

/*
 * We cannot do:
#define SEGMENT_EXTRA(fct) __attribute__ ((section (STRING(.xcode. ## fct))))
 * Because of the CPP warning:
 * pasting "." and "xxx" does not give a valid preprocessing token
 *
 * Other note: how about: SEGMENT_EXTRA(__FUNCTION__) working?
 */
#define _str(s)	#s
#define _STR(s)	_str(s)
#define _EXP(X)	X

#if (SETUP & XCODE_SEGMENT)
// see user.c::fill_exported_togpl() when changing those two:
#define UTIL_FCT_PREFIX(fct)	__attribute__ ((section (_STR(_EXP(.loadcode.)_EXP(fct))) ))
#define KBD_FCT_PREFIX(fct)	__attribute__ ((section (_STR(_EXP(.loadcode.)_EXP(fct))) ))
#define GZIP_FCT_PREFIX(fct)	__attribute__ ((section (_STR(_EXP(.loadcode.)_EXP(fct))) ))
#define LOAD_FCT_PREFIX(fct)	__attribute__ ((section (_STR(_EXP(.loadcode.)_EXP(fct))) ))
#define DISK_FCT_PREFIX(fct)	__attribute__ ((section (_STR(_EXP(.diskcode.)_EXP(fct))) ))
#define FS_FCT_PREFIX(fct)	__attribute__ ((section (_STR(_EXP(.fscode.)_EXP(fct))) ))
#define VIDEO_FCT_PREFIX(fct)	__attribute__ ((section (_STR(_EXP(.usercode.)_EXP(fct))) ))
#endif

#ifndef DISK_FCT_PREFIX
#define DISK_FCT_PREFIX(fct)	/* */
#endif

#ifndef FS_FCT_PREFIX
#define FS_FCT_PREFIX(fct)	/* */
#endif

#ifndef LOAD_FCT_PREFIX
#define LOAD_FCT_PREFIX(fct)	/* */
#endif

#ifndef GZIP_FCT_PREFIX
#define GZIP_FCT_PREFIX(fct)	/* */
#endif

#ifndef UTIL_FCT_PREFIX
#define UTIL_FCT_PREFIX(fct)	/* */
#endif

#ifndef KBD_FCT_PREFIX
#define KBD_FCT_PREFIX(fct)	/* */
#endif

#ifndef VIDEO_FCT_PREFIX
#define VIDEO_FCT_PREFIX(fct)	/* */
#endif

/*
 * For vmlinux.c / fs.c:
 */

/*
 * Define a minimum size for kernel/initrd/\*.kgz files to not take a checksum or
 * md5sum or signature as a real file:
 */
#define MIN_KERNEL_INITRD_SIZE 4095
#define IS_KERNEL_OR_INITRD(elem) \
	((elem)->boottype == is_linux || (elem)->boottype == is_initrd || (elem)->boottype == is_initramfs)

/*
 * At least one of LINUZNAME1, LINUZNAME2, LINUZNAME3, KERNELEXT
 * has to be defined.
 *
 * This bootloader will recognise "LINUZNAME[123]*" if they are defined,
 * and treat them as usual Linux kernel files, and recognise also
 * "*KERNELEXT" to treat them as compressed ELF files.
 *
 * Existance of LINUZNAME* is also tested in "vmlinuz.c" to include
 * corresponding code. Those strings are mainly used in "fs.c".
 *
 * KERNELEXT is case sensitive in E2FS, you would better _not_ write
 * it in capitals for FAT*
 */
#define LINUZNAME1		"vmlinuz"
#define LINUZNAME2		"bzImage"
#define LINUZNAME3		"linux"

 /* Big code simplification: has to be 4 bytes long plus ending zero. */
#define KERNELEXT		".kgz" /* for instance linux-2.6.kgz */
#define KERNELEXT_UPPERCASE	".KGZ"

 /* Big code simplification: has to be 4 bytes long plus ending zero. */
#define ISOEXT			".iso"
#define ISOEXT_UPPERCASE	".ISO"

#define DEFAULTSEARCHPATH	"boot"
#define ALT1SEARCHPATH		"install.386"
#define ALT2SEARCHPATH		"install.amd"
#define ALT3SEARCHPATH		"isolinux"
#define ALT4SEARCHPATH		"casper"
#define ALT5SEARCHPATH		"live"
// debian-testing-i386-netinst.iso contains /install.386/{vmlinuz,initrd.gz}
// debian-live-504-i386-rescue.iso contains /live/{vmlinuz,initrd.gz}

/*
 * If, in the same disk/partition/directory of "vmlinuz*", there is
 * at least one file named "initrd*", then an initial RAMdisk is loaded.
 *
 * The file loaded is the "initrd*" with the nearest ending,
 * i.e. if you have those files in /boot:
 *  vmlinuz-2.2.0.img   zImage-3.test   bzImage-2.4 kernel
 *  initrd     initrd-2.gz   initrd-2.2.gz
 *
 * Then "vmlinuz-2.2.0.img" will load "initrd-2.2.gz"
 *      "zImage-3.test"     will load "initrd"
 *      "bzImage-2.4"       will load "initrd-2.gz"
 *      "kernel"            will load "initrd"
 *
 * As an extension, if the initrd file is named "initrd.img-2.4.18", the
 * part ".img" is ignored in the comparisson.
 * When using KERNELEXT, you should have at least the same name,
 * for instance kernel-2.5.1.kgz and initrd-2.5.1*, let's say
 * initrd-2.5.1.img. If you build a monolitic kernel (without
 * modules) you can disable loading the nearest initrd by using
 * the '+' sign, like "/boot/linux+2.6.16.kgz.
 * This is when KERNELEXT is exactly ".kgz" (with the dot).
 * The complete initrd processing is excluded if INITRDNAME
 * is undefined (saves 1.5 Kb in fscodeseg).
 */
#define INITRDNAME	"initrd"
//#define INITRDNAME2	"miniroot"	/* KANOTIX-2005-04.iso */
#define INITRDNAME2	"tinycore" /* tinycore_2.6.iso */

/*
 * The initramfs is treated completely as an initrd, but not decompressed
 * when loaded (so no check of the validity of what is loaded).
 * If the last letter is 'a', accept also finishes by "mfs" with or without
 * ".img" for kernel name association, i.e. vmlinuz-2.6.18 will be
 * associated to initra-2.6.18, initramfs-2.6.18 and initramfs.img-2.6.18
 */
#define INITRAMFSNAME	"initra"

/*
 * Those are used to know if a FAT partition is bootable or
 * not, i.e. if you have created two partition on DOS
 * but did not do "sys d:", the disk "d:" is not bootable.
 * Same to know if a floppy is bootable.
 *
 * We consider that a FAT filesystem with at least one
 * of these file in the root directory is bootable, it
 * is not completely true for DOS6 or before: it has to
 * be the _first_ file of the directory but we do not check
 * that. Also for DOS6 and before, it does not need to be
 * in the root directory but simply the first allocated in
 * the data area of the FAT filesystem.
 * Note also that usually the bootloaders (even Gujin) are
 * physically on a FAT partition, so you can force to see
 * (most) of those by "keep_all_part_mbr", or individually
 * by creating an empty file named "ibmbio.com"
 * Someone around is talking of filename "ibmio.sys", never
 * seen it myself.
 *
 */
#define DOSNAME1		"io.sys"
#define DOSNAME2		"ibmbio.com"

/*
 * This file contains the previous bootsector if Windows
 * has been installed over a previous DOS installation,
 * see "http://www.goodells.net/multiboot/principles.htm"
 */
#define ALTMBRNAME	"bootsect.dos"

/*
 * Boot Device Image files: image of a floppy or a hard disk
 * which can be used like El-Torito images to boot the PC but
 * do not need to be on a CDROM/DVD and in the
 * El-Torito Boot Directory. They will be searched in the same
 * places as *.kgz files and an error will be flagged when
 * selected if they do not contain a standard boot record.
 * This boot record is used at least to detect which disk
 * (fd/hd) to emulate, its geometry when accessed using
 * standard BIOS and its size for BIOS/EBIOS.
 * If the BIOS hd number is too high (like 0xFF), it will be
 * reduced to maintain consecutive BIOS hard disk number,
 * and may boot if the boot code is ready to get this number
 * in %edx register.
 * Note that there isn't any point in searching those files under
 * the DOS interface.
 * Big code simplification: has to be 4 bytes long plus ending zero.
 */
#define BOOTDEVICEIMAGE	".bdi"	/* for instance memtest.bdi */
#define BOOTDEVICEIMAGE_UPPERCASE	".BDI"

/* Those are treated exactly as BDI files, but only if they have
 * the 0xAA55 signature at offset 510, which is the standard
 * way to detect a bootable device image:
 */
#define BOOTDEVICEIMAGE_MAYBE1	".img"
#define BOOTDEVICEIMAGE_MAYBE1_UPPERCASE	".IMG"
#define BOOTDEVICEIMAGE_MAYBE2	".bin"
#define BOOTDEVICEIMAGE_MAYBE2_UPPERCASE	".BIN"

/*
 * The unawaited command file for command lines in Linux
 */
#define COMMANDFILE		"gujin.cmd"
#endif /* MAKE_H */
