/* main.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 ** As they say here "standing on the shoulders of giants" ...
 **
 ** Thanks to: Theodore Tso <tytso@mit.edu> for the e2fs library.
 ** Thanks to: Jean-loup Gailly <jloup@gzip.org> and
 **            Mark Adler <madler@alumni.caltech.edu> for the zlib library.
 **              http://www.freesoftware.com/pub/infozip/zlib/
 ** Thanks to: Colin Plumb <colin@nyx.net> for his guide to use GCC/asm()
 **		sent in the list linux-kernel@vger.rutgers.edu on
 **		Mon, 20 Apr 1998 - Subject: Re: Using inline asm on the i386
 **            See also: "info GCC"
 **              "Assembler Instructions with C Expression Operands"
 ** Thanks to: Ralf Brown for his list of interrupts INTER61:
 **		http://www.ctyme.com/rbrown.htm and all contributers.
 **         David Jurgens for the HelpPC Reference Library:
 **		http://heim.ifi.uio.no/~stanisls/helppc/
 **		And also: http://www.mcamafia.de/pdf/ps2bios2.pdf
 **			http://www.mcamafia.de/pdf/pdfref.htm
 ** Thanks to: The GAS developper team.
 ** Thanks to: opcode databases like http://www.sandpile.org/80x86
 ** Thanks to: "flat" real mode description from John Fine
 **		(johnfine@erols.com , http://users.erols.com/johnfine/)
 **            and Robert Collins
 **		(rcollins@x86.org , http://www.x86.org/ftp/articles/)
 ** Thanks to: Bart Lagerweij for saving me time to understand that
 **            I will not be able to call the Bootable CDROM BIOS
 **            services (see the list of really implemented calls)
 **            http://www.nu2.nu/bootcd/
 ** Thanks to: Michel LESPINASSE for explaining why EMM386 would not
 **            unload cleanly using its documented external interface.
 **            http://www.nondot.org/sabre/os/files/ProtectedMode/GEMMIS.txt
 **        See also: http://ata-atapi.com/
 ** Thanks to: jouni the TooLate author (jon@stekt.oulu.fi)
 **            http://stekt.oulu.fi/~jon/jouninfo/
 **            for his nice font editor.
 */

#include "make.h"
#include "instboot.h"
#include "library.h"
#include "boot.h"
#include "debug.h"
#include "messages.h"
#include "bios.h"	/* _DOSEMU_check(), CDROMbootspec */
#include "util.h"	/* initrd_file(), detect_processor() ... */
#include "kbd.h"
#include "disk.h"
#include "fs.h"
#include "user.h"	/* define UI which is the user interface (UI for vesabios.h) */
#include "vgabios.h"	/* _VGA_autoload_default_palette()... */
#include "vesabios.h"	/* _VESA_setmode(), needs user.h for UI, VESA_setwin(), mingujin.exe */

#if DISK_SUPPORT & DOS_SUPPORT
#include "dos.h"
#endif

#if USER_SUPPORT != 0
struct {
    unsigned short curpage, totalpage, nbperpage, nbtotal, interline;
    enum { kernel_bottom_menu, setup_bottom_menu } type;
    } Menu; /* init to BSS */

#include "mouse.h"
#endif /* USER_SUPPORT != 0 */

#include "vmlinuz.h"	/* menu_load_system() */
static inline void list_disks (void)
  {
#if !(SETUP & QUITE)
  struct diskparam_str *dp;
  char chs_string[32]; /* sizeof (" C/H/S: %u/%u/%u") + 8, C can have more than 2 chars, or DISK_BYTES_PER_SECTOR */

  for (dp = &DI.param[0]; dp < &DI.param[DI.nbdisk]; dp++) {
      unsigned sizeKB, nbsector_in_16k;
      unsigned char remark[12], *ptr;

      if (dp->bytepersector == 0) {
	  PRINTF (DISK_NAME_NO_MEDIA, dp - &DI.param[0], dp->diskname);
	  continue;
	  }

      /* ideAtapiReadCdRecordedCapacity done, blocksize 2352, lba 204,506 */
      /* nbsector_in_16k = DIVROUNDUP (16 * 1024, dp->bytepersector); */
      nbsector_in_16k = (16 * 1024) / dp->bytepersector;
      unsigned char bitpos = __builtin_ffs (nbsector_in_16k) - 1;
      if (nbsector_in_16k == 0 || nbsector_in_16k == 1U << bitpos)
	  sizeKB = 16 * (dp->nbtotalsector >> bitpos);
	else {
	  unsigned remainder;
	  sizeKB = 16 * ull_div_ul (dp->nbtotalsector, nbsector_in_16k, &remainder);
	  }

      ptr = remark;
      *ptr++ = ' ';
      *ptr++ = '(';
      if (dp->BEER_sector != 0)
	  *ptr++ = '*';
      if (dp->ide_attribute.host_protected_area)
	  *ptr++ = 'H';
      if (dp->ide_attribute.config_overlay)
	  *ptr++ = 'C';
      if (dp->ide_attribute.security)
	  *ptr++ = 'P';
      if (dp->ide_attribute.removable)
	  *ptr++ = 'R';
      if (dp->ide_attribute.SAORAB)
	  *ptr++ = 'B';
//    if (dp->ide_attribute != (struct ide_attribute_str) {}) {}
      if (ptr[-1] == '(')
	  remark[0] = '\0';
      *ptr++ = ')';
      *ptr++ = '\0';

      if (dp->access == bios_chs || dp->access == hardide_chs)
	  sprintf (chs_string, " C/H/S: %u/%u/%u",
			dp->nbcylinder,
			dp->nbhead,
			dp->nbsectorpertrack);
       else if (dp->access == ebios_lba && dp->bytepersector != 512)
	  sprintf (chs_string, " %u %s", dp->bytepersector, DISK_BYTES_PER_SECTOR);
	else
	  chs_string[0] = '\0';

      {
      unsigned i, nbdisplayed;

      for (i = nbdisplayed = 0; i < dp->nbpartition; i++)
	  if (dp->partition[i].type != EMPTY && dp->partition[i].misc.active != part_extended)
	      nbdisplayed++;
      if (nbdisplayed == 1 && dp->nbpartition == 5)
	  nbdisplayed = 0; /* Dummy floppy partition */
      PRINTF (DISK_NAME_REMARK_SIZE_NBPARTITION,
		dp - &DI.param[0],
		dp->diskname,
		chs_string,
		remark,
		(sizeKB > 9*1024*1024) ? sizeKB/1024/1024 : ((sizeKB > 9*1024) ? sizeKB/1024 : sizeKB),
		(sizeKB > 9*1024*1024) ? "Gb" : ((sizeKB > 9*1024) ? "Mb" : "Kb"),
		nbdisplayed
		);
      }
#if DISK_SUPPORT & IDE_SUPPORT
      if (dp->ide_attribute.host_protected_area) {
	  unsigned long long ide_max_native48_hpa = get_native_max_hpa(dp);
	  if (ide_max_native48_hpa != 0 && ide_max_native48_hpa > dp->nbtotalsector) {
	      unsigned diff = (ide_max_native48_hpa - dp->nbtotalsector) / 2; /* Assumes 512bytes/sectors when HPA */
	      PRINTF (DISK_HPA,
			(diff > 9*1024*1024) ? diff/1024/1024 : ((diff > 9*1024) ? diff/1024 : diff),
			(diff > 9*1024*1024) ? "Gb" : ((diff > 9*1024) ? "Mb" : "Kb")
			);
	      }
	  }
      if (dp->error_log.ideReadConfig_failed)
	  PRINTF (DISK_CONFIG_LOCKED);
#endif
      PRINTF ("." CRLF);
      }
#endif
  }

#if USER_SUPPORT & VGA_SUPPORT
static inline void listmode (struct user_interface_str *ui)
  {
#if !(SETUP & QUITE)
  unsigned short start = 0, cpt, firsttime = 1,
		 valid = VIDEO_mode_is_valid(0);

  for (cpt = 1; cpt <= VIDEO_mode_max(); cpt++) {
      if (cpt == VIDEO_mode_max() || VIDEO_mode_is_valid(cpt) != valid) {
	  if (valid == 0)
	      start = cpt;
	    else {
	      const char *format = (start < cpt-2)? ", 0x%X..0x%X" : ", 0x%X, 0x%X";
	      if (start > cpt-2)
		  format += sizeof (", 0x%X")-1;
	      PRINTF (firsttime? (format+2) : format, start, (cpt == VIDEO_mode_max())? cpt : cpt-1);
	      firsttime = 0;
	      }
	  valid = !valid;
	  }
      }
  if (firsttime)
      PRINT (MSG_NONE);
  PRINT ("." CRLF);
#endif
  }
#endif /* VGA_SUPPORT */

static inline void detect_joystick_mouse (struct gujin_param_attrib gujin_attr)
  {
  PDBG (("%s: ", __FUNCTION__));
#if USER_SUPPORT & JOYSTICK_SUPPORT
  if (MOUSE_joystick_init (gujin_attr.enable_joystick)) {
      PRINT (JOYSTICK_PRESENT);
      PDBG (("[joystick found] "));
      }
#endif

#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT | SERIAL_MOUSE_SUPPORT)
  {
  const char *msg = NO_MOUSE, *extramsg = "";
  unsigned nb = 0;

  if (MOUSE_init (_BIOS_equipment_flags().nb_serial_port) == 0) {
      PDBG (("MOUSE_init passed with type 0x%X\r\n", MOUSE.type));
      if (MOUSE.type == MOUSE_PS2) {
	  msg = BIOS_MOUSE_S_X;
	  nb = MOUSE.data.PS2.device_id;
	  }
	else if ((MOUSE.type & MOUSE_SERIAL_MASK) == MOUSE_SERIAL_COM1) {
	  msg = SERIAL_ON_COMX_MOUSE;
	  nb = MOUSE.type - MOUSE_SERIAL_COM1 + 1;
	  }
	else if ((MOUSE.type & MOUSE_SERIAL_MASK) == MOUSE_SERIAL3_COM1) {
	  msg = SERIAL_ON_COMX_MOUSE;
	  extramsg = MOUSE_3BUTTONS;
	  nb = MOUSE.type - MOUSE_SERIAL3_COM1 + 1;
	  }
	else if ((MOUSE.type & MOUSE_SERIAL_MASK) == MOUSE_SERIALM_COM1) {
	  msg = SERIAL_ON_COMX_MOUSE;
	  extramsg = MOUSE_MOUSESYSTEM;
	  nb = MOUSE.type - MOUSE_SERIALM_COM1 + 1;
	  }
	else
	  msg = ERROR_MOUSE;
      }
    else
      PDBG (("no mouse found\r\n"));
  PRINTF (msg, extramsg, nb);
  }
#endif /* MOUSE_SUPPORT */
  }

static inline void detect_interface (struct gujin_param_attrib gujin_attr)
  {
  if (BOOT1_COM_port() >= 0) {
#if USER_SUPPORT & SERIAL_SUPPORT
      _BIOS_serialstatus status __attribute__ ((unused));
      PRINT (SERIAL_TERMINAL);
      status = _BIOS_getserialstatus(BOOT1_COM_port());
      DBG ((" [ COM%u status: CTS=%u , DSR=%u, RI=%u, CD=%u ] ",
		BOOT1_COM_port() + 1,
		status.al.modem.cts, status.al.modem.dsr,
		status.al.modem.ri, status.al.modem.cd));
      if (SERIAL_init() == 0)
	  PRINTF (TERMINAL_MODES, UI.info.serial.terminalname, UI.info.serial.max_mode);
	else
	  PUTS (SERIAL_INIT_FAILED);
#endif /* SERIAL_SUPPORT */
      }
    else {
#if USER_SUPPORT & (VGA_SUPPORT|VESA_SUPPORT)
      unsigned setmode_result = 0xFFFFFFFF;
#endif
#if USER_SUPPORT & VGA_SUPPORT
      unsigned vga_result = VGA_init();

//      if (vga_result == 0) {	// also for MDA/CGA/EGA
	  /* If we are in a VESA mode, VGA without s/getpixel,
	     i.e. more than 256 colors, we will have an error here: */
	  unsigned curvidmode = UI.function.getmode();
	  setmode_result = UI.function.setmode (curvidmode | 0x8000);
//        setmode_result = UI.function.getsize (curvidmode, &UI.parameter);
	  if (setmode_result != 0)
	      DBG (("Initial VGA_setmode(VGA_getmode) failed: 0x%X\r\n", setmode_result));
//	  }
      if (   !(UI.parameter.attr.isgraphic || _VGA_isgraphic())
	  && UI.parameter.attr.ansi_font) {
	  /* Reverse the conversion done by BOOT1_ansi_putstr() to see the same
		letters after loading the ANSI font: */
	  /* Maybe only do it from top of screen to cursor? */
	  unsigned row, col;
	  unsigned char curcol, currow, cursorstart, cursorstop;

	  _VGA_getcursor (0, &currow, &curcol, &cursorstart, &cursorstop);
	  for (row = 0; row < UI.parameter.nbrow; row++) {
	      for (col = 0; col < UI.parameter.nbcol; col++) {
		  unsigned char ch, attr, page = 0;
		  _VGA_setcursor (page, row, col);
		  _VGA_readchar (&ch, page, &attr);
		  _VGA_writechar_attr (ansi2pc[ch & 0xFF], page, attr, 1);
		  }
	      }
	  _VGA_setcursor (0, currow, curcol);
	  }
      /* after on-screen char conversion, we can print accented letters: */
      if (vga_result != 0)
	  PRINT (UI.monitor? EGA_CARD_PRESENT : CGA_MGA_HGC_CARD_PRESENT);
	else {
	  PRINTF (VGA_CARD_PRESENT_MODES, curvidmode & 0x7F);
	  listmode (&UI);
	  }
#endif
#if USER_SUPPORT & VESA_SUPPORT
      if (VESA_preinit() == 0)
	  PRINTF (S_CARD_VESA_VERSION_U_U, UI.info.vesa.cardname,
		  UI.info.vesa.version >> 8, UI.info.vesa.version & 0xFF);
#endif /* VESA_SUPPORT */

#if USER_SUPPORT & (VGA_SUPPORT|VESA_SUPPORT)
      if (setmode_result == 0 && gujin_attr.verbose) {
	  unsigned char row, col, cpt;

	  /* Change the bgcolor of the end of the screen: */
	  /* If setmode_result != 0, we do not have UI.function.[sg]etcursor() :*/
	  if (UI.function.getcursor (&row, &col) == 0) {
	      for (cpt = row + 1; cpt < UI.parameter.nbrow; cpt++)
		  puts ("");
	      UI.function.setcursor (row, col);
	      }
	  }
#endif
      }
  }

#if (SETUP & MULTILINGUAL) && (DISK_SUPPORT & DOS_SUPPORT)
static inline unsigned char getDOSenvironLANG (farptr psp)
  {
  unsigned char envir[80];
  // 2Ch	WORD	DOS 2+ segment of environment for process (see #01379)
  farptr DOSenviron = peekw (psp + 0x2C) << 16;
  if (DOSenviron == 0)
      return 0;
  for (;;) {
      unsigned char *ptr = envir;
      for (;;) {
	  *ptr = peekb(DOSenviron++);
	  if (*ptr == '\0' || ptr == &envir[lastof(envir)])
	      break;
	  ptr++;
	  }
      *ptr = '\0';
      if (ptr == envir)
	  break;
      if ((*(unsigned *)envir == *(unsigned *)"LANG" || *(unsigned *)envir == *(unsigned *)"lang")
		&& envir[4] == '=' && envir[7] == '\0') {
	  unsigned short twolettercode = *(unsigned short *)&envir[5];
//#define LANG_CMDLINE	MLTSTR ("lang=en", "lang=fr", "lang=ru", "lang=sp", "lang=it", "lang=pt", "lang=de", "lang=nl")
	  if (twolettercode == *(unsigned short *)"en")
	      return 0;
	    else if (twolettercode == *(unsigned short *)"fr")
	      return 1;
	    else if (twolettercode == *(unsigned short *)"ru")
	      return 2;
	    else if (twolettercode == *(unsigned short *)"sp")
	      return 3;
	    else if (twolettercode == *(unsigned short *)"it")
	      return 4;
	    else if (twolettercode == *(unsigned short *)"pt")
	      return 5;
	    else if (twolettercode == *(unsigned short *)"de")
	      return 6;
	    else if (twolettercode == *(unsigned short *)"nl")
	      return 7;
	  printf ("DOS environment language '%s' not understood.\r\n", envir);
	  return 0;
	  }
      }
  return 0;
  }
#endif

#if USER_SUPPORT != 0

#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
static inline unsigned get_refresh_freq (void)
  {
  const unsigned short addr = 0x3DA;	/* 0x3BA for MDA */

  if (UTIL.processor.calibrate_rdtsc == 0)
      return 0;
  if (BOOT1_COM_port() >= 0)
      return 0;
  unsigned long long rdtsc_timeout = rdtsc() + 16 * UTIL.processor.calibrate_rdtsc;
  unsigned char ioval = inb (addr) & 0x08;
  while (rdtsc() < rdtsc_timeout && ioval == (inb (addr) & 0x08))
      continue;
  if (rdtsc() >= rdtsc_timeout) {
      DBG ((" [%s: timeout VSync] ", __FUNCTION__));
      return 0;
      }
    else {
      VDBG ((" [%s: got Vsync] ", __FUNCTION__));
      ioval = inb (addr) & 0x08;
      while (ioval == (inb (addr) & 0x08))
	  continue;
      unsigned long long rdtsc_start = rdtsc();
      while (ioval != (inb (addr) & 0x08))
	  continue;
      while (ioval == (inb (addr) & 0x08))
	  continue;
      unsigned total_rdtsc = rdtsc() - rdtsc_start;
      /* UTIL.processor.calibrate_rdtsc is per 1/100 seconds */
      VDBG ((" [%s: refresh = 100 * %u / %u = %u Hz] ", __FUNCTION__, UTIL.processor.calibrate_rdtsc, total_rdtsc, 100 * UTIL.processor.calibrate_rdtsc/total_rdtsc));
      return DIVROUNDUP (100 * UTIL.processor.calibrate_rdtsc, total_rdtsc);
      }
  }

static inline void report_screen(void)
  {
  const char *msg;

  PRINT (SCREEN_INTERFACE);
  if (UI.monitor == 0)
      msg = UNKNOWN_SCREEN;
    else if (UI.monitor == 1)
      msg = BLACKWHITE_SCREEN;
    else if (UI.monitor == 2)
      msg = MONOCHROME_SCREEN;
    else if (UI.monitor == 3)
      msg = COLOR_SCREEN;
    else
      msg = COLOR_SCREEN_TYPE_X;
  PRINTF (msg, UI.monitor);

#if USER_SUPPORT & VESA_EDID
  if (BOOT1_COM_port() < 0) {
      if (UI.edid.horizontal_resolution != 0)
	  PRINTF (EDID_TYPE,
		UI.edid.horizontal_resolution,
		UI.edid.vertical_resolution,
		UI.edid.frequency,
		(UI.edid.Vsize_cm + 3 * UI.edid.Hsize_cm / 4 + 2) / 3
		);
      if (UI.edid.lowHfreq != 0)
	  PRINTF (EDID_FREQ, UI.edid.lowHfreq, UI.edid.highHfreq,
			 UI.edid.lowVfreq, UI.edid.highVfreq);
      if (UI.edid.Hsize_cm == 0 && UI.edid.Vsize_cm == 0)
	  PUTS (NON_EDID_SCREEN);
      }
#endif /* VESA_EDID */
  }
#endif /* VGA_SUPPORT | VESA_SUPPORT */

/*
 * Video mode selection:
 */
static inline int
prev_next_height (unsigned next, unsigned char currbpp, struct gujin_param_attrib gujin_attr)
  {
  struct videomode_str *array;
  unsigned char cpt;
  int found = -1;

  for (array = UI.mode, cpt = 0; cpt < UI.nbmode; cpt++, array++) {
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
      if (array->vesa? !gujin_attr.VESA_interface : !gujin_attr.VGA_interface)
	  continue;
#endif
      if (gujin_attr.lock_bpp && array->bpp != currbpp)
	  continue;
      if (gujin_attr.lock_text_graphic && array->text != !UI.parameter.attr.isgraphic)
	  continue;
      if (array->width != UI.parameter.width)
	  continue;
      if (array->height == UI.parameter.height)
	  continue;
      if ((array->height < UI.parameter.height) ^ !next)
	  continue;
      if (found == -1)
	  found = cpt;
	else if (array->height == UI.mode[found].height) {
	  if (ABS((int)currbpp - array->bpp) < ABS((int)currbpp - UI.mode[found].bpp))
	      found = cpt;
	  }
	else if ((array->height < UI.mode[found].height) ^ !next)
	  found = cpt;
      }
  return found;
  }

static inline int
prev_next_width (unsigned next, unsigned char currbpp, struct gujin_param_attrib gujin_attr)
  {
  struct videomode_str *array;
  unsigned char cpt;
  int found = -1;

  for (array = UI.mode, cpt = 0; cpt < UI.nbmode; cpt++, array++) {
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
      if (array->vesa? !gujin_attr.VESA_interface : !gujin_attr.VGA_interface)
	  continue;
#endif
      if (gujin_attr.lock_bpp && array->bpp != currbpp)
	  continue;
      if (gujin_attr.lock_text_graphic && array->text != !UI.parameter.attr.isgraphic)
	  continue;
      if (array->width == UI.parameter.width)
	  continue;
      if ((array->width < UI.parameter.width) ^ !next)
	  continue;
      if (found == -1)
	  found = cpt;
	else if (array->width == UI.mode[found].width) {
	  if (array->height == UI.mode[found].height) {
	      if (ABS((int)currbpp - array->bpp) < ABS((int)currbpp - UI.mode[found].bpp))
		  found = cpt;
	      }
	    else if ((array->height < UI.mode[found].height) ^ !next)
	      found = cpt;
	  }
	else if ((array->width < UI.mode[found].width) ^ !next)
	  found = cpt;
      }
  return found;
  }

static inline int
prev_next_definition (unsigned next, struct gujin_param_attrib gujin_attr)
  {
  unsigned currbpp = get_bpp (&UI.parameter);
  int found;

  found = prev_next_height (next, currbpp, gujin_attr);
  if (found == -1)
      found = prev_next_width (next, currbpp, gujin_attr);
  return found;
  }

static inline int
prev_next_color (unsigned next, struct gujin_param_attrib gujin_attr)
  {
  struct videomode_str *array;
  unsigned currbpp = get_bpp (&UI.parameter);
  unsigned char cpt;
  int found = -1;

  for (array = UI.mode, cpt = 0; cpt < UI.nbmode; cpt++, array++) {
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
      if (array->vesa? !gujin_attr.VESA_interface : !gujin_attr.VGA_interface)
	  continue;
#endif
      if (array->width != UI.parameter.width)
	  continue;
      if (array->height != UI.parameter.height)
	  continue;
      if (array->bpp == currbpp)
	  continue;
      if ((array->bpp < currbpp) ^ !next)
	  continue;
      if (found == -1)
	  found = cpt;
	else if (array->bpp == UI.mode[found].bpp)
	  continue; /* take only the first one */
	else if ((array->bpp < UI.mode[found].bpp) ^ !next)
	  found = cpt;
      }
  return found;
  }

static inline int
prev_next_alternate (struct gujin_param_attrib gujin_attr)
  {
  struct videomode_str *array;
  unsigned currbpp = get_bpp (&UI.parameter);
  unsigned char cpt, found = 0;
  int first= -1;

  for (array = UI.mode, cpt = 0; cpt < UI.nbmode; cpt++, array++) {
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
      if (array->vesa? !gujin_attr.VESA_interface : !gujin_attr.VGA_interface)
	  continue;
#endif
      if (array->width != UI.parameter.width)
	  continue;
      if (array->height != UI.parameter.height)
	  continue;
      if (array->bpp != currbpp)
	  continue;
      if (array->number == UI.parameter.identification)
	  found = 1;
	else if (first == -1)
	  first = cpt;	/* if none after, start with first */
	else if (found == 1)
	  return cpt;	/* return the following mode */
      }
  return first;
  }

static inline int
switch_text_graphic (struct gujin_param_attrib gujin_attr)
  {
  unsigned short tmp;
  unsigned char cpt;

  if (UI.parameter.attr.isgraphic)
      tmp = copy_gujin_param.default_text_video_mode;
    else
      tmp = copy_gujin_param.default_graphic_video_mode;
  if (tmp != 0xFFFF) {
      for (cpt = 0; cpt < UI.nbmode; cpt++) {
	  /* exclude clear screen and linear bits: */
	  if (((UI.mode[cpt].number ^ tmp) & 0x3FF) == 0) {
	      return cpt;
	      }
	  }
      }
  for (cpt = 0; cpt < UI.nbmode; cpt++) {
      if (UI.mode[cpt].text ^ !UI.parameter.attr.isgraphic)
	  break;
      }
  if (cpt < UI.nbmode)
      return cpt;
  return -1;
  }

static int
find_video_mode (unsigned char key)
  {
#if USER_SUPPORT & SERIAL_SUPPORT
  if (key == ',')
      key = '+'; /* VTxxx keyboard has no '+' on numeric pad */
#endif

  if (key == '+' || key == '-')
      return prev_next_definition (key == '+', copy_gujin_param.attrib);
    else if (key == '*' || key == '/')
      return prev_next_color (key == '*', copy_gujin_param.attrib);
    else if (key == '\t')
      return prev_next_alternate (copy_gujin_param.attrib);
    else /* key == '.' */
      return switch_text_graphic (copy_gujin_param.attrib);
  }


static unsigned inline
change_video_mode (unsigned char key, unsigned update_bootparam, struct gujin_param_attrib gujin_attr)
  {
  unsigned result, mode;
  int found = find_video_mode (key);

#if USER_SUPPORT & VESA_SUPPORT
  unsigned short linear_vesa;

  if ((UI.info.vesa.version >> 8) >= 2 && gujin_attr.VESA2_interface)
      linear_vesa = 0x4000;
    else
      linear_vesa = 0;
#else /* VESA_SUPPORT */
  /* inline */ const unsigned short linear_vesa = 0;
#endif /* VESA_SUPPORT */

  if (found < 0) {
      PRINT (NO_MODE_FOUND);
      return 1;
      }
#if USER_SUPPORT & VESA_EDID
  if (   (   UI.edid.horizontal_resolution
	  && UI.edid.vertical_resolution)
      && (   UI.mode[found].width > UI.edid.horizontal_resolution
	  || UI.mode[found].height > UI.edid.vertical_resolution)) {
      PRINTF (MODE_X_U_U_OVER_MONITOR_SPEC,
		UI.mode[found].number,
		UI.mode[found].width,
		UI.mode[found].height);
      return 2;
      }
#endif /* VESA_EDID */
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
  if (UI.mode[found].vesa) {
      if (VGA_ACTIVE()) {
	  if (VESA_init() != 0) {
	      PRINTF (ERROR_VGAVESA_SWITCH_FAILED, "VESA");
	      return 3;
	      }
	  }
      }
    else {
      if (VESA_ACTIVE() && BOOT1_COM_port() < 0) {
	  if (VGA_init() != 0) {
	      PRINTF (ERROR_VGAVESA_SWITCH_FAILED, "VGA");
	      return 4;
	      }
	  }
      linear_vesa = 0;
      }
#endif /* VESA_SUPPORT && VGA_SUPPORT */

  /* do not clear the screen : */
  mode = UI.mode[found].number | linear_vesa | 0x8000;
  PRINTF (SETTING_MODE_X, UI.mode[found].number | linear_vesa);

  result = UI.function.setmode (mode);
  if (result != 0) {
      PRINTF (": %s 0x%X", WORD_ERROR, result);
      return 5;
      }
  mode &= ~0x8000;
  if (update_bootparam) {
      copy_gujin_param.default_video_mode = mode;
      if (UI.parameter.attr.isgraphic)
	  copy_gujin_param.default_graphic_video_mode = mode;
	else
	  copy_gujin_param.default_text_video_mode = mode;
      }
  return 0;
  }

/**
 ** Decide which mode to set initially, never re-used after init
 **/
static inline int
best_max_width (unsigned short maxwidth,
		struct gujin_param_attrib gujin_attr)
  {
  struct videomode_str *array;
  unsigned char cpt;
  int found = -1;

  for (array = UI.mode, cpt = 0; cpt < UI.nbmode; cpt++, array++) {
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
      if (array->vesa? !gujin_attr.VESA_interface
		     : !gujin_attr.VGA_interface)
	  continue;
#endif
      if (array->width > maxwidth)
	  continue;
      if (found == -1)
	  found = cpt;
	else if (array->width >= UI.mode[found].width) {
	  if (   array->height > UI.mode[found].height
	      || array->bpp > UI.mode[found].bpp)
	      found = cpt;
	  }
      }
  DBG (("\r\n%s: found mode index %u (maxwidth %u) out of UI.nbmode = %u\r\n",
	__FUNCTION__, found, maxwidth, UI.nbmode));
  return found;
  }

static inline int
next_lower_quality (unsigned short maxheight,
		    unsigned short maxwidth,
		    unsigned short maxbpp,
		    unsigned char  lastfound,
		    struct gujin_param_attrib gujin_attr)
  {
  struct videomode_str *array;
  unsigned char cpt;
  int found = -1;

  for (array = UI.mode, cpt = 0; cpt < UI.nbmode; cpt++, array++) {
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
      if (array->vesa? !gujin_attr.VESA_interface
		     : !gujin_attr.VGA_interface)
	  continue;
#endif
      if (array->height > maxheight)
	  continue;
      if (array->width > maxwidth)
	  continue;
      if (array->bpp > maxbpp)
	  continue;
      if (cpt == lastfound)
	  continue;
      if (found == -1)
	  found = cpt;
	else if (   array->height >= UI.mode[found].height
		 || array->width >= UI.mode[found].width
		 || array->bpp >= UI.mode[found].bpp)
	  found = cpt;
      }
  return found;
  }

/* NOTE:
 * We are maybe still in an invalid mode, for instance VGA
 * graphic with more than 256 colors, no S/Getpixel,
 * no cursor positionning... take care.
 * (see 'detect_interface()')
 */
static void
set_initial_mode (struct gujin_param_attrib gujin_attr, unsigned shall_complain)
  {
  unsigned short mode = copy_gujin_param.default_video_mode, tmp, nb_inch;

  if (BOOT1_COM_port() >= 0) {
      if (mode == 0xFFFF)
	  mode = 0;
      if ((tmp = UI.function.setmode (mode)) != 0) {
	  VDBG (("Setting serial default mode 0x%X failed: 0x%X\r\n", mode, tmp));
	  if ((tmp = UI.function.setmode (0)) != 0)
	      VDBG (("Setting serial default mode 0 failed: 0x%X\r\n", tmp));
	  }
      return;
      }

  if (mode >= 0x100) {
      // UI.info.vesa.version is still null because still not initialised...
      if (!gujin_attr.VESA_interface)
	  mode = 0xFFFF;
      }
#if USER_SUPPORT & VGA_SUPPORT
    else {
      if (!VIDEO_mode_is_valid(mode))
	  mode = 0xFFFF;
      }
#endif

  if (mode != 0xFFFF) {
#if USER_SUPPORT & VESA_SUPPORT
      if (mode >= 0x100)
	  VESA_init();
#endif
      if ((tmp = UI.function.setmode (mode)) == 0) {
#if (SETUP & UPDATE_BOOTPARAM) && (USER_SUPPORT & VGA_SUPPORT)
	  restore_gujin_param_vga_mode();
#endif
	  return;
	  }
      DBG (("Setting default mode 0x%X (in %s) failed: 0x%X\r\n", mode, (mode >= 0x100)? "VESA" : "VGA", tmp));
#if USER_SUPPORT & VESA_SUPPORT
      if (mode < 0x100) {
	  DBG (("Forcing to VESA and try again\r\n"));
	  VESA_init();
	  if ((tmp = UI.function.setmode (mode)) == 0) {
#if (SETUP & UPDATE_BOOTPARAM) && (USER_SUPPORT & VGA_SUPPORT)
	      restore_gujin_param_vga_mode();
#endif
	      return;
	      }
	  DBG (("Setting default mode 0x%X (forced in VESA) failed: 0x%X\r\n", mode, tmp));
	  }
#endif
      }

  /* That is the first power-on mode selection: */
#if USER_SUPPORT & VESA_EDID
  if (UI.edid.Hsize_cm != 0 && UI.edid.Vsize_cm != 0) {
      /*
       * After asking to a professionnal mathematician, we
       * suppose a display size approx 4/3 :
       *  sqrt(sqr(Vsize_cm) + sqr(Hsize_cm)) == 5/3*Vsize_cm
       * 50 inches = 50 * 2.54 = 127 cm ; and 50 * 5 == 2 * 127
       * We round up the result.
       */
      nb_inch = (UI.edid.Vsize_cm + 3 * UI.edid.Hsize_cm / 4 + 2) / 3;
      }
    else
#endif /* VESA_EDID */
      nb_inch = 13;

#if USER_SUPPORT & VESA_SUPPORT
  if (UI.info.vesa.version != 0) {
      unsigned short maxwidth;
      int found;

      if (nb_inch < 14)
	  maxwidth = 640;
	else if (nb_inch < 15)
	  maxwidth = 800;
	else if (nb_inch < 16)
	  maxwidth = 1024;
	else if (nb_inch < 18)
	  maxwidth = 1152;
	else
	  maxwidth = 1280;

#if USER_SUPPORT & VESA_EDID
      if (   UI.edid.Hsize_cm != 0
	  && UI.edid.Vsize_cm != 0
	  && maxwidth > UI.edid.horizontal_resolution)
	  maxwidth = UI.edid.horizontal_resolution;
#endif /* VESA_EDID */

      found = best_max_width (maxwidth, gujin_attr);
      if (found != -1) {
	  if (!VESA_ACTIVE()) // && UI.mode[found].vesa
	      VESA_init();
	  do {
	      unsigned short mode = UI.mode[found].number;
	      /* UI.mode[] updated if setmode error: */
	      unsigned short maxheight = UI.mode[found].height,
			 maxbpp = UI.mode[found].bpp;
	      maxwidth = UI.mode[found].width;

	      if ((UI.info.vesa.version >> 8) >= 2 && copy_gujin_param.attrib.VESA2_interface)
		  mode |= 0x4000;
	      tmp = UI.function.setmode (mode);
	      if (tmp) {
		  DBG (("Setting mode 0x%X failed: 0x%X\r\n", mode, tmp));
		  if (mode & 0x4000) {
		      mode &= ~0x4000;
		      tmp = UI.function.setmode (mode);
		      if (tmp)
			  DBG (("Setting mode 0x%X failed: 0x%X\r\n", mode, tmp));
		      }
		  }
	      if (tmp == 0) {
		  copy_gujin_param.default_video_mode = mode;
		  if (UI.parameter.attr.isgraphic)
		      copy_gujin_param.default_graphic_video_mode = UI.mode[found].number;
		    else
		      copy_gujin_param.default_text_video_mode = UI.mode[found].number;
		  return;
		  }
	      found = next_lower_quality (maxheight, maxwidth, maxbpp,
					(unsigned char)found, gujin_attr);
	      } while (found != -1);
	  }
      DBG (("No even one good VESA mode found!\r\n"));
      }
#endif /* VESA_SUPPORT */

#if USER_SUPPORT & VGA_SUPPORT
  if (VESA_ACTIVE())
      VGA_init();

  {
  unsigned char nbcol, page;
  mode = _VGA_getmode (&nbcol, &page) & 0x7FU;
  }

  /* No default mode selected and no working VESA */
  UI.function.setmode (mode);

#if (SETUP & UPDATE_BOOTPARAM) && (USER_SUPPORT & VGA_SUPPORT)
  copy_gujin_param.default_video_mode = mode;
  setup_gujin_param_vga_mode ();
#endif

#else /* VGA_SUPPORT */
  puts (NEED_VGA_SUPPORT);
  _BIOS_wait (4 * 1000000);
  BIOS_killme();
#endif /* VGA_SUPPORT */
  }

static void menu_reset_saved_parameters (void)
  {
  struct gujin_param_attrib new_gujin_attr = {
#define Apply(field, deflt)	new_gujin_attr.field = deflt,
      APPLY_ATTRIBUTE_ORDER(Apply)
#undef Apply
      };
  if (!copy_gujin_param.attrib.disk_write_enable && copy_gujin_param.attrib.write_disabled_written_once)
      new_gujin_attr.write_disabled_written_once = 1;

  FORCED_WHEN_DOS(BOOT1_DOS_running(), new_gujin_attr);
  copy_gujin_param.attrib = new_gujin_attr;
  copy_gujin_param.timeout_autoload = 0;

#if USER_SUPPORT & VGA_SUPPORT
  VIDEO_mode_reset();
#endif
  }

/**
 ** Menu management:
 **/
static void centered_print (const char *str, unsigned row, unsigned nbcol)
  {
  char line[2 * UI.parameter.nbcol + 1]; /* mange UTF-8 up to 2 bytes/chars */

  while (*str != '\0') {
      char *ptr = line;
      while (*str != '\0') {
	  if (*str == '\r') {
	      str++;
	      continue;
	      }
	  if (*str == '\n') {
	      str++;
	      break;
	      }
	  if (ptr >= &line[sizeof(line)-2])
	      return; /* too long message */
	  *ptr++ = *str++;
	  }
      *ptr = '\0';
      unsigned col = (nbcol - NbUtf8Char(line)) / 2; /* stops "movl UI.function.setcursor,%ecx ; movl %ecx,-28(%ebp) ; call *-28(%ebp)" */
      UI.function.setcursor (row++, col);
      print (line);
      }
  }

static inline void redraw_first_lines (unsigned refresh)
  {
  const char *msg2 = "", *msg1;
  char topline[100];
  unsigned length;

  draw_bg_box (1, 1, FIRST_LINES_BACKGROUND);

  if (BOOT1_COM_port() >= 0)
      msg1 = "";
    else
#if USER_SUPPORT & VGA_SUPPORT
	 if (VGA_ACTIVE()) {
      if (UI.videocard._VGA_get_functionality_info_works)
	  msg1 = "[VGA]";
	else if (UI.videocard._EGA_getdisplay_works)
	  msg1 = "[EGA]";
	else
	  msg1 = "[CGA]";
      }
    else
#endif
#if USER_SUPPORT & VESA_SUPPORT
	 if (!VESA_ACTIVE())
      msg1 = "[]";
    else
#if USER_SUPPORT & VESA_LINEAR
	 if (UI.parameter.winsize == VESA2_MARKER)
      msg1 = "[lin]"; /* using VESA2 linear framebuffer */
    else
#endif
	 if ((unsigned) UI.parameter.linelength * UI.parameter.height < 64*1024)
      msg1 = "[nowin]"; /* no window switching needed */
    else
#if USER_SUPPORT & VESA_WINFUNCTION
	 if (UI.parameter.addrpositionfarfct == 0)
	  msg1 = "[int]"; /* using VESA interrupt to change windows */
    else
#if USER_SUPPORT & VESA_HARDWINDOW
	 if (UI.parameter.hardwindow) {
      if (UI.parameter.offsetarray)
	  msg1 = "[hard]"; /* using hard method to change windows */
	else
	  msg1 = "[hardc]"; /* using hard compressed method to change windows */
      }
    else
#endif /* VESA_HARDWINDOW */
      msg1 = "[fct]"; /* using VESA function to change windows */
#else /* VESA_WINFUNCTION */
      msg1 = "[int]"; /* using VESA interrupt to change windows */
#endif /* VESA_WINFUNCTION */

#if USER_SUPPORT & VESA_2WINDOWS
  if (   UI.parameter.NbWin == 2
      && (unsigned) UI.parameter.linelength * UI.parameter.height >= 64*1024) {
#if USER_SUPPORT & VESA_HARDWINDOW
      if (UI.parameter.readhardwindow) {
	  if (UI.parameter.readoffsetarray)
	      msg2 = "[hard]"; /* using hard method to change windows */
	    else
	      msg2 = "[hardc]"; /* using hard compressed method to do so */
	  }
	else
#endif /* VESA_HARDWINDOW */
	  msg2 = msg1;
      }
#endif /* VESA_2WINDOWS */
#else /* VESA_SUPPORT */
      msg1 = "[]";
#endif /* VESA_SUPPORT */

  sprintf (topline, MSG_TOPLINE,
	   UI.parameter.identification,
	   UI.parameter.attr.isgraphic? MSG_GRAPHIC : MSG_TEXT,
	   msg1, msg2);
  length = NbUtf8Char (topline);

  const char *str_tmp = TOPLINE_CHARSIZE;
  if (length + NbUtf8Char (str_tmp) + sizeof (" 12:34:56 ") < UI.parameter.nbcol) {
      sprintf (&topline[strlen(topline)], str_tmp, UI.parameter.charwidth, UI.parameter.charheight);
      length = NbUtf8Char (topline);
      }
  str_tmp = TOPLINE_NBMODES;
  if (length + NbUtf8Char (str_tmp) + sizeof (" 12:34:56 ") < UI.parameter.nbcol) {
      sprintf (&topline[strlen(topline)], str_tmp, UI.nbmode);
      length = NbUtf8Char (topline);
      }
  str_tmp = TOPLINE_REFRESH;
  if (length + NbUtf8Char (str_tmp) + sizeof (" 12:34:56 ") < UI.parameter.nbcol && refresh != 0)
      sprintf (&topline[strlen(topline)], str_tmp, refresh);
  centered_print (topline, 1, UI.parameter.nbcol - sizeof ("12:34:56"));
  }

static unsigned redraw_top_field (int field, unsigned short row, unsigned short col)
  {
  char middle_buffer[40];
  const char *startstr = 0;
  unsigned    strlen_startstr;
  const char *field1name;	/* always exist */
  unsigned    strlen_field1name;
  unsigned    strlen_middlestr = 0;
  const char *field2name = 0;
  unsigned    strlen_field2name;
  unsigned short field1key __attribute__ ((unused));
  unsigned short field2key = field2key; /* inited B4 used */

  MOUSE_declare_attribute (mouse_attr1);
  MOUSE_declare_attribute (mouse_attr2);
  MOUSE_attrset_upperfield (mouse_attr1, 1);
  MOUSE_attrset_upperfield (mouse_attr2, 1);

  unsigned short cpt, cptgraphic, cpttext;

  for (cpt = cpttext = cptgraphic = 0; cpt < UI.nbmode; cpt++)
      if (UI.mode[cpt].text)
	  cpttext++;
	else
	  cptgraphic++;
  unsigned char has_text_graphic_menu = !!((UI.parameter.attr.isgraphic? cpttext : cptgraphic) > 0);
  unsigned char has_alternate_menu = !(find_video_mode ('\t') == -1);

  if (field == 0) { /* Returns the number of fields to draw */
#if SETUP & MULTILINGUAL
      return 4 + has_text_graphic_menu + has_alternate_menu;
#else
      return 3 + has_text_graphic_menu + has_alternate_menu;
#endif
      }
  if (field == -1 || field == 1) {
      startstr = MENU_RESOLUTION;
      strlen_startstr = NbUtf8Char (startstr);
      field1name = "<->";
      strlen_field1name = sizeof ("<->") - 1;
      MOUSE_attrset_activefield (mouse_attr1, find_video_mode ('-') != -1);
      field1key = MINUS_KEYCODE();
      /* With our implementation, UI.parameter.{width,height} is
	equal to UI.parameter.{nbcol,nbrow} in text modes */
      strlen_middlestr = sprintf (middle_buffer, "%ux%u ", UI.parameter.width, UI.parameter.height);
      field2name = "<+>";
      strlen_field2name = sizeof ("<+>") - 1;
      MOUSE_attrset_activefield (mouse_attr2, find_video_mode ('+') != -1);
      field2key = PLUS_KEYCODE();
      }
    else if (field == -2 || field == 2) {
      startstr = MENU_COLORS;
      strlen_startstr = NbUtf8Char (startstr);
      field1name = "</>";
      strlen_field1name = sizeof ("</>") - 1;
      MOUSE_attrset_activefield (mouse_attr1, find_video_mode ('/') != -1);
      field1key = DIVIDE_KEYCODE();
      strlen_middlestr = sprintf (middle_buffer, MENU_DISPLAY_COLORS, get_bpp (&UI.parameter));
      field2name = "<*>";
      strlen_field2name = sizeof ("<*>") - 1;
      MOUSE_attrset_activefield (mouse_attr2, find_video_mode ('*') != -1);
      field2key = MULTIPLY_KEYCODE();
      }
#if SETUP & MULTILINGUAL
    else if (field == -3 || field == 3) {
      /* We just use middle_buffer as a temporary, do not set strlen_middlestr */
      const char *kbdmap_str = kbdname (copy_gujin_param.kbdmap), *langmsg = MENU_LANGUAGE_KEYBOARD;
      if (!kbdmap_str)
	  kbdmap_str = "??";
      if (UTIL.keyboard == serialkbd)
	  langmsg = MENU_LANGUAGE;
      sprintf (middle_buffer, langmsg, LANG_CODE, kbdmap_str);
      startstr = middle_buffer;
      //startstr = MENU_LANGUAGE;
      strlen_startstr = NbUtf8Char (startstr);
      strlen_field2name = 0;
      field1name = MENU_NEXTLANG;
      strlen_field1name = NbUtf8Char (field1name);
      MOUSE_attrset_activefield (mouse_attr1, 1);
      field1key = CTRL_T_KEYCODE();
      }
#endif
#if SETUP & MULTILINGUAL
    else if (field == -4 || field == 4) {
#else
    else if (field == -3 || field == 3) {
#endif
      strlen_startstr = 0;
      strlen_field2name = 0;
      if (UI.attributes.valid.transparent) {
	  /* can do 3D buttons */
	  MOUSE_attrset_pressedfield(mouse_attr1, (Menu.type != kernel_bottom_menu));
	  field1name = MENU_SETUP;
	  }
	else {
	  if (Menu.type == kernel_bottom_menu)
	      field1name = MENU_SETUP;
	    else
	      field1name = MENU_APPLICATION;
	  }
      strlen_field1name = NbUtf8Char (field1name);
      MOUSE_attrset_activefield (mouse_attr1, 1);
      field1key = SPACE_KEYCODE();
      }
#if SETUP & MULTILINGUAL
    else if ((field == -5 || field == 5) && has_text_graphic_menu) {
#else
    else if ((field == -4 || field == 4) && has_text_graphic_menu) {
#endif
      strlen_startstr = 0;
      strlen_field2name = 0;
      if (UI.parameter.attr.isgraphic) {
	  field1name = MENU_TO_TEXT;
	  MOUSE_attrset_activefield (mouse_attr1, 1);
	  }
	else {
	  field1name = MENU_TO_GRAPHIC;
	  MOUSE_attrset_activefield (mouse_attr1, 1);
	  }
      strlen_field1name = NbUtf8Char (field1name);
      field1key = DOT_KEYCODE();
      }
#if SETUP & MULTILINGUAL
    else if (has_alternate_menu && (has_text_graphic_menu? (field == -6 || field == 6) : (field == -5 || field == 5))) {
#else
    else if (has_alternate_menu && (has_text_graphic_menu? (field == -5 || field == 5) : (field == -4 || field == 4))) {
#endif
      strlen_startstr = 0;
      strlen_field2name = 0;
      field1name = MENU_ALTERNATE;
      strlen_field1name = NbUtf8Char (field1name);
      MOUSE_attrset_activefield (mouse_attr1, 1);
      field1key = TAB_KEYCODE();
      }
    else
      return 0;

  if (field > 0) {
      if (strlen_startstr) {
	  print (startstr);
	  col += strlen_startstr;
	  }
      PUT_FIELD (field1name, mouse_attr1, field1key, row, col);
      col += strlen_field1name + 1;
      UI.function.setcursor (row, col);
      if (strlen_middlestr) {
	  print (middle_buffer);
	  col += strlen_middlestr;
	  }
      if (strlen_field2name) {
	  PUT_FIELD (field2name, mouse_attr2, field2key, row, col);
	  }
      return 0;
      }
    else {
      unsigned len = strlen_field1name + 1;
//      if (strlen_startstr)
	  len += strlen_startstr;
//      if (strlen_middlestr)
	  len += strlen_middlestr;
      if (field2name)
	  len += strlen_field2name + 1;
      return len + 1; /* at least one space after */
      }
  }

static unsigned short
redraw_top_menu (int curline)
  {
  unsigned space[redraw_top_field(0, 0, 0)];
  unsigned curfield, just_report, oneperline = 0;

  for (curfield = 0; curfield < nbof (space); curfield++)
      space[curfield] = redraw_top_field (-(curfield + 1), 0, 0);

  if (curline != 0) {
      unsigned nblines = redraw_top_menu (0);
      just_report = 0;
      draw_bg_box (curline, nblines, TOP_MENU_BACKGROUND);
      }
    else
      just_report = 1;

  for (curfield = 0; curfield < nbof (space); curline ++) {
      unsigned field = curfield, interspace, curcol;
      unsigned freespace = UI.parameter.nbcol ?: 40;

      while (field < nbof (space)) {
	  if (freespace < space[field])
	      break;
	  freespace -= space[field++];
	  if (oneperline)
	      break;
	  }
      if (field == 1)
	  oneperline = 1;
      interspace = freespace / (field - curfield + 1);
      curcol = interspace + (freespace % (field - curfield + 1))/2;
      while (curfield < field) {
	  if (!just_report) {
		/* setcursor: home at 0,0 => add 1 to curcol */
	      UI.function.setcursor (curline, curcol + 1);
	      redraw_top_field (curfield + 1, curline, curcol + 1);
	      curcol += space[curfield] + interspace;
	      }
	  curfield ++;
	  }
      }
  return curline;
  }

static inline unsigned
begin_line_kernel_field (char *buffer, struct diskparam_str *dp, struct desc_str *ptr,
			struct gujin_param_attrib gujin_attr)
  {
  char *buf = buffer;

  if (gujin_attr.menu_with_disk) {
      unsigned char *ptr = dp->diskname;

      while (*ptr) *buf++ = *ptr++;
      if (dp->nbpartition == 0)
	  buf = STRINIT (buf, " ");	/* that is better after DOS disks */
	else
	  buf = STRINIT (buf, ": ");
      }
  if (gujin_attr.menu_with_parttype && dp->nbpartition != 0) {	/* not after "DOS C:" */
      if (ptr->boottype == is_MBR)
	  buf = STRINIT (buf, "MBR: ");
	else if (ptr->boottype == is_el_torito)
	  buf = STRINIT (buf, "El Torito: ");
	else if (ptr->boottype == is_bdi_file)
	  buf = STRINIT (buf, "Boot Device Image: ");
	else {
#if 0 /* code size optimise */
	  const char *parttypename;
#define P(ident, nb, str) if (nb == dp->partition[ptr->partition].type) parttypename = str; else
	PARTITIONLIST()	parttypename = "unknown";
#undef P
	  while (*parttypename) *buf++ = *parttypename++;
#else
	  static const unsigned char partnb[] = {
#define P(ident, nb, str)	nb,
	      PARTITIONLIST()
#undef P
	      };
	  static const char partname_concat[] = " unknown\0"
#define P(ident, nb, str)	str "\0"
		PARTITIONLIST()
#undef P
		;
	  const char *parttypename = partname_concat;
	  int cpt = nbof (partnb);

	  while (--cpt >= 0)
	      if (partnb[cpt] == dp->partition[ptr->partition].type) {
		  do do parttypename++; while (*parttypename); while (--cpt >= 0);
		  break;
		  }
	  while (*++parttypename) *buf++ = *parttypename;
#endif

	  buf = STRINIT (buf, ": ");
	  }
      }
  if (gujin_attr.menu_with_partition && ptr->boottype != is_MBR) {	/* not after "MBR:" */
      char sep, *nameptr;

      if (dp->nbpartition == 0) {
	  /*
	   * Dirty if (access == dos_part), the "partition" pointer is
	   * used as the DOS name pointer because nbpartition == 0.
	   */
	  sep = '"';
	  nameptr = (char *)dp->partition;
	  }
	else {
	  struct partition_str *curpart = &dp->partition[ptr->partition];

	  nameptr = (char *)curpart->name;
	  if (curpart->misc.beer_partition)
	      sep = '`';
	    else if (curpart->misc.active == part_active)
	      sep = '"';
	    else
	      sep = '\'';
	  }
      if (nameptr) {
	  *buf++ = sep;
	  while (*nameptr) *buf++ = *nameptr++;
	  *buf++ = sep;
	  }
      buf = STRINIT (buf, ": ");
      }
  *buf = 0;
  return buf - buffer;
  }

static inline void
redraw_kernel_field (int field, unsigned short row, unsigned short col)
  {
  /* This assumes that initrd's are at the end of the array */
  struct desc_str *ptr = &BOOTWAY.desc[ABS(field) - 1];
  char buffer[128], *dst = buffer; /* Hopes it is enough */

  MOUSE_declare_attribute (mouse_attr);

  begin_line_kernel_field (buffer, &DI.param[ptr->disk], ptr, copy_gujin_param.attrib);
  print (buffer);
  col += NbUtf8Char (buffer);

#if defined (INITRDNAME) || defined (INITRDNAME2)
  unsigned index = 0;
  if (ptr->boottype == is_linux || ptr->boottype == is_elf)
      index = initrd_file (ABS(field) - 1);
#endif

#ifdef COMMANDFILE
  unsigned ret;
  if (ptr->boottype == is_linux
	&& (ret = commandfile_get (ptr->iso_fsname ?: (char *)DI.param[ptr->disk].partition[ptr->partition].name, ptr->filename,
		index? BOOTWAY.desc[index].filename : 0, cmdf_getmenuname, ptr->variant_number, buffer, sizeof(buffer)-1)) != 0) {
      dst = buffer;
      /* search the last star in dst and replace it by the common part of the vmlinuz filename: */
      char *starptr = 0, *dstptr = dst;
      while (*dstptr) {
	  if (*dstptr == '*')
	      starptr = dstptr;
	  dstptr++;
	  }
      if (starptr) {
	  char *fnsrc = &ptr->filename[ptr->name_offset], first = 1;
	  if (*fnsrc == '-')
	      fnsrc++;
#if 0
	  while (*fnsrc && strchr("0123456789-.", *fnsrc)) {
#else
	  char *fnend = fnsrc;
	  while (*fnend && *fnend != '-')
	      fnend++;
//	  to get same code size if -fsigned-char or -funsigned-char, do the cast:
//	  if (fnend[0] == '-' && fnend[1] >= '0' && fnend[1] <= '9') {
	  if (fnend[0] == '-' && (unsigned char)fnend[1] >= '0' && (unsigned char)fnend[1] <= '9') {
	      fnend++;
	      while (*fnend >= '0' && *fnend <= '9')
		  fnend++;
	      }
	  if (*(unsigned *)fnend == *(unsigned *)"-pae")
	      fnend += 4;
	  while (*fnsrc && fnsrc < fnend) {
#endif
	      char *xptr = dstptr;
	      while (!first && xptr >= starptr) {
		  xptr[1] = xptr[0];
		  xptr--;
		  }
	      *starptr = *fnsrc++;
	      dstptr++;
	      starptr++;
	      first = 0;
	      }
	  }
      while (*dst++ != 0)
	  continue;
      }
    else
#endif

#if defined (INITRDNAME) || defined (INITRDNAME2)
  if (  copy_gujin_param.attrib.menu_with_initrd && index != 0) {
     /* "/boot/{vmlinuz,initrd}-2.24.0{,.img}" - but skip ".img" for initrd, whatever position */
     char *fnsrc = ptr->filename, *insrc = BOOTWAY.desc[index].filename;
      while (*fnsrc == *insrc && *fnsrc) {
	  *dst++ = *fnsrc++;
	  insrc++;
	  }
      *dst++ = '{';
      unsigned nb = ptr->name_offset - (fnsrc - ptr->filename) + 1;
      while (--nb)
	  *dst++ = *fnsrc++;
      *dst++ = ',';
      nb = BOOTWAY.desc[index].name_offset - (insrc - BOOTWAY.desc[index].filename) + 1;
      while (--nb)
	  *dst++ = *insrc++;
      if (*(unsigned *)insrc == *(unsigned *)".img")
	  insrc += 4;
      *dst++ = '}';
      while (*fnsrc == *insrc && *fnsrc) {
	  *dst++ = *fnsrc++;
	  insrc++;
	  }
      if (*(unsigned *)insrc == *(unsigned *)".img")
	  insrc += 4;
      if (*fnsrc || *insrc) {
	  *dst++ = '{';
	  while (*fnsrc)
	      *dst++ = *fnsrc++;
	  *dst++ = ',';
	  while (*insrc)
	      *dst++ = *insrc++;
	  *dst++ = '}';
	  }
      *dst = '\0';
      }
    else
#endif
      {
      char *src = ptr->filename;
      while (*src)
	  *dst++ = *src++;
      *dst = '\0';
      }

  if (dst - buffer >= UI.parameter.nbcol - col)
      *(unsigned *)&buffer[UI.parameter.nbcol - col - 4] = *(unsigned *)"...";

  MOUSE_attrset_mainfield (mouse_attr, 1);
  MOUSE_attrset_activefield (mouse_attr, 1);
  PUT_FIELD (buffer, mouse_attr,
	     FCT_KEYCODE(ABS(field) - Menu.curpage * Menu.nbperpage),
	     row, col);
  }

static inline unsigned
attr_is_greyed (unsigned absnb)
  {
  if (BOOT1_DOS_running()) {
      if (   absnb == enum_search_disk_mbr
	  || absnb == enum_search_part_mbr
	  || absnb == enum_probe_ide_disk
	  || absnb == enum_probe_cdrom)
	  return 1;
      }
  return 0;
  }

static inline int
redraw_setup_attribute_field (int field, unsigned short keyNo,
			      unsigned short row, unsigned short col)
  {
  const char *startstr = 0;
  char active = 0;
  unsigned absnb = ABS(field) - 1;

#if SETUP & MULTILINGUAL /* code size optimise, cannot use it if strings not constant */
  if (absnb == enum_probe_dos_disk && !BOOT1_DOS_running()) {
      startstr = SMENU_probe_BDI_FILE;
      active = copy_gujin_param.attrib.probe_dos_disk;
      }
    else if (absnb == enum_disk_write_enable && UTIL.boot_device_read_only) {
      startstr = SMENU_disk_write_enable_but_bootdevice;
      active = copy_gujin_param.attrib.disk_write_enable;
      }
    else
      switch (absnb) {
#define APPLY_REDRAW(field, deflt) \
	case enum_ ## field: startstr = SMENU_ ## field ; \
		active = copy_gujin_param.attrib.field ; break;

      APPLY_ATTRIBUTE_ORDER(APPLY_REDRAW)
#undef APPLY_REDRAW
	  }

#else /* code size optimise */
  static const char * const Smenu[] = {
#define APPLY_STRING(field, deflt)	SMENU_ ## field,
      APPLY_ATTRIBUTE_ORDER(APPLY_STRING)
#undef APPLY_STRING
      };
  union { unsigned value; struct gujin_param_attrib params; } tmp;

  if (absnb == enum_probe_dos_disk && !BOOT1_DOS_running())
      startstr = SMENU_probe_BDI_FILE;
#if 1
    else if (absnb == enum_disk_write_enable && UTIL.boot_device_read_only)
      startstr = SMENU_disk_write_enable_but_bootdevice;
#else
    else if (absnb == enum_disk_write_enable && UTIL.boot_device_read_only) {
      const char *msg = SMENU_disk_write_enable_but_bootdevice;
      unsigned len = strlen (msg), idx;
      static char newmsg[80];
      for (idx = 0; idx < len-2 && idx < sizeof(newmsg)-6; idx++)
	  newmsg[idx] = msg[idx];
      newmsg[idx++] = ' ';
      newmsg[idx++] = '0' + UTIL.boot_device_read_only/10;
      newmsg[idx++] = '0' + UTIL.boot_device_read_only%10;
      newmsg[idx++] = ')';
      newmsg[idx++] = '\0';
      startstr = newmsg;
      }
#endif
    else
      startstr = Smenu[absnb];
  tmp.params = copy_gujin_param.attrib;
  active = ((tmp.value & (1U << absnb)) != 0);

#endif /* code size optimise */

  if (startstr) {
      char field[2];
      unsigned oldbgcolor = 0;
      MOUSE_declare_attribute (mouse_attr);

      if (field < 0)
	  return NbUtf8Char (startstr) + sizeof ("[X]") - 1
			+ 1 + 1;	/* clickable + space needed after */
      if (UI.attributes.valid.transparent) {
	  MOUSE_attrset_pressedfield (mouse_attr, 1);
	  oldbgcolor = UI.bgcolor;
	  UI.function.setbgcolor (UI.stdcolor[white]);
	  }
	else {
	  print ("[ ]");
	  col += 1; /* "[" */
	  UI.function.setcursor (row, col); /* \010 do not always work */
	  }
      if (!active)
	  field[0] = ' ';
//	else if (UI.parameter.attr.null_del_available) // still works if BOOT1_ansi_putstr() is called
	else if (!BOOT1_DOS_running() && BOOT1_COM_port() < 0)
	  field[0] = '\x7F'; /* box ticker */
//	else if (!UI.parameter.attr.ansi_font)
//	  field[0] = '\xFB'; /* square root box ticker, conversion handled by ansi2pc[]  */
	else
	  field[0] = 'X'; /* Do not know which codepage is active if DOS or serial... */
      field[1] = '\0';

      MOUSE_attrset_chkboxfield(mouse_attr, 1);
      MOUSE_attrset_mainfield (mouse_attr, 1);
      MOUSE_attrset_activefield (mouse_attr, 1);
      MOUSE_attrset_greyfield (mouse_attr, attr_is_greyed(absnb));
#if __GNUC__ == 4 && __GNUC_MINOR__ == 4 && __GNUC_PATCHLEVEL__ == 0
//      asm ("" : : : "memory"); /* stops "movl MOUSE.function.print_field, %ebx ; call *%ebx" */
      asm ("" : "=m" (MOUSE.function.print_field)); /* stops "movl MOUSE.function.print_field, %ebx ; call *%ebx" */
#endif
      PUT_FIELD (field, mouse_attr, FCT_KEYCODE(keyNo), row, col);
      if (UI.attributes.valid.transparent)
	  UI.function.setbgcolor (oldbgcolor);
      if (!UI.attributes.valid.transparent)
	  col += 1; /* "]" */
      UI.function.setcursor (row, col + 2);
      print (startstr);
      }
  return 0;
  }

static inline int
redraw_setup_action_field (int field, unsigned short keyNo,
			   unsigned short row, unsigned short col)
  {
  const char *fieldname;
  MOUSE_declare_attribute (mouse_attr);

  MOUSE_attrset_mainfield (mouse_attr, 1);

  switch ((field >= 0)? field : -field) {
      case 1:
	fieldname = SMENU_PROBE_VESA;
	MOUSE_attrset_activefield (mouse_attr, !!(USER_SUPPORT & VESA_SUPPORT));
	break;
      case 2:
	fieldname = SMENU_PROBE_VGA_0_13;
	MOUSE_attrset_activefield (mouse_attr, !!(USER_SUPPORT & VGA_SUPPORT));
	break;
      case 3:
	fieldname = SMENU_PROBE_VGA_14_127;
	MOUSE_attrset_activefield (mouse_attr, !!(USER_SUPPORT & VGA_SUPPORT));
	break;
      case 4:
	fieldname = SMENU_RESET_VGA_MODES;
	MOUSE_attrset_activefield (mouse_attr, !!(USER_SUPPORT & VGA_SUPPORT));
	break;
      case 5:
	fieldname = SMENU_RESET_ALL_PARAMS;
	MOUSE_attrset_activefield (mouse_attr, 1);
	break;
      case 6:
	fieldname = BOOT1_DOS_running()? SMENU_EXIT_TO_DOS : SMENU_POWER_DOWN;
	MOUSE_attrset_activefield (mouse_attr, 1);
	break;
      case 7:
	fieldname = SMENU_STOP_CDROMEMUL;
	MOUSE_attrset_activefield (mouse_attr, !BOOT1_DOS_running());
	break;
      case 8:
//#if defined (INITRDNAME) || defined (INITRDNAME2)
	fieldname = SMENU_SET_MIN_INIRTD_CHAR;
	MOUSE_attrset_activefield (mouse_attr, 1);
//#endif
	break;
      case 9:
	fieldname = SMENU_SET_TIME_OFFSET;
	MOUSE_attrset_activefield (mouse_attr, 1);
	break;
      case 10:
	fieldname = SMENU_SET_TIMEOUT_AUTOLOAD;
	MOUSE_attrset_activefield (mouse_attr, 1);
	break;
      case 11:
	fieldname = SMENU_SET_KEYBOARD;
	MOUSE_attrset_activefield (mouse_attr, !BOOT1_DOS_running()
					|| BOOT1_COM_port() >= 0);
	break;
      case 12:
	fieldname = SMENU_SET_COMMANDLINE;
	MOUSE_attrset_activefield (mouse_attr, 1);
	break;
      case 13:
	fieldname = SMENU_SET_PARSED_DIRECTORY_NAME;
	MOUSE_attrset_activefield (mouse_attr, 1);
	break;
      case 14:
	fieldname = SMENU_SET_ENDEXTPART;
	MOUSE_attrset_activefield (mouse_attr, !BOOT1_DOS_running());
	break;
#if SETUP & UPDATE_BOOTPARAM
      case 15:
	fieldname = SMENU_UNINSTALL;
	unsigned char active = STATE.has_just_been_uninstalled != 1367;
	extern bootloader2_t uninstall_mbr;
	if (active)
	    if (codeseg_peekw(&uninstall_mbr.header.nbword) != 256)
		active = 0;
	MOUSE_attrset_activefield (mouse_attr, active);
	break;
      case 0:
	return 15; /* number of fields */
#else
      case 0:
	return 14; /* number of fields */
#endif
      default:
	return 0;
      }

  if (field > 0) {
      PUT_FIELD (fieldname, mouse_attr, FCT_KEYCODE(keyNo), row, col);
      return 0;
      }
    else {
      return NbUtf8Char (fieldname) + 1 + 1; /* at least one space after */
      }
  }

static inline int
redraw_setup_field (int field, unsigned short row, unsigned short col)
  {
  unsigned short keyNo = ABS(field) - Menu.curpage * Menu.nbperpage;

  if (field == 0)
      return TOTAL_ATTRIBUTE + redraw_setup_action_field (0, 0, 0, 0);
    else if (ABS(field) <= TOTAL_ATTRIBUTE)
      return redraw_setup_attribute_field (field, keyNo, row, col);
    else
      return redraw_setup_action_field (field + ((field > 0)
						 ? - TOTAL_ATTRIBUTE
						 : TOTAL_ATTRIBUTE),
					keyNo, row, col);
  }

static inline void invert_setup_attribute_field (unsigned absnb)
  {
#if 0 /* code size optimise */
  switch (absnb) {
#define APPLY_INVERT(field, deflt) \
      case enum_ ## field: copy_gujin_param.attrib.field = !copy_gujin_param.attrib.field ; break;
      APPLY_ATTRIBUTE_ORDER(APPLY_INVERT)
#undef APPLY_INVERT
      }
#if !(DEBUG & (DEBUG_DISK|DEBUG_FS))	/* quick and dirty to get a log */
  FORCED_WHEN_DOS (BOOT1_DOS_running(), copy_gujin_param.attrib);
#endif
#else /* code size optimise */
  union { unsigned value; struct gujin_param_attrib params; } tmp;

  tmp.params = copy_gujin_param.attrib;
  tmp.value ^= 1U << absnb;
#if !(DEBUG & (DEBUG_DISK|DEBUG_FS))	/* quick and dirty to get a log */
  FORCED_WHEN_DOS (BOOT1_DOS_running(), tmp.params);
#endif
  copy_gujin_param.attrib = tmp.params;
#endif /* code size optimise */

#if USER_SUPPORT & JOYSTICK_SUPPORT
  if (absnb == enum_enable_joystick)
      MOUSE_joystick_init (tmp.params.enable_joystick);
#endif
  }

struct todraw_str {
    unsigned clearbackground		: 1;
    unsigned firstlines		: 1;
    unsigned topmenu		: 1;
    unsigned bottommenu		: 1;
    unsigned advertisement	: 1;
    unsigned copyright		: 1;
    unsigned update_bootparam	: 1;
    unsigned refresh		: 1;
    unsigned reparse		: 1;
    unsigned presskey		: 1;
    } __attribute__ ((packed));

static inline struct todraw_str
activate_setup_action_field (unsigned absnb, unsigned short advline)
  {
  unsigned char min, max;
  unsigned newbgcolor = UI.parameter.nbcolor <= 2? UI.bgcolor : UI.stdcolor[lightgray];

  switch (absnb) {
      case 1:
#if USER_SUPPORT & VESA_SUPPORT
	  if (BOOT1_COM_port() < 0) {
	      unsigned short i;

		/* FIXME: Is it needed to switch to VESA? */
		/* Call of VESA_getmode in VGA */
#if USER_SUPPORT & VGA_SUPPORT
	      unsigned in_vesa = VESA_ACTIVE();
	      unsigned short current_mode = current_mode;

	      if (!in_vesa) {
		  current_mode = UI.function.getmode() & ~0x8000;
		  if (VESA_init() != 0)
		      break;
		  }
#endif

#if USER_SUPPORT & VESA_WINDOW
	      {
	      VESA_modeinfo_t modeinfo;
	      unsigned tmp = 0;

	  /* "CHIPS 65520/525/530 Flat Panel VGA" on 486 portable T2130CS always
	  answer "no error" and the data of the current mode if mode <= 0x7F !!!
	  It is a VESA 1.2 card */
	      i = 0x80;
	      while (i-- && tmp == 0)
		  tmp = _VESA_getmodeinfo (&modeinfo, i);
	      if (tmp == 0) {
		  DBG ((" [Probing VESA 0..0x7F, never seen an error, "
			"this probe is not working] "));
		  }
		else {
		  DBG ((" [Probing VESA 0..0x7F, works (stop at 0x%X)] ", i));
#endif
		  for (i = 0; i < 0x80; i++) {
		      struct video_parameter_str param;
		      UI.function.getsize (i, &param);
		      }
#if USER_SUPPORT & VESA_WINDOW
		  }
	      }
#endif


#if USER_SUPPORT & VGA_SUPPORT
	      if (!in_vesa) {
		  VGA_init();
		  UI.function.setmode (current_mode);
		  }
#endif
	      }
#endif
	  /* if we have switch VGA->VESA->VGA: */
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 1,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam	= 0,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 0
		};
      case 2:
	  min = 0; max = 0x13;
	  goto case_2_3;
      case 3: {
	  unsigned short key;

	  if (advline + TURN_OFF_MONITOR_NBLINES >= UI.parameter.nbrow)
	      advline = UI.parameter.nbrow - TURN_OFF_MONITOR_NBLINES;
	  centered_print (TURN_OFF_MONITOR, advline, UI.parameter.nbcol);
	  centered_print (PRESS_KEY_TO_CONTINUE,
			advline + TURN_OFF_MONITOR_NBLINES, UI.parameter.nbcol);
	  key = UI.function.getkey (30 * TICKS_PER_SECOND);
	  if (key == TIMEOUT_KEYCODE() || (key & 0xFF) == '\033')
	      return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 0,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 0,
		.update_bootparam	= 0,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 0
		};
	  min = 0x14; max = 0x7F;
	  }
	  goto case_2_3;
      case_2_3:
#if USER_SUPPORT & VGA_SUPPORT
	  if (BOOT1_COM_port() < 0) {
	      struct video_parameter_str param;
	      unsigned char i;
	      unsigned short current_mode = UI.function.getmode() & ~0x8000;

#if USER_SUPPORT & VESA_SUPPORT
	      unsigned in_vesa = VESA_ACTIVE();

	      if (in_vesa && VGA_init() != 0)
		  break;
#endif
	      UI.function.setmode (3);
	      UI.function.setbgcolor (0); /* due to  probing */
	      UI.function.setfgcolor (newbgcolor);
	      UI.function.clearscreen();

	      for (i = min; i <= max; i++) {
		  unsigned err;
		  unsigned char j;

		  for (j = 0; j < UI.nbmode; j++)
		      if (i == UI.mode[j].number && UI.mode[j].vesa)
			  break;
		  if (j < UI.nbmode)
		      continue;
		  printf (SETTING_MODE_X, i);
		  err = UI.function.getsize (i, &param);
		  if (err)
		      printf (": %s 0x%X", WORD_ERROR, err);
#ifndef KEEP_ALL_VGA_MODE
		    else if (param.identification > 0x37 && param.nbcolor <= 4)
		      print (": <= 2 BPP");
#endif
		    else
		      printf (": %s.", WORD_OK);
		  puts ("");
		  }
	      print (MSG_BEEP);
#if SETUP & UPDATE_BOOTPARAM
	      setup_gujin_param_vga_mode ();
#endif

#if USER_SUPPORT & VESA_SUPPORT
	      if (in_vesa)
		  VESA_init();
#endif
	      UI.function.setmode (current_mode);
	      }
#endif
	  /* if we have switch VESA->VGA->VESA: */
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 1,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam	= 1,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 0
		};
      case 4: {
#if USER_SUPPORT & VGA_SUPPORT
	  VIDEO_mode_reset();
#endif
	  reset_low_valid_mode(128);
#if (SETUP & UPDATE_BOOTPARAM) && (USER_SUPPORT & VGA_SUPPORT)
	  setup_gujin_param_vga_mode ();
#endif
	  }
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 1,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 0,
		.update_bootparam	= 1,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 0
		};
      case 5:
	  menu_reset_saved_parameters();
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 0,
		.bottommenu	= 1, /* redraw the pressed button */
		.advertisement	= 1,
		.copyright	= 0,
		.update_bootparam	= 1,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 0
		};
      case 6:
	  if (!BOOT1_DOS_running()) {
	      APM_power_OFF();
	      _BIOS_power_OFF();
	      }
	    else
	      BIOS_killme();
	  /* if failed: */
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 1,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam	= 0,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 1
		};
      case 7: {
	  unsigned bgcolor = UI.bgcolor;
	  unsigned char tmpcol = UI.parameter.col;
	  const char *msg = MSG_STOP_CDROMEMUL;

	  UI.function.setbgcolor (newbgcolor);
	  printf (msg);
	  if (tmpcol == UI.parameter.col)
	      UI.parameter.col += NbUtf8Char (msg);
	  copy_gujin_param.stop_emulation = get_number (copy_gujin_param.stop_emulation, 2);
	  UI.function.setbgcolor (bgcolor);
	  }
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 0,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1, /* 40 cols mode */
		.update_bootparam	= 1,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 0
		};
      case 8: {
//#if defined (INITRDNAME) || defined (INITRDNAME2)
	  unsigned bgcolor = UI.bgcolor;
	  unsigned char tmpcol = UI.parameter.col;
	  const char *msg = MSG_MIN_NB_CHAR_INITRD;

	  UI.function.setbgcolor (newbgcolor);
	  printf (msg);
	  if (tmpcol == UI.parameter.col)
	      UI.parameter.col += NbUtf8Char (msg);
	  copy_gujin_param.min_nb_initrd = get_number (copy_gujin_param.min_nb_initrd, 9);
	  UI.function.setbgcolor (bgcolor);
//#endif
	  }
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 0,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1, /* 40 cols mode */
		.update_bootparam	= 1,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 0
		};
      case 9: {
	  unsigned bgcolor = UI.bgcolor;
	  unsigned char row, col, tmpcol = UI.parameter.col, badkey = 0;
	  const char *msg = MSG_TIME_OFFSET;

	  UI.function.setbgcolor (newbgcolor);
	  printf (msg);
	  if (tmpcol == UI.parameter.col) {
	      /* TOCHECK: is that posssible? probably,
		printf does not maintain cursor in all cases. */
	      /* Update UI.parameter.{row,col} in case
		getcursor not working (bad VT100 emulation): */
	      UI.parameter.col += NbUtf8Char (msg);
	      }
	  row = UI.parameter.row;
	  col = UI.parameter.col;
	  while (badkey == 0) {
	      unsigned short key;
	      printf ("%c%u ", UTIL.time_hour_offset >= 0? '+' : '-', UTIL.time_hour_offset >= 0? UTIL.time_hour_offset : -UTIL.time_hour_offset);
	      UI.function.setcursor (row, col);
	      key = UI.function.getkey (30 * TICKS_PER_SECOND);
	      if ((key & 0xFF) == '+')
		  UTIL.time_hour_offset = (UTIL.time_hour_offset == 12)? -12 : (UTIL.time_hour_offset+1);
		else if ((key & 0xFF) == '-')
		  UTIL.time_hour_offset = (UTIL.time_hour_offset == -12)? 12 : (UTIL.time_hour_offset-1);
		else {
		  printf (MSG_ERROR_KEYCODE, key);
		  badkey = 1;
		  }
	      };
	  if (badkey == 0)
	      copy_gujin_param.time_hour_offset = UTIL.time_hour_offset;
	  UI.function.setbgcolor (bgcolor);
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 0,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1, /* 40 cols mode */
		.update_bootparam	= 1,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= badkey
		};
	  }
      case 10: {
	  unsigned bgcolor = UI.bgcolor;
	  unsigned char tmpcol = UI.parameter.col;
	  const char *msg = MSG_TIMEOUT_AUTOLOAD;

	  UI.function.setbgcolor (newbgcolor);
	  print (msg);
	  if (tmpcol == UI.parameter.col)
	      UI.parameter.col += NbUtf8Char (msg);
	  unsigned char last_loaded = 0xFF, total_loadable = 0, init_page = 0, total_page = 0;
	  copy_gujin_param.timeout_autoload = get_number (copy_gujin_param.timeout_autoload, 99);
	  /* If we disable write enable staying in the setup menu,
		try to re-enable quickboot behaviour */
	  if (copy_gujin_param.timeout_autoload == 1) {
	      last_loaded = 0; total_loadable = 1; init_page = 0; total_page = 1;
	      }
	  copy_gujin_param.autoload.last_loaded = last_loaded;
	  copy_gujin_param.autoload.total_loadable = total_loadable;
	  copy_gujin_param.autoload.init_page = init_page;
	  copy_gujin_param.autoload.total_page = total_page;
	  UI.function.setbgcolor (bgcolor);
	  }
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 0,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1, /* 40 cols mode */
		.update_bootparam	= 1,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 0
		};
      case 11: {
	  unsigned bgcolor = UI.bgcolor, tmp;

	  UI.function.setbgcolor (newbgcolor);
	  tmp = kbd_detect ();
	  if (tmp == 0) {
	      kbd_setup (copy_gujin_param.kbdmap);
	      _BIOS_wait (2 * 1000 * 1000);
	      }
	  UI.function.setbgcolor (bgcolor);
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 0,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1, /* 40 cols mode */
		.update_bootparam	= 1,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= tmp
		};
	  }
      case 12: {
	  unsigned bgcolor = UI.bgcolor, tmpret;
	  unsigned char tmpcol = UI.parameter.col;
	  const char *msg = MSG_COMMAND_LINE;

	  printf (msg);
	  if (tmpcol == UI.parameter.col) {
	      /* TOCHECK: is that posssible? probably,
		printf does not maintain cursor in all cases. */
	      /* Update UI.parameter.{row,col} in case
		getcursor not working (bad VT100 emulation): */
	      UI.parameter.col += NbUtf8Char (msg);
	      }
	  UI.cursor_bgcolor = UI.stdcolor[brown];
	  UI.function.setbgcolor (newbgcolor);
	  char buffer[sizeof (copy_gujin_param.extra_cmdline)];
	  strcpy (buffer, copy_gujin_param.extra_cmdline);
	  tmpret = get_line (buffer, sizeof (buffer), '\0', ' ', '\377');
	  if (tmpret == 0)
	      strcpy (copy_gujin_param.extra_cmdline, buffer);
	  UI.function.setbgcolor (bgcolor);
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 0,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1, /* 40 cols mode */
		.update_bootparam = tmpret != 0,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= tmpret != 0
		};
	  }
      case 13: {
	  unsigned bgcolor = UI.bgcolor, tmpret;
	  unsigned char tmpcol = UI.parameter.col;
	  const char *msg = MSG_DIRECTORY_NAME;

	  printf (msg);
	  if (tmpcol == UI.parameter.col) {
	      /* TOCHECK: is that posssible? probably,
		printf does not maintain cursor in all cases. */
	      /* Update UI.parameter.{row,col} in case
		getcursor not working (bad VT100 emulation): */
	      UI.parameter.col += NbUtf8Char (msg);
	      }
	  UI.cursor_bgcolor = UI.stdcolor[brown];
	  UI.function.setbgcolor (newbgcolor);
	  char buffer[sizeof (copy_gujin_param.scanpath)];
	  strcpy (buffer, copy_gujin_param.scanpath);
	  tmpret = get_line (buffer, sizeof (buffer), '\0', '0', 'z');
	  if (tmpret == 0)
	      strcpy (copy_gujin_param.scanpath, buffer);
	  UI.function.setbgcolor (bgcolor);
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 0,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1, /* 40 cols mode */
		.update_bootparam = tmpret != 0,
		.refresh	= 0,
		.reparse	= 0,	/* reparse only when going back to kernels menu */
		.presskey	= tmpret != 0
		};
	  }
      case 14: {
	  unsigned bgcolor = UI.bgcolor;
	  printf (EP_SET_EXTENDED_PARTITION_SIZE);
	  UI.function.setbgcolor (newbgcolor);
	  adjust_extended_partition_length ();
	  UI.function.setbgcolor (bgcolor);
	  }
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 0,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam	= 0,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 1
		};
#if SETUP & UPDATE_BOOTPARAM
      case 15:
	  BOOT1_uninstall_mbr();
	  return (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 1,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam = 0,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 1
		};
#endif
      }
  /* VESA_init() failed */
  return (struct todraw_str) {
	.clearbackground	= 0,
	.firstlines	= 0,
	.topmenu	= 0,
	.bottommenu	= 1, /* redraw the pressed button */
	.advertisement	= 1,
	.copyright	= 0,
	.update_bootparam	= 0,
	.refresh	= 0,
	.reparse	= 0,
	.presskey	= 0
	};
  }

static inline struct todraw_str
activate_setup_field (unsigned absnb, unsigned short advline)
  {
  struct todraw_str returned;

  if (absnb < TOTAL_ATTRIBUTE) {
      invert_setup_attribute_field (absnb);
      unsigned char redraw_copyright = 0;

      if (absnb == enum_use_gujin_embedded_font && BOOT1_COM_port() < 0) {
	  unsigned currentmode = UI.function.getmode();
	  UI.function.setmode (currentmode);
	  redraw_copyright = 1;
	  }
	/* number of modes, +/- through BPP... : */
      returned = (struct todraw_str) {
	.clearbackground	= 0,
	.firstlines	= 1,
	.topmenu	= 1,
	.bottommenu	= 1,
	.advertisement	= 1,
	.copyright	= redraw_copyright,
	.update_bootparam	= 1,
	.refresh	= 0,
	.reparse	= 0,
	.presskey	= 0
	};
      }
    else
      returned = activate_setup_action_field (absnb - TOTAL_ATTRIBUTE + 1,
						advline);

  return returned;
  }

static inline void
redraw_bottom_menu (unsigned nbline, unsigned curline)
  {
  unsigned char fctkeyno;
  unsigned endline = curline + nbline - 1;

  if (Menu.totalpage > 1) {
      MOUSE_declare_attribute (mouse_attr);
      MOUSE_attrset_mainfield (mouse_attr, 1);
      MOUSE_attrset_activefield (mouse_attr, (Menu.curpage != 0));

      UI.function.setcursor (curline, UI.parameter.nbcol / 4 - 6);
      PUT_FIELD (PAGEUP_KEYNAME, mouse_attr, PAGEUP_KEYCODE(),
			curline, UI.parameter.nbcol / 4 - 6);
      UI.function.setcursor (curline, UI.parameter.nbcol / 2 + 4);
      PUT_FIELD (HOME_KEYNAME, mouse_attr, HOME_KEYCODE(),
			curline, UI.parameter.nbcol / 2 + 4);
      curline++;
      }

  curline -= Menu.interline / 2;

  for (fctkeyno = 1;
	  fctkeyno <= Menu.nbperpage
       && fctkeyno <= Menu.nbtotal - Menu.curpage * Menu.nbperpage;
       fctkeyno++) {
      const char *format;
      unsigned short column = sizeof (" F%u: ") - 1;
      curline += Menu.interline;
      /* We could also do 'Menu.interline' times '\r\n', it would
	 not need a draw_box()... inteface would be quicker,
	 but it may be better if we want to display something
	 on the right part of the screen ... */
      UI.function.setcursor (curline, 0);
      if (BOOT1_COM_port() >= 0 && fctkeyno <= 4) {
	  /* Here, on a real VT, if you cannot have F5, use the 'Do' key */
	  format = " PF%u: ";
	  }
	else if (fctkeyno <= 9 && Menu.nbperpage >= 10)
	  format = " F%u:  ";
	else {
	  format = " F%u: ";
	  if (fctkeyno <= 9)
	      column --; /* start the proposed line one char earlier */
	  }
      /* This man, he was perfect. Just a small detail,
	he didn't like my hat. */
      printf (format, fctkeyno);
      if (Menu.type == kernel_bottom_menu)
	  redraw_kernel_field (Menu.curpage * Menu.nbperpage + fctkeyno, curline, column);
	else
	  redraw_setup_field (Menu.curpage * Menu.nbperpage + fctkeyno, curline, column);
      }

  if (Menu.totalpage > 1) {
      MOUSE_declare_attribute (mouse_attr);
      MOUSE_attrset_mainfield (mouse_attr, 1);
      MOUSE_attrset_activefield (mouse_attr,
		 (Menu.totalpage > 1 && Menu.curpage < Menu.totalpage - 1));

      UI.function.setcursor (endline, UI.parameter.nbcol / 4 - 6);
      PUT_FIELD (PAGEDOWN_KEYNAME, mouse_attr, PAGEDOWN_KEYCODE(),
			endline, UI.parameter.nbcol / 4 - 6);
      UI.function.setcursor (endline, UI.parameter.nbcol / 2 + 4);
      PUT_FIELD (END_KEYNAME, mouse_attr, END_KEYCODE(),
			endline, UI.parameter.nbcol / 2 + 4);
      }
  }

static inline void init_Menu (unsigned nblines)
  {
  unsigned max_Fkey;

  if (UTIL.keyboard == serialkbd)
      max_Fkey = 12; /* see is_valid_fct_keycode() just after */
    else if (UTIL.keyboard == stdkbd)
      max_Fkey = 10;
    else
      max_Fkey = 12;
  /* minimum one empty line in between two menu lines: */
  if (Menu.nbtotal < nblines / 2 && Menu.nbtotal <= max_Fkey) {
      Menu.nbperpage = Menu.nbtotal;
      Menu.totalpage = 1;
      Menu.interline = nblines / (Menu.nbperpage + 1);
      }
    else {
      /* 2 lines for the PgUp/Home menu, one for PgDn/end: */
      Menu.nbperpage = (nblines - 3) / 2;
      if (Menu.nbperpage > max_Fkey)
	  Menu.nbperpage = max_Fkey;
      Menu.totalpage = DIVROUNDUP (Menu.nbtotal, Menu.nbperpage);
      Menu.interline = (nblines - 3) / Menu.nbperpage;
      }
  }

static inline unsigned short
redraw_screen (struct todraw_str toredraw, unsigned refresh, unsigned timeout, struct autoload_str *autoload)
  {
  unsigned line;
  unsigned short advline = UI.parameter.nbrow - 1;

  if (toredraw.clearbackground || UI.parameter.attr.has_scrolled) { /* Sometimes, scrolled without flag set: SERIAL */
      UI.parameter.attr.has_scrolled = 0;
      UI.function.clearscreen();
      toredraw =  (struct todraw_str) {
	.clearbackground	= 0,
	.firstlines		= 1,
	.topmenu		= 1,
	.bottommenu	= 1,
	.advertisement	= 1,
	.copyright	= 1,
	};
      }

  if (UI.parameter.nbcol >= ADVERTISEMENT_BANNER_MINWIDTH)
      advline -= ADVERTISEMENT_BANNER_NBLINES + 1;

  if (toredraw.firstlines)
      redraw_first_lines (refresh);

#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
  {
  struct mousefieldattr_str reset_fields = {
	.fullscreenfield = 1,
	.upperfield = toredraw.topmenu,
	.mainfield  = toredraw.bottommenu
	};
  MOUSE_reset_field (reset_fields);
  }
#endif

  if (toredraw.topmenu)
      line = redraw_top_menu (3) + 1;
    else
      line = redraw_top_menu (0) + 4; /* still get line */

  if (toredraw.bottommenu /* FIXME: && advline > line */) {
      unsigned nblines = advline - 1 - line;

      draw_bg_box (line, nblines, BOTTOM_MENU_BACKGROUND);
      if (Menu.nbtotal == 0)
	  centered_print (NOT_EVEN_ONE_SELECTABLE_ITEM, line + 3, UI.parameter.nbcol);
	else {
	  if (Menu.nbperpage == 0) {
	      init_Menu (nblines);
	      /* autoload->init_page/autoload->total_page also saves the page displayed
		when switching to the setup screen, to return to it when leaving setup screen */
	      if (Menu.type == kernel_bottom_menu
			&& (autoload->total_page == Menu.totalpage || autoload->total_page == 0)) {
		  autoload->total_page = Menu.totalpage;
		  if (autoload->last_loaded && Menu.nbperpage && timeout)
		      Menu.curpage = autoload->last_loaded / Menu.nbperpage;
		    else if (autoload->init_page < autoload->total_page)
		      Menu.curpage = autoload->init_page;
		    else
		      Menu.curpage = autoload->init_page = 0;
		  }
		else
		  Menu.curpage = 0;
	      }
	  redraw_bottom_menu (nblines, line);
	  }
      }

  if (toredraw.copyright) {
      unsigned savebgcolor, savefgcolor, len = strlen((char *)0xFFAC);

      if (BOOT1_COM_port() < 0) {
	  show_ladders (UI.stdcolor[black]);
	  UI.function.setcursor (UI.parameter.nbrow - 1, UI.parameter.nbcol - len - 1);
	  savebgcolor = UI.bgcolor;
	  UI.function.setbgcolor (UI.stdcolor[black]);
	  }
	else {
	  unsigned short cpt;

	  savebgcolor = UI.bgcolor;
	  UI.function.setbgcolor (UI.stdcolor[black]);
	  UI.function.setcursor (UI.parameter.nbrow - 1, 0);
	  cpt = UI.parameter.nbcol - len;
	  while (--cpt)
	      print (" ");
	  }
      savefgcolor = UI.fgcolor;
      UI.function.setfgcolor (UI.stdcolor[lightgray]);
      print ((char *)0xFFAC);
      UI.function.setbgcolor (savebgcolor);
      UI.function.setfgcolor (savefgcolor);
      }

  if (   toredraw.advertisement
      && UI.parameter.nbcol >= ADVERTISEMENT_BANNER_MINWIDTH) {
      const char *autoload_msg = IN_U_SECOND_AUTOLOAD_S;
      unsigned buflen_needed = strlen(autoload_msg) + 10;
      if (BOOTWAY.desc[autoload->last_loaded].filename)
	  buflen_needed += strlen (BOOTWAY.desc[autoload->last_loaded].filename);
      char buffer[buflen_needed];
      const char *advert = buffer;

      if (timeout == 0)
	  advert = ADVERTISEMENT_BANNER;
	else
	  sprintf (buffer, IN_U_SECOND_AUTOLOAD_S, timeout, BOOTWAY.desc[autoload->last_loaded].filename);

      draw_bg_box (advline, ADVERTISEMENT_BANNER_NBLINES, ADVERT_BACKGROUND);
      centered_print (advert, advline, UI.parameter.nbcol);
      }
  return advline;
  }

/*
 * This function treat the key pressed / mouse clics, it is completely
 * independant of what is displayed on the screen:
 */

/* simple helpers */
static inline unsigned short
is_valid_fct_keycode (unsigned short key, unsigned short nbfctkey)
  {
  unsigned short cpt = nbfctkey < nbof(all_fct_keycode) ? nbfctkey : nbof(all_fct_keycode);
  while (cpt-- > 0)
      if (key == all_fct_keycode[cpt])
	  return cpt + 1;	/* the function key number */

  cpt = nbfctkey < nbof(all_sfct_keycode) ? nbfctkey : nbof(all_sfct_keycode);
  while (cpt-- > 0)
      if (key == all_sfct_keycode[cpt])
	  return 0x0100 + cpt + 1;	/* the shift function key number */
  return 0;
  }

static inline unsigned
is_valid_chgmode_keycode (unsigned short key)
  {
  /* We include the comma if serial because on a real VT,
     the comma replaces the plus on the numeric keypad */
  if (BOOT1_COM_port() >= 0)
#if 0 /* I have a bug here, with constant strings? */
      return strchr ("+-*/.,\t", key & 0xFF) != 0;
#elif 1
      {
      const char *ptr = "+-*/.,\t";
      while (*ptr != '\0')
	  if (*ptr++ == (key & 0xFF))
	      return 1;
      return 0;
      }
#else
      return    (key & 0xFF) == '+' || (key & 0xFF) == '-'
	     || (key & 0xFF) == '*' || (key & 0xFF) == '/'
	     || (key & 0xFF) == '.' || (key & 0xFF) == ','
	     || (key & 0xFF) == '\t';
#endif
#if 1
  {
#if __GNUC__ != 4 || __GNUC_MINOR__ != 6 || __GNUC_PATCHLEVEL__ != 0
  static const unsigned short array[] = (const unsigned short []) {
#else
  static unsigned short array[] = (unsigned short []) {
#endif

	PLUS_KEYCODE(),
	MINUS_KEYCODE(),
	MULTIPLY_KEYCODE(),
	DIVIDE_KEYCODE(),
	DOT_KEYCODE(),
	TAB_KEYCODE(),
	KP_DIVIDE_KEYCODE(),
	KP_PLUS_KEYCODE(),
	KP_MINUS_KEYCODE(),
	KP_MULTIPLY_KEYCODE(),
	KP_DOT_KEYCODE(),
	0
	};
  const unsigned short *ptr = array;
  while (*ptr != 0)
      if (*ptr++ == key)
	  return 1;
  }
#else
  if (   key == PLUS_KEYCODE()
      || key == MINUS_KEYCODE()
      || key == MULTIPLY_KEYCODE()
      || key == DIVIDE_KEYCODE()
      || key == DOT_KEYCODE()
      || key == TAB_KEYCODE())
      return 1;
  /* only extkbd: */
  if (   key == KP_DIVIDE_KEYCODE()
      || key == KP_PLUS_KEYCODE()
      || key == KP_MINUS_KEYCODE()
      || key == KP_MULTIPLY_KEYCODE()
      || key == KP_DOT_KEYCODE())
      return 1;
#endif
  return 0;
  }

static inline unsigned
is_valid_chgmenu_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0)
      return (key & 0xFF) == ' ';
    else
      return key == SPACE_KEYCODE();
  }

static inline unsigned
is_valid_ONOFF_keycode (unsigned short key)
  {
  if (key == ON_OFF_KEYCODE())
      return 1;
  if (key == POWER_KEYCODE())
      return 1;
  return 0;
  }

static inline unsigned
is_valid_failedboot (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0)
      return (key & 0xFF) == '\034'; // ^\ because '\033'; is part of function keys
    else
      return key == ESC_KEYCODE();
  }

/*
 * Control R or Control L to redraw? This question is like arguing how
 * many fingers you have in your hand with a english native speaker...
 * Lets say the two redraw the complete screen, but ^R also reprobe all
 * the disk subsystem (if your CDROM was not reported present at boot)
 */
static inline unsigned
is_Redraw_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0)
      return (key & 0xFF) == '\014';
    else
      return key == CTRL_L_KEYCODE();
  }

static inline unsigned
is_reprobe_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0)
      return (key & 0xFF) == '\022';
    else
      return (key == CTRL_R_KEYCODE());
  }

static inline unsigned
is_PgUp_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0 && (key & 0xFF) == '\020')
      return 1; /* ^P */
  if (key == PAGEUP_KEYCODE())
      return 1;
  /* only extkbd: */
  if (key == (PAGEUP_KEYCODE() | 0xE0))
      return 1;
  return 0;
  }

static inline unsigned
is_PgDn_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0 && (key & 0xFF) == '\016')
      return 1; /* ^N */
  if (key == PAGEDOWN_KEYCODE())
      return 1;
  /* only extkbd: */
  if (key == (PAGEDOWN_KEYCODE() | 0xE0))
      return 1;
  return 0;
  }

static inline unsigned
is_Home_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0 && (key & 0xFF) == '\001')
      return 1; /* ^A */
  if (key == HOME_KEYCODE())
      return 1;
  /* only extkbd: */
  if (key == (HOME_KEYCODE() | 0xE0))
      return 1;
  return 0;
  }

static inline unsigned
is_End_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0 && (key & 0xFF) == '\005')
      return 1; /* ^E */
  if (key == END_KEYCODE())
      return 1;
  /* only extkbd: */
  if (key == (END_KEYCODE() | 0xE0))
      return 1;
  return 0;
  }

#if DISK_SUPPORT & BOOTSECT_PROBE
/**
 ** Floppy emulation:
 **/
static inline unsigned update_bios_floppy_emul (unsigned char emulated_bios)
  {
  extern union bios_0x0410_u {
      unsigned short all;
      struct _BIOS_equipment_flags_str bits;
      } __attribute__ ((packed)) CS_bios_0x0410;
  unsigned current_nb_floppy = 0;

  CS_bios_0x0410.bits = _BIOS_equipment_flags();
  if (CS_bios_0x0410.bits.floppy_present)
      current_nb_floppy = 1 + CS_bios_0x0410.bits.nb_floppy_less_1;
  /* if dp->OSdisknumber < 0x80, we want to partially hide the floppy and not count it in the usual list,
	like using instboot --disk=BIOS:0x10,auto
	Same if dp->OSdisknumber == 0x01 and no floppy enabled, hide it.
	if it is a hard disk, dp->OSdisknumber >= 0x80 */
  if (emulated_bios <= current_nb_floppy) {
      union bios_0x0410_u new_bios_0x0410 = CS_bios_0x0410;
      new_bios_0x0410.bits.floppy_present = 1;
      new_bios_0x0410.bits.nb_floppy_less_1 = current_nb_floppy; /* so add one */
      pokew (0x00400010, new_bios_0x0410.all);
      ZDBG (("report floppy %s, 1+%u floppies\r\n",
		new_bios_0x0410.bits.floppy_present? "present" : "abscent",
		new_bios_0x0410.bits.nb_floppy_less_1));
      current_nb_floppy += 1;
      }
  return current_nb_floppy;
  }

static inline void finalise_emul (unsigned byte_to_copy)
  {
  extern unsigned short CS_oldINT13entryOFF, CS_oldINT13entrySEG;
  extern unsigned short CS_newINT13entryOFF, CS_newINT13entrySEG;
  extern char newINT13entry[];
  extern unsigned short CS_oldMEMsize, CS_newMEMsize;
  extern unsigned CS_oldintaddr;

  farptr faradr = peekl(4 * 0x00000013);	/* INT 0x13 */

  CS_oldintaddr = faradr;
  CS_oldINT13entryOFF = faradr & 0xFFFF;
  CS_oldINT13entrySEG = faradr >> 16;

  CS_oldMEMsize = peekw (0x00400013);	/* BIOS nb of Kbytes */
  /* If MSDOS floppy is emulated, the number of free Kbytes seems ignored
     and the EBDA is anyway loaded at 0x9FC00, so do not put our code
     in between 0x9FC00 and 0xA0000... */
  if (CS_oldMEMsize > 0x9FC00 / 1024)
      CS_oldMEMsize = 0x9FC00 / 1024;	/* Maybe we should not, free memory if no video card */

  CS_newMEMsize = CS_oldMEMsize - (byte_to_copy + 1023) / 1024;
  pokew (0x00400013, CS_newMEMsize);	/* BIOS nb of Kbytes */

  /* at run time, data accessed with %cs prefix; then %cs shall be equal to current %ds */
  /* GCC-4.2: cast from pointer to integer of different size
  CS_newINT13entryOFF = (unsigned short)newINT13entry;
   */
  CS_newINT13entryOFF = (unsigned)newINT13entry & 0x0000FFFF;
  CS_newINT13entrySEG = (1024 * (unsigned)CS_newMEMsize - (unsigned)newINT13entry) / 16;
  faradr = ((unsigned)CS_newINT13entrySEG << 16) | CS_newINT13entryOFF;

  ZDBG (("CS_oldMEMsize %u, CS_newMEMsize %u, old IRQ 0x%X:0x%X, new IRQ 0x%X:0x%X i.e. 0x%X\r\n",
	CS_oldMEMsize, CS_newMEMsize, CS_oldINT13entrySEG, CS_oldINT13entryOFF,
	CS_newINT13entrySEG, CS_newINT13entryOFF, faradr));

  smemcpy (faradr, newINT13entry, byte_to_copy & 0xFFFF);
//  pokel (4 * 0x00000013, faradr);	/* INT 0x13 */
  UTIL.swap_BIOSint0x13_irqhandler = faradr;

  ZDBG (("Copied %u bytes from %%ss:0x%X to farptr 0x%X\r\n", byte_to_copy, newINT13entry, faradr));
 }

/* This function setup all assembler for disk emulation */
/* We manage up to 4 Kbytes TSR if !FLOPPYDBG, code size 2400 bytes,
    so max (4096 - 2400) / 12 = 141 fragments */
#define FRAGMENTED_DISK_NBFRAG 128

static unsigned char
setup_emulation (const struct diskparam_str *dp, const FAT_bootsect1_t *bbfpart1, const FAT_bootsect2_t *bbfpart2, unsigned boot_media_type,
		const char *filename, unsigned emulation_nb_sectors, unsigned cdrom_lba,
		unsigned long long partition_start, unsigned long long partition_length, unsigned char partition_type, unsigned enable_disk_write,
		struct sector_chain_str remap_array[FRAGMENTED_DISK_NBFRAG])
  {
  extern unsigned char CS_emulated_bios, CS_floppy_disabled;
  extern unsigned char CS_disk_base_table_bps, CS_disk_base_table_spt;
  extern unsigned char CS_max_head, CS_nb_sector, CS_drive_type, CS_boot_media_type, CS_request_al;
  extern unsigned short CS_max_cylinders;
  extern unsigned short CS_bit48_total_sector, CS_bit32_total_sector, CS_bit16_total_sector;
  extern unsigned short CS_bit48_total_sector512, CS_bit32_total_sector512, CS_bit16_total_sector512;
  extern unsigned char CS_nb_disk;
  extern unsigned char CS_target_disk, CS_target_max_head, CS_target_nb_sector, CS_target_bps;
  extern unsigned short CS_target_ide_dcr, CS_target_ide_base;
  extern unsigned char CS_target_lba_head;
  extern unsigned short CS_cdrom_controller_index;
  extern unsigned char CS_cdrom_controller_number;
  extern unsigned char CS_gap_sectors_shift, CS_intergap_shift;
  extern unsigned short CS_cdrom_lba_msb, CS_cdrom_lba_lsb;
  extern char CS_int13filename[64], CS_int13biosdisk;
  extern unsigned short CS_ide_base;
  extern unsigned char CS_ide_mask, CS_host_bios, CS_partition_type;
  extern unsigned CS_ebios_bustype;
  extern unsigned long long CS_ebios_Interface, CS_partition_start, CS_partition_length;
  extern union interface_path_u CS_ebios_bus;
  extern union device_path_u CS_ebios_device;
  extern struct sector_chain_str CS_remap_adr[FRAGMENTED_DISK_NBFRAG];
  extern char max_byte_to_copy[];

//#define FLOPPYDBG	/* only when video mode BIOS can write chars... */
#ifndef FLOPPYDBG
  extern char newINT13entry[];
  memset (fourKbuffer, 0, sizeof (fourKbuffer));
  lmemcpy(fourKbuffer, xdata_adr(newINT13entry), (unsigned)max_byte_to_copy);	/* .xdatacode */
#endif

#if USER_SUPPORT != 0
//  printf ("max_byte_to_copy %u, max FRAGMENTED_DISK_NBFRAG %u\r\n", max_byte_to_copy, (4096 - (int)max_byte_to_copy) / 12); _BIOS_getkey();
#endif

  if (enable_disk_write)
      CS_floppy_disabled = 0;
    else
      CS_floppy_disabled = 2;

  unsigned cpt_frag = 0, byte_to_copy = (unsigned)max_byte_to_copy;
  do
      CS_remap_adr[cpt_frag] = remap_array[cpt_frag];
      while (remap_array[cpt_frag++].nb != 0);
#ifdef FLOPPYDBG
  ZDBG (("CS_remap_adr uses %u elements, reducing copied size from %u to %u\r\n", cpt_frag, byte_to_copy,
	byte_to_copy - (FRAGMENTED_DISK_NBFRAG - cpt_frag - 2)*sizeof(CS_remap_adr[0])));
  byte_to_copy -= (FRAGMENTED_DISK_NBFRAG - cpt_frag - 2)*sizeof(CS_remap_adr[0]);
#else
  byte_to_copy += cpt_frag *sizeof(CS_remap_adr[0]);
#endif

  /* Note: drive_type is reported as 0x10 for my USB thumb drive booting in USB-FDD emulation,
     it may be used to know if the total size of the disk is C*H*S or (C+1)*H*S */

  if ((bbfpart2->Signaturebyte0x29 != 0x29 && bbfpart2->Signaturebyte0x29 != 0x28)
	/* || bbfpart1->NbHead == 0 || bbfpart1->NbSectorpertrack == 0 */
	|| bbfpart1->NbSectorpertrack > 63) {
#define CD_NOEMUL_DISK 0xE0
      static const struct predef_t {
	  unsigned char emulated_bios;
	  unsigned char disk_base_table_bps; /* 2: 512 bytes/sectors, 4: 2048 bytes/sectors */
	  unsigned char max_head;
	  unsigned short max_cylinders;
	  unsigned char nb_sector;
	  unsigned char drive_type; /* 1: 360K, 2: 1.2M, 3: 720K, 4: 1.44M, 6: 2.88M, 0x10: ATAPI */
	  unsigned char boot_media_type;	/* 0: no emul, 1: 1.2M, 2: 1.44M, 3:2.88M, 4: HD (+msb) */
	  } __attribute__ ((packed)) predefined[] = {
	  [boot_media_noemul] = {CD_NOEMUL_DISK, 4, 254, 1023, 63, 0x10, 0},
	  [boot_media_120] =    {0,    2,  1,   79,   15, 2,    1},
	  [boot_media_144] =    {0,    2,  1,   79,   18, 4,    2},
	  [boot_media_288] =    {0,    2,  1,   79,   36, 6,    3},
	  [boot_media_hd] =     {0x80, 2,  254, 1023, 63, 0x10,    4}
	  };
      const struct predef_t *predef = &predefined[boot_media_type <= 4 ? boot_media_type : 4];

      CS_emulated_bios = predef->emulated_bios;
      CS_disk_base_table_bps = predef->disk_base_table_bps;
      CS_max_head = predef->max_head;
      CS_max_cylinders = predef->max_cylinders;
      CS_nb_sector = predef->nb_sector;
      CS_drive_type = predef->drive_type; /* Get Drive Parameter */
      CS_boot_media_type = predef->boot_media_type | 0x40; /* CDROM BIOS get status + have ATAPIdrv */
      }
    else {
      unsigned char first_free_hd_bios = 0x80 + DI.nb_bios_hd;

      if (bbfpart1->Bytepersector == 2048)
	  CS_disk_base_table_bps = 4;
	else if (bbfpart1->Bytepersector == 4096)
	  CS_disk_base_table_bps = 5; // not handled in assembly for ATAPI
	else if (bbfpart1->Bytepersector == 1024)
	  CS_disk_base_table_bps = 3; // not handled in assembly for ATAPI
	else /* maybe bbfpart1->Bytepersector % 512 != 0 */
	  CS_disk_base_table_bps = 2;
      if (bbfpart2->PhysicaldriveNb < first_free_hd_bios)
	  CS_emulated_bios = bbfpart2->PhysicaldriveNb;
	else if (boot_media_type == boot_media_noemul)
	  CS_emulated_bios = CD_NOEMUL_DISK;
	else /* if set to 0xFF */
	  CS_emulated_bios = first_free_hd_bios;
      CS_max_head = bbfpart1->NbHead - 1;
      CS_nb_sector = bbfpart1->NbSectorpertrack;
      /* Overwrite size of file with the size the disk pretend to be, non present data will be treated as filesystem hole: */
      if ((bbfpart1->NbTotalsector ?: bbfpart1->NbTotalsector2) > emulation_nb_sectors)
	  emulation_nb_sectors = bbfpart1->NbTotalsector ?: bbfpart1->NbTotalsector2;
      if (bbfpart1->NbHead != 0 && CS_nb_sector > 1)
	  CS_max_cylinders = emulation_nb_sectors / (CS_max_head+1) / CS_nb_sector - 1;
	else
	  CS_max_cylinders = 0xFFFF;
      /* Floppy Disk Base Table: */
      if (emulation_nb_sectors == 40*2*9)
	  CS_drive_type = 1, CS_boot_media_type = 0x9 | 0x40; /* have ATAPIdrv */
	else if (emulation_nb_sectors == 80*2*15)
	  CS_drive_type = 2, CS_boot_media_type = 1 | 0x40; /* have ATAPIdrv */
	else if (emulation_nb_sectors == 80*2*9)
	  CS_drive_type = 3, CS_boot_media_type = 0xA | 0x40; /* have ATAPIdrv */
	else if (emulation_nb_sectors == 80*2*18)
	  CS_drive_type = 4, CS_boot_media_type = 2 | 0x40; /* have ATAPIdrv */
	else if (emulation_nb_sectors == 80*2*36)
	  CS_drive_type = 6, CS_boot_media_type = 3 | 0x40; /* have ATAPIdrv */
	else
	  CS_drive_type = 0x10, CS_boot_media_type = ((CS_disk_base_table_bps == 4)? 0 : 4) | 0x40; /* have ATAPIdrv */
      emulation_nb_sectors <<= (CS_disk_base_table_bps - 2); /* this variable is in 512 bytes sectors */
      }

  /* We want the name of the iso if full BDI or El-torito, but we want Kanotix-2005-04.iso:/boot/memtest.bin */
  char *dst = CS_int13filename;
  while (dst < &CS_int13filename[sizeof(CS_int13filename)])
      if (*filename == '\0' || (*filename == ':' && CS_disk_base_table_bps == 4))
	  *dst++  = '\0';
	else
	  *dst++  = *filename++;
  CS_int13filename[sizeof(CS_int13filename)-1] = '\0';

  /* following is in 512 bytes sectors on floppy/hd emulation CDROM: */
  CS_bit48_total_sector512 = 0;
  CS_bit32_total_sector512 = emulation_nb_sectors >> 16;
  CS_bit16_total_sector512 = emulation_nb_sectors;

  /* Those are the reported values: */
  CS_bit48_total_sector = 0;
  CS_bit32_total_sector = (emulation_nb_sectors >> (CS_disk_base_table_bps - 2)) >> 16;
  CS_bit16_total_sector = emulation_nb_sectors >> (CS_disk_base_table_bps - 2);

  CS_request_al = 0;
  CS_disk_base_table_spt = CS_nb_sector;
  CS_int13biosdisk = CS_emulated_bios;

  if (CS_emulated_bios < 0x80)
      CS_nb_disk = update_bios_floppy_emul (CS_emulated_bios);
    else {
//      CS_nb_disk = _BIOSDISK_get_nb_harddisk();
      CS_nb_disk = DI.nb_bios_hd;
      if (CS_emulated_bios < CD_NOEMUL_DISK)
	  CS_nb_disk ++;
      }

#ifdef NB_ISO
  CS_gap_sectors_shift = BOOTWAY.iso.gap_sectors_shift;
  CS_intergap_shift = BOOTWAY.iso.intergap_shift;
#else
  CS_gap_sectors_shift = 0;
  CS_intergap_shift = 0;
#endif
  ZDBG (("%s: CS_gap_sectors_shift %u, CS_intergap_shift %u\n", __FUNCTION__, CS_gap_sectors_shift, CS_intergap_shift));

  CS_partition_start = partition_start;
  CS_partition_length = partition_length;
  CS_partition_type = partition_type;

  for (CS_target_bps = 0; (128 << CS_target_bps) < dp->bytepersector; CS_target_bps++)
      continue;
  if (dp->access == ebios_lba || dp->access == bios_chs) {
      if (dp->access == ebios_lba) {
	  CS_target_nb_sector = 0;
	  CS_host_bios = dp->disknb;
	  CS_ide_mask = dp->lba_slave_mask;
	  CS_ide_base = dp->ideIOadr;
	  CS_ebios_bustype = *(unsigned *)&dp->ebios_bustype;
	  CS_ebios_Interface = *(unsigned long long*)&dp->ebios_Interface;
	  CS_ebios_bus = dp->ebios_bus;
	  CS_ebios_device = dp->ebios_device;
	  }
	else
	  CS_target_nb_sector = dp->nbsectorpertrack;
      CS_target_disk = dp->disknb;
      CS_target_max_head = dp->nbhead - 1;
      /*  CS_request_al = 0; INT13/0x43: write with verify stuff, EBIOS version dependant */
      /* if target_nb_sector == 0 and CS_target_ide_dcr == 0: EBIOS */
      CS_target_ide_dcr = CS_target_ide_base = CS_target_lba_head = 0;	/* unused if (E)BIOS */
      ZDBG (("%s FD/HD (BIOS nb 0x%X, max heads %u, nb_sectors %u)",
		dp->access == bios_chs? "BIOS" : "EBIOS", CS_target_disk, CS_target_max_head, CS_target_nb_sector));
      }
    else if (dp->access == hardide_chs || dp->access == hardide_lba || dp->access == hardide_lba48) {
      /* FIXME: direct IDE access simulated disks */
      }
    else {
      CS_target_disk = CS_target_max_head = CS_target_nb_sector = 0; /* unused if ATAPI */
      /* if target_nb_sector == 0 and CS_target_ide_dcr != 0: ATAPI */
//      CS_target_ide_dcr = 0x3F6; CS_target_ide_base = 0x1F0; CS_target_lba_head = 0xA0 /*| 0x10*/;
      CS_target_ide_dcr = dp->ideIOctrladr;
      CS_ide_base = CS_target_ide_base = dp->ideIOadr;
      CS_ide_mask = CS_target_lba_head = 0xA0 | dp->lba_slave_mask;

      CS_cdrom_controller_index = dp->disknb; /* 0: master, 1: slave */
#if (DISK_SUPPORT & (EBIOS_SUPPORT | IDE_SUPPORT))
      CS_cdrom_controller_number = DI.nb_IDE_found;
      for (;;) {
	  CS_cdrom_controller_number--;
	  if (DI.IDE_found[CS_cdrom_controller_number].ideIOadr == CS_target_ide_base)
	      break;
	  if (CS_cdrom_controller_number == 0)
	      break;
	  }
#endif
      ZDBG (("ATAPI FD/HD (0x%X, 0x%X, 0x%X, %s on controller %u)",
		CS_target_ide_base, CS_target_ide_dcr, CS_target_lba_head,
		CS_cdrom_controller_index? "slave" : "master", CS_cdrom_controller_number));
      }
  /* FIXME: I am not sure the LBA in ElTorito shall be absolute or relative to
	the beginning of the session, to be check with major providers of CDROM/DVD writer software */
  CS_cdrom_lba_msb = cdrom_lba >> 16;
  CS_cdrom_lba_lsb = cdrom_lba;
  ZDBG ((" CDROM BIOS reports offset %u.\r\n", cdrom_lba));

  finalise_emul (byte_to_copy);

  ZDBG (("emulated_bios 0x%X, drive_type %u, disk_base_table_bps %u, disk_base_table_spt %u, "
	"boot_media_type 0x%X, max_head %u, max_cylinders %u, nb_sector %u, total %u<<16 + %u\r\n",
	CS_emulated_bios, CS_drive_type, CS_disk_base_table_bps, CS_disk_base_table_spt,
	CS_boot_media_type, CS_max_head, CS_max_cylinders, CS_nb_sector, CS_bit32_total_sector, CS_bit16_total_sector));

#if 0
printf ("emulated_bios 0x%X, floppy_disabled 0x%X, drive_type %u, disk_base_table_bps %u, CS_target_bps %u, disk_base_table_spt %u, "
	"boot_media_type 0x%X, max_head %u, max_cylinders %u, nb_sector %u, total %u<<32 + %u<<16 + %u\r\n",
	CS_emulated_bios, CS_floppy_disabled, CS_drive_type, CS_disk_base_table_bps, CS_target_bps, CS_disk_base_table_spt,
	CS_boot_media_type, CS_max_head, CS_max_cylinders, CS_nb_sector,
	CS_bit48_total_sector, CS_bit32_total_sector, CS_bit16_total_sector);
printf ("CS_request_al %u, CS_nb_disk %u, target_disk 0x%X, target_max_head %u, target_nb_sector %u, "
	"target_ide_dcr 0x%X, target_ide_base 0x%X, CS_target_lba_head 0x%X\r\n",
	CS_request_al, CS_nb_disk, CS_target_disk, CS_target_max_head, CS_target_nb_sector,
	CS_target_ide_dcr, CS_target_ide_base, CS_target_lba_head);
printf ("cdrom_controller_index 0x%X, cdrom_controller_number %u, gap_sectors_shift %u, intergap_shift %u, "
	"cdrom_lba_msb 0x%X, cdrom_lba_lsb 0x%X, int13biosdisk 0x%X filename: %s\r\n",
	CS_cdrom_controller_index, CS_cdrom_controller_number, CS_gap_sectors_shift, CS_intergap_shift,
	CS_cdrom_lba_msb, CS_cdrom_lba_lsb, CS_int13biosdisk, CS_int13filename);
printf ("CS_remap_adr: %u sectors at %llu, %u sectors at %llu, %u sectors at %llu...\r\n",
	CS_remap_adr[0].nb, CS_remap_adr[0].lba, CS_remap_adr[1].nb, CS_remap_adr[1].lba, CS_remap_adr[2].nb, CS_remap_adr[2].lba);
printf ("Press a key to continue...");
_BIOS_getkey();
#endif

  return CS_emulated_bios;
  }

/* I think we need to switch language now... but it doesn't matter:
	"Whatever the language, the world does not compile." */

	asm (
#ifdef FLOPPYDBG
"	.section	.rodata					\n"
#else
"	.section	.xdatacode				\n"
#endif
"max_byte_to_copy = newINT13end - newINT13entry			\n"
"								\n"
"emulated_bios = 0x28						\n"
"newINT13entryOFF = 0x4329					\n"
"newINT13entrySEG = 0x2032					\n"
"newMEMsize = 0x3030						\n"
"oldMEMsize = 0x3420						\n"
"oldINT13entryOFF = 0x4574					\n"
"oldINT13entrySEG = 0x6965					\n"
"max_cylinders = 0x6E6E						\n"
"target_nb_sector = 0x65	# if 0, use EBIOS and LBA	\n"
"target_max_head = 0x20						\n"
"nb_disk = 0x4C							\n"
"bit16_total_sector512 = 0x4F52	# of emulated disk		\n"
"bit32_total_sector512 = 0x5241					\n"
"bit48_total_sector512 = 0x494E	# upper bits			\n"
"bit16_total_sector = 0x4F52	# of emulated disk		\n"
"bit32_total_sector = 0x5241					\n"
"bit48_total_sector = 0x494E	# upper bits			\n"
"nb_sector = 0							\n"
"max_head = 0							\n"
"target_disk = 0x00	# container disk			\n"
"drive_type = 0x00						\n"
"request_al = 0x00	# INT13/0x43 write with verify if ...	\n"
"target_ide_dcr = 0x0000	# if ATAPI			\n"
"target_dcr_val = 0x0A						\n"
"target_ide_base = 0x0000					\n"
"target_lba_head = 0x00						\n"
"bios_0x0410 = 0x00001111					\n"
"								\n"
"boot_media_type = 0x02	# 0: no emul, 1: 1.2M, 2: 1.44M,	\n"
"			# 3:2.88M, 4: HD (+msb)			\n"
"cdrom_controller_number = 0x00		# IDE number?		\n"
"cdrom_controller_index = 0x0000	# 1 for slave		\n"
"cdrom_lba_lsb = 0x0000	# bit16_lba_offset in 2048 sectors	\n"
"cdrom_lba_msb = 0x0000	# bit32_lba_offset in 2048 sectors	\n"
"cdrom_read_cache_seg = 0x0000					\n"
"cdrom_load_segment = 0x07C0					\n"
"cdrom_nb_512b_sector = 4					\n"
"								\n"
"	.align 16						\n"
"newINT13entry:							\n"

"	jmp 1f							\n"
"	.ascii	\"$INT13SF\"					\n"
"	.ascii	\"Gujin   \"					\n"
"CS_oldintaddr:							\n"
"	.long	0						\n"
"	.long	0	# flags					\n"
"CS_int13biosdisk:	.byte	0xFF				\n"
"CS_int13filename:						\n"
"	.ascii	\"unknown\"					\n"
"	.skip	64 - 7						\n"
"CS_host_bios:		.skip 1					\n"
"CS_ide_mask:		.skip 1					\n"
"CS_ide_base:		.skip 2					\n"
"CS_ebios_bustype:	.skip 4					\n"
"CS_ebios_Interface:	.skip 8					\n"
"CS_ebios_bus:		.skip 8					\n"
"CS_ebios_device:	.skip 16				\n"
"CS_partition_start:	.skip 8					\n"
"CS_partition_length:	.skip 8					\n"
"CS_partition_type:	.skip 1					\n"
"	1:							\n"

#if 0
"	pushfw				\n"
"	pushaw				\n"
"	mov	$0x2000 + 'D',%cx	\n"
"	callw	all_print_CXDX		\n"
"	popaw				\n"
"	pushaw				\n"
"	mov	$0x2000 + 'A',%cx	\n"
"	mov	%ax,%dx			\n"
"	callw	all_print_CXDX		\n"
"	popaw				\n"
"	popfw				\n"
"	jmp	2f			\n"
"all_print_CXDX:			\n"
"	movw	$0x0007,%bx		\n"
"	movb	$0x0E,%ah		\n"
"	movb	%cl,%al			\n"
"	int	$0x10			\n"
"	movb	%ch,%al			\n"
"	int	$0x10			\n"
"	movw	$12,%cx			\n"
"	1:				\n"
"	mov	%dx,%ax			\n"
"	shrw	%cl,%ax			\n"
"	andw	$0x000F,%ax # ALcarry	\n"
"	sahf		# clear AF	\n"
"	aaa				\n"
"	aad	$0x11			\n"
"	add	$0x0E30,%ax		\n"
"	int	$0x10			\n"
"	subw	$4,%cx			\n"
"	jns	1b			\n"
"	mov	$0x20,%al		\n"
"	int	$0x10			\n"
"	retw				\n"
"	2:				\n"
#endif

"	pushfw				# iretw			\n"
"	cmpb	$emulated_bios,%dl				\n"
"CS_emulated_bios = . - 1					\n"
"	je	emulate_disk					\n"
"	cmpb	$0x7F,%dl					\n"
"	je	floppy_general_call				\n"
"	mov	%dl,%cs:CS_restore_dl				\n"
"	xorb	%cs:CS_emulated_bios,%dl			\n"
"	rcl	%dl						\n"
"	mov	%cs:CS_restore_dl,%dl				\n"
"	jc	CALLoldINT13					\n"
"	cmpb	$0x08,%ah					\n"
"	jne	not_INT1308					\n"
"	mov	%cs:CS_nb_disk,%dl				\n"
"	xchgb	%dl,%cs:CS_restore_dl				\n"
"not_INT1308:							\n"
"	cmpb	%dl,%cs:CS_emulated_bios			\n"
"	sbb	$0,%dl						\n"
"	cmpb	$0x52,%cs:22+CS_restore_dl			\n"
"	jne	CALLoldINT13					\n"
"	testb	$0x80,%dl					\n"
"	jz	1f						\n"
"	cmpb	$0x15,%ah					\n"
"	je	CALLoldINT13					\n"
"	1:							\n"
"	pushw	%cs			# iretw			\n"
"	pushw	$INT13return_setdl	# iretw			\n"
"	pushfw							\n"
"CALLoldINT13:							\n"
"	popfw			# so sti			\n"
"pushIPjumpOLDINT13:						\n"
"	ljmpw	$oldINT13entrySEG,$oldINT13entryOFF		\n"
"CS_oldINT13entryOFF = . - 4					\n"
"CS_oldINT13entrySEG = . - 2					\n"
"INT13return_setdl:						\n"
"	mov	%cs:CS_restore_dl,%dl				\n"
"	lretw	$2						\n"
"								\n"
"floppy_general_call:						\n"
"	cmpw	$0x4B00,%ax	# CDROM BIOS disinstall emul	\n"
"	jne	CALLoldINT13					\n"
"	orb	$0x80,%cs:CS_floppy_disabled			\n"
"	pushw	%ds		# start disinstall tests	\n"
"	pushw	$0						\n"
"	popw	%ds		# interrupt & BIOS area		\n"
"	cmpw	$newINT13entryOFF,%ds:0x13*4	# IDT at 0	\n"
"CS_newINT13entryOFF = . - 2					\n"
"	jne	cannot_disinstall				\n"
"	cmpw	$newINT13entrySEG,%ds:0x13*4+2	# IRQ 0x13	\n"
"CS_newINT13entrySEG = . - 2					\n"
"	jne	cannot_disinstall				\n"
"	cmpw	$newMEMsize,%ds:0x413	# Memsize Kb (INT 0x12)	\n"
"CS_newMEMsize = . - 2						\n"
"	jne	cannot_disinstall				\n"
"	cli			# start real disinstall		\n"
"	movw	$oldMEMsize,%ds:0x413				\n"
"CS_oldMEMsize = . - 2						\n"
"	pushw	%cs:CS_bios_0x0410				\n"
"	popw	%ds:0x410					\n"
"	pushw	%cs:CS_oldINT13entryOFF				\n"
"	popw	%ds:0x13*4					\n"
"	pushw	%cs:CS_oldINT13entrySEG				\n"
"	popw	%ds:0x13*4+2					\n"
"cannot_disinstall:						\n"
"	popw	%ds		# end real disinstall		\n"
"	jmp	CALLoldINT13					\n"
"								\n"
"								\n"
"CS_floppy_disabled:	.byte 0x02	# write disabled	\n"
"CS_last_status:	.byte 0x06	# diskette changed	\n"
"CS_first_read_mask:	.byte 0x0F	# 2048->512 reads	\n"
"CS_last_read_mask:	.byte 0x0F	# 2048->512 reads	\n"
"CS_bios_0x0410:	.long bios_0x0410			\n"
"CS_restore_dl:		.byte 0					\n"
"CS_target_bps:		.byte 2					\n"
"								\n"
"CS_disk_base_table:	# HelpPC 2.10, Copyright 1991 David Jurgens	\n"
"	.byte	0xAF	# specify byte 1; step-rate time, head unload time, 0xDF Compaq	\n"
"	.byte	0x02	# specify byte 2; head load time, DMA mode	\n"
"	.byte	0x25	# timer ticks to wait before disk motor shutoff	\n"
"CS_disk_base_table_bps:						\n"
"	.byte	0x02	# bytes per sector/sector size: 128 << value	\n"
"CS_disk_base_table_spt:						\n"
"	.byte	0x12	# sectors per track (last sector number)	\n"
"	.byte	0x1B	# inter-block gap length/gap between sectors	\n"
"	.byte	0xFF	# data length, if sector length not specified	\n"
"	.byte	0x6C	# gap length between sectors for format, 0x65 Compaq1.44, 0x54 Compaq1.20	\n"
"	.byte	0xF6	# fill byte for formatted sectors		\n"
"	.byte	0x0F	# head settle time in milliseconds		\n"
"	.byte	0x08	# motor startup time in eighths of a second	\n"
"	.byte	0x20, 0x47, 0x50, 0x4c, 0x20, 0x45, 0x2e, 0x4c		\n"
"	.byte	0x4f, 0x52, 0x52, 0x41, 0x49, 0x4e, 0x2e, 0x00		\n"
"									\n"
"emulate_disk:							\n"
#ifdef FLOPPYDBG /* PRINT regiters at entry */
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'a',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %bx,%dx ; mov $0x2000 + 'b',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %cx,%dx ; mov $0x2000 + 'c',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %dx,%dx ; mov $0x2000 + 'd',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %es,%dx ; mov $0x2000 + 'e',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
/* FIXME: Can we safely call int 0x13/ah=0 when there is no floppy drive? */
"	cmpb	$0,%ah		# DISK				\n"
"	je	CALLoldINT13	# reset disks			\n"
"	popfw							\n"
"test_disable_me:						\n"
"	cmpw	$0x4B00,%ax	# bootable CDROM		\n"
"	jne	test_reenable_me				\n"
"	orb	$0x80,%cs:CS_floppy_disabled			\n"
"	pushw	%ds		# start disinstall tests	\n"
"	pushw	%ax						\n"
"	pushw	$0						\n"
"	popw	%ds		# interrupt & BIOS area		\n"
"	mov	%cs:CS_newINT13entryOFF,%ax			\n"
"	cmpw	%ax,%ds:0x13*4	# IDT at 0			\n"
"	jne	no_disinstall					\n"
"	mov	%cs:CS_newINT13entrySEG,%ax			\n"
"	cmpw	%ax,%ds:0x13*4+2	# IRQ 0x13		\n"
"	jne	no_disinstall					\n"
"	mov	%cs:CS_newMEMsize,%ax				\n"
"	cmpw	%ax,%ds:0x413	# Memsize Kb (INT 0x12)		\n"
"	jne	no_disinstall					\n"
"	pushw	%cs:CS_bios_0x0410				\n"
"	popw	%ds:0x410					\n"
"	pushw	%cs:CS_oldINT13entryOFF				\n"
"	popw	%ds:0x13*4					\n"
"	pushw	%cs:CS_oldINT13entrySEG				\n"
"	popw	%ds:0x13*4+2					\n"
"	pushw	%cs:CS_oldMEMsize				\n"
"	popw	%ds:0x413					\n"
"no_disinstall:							\n"
"	popw	%ax						\n"
"	popw	%ds						\n"
"direct_return_carry_clear_ah_0:				\n"
"	movb	$0x00,%ah	# no error			\n"
"direct_return_carry_clear:					\n"
"	clc							\n"
"direct_return:							\n"
"	pushw	%ax				\n"
"	jc	dont_clear_last_status		\n"
"	mov	$0,%ah				\n"
"dont_clear_last_status:				\n"
"	movb	%ah,%cs:CS_last_status				\n"
"	pushfw							\n"
"	pushw	%ds						\n"
"	pushw	$0						\n"
"	popw	%ds						\n"
"	cmpb	$0x80,%cs:CS_emulated_bios			\n"
"	jl	status_last_floppy				\n"
#ifdef FLOPPYDBG /* PRINT status last HD */
	" pushaw ; movzbw %ah,%dx ; mov $0x2000 + 'H',%cx; callw print_cxdx ; popaw \n"
#endif
"	mov	%ah,%ds:0x474	# status last hard disk operation	\n"
"	jmp	bios_status_updated				\n"
"status_last_floppy:						\n"
#ifdef FLOPPYDBG /* PRINT status last floppy */
	" pushaw ; movzbw %ah,%dx ; mov $0x2000 + 'F',%cx; callw print_cxdx ; popaw \n"
#endif
"	mov	%ah,%ds:0x441	# status last floppy operation	\n"
"bios_status_updated:						\n"
"	popw	%ds						\n"
"	popfw							\n"
"	popw	%ax				\n"
#ifdef FLOPPYDBG /* PRINT regs at exit, exit code 'R' if carry, 'r' if !carry */
	" pushfw			\n"
"	push	%ax		\n"
"	mov	$0x02,%ah	\n"
"	int	$0x16		\n"
"	andb	$0x20,%al	\n"	/* only when num-lock active */
"	pop	%ax		\n"
"	jz	2f		\n"
	" popfw				\n"
	" pushfw			\n"
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'a',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %bx,%dx ; mov $0x2000 + 'b',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %cx,%dx ; mov $0x2000 + 'c',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %dx,%dx ; mov $0x2000 + 'd',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %es,%dx ; mov $0x2000 + 'e',%cx; callw print_cxdx ; popaw \n"
	" pushaw			\n"
	" mov	%ax,%dx			\n"
	" setnc	%cl			\n"
	" shl	$5,%cl			\n"
	" add	$'R',%cl		\n"
	" mov	$' ',%ch		\n"
	" callw print_cxdx		\n"
	" xor	%dx,%dx			\n"
	" mov	$30,%cx			\n"
	" mov	$0x8600,%ax		\n"
	" int	$0x15   # _BIOS_wait	\n"
	" popaw				\n"
	" callw	print_crlf		\n"
"	2:			\n"
	" popfw				\n"
#endif /* FLOPPYDBG */
"	lretw	$2						\n"
"dosayaword:							\n"
"	mov	$815508,%eax					\n"
"	and	%eax,%eax					\n"
"	not	%eax						\n"
"	mov	$802218,%eax					\n"
"test_reenable_me:						\n"
"	cmpw	$0x4BFF,%ax	# not documented		\n"
"	jne	test_enabled					\n"
"	andb	$0x7F,%cs:CS_floppy_disabled			\n"
"	jmp	direct_return_carry_clear_ah_0			\n"
"test_enabled:							\n"
"	testb	$0x80,%cs:CS_floppy_disabled			\n"
"	jz	test_get_status					\n"
"direct_return_carry_set_ah_80:					\n"
"	mov	$0x80,%ah	# timeout, drive not ready	\n"
"direct_return_carry_set:					\n"
"	stc							\n"
"	jmp	direct_return					\n"
"test_get_status:						\n"
"	cmpw	$0x4B01,%ax	# _BIOSCDROM_getstatus		\n"
"	jne	test_extended_present				\n"
//"	mov	(%si),%al					\n"
//"	cmpb	$0x13,%al					\n"
//"	jb	buffer_too_small				\n"
/* We have to fill the specification packet at %ds:%si: */
"	movw	$256*boot_media_type+0x13,%ax			\n"
"CS_boot_media_type = . - 1					\n"
"	movw	%ax,0(%si)					\n"
"	movb	%cs:CS_emulated_bios,%al			\n"
"	movb	$cdrom_controller_number,%ah			\n"
"CS_cdrom_controller_number = . - 1				\n"
"	movw	%ax,2(%si)					\n"
"	movw	$cdrom_lba_lsb,4(%si)				\n"
"CS_cdrom_lba_lsb = . - 2					\n"
"	movw	$cdrom_lba_msb,6(%si)				\n"
"CS_cdrom_lba_msb = . - 2					\n"
"	movw	$cdrom_controller_index,8(%si)			\n"
"CS_cdrom_controller_index = . - 2				\n"
"	movw	$cdrom_read_cache_seg,0xA(%si)			\n"
"CS_cdrom_read_cache_seg = . - 2				\n"
"	movw	$cdrom_load_segment,0xC(%si)			\n"
"CS_cdrom_load_segment = . - 2					\n"
"	movw	$cdrom_nb_512b_sector,0xE(%si)			\n"
"CS_cdrom_nb_512b_sector = . - 2				\n"
"	movw	%cs:CS_max_cylinders,%ax			\n"
"	# not the logical:	rol $8,%ax			\n"
"	shlb	$6,%ah						\n"
"	orb	%cs:CS_nb_sector,%ah				\n"
"	movw	%ax,0x10(%si)					\n"
"	movb	%cs:CS_max_head,%al				\n"
"	mov	%al,0x12(%si)					\n"
"	movb	$0x0,%al					\n"
//"buffer_too_small:						\n"
"	jmp	direct_return_carry_clear_ah_0			\n"
"test_extended_present:						\n"
"	cmpb	$0x41,%ah					\n"
"	jne	test_extended_info				\n"
"	cmpw	$0x55AA,%bx					\n"
"	jne	direct_return_carry_set_ah_1			\n"
"	rol	$8,%bx						\n"
"	mov	$0x2100,%ax					\n"
"	mov	$1,%cx						\n"
//"	mov	$0,%dh	\n" // RBIL say "DH extension version"
"	jmp	direct_return_carry_clear			\n"
"test_extended_info:						\n"
"	cmpb	$0x48,%ah					\n"
"	jne	test_extended_write				\n"
"	cmpw	$0x1E,(%si)					\n"
"	jb	direct_return_carry_set_ah_1			\n"
"	movw	$0x1E,(%si)					\n"
"	movw	$0x7,2(%si)					\n"
"	movw	%cs:CS_max_cylinders,%ax			\n"
"	movw	%ax,4(%si)					\n"
"	movw	$0,6(%si)					\n"
"	incl	4(%si)						\n"
"	movb	%cs:CS_max_head,%al				\n"
"	movb	$0,%ah						\n"
"	movw	%ax,8(%si)					\n"
"	movw	$0,10(%si)					\n"
"	incl	8(%si)						\n"
"	movb	%cs:CS_nb_sector,%al				\n"
"	movb	$0,%ah						\n"
"	movw	%ax,12(%si)					\n"
"	movw	$0,14(%si)					\n"
"	movw    $bit16_total_sector,%ax				\n"
"CS_bit16_total_sector = . - 2					\n"
"	movw	%ax,16(%si)					\n"
"	movw    $bit32_total_sector,%ax				\n"
"CS_bit32_total_sector = . - 2					\n"
"	movw	%ax,18(%si)					\n"
"	movw    $bit48_total_sector,%ax				\n"
"CS_bit48_total_sector = . - 2					\n"
"	movw	%ax,20(%si)					\n"
"	movw	$0,22(%si)					\n"
"	push	%cx						\n"
"	movb	%cs:CS_disk_base_table_bps,%cl	# 128 << 2 = 512	\n"
"	mov	$128,%ax					\n"
"	shl	%cl,%ax						\n"
"	mov	%ax,24(%si)					\n"
"	pop	%cx						\n"
"	movw	$0xFFFF,26(%si)					\n"
"	movw	$0xFFFF,28(%si)					\n"
"	jmp	direct_return_carry_clear_ah_0			\n"
"test_extended_write:						\n"
"	cmpb	$0x43,%ah					\n"
"	jne	test_extended_read				\n"
"	testb	$0x02,%cs:CS_floppy_disabled # write & write protected?	\n"
"	jnz	direct_return_carry_set_ah_3			\n"
"	jmp	L_extended_read_write				\n"
"test_extended_read:						\n"
"	cmpb	$0x42,%ah					\n"
// FIXME: also extended verify sectors?
"	jne	test_get_status_last_operation			\n"
"L_extended_read_write:						\n"
"	cmpw	$0x0010,(%si)					\n"
"	jne	direct_return_carry_set_ah_1			\n"
"	pushw	%ds						\n"
"	pushw	%es						\n"
"	pushaw	   # push ax, cx, dx, bx, initsp, bp, si, di	\n"
"	pushw	14(%si)		# msb LBA			\n"
"	pushw	12(%si)						\n"
"	pushw	10(%si)						\n"
"	pushw	8(%si)		# lsb LBA			\n"
"	mov	%sp,%bp		# (%bp) = LBA			\n"
"	mov	4(%si),%bx					\n"
"	mov	6(%si),%es					\n"
"	mov	2(%si),%si					\n"
#ifdef FLOPPYDBG /* PRINT number of sectors requested and at which address */
	" pushaw ; mov %si,%dx ; mov $0x2000 + '#',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %es,%dx ; mov $0x2000 + '@',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %bx,%dx ; mov $0x2000 + ' ',%cx; callw print_cxdx ; popaw \n"
#endif
"	andw	$0x00FF,%si	# now %sil = nb sector requested\n"
//"	andb	$0xBF,%ah	# 0x42->0x02, 0x43->0x03	\n"
"	jmp	lba_requested_known				\n"
"test_get_status_last_operation:	# DISK			\n"
"	cmpb	$1,%ah						\n"
"	jne	test_format_track				\n"
"	mov	%cs:CS_last_status,%ah				\n"
"	jmp	direct_return_carry_clear			\n"
"test_format_track:			# FLOPPY		\n"
"	cmpb	$5,%ah						\n"
"	jne	test_detect_disk_change				\n"
"direct_return_carry_set_ah_3:					\n"
"	mov	$3,%ah		# diskette write protect error	\n"
"	jmp	direct_return_carry_set				\n"
"test_detect_disk_change:		# FLOPPY		\n"
"	cmpb	$0x16,%ah					\n"
"	jne	test_get_disk_type				\n"
"	mov	$6,%ah	# change line active or not supported	\n"
"	jmp	direct_return_carry_set				\n"
// If CF clear, AH = 00 no disk change; AH = 01 disk changed
//"	mov	$0,%ah	# no disk change			\n"
//"	jmp	direct_return_carry_clear			\n"
"test_get_disk_type:			# DISK			\n"
"	cmpb	$0x15,%ah					\n"
"	jne	test_get_drive_parameter			\n"
"	testb	$0x80,%cs:CS_emulated_bios			\n"
"	jz	floppy_params					\n"
"	mov	%cs:CS_bit16_total_sector,%dx			\n"
"	mov	%cs:CS_bit32_total_sector,%cx			\n"
//"	sub	$1,%dx						\n"
//"	sbb	$0,%cx						\n"
"	mov	$0x03,%ah	# hard disk			\n"
"	jmp	direct_return_carry_clear			\n"
"floppy_params:							\n"
"	mov	$0x01,%ah	# floppy w/o change line	\n"
//"	mov	$0x02,%ah	# floppy with change line	\n"
"	jmp	direct_return_carry_clear			\n"
"test_get_drive_parameter:		# DISK			\n"
"	cmpb	$0x08,%ah					\n"
"	jne	test_read_write_verify				\n"
"	mov	$0,%al	# at least on some BIOS			\n"
"	mov	$drive_type,%bl					\n"
"CS_drive_type = . - 1						\n"
"	cmpb	$6,%bl						\n"
"	jbe	1f	# do not report if not floppy		\n"
"	pushw	%cs						\n"
"	popw	%es						\n"
"	mov	$CS_disk_base_table,%di				\n"
"	1:							\n"
"	movw	$max_cylinders,%cx				\n"
"CS_max_cylinders = . - 2					\n"
"	rolw	$8,%cx						\n"
"	shlb	$6,%cl						\n"
"	orb	$nb_sector,%cl					\n"
"CS_nb_sector = . - 1						\n"
"	movb	$max_head,%dh					\n"
"CS_max_head = . - 1						\n"
"	movb	$nb_disk,%dl					\n"
"CS_nb_disk = . - 1						\n"
"	jmp	direct_return_carry_clear_ah_0			\n"
"test_read_write_verify:					\n"
"	cmpb	$0x04,%ah					\n"
// FIXME: verify sectors %ah=0x04/0x44?
//"	jbe	read_write_verify		# unsigned	\n"
"	jb	read_write_verify		# unsigned	\n"
"test_set_disk_type_for_format:		# FLOPPY 0x17		\n"
"test_set_media_type_for_format:	# DISK	0x18		\n"
"direct_return_carry_set_ah_1:					\n"
"	mov	$1,%ah	# bad command or parameter		\n"
"	jmp	direct_return_carry_set				\n"
"read_write_verify:						\n"
"	test	%al,%al		# nb sector cannot be null	\n"
"	jz	direct_return_carry_set_ah_1			\n"
"	cmpb	%cs:CS_max_head,%dh				\n"
"	ja	direct_return_carry_set_ah_1	# unsigned	\n"
"	test	$0x3F,%cl	# sector cannot be null		\n"
"	jz	direct_return_carry_set_ah_1			\n"
"	pushw	%cx						\n"
"	andw	$0x003F,%cx					\n"
"	cmpb	%cs:CS_nb_sector,%cl				\n"
"	popw	%cx						\n"
"	ja	direct_return_carry_set_ah_1	# unsigned	\n"
"	pushw	%cx						\n"
"	pushw	%ax						\n"
"	# refuse here access cylinder over 80,			\n"
"	# i.e. 80 is accepted (will check max lba later):	\n"
"	movw	%cs:CS_max_cylinders,%ax			\n"
"	inc	%ax						\n"
"	rolw	$8,%cx						\n"
"	shrb	$6,%ch						\n"
"	cmpw	%ax,%cx						\n"
"	popw	%ax						\n"
"	popw	%cx						\n"
"	ja	direct_return_carry_set_ah_1	# unsigned	\n"
"	cmpb	$0x03,%ah	# write & write protected?	\n"
"	jne	parameter_basic_test_passed			\n"
"	testb	$0x02,%cs:CS_floppy_disabled			\n"
"	jnz	direct_return_carry_set_ah_3			\n"
"parameter_basic_test_passed:					\n"
"	pushw	%ds						\n"
"	pushw	%es						\n"
"	pushaw	   # push ax, cx, dx, bx, initsp, bp, si, di	\n"
"	movw	%ax,%si						\n"
"	andw	$0x00FF,%si	# now %sil = nb sector requested\n"
// Limit: max_head and nb_sector fit in a byte, we handle here
// max_head=255 and so nb_head = 256 but we need nb_sector small
// enought for (nb_head * nb_sector + sector-1) to fit in 16 bits.
"	mov	%cx,%ax		# sector in %cl			\n"
"	andw	$0x003F,%ax	# sector 1..63 in %ax		\n"
"	dec	%ax		# %ax = S - 1			\n"
"	# begin to build the INT 0x13/0x42, 0x43 or 0x44	\n"
"	# request block in stack:				\n"
"	pushw	$0		# msb LBA			\n"
"	pushw	$0						\n"
"	pushw	$0						\n"
"	pushw	%ax		# lsb LBA			\n"
"	mov	%sp,%bp		# (%bp) = S - 1			\n"
"	mov	%cs:CS_nb_sector,%al	# %al = S0 , %dh = head	\n"
"	xchgb	%al,%dh			# %dh = S0 , %al = head	\n"
"	mul	%dh		# ax = al * dh			\n"
"	addw	%ax,(%bp)	# (%bp) = H * S0 + S - 1	\n"
"#	adcw	$0,2(%bp)	# not needed - no overflow	\n"
"	shrw	$8,%dx		# %dx = S0			\n"
"	mov	%cs:CS_max_head,%al				\n"
"	mul	%dl		# ax = al * dl			\n"
"	add	%dx,%ax		# ax = (max_head+1) * nb_sector	\n"
"	mov	%ch,%dl						\n"
"	mov	%cl,%dh						\n"
"	shrb	$6,%dh		# cylinder 0..1023 in %dx	\n"
"	mul	%dx		# dx:ax = ax * dx		\n"
"	addw	%ax,(%bp)					\n"
"	adcw	%dx,2(%bp)	# (%bp) = C * H0 * S0 + H * S0	+ S - 1\n"
"	adcw	$0,4(%bp)	# just in case of overflow	\n"
"lba_requested_known:						\n"
// CDROM ELTorito noemul 2 Kb sectors simultated on HD partition:
"	mov	%cs:CS_target_bps,%cl				\n"
"	sub	%cs:CS_disk_base_table_bps,%cl			\n"
"	jae	no_compensate					\n"
"	neg	%cl						\n"
"	mov	$0,%ch						\n"
"loop_rotate:							\n"
"	shll	(%bp)	# set/clears carry			\n"
"	rcll	4(%bp)						\n"
"	shlw	%si	# FIXME: overflow NOT handled!!!	\n"
"	loop	loop_rotate					\n"
"no_compensate:							\n"
#ifdef FLOPPYDBG /* PRINT LBA requested */
	" pushaw ; mov 4(%bp),%dx ; mov $0x2000 + 'l',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov 2(%bp),%dx ; mov $0x2000 + ' ',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov (%bp),%dx  ; mov $0x2000 + ' ',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
// Check Max (last cylinder may not be complete) by calculating Max
// Request Size in sector from this LBA: 48 bits arithmetic in %cx:%dx:%ax
"	movw	$bit16_total_sector512,%ax			\n"
"CS_bit16_total_sector512 = . - 2				\n"
"	movw	$bit32_total_sector512,%dx			\n"
"CS_bit32_total_sector512 = . - 2				\n"
"	movw	$bit48_total_sector512,%cx			\n"
"CS_bit48_total_sector512 = . - 2				\n"
"	subw	(%bp),%ax					\n"
"	sbbw	2(%bp),%dx					\n"
"	sbbw	4(%bp),%cx					\n"
"	jnc	no_overflow_lba					\n"
"not_even_one_sector_for_cmd:					\n"
"	addw	$8,%sp						\n"
"	popaw							\n"
"	popw	%es						\n"
"	popw	%ds						\n"
"	jmp	direct_return_carry_set_ah_1			\n"
"no_overflow_lba:						\n"
#ifdef FLOPPYDBG /* PRINT maximum nb_sector request */
	" pushaw ; mov %cx,%dx ; mov $0x2000 + 'm',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %dx,%dx ; mov $0x2000 + ' ',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %ax,%dx ; mov $0x2000 + ' ',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	xorw	%di,%di	# mark NO short rw error		\n"
"	orw	%dx,%cx						\n"
"	jnz	full_request					\n"
"	orw	%ax,%cx						\n"
"	jz	not_even_one_sector_for_cmd			\n"
"	cmpw	%ax,%si		# ax contains maximum request	\n"
"	jbe	full_request					\n"
//"	jb	full_request					\n"
"	mov	%ax,%di		# mark short rw error		\n"
"	mov	%ax,%si		# request less sectors, < 256	\n"
"full_request:							\n"
#ifdef FLOPPYDBG /* PRINT nb sector really requested */
	" pushaw ; mov %si,%dx ; mov $0x2000 + 'N',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	mov	%bx,%cs:CS_FRAGdstaddr				\n"
"	mov	%si,%cs:CS_FRAGnbsector				\n"
#ifdef FLOPPYDBG /* PRINT total nb sector requested up to now */
	" pushaw ; mov %cs:CS_FRAGtotalSectorRead,%dx ; mov $0x2000 + '+',%cx; callw print_cxdx ; popaw	\n"
	" add	%si,%cs:CS_FRAGtotalSectorRead			\n"
	" jmp	1f						\n"
	"CS_FRAGtotalSectorRead:	.short 0		\n"
	" 1:							\n"
#endif /* FLOPPYDBG */
"	pushl	(%bp)						\n"
"	popl	%cs:CS_FRAGlbalsb				\n"
"	pushl	4(%bp)						\n"
"	popl	%cs:CS_FRAGlbamsb				\n"
"	movw	$0,%cs:CS_FRAGnbSectorRead			\n"
"	jmp	FRAG_first_treat	# no code cache flush	\n"
"CS_FRAGnbSectorRead:	.short 0				\n"
"CS_FRAGcurnb:	.short 0					\n"
"FRAG_next_treat:						\n"
"FRAGdstaddr = 5						\n"
"	mov	$FRAGdstaddr,%bx				\n"
"CS_FRAGdstaddr = . - 2						\n"
"FRAGnbsector = 6						\n"
"	mov	$FRAGnbsector,%si				\n"
"CS_FRAGnbsector = . - 2					\n"
"FRAGlbalsb = 7							\n"
"	movl	$FRAGlbalsb,(%bp)				\n"
"CS_FRAGlbalsb = . - 4						\n"
"FRAGlbamsb = 8							\n"
"	movl	$FRAGlbamsb,4(%bp)				\n"
"CS_FRAGlbamsb = . - 4						\n"
"	add	$16,%sp						\n"
"	pushl	4(%bp)						\n"
"	pushl	(%bp)						\n"
#ifdef FLOPPYDBG /* PRINT base addr and lba still to process */
	" pushaw ; mov %bx,%dx ; mov $0x2000 + 'B',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov 4(%bp),%dx ; mov $0x2000 + 'L',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov (%bp),%dx ; mov $0x2000 + ' ',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	cmpw	$0,%cs:CS_FRAGnbSectorRead			\n"
"	jne	first_request_not_too_big			\n"
"	shr	$1,%si						\n"
"	jnz	first_request_not_too_big			\n"
"	inc	%si	# at least request 1			\n"
"first_request_not_too_big:					\n"
"FRAG_first_treat:						\n"
"	pushal							\n"
#ifdef FLOPPYDBG /* PRINT stack addr */
	" pushaw ; mov %sp,%dx ; mov $0x2000 + 'S',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
#ifdef FLOPPYDBG
"	mov	$0,%cx						\n"
#endif
"	mov	$CS_remap_adr,%bx				\n"
"	movl	(%bp),%eax					\n"
"	movzwl	4(%bp),%edx	# %edx:%eax = cur_lba		\n"
"next_segment:							\n"
"	or	%edx,%edx					\n"
"	jnz	isbigger					\n"
"	cmpl	%cs:8(%bx),%eax	# remap[%cx].nb			\n"
"	jl	found_segment					\n"
"isbigger:							\n"
#ifdef FLOPPYDBG
"	inc	%cx						\n"
#endif
"	subl	%cs:8(%bx),%eax					\n"
"	sbb	$0,%edx						\n"
"	add	$12,%bx						\n"
"	cmpl	$0,%cs:8(%bx)					\n"
"	je	manage_hole					\n"
"	jmp	next_segment					\n"
"found_segment:							\n"
#ifdef FLOPPYDBG /* PRINT segment index request */
	" pushaw ; mov %cx,%dx ; mov $0x2000 + 'I',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	mov	%cs:8(%bx),%ecx					\n"
"	sub	%eax,%ecx	# max nb in this segment	\n"
"	movzwl	%si,%esi					\n"
"	cmpl	%esi,%ecx					\n"
"	jae	do_full_request					\n"
"	mov	%ecx,%esi					\n"
"do_full_request:						\n"
#ifdef FLOPPYDBG /* PRINT request size */
	" pushaw ; mov %si,%dx ; mov $0x2000 + 'Z',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	mov	%cs:(%bx),%ecx					\n"
"	or	%cs:4(%bx),%ecx					\n"
"	jnz	no_hole						\n"
"	cmpw	$CS_remap_adr,%bx	# test CDnoemul		\n"
"	jne	manage_hole					\n"
"	cmpl	$0,%cs:12+8(%bx)	# test CDnoemul		\n"
"	je	no_hole						\n"
"manage_hole:							\n"
#ifdef FLOPPYDBG /* PRINT hole cmd %ax */
	" pushaw ; mov 22(%bp),%dx ; mov $0x2000 + 'H',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	cmpb	$0x43,22+1(%bp)	# saved %ah is write?		\n"
"	je	hole_write					\n"
"	cmpb	$0x03,22+1(%bp)	# saved %ah is write?		\n"
"	jne	hole_read_verify				\n"
"hole_write:							\n"
"	popal							\n"
"	subw	$8,%sp						\n"
"	mov	%cs:CS_FRAGnbSectorRead,%ax			\n"
"	mov	10(%bp),%si	# restore initial %si		\n"
"	mov	26(%bp),%dx	# get initial %ds		\n"
"	mov	%dx,%es						\n"
#ifdef FLOPPYDBG /* PRINT write in hole */
	" pushaw ; mov %es:2(%si),%dx ; mov $0x2000 + 'W',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'w',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	mov	%ax,%es:2(%si)	# initial nb_block		\n"
"	mov	$0x0A,%ah	# bad sector detected		\n"
"	mov	$1,%dx		# set carry in %dx		\n"
"	jmp	no_mark_short_rw_error				\n"
"hole_read_verify:						\n"
"	addw	%si,%cs:CS_FRAGnbSectorRead			\n"
"	sub	%si,%cs:CS_FRAGnbsector	# %si contains nbsect	\n"
"	movzwl	%si,%ecx					\n"
"	add	%ecx,%cs:CS_FRAGlbalsb				\n"
"	adc	$0,%cs:CS_FRAGlbamsb				\n"
"	mov	%cs:CS_FRAGdstaddr,%di				\n"
#ifdef FLOPPYDBG /* PRINT hole nb sectors */
	" pushaw ; mov %si,%dx ; mov $0x2000 + 'h',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	shl	$7-2,%si  # in 128 bytes sectors but stosl	\n"
"	movb	%cs:CS_target_bps,%cl # 128 << 2 = 512		\n"
"	shl	%cl,%si						\n"
"	movzwl	%si,%ecx					\n"
"	shl	$2,%si		# stosl				\n"
"	addw	%si,%cs:CS_FRAGdstaddr				\n"
#ifdef FLOPPYDBG /* PRINT hole size, %es:%bx */
	" pushaw ; mov %si,%dx ; mov $0x2000 + '&',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %es,%dx ; mov $0x2000 + 'e',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %di,%dx ; mov $0x2000 + 'd',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	shr	$2,%si		# stosl				\n"
"	cmpb	$0x44,22+1(%bp)	# saved %ah is verify?		\n"
"	je	hole_verify					\n"
"	cmpb	$0x04,22+1(%bp)	# saved %ah is verify?		\n"
"	je	hole_verify					\n"
"	mov	%es,%dx						\n"
"	movzwl	%dx,%eax					\n"
"	shl	$4,%eax						\n"
"	movzwl	%di,%edi					\n"
"	add	%edi,%eax					\n"
"	ror	$4,%eax						\n"
"	mov	%ax,%es						\n"
"	shr	$28,%eax					\n"
"	mov	%ax,%di						\n"
#ifdef FLOPPYDBG /* PRINT hole %cx, %es:%bx */
	" pushaw ; mov %cx,%dx ; mov $0x2000 + 'c',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %es,%dx ; mov $0x2000 + 'e',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %di,%dx ; mov $0x2000 + 'd',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	xor	%eax,%eax					\n"
"	stosl	%eax,%es:(%di)					\n"
"	mov	%dx,%es						\n"
"hole_verify:							\n"
"	popal							\n"
"	subw	$8,%sp						\n"
"	cmpw	$0,%cs:CS_FRAGnbsector				\n"
"	jnz	FRAG_next_treat					\n"
"	xor	%dx,%dx		# flags maybe to modify in %dx	\n"
"	movzbw	%cs:CS_FRAGnbSectorRead,%ax			\n"
#ifdef FLOPPYDBG /* PRINT return value to set */
	" pushaw ; mov %ax,%dx ; mov $0x2000 + '*',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	jmp	CS_FRAG_end					\n"
"no_hole:							\n"
"	mov	%cs:4(%bx),%ecx					\n"
"	rcl	%ecx						\n"
"	shr	%ecx						\n"
"	jnc	simple_remap					\n"
#ifdef FLOPPYDBG /* PRINT complex remap */
	" pushaw ; mov %cx,%dx ; mov $0x2000 + '$',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	pushl	%ebp						\n"
"	push	%edi						\n"
"	push	%ecx						\n"
"	push	%edx						\n"
"	push	%eax						\n"
"	push	%edx						\n"
"	push	%eax						\n"
"	add	%esi,%eax					\n"
"	adc	$0,%edx						\n"
"intergap_shift = 0						\n"
"	mov	$intergap_shift,%cl				\n"
"CS_intergap_shift = . - 1					\n"
"	mov	$1,%ebp						\n"
"	shl	%cl,%ebp					\n"
"	dec	%ebp						\n"
"	and	%eax,%ebp	# %ebp: (cur_lba + nbsect) & ((1 << intergap_shift) - 1)	\n"
"	shrd	%cl,%edx,%eax					\n"
"	mov	%eax,%edi	# %edi: (cur_lba + nbsect) >> intergap_shift	\n"
"	pop	%eax						\n"
"	pop	%edx						\n"
"	shrd	%cl,%edx,%eax	# %edx:%eax ((cur_lba >> intergap_shift)	\n"
"	cmpl	%eax,%edi					\n"
"	je	no_saturation					\n"
#ifdef FLOPPYDBG /* PRINT saturation */
	" pushaw ; mov %bp,%dx ; mov $0x2000 + 'S',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	sub	%ebp,%esi					\n"
"no_saturation:							\n"
"gap_sectors_shift = 0						\n"
"	mov	$gap_sectors_shift,%cl				\n"
"CS_gap_sectors_shift = . - 1					\n"
"	shld	%cl,%edx,%eax	# %edx:%eax ((cur_lba >> intergap_shift) << gap_sectors_shift)	\n"
#ifdef FLOPPYDBG /* PRINT delta */
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'D',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	pop	%ecx	# old eax				\n"
"	add	%ecx,%eax					\n"
"	pop	%ecx	# old edx				\n"
"	adc	%ecx,%edx					\n"
"	pop	%ecx						\n"
"	pop	%edi						\n"
"	pop	%ebp						\n"
"simple_remap:							\n"
"	add	%cs:(%bx),%eax					\n"
"	adc	%ecx,%edx	# %edx:%eax = reallba		\n"
"	mov	%eax,(%bp)					\n"
"	mov	%edx,4(%bp)					\n"
"	mov	%si,%cs:CS_FRAGcurnb				\n"
#ifdef FLOPPYDBG /* PRINT nb sectors this time */
	" pushaw ; mov %si,%dx ; mov $0x2000 + 's',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	popal							\n"
"	mov	%cs:CS_FRAGcurnb,%si				\n"
#ifdef FLOPPYDBG /* PRINT LBA after offset */
	" pushaw ; mov 4(%bp),%dx ; mov $0x2000 + 'o',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov 2(%bp),%dx ; mov $0x2000 + ' ',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov (%bp),%dx  ; mov $0x2000 + ' ',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	mov	$0,%dh						\n"
"	mov	$target_nb_sector,%dl				\n"
"CS_target_nb_sector = . - 1					\n"
"	mov	$target_max_head,%al				\n"
"CS_target_max_head = . - 1					\n"
"	mul	%dl		# ax = al * dl			\n"
"	addw	%dx,%ax		# ax = nb heads * nb sect/track	\n"
"	jnz	bios_access	# i.e. target_nb_sector != 0	\n"
"	mov	$target_ide_dcr,%dx				\n"
"CS_target_ide_dcr = . - 2	# 0x3F6	(base + 0x206)		\n"
"	orw	%dx,%dx						\n"
"	jnz	atapi_access					\n"
"ebios_access:							\n"
"	movzbw	%cs:CS_target_disk,%dx				\n"
#ifdef FLOPPYDBG /* PRINT what we will really request to that EBIOS */
	" pushaw ; mov $0x2000 + 'E',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %es,%dx ; mov $0x2000 + 'e',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %bx,%dx ; mov $0x2000 + 'b',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %si,%dx  ; mov $0x2000 + 's',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
// finish INT 0x13/0x4* request block and do the request:
"	pushw	%es		# now sp != bp			\n"
"	pushw	%bx		# saved %bx			\n"
"	pushw	%si		# request nb sector		\n"
"	pushw	$16						\n"
"	mov	%sp,%si		# %ds:%si base addr		\n"
"	movw	22(%bp),%ax	# saved %ax			\n"
"	orb	$0x40,%ah	# 2/3/4 -> 0x42/43/44		\n"
"	movb	$request_al,%al	# write verify stuff		\n"
"CS_request_al = . - 1						\n"
"	pushw	%ds						\n"
"	pushw	%ss						\n"
"	popw	%ds						\n"
//"	int	$0x13						\n"
"	pushfw							\n"
"	pushw	%cs						\n"
"	callw	pushIPjumpOLDINT13				\n"
"	popw	%ds						\n"
"	jnc	all_sectors_rw_done				\n"
"	mov	%ss:2(%si),%dx					\n"
"	jmp	xbios_nbrw_dx					\n"
"all_sectors_rw_done:						\n"
"	mov	%cs:CS_FRAGcurnb,%dx				\n"
"xbios_nbrw_dx:							\n"
#ifdef FLOPPYDBG /* PRINT how many sectors have been read/write */
	" pushaw ; mov $0x2000 + '|',%cx; callw print_cxdx ; popaw \n"
#endif
"	pushfw							\n"
"	addw	%dx,%cs:CS_FRAGnbSectorRead			\n"
"	sub	%dx,%cs:CS_FRAGnbsector				\n"
"	pushw	%cx						\n"
"	pushl	%edx						\n"
"	movzwl	%dx,%edx					\n"
"	add	%edx,%cs:CS_FRAGlbalsb				\n"
"	adcl	$0,%cs:CS_FRAGlbamsb				\n"
"	shl	$7,%edx		# in 128 bytes sectors		\n"
"	movb	%cs:CS_target_bps,%cl # 128 << 2 = 512	\n"
"	shl	%cl,%edx					\n"
"	addw	%dx,%cs:CS_FRAGdstaddr				\n"
"	popl	%edx						\n"
"	sub	%cs:CS_disk_base_table_bps,%cl			\n"
"	jae	no_compensate2					\n"
"	neg	%cl						\n"
"	shrw	%cl,%dx	# FIXME: overflow NOT handled!!!	\n"
"no_compensate2:						\n"
"	popw	%cx						\n"
"	popfw							\n"
"	movb	%dl,%al	# nb sector successfull in %al	\n"
"finish_fixup_irq:						\n"
"	pushfw							\n"
"	popw	%dx		# flags maybe to modify in %dx	\n"
"	jc	no_mark_short_rw_error	# another error		\n"
/* handle fragmented floppy image file, max NB_FRAG fragments - next read */
#ifdef FLOPPYDBG /* PRINT CS_FRAGnbsector, sector still to process */
	" pushaw ; mov %cs:CS_FRAGnbsector,%dx ; mov $0x2000 + 'Y',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	cmpw	$0,%cs:CS_FRAGnbsector				\n"
"	jne	FRAG_next_treat					\n"
"CS_FRAG_end:							\n"
"	cmpb	$0x40,23(%bp)	# %ah extended BIOS?		\n"
"	jae	1f						\n"
"	mov	%cs:CS_FRAGnbSectorRead,%al			\n"
"	1:							\n"
#ifdef FLOPPYDBG /* PRINT is that short read */
	" pushaw ; mov %di,%dx ; mov $0x2000 + '~',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	testw	%di,%di		# test short rw error		\n"
"	jz	no_mark_short_rw_error	# no error anyways	\n"
"	cmpb	$0x40,23(%bp)	# %ah extended BIOS?		\n"
"	jl	not_status_extended				\n"
"	mov	10(%bp),%si	# restore initial %si		\n"
"	pushw	26(%bp)		# get initial %ds		\n"
"	popw	%es						\n"
#ifdef FLOPPYDBG /* PRINT nb_block request and restore */
	" pushaw ; mov %es:2(%si),%dx ; mov $0x2000 + '<',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %di,%dx ; mov $0x2000 + '>',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	mov	%di,%es:2(%si)	#  or CS_FRAGnbSectorRead ?	\n"
"not_status_extended:						\n"
// How is usually encoded a short read/write in %ah?
//"	mov	$1,%ah		# bad command or parameter	\n"
// As a read with carry set but AH==0
"	mov	$0,%ah		# no error			\n"
"	or	$0x01,%dl	# set carry			\n"
"	jmp	mark_done					\n"
"no_mark_short_rw_error:					\n"
"	cmpb	$0x40,23(%bp)	# %ah extended BIOS?		\n"
"	jl	mark_done					\n"
"	mov	10(%bp),%si	# restore initial %si		\n"
"	pushw	26(%bp)		# get initial %ds		\n"
"	popw	%es						\n"
#ifdef FLOPPYDBG /* PRINT nb_block request and restore */
	" pushaw ; mov %es:2(%si),%dx ; mov $0x2000 + '{',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; movzbw %al,%dx ; mov $0x2000 + '}',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
//"	movzbw	%al,%es:2(%si)	# CS_FRAGnbSectorRead		\n"
"	movb	%al,%es:2(%si)	# CS_FRAGnbSectorRead		\n"
"	movb	22(%bp),%al	# keep old al			\n"
"mark_done:							\n"
#ifdef FLOPPYDBG /* PRINT old ax value, new ax dx */
	" pushaw ; mov 22(%bp),%dx ; mov $0x2000 + '@',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'a',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %dx,%dx ; mov $0x2000 + 'f',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	movw	%ax,22(%bp)	# will be restored as %ax	\n"
"	addw	$16,%sp						\n"
"	pushw	%dx						\n"
"	popfw							\n"
"	popaw							\n"
"	popw	%es						\n"
"	popw	%ds						\n"
"	jmp	direct_return					\n"
"bios_access:							\n"
#ifdef FLOPPYDBG /* PRINT LBA after offset */
	" pushaw ; mov $0x2000 + 'Q',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	subw	$8,%sp						\n"
"	movw	%ax,%cx						\n"
"	movw	2(%bp),%dx					\n"
"	movw	(%bp),%ax					\n"
"	div	%cx	# dx = dx:ax % cx, ax = dx:ax / cx	\n"
"	movb	%al,%ch						\n"
"	movb	%ah,%cl						\n"
"	shlb	$6,%cl		# cylinder 0..1023 in %cx	\n"
"	movw	%dx,%ax						\n"
"	divb	%cs:CS_target_nb_sector	# al = ax / byte, ah = ax % byte	\n"
"	mov	%al,%dh		# head in %dh			\n"
"	inc	%ah						\n"
"	orb	%ah,%cl		# sector low 6 bits of %cl	\n"
"	movw	%si,%ax		# request nb sector		\n"
"	movb	$target_disk,%dl				\n"
"CS_target_disk = . - 1						\n"
"	movb	23(%bp),%ah	# saved %ah			\n"
"	andb	$0xBF,%ah	# 0x42->0x02, 0x43->0x03	\n"
#ifdef FLOPPYDBG /* PRINT register before INT 0x13 */
	" callw	print_crlf \n"
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'A',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %bx,%dx ; mov $0x2000 + 'B',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %cx,%dx ; mov $0x2000 + 'C',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %dx,%dx ; mov $0x2000 + 'D',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %es,%dx ; mov $0x2000 + 'E',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
//"	int	$0x13						\n"
"	pushfw							\n"
"	pushw	%cs						\n"
"	callw	pushIPjumpOLDINT13				\n"
"	jnc	all_sectors_rw_done				\n"
"	movzbw	%al,%dx						\n"
"	jmp	xbios_nbrw_dx					\n"
"atapi_access:							\n"
#ifdef FLOPPYDBG
	" pushaw ; mov $0x2000 + 'a',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	subw	$8,%sp						\n"
"	mov	$target_dcr_val,%al	# 0x0A			\n"
"	out	%al,%dx						\n"
"	mov	$target_ide_base,%dx				\n"
"CS_target_ide_base = . - 2	# 0x1F0				\n"
"	addw	$7,%dx						\n"
"	inb	%dx,%al		# short wait			\n"
"	xor	%cx,%cx		# 65536*1us (approx) = 65 ms	\n"
"La_waitnotbusy:						\n"
"	inb	%dx,%al		# we need to wait longer	\n"
"	inb	%dx,%al		# with newer ultra speed	\n"
"	inb	%dx,%al		# DVD-RAM readers		\n"
"	inb	%dx,%al						\n"
"	test	$0x80,%al	# busy				\n"
"	loopnz	La_waitnotbusy					\n"
"	jcxz	La_ide_error_relay	# timeout		\n"
"	dec	%dx		# .lba_head			\n"
"	mov	$target_lba_head,%al				\n"
"CS_target_lba_head = . - 1					\n"
"	out	%al,%dx						\n"
"	inc	%dx		# .command			\n"
"	inb	%dx,%al		# short wait			\n"
"	xor	%cx,%cx		# 65536*1us (approx) = 65 ms	\n"
"La_waitdiskready:						\n"
"	loop	La_waitcontinue	# timeout BUSY or !DRDY or !RDY	\n"
"La_ide_error_relay:						\n"
"	jmp	La_ide_error					\n"
"La_waitcontinue:						\n"
"	inb	%dx,%al		# we need to wait longer	\n"
"	inb	%dx,%al		# with newer ultra speed	\n"
"	inb	%dx,%al		# DVD-RAM readers		\n"
"	inb	%dx,%al						\n"
"	test	$0xD0,%al	# clear carry			\n"
"	jle	La_waitdiskready				\n"
"	jnp	La_waitdiskready				\n"
"	xor	%ax,%ax						\n"
"	cmpb	$4,%cs:CS_disk_base_table_bps	# 128 << 2 = 512	\n"
"	je	L_first_last_different	# skip if 2048 byte/s	\n"
"	mov	(%bp),%cl					\n"
"	and	$0x03,%cl					\n"
"	mov	$0xF0F0,%ax					\n"
"	rol	%cl,%al						\n"
"	add	%si,%cx						\n"
//"	and	$0x03,%cl	# not needed			\n"
"	neg	%cl						\n"
"	add	$4,%cl						\n"
"	and	$0x03,%cl					\n"
"	ror	%cl,%ah						\n"
"	and	$0x0F0F,%ax					\n"
"	mov	(%bp),%cx					\n"
"	clc							\n"
"	rcrw	4(%bp)						\n"
"	rcrw	2(%bp)						\n"
"	rcrw	(%bp)						\n"
"	clc							\n"
"	rcrw	4(%bp)						\n"
"	rcrw	2(%bp)						\n"
"	rcrw	(%bp)						\n"
"	and	$0x03,%cx					\n"
"	add	%cx,%si						\n"
"	add	$3,%si						\n"
"	shr	$2,%si						\n"
"	cmpw	$1,%si						\n"
"	jne	L_first_last_different				\n"
"	or	%al,%ah						\n"
"	xor	%al,%al		# not needed			\n"
"L_first_last_different:					\n"
"	# This is ignored if only one 2Kb sector read:		\n"
"	movb	%al,%cs:CS_first_read_mask			\n"
"	# bit set: skip from reading this 512byte sector:	\n"
"	movb	%ah,%cs:CS_last_read_mask			\n"
"	cmpb	$0x03,23(%bp)	# saved %ah			\n"
"	jne	L_not_partial_sector_write			\n"
"	cmpb	$0x43,23(%bp)	# saved %ah			\n"
"	jne	L_not_partial_sector_write			\n"
"	orb	%ah,%al						\n"
"	jnz	direct_return_carry_set_ah_3			\n"
"L_not_partial_sector_write:					\n"
#ifdef FLOPPYDBG
	" callw	print_crlf \n"
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'M',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov (%bp),%dx ; mov $0x2000 + 'L',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %si,%dx ; mov $0x2000 + 'N',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	sub	$6,%dx		# .feature			\n"
"	mov	$0,%al						\n"
"	outb	%al,%dx						\n"
"	inc	%dx		# .sector_count			\n"
"	outb	%al,%dx						\n"
"	inc	%dx		# .sector			\n"
"	outb	%al,%dx						\n"
"	mov	%si,%ax		# request nb sector		\n"
"	shl	$11,%ax		# * 2048			\n"
"	inc	%dx		# .cylinder_low			\n"
"	outb	%al,%dx						\n"
"	inc	%dx		# .cylinder_high		\n"
"	mov	%ah,%al						\n"
"	outb	%al,%dx						\n"
"	inc	%dx		# .lba_head			\n"
"	inc	%dx		# .command			\n"
"	movb	$0xA0,%al	# IDE_PACKET			\n"
"	outb	%al,%dx						\n"
"	inb	%dx,%al		# short wait			\n"
"	xor	%cx,%cx		# 65536*1us (approx) = 65 ms	\n"
"La_waitpacket:							\n"
"	inb	%dx,%al		# we need to wait longer	\n"
"	inb	%dx,%al		# with newer ultra speed	\n"
"	inb	%dx,%al		# DVD-RAM readers		\n"
"	inb	%dx,%al						\n"
"	test	$0x81,%al	# clear carry			\n"
"	loopnz	La_waitpacket					\n"
"	jcxz	La_ide_error	# timeout/error/no DRQ		\n"
"	test	$0x08,%al					\n"
"	jz	La_waitpacket					\n"
"	subw	$5,%dx		# .interrupt_reason		\n"
"	inb	%dx,%al						\n"
"	andb	$0x03,%al					\n"
"	cmpb	$0x01,%al	# needs C/D=1, I/O = 0		\n"
"	je	L_write_scsi_cmd				\n"
"La_ide_error:							\n"
"	stc							\n"
"	jmp	finish_fixup_irq				\n"
"	sub	%al,0x29(%bp,%di)				\n"
"	and	%al,0x74(%di)					\n"
"	imul	$0x656E,0x6E(%di),%sp				\n"
"	and	%cl,0x4F(%si)					\n"
"	push	%dx						\n"
"	push	%dx						\n"
"	inc	%cx						\n"
"	dec	%cx						\n"
"	dec	%si						\n"
"	and	%dh,(%bp,%si)					\n"
"	xor	%dh,(%bx,%si)					\n"
"	xor	$0xD2E,%ax					\n"
"	or	(%bx,%si),%al					\n"
"L_write_scsi_cmd:						\n"
"	sub	$2,%dx		# .data				\n"
"	mov	$0x002A,%ax	# SCSI WRITE			\n"
"	cmpb	$0x03,23(%bp)	# saved %ah			\n"
"	je	L_atapi_write					\n"
"	cmpb	$0x43,23(%bp)	# saved %ah			\n"
"	je	L_atapi_write					\n"
"	mov	$0x00BE,%ax	# old SCSI READ			\n"
"	mov	$0x0028,%ax	# new SCSI READ			\n"
"L_atapi_write:							\n"
#ifdef FLOPPYDBG
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'P',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	outw	%ax,%dx						\n"
"	movw	2(%bp),%ax					\n"
"	rolw	$8,%ax						\n"
#ifdef FLOPPYDBG
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'P',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	outw	%ax,%dx						\n"
"	movw	0(%bp),%ax					\n"
"	rorw	$8,%ax						\n"
#ifdef FLOPPYDBG
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'P',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	outw	%ax,%dx						\n"
"	xorw	%ax,%ax						\n"
#ifdef FLOPPYDBG
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'P',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	outw	%ax,%dx						\n"
"	mov	%si,%ax		# request nb sector < 256	\n"
"	mov	$0,%ah						\n"
#ifdef FLOPPYDBG
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'P',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	outw	%ax,%dx						\n"
"	xorw	%ax,%ax						\n"
#ifdef FLOPPYDBG
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'P',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	outw	%ax,%dx						\n"
"	mov	$30,%ah		# 30*65ms = 1950 ms		\n"
"L_atapi_continue_io:						\n"
"	add	$7,%dx		# .status			\n"
"	inb	%dx,%al		# short wait			\n"
"	xor	%cx,%cx		# 65536*1us (approx) = 65 ms	\n"
"La_waitcommandcomplete:					\n"
"	inb	%dx,%al		# we need to wait longer	\n"
"	inb	%dx,%al		# with newer ultra speed	\n"
"	inb	%dx,%al		# DVD-RAM readers		\n"
"	inb	%dx,%al						\n"
"	test	$0x81,%al	# clear carry			\n"
"	loopnz	La_waitcommandcomplete				\n"
"	jcxz	La_ide_timeout	# timeout/error/no DRQ		\n"
"	jmp	1f						\n"
"La_ide_timeout:						\n"
"	dec	%ah						\n"
"	jnz	La_waitcommandcomplete				\n"
"	jmp	La_ide_error					\n"
"	1:							\n"
"	test	$0x08,%al					\n"
"	jz	La_waitcommandcomplete				\n"
"	movw	%bx,%ax		# accept up to 64 K at a time	\n"
"	shr	$4,%ax						\n"
"	mov	%es,%cx						\n"
"	addw	%ax,%cx						\n"
"	mov	%cx,%es						\n"
"	shl	$4,%ax						\n"
"	sub	%ax,%bx						\n"
#ifdef FLOPPYDBG
	" callw	print_crlf \n"
	" pushaw ; mov %es,%dx ; mov $0x2000 + 'E',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %bx,%dx ; mov $0x2000 + 'B',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	sub	$3,%dx		# .cylinder_low			\n"
"	inb	%dx,%al						\n"
"	mov	%al,%ah						\n"
"	inc	%dx		# .cylinder_high		\n"
"	inb	%dx,%al						\n"
"	rol	$8,%ax						\n"
"	subw	$3,%dx		# .interrupt_reason		\n"
"	mov	%ax,%cx						\n"
"	shr	%cx		# read word by word		\n"
#ifdef FLOPPYDBG
	" pushaw ; mov %ax,%dx ; mov $0x2000 + 'n',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %si,%dx ; mov $0x2000 + 's',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	shr	$11,%ax		# / 2048			\n"
"	jz	La_ide_error	# something seriously wrong	\n"
"	subw	%ax,%si		# request nb sector - what left	\n"
"	js	La_ide_error	# something seriously wrong	\n"
"	inb	%dx,%al						\n"
"	testb	$0x01,%al					\n"
"	jnz	La_ide_error	# needs C/D=0			\n"
"	sub	$2,%dx		# .data				\n"
"	cmpb	$0x03,23(%bp)	# saved %ah			\n"
"	je	L_atapi_do_write				\n"
"	cmpb	$0x43,23(%bp)	# saved %ah			\n"
"	je	L_atapi_do_write				\n"
"	testb	$0x02,%al					\n"
"	jz	La_ide_error	# needs I/O = 1			\n"
//"	xchgw	%bx,%di						\n"
//"	rep	insw	%dx,%es:(%di)				\n"
//"	xchgw	%di,%bx						\n"
"L_atapi_do_read:						\n"
"	inc	%cl	# keep carry				\n"
"	dec	%cl	# keep carry				\n"
"	jnz	L_do_input_word	# each 256 16 bits words	\n"
"	test	%si,%si			# clear carry		\n"
"	jnz	L_first_sector					\n"
"	cmpw	$1024,%cx		# last sector read ?	\n"
"	ja	L_first_sector					\n"
#ifdef FLOPPYDBG
	" pushaw ; mov %cx,%dx ; mov $0x2000 + '*',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	rcrb	%cs:CS_last_read_mask	# use/set/clear carry	\n"
"	jmp	L_do_input_word		# start skipping	\n"
"L_first_sector:						\n"
#ifdef FLOPPYDBG
	" pushaw ; mov %cx,%dx ; mov $0x2000 + '.',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	clc							\n"
"	rcrb	%cs:CS_first_read_mask	# use/set/clear carry	\n"
"L_do_input_word:						\n"
"	inw	%dx,%ax						\n"
"	jc	L_no_read_512b_sector	# use carry		\n"
"	mov	%al,%es:(%bx)					\n"
"	inc	%bx						\n"
"	mov	%ah,%al						\n"
"	lahf			# save carry			\n"
"	jnz	L_read_no_inc_es_1				\n"
"	mov	%es,%bx						\n"
"	add	$0x10,%bh					\n"
"	mov	%bx,%es						\n"
"	xor	%bx,%bx						\n"
"	L_read_no_inc_es_1:					\n"
"	mov	%al,%es:(%bx)					\n"
"	inc	%bx						\n"
"	jnz	L_read_no_inc_es_2				\n"
"	mov	%es,%bx						\n"
"	add	$0x10,%bh					\n"
"	mov	%bx,%es						\n"
"	xor	%bx,%bx						\n"
"	L_read_no_inc_es_2:					\n"
"	sahf			# restore carry			\n"
"L_no_read_512b_sector:						\n"
"	loop	L_atapi_do_read					\n"
"	jmp	L_atapi_end_readwrite				\n"
//
"L_atapi_do_write:						\n"
"	testb	$0x02,%al					\n"
"	jnz	La_ide_error	# needs I/O = 0			\n"
//"	xchgw	%bx,%si						\n"
//"	rep	outsw	%es:(%si),%dx				\n"
//"	xchgw	%si,%bx						\n"
"	L_loop_write:						\n"
#ifdef FLOPPYDBG
	" pushaw ; mov %cx,%dx ; mov $0x2000 + 'W',%cx; callw print_cxdx ; popaw \n"
#endif /* FLOPPYDBG */
"	mov	%es:(%bx),%al					\n"
"	inc	%bx						\n"
"	jnz	L_write_no_inc_es_1				\n"
"	mov	%es,%bx						\n"
"	add	$0x10,%bh					\n"
"	mov	%bx,%es						\n"
"	xor	%bx,%bx						\n"
"	L_write_no_inc_es_1:					\n"
"	mov	%es:(%bx),%ah					\n"
"	inc	%bx						\n"
"	jnz	L_write_no_inc_es_2				\n"
"	mov	%es,%bx						\n"
"	add	$0x10,%bh					\n"
"	mov	%bx,%es						\n"
"	xor	%bx,%bx						\n"
"	L_write_no_inc_es_2:					\n"
"	outw	%ax,%dx						\n"
"	loop	L_loop_write					\n"
//
"L_atapi_end_readwrite:						\n"
"	orw	%si,%si						\n"
"	jnz	L_atapi_continue_io				\n"
"	clc							\n"
"	mov	$0,%ah						\n"
#ifdef FLOPPYDBG
	" pushf									 \n"
	" pushaw ; mov %es,%dx ; mov $0x2000 + 'E',%cx; callw print_cxdx ; popaw \n"
	" pushaw ; mov %bx,%dx ; mov $0x2000 + 'B',%cx; callw print_cxdx ; popaw \n"
	" popf									 \n"
#endif /* FLOPPYDBG */
"	jmp	all_sectors_rw_done				\n"

#ifdef FLOPPYDBG
"print_cxdx:				\n"
"	pushfw				\n"
"	push	%ax		\n"
"	mov	$0x02,%ah	\n"
"	int	$0x16		\n"
"	andb	$0x20,%al	\n"	/* only when num-lock active */
"	pop	%ax		\n"
"	jz	2f		\n"
"	movw	$0x0007,%bx		\n"
"	movb	$0x0E,%ah		\n"
"	movb	%cl,%al			\n"
"	int	$0x10			\n"
"	movb	%ch,%al			\n"
"	int	$0x10			\n"
"	movw	$12,%cx			\n"
"	1:				\n"
"	mov	%dx,%ax			\n"
"	shrw	%cl,%ax			\n"
"	andw	$0x000F,%ax # ALcarry	\n"
"	sahf		# clear AF	\n"
"	aaa				\n"
"	aad	$0x11			\n"
"	add	$0x0E30,%ax		\n"
"	int	$0x10			\n"
"	subw	$4,%cx			\n"
"	jns	1b			\n"
"	mov	$0x20,%al		\n"
"	int	$0x10			\n"
"	2:			\n"
"	popfw				\n"
"	retw				\n"

"print_crlf:				\n"
"	pushfw				\n"
"	push	%ax		\n"
"	mov	$0x02,%ah	\n"
"	int	$0x16		\n"
"	andb	$0x20,%al	\n"	/* only when num-lock active */
"	pop	%ax		\n"
"	jz	2f		\n"
"	pushw	%es			\n"
"	pushw	%ds			\n"
"	pushaw				\n"
"	movw	$0x0007,%bx		\n"
"	movw	$0x0E00 + '\r',%ax	\n"
"	int	$0x10			\n"
"	movw	$0x0E00 + '\n',%ax	\n"
"	int	$0x10			\n"
//"	mov	$0x8600,%ax		\n"
//"	cwd	# %dx= 0xFFFF		\n"
//"	mov	$30,%cx # 30 * 15 ms	\n"
//"	int	$0x15   # _BIOS_wait	\n"
"	popaw				\n"
"	popw	%ds			\n"
"	popw	%es			\n"
"	2:			\n"
"	popfw				\n"
"	retw				\n"
#endif /* FLOPPYDBG */
"	.align 4			\n"
"CS_remap_adr:				\n"
#ifdef FLOPPYDBG
"	.long	0, 0	# lba index 0	\n"
"	.long	0	# nb index 0	\n"
"	.skip	12 * " STRING(FRAGMENTED_DISK_NBFRAG) " \n"
#endif
"newINT13end:				\n"

#define WINNT_BIOS_SWAP_NEEDED
#ifdef WINNT_BIOS_SWAP_NEEDED
// Totally different, BIOS swap disk 0x80 and 0x81:
"	.previous			\n"
"	.section	.rodata		\n"
"	.align 16			\n"
"int0x13_intercept:			\n"
#if 0 // TOTEST: behaves better if Windows95+ started?
"	jmp 1f					\n"
"	nop					\n"
"	.ascii	"$INT13SF"			\n"
"	.ascii	"Gujin   "			\n"
"oldintaddr:					\n"
"	.long	0				\n"
"	.long	1	# or 0 ?		\n"
"	1:					\n"
"	pushl	%cs:old_int0x13_anchor + 1	\n"
"	popl		%cs:oldintaddr		\n"
#endif
// FIXME: %dl is changed at return to INT0x13...
"	pushfw				\n"
"testdisk1_anchor:			\n"
"	cmpb	$0x80,%dl		\n"
"	jne	int0x13_next		\n"
"replacedisk1_anchor:			\n"
"	mov	$0x81,%dl		\n"
"	jmp	int0x13_done		\n"
"int0x13_next:				\n"
"testdisk2_anchor:			\n"
"	cmpb	$0x81,%dl		\n"
"	jne	int0x13_done		\n"
"replacedisk2_anchor:			\n"
"	mov	$0x80,%dl		\n"
"int0x13_done:				\n"
"	popfw				\n"
"old_int0x13_anchor:			\n"
"	ljmpw	$0xfedc,$0xba98		\n"
"int0x13_intercept_end:			\n"

#endif /* WINNT_BIOS_SWAP_NEEDED */
"	.previous			\n"
	);

/*
  Floppy stuff NOT managed:
	40:3E   byte    Drive recalibration status

	76543210 drive recalibration status
	        +-- 1=recalibrate drive 0
	       +--- 1=recalibrate drive 1
	      +---- 1=recalibrate drive 2
	     +----- 1=recalibrate drive 3
	  +---------- unused
	 +----------- 1=working interrupt flag

Bitfields for diskette recalibrate status:
Bit(s)	Description	(Table M0012)
 7	diskette hardware interrupt occurred
 6-4	reserved
 3	recalibrate diskette 3 (PC,XT only)
 2	recalibrate diskette 2 (PC,XT only)
 1	recalibrate diskette 1
 0	recalibrate diskette 0

	40:3F   byte    Diskette motor status

	76543210 diskette motor status
	        +-- 1=drive 0 motor on
	       +--- 1=drive 1 motor on
	      +---- 1=drive 2 motor on
	     +----- 1=drive 3 motor on
	  +---------- unused
	 +----------- 1=write operation

Bitfields for diskette motor status:
Bit(s)	Description	(Table M0013)
 7	current operation is write or format, rather than read or verify
 6	reserved (DMA enabled on 82077)
 5-4	diskette drive number selected (0-3)
 3	diskette 3 motor on (PC,XT only)
 2	diskette 2 motor on (PC,XT only)
 1	diskette 1 motor on
 0	diskette 0 motor on

	40:40   byte    Motor shutoff counter (decremented by INT 8)
	40:41   byte    Status of last diskette operation (see INT 13,1)

	76543210 status of last diskette operation
	        +--- invalid diskette command
	       +---- diskette address mark not found
	      +----- sector not found
	     +------ diskette DMA error
	    +------- CRC check / data error
	   +-------- diskette controller failure
	  +--------- seek to track failed
	 +---------- diskette time-out

Bitfields for diskette last operation status:
Bit(s)	Description	(Table M0014)
 7	drive not ready
 6	seek error
 5	general controller failure
 4-0	error reason
	00h no error
	01h invalid request/parameter
	02h address mark not found
	03h write-protect error
	04h sector not found
	06h diskette change line active
	08h DMA overrun
	09h DMA across 64k boundary
	0Ch media type unknown
	10h CRC error on read
Note:	the following values for this byte differ somewhat from the
	  bitfield definition above:
	    30h drive does not support media sense
	    31h no media in drive
	    32h drive does not support media type
	    AAh diskette drive not ready

	40:42  7 bytes  NEC diskette controller status (see FDC)

	40:8B   byte    Last diskette data rate selected

	76543210 last diskette data rate selected
	     +--------- reserved
	   +------------ last floppy drive step rate selected
	 +-------------- last floppy data rate selected

	Data Rate                       Step Rate
	00  500K bps            00  step rate time of 0C
	01  300K bps            01  step rate time of 0D
	10  250K bps            10  step rate time of 0A
	11  reserved            11  reserved

Bitfields for diskette media control:
Bit(s)	Description	(Table M0028)
 7-6	last data rate set by controller
	00=500kbps, 01=300kbps, 10=250kbps, 11=1Mbps
 5-4	last diskette drive step rate selected
	00=0Ch, 01=0Dh, 10=0Eh, 11=0Ah
 3-2	{data rate at start of operation}
 1-0	reserved
Note:	EHD BIOS sets this byte to 01h and never reads it back

	40:8F   byte    Combination hard/floppy disk card when bit 0 set

Bitfields for diskette controller information:
Bit(s)	Description	(Table M0029)
 7	reserved
 6	=1 drive 1 determined
 5	=1 drive 1 is multi-rate, valid if drive determined
 4	=1 drive 1 supports 80 tracks, always valid
 3	reserved
 2	=1 drive 0 determined
 1	=1 drive 0 is multi-rate, valid if drive determined
 0	=1 drive 0 supports 80 tracks, always valid
Note:	EHD BIOS sets this byte to 01h and never alters it again

	40:90  4 bytes  Drive 0,1,2,3 media state

	76543210 drive media state (4 copies)
	      +------- drive/media state (see below)
	     +------- reserved
	    +------- 1=media/drive established
	   +------- double stepping required
	 +--------- data rate:  00=500K bps    01=300K bps
	10=250K bps    11=reserved
	Bits
	210  Drive Media State
	000  360Kb diskette/360Kb drive not established
	001  360Kb diskette/1.2Mb drive not established
	010  1.2Mb diskette/1.2Mb drive not established
	011  360Kb diskette/360Kb drive established
	100  360Kb diskette/1.2Mb drive established
	101  1.2Mb diskette/1.2Mb drive established
	110  Reserved
	111  None of the above

Bitfields for diskette drive media state:
Bit(s)	Description	(Table M0030)
 7-6	data rate
	00=500kbps, 01=300kbps, 10=250kbps, 11=1Mbps
 5	double stepping required (e.g. 360kB in 1.2MB)
 4	media type established
 3	drive capable of supporting 4MB media
 2-0	on exit from BIOS, contains
	000 trying 360kB in 360kB
	001 trying 360kB in 1.2MB
	010 trying 1.2MB in 1.2MB
	011 360kB in 360kB established
	100 360kB in 1.2MB established
	101 1.2MB in 1.2MB established
	110 reserved
	111 all other formats/drives

Note:	officially "Drive 2 media state"
Bitfields for diskette drive 0 media state at start of operation:
Bit(s)	Description	(Table M0031)
 7-3 (see #M0030)
 2	multiple data rate capability determined
 1	multiple data rate capability
 0	=1 if drive has 80 tracks, =0 if 40 tracks

	40:94   byte    Track currently seeked to on drive 0
	40:95   byte    Track currently seeked to on drive 1
*/

static unsigned
menu_load_other (unsigned index, struct registers *regs, struct gujin_param_attrib gujin_attr)
  {
  struct desc_str *bootwaydesc = &BOOTWAY.desc[index]; /* not const, bootwaydesc->filesize may be modified by file_treat() */
  struct diskparam_str *dp = &DI.param[bootwaydesc->disk], *dp1 = dp;
  unsigned char dskfound = 0;

  const unsigned long long base_sector = dp->partition[bootwaydesc->partition].start + dp->fulldisk_offset;
  const unsigned long long partlength = dp->partition[bootwaydesc->partition].length;
  const unsigned char parttype = BOOTWAY.iso.external_fs;
  unsigned long long sector_lba = bootwaydesc->inode;
  unsigned nb_sector = DEFAULT_BUFFER_SIZE/dp->bytepersector, emulation_nb_sectors = 0, cdrom_lba = 0;
  int partition = bootwaydesc->partition;
  farptr loadaddr = 0x7C00;
  unsigned char boot_media_type = 0xFF;	/* != 0xFF when simulating a disk */
  struct sector_chain_str remap_array[FRAGMENTED_DISK_NBFRAG];

  /* MSDOS 5/6 file IO.SYS consider %dl to contain the
   * boot device (saved at offset 301):
   * objdump --disassemble-all -b binary -m i8086 /mnt/floppy/io.sys | less
   * %dh should be zero if boot device supports
   * int $0x13, else (?) 0x20 (bit 5 set).
   * Gujin may also need it if "--disk=EBIOS:0x80,auto" on MBR
   */
  if (dp->access == bios_chs || dp->access == ebios_lba) {
      regs->edx &= ~0xFFFF;
      regs->edx |= dp->disknb;
      dskfound = 1;
// printf ("reboot BIOS disk edx = 0x%X\r\n", regs->edx); _BIOS_wait (5000000);
      }
    else {
      unsigned short cpt;

      /* works only for Phoenix extended BIOSes: */
      for (cpt = 0; cpt < DI.nbdisk; cpt++) {
	  if (   DI.param[cpt].access != bios_chs
	      && DI.param[cpt].access != ebios_lba)
	      continue;
	  if (dp->ideIOadr != DI.param[cpt].ideIOadr)
	      continue;
	  if ((dp->lba_slave_mask & 0x30) == (DI.param[cpt].lba_slave_mask & 0x30)) {
	      regs->edx &= ~0xFFFF;
	      regs->edx |= DI.param[cpt].disknb;
	      dp1 = &DI.param[cpt];
	      dskfound = 1;
// printf ("reboot IDE disk edx = 0x%X (ideIOadr/lba_slave_mask 0x%X/0x%X)\r\n", regs->edx, DI.param[cpt].ideIOadr, DI.param[cpt].lba_slave_mask);
	      break;
	      }
	  }
      if (cpt >= DI.nbdisk) {
	  DBG (("NO BIOS drive found for dp->access %u dp->ideIOadr 0x%X dp->lba_slave_mask 0x%X, Shall we abort?\r\n", dp->access, dp->ideIOadr, dp->lba_slave_mask));
	  // return 4; /* is there a valid way to come here? (IDE disk not supported by Phoenix extended BIOS booting MBR/PBR) */
	  }
      }

  if (bootwaydesc->boottype == is_MBR) {
      /* MBR booting, cannot really check anything */
      partition = -1;
      }
    else if (dp->partition[bootwaydesc->partition].misc.beer_partition) {
      /* Reboot to the BEER without too many problems, and no hide_unhide_partitions(): */
      sector_lba = 0;
      }
    else if (bootwaydesc->boottype == is_PBR && bootwaydesc->name_offset == 0) { /* if >= 1 it is MBR/PBR in a file */
      /* Booting a boot record of a partition: */
      partition = -1;
      if (dp->bytepersector != 512)
	  boot_media_type = boot_media_144;
	else if ((dp->access == bios_chs || dp->access == ebios_lba) &&  dp->disknb < 0x80)
	  /* floppy boot: do not hide anything */ ;
	else if (dp->partition[bootwaydesc->partition].type >= 0x20)
	  /* *BSD boot, no hidding, no joke! */ ;
	else if (gujin_attr.hide_unhide_partitions) {
	  unsigned tmp = hide_unhide_partitions (index);
	  if (tmp != 0) {
	      PRINTF (ERROR_HIDE_UNHIDE_X, tmp);
	      if (tmp < 0x10)
		  PRINT (ERROR_HIDE_UNHIDE_UNCHANGED);
		else {
		  if (!gujin_attr.disk_write_enable)
		      PRINTF (ERROR_HIDE_UNHIDE_READONLY, SMENU_disk_write_enable);
		    else
		      PRINT (ERROR_HIDE_UNHIDE_VIRUS);
		  }
	      return 2;
	      }
#if 0	/* Window XP do not start when HPA set and locked, nor when disksize reduced (unless installed with it?) */
	  if (gujin_attr.ATA_protect_partition && bootwaydesc->partition == 0) {
	      /* We are booting a boot record of a partition, if it is DOS/window,
		protect the Linux/BSD partition from viruses infections */
	      /* We only handle Windows as first partition, a possible FAT partition
		to exchange data, and a non window compatible partition - the
		HPA will be set before the non window compatible partition.
		We know we are booting a partition type < 0x20,
		extended partitions are at end of the array dp->partition */
	      printf ("Window booting %u/%u partitions, type 0x%X, next: 0x%X, 0x%X\r\n",
		bootwaydesc->partition, dp->nbpartition, dp->partition[0].type, dp->partition[1].type, dp->partition[2].type);
	      if (dp->nbpartition > 2 && dp->partition[1].type < 0x20 && dp->partition[2].type >= 0x20)
		  printf ("set disksize at begin of 3rd partition, i.e. 0x%llX\r\n", dp->BEER_disksize = dp->partition[2].start);
//		  printf ("set disksize at begin of 3rd partition, i.e. 0x%llX\r\n", dp->BEER_HostProtectedArea_start = dp->partition[2].start);
		else if (dp->nbpartition >= 2 && dp->partition[1].type >= 0x20)
		  printf ("set disksize at begin of 2nd partition, i.e. 0x%llX\r\n", dp->BEER_disksize = dp->partition[1].start);
//		  printf ("set disksize at begin of 2nd partition, i.e. 0x%llX\r\n", dp->BEER_HostProtectedArea_start = dp->partition[1].start);
	      /* we should also modify disksize/HPA of the EBIOS disk if dp points to ide/lba disk */
	      _BIOS_wait (3000000);
	      }
#endif
	  }
      }
    else {
      const ISO9660_Default_entry *catalog = 0;

      if (bootwaydesc->boottype != is_PBR && bootwaydesc->name_offset != 0 && dp->partition[bootwaydesc->partition].ElToritoCatalog != 0) {
	  /* DIRTY: name_offset is the index inside boot catalog, start at 1 for El-torito else BDI */
	  catalog = &dp->partition[bootwaydesc->partition].ElToritoCatalog[bootwaydesc->name_offset];
	  ZDBG (("Booting El-Torito entry %u, boot_media_type 0x%X, system_type 0x%X, load_segment 0x%X\r\n", bootwaydesc->name_offset, catalog->boot_media_type, catalog->system_type, catalog->load_segment));

	  if (catalog->sector_lba == 0 || catalog->sector_count == 0) {
	      ZDBG (("EL-TORITO sector_lba %u sector_count %u cannot work!\r\n", catalog->sector_lba, catalog->sector_count));
	      return 1;
	      }
	  if (catalog->load_segment != 0)
	      loadaddr = catalog->load_segment << 4;
	  if (loadaddr + catalog->sector_count * 2048 > STATE.codeseg_adr >> 12) {
	      ZDBG (("EL-TORITO load address 0x%X plus sector_count %u times 2048 bytes sectors overwrite Gujin code segment 0x%X\r\n",
			loadaddr, catalog->sector_count, STATE.codeseg_adr >> 16));
	      return 5;
	      }
	    else if (loadaddr > (STATE.codeseg_adr >> 12) && loadaddr < (STATE.dataseg_adr >> 12) + 0x10000) { /* 64 Kb after stack safety, FIXME BIG_MALLOC */
	      ZDBG (("EL-TORITO load address 0x%X overwrite Gujin code/data/stack segment 0x%X..0x%X + 0x1000\r\n",
			loadaddr, STATE.codeseg_adr >> 16, STATE.dataseg_adr >> 16));
	      return 6;
	      }
	  boot_media_type = catalog->boot_media_type;
	  cdrom_lba = catalog->sector_lba;
	  sector_lba = catalog->sector_lba;
	  nb_sector = catalog->sector_count;
	  if (boot_media_type == boot_media_120)
	      emulation_nb_sectors = 2400;
	    else if (boot_media_type == boot_media_144)
	      emulation_nb_sectors = 2880;
	    else if (boot_media_type == boot_media_288)
	      emulation_nb_sectors = 5760;
	    else if (boot_media_type == boot_media_noemul || boot_media_type == boot_media_hd) {
	      if (bootwaydesc->iso_inode != 0)
		  emulation_nb_sectors = bootwaydesc->iso_filesize / 512;
		else
		  emulation_nb_sectors = dp->partition[bootwaydesc->partition].length;
	      if (boot_media_type == boot_media_hd)
		  emulation_nb_sectors -= catalog->sector_lba * (2048 / 512);
	      }
	  if (dp->bytepersector == 512) {
	      sector_lba *= 4;
	      nb_sector *= 4;
	      }
	  }

      if (!catalog || bootwaydesc->iso_inode != 0) {
	  unsigned nb_remap_fragment = sizeof (remap_array)/sizeof (remap_array[0]);
	  ZDBG (("Booting %s: get fragment array (max frags %u)\r\n", catalog? "a El-Torito in an iso image" : "a BDI/BOOTSECT.DOS", nb_remap_fragment));

	  /* scientists have detected explosions on the surface of Mars */
	  if (file_get_chain (bootwaydesc, &nb_remap_fragment, remap_array) != 0) {
	      ZDBG (("cannot get the fragment list for BDI/BOOTSECT.DOS/ISO file\r\n"));
	      return 9;
	      }

	  ZDBG (("fragment list for file (size %u) of %u elements: ", bootwaydesc->filesize, nb_remap_fragment));

	  if (!catalog) { /* The DI.readsector() is in current sector size */
	      sector_lba = remap_array[0].lba + base_sector;
	      nb_sector = remap_array[0].nb;
	      /* We could write anything here, just write the beginning of the BDI file: */
	      cdrom_lba = remap_array[0].lba + base_sector;
	      }

	  unsigned total_nb_sectors = 0;
	  unsigned cptfrag;
	  unsigned char shift = 0;
	  if (!catalog && dp->bytepersector == 2048)
	      shift = 2;
	  /* Undo if emulated disk sector size is 2048 bytes: */
	  if (shift == 2 && DI.readsector (dp, -1, sector_lba, 1, loadaddr) == 0) {
	      unsigned char sig0x29_fat16 = peekb (loadaddr + 0x26);
	      unsigned char sig0x29_fat32 = peekb (loadaddr + 0x42);
	      unsigned short emulated_sectsize = peekw (loadaddr + 0x0B);
	      if (sig0x29_fat16 == 0x29 || sig0x29_fat16 == 0x28 || sig0x29_fat32 == 0x29 || sig0x29_fat32 == 0x28)
		  switch (emulated_sectsize) {
		    case 2048: shift = 0; break;
		    }
	      }
//	  if (base_sector && shift) { printf ("base_sector %lld shift %u", base_sector, shift); _BIOS_getkey(); }
	  for (cptfrag = 0; cptfrag < nb_remap_fragment; cptfrag++) {
	      remap_array[cptfrag].lba <<= shift;
	      remap_array[cptfrag].nb <<= shift;
	      ZDBG (("%u sector at %llU+%llU, ", remap_array[cptfrag].nb, remap_array[cptfrag].lba, base_sector));
	      if (remap_array[cptfrag].lba || (cptfrag == 0 && remap_array[1].nb == 0)) /* not for holes, but for CDnoemul */
		  remap_array[cptfrag].lba += base_sector;
	      total_nb_sectors += remap_array[cptfrag].nb;
	      }
	  ZDBG ((" (total_nb_sectors %u) end.\r\n", total_nb_sectors));
	  if (!catalog) {
	      emulation_nb_sectors = total_nb_sectors;
	      if (dp->bytepersector == 2048)
		  emulation_nb_sectors <<= shift;
	      /* KANOTIX-2005-04 contains memtest.bin of 195 sectors, filesize 99784 bytes
		without 0x29 signature, this file loads more sectors than that,
		setup a minimum for the emulated size of 2*80*15 = 2400 sectors, end will be a hole */
	      if (emulation_nb_sectors < 2400 / (2048/512))
		  emulation_nb_sectors = 2400 / (2048/512);
	      if (bootwaydesc->boottype != is_PBR) {
		  /* We will start a simulated BIOS disk for this BDI file: */
		  if (total_nb_sectors * dp->bytepersector > 5760 * 512)
		      boot_media_type = boot_media_hd;
		    else if (total_nb_sectors * dp->bytepersector > 2880 * 512)
		      boot_media_type = boot_media_288;
		    else if (total_nb_sectors * dp->bytepersector > 2400 * 512)
		      boot_media_type = boot_media_144;
		    else
		      boot_media_type = boot_media_120;
		  }
	      }
	  }

      if (catalog) {
	  if (bootwaydesc->iso_inode != 0) {
	      unsigned segment = 0;
	      while (sector_lba >= remap_array[segment].nb)
		  sector_lba -= remap_array[segment++].nb;
	      unsigned last_offset = sector_lba;
	      sector_lba += remap_array[segment].lba;
	      /* FIXME: DI.readsector() assumes we have catalog->sector_count without segment discontinuity */
	      if (boot_media_type != boot_media_noemul) {
		  /* we need to shift and limit remap_array[] to catalog->sector_lba and catalog->sector_count */
		  if (segment) {
		      struct sector_chain_str *dst = remap_array, *src = &remap_array[segment];
		      do
			  *dst++ = *src;
			  while ((src++)->nb != 0);
		      }
		  segment = 0;
		  remap_array[segment].lba += last_offset;
		  remap_array[segment].nb -= last_offset;
		  unsigned nbsect = emulation_nb_sectors;
		  while (nbsect > remap_array[segment].nb)
		      nbsect -= remap_array[segment++].nb;
		  remap_array[segment].nb = nbsect;
		  remap_array[segment + 1].lba = 0;
		  remap_array[segment + 1].nb = 0;
		  }
	      }
	    else {
	      sector_lba += base_sector;
	      if (boot_media_type == boot_media_noemul)
		  remap_array[0].lba = base_sector;
		else /* We assume the 120/144/288 floppy or hd is contiguous, always true on ISO9660: */
		  remap_array[0].lba = sector_lba * (dp->bytepersector / 512);
	      remap_array[0].nb = emulation_nb_sectors;
	      remap_array[1].lba = 0;
	      remap_array[1].nb = 0;
	      }
	  }

      unsigned max_nb_sector = ((STATE.codeseg_adr >> 12) - loadaddr) / dp->bytepersector;
      if (max_nb_sector * dp->bytepersector > 0x8000)
	  max_nb_sector = 0x8000 / dp->bytepersector;
      if (nb_sector > max_nb_sector)
	  nb_sector = max_nb_sector;
      partition = -1; /* all LBA relative to beginning of disk */
      }

  if (DI.readsector (dp, partition, sector_lba, nb_sector, loadaddr) != 0) {
      ZDBG (("cannot read %u sectors at lba %llu (partition %d) to address 0x%X\r\n", nb_sector, sector_lba, partition, loadaddr));
      return 7;
      }
  ZDBG (("read %u sectors at lba %llu partition %d to 0x%X success\r\n", nb_sector, sector_lba, partition, loadaddr));

#if 0
  unsigned nbnull = 0, cpt;
  while (peekb(loadaddr + nbnull) == 0)
      nbnull++;
  printf ("Has read %u sectors at lba %llu partition %d (%llu) to 0x%X, starts with %u nulls, then:\r\n", nb_sector, sector_lba, partition, base_sector, loadaddr, nbnull);
  for (cpt = 0; cpt < 64; cpt++)
      printf ("0x%X, ", peekb(loadaddr + nbnull + cpt));
  printf ("\r\nPress a key to continue...");
  _BIOS_getkey();
#endif

  union {
      FAT_bootsect1_t fatcommon;
      bootbefore_FAT32_t fat32;
      bootbefore_t fat16;
      } bootbefore;	/* temporary copy of MBR/PBR */
  lmemcpy (&bootbefore, loadaddr, sizeof (bootbefore));
  ZDBG (("MBR NbHead: %u , NbSectorpertrack: %u\r\n", bootbefore.fatcommon.NbHead, bootbefore.fatcommon.NbSectorpertrack));

  if (boot_media_type != 0xFF) {
      const FAT_bootsect2_t *part2 = &bootbefore.fat16.part2;
      if (part2->Signaturebyte0x29 != 0x29 && part2->Signaturebyte0x29 != 0x28
	&& (bootbefore.fat32.part2.Signaturebyte0x29 == 0x29 || bootbefore.fat32.part2.Signaturebyte0x29 == 0x28))
	part2 = &bootbefore.fat32.part2;
      regs->edx = setup_emulation (dp1, &bootbefore.fat16.part1, part2, boot_media_type, bootwaydesc->filename,
		emulation_nb_sectors, cdrom_lba, base_sector, partlength, parttype, !bootwaydesc->ReadOnly, remap_array);

      /* We cannot stop emulation: it may change the amount of memory < 640K available and change the INT 0x13 vector
	that we have already copied inside the new IRQ treatment - or we should do it before finalise_emul() */
      if (/*UTIL.bios_boot_dl == dp1->disknb &&*/ UTIL.CDROM_HDFD_emulation_state == 1) {
	  ZDBG (("Forbid 'late stop CDROM emulation' of BIOS 0x%X to keep the new floppy emulation!.\r\n", UTIL.bios_boot_dl));
	  UTIL.CDROM_HDFD_emulation_state = 0;
	  }
      }
   /*
    * Check if we have a FAT16/FAT32 partition or something else:
    * So you should be able to install Windows to one disk and move this disk
    * to another BIOS number, or add a disk to a lower IDE interface without problem.
    * You should also be able to install Windows to a primary partition and then
    * copy _all_ its files (better use Linux "cp -a" for that) to an extended partition, and
    * then boot this extended partition. This extended partition need to have  type
    * Win95_FAT32LBA or Win95_FAT16LBA if you need extended BIOS to access it
    * because it is over the standard BIOS limit - and that is not the default of DOS FDISK.
    */
#ifdef WINNT_BIOS_SWAP_NEEDED
  if (*CAST (unsigned long long *, bootbefore.fatcommon.String) == *CAST (unsigned long long *, "NTFS    ")
//	&& bootbefore.part2.PhysicaldriveNb != (regs->edx & 0xFF)) { /* regs->edx may be modified at top of this function */
	&& (regs->edx & 0x7F) != 0) { /* regs->edx may be modified at top of this function */
      /* NTFS does not boot by modifying the bootsector, we need to swap disks at BIOS level (DELL USB simulated on 0x80) */
      extern char int0x13_intercept[], old_int0x13_anchor[], int0x13_intercept_end[];
      extern char testdisk1_anchor[], replacedisk1_anchor[], testdisk2_anchor[], replacedisk2_anchor[];
      unsigned copy_size = (unsigned)int0x13_intercept_end - (unsigned)int0x13_intercept;
#if 1
      unsigned copy_segment = _BIOS_getEBDA_segment ();
      if (!copy_segment)
	  copy_segment = _BIOS_getBaseMemory() * (1024 / 16); /* 16 bytes segments */
      copy_segment -= (copy_size + 15)/16;
#else
      unsigned copy_segment = 0x0030; /* Bottom of stack area used during post and bootstrap */
#endif
      ZDBG (("NFTS signature: '%s', swap disk 0x%X and 0x%X by intercepting INT0x13 at 0x%X:0x0\r\n",
		bootbefore.fatcommon.String, bootbefore.fat16.part2.PhysicaldriveNb, (regs->edx & 0xFF), copy_segment));

      /* NTFS PhysicaldriveNb at 0x24 and NbHiddensector at 0x1c (recalculated disk size at 0x20), same as FAT16 */
      ZDBG (("Adjust NbHiddensector from %u to %llu\r\n", bootbefore.fatcommon.NbHiddensector, base_sector));
      /* Try to modify that field, even when there isn't any 0x29 signature: */
//      bootbefore.fatcommon.NbHiddensector = base_sector; // is that needed or even working?
      testdisk1_anchor[2] = replacedisk2_anchor[1] = bootbefore.fat16.part2.PhysicaldriveNb; /* NTFS disk Nb same as FAT16 */
      testdisk2_anchor[2] = replacedisk1_anchor[1] = (regs->edx & 0xFF);
      regs->edx = (regs->edx & 0xFFFF0000) | bootbefore.fat16.part2.PhysicaldriveNb; /* NTFS disk Nb same as FAT16 */
      *(unsigned *)&old_int0x13_anchor[1] = peekl (0x13 * 4);
      ZDBG (("Copy %u bytes from 0x%X to farptr 0x%X\r\n", copy_size, int0x13_intercept, copy_segment << 16));
      smemcpy (copy_segment << 16, int0x13_intercept, copy_size);
      //pokel (0x13 * 4, copy_segment << 16);
      UTIL.swap_BIOSint0x13_irqhandler = copy_segment << 16;
#if 0
      if (UTIL.CDROM_HDFD_emulation_state == 1) {
	  ZDBG (("Forbid 'late stop CDROM emulation' to keep the BIOS 0x80 emulation!.\r\n"));
	  UTIL.CDROM_HDFD_emulation_state = 0;
	  }
#endif
      }
    else if (bootbefore.fat16.part2.Signaturebyte0x29 == 0x29 || bootbefore.fat16.part2.Signaturebyte0x29 == 0x28) {
#else /* WINNT_BIOS_SWAP_NEEDED */
    if (bootbefore.fat16.part2.Signaturebyte0x29 == 0x29 || bootbefore.fat16.part2.Signaturebyte0x29 == 0x28
	|| *CAST (unsigned long long *, bootbefore.fatcommon.String) == *CAST (unsigned long long *, "NTFS    ")) {
#endif /* WINNT_BIOS_SWAP_NEEDED */
      ZDBG (("FAT16 0x29 signature: 0x%X", bootbefore.fat16.part2.Signaturebyte0x29));
      if (bootwaydesc->boottype == is_PBR && bootwaydesc->name_offset != 0) {	/* if >= 1 it is MBR/PBR in a file */
	  /* BOOTSECT.DOS system loaded is assumed not to be on a floppy, unless BOOTSECT.DOS is itself on it */
	  if (bootbefore.fat16.part2.PhysicaldriveNb)
	      regs->edx = (regs->edx & ~0xFFFF) | bootbefore.fat16.part2.PhysicaldriveNb;
	    else
	      bootbefore.fat16.part2.PhysicaldriveNb = regs->edx;
	  ZDBG ((", MBR/PBR in a file, set regs->edx to 0x%X", regs->edx));
	  }
	else {
	  if (dskfound && bootbefore.fat16.part2.PhysicaldriveNb != (regs->edx & 0xFF)) {
	      ZDBG ((", adjust PhysicaldriveNb from 0x%X to 0x%X", bootbefore.fat16.part2.PhysicaldriveNb, (regs->edx & 0xFF)));
	      bootbefore.fat16.part2.PhysicaldriveNb = regs->edx;
	      }
	  if (boot_media_type == 0xFF) {	/* NOT for floppy emulation */
	      ZDBG ((", adjust NbHiddensector from %u to %llu", bootbefore.fat16.part1.NbHiddensector, base_sector));
	      /* If we have a non corrupted FAT boot record in a PBR, we already have part1.NbHiddensector setup,
		but DOS FDISK can put 63 in NbHiddensector if it has been used in its 2Gb compatibility mode once: */
	      bootbefore.fat16.part1.NbHiddensector = base_sector;
	      }
	  }
      }
    else if ((bootbefore.fat32.part2.Signaturebyte0x29 == 0x29 || bootbefore.fat32.part2.Signaturebyte0x29 == 0x28)) {
      ZDBG (("FAT32 0x29 signature 0x%X", bootbefore.fat32.part2.Signaturebyte0x29));
      if (bootwaydesc->boottype == is_PBR && bootwaydesc->name_offset != 0) {	/* if >= 1 it is MBR/PBR in a file */
	  /* BOOTSECT.DOS system loaded is assumed not to be on a floppy, unless BOOTSECT.DOS is itself on it */
	  if (bootbefore.fat32.part2.PhysicaldriveNb)
	      regs->edx = (regs->edx & ~0xFFFF) | bootbefore.fat32.part2.PhysicaldriveNb;
	    else
	      bootbefore.fat32.part2.PhysicaldriveNb = regs->edx;
	  ZDBG ((", MBR/PBR in a file, set regs->edx to 0x%X", regs->edx));
	  }
	else {
	  if (dskfound && bootbefore.fat32.part2.PhysicaldriveNb != (regs->edx & 0xFF)) {
	      ZDBG ((", adjust PhysicaldriveNb from 0x%X to 0x%X", bootbefore.fat32.part2.PhysicaldriveNb, (regs->edx & 0xFF)));
	      bootbefore.fat32.part2.PhysicaldriveNb = regs->edx;
	      }
	  if (boot_media_type == 0xFF) {	/* NOT for floppy emulation */
	      ZDBG ((", adjust NbHiddensector from %u to %llu", bootbefore.fat32.part1.NbHiddensector, base_sector));
	      /* If we have a non corrupted FAT boot record in a PBR, we already have part1.NbHiddensector setup,
		but DOS FDISK can put 63 in NbHiddensector if it has been used in its 2Gb compatibility mode once: */
	      bootbefore.fat32.part1.NbHiddensector = base_sector;
	      }
	  }
      }
  ZDBG (("\r\n"));
  if (   bootbefore.fatcommon.jmpinstruction[0] == 0xeb /* short jmp */
     && bootbefore.fatcommon.EBIOSaccess == 0x90
     && dp->access != bios_chs /* safety, shall not  have Win95_FAT*LBA then! */
     && (   dp->partition[bootwaydesc->partition].type == Win95_FAT32LBA
	 || dp->partition[bootwaydesc->partition].type == Win95_FAT16LBA))
	 bootbefore.fatcommon.EBIOSaccess = Win95_FAT16LBA;
  if (bootwaydesc->boottype != is_PBR || bootwaydesc->name_offset == 0)
      smemcpy (loadaddr, &bootbefore, sizeof(bootbefore));

  if (bootwaydesc->boottype == is_PBR) {
      /*
       * SYSLINUX, more exactly extlinux, would like %ds:%si pointing to the partition description.
       * That would be pointing to the initial MBR which has been moved usually at 0x600 (note
       * that the address is 0x600 and not 0x6000, that is 0x200 over the interrupt vector table,
       * not just below 0x7C00).
       * There is a serious problem if the partition booted is an extended partition, because the
       * start in the partition table is relative to the start of the extended partition. This
       * is not usually seen as a problem because bootloaders which loads extended partition
       * will not usually preserve content of %si so %ds:%si will not look like a partition
       * description. Gujin preserves registers and does not touch 0x600..0x800 so we will
       * use a sonic screwdriver to fix that the way other bootloaders like it.
       */
      if (   bootwaydesc->inode != dp->fulldisk_offset /* floppies do not participate */
	  && DI.readsector (dp, -1, dp->fulldisk_offset, 1, 0x600) == 0) {
	  unsigned order = dp->partition[bootwaydesc->partition].misc.order;
	  if (order <= 3)	/* primary partition */
	      regs->esi = 0x600 + offsetof(bootsector_t, after.bootsect_partition[order]);
	    else {
#if 0
	      struct bootsect_partition_str bp_str = {
		  .indicator = bootable,
		  .begin_head = 0,
		  .begin_sector = 0,
		  .begin_cylinder_msb = 0,
		  .begin_cylinder_lsb = 0,
		  .system_indicator = dp->partition[bootwaydesc->partition].type,
		  .end_head = 0,
		  .end_sector = 0,
		  .end_cylinder_msb = 0,
		  .end_cylinder_lsb = 0,
		  .nb_sector_before = dp->partition[bootwaydesc->partition].start,
		  .nb_sector = dp->partition[bootwaydesc->partition].length,
		  };
#else
	      struct bootsect_partition_str bp_str;
	      memset (&bp_str, 0, sizeof (struct bootsect_partition_str));
	      bp_str.indicator = bootable,
	      bp_str.system_indicator = dp->partition[bootwaydesc->partition].type,
	      bp_str.nb_sector_before = dp->partition[bootwaydesc->partition].start,
	      bp_str.nb_sector = dp->partition[bootwaydesc->partition].length,
#endif
	      regs->esi = 0x600 + offsetof(bootsector_t, after.bootsect_partition[3]);
	      smemcpy (regs->esi, &bp_str, sizeof (bp_str));
	      }
	  if (regs->esi >= 16U * regs->ds)
	      regs->esi -= 16U * regs->ds;	/* do not modify %ds if possible */
	    else
	      regs->ds = 0;
	  }

      if (!dp->partition[bootwaydesc->partition].misc.beer_partition) {
	  /* Codepage preparing/loading strange errors if booted in anything else than mode 3: */
	  _VGA_autoload_default_palette (1);
#if (USER_SUPPORT & VGA_SUPPORT) && (USER_SUPPORT & VESA_SUPPORT)
	  if (VESA_ACTIVE())
	      VGA_init();
#endif
	  UI.function.setmode (3);
	  segment_door (0);	/* other himem (hiren CD) needs it */
	  set_A20 (0);	/* Win98 himem.sys needs it */
	  }
      }
  return 0;
  }
#endif

awk_farret (final_loadrun);
unsigned
final_loadrun (unsigned index, struct registers *regs,
		struct gujin_param_attrib gujin_attr, unsigned char edit_cmdline_b4_run)
  {
  if (   BOOTWAY.desc[index].boottype == is_linux
      || BOOTWAY.desc[index].boottype == is_elf) {
      if (menu_load_system (index, regs, edit_cmdline_b4_run) & 0x20000000)
	  return 0xA0000000; /* display "press a key to continue" ? */
	else
	  return 0x80000000; /* Error message is already displayed */
      }
#if DISK_SUPPORT & BOOTSECT_PROBE
    else
      return menu_load_other (index, regs, gujin_attr); /* complete restart if returns 0 ! */
#else
  return 4;
#endif /* DISK_SUPPORT & BOOTSECT_PROBE */
  }

#if !(DEBUG & DEBUG_DISK)
//#define INCLUDE_CLEANDISK
#endif

#ifdef INCLUDE_CLEANDISK
static inline unsigned
is_cleandisk_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0)
      return (key & 0xFF) == '\030';
    else
      return (key == CTRL_X_KEYCODE());
  }

static void do_clean_disk (void)
  {
  struct diskparam_str *dp;
  unsigned nb, freeblock;
  unsigned short key = 0;

  printf (FL_FREELIST_DISK_0_N, DI.nbdisk);
  for (dp = &DI.param[0]; dp < &DI.param[DI.nbdisk]; dp++) {
      printf (INDEX_DISK_NAME, dp - &DI.param[0], dp->diskname);
      for (freeblock = 0; freeblock < dp->nbfreelist; freeblock++)
	  printf ("%llU + %llu; ", dp->freelist[freeblock].start, dp->freelist[freeblock].length);
      printf (FL_TOTAL_DISK_SIZE, dp->nbtotalsector);
      }
  printf (FL_TYPE_DISK_INDEX);
  nb = get_number (DI.nbdisk, DI.nbdisk); /* invalid default for ESC */
  if (nb < DI.nbdisk) {
      dp = &DI.param[nb];
      printf (FL_CHECKING_CONTENT_OF_DISK, dp->diskname);
      for (freeblock = 0; freeblock < dp->nbfreelist && key != '\033'; freeblock++) {
	  unsigned long long i;
	  for (i = 0; i < dp->freelist[freeblock].length; i++) {
	      printf (FL_CHECKING_SECTOR_AT_LBA, dp->freelist[freeblock].start + i);
	      if (DI.readsector (dp, -1, dp->freelist[freeblock].start + i, 1, data_adr(fourKbuffer)) != 0) {
		  puts (FL_READ_SECTOR_FAILED);
		  break;
		  }
	      if (UI.function.getkey(0) != TIMEOUT_KEYCODE()) {
		  puts (FL_INTERRUPTED);
		  key = ' ';	/* stop CTRLENTER */
		  }
		else {
		  unsigned short j;

		  for (j = 0; j < dp->bytepersector / sizeof (unsigned); j++)
		      if (((unsigned *)fourKbuffer)[j] != 0)
			  break;
		  if (j >= dp->bytepersector / sizeof (unsigned))
		      continue;
		  if (key == CTRLENTER)
		      puts (FL_AUTOCLEAR);
		    else
		      puts (FL_NOT_EMPTY);
		  }
	      for (;;) {
		  if (key != CTRLENTER) {
		      puts (FL_TYPE_SPACE_ENTER_TAB);
		      key = UI.function.getkey(30 * TICKS_PER_SECOND);
		      if (key == CTRLENTER) {
			  unsigned short key2;
			  print (FL_CONFIRM_CLEAR_ALL);
			  key2 = UI.function.getkey(30 * TICKS_PER_SECOND) & 0xFF;
			  if (key2 != 'y' && key2 != 'Y') {
			      key = ' ';
			      continue;
			      }
			  }
			else
			  key &= 0xFF;
		      }
		  if (key == ' ') {
		      unsigned short j = 0;
		      if (DI.readsector (dp, -1, dp->freelist[freeblock].start + i, 1, data_adr(fourKbuffer)) != 0) {
			  printf (FL_REREAD_DISK_FAILED, dp->diskname, dp->freelist[freeblock].start + i);
			  break;
			  }
		      while (j < dp->bytepersector) {
			  extern const char *const itoa_array;
			  const unsigned strlen_pattern = sizeof ("0x000: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  ................\r\n") - 1;
			  /* 4 lines at a time for scrolling speed: */
			  char buf[4 * strlen_pattern + 1], *endbuf = &buf[4 * strlen_pattern], *ptr = buf;

			  while (ptr < endbuf) {
			      unsigned k;
			      *ptr++ = '0';
			      *ptr++ = 'x';
			      *ptr++ = itoa_array[j / 256];
			      *ptr++ = itoa_array[(j / 16) % 16];
			      *ptr++ = '0';
			      *ptr++ = ':';
			      for (k = 0; k < 16; k++) {
				  unsigned char c = fourKbuffer[j+k];
				  *ptr++ = ' ';
				  *ptr++ = itoa_array[c / 16];
				  *ptr++ = itoa_array[c % 16];
				  }
			      *ptr++ = ' ';
			      *ptr++ = ' ';
			      for (k = 0; k < 16; k++) {
				  unsigned char c = fourKbuffer[j++];
				  if (c >= ' ' && c <= '~')
				      *ptr++ = c;
				     else
				      *ptr++ = '.';
				  }
			      *ptr++ = '\r';
			      *ptr++ = '\n';
			      }
			  *ptr = '\0';
			  print (buf);
			  }
		      }
		    else if (key == '\r' || key == '\n' || key == CTRLENTER) {
		      unsigned short j;
		      for (j = 0; j < dp->bytepersector / sizeof (unsigned); j++)
			  ((unsigned *)fourKbuffer)[j] = 0;
		      if (DI.writesector (dp, -1, dp->freelist[freeblock].start + i, 1, data_adr(fourKbuffer)) != 0) {
			  printf (FL_WRITING_DISK_FAILED, dp->diskname, dp->freelist[freeblock].start + i);
			  break;
			  }
			else if (key == CTRLENTER)
			  break;
		      }
		    else
		      break;
		  } /* for (;;) */
	      if (key != '\t' && key != CTRLENTER) {
		  printf (FL_SKIP_TO_NEXT_BLOCK, freeblock);
		  break;
		  }
	      }
	  }
      printf (FL_FINISHED_TREATMENT_DISK, dp->diskname);
      }
  }
#endif /* INCLUDE_CLEANDISK */

//#define INCLUDE_INSTALL
#ifdef INCLUDE_INSTALL
static inline unsigned
is_install_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0)
      return (key & 0xFF) == '\020';
    else
      return (key == CTRL_P_KEYCODE());
  }

static void recal_checksum (char *file,
	instboot_info_t *instboot_info,
	boot1param_t *boot1param,
	bootloader2_t *bootchain)
  {
  unsigned short *ptr, tmp_checksum;
  signed index = 0;
  unsigned cpt;

//  if (Bytepersector == 512)
      ptr = (unsigned short *)(file + 1024);
//    else
//      ptr = (unsigned short *)(file + Bytepersector);

  do
      ptr += bootchain[index].header.nbword;
      while (bootchain[index++].header.next != 0);

  while (--index >= 0) {
      bootchain[index].header.checksum = 0;
      cpt = bootchain[index].header.nbword + 1;
      while (--cpt)
	  bootchain[index].header.checksum -= *--ptr;
      }

  boot1param->bootloader2.header.checksum = 0;
  cpt = boot1param->bootloader2.header.nbword;
  while (cpt --)
      boot1param->bootloader2.header.checksum -= *--ptr;

  ptr = (unsigned short *)(file + instboot_info->checksum_start_adr);
  tmp_checksum = 0;
  boot1param->boot1_checksum = 0;	/* skip the actual checksum by substracting 0 */
  while (ptr < (unsigned short *)(file + instboot_info->bootend_adr))
      tmp_checksum -= *ptr++;
  boot1param->boot1_checksum = tmp_checksum;
  }

/* TODO: add set_bootchain() and chs_chain() from instboot.c - that will not fit! */
static void do_intall_gujin (struct registers *regs)
  {
  struct diskparam_str *dp;

  printf (IN_LIST_DISK_0_N, DI.nbdisk);
  for (dp = &DI.param[0]; dp < &DI.param[DI.nbdisk]; dp++) {
      unsigned last_partition_used = 0, last_partition_used_extended = 0;
      unsigned i, beer_index = 0xFFFFFFFF;	/* not related to dp->BEER_device_index */

      printf (INDEX_DISK_NAME, dp - &DI.param[0], dp->diskname);
      if (dp->bytepersector == 0) {
	    puts ("drive without media");
	    continue;
	    }

      for (i = 0; i < dp->nbpartition; i++) {
	  unsigned endpart = dp->partition[i].start + dp->partition[i].length;

	  if (dp->partition[i].misc.beer_partition) {
		beer_index = i;
		continue;
		}
	  if (dp->partition[i].misc.active == part_extended) {
	      if (endpart > dp->partition[last_partition_used_extended].start + dp->partition[last_partition_used_extended].length)
		  last_partition_used_extended = i;
	      }
	    else {
	      if (endpart > dp->partition[last_partition_used].start + dp->partition[last_partition_used].length)
		  last_partition_used = i;
	      }
	  }

      printf ("\r\n disk size %llU, sector size %u, fulldisk_size %llU, fulldisk_offset %llU,\r\n"
		" BEER partition index %u, last extended %llU, last sector used %llU\r\n",
		dp->nbtotalsector, dp->bytepersector, dp->fulldisk_size, dp->fulldisk_offset, beer_index,
		dp->partition[last_partition_used_extended].start + dp->partition[last_partition_used_extended].length,
		dp->partition[last_partition_used].start + dp->partition[last_partition_used].length);
      if (dp->BEER_size)
	  printf ("BEER_size %U, BEER_sector %llU, BEER_HostProtectedArea_start %llU,\r\n"
		" BEER_ReservedAreaBootCodeAddress %ll, BEER_disksize %llU\r\n",
		dp->BEER_size, dp->BEER_sector, dp->BEER_HostProtectedArea_start,
		dp->BEER_ReservedAreaBootCodeAddress, dp->BEER_disksize);

      if (DI.readsector (dp, -1, 0, sizeof (fourKbuffer)/dp->bytepersector, data_adr(fourKbuffer)) != 0)
	  printf ("Read first %u sectors of MBR failed\r\n", sizeof (fourKbuffer)/dp->bytepersector);
	else if (((bootbefore_t *)fourKbuffer)->part1.String[0] == 'G') {
	  printf ("MBR: ");
	  get_remote_uninstall_lba (dp, (bootbefore_t *)fourKbuffer);
	  }

      if (dp->ide_attribute.config_overlay && dp->config_max_lba == 0)
	  printf ("IDE config_overlay unaccessible, cannot install (try full power cycle)\r\n");
	else if (dp->ide_attribute.host_protected_area && dp->config_max_lba != 0 && dp->nbtotalsector != dp->config_max_lba + 1)
	  printf ("HPA and dp->nbtotalsector %llU, dp->config_max_lba %llU, HPA already defined\r\n", dp->nbtotalsector, dp->config_max_lba);
	else if (beer_index != 0xFFFFFFFF) {
#define BEER_MIN_SIZE	(200 * 1024)
#define BEER_MAX_LENGTH	(64 * 1024 * 1024)
	  if (dp->partition[beer_index].length < BEER_MIN_SIZE / dp->bytepersector)
	      printf ("B.E.E.R. partition too small: only %llu sectors", dp->partition[beer_index].length);
	    else {
	      printf ("install on B.E.E.R. partition index %u: %llu+%llu, structure at %llU\r\n",
			beer_index, dp->partition[beer_index].start, dp->partition[beer_index].length, dp->BEER_sector);
	      if (DI.readsector (dp, beer_index, 0, sizeof (fourKbuffer)/dp->bytepersector, data_adr(fourKbuffer)) != 0)
		  printf ("Read first %u sectors of BEER partition failed\r\n", sizeof (fourKbuffer)/dp->bytepersector);
		else {
		  printf ("B.E.E.R. PBR: ");
		  get_remote_uninstall_lba (dp, (bootbefore_t *)fourKbuffer);
		  }
	      }
	  }
	else if (dp->nbfreelist == 0 || dp->nbpartition == 0)
	  puts ("no free space at all, no predefined B.E.E.R.");
	/* Note that freelist has to be in increasing order to be calculated anyway: */
	else if (dp->freelist[dp->nbfreelist - 1].start < dp->partition[last_partition_used].start
					+ dp->partition[last_partition_used].length)
	  puts ("no free space at end of the disk for B.E.E.R.");
	else if (dp->freelist[dp->nbfreelist - 1].length < BEER_MIN_SIZE / dp->bytepersector)
	  printf ("not enough free space at end of the disk for B.E.E.R., only %u sectors\r\n", dp->freelist[dp->nbfreelist - 1].length);
	else {
	  unsigned long long BEER_start, BEER_length, BEER_struct;
	  unsigned short nb_sector_per_4Kbytes = 4096 / dp->bytepersector;

	  BEER_start = dp->freelist[dp->nbfreelist - 1].start;
	  BEER_length = dp->freelist[dp->nbfreelist - 1].length;
	  if (BEER_length > BEER_MAX_LENGTH / dp->bytepersector) {
	      BEER_start += BEER_length - BEER_MAX_LENGTH / dp->bytepersector;
	      BEER_length = BEER_MAX_LENGTH / dp->bytepersector;
	      }
	  /* We want start & size multiple of 4 Kbytes, and keep few sectors at end for B.E.E.R. structure */
//	  while (BEER_start % nb_sector_per_4Kbytes != 0) {
	  while (BEER_start & (nb_sector_per_4Kbytes - 1)) {
		BEER_start++;
		BEER_length--;
		}
	  /* last sector could be used for testing purpose by the BIOS: */
	  BEER_length --;
	  do /* at least one sector for BEER_struct */
	      BEER_length --;
//	      while (BEER_length % nb_sector_per_4Kbytes != 0);
	      while (BEER_length & (nb_sector_per_4Kbytes - 1));
	  BEER_struct = BEER_start + BEER_length;
	  BEER_length -= nb_sector_per_4Kbytes;
	  printf ("Create B.E.E.R. partition at %llU length %llu, structure at %llU\r\n", BEER_start, BEER_length, BEER_struct);
	  }
      puts ("");
      print (PRESS_KEY_TO_CONTINUE);
      UI.function.getkey (30 * TICKS_PER_SECOND);
      }
#if 0
  /* Where is old uninstall sector & safe MBR sector for this disk? */
  printf (IN_TYPE_DISK_INDEX);
  unsigned nb = get_number (DI.nbdisk, DI.nbdisk); /* invalid default for ESC */
  if (nb < DI.nbdisk) {
      dp = &DI.param[nb];
      printf (FL_CHECKING_CONTENT_OF_DISK, dp->diskname);
      printf (FL_FINISHED_TREATMENT_DISK, dp->diskname);

      /* DO NOT FORGET TO REPROBE DISKS */
#ifdef FS_USE_MALLOC
      if (BOOTWAY.desc != 0) {
	  unsigned cptdesc;
	  for (cptdesc = 0; cptdesc < BOOTWAY.nb + BOOTWAY.nb_initrd; cptdesc++)
	      if (BOOTWAY.desc[cptdesc].iso_fsname && BOOTWAY.desc[cptdesc].variant_number == 0)
		  FREE (BOOTWAY.desc[cptdesc].iso_fsname);
	  FREE (BOOTWAY.desc);
	  BOOTWAY.desc = 0;
	  }
#endif
      probedisk ();
      disk_analyse ();
      }
#endif
  }

#endif /* INCLUDE_INSTALL */

#if (DISK_SUPPORT & IDE_SUPPORT) && (USER_SUPPORT != 0)
#define INCLUDE_IDEPW
#endif

#ifdef INCLUDE_IDEPW
/* Really really big warning: Do not use this feature.
   Do not use it before reading the ATA specification.
   Do never program a user password without specifying
   a master password - It is impossible (or say me how)
   to get the default master password from some disk
   manufacturers, if you fail to program/remember the
   user password.
   (Few disk manufacturers are good enough to program
    an empty (all zero) or blanks (all spaces) default
    master password - i.e. when the revision is 0xFFFE,
    but some do not _care_ to help their costumers).
   Take care: unlike what you are waiting for, a disk protected
   by a password is not protected at all until the disk is
   frozen: you do not need to know the previous password
   when you change the password (before power off).
   If you need to security erase an hard disk and have the
   master password - or analyse the SMART stuff, you may
   be interrested by a package called atapwd.zip, search it
   on Internet; see also newer hdparm parameters.
   OBVIOUSLY you cannot boot off an hard disk protected by
   password, you will not be able to read the MBR.
     Repeat after me:
   Do not use a CNP navigation program on networked computers.
   */
static void do_ide_password (unsigned unlock)
  {
  unsigned nb;
  if (UTIL.keyboard != serialkbd && !kbdmap_is_precise(copy_gujin_param.kbdmap)) {
      const char *msg = (copy_gujin_param.kbdmap == kbd_unknown)? PW_KEYBOARD_UNKNOWN : PW_KEYBOARD_PARTIALLY_KNOWN;
      puts (msg);
      }
    else {
      unsigned nb_possible_disk = 0;
      for (nb = 0; nb < DI.nbdisk; nb ++)
	  if (DI.param[nb].access == hardide_chs || DI.param[nb].access == hardide_lba || DI.param[nb].access == hardide_lba48) {
	      printf (INDEX_DISK_NAME, nb, DI.param[nb].diskname);
	      nb_possible_disk++;
	      }
      if (nb_possible_disk == 0) {
	  puts ("No useable disk detected, probe_ide enabled?");
	  // for EBIOS, would have to init dp->ideIOctrladr, dp->ideIOadr,
	  // dp->security_status.frozen, dp->ide_master_password_revision
	  return;
	  }
      printf (PW_TYPE_DISK_INDEX, unlock? PW_UNLOCK : PW_LOCK);
      nb = get_number (DI.nbdisk, DI.nbdisk); /* invalid default for ESC */
      if (nb < DI.nbdisk && DI.param[nb].security_status.frozen)
	  printf (PW_DRIVE_FROZEN, DI.param[nb].diskname);
	else if (nb < DI.nbdisk) {
	  struct diskparam_str *dp = &DI.param[nb];
	  char password[33], retype[33];	/* get_line set a '\0', ideSecurity just needs an array */
	  unsigned ret, tmp;
	  enum IDE_security_enum cmd = unlock? IDE_security_disable_user : IDE_security_set_pw_user;
	  const char *msg = unlock? PW_UNLOCKING : PW_LOCKING;

	  DDBG (("Security: supported %u, enabled %u, locked %u, frozen %u, count_expired %u, enhanced_erase %u, high/max level %u\r\n",
		dp->security_status.supported, dp->security_status.enabled, dp->security_status.locked,
		dp->security_status.frozen, dp->security_status.count_expired, dp->security_status.enhanced_erase,
		dp->security_status.high_level));

	  printf (PW_TREATING_DRIVE, msg, dp->diskname);
	  for (;;) {
	      print (TYPE_PASSWORD_OR_ESC);
	      password[0] = '\0';
	      ret = get_line (password, sizeof (password), '*', ' ', '~');
	      if (ret == 0 && STREQL(password, "MASTER PASSWORD")) {
		  printf (WARNING_MASTER_PASSWORD_X, dp->ide_master_password_revision);
		  cmd = unlock? IDE_security_disable_master : IDE_security_set_pw_master;
		  continue;
		  }
	      tmp = 0;
	      while (password[tmp] != 0)
		  tmp++;
	      while (tmp < sizeof (password))
		  password[tmp++] = '\0';
	      break;
	      }
	  if (ret == 0) {
	      print (CONFIRM_PASSWORD);
	      print (TYPE_PASSWORD_OR_ESC);
	      retype[0] = '\0';
	      ret = get_line (retype, sizeof (retype), '*', ' ', '~');
	      tmp = 0;
	      while (retype[tmp] != 0)
		  tmp++;
	      while (tmp < sizeof (retype))
		  retype[tmp++] = '\0';
	      if (ret != 0 || !memeql(retype, password, sizeof (retype)))
		  puts (MSG_COLON_FAILED);
		else {
		  ret = DI.ideSecurity (dp, password, cmd);
		  if (ret == 0 && cmd == IDE_security_set_pw_master)
		      puts (PW_MASTER_SUCCESS);
		    else if (ret == 0)
		      puts (PW_PLEASE_POWER_OFF);
		    else
		      printf (MSG_COLON_FAILED_ERROR, ret);
		  }
	      }
	    else
	      puts (MSG_ABORTED);
	  }
      }
  }
#endif

static inline unsigned
is_nextlang_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0)
      return (key & 0xFF) == '\024';
    else
      return (key == CTRL_T_KEYCODE());
  }

static inline unsigned
is_prevlang_keycode (unsigned short key)
  {
  if (BOOT1_COM_port() >= 0)
      return 0;
    else
      return (key == ALT_T_KEYCODE());
  }

static void reparse_disks(void)
  {
#ifdef FS_USE_MALLOC
  /* To not create too many holes in our malloc-able memory - with our simple
    malloc management, we release the memory malloced by disk_analyse()
    before the call to probedisk() : */
  if (BOOTWAY.desc != 0) {
      unsigned cptdesc;
      for (cptdesc = 0; cptdesc < BOOTWAY.nb + BOOTWAY.nb_initrd; cptdesc++)
	  if (BOOTWAY.desc[cptdesc].iso_fsname && BOOTWAY.desc[cptdesc].variant_number == 0)
	      FREE (BOOTWAY.desc[cptdesc].iso_fsname);
      BOOTWAY.nb = BOOTWAY.nb_initrd = BOOTWAY.nb_bdi = BOOTWAY.nb_iso = 0;
      FREE (BOOTWAY.desc);
      BOOTWAY.desc = 0;
      }
#endif
  /* That memory is allocated before disk probing, will not create holes...
  if (UI.nbmode != 0) {
      struct videomode_str tmp[UI.nbmode];
      _memcpy (tmp, UI.mode, UI.nbmode * sizeof (struct videomode_str));
      FREE (UI.mode);
      UI.mode = MALLOC (UI.nbmode * sizeof (struct videomode_str), "VIDmode");
      _memcpy (UI.mode, tmp, UI.nbmode * sizeof (struct videomode_str));
      }
  DBG (("MOUSE.fields = 0x%X, UI.mode = 0x%X\r\n", MOUSE.fields, UI.mode));
  */
  probedisk ();
  disk_analyse ();
  }

static unsigned
menu (struct registers *regs, struct gujin_param_attrib gujin_attr)
  {bound_stack();{
  unsigned refresh_freq = refresh_freq; /* inited B4 used */
  unsigned timeout = copy_gujin_param.timeout_autoload;
  struct autoload_str autoload = copy_gujin_param.autoload;
#if SETUP & UPDATE_BOOTPARAM
  struct autoload_str initial_autoload = copy_gujin_param.autoload;
  unsigned char initial_timeout_autoload = copy_gujin_param.timeout_autoload;
#endif
  /* toredraw.mainfield has to be set to remove the "whole screen"
     "press a key to continue" at startup: */
  struct todraw_str toredraw = {
	.clearbackground	= 0,
	.firstlines	= 1,
	.topmenu	= 1,
	.bottommenu	= 1,
	.advertisement	= 1,
	.copyright	= 1,
	.update_bootparam = 1, /* to save the initial mode, also we no more do it in treat_startup_keys() */
	.refresh = 1,
	.reparse = 0,
	.presskey = 0
	};
  unsigned short advline, quickboot = timeout;
  char local_scanpath[sizeof (copy_gujin_param.scanpath)];
  strcpy (local_scanpath, copy_gujin_param.scanpath);

  /* First check for "quickboot" conditions: */
  if (gujin_attr.verbose || gujin_attr.disk_write_enable)
      quickboot = 0;
  if (BOOTWAY.nb == 0 || BOOTWAY.nb > 2)
      quickboot = 0;
    else if (BOOTWAY.nb == 2) {
      if (BOOTWAY.desc[0].inode != BOOTWAY.desc[1].inode)
	  quickboot = 0;
      if (BOOTWAY.desc[0].filesize != BOOTWAY.desc[1].filesize)
	  quickboot = 0;
      if (BOOTWAY.desc[0].partition != BOOTWAY.desc[1].partition)
	  quickboot = 0;
      if (DI.param[BOOTWAY.desc[0].disk].access == DI.param[BOOTWAY.desc[1].disk].access)
	  quickboot = 0;
      }

  if (quickboot && !_BIOS_checkkey()) {
      if (BOOTWAY.desc[0].boottype != is_MBR && BOOTWAY.desc[0].boottype != is_PBR
		&& BOOTWAY.desc[0].boottype != is_el_torito)
	  set_initial_mode (gujin_attr, 0);

      /* If "quick autoload" when only one single MBR/ElTorito exists, do not even initialise
		the screen, just boot that (see --quickboot=) : */
      if (   autoload.last_loaded == 0 && autoload.total_loadable == 1
	  && autoload.init_page == 0 && autoload.total_page == 1) {
	  unsigned char timeout_autoload = quickboot;
	  unsigned error_loadrun;

	  DBG (("Autoload conditions detected - wait %u second before\r\n",
		timeout_autoload - 1));
	  if (timeout_autoload > 1)
	      _BIOS_wait (1000000 * (timeout_autoload - 1));
	  LOADER.edit_cmdline_b4_run = 0;
	  if ((error_loadrun = final_loadrun (0, regs, gujin_attr, 0)) == 0)
	      return 0;
	  if ((error_loadrun & 0x80000000) == 0 && (error_loadrun != 2))
	      printf (ERROR_LOADING_FILE, error_loadrun);
	  }
      }
    else
      set_initial_mode (gujin_attr, 1);
  /* set_initial_mode() can have reseted copy_gujin_param.attrib.VESA2_interface (VirtualBox) */
  gujin_attr.VESA2_interface = copy_gujin_param.attrib.VESA2_interface;

  Menu.nbtotal = BOOTWAY.nb;
  Menu.nbperpage = 0;
  Menu.type = kernel_bottom_menu;

  if (timeout == 0) {
      struct desc_str *p;
      unsigned cmdfile_timeout = 0xFFFFFFFFU, cmdfile_toload = 0xFFFFFFFFU;
      for (p = &BOOTWAY.desc[0]; p < &BOOTWAY.desc[BOOTWAY.nb]; p++)
	  if (p->variant_timeout && p->variant_timeout < cmdfile_timeout) {
	      cmdfile_timeout = p->variant_timeout;
	      cmdfile_toload = p - &BOOTWAY.desc[0];
	      }
      if (cmdfile_toload != 0xFFFFFFFFU) {
	  autoload.last_loaded = cmdfile_toload;
	  autoload.total_loadable = BOOTWAY.nb;
	  autoload.total_page = 0;	/* still unknown */
	  timeout = cmdfile_timeout;
	  }
      }

  if (timeout != 0 && BOOTWAY.nb > 0) {
      if (autoload.last_loaded == 0xFF) {
	  /* load the newest kernel if timeout >= 2s && no previous kernel selected and saved */
	  /* Enter this mode by configuring autoload timeout but disable writing before the first reboot */
	  unsigned short cpt, newest = 0;
	  for (cpt = 0; cpt < BOOTWAY.nb; cpt++)
	      if (BOOTWAY.desc[cpt].last_modification_date > BOOTWAY.desc[newest].last_modification_date)
		  newest = cpt;
	  autoload.last_loaded = newest;
	  autoload.total_loadable = BOOTWAY.nb;
	  autoload.total_page = 0;	/* still unknown */
	  }
	else if (autoload.total_loadable != BOOTWAY.nb) {
	  /* If the last kernel loaded has been saved, but there are a
		different number of kernel loadable, cancel autoload: */
	  timeout = 0;
	  }
      }

  for (;;) {
      unsigned short key;

      if (toredraw.refresh == 1) {
#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
	  refresh_freq = get_refresh_freq ();
#endif
	  toredraw.refresh = 0; /* refresh_freq updated */
	  }

      if (toredraw.update_bootparam) {
#if SETUP & UPDATE_BOOTPARAM
	  struct autoload_str save_autoload = copy_gujin_param.autoload;
	  /* If we are disabling writing and have autoload configured and used once, we
		should autoload something valid and not something in the last kernel page displayed: */
	  if (!copy_gujin_param.attrib.disk_write_enable
		&& !copy_gujin_param.attrib.write_disabled_written_once
		&& copy_gujin_param.timeout_autoload != 0
		&& initial_autoload.last_loaded != 0xFF
		&& initial_timeout_autoload != 0)
	      copy_gujin_param.autoload = initial_autoload;
	  /* Will update UTIL.boot_device_read_only if on a read-only CDROM or floppy to manage hidden partitions: */
	  BOOT1_update_bootparam ();
	  //if (UTIL.boot_device_read_only) { printf ("boot_device_read_only %u ", UTIL.boot_device_read_only); _BIOS_getkey(); }
	  copy_gujin_param.autoload = save_autoload;
#endif
	  toredraw.update_bootparam = 0;
	  }

      /* This may display few lines of warning from the disk/partition analysis... */
      if (toredraw.reparse) {
	  reparse_disks ();
	  Menu.nbtotal = BOOTWAY.nb;
	  Menu.nbperpage = 0; /* recalculate nbpage */
	  /* if scroll / copyright line partly overwritten / line before copyright background modified: */
	  unsigned char row, col;
	  UI.function.getcursor (&row, &col); /* ignore errors */
	  if (row >= UI.parameter.nbrow - 2)
	      toredraw =  (struct todraw_str) {
		.clearbackground	= 1,
		.firstlines	= 1,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam	= 0,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 0
		};
	  }

      advline = redraw_screen (toredraw, refresh_freq, timeout, &autoload);
      toredraw =  (struct todraw_str) {
	.clearbackground	= 0,
	.firstlines	= 0,
	.topmenu	= 0,
	.bottommenu	= 0,
	.advertisement	= 0,
	.copyright	= 0,
	.update_bootparam	= 0,
	.refresh	= 0,
	.reparse	= 0,
	.presskey	= 0
	};

      /* GCC-4.7 does "mov $UI.function.getkey,%edx; call *%edx if not: */
      unsigned this_timeout = (timeout != 0)? TICKS_PER_SECOND : 0xFFFFFFFFU;
      key = UI.function.getkey (this_timeout);
      if (timeout != 0 && autoload.last_loaded != 0xFF) {
	  if (key != TIMEOUT_KEYCODE() || MOUSE_has_moved()) {
	      /* Cancel autoboot & redraw advert: */
	      timeout = 0;
	      toredraw.advertisement = 1;
	      continue;
	      }
	    else if (--timeout != 0) {
	      /* No, the answer is an orange and two lemons. */
	      /* If I have three lemons and three oranges and
		 I lose two oranges and a lemon what do I have left? */
	      /* OK, so you think that time flows _that_ way, do you? */
	      toredraw.advertisement = 1;
	      continue;
	      }
	  }

      /*
       * Choices which will not fail, so will not
       * display any messages in the "advert" window:
       * We no more consider PageUp/PageDown as error
       * when at first/last page, more user friendly.
       */
      if (is_Redraw_keycode(key)) {
	  toredraw =  (struct todraw_str) {
		.clearbackground	= 1,
		.firstlines	= 1,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam	= 0,
		.refresh	= 1,
		.reparse	= 0,
		.presskey	= 0
		};
	  continue;
	  }
#if SETUP & MULTILINGUAL
	else if (is_nextlang_keycode (key)) {
	  MltStrLanguage = (MltStrLanguage + 1) % NB_TOTAL_LANGUAGE;
	  copy_gujin_param.MltStrLanguage = MltStrLanguage;
	  toredraw =  (struct todraw_str) {
		.clearbackground	= 1,
		.firstlines	= 1,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam	= 1,
		.refresh	= 1,
		.reparse	= 0,
		.presskey	= 0
		};
	  continue;
	  }
	else if (is_prevlang_keycode (key)) {
	  MltStrLanguage = (MltStrLanguage + NB_TOTAL_LANGUAGE - 1) % NB_TOTAL_LANGUAGE;
	  copy_gujin_param.MltStrLanguage = MltStrLanguage;
	  toredraw =  (struct todraw_str) {
		.clearbackground	= 1,
		.firstlines	= 1,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam	= 1,
		.refresh	= 1,
		.reparse	= 0,
		.presskey	= 0
		};
	  continue;
	  }
#endif
	else if (is_PgUp_keycode (key) /* && Menu.curpage != 0 */ ) {
	  if (Menu.curpage != 0) {
	      Menu.curpage -= 1;
	      toredraw.bottommenu = 1;
	      }
	  continue;
	  }
	else if (is_PgDn_keycode (key) /* && Menu.curpage < Menu.totalpage - 1 */) {
	  if (Menu.curpage < Menu.totalpage - 1) {
	      Menu.curpage += 1;
	      toredraw.bottommenu = 1;
	      }
	  continue;
	  }
	else if (is_Home_keycode (key)) {
	  if (Menu.curpage != 0) {
	      Menu.curpage = 0;
	      toredraw.bottommenu = 1;
	      }
	  continue;
	  }
	else if (is_End_keycode (key)) {
	  if (Menu.curpage < Menu.totalpage - 1) {
	      Menu.curpage = Menu.totalpage - 1;
	      toredraw.bottommenu = 1;
	      }
	  continue;
	  }

      if (is_valid_chgmenu_keycode (key) && Menu.type == kernel_bottom_menu) {
	  /* to come back to the same page at next switch, save into autoload: */
	  copy_gujin_param.autoload.last_loaded = 0xFF;
	  copy_gujin_param.autoload.total_loadable = BOOTWAY.nb;
	  copy_gujin_param.autoload.init_page = Menu.curpage;
	  copy_gujin_param.autoload.total_page = Menu.totalpage;
	  Menu.type = setup_bottom_menu;
	  Menu.nbtotal = redraw_setup_field (0, 0, 0);
	  Menu.nbperpage = 0; /* recalculate nbpage */
	  // I do not remember nor understand why I have put .update_bootparam = 1 here...
	  // ... Now I remember, it is to come back to the same page at next power-on...
	  toredraw = (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 0,
		.copyright	= 0,
		.update_bootparam	= 1,
		.refresh	= 0,
		.reparse	= 0,
		.presskey	= 0
		};
	  continue;
	  }

      if (UI.parameter.nbcol >= ADVERTISEMENT_BANNER_MINWIDTH) {
	  unsigned short row = UI.parameter.nbrow
				- ADVERTISEMENT_BANNER_NBLINES - 2;
	  draw_bg_box (row, ADVERTISEMENT_BANNER_NBLINES,
				WORK_ADVERT_BACKGROUND);
	  toredraw.advertisement = 1;
	  UI.function.setcursor (row, 5);
	  }
	else
	  UI.function.setcursor (UI.parameter.nbrow - 2, 0);

      /*
       * Choices which can fail / display in advert banner / treat toredraw.presskey:
       */
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
      MOUSE_reset_field (reset_all_fields);
#endif
      unsigned short tmp_fctkeynum = 0;

      if (is_valid_chgmode_keycode (key)) {
	  if (change_video_mode (key & 0xFF, 1, gujin_attr) == 0) {
	      Menu.nbperpage = 0; /* recalculate nbpage */
	      toredraw = (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 1,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam	= 1,
		.refresh	= 1,
		.reparse	= 0,
		.presskey	= 0
		};
	      }
	    else {
	      toredraw.topmenu = 1; /* over monitor spec */
	      toredraw.presskey = 1;
	      }
	  }
	else if ((Menu.type == kernel_bottom_menu)? is_reprobe_keycode (key)
						: is_valid_chgmenu_keycode (key)) {
	  /* We want to compare the local copy inside this function "gujin_attr" to the value of the real structure copy_gujin_param.attrib: */
	  unsigned char reparse_disks_bit = strcmp(local_scanpath, copy_gujin_param.scanpath);
	  strcpy (local_scanpath, copy_gujin_param.scanpath);

	  if (Menu.type == kernel_bottom_menu || diskconfig_changed (gujin_attr, copy_gujin_param.attrib)
		|| searchconfig_changed (gujin_attr, copy_gujin_param.attrib))
	      reparse_disks_bit = 1;
	    else {
	      Menu.nbtotal = BOOTWAY.nb;
	      Menu.nbperpage = 0; /* recalculate nbpage, use autoload to go to last active page */
	      autoload = copy_gujin_param.autoload;
	      }

	  gujin_attr = copy_gujin_param.attrib;
	  Menu.type = kernel_bottom_menu;
	  toredraw = (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines	= 0,
		.topmenu	= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 0,
		.update_bootparam	= 0,
		.refresh	= 0,
		.reparse	= reparse_disks_bit,
		.presskey	= 0
		};
	  }
	else if (   key == TIMEOUT_KEYCODE()
		 || (tmp_fctkeynum = is_valid_fct_keycode (key, MIN (Menu.nbperpage,
			Menu.nbtotal - (Menu.curpage * Menu.nbperpage) ))) != 0) {
	  unsigned index;
	  if (key != TIMEOUT_KEYCODE()) {
	      index = Menu.curpage * Menu.nbperpage + (tmp_fctkeynum & 0xFF) - 1;
	      }
	    else
	      index = autoload.last_loaded;
	  unsigned char edit_cmdline_b4_run = tmp_fctkeynum >> 8;

	  if (Menu.type == kernel_bottom_menu && index < BOOTWAY.nb) {
	      unsigned error_loadrun;
	      if (key != TIMEOUT_KEYCODE()) {
		  copy_gujin_param.autoload.last_loaded = index;
		  copy_gujin_param.autoload.total_loadable = BOOTWAY.nb;
		  copy_gujin_param.autoload.init_page = Menu.curpage;
		  copy_gujin_param.autoload.total_page = Menu.totalpage;
#if SETUP & UPDATE_BOOTPARAM
		  /* probably not usefull, but do it conditionnally: */
		  BOOT1_update_bootparam();
#endif
		  }
	      if ((error_loadrun = final_loadrun (index, regs, gujin_attr, edit_cmdline_b4_run)) == 0)
		  return 0;
	      if ((error_loadrun & 0x80000000) == 0) {
		  if (error_loadrun != 2)
		      printf (ERROR_LOADING_FILE, error_loadrun);
		  if (!gujin_attr.verbose) {
		      print (MSG_ERROR);
		      print (SMENU_verbose);
		      _BIOS_wait (1500000);
		      }
		  }
	      toredraw = (struct todraw_str) {
		.clearbackground	= 0,
		.firstlines		= 1,
		.topmenu		= 1,
		.bottommenu	= 1,
		.advertisement	= 1,
		.copyright	= 1,
		.update_bootparam	= 1,
		.refresh		= 0,
		.reparse		= !!(error_loadrun & 0x10000000),
		.presskey	= !(error_loadrun & 0x20000000)
		};
	      }
	    else if (index < (unsigned)redraw_setup_field (0, 0, 0))
	      toredraw = activate_setup_field (index, advline);
	  }
	else if (is_valid_ONOFF_keycode (key)) {
	  APM_power_OFF();
	  /* if failed: */
	  toredraw = (struct todraw_str) {0,1,1,1,1,1,0, .presskey = 1};
	  }
	else if (is_valid_failedboot (key) && !BOOT1_DOS_running()) {
	  before_exit();
	  _BIOS_failedboot();
	  toredraw = (struct todraw_str) {0,1,1,1,1,1,0, .presskey = 1};
	  }
#ifdef COMMANDFILE
	else if ((key & 0xFF) == 'C') {
	  UI.function.clearscreen();
	  printf ("Memory: %d bytes free, cmdfile_base 0x%X, cmdfile_len %u:\r\n", freemem(), UTIL.cmdfile_base, UTIL.cmdfile_len);
	  unsigned ccpt;
	  for (ccpt = 0; ccpt < UTIL.cmdfile_len; ccpt++)
	      if (UTIL.cmdfile_base[ccpt] == '\n')
		  printf ("\\n\r\n");
		else if (UTIL.cmdfile_base[ccpt] < ' ')
		  printf ("\\x%X", UTIL.cmdfile_base[ccpt]);
		else
		  printf ("%c", UTIL.cmdfile_base[ccpt]);
	  toredraw = (struct todraw_str) {1,1,1,1,1,1,0, .presskey = 1};
	  }
#endif
#ifdef INCLUDE_CLEANDISK
	else if (is_cleandisk_keycode(key)) {
	  do_clean_disk ();
	  toredraw = (struct todraw_str) {1,1,1,1,1,1,0, .presskey = 1};
	  }
#endif
#ifdef INCLUDE_INSTALL
	else if (is_install_keycode(key)) {
	  do_intall_gujin (regs);
	  toredraw = (struct todraw_str) {1,1,1,1,1,1,0, .presskey = 1};
	  }
#endif
#ifdef INCLUDE_IDEPW
	else if (   gujin_attr.verbose
		 && Menu.type == setup_bottom_menu
		 && ((key & 0xFF) == '}' || (key & 0xFF) == '{')
		 ) {
	  do_ide_password ((key & 0xFF) == '{');
	  toredraw = (struct todraw_str) {0,1,1,1,1,1,0, .presskey = 1};
	  }
#endif
#if 0
	else if ((key & 0xFF) == 'M') {
	  printf ("Memory: freemem %d bytes, usedstack %u bytes, array_size %u, quantum %u\r\n",
		freemem(), usedstack(), UTIL.memalloc.array_size, UTIL.memalloc.quantum);
	  unsigned i;
	  for (i = 0; i < nbof (UTIL.memalloc.malloc_array); i++)
	      if (UTIL.memalloc.malloc_array[i].size > 0)
#ifdef DBG_MALLOC
		  printf ("block %u allocated for %s size %u, ", i, UTIL.memalloc_content[i], UTIL.memalloc.malloc_array[i].size);
#else
		  printf ("block %u allocated size %u, ", i, UTIL.memalloc.malloc_array[i].size);
#endif
		else if (UTIL.memalloc.malloc_array[i].size < 0)
		  printf ("block %u free size %u, ", i, -UTIL.memalloc.malloc_array[i].size);
		else {
		  printf ("block %u size null, end of used array\r\n", i);
		  break;
		  }
	  toredraw = (struct todraw_str) {0,1,1,1,1,1,0, .presskey = 1};
	  }
#endif
#if 0
	else if ((key & 0xFF) == '0') {
	  unsigned cpt;
	  for (cpt = 0 ; cpt <= 0xFF; cpt++) {
	      unsigned char status;
	      unsigned short nbchannel, scsibios_codeseg;
	      printf ("_SCSIBIOSDISK_sanitycheck 0x%X: ", cpt);
	      if (_SCSIBIOSDISK_sanitycheck(cpt, &status, &nbchannel, &scsibios_codeseg) == 0) {
		  printf ("success status = 0x%X, nbchannel %u, scsibios_codeseg 0x%X\r\n", status, nbchannel, scsibios_codeseg);
		  }
		else
		  printf ("Failed status = 0x%X", status);
	      _BIOS_wait (500000);
	      printf ("\r                                                      \r");
	      }
	  printf ("\r\n_SCSI_CAM_BIOS_installcheck() returns %u\r\n", _SCSI_CAM_BIOS_installcheck());
	  _BIOS_wait (5000000);
	  }
#endif
#if 0
	else if ((key & 0xFF) == '0') {
	  union PCI_hardware_characteristics_u pciaccess;
	  unsigned short version_bcd;
	  struct PCIv3_support PCI_support;

	  printf ("_PCIBIOS_install_check() returns %u\r\n",
		_PCIBIOS_install_check (&pciaccess, &version_bcd, &PCI_support));
	  printf ("  i.e. pciaccess = 0x%X, version_bcd = 0x%X, PCI_support 0x%X\r\n",
		pciaccess.all, version_bcd, *(unsigned short *)&PCI_support);

	  {
		/* 0x00 for SCSI, 0x01 for IDE, 0x02 for floppy, 0x03 for IPI, 0x04 for RAID, 0x05 for ADMA, 0x80 for other */
		/* subclass 0x05 for PCMCIA bridge, 0x07 for CARDBUS bridge */
#if 0
	  struct PCI_classcode classcode_array[] = {
		{ .pci_class = 0x01, .pci_subclass = 0x01 , .programming_interface = 0x80 },
		{ .pci_class =  1, .pci_subclass = 31 },
		{ .pci_class = 31, .pci_subclass =  1 },
		{ .pci_class = 0x11, .pci_subclass =  1 },
		{ .pci_class =  1, .pci_subclass = 0x11 },
		{ .pci_class =  0, .pci_subclass = 0x11 , .programming_interface = 1 },
		{ .pci_class =  0, .pci_subclass = 1 , .programming_interface = 0x11 },
		}, *classcode = classcode_array;

	  for (; classcode <= &classcode_array[sizeof (classcode_array) / sizeof (classcode_array[0])]; classcode++) {
	      unsigned short device_index = 0;
	      struct PCI_location location = (struct PCI_location) { .bus_number = 0, .device = 0x31, .function = 1 };
	      unsigned ret = 0;

	      if (classcode < &classcode_array[sizeof (classcode_array) / sizeof (classcode_array[0])]) {
		  ret = _PCIBIOS_find_classcode (*classcode, device_index, &location);
		  printf ("_PCIBIOS_find_classcode({0, 0x%X, 0x%X, 0x%X}, %u, &location) returns 0x%X\r\n",
			classcode->pci_class, classcode->pci_subclass, classcode->programming_interface, device_index, ret);
		  }
#else
	  struct PCI_classcode classcode_array = { .pci_class = 0x01, .pci_subclass = 0x01 },
		 *classcode = &classcode_array;
	  for (classcode->pci_subclass = 0; ++(classcode->pci_subclass) != 0; )
	  for (classcode->programming_interface = 0; ++(classcode->programming_interface) != 0; ) {
	      unsigned short device_index = 0;
	      struct PCI_location location = {};
	      unsigned ret = 0;

	      if ((ret = _PCIBIOS_find_classcode (*classcode, device_index, &location)) == 0)
		  printf ("_PCIBIOS_find_classcode({0, 0x%X, 0x%X, 0x%X}, %u, &location) returns 0x%X\r\n",
			classcode->pci_class, classcode->pci_subclass, classcode->programming_interface, device_index, ret);
#endif

	      if (ret == 0) {
		  unsigned char cval;
		  unsigned short sval;
		  unsigned lval, cpt;

		  printf ("    location: bus_number %u, device 0x%X, function %u, ",
			location.bus_number, location.device, location.function);
		  if (_PCIBIOS_read_config_word(location, 0x04, &sval) == 0)
		      printf ("command 0x%X (I/O %sable, mem %sable) ", sval, sval & 1? "en":"dis", sval & 2? "en":"dis");
		  if (_PCIBIOS_read_config_byte(location, 0x0E, &cval) == 0)
		      printf ("header type 0x%X, ", cval);
		  if (_PCIBIOS_read_config_byte(location, 0x0B, &cval) == 0)
		      printf ("base class code 0x%X, ", cval);
		  if (_PCIBIOS_read_config_byte(location, 0x0A, &cval) == 0)
		      printf ("sub class code 0x%X, ", cval);
		  if (_PCIBIOS_read_config_byte(location, 0x09, &cval) == 0)
		      printf ("prog interface 0x%X, ", cval);
		  for (cpt = 0; cpt < 5; cpt++)
		      if (_PCIBIOS_read_config_dword(location, 0x10 + 4*cpt, &lval) == 0)
			  printf ("address[%u] is 0x%X i.e. %s ", cpt, lval, lval & 1 ? "I/O" : "mem");
		  if ((sval & 1) && (lval & 1)) {
		      lval &= ~1;
		      printf ("reading long I/O at 0x%X: 0x%X, 0x%X, 0x%X", lval, inl (lval), inl (lval+4), inl (lval+8));
		      }
		  puts ("");
		  }
	      }
	  }
	  toredraw = (struct todraw_str) {0,1,1,1,1,1,0, .presskey = 1};
	  }
#endif
	else {
	  PRINTF (MENU_UNKNOWN_KEY, key);
	  toredraw = (struct todraw_str) {0,0,0,0,1,0,0, .presskey = 1};
	  }

      /* gujin_attr.verbose may not be up to date if we are in the
	 "setup" screen, use copy_gujin_param.attrib value: */
      if (toredraw.presskey) {
	  if (copy_gujin_param.attrib.verbose) {
	      MOUSE_declare_attribute (atr);

	      MOUSE_attrset_fullscreenfield(atr, 1); /* full screen sensitive to clicks here */
	      MOUSE_attrset_activefield(atr, 1);
	      printf ("\r\n    ");
	      PUT_FIELD (PRESS_KEY_TO_CONTINUE, atr, SPACE_KEYCODE(), 0, 0);
	      UI.function.getkey (30 * TICKS_PER_SECOND);
	      }
	  }

      { /* If we have written too many lines, we may have scrolled: */
      unsigned char row, col;

      if (   UI.function.getcursor (&row, &col) != 0
	  || row == UI.parameter.nbrow - 1) {
	  toredraw.firstlines = 1;
	  toredraw.topmenu = 1;
	  toredraw.bottommenu = 1;
	  toredraw.advertisement = 1;
	  toredraw.copyright = 1;
	  }
      }
      } /* for (;;) */
  }}

static inline void treat_startup_keys (void)
  {
  struct shift_flags_str flags = _BIOS_get_shift_flags();

  if (flags.caps_lock_active || flags.control_key_pressed || flags.alternate_key_pressed) {
      if (flags.caps_lock_active) {
	  /*
	   * give a way to reset saved parameters - else it may
	   * be impossible to boot after changing the video card...
	   */
	  PUTS (IMPORTANT_PARAMETERS_RESETED);
	  menu_reset_saved_parameters();
	  }
      if (flags.control_key_pressed || flags.alternate_key_pressed) {
	  PUTS (FORCE_VIDEO_MODE_3);
	  copy_gujin_param.default_video_mode = 3;
	  }
      }
  }

#else /* USER_SUPPORT != 0 */

/*
 * Functions to read a key pressed are not even linked-in,
 * so decide alone what to load:
 */
static inline void displayDOSenviron (farptr psp, char *cmd, unsigned display)
  {
  unsigned char envir[80], *ptr;
  // 2Ch	WORD	DOS 2+ segment of environment for process (see #01379)
  farptr DOSenviron = peekw (psp + 0x2C) << 16;
  if (DOSenviron != 0) {
      if (display)
	  printf ("DOS environment at 0x%X: ", DOSenviron);
      for (;;) {
	  ptr = envir;
	  for (;;) {
	      *ptr = peekb(DOSenviron++);
	      if (*ptr == '\0')
		  break;
	      if (ptr == &envir[lastof(envir)]) {
		  *ptr = '\0';
		  break;
		  }
	      ptr++;
	      }
	  if (ptr == envir) {
	      if (display)
		  puts("");
	      break;
	      }
	  if (display)
	      printf ("'%s', ", envir);
	  }
      /* here on MSDOS 6.2, I have bytes 0x01, 0x00: skip up to the next zero */
      while (peekb (DOSenviron++) != 0)
	  /* nothing */;
      ptr = envir;
      for (;;) {
	  *cmd = *ptr = peekb(DOSenviron++);
	  if (*ptr == '\0')
	      break;
	  if (ptr == &envir[lastof(envir)]) {
	      *cmd = *ptr = '\0';
	      break;
	      }
	  ptr++;
	  cmd++;
	  }
      if (display)
	  printf ("Command line: '%s'\r\n", envir);
      }
    else
      strcpy (cmd, "gujin.exe");
  }

#if DISK_SUPPORT == DOS_SUPPORT
unsigned short selected_video_mode;
char extraexecute[80];
#endif

static inline unsigned
menu (struct registers *regs, struct gujin_param_attrib gujin_attr)
  {
#if DISK_SUPPORT != DOS_SUPPORT
  unsigned short cpt;

#if DEBUG & DEBUG_VIDEO
  print ("List of kernel found: ");
  for (cpt = 0; cpt < BOOTWAY.nb; cpt++) {
      print (BOOTWAY.desc[cpt].filename);
      print (", ");
      }
  print (".\r\nPress a key to continue and load kernel\r\n");
   _BIOS_getkey();
#endif

  for (cpt = 0; cpt < BOOTWAY.nb; cpt++) {
      menu_load_system (cpt, regs, 0);
#if DEBUG & DEBUG_VIDEO
      puts ("Press a key to continue and load next kernel");
      _BIOS_getkey();
#else
      puts ("");
#endif
      }
  _BIOS_failedboot();

#else

  unsigned char cmdline[256];
  char command[80];
  unsigned char *parse, *cptr;
  static struct diskparam_str _disk_param[NB_DISK] = {};
  static struct partition_str _disk_partition[NB_DISK][NB_PARTITION];
  static struct desc_str _fs_desc[NB_DESC_ARRAY] = {};

  BOOTWAY.desc = _fs_desc;
  BOOTWAY.max_name_length = NAME_LENGTH;
  BOOTWAY.max_desc_array = NB_DESC_ARRAY;
  BOOTWAY.sizeof_desc_str = sizeof (struct desc_str);

  DI.max_IDE_found = NB_IDE;
  DI.max_disk = NB_DISK;
  DI.max_partition = NB_PARTITION;
  DI.sizeof_diskparam_str = sizeof (struct diskparam_str);
  DI.sizeof_partition_str = sizeof (struct partition_str);

  DI.param = _disk_param;
  DI.param->partition = _disk_partition[0];

  {
  unsigned short cpt;
  farptr psp = (unsigned)DOS_GetCurrentPSPSegment() << 16;
  displayDOSenviron (psp, command, 0);

  parse = cmdline;
  psp += 0x80; /* points to the lenght */
  cpt = peekb (psp);
  if (cpt > sizeof (cmdline))
      cpt = sizeof (cmdline);
  while (cpt--)
      *parse++ = peekb (++psp);
  }

  if (*(parse - 1) == '\r' || *(parse - 1) == '\n')
      parse--; /* seems to be DOSEMU only */
  *parse = '\0';

  parse = cmdline;
  while (*parse == ' ' || *parse == '\t')
      parse++;

  if (   parse[0] == '\0'
      || (   (parse[0] == '-'|| parse[0] == '/')
	  && (parse[1] == 'h' || parse[1] == 'H' || parse[1] == '?'))) {
      /* Do not display command path: */
      char *cmdptr = command;
      while (*cmdptr) cmdptr++;
      while (cmdptr >= command && *cmdptr != '/' && *cmdptr != '\\')
	  cmdptr--;
      cmdptr++;
      printf (BASIC_LOADER_HELP_MESSAGE, cmdptr);
      puts (BASIC_LOADER_HELP_MESSAGE_2);
      return 0;
      }

#if 1 /* FIXME: has to be before /V if /V is present */
  extraexecute[0] = '\0';
  if (     (parse[0] == '/' || parse[0] == '-')
	&& (parse[1] == 'x' || parse[1] == 'X')
	&& parse[2] == '=') {
//      unsigned cpt = 0;
      unsigned cpt = 1; // command line format
      parse += 3;
      if (*parse == '\'' || *parse == '"') {
	  char firstchar = *parse++;
	  while (*parse && *parse != firstchar && cpt < sizeof (extraexecute)-3)
	      extraexecute[cpt++] = *parse++;
	  parse++;
	  }
	else
	  while (*parse && *parse != ' ' && *parse != '\t' && cpt < sizeof (extraexecute)-3)
	      extraexecute[cpt++] = *parse++;
      extraexecute[cpt++] = '\r';
      extraexecute[cpt] = '\0';
      extraexecute[0] = cpt - 2; // command line format, 1st byte is length w/o CR
      while (*parse == ' ' || *parse == '\t')
	  parse++;
      }
#endif

#if 1
  selected_video_mode = 0xFFFF;

  if (   (parse[0] == '-'|| parse[0] == '/')
      && (parse[1] == 'v'|| parse[1] == 'V')) {
      VESA_VbeInfoBlock VESA_info;
      VESA_modeinfo_t VESA_modeinfo;
      farptr videomode_farptr;
      unsigned short videomode, nbselect = 0, cpt = 0;
      char videostr[32];

      if (parse[1] == 'V')
	  parse[1] = 'v';
      VESA_info.VbeSignature[0] = 'V';
      VESA_info.VbeSignature[1] = 'B';
      VESA_info.VbeSignature[2] = 'E';
      VESA_info.VbeSignature[3] = '2';
      if (   _VESA_getinfo(&VESA_info) != 0
	  || VESA_info.VbeSignature[0] != 'V' || VESA_info.VbeSignature[1] != 'E'
	  || VESA_info.VbeSignature[2] != 'S' || VESA_info.VbeSignature[3] != 'A'
	  || VESA_info.VideoModePtr == 0 || (VESA_info.VideoModePtr >> 16) > 0xFFF0
	  ) {
	  puts (BASIC_VESA_NOTCOMPATIBLE);
	  return 1;
	  }

      if (parse[2] != '=')
	  printf (BASIC_AVAILABLE_VESA_MODE_MESSAGE, VESA_info.VbeVersion >> 8, VESA_info.VbeVersion & 0xFF);
      videomode_farptr = VESA_info.VideoModePtr;
      while ((videomode = peekw(videomode_farptr)) != 0xFFFF) {
//	  if (   _VESA_getmodeinfo (&VESA_modeinfo, videomode | 0x4000) == 0
	  if (   _VESA_getmodeinfo (&VESA_modeinfo, videomode) == 0
// Some VESA2 BIOS do not set this bit (but for text mode) even when they can display chars... maybe they cannot scroll.
//	      && VESA_modeinfo.ModeAttributes.BIOS_output_supported
	      && VESA_modeinfo.ModeAttributes.optional_info_available
	      && (!VESA_modeinfo.ModeAttributes.graphic_mode || VESA_modeinfo.ModeAttributes.linear_framebuffer_supported)
	      && VESA_modeinfo.ModeAttributes.mode_supported) {
	      sprintf (videostr, "/V=%ux%ux%u", VESA_modeinfo.XResolution, VESA_modeinfo.YResolution, VESA_modeinfo.BitsPerPixel);
	      if (parse[2] != '=') {
		  printf ("  %s", videostr);
		  if (++cpt % 3 == 0)
		      puts ("");
		  }
		else {
		  char *ptr = videostr + 3;
		  while (*ptr && *ptr == parse[ptr - videostr])
		      ptr++;
		  if (*ptr == '\0') {
		      selected_video_mode = videomode;
		      break;
		      }
		  if (ptr - videostr > nbselect) {
		      nbselect = ptr - videostr;
		      selected_video_mode = videomode;
		      }
		  }
	      }
	  videomode_farptr += 2;
	  }
      if (parse[2] != '=')
	  return 0;
      while (*parse != '\0' && *parse != ' ' && *parse != '\t')
	  parse++;
      while (*parse == ' ' || *parse == '\t')
	  parse++;
      }
#endif

  /* set DI.param[0].access == dos_part */
  DI.nbdisk = 1;
  DI.param[0].access = dos_part;

  {
  char *ptr, *end;

  ptr = BOOTWAY.desc[0].filename;
  end = &BOOTWAY.desc[0].filename[lastof (BOOTWAY.desc[0].filename)];
  while (ptr < end && (*parse > ' ' && *parse <= '~'))
      *ptr++ = *parse++;
  *ptr = '\0';
  ptr -= 4;
  if (*(unsigned *)ptr == *(unsigned *)".kgz")
      BOOTWAY.desc[0].boottype = is_elf;
    else
      BOOTWAY.desc[0].boottype = is_linux;
  }
  {
  unsigned short handle, nbread_errorcode = 0;

  if (DOS_OpenFile (BOOTWAY.desc[0].filename, DOS_open_read_only, &handle) != 0) {
      printf (ERROR_FILE_S_DOES_NOT_EXIST, BOOTWAY.desc[0].filename);
      return 0x2100;
      }
    else if (DOS_SeekFile (handle, DOS_SEEK_END, 0, &BOOTWAY.desc[0].filesize, &nbread_errorcode) != 0)
      return 0x2100;
    else if (DOS_CloseFile (handle, &nbread_errorcode) != 0)
      return 0x2100;
  }

  BOOTWAY.nb = 1;
  BOOTWAY.desc[0].disk = 0;

  while (*parse == ' ' || *parse == '\t')
      parse++;

#if 0 /* NO, do not change it: console=ttyS0,115200 */
  /* The DOS command is uppercased by DOS, re-lowercase it back for Linux: */
  for (cptr = parse; *cptr != '\0'; cptr++)
      if (*cptr >= 'A' && *cptr <= 'Z')
	  *cptr = *cptr - 'A' + 'a';
#endif

  /* look if first parameter contains an equal sign "=", if yes it has to
	be the first parameter (after kernel name), else it is the initrd.
	If first parameter is "initrd=" then do the right thing */
  if (parse[0] == 'i' && parse[1] == 'n' && parse[2] == 'i' && parse[3] == 't'
	&& parse[4] == 'r' && parse[5] == 'd' && parse[6] == '=')
      parse += 7;

  cptr = parse;
  while (*cptr != '\0' && *cptr != '-' && *cptr != ' ' && *cptr != '=')
      cptr++;

  if (*parse != '\0' && *parse != ' ' && *parse != '-' && *cptr != '=') {
      char *ptr, *end;
      unsigned short handle, nbread_errorcode = 0;

      ptr = BOOTWAY.desc[1].filename;
      end = &BOOTWAY.desc[1].filename[lastof (BOOTWAY.desc[1].filename)];
      while (ptr < end && (*parse > ' ' && *parse <= '~'))
	  *ptr++ = *parse++;
      *ptr = '\0';

      BOOTWAY.nb_initrd = 1;
      BOOTWAY.desc[1].disk = 0;
      BOOTWAY.desc[1].boottype = is_initrd;
      if (DOS_OpenFile (BOOTWAY.desc[1].filename, DOS_open_read_only, &handle) != 0) {
	  printf (ERROR_FILE_S_DOES_NOT_EXIST, BOOTWAY.desc[1].filename);
	  return 0x2100;
	  }
	else if (DOS_SeekFile (handle, DOS_SEEK_END, 0, &BOOTWAY.desc[1].filesize, &nbread_errorcode) != 0)
	  return 0x2100;
	else if (DOS_CloseFile (handle, &nbread_errorcode) != 0)
	  return 0x2100;
      }
    else {
      if (*parse == '-')
	  parse++;
      BOOTWAY.desc[1].filename[1] = '\0';
      BOOTWAY.nb_initrd = 0;
      BOOTWAY.desc[1].filesize = 0;
      }

  {
  extern const gujin_param_t gujin_param;
  unsigned cpt;
  char last = ' ';
  char filename_opt[80], *fnopt = filename_opt, filename_buf[256];
  unsigned short handle = 0;

  for (cpt = 0; cpt < sizeof (gujin_param.extra_cmdline) - 1; cpt++)
      if (copy_gujin_param.extra_cmdline[cpt] == '\0')
	  break;
  while (*parse != '\0' && (*parse == ' ' || *parse == '\t'))
      parse++;
  if (*parse != '\0') for (;;) {
      if ((last == ' ' || last == '\t') && *parse == '@' && parse[1] != '\0' && parse[1] != ' ' && parse[1] != '\t') {
	  unsigned short nbread_errorcode = 0, errorclose;
//	  char *save_fnopt = parse;
	  do {
	      *fnopt++ = *++parse;
	      } while (*parse != '\0' && *parse != ' ' && *parse != '\t');
	  *fnopt = '\0';
	  if (DOS_OpenFile (filename_opt, DOS_open_read_only, &handle) != 0) {
	      print ("Getting parameter from file \"");
	      print (filename_opt);
	      puts ("\": open failed.");
	      return 1;
//	      parse = save_fnopt;
//	      fnopt = filename_opt;
	      }
	    else if (DOS_ReadFile (handle, filename_buf, sizeof (filename_buf) - 1, &nbread_errorcode) != 0
			// || nbread_errorcode == 0 : accept empty file
			|| nbread_errorcode > sizeof (filename_buf) - 1) {

	      print ("Getting parameter from file \"");
	      print (filename_opt);
	      puts ("\": read failed.");
//	      parse = save_fnopt;
//	      fnopt = filename_opt;
	      DOS_CloseFile (handle, &errorclose);
	      return 1;
	      }
	    else if (DOS_CloseFile (handle, &errorclose) == 0) {
//printf ("getting parameter from file '%s': got %u chars, i.e. '%s'.\r\n|", filename_opt, nbread_errorcode, filename_buf);
	      fnopt = filename_buf;
	      while (   nbread_errorcode-- != 0 /* accept empty file */
		     && (   filename_buf[nbread_errorcode] == ' ' || filename_buf[nbread_errorcode] == '\0'
			 || filename_buf[nbread_errorcode] == '\t' || filename_buf[nbread_errorcode] == '\r'
			 || filename_buf[nbread_errorcode] == '\n'));
	      filename_buf[++nbread_errorcode] = '\0';
	      }
	    else {
//puts ("Close failed???");
//	      parse = save_fnopt;
	      fnopt = filename_opt;
	      }
	  }

      copy_gujin_param.extra_cmdline[cpt] = last;
//printf ("%c", last);
      if (last == '\0')
	  break;
// DOS has a 128 char limit for command line, but FreeDOS has more;
// we use the unused space after extra_cmdline field to store
// end of command line for mingujin.exe.
//	else if (cpt == sizeof (gujin_param.extra_cmdline) - 1) {
	else if (cpt == sizeof (gujin_param.extra_cmdline) - 1
			+ sizeof (gujin_param.cardname)
			+ sizeof (gujin_param.VGA_valid_mode)
			+ sizeof (gujin_param.vga_mode)
			) { /* USER_SUPPORT == 0 here, max 232 char cmdline */
	  puts (CMDLINE_TOO_LONG);
	  last = '\0';
	  }
	else {
	  cpt++;
	  if (fnopt == filename_opt)
	      last = *parse++;
	    else { /* getting command from file for now */
	      last = *fnopt++;
	      if (last == '\0') {
		  fnopt = filename_opt;
		  last = *parse++;
		  }
	      if (last == '\r' || last == '\n')
		  last = ' ';
	      }
	  }
      }
//printf ("|\r\n end of test.\r\n"); return 0x2100;
  if (handle != 0) { /* at least one file @param given */
      for (cpt = 0; cpt < sizeof (filename_buf) - 1; cpt++) {
	  filename_buf[cpt] = copy_gujin_param.extra_cmdline[cpt];
	  if (filename_buf[cpt] == '\0')
	      break;
	  }
      filename_buf[cpt] = '\0';
      print ("Parameters: ");
      puts (filename_buf);
      }
  }

  menu_load_system (0, regs, 0);

#endif /* DISK_SUPPORT != DOS_SUPPORT */

  return 1;
  }

#endif /* USER_SUPPORT != 0 */

__attribute__ ((cdecl, regparm (0)))
int main (struct registers *regs)
  {
  int tmp;

//  BOOT1_putstr ("In Main\r\n");
#if USER_SUPPORT != 0 /* not for tiny_img nor tiny_exe */
  init_malloc();
#endif
  DBG_INIT();

#ifdef SERIAL_INIT_MODEM_STRING
  if (BOOT1_COM_port() >= 0)
      BOOT1_putstr (SERIAL_INIT_MODEM_STRING);
#endif

  UTIL.boot_device_read_only = BOOT1_DOS_running();
  extern char eltoritodisk[];
  UTIL.bios_boot_dl = codeseg_peekb(eltoritodisk); // regs->edx & 0xFF;
  extern const gujin_param_t gujin_param;
  lmemcpy(&copy_gujin_param, code_adr(&gujin_param), sizeof (copy_gujin_param));

#if (SETUP & MULTILINGUAL) && (DISK_SUPPORT & DOS_SUPPORT)
  unsigned char real_MltStrLanguage = copy_gujin_param.MltStrLanguage;
  if (BOOT1_DOS_running())
      real_MltStrLanguage = getDOSenvironLANG (0x10000 * DOS_GetCurrentPSPSegment());
  if (real_MltStrLanguage != MLTSTR_RUSSIAN) /* Else we can't print the first few lines! */
      MltStrLanguage = real_MltStrLanguage;
#endif

#if USER_SUPPORT != 0
  UI_init (FIRST_BOOT1_COM_port() >= 0);
#endif

  if (copy_gujin_param.attrib.disk_write_enable == 0) {
      /* For instance, caps_lock at startup shall not write
	 once the configuration if not already write enabled: */
      copy_gujin_param.attrib.write_disabled_written_once = 1;
      }

#if USER_SUPPORT != 0
  treat_startup_keys ();
#endif

#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
  if (BOOT1_COM_port() < 0)
      UI.monitor = BIOS_probemonitor();
#endif

  detect_interface(copy_gujin_param.attrib); /* After that we can print cyrillic letters! */

#if (SETUP & MULTILINGUAL) && (DISK_SUPPORT & DOS_SUPPORT)
  if (real_MltStrLanguage == MLTSTR_RUSSIAN)
      MltStrLanguage = real_MltStrLanguage;
  update_msg_exiting();
#endif

#if 0
  farptr int13addr = peekl(4 * 0x13);
  unsigned char simdisk = peekll(int13addr + 3) == *(unsigned long long *)"$INT13SF"
	&& peekb(int13addr + 28) > ' ' && peekb(int13addr + 28) <= '~'
	&& peekll(int13addr + 28) != *(unsigned long long *)"unknown";
  printf ("int13addr = 0x%X, simdisk= %s, flags 0x%X, disk emulated 0x%X\r\n", int13addr, simdisk? "true" : "false", peekl(int13addr + 23), peekb(int13addr + 27));
  if (simdisk) {
      char CS_int13filename[64];
      unsigned cpt = 0;
      int13addr += 28;
      while (cpt < sizeof (CS_int13filename) && (CS_int13filename[cpt++] = peekb(int13addr++)) != '\0')
	  continue;
      farptr xaddr = peekl(4 * 0x13) + 28 + sizeof (CS_int13filename);
      unsigned bus_string = peekl(xaddr + 4);
      printf ("Booted from simulated disk, filename '%s' on BIOS 0x%X on IDE I/O 0x%X %s on bus %s\r\n",
	CS_int13filename, peekb(xaddr), peekw(xaddr+2), (peekb(xaddr + 1) & 0x10)? "slave" : "master", (char *)&bus_string);
	/* there is other parameters there to totally identify the disk/partition:
"	.ascii	\"$INT13SF\"					\n"
"	.ascii	\"Gujin   \"					\n"
"CS_oldintaddr:							\n"
"	.long	0						\n"
"	.long	0	# flags					\n"
"CS_int13biosdisk:	.byte	0xFF				\n"
"CS_int13filename:						\n"
"	.ascii	\"unknown\"					\n"
"	.skip	64 - 7						\n"
"CS_host_bios:		.skip 1					\n"
"CS_ide_mask:		.skip 1					\n"
"CS_ide_base:		.skip 2					\n"
"CS_ebios_bustype:	.skip 4					\n"
"CS_ebios_Interface:	.skip 8					\n"
"CS_ebios_bus:		.skip 8					\n"
"CS_ebios_device:	.skip 16				\n"
"CS_partition_start	.skip 8					\n"
"CS_partition_length:	.skip 8					\n"
"CS_partition_type:	.skip 1					\n"
	*/
      }
#endif

  detect_environment ();
  detect_joystick_mouse (copy_gujin_param.attrib);

#if USER_SUPPORT & (VGA_SUPPORT | VESA_SUPPORT)
#if !(SETUP & QUITE)
  report_screen();
#endif
#endif

//printf ("initial %edx (i.e. boot disk) = 0x%X\r\n", regs->edx);
//gdt_idt_param_t gdtptr, idtptr;
//get_gdt(&gdtptr); get_idt(&idtptr);
//printf ("GDT: base 0x%X limit 0x%X, IDT: base 0x%X limit 0x%X\n ", gdtptr.base, gdtptr.limit, idtptr.base, idtptr.limit);

  if (!BOOT1_DOS_running()) {
      unsigned short returned_code;
      unsigned char ret;

#if 0
      unsigned short disk;
      extern const boot1param_t boot1param;

      /* Take care, registers at initialisation of Gujin are no more 0xFFEC..0xFFFE
       * because there has been a popaw and a pushal since boot1 */
      printf("CDROMpatch boot drive: (0x%X, offset 0x%X); initial dx: 0x%X\r\n",
		codeseg_peekb((void *)&boot1param.bootloader2.bootloader2_cmd.int1342.reg_dx),
		codeseg_peekl((void *)0x1C), regs->edx);

      printf ("Scanning lock status from 0xFF to 0:\r\n");
	// usually nothing locked
      disk = 0x100;
      while (disk--)
	  if (_EBIOSDISK_locker (disk, _EBIOSDISK_getlock, &returned_code) != 0)
	      printf (".");
	    else
	      printf ("\r\n_EBIOSDISK_getlock 0x%X success, lock state 0x%X\r\n",
			disk, returned_code);

      printf ("\r\nScanning CDROM 0xFF to 0:\r\n");
	// Sometimes in no-emul modes, the CDROM BIOS call works once
	// and then disappear so you do no more see it here:
      disk = 0x100;
      while (disk--)
	  if (_BIOSCDROM_getstatus (&UTIL.CDROM_HDFD_emulation_spec, disk, &returned_code) != 0)
	      printf (".");
	    else {
	      printf ("\r\nCDROM 0x%X valid, returned_code 0x%X, boot drive 0x%X at 0x%X\r\n",
			disk, returned_code,
			UTIL.CDROM_HDFD_emulation_spec.bootdrive,
			UTIL.CDROM_HDFD_emulation_spec.LBAimage);
	      {
	      CDROMcatalog catalog = {
		.packetsize = sizeof (CDROMcatalog),
		.nbsector = 1,
		.buffer = fourKbuffer,
		.firstsector = 0
		};
	      if (_BIOSCDROM_getcatalog(&catalog, UTIL.CDROM_HDFD_emulation_spec.bootdrive, &returned_code) != 0)
		  printf ("Error getting catalog 0x%X\r\n", returned_code);
		else
		  printf ("Got catalog 0x%X\r\n",returned_code);
	      }
	      }
      printf ("\r\nend Scan\r\n");

      for (disk = 0, ret = ~0; disk <= 0x80 && ret != 0; disk += 0x80) {
	  /* CDROM in HD or FD emulation mode */
	  DDBG (("_BIOSCDROM_getstatus %s 0x%X: ", disk ? "floppy" : "hard disk", disk));
	  ret = _BIOSCDROM_getstatus (&UTIL.CDROM_HDFD_emulation_spec, disk, &returned_code);
	  DDBG (("carry %s, return code 0x%X\r\n", ret? "set" : "clear", returned_code));
	  }

      if (ret == 0) {
	  const CDROMbootspec *const spec = &UTIL.CDROM_HDFD_emulation_spec;
	  printf ("Booting from CDROM, packetsize==0x13? 0x%X, bootmedia: reserved %u, "
		  "ATAPIdrv %u, SCSIdrv %u, bootmedia.mediatype = 0x%X, i.e. ",
		  spec->packetsize, spec->bootmedia.reserved,
		  spec->bootmedia.ATAPIdrv, spec->bootmedia.SCSIdrv,
		  spec->bootmedia.mediatype);
	  puts ( ((const unsigned char *const []){"noemul", "floppy120",
			"floppy144", "floppy288", "harddisk", "invalid"})
			[spec->bootmedia.mediatype <= harddisk ? spec->bootmedia.mediatype : harddisk+1]);
	  printf ("bootdrive 0x%X, CDROMcontrol 0x%X, LBAimage 0x%X, devspec 0x%X, buffer3Kseg 0x%X\r\n",
		  spec->bootdrive, spec->CDROMcontrol, spec->LBAimage, spec->devspec, spec->buffer3Kseg);
	  printf ("bootloadseg 0x%X, nb512bytesect %u, lsbnbcyl %u, nbsector_msbnbcyl 0x%X, nbhead %u\r\n",
		  spec->bootloadseg, spec->nb512bytesect, spec->lsbnbcyl, spec->nbsector_msbnbcyl, spec->nbhead);
	  _BIOS_wait (5000000);
#else
      UTIL.CDROM_HDFD_emulation_spec.bootdrive = UTIL.bios_boot_dl;
      UTIL.CDROM_HDFD_emulation_spec.packetsize = sizeof (UTIL.CDROM_HDFD_emulation_spec);
      DDBG (("_BIOSCDROM_getstatus on 0x%X packetsize 0x%X (== 0x13?): ", UTIL.bios_boot_dl, UTIL.CDROM_HDFD_emulation_spec.packetsize));
      ret = _BIOSCDROM_getstatus (&UTIL.CDROM_HDFD_emulation_spec, UTIL.bios_boot_dl, &returned_code);
//extern short boot1param_reg_dx[], bootchain_reg_dx[];
//printf ("boot1param_reg_dx 0x%X, bootchain_reg_dx 0x%X, eltoritodisk 0x%X, UTIL.bios_boot_dl 0x%X, _BIOSCDROM_getstatus returns carry %s, return code 0x%X, emulation_spec.bootdrive 0x%X\r\n", codeseg_peekw(boot1param_reg_dx), codeseg_peekw(bootchain_reg_dx), codeseg_peekb(eltoritodisk), UTIL.bios_boot_dl, ret? "set" : "clear", returned_code, UTIL.CDROM_HDFD_emulation_spec.bootdrive); _BIOS_getkey();
      DDBG (("carry %s, return code 0x%X\r\n", ret? "set" : "clear", returned_code));
      if (ret == 0 && UTIL.CDROM_HDFD_emulation_spec.packetsize != 0) {
#endif
	  UTIL.CDROM_HDFD_emulation_state = 1;
	  if (copy_gujin_param.stop_emulation == 1) {
	      /* Problem reported when BIOS boot CDROM containning Gujin which boots itself from the
		same CDROM: The BIOS stack sees twice _BIOSCDROM_terminate(all disks) and react badly.
		Try to send _BIOSCDROM_terminate only to the disk booted:
	      DDBG (("call _BIOSCDROM_terminate, all floppy (0x7F as disk): carry "));
	      if (_BIOSCDROM_terminate (&UTIL.CDROM_HDFD_emulation_spec, 0x7F, &returned_code) == 0) {
		*/
	      DDBG (("call early _BIOSCDROM_terminate on floppy 0x%X: carry ", UTIL.bios_boot_dl));
	      if (_BIOSCDROM_terminate (&UTIL.CDROM_HDFD_emulation_spec, UTIL.bios_boot_dl, &returned_code) == 0) {
		  DDBG (("cleared"));
		  UTIL.CDROM_HDFD_emulation_state = 2;
		  }
		else
		  DDBG (("set"));
	      DDBG ((", return code 0x%X.\r\n", returned_code));
	      }
	  }
      }

  UTIL.time_hour_offset = copy_gujin_param.time_hour_offset;
  UTIL.time_minute_offset = copy_gujin_param.time_minute_offset;

#if USER_SUPPORT != 0
  /* BEFORE probing password protected disks */
  if (UTIL.keyboard != serialkbd) {
      enum kbd_enum keyboard = copy_gujin_param.kbdmap;

      if (keyboard != kbd_unknown) {
	  DBG (("Setting keyboard %s\r\n", kbdname(keyboard)));
	  kbd_setup (keyboard);
	  }
      }
#endif

#if (DEBUG & DEBUG_LOADER) && (USER_SUPPORT != 0)
  ZDBG (("BIOS data area content:"));
  for (tmp = 0; tmp < 0x100; tmp++) {
      if (tmp % 16 == 0)
	  ZDBG (("\r\n0x%X: ", tmp));
      ZDBG (("0x%X, ", peekb (0x400000 + tmp)));
      }
  {
  unsigned short XtAt_carry;
  farptr sysconfptr = _BIOS_getConfiguration (&XtAt_carry);
  ZDBG (("\r\n\r\nget_system_configuration: carry %u, Xt(0x80)/At(0x86) 0x%X farptr 0x%X\r\n", XtAt_carry & 0xFF, XtAt_carry >> 8, sysconfptr));
  if (((XtAt_carry & 0xFF) == 0) && (XtAt_carry >> 8 == 0x86) && sysconfptr) {
      ZDBG (("\tdescriptor length %u, model 0x%X/0x%X BIOS revision 0x%X\r\n", peekw(sysconfptr), peekb(sysconfptr+2), peekb(sysconfptr+3), peekb(sysconfptr+4)));
      ZDBG (("\tFeature information 0x%X i.e. EBDA allocated: %u\r\n", peekb(sysconfptr+5), !!(peekb(sysconfptr+5) & 0x4)));
      }
  }
  {
  farptr ebda_seg = _BIOS_getEBDA_segment();
  unsigned cpt;
  ZDBG (("\r\n\r\ndisk size: "));
  for (cpt = 0; cpt < DI.nbdisk; cpt++)
      ZDBG (("%s: %u i.e. 0x%X \r\n", DI.param[cpt].diskname, (unsigned)DI.param[cpt].BEER_disksize, (unsigned)DI.param[cpt].BEER_disksize));
  ZDBG (("\r\nEBDA at 0x%X size %u Kb, data area content (disks at 0x3D, 0x4D):", ebda_seg, peekw (ebda_seg << 16)));
  ebda_seg <<= 16;
  if (ebda_seg) for (tmp = 0; tmp < 0x100; tmp++) {
      if (tmp % 16 == 0)
	  ZDBG (("\r\n0x%X: ", tmp));
      ZDBG (("0x%X, ", peekb (ebda_seg + tmp)));
      }
  }
  ZDBG (("\r\n"));
#endif

#if (DISK_SUPPORT != DOS_SUPPORT) || (USER_SUPPORT != 0) /* not for tiny_exe */
  if (!BOOT1_DOS_running())
      PRINTF (DETECTING_DISKS_BOOTED_FROM, UTIL.bios_boot_dl);
    else
      PRINTF (DETECTING_DISKS);
  if (UTIL.CDROM_HDFD_emulation_state)
      PRINTF (UTIL.CDROM_HDFD_emulation_state == 2 ? CDROM_EMULATION_STOP : CDROM_EMULATION_KEPT, UTIL.CDROM_HDFD_emulation_spec.bootdrive);
  PUTS ("");
  probedisk();
  PRINTF (HAVE_FOUND_U_DISKS, DI.nbdisk);
  list_disks();

  disk_analyse ();
  if (!copy_gujin_param.attrib.probe_file_in_iso_image)
      PRINTF (FOUND_D_WAY_TO_BOOT_U_INITRD, BOOTWAY.nb, BOOTWAY.nb_initrd);
    else
      PRINTF (FOUND_D_ISO_D_WAY_TO_BOOT_U_INITRD, BOOTWAY.nb_iso, BOOTWAY.nb, BOOTWAY.nb_initrd);
#endif

#if !(SETUP & QUITE) && (USER_SUPPORT != 0)
  if (copy_gujin_param.attrib.verbose) {
      MOUSE_declare_attribute (atr);

      MOUSE_attrset_fullscreenfield(atr, 1); /* full screen sensitive to clicks here */
      MOUSE_attrset_activefield(atr, 1);
      PUT_FIELD (PRESS_PAUSE_TO_STOP_DISPLAY, atr, HOME_KEYCODE(), 0, 0);
      UI.function.getkey (5 * TICKS_PER_SECOND);
      while (_BIOS_get_shift_flags().control_key_pressed)
	  continue;
      }
#endif

  tmp = menu (regs, copy_gujin_param.attrib);

  if (!BOOT1_DOS_running() && UTIL.CDROM_HDFD_emulation_state == 1 && copy_gujin_param.stop_emulation == 2) {
      unsigned short returned_code;
      /* Problem reported when BIOS boot CDROM containning Gujin which boots itself from the
	same CDROM: The BIOS stack sees twice _BIOSCDROM_terminate(all disks) and react badly.
	Try to send _BIOSCDROM_terminate only to the disk booted:
      DDBG (("call _BIOSCDROM_terminate, all floppy (0x7F as disk): "));
      if (_BIOSCDROM_terminate (&UTIL.CDROM_HDFD_emulation_spec, 0x7F, &returned_code) == 0) {
	*/
      DDBG (("call late _BIOSCDROM_terminate on floppy 0x%X: carry ", UTIL.bios_boot_dl));
      if (_BIOSCDROM_terminate (&UTIL.CDROM_HDFD_emulation_spec, UTIL.bios_boot_dl, &returned_code) == 0) {
	  DDBG (("cleared, return code 0x%X.\r\n", returned_code));
	  UTIL.CDROM_HDFD_emulation_state = 2;
	  PRINTF (CDROM_EMULATION_STOP, UTIL.CDROM_HDFD_emulation_spec.bootdrive);
	  }
	else
	  DDBG (("set, return code 0x%X.\r\n", returned_code));
      }
  before_exit();
  if (UTIL.swap_BIOSint0x13_irqhandler)
      pokel (0x13 * 4, UTIL.swap_BIOSint0x13_irqhandler);
  return tmp;
  }
