/* library.h */
#ifndef LIBRARY_H
#define LIBRARY_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef MAKE_H
#error I need file "make.h" included!
#endif

#define ASMREAD volatile  /* inline function with asm("") are const fcts */

//#if __GNUC__ >= 4
// problem with "m" is it does not cover a structure defined in the stack...
//#define ASM_STRUCT_INPUT(str)	"m"(str)
//#define ASM_STRUCT_OUTPUT(str)	"=m"(str)
//#define ASM_STRUCT_INOUT(str)	"+m" (str)
//#else
#define ASM_STRUCT_INPUT(str)	"X" (str)
#define ASM_STRUCT_OUTPUT(str)	"=X" (str)
#define ASM_STRUCT_INOUT(str)	"+X" (str)
//#endif

#if (__GNUC__ > 3) || (__GNUC__ == 3 && __GNUC_MINOR__ >= 4)
#define attr_used	used
#define FASTCALL        __attribute__ (( fastcall ))
#else
#define attr_used	unused
#define FASTCALL        __attribute__ (( regparm(1) ))
#endif

/*
 * Well, I am using awk to transform most near call to far calls:
 * If you get strait "awk" in your assembly, and no "#awk",
 * then you forgot to run "awk -f awk.cmd < xxx.S > xxx.s".
 */
#if SETUP & XCODE_SEGMENT	/* Makefile .S.s */
#define awk_farcall(fct)	asm ("awk farcall " #fct);
#define awk_farret(fct)		asm ("awk farret " #fct);
#define awk_farfct(ptr)		asm volatile ("awk farptr %0" : : "m" (ptr));
#else
#define awk_farcall(fct)	/* */
#define awk_farret(fct)		/* */
#define awk_farfct(ptr)		/* */
#endif

/*
 * If you do not understand this general stuff,
 * re-read your favorite C reference book...
 */
#define UINT_MAX	0xFFFFFFFFU
#define string(str)	#str
#define STRING(str)	string(str)
#define nbof(X)		(sizeof (X) / sizeof (X[0]))
#define lastof(X)	(nbof(X) - 1)
#define DIVROUNDUP(X,Y)	(((X)+(Y)-1)/(Y))
#define ROUNDUP(X,Y)	((Y) * DIVROUNDUP(X,Y))
#define ABS(X)		(((X) < 0)? -(X) : (X))
#if 1
#define MIN(X,Y)	(((X) <= (Y))? (X) : (Y))
#define MAX(X,Y)	(((X) >= (Y))? (X) : (Y))
#else
#define MIN(X,Y) \
	( (((X) < 0) && ((Y) > 0))					\
	 ? (X)								\
	 : (((X) > 0) && ((Y) < 0)) ? (Y)				\
				    : (((X) <= (Y))? (X) : (Y)))

#define MAX(X,Y) \
	( (((X) < 0) && ((Y) > 0))					\
	 ? (Y)								\
	 : (((X) > 0) && ((Y) < 0)) ? (X)				\
				    : (((X) >= (Y))? (X) : (Y))
#endif

#ifdef offsetof
#undef offsetof
#endif
#define offsetof(type, field)	((unsigned long)&((type *)0)->field)
//#define CAST(type, val) ({ union { typeof(val) before; type after; } tmp = { val }; /*return */ tmp.after; })
#define CAST(type, val) ((type)val)

#if 0
//#define ENDIAN16(x) ({ unsigned short tmp = (x); /* return */ (unsigned short)((tmp >> 8) | (tmp << 8)); })
#define ENDIAN16(x) ({ unsigned short tmp = (x); tmp = (tmp >> 8) | (tmp << 8); /* return */ tmp; })
#define ENDIAN32(x) ({ union { unsigned ul; unsigned short us; } tmp = { .ul = (x) }; \
    tmp.us = (tmp.us >> 8) | (tmp.us << 8);	    \
    tmp.ul = (tmp.ul >> 16) | (tmp.ul << 16);	    \
    tmp.us = (tmp.us >> 8) | (tmp.us << 8);	    \
    /* return */ tmp.ul; })
#else
#define ENDIAN16(x) ({ unsigned short tmp = (x); asm (" rolw $8,%w0 " : "+g" (tmp)); /* return */ tmp; })
#if defined (__i386__)
#define ENDIAN32(x) ({ unsigned tmp = (x); asm (" rolw $8,%w0 ; roll $16,%0 ; rolw $8,%w0 " : "+g" (tmp)); /* return */ tmp; })
#else
#define ENDIAN32(x) ({ unsigned tmp = (x); asm (" bswap %0 " : "+r" (tmp)); /* return */ tmp; })
#endif
#endif

#define UNUSUAL(X)	__builtin_expect (X, 0)
#define USUAL(X)	__builtin_expect (X, 1)

/*
 * The linker will refuse to link this symbol, used to check
 * for compile/link time error:
 */
void __ERROR(void);

/*
 * For DOS & e2fs load file in fs.c // _not_ malloced (DOS)
 * For disk.c used in different places.
 */
#define DEFAULT_BUFFER_SIZE (4U * 1024U)
extern unsigned char fourKbuffer[DEFAULT_BUFFER_SIZE];

/*
 * The C language is a bit subtile times to times,
 * "extern char memory[];" will not cover local variables,
 * and "extern char *memory;" can be transmitted by a register.
 * Re-read the last sentence - it is seriously important.
 * It means that the following is not a real memory barrier,
 * and cannot be used by "memcpy" - it is no more efficient
 * than asm (""::: "memory");
 * The form "extern char *memory;" will in some case put the
 * address of the pointer "memory" in a register so add some
 * assembly code - that I do not want.
 * I still do not know the solution - leave it this way for now.
 * Maybe create the form 'asm ("" : : "memory", "stack");' ?
 */
extern char memory[];
//union { char *pc, c; short *ps, s; long *pl, l; } *mem;

/* ONLY to be used with cst_str as a constant inline string, i.e. "abc" */
#define STRINIT(dst, cst_str) \
    ({ memcpy (dst, cst_str, sizeof (cst_str)); dst + sizeof (cst_str) - 1; })

#define STREQL(str, cst_str) \
    memeql (str, cst_str, sizeof (cst_str))

#define STRBEGINEQL(str, cst_str) \
    memeql (str, cst_str, sizeof (cst_str) - 1)

/* Maybe you should also re-read the GCC documentation ? */
#define Memset(ptr, value, size) do { \
    union { char tmparray[size]; typeof (*ptr) noalias[0];	\
		} *unused  __attribute__((unused));		\
    *((typeof(unused)) (ptr)) = (typeof(*unused)) {		\
	.tmparray = {[0 ... size-1] = value }};			\
    } while (0)

#define Bzero(ptr, size) do { \
    union { char tmparray[size]; typeof (*ptr) noalias[0];	\
		} *unused  __attribute__((unused));		\
    *((typeof(unused)) (ptr)) = ((typeof(*unused)) {});		\
    } while (0)

/* problem if dst is void* => noalias is array of void */
#define Memcpy(dst, src, size) do { \
    union { char tmparray[size]; typeof (*dst) noalias[0];	\
		} *unused  __attribute__((unused));		\
    *(typeof(unused)) (dst) = *(typeof(unused)) (src);		\
    } while (0)

/* Unfortunately that cannot be compiled: */
#define Memcmp(dst, src, size) ({ \
    union { char tmparray[size]; typeof (*dst) noalias[0];	\
		} *unused  __attribute__((unused));		\
    *(typeof(unused)) dst != *(typeof(unused)) src;		\
    })

/*
 * The gas macroes are not expanded if they appear after a
 * semicolon in functions memcpy,... later in this file.
 */

extern inline void _memset (void *buffer, int value, unsigned size)
  {
#if ASSEMBLY_TYPE == ASSEMBLY_NONE
  unsigned char *_buffer = (unsigned char *)buffer;
  size++;
  while (--size)
      *_buffer++ = value;
#else
  asm (
"	cld	# _memset modify %2	\n"
"	rep stosb %%al,%%es:(%%edi)	\n"
	: "+c" (size), "+D" (buffer), /* in fact %es:%edi */
	  "=m" (*memory)
	: "a" (value)
	: "cc", "memory");
#endif
  }

extern inline void _memcpy (void *dst, const void *src, unsigned size)
  {
#if ASSEMBLY_TYPE == ASSEMBLY_NONE
  unsigned char *_dst = (unsigned char *)dst;
  const unsigned char *_src = (const unsigned char *)src;
  size++;
  while (--size)
      *_dst++ = *_src++;
#else
  asm (
"	cld	# _memcpy modify %3		\n"
"	rep movsb %%ds:(%%esi),%%es:(%%edi)	\n"
	: "+c" (size), "+D" (dst), "+S" (src), "=m" (*memory)
	: "m" (*memory)
	: "cc", "memory");
#endif
  }

// __attribute__ ((pure))
extern inline int _memcmp (const void *_s1, const void *_s2, unsigned nb)
  {
  /* GCC-3.4: warning: use of cast expressions as lvalues is deprecated */
  const unsigned char *s1 = (const unsigned char *)_s1,
		      *s2 = (const unsigned char *)_s2;
#if ASSEMBLY_TYPE == ASSEMBLY_NONE
  nb++;
  while (--nb != 0) {
      if (*s1 != *s2)
	  break;
      s1++;
      s2++;
      }
#else
  asm (
"	cld	# _memcmp depends on %3		\n"
"	repe cmpsb %%es:(%%edi),%%ds:(%%esi)	\n"
	: "+c" (nb), "+S" (s1), "+D" (s2)
	: "m" (*memory)
	: "cc" );
  s1--;
  s2--;
#endif
  if (*s1 == *s2)
      return 0;
  if (*s1 > *s2)
      return 1;
    else
      return -1;
  }

extern inline int _memeql (const void *s1, const void *s2, unsigned nb)
  {
#if ASSEMBLY_TYPE == ASSEMBLY_NONE
  const unsigned char *_s1 = (const unsigned char *)s1;
  const unsigned char *_s2 = (const unsigned char *)s2;
  nb++;
  while (--nb != 0) {
      if (UNUSUAL(*_s1 != *_s2))
	  break;
      _s1++;
      _s2++;
      }
#else
  nb++;
  asm ("cld; repe cmpsb %%es:(%%edi),%%ds:(%%esi) "
	: "+c" (nb), "+S" (s1), "+D" (s2)
	: "m" (*memory));
#endif
  return !nb;
  }

extern inline void memcpy_door_open (void *dst, const void *src, unsigned size)
  {
#if ASSEMBLY_TYPE == ASSEMBLY_NONE
  unsigned char *_dst = (unsigned char *)dst;
  const unsigned char *_src = (const unsigned char *)src;
  size ++;
  while (--size)
      *_dst++ = *_src++;
#else
  asm (
"	cld	# memcpy_door_open modify %3	\n"
"	rep movsb %%ds:(%%esi),%%es:(%%edi)	\n"
	: "+c" (size), "+D" (dst), "+S" (src), "=m" (*memory)
	: "m" (*memory)
	: "cc", "memory");
#endif
  }

// __attribute__ ((pure))
extern inline const char *_strnchr (const char *str, char c, unsigned size)
  {
#if ASSEMBLY_TYPE == ASSEMBLY_NONE
  while (*str++ != c && size--) ;
#else
  asm ("cld ; repne scasb %%es:(%%edi),%%al"
	: "+c" (size), "+D" (str) /* in fact %es:%edi */
	: "a" (c), ASM_STRUCT_INPUT (*str)
	: "cc");
#endif
  return str - 1;
  }

extern inline int hexval (char c)
  {
  extern const char *const itoa_array;
  extern const unsigned itoa_len;
  const char *ptr = _strnchr (itoa_array, c, itoa_len);

  if (*ptr == c)
      return (ptr - itoa_array) & 0x0F;
  return -1;
  }

extern inline void *memcpy (void *dst, const void *src, const unsigned size)
  {
  if (__builtin_constant_p (size)) {
      if (size == 1) {
	  ((unsigned char *)dst)[0] = ((const unsigned char *)src)[0];
	  return dst;
	  }
	else if (size == 2) {
	  ((unsigned short *)dst)[0] = ((const unsigned short *)src)[0];
	  return dst;
	  }
	else if (size == 3) {
	  ((unsigned short *)dst)[0] = ((const unsigned short *)src)[0];
	  ((unsigned char *)dst)[2] = ((const unsigned char *)src)[2];
	  return dst;
	  }
	else if (size == 4) {
	  ((unsigned *)dst)[0] = ((const unsigned *)src)[0];
	  return dst;
	  }
	else if (size == 5) {
	  ((unsigned *)dst)[0] = ((const unsigned *)src)[0];
	  ((unsigned char *)dst)[4] = ((const unsigned char *)src)[4];
	  return dst;
	  }
	else if (size == 6) {
	  ((unsigned *)dst)[0] = ((const unsigned *)src)[0];
	  ((unsigned short *)dst)[2] = ((const unsigned short *)src)[2];
	  return dst;
	  }
	else if (size == 8) {
	  ((unsigned *)dst)[0] = ((const unsigned *)src)[0];
	  ((unsigned *)dst)[1] = ((const unsigned *)src)[1];
	  return dst;
	  }
      }
  _memcpy (dst, src, size);
  return dst;
  }

/* This returns !memcmp(), so not the -1/0/1 system: */
extern inline unsigned memeql (const void *dst, const void *src, const unsigned size)
  {
  if (__builtin_constant_p (size)) {
      if (size == 0) {
	  __ERROR();
	  }
	else if (size == 1) {
	  return ((const unsigned char *)dst)[0] == ((const unsigned char *)src)[0];
	  }
	else if (size == 2) {
	  return ((const unsigned short *)dst)[0] == ((const unsigned short *)src)[0];
	  }
	else if (size == 3) {
#if 1
	  return ((((const unsigned *)dst)[0] ^ ((const unsigned *)src)[0]) & 0x00FFFFFF) == 0; /* little endian */
#else
	  return ((const unsigned short *)dst)[0] == ((const unsigned short *)src)[0]
	      && ((const unsigned char *)dst)[2] == ((const unsigned char *)src)[2];
#endif
	  }
	else if (size == 4) {
	  return ((const unsigned *)dst)[0] == ((const unsigned *)src)[0];
	  }
#if 0 /* does not work (set memeql() out of line), but that is still what I am thinking: */
	else if (size <= 8) {
	  if (((const unsigned *)dst)[0] != ((const unsigned *)src)[0])
	      return 0;
	  return memeql ((const char *)dst + 4, (const char *)src + 4, size - 4);
	  }
#else
	else if (size == 5) {
	  return ((const unsigned *)dst)[0] == ((const unsigned *)src)[0]
	      && ((const unsigned char *)dst)[4] == ((const unsigned char *)src)[4];
	  }
	else if (size == 6) {
	  return ((const unsigned *)dst)[0] == ((const unsigned *)src)[0]
	      && ((const unsigned short *)dst)[2] == ((const unsigned short *)src)[2];
	  }
	else if (size == 7) {
	  return ((const unsigned *)dst)[0] == ((const unsigned *)src)[0]
#if 1
	      && ((((const unsigned *)dst)[1] ^ ((const unsigned *)src)[1]) & 0x00FFFFFF) == 0; /* little endian */
#else
	      && ((const unsigned short *)dst)[2] == ((const unsigned short *)src)[2]
	      && ((const unsigned char *)dst)[6] == ((const unsigned char *)src)[6];
#endif
	  }
	else if (size == 8) {
	  return ((const unsigned *)dst)[0] == ((const unsigned *)src)[0]
	      && ((const unsigned *)dst)[1] == ((const unsigned *)src)[1];
	  }
#endif
      }
  return _memeql (dst, src, size);
  }

extern inline void *memset (void *dst, int value, const unsigned size)
  {
  if (__builtin_constant_p (size)) {
      if (size == 1) {
	  ((unsigned char *)dst)[0] = value;
	  return dst;
	  }
	else if (size == 2) {
	  ((unsigned short *)dst)[0] = 0x0101 * (value & 0xFF);
	  return dst;
	  }
	else if (size == 3) {
	  ((unsigned short *)dst)[0] = 0x0101 * (value & 0xFF);
	  ((unsigned char *)dst)[2] = value;
	  return dst;
	  }
	else if (size == 4) {
	  ((unsigned *)dst)[0] = 0x01010101 * (value & 0xFF);
	  return dst;
	  }
	else if (size == 5) {
	  ((unsigned *)dst)[0] = 0x01010101 * (value & 0xFF);
	  ((unsigned char *)dst)[4] = value;
	  return dst;
	  }
	else if (size == 6) {
	  ((unsigned *)dst)[0] = 0x01010101 * (value & 0xFF);
	  ((unsigned short *)dst)[2] = 0x0101 * (value & 0xFF);
	  return dst;
	  }
	else if (size == 8) {
	  ((unsigned *)dst)[0] = 0x01010101 * (value & 0xFF);
	  ((unsigned *)dst)[1] = 0x01010101 * (value & 0xFF);
	  return dst;
	  }
      }
  _memset (dst, value, size);
  return dst;
  }

#if 0
// __attribute__ ((pure))
extern inline int memcmp (const void *s1, const void *s2, unsigned nb)
  {
  return _memcmp (s1, s2, nb);
  }
#else
extern inline int memcmp (const void *s1, const void *s2, unsigned nb)
  {
  return __builtin_memcmp (s1, s2, nb);
  }
#endif

/*
 * It is easy to set a register to -1, "orl $-1,%reg"
 * which do not contain a 32 bit inline constant on
 * 80x86. Unfortunately, GCC-3.0 does not do it very
 * often...
 */
#define MAXSTRING (~0)

// __attribute__ ((pure))
extern inline unsigned strlen (const char *str)
  {
  return _strnchr (str, '\0', MAXSTRING) - str;
  }

extern inline void _strncpy (char *dst, const char *src, unsigned nb)
  {
#if ASSEMBLY_TYPE == ASSEMBLY_NONE
  do {
      if (UNUSUAL(nb-- == 0)) {
	  *dst = '\0';
	  break;
	  }
      *dst++ = *src;
      } while (*src++);
#else
#if 0
  unsigned len = _strnchr (s1, '\0', nb) - s1;
#else
  unsigned len = strlen(src);
  if (len > nb)
      len = nb;
#endif
  memcpy (dst, src, len);
  dst[len] = '\0';
#endif
  }

extern inline int _strncmp (const char *s1, const char *s2, unsigned nb)
  {
#if ASSEMBLY_TYPE == ASSEMBLY_NONE
  while (*s1) {
      if (--nb == 0)
	  break;
      if (*s1 != *s2)
	  break;
      s1++; s2++;
      }
  return (*s1 > *s2)? 1 : ((*s1 < *s2)? -1 : 0);
#else
#if 0
  unsigned len = _strnchr (s1, '\0', nb) - s1 + 1;
#else
  unsigned len = strlen(s1) + 1;
  if (len > nb)
      len = nb;
#endif
  return _memcmp (s1, s2, len);
#endif
  }

extern inline const char *strchr (const char *str, char c)
  {
  while (*str) {
      if (*str == c)
	  return str;
      str++;
      }
  return 0;
  }

extern inline const char *strstr (const char *str1, const char *str2)
  {
  unsigned strlen_str2 = strlen (str2);
  if (strlen_str2 == 0)
      return str1;
  while ((str1 = strchr (str1, str2[0])) != 0)
      if (memeql (str1, str2, strlen_str2))
	  return str1;
	else
	  str1++;
  return 0;
  }

extern inline char *strcpy (char *dst, const char *src)
  {
  _strncpy (dst, src, MAXSTRING);
  return dst;
  }

extern inline char *strcat (char *dst, const char *src)
  {
  strcpy (dst + strlen (dst), src);
  return dst;
  }

extern inline char *stradd (char *dst, const char *src)
  {
  unsigned len = strlen (src);
  memcpy (dst, src, len + 1); /* do not forget '\0' */
  return dst + len; /* points to the zero char */
  }

extern inline int strncmp (const char *s1, const char *s2, unsigned nb)
  {
  return _strncmp (s1, s2, nb);
  }

extern inline int strcmp (const char *s1, const char *s2)
  {
  return strncmp (s1, s2, MAXSTRING);
  }

extern inline int strnicmp (const char *s1, const char *s2, unsigned nb)
  {
  unsigned char tmp1 = *s1, tmp2 = *s2;

  if (!tmp1)
      return (tmp2 == 0)? 0 : 1;

  while (tmp1) {
      if (tmp1 >= 'a')
	  tmp1 -= 'a' - 'A';
      if (tmp2 >= 'a')
	  tmp2 -= 'a' - 'A';
      if (--nb == 0)
	  break;
      if (tmp1 != tmp2)
	  break;
      tmp1 = *++s1;
      tmp2 = *++s2;
      }
  return (tmp1 > tmp2)? 1 : ((tmp1 < tmp2)? -1 : 0);
  }

extern inline int stricmp (const char *s1, const char *s2)
  {
  unsigned char tmp1 = *s1, tmp2 = *s2;

  if (!tmp1)
      return (tmp2 == 0)? 0 : 1;

  while (tmp1) {
      if (tmp1 >= 'a')
	  tmp1 -= 'a' - 'A';
      if (tmp2 >= 'a')
	  tmp2 -= 'a' - 'A';
      if (tmp1 != tmp2)
	  break;
      tmp1 = *++s1;
      tmp2 = *++s2;
      }
  return (tmp1 > tmp2)? 1 : ((tmp1 < tmp2)? -1 : 0);
  }

extern inline void strlwr (char *str)
  {
  if (!str)
      return;
  while (*str) {
      if (*str >= 'A' && *str <= 'Z')
	  *str += 'a' - 'A';
      str++;
      }
  }

/*
 * Few standard functions:
 */
extern inline unsigned long long rdmsr (unsigned addr)
  {
  unsigned long long result;
  /* Add volatile here to not read incompatible addr? */
  asm volatile (" rdmsr " : "=A" (result) : "c" (addr));
  return result;
  }

extern inline void wrmsr (unsigned addr, unsigned long long value)
  {
  asm (" wrmsr " : : "A" (value), "c" (addr));
  }

extern inline void disable_serial_cpu (void)
  {
  wrmsr(119, rdmsr (119) & 0xFFFFFFFFFFDFFFFFULL); /* bit 21 */
  }

extern inline unsigned long long rdpmc (unsigned addr)
  {
  unsigned long long result;
  asm (" rdpmc # non-serialising " : "=A" (result) : "c" (addr));
  return result;
  }

/*
 * Self explaining name - keep it big enough for BIOS interrupts:
 * (used in malloc and DEBUG_ADDR)
 */
#define stack_safety 0x400

/*
 * General state of Gujin, constant at C level:
 * Exception: stack limit, BoundMsg are changing with the malloced
 * memory, but it is just changed/used in assembly...
 */
extern volatile const struct state_str {
    unsigned dataseg_adr;
    unsigned codeseg_adr;
      signed short serial_port;	/* -1 for screen */
    unsigned short has_just_been_uninstalled;
#if KEEP_STRUCT_MAPPING || (SETUP & XCODE_SEGMENT)
    unsigned diskcodeseg_adr;
    volatile const gujin_param_t *gujin_param_ptr;
    volatile const bootloader2_t *uninstall_mbr_ptr;
#endif
#if KEEP_STRUCT_MAPPING || !(SETUP & BIOS_ONLY)
    unsigned char dos_running;
    unsigned char reserved[3];
#endif
    unsigned char *color_overwrite_adr;
#if KEEP_STRUCT_MAPPING || (DEBUG & DEBUG_ADDR)
    struct {
	signed  low_limit;
	signed  high_limit;
	} __attribute__ ((packed)) data_limit;
    const char *BoundMsg;
#endif
#if KEEP_STRUCT_MAPPING || (SETUP & XDATA_SEGMENT)
    unsigned xdataseg_adr;
#endif
#if KEEP_STRUCT_MAPPING || (DEBUG & DEBUG_STACK)
    struct {
	signed low_limit;
	signed high_limit;
	} __attribute__ ((packed)) stack_limit;
#endif
#if KEEP_STRUCT_MAPPING || (SETUP & XCODE_SEGMENT)
    unsigned fscodeseg_adr;
    unsigned loadcodeseg_adr;
    unsigned usercodeseg_adr;
#endif
#if KEEP_STRUCT_MAPPING || (SETUP & XSTRING_SEGMENT)
    unsigned xstring1seg_adr, xstring2seg_adr;
#endif
    unsigned reserved2[15]; /* present when xstring1seg_adr != 0 */
    } STATE;

/*
 * these are 32bit aligned, defined by linker:
 */
extern unsigned _etext[], _srodata[], _end[], _sbss[], _edata[];

#if SETUP & CODE_SEGMENT
extern unsigned char deltaseg[]; /* 8 bits aligned, defined in linker file */
#endif

#if SETUP & XCODE_SEGMENT
extern unsigned char diskcodeseg[], fscodeseg[], loadcodeseg[], usercodeseg[];
#endif

#if SETUP & XDATA_SEGMENT
extern unsigned char xdataseg[];
#endif

#if SETUP & XSTRING_SEGMENT
extern unsigned char xstring1seg[], xstring2seg[];
#endif

int printf (const char *format, ...) __attribute__ ((format (printf, 1, 2)));
awk_farcall (printf);
void print (const char *msg);
awk_farcall (print);
void puts (const char *msg);
awk_farcall (puts);
#if !(SETUP & QUITE)
int verbose_printf (const char *format, ...) __attribute__ ((format (printf, 1, 2)));
awk_farcall (verbose_printf);
void verbose_print (const char *msg);
awk_farcall (verbose_print);
void verbose_puts (const char *msg);
awk_farcall (verbose_puts);
#endif
int sprintf (char *dest, const char *format, ...) __attribute__ ((format (printf, 2, 3)));
awk_farcall (sprintf);
void before_exit (void);
awk_farcall (before_exit);
void BIOS_killme (void);
awk_farcall (BIOS_killme);

#ifdef TREAT_EXCEPTION
void restore_vectors (void);
awk_farcall (restore_vectors);
#endif
void APM_power_OFF(void);
awk_farcall (APM_power_OFF);

#ifdef TREAT_TIMER
void set_timeout (unsigned nbtick);
awk_farcall (set_timeout);
#else
#define set_timeout(X) /* */
#endif


/* This one is also used in DEBUG, in codeseg: */
char *snprtf (char *buffer, unsigned buffersize, const char *format, void **param) __attribute__ ((format (printf, 3, 0)));
awk_farcall (snprtf);

#if !(SETUP & QUITE)
extern const char *const mem_name[];
#endif

/*
 * Segment playing:
 */

#define FCT_SET_SEGREG(seg) \
    extern inline void set##seg (unsigned short val) {		\
	asm volatile (" movw %w0,%%"#seg" " : : "r" (val));	\
	}

#define FCT_GET_SEGREG(seg) \
    extern inline unsigned short get##seg (void) {		\
	unsigned short returned;				\
								\
	asm ASMREAD (" movw %%"#seg",%w0 " : "=r" (returned));	\
	return returned;					\
	}

#define FCT_UPDATE_SEG(seg) \
    extern inline void update_##seg (void) {			\
	asm volatile (" movw %%"#seg":(0),%%ax " : : : "ax");	\
	}

/*
 * Segment door playing:
 */
#define FCT_TRYDOOR(seg) \
    extern inline void trydoor_peek_##seg (void) {	\
	unsigned dummy = 0x10000;			\
	asm volatile (" movl  %%"#seg":(%0),%0 ; nop "	\
			: "+a" (dummy));		\
	}

FCT_SET_SEGREG(ds)
FCT_SET_SEGREG(es)
FCT_SET_SEGREG(fs)
FCT_SET_SEGREG(gs)
FCT_SET_SEGREG(ss)

FCT_GET_SEGREG(cs)
FCT_GET_SEGREG(ds)
FCT_GET_SEGREG(es)
FCT_GET_SEGREG(fs)
FCT_GET_SEGREG(gs)
FCT_GET_SEGREG(ss)

FCT_UPDATE_SEG(ds)
FCT_UPDATE_SEG(es)
FCT_UPDATE_SEG(fs)
FCT_UPDATE_SEG(gs)
FCT_UPDATE_SEG(ss)

FCT_TRYDOOR(ds)
FCT_TRYDOOR(es)
FCT_TRYDOOR(fs)
FCT_TRYDOOR(gs)
FCT_TRYDOOR(cs)
FCT_TRYDOOR(ss)

/*
 * Get standart regs:
 */
extern inline unsigned geteax (void)
  {
  unsigned returned;
  asm ASMREAD (" # geteax " : "=a" (returned));
  return returned;
  }

extern inline unsigned getebx (void)
  {
  unsigned returned;
  asm ASMREAD (" # getebx " : "=b" (returned));
  return returned;
  }

extern inline unsigned getecx (void)
  {
  unsigned returned;
  asm ASMREAD (" # getecx " : "=c" (returned));
  return returned;
  }

extern inline unsigned getedx (void)
  {
  unsigned returned;
  asm ASMREAD (" # getedx " : "=d" (returned));
  return returned;
  }

extern inline unsigned getesi (void)
  {
  unsigned returned;
  asm ASMREAD (" # getesi " : "=S" (returned));
  return returned;
  }

extern inline unsigned getedi (void)
  {
  unsigned returned;
  asm ASMREAD (" # getedi " : "=D" (returned));
  return returned;
  }


#define FCT_GET_REG(reg) \
    extern inline unsigned get##reg (void) {			\
	unsigned returned;					\
								\
	asm ASMREAD (" movl %%"#reg",%0 " : "=g" (returned));	\
	return returned;					\
	}

FCT_GET_REG(ebp)
FCT_GET_REG(esp)

/*
 * Intel segment %%fs is considered as non preserved,
 * and segment %%gs as preserved across calls.
 * The complete software should keep %%es == %%ds
 *
 * Everywhere where a pointer is passed as an argument
 * to a function, it is assumed that %ss == %es if the
 * address is taken from a variable in stack, or
 * %ds == %es if it is taken from a global variable.
 *
 * So here, to simplify, %ss == %ds == %es.
 *
 * "farptr" type:
 * The segment is at upper address, offset at lower address,
 * "farptr" can do address calculus: +=4; ... within 64 Kbytes.
 * If this is not accepted, one can use these
 * assembly instructions to set up %es each times:
 * (after a push, you can no more access local variables)
 *
 *  " pushl	%%es
 *    rol	$16,%%edi	# changes carry
 *    mov	%%di,%%es
 *    rol	$16,%%edi
 *    ...
 *    popl	%%es " : : "D" (myfarptr)
 *
 * Then, to call it, the parameter is a "farptr" (no strong
 * type checking) and you get it using: "data_adr(&param)"
 */

typedef unsigned farptr;

__attribute__ ((const)) extern inline unsigned
farptr2linear (farptr addr)
  {
  return (addr & 0xFFFF) + ((addr >> 16) << 4);
  }

__attribute__ ((const)) extern inline farptr
linear2farptr (unsigned addr)
  {
  /* consider segment as counter of 64 Kb, used for DMA floppy overflow tests */
  return (addr & 0xFFFF) + ((addr >> 16) << 28);
  }

__attribute__ ((const)) extern inline farptr
smallestoffset (farptr addr)
  {
  return (((addr >> 16) + (addr >> 4)) << 16) + (addr & 0x0F);
  }

__attribute__ ((const)) extern inline farptr
data_adr (const void *data)
  {
#if SETUP & BIG_MALLOC
  if ((unsigned) data > 0x10000) {
      unsigned linear = (STATE.dataseg_adr >> 12) + (unsigned) data;
      return (farptr)(((linear & 0xFFFF0) << 12) | (linear & 0x0F));
      }
    else
#endif
      return (farptr)(STATE.dataseg_adr | (unsigned) data);
  }

#if SETUP & XDATA_SEGMENT
__attribute__ ((const)) extern inline farptr
xdata_adr (const void *data)
  {
  return (farptr)(STATE.xdataseg_adr | (unsigned) data);
  }
#else
__attribute__ ((const)) extern inline farptr
xdata_adr (const void *data)
  {
  return data_adr (data);
  }
#endif

#if SETUP & XSTRING_SEGMENT
__attribute__ ((const)) extern inline farptr
xstring1_adr (const void *data)
  {
  return (farptr)(STATE.xstring1seg_adr | (unsigned) data);
  }
__attribute__ ((const)) extern inline farptr
xstring2_adr (const void *data)
  {
  return (farptr)(STATE.xstring2seg_adr | (unsigned) data);
  }
#endif

__attribute__ ((const)) extern inline farptr
stack_adr (const void *data)
  {
  /* stack is max 64 K */
  return (farptr)(STATE.dataseg_adr | (unsigned) data);
  }

__attribute__ ((const)) extern inline farptr
code_adr (volatile const void *data)
  {
  return (farptr)(STATE.codeseg_adr | (unsigned) data);
  }

#if SETUP & XCODE_SEGMENT
__attribute__ ((const)) extern inline farptr
diskcode_adr (const void *data)
  {
  return (farptr)(STATE.diskcodeseg_adr | (unsigned) data);
  }
__attribute__ ((const)) extern inline farptr
fscode_adr (const void *data)
  {
  return (farptr)(STATE.fscodeseg_adr | (unsigned) data);
  }
__attribute__ ((const)) extern inline farptr
loadcode_adr (const void *data)
  {
  return (farptr)(STATE.loadcodeseg_adr | (unsigned) data);
  }
__attribute__ ((const)) extern inline farptr
usercode_adr (const void *data)
  {
  return (farptr)(STATE.usercodeseg_adr | (unsigned) data);
  }
#endif

__attribute__ ((const)) extern inline unsigned
linear2dsrelative (unsigned addr)
  {
  return addr - (STATE.dataseg_adr >> 12);
  }

__attribute__ ((const)) extern inline unsigned
dsrelative2linear (unsigned addr)
  {
  return addr + (STATE.dataseg_adr >> 12);
  }

__attribute__ ((const)) extern inline unsigned
farptr2dsrelative (farptr addr)
  {
  return linear2dsrelative (farptr2linear (addr));
  }

extern inline void fmemcpy (farptr dst, farptr src, unsigned short size)
  {
  asm (
"	pushw	%%es						\n"
"	pushl	%%esi						\n"
"	popw	%%si						\n"
"	popw	%%fs						\n"
"	pushl	%%edi						\n"
"	popw	%%di						\n"
"	popw	%%es						\n"
"	cld		# fmemcpy modify %3			\n"
"	rep movsb %%fs:(%%si),%%es:(%%di)	# no macro	\n"
"	popw	%%es						\n"
	: "+S" (src), "+D" (dst), "+c" (size), "=m" (*memory)
	: "m" (*memory)
	: "cc", "memory"
	);
  }

extern inline void fmemset (farptr dst, char val, unsigned short size)
  {
  asm (
"	pushw	%%es						\n"
"	pushl	%%edi						\n"
"	popw	%%di						\n"
"	popw	%%es						\n"
"	cld		# fmemset modify %2			\n"
"	rep stosb %%al,%%es:(%%di)		# no macro	\n"
"	popw	%%es						\n"
	: "+c" (size), "+D" (dst), "=m" (*memory)
	: "a" (val)
	: "cc", "memory"
	);
  }

/* Copy from the stack to a farptr: */
extern inline void smemcpy (farptr dst, const void *src, unsigned short size)
  {
  asm (
"	pushw	%%es						\n"
"	pushl	%%edi						\n"
"	popw	%%di						\n"
"	popw	%%es						\n"
"	cld		# smemcpy modify %3			\n"
#if (SETUP & BIG_MALLOC)
"	rep movsb %%ss:(%%si),%%es:(%%di)	# no macro	\n"
#else
"	rep movsb %%ds:(%%si),%%es:(%%di)	# no macro	\n"
#endif
"	popw	%%es						\n"
	: "+S" (src), "+D" (dst), "+c" (size), "=m" (*memory)
	: "m" (*memory)
	: "cc", "memory"
	);
  }

/* See asm() documentation for that: */
//#define ASM_MEM_CLOBBER(ptr, nbbytes) ({ struct { char x[nbbytes]; } *p = (void *)ptr ; *p; })
#define ASM_MEM_CLOBBER(ptr, nbbytes)	*memory

/* Copy from a farptr to the current data area: */
extern inline void lmemcpy (void *dst, farptr src, unsigned short size)
  {
  unsigned dummy;
  asm (
"	pushl	%4						\n"
"	popw	%%si						\n"
"	popw	%%fs						\n"
"	cld		# lmemcpy modify %3			\n"
"	rep movsb %%fs:(%%si),%%es:(%%di)	# no macro	\n"
	: "=S" (dummy), "+D" (dst), "+c" (size), "=m" (ASM_MEM_CLOBBER(dst, size))
	: "g" (src)
	: "cc", "memory"
	);
  }

/* Copy from a code pointer to the current data area:(take care of EXTRASEG!) */
extern inline void cmemcpy (void *dst, volatile const void *src, unsigned short size)
  {
  asm (
"	cld		# cmemcpy modify %3			\n"
"	rep movsb %%cs:(%%si),%%es:(%%di)	# no macro	\n"
	: "+S" (src), "+D" (dst), "+c" (size), "=m" (ASM_MEM_CLOBBER(dst, size)) : : "memory"
	);
  }

/*
 * This function copies memory from either a farptr or
 * the current data segment to either a farptr or
 * the current data segment.
 * If the source is in the data segment, the upper part
 * of the farptr is null - same for destination.
 */
extern inline void
farmemcpy (unsigned char *dst, const unsigned char *src, unsigned short size)
  {
  asm (
"	pushw	%%ds		\n"
"	pushw	%%es		\n"
"	test	%4,%%esi	\n"
"	jz	1f		\n"
"	pushl	%%esi		\n"
"	popw	%%si		\n"
"	popw	%%ds		\n"
"	1:			\n"
"	test	%4,%%edi	\n"
"	jz	1f		\n"
"	pushl	%%edi		\n"
"	popw	%%di		\n"
"	popw	%%es		\n"
"	1:			\n"
"	cld	# farmemcpy modify %3	\n"
"	rep movsb %%ds:(%%si),%%es:(%%di) # no macro, ignore (e[sd]i >> 16)\n"
"	popw	%%es		\n"
"	popw	%%ds		\n"
	: "+S" (src), "+D" (dst), "+c" (size), "=m" (ASM_MEM_CLOBBER(dst, size))
	: "ri" (0xFFFF0000), "m" (*memory)
	: "cc", "memory");
}


/*
 * more standart stuff:
 */
extern inline void short_delay (unsigned count)
  {
  asm volatile (
"	loopl .		\n"
	: "+c" (count));
  }

extern inline void aligned_short_delay (unsigned count)
  {
  asm volatile (
"	.p2align 2	\n"
"addr_aligned:		\n"
"	loopl .		\n"
	: "+c" (count));
  }

extern inline void unaligned_short_delay (unsigned count)
  {
  asm volatile (
"	.p2align 2	\n"
"	nop		\n"
"addr_unaligned:	\n"
"	loopl .		\n"
	: "+c" (count));
  }

extern inline void disabled_aligned_short_delay (unsigned count)
  {
  asm volatile (
"	.p2align 2	\n"
"addr_disabled:		\n"
"	loopl .		\n"
	: "+c" (count));
  }

extern inline void very_short_delay (unsigned short count)
  {
  asm volatile (
"	loopw .		\n"
	: "+c" (count));
  }

extern inline void aligned_very_short_delay (unsigned short count)
  {
  asm volatile (
"	.p2align 2	\n"
"	loopw .		\n"
	: "+c" (count));
  }

extern inline void unaligned_very_short_delay (unsigned short count)
  {
  asm volatile (
"	.p2align 2	\n"
"	nop		\n"
"	loopw .		\n"
	: "+c" (count));
  }

extern inline unsigned long long measure_short_delay (unsigned count)
  {
  unsigned long long measure1 , measure2;
  unsigned dummy;

  /* Measure overhead */
  asm volatile (
"	nop			\n"
"	rdtsc			\n"
"	pushl	%%edx		\n"
"	pushl	%%eax		\n"
"	rdtsc			\n"
"	popl	%%ecx		\n"
"	subl	%%ecx,%%eax	\n"
"	popl	%%ecx		\n"
"	sbbl	%%ecx,%%edx	\n"
"				\n"
	: "=A" (measure1), "=c" (dummy));

  /* Measure exact (rdtsc is not serialising, but...) */
  asm volatile (
"	nop			\n"
"	rdtsc			\n"
"	pushl	%%edx		\n"
"	pushl	%%eax		\n"
"	loopl .			\n"
"	rdtsc			\n"
"	popl	%%ecx		\n"
"	subl	%%ecx,%%eax	\n"
"	popl	%%ecx		\n"
"	sbbl	%%ecx,%%edx	\n"
"				\n"
	: "=A" (measure2), "=c" (dummy)
	: "c" (count));
  return measure2 - measure1;
  }

extern inline unsigned char inb (unsigned short port)
  {
  unsigned char result;
  asm ASMREAD (" inb %1,%0 " : "=a" (result) : "dN" (port));
  return result;
  }

extern inline unsigned short inw (unsigned short port)
  {
  unsigned short result;
  asm ASMREAD (" inw %1,%0 " : "=a" (result) : "dN" (port));
  return result;
  }

extern inline unsigned inl (unsigned short port)
  {
  unsigned result;
  asm ASMREAD (" inl %1,%0 " : "=a" (result) : "dN" (port));
  return result;
  }

extern inline void outb (unsigned short port, unsigned char data)
  {
  asm volatile (" outb %0,%1 " : : "a" (data), "dN" (port));
  }

extern inline void outw (unsigned short port, unsigned short data)
  {
  asm volatile (" outw %0,%1 " : : "a" (data), "dN" (port));
  }

extern inline void outl (unsigned short port, unsigned data)
  {
  asm volatile (" outl %0,%1 " : : "a" (data), "dN" (port));
  }

extern inline unsigned char getnvram (unsigned char addr)
  {
  outb (0x70, addr & ~0x80); /* bit 7 cleared: NMI enabled */
  short_delay (1000); /* 1000 assembly instructions */
  return inb (0x71);
  }

extern inline void reset_387 (void)
  {
  outb (0xF0, 0);	/* reset copro */
  outb (0xF1, 0);	/* reset copro part 2 */
  }

extern inline void disable_nmi (void)
  {
  outb (0x70, 0x80);
  short_delay (1000);
  inb (0x71);
  }

extern inline void enable_nmi (void)
  {
  outb (0x70, 0x00);
  short_delay (1000);
  inb (0x71);
  }

extern inline unsigned getflags (void)
  {
  unsigned returned;

  // volatile here: see has_cpuid() !
  asm volatile (" pushfl ; popl %0 " : "=r" (returned) );
  return returned;
  }

extern inline void setflags (unsigned value)
  {
  asm volatile (" pushl %0 ; popfl " : : "r" (value) );
  }

extern inline void setTFflag (void)
  {
  setflags (getflags () | 0x100);
  }

extern inline void clearTFflag (void)
  {
  setflags (getflags () & ~0x100);
  }

extern inline unsigned short getsw (void)
  {
  unsigned short returned;

  asm ASMREAD (" smsw %0 " : "=r" (returned) );
  return returned;
  }

extern inline void setsw (unsigned short SW)
  {
  asm volatile (" lmsw %0 " : : "r" (SW) );
  }

extern inline void disable_interrupt (void)
  {
  asm volatile (" cli ");
  }

extern inline void enable_interrupt (void)
  {
  asm volatile (" sti ");
  }

extern inline unsigned interrupt_enabled (void)
  {
  return (getflags() & 0x200);
  }

/*
 * Few inline functions:
 */

extern inline unsigned long long rdtsc (void)
  {
  unsigned long long returned;

  asm volatile (" rdtsc " : "=A" (returned) ); /* real volatile! */
  return returned;
  }

/*
 * WARNING: Intel documented.
 *  Get a DIVIDE ERROR if:
 *  If the source operand (divisor) is 0.
 *  If the quotient is too large for the designated register.
 * But we cannot call __umoddi3/__udivdi3 library function...
 */
/* NOT "__attribute__ ((const))" because  modify *remainder !! */
extern inline unsigned
ull_div_ul (unsigned long long ull, unsigned ul, unsigned *remainder)
  {
  unsigned result;

  asm (" divl %3 " : "=a" (result), "=d" (*remainder) : "A" (ull), "rm" (ul) : "cc" );

  return result;
  }

__attribute__ ((const)) extern inline unsigned long long
ul_mul_ul (unsigned ul1, unsigned ul2)
  {
  unsigned long long ull;

  asm (" mull %2 " : "=A" (ull) : "a" (ul1), "rm" (ul2) : "cc" );

  return ull;
  }

/*
 * Functions I cannot test
 */
__attribute__ ((const)) extern inline unsigned char cpu_is_cyrix (void)
  {
  unsigned short ax;

  asm (
 "	sahf	# i.e. 0	\n"
 "	div	%%cl		\n"
 "	lahf			\n"
	: "=a" (ax) : "a" (5), "c" (2));
  return ((ax >> 8) != 2);
  }

extern inline unsigned char cyrix_get_register (unsigned char addr)
  {
  outb (0x22, addr);
  return inb(0x23);
  }

extern inline void cyrix_set_register (unsigned char addr, unsigned char val)
  {
  outb (0x22, addr);
  outb (0x23, val);
  }

extern inline void cyrix_enable_cpuid(void)
  {
  unsigned char init_c3;

  disable_interrupt ();
  init_c3 = cyrix_get_register (0xc3);
  cyrix_set_register (0xc3, 0x10 | (init_c3 & 0xF)); /* enable config access */
  cyrix_set_register (0xe8, cyrix_get_register (0xe8) | 0x80); /* enable CPUID */
  if ((cyrix_get_register (0xfe) & 0xf0) == 0x30) /* 6x86 */
	cyrix_set_register (0xe9, cyrix_get_register (0xe9) & 0xfd); /* SLOP bug */
  cyrix_set_register (0xc3, init_c3);
  enable_interrupt ();
  }

/*
 * Processor stuff:
 */
__attribute__ ((const)) extern inline unsigned is_80486 (void)
  {
  unsigned returned, flags = getflags(),
	   mask = 1 << 18;	/* AC, bit 18 */

  setflags (flags ^ mask);
  returned = ((getflags() & mask) != (flags & mask));
  setflags (flags & ~mask);
  return returned;
  }

extern inline unsigned has_cpuid (void)
  {
  unsigned returned, flags = getflags(), mask = 1 << 21; /* ID, bit 21 */

  setflags (flags ^ mask);
  returned = (getflags() != flags);
  setflags (flags);
  return returned;
  }

union cpuid_string {
    struct {
	unsigned str1, str2, str3;
	} number;
    char string[12];
    };

extern inline int cpuid0 (union cpuid_string *result)
  {
  int returned; /* the max CPUID input value, 2 for Pentium+ */

  asm ( " cpuid "
	: "=a" (returned),
	  "=b" (result->number.str1),
	  "=d" (result->number.str2),
	  "=c" (result->number.str3)
	: "a" (0));
  return returned;
  }

struct cpuid_type_str {
    unsigned	setpping: 4;
    unsigned	model	: 4;
    unsigned	family	: 4;
    unsigned	type	: 2;
    unsigned	unused4	: 2;
    unsigned	extended_model	: 4;
    unsigned	extended_family : 8;
    unsigned	unused5	: 4;
    } __attribute__ ((packed));

struct cpuid_flags_str {
    unsigned	fpu	: 1;
    unsigned	vme	: 1;
    unsigned	de	: 1;
    unsigned	pse	: 1;
    unsigned	tsc	: 1;
    unsigned	msr	: 1;
    unsigned	pae	: 1;
    unsigned	mce	: 1;
    unsigned	cx8	: 1;
    unsigned	apic	: 1;
    unsigned	unused3	: 1;
    unsigned	sep	: 1;
    unsigned	mtrr	: 1;
    unsigned	pge	: 1;
    unsigned	mca	: 1;
    unsigned	cmov	: 1;
    unsigned	pat	: 1;
    unsigned	pse36	: 1;
    unsigned	psn	: 1;
    unsigned	clfl	: 1;
    unsigned	unused2	: 1;
    unsigned	dtes	: 1;
    unsigned	acpi	: 1;
    unsigned	mmx	: 1;
    unsigned	fxsr	: 1;
    unsigned	sse	: 1;
    unsigned	sse2	: 1;
    unsigned	ss	: 1;
    unsigned	htt	: 1;
    unsigned	tm	: 1;
    unsigned	ia64	: 1;
    unsigned	unused1	: 1;

    unsigned	mapping_to_define;
    } __attribute__ ((packed));

struct cpuid_brand_str {
    unsigned char brand_id;
    unsigned char clflush_chunk;
    unsigned short apic_id;
    } __attribute__ ((packed));

extern inline struct cpuid_type_str
cpuid1 (struct cpuid_flags_str *flags, struct cpuid_brand_str *brand)
  {
  struct cpuid_type_str returned;
  struct cpuid_flags_str tmp_flags;
  struct cpuid_brand_str tmp_brand;

  asm (
"	cpuid			\n"
"	xchgl	%%eax,%%ecx	\n"
"	xchgl	%%eax,%%edx	\n"
	: "=A" (tmp_flags), "=b" (tmp_brand), "=c" (returned)
	: "a" (1)
	);
  if (flags)
      *flags = tmp_flags;
  if (brand)
      *brand = tmp_brand;
  return returned;
  }

/*
 * cpuid_string: AuthenticAMD or TransmetaCPU,
 * reserved for other processors.
 */
extern inline unsigned cpuid8000 (union cpuid_string *result)
  {
  int returned;

  asm ( " cpuid "
	: "=a" (returned),
	  "=b" (result->number.str1),
	  "=d" (result->number.str2),
	  "=c" (result->number.str3)
	: "a" (0x80000000));
  return returned;
  }

struct cpuid_amd_type_str {
    unsigned	setpping: 4;
    unsigned	model	: 4;
    unsigned	family	: 4;
    unsigned	unused4	: 20;
    } __attribute__ ((packed));

struct cpuid_amd_flags_str {
    unsigned	fpu	: 1;
    unsigned	vme	: 1;
    unsigned	de	: 1;
    unsigned	pse	: 1;
    unsigned	tsc	: 1;
    unsigned	msr	: 1;
    unsigned	pae	: 1;
    unsigned	mce	: 1;
    unsigned	cx8	: 1;
    unsigned	apic	: 1;
    unsigned	unused3	: 1;
    unsigned	sep	: 1;
    unsigned	mtrr	: 1;
    unsigned	pge	: 1;
    unsigned	mca	: 1;
    unsigned	cmov	: 1;
    unsigned	pat	: 1; /* or FCMOV */
    unsigned	pse36	: 1;
    unsigned	unused2	: 4;
    unsigned	mmxplus	: 1;
    unsigned	mmx	: 1;
    unsigned	fxsr	: 1; /* or emmx, Cyrix */
    unsigned	unused1	: 4;
    unsigned	lm	: 1;
    unsigned	E3Dnow	: 1; /* Extended 3DNow!+ */
    unsigned	S3Dnow	: 1; /* Standard 3DNow! */
    } __attribute__ ((packed));

extern inline struct cpuid_amd_type_str
cpuid8001 (struct cpuid_amd_flags_str *flags)
  {
  struct cpuid_amd_type_str returned;
  struct cpuid_amd_flags_str tmp_flags;
  unsigned reserved1, reserved2;

  asm (" cpuid "
	: "=a" (returned), "=d" (tmp_flags), "=b" (reserved1), "=c" (reserved2)
	: "a" (0x80000001)
	);
  if (flags)
      *flags = tmp_flags;
  return returned;
  }

union cpuid_extended_string {
    struct {
	unsigned str1, str2, str3, str4;
	unsigned str5, str6, str7, str8;
	unsigned str9, str10, str11, str12;
	} number;
    char string[3 * 4 * 4];
    };

extern inline void cpuid_estring (union cpuid_extended_string *result)
  {
  asm ( " cpuid "
	: "=a" (result->number.str1),
	  "=b" (result->number.str2),
	  "=c" (result->number.str3),
	  "=d" (result->number.str4)
	: "a" (0x80000002));
  asm ( " cpuid "
	: "=a" (result->number.str5),
	  "=b" (result->number.str6),
	  "=c" (result->number.str7),
	  "=d" (result->number.str8)
	: "a" (0x80000003));
  asm ( " cpuid "
	: "=a" (result->number.str9),
	  "=b" (result->number.str10),
	  "=c" (result->number.str11),
	  "=d" (result->number.str12)
	: "a" (0x80000004));
  }

extern inline void cool_processor (void)
  {
  asm (" rep\n nop ");
  }

extern inline void halt_processor (void)
  {
  asm (" hlt ");
  }

/**
 ** Peek and Poke: Are you old enough to remember BASIC?
 **/

/*
 * For A20 opening, before peekw can be used because
 * the segment doors are still not opened.
 * (needed at least when ASSEMBLY_DSES is defined).
 */

/* This produces bigger code:
extern inline unsigned char peekb (farptr adr)
  {
  unsigned char returned;
  if (__builtin_constant_p (adr)) {
      setfs (adr >> 16);

      asm ASMREAD (" movb %%fs:%1,%0 "
	: "=q" (returned) : "m" (*(char *)(adr & 0xFFFF)));
      }
    else {
#if 0
      asm ASMREAD (
"	lfs	%1,%0		\n"
"	movb	%%fs:(%0),%0	\n"
	: "=BbSD" (returned) : "m" (adr));
#else
      asm ASMREAD (
"	roll	$16,%1		\n"
"	movw	%w1,%%fs	\n"
"	roll	$16,%1		\n"
"	movb	%%fs:(%w1),%0	\n"
	: "=q" (returned) : "bSDB" (adr) : "cc");
#endif
      }
  return returned;
  }
*/

extern inline unsigned peekb_doorclosed (farptr adr)
  {
  unsigned returned;
  unsigned short scratch;

  asm (
"	pushl	%2		\n"
"	popw	%w0		\n"
"	popw	%%fs		\n"
"	movzbl	%%fs:(%w0),%1	\n"
	: "=bSDB" (scratch), "=r" (returned)
	: "g" (adr), "m" (*memory)
	);
  return returned;
  }

extern inline unsigned peekw_doorclosed (farptr adr)
  {
  unsigned returned;
  unsigned short scratch;

  asm (
"	pushl	%2		\n"
"	popw	%w0		\n"
"	popw	%%fs		\n"
"	movzwl	%%fs:(%w0),%1	\n"
	: "=bSDB" (scratch), "=r" (returned)
	: "g" (adr), "m" (*memory)
	);
  return returned;
  }

extern inline unsigned peekl_doorclosed (farptr adr)
  {
  unsigned returned;
  unsigned short scratch;

  asm (
"	pushl	%2		\n"
"	popw	%w0		\n"
"	popw	%%fs		\n"
"	movl	%%fs:(%w0),%1	\n"
	: "=bSDB" (scratch), "=r" (returned)
	: "g" (adr), "m" (*memory)
	);
  return returned;
  }

extern inline unsigned long long peekll_doorclosed (farptr adr)
  {
  unsigned long long returned;
  unsigned short scratch;

  asm (
"	pushl	%2			\n"
"	popw	%w0			\n"
"	popw	%%fs			\n"
"	movl	%%fs:(%w0),%%eax	\n"
"	movl	%%fs:4(%w0),%%edx	\n"
	: "=bSDB" (scratch), "=A" (returned)
	: "g" (adr), "m" (*memory)
	);
  return returned;
  }

extern inline void pokeb_doorclosed (farptr adr, unsigned char value)
  {
  unsigned short scratch;
  asm (
"	pushl	%3		\n"
"	popw	%w0		\n"
"	popw	%%fs		\n"
"	movb	%2,%%fs:(%w0)	\n"
	: "=&bSDB" (scratch), "=m" (*memory)
	: "qi" (value), "g" (adr)
	);
  }

extern inline void pokew_doorclosed (farptr adr, unsigned short value)
  {
  unsigned short scratch;
  asm (
"	pushl	%3		\n"
"	popw	%w0		\n"
"	popw	%%fs		\n"
"	movw	%2,%%fs:(%w0)	\n"
	: "=&bSDB" (scratch), "=m" (*memory)
	: "qi" (value), "g" (adr)
	);
  }

extern inline void pokel_doorclosed (farptr adr, unsigned value)
  {
  unsigned short scratch;
  asm (
"	pushl	%3		\n"
"	popw	%w0		\n"
"	popw	%%fs		\n"
"	movl	%2,%%fs:(%w0)	\n"
	: "=&bSDB" (scratch), "=m" (*memory)
	: "ri" (value), "g" (adr)
	);
  }

extern inline void pokell_doorclosed (farptr adr, unsigned long long value)
  {
  unsigned short scratch;
  asm (
"	pushl	%3			\n"
"	popw	%w0			\n"
"	popw	%%fs			\n"
"	movl	%%eax,%%fs:(%w0)	\n"
"	movl	%%edx,%%fs:4(%w0)	\n"
	: "=&bSDB" (scratch), "=m" (*memory)
	: "A" (value), "g" (adr)
	);
  }

extern inline unsigned xchl_doorclosed (farptr adr, unsigned value)
  {
  unsigned short scratch;

  asm (
"	pushl	%4		\n"
"	popw	%w0		\n"
"	popw	%%fs		\n"
"	xchgl	%%fs:(%w0),%1	\n"
	: "=bSDB" (scratch), "=r" (value), "=m" (*memory)
	: "1" (value), "g" (adr)
	);
  return value;
  }

extern inline void ss_pokeb (void *offset, unsigned char value) {
  asm volatile (" movb	%0,%%ss:%a1 " : : "qi" (value), "p" ((unsigned char *)offset) );
  }
extern inline void ss_pokew (void *offset, unsigned short value) {
  asm volatile (" movw	%0,%%ss:%a1 " : : "ri" (value), "p" ((unsigned short *)offset) );
  }
extern inline void ss_pokel (void *offset, unsigned value) {
  asm volatile (" movl	%0,%%ss:%a1 " : : "ri" (value), "p" ((unsigned *)offset) );
  }

extern inline unsigned ss_peekb (const void *offset) {
  unsigned returned;
  asm volatile (" movzbl	%%ss:%a1,%0 " : "=r" (returned) : "p" ((const unsigned char *)offset));
  return returned;
  }
extern inline unsigned ss_peekw (const void *offset) {
  unsigned returned;
  asm volatile (" movzwl	%%ss:%a1,%0 " : "=r" (returned) : "p" ((const unsigned short *)offset));
  return returned;
  }
extern inline unsigned ss_peekl (const void *offset) {
  unsigned returned;
  asm volatile (" movl	%%ss:%a1,%0 " : "=r" (returned) : "p" ((const unsigned *)offset));
  return returned;
  }

/* For gs_*(), to test 4Gb segments, use "r" and not "p" because:
	library.h:1895: Warning: 0000000000100500 shortened to 0000000000000500 */
extern inline void gs_pokel (void *offset, unsigned value) {
  asm volatile (" movl	%0,%%gs:%a1 " : : "ri" (value), "r" ((unsigned *)offset) );
  }
extern inline unsigned gs_peekl (const void *offset) {
  unsigned returned;
  asm volatile (" movl	%%gs:%a1,%0 " : "=r" (returned) : "r" ((const unsigned *)offset));
  return returned;
  }

extern inline void cs_pokeb (void *offset, unsigned char value) {
  asm volatile (" movb	%0,%%cs:%a1 " : : "qi" (value), "p" ((unsigned char *)offset) );
  }
extern inline void cs_pokew (void *offset, unsigned short value) {
  asm volatile (" movw	%0,%%cs:%a1 " : : "ri" (value), "p" ((unsigned short *)offset) );
  }
extern inline void cs_pokel (void *offset, unsigned value) {
  asm volatile (" movl	%0,%%cs:%a1 " : : "ri" (value), "p" ((unsigned *)offset) );
  }

extern inline unsigned cs_peekb (volatile const void *offset) {
  unsigned returned;
  asm volatile (" movzbl	%%cs:%a1,%0 " : "=r" (returned) : "p" ((const unsigned char *)offset));
  return returned;
  }
extern inline unsigned cs_peekw (volatile const void *offset) {
  unsigned returned;
  asm volatile (" movzwl	%%cs:%a1,%0 " : "=r" (returned) : "p" ((const unsigned short *)offset));
  return returned;
  }
extern inline unsigned cs_peekl (volatile const void *offset) {
  unsigned returned;
  asm volatile (" movl	%%cs:%a1,%0 " : "=r" (returned) : "p" ((const unsigned *)offset));
  return returned;
  }

extern inline unsigned codeseg_peekb (const void *offset)
  {
#ifdef CALLING_FROM_XCODESEG
  return peekb_doorclosed (code_adr (offset));
#else
  unsigned returned;

  asm (" movzbl	%%cs:%a1,%0 " : "=r" (returned) : "p" ((const unsigned char *)offset));
  return returned;
#endif
  }

extern inline unsigned codeseg_peekw (volatile const void *offset)
  {
#ifdef CALLING_FROM_XCODESEG
  return peekw_doorclosed (code_adr (offset));
#else
  unsigned returned;

  asm (" movzwl	%%cs:%a1,%0 " : "=r" (returned) : "p" ((const unsigned short *)offset));
  return returned;
#endif
  }

extern inline unsigned codeseg_peekl (const void *offset)
  {
#ifdef CALLING_FROM_XCODESEG
  return peekl_doorclosed (code_adr (offset));
#else
  unsigned returned;

  asm (" movl	%%cs:%a1,%0 " : "=r" (returned) : "p" ((const unsigned *)offset));
  return returned;
#endif
  }

extern inline void codeseg_pokeb (void *offset, unsigned char value)
  {
#ifdef CALLING_FROM_XCODESEG
  pokeb_doorclosed (code_adr (offset), value);
#else
  asm (" movb	%0,%%cs:%a1 " : : "qi" (value), "p" ((unsigned char *)offset));
#endif
  }

extern inline void codeseg_pokew (void *offset, unsigned short value)
  {
#ifdef CALLING_FROM_XCODESEG
  pokew_doorclosed (code_adr (offset), value);
#else
  asm (" movw	%0,%%cs:%a1 " : : "qi" (value), "p" ((unsigned short *)offset));
#endif
  }

extern inline void codeseg_pokel (void *offset, unsigned value)
  {
#ifdef CALLING_FROM_XCODESEG
  pokel_doorclosed (code_adr (offset), value);
#else
  asm (" movl	%0,%%cs:%a1 " : : "ri" (value), "p" ((unsigned *)offset));
#endif
  }

/*
 * The generic ones:
 */

//extern inline unsigned char peekb (farptr adr)
extern inline unsigned peekb (farptr adr)
  {
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
  return peekb_doorclosed (adr);
#else
  return *(unsigned char *)(farptr2dsrelative (adr));
#endif /* ASSEMBLY_DSES */
  }

//extern inline unsigned short peekw (farptr adr)
extern inline unsigned peekw (farptr adr)
{
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
  return peekw_doorclosed (adr);
#else
  return *(unsigned short *)(farptr2dsrelative (adr));
#endif
  }

extern inline unsigned peekl (farptr adr)
  {
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
  return peekl_doorclosed (adr);
#else
  return *(unsigned *)(farptr2dsrelative (adr));
#endif
  }

extern inline unsigned long long peekll (farptr adr)
  {
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
  return peekll_doorclosed (adr);
#else
  return *(unsigned long long *)(farptr2dsrelative (adr));
#endif
  }

extern inline void pokeb (farptr adr, unsigned char value)
  {
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
  pokeb_doorclosed (adr, value);
#else
  *(unsigned char *)(farptr2dsrelative (adr)) = value;
#endif
  }

extern inline void pokew (farptr adr, unsigned short value)
  {
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
  pokew_doorclosed (adr, value);
#else
  *(unsigned short *)(farptr2dsrelative (adr)) = value;
#endif
  }

extern inline void pokel (farptr adr, unsigned value)
  {
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
  pokel_doorclosed (adr, value);
#else
  *(unsigned *)(farptr2dsrelative (adr)) = value;
#endif
  }

extern inline void pokell (farptr adr, unsigned long long value)
  {
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
  pokell_doorclosed (adr, value);
#else
  *(unsigned long long *)(farptr2dsrelative (adr)) = value;
#endif
  }

extern inline unsigned xchgl (farptr adr, unsigned value)
  {
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
  return xchl_doorclosed (adr, value);
#else
  unsigned tmp = *(unsigned *)(farptr2dsrelative (adr));

  *(unsigned *)(farptr2dsrelative (adr)) = value;
  return tmp;
#endif
  }

/*
 * setjmp/ljmp:
 */
typedef struct {
    unsigned esi, edi, ebp, ebx;
    unsigned short gs, fs, es, ds;
    unsigned short flags, ip, cs, sp, ss;
    } __attribute__ ((packed)) setjmp_buf;

// __attribute__ ((noreturn))
extern inline void longjmp (setjmp_buf *ptr)
  {
  asm volatile (
"	cli						\n"
"	movw	%%cs,%%dx				\n"
#if SETUP & CODE_SEGMENT
"	addw	$deltaseg,%%dx				\n"
#endif
"	movw	%%dx,%%ss				\n"
"	movzwl	%%ax,%%esp				\n"
"	popl	%%esi					\n"
"	popl	%%edi					\n"
"	popl	%%ebp					\n"
"	popl	%%ebx					\n"
"	popw	%%gs					\n"
"	popw	%%fs					\n"
"	popw	%%es					\n"
"	popw	%%ds					\n"
"	popw	%%dx					\n"
"	popl	%%eax					\n"
"	lss	(%%esp),%%sp				\n"
"	pushw	%%dx					\n"
"	pushl	%%eax		# %%eax not null	\n"
"	iretw						\n"
	: "+a" (ptr) : : "ecx" , "edx", "memory", "cc");
  }

// __attribute__ ((returns_twice))
extern inline unsigned setjmp (setjmp_buf *ptr)
  {
  asm volatile (
"	movl	%%esi,(%%eax)				\n"
"	movl	%%edi,4(%%eax)				\n"
"	movl	%%ebp,8(%%eax)				\n"
"	movl	%%ebx,12(%%eax)				\n"
"	movw	%%gs,16(%%eax)				\n"
"	movw	%%fs,18(%%eax)				\n"
"	movw	%%es,20(%%eax)				\n"
"	movw	%%ds,22(%%eax)				\n"
"	pushfw						\n"
"	popw	%%dx					\n"
"	movw	%%dx,24(%%eax)				\n"
"	movw	$setjump_%=,26(%%eax)			\n"
"	movw	%%cs,28(%%eax)				\n"
"	movw	%%sp,30(%%eax)				\n"
"	movw	%%ss,32(%%eax)				\n"
"	xor	%%eax,%%eax	# %%eax null		\n"
"	setjump_%=:					\n"
	: "+a" (ptr)
	: : /* if eax != 0 */ "ecx", "edx", "memory");
  return (unsigned) ptr;
  }

/*
 * This one has to be inline, else will not work.
 */

#define shit_handler(active, nbsecond) ({ \
  extern setjmp_buf setjump_buffer;                     \
  extern volatile unsigned setjump_key;                 \
  unsigned returned;                                    \
							\
  if (!active || setjmp (&setjump_buffer) != 0) {	\
      set_timeout (0);					\
      setjump_key = 0;					\
      returned = 0xFFFFFFFFU;				\
      }							\
    else {						\
      setjump_key = 0x45746965;				\
      if (nbsecond)					\
	  set_timeout (nbsecond * 18);			\
      returned = 0;					\
      }							\
  /*return*/ returned;					\
  })

/*
 * Basic protected mode segment description:
 */

typedef struct {
    unsigned	limit : 16;
    unsigned	base : 24;
    unsigned	accessed : 1;			/* if 1 */
    unsigned	writable : 1;			/* if 1 */
    unsigned	expansion_direction : 1;	/* down if 1 */
    unsigned	cste_10 : 2;
    unsigned	dpl : 2;
    unsigned	present : 1;
    unsigned	limit_msb : 4;
    unsigned	available : 1;
    unsigned	cste_0 : 1;
    unsigned	big : 1;
    unsigned	granularity : 1;
    unsigned	base_msb : 8;
    } __attribute__ ((packed)) dataseg_t;

typedef struct {
    unsigned	limit : 16;
    unsigned	base  : 24;
    unsigned	accessed : 1;
    unsigned	readable : 1;
    unsigned	conforming : 1;
    unsigned	cste_11 : 2;
    unsigned	dpl : 2;
    unsigned	present : 1;
    unsigned	limit_msb : 4;
    unsigned	available : 1;
    unsigned	cste_0 : 1;
    unsigned	deflt : 1;
    unsigned	granularity : 1;
    unsigned	base_msb : 8;
    } __attribute__ ((packed)) codeseg_t;

typedef struct {
    unsigned	limit : 16;
    unsigned	base  : 24;
    enum { reserved0 = 0,
	TSS_16bit_avail, LDT, TSS_16bit_busy, callgate_16bit,
	taskgate, interruptgate_16bit, trapgate_16bit,
	reserved8,
	TSS_32bit_avail, reserved10, TSS_32bit_busy, callgate_32bit,
	reserved13, interruptgate_32bit, trapgate_32bit
	} type : 4;
    unsigned	cste_0 : 1;
    unsigned	dpl : 2;
    unsigned	present : 1;
    unsigned	limit_msb : 4;
    unsigned	reserved : 3;
    unsigned	granularity : 1;
    unsigned	base_msb : 8;
    } __attribute__ ((packed)) systemseg_t;

typedef union {
    dataseg_t	data;
    codeseg_t	code;
    systemseg_t	system;
    } seg_t;

typedef struct {
    unsigned short	previous_task_link;
    unsigned short	reserved1;
    unsigned		ESP0;
    unsigned short	SS0;
    unsigned short	reserved2;
    unsigned		ESP1;
    unsigned short	SS1;
    unsigned short	reserved3;
    unsigned		ESP2;
    unsigned short	SS2;
    unsigned short	reserved4;
    unsigned		CR3;
    unsigned		EIP;
    unsigned		EFLAGS;
    unsigned		EAX;
    unsigned		ECX;
    unsigned		EDX;
    unsigned		EBX;
    unsigned		ESP;
    unsigned		EBP;
    unsigned		ESI;
    unsigned		EDI;
    unsigned short	ES;
    unsigned short	reserved5;
    unsigned short	CS;
    unsigned short	reserved6;
    unsigned short	SS;
    unsigned short	reserved7;
    unsigned short	DS;
    unsigned short	reserved8;
    unsigned short	FS;
    unsigned short	reserved9;
    unsigned short	GS;
    unsigned short	reserved10;
    unsigned short	LDT_seg_selector;
    unsigned short	reserved11;
    unsigned		debug_trap	: 1;
    unsigned		reserved	: 14;
    unsigned		IOMap_base_addr : 15;
    } __attribute__ ((packed)) TSS_t;

typedef union {
    unsigned	all;
    struct {
	unsigned protected_mode		: 1;
	unsigned wait_fwait_trapped	: 1;
	unsigned fpu_trapped		: 1;
	unsigned task_switch		: 1;
	unsigned cste_1			: 1;
	unsigned internal_fpu_report	: 1;	/* when set */
	unsigned reserved1		: 10;
	unsigned write_protect		: 1;	/* supervisor to user read only */
	unsigned reserved2		: 1;
	unsigned alignment_check	: 1;	/* enabled if set, CPL=3, protected */
	unsigned reserved3		: 10;
	unsigned no_writethrough	: 1;	/* disabled if set */
	unsigned no_caching		: 1;	/* disabled if set */
	unsigned paging			: 1;	/* enabled if set */
	} __attribute__ ((packed)) bit;
    } __attribute__ ((packed)) CR0_t;

typedef union {
    unsigned	all;
    struct {
	unsigned unused		: 3;
	unsigned PWT		: 1;
	unsigned PCD		: 1;
	unsigned reserved	: 7;	/* 36 bits addr cake */
	unsigned pagedir_base	: 20;
	} __attribute__ ((packed)) bit;
    } __attribute__ ((packed)) CR3_t;

typedef union {
    unsigned	all;
    struct {
	unsigned present	: 1;
	unsigned read_write	: 1;
	unsigned user_super	: 1;
	unsigned write_through	: 1;
	unsigned cache_disabled	: 1;
	unsigned accessed	: 1;
	unsigned reserved_0	: 1;
	unsigned page_size_0	: 1;
	unsigned global_page	: 1;
	unsigned available	: 3;
	unsigned base_address	: 20;
	} __attribute__ ((packed)) bit;
    } __attribute__ ((packed)) pagedir4K_t;

typedef union {
    unsigned	all;
    struct {
	unsigned present	: 1;
	unsigned read_write	: 1;
	unsigned user_super	: 1;
	unsigned write_through	: 1;
	unsigned cache_disabled	: 1;
	unsigned accessed	: 1;
	unsigned dirty		: 1;
	unsigned page_size_1	: 1;
	unsigned global_page	: 1;
	unsigned available	: 3;
	unsigned reserved	: 10;
	unsigned base_address	: 10;
	} __attribute__ ((packed)) bit;
    } __attribute__ ((packed)) pagedir4M_t;

typedef union {
    unsigned	all;
    struct {
	unsigned present	: 1;
	unsigned read_write	: 1;
	unsigned user_super	: 1;
	unsigned write_through	: 1;
	unsigned cache_disabled	: 1;
	unsigned accessed	: 1;
	unsigned dirty		: 1;
	unsigned reserved_0	: 1;
	unsigned global_page	: 1;
	unsigned available	: 3;
	unsigned base_address	: 20;
	} __attribute__ ((packed)) bit;
    } __attribute__ ((packed)) pagetable_t;

typedef union {
    unsigned	all;
    struct {
	unsigned VME : 1;
	unsigned PVI : 1;
	unsigned TSD : 1;
	unsigned DE  : 1;
	unsigned PSE : 1;
	unsigned PAE : 1;
	unsigned MCE : 1;
	unsigned PGE : 1;
	unsigned PCE : 1;
	unsigned OSFXSR : 1;
	unsigned OSXMMEXCPT : 1;
	unsigned reserved : 21;
	} __attribute__ ((packed)) bit;
    } __attribute__ ((packed)) CR4_t;

typedef struct {
    unsigned short limit;
    unsigned       base;
    } __attribute__ ((packed)) gdt_idt_param_t;

/* lgdt produces an exception with DOSEMU... anyway you cannot
   open the A20 gate with DOSEMU - at least I think so. */
/* 32 bit (and not 286 24 bits) forced by the use of addr32 : */
extern inline void set_gdt (const gdt_idt_param_t *gdtptr)
  {
  asm volatile (" lgdtl %0 " : /* no output */ : "m" (*gdtptr));
  }

extern inline void set_gdt_24 (const gdt_idt_param_t *gdtptr)
  {
  asm volatile (" lgdtw %0 " : /* no output */ : "m" (*gdtptr));
  }

extern inline void get_gdt (gdt_idt_param_t *gdtptr)
  {
  asm (" sgdtl %0 " : "=m" (*gdtptr));
  }

extern inline void set_idt (const gdt_idt_param_t *idtptr)
  {
  asm volatile (" lidtl %0 " : /* no output */ : "m" (*idtptr));
  }

extern inline void set_idt_24 (const gdt_idt_param_t *idtptr)
  {
  asm volatile (" lidtw %0 " : /* no output */ : "m" (*idtptr));
  }

extern inline void get_idt (gdt_idt_param_t *idtptr)
  {
  asm (" sidtl %0 " : "=m" (*idtptr));
  }

extern inline void set_task (unsigned short seg)
  {
  asm volatile (" ltr %0 " : : "q" (seg));
  }

extern inline unsigned short get_task (void)
  {
  unsigned short seg;

  asm volatile (" str %0 " : "=q" (seg));
  return seg;
  }

/* Not sure the following is working/is usable, but it is here: */
extern inline unsigned loadseglimit (unsigned descriptor)
  {
  unsigned returned;

  asm volatile (" lsll %1,%0 " : "=rm" (returned) : "r" (descriptor * 8));
  return returned;
  }

extern inline CR0_t get_CR0 (void)
  {
  CR0_t returned;

  asm ASMREAD (" mov %%cr0,%0 " : "=r" (returned));
  return returned;
  }

extern inline void set_CR0 (CR0_t CR0)
  {
  asm volatile (" mov %0,%%cr0\n jmp 1f\n 1:" : /* no output */ : "r" (CR0));
  }

extern inline CR4_t get_CR4 (void)
  {
  CR4_t returned;
  asm (" mov %%cr4,%0 " : "=r" (returned));
  return returned;
  }

extern inline void set_CR4 (CR4_t CR4)
  {
  asm volatile (" mov %0,%%cr4 " : /* no output */ : "r" (CR4));
  }

#define GETREG(regname) \
    extern inline unsigned get_ ## regname (void) {		\
	unsigned returned;					\
	asm (" mov %%" #regname ",%0 " : "=r" (returned));	\
	return returned;					\
	}

#define SETREG(regname) \
    extern inline void set_ ## regname (unsigned val) {		\
	asm volatile (" mov %0,%%" #regname : : "r" (val));	\
	}

GETREG(CR2)
GETREG(CR3) SETREG(CR3)

SETREG(DR0) GETREG(DR0)
SETREG(DR1) GETREG(DR1)
SETREG(DR2) GETREG(DR2)
SETREG(DR3) GETREG(DR3)
SETREG(DR4) GETREG(DR4)
SETREG(DR5) GETREG(DR5)
SETREG(DR6) GETREG(DR6)
SETREG(DR7) GETREG(DR7)

extern inline void clear_a_bit (void)
  {
  /* To know if you are inside or outside the matrix, try to
     read debug register %dr4 or %dr5: if you do not have
     any exception, you'd better follow the white rabit and
     take the red pill... */
  asm volatile (
"	pushl	%edx		\n"
"	pushw	$0		\n"
"	popw	%fs		\n"
"	pushw	%cs		\n"
"	pushw	$1f		\n"
"	popl	%edx		\n"
"	xchgl	%fs:4,%edx	\n"
"	pushfw			\n"
"	pushw	%cs		\n"
"	pushw	$2f		\n"
"	1:			\n"
"	pushl	%eax		\n"
"	movl	$0x400,%eax	\n"
"	movl	%eax,%dr7	\n"
"	popl	%eax		\n"
"	iretw			\n"
"	2:			\n"
"	movl	%edx,%fs:4	\n"
"	popl	%edx		\n"
	);
  }

/*
 * I do the following my way, is that how real programmers
 * do it ? I wonder why everybody want to use assembly here...
 */

#if 0
extern inline unsigned protected_mode (void)
  {
  CR0_t CR0 = get_CR0();

  CR0.bit.protected_mode = 1;
  set_CR0 (CR0);

  if (get_CR0().bit.protected_mode != 0)
      return 0;
    else
      return 1; /* error */
  }
#else
extern inline unsigned protected_mode (void)
  {
  setsw (getsw() | 1);
  asm volatile (" jmp 1f ; 1: ");

  if ((getsw() & 1) != 0)
      return 0;
    else
      return 1; /* error */
  }
#endif

extern inline unsigned real_mode (void)
  {
  CR0_t CR0 = get_CR0();

  CR0.bit.protected_mode = 0;
  set_CR0 (CR0);

  if (get_CR0().bit.protected_mode == 0)
      return 0;
    else
      return 1; /* error */
  }

/*
 * The cache playing:
 */
extern inline CR0_t disable_cache (void)
  {
  CR0_t CR0 = get_CR0(), save_CR0;

  save_CR0 = CR0;

  if (CR0.bit.no_caching == 0) { /* else 80386 or cache already disabled */
      CR0.bit.no_writethrough = 1;
      CR0.bit.no_caching = 1;
      set_CR0 (CR0);
      }
  return save_CR0;
  }

extern inline void enable_cache (CR0_t save_CR0)
  {
  if (save_CR0.bit.no_caching == 0) { /* else 80386 or cache already disabled */
      set_CR0 (save_CR0);
      }
  }

extern inline void force_cache (void *adr, unsigned len)
  {
  unsigned unused, modified, change;

  len /= 4;
  asm volatile (
	" rep lodsl  %%ds:(%1),%0"
	: "=a" (unused), "=S" (modified), "=c" (change)
	: "S" (adr), "c" (len)
	);
  asm volatile (
	" rep lodsl  %%ds:(%1),%0"
	: "=a" (unused), "=S" (modified), "=c" (change)
	: "S" (adr), "c" (len)
	);
  asm volatile (
	" rep lodsl  %%ds:(%1),%0"
	: "=a" (unused), "=S" (modified), "=c" (change)
	: "S" (adr), "c" (len)
	);
  asm volatile (
	" rep lodsl  %%ds:(%1),%0"
	: "=a" (unused), "=S" (modified), "=c" (change)
	: "S" (adr), "c" (len)
	);
  }

#endif /* LIBRARY_H */

