/* library.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"
#include "instboot.h"
#include "library.h"
#include "boot.h"	/* BOOT1_DOS_running() */
#include "debug.h"
#include "bios.h"	/* _BIOS_getBaseMemory(), CDROMbootspec */
#include "util.h"	/* set_A20(), segment_door() */
#if USER_SUPPORT != 0
#include "user.h"	/* UI.function.putstr */
#endif
#include "disk.h"	/* freeze_IDE() */
#include "dos.h"	/* DOS_SetInterruptVector() */
#include "vgabios.h"	/* _VGA_setmode(), _VGA_autoload_default_palette() */
#include "vesabios.h"	/* _VESA_setmode() */
#include "messages.h"	/* MSG_EXITING */
#include "fs.h"		/* initrd_file(): BOOTWAY */

/**
 ** Some standart functions:
 **/
#if SETUP & MULTILINGUAL
enum MltStr_e MltStrLanguage = MLTSTR_ENGLISH; /* inited in main() */

#if !(SETUP & XSTRING_SEGMENT) /* only fewer languages will fit, maybe only one or two... */
__attribute__ (( fastcall, pure )) const char *MltStr(const char *multistring)
  {
  enum MltStr_e cptlang;
  const char *ptr = multistring;
  for (cptlang = MLTSTR_ENGLISH; cptlang < MltStrLanguage; cptlang++)
     while (*ptr++) {}
  return (*ptr)? ptr : multistring;
  }
#else
__attribute__ (( fastcall, pure )) const char *MltStr(const char *multistring) // or const ?
  {
  enum MltStr_e cptlang;
  const char *ptr = multistring;

  if ((*(unsigned *)&ptr[1] >> 16) != 0 || (*ptr != '\xFF' && *ptr != '\xFE')) {
      for (cptlang = MLTSTR_ENGLISH; cptlang < MltStrLanguage; cptlang++)
	  while (*ptr++) {}
      return (*ptr)? ptr : multistring;
      }

  farptr initial_fptr;
  if (*ptr == '\xFE')
      initial_fptr = xstring2_adr((void *)*(unsigned *)&ptr[1]);
    else /* if (*ptr == '\xFF') */
      initial_fptr = xstring1_adr((void *)*(unsigned *)&ptr[1]);

  // we need to use an intermediate buffer because we may have multiple in flight, like "printf (mltstr1, mltstr2);"
  farptr fptr = initial_fptr;
//  static char buffer[1024], *buffer_index = buffer; creates a .data (initialised variable)
  static char buffer[1024], *buffer_index;
  if (!buffer_index)
      buffer_index = buffer;

  for (cptlang = MLTSTR_ENGLISH; cptlang < MltStrLanguage; cptlang++)
      while (peekb(fptr++)) {}
  if (peekb(fptr) == '\0')	/* use default as english */
      fptr = initial_fptr;
  int strlen = 0;
  while (peekb(fptr + strlen) != 0 && strlen < &buffer[sizeof(buffer) - 1] - buffer_index)
      strlen++;
  if (strlen >= &buffer[sizeof(buffer)- 1] - buffer_index)
      buffer_index = buffer;
  char *returned = buffer_index;
  while ((*buffer_index++ = peekb(fptr++)) != 0) {}
  return returned;
  }
#endif
#endif /* SETUP & MULTILINGUAL */

/* See hexval() in library.h: */
const char *const itoa_array =      "0123456789ABCDEF0123456789abcdef";
const unsigned itoa_len =   sizeof ("0123456789ABCDEF0123456789abcdef") - 1;

/*
 * More or less subroutine of printf() rewrite, limited to stack growing
 * down, and limited to formats '%%', '%c', '%s', '%d', '%x', '%X', '%u',
 * '%llu', '%llX' without size like '%5.5d'.
 * "%D"/"%U" is treated as "%d"/"%u" but adds comma every 3 digit like: 1,000,000
 *
 * Used also in user.c::BOOT1_printf
 */
char *snprtf (char *buffer, unsigned len, const char *format, void **param)
  {bound_stack();{
  char *dest = buffer;
  char *endbuffer = buffer + len - 1;	/* for the ending zero */

  for (dest = buffer; dest < endbuffer && *format != '\0'; format++) {
      if (*format != '%')
	  *dest++ = *format;
	else if (*++format == '%')
	  *dest++ = '%';
	else if (*format == 'c')
	  *dest++ = *(char *)param++;
	else if (*format == 's') {
	  const char *cptr = (char *)*param++;

//	  checkptr (cptr, "snprtf string error");
#if !(SETUP & BIG_MALLOC)
	  /* Do not crash if format error, not a pointer */
	  if ((unsigned)cptr >= 0xFFFF || (unsigned)cptr < 3)
	      cptr = "Not A Pointer";
#endif
	  while (*cptr != '\0' && dest < endbuffer)
	      *dest++ = *cptr++;
	  }
	else if (strchr ("lDdUuXxp", *format)) {
	  const char *itoa = itoa_array;
	  /* enought for "long long" in decimal with commas, 20 digit + 7 comma: */
	  char buff[32], *buffptr = buff, hexadisplay = 1;
	  unsigned char cpt = 3;
	  unsigned long long todisplay;

	  if (format[0] == 'l' && format[1] == 'l') {
	      format += 2;
	      if (*format == 'x' || *format == 'X')
		  todisplay = *(unsigned long long *)param;
		else {
		  if (*(long long *)param >= 0)
		      todisplay = *(unsigned long long *)param;
		    else {
		      todisplay = - *(long long *)param;
		      *dest++ = '-';
		      if (dest >= endbuffer)
			  break;
		      }
		  if (todisplay < 10 * 0xFFFFFFFFULL)
		      hexadisplay = 0;	/* no Divide Error Exception */
		  }
	      param += 2;
	      }
	    else {
	      if (format[0] == 'l')
		  format++;
	      if (*format == 'x' || *format == 'X' || *format == 'p')
		  todisplay = *(unsigned *)param;
		else {
		  if (*(int *)param >= 0)
		      todisplay = *(unsigned *)param;
		    else {
		      todisplay = - *(int *)param;
		      *dest++ = '-';
		      if (dest >= endbuffer)
			  break;
		      }
		  hexadisplay = 0;
		  }
	      param += 1;
	      }

	  if (*format == 'x' || *format == 'p')
	      itoa = &itoa[0x10]; /* small letters needed for idex= parameter */

	  for (;;) {
	      unsigned index;
	      if (hexadisplay) {
		  index = todisplay & 0xF;
		  todisplay >>= 4;
		  }
		else {
#if 1
		  /* Divide Error Exception if the quotient is too large for the designated register! */
		  /* That is not equivalent for numbers over (10 * 0xFFFFFFFF)
		     but good enought for us and it works in real mode
		     (no long long precompiled library called) : */
		  /* We may also try to do " ((unsigned long long)(todisplay * 0xCCCCCCCDU)) >> 35 ",
			i.e. " ul_mul_ul(0xCCCCCCCDU, nb) >> 35 " but with long long */
		  todisplay = ull_div_ul (todisplay, 10, &index);
#else
		  index = todisplay % 10;
		  todisplay /= 10;
#endif
		  }
	      *buffptr++ = itoa[index];
	      if (todisplay == 0)
		  break;
	      if (*format == 'U' || *format == 'D')
		  if (--cpt == 0) { cpt = 3; *buffptr++ =','; }
	      }
	  while (--buffptr >= buff && dest < endbuffer)
	      *dest++ = *buffptr;
	  }
	else { /* unrecognised format */
	  *dest++ = '%';
	  format--;
	  }
      }
  *dest = '\0';

  return buffer;
  }}

#if USER_SUPPORT != 0
#define PRINTSTRING	UI.function.putstr
#else
#define PRINTSTRING	BOOT1_putstr
#endif

int printf (const char *format, ...)
  {bound_stack();{
  char buffer[256];
  const char *tmp = snprtf ( buffer, sizeof (buffer), format,
		(void **)((unsigned)&format + sizeof(const char *)) );

  PRINTSTRING (tmp);
  return 0;
  }}

#if !(SETUP & QUITE)
int verbose_printf (const char *format, ...)
  {bound_stack();{
  if (copy_gujin_param.attrib.verbose) {
      char buffer[256];
      const char *tmp = snprtf ( buffer, sizeof(buffer), format,
		(void **)((unsigned)&format + sizeof(const char *)) );

      PRINTSTRING (tmp);
      }
  return 0;
  }}
#endif

void print (const char *msg)
  {
  PRINTSTRING (msg);
  }

#if !(SETUP & QUITE)
void verbose_print (const char *msg)
  {
  if (copy_gujin_param.attrib.verbose)
      PRINTSTRING (msg);
  }
#endif

void puts (const char *msg)
  {
  PRINTSTRING (msg);
  PRINTSTRING ("\r\n");
  }

#if !(SETUP & QUITE)
void verbose_puts (const char *msg)
  {
  if (copy_gujin_param.attrib.verbose) {
      PRINTSTRING (msg);
      PRINTSTRING ("\r\n");
      }
  }
#endif

int sprintf (char *dest, const char *format, ...)
  {
  /* Maximum string: 4096 bytes */
  snprtf (dest, 4096, format,
		 (void **)((unsigned)&format + sizeof(const char *)));
  return strlen (dest);
  }

/* That is even for standard exit, but not Linux exit: */
void before_exit (void)
  {
#if DISK_SUPPORT & IDE_SUPPORT
  DBG ((" [freeze_IDE(0): "));
  freeze_IDE ((struct freeze_IDE_s) {});
  DBG (("\r\nfreeze_IDE done] "));
#endif

#if USER_SUPPORT & BIOS_MOUSE_SUPPORT
  if (_BIOS_equipment_flags().pointing_device_installed) {
      struct PS_resetID result;
      /* _PS_enable() and _PS_set_handler() should not be needed,
	but running Gujin from Gujin makes sometimes the mouse disappear...
	does it help?
      _PS_enable (0); */
      _PS_set_handler (0);
      _PS_reset (&result);
      }
#endif

#ifdef TREAT_EXCEPTION
  restore_vectors();
#endif

  DBG_END();

  /*
   * After the following, UI is no more up-to-date, so no more printf...
   */
#if (USER_SUPPORT & VESA_SUPPORT) && (USER_SUPPORT & VGA_SUPPORT)
  if (BOOT1_COM_port() < 0) {
      _VGA_autoload_default_palette (1);
      if (VESA_ACTIVE() && !UI.parameter.BIOS_compatible) {
	  _VGA_setmode (3);
	  }
	else {
	  unsigned char nbcol, page;
	  /* clear screen and reset palette, needed? */
	  _VGA_setmode (_VGA_getmode (&nbcol, &page) & 0x7F);
	  }
      }
#endif
  }

/* That is not a standard exit: */
__attribute__ ((noreturn))
void BIOS_killme (void)
  {
  DBG (("\r\n%s called\r\n", __FUNCTION__));

#if USER_SUPPORT != 0
  _VGA_setcursor (0, UI.parameter.nbrow - 1, 0);
#endif
  BOOT1_putstr ("\r\n");

  before_exit();

#if DEBUG & DEBUG_VIDEO
  puts ("Press a key to continue BIOS_killme");
  _BIOS_getkey();
#endif

  /* Normally we should not do DBG(), but it works if
     we have a serial line/printer connected: */
  if (BOOT1_DOS_running()) {
      DBG ((" Trying to stop 'exe' executable"));
      _DOS_stop_exe ();
      DBG ((" Trying to stop 'com' executable"));
      _DOS_stop_com ();
      DBG ((" not working!"));
      }
    else {
      DBG ((" not in DOS environment"));
      _BIOS_wait (4 * 1000 * 1000);
      /* Not when VCPI is running: */
      set_A20 (0);
      _BIOS_failedboot ();
      }
  DBG ((" Trying to stop DOSEMU"));
  _DOSEMU_stop ();
  DBG ((" Trying a segment violation"));
  _BIOS_sigsegv ();
  DBG ((" Trying the HLT instruction"));
  _BIOS_halt ();
  DBG ((" starting infinite loop"));
  while (1) ;
  }

/**
 ** Exception & timeout related things:
 **/

#if defined (TREAT_EXCEPTION) || defined (TREAT_TIMER)
volatile unsigned setjump_key; /* init at 0 : BSS */
setjmp_buf setjump_buffer;

static farptr __attribute__ (( regparm(1) ))
get_vector (unsigned irq)
  {
  if (BOOT1_DOS_running())
      return DOS_GetInterruptVector (irq);
    else
      return peekl (4*irq);
  }

static farptr
set_vector_farptr (unsigned irq, farptr fct_faradr)
  {
  if (BOOT1_DOS_running()) {
      farptr returned = get_vector (irq);
      DOS_SetInterruptVector (irq, fct_faradr);
      return returned;
      }
    else
      return xchgl (4*irq, fct_faradr);
  }
#endif

#ifdef TREAT_EXCEPTION
/*
 * "bound" checking and exceptions related things:
 */

static struct {
    unsigned initialised;
    unsigned divide_error,
	     instruction_invalid,
	     stack_segment_fault,
	     general_protection,
	     reserved;
#if DEBUG & (DEBUG_STACK|DEBUG_ADDR)
    unsigned  bound_error;
#endif
#ifdef TREAT_COUNTER_TRACE_FLAG
    unsigned  debug;
#endif
    } savedvector; /* init at 0 : BSS */

/* to save code space: */
static unsigned set_vector (unsigned irq, void *fctadr)
  {
  return set_vector_farptr (irq, code_adr (fctadr));
  }

/*
 * Setup vectors, only once if restart:
 * Called from assembly, do not prototype it.
 */
void setup_vectors (void)
  {
  extern unsigned char  divide_error[], instruction_invalid[],
			stack_segment_fault[], general_protection[];

  if (savedvector.initialised++ != 0)
      return;
  savedvector.divide_error = set_vector (0, divide_error);
  savedvector.instruction_invalid = set_vector (6, instruction_invalid);
  savedvector.stack_segment_fault = set_vector (12, stack_segment_fault);
  savedvector.general_protection = set_vector (13, general_protection);
#ifdef TREAT_COUNTER_TRACE_FLAG
  {
  extern unsigned char  TFflagHandler[];
  savedvector.debug = set_vector (1, TFflagHandler);
  }
#endif
#if DEBUG & (DEBUG_STACK|DEBUG_ADDR)
  {
  extern unsigned char  bound_error[];
  savedvector.bound_error = set_vector (5, bound_error);
  }
#endif
  }

void restore_vectors (void)
  {
  if (savedvector.initialised-- == 0)
      return;

  set_vector_farptr (0, savedvector.divide_error);
  set_vector_farptr (6, savedvector.instruction_invalid);
  set_vector_farptr (12, savedvector.stack_segment_fault);
  set_vector_farptr (13, savedvector.general_protection);
#ifdef TREAT_COUNTER_TRACE_FLAG
  set_vector_farptr (1, savedvector.debug);
#endif
#if DEBUG & (DEBUG_STACK|DEBUG_ADDR)
  set_vector_farptr (5, savedvector.bound_error);
#endif
  }

/**
 ** That is really for fun...
 **/
void APM_power_OFF (void)
  {
  unsigned short BCD_version, flags, tmp;

  if (BOOT1_DOS_running())
      return;
  PRINT (MSG_POWERING_OFF);
  /* La grande prophetie va-t-elle se realiser ? / Will the dark prophecy happen? */

  tmp = _APM_installation_check (&BCD_version, &flags);
  ZDBG (("\r\n%s: check result 0x%X, version 0x%X, flags 0x%X\r\n",
	__FUNCTION__, tmp, BCD_version, flags));
  if (tmp != 0 || BCD_version < 0x102)
      ZDBG (("So cannot Power Down (works only with APM v1.2+)\r\n"));
    else {
      tmp = _APM_connect_realmode();
      if (tmp != 0)
	  ZDBG (("_APM_connect_realmode failed 0x%X\r\n", tmp));
	else {
	  tmp = _APM_declare_driver_version (0x102);
	  if (tmp != 0)
	      ZDBG (("_APM_declare_driver_version(0x102) refused: 0x%X\r\n",
			tmp));
	    else {
	      tmp = _APM_state (APM_alldevice, APM_off);
	      if (tmp != 0)
		  ZDBG (("_APM_state(OFF) failed 0x%X\r\n", tmp));
	      /* If we return there for any reason: */
	      tmp = _APM_state (APM_alldevice, APM_ready);
	      if (tmp != 0)
		  ZDBG (("_APM_state(ON) failed 0x%X\r\n", tmp));
	      }
	  tmp = _APM_disconnect();
	  if (tmp != 0)
	      ZDBG (("_APM_disconnect failed: 0x%X\r\n", tmp));
	  }
      }
  PRINT (MSG_COLON_FAILED);
  }

static void puthex (const char *msg, unsigned hex, unsigned crlf)
  {
  char buff[8+2+1], *buffptr = &buff[nbof(buff) - 1];

  *buffptr-- = '\0';
  do {
      *buffptr-- = itoa_array[hex % 16];
      hex /= 16;
      } while (hex != 0);
  *buffptr-- = 'x',
  *buffptr = '0';
  print (msg);
  if (crlf)
      puts (buffptr);
    else
      print (buffptr);
  }

/* Special calling convention:
 *   DO NOT MODIFY parameters here, if you want to return!
 */
void __attribute__ ((cdecl, regparm (0), noreturn))
exceptions (unsigned edi, unsigned esi, unsigned ebp, unsigned initsp,
	    unsigned ebx, unsigned edx, unsigned ecx, unsigned eax,
	    unsigned ds,  unsigned es,
	    unsigned vector,
	    unsigned cs_ip, unsigned flags)
  {
#if DEBUG & DEBUG_STACK
  unsigned save_stack_limit_low = STATE.stack_limit.low_limit;
#if !(SETUP & QUITE)
  unsigned save_esp = getesp();
#endif
#endif
  const char *string;

  /* Now rescue the ISO9001 video BIOSes which executes invalid instructions
	 or divisions by zero when trying to set invalid modes */
  if (setjump_key == 0x45746965 && setjump_buffer.cs != 0) {
      setjump_key = -setjump_key;
      longjmp (&setjump_buffer);
      }

#if DEBUG & DEBUG_STACK
  /* disable the stack checking & force update save_stack_limit_low: */
  asm volatile (
"	movl	$0,%1	"
	: "+g" (save_stack_limit_low)	/* force update */
	: "m" (STATE.stack_limit.low_limit)
	);
#endif /* DEBUG_STACK */

#if USER_SUPPORT != 0
  UI.function.setcursor (0, 0);
  UI.function.setfgcolor (UI.stdcolor[red]);
  UI.function.setbgcolor (UI.stdcolor[lightgray]);
#endif

  switch (vector) {
    case 0:
	string = "DIVISION BY ZERO / DIVISION OVERFLOW";
	break;
    case 5:
#if (DEBUG & DEBUG_ADDR) && (DEBUG & DEBUG_STACK)
	string = "BOUND: STACK OVERFLOW OR POINTER OUT OF RANGE";
#elif DEBUG & DEBUG_ADDR
	string = "BOUND: POINTER OUT OF RANGE";
#elif DEBUG & DEBUG_STACK
	string = "BOUND: STACK OVERFLOW";
#else
	string = "BOUND: Who has executed this instruction?";
#endif
	break;
    case 6:
	string = "INVALID INSTRUCTION";
	break;
    case 12:
	string = "STACK SEGMENT FAULT";
	break;
    case 13:
	string = "GENERAL PROTECTION";
	break;
    default:
	string = "??UNKNOWN??";
	break;
    }
  DBG (("\r\n %s at address 0x%X (content: 0x%X 0x%X), codeseg 0x%X, diskcodeseg 0x%X, fscodeseg 0x%X, loadcodeseg 0x%X, usercodeseg 0x%X\r\n",
	string, cs_ip, peekl(cs_ip - 4), peekl(cs_ip), STATE.codeseg_adr >> 16, STATE.diskcodeseg_adr >> 16, STATE.fscodeseg_adr >> 16, STATE.loadcodeseg_adr >> 16, STATE.usercodeseg_adr >> 16));
  print (string);
  puthex (" at address ", cs_ip, 0);
  puthex (", codeseg ", STATE.codeseg_adr >> 16, 0);
#if SETUP & XCODE_SEGMENT
  puthex (" diskcodeseg ", STATE.diskcodeseg_adr >> 16, 0);
  puthex (" fscodeseg ", STATE.fscodeseg_adr >> 16, 0);
  puthex (" loadcodeseg ", STATE.loadcodeseg_adr >> 16, 0);
  puthex (" usercodeseg ", STATE.usercodeseg_adr >> 16, 0);
#endif
  puthex (" dataseg ", STATE.dataseg_adr >> 16, 1);

#if !(SETUP & QUITE)
  if (vector == 5) {
#if DEBUG & DEBUG_STACK
      puthex (" stack limit low: ", save_stack_limit_low, 0);
      puthex (" stack limit high: ", STATE.stack_limit.high_limit, 0);
      puthex (" value:", save_esp, 1);
#endif /* DEBUG_STACK */

#if DEBUG & DEBUG_ADDR
      puthex (" data limit low: ", STATE.data_limit.low_limit, 0);
      puthex (" data limit high: ", STATE.data_limit.high_limit, 1);

      if (STATE.BoundMsg != 0 && strlen((char *)STATE.BoundMsg) < 80) {
	  print ("Bound Msg: ");
	  puts (STATE.BoundMsg);
	  }
#endif /* DEBUG_ADDR */
      }
    else if (vector == 6 || vector == 13) {
      puthex ("invalid code (check corruption with boot.asm): ",
		peekb(cs_ip), 0);
      puthex (", ", peekb(cs_ip+1), 0);
      puthex (", ", peekb(cs_ip+2), 0);
      puthex (", ", peekb(cs_ip+3), 0);
      puthex (", ", peekb(cs_ip+4), 0);
      puthex (", ", peekb(cs_ip+5), 1);
      }
#if 0
    else if (vector == 13) {
      /* There is no error code in real mode ! */
      puthex (" CR2 ", get_CR2(), 1); // seems not filled in real mode
      }
#endif

  puthex (" eax = ", eax, 0);
  puthex (" ebx = ", ebx, 0);
  puthex (" ecx = ", ecx, 0);
  puthex (" edx = ", edx, 1);

//  puthex (" initsp = ", initsp, 0);
  puthex (" edi = ", edi, 0);
  puthex (" esi = ", esi, 0);
  puthex (" ebp = ", ebp, 0);
  puthex (" flags = ", flags & 0xFFFF, 1);

  puthex (" cs = ", getcs(), 0);
  puthex (" ds = ", ds & 0x0000FFFF, 0);
  puthex (" es = ", es & 0x0000FFFF, 0);
  puthex (" ss = ", getss(), 0);
  puthex (" esp = ", getesp(), 1);

  {
#if 0
  farptr esp = (((unsigned)getss()) << 16) + getesp();
  puthex (" *(esp+00) = ", peekl(esp + 0x00), 0);
  puthex (" *(esp+04) = ", peekl(esp + 0x04), 0);
  puthex (" *(esp+08) = ", peekl(esp + 0x08), 0);
  puthex (" *(esp+0C) = ", peekl(esp + 0x0C), 1);
  puthex (" *(esp+10) = ", peekl(esp + 0x10), 0);
  puthex (" *(esp+14) = ", peekl(esp + 0x14), 0);
  puthex (" *(esp+18) = ", peekl(esp + 0x18), 0);
  puthex (" *(esp+1C) = ", peekl(esp + 0x1C), 1);
  puthex (" *(esp+20) = ", peekl(esp + 0x20), 0);
  puthex (" *(esp+24) = ", peekl(esp + 0x24), 0);
  puthex (" *(esp+28) = ", peekl(esp + 0x28), 0);
  puthex (" *(esp+2C) = ", peekl(esp + 0x2C), 1);
  puthex (" *(esp+30) = ", peekl(esp + 0x30), 0);
  puthex (" *(esp+34) = ", peekl(esp + 0x34), 0);
  puthex (" *(esp+38) = ", peekl(esp + 0x38), 0);
  puthex (" *(esp+3C) = ", peekl(esp + 0x3C), 1);
#endif
  }
#endif /* QUITE */

  enable_interrupt();	/* control alt del */
  print ("\r\nPress PAUSE to stop display...\r\n");
  _BIOS_wait (10 * 1000 * 1000);
  BIOS_killme() ;
  }

#endif /* TREAT_EXCEPTION */

#ifdef TREAT_TIMER

/*
 * Define the timer interrupt we manage, may be 0x1C or 0x08,
 * but keep it to 0x08:
 */
#define TIMER_IRQ 0x08

/* IMPORTANT NOTE:
 * This function, "timeout_handler" shall not use %ebp as a pointer
 * because %ss is not (and cannot be) set. Keep it simple!
 * Also segment %gs is not set, so you probably can only display
 * something in VGA text modes.
 */
__attribute__ ((cdecl, regparm (0))) void
timeout_handler (const unsigned ip_flags,
		 const unsigned seggsfs,
		 const unsigned segdses,
		 const unsigned edx,
		 const unsigned ecx,
		 const unsigned eax)
  {
  if (setjump_key == 0x45746965 && setjump_buffer.cs != 0) {
      setjump_key = ~setjump_key;
      longjmp (&setjump_buffer);
      }
  }

/* NOTE: no re-entrancy checking */
void set_timeout (unsigned nbtick)
  {
  extern unsigned char timer_handler_oldptr[], timer_counter[];

  if (nbtick == 0) {
      farptr oldptr = codeseg_peekl (timer_handler_oldptr);

      if (oldptr != 0) {
	  if (BOOT1_DOS_running())
	      DOS_SetInterruptVector (TIMER_IRQ, oldptr);
	    else
	      pokel (4*TIMER_IRQ, oldptr);
	  }
      codeseg_pokel (timer_counter, 0);
      codeseg_pokel (timer_handler_oldptr, 0);
      }
    else if (nbtick != 0xFFFFFFFFU) {
      extern unsigned char timer_handler_relay[];

      codeseg_pokel (timer_counter, nbtick);
      if (BOOT1_DOS_running()) {
	  codeseg_pokel (timer_handler_oldptr,
				DOS_GetInterruptVector (TIMER_IRQ));
	  DOS_SetInterruptVector (TIMER_IRQ, code_adr (timer_handler_relay));
	  }
	else {
	  codeseg_pokel (timer_handler_oldptr, peekl (4*TIMER_IRQ));
	  pokel (4*TIMER_IRQ, code_adr (timer_handler_relay));
	  }
      }
    else {
      /* Never call this with nbtick == 0xFFFFFFFFU, but here we want the
	 asm() to be in the same segment than set_timeout, think about
	 ld --gc-section */
      asm (
"									\n"
"	jmp	end_of_timeout_assembly					\n"
"									\n"
"	.GLOBAL timer_handler_oldptr					\n"
"	.GLOBAL timer_handler_relay, timer_counter			\n"
"	.align 4							\n"
"timer_counter:								\n"
"	.long	0							\n"
"timer_handler_oldptr:							\n"
"	.long	0							\n"
"timer_handler_relay:							\n"
"	cmpl	$0,%cs:timer_counter					\n"
"	je	timer_no_treat						\n"
"	decl	%cs:timer_counter					\n"
"	jne	timer_no_treat						\n"
"									\n"
"	xchgw	4(%esp),%bp						\n"
"	pushw	%bp							\n"
"	xchgw	6(%esp),%bp						\n"
"	pushw	%cs							\n"
"	pushw	$timer_handler_treat					\n"
"timer_no_treat:							\n"
"	ljmpw	*%cs:timer_handler_oldptr				\n"
"timer_handler_treat:							\n"
"	pushfw		# align stack					\n"
"	pushw	%gs							\n"
"	pushw	%fs							\n"
"	pushw	%ds							\n"
"	pushw	%es							\n"
"	pushl	%edx							\n"
"	pushl	%ecx							\n"
"	pushl	%eax							\n"
"	movw	%cs,%ax							\n"
//#ifdef CALLING_FROM_SEG
//"	subw	$xxxcodeseg,%ax					\n"
//#endif
"	addw	$deltaseg,%ax   # not necessary if no CODE_SEGMENT	\n"
"	movw	%ax,%ds							\n"
"	movw	%ax,%es							\n"
"	calll	timeout_handler	# in current segment			\n"
"	popl	%eax		# timer_handler_ptr shall not use %ebp	\n"
"				# as a pointer (%ss not set)		\n"
"	popl	%ecx							\n"
"	popl	%edx							\n"
"	popw	%es							\n"
"	popw	%ds							\n"
"	popw	%fs							\n"
"	popw	%gs							\n"
"	popfw		# align stack					\n"
"	iretw								\n"
"									\n"
"	end_of_timeout_assembly:					\n"
"									\n"
	);
      }
  }

#endif /* TREAT_TIMER */


#if ASSEMBLY_TYPE == ASSEMBLY_DSES
int initial_open_door (void)
  {
  /* malloc is still not initialised here => no printf */
  if (BOOT1_DOS_running()) {
      unsigned short handle;

      if (   peekw_doorclosed (0x67*4) != 0
	  && (   DOS_OpenFile ("EMMXXXX0", DOS_open_read_only, &handle) == 0
	      || DOS_OpenFile ("EMMQXXX0", DOS_open_read_only, &handle) == 0)
	  && EMM386_check() != 0) {
	  unsigned short errorcode;

	  DOS_CloseFile (handle, &errorcode);
	  BOOT1_putstr (EMM386_CANNOT_RUN);
	  return 1;
	  }
      }

  if (segment_door (1) != 0) { /* before the following xchgl() */
      BOOT1_putstr (OPENDOOR_CANNOT_RUN);
      _BIOS_wait (4 * 1000 * 1000); /* 4 seconds */
      return 1;
      }
  return 0;
  }
#endif /* ASSEMBLY_TYPE == ASSEMBLY_DSES */

