/* kbd.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"
#if (SETUP & XCODE_SEGMENT)
#define CALLING_FROM_XCODESEG	/* before other includes */
#endif

#include "instboot.h"
#include "library.h"
#include "boot.h"	/* BOOT1_COM_port() */
#include "debug.h"
#include "bios.h"	/* _BIOS_nbtick(), _BIOS_equipment_flags(), CDROMbootspec */
#include "util.h"
#include "disk.h"
#include "kbd.h"
#include "user.h"
#include "messages.h"
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
#include "mouse.h"
#endif

#if USER_SUPPORT != 0
/*****************************************************************************************
 *     US Keyboard left to right, top to bottom.          ** Local/UK symbol drawn on key*
 *****************************************************************************************
 * key * Shift * Hex * keycode * Shift  *  Ctrl  *  Alt   **   Local  *  Shift  *  Alt   *
 *****************************************************************************************
 *  `  *   ~   *60/7E*  2960   *  297E  *  none  *  2900  **     `    *    ¬    *  none  * *
 *  1  *   !   *31/21*  0231   *  0221  *  none  *  7800  **     1    *    !    *  none  *
 *  2  *   @   *32/40*  0332   *  0340  *  0300  *  7900  **     2    *    "    *  none  * *
 *  3  *   #   *33/23*  0433   *  0423  *  none  *  7A00  **     3    *    £    *  none  * *
 *  4  *   $   *34/24*  0534   *  0524  *  none  *  7B00  **     4    *    $    *  none  *
 *  5  *   %   *35/25*  0635   *  0625  *  none  *  7C00  **     5    *    %    *  none  *
 *  6  *   ^   *36/5E*  0736   *  075E  *  071E  *  7D00  **     6    *    ^    *  none  *
 *  7  *   &   *37/26*  0837   *  0826  *  none  *  7E00  **     7    *    &    *  none  *
 *  8  *   *   *38/2A*  0938   *  092A  *  none  *  7F00  **     8    *    *    *  none  *
 *  9  *   (   *39/28*  0A39   *  0A28  *  none  *  8000  **     9    *    (    *  none  *
 *  0  *   )   *30/29*  0B30   *  0B29  *  none  *  8100  **     0    *    )    *  none  *
 *  -  *   _   *2D/5F*  0C2D   *  0C5F  *  0C1F  *  8200  **     -    *    _    *  none  *
 *  =  *   +   *3D/2B*  0D3D   *  0D2B  *  none  *  8300  **     =    *    +    *  none  *
 *****************************************************************************************
 * Right of key P:                                        **
 * key * Shift * Hex * keycode * Shift  *  Ctrl  *  Alt   **   Local  *  Shift  *  Alt   *
 *****************************************************************************************
 *  [  *   {   *5B/7B*  1A5B   *  1A7B  *  1A1B  *  1A00  **     [    *    {    *  none  *
 *  ]  *   }   *5D/7D*  1B5D   *  1B7D  *  1B1D  *  1B00  **     ]    *    }    *  none  *
 *  \  *   |   *5C/7C*  2B5C   *  2B7C  *  2B1C  *  2600  **     US keyboard only        * ?alt?
 *****************************************************************************************
 * Right of key L:                                        **
 * key * Shift * Hex * keycode * Shift  *  Ctrl  *  Alt   **   Local  *  Shift  *  Alt   *
 *****************************************************************************************
 *  ;  *   :   *3B/3A*  273B   *  273A  *  none  *  2700  **     ;    *    :    *  none  *
 *  '  *   "   *27/22*  2827   *  2822  *  none  *  2800  **     '    *    @    *  none  * *
 *  non US kbd only  *  2B5C   *  2B7C  *  2B1C  *  2B00  **     #    *    ~    *  none  *
 *****************************************************************************************
 * Left of key Z:                                         **
 * key * Shift * Hex * keycode * Shift  *  Ctrl  *  Alt   **   Local  *  Shift  *  Alt   *
 *****************************************************************************************
 *  non US kbd only  *  565C   *  567C  *  561C  *  5600  **     \    *    |    *  none  * *
 *  Warning: On some keyboards (a HP), this key generate 0x567C when alt is pressed, same as shift !
 *  Warning: On some BIOS (a COMPAQ P120), this key does not generate any keycode whatsoever!
 *****************************************************************************************
 * Right of key M                                         **
 * key * Shift * Hex * keycode * Shift  *  Ctrl  *  Alt   **   Local  *  Shift  *  Alt   *
 *****************************************************************************************
 *  ,  *   <   *2C/3C*  332C   *  333C  *  none  *  3300  **    ,     *    <    *  none  *
 *  .  *   >   *2E/3E*  342E   *  343E  *  none  *  3400  **    .     *    >    *  none  *
 *  /  *   ?   *2F/3F*  352F   *  353F  *  none  *  3500  **    /     *    ?    *  none  *
 *****************************************************************************************
 * To convert from AZERTY, you'll also need:              **
 * key * Shift * Hex * keycode * Shift  *  Ctrl  *  Alt   **   Local  *  Shift  *  Alt   *
 *****************************************************************************************
 *  q  *   Q   *71/51*  1071   *  1051  *  1011  *  1000  **    q     *    Q    *        *
 *  w  *   W   *77/57*  1177   *  1157  *  1117  *  1100  **    w     *    W    *        *
 *  a  *   A   *61/41*  1E61   *  1E41  *  1E01  *  1E00  **    a     *    A    *        *
 *  z  *   Z   *7A/5A*  2C7A   *  2C5A  *  2C1A  *  2C00  **    z     *    Z    *        *
 *  m  *   M   *6D/4D*  326D   *  324D  *  320D  *  3200  **    m     *    M    *        *
 *****************************************************************************************/

#if (SETUP & XDATA_SEGMENT)
#define XSEG() __attribute__((section (".xdata")))
#else
#define XSEG() /**/
#endif

/*
 *  For most keyboards, us,fr,uk,no excluded, data comes from book:
 * DOS The Complete Reference 4th edition, ISBN 0-07-881904-0 by Kris Jamsa
 * So this may need some corrections (quote, comma, apostrophe looks the same)...
 */
static const union kbd_map_u {
    struct keycode_str { unsigned short
	open_parenthesis,	close_parenthesis,	open_bracket,		close_bracket,
	open_accolade,		close_accolade,		inferior,		superior,
	plus,			minus,			multiply,		equal,
	slash,			backslash,		exclamation,		interrogation,
	diese,			percent,		ampersand,		underscore,
	comma,			semicolon,		dot,			colon,
	quote,			double_quote,		backquote,		tilda,
	circumflex,		arobasse,		tube,			dollard,

	pound,			euro,			cent,			yen, /**/
	currency,		cedillia,		plus_minus,		non_blank_space,
	math_not,		power1,			square,			cubic,
	open_quotes,		close_quotes,		degree,			degraa, /*?*/
	plain_tube,		inv_exclamation,	inv_interrogation,	inv_circumflex,
	apostrophe,		one_half,		one_quarter,		three_quarter,
	paragraph,		chapter,		mu,			trema,
	grave_a,		acute_a,		trema_a,		trema_A,
	round_a,		round_A,		ae,			AE,
	grave_e,		acute_e,		grave_E,		acute_E,
	grave_o,		acute_o,		trema_o,		trema_O,
	slash_o,		slash_O,		grave_I,		beta,
	grave_u,		acute_u,		trema_u,		trema_U,
	tilda_n,		tilda_N,		cedillia_c,		cedillia_C;
	} keycode;
    unsigned short array[sizeof(struct keycode_str) / sizeof(unsigned short)];
    }
    kbd_us_map XSEG() = { .keycode = {
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.backquote = 0x2960,	.tilda = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	.exclamation = 0x0221,
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.arobasse = 0x0340,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.diese = 0x0423,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	.dollard = 0x0524,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	.percent = 0x0625,
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.circumflex = 0x075E,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.ampersand = 0x0826,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.multiply = 0x092A,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.open_parenthesis = 0x0A28,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.close_parenthesis = 0x0B29,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.minus = 0x0C2D,	.underscore = 0x0C5F,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.equal = 0x0D3D,	.plus = 0x0D2B,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.open_bracket = 0x1A5B,	.open_accolade = 0x1A7B,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.close_bracket = 0x1B5D,	.close_accolade = 0x1B7D,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.semicolon = 0x273B,	.colon = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.quote = 0x2827,	.double_quote = 0x2822,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.backslash = 0x2B5C,	.tube = 0x2B7C, /* : those are US only */
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.comma = 0x332C,	.inferior = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.dot = 0x342E,		.superior = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.slash = 0x352F, .interrogation = 0x353F,

	/* still display those two with ASCII font - exception simplifying problems... */
	.inv_circumflex = 0x075E,	.plain_tube = 0x2B7C,

	/* as when typing alt-3digits on BIOS: */
	/* Assumes -finput-charset=UTF-8 which is GCC default */
	.grave_I = 'Ì',	.beta = 'ß',
	.grave_a = 'à', .acute_a = 'á', .trema_a = 'ä', .trema_A = 'Ä',
	.round_a = 'å', .round_A = 'Å', .ae = 'æ', .AE = 'Æ',
	.trema_o = 'ö', .trema_O = 'Ö', .slash_o = 'ø', .slash_O = 'Ø',
	.acute_e = 'é', .grave_e = 'è', .acute_E = 'É', .grave_E = 'È',
	.acute_o = 'ó', .grave_o = 'ò',
	.grave_u = 'ù', .acute_u = 'ú', .trema_u = 'ü', .trema_U = 'Ü',
	.tilda_n = 'ñ', .tilda_N = 'Ñ', .cedillia_c = 'ç', .cedillia_C = 'Ç',

	.pound = '£',	.cent = '¢',	.yen = '¥',
	.currency = '¤',	.cedillia = '¸',	.plus_minus = '±',

	.math_not = '¬',	.power1 = '¹',	.square = '²',	.cubic = '³',
	.open_quotes = '«', .close_quotes = '»',	.degree = 'º', .degraa = 'ª',
	.inv_exclamation = '¡',	.inv_interrogation = '¿',	.apostrophe = '´',
	.one_half = '½',	.one_quarter = '¼',		.three_quarter = '¾',
	.paragraph = '§',	.chapter = '¶',	.mu = 'µ',	.trema = '¨',
	.non_blank_space ='\xa0',
    }},
    kbd_ca_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.diese = 0x2960,	.tube = 0x297E,	.backslash = 0x2900,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	.plus_minus = 0x7800,
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,	.arobasse = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.slash = 0x0423,	.pound = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	.cent = 0x7B00,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	.currency = 0x7C00,
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.interrogation = 0x075E,	.math_not = 0x7D00,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.plain_tube = 0x7E00,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.square = 0x7F00,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.cubic = 0x8000,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.one_quarter = 0x8100,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	/* .three_half??? */
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.one_half = 0x8300,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.circumflex = 0x1A5B,	.inv_circumflex = 0x1A7B,	.open_bracket = 0x1A00,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.degraa = 0x1B5D,	.trema = 0x1B7D,	.close_bracket = 0x1B00,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.tilda = 0x2700,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.backquote = 0x2822,	.open_accolade = 0x2800,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.inferior = 0x2B5C,	.superior = 0x2B7C,	.close_accolade = 0x2B00,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.open_quotes = 0x565C,	.close_quotes = 0x567C,	.degree = 0x5600,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.comma = 0x333C, /* also */		/* .??? = 0x3300, */
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.dot = 0x343E, /* also */		/* .??? = 0x3400, */
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.acute_e = 0x352F,	.acute_E = 0x353F, /* .??? = 0x3500, */
	/* altM: */ .mu = 0x3200,
	/* altO: */ .paragraph = 0x1800,
	/* altP: */ .chapter = 0x1900,
    }},
    kbd_br_map XSEG() = { .keycode = { /* Brasil - i.e. Latin America - not an offence for Argentina... */
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.backslash = 0x2960,	.tube = 0x297E,	.math_not = 0x2900,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.ampersand = 0x075E,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.slash = 0x0826,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.open_parenthesis = 0x092A,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.close_parenthesis = 0x0A28,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.equal = 0x0B29,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.apostrophe = 0x0C2D,	.interrogation = 0x0C5F,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.inv_interrogation = 0x0D3D,	.inv_exclamation = 0x0D2B,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.quote = 0x1A5B,	.degree = 0x1A7B,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.plus = 0x1B5D,	.multiply = 0x1B7D,	.tilda = 0x1B00,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.tilda_n = 0x273B,	.tilda_N = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.open_accolade = 0x2827,	.open_bracket = 0x2822,	.circumflex = 0x2800,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.close_accolade = 0x2B5C,	.close_bracket = 0x2B7C,	.backquote = 0x2B00,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.minus = 0x352F,	.underscore = 0x353F,
	/* altQ: */ .arobasse = 0x1000,
	/* altM: */ .mu = 0x3200,
    }},
    kbd_uk_map XSEG() = { .keycode = {
	/* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.math_not = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.pound = 0x0423,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.arobasse = 0x2822,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.diese = 0x2B5C,	.tilda = 0x2B7C,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.backslash = 0x565C,	.tube = 0x567C,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.plain_tube = 0x2900, /* alt` */

	/* alt4: */ .euro = 0x7B00,
	/* altM: */ .mu = 0x3200,
    }},
    kbd_no_map XSEG() = { .keycode = {
	/* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.tube = 0x2960,	.paragraph = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,	.arobasse = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.pound = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	.currency = 0x0524,	.dollard = 0x7B00,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	.euro = 0x7C00,
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.ampersand = 0x075E,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.slash = 0x0826,	.open_accolade = 0x7E00,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.open_parenthesis = 0x092A, .open_bracket = 0x7F00,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.close_parenthesis = 0x0A28, .close_bracket = 0x8000,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.equal = 0x0B29,	.close_accolade = 0x8100,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.plus = 0x0C2D,	.interrogation = 0x0C5F,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.backslash = 0x0D3D,	.backquote = 0x0D2B,	.apostrophe = 0x8300,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.round_a = 0x1A5B,	.round_A = 0x1A7B,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.trema = 0x1B5D,	.circumflex = 0x1B7D,	.tilda = 0x1B00,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.slash_o = 0x273B, .slash_O = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.ae = 0x2827,	.AE = 0x2822,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.quote = 0x2B5C,	.multiply = 0x2B7C,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.minus = 0x352F,	.underscore = 0x353F,
	/* altM: */ .mu = 0x3200,
    }},
    kbd_fi_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.paragraph = 0x2960,	.one_half = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,	.arobasse = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.pound = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	.currency = 0x0524,	.dollard = 0x7B00,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.ampersand = 0x075E,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.slash = 0x0826,	.open_accolade = 0x7E00,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.open_parenthesis = 0x092A,	.open_bracket = 0x7F00,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.close_parenthesis = 0x0A28,	.close_bracket = 0x8000,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.equal = 0x0B29,	.close_accolade = 0x8100,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.interrogation = 0x0C5F,	.backslash = 0x8200,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.quote = 0x0D3D,	.backquote = 0x0D2B,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.round_a = 0x1A5B,	.round_A = 0x1A7B,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.trema = 0x1B5D,	.circumflex = 0x1B7D,	.tilda = 0x1B00,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.trema_o = 0x273B,	.trema_O = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.trema_a = 0x2827,	.trema_A = 0x2822,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.plus = 0x2B5C,		.multiply = 0x2B7C,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,	.tube = 0x5600,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.minus = 0x352F,	.underscore = 0x353F,
	/* altM: */ .mu = 0x3200,
    }},
    kbd_dk_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.one_half = 0x2960,	.paragraph = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,	.arobasse = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.pound = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	.currency = 0x0524,	.dollard = 0x7B00,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.ampersand = 0x075E,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.slash = 0x0826,	.open_accolade = 0x7E00,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.open_parenthesis = 0x092A,	.open_bracket = 0x7F00,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.close_parenthesis = 0x0A28,	.close_bracket = 0x8000,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.equal = 0x0B29,	.close_accolade = 0x8100,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.interrogation = 0x0C5F,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.quote = 0x0D3D,	.backquote = 0x0D2B,	.tube = 0x8300,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.round_a = 0x1A5B,	.round_A = 0x1A7B,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.trema = 0x1B5D,	.circumflex = 0x1B7D,	.tilda = 0x1B00,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.ae = 0x273B,	.AE = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.slash_o = 0x2827,	.slash_O = 0x2822,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.plus = 0x2B5C,	.multiply = 0x2B7C,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,	.backslash = 0x5600,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.minus = 0x352F,	.underscore = 0x353F,
	/* altM: */ .mu = 0x3200,
	}},
    kbd_nl_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.arobasse = 0x2960,	.paragraph = 0x297E,	.math_not = 0x2900,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	.power1 = 0x7800,
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,	.square = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.cubic = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	.one_quarter = 0x7B00,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	.one_half = 0x7C00,
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.ampersand = 0x075E,	.three_quarter = 0x7D00,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.underscore = 0x0826,	.pound = 0x7E00,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.open_parenthesis = 0x092A,	.open_accolade = 0x7F00,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.close_parenthesis = 0x0A28,	.close_accolade = 0x8000,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.apostrophe = 0x0B29,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.slash = 0x0C2D,	.interrogation = 0x0C5F,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.degree = 0x0D3D,	.tilda = 0x0D2B,	.cedillia = 0x8300,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.trema = 0x1A5B,	.circumflex = 0x1A7B,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.multiply = 0x1B5D,	.tube = 0x1B7D,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.plus = 0x273B,		.plus_minus = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.quote = 0x2827,	.backquote = 0x2822,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.inferior = 0x2B5C,	.superior = 0x2B7C,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.close_bracket = 0x565C,	.open_bracket = 0x567C,	.plain_tube = 0x5600,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.minus = 0x352F,	.equal = 0x353F,
	/* altM: */ .mu = 0x3200,
	/* altZ: */ .close_quotes = 0x2C00,
	/* altX: */ .open_quotes = 0x2D00,
	/* altS: */ .beta = 0x1F00,
	/* lacks \ , add as alt/ : */ .backslash = 0x8200
    }},
    kbd_sp_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.degree = 0x2960,	.degraa = 0x297E,	.backslash = 0x2900,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	.tube = 0x7800,
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,	.arobasse = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.quote = 0x0423,	.diese = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
// do not know!	.euro = 0x7B00,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.ampersand = 0x075E,	.math_not = 0x7D00,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.slash = 0x0826,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.open_parenthesis = 0x092A,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.close_parenthesis = 0x0A28,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.equal = 0x0B29,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.apostrophe = 0x0C2D,	.interrogation = 0x0C5F,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.inv_exclamation = 0x0D3D,	.inv_interrogation = 0x0D2B,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.backquote = 0x1A5B,	.circumflex = 0x1A7B,	.open_bracket = 0x1A00,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.plus = 0x1B5D,	.multiply = 0x1B7D,	.close_bracket = 0x1B00,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.tilda_n = 0x273B,	.tilda_N = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.open_accolade = 0x2800,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.cedillia_c = 0x2B5C,	.cedillia_C = 0x2B7C,	.close_accolade = 0x2B00,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.minus = 0x352F,	.underscore = 0x353F,
	/* altM: */ .mu = 0x3200,
    }},
    kbd_pt_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.backslash = 0x2960,	.tube = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,	.arobasse = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.pound = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	.paragraph = 0x7B00,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.ampersand = 0x075E,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.slash = 0x0826,	.open_accolade = 0x7E00,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.open_parenthesis = 0x092A,	.open_bracket = 0x7F00,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.close_parenthesis = 0x0A28,	.close_bracket = 0x8000,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.equal = 0x0B29,	.close_accolade = 0x8100,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.apostrophe = 0x0C2D,	.interrogation = 0x0C5F,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.open_quotes = 0x0D3D,	.close_quotes = 0x0D2B,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.plus = 0x1A5B,	.multiply = 0x1A7B,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.quote = 0x1B5D,	.backquote = 0x1B7D,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.cedillia_c = 0x273B,	.cedillia_C = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.degree = 0x2827,	.degraa = 0x2822,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.tilda = 0x2B5C,	.circumflex = 0x2B7C,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.minus = 0x352F,	.underscore = 0x353F,
	/* altM: */ .mu = 0x3200,
    }},
    kbd_it_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.backslash = 0x2960,	.tube = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.pound = 0x0423,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.ampersand = 0x075E,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.slash = 0x0826,	.open_accolade = 0x7E00,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.open_parenthesis = 0x092A,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.close_parenthesis = 0x0A28,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.equal = 0x0B29,	.close_accolade = 0x8100,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.quote = 0x0C2D,	.interrogation = 0x0C5F,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.grave_I = 0x0D3D,	.circumflex = 0x0D2B,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.grave_e = 0x1A5B,	.acute_e = 0x1A7B,	.open_bracket = 0x1A00,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.plus = 0x1B5D,		.multiply = 0x1B7D,	.close_bracket = 0x1B00,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.grave_o = 0x273B,	.cedillia_c = 0x273A,	.arobasse = 0x2700,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.grave_a = 0x2827,	.degree = 0x2822,	.diese = 0x2800,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.grave_u = 0x2B5C,	.paragraph = 0x2B7C,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.minus = 0x352F,	.underscore = 0x353F,
	/* altM: */ .mu = 0x3200,
    }},
//    kbd_xx_map XSEG() = { .keycode = {
//	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
//	open_parenthesis,	close_parenthesis,	open_bracket,		close_bracket,
//	open_accolade,		close_accolade,		inferior,		superior,
//	plus,			minus,			multiply,		equal,
//	slash,			backslash,		exclamation,		interrogation,
//	diese,			percent,		ampersand,		underscore,
//	comma,			semicolon,		dot,			colon,
//	quote,			double_quote,		backquote,		tilda,
//	circumflex,		arobasse,		tube,			dollard,
//
//	pound,			euro,			cent,			yen, /**/
//	currency,		cedillia,		plus_minus,		non_blank_space,
//	math_not,		power1,			square,			cubic,
//	open_quotes,		close_quotes,		degree,			degraa, /*?*/
//	plain_tube,		inv_exclamation,	inv_interrogation,	inv_circumflex,
//	apostrophe,		one_half,		one_quarter,		three_quarter,
//	paragraph,		chapter,		mu,			trema,
//	grave_a,		acute_a,		trema_a,		trema_A,
//	round_a,		round_A,		ae,			AE,
//	grave_e,		acute_e,		grave_E,		acute_E,
//	grave_o,		acute_o,		trema_o,		trema_O,
//	slash_o,		slash_O,		grave_I,		beta,
//	grave_u,		acute_u,		trema_u,		trema_U,
//	tilda_n,		tilda_N,		cedillia_c,		cedillia_C;
//    }},
    kbd_de_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.circumflex = 0x2960,	.degree = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,	.square = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.paragraph = 0x0423,	.cubic = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.ampersand = 0x075E,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.slash = 0x0826,		.open_accolade = 0x7E00,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.open_parenthesis = 0x092A,	.open_bracket = 0x7F00,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.close_parenthesis = 0x0A28,	.close_bracket = 0x8000,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.equal = 0x0B29,	.close_accolade = 0x8100,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.beta = 0x0C2D,		.interrogation = 0x0C5F,	.backslash = 0x8200,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.quote = 0x0D3D,	.backquote = 0x0D2B,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.trema_u = 0x1A5B,	.trema_U = 0x1A7B,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.plus = 0x1B5D,		.multiply = 0x1B7D,	.tilda = 0x1B00,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.trema_o = 0x273B,	.trema_O = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.trema_a = 0x2827,	.trema_A = 0x2822,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.diese = 0x2B5C,	.apostrophe = 0x2B7C,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,	.tube = 0x5600,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.comma = 0x332C,	.semicolon = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.dot = 0x342E,		.colon = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.minus = 0x352F,	.underscore = 0x353F,
	/* altQ: */ .arobasse = 0x1000,
	/* altM: */ .mu = 0x3200,
    }},
    kbd_ch_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/****** Encode here only the difference with a US keyboard *****/
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.paragraph = 0x2960,	.degree = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	.plus = 0x0221,	.tube = 0x7800,
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.double_quote = 0x0340,	.arobasse = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.multiply = 0x0423,	.diese = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	.cedillia_c = 0x0524,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.ampersand = 0x075E,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.slash = 0x0826,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.open_parenthesis = 0x092A,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.close_parenthesis = 0x0A28,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.equal = 0x0B29,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.backquote = 0x0C2D,	.interrogation = 0x0C5F,	.apostrophe = 0x8200,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */	// check the .inv_circumflex!!
	.circumflex = 0x0D3D,	.inv_circumflex = 0x0D2B,	.tilda = 0x8300,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.trema_u = 0x1A5B,	.grave_e = 0x1A7B,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.trema = 0x1B5D,	.exclamation = 0x1B7D,
	/* ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.trema_o = 0x273B,	.acute_e = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.trema_a = 0x2827,	.grave_a = 0x2822,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.dollard = 0x2B5C,	.pound = 0x2B7C,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,	.backslash = 0x5600,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.minus = 0x352F,	.underscore = 0x353F,

	/* Those do not seems to exist:
		<g> Show me your keyboard, I will say you who you are. </g>
	   Take the ones of German keyboard: */
	/* alt8: */ .open_bracket = 0x7F00,	/* alt9: */ .close_bracket = 0x8000,
	/* alt7: */ .open_accolade = 0x7E00,	/* alt0: */ .close_accolade = 0x8100,
	/* altM: */ .mu = 0x3200,
    }},
    kbd_fr_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.square = 0x2960,	.non_blank_space = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	.ampersand = 0x0221,
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.acute_e = 0x0340,	.tilda = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.double_quote = 0x0423,	.diese = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	.quote = 0x0524,	.open_accolade = 0x7B00,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	.open_parenthesis = 0x0625,	.open_bracket = 0x7C00,
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.minus = 0x075E,	.tube = 0x7D00,
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.grave_e = 0x0826,	.backquote = 0x7E00,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.underscore = 0x092A,	.backslash = 0x7F00,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.cedillia_c = 0x0A28,	.inv_circumflex = 0x8000, // two ^ on fr keyboard!
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.grave_a = 0x0B29,	.arobasse = 0x8100,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.close_parenthesis = 0x0C2D,	.degree = 0x0C5F,	.close_bracket = 0x8200,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.equal = 0x0D3D,	.plus = 0x0D2B,	.close_accolade = 0x8300,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.circumflex = 0x1A5B,	.trema = 0x1A7B,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.dollard = 0x1B5D,	.pound = 0x1B7D,	.currency = 0x1B00,
	/* *** invert M and ; *** ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.comma = 0x273B,	.interrogation = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.grave_u = 0x2827,	.percent = 0x2822,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.multiply = 0x2B5C,	.mu = 0x2B7C,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x332C,	.dot = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x342E,	.slash = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.exclamation = 0x352F,	.paragraph = 0x353F,
    }},
    kbd_be_map XSEG() = { .keycode = {
	/* altE: */ .euro = 0x1200, /* altY: */ .yen = 0x1500, /* altC: */ .cent = 0x2E00,
	/*  US key * shift US key * keycode * shift keycode * control keycode * alt keycode */
	/* ` * ~ * 0x2960 * 0x297E *  none  * 0x2900 */
	.square = 0x2960,	.cubic = 0x297E,
	/* 1 * ! * 0x0231 * 0x0221 *  none  * 0x7800 */
	.ampersand = 0x0221,	.tube = 0x7800,
	/* 2 * @ * 0x0332 * 0x0340 * 0x0300 * 0x7900 */
	.acute_e = 0x0340,	.arobasse = 0x7900,
	/* 3 * # * 0x0433 * 0x0423 *  none  * 0x7A00 */
	.double_quote = 0x0423,	.diese = 0x7A00,
	/* 4 * $ * 0x0534 * 0x0524 *  none  * 0x7B00 */
	.quote = 0x0524,
	/* 5 * % * 0x0635 * 0x0625 *  none  * 0x7C00 */
	.open_parenthesis = 0x0625,
	/* 6 * ^ * 0x0736 * 0x075E * 0x071E * 0x7D00 */
	.paragraph = 0x075E,	.inv_circumflex = 0x7D00, // two ^ on be keyboard!
	/* 7 * & * 0x0837 * 0x0826 *  none  * 0x7E00 */
	.grave_e = 0x0826,
	/* 8 * * * 0x0938 * 0x092A *  none  * 0x7F00 */
	.exclamation = 0x092A,
	/* 9 * ( * 0x0A39 * 0x0A28 *  none  * 0x8000 */
	.cedillia_c = 0x0A28,	.open_accolade = 0x8000,
	/* 0 * ) * 0x0B30 * 0x0B29 *  none  * 0x8100 */
	.grave_a = 0x0B29,
	/* - * _ * 0x0C2D * 0x0C5F * 0x0C1F * 0x8200 */
	.close_parenthesis = 0x0C2D,	.degree = 0x0C5F,	.close_accolade = 0x8200,
	/* = * + * 0x0D3D * 0x0D2B *  none  * 0x8300 */
	.minus = 0x0D3D,	.underscore = 0x0D2B,
	/* [ * { * 0x1A5B * 0x1A7B * 0x1A1B * 0x1A00 (right of P) */
	.circumflex = 0x1A5B,	.trema = 0x1A7B,	.open_bracket = 0x1A00,
	/* ] * } * 0x1B5D * 0x1B7D * 0x1B1D * 0x1B00 */
	.dollard = 0x1B5D,	.multiply = 0x1B7D,	.close_bracket = 0x1B00,
	/* *** invert M and ; *** ; * : * 0x273B * 0x273A *  none  * 0x2700 (right of L) */
	.comma = 0x273B,	.interrogation = 0x273A,
	/* ' * " * 0x2827 * 0x2822 *  none  * 0x2800 */
	.grave_u = 0x2827,	.percent = 0x2822,	.backquote = 0x2800,
	/* # * ~ * 0x2B5C * 0x2B7C * 0x2B1C * 0x2B00 (left of ENTER - UK symbols: non US keyboard only) */
	.mu = 0x2B5C,	.pound = 0x2B7C,	.backslash = 0x2B00,
	/* \ * | * 0x565C * 0x567C * 0x561C * 0x5600 (left of Z - UK symbols: non US keyboard only) */
	.inferior = 0x565C,	.superior = 0x567C,
	/* , * < * 0x332C * 0x333C *  none  *  3300 (right of M) */
	.semicolon = 0x332C,	.dot = 0x333C,
	/* . * > * 0x342E * 0x343E *  none  *  3400  */
	.colon = 0x342E,	.slash = 0x343E,
	/* / * ? * 0x352F * 0x353F *  none  *  3500  */
	.equal = 0x352F,	.plus = 0x353F,		.tilda = 0x3500,
    }};
  /* other known but not encoded: Czech, Slovak, Hungary, Poland, Yugoslavia
     mostly because I do not know how to put a name to describe the keys!!! */

/*
 * This function do not translate the ALT-letter keys for dvorak.
 * Dvorak cannot generate ^W and ^Z and more...
 */
KBD_FCT_PREFIX(kbd_translate) unsigned short
kbd_translate (unsigned short key)
  {
  unsigned short i, key_without_modifier = key & ~0x0060;

  if (kbdmap_is_QWERTZ(copy_gujin_param.kbdmap)) {
      if (key_without_modifier == 0x2C1A || key_without_modifier == 0x1519)
	  key ^= 0x2C1A ^ 0x1519;	/* invert z and y */
	else if (key == 0x2C00 || key == 0x1500)
	  key ^= 0x1500 ^ 0x2C00;	/* invert altZ and altY */
      }
    else if (kbdmap_is_AZERTY(copy_gujin_param.kbdmap)) {
      if (key_without_modifier == 0x1011 || key_without_modifier == 0x1E01)
	  key ^= 0x1011 ^ 0x1E01;	/* invert q and a */
	else if (key == 0x1000 || key == 0x1E00)
	  key ^= 0x1000 ^ 0x1E00;	/* invert altA and altQ */
	else if (key_without_modifier == 0x1117 || key_without_modifier == 0x2C1A)
	  key ^= 0x1117 ^ 0x2C1A;	/* invert z and w */
	else if (key == 0x1100 || key == 0x2C00)
	  key ^= 0x1100 ^ 0x2C00;	/* invert altZ and altW */
	/* Here we inverse _locally_ the two keys, do not care now
	   what is written on the AZERTY key position QWERTY M:
	   Note that we cannot produce Control M with AZERTY keyboard. */
	else if (key == 0x326D || key == 0x273B) /* QWERTY m and QWERTY ; */
	  key ^= 0x326D ^ 0x273B;
	else if (key == 0x324D || key == 0x273A) /* QWERTY M and QWERTY : */
	  key ^= 0x324D ^ 0x273A;
	else if (key == 0x3200 || key == 0x2700) /* QWERTY altM and QWERTY alt; */
	  key ^= 0x3200 ^ 0x2700;
	//else if (key == 320D) /* QWERTY ctrlM */
	//  key ^= 0x320D ^ 0x320D;
	else {
	  /* Here we invert the "top row digit on shift" braindamaged stuff: */
	  static const unsigned short azerty_shift_digit[] = {
		0x0231, 0x0221, /* 1 , ! */
		0x0332, 0x0340, /* 2 , @ */
		0x0433, 0x0423, /* 3 , # */
		0x0534, 0x0524, /* 4 , $ */
		0x0635, 0x0625, /* 5 , % */
		0x0736, 0x075E, /* 6 , ^ */
		0x0837, 0x0826, /* 7 , & */
		0x0938, 0x092A, /* 8 , * */
		0x0A39, 0x0A28, /* 9 , ( */
		0x0B30, 0x0B29  /* 0 , ) */
		};
	  for (i = 0; i < nbof (azerty_shift_digit); i++) {
	      if (azerty_shift_digit[i] == key) {
		  key = azerty_shift_digit[i ^ 1];
		  break;
		  }
	      }
	  }
      if (_BIOS_get_shift_flags().caps_lock_active) {
	  /* strange a keyboard, in'it? */
	  if (key == 0x324D || key == 0x326D)
	      key ^= 0x324D ^ 0x326D;
	    else if (key == 0x273A)
	      key = 0x273B;
	    else if (key == 0x273B)
	      key = 0x273A;
	  }
      }
#ifdef INCLUDE_DVORAK_KEYBOARD
    else if (kbdmap_is_DVORAK(copy_gujin_param.kbdmap)) {
      static const unsigned short dvorak_shift1[] = {
	/*   s  ->   o  ->   r  ->   p  ->   l  ->   n  ->   b  ->   x  ->   q    as control:*/
	  0x1F13, 0x180F, 0x1312, 0x1910, 0x260C, 0x310E, 0x3002, 0x2D18, 0x1011
	  };
      static const unsigned short dvorak_shift2[] = {
	/*   v  ->   k  ->   t  ->   y  ->   f  ->   u  ->   g  ->   i  ->   c  ->   j  ->   h  ->   d  ->   e    as control: */
	  0x2F16, 0x250B, 0x1414, 0x1519, 0x2106, 0x1615, 0x2207, 0x1709, 0x2E03, 0x240A, 0x2308, 0x2004, 0x1205
	  };
      static const unsigned short dvorak_shift3[] = {
	/*   ' ->   -  ->   [  ->   /  ->   z  ->   ;  ->   s */
	  0x2827, 0x0C2D, 0x1A5B, 0x352F, 0x2C7A, 0x273B, 0x1F73
	  };
      static const unsigned short dvorak_shift4[] = {
	/*   " ->   _  ->   {  ->   ?  ->   Z  ->   :  ->   S  */
	  0x2822, 0x0C5F, 0x1A7B, 0x353F, 0x2C5A, 0x273A, 0x1F53
	  };
      static const unsigned short dvorak_invert[] = {
	  0x1177, 0x332C, /*  w  <->  , */
	  0x1157, 0x333C, /*  W  <->  < */
	  /* dvorak cannot generate ^W */
	  0x1100, 0x3300, /*  altW  <->  alt< */
	  };
      static const unsigned short dvorak_replace[] = {
	  0x1071, 0x2827, /* q -> ' */
	  0x1051, 0x2822, /* Q -> " */
	  /* key == 0x1011 = ^Q on QWERTY not treated */
	  0x342E, 0x2F76, /* . -> v */
	  0x343E, 0x2F56, /* > -> V */
	  /* dvorak cannot generate ^V , it would be ctrl. on QWERTY */
	  0x1265, 0x342E, /* e -> . */
	  0x1245, 0x343E, /* E -> > */
	  };

      for (i = 0; i < nbof (dvorak_shift1) - 1; i++)
	  if (dvorak_shift1[i] == key_without_modifier) {
	      key = (key & ~key_without_modifier) | dvorak_shift1[i+1];
	      goto dvorak_found;
	      }
      for (i = 0; i < nbof (dvorak_shift2) - 1; i++)
	  if (dvorak_shift2[i] == key_without_modifier) {
	      key = (key & ~key_without_modifier) | dvorak_shift2[i+1];
	      goto dvorak_found;
	      }
      /* dvorak cannot generate ^Z , it would be ctrl/ on QWERTY */
      /* dvorak cannot generate ^S , it would be ctrl; on QWERTY */
      for (i = 0; i < nbof (dvorak_shift3) - 1; i++)
	  if (dvorak_shift3[i] == key) {
	      key = dvorak_shift3[i+1];
	      goto dvorak_found;
	      }
      for (i = 0; i < nbof (dvorak_shift4) - 1; i++)
	  if (dvorak_shift4[i] == key) {
	      key = dvorak_shift4[i+1];
	      goto dvorak_found;
	      }
      for (i = 0; i < nbof (dvorak_invert); i++)
	  if (dvorak_invert[i] == key) {
	      key = dvorak_invert[i ^ 1];
	      goto dvorak_found;
	      }
      for (i = 0; i < nbof (dvorak_replace); i += 2)
	  if (dvorak_replace[i] == key) {
	      key = dvorak_replace[i+1];
	      goto dvorak_found;
	      }
      dvorak_found:
      if (copy_gujin_param.kbdmap == kbd_dvorak) {
	    static const unsigned short dvorak_nonansi_invert[] = {
		0x1B7D, 0x0D2B, /* only non ANSI: }  <->  + */
		0x1B5D, 0x0D3D,  /* only non ANSI: ]  <->  = */
		};
	    for (i = 0; i < nbof (dvorak_nonansi_invert); i++)
		if (dvorak_nonansi_invert[i] == key) {
		    key = dvorak_nonansi_invert[i ^ 1];
		    break;
		    }
	    }
	  else /* if (copy_gujin_param.kbdmap == kbd_dvorak_ansi) */ {
	    static const unsigned short dvorak_ansi_replace[] = {
		0x1B5D, 0x1A7B, /* ] -> { */
		0x0C2D, 0x1B5D, /* - -> ] */
		0x0C5F, 0x1A5B, /* _ -> [ */
		};
	    for (i = 0; i < nbof (dvorak_ansi_replace); i += 2)
		if (dvorak_ansi_replace[i] == key) {
		    key = dvorak_ansi_replace[i+1];
		    break;
		    }
	    }
      }
#endif /* INCLUDE_DVORAK_KEYBOARD */
  /* Here at least the letters (and their shift, ctrl, alt)
     have the same keycode as a US keyboard, now treat symbols: */
  if (UTIL.keyptr != 0) {
      i = nbof (kbd_us_map.array);
      while (i--) {
	  if (UTIL.keyptr[i] == key)
#if (SETUP & XDATA_SEGMENT)
	      return peekw (xdata_adr (&kbd_us_map.array[i]));
#else
	      return kbd_us_map.array[i];
#endif
	  }
      }
  return key;
  }

KBD_FCT_PREFIX(kbd_setup) void
kbd_setup (enum kbd_enum kbdmap)
  {
#if 0	/* just space saving stuff... 320 bytes of code for gcc-3.3, not working for (SETUP & XDATA_SEGMENT) */
  switch (kbdmap) {
    case kbd_qwerty:
    case kbd_azerty:
    case kbd_qwertz:
#ifdef INCLUDE_DVORAK_KEYBOARD
    case kbd_dvorak:
    case kbd_dvorak_ansi:
#endif
    case kbd_unknown:
    case kbd_us: UTIL.keyptr = 0;	break;

    case kbd_ca: UTIL.keyptr = kbd_ca_map.array; break;
    case kbd_br: UTIL.keyptr = kbd_br_map.array; break;
    case kbd_uk: UTIL.keyptr = kbd_uk_map.array; break;
    case kbd_no: UTIL.keyptr = kbd_no_map.array; break;
    case kbd_fi: UTIL.keyptr = kbd_fi_map.array; break;
    case kbd_dk: UTIL.keyptr = kbd_dk_map.array; break;
    case kbd_nl: UTIL.keyptr = kbd_nl_map.array; break;
    case kbd_sp: UTIL.keyptr = kbd_sp_map.array; break;
    case kbd_pt: UTIL.keyptr = kbd_pt_map.array; break;
    case kbd_it: UTIL.keyptr = kbd_it_map.array; break;

    case kbd_fr: UTIL.keyptr = kbd_fr_map.array; break;
    case kbd_be: UTIL.keyptr = kbd_be_map.array; break;

    case kbd_de: UTIL.keyptr = kbd_de_map.array; break;
    case kbd_ch: UTIL.keyptr = kbd_ch_map.array; break;
    }
#else	/* just space saving stuff... */

#if (SETUP & XDATA_SEGMENT)
      if (UTIL.keyptr != 0) /* should not be called twice, but just in case */
	  FREE (UTIL.keyptr); /* try to remove malloc'ed memory hole */
      UTIL.keyptr = 0;
#endif

  if (kbdmap == kbd_unknown)
      UTIL.keyptr = 0;
#ifdef INCLUDE_DVORAK_KEYBOARD
    else if (kbdmap_is_DVORAK(kbdmap))
      UTIL.keyptr = 0;
#endif
    else if (kbdmap_is_QWERTY(kbdmap)) {
      static const union kbd_map_u *const qwerty_map[] = {
	  [kbdmap_subclass(kbd_qwerty)] = &kbd_us_map,
	  [kbdmap_subclass(kbd_us)] = &kbd_us_map,
	  [kbdmap_subclass(kbd_ca)] = &kbd_ca_map,
	  [kbdmap_subclass(kbd_br)] = &kbd_br_map,
	  [kbdmap_subclass(kbd_uk)] = &kbd_uk_map,
	  [kbdmap_subclass(kbd_no)] = &kbd_no_map,
	  [kbdmap_subclass(kbd_fi)] = &kbd_fi_map,
	  [kbdmap_subclass(kbd_dk)] = &kbd_dk_map,
	  [kbdmap_subclass(kbd_nl)] = &kbd_nl_map,
	  [kbdmap_subclass(kbd_sp)] = &kbd_sp_map,
	  [kbdmap_subclass(kbd_pt)] = &kbd_pt_map,
	  [kbdmap_subclass(kbd_it)] = &kbd_it_map,
	  };
      UTIL.keyptr = qwerty_map[kbdmap_subclass(kbdmap)]->array;
      }
    else if (kbdmap_is_AZERTY(kbdmap)) {
      static const union kbd_map_u *const azerty_map[] = {
	  [kbdmap_subclass(kbd_azerty)] = &kbd_fr_map,
	  [kbdmap_subclass(kbd_fr)] = &kbd_fr_map,
	  [kbdmap_subclass(kbd_be)] = &kbd_be_map,
	  };
      UTIL.keyptr = azerty_map[kbdmap_subclass(kbdmap)]->array;
      }
    else if (kbdmap_is_QWERTZ(kbdmap)) {
      static const union kbd_map_u *const qwertz_map[] = {
	  [kbdmap_subclass(kbd_qwertz)] = &kbd_de_map,
	  [kbdmap_subclass(kbd_de)] = &kbd_de_map,
	  [kbdmap_subclass(kbd_ch)] = &kbd_ch_map,
	  };
      UTIL.keyptr = qwertz_map[kbdmap_subclass(kbdmap)]->array;
      }

#if (SETUP & XDATA_SEGMENT)
  if (UTIL.keyptr) {
      void *ptr = MALLOC (sizeof (union kbd_map_u), "keyptr");
      if (ptr) {
	  lmemcpy (ptr, xdata_adr(UTIL.keyptr), sizeof (union kbd_map_u));
	  UTIL.keyptr = ptr;
	  }
	else
	  UTIL.keyptr = 0;
      }
#endif

#endif	/* just space saving stuff... */
  }

KBD_FCT_PREFIX(kbd_detect) unsigned
kbd_detect (void)
  {
  unsigned short key, realkey;
  enum kbd_enum old = copy_gujin_param.kbdmap, new_kbd = 0;

  copy_gujin_param.kbdmap = kbd_unknown;
  kbd_setup (kbd_unknown);
  printf (MSG_KEYBOARD_PRESS, 'Z');
  realkey = UI.function.getkey(15 * TICKS_PER_SECOND);
  key = realkey & 0xFF9F;
  if (key == 0x1519) {
      copy_gujin_param.kbdmap = kbd_qwertz;
      printf (MSG_KBD_NOW_PRESS, '$');
      realkey = key = UI.function.getkey(15 * TICKS_PER_SECOND);
      if (key == 0x0524) /* US: = shift4 */
	  new_kbd = kbd_de;
	else if (key == 0x2B5C) /* UK: # */
	  new_kbd = kbd_ch;
#if 0
	else if (key == 0x273B) /* US: ; */
	  new_kbd = kbd_pl;
#endif
	else
	  new_kbd = kbd_qwertz;
      }
    else if (key == 0x1117) {
      copy_gujin_param.kbdmap = kbd_azerty;
      printf (MSG_KBD_NOW_PRESS, '=');
      realkey = key = UI.function.getkey(15 * TICKS_PER_SECOND);
      if (key == 0x0D3D) /* US: = */
	  new_kbd = kbd_fr;
	else if (key == 0x352F) /* US: / */
	  new_kbd = kbd_be;
	else
	  new_kbd = kbd_azerty;
      }
    else if (key == 0x2C1A) {
      copy_gujin_param.kbdmap = kbd_qwerty;
      printf (MSG_KBD_NOW_PRESS, '@');
      realkey = key = UI.function.getkey(15 * TICKS_PER_SECOND);
      if (key == 0x0340)	/* shift2 */
	  new_kbd = kbd_us;
	else if (key == 0x2822) /* shift' */
	  new_kbd = kbd_uk;
	else if (key == 0x2700) /* right of L */
	  new_kbd = kbd_it;
	else if (key == 0x1000) /* altQ */
	  new_kbd = kbd_br;
	else if (key == 0x2960) /* ` */
	  new_kbd = kbd_nl;
	else if (key == 0x7900) { /* alt2 */
	  printf (MSG_KBD_NOW_PRESS, '+');
	  realkey = key = UI.function.getkey(15 * TICKS_PER_SECOND);
	  if (key == 0x1B5D) /* ] */
	      new_kbd = kbd_sp;
	    else if (key == 0x2B5C) { /* UK # */
	      printf (MSG_KBD_NOW_PRESS, '|');
	      realkey = key = UI.function.getkey(15 * TICKS_PER_SECOND);
	      if (key == 0x8300) /* alt= */
		  new_kbd = kbd_dk;
		else if (key == 0x5600) /* alt\ */
		  new_kbd = kbd_fi;
		else
		  new_kbd = kbd_qwerty;
	      }
	    else if (key == 0x1A5B) /* [ */
	      new_kbd = kbd_pt;
	    else if (key == 0x0C2D) /* - */
	      new_kbd = kbd_no;
	    else if (key == 0x0D2B) /* + */
	      new_kbd = kbd_ca;
	    else
	      new_kbd = kbd_qwerty;
	  }
	else
	  new_kbd = kbd_qwerty;
      }
#ifdef INCLUDE_DVORAK_KEYBOARD
    else if (key == 0x350F) {
      printf (MSG_KBD_NOW_PRESS, '[');
      realkey = key = UI.function.getkey(15 * TICKS_PER_SECOND);
      if (key == 0x0C5F)
	  new_kbd = kbd_dvorak_ansi;
      else /* if (key == 0x0C2D) */
	  new_kbd = kbd_dvorak;
      }
#endif
  if (new_kbd != 0) {
      print (": ");
      print (kbdname (new_kbd));
      copy_gujin_param.kbdmap = new_kbd;
      return 0;
      }
  printf (MSG_ERROR_KEYCODE, realkey);
  copy_gujin_param.kbdmap = old;
  kbd_setup (old);
  return 1;
  }

/** get_line() can be used for free text typing, but also to get numbers:
 **  If lowlimit = '0' and highlimit = '9' we can get an unsigned,
 **  If lowlimit = '9' and highlimit = '0' we allow as before but also a '-' sign in front.
 **  If lowlimit = 'f' and highlimit = '0' we allow an hexadecimal number IF prefixed with '0x'
 **  If lowlimit = 'F' and highlimit = '0' we allow a signed hexadecimal number, prefix '-0x' or '0x'
 **/
KBD_FCT_PREFIX(get_line) unsigned
get_line (char *buffer, unsigned maxlen, char display, unsigned char lowlimit, unsigned char highlimit)
  {
  unsigned len, cursor = 0; /* nb char to go backward from end of line */
  unsigned char row, col, replace = 0, quit = 0;
  unsigned char accept_leading_minus = 0, accept_hexadecimal = 0;
  char curs[2] = " ";
  struct optimise_redisplay_s {
      unsigned char no_change		: 1;
      unsigned char just_move_cursor	: 1;
      unsigned char simple_add_char	: 1;
      unsigned char delete_last		: 1;
      unsigned char cursor_positioned	: 1;
      unsigned char cursor_at_EOL	: 1;
      unsigned char force_clear		: 1;
      /* Maybe also only print the buffer starting from the cursor -> offset */
      } optimise_redisplay = {0};
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
  void (*save_print_field) (const char *str, struct mousefieldattr_str attr,
	unsigned short key, unsigned short row, unsigned short col)
	= MOUSE.function.print_field;
  MOUSE.function.print_field = 0; /* MOUSE_inited() will return false */
#endif


  if (lowlimit == '9' && highlimit == '0') {
      accept_leading_minus = 1;
      lowlimit = '0';
      highlimit = '9';
      }
    else if ((lowlimit == 'F' || lowlimit == 'f') && highlimit == '0') {
      accept_leading_minus = (lowlimit == 'F');
      accept_hexadecimal = 1;
      lowlimit = '0';
      highlimit = '9';
      }

  for (len = 0; (len < maxlen) && (buffer[len] != '\0'); len++)
      continue;

  if (UI.function.getcursor (&row,&col) != 0) {
      /* If getcursor not working, like bad serial VT100 emulations,
	we assume UI.parameter.{row,col} are up to date, set them
	manually if you know their value... */
      row = UI.parameter.row;
      col = UI.parameter.col;
      }

  for (;;) {
      unsigned short key, cpt;

      /* if there were a problem, try to disable optimisation by: * /
      optimise_redisplay = (struct optimise_redisplay_s) {0}; */

      if (optimise_redisplay.no_change)
	  optimise_redisplay.no_change = 0;
	else {
	  if (optimise_redisplay.just_move_cursor || optimise_redisplay.delete_last) {
	      optimise_redisplay.just_move_cursor = 0;
	      if (UI.parameter.attr.isgraphic)
		  print (curs);	/* remove old cursor, standard bgcolor */
	      /* more treatment for optimise_redisplay.delete_last later */
	      }
	    else if (optimise_redisplay.simple_add_char) {
	      optimise_redisplay.simple_add_char = 0;
	      if (!UI.parameter.attr.isgraphic)
		  optimise_redisplay.cursor_positioned = 1;	/* but not drawn */
	      if (display != 0)
		  curs[0] = display;
		else
		  curs[0] = buffer[len - cursor - 1];
	      print (curs);	/* overwrite old cursor */
	      }
	    else {
	      unsigned nb_end_space;
	      UI.function.setcursor (row, col);
	      /* FIXME: We rely on VT serial terminal being in auto-newline here... */
	      if (display != 0) {
		  curs[0] = display;
		  for (cpt = 0; cpt < len; cpt++)
		      print (curs);
		  }
		else
		  print (buffer);
	      nb_end_space = maxlen - len;
	      if (col + maxlen >= UI.parameter.nbcol && len + col + 2 < UI.parameter.nbcol
			&& !optimise_redisplay.force_clear)
		  nb_end_space = UI.parameter.nbcol - col - len - 1;
	      optimise_redisplay.force_clear = 0;
	      while (nb_end_space--)
		  print (" ");
	      }
	  if (optimise_redisplay.cursor_positioned)
	      optimise_redisplay.cursor_positioned = 0;
	    else { /* line edited may be larger than screen line length: */
	      unsigned short tmp = col + len - cursor;
	      unsigned char realrow = row + tmp / UI.parameter.nbcol;
	      unsigned char realcol = tmp % UI.parameter.nbcol;
	      UI.function.setcursor (realrow, realcol);

	      if (optimise_redisplay.delete_last) {
		  optimise_redisplay.delete_last = 0;
		  /* Try to not rely on ^H to go backward, mostly at beginning of line */
		  print (" ");
		  UI.function.setcursor (realrow, realcol);
		  }
	      if (UI.parameter.attr.isgraphic) {
		  unsigned oldbgcolor = UI.bgcolor;
		  if (cursor == 0)
		      curs[0] = ' ';
		    else if (display != 0)
		      curs[0] = display;
		    else
		      curs[0] = buffer[len - cursor];
		  UI.function.setbgcolor (UI.cursor_bgcolor);
		  print (curs);
		  UI.function.setbgcolor (oldbgcolor);
		  /* We can handle ^H in graphic modes */
		  print ("\010");	/* row and col are same in text and graphic modes */
		  }
	      }
	  optimise_redisplay.cursor_at_EOL = ((col + len - cursor + 1) % UI.parameter.nbcol == 0);
	  }
      if (quit)	{ /* returns only with the cursor at end, so "\r\n press a key to continue" is OK */
#if USER_SUPPORT & (BIOS_MOUSE_SUPPORT|SERIAL_MOUSE_SUPPORT|JOYSTICK_SUPPORT)
	  MOUSE.function.print_field = save_print_field;
#endif
	  return quit - 1;
	  }
      key = UI.function.getkey(15 * TICKS_PER_SECOND);
      if (_BIOS_get_shift_flags().num_lock_active) {
	  static const unsigned short numpad_keycode[] = {
	      NUMPAD_DIGIT_KEYCODE(0), NUMPAD_DIGIT_KEYCODE(1), NUMPAD_DIGIT_KEYCODE(2), NUMPAD_DIGIT_KEYCODE(3),
	      NUMPAD_DIGIT_KEYCODE(4), NUMPAD_DIGIT_KEYCODE(5), NUMPAD_DIGIT_KEYCODE(6), NUMPAD_DIGIT_KEYCODE(7),
	      NUMPAD_DIGIT_KEYCODE(8), NUMPAD_DIGIT_KEYCODE(9), KP_DOT_KEYCODE(),
	      KP_DIVIDE_KEYCODE(), KP_MULTIPLY_KEYCODE(), KP_MINUS_KEYCODE(), KP_PLUS_KEYCODE(), KP_RETURN_KEYCODE()
	      };
	  static const unsigned short numpad_translate[] = {
	      DIGIT_KEYCODE(0), DIGIT_KEYCODE(1), DIGIT_KEYCODE(2), DIGIT_KEYCODE(3),
	      DIGIT_KEYCODE(4), DIGIT_KEYCODE(5), DIGIT_KEYCODE(6), DIGIT_KEYCODE(7),
	      DIGIT_KEYCODE(8), DIGIT_KEYCODE(9), DOT_KEYCODE(),
	      DIVIDE_KEYCODE(), MULTIPLY_KEYCODE(), MINUS_KEYCODE(), PLUS_KEYCODE(), RETURN_KEYCODE()
	      };
	  unsigned short i = nbof (numpad_keycode) + 1;
	  /* Who is the more foolish, the fool or the one who follows him? */
	  while (i-- > 0)
	      if (key == numpad_keycode[i]) {
		  key = numpad_translate[i];
		  break;
		  }
	  }

      if (   (key & 0xFF00) == RETURN_KEYCODE() || (key & 0x00FF) == '\r'
	  || (key & 0xFF00) == KP_RETURN_KEYCODE() || (key & 0x00FF) == '\n') {
	  quit = 1;
	  cursor = 0;
	  continue;
	  }
	else if (key == TIMEOUT_KEYCODE() || (key & 0xFF00) == ESC_KEYCODE() || (key & 0x00FF) == '\033') {
	  quit = 2;
	  cursor = 0;
	  continue;
	  }
	else if ((key & 0xFF00) == LEFT_KEYCODE() || (key & 0x00FF) == '\002') {
	  if (cursor < len)
	      cursor++;
	  optimise_redisplay.just_move_cursor = !optimise_redisplay.cursor_at_EOL;
	  }
	else if ((key & 0xFF00) == RIGHT_KEYCODE() || (key & 0x00FF) == '\006') {
	  if (cursor > 0)
	      cursor--;
	  optimise_redisplay.just_move_cursor = !optimise_redisplay.cursor_at_EOL;
	  }
	else if ((key & 0xFF00) == HOME_KEYCODE() || (key & 0x00FF) == '\001') {
	  cursor = len;
	  optimise_redisplay.just_move_cursor = !optimise_redisplay.cursor_at_EOL;
	  }
	else if ((key & 0xFF00) == END_KEYCODE() || (key & 0x00FF) == '\005') {
	  cursor = 0;
	  optimise_redisplay.just_move_cursor = !optimise_redisplay.cursor_at_EOL;
	  }
	else if ((key & 0xFF00) == INSERT_KEYCODE()) {
	  replace = !replace;
	  optimise_redisplay.no_change = 1;
	  }
	else if ((key & 0xFF00) == BACKSPACE_KEYCODE() || (key & 0x00FF) == '\010') {
	  if (cursor < len) {
	      for (cpt = len - cursor - 1; cpt < len; cpt++)
		  buffer[cpt] = buffer[cpt + 1];
	      len --;
	      if (cursor == 0)
		  optimise_redisplay.delete_last = !optimise_redisplay.cursor_at_EOL;
	      }
	  }
	else if ((key & 0xFF00) == DELETE_KEYCODE() || (key & 0x00FF) == '\177' || (key & 0x00FF) == '\004') {
	  if (cursor > 0) {
	      for (cpt = len - cursor; cpt < len; cpt++)
		  buffer[cpt] = buffer[cpt + 1];
	      cursor--;
	      len --;
	      }
	    else /* if (cursor == 0) */
	      optimise_redisplay.delete_last = 1;
	  }
	else if ((key & 0xFF00) == PAGEUP_KEYCODE() || (key & 0x00FF) == '\020') {
	  optimise_redisplay.no_change = 1;
	  }
	else if ((key & 0xFF00) == PAGEDOWN_KEYCODE() || (key & 0x00FF) == '\016') {
	  optimise_redisplay.no_change = 1;
	  }
	else if ((key & 0xFF00) == UP_KEYCODE()) {
	  optimise_redisplay.no_change = 1;
	  }
	else if ((key & 0xFF00) == DOWN_KEYCODE()) {
	  optimise_redisplay.no_change = 1;
	  }
	else if ((key & 0x00FF) == '\013') { /* ^K to kill end of line */
	  len = len - cursor;	/* we'll do buffer[len] = '\0'; later */
	  cursor = 0;
	  optimise_redisplay.force_clear = 1;
	  }
	else if ((key & 0x00FF) == '\027') { /* ^W to delete word */
	  int nbchar = len - cursor - 1;
	  while (nbchar >= 0 && (buffer[nbchar] == ' ' || buffer[nbchar] == '\t'))
	      nbchar--;
	  while (nbchar >= 0 && !(buffer[nbchar] == ' ' || buffer[nbchar] == '\t'))
	      nbchar--;
	  nbchar = (len - cursor) - nbchar;
	  if (buffer[len - cursor] != ' ')
	      nbchar--;
	  for (cpt = len - cursor - nbchar; cpt < len - nbchar; cpt++)
	      buffer[cpt] = buffer[cpt + nbchar];
	  len -= nbchar;
	  optimise_redisplay.force_clear = 1;
	  }
	/* Shall we manage ^T or others, see man bash | grep "C-" */
	else {
	  unsigned has_hexa_header (char *buffer, unsigned len) {
		if (buffer[0] == '-')
		    return (len >= 3) && buffer[1] == '0' && (buffer[2] == 'x' || buffer[2] == 'X');
		  else
		    return (len >= 2) && buffer[0] == '0' && (buffer[1] == 'x' || buffer[1] == 'X');
		}
	  if (len >= maxlen - 1 && (!replace || cursor == 0)) {
	      /* only delete/backspace possible */
	      optimise_redisplay.no_change = 1;
	      }
	    else if (  ((key & 0xFF) >= lowlimit && (key & 0xFF) <= highlimit)
		     || (accept_leading_minus && (key & 0xFF) == '-' && len == 0)
		     || (accept_hexadecimal && ((key & 0xFF) == 'x' || (key & 0xFF) == 'X') && (len == ((buffer[0] == '-')? 2 : 1)) && buffer[len-1] == '0')
		     || (accept_hexadecimal && has_hexa_header (buffer, len) && (   ((key & 0xFF) >= 'a' && (key & 0xFF) <= 'f')
										 || ((key & 0xFF) >= 'A' && (key & 0xFF) <= 'F')))) {
		/* FIXME: previous test is not foolproof */
	      if (!replace || cursor == 0) {
		  for (cpt = len; cpt > len - cursor; cpt--)
		      buffer[cpt] = buffer[cpt - 1];
		  len++;
		  if (cursor == 0 && len + col + 2 != UI.parameter.nbcol) /* line extention */
		      optimise_redisplay.simple_add_char = !optimise_redisplay.cursor_at_EOL;
		  }
		else {
		  cpt = len - cursor--;
		  optimise_redisplay.simple_add_char = !optimise_redisplay.cursor_at_EOL;
		  }
	      buffer[cpt] = key & 0xFF;
	      }
	    else
	      print ("\007");
	  }
      buffer[len] = '\0';
      }
  }

/**
 ** When needing a number.
 **/
/* FIXME: this function is misnamed, should be get_unsigned() - it accepts
   hexadecimal since Gujin-1.6 but only display the default in decimal */
KBD_FCT_PREFIX(get_number) unsigned
get_number (unsigned deflt, unsigned max)
  {
  /* 0xFFFFFFFF = 4294967295 : 10 digit ; 0x80000001 = -2147483649 : 11 chars */
  char buffer[11]; /* 10 digit with ending zero, enought for unsigned decimal & hexadecimal */
  unsigned ret =  1000000000, len = 0, tmp = deflt;

  for (;;) {
      unsigned div = tmp / ret;
      tmp = tmp % ret;
      ret /= 10;
      if (div != 0 || len != 0 || ret == 0)
	  buffer[len++] = div + '0';
      if (ret == 0) {
	  buffer[len] = '\0';
	  break;
	  }
      }
   /* What to do when the user validate an empty string as number?
      It may be not equivalent to type zero... */
  if (get_line (buffer, sizeof (buffer), '\0', 'f', '0') || buffer[0] == '\0')
      return deflt;
  ret = 0;
  if (buffer[0] == '0' && (buffer[1] == 'x' || buffer[1] == 'X')) {
      for (len = 2; buffer[len] != '\0'; len++) {
	  int hex = hexval (buffer[len]);
	  if (hex < 0)
	      break;
	  ret = (ret << 4) + hex;
	  }
      }
    else {
      for (len = 0; buffer[len] != '\0'; len++) {
	  int hex = hexval (buffer[len]);
	  if (hex < 0 || hex >= 0xA)
	      break;
	  ret = 10 * ret + hex;
	  }
      }
  return (ret > max)? deflt : ret;
  }

KBD_FCT_PREFIX(adjust_extended_partition_length) void
adjust_extended_partition_length (void)
{
  struct diskparam_str *dp;

  for (dp = &DI.param[0]; dp < &DI.param[DI.nbdisk]; dp++) {
      unsigned cpt, extended_partition;
      /* Anyways we are limited to 32 bits for the result: */
      unsigned minlba = 0, maxlba = (unsigned)(dp->nbtotalsector - 1), lba, best = maxlba;
      bootafter_t *bootafter = (bootafter_t *)(fourKbuffer + 0x1B8);
      struct bootsect_partition_str *bptr;

      for (cpt = 0; cpt < dp->nbpartition; cpt++)  /* primary: OSnumber is 1..4 */
	  if (dp->partition[cpt].OSnumber <= 4 && dp->partition[cpt].misc.active == part_extended)
	      break;
      if (cpt >= dp->nbpartition) {
	  printf (EP_NO_EXTENDED_PARTITION_FOUND, dp->diskname);
	  continue;
	  }
      extended_partition = cpt;

      if (DI.readsector (dp, -1, 0, 1, data_adr(fourKbuffer)) != 0 || bootafter->Signature0xAA55 != 0xAA55) {
	  printf (EP_CANNOT_READ_MBR, dp->diskname);
	  continue;
	  }

      for (bptr = bootafter->bootsect_partition; bptr < &bootafter->bootsect_partition[4]; bptr++) {
	  if ((bptr->indicator & 0x7F) != 0)
	      continue;
	  if (bptr->nb_sector_before != dp->partition[extended_partition].start)
	      continue;
	  if (bptr->nb_sector != dp->partition[extended_partition].length)
	      continue;
	  if (bptr->system_indicator != DOS_extended
	    && bptr->system_indicator != Win95_extendedLBA
	    && bptr->system_indicator != Linux_extended)
	      continue;
	  break;
	  }
      if (bptr >= &bootafter->bootsect_partition[4]) {
	  printf (EP_NO_CORRESPONDING_EXTENDED_PARTITION, dp->diskname);
	  continue;
	  }

      for (cpt = 0; cpt < dp->nbpartition; cpt++) {
	  if (dp->partition[cpt].misc.beer_partition)
	      continue;
	  if (dp->partition[cpt].start > dp->partition[extended_partition].start + dp->partition[extended_partition].length) {
	      /* search the first sector of the first partition after the extended one: */
	      if (maxlba > (unsigned)(dp->partition[cpt].start - 1))
		  maxlba = (unsigned)(dp->partition[cpt].start - 1);
	      }
	    else if (dp->partition[cpt].start > dp->partition[extended_partition].start) {
	      /* search the last sector of the last partition inside the extended one: */
	      if (minlba < (unsigned)(dp->partition[cpt].start + dp->partition[cpt].length))
		  minlba = (unsigned)(dp->partition[cpt].start + dp->partition[cpt].length);
	      }
	  }
      printf (EP_ENTER_NEW_VALUE_CURRENT_MIN_MAX,
		dp->diskname,
		dp->partition[extended_partition].start + dp->partition[extended_partition].length,
		minlba, maxlba);

      {
      unsigned short unit;
      unsigned rounded;

      if (bptr->end_sector == 0)
	  bptr->end_sector = dp->bios_nbsectorpertrack ?: 63;
      if (bptr->end_head == 0)
	  bptr->end_head = dp->bios_maxhead ?: 254; /* newer LBA disks have usually 255 heads */
      unit = bptr->end_sector * (bptr->end_head + 1);

      if (best == maxlba) {
	  /* maxlba did not change, extended partition is the last one in LBA position,
		keep some space for B.E.E.R. partition: */
	  best -= (64 * 1024 * 1024) / dp->bytepersector;
	  best = (best / unit) * unit;
	  }
	else
	  best = 0; // -= bptr->end_sector;
	/* get_number() needs getcursor to work, possible problem on VT emulations */
      lba = get_number (best, 0xFFFFFFFF);
      if (lba < minlba || lba > maxlba) {
	  puts (EP_OUT_OF_RANGE);
	  continue;
	  }
      rounded = (lba / unit) * unit;
      if (rounded < minlba)
	 rounded += unit;
      if (lba != rounded && rounded >= minlba && rounded < maxlba) {
	  unsigned short key;
	  printf (EP_ROUND_VALUE_U_TO_U, lba, rounded);
	  key = UI.function.getkey(20 * TICKS_PER_SECOND) & 0xFF;
	  if (key == 'Y' || key == 'y' || key == '\r' || key == '\n') {
	      lba = rounded;
	      print ("Y");
	      }
	    else
	      print ("N");
	  printf (EP_USING_VALUE_U, lba);
	  }

      bptr->nb_sector = lba - bptr->nb_sector_before;
      if (lba / unit <= 1023) {
	  unsigned short cylinder = lba / unit, remainder = lba % unit;
	  bptr->end_cylinder_lsb = cylinder & 0xFF;
	  bptr->end_cylinder_msb = cylinder >> 8;
	  bptr->end_head = remainder / bptr->end_sector;
	  bptr->end_sector = remainder % bptr->end_sector + 1;
	  }
      }
      if (DI.writesector (dp, -1, 0, 1, data_adr(fourKbuffer)) != 0)
	  puts (EP_WRITE_FAILED);
	else
	  puts (EP_DONE);
      }
}

#endif /* USER_SUPPORT != 0 */
