/* instboot.h */
#ifndef INSTBOOT_H
#define INSTBOOT_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/**
 ** This file can be compiled in real mode and in protected mode,
 ** take care...
 **/

/*
 * The boot sector contains this structure at its beginning:
 */
typedef struct {
    unsigned char	jmpinstruction[2];
    unsigned char	EBIOSaccess;	/* 0x90 = nop if BIOS only; 0x0E = Win95_FAT16LBA if EBIOS accessible */
    unsigned char	String[8];	/* blank padded ! */
    unsigned short	Bytepersector;
    unsigned char	Sectorpercluster;
    unsigned short	Reservedsector;	/* nb sectors before FAT */
    unsigned char	NbFAT;
    unsigned short	NbRootdirentry;	/* = 0 for FAT32 */
    unsigned short	NbTotalsector;	/* = 0 if >= 0x10000 or FAT32 */
    unsigned char	Mediadescriptor; /* Some people use 0xF8 for non removeable, 0xF0 for removeable... */
    unsigned short	NbSectorperFAT;	/* = 0 for FAT32 */
    /* DOS 3+ : */
    unsigned short	NbSectorpertrack;
    unsigned short	NbHead;
    unsigned		NbHiddensector;	/* nb of sectors before this partition */
    /* DOS 4+ : */
    unsigned		NbTotalsector2;
    } __attribute__ ((packed, aligned(1))) FAT_bootsect1_t;

typedef struct {
    unsigned		NbSectorperFAT2;
    struct FAT32_flags_str {
	unsigned short	active_fat	: 4;
	unsigned short	reserved1	: 3;
	unsigned short	one_fat_active	: 1;
	unsigned short	reserved2	: 8;
	} __attribute__ ((packed)) flags;
    unsigned short	version;
    unsigned		root_cluster; /* usually 2 */
    unsigned short	FSinfosector; /* below Reservedsector */
    unsigned short	save_bootrecord; /* usually 6 if != 0 */
    unsigned char	reserved[12];	/* fill with 0 */
    } __attribute__ ((packed, aligned(1))) FAT32_bootsect_t;

typedef struct {
    unsigned char	PhysicaldriveNb;
    unsigned char	dirty_flag	: 1;	/* need checkdisk at next boot */
    unsigned char	surface_scan	: 1;	/* checkdisk shall also do surface scan */
    unsigned char	FATreserved	: 6;	/* was current head, head for the boot sector */
    unsigned char	Signaturebyte0x29;	/* can also be 0x28 for Window NT */
    unsigned		VolumeserialNb;
    unsigned char	Volumelabel[11];	/* blank padded ! */
    unsigned char	FileSysName[8];		/* blank padded ! */
    } __attribute__ ((packed, aligned(1))) FAT_bootsect2_t;

typedef struct {
    FAT_bootsect1_t	part1;
    FAT_bootsect2_t	part2;
    } __attribute__ ((packed, aligned(1))) bootbefore_t;

typedef struct {
    FAT_bootsect1_t	part1;
    FAT32_bootsect_t	partFAT32;
    FAT_bootsect2_t	part2;
    } __attribute__ ((packed, aligned(1))) bootbefore_FAT32_t;

/*
 * The "diskread" command structure for disk interface:
 */
typedef struct {
    struct {
	unsigned short next;
	unsigned short checksum;	/* sum of sector + checksum = 0 */
	unsigned short nbword;		/* size to calculate checksum */
	} __attribute__ ((packed)) header;

    union diskcmd {
	struct diskcmd_bios {	/* 12 bytes, used if reg_ah != 0xF0 */
	    unsigned short	reg_dx;
	    unsigned short	reg_ax;
	    unsigned short	reg_cx;
	    /* no need to initialise the following, just a remainder: */
	    /* but checksum of boot1 is recalculated if first attempt failed! */
	    unsigned short	dummy;
	    unsigned long	will_be_set_to_esbx;
	    } __attribute__ ((packed)) int1302;
	struct diskcmd_ebios {	/* 20 bytes, used (logically) if reg_ah = 0x42, 0x43... */
	    unsigned short	reg_dx;
	    unsigned short	reg_ax;
	    unsigned char	cst_16;
	    unsigned char	cst_0;		/* has to be zero */
	    unsigned char	nb_block;	/* max 127 */
	    unsigned char	cst2_0;		/* has to be zero */
	    unsigned short	offset, segment;
	    unsigned long long	lba;
	    } __attribute__ ((packed)) int1342;
	struct diskcmd_hard {	/* 20 bytes, used if reg_ah == 0xF0 */
	    unsigned short	ide_dcr_address; /* usually base_ide+0x206 */
	    unsigned short	reg_ax;		/* %al has to be 0x0A */
	    unsigned short	base_ide_plus7;	/* added by 7, i.e. 1F7, 3F7 */
	    unsigned char	lba_head;
	    unsigned char	lba48_cylinder_high; /* lba48_* ignored if lba28 */
	    unsigned char	lba48_cylinder_low;
	    unsigned char	lba48_sector;
	    unsigned char	lba48_nb_sect;
	    unsigned char	cylinder_high;
	    unsigned char	cylinder_low;
	    unsigned char	sector;
	    unsigned char	nb_sect;
	    unsigned char       ide_command;    /* to read with retry, 0x20 */
	    unsigned short	nb_sect_read;	/* keep it equal to nb_sect */
	    unsigned short	sector_size_div2;
	    } __attribute__ ((packed)) hardide;
	} bootloader2_cmd;
    } __attribute__ ((packed, aligned(1))) bootloader2_t;

/*
 * The parameter block used by bootloader 1 to get
 * bootloader 2 in memory. Its size can not be modified !
 * Checksum calculus (also in instboot.c) is done on 16 bits,
 * so "boot1_checksum" has to be aligned like symbol "start",
 * and like symbol "bootend".
 */
typedef struct {
    unsigned short	boot1_checksum;
    bootloader2_t	bootloader2;
    } __attribute__ ((packed, aligned(2))) boot1param_t;

/*
 * To use the serial (BIOS) interface, you need hardware flow control,
 * i.e. either a true VT connected or a null-modem cable (DTR-> DCD, RI,
 * CTS, DSR), and fill in the following structure (serialconf) with
 * the configuration you need and the port in serialconf_port_plus_1.
 * Minicom works on another Linux box with a standard twisted cable.
 */
typedef struct {
    unsigned char	eightbit   :1;	/* else seven bits */
    unsigned char	constant_1 :1;
    unsigned char	doublestop :1;	/* else one stop bit */
    unsigned char	parity     :1;	/* else no parity generation/check */
    unsigned char	even       :1;	/* else odd */
    enum { B110 = 0, B150, B300, B600, B1200, B2400, B4800, B9600 }
			speed      :3;
    unsigned char	constant_0;	/* loaded in %ah before INT 0x14 */
    } __attribute__ ((packed, aligned(1))) serialconf_t;

/*
 * The boot sector contains this structure at its end:
 */

/*
 * The list of partition kind:
 */
#define DOS_FAT12_PARTITIONLIST() \
	P (DOS_FAT12,		0x1,	"FAT12")		\
	P (HiddenDOS_FAT12,	0x11,	"Hidden FAT12")

#define DOS_FAT16_PARTITIONLIST() \
	/* below 32Mb partition : */				\
	P (DOS_FAT16,		0x4,	"FAT16")		\
	P (HiddenDOS_FAT16,	0x14,	"Hidden FAT16")		\
	/* over 32Mb partition : */				\
	P (BIG_DOS_FAT16,	0x6,	"Big FAT16")		\
	P (HiddenBIG_DOS_FAT16,	0x16,	"Hidden Big FAT16")	\
	P (Win95_FAT16LBA,	0x0E,	"FAT16 LBA")		\
	P (HiddenWin95_FAT16LBA,0x1E,	"Hidden FAT16 LBA")

#define DOS_FAT32_PARTITIONLIST() \
	P (Win95_FAT32,		0x0B,	"FAT32")		\
	P (Win95_FAT32LBA,	0x0C,	"FAT32 LBA")		\
	P (HiddenWin95_FAT32,	0x1B,	"Hidden FAT32")		\
	P (HiddenWin95_FAT32LBA,0x1C,	"Hidden FAT32 LBA")

#define NTFS_PARTITIONLIST() \
	P (OS2_HPFS_NTFS,	0x7,	"NTFS/HPFS")		\
	P (Hidden_NTFS,		0x17,	"Hidden NTFS")

#define EXTENDED_PARTITIONLIST() \
	P (DOS_extended,	0x5,	"extended")		\
	P (Win95_extendedLBA,	0x0F,	"extended LBA")		\
	P (Linux_extended,	0x85,	"Linux extended")	\

#define PARTITIONLIST() \
	P (EMPTY,		0x0,	"empty")		\
	DOS_FAT12_PARTITIONLIST()				\
	DOS_FAT16_PARTITIONLIST()				\
	DOS_FAT32_PARTITIONLIST()				\
	NTFS_PARTITIONLIST()					\
	EXTENDED_PARTITIONLIST()				\
								\
	P (LDM,			0x42,	"LDM")			\
	P (GNU_HURD,		0x63,	"GNU HURD")		\
	P (MINIX,		0x81,	"MINIX")		\
	P (Linux_Swap,		0x82,	"Linux swap")		\
	P (Linux,		0x83,	"Linux")		\
	P (BSD_386,		0xA5,	"FreeBSD, BSD/386")	\
	P (OpenBSD,		0xA6,	"OpenBSD")		\
	P (NetBSD,		0xA9,	"NetBSD")		\
	P (BSDI_fs,		0xB7,	"BSDI fs")		\
	P (BSDI_swap,		0xB8,	"BSDI swap")		\
	P (Solaris_boot,	0xBE,	"Solaris boot")		\
	P (DOS_R_O,		0xE3,	"DOS R/O")		\
	P (GPT_scheme,		0xEE,	"GPT_scheme")	\
	P (DOS_secondary,	0xF2,	"DOS secondary")	\
	P (LinuxAutoRaid,	0xFD,	"Linux RAID autodetect")

enum system_indicator_enum {
#define P(shortname, value, longname) shortname = value,
	PARTITIONLIST()
#undef P /*(shortname, value, longname)*/
    };

typedef struct {
    unsigned		WindowsNTmarker;
    unsigned short	Unknown;
    struct bootsect_partition_str {
	enum {non_bootable = 0, bootable = 0x80, invalid}
			indicator			:8;
	unsigned char	begin_head			:8;
	unsigned char	begin_sector			:6;
	unsigned char	begin_cylinder_msb		:2;
	unsigned char	begin_cylinder_lsb		:8;
//	unsigned short	begin_sector				   : 6;
//	unsigned short	begin_cylinder __attribute__((big_endian)) : 10;

	enum system_indicator_enum system_indicator	:8;
	unsigned char	end_head			:8;
	unsigned char	end_sector			:6;
	unsigned char	end_cylinder_msb		:2;
	unsigned char	end_cylinder_lsb		:8;
//	unsigned short	end_sector				 : 6;
//	unsigned short	end_cylinder __attribute__((big_endian)) : 10;

	unsigned	nb_sector_before;
	unsigned	nb_sector;
	} __attribute__ ((packed)) bootsect_partition[4];
    unsigned short	Signature0xAA55;
    } __attribute__ ((packed, aligned(1))) bootafter_t;

struct GPT_header_s {
    unsigned char signature[8]; /* "EFI PART" */
    unsigned revision; /* for v1.0, 0x10000 */
    unsigned header_size; /* usually 0x5C = 92 */
    unsigned crc32_header;
    unsigned reserved; /* must be 0 */
    unsigned long long current_lba;
    unsigned long long backup_lba;
    unsigned long long first_useable_lba;
    unsigned long long last_useable_lba;
    unsigned char disk_guid[16];
    unsigned long long partition_entry_start_lba;
    unsigned number_of_partition_entry;
    unsigned sizeof_partition_entry; /* usually 128 */
    unsigned crc32_partition_array;
    } __attribute__((packed));
struct GPT_entry_s {
    unsigned char partition_type_guid[16];
    unsigned char partition_guid[16];
    unsigned long long first_lba;
    unsigned long long last_lba; /* inclusive, usually odd */
    unsigned long long attribute;
    unsigned short name_utf_16le[36];
    } __attribute__((packed));

/*
 * End of disk, B.E.E.R., see ATA T13 D1367
 */
struct beer_directory_s {
    struct {
	unsigned char	available_as_B	: 1;
	unsigned char	reserved	: 1;
	unsigned char	diagnostic	: 1;
	unsigned char	read_only	: 1;
	unsigned char	this_boot	: 1;
	unsigned char	empty		: 1;
	unsigned char	hidden		: 1;
	unsigned char	bootable_as_A	: 1;
	} __attribute__ ((packed)) flags;
    unsigned char	reserved1;
	/* Thou shalt align data ... */
    unsigned long long	service_area_start;
    unsigned long long	service_area_size;
    unsigned		load_sector;
    unsigned		load_address;
    unsigned short	service_area_id;
    unsigned char	ID_string[32];
    unsigned short	reserved2;
    unsigned short	checksum;
    } __attribute__ ((packed));

struct beer_s {
    unsigned short	signature_0xBEEF;
    unsigned short	beer_size;
    struct beer_attr_s {
	unsigned short	reported_geometry_valid : 1;
	unsigned short	formated_geometry_valid : 1;
	unsigned short	service_dir_present	: 1;
	unsigned short	device_support_lba	: 1;
	unsigned short	config_time_stamp_valid	: 1;
	unsigned short	use_RABCA		: 1;
	unsigned short	generated_record	: 1;
	unsigned short	read_only		: 1;
	unsigned short	reserved		: 8;
	} __attribute__ ((packed)) attr;
    struct {
	/* Thou shalt align data ... */
	unsigned	cylinder, head, sector, bytepersector;
	unsigned long long	sectorperdevice;
	} __attribute__ ((packed)) reported, formatted;
    unsigned short	bcd_year;
    unsigned short	linear_day;
    unsigned		config_time_stamp;
    unsigned char	reserved_1;
    unsigned char	device_index;
    unsigned long long	HostProtectedArea_start;
    unsigned long long	ReservedAreaBootCodeAddress;
    unsigned short	BEERdir_nb_entry;
    unsigned short	BEERdir_length;
    unsigned char	reserved_2;
    unsigned char	revision_3;
    unsigned char	device_name[40];
    unsigned short	checksum;
    struct beer_directory_s	directory[0];
    } __attribute__ ((packed));

/*
 * The variables of Gujin saved between boots:
 */
#if defined (INITRDNAME) || defined (INITRDNAME2)
#define INITIAL_menu_with_initrd 1
#else
#define INITIAL_menu_with_initrd 0
#endif

/* Very special defines: */
#define DISPLAY_APPLY_ATTRIBUTE_ORDER(apply) \
	/* name                 default value */ \
    apply(verbose,		  1) \
    apply(lock_bpp,		  1) \
    apply(lock_text_graphic,	  0) \
    apply(menu_with_disk,	  1) \
    apply(menu_with_parttype,	  1) \
    apply(menu_with_partition,	  1) \
    apply(menu_with_initrd,	  INITIAL_menu_with_initrd)

#define MISC_APPLY_ATTRIBUTE_ORDER(apply) \
	/* name                 default value */ \
    apply(force_textmode,		0) \
    apply(probe_file_in_iso_image,	1) \
    apply(force_probe_root,		1) \
    apply(IDE_in_BIOS_order,		0) \
    apply(ignore_kernel_option,		1) \
    apply(disk_write_enable,		0) \
    apply(hide_unhide_partitions,	0)

#define USER_APPLY_ATTRIBUTE_ORDER(apply) \
	/* name                 default value */ \
    apply(VGA_interface,	  !!(USER_SUPPORT & VGA_SUPPORT) )	\
    apply(use_gujin_embedded_font,	1)				\
    apply(VESA_interface,	  !!(USER_SUPPORT & VESA_WINDOW) )	\
    apply(enable_VESA_hardwin,	  !!(USER_SUPPORT & VESA_HARDWINDOW) )	\
    apply(VESA2_interface,	  !!(USER_SUPPORT & VESA_LINEAR) )	\
    apply(enable_joystick,	  !!(USER_SUPPORT & JOYSTICK_SUPPORT) )

#define SEARCH_APPLY_ATTRIBUTE_ORDER(apply) \
	/* name                 default value */ \
    apply(search_disk_mbr,	  0) \
    apply(search_part_mbr,	  1) \
    apply(keep_all_part_mbr,	  0) \
    apply(search_el_torito,	  1) \
    apply(search_topdir_files,	  1) \
    apply(search_subdir_files,	  1)

#define PROBE_APPLY_ATTRIBUTE_ORDER(apply) \
	/* name                 default value */ \
    apply(probe_bios_floppy_disk, !!(DISK_SUPPORT & (BIOS_SUPPORT | EBIOS_SUPPORT)) ) \
    apply(probe_bios_hard_disk,   !!(DISK_SUPPORT & (BIOS_SUPPORT | EBIOS_SUPPORT)) ) \
    apply(probe_ide_disk,	  0)	\
    apply(probe_cdrom,		  1) \
    apply(probe_dos_disk,	  ((DEBUG & DEBUG_OUTPUT) != DOSFILE)) /* is probe_bdi_file in BIOS environment */

#define APPLY_ATTRIBUTE_ORDER(apply) \
    DISPLAY_APPLY_ATTRIBUTE_ORDER (apply)	\
    MISC_APPLY_ATTRIBUTE_ORDER (apply)		\
    USER_APPLY_ATTRIBUTE_ORDER (apply)		\
    SEARCH_APPLY_ATTRIBUTE_ORDER (apply)	\
    PROBE_APPLY_ATTRIBUTE_ORDER (apply)

/* list of attributes: */
enum gujin_param_attrib_enum {
#define Apply(field, deflt)	enum_ ## field,
    APPLY_ATTRIBUTE_ORDER (Apply)
#undef Apply
    enum_gujin_param_attrib_end
    };
#define TOTAL_ATTRIBUTE		enum_gujin_param_attrib_end

#define FORCED_WHEN_DOS(test, gujin_attr) \
  if (test) {					\
      gujin_attr.search_disk_mbr = 0;		\
      gujin_attr.search_part_mbr = 0;		\
      gujin_attr.keep_all_part_mbr = 0;		\
      gujin_attr.search_el_torito = 0;		\
      gujin_attr.probe_ide_disk = 0;		\
      gujin_attr.probe_cdrom = 0;		\
      gujin_attr.use_gujin_embedded_font = 0;	\
      gujin_attr.disk_write_enable = 0;		\
      }

/* list of recognised keyboard: */
enum kbd_enum {
    kbd_qwerty = 0x00,
    kbd_us,
    kbd_ca,
    kbd_br,
    kbd_uk,
    kbd_no,
    kbd_fi,
    kbd_dk,
    kbd_nl,
    kbd_sp,
    kbd_pt,
    kbd_it,
    kbd_azerty = 0x40,
    kbd_fr,
    kbd_be,
    kbd_qwertz = 0x80,
    kbd_de,
    kbd_ch,

#define INCLUDE_DVORAK_KEYBOARD		/* NOT TESTED */
#ifdef INCLUDE_DVORAK_KEYBOARD
    kbd_dvorak = 0xC0,
    kbd_dvorak_ansi = 0xC1,
#endif

    kbd_unknown = 0xFF
    };

#define kbdmap_is_QWERTY(kbdmap)	(((kbdmap) >> 6) == 0)
#define kbdmap_is_AZERTY(kbdmap)	(((kbdmap) >> 6) == 1)
#define kbdmap_is_QWERTZ(kbdmap)	(((kbdmap) >> 6) == 2)
#define kbdmap_is_DVORAK(kbdmap)	(((kbdmap) >> 6) == 3 && (kbdmap) != kbd_unknown)
#define kbdmap_subclass(kbdmap)		((kbdmap) & 0x3F)

#define kbdmap_is_precise(kbdmap)	(kbdmap_subclass(kbdmap) != 0 && (kbdmap) != kbd_unknown)

/* Used at least in vmlinux.c : */
extern inline const char *kbdname (enum kbd_enum kbdmap) {
#if 0	/* just space saving stuff... 448 bytes of code on gcc-3.3 */
    switch (kbdmap) {
	case kbd_qwerty:	return "qwerty";
	case kbd_us:		return "us";
	case kbd_ca:		return "ca";
	case kbd_br:		return "br";
	case kbd_uk:		return "uk";
	case kbd_no:		return "no";
	case kbd_fi:		return "fi";
	case kbd_dk:		return "dk";
	case kbd_nl:		return "nl";
	case kbd_sp:		return "sp";
	case kbd_pt:		return "pt";
	case kbd_it:		return "it";
	case kbd_azerty:	return "azerty";
	case kbd_fr:		return "fr";
	case kbd_be:		return "be";
	case kbd_qwertz:	return "qwertz";
	case kbd_de:		return "de";
	case kbd_ch:		return "ch";
#ifdef INCLUDE_DVORAK_KEYBOARD
	case kbd_dvorak:	return "dvorak";
	case kbd_dvorak_ansi:	return "dvorak_ansi";
#endif
	default:
	case kbd_unknown:	return 0;
	}
#else	/* just space saving stuff... */
  if (kbdmap_is_QWERTY(kbdmap)) {
#if 0	/* more space saving ... */
      static const unsigned char *const qwerty_map[] = {
	  [kbdmap_subclass(kbd_qwerty)] = "qwerty",
	  [kbdmap_subclass(kbd_us)] = "us",
	  [kbdmap_subclass(kbd_ca)] = "ca",
	  [kbdmap_subclass(kbd_br)] = "br",
	  [kbdmap_subclass(kbd_uk)] = "uk",
	  [kbdmap_subclass(kbd_no)] = "no",
	  [kbdmap_subclass(kbd_fi)] = "fi",
	  [kbdmap_subclass(kbd_dk)] = "dk",
	  [kbdmap_subclass(kbd_nl)] = "nl",
	  [kbdmap_subclass(kbd_sp)] = "sp",
	  [kbdmap_subclass(kbd_pt)] = "pt",
	  [kbdmap_subclass(kbd_it)] = "it",
	  };
      return qwerty_map[kbdmap_subclass(kbdmap)];
#else
      /* Here we will not have an array of 4 bytes pointers to store
	pointers to string of less than 4 bytes... */
      static const char qwerty_multistr[] =
	"qwerty"	"\0"
	"us"		"\0"
	"ca"		"\0"
	"br"		"\0"
	"uk"		"\0"
	"no"		"\0"
	"fi"		"\0"
	"dk"		"\0"
	"nl"		"\0"
	"sp"		"\0"
	"pt"		"\0"
	"it"		"\0";	/* finishes with two '\0' */
      const char *stringptr = qwerty_multistr;
      unsigned cpt = kbdmap_subclass(kbdmap);
      while (cpt--) { /* zero is first choice */
	  while (*stringptr++) /* nothing */;
	  if (*stringptr == '\0')
	      break; /* double zero is after last choice */
	  }
      return stringptr;
#endif
      }
    else if (kbdmap_is_AZERTY(kbdmap)) {
#if 0	/* more space saving ... */
      static const unsigned char *const azerty_map[] = {
	  [kbdmap_subclass(kbd_azerty)] = "azerty",
	  [kbdmap_subclass(kbd_fr)] = "fr",
	  [kbdmap_subclass(kbd_be)] = "be",
	  };
      return azerty_map[kbdmap_subclass(kbdmap)];
#else
      /* Here we will not have an array of 4 bytes pointers to store
	pointers to string of less than 4 bytes... */
      static const char azerty_multistr[] =
	  "azerty"	"\0"
	  "fr"		"\0"
	  "be"		"\0";	/* finishes with two '\0' */
      const char *stringptr = azerty_multistr;
      unsigned cpt = kbdmap_subclass(kbdmap);
      while (cpt--) { /* zero is first choice */
	  while (*stringptr++) /* nothing */;
	  if (*stringptr == '\0')
	      break; /* double zero is after last choice */
	  }
      return stringptr;
#endif
      }
    else if (kbdmap_is_QWERTZ(kbdmap)) {
#if 0	/* more space saving ... */
      static const unsigned char *const qwertz_map[] = {
	  [kbdmap_subclass(kbd_qwertz)] = "qwertz",
	  [kbdmap_subclass(kbd_de)] = "de",
	  [kbdmap_subclass(kbd_ch)] = "ch",
	  };
      return qwertz_map[kbdmap_subclass(kbdmap)];
#else
      /* Here we will not have an array of 4 bytes pointers to store
	pointers to string of less than 4 bytes... */
      static const char qwerty_multistr[] =
	  "qwerty"	"\0"
	  "de"		"\0"
	  "ch"		"\0";	/* finishes with two '\0' */
      const char *stringptr = qwerty_multistr;
      unsigned cpt = kbdmap_subclass(kbdmap);
      while (cpt--) { /* zero is first choice */
	  while (*stringptr++) /* nothing */;
	  if (*stringptr == '\0')
	      break; /* double zero is after last choice */
	  }
      return stringptr;
#endif
      }
#ifdef INCLUDE_DVORAK_KEYBOARD
    else if (kbdmap_is_DVORAK(kbdmap)) {
      static const char *const dvorak_map[] = {
	  [kbdmap_subclass(kbd_dvorak)] = "drovak",
	  [kbdmap_subclass(kbd_dvorak_ansi)] = "drovak_ansi",
	  };
      return dvorak_map[kbdmap_subclass(kbdmap)];
      }
#endif
    return 0; /* kbd_unknown */
#endif	/* just space saving stuff... */
    }

/*
 * The big variable saved between boots:
 * Located in first sector of boot2 to be easy to modify,
 * it has its own subchecksum to keep total checksum constant.
 */
typedef struct {
    unsigned		magic;
    unsigned short	version;
    struct gujin_param_compilation_attrib {
	unsigned short vga_support	: 1;
	unsigned short big_malloc	: 1;
	unsigned short bios_only	: 1;
	unsigned short initrd_loader	: 1;
	unsigned short vmlinuz_loader	: 1;
	unsigned short elf_loader	: 1;
	unsigned short multiboot_loader	: 1;
	unsigned short multilingual	: 1;
	unsigned short unicode	: 1;
	unsigned short reserved		: 7;
	} __attribute__ ((packed)) compilation_attrib;
    char		dot_msg[3];
    unsigned char	MltStrLanguage;
    char		go_msg[8];
    char		scanpath[16];	/* "boot" !NEVER empty string! */
    unsigned short	default_video_mode,
			default_text_video_mode,
			default_graphic_video_mode;
    unsigned char	min_nb_initrd;
    unsigned char	stop_emulation; /* of CDROM HD or FLOPPY */
    struct gujin_param_attrib {

#define Apply(field, deflt)	unsigned field : 1;
	APPLY_ATTRIBUTE_ORDER(Apply)
#undef Apply

//	unsigned available		: 31 - TOTAL_ATTRIBUTE; last free taken by force_bzimage_compatibility !
	unsigned write_disabled_written_once : 1;
	} __attribute__ ((packed)) attrib;

    struct autoload_str {
	unsigned char last_loaded;
	unsigned char total_loadable;
	unsigned char init_page;
	unsigned char total_page;
	} __attribute__ ((packed)) autoload;
    unsigned char	timeout_autoload;
#if 0
    enum kbd_enum kbdmap : 8; /* cannot take its address */
#else
    unsigned char	kbdmap;
#endif
    signed char		time_hour_offset, time_minute_offset;
    unsigned		reserved2[4];
    char		extra_cmdline[64];

// instboot.c has to treat a binary even if not compiled with KEEP_STRUCT_MAPPING
// Moreover, for mingujin.exe, that is used (overwritten) to store a longer
// DOS/FreeDOS command line.
//#if KEEP_STRUCT_MAPPING || (USER_SUPPORT & VGA_SUPPORT)
    unsigned char	cardname[32];	/* for autoreset, from VESA */
    unsigned char	VGA_valid_mode[16];
    struct vga_mode_str {
	unsigned isgraphic	: 1;
	unsigned reserved	: 1;
	unsigned number		: 7;
	unsigned bpp		: 4; /* 0 to 15, 8bpp max for VGA BIOS */
	unsigned height		: 11; /* 0 to 2048 */
	unsigned width_div8	: 8; /* 0 to 2048, div8 only for graphic */
	} __attribute__ ((packed)) vga_mode[30];
//#endif

    unsigned short	partial_checksum  __attribute__ ((aligned (2)));
    } __attribute__ ((packed)) gujin_param_t;

/*
 * The structure of directories (BIOS/IDE access to FAT12/FAT16/FAT32)
 */
typedef struct {
    char name[8]; /* offset 0: end_dir=0, firstcharE5=0x05, erased=0xE5 */
    char extension[3];
    struct file_attr_str {
	unsigned char	readonly	: 1;
	unsigned char	hidden		: 1;
	unsigned char	system		: 1;
	unsigned char	volumelabel	: 1;
	unsigned char	subdirectory	: 1;
	unsigned char	archive		: 1;
	unsigned char	unused		: 2;
	} __attribute__ ((packed)) attributes;
    char reserved[8];
    unsigned short	cluster_msb;
    struct {
	unsigned seconddiv2	: 5;
	unsigned minute		: 6;
	unsigned hour		: 5;
	unsigned day		: 5;	/* 1..31 */
	unsigned month		: 4;	/* 1..12 */
	unsigned year		: 7;	/* 1980 + x */
	} __attribute__ ((packed)) date;
    unsigned short	cluster;
    unsigned		size;
    } __attribute__ ((packed)) directory_t;

#if DISK_SUPPORT & ISOFS_PROBE
typedef struct { /* ISO9660: shall be at sector 0x10 of each session */
    unsigned char	volume_descriptor_type;		/* 1 */
    unsigned char	standard_identifier[5];		/* "CD001" */
    unsigned char	volume_descriptor_version;	/* 1 */
    unsigned char	unused1;			/* 0 */
    char			system_identifier[32];		/* [A-Z0-9_ !"%&'()*+,-./:;<=>?] */
    char			volume_identifier[32];		/* [A-Z0-9_] */
    unsigned char	unused2[8];			/* 0 */
    unsigned		volume_space_size_le;		/* little endian */
    unsigned		volume_space_size_be;		/* big endian */
    unsigned char	unused3[32];
    unsigned short	volume_set_size_le;
    unsigned short	volume_set_size_be;
    unsigned short	volume_sequence_number_le;
    unsigned short	volume_sequence_number_be;
    unsigned short	logical_block_size_le;
    unsigned short	logical_block_size_be;
    unsigned		path_table_size_le;
    unsigned		path_table_size_be;
    unsigned		path_table_location_le;		/* table itself is little endian */
    unsigned		optional_path_table_location_le;	/* table itself is little endian or 0 */
    unsigned		path_table_location_be;		/* table itself is big endian */
    unsigned		optional_path_table_location_be;	/* table itself is big endian or 0 */
    struct CD_directory_record_str {	/* size 34 bytes */
	unsigned char	length;	/* LEN_DR */
	unsigned char	extended_attribute_record_length;
	unsigned	first_logical_block_number_le;
	unsigned	first_logical_block_number_be;
	unsigned	data_length_le;
	unsigned	data_length_be;
	struct CD_date_time_str {
	    unsigned char	years_since_1900;
	    unsigned char	month_1_12;
	    unsigned char	day_1_31;
	    unsigned char	hour_0_23;
	    unsigned char	minute_0_59;
	    unsigned char	second_0_59;
	      signed char	nb_15minute_GMT_offset; /* -48(West) to +52(East) */
	    } __attribute__ ((packed)) date_time;
	struct CD_file_attribute {
	    unsigned char	hidden		: 1;
	    unsigned char	directory	: 1;
	    unsigned char	associated_file	: 1;
	    unsigned char	extended_attr	: 1;
	    unsigned char	extended_info	: 1;
	    unsigned char	reserved1	: 1;
	    unsigned char	reserved2	: 1;
	    unsigned char	has_hard_link	: 1;
	    } __attribute__ ((packed)) attributes;
	unsigned char	file_unit_size;		/* for interleaved files else 0 */
	unsigned char	interleave_gap_size;	/* for interleaved files else 0 */
	unsigned short	volume_sequence_number_le;
	unsigned short	volume_sequence_number_be;
	unsigned char	file_identifier_length;	/* LEN_FI */
	char		file_identifier[0];	/* size LEN_FI */
	unsigned char	padding;		/* if LEN_FI is even - if present has to be null */
	unsigned char	system_use[0];		/* if LEN_DR is odd: sizeof(struct CD_directory_record_str) has to be even */
	} __attribute__ ((packed)) root_directory_record;
    unsigned char volume_set_identifier[128];	/* offset 191..318 */
    unsigned char publisher_identifier[128];	/* offset 319..446 */
    unsigned char data_preparer_identifier[128];	/* offset 447..574 */
    unsigned char application_identifier[128];	/* offset 575..702 */
    unsigned char copyright_file_identifier[37];	/* offset 703..739 : name of file in root directory or only space chars */
    unsigned char abstract_file_identifier[37];	/* offset 740..776 : name of file in root directory or only space chars */
    unsigned char bibliographic_file_identifier[37];	/* offset 777..813 : name of file in root directory or only space chars */
    struct CD_date_time_str volume_creation;	/* offset 814..830 */
    unsigned char filler1[10];
    struct CD_date_time_str volume_modification;	/* offset 831..847 */
    unsigned char filler2[10];
    struct CD_date_time_str volume_expiration;	/* offset 848..864 */
    unsigned char filler3[10];
    struct CD_date_time_str volume_effective;	/* offset 865..881 : "can be used after this date" */
    unsigned char filler4[10];
    unsigned char file_structure_version;	/* 1 */
    unsigned char reserved;			/* 0 at offset 883 */
    unsigned char application_use[1395 - 884 +1];
    unsigned char reserved2[2048 - 1396 +1];	/* 0 */
    } __attribute__ ((packed)) ISO9660_Primary_volume_descriptor;

typedef struct { /* ISO9660: shall be at sector 0x11 of each bootable session */
    unsigned char  header_id_0x00;
    unsigned char  ident_name[5]; /* 'CD001' */
    unsigned char  version_0x01;
    unsigned char  bootsys_name[0x26 - 0x07 +1]; /* 'EL TORITO SPECIFICATION\0\0\0\0\0\0\0\0\0' */
    unsigned char  unused1[0x46 - 0x27 +1]; /* must be zero, size 32 */
    unsigned       boot_catalog_lba;
    unsigned char  unused2[0x7FF - 0x4B +1]; /* must be zero, size 1973 */
    } __attribute__ ((packed)) ISO9660_Boot_record_volume_descriptor;

#endif

typedef struct {
    unsigned char  header_id_0x01;
    enum { platform_80x86 = 0, platform_PowerPC = 1, platform_Mac = 2 } platform_id : 8;
    unsigned short reserved;
    unsigned char  manufacturer[0x1B - 0x04 +1];
    unsigned short checksum;
    unsigned char  signature1_0x55;
    unsigned char  signature2_0xAA;
    } __attribute__ ((packed)) ISO9660_Validation_entry;

typedef struct {
    unsigned char  boot_indicator; /* 0x88: bootable, 0x00: non bootable */
    enum { boot_media_noemul = 0, boot_media_120 = 1, boot_media_144 = 2,
		boot_media_288 = 3, boot_media_hd = 4 } boot_media_type : 8;
    unsigned short load_segment; /* 0x7c0 if null */
    unsigned char  system_type;
    unsigned char  unused;
    unsigned short sector_count;
    unsigned       sector_lba;
    unsigned char  reserved[0x1F - 0x0C +1];
    } __attribute__ ((packed)) ISO9660_Default_entry;

/*
 * Some references for instboot to patch executable:
 * Only to be used by the installer.
 */
typedef struct {
    unsigned checksum_start_adr;
    unsigned boot1param_adr;
    unsigned bootend_adr;
    unsigned uninstall_mbr_adr;
    unsigned bootchain_adr;
    unsigned bootchain_nbblock;
    unsigned gujin_param_adr;
    unsigned code_start_adr, patch2adr, patch3adr, patch4adr;
    unsigned sbss, edata, deltaseg, diskcodeseg, xdataseg;
    unsigned password_serial_adr, serialconf_port_plus_1_adr;
    unsigned read_disk_adr;
    unsigned partition_relative_read_disk_adr;
    unsigned read_disk_1st_adr;	/* CDROM EL TORITO */
    unsigned usbhdd_patch1, usbhdd_patch2;
    unsigned singlesectorload_patch1, singlesectorload_patch2;
    unsigned fscodeseg, loadcodeseg, usercodeseg;
    unsigned xstring1seg, xstring2seg;
    unsigned color_overwrite_adr;
    } instboot_info_t;

/*
 * not really usefull here but used in so many places, i.e.
 * user.h, vmlinuz.h, vesabios.h :
 */
struct vesa_color_layout_str {
    struct {
	unsigned char  MaskSize;
	unsigned char  FieldPosition;
	} __attribute__ ((packed)) Red, Green, Blue, Rsvd;
    } __attribute__ ((packed));

/*
 * To be used either to store background/foreground colors
 * or a complete image: with "compressed" discriminant.
 */
typedef union {
    struct {
	unsigned nbgroup	: 16;
	unsigned reserved1	: 1;
	unsigned compressed	: 1; /* = 0 for single color */
	unsigned reserved2	: 6;
	unsigned nbpixel	: 8;
	} compress;
    struct {	/* only 6 bits DAC: */
	unsigned reserved1	: 2;
	unsigned red		: 6;
	unsigned reserved2	: 2;
	unsigned green		: 6;
	unsigned reserved3	: 2;
	unsigned blue		: 6;
	unsigned reserved4	: 8;
	} color24bpp;
    struct {	/* 15 and 16 bpp treated the same: */
	unsigned reserved1	: 3;
	unsigned red		: 5;
	unsigned reserved2	: 3;
	unsigned green		: 5;
	unsigned reserved3	: 3;
	unsigned blue		: 5;
	unsigned reserved4	: 8;
	} color15bpp;
    struct {
	unsigned reserved	: 24;
	unsigned red		: 3;
	unsigned green		: 3;
	unsigned blue		: 2;
	} color8bpp;
    struct {
	unsigned lsb		: 2;
	unsigned reserved1	: 6;
	unsigned msb		: 2;
	unsigned reserved2	: 22;
	} color4bpp;
    struct {
	unsigned reserved1	: 16;
	unsigned white		: 1;
	unsigned reserved2	: 15;
	} color1bpp;
    } compressed_color;

/*
 * used in disk.[ch] and in the installer:
 */
typedef struct {
    struct {
	unsigned short	reserved		: 1; /* bit 0 */
	unsigned short	retired1		: 1; /* bit 1 */
	unsigned short	response_incomplete	: 1; /* bit 2 */
	unsigned short	retired2		: 3; /* bit 3..5 */
	unsigned short	obsolete		: 1; /* bit 6 */
	unsigned short	removable		: 1; /* bit 7 */
	unsigned short	retired3		: 6; /* bit 8 .. 13 */
	unsigned short	ATAPI_device		: 2; /* bit 14,15 has to be 0x2 for ATAPI */
	} __attribute__ ((packed)) general_config;
    unsigned short	number_of_cylinder;	/* obsolete */
    unsigned short	specific_configuration;
    unsigned short	number_of_head;		/* obsolete */
    unsigned short	retired1[2];
    unsigned short	number_of_sector_per_track;/* obsolete */
    unsigned short	reserved_compact_flash1[8 - 7 +1];
    unsigned short	retired2;
    unsigned char	ASCII_serial_number[20]; /* 0 : not specified */
    unsigned short	retired3[21 - 20 +1];
    unsigned short	obsolete1;	/* [word 22] */
    unsigned char	ASCII_firmware_revision[8];
    unsigned char	ASCII_model_vendor[40];
    unsigned char	RWmultiple_max_sector; /* if != 0 */
    unsigned char	constant_0x80; /* char */
    unsigned short	reserved1;
    struct capabilities_str {
	unsigned short	retired			: 8;
	unsigned short	dma_supported		: 1;
	unsigned short	lba_supported		: 1;
	unsigned short	IORDY_may_be_disabled	: 1;
	unsigned short	IORDY_supported		: 1;
	unsigned short	ATAPI_reserved		: 1;
	unsigned short	standby			: 1;
	unsigned short	ATAPI_reserved2		: 2;
	} __attribute__ ((packed)) capabilities;
    unsigned short	capabilities2;		/* [word 50] */
    unsigned short	obsolete2[52 - 51 +1];
    struct valid_description_str {
	unsigned short	translation_fields_valid : 1;
	unsigned short	words_64_70_valid	 : 1;
	unsigned short	word_88_valid		 : 1;
	unsigned short	description_reserved	 : 13;
	} __attribute__ ((packed)) valid_description;	/* [word 53] */
    unsigned short	number_of_current_cylinder; /* obsolete */
    unsigned short	number_of_current_head;	/* obsolete */
    unsigned short	number_of_current_sector_per_track; /* obsolete */
    unsigned		current_capacity_in_sector; /* obsolete */
    struct {
	unsigned short	current_multiple	: 7;
	unsigned short	current_multiple_valid	: 1;
	unsigned short	reserved		: 8;
	} __attribute__ ((packed)) RWmultiple;
    unsigned		lba_capacity;	/* BIG disk, > 20 Go [word 60] */
    unsigned short	obsolete3;
    struct multiword_dma_str {
	unsigned short	mode0supported	: 1;
	unsigned short	mode1supported	: 1;
	unsigned short	mode2supported	: 1;
	unsigned short	reserved1	: 5;
	unsigned short	mode0selected	: 1;
	unsigned short	mode1selected	: 1;
	unsigned short	mode2selected	: 1;
	unsigned short	reserved2	: 5;
	} __attribute__ ((packed)) multiword_dma;
    struct pio_modes_str {
	unsigned short	supported	: 8;
	unsigned short	reserved	: 8;
	} __attribute__ ((packed)) pio_modes;
    unsigned short	PIO_DMA_timings[68 - 65 +1];
    unsigned short	OVERLAP_reserved[70 - 69 +1];
    unsigned short	ATAPI_reserved[74 - 71 +1];
    unsigned short	queue_depth;
    unsigned short	reserved2[79 - 76 +1];
    unsigned short	major_version_number; /* [word 80] */
    unsigned short	minor_version_number;
    struct command1_str {
	unsigned short	smart		: 1;
	unsigned short	security	: 1;
	unsigned short	removable	: 1;
	unsigned short	power_management: 1;
	unsigned short	packet		: 1;
	unsigned short	write_cache	: 1;
	unsigned short	look_ahead	: 1;
	unsigned short	release_inter	: 1;
	unsigned short	service_inter	: 1;
	unsigned short	device_reset	: 1;
	unsigned short	host_protected_area: 1;
	unsigned short	obsolete1	: 1;
	unsigned short	write_buffer	: 1;
	unsigned short	read_buffer	: 1;
	unsigned short	nop		: 1;
	unsigned short	obsolete2	: 1;
	} __attribute__ ((packed)) command_supported1;
    struct command2_str {
	unsigned short	download_microcode: 1;
	unsigned short	dma_rw_queued	: 1;
	unsigned short	cfa_feature	: 1;
	unsigned short	advanced_pm	: 1;
	unsigned short	remove_notification: 1;
	unsigned short	powerup_standby	: 1;
	unsigned short	spinup_setfeature: 1;
	unsigned short	SAOReservedAreaBoot: 1;
	unsigned short	set_max_security: 1;
	unsigned short	AAcousticMngmt	: 1;
	unsigned short	lba48		: 1;
	unsigned short	DevConfOverlay	: 1;
	unsigned short	flush_cache	: 1;
	unsigned short	flush_cache_ext	: 1;
	unsigned short	cste_one	: 1;
	unsigned short	cste_zero	: 1;
	} __attribute__ ((packed)) command_supported2;
    struct command3_str {
	unsigned short	SMART_error_log	: 1;
	unsigned short	SMART_self_test	: 1;
	unsigned short	media_serial_nb	: 1;
	unsigned short	mediacard_passthrough	: 1;
	unsigned short	reserved1	: 1;
	unsigned short	general_logging	: 1;
	unsigned short	reserved2	: 8;
	unsigned short	cste_one	: 1;
	unsigned short	cste_zero	: 1;
	} __attribute__ ((packed)) command_supported3;
    struct command1_str command_enabled1;
    struct command2_str command_enabled2;
    struct command3_str command_enabled3;
    struct multiword_udma_str {
	unsigned short	mode0supported	: 1;
	unsigned short	mode1supported	: 1;
	unsigned short	mode2supported	: 1;
	unsigned short	mode3supported	: 1;
	unsigned short	mode4supported	: 1;
	unsigned short	mode5supported	: 1;
	unsigned short	reserved1	: 2;
	unsigned short	mode0selected	: 1;
	unsigned short	mode1selected	: 1;
	unsigned short	mode2selected	: 1;
	unsigned short	mode3selected	: 1;
	unsigned short	mode4selected	: 1;
	unsigned short	mode5selected	: 1;
	unsigned short	reserved2	: 2;
	} __attribute__ ((packed)) multiword_udma; /* [word 88] */
    unsigned short security_erase_unit_time;
    unsigned short enhanced_security_erase_time; /* [word 90] */
    unsigned short current_advancedPM;
    unsigned short master_password_revision;
    unsigned short hardware_reset_result;
    unsigned short acoustic_management;
    unsigned short reserved3[99 - 95 +1];
    unsigned long long	max_user_lba48; /* [word 100] */
    unsigned short streaming_transfert_time_PIO;
    unsigned short reserved4;
    struct sector_size_t {
	unsigned short	logical_per_physical		: 4;
	unsigned short	reserved			: 8;
	unsigned short	logical_sector_longer_than_256	: 1;
	unsigned short	multiple_logical_per_physical	: 1;
	unsigned short	shall_be_1			: 1;
	unsigned short	shall_be_0			: 1;
	} __attribute__ ((packed)) sector_size;	/* [word 106] */
    unsigned short inter_seek_delay_acoustic;
    unsigned short NAA_IEEEoui;
    unsigned short IEEEoui_uniqueID;
    unsigned short uniqueID_msb;
    unsigned short uniqueID_lsb;
    unsigned long long WWname128bits;
    unsigned short reserved_technical_report;
    unsigned	word_per_logical_sector;	/* [word 117/118] */
    unsigned short reserved5[126 - 119 +1];
    unsigned short removeable_status;
    struct security_status_str {
	unsigned short	supported	: 1;
	unsigned short	enabled		: 1;
	unsigned short	locked		: 1;
	unsigned short	frozen		: 1;
	unsigned short	count_expired	: 1;
	unsigned short	enhanced_erase	: 1;
	unsigned short	reserved1	: 2;
	unsigned short	high_level	: 1;
	unsigned short	reserved2	: 7;
	} __attribute__ ((packed)) security_status;	/* [word 128] */
    unsigned short vendor_specific[159 - 129 +1];
    unsigned short cfa_power_mode;	/* [word 160] */
    unsigned short reserved_compact_flash[175 - 161 +1];
    unsigned short current_media_serial_number[205 - 176 +1];
    unsigned short reserved6[254 - 206 +1];
    struct {
	unsigned short	signature	: 8;
	unsigned short	checksum	: 8;
	} integrity;
    } __attribute__ ((packed)) ata_identify_t;

/*
 * the ATAPI version:
 */
typedef struct {
    struct {
	enum {  len_12_bytes = 0,
		len_16_bytes = 1
		} cmd_packet_len		: 2; /* bit 0,1 */
	unsigned short	incomplete_response	: 1; /* bit 2 */
	unsigned short	reserved1		: 2; /* bit 3,4 */
	unsigned short	DRQ_INTRQ_timings	: 2; /* bits 5,6 */
	unsigned short	removable		: 1; /* bit 7 */
	enum {
		direct_acces = 0,
		sequencial_access = 1,
		printer = 2,
		processor = 3,
		write_once = 4,
		cdrom = 5,
		scanner = 6,
		optical_memory = 7,
		medium_changer = 8,
		communication = 9,
		acs_it8_1 = 10, acs_it8_2 = 11,
		array_controller = 12,
		unknown = 0x1F
		} command_packet_set		: 5; /* bit 8 .. 12 */
	unsigned short	reserved2		: 1; /* bit 13 */
	unsigned short	ATAPI_device		: 2; /* bit 14,15 has to be 0x2 */
	} __attribute__ ((packed)) general_config;
    unsigned short	reserved1;
    unsigned short	unique_configuration;
    unsigned short	reserved2[9 - 3 +1];
    unsigned char	serial_number[20];
    unsigned short	reserved3[22 - 20 +1];
    unsigned char	firmware_revision[8];
    unsigned char	model_number[40];
    unsigned short	reserved4[48 - 47 +1];
    struct capabilities_str	capabilities;
    unsigned short      capabilities2;		/* [word 50] */
    unsigned short      obsolete2[52 - 51 +1];
    struct valid_description_str	valid_description;
    unsigned short		reserved7[62 - 54 +1];
    struct multiword_dma_str	multiword_dma;	/* word 63 */
    struct pio_modes_str	pio_modes;
    unsigned short	PIO_DMA_timings[4];
    unsigned short	OVERLAP_reserved[70 - 69 +1];
    unsigned short	time_packet2release;
    unsigned short	time_service2busy;
    unsigned short	reserved9[74 - 73 +1];
    unsigned short	queue_depth;
    unsigned short	reserved10[79 - 76 +1];
    unsigned short	major_version_number; /* [word 80] */
    unsigned short	minor_version_number;
    struct command1_str	command_supported1;
    struct command2_str	command_supported2;
    struct command3_str	command_supported3;
    struct command1_str command_enabled1;
    struct command2_str command_enabled2;
    struct command3_str command_enabled3;
    struct multiword_udma_str multiword_udma; /* [word 88] */
    unsigned short reserved11[92 - 89 +1];
    unsigned short hardware_reset_result;
    unsigned short reserved12[124 - 94 +1];
    unsigned short ATAPI_byte_count_0_behavior;
    unsigned short obsolete;
    unsigned short removeable_status;	/* [word 127] */
    struct security_status_str security_status;		/* [word 128] */
    unsigned short vendor_specific[159 - 129 +1];
    unsigned short compactflash_reserved[175-160 +1];
    unsigned short reserved13[254-176 +1];
    struct {
	unsigned short	signature	: 8;
	unsigned short	checksum	: 8;
	} integrity;
    } __attribute__ ((packed)) atapi_identify_t;

#endif /* INSTBOOT_H */

