/* instboot.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#define _LARGEFILE64_SOURCE
#define _FILE_OFFSET_BITS 64
#include <errno.h>
#include <sys/types.h>		/* open() */
#include <sys/stat.h>
#include <sys/ioctl.h>		/* BLKRRPART in linux/fs.h to reread partition table */
#include <fcntl.h>		/* open() */
#include <unistd.h>		/* readlink(), ... */
#include <stdlib.h>
#include <dirent.h>		/* scandir(), readdir()  (see also man fts or man nftw) */
#include <string.h>		/* strchr(), memset() */
#include <stdio.h>
#include <time.h>		/* gmtime() / localtime() */

#include <linux/fd.h>		/* struct floppy_struct */
#include <linux/hdreg.h>	/* struct hd_geometry */
#include <linux/fs.h> /* MAJOR() and MINOR(), there could be a S_ISNUL(st_mode) for /dev/null */
#include <linux/msdos_fs.h>	/* FAT_IOCTL_SET_ATTRIBUTES */

#include "instboot.h"
//#include "messages.h"	/* ONLY for LANG_CMDLINE, for "--lang=" parameter, see also main.c:mingujin.exe */
#define MLTSTR(ENGLISH, FRENCH, RUSSIAN, SPANISH, ITALIAN, PORTUGESE, GERMAN, DUTCH) \
	ENGLISH, FRENCH, RUSSIAN, SPANISH, ITALIAN, PORTUGESE, GERMAN, DUTCH
#define LANG_CMDLINE	MLTSTR ("lang=en", "lang=fr", "lang=ru", "lang=sp", "lang=it", "lang=pt", "lang=de", "lang=nl")

#define DIVROUNDUP(X,Y)		(((X)+(Y)-1)/(Y))

#ifdef offsetof
#undef offsetof
#endif
#define offsetof(type, field)	((unsigned long)(&(((type *)0)->field)))

/*
 * the letter following /proc/ide/ide i.e. most time [0-9]
 * and a maximum of 10 IDE interfaces.
 */
#define IS_IDE_CHAR(x)	((x) >= '0' && (x) <= '9')

/*
 * The verbose global message selector:
 */
#define VERBOSE_INSTANCIATE(TREAT) \
	TREAT('s', 0, size)		\
	TREAT('c', 0, chain)		\
	TREAT('p', 0, proc)		\
	TREAT('f', 0, filesys)		\
	TREAT('l', 0, location)		\
	TREAT('m', 0, mbr)		\
	TREAT('r', 1, reprint)		\
	TREAT('e', 0, extra)

static struct verbose_s {
	unsigned inited : 1;
	unsigned warning : 1;
#define TREAT(letter, deflt, field)	unsigned field : 1;
	VERBOSE_INSTANCIATE(TREAT)
#undef TREAT
    } verbose = {
	.warning = 1,
#define TREAT(letter, deflt, field)	.field = deflt,
	VERBOSE_INSTANCIATE(TREAT)
#undef TREAT
    };

/* Link result: size of memory regions stuff: */
#define SIZE(args...) do { \
    if (verbose.size)	fprintf (stdout, args);	\
    } while (0)
/* chaining the 2nd level bootloader stuff: */
#define CHAIN(args...) do { \
    if (verbose.chain)	fprintf (stdout, args);	\
    } while (0)
/* /proc stuff: */
#define PROC(args...) do { \
    if (verbose.proc)	fprintf (stdout, args);	\
    } while (0)
/* FAT12/16 filesystem creation stuff: */
#define FILESYS(args...) do { \
    if (verbose.filesys) fprintf (stdout, args);	\
    } while (0)
/* how much of the output device written stuff: */
#define LOCATION(args...) do { \
    if (verbose.location) fprintf (stdout, args);	\
    } while (0)
/* content written in the real MBR (--mbr-device=)
 * and in the PBR (Partition Boot Record) stuff: */
#define MBR(args...) do { \
    if (verbose.mbr)	fprintf (stdout, args);	\
    } while (0)
/* extra stuff: */
#define EXTRA(args...) do { \
    if (verbose.extra)	fprintf (stdout, args);	\
    } while (0)
/* reprint command line main options */
#define REPRINTF(args...) do { \
    if (verbose.reprint)  fprintf (stdout, args); \
    } while (0)

/* Those prints without any conditions: */
#define ERROR(args...) do { \
    fprintf (stdout , "Error: " args);				\
    if (!verbose.inited)					\
	fprintf (stdout , "Are you root?, try verbose mode with '-w'\n");	\
    } while (0)

#define PARAMERR(args...) do { \
    fprintf (stdout , "Error: " args);		\
    } while (0)

#define WARNING(args...) do { \
    if (verbose.warning)			\
	fprintf (stdout , "Warning: " args);	\
    } while (0)

#define HINT(args...) do { \
    fprintf (stdout , "Hint: " args);		\
    } while (0)

#define PRINTF	printf

typedef unsigned farptr;

/* Parameter of function get_hard_disk_geometry() : */
struct get_hard_disk_geometry_s {
    unsigned long long total_sector;
    unsigned long long start_sector;
    unsigned cylinders;
    unsigned short heads;
    unsigned short sectpertrack;
    };

/* Parameter of search_access() : */
struct ide_capability_s {
    unsigned security			: 1;
    unsigned host_protected_area	: 1;
    unsigned SAOReservedAreaBoot	: 1;
    unsigned DevConfOverlay		: 1;
    };

/* manage efficiently a string of less than 4 chars: */
union fourbyte_u {
    char array[4];
    unsigned all;
    };

/*
 * The main variable.
 * We use "unsigned short" instead of "unsigned char" for some fields
 * (heads, sectpertrack, diskindex) even when we know they are < 256,
 * because they may be read by scanf() and I do not know about
 * the "%hhu" support in scanf conversion.
 */
struct install_s {
    enum gujin2_load_t {
	gujin2_load_error = 0,
	gujin2_load_dos_com,
	gujin2_load_dos_exe,
	gujin2_load_PIC,	/* position independant, Network EEPROM? */
	gujin2_load_CDROM_noemul,
	gujin2_load_bios,
	gujin2_load_ebios,
	gujin2_load_ide_chs,
	gujin2_load_ide_lba,
	gujin2_load_ide_lba48
	} gujin2_load;
    /*
     * two constants used anyway, i.e. even DOS:com/exe
     * loader use them for checksum calculus:
     */
    struct bootchain_s {
	unsigned short adr, nbblock;
	} bootchain;
    /*
     * If we know them, we write them here whatever the loader used:
     */
    struct ide_s {
	unsigned short data_adr, dcr_adr;
	} ide;
    /*
     * Geometry: basically, the partition stuff 'partdisk' is what is
     * written in the Partition Boot Record - and used for the filesystem,
     * and the 'wholedisk' stuff is what is used to build the load chain.
     * The 'wholedisk' stuff is equal by default to the 'partition' stuff if
     * 'partition.start_sector' is null, and then we have the usual floppy format.
     */
    struct geometry_s {
	/* 'wholedisk.start_sector' would be the number of sector before MBR
	 * in case of SAORAB use - but is only tested being 0 now.
	 */
	unsigned long long total_sector, start_sector;
	/* wholedisk.[C/H/S] only used if IDE:CHS (shall be physical) or BIOS (
	 * shall then be logical).
	 * cylinders is deduced from the set: total_sector, heads, sectpertrack.
	 * so I am not sure if it needs to be kept (information loss: how much
	 * of the last cylinder is used / seen when you play with IDE SET MAX LBA).
	 */
	unsigned cylinders;
	/* not really tested with BytePerSector != 512, but for CDROM
	 * Only wholedisk.BytePerSector used, partdisk.BytePerSector could
	 * even be set to zero if you want such a value in the bootsector.
	 */
	unsigned short BytePerSector;
	/* If heads == 0 and sectpertrack != 0, the device can be accessed in
	 * LBA modes and the maximum number of sector requested is
	 * sectpertrack included, zero is no limit.
	 */
	unsigned short heads, sectpertrack;
	/* For partdisk, that is the field bootbefore.part2.PhysicalDriveNb, if
	 * not overwritten by install->fs.PhysicalDriveNb parameter in  --fs=""
	 * For wholedisk, if we have a BIOS Gujin2 loader, that is the
	 * disk number (0x80 for first HD, 0 for the first floppy disk, 0x7F is auto BIOS %dl);
	 * if it is an IDE loader, it is 0 for master, 1 for slave.
	 * (IDE slave 2 and 3 are not tested)
	 * if wholedisk.diskindex = 0x7F (auto), and the sector size is != 512 (CDROM),
	 * the first dummy BIOS call (reset disk) uses partdisk.diskindex as parameter
	 */
	unsigned short diskindex;
	} partdisk, wholedisk;
    /*
     * Keep field sizes to the size of FAT_bootsect1_t type because
     * their maximum is used in GETFSCMD() reading macro (256 * sizeof()).
     */
    struct fs_s {
	unsigned long long NbFsSector;
	unsigned NbFat32Sector;	/* 32 bits for FAT32, 16 bits else */
	unsigned short NbReservedSector, NbRootSector;
	unsigned char NbFat, FatSize;
	unsigned char NbSectorPerCluster, MediaDescriptor, PhysicalDriveNb;
	unsigned nb_sector_FAT_wasted; // forced unsused FAT sector at end
	} fs;
    /*
     * just constants (for this install) which are used a bit everywhere:
     */
    struct misc_s {
	unsigned long long lba_start_chain, lba_end_chain, begin_file_sector_nr;
	unsigned long long output_partition_offset, restricted_disk_size;
	unsigned long long reported_disk_size, formatted_disk_size, beer_sector_offset;
	unsigned short forced_Heads, forced_SectPerTrack;
	unsigned firstDataSectorInPartition, nb_cluster_on_filesystem;
	unsigned nbsectorinfullfile; /* includes uninstall_mbr if present */
	unsigned nb_cluster_at_begin, nb_cluster_in_file,
		 nb_empty_sector_in_cluster, nb_cluster_at_end, nb_sector_at_end;
	unsigned infs_file, infs_position, iso_file;
	unsigned holy_filesystem, holy_unit_size; /* holy_filesystem: if filesystem support holes, size of initial hole */
	char default_ide_password[33]; /* with the ending zero */
	char maxsectloadtouse;
	unsigned char mbr_codesig_bug;
	} misc;
    /*
     * BEER constants, how to change them is undocumented: I may rely on
     * something one day... --beer=8,4,720,65536
     */
    struct beersect_s {
	unsigned BeerNbSector; /* Nb sector at real end of disk for the single BEER sector = 8 */
	unsigned BeerPosition; /* half the previous value: that is 4 */
	struct {
	    unsigned minSector, maxSector;
	    } BeerPartition;
	} beersect;
    };

bootafter_t *udev_create_partition = 0;
unsigned long long primary_GPT_sector = 0, secondary_GPT_sector = 0; /* != 0 if GPT partition table found */
const unsigned start_BEER_if_GPT = 64; /* a lot later than end of GPT usually at 34, max 250 partitions */

static inline unsigned
gujin2_loaded_by_dos (enum gujin2_load_t gujin2_load)
  {
  return    (gujin2_load == gujin2_load_dos_com)
	 || (gujin2_load == gujin2_load_dos_exe);
  }

static inline unsigned
gujin2_loaded_externally (enum gujin2_load_t gujin2_load)
  {
  return    (gujin2_load == gujin2_load_dos_com)
	 || (gujin2_load == gujin2_load_dos_exe)
	 || (gujin2_load == gujin2_load_PIC);
  }

static inline unsigned
gujin2_loaded_by_bios (enum gujin2_load_t gujin2_load)
  {
  return    (gujin2_load == gujin2_load_bios)
	 || (gujin2_load == gujin2_load_ebios);
  }

static inline unsigned
gujin2_loaded_by_ide_hard (enum gujin2_load_t gujin2_load)
  {
  return    (gujin2_load == gujin2_load_ide_chs)
	 || (gujin2_load == gujin2_load_ide_lba)
	 || (gujin2_load == gujin2_load_ide_lba48);
  }

static inline unsigned
gujin2_loaded_with_chs (enum gujin2_load_t gujin2_load)
  {
  return    (gujin2_load == gujin2_load_ide_chs)
	 || (gujin2_load == gujin2_load_bios);
  }

#if 0
static inline char *
str_gujin2_load (enum gujin2_load_t gujin2_load)
  {
  switch (gujin2_load) {
      case gujin2_load_error: return "gujin2_load_error";
      case gujin2_load_dos_com: return "gujin2_load_dos_com";
      case gujin2_load_dos_exe: return "gujin2_load_dos_exe";
      case gujin2_load_PIC: return "gujin2_load_PIC";
      case gujin2_load_CDROM_noemul: return "gujin2_load_CDROM_noemul";
      case gujin2_load_bios: return "gujin2_load_bios";
      case gujin2_load_ebios: return "gujin2_load_ebios";
      case gujin2_load_ide_chs: return "gujin2_load_ide_chs";
      case gujin2_load_ide_lba: return "gujin2_load_ide_lba";
      case gujin2_load_ide_lba48: return "gujin2_load_ide_lba48";
      default: return "gujin2_load value unknown";
      }
  }

static void
report_install_content (struct install_s *install)
  {
  printf ("\n%s: install.gujin2_load = %s, loaded_by_dos %u, loaded_by_ide_hard %u,\n"
		"\tloaded_with_chs %u\n",
	__FUNCTION__,
	str_gujin2_load (install->gujin2_load),
	gujin2_loaded_by_dos(install->gujin2_load),
	gujin2_loaded_by_ide_hard(install->gujin2_load),
	gujin2_loaded_with_chs(install->gujin2_load));
  printf ("  bootchain: { adr 0x%X, nbblock %u } ide: { data_adr 0x%X, dcr_adr 0x%X}\n",
	install->bootchain.adr, install->bootchain.nbblock,
	install->ide.data_adr, install->ide.dcr_adr);
  printf ("  partdisk:  { total_sector %llu, start_sector %llu, cylinders %u, "
	"BytePerSector %u, heads %u, sectpertrack %u, diskindex 0x%X }\n",
	install->partdisk.total_sector, install->partdisk.start_sector,
	install->partdisk.cylinders, install->partdisk.BytePerSector,
	install->partdisk.heads, install->partdisk.sectpertrack,
	install->partdisk.diskindex);
  printf ("  wholedisk: { total_sector %llu, start_sector %llu, cylinders %u, "
	"BytePerSector %u, heads %u, sectpertrack %u, diskindex 0x%X }\n",
	install->wholedisk.total_sector, install->wholedisk.start_sector,
	install->wholedisk.cylinders, install->wholedisk.BytePerSector,
	install->wholedisk.heads, install->wholedisk.sectpertrack,
	install->wholedisk.diskindex);
  printf ("  fs: { NbFsSector %llu, NbReservedSector %u, NbRootSector %u, NbFatSector %u, NbFat %u, FatSize %u,\n"
		"\tNbSectorPerCluster %u, MediaDescriptor 0x%X, PhysicalDriveNb 0x%X }\n",
	install->fs.NbFsSector, install->fs.NbReservedSector,
	install->fs.NbRootSector, install->fs.NbFat32Sector,
	install->fs.NbFat, install->fs.FatSize,
	install->fs.NbSectorPerCluster, install->fs.MediaDescriptor,
	install->fs.PhysicalDriveNb);
  printf ("  misc: { lba_start_chain %llu, lba_end_chain %llu, \n"
		"begin_file_sector_nr %llu, output_partition_offset %llu,\n"
		"restricted_disk_size %llu, reported_disk_size %llu, formatted_disk_size %llu,\n"
		"\tfirstDataSectorInPartition %u, nb_cluster_on_filesystem %u, nbsectorinfullfile %u,\n"
		"\tnb_cluster_at_begin %u, nb_cluster_in_file %u,\n"
		"\tnb_empty_sector_in_cluster %u, nb_cluster_at_end %u, nb_sector_at_end %u }\n",
	install->misc.lba_start_chain, install->misc.lba_end_chain,
	install->misc.begin_file_sector_nr, install->misc.output_partition_offset,
	install->misc.restricted_disk_size, install->misc.reported_disk_size, install->misc.formatted_disk_size,
	install->misc.firstDataSectorInPartition, install->misc.nb_cluster_on_filesystem, install->misc.nbsectorinfullfile,
	install->misc.nb_cluster_at_begin, install->misc.nb_cluster_in_file,
	install->misc.nb_empty_sector_in_cluster, install->misc.nb_cluster_at_end, install->misc.nb_sector_at_end);
  }
#else
static inline void
report_install_content (struct install_s *install)
  {
  }
#endif

/**
 ** Linux dependant stuff:
 **/
static inline int
search_ide_for_disk (unsigned char devdisk)
  {
  int save_errno = errno;
  char link[80], linkpath[] = "/proc/ide/hd?",
	      syslinkpath[] = "/sys/block/hd?/device";
  int ret;

  linkpath[12] = syslinkpath[13] = devdisk;
  PROC ("Proc: %s with '%s' or '%s': ", __FUNCTION__, syslinkpath, linkpath);

  if ((ret = readlink (syslinkpath, link, sizeof (link))) >= 0)
      PROC ("(using %s) ", syslinkpath);
    else if ((ret = readlink (linkpath, link, sizeof (link))) >= 0)
      PROC ("(using %s) ", linkpath);
    else {
      PROC ("failed.\n");
      errno = save_errno;
      return -1;
      }

  if (ret >= sizeof (link)) {
      PROC ("too long.\n");
      errno = save_errno;
      return -1;
      }
    else {
      char *ptr = link;

      link[ret] = '\0';
      /* either /proc/ide/hda -> ide0/hda or /sys/block/hda/device -> ../../devices/pci0000:00/0000:00:11.1/ide0/0.0 */
      while (*ptr != '\0' && (ptr - link <= ret))
	  if (ptr[0] == 'i' && ptr[1] == 'd' && ptr[2] == 'e' && ptr[4] == '/')
	      break;
	    else
	      ptr++;
      if (*ptr == '\0' || !IS_IDE_CHAR (ptr[3])) {
	  PROC ("\"ide[0-9]/\" search failed in '%s'.\n", link);
	  return -1;
	  }
      PROC ("link is '%s', so ide%u.\n", link, ptr[3] - '0');
      return ptr[3] - '0';
      }
  }

static inline int
search_geometry_for_disk (unsigned char devdisk,
		struct geometry_s *physical, struct geometry_s *logical)
  {
  FILE *filepattern;
  char geopath[] = "/proc/ide/hd?/geometry";

  geopath[12] = devdisk;
  PROC ("Proc: %s with '%s': ", __FUNCTION__, geopath);
  if ((filepattern = fopen (geopath, "r")) == NULL) {
      PROC ("cannot open '%s' for reading!\n", geopath);
      return 1;
      }
  PROC ("\n");
  if (physical)
      *physical = (struct geometry_s) {};
  if (logical)
      *logical = (struct geometry_s) {};
  while (!feof(filepattern)) {
      char linebuffer[80];

      if (fgets (linebuffer, sizeof (linebuffer), filepattern) !=0) {
	 if (   logical
	     && sscanf (linebuffer, "logical %u/%hu/%hu" "%*[^\n]",
			&logical->cylinders,
			&logical->heads,
			&logical->sectpertrack) == 3) {
	      PROC ("Proc:     found logical geometry %u/%u/%u\n",
			logical->cylinders,
			logical->heads,
			logical->sectpertrack);
	      logical = 0;
	      }
	  if (   physical
	      && sscanf (linebuffer, "physical %u/%hu/%hu" "%*[^\n]",
			&physical->cylinders,
			&physical->heads,
			&physical->sectpertrack) == 3) {
	      PROC ("Proc:     found physical geometry %u/%u/%u\n",
			physical->cylinders,
			physical->heads,
			physical->sectpertrack);
	      physical = 0;
	      }
	  }
      }
  if (fclose (filepattern) == EOF) {
      PROC ("cannot close '%s'.\n", geopath);
      return 1;
      }
  if (logical) {
      PROC ("did not find the logical description.\n");
      return 1;
      }
  if (physical) {
      PROC ("did not find the physical description.\n");
      return 1;
      }
  PROC ("Proc: %s done.\n", __FUNCTION__);
  return 0;
  }

static inline unsigned long long
search_capacity_for_disk (unsigned char ata, unsigned char devdisk)
  {
  char cappath[80] = "/proc/ide/?d?/capacity";
  char syssize[80] = "/sys/block/?d?/size";
  FILE *filepattern;
  unsigned long long capacity = 0;

  if (ata != '\0') {
      cappath[10] = syssize[11] = ata;
      cappath[12] = syssize[13] = devdisk;
      }
    else {
      strcpy (syssize, "/sys/block/mmcblk0/size");
      syssize[17] = devdisk;
      cappath[0] = '\0';
      }
  PROC ("Proc: %s with '%s' or '%s': ", __FUNCTION__, syssize, cappath);
  if ((filepattern = fopen (syssize, "r")) != NULL)
      PROC ("(using %s) ", syssize);
    else if ((filepattern = fopen (cappath, "r")) != NULL)
      PROC ("(using %s) ", cappath);
    else {
      PROC ("cannot open any of them for reading!\n");
      return 0;
      }
  for (;;) {
      int got;
      if ((got = fgetc (filepattern)) == EOF)
	  break;
      /* ignore leading whitespace: */
      if ((got == ' ' || got == '\t' || got == '\n') && capacity == 0)
	  continue;
      /* The value is in decimal: */
      if (got >= '0' && got <= '9') {
	  capacity *= 10;
	  capacity += got - '0';
	  continue;
	  }
      break;
      }
  PROC ("found %llu.\n", capacity);
  if (fclose (filepattern) == EOF) {
      PROC ("cannot close '%s'.\n", cappath);
      return 0;
      }
  return capacity;
  }

/*
 * This code suppose the DCR address of IDE is only one byte length,
 * and that the IDE base address is first, below its own DMA address.
 * Maybe we could have /proc/ide/ide[0-9]/ioaddress with few lines:
 * base 0x1F0-0x1F7
 * dcr  0x3F6-0x3F6 # meaning only one byte here, no "drive address" at 0x3F7
 * dma  0xd400-0xd407
 * Note: I have one box with line: "0320-0323 : " and "0350-0353 : " i.e. no name...
 * sometimes there is also spacing at beginning of the line
 */
static inline int
search_ide_io_addr (unsigned idenb,
		    unsigned short *data_adr, unsigned short *dcr_adr)
  {
  FILE *filepattern;
  char ioportspath[] = "/proc/ioports";

  PROC ("Proc: %s with '%s': ", __FUNCTION__, ioportspath);
  if ((filepattern = fopen (ioportspath, "r")) == NULL) {
      PROC ("cannot open %s for reading!\n", ioportspath);
      return 1;
      }
  *data_adr = 0;
  *dcr_adr = 0;

  while (!feof(filepattern) && (!*data_adr || !*dcr_adr)) {
      unsigned baseadr = 0, endadr = 0, nextline = 0;
      enum { get_begin_space, get_start, get_stop, get_colon_i,
	      get_d, get_e, get_idenb, get_end} state = get_begin_space;

      while (!nextline) {
	  int got;
	  if ((got = fgetc (filepattern)) == EOF)
	      break;
	  switch (state) {
	      case get_begin_space:
		  if (got > ' ')
		      state = get_start;
		    else
		      break;
	      case get_start:
		  if (got >= '0' && got <= '9')
		      baseadr = (baseadr << 4) + got - '0';
		    else if (got >= 'A' && got <= 'F')
		      baseadr = (baseadr << 4) + got - 'A' + 10;
		    else if (got >= 'a' && got <= 'f')
		      baseadr = (baseadr << 4) + got - 'a' + 10;
		    else if (got == '-')
		      state = get_stop;
		    else
		      state = get_end;
		  break;
	      case get_stop:
		  if (got >= '0' && got <= '9')
		      endadr = (endadr << 4) + got - '0';
		    else if (got >= 'A' && got <= 'F')
		      endadr = (endadr << 4) + got - 'A' + 10;
		    else if (got >= 'a' && got <= 'f')
		      endadr = (endadr << 4) + got - 'a' + 10;
		    else if (got == ' ')
		      state = get_colon_i;
		    else
		      state = get_end;
		  break;
	      case get_colon_i:
		  if (got == '\n')
		      nextline = 1;
		    else if (got != ' ' && got != '\t' && got != ':')
		      state = (got == 'i')? get_d : get_end;
		  break;
	      case get_d:
		  state = (got == 'd')? get_e : get_end;
		  break;
	      case get_e:
		  state = (got == 'e')? get_idenb : get_end;
		  break;
	      case get_idenb:
		  if (got == idenb + '0') {
		      PROC ("\nProc:     found I/O addr 0x%X-0x%X ", baseadr, endadr);
		      if (baseadr == endadr) {
			  PROC ("as dcr address, ");
			  *dcr_adr = baseadr;
			  }
			else {
			  PROC ("as base address, ");
			  *data_adr = baseadr;
			  }
		      }
		  state = get_end;
		  break;
	      case get_end:
		  if (got == '\n')
		      nextline = 1;
		  break;
	      }
	  }
    }
  if (fclose (filepattern) == EOF) {
      PROC ("\ncannot close %s\n", ioportspath);
      return 1;
      }
  if (*data_adr == 0 || *dcr_adr == 0) {
      PROC ("\nProc: did not find all the IDE I/O addresses!\n");
      return 1;
      }
  PROC ("\nProc: %s done.\n", __FUNCTION__);
  return 0;
  }

static inline unsigned
search_masterslave (unsigned char devdisk, unsigned short *diskindex)
  {
/*
$ cat /sys/block/hda/dev
3:0
$ cat /sys/block/hdb/dev
3:64
$ cat /sys/block/hdc/dev
22:0
$ cat /sys/block/hdd/dev
22:64
*/
  char sysdev[] = "/sys/block/hd?/dev";
  FILE *filepattern;

  sysdev[13] = devdisk;
  PROC ("Proc: %s with '%s': ", __FUNCTION__, sysdev);
  if ((filepattern = fopen (sysdev, "r")) != NULL) {
      int got_semicolon = 0, value = 0;

      for (;;) {
	  int got;
	  if ((got = fgetc (filepattern)) == EOF)
	      break;
	  if (got_semicolon == 0) {
	      if (got == ':')
		  got_semicolon = 1;
	      }
	    else if (got >= '0' && got <= '9')
	      value = value * 10 + got - '0';
	    else {
	      if (got != ' ' && got != '\n' && got != '\0')
		  got_semicolon = 0; /* anyways... */
	      break;
	      }
	  }
      fclose (filepattern);
      if (got_semicolon == 0) {
	  *diskindex = (devdisk - 'a') & 0x01; /* 0 for master */
	  PROC ("failed finding pattern, using %u, i.e. %s.\n", *diskindex, *diskindex? "slave" : "master");
	  }
	else {
	  *diskindex = (value != 0); /* 0 for master, 64 for slave */
	  PROC ("found %u, using %u i.e. %s.\n", value, *diskindex, *diskindex? "slave" : "master");
	  }
      }
    else {
      *diskindex = (devdisk - 'a') & 0x01; /* 0 for master */
      PROC ("failed openning file, using %u, i.e. %s.\n", *diskindex, *diskindex? "slave" : "master");
      }
  return 0;
  }

static inline enum gujin2_load_t
search_access (unsigned char devdisk, struct geometry_s *geo,
	struct ide_capability_s *cap_supported, struct ide_capability_s *cap_enabled)
  {
  enum gujin2_load_t returned;
  FILE *fileident;
  char identpath[]   = "/proc/ide/hdx/identify";
  union {
      ata_identify_t ata_identify;
      short word_array[256];
      } buffer;

  identpath[12] = devdisk;
  PROC ("Proc: %s with '%s': ", __FUNCTION__, identpath);

  if ((fileident = fopen (identpath, "r")) == NULL) {
      PROC ("cannot open '%s' for reading!\n", identpath);
      return gujin2_load_error;
      }
  {
  short *shortbuf = buffer.word_array;
  unsigned nbread = 0;
  while (   !feof(fileident)
	 && fscanf (fileident, "%hx %hx %hx %hx %hx %hx %hx %hx",
		&shortbuf[0], &shortbuf[1], &shortbuf[2], &shortbuf[3],
		&shortbuf[4], &shortbuf[5], &shortbuf[6], &shortbuf[7]) == 8) {
      shortbuf = &shortbuf[8];
      nbread += 8 * sizeof (short);
      }
  if (nbread != sizeof (buffer)) {
      PROC ("file %s is not long enough: %u instead of %u\n", identpath, nbread, (unsigned)sizeof (buffer));
      fclose (fileident);
      return gujin2_load_error;
      }

  shortbuf = (short *)(void *)&buffer.ata_identify.command_supported1;
  if (   (shortbuf[1] == 0  && shortbuf[2] == 0  && shortbuf[3] == 0)
      || (shortbuf[1] == -1 && shortbuf[2] == -1 && shortbuf[3] == -1)) {
      PROC ("Pre-ATA3 device, ");
      shortbuf[1] = shortbuf[2] = shortbuf[3] = 0;
      }

  if (   !buffer.ata_identify.capabilities.lba_supported
      && buffer.ata_identify.current_capacity_in_sector != buffer.ata_identify.lba_capacity
      && (buffer.ata_identify.current_capacity_in_sector & 0x0FE00000) != 0
      && (buffer.ata_identify.current_capacity_in_sector & 0x00000FE0) == 0) {
      unsigned swapped =  (buffer.ata_identify.current_capacity_in_sector << 16)
			| (buffer.ata_identify.current_capacity_in_sector >> 16);
      EXTRA ("\nExtra:    no lba on a > 1 Gb disk, swap size: 0x%X -> 0x%X, "
		"i.e. capacity %u sectors\nExtra:    ",
		buffer.ata_identify.current_capacity_in_sector,
		swapped, swapped);
      buffer.ata_identify.current_capacity_in_sector = swapped;
      }
  }
  if (fclose (fileident) == EOF) {
      PROC ("cannot close %s\n", identpath);
      return gujin2_load_error;
      }

  if (geo)
      *geo = (struct geometry_s) {};

  if (   buffer.ata_identify.command_supported2.lba48
      && buffer.ata_identify.command_enabled2.lba48) {
      PROC ("lba48 supported.\n");
      returned = gujin2_load_ide_lba48;
      }
    else if (buffer.ata_identify.capabilities.lba_supported) {
      PROC ("lba supported.\n");
      returned = gujin2_load_ide_lba;
      }
    else {
	/* physical seems wrong, but checked to be wrong on the disk,
	   or someone sent an IDE_INITIALISE_DRIVE cmd? :
	$ cat /proc/ide/hde/geometry
	physical     632960/1/63
	logical      39560/16/63
	...
	Proc: translation valid, C/H/S = 65535/1/63, C/H/S = 16383/16/63
	*/
      PROC ("CHS only supported, will need to overwrite logical by physical C/H/S\n");
      if (buffer.ata_identify.valid_description.translation_fields_valid) {
	  PROC ("Proc: translation fields valid, C/H/S = %u/%u/%u (base C/H/S %u/%u/%u).\n",
		buffer.ata_identify.number_of_current_cylinder,
		buffer.ata_identify.number_of_current_head,
		buffer.ata_identify.number_of_current_sector_per_track,
		buffer.ata_identify.number_of_cylinder,
		buffer.ata_identify.number_of_head,
		buffer.ata_identify.number_of_sector_per_track);
	  if (geo) {
	      geo->cylinders = buffer.ata_identify.number_of_current_cylinder;
	      geo->heads = buffer.ata_identify.number_of_current_head;
	      geo->sectpertrack = buffer.ata_identify.number_of_current_sector_per_track;
	      }
	  }
	else {
	  PROC ("Proc: translation fields invalid, C/H/S = %u/%u/%u (current C/H/S %u/%u/%u).\n",
		buffer.ata_identify.number_of_cylinder,
		buffer.ata_identify.number_of_head,
		buffer.ata_identify.number_of_sector_per_track,
		buffer.ata_identify.number_of_current_cylinder,
		buffer.ata_identify.number_of_current_head,
		buffer.ata_identify.number_of_current_sector_per_track);
	  if (geo) {
	      geo->cylinders = buffer.ata_identify.number_of_cylinder;
	      geo->heads = buffer.ata_identify.number_of_head;
	      geo->sectpertrack = buffer.ata_identify.number_of_sector_per_track;
	      }
	  }
      returned = gujin2_load_ide_chs;
      }

  if (cap_supported)
      *cap_supported = (struct ide_capability_s) {
		.security		= buffer.ata_identify.command_supported1.security,
		.host_protected_area	= buffer.ata_identify.command_supported1.host_protected_area,
		.SAOReservedAreaBoot	= buffer.ata_identify.command_supported2.SAOReservedAreaBoot,
		.DevConfOverlay		= buffer.ata_identify.command_supported2.DevConfOverlay,
		};

  if (cap_enabled)
      *cap_enabled = (struct ide_capability_s) {
		.security		= buffer.ata_identify.command_enabled1.security,
		.host_protected_area	= buffer.ata_identify.command_enabled1.host_protected_area,
		.SAOReservedAreaBoot	= buffer.ata_identify.command_enabled2.SAOReservedAreaBoot,
		.DevConfOverlay		= buffer.ata_identify.command_enabled2.DevConfOverlay,
		};

  if (buffer.ata_identify.command_supported1.security)
      PROC ("Proc: security supported and %sactive\n",
		buffer.ata_identify.command_enabled1.security? "" : "in");

  if (buffer.ata_identify.command_supported1.host_protected_area)
      PROC ("Proc: host_protected_area supported and %sactive\n",
		buffer.ata_identify.command_enabled1.host_protected_area? "" : "in");

  if (buffer.ata_identify.command_supported2.SAOReservedAreaBoot)
      PROC ("Proc: SAOReservedAreaBoot supported and %sactive\n",
		buffer.ata_identify.command_enabled2.SAOReservedAreaBoot? "" : "in");

  if (buffer.ata_identify.command_supported2.DevConfOverlay)
      PROC ("Proc: DevConfOverlay supported and %sactive\n",
		buffer.ata_identify.command_enabled2.DevConfOverlay? "" : "in");

  if (geo) {
      if (   returned == gujin2_load_ide_lba48
	  && buffer.ata_identify.max_user_lba48 >= buffer.ata_identify.lba_capacity)
	  geo->total_sector = buffer.ata_identify.max_user_lba48;
	else if (buffer.ata_identify.lba_capacity != 0
//	         && buffer.ata_identify.capabilities.lba_supported
		)
	  geo->total_sector = buffer.ata_identify.lba_capacity;
	else if (buffer.ata_identify.current_capacity_in_sector != 0)
	  geo->total_sector = buffer.ata_identify.current_capacity_in_sector;
	else /* FIXME if one of the pointers is zero: */
	  geo->total_sector = geo->cylinders * geo->heads * geo->sectpertrack;
      PROC ("Proc: IDE disk size: lba48: %llu, lba %u, basic %u, current C*H*S %u, C*H*S %u, so %llu sectors.\n",
		buffer.ata_identify.max_user_lba48,
		buffer.ata_identify.lba_capacity,
		buffer.ata_identify.current_capacity_in_sector,
		(unsigned)buffer.ata_identify.number_of_current_cylinder
			* buffer.ata_identify.number_of_current_head
			* buffer.ata_identify.number_of_current_sector_per_track,
		(unsigned)buffer.ata_identify.number_of_cylinder
			* buffer.ata_identify.number_of_head
			* buffer.ata_identify.number_of_sector_per_track,
		geo->total_sector);
      }

  return returned;
  }

/* This also gives the geometry for SCSI drives for BIOS access; that is
   the logical geometry so not to be used with '--disk=ide:chs' : */
/* This works also for floppies, because they also answer to HDIO_GETGEO_BIG */
static inline unsigned
get_hard_disk_geometry (int device_descriptor, struct get_hard_disk_geometry_s *geo)
  {
#ifdef HDIO_GETGEO
  struct hd_geometry   hd_param;
#endif
#ifdef HDIO_GETGEO_BIG
  struct hd_big_geometry hd_big_param;

  if (ioctl (device_descriptor, HDIO_GETGEO_BIG, &hd_big_param) >= 0) {
      EXTRA ("Extra:  HDIO_GETGEO_BIG ioctl: ");
      geo->total_sector = hd_big_param.cylinders * hd_big_param.sectors * hd_big_param.heads;
      geo->start_sector = hd_big_param.start;
      geo->cylinders = hd_big_param.cylinders;
      geo->heads = hd_big_param.heads;
      geo->sectpertrack = hd_big_param.sectors;
      EXTRA ("total_sector: %llu, start_sector: %llu, cylinders: %u, heads: %u, sectpertrack: %u.\n",
		geo->total_sector, geo->start_sector, geo->cylinders, geo->heads, geo->sectpertrack);
      }
    else
#endif
#ifdef HDIO_GETGEO
	 if (ioctl (device_descriptor, HDIO_GETGEO, &hd_param) >= 0) {
      EXTRA ("Extra:  HDIO_GETGEO ioctl: ");
      geo->total_sector = hd_param.cylinders * hd_param.sectors * hd_param.heads;
      geo->start_sector = hd_param.start;
      geo->cylinders = hd_param.cylinders;
      geo->heads = hd_param.heads;
      geo->sectpertrack = hd_param.sectors;
      EXTRA ("total_sector: %llu, start_sector: %llu, cylinders: %u, heads: %u, sectpertrack: %u.\n",
		geo->total_sector, geo->start_sector, geo->cylinders, geo->heads, geo->sectpertrack);
      }
    else
#endif
      {
      EXTRA ("Error: %s cannot get geometry.\n",  __FUNCTION__);
      return 1;
      }
  return 0;
  }

static int
get_from_statbuf_st_dev (unsigned statbuf_st_dev, char buffname[256], unsigned long long *start, unsigned long long *nb512sectors)
  {
#if 0	/* scandir() does not return devices so cannot be used on "/dev" */
  int filter_none (const struct dirent *unused) { return 1; }

  char pattern[256];	/* if replace by buffname, take care of sizeof(pattern) */
  struct stat stat_buf;
  struct dirent **namelist;
  int n;

  strcpy (pattern, "/dev/");
  if (stat (pattern, &stat_buf) != 0)
      return -1;

  EXTRA ("Extra: scandisk %s for stat_buf.st_rdev 0x%X\n", pattern, statbuf_st_dev);
//  n = scandir (pattern, &namelist, 0, alphasort);
  n = scandir (pattern, &namelist, filter_none, alphasort);
  if (n < 0) {
      perror("scandir");
      return -2;
      }

  while (n--) {
      if (statbuf_st_dev && stat (namelist[n]->d_name, &stat_buf) == 0) {
	  EXTRA ("file %s dev 0x%X rdev 0x%X\n", namelist[n]->d_name, (unsigned)stat_buf.st_dev, (unsigned)stat_buf.st_rdev);
	  if (stat_buf.st_rdev == statbuf_st_dev) {
	      if (buffname) {
		  strcpy (buffname, pattern);
		  strcat (buffname, namelist[n]->d_name);
		  }
	      statbuf_st_dev = 0; /* found */
	      }
	  }
      free(namelist[n]);
      }
  free(namelist);
  if (statbuf_st_dev == 0)
      return 0;
  return 1;
#else
  struct stat stat_buf;
  struct dirent **namelist_major, **namelist_minor;
  int n_major, n_minor, found = 0;
//  char filter_name[80] = ""; // That would need an executable stack.

  int filter (const struct dirent *d) {
//	const char *src1 = filter_name, *src2 = d->d_name;
//	if (*src2 == '.')
//	    return 0;
//	while (*src1)
//	    if (*src1++ != *src2++)
//		return 0;
	if (d->d_name[0] == '.')
	    return 0;
	return 1;
	}

  if (stat ("/sys/block/", &stat_buf) != 0)
      return -1;

  EXTRA ("Extra: scandisk() %s*/dev for 0x%X\n", "/sys/block/", (unsigned)statbuf_st_dev);
  n_major = scandir ("/sys/block/", &namelist_major, filter, alphasort);
  if (n_major < 0) {
      perror("scandir");
      return -2;
      }

  while (n_major--) {
      if (!found) {
	  unsigned major, minor;
	  FILE *fin;
	  char filename[80];

	  strcpy (filename, "/sys/block/");
	  strcat (filename, namelist_major[n_major]->d_name);
	  strcat (filename, "/dev");

	  EXTRA ("Extra: checking '%s' ", filename);
	  if ((fin = fopen (filename, "r")) == NULL || fscanf (fin, "%d:%d", &major, &minor) != 2) {
	      if (fin != NULL)
		  fclose (fin);
	      EXTRA ("cannot get major/minor\n");
	      free(namelist_major[n_major]);
	      continue;
	      }
	  fclose (fin);

	  EXTRA ("major 0x%X minor 0x%X\n", major, minor);
	  if (major(statbuf_st_dev) == major) { /* note that sda and sdb have the same major */
	      if (minor(statbuf_st_dev) == minor) {
		  found = 1;
		  if (buffname) {
		      strcpy (buffname, "/dev/");
		      strcat (buffname, namelist_major[n_major]->d_name);
		      }
		  if (nb512sectors) {
		      *nb512sectors = 0;
		      strcpy (filename, "/sys/block/");
		      strcat (filename, namelist_major[n_major]->d_name);
		      strcat (filename, "/size");
		      if ((fin = fopen (filename, "r")) != NULL) {
			  if (fscanf (fin, "%llu", nb512sectors) != 1)
			      *nb512sectors = 0;
			  fclose (fin);
			  }
			else {
			  //EXTRA ("major 0x%X minor 0x%X found, but failed openning '%s'\n", major, minor, filename);
			  //found = 0;
			  }
		      }
		  if (start) {
		      *start = 0; /* Normally, this file do not exist when no partition, but maybe one day (SOARB)? */
		      strcpy (filename, "/sys/block/");
		      strcat (filename, namelist_major[n_major]->d_name);
		      strcat (filename, "/start");
		      if ((fin = fopen (filename, "r")) != NULL) {
			  if (fscanf (fin, "%llu", start) != 1)
			      *start = 0;
			  fclose (fin);
			  }
			else {
			  //EXTRA ("major 0x%X minor 0x%X found, but failed openning '%s'\n", major, minor, filename);
			  //found = 0;
			  }
		      }
		  }
		else {
		  char pattern[32] = "/sys/block/";
		  strcat (pattern, namelist_major[n_major]->d_name);
		  strcat (pattern, "/");
		  EXTRA ("Extra: scandisk() %s*/dev for 0x%X\n", pattern, (unsigned)statbuf_st_dev);
		  n_minor = scandir (pattern, &namelist_minor, filter, alphasort);
		  if (n_minor >= 0) {
		      while (n_minor--) {
			  if (!found) {
			      strcpy (filename, pattern);
			      strcat (filename, namelist_minor[n_minor]->d_name);
			      strcat (filename, "/dev");

			      // EXTRA ("Extra: checking '%s'\n", filename);
			      if ((fin = fopen (filename, "r")) != NULL && fscanf (fin, "%d:%d", &major, &minor) == 2) {
				  if (major(statbuf_st_dev) != major)
				      EXTRA ("Extra: /sys/block data inconsistent!\n");
				    else if (minor(statbuf_st_dev) == minor) {
				      found = 1;
				      if (buffname) {
					  strcpy (buffname, "/dev/");
					  strcat (buffname, namelist_minor[n_minor]->d_name);
					  }
				      if (nb512sectors) {
					  *nb512sectors = 0;
					  strcpy (filename, pattern);
					  strcat (filename, namelist_minor[n_minor]->d_name);
					  strcat (filename, "/size");
					  fclose (fin);
					  if ((fin = fopen (filename, "r")) != NULL) {
					      if (fscanf (fin, "%llu", nb512sectors) != 1)
						  *nb512sectors = 0;
					      }
					    else {
					      //EXTRA ("major 0x%X minor 0x%X found, but failed openning '%s'\n", major, minor, filename);
					      //found = 0;
					      }
					  }
				      if (start) {
					  *start = 0;
					  strcpy (filename, pattern);
					  strcat (filename, namelist_minor[n_minor]->d_name);
					  strcat (filename, "/start");
					  fclose (fin);
					  if ((fin = fopen (filename, "r")) != NULL) {
					      if (fscanf (fin, "%llu", start) != 1)
						  *start = 0;
					      }
					    else {
					      //EXTRA ("major 0x%X minor 0x%X found, but failed openning '%s'\n", major, minor, filename);
					      //found = 0;
					      }
					  }
				      }
				  }
			      if (fin != NULL)
				  fclose (fin);
			      }
			  free(namelist_minor[n_minor]);
			  }
		      free(namelist_minor);
		      }
		  }
	      }
	  }
      free(namelist_major[n_major]);
      }
  free(namelist_major);
  if (!found)
      return -3;
  return 0;
#endif
  }

#define GET_PHYSICAL_BLOCKSIZE(fd) ({ \
	unsigned device_block_size;				\
	if (ioctl (fd, BLKSSZGET, &device_block_size) != 0)	\
	    device_block_size = 0;				\
	/* return */ device_block_size;				\
      })

#define GET_PHYSICAL_SIZE(fd) ({ \
	unsigned long device_size;			\
	if (ioctl (fd, BLKGETSIZE, &device_size) != 0)	\
	    device_size = 0;				\
	/* return */ device_size;			\
      })

static inline unsigned
get_device_start_sector (int device_descriptor, unsigned long long *start_sector, unsigned long long *nbsectors)
  {
  struct get_hard_disk_geometry_s geo;
  struct stat stat_buf;
  unsigned BytePerSector = GET_PHYSICAL_BLOCKSIZE(device_descriptor);	/* usually 512 */
  unsigned adjust = BytePerSector/512;

  if (fstat (device_descriptor, &stat_buf) == 0 && get_from_statbuf_st_dev (stat_buf.st_rdev, 0, start_sector, nbsectors) == 0) {
      if (start_sector && nbsectors) {
	  *start_sector /= adjust;
	  *nbsectors /= adjust;
	  EXTRA ("Extra: %s by /sys/block/[hs]d?/[hs]d[0-9]/start: %llu nbsectors: %llu\n", __FUNCTION__, *start_sector, *nbsectors);
	  }
      else if (start_sector) {
	  *start_sector /= adjust;
	  EXTRA ("Extra: %s by /sys/block/[hs]d?/[hs]d[0-9]/start: %llu\n", __FUNCTION__, *start_sector);
	  }
      else
	  EXTRA ("Extra: %s by /sys/block/[hs]d?/[hs]d[0-9]/ no start/nb512sectors\n", __FUNCTION__);
      }
    else if (get_hard_disk_geometry (device_descriptor, &geo) == 0) {
      *start_sector = geo.start_sector / adjust;
      EXTRA ("Extra: %s by IOCTL: %llu in 512 bytes units, i.e. %llu real sectors\n", __FUNCTION__, geo.start_sector, *start_sector);
      }
    else {
      EXTRA ("Extra: %s FAILED!\n", __FUNCTION__);
      return 1;
      }
  return 0;
  }

/* create a contiguous file with a hole at its start of size initial_hole_in_blocks.
   We want a file countigous on disk, so we create a dummy one now and will fill it later on */
static int infs_filemap (const char *filename, unsigned filesize, unsigned *unit_size,
		unsigned long long *start_chain, unsigned long long *end_chain,
		unsigned *statbuf_st_dev, unsigned initial_hole_in_blocks)
  {
  char filebuffer[filesize];
  int fout_desc, ret = 0;
  unsigned cpt, bmap_block_size;
#define AUTO_RETRY 16	/* less than ('Z' - 'A'), tmp file created "tmp[A-Z]" */
  char *to_unlink[AUTO_RETRY] = { 0 }; /* all to zero */

  EXTRA ("Extra: try to create contiguous file with initial hole of %u blocks named '%s' size %u bytes\n", initial_hole_in_blocks, filename, filesize);
  for (cpt = 0; cpt < sizeof(filebuffer); cpt++)
      filebuffer[cpt] = 0x5F;		/* we do not want a file with holes but the first one */
  if (statbuf_st_dev)
      *statbuf_st_dev = 0;

  for (cpt = 0; cpt < AUTO_RETRY; cpt++) {
      unsigned long block_index, current_block = 0, start_hole = 0;

      *start_chain = *end_chain = 0;
      if ((fout_desc = open (filename, O_CREAT|O_WRONLY|O_SYNC, S_IRUSR | S_IWUSR)) == -1) {
	  perror (filename);
	  ret = 1;
	  }
	else if (ioctl (fout_desc, FIGETBSZ, &bmap_block_size) || bmap_block_size == 0) {
	  perror ("FIGETBSZ");
	  ret = 2;
	  }
	else if (initial_hole_in_blocks && lseek(fout_desc, initial_hole_in_blocks * bmap_block_size, SEEK_SET) != initial_hole_in_blocks * bmap_block_size) {
	  perror ("lseek");
	  ret = 3;
	  }
	else if (write (fout_desc, filebuffer, sizeof(filebuffer)) != sizeof(filebuffer)) {
	  perror ("write");
	  ret = 4;
	  }
#if 1
	else if (fsync (fout_desc) != 0) {
	  perror ("fsync");
	  ret = 5;
	  }
#else
	else if ((ret = dup (fout_desc)) == -1 || close (ret) == -1) {
	  perror ("dup/close (sync to disk)");
	  ret = 5;
	  }
#endif
      if (ret) {
error_cleanfiles:
	  if (ret != 1) {
	      EXTRA ("Extra: ret = %d, closing and removing target file '%s'\n", ret, filename);
	      if (ret < 8 && close (fout_desc) != 0)
		  perror (filename);
	      if (unlink (filename) != 0)
		  perror (filename);
	      }
	  while (cpt-- != 0) {
	      EXTRA ("Extra: removing temporary file '%s'\n", to_unlink[cpt]);
	      if (unlink (to_unlink[cpt]) != 0) {
		  perror (to_unlink[cpt]);
		  free (to_unlink[cpt]);
		  }
	      }
	  return ret;
	  }
      if (unit_size)
	  *unit_size = bmap_block_size;
      if (statbuf_st_dev && *statbuf_st_dev == 0) {
	  struct stat statbuf;

	  if (fstat (fout_desc, &statbuf)) {
	      perror ("fstat");
	      ret = 6;
	      goto error_cleanfiles;
	      }
	  *statbuf_st_dev = statbuf.st_dev;
	  }

      const unsigned nb_block = initial_hole_in_blocks + (sizeof(filebuffer) + bmap_block_size - 1) / bmap_block_size;
      for (block_index = 0; block_index < nb_block; block_index++) {
	  unsigned long block = block_index;
	  if (ioctl (fout_desc, FIBMAP, &block)) {
	      perror ("FIBMAP");
	      ret = 7;
	      goto error_cleanfiles;
	      }
	  if (block == 0)
	      start_hole++;
	    else if (   (start_hole != 0 && start_hole != initial_hole_in_blocks)
		     || (current_block != 0 && block != current_block + 1)) {
	      /* wrong beginning hole, another hole or non contiguous */
	      EXTRA ("Extra: closing '%s' ", filename);
	      if (close (fout_desc) != 0) {
		  perror (filename);
		  ret = 8;
		  goto error_cleanfiles;
		  }
	      to_unlink[cpt] = malloc (strlen (filename) + strlen ("tmpA") + 1);
	      strcpy (to_unlink[cpt], filename);
	      unsigned xcpt;
	      for (xcpt = strlen (to_unlink[cpt]); xcpt != 0; xcpt--) {
		  if (to_unlink[cpt][xcpt] == '/' || to_unlink[cpt][xcpt] == '\\') {
		      xcpt++;
		      break;
		      }
		  }
	      strcpy (&to_unlink[cpt][xcpt], "tmpA");
	      to_unlink[cpt][xcpt+strlen("tmpA")-1] += cpt;
	      EXTRA ("and renaming to '%s', then retrying\n", to_unlink[cpt]);
	      if (rename (filename, to_unlink[cpt]) != 0) {
		  strcpy (to_unlink[cpt], filename);
		  perror (filename);
		  ret = 9;
		  goto error_cleanfiles;
		  }
	      fout_desc = 0;
	      break;
	      }
	  current_block = block;
	  if (*start_chain == 0)
	      *end_chain = *start_chain = block;
	    else
	      *end_chain += 1;
	  }
      if (block_index == nb_block)
	  break; /* Success */
      }
  for (cpt = 0; cpt < AUTO_RETRY; cpt++)
      if (to_unlink[cpt]) {
	  EXTRA ("Extra: removing temporary file '%s'\n", to_unlink[cpt]);
	  if (unlink (to_unlink[cpt]) != 0) {
	      perror (to_unlink[cpt]);
	      free (to_unlink[cpt]);
	      }
	    else {
	      free (to_unlink[cpt]);
	      to_unlink[cpt] = 0;
	      }
	  }
  for (cpt = 0; cpt < AUTO_RETRY; cpt++)
      if (to_unlink[cpt])
	  return 9; /* we even failed to unlink temporary files! */
  /* if fout_desc is still != 0, we have succedded */

  if (fout_desc == 0) {
      EXTRA ("Extra: FAILED creating contiguous file %s\n", filename);
      return 10;
      }
  if (close (fout_desc) != 0)
      perror (filename);
  EXTRA ("Extra: Success creating contiguous file %s at %llu to %llu block size %u (length %u bytes)\n",
	filename, *start_chain, *end_chain, bmap_block_size, (unsigned)(*end_chain + 1 - *start_chain) * bmap_block_size);
  return 0;
  }

/**
 ** Chaining:
 **/

/*
 * For 1.68M floppy disk, do not try to defragment the floppy: It is
 * specially fragmented because most BIOSes cannot access over the 18th
 * sector, even if the floppy is described to have more. Some other BIOSes
 * will read them but cannot read more than 18 sectors at once.
 * Most such tools to make the file contiguous on the magnetic support will
 * not try to touch files with attribute "readonly", "hidden" and "system",
 * but do not play (too much) with fire...
 * The 1.68 Mbytes floppy patch touches chs_chain(), the FAT and the file
 * written (gap added), the two last part are _not_ implemented.
 *
 * Note that if the BIOS driver cannot access over the 18th sector, Gujin
 * will not be able to access those files - it can only be accessed after
 * the Linux kernel has been loaded. That means neither vmlinux nor initrd
 * shall map a sector over sector 18: it is not so interresting.
 * Right now I "#ifndef EXTENDED_FLOPPY_SIZE" the chs_chain() patch and
 * "#ifdef EXTENDED_FLOPPY_SIZE" a patch which will work only if the BIOS
 * cannot read more than 18 sectors at once, but is able to read all the
 * sectors.
 */
static unsigned
set_bootchain (bootloader2_t *bootchain, unsigned long long lba, unsigned nb,
		farptr loadfarptr, enum gujin2_load_t gujin2_load,
		const struct geometry_s *geo, unsigned short diskindex,
		const struct ide_s *ide, unsigned long long parameter_lba)
  {
  if (nb == 0 &&  !(loadfarptr == 0x07C00000 + 512 && geo->BytePerSector != 512)) {
      CHAIN ("Chain: %s: Cannot chain a read of zero sectors!\n",
		__FUNCTION__);
      return 1;
      }
  if (geo->BytePerSector == 0) {
      CHAIN ("Chain: %s: Cannot chain a read with BytePerSector == 0!\n",
		__FUNCTION__);
      return 1;
      }
  if (lba >= geo->total_sector) {
      CHAIN ("Chain: %s: Cannot chain a read at lba = %llu when total_sector = %llu!\n",
		__FUNCTION__, lba, geo->total_sector);
      return 1;
      }
  if (geo->BytePerSector & 1) {
      CHAIN ("Chain: %s: Cannot chain a read with BytePerSector = %u, will not work!\n",
		__FUNCTION__, geo->BytePerSector);
      return 1;
      }
  /* header.nbword is used for checksum calculus: */
  if (nb != 0)
      bootchain->header.nbword = (nb * geo->BytePerSector) / 2;
    else {
      bootchain->header.nbword = (geo->BytePerSector - 512) / 2;
      CHAIN ("Chain: first checksum calculus on %u bytes instead of %u\n",
		2 * bootchain->header.nbword, geo->BytePerSector);
      }

  if (gujin2_loaded_by_ide_hard (gujin2_load)) {
      bootchain->bootloader2_cmd.hardide.ide_dcr_address = ide->dcr_adr;
      bootchain->bootloader2_cmd.hardide.reg_ax = 0xF00A;
      bootchain->bootloader2_cmd.hardide.base_ide_plus7 = ide->data_adr + 7;

      if (nb == 0) {
	  ERROR("Cannot chain 0 sector to read with IDE interface, sorry.\n");
	  return 1;
	  }
      bootchain->bootloader2_cmd.hardide.lba48_nb_sect = 0;
      bootchain->bootloader2_cmd.hardide.nb_sect = nb;
      bootchain->bootloader2_cmd.hardide.nb_sect_read = nb;
      /* number of words which are physically read by an IDE read command: */
      bootchain->bootloader2_cmd.hardide.sector_size_div2 = 512 / 2;
      if (diskindex >= 4) {
	  CHAIN ("Chain: Too high diskindex: %u max 3 in HARD IDE!\n", diskindex);
	  return 1;
	  }
      /* shall work with IDE slave 2 & 3: */
      bootchain->bootloader2_cmd.hardide.lba_head = 0x80 | ((diskindex ^ 0x02) << 4);
      }
    else {
      if (diskindex >= 0x100) {
	  CHAIN ("Chain: Too high diskindex: %u max 0xFF in BIOS!\n", diskindex);
	  return 1;
	  }
      }

  if (gujin2_loaded_with_chs (gujin2_load)) {
      unsigned HtimeS = (unsigned)geo->heads * geo->sectpertrack;
      unsigned tmp = lba % HtimeS;
      unsigned cylinder = lba / HtimeS;
      unsigned head = tmp / geo->sectpertrack;
      unsigned sector = tmp % geo->sectpertrack + 1;	/* 0 does not exist */

      if (cylinder > geo->cylinders || cylinder > 0xFFFF) {
	  CHAIN ("Chain: Too high cylinder, calculated %u max %u!\n",
			cylinder, geo->cylinders);
	  return 1;
	  }

      if (gujin2_load == gujin2_load_ide_chs) {
	  if (geo->heads > 16) {
	      CHAIN ("Chain: Too high nb heads: %u, max 16 in IDE/CHS!\n",
				geo->heads);
	      return 1;
	      }
	  if (geo->sectpertrack >= 256) {
	      CHAIN ("Chain: Too high nb sectors per track: %u max 255 in IDE/CHS!\n",
				geo->sectpertrack);
	      return 1;
	      }
	  if (cylinder >= 65536) {
	      CHAIN ("Chain: Too high cylinder, calculated %u max 65536 in IDE/CHS!\n",
				cylinder);
	      return 1;
	      }
	  bootchain->bootloader2_cmd.hardide.lba_head |= head;
	  bootchain->bootloader2_cmd.hardide.lba48_cylinder_high = 0;
	  bootchain->bootloader2_cmd.hardide.lba48_cylinder_low = 0;
	  bootchain->bootloader2_cmd.hardide.lba48_sector = 0;
	  bootchain->bootloader2_cmd.hardide.lba48_nb_sect = 0;
	  bootchain->bootloader2_cmd.hardide.cylinder_high = cylinder >> 8;
	  bootchain->bootloader2_cmd.hardide.cylinder_low = cylinder & 0xFF;
	  bootchain->bootloader2_cmd.hardide.sector = sector;

	  bootchain->bootloader2_cmd.hardide.ide_command = 0x20;
	  }
	else { /* gujin2_load == gujin2_load_bios */
	  if (geo->heads > 256) {
	      CHAIN ("Chain: Too high heads: %u, max 256 in BIOS!\n",
				geo->heads);
	      return 1;
	      }
	  if (geo->sectpertrack >= 64) {
	      CHAIN ("Chain: Too high sectpertrack: %u max 63 in BIOS!\n",
				geo->sectpertrack);
	      return 1;
	      }
	  bootchain->bootloader2_cmd.int1302.reg_cx = cylinder << 8;
	  bootchain->bootloader2_cmd.int1302.reg_cx |= sector & 0x3F;
	  bootchain->bootloader2_cmd.int1302.reg_cx |= (cylinder >> 2) & 0xC0;
	  if ((cylinder & 0xC00) != 0 && geo->heads <= 16) {
	      /* old bios with 4096 sectors, elsewhere compatible */
	      head |= ((cylinder & 0x0C00) >> 4);
	      }
	  if (nb == 0) {
	      /* so we have (loadadr == 0x07C00000 + 512 && geo->BytePerSector != 512),
		 so we are patching to not load anything, this is just to check the checksum of the
		 first partial sector, in between 0x7c00 + 512 and 0x7C00+geo->BytePerSector.
		 Call instead INT 0x13 %ah=0x10, %dl=drvnb: Test for drive ready ret %ah and carry
		 it is just a dummy call which shall not fail...
		 Does not work for me, INT 0x13 %ah=0x10 returns carry set for booted CDROM,
		 call INT 0x13 %ah=0 instead for disk 0x80.
	       */
#if 0
	      CHAIN ("Chain: Replace first read by INT 0x13 ah=0x10, test drive ready\n");
	      bootchain->bootloader2_cmd.int1302.reg_ax = 0x1000;
	      bootchain->bootloader2_cmd.int1302.reg_dx = (head << 8) | diskindex;
#else
	      CHAIN ("Chain: Replace first read by INT 0x13 ah=0x0, dl = 0x80, reset disks\n");
	      bootchain->bootloader2_cmd.int1302.reg_ax = 0x0000;
	      bootchain->bootloader2_cmd.int1302.reg_dx = 0x0080;
#endif
	      }
	    else {
	      bootchain->bootloader2_cmd.int1302.reg_ax = 0x200 | nb;
	      bootchain->bootloader2_cmd.int1302.reg_dx = (head << 8) | diskindex;
	      }

	  bootchain->bootloader2_cmd.int1302.dummy = 0;
	  bootchain->bootloader2_cmd.int1302.will_be_set_to_esbx = loadfarptr;
	  }
      return 0;
      }
    else if (   gujin2_load == gujin2_load_ebios
	     || gujin2_load == gujin2_load_CDROM_noemul) {
      if (nb == 0) {
	  /* so we have (loadadr == 0x07C00000 + 512 && geo->BytePerSector != 512),
	     so we are patching to not load anything, this is just to check the checksum of the
	     first partial sector, in between 0x7c00 + 512 and 0x7C00+geo->BytePerSector.
	     Call instead INT 0x13 %ah=0x10, %dl=drvnb: Test for drive ready ret %ah and carry
	     it is just a dummy call which shall not fail...
	     Does not work for me, INT 0x13 %ah=0x10 returns carry set for booted CDROM,
	     call INT 0x13 %ah=0 instead for disk 0x80.
	   */
#if 0
	  CHAIN ("Chain: Replace first read by INT 0x13 ah=0x10, test drive ready\n");
	  bootchain->bootloader2_cmd.int1342.reg_ax = 0x1000;
	  bootchain->bootloader2_cmd.int1342.reg_dx = diskindex;
	  bootchain->bootloader2_cmd.int1342.cst_0 = diskindex; /* saves for BOOT1_update_bootparam() */
	  bootchain->bootloader2_cmd.int1342.lba = parameter_lba;
#elif 1
	  /* This goes a lot faster, for CD_BDI Makefile target */
	  /* TODO: Check that it is working when no disk present as BIOS 0x80, status cleared in assembler */
	  CHAIN ("Chain: Replace first read by INT 0x13 ah=0x01, dl = 0x80, get status last operation (will chksm 512..bytepersector)\n");
	  bootchain->bootloader2_cmd.int1342.reg_ax = 0x0100;
	  bootchain->bootloader2_cmd.int1342.reg_dx = 0x0080;
	  bootchain->bootloader2_cmd.int1342.cst_0 = diskindex; /* saves for BOOT1_update_bootparam() */
	  bootchain->bootloader2_cmd.int1342.lba = parameter_lba;
#elif 0
	  CHAIN ("Chain: Replace first read by INT 0x13 ah=0x12, dl = 0x80, controller RAM diagnostic\n");
	  bootchain->bootloader2_cmd.int1342.reg_ax = 0x1200;
	  bootchain->bootloader2_cmd.int1342.reg_dx = 0x0080;
	  bootchain->bootloader2_cmd.int1342.cst_0 = diskindex; /* saves for BOOT1_update_bootparam() */
	  bootchain->bootloader2_cmd.int1342.lba = parameter_lba;
#else
	  CHAIN ("Chain: Replace first read by INT 0x13 ah=0x0, dl = 0x80, reset disks\n");
	  bootchain->bootloader2_cmd.int1342.reg_ax = 0x0000;
	  bootchain->bootloader2_cmd.int1342.reg_dx = 0x0080;
	  bootchain->bootloader2_cmd.int1342.cst_0 = diskindex; /* saves for BOOT1_update_bootparam() */
	  bootchain->bootloader2_cmd.int1342.lba = parameter_lba;
#endif
	  }
	else {
	  bootchain->bootloader2_cmd.int1342.reg_ax = 0x4200;
	  bootchain->bootloader2_cmd.int1342.reg_dx = diskindex;
	  bootchain->bootloader2_cmd.int1342.cst_0 = 0;
	  bootchain->bootloader2_cmd.int1342.lba = lba;
	  }
      bootchain->bootloader2_cmd.int1342.cst_16 = 16;
      bootchain->bootloader2_cmd.int1342.nb_block = nb;
      bootchain->bootloader2_cmd.int1342.cst2_0 = 0;
      bootchain->bootloader2_cmd.int1342.offset = loadfarptr & 0xFFFF;
      bootchain->bootloader2_cmd.int1342.segment = loadfarptr >> 16;
#if 0
      printf ("\nEBIOS %s %u sector at %u addr 0x%X:0x%X (%u bytes)\n",
		bootchain->bootloader2_cmd.int1342.reg_ax == 0x4200 ? "read" : "checksum",
		bootchain->bootloader2_cmd.int1342.nb_block,
		(unsigned)bootchain->bootloader2_cmd.int1342.lba,
		bootchain->bootloader2_cmd.int1342.segment,
		bootchain->bootloader2_cmd.int1342.offset,
		2 * bootchain->header.nbword);
#endif
      return 0;
      }
    else if (   gujin2_load == gujin2_load_ide_lba
	     || gujin2_load == gujin2_load_ide_lba48) {
      bootchain->bootloader2_cmd.hardide.lba48_cylinder_high = (lba >> 40) & 0xFF;
      bootchain->bootloader2_cmd.hardide.lba48_cylinder_low = (lba >> 32) & 0xFF;
      bootchain->bootloader2_cmd.hardide.lba48_sector = (lba >> 24) & 0xFF;
      bootchain->bootloader2_cmd.hardide.lba48_nb_sect = nb >> 8;
      bootchain->bootloader2_cmd.hardide.lba_head |= 0x40; /* LBA bit */
      bootchain->bootloader2_cmd.hardide.cylinder_high = (lba >> 16) & 0xFF;
      bootchain->bootloader2_cmd.hardide.cylinder_low = (lba >> 8) & 0xFF;
      bootchain->bootloader2_cmd.hardide.sector = lba & 0xFF;

      if (gujin2_load == gujin2_load_ide_lba) {
	  if ((lba >> 28) != 0) {
	      CHAIN ("Chain: Too high LBA, needs 48 bits LBA!\n");
	      return 1;
	      }
	  if ((nb >> 8) != 0) {
	      CHAIN ("Chain: Too high number of sector, needs 48 bits LBA!\n");
	      return 1;
	      }
	  bootchain->bootloader2_cmd.hardide.lba_head |= lba >> 24;
	  bootchain->bootloader2_cmd.hardide.ide_command = 0x20;
	  }
	else /* (gujin2_load == gujin2_load_ide_lba48) */{
	  bootchain->bootloader2_cmd.hardide.ide_command = 0x24;
	  }
      return 0;
      }
    else if (gujin2_loaded_externally (gujin2_load)) {
      /* bootchain->header.nbword already set up */
      return 0;
      }
    else {
      CHAIN ("Chain: %s: gujin2_load == %u!\n", __FUNCTION__, gujin2_load);
      return 1;
      }
  }

static unsigned
chs_chain (const struct install_s *install, bootloader2_t *chain, bootloader2_t *bootchain,
		bootloader2_t *uninstall_mbr, unsigned long long parameter_lba)
  {
  unsigned long long lba = install->misc.lba_start_chain,
		     last_lba = install->misc.lba_end_chain;
  unsigned cpt = 0;
  unsigned short size = 1;
  unsigned short NbSectorpertrack;
  const struct geometry_s *geo = &install->wholedisk;
  unsigned short diskindex = geo->diskindex; /* will change for auto BIOS disks */
  farptr loadadr = 0x07C00000 + 512; /* even if geo->BytePerSector != 512, to calculate checksum 512..2048 or 4096 */

  CHAIN ("Chain: MBR sector (lba = %llu) of disk 0x%X at 0x7C00, ",
		geo->start_sector, geo->diskindex);

  CHAIN ("%s: lba=%llu, last_lba=%llu, %llu - %llu = %llu, ",
	__FUNCTION__, lba, last_lba, last_lba, lba, last_lba - lba);
  if (geo->BytePerSector < 512) {
      CHAIN ("geo->Bytepersector = %u, cannot work!", geo->BytePerSector);
      return 1;
      }
    else if (geo->BytePerSector > 512) {
      CHAIN ("\nChain: geo->Bytepersector = %u, first block not loaded - just calculate checksum", geo->BytePerSector);
      size = 0;
      }
  if (gujin2_loaded_by_ide_hard (install->gujin2_load)) {
      CHAIN ("\nChain: IDE %s disk at I/O 0x%X, 0x%X, ",
		geo->diskindex ? "slave" : "master",
		install->ide.data_adr,
		install->ide.dcr_adr);
      if (install->gujin2_load == gujin2_load_ide_chs)
	  CHAIN ("CHS, C/H/S %u/%u/%u\n",
		geo->cylinders,
		geo->heads,
		geo->sectpertrack);
	else if (install->gujin2_load == gujin2_load_ide_lba)
	  CHAIN ("LBA\n");
	else
	  CHAIN ("LBA48\n");
      if (install->ide.data_adr == 0 || install->ide.dcr_adr == 0) {
	  CHAIN ("%s: Cannot chain an IDE read with ide.data_adr = 0x%X, ide.dcr_adr = 0x%X!\n",
		__FUNCTION__, install->ide.data_adr, install->ide.dcr_adr);
	  return 1;
	  }
      }
    else if (install->gujin2_load == gujin2_load_bios) {
      if (diskindex == 0x7F)
	  diskindex = install->partdisk.diskindex;
      CHAIN ("\nChain: BIOS disk 0x%X, C/H/S %u/%u/%u\n",
		diskindex,
		geo->cylinders,
		geo->heads,
		geo->sectpertrack);
      }
    else if (install->gujin2_load == gujin2_load_ebios) {
      if (diskindex == 0x7F)
	  diskindex = install->partdisk.diskindex;
      CHAIN ("\nChain: EBIOS disk 0x%X\n", diskindex);
      }
    else if (install->gujin2_load == gujin2_load_CDROM_noemul) {
      if (diskindex == 0x7F)
	  diskindex = install->partdisk.diskindex;
      CHAIN ("\nChain: noemul patched CDROM\n");
      }
    else if (gujin2_loaded_by_dos (install->gujin2_load))
      CHAIN ("\nChain: DOS\n");
    else if (gujin2_load_PIC == install->gujin2_load)
      CHAIN ("\nChain: Position Independant Code\n");
    else {
      CHAIN ("\nChain: install->gujin2_load = %u!\n", install->gujin2_load);
      return 1;
      }

#ifndef EXTENDED_FLOPPY_SIZE
  if (geo->heads == 2 && geo->sectpertrack > 18) {
      NbSectorpertrack = 18;	/* assume that only locally */
      CHAIN ("Chain: limit read size to %u sectors\n", NbSectorpertrack);
      }
    else
#endif
	 if (geo->heads == 0 && geo->sectpertrack != 0) {
      NbSectorpertrack = geo->sectpertrack & 0x7F;
      CHAIN ("Chain: limit read size to %u sectors\n", NbSectorpertrack);
      if (geo->sectpertrack & 0x80) {
	  /* NOTE: needed even for uninstall_mbr! */
	  CHAIN ("Chain: Partition relocation base address %llu\n", install->partdisk.start_sector);
	  lba -= install->partdisk.start_sector;
	  last_lba -= install->partdisk.start_sector;
	  }
      }
    else
      NbSectorpertrack = geo->sectpertrack;

  chain->header.next = install->bootchain.adr;

  do {
      farptr nextloadadr;

      if (NbSectorpertrack != 0 && geo->heads != 0) {
	  if (gujin2_loaded_with_chs (install->gujin2_load)
		&& NbSectorpertrack > 1 && size > NbSectorpertrack - lba % NbSectorpertrack)
	      size = NbSectorpertrack - lba % NbSectorpertrack;

	  /* Chain[01]: 15 sector at LBA 153, C/H/S: 0/02/28 loaded at 0x08400, i.e. 0x07C00800..0x07C07FFF
		so next load address at 0x08400 + 15 * 2048 = 0xFC00, would set size = 0 */
	  if (geo->BytePerSector == 512) {
#ifdef EXTENDED_FLOPPY_SIZE
	      if (geo->heads == 2 && NbSectorpertrack > 18 && size > 18)
		  size = 18;
#endif

#define LINADR(adr) (((adr) & 0xFFFF) + (((adr) & 0xFFFF0000) >> 12))

	      /* If we have sector size > 512, we need to assume there is no DMA problem else we cannot load (load addr aligned only to 512) */
	      if (geo->BytePerSector <= 512 && (LINADR(loadadr) + size * geo->BytePerSector) >> 16 != LINADR(loadadr) >> 16) /* floppy DMA problem */
		  size = (0x10000 - (LINADR(loadadr) & 0xFFFF)) / geo->BytePerSector;
	      }
	  }
	else if (NbSectorpertrack != 0 && size > NbSectorpertrack) {
	  /* TODO: document that NbSectorpertrack != 0 && geo->heads == 0
	   * means set maximum read size in sectors to NbSectorpertrack
	   * included, zero meanning no limit. */
	  CHAIN ("[limit %u sectors] ", NbSectorpertrack);
	  size = NbSectorpertrack;
	  }

      if (install->misc.maxsectloadtouse && size > install->misc.maxsectloadtouse) {
	  CHAIN ("[limit %u sectors] ", install->misc.maxsectloadtouse);
	  size = install->misc.maxsectloadtouse;
	  }

      if (geo->BytePerSector > 512 && loadadr == 0x07C00000 + 512 && size == 0)
	  nextloadadr = 0x07C00000 + geo->BytePerSector;
	else
	  nextloadadr = loadadr + size * geo->BytePerSector;
      if ((nextloadadr >> 16) != (loadadr >> 16)) {
	  /* We need to adjust %es in the loader, so stop with %bx == 0 */
	  size = (0x10000 - (loadadr & 0xFFFF)) / geo->BytePerSector;
	  nextloadadr = loadadr + size * geo->BytePerSector;
	  nextloadadr -= 0x10000;
	  nextloadadr += 0x10000000;
	  }

      if (set_bootchain (chain, lba, size, loadadr, install->gujin2_load,
			geo, diskindex, &install->ide, parameter_lba) != 0) {
	  EXTRA ("Error in set_bootchain for main loader chain.\n");
	  return 1;
	  }

      CHAIN ("Chain[%02u]: %2u sector at LBA %3u, ", cpt, size, (unsigned)lba);
      if (geo->sectpertrack != 0 && geo->heads != 0)
	  CHAIN ("C/H/S: %u/%02u/%02u",
		(unsigned)(lba / geo->sectpertrack / geo->heads),
		(unsigned)(lba / geo->sectpertrack % geo->heads),
		(unsigned)(lba % geo->sectpertrack + 1));
      CHAIN (" loaded at 0x%05X, i.e. 0x%08X..0x%08X\n",
		LINADR(loadadr), loadadr,
		//loadadr + size * geo->BytePerSector);
		loadadr + 2 * chain->header.nbword - 1);
#undef LINADR

  /* If BytePerSector > 512, chain the first sector to calculate the end of the sector checksum: */
      if (size == 0 && cpt == 0 && geo->BytePerSector > 512)
	  lba ++;
	else
	  lba += size;
      loadadr = nextloadadr;
      chain = &bootchain[cpt];
      chain->header.next = (cpt + 1) * sizeof (bootloader2_t) + install->bootchain.adr;
      size = last_lba - lba;
      cpt ++;
      } while (size != 0 && cpt < 100);
  cpt --;

  if (cpt >= install->bootchain.nbblock) {
      /* We have had memory corruption, but we know the number
	 of element we really need: */
      CHAIN ("cannot fit %u blocks in %u, increase bootchain[] size "
		"in boot.c!\n", cpt + 1, install->bootchain.nbblock);
      return 1;
      }
  bootchain[cpt - 1].header.next = 0;
  if (cpt < install->bootchain.nbblock - 2)
      strcpy ((char *)&bootchain[install->bootchain.nbblock-2],
		"Copyright 2010 Etienne Lorrain");
  CHAIN ("Chain: using %u element (max %u), i.e %u bytes for bootchain[] (max %u)\n",
		cpt,
		install->bootchain.nbblock,
		cpt * (unsigned)sizeof(bootchain[0]),
		install->bootchain.nbblock * (unsigned)sizeof(bootchain[0]));

  if (uninstall_mbr == 0)
      CHAIN ("Chain: do not chain any MBR recovery\n");
    else if (set_bootchain (&uninstall_mbr[0], last_lba, 1, 0x7c00,
		install->gujin2_load, geo, diskindex, &install->ide, 0) != 0) {
      EXTRA ("Error in set_bootchain for MBR recovery error.\n");
      return 1;
      }
    else if (set_bootchain (&uninstall_mbr[1], 0, 1, 0x7c00 + 4096,
		install->gujin2_load, geo, diskindex, &install->ide, 0) != 0) {
      EXTRA ("Error in set_bootchain for read MBR (partition table) error.\n");
      return 1;
      }
    else if (set_bootchain (&uninstall_mbr[2], 0, 1, 0x7c00,
		install->gujin2_load, geo, diskindex, &install->ide, 0) != 0) {
      EXTRA ("Error in set_bootchain for write MBR (recovery + partition table) error.\n");
      return 1;
      }
    else {
      /* transform the read of uninstall_mbr[2] in write: */
	if (gujin2_loaded_by_ide_hard (install->gujin2_load)) {
	    if (uninstall_mbr[2].bootloader2_cmd.hardide.ide_command == 0x24)
		uninstall_mbr[2].bootloader2_cmd.hardide.ide_command = 0x34;
	      else if (uninstall_mbr[2].bootloader2_cmd.hardide.ide_command == 0x20)
		uninstall_mbr[2].bootloader2_cmd.hardide.ide_command = 0x30;
	      else {
		EXTRA ("Error did not recognised hardide.ide_command 0x%X\n",
			uninstall_mbr[2].bootloader2_cmd.hardide.ide_command);
		return 1;
		}
	    }
	  else if (install->gujin2_load == gujin2_load_ebios) {
	    if (uninstall_mbr[2].bootloader2_cmd.int1342.reg_ax != 0x4200) {
		EXTRA ("Error did not recognised EBIOS command 0x%X for MBR recovery\n",
			uninstall_mbr[2].bootloader2_cmd.int1342.reg_ax);
		return 1;
		}
	    uninstall_mbr[2].bootloader2_cmd.int1342.reg_ax = 0x4300;
	    }
	  else if (install->gujin2_load == gujin2_load_bios) {
	    if (uninstall_mbr[2].bootloader2_cmd.int1302.reg_ax >> 8 != 0x02) {
		EXTRA ("Error did not recognised BIOS command 0x%X for MBR recovery\n",
			uninstall_mbr[2].bootloader2_cmd.int1302.reg_ax >> 8);
		return 1;
		}
	    uninstall_mbr[2].bootloader2_cmd.int1302.reg_ax += 0x0100; /* i.e. 0x03 << 8 | nb */
	    }
	  else {
	    EXTRA ("Error not loaded IDE hard nor BIOS and uninstall_mbr not null, abort.\n");
	    return 1;
	    }

      CHAIN ("Chain: will save MBR (for MBR recovery) at lba: %u\n", (unsigned)last_lba);
      uninstall_mbr[0].header.nbword = geo->BytePerSector / 2; /* no more gray in menu */
      }
  return 0;
  }

/**
 ** Checksuming:
 **/
/* Check the checksum of /part of/ the initial sector (loaded by MBR),
   in case of bad assembler: */
static inline void
check_initial_boot1_checksum (unsigned char *file,
			      instboot_info_t *instboot_info,
			      boot1param_t *boot1param)
  {
  unsigned short *ptr = (unsigned short *)(file + instboot_info->checksum_start_adr);
  unsigned short *tmp_patch = (unsigned short *)(file+instboot_info->patch2adr);
  unsigned short checksum = 0;

  *tmp_patch -= instboot_info->deltaseg;

  while (ptr < &boot1param->boot1_checksum)
      checksum -= *ptr++;

  *tmp_patch += instboot_info->deltaseg;

  if (boot1param->boot1_checksum != checksum)
      EXTRA ("WARNING: checksum for Level 1 is incorrect, it is 0x%X "
		"and should be 0x%X (corrected).\n",
		boot1param->boot1_checksum, checksum);

  }

static inline void
checksum_first (unsigned char	*file,
		unsigned short	*ptr,
		boot1param_t	*boot1param,
		unsigned	Bytepersector)
  {
  unsigned nbword = boot1param->bootloader2.header.nbword;
  unsigned short checksum = 0;

//printf ("initial checksum from 0x%lX ", (unsigned long)ptr - (unsigned long)file - 2);
  while (nbword --)
      checksum -= *--ptr;
//printf ("downto 0x%lX\n", (unsigned long)ptr - (unsigned long)file);
  boot1param->bootloader2.header.checksum = checksum;
  }

static inline void
checksum_boot1 (unsigned char *file, const instboot_info_t *instboot_info, boot1param_t *boot1param)
  {
  unsigned short *ptr = (unsigned short *)(file + instboot_info->checksum_start_adr);
  unsigned short checksum = 0;

  /* Checksum of the (patched) sector loaded by the BIOS: */
//printf ("boot1 checksum from 0x%lX ", (unsigned long)ptr - (unsigned long)file);
  while (ptr < &boot1param->boot1_checksum)
      checksum -= *ptr++;

  ptr ++;	/* skip the actual checksum */

  while (ptr < (unsigned short *)(file + instboot_info->bootend_adr))
      checksum -= *ptr++;
//printf ("to 0x%lX\n", (unsigned long)ptr - (unsigned long)file - 2);
  boot1param->boot1_checksum = checksum;
  }

static unsigned
calculate_checksum (unsigned char *file, unsigned filesize,
		    instboot_info_t *instboot_info,
		    boot1param_t *boot1param,
		    bootloader2_t *bootchain,
		    unsigned Bytepersector)
  {
#if 1
  unsigned short *ptr = (unsigned short *)(file + filesize);
  signed index = 0;
  unsigned nbwordloaded = 0;

  for (;;) {
      bootchain[index].header.checksum = 0;
      nbwordloaded += bootchain[index].header.nbword;
      if (bootchain[index].header.next == 0)
	  break;
      index ++;
      }
  if (Bytepersector == 512) {
      if (2 * nbwordloaded != filesize - 2 * Bytepersector) {
	  EXTRA ("chain loads %u bytes instead of filesize-2*byte/sector = %u bytes (delta %d)!\n",
		2 * nbwordloaded, filesize - 2 * Bytepersector,
		2 * nbwordloaded - (filesize - 2 * Bytepersector));
	  return 1;
	  }
      }
    else /* if (boot1param->bootloader2.header.nbword == (2048-512) / 2) */ { /* first slot to calculate remaining checksum */
      if (2 * nbwordloaded != filesize - Bytepersector) {
	  EXTRA ("chain loads %u bytes instead of filesize-byte/sector = %u bytes (delta %d)!\n",
		2 * nbwordloaded, filesize - Bytepersector,
		2 * nbwordloaded - (filesize - Bytepersector));
	  return 1;
	  }
      }

  index++;
  while (--index >= 0) {
      unsigned short checksum = 0;
      unsigned cpt = bootchain[index].header.nbword + 1;

//printf ("boot2 checksum from 0x%lX ", (unsigned long)ptr - 2 - (unsigned long)file);
      while (--cpt)
	  checksum += *--ptr;
//printf ("downto 0x%lX\n", (unsigned long)ptr - (unsigned long)file);
      bootchain[index].header.checksum = - checksum;
      }

  if (Bytepersector == 512) {
      if ((unsigned char *)ptr != file + 2 * Bytepersector) {
	  EXTRA ("Checksum for chain does not cover all of boot2 (delta = %d)!\n",
		(int)((unsigned char *)ptr - file - 2 * Bytepersector));
	  return 1;
	  }
      }
    else {
      if ((unsigned char *)ptr != file + Bytepersector) {
	  EXTRA ("Checksum for chain does not cover all of boot2 (delta = %d)!\n",
		(int)((unsigned char *)ptr - file - Bytepersector));
	  return 1;
	  }
      }

  checksum_first (file, ptr, boot1param, Bytepersector);
  checksum_boot1 (file, instboot_info, boot1param);

  return 0;
#else  /* check that the simplified version used in the self installer works: */
  unsigned short *ptr, tmp_checksum;
  signed index = 0;
  unsigned cpt;

  if (Bytepersector == 512)
      ptr = (unsigned short *)(file + 1024);
    else
      ptr = (unsigned short *)(file + Bytepersector);

  do
      ptr += bootchain[index].header.nbword;
      while (bootchain[index++].header.next != 0);

  while (--index >= 0) {
      bootchain[index].header.checksum = 0;
      cpt = bootchain[index].header.nbword + 1;
      while (--cpt)
	  bootchain[index].header.checksum -= *--ptr;
      }

  boot1param->bootloader2.header.checksum = 0;
  cpt = boot1param->bootloader2.header.nbword;
  while (cpt --)
      boot1param->bootloader2.header.checksum -= *--ptr;

  ptr = (unsigned short *)(file + instboot_info->checksum_start_adr);
  tmp_checksum = 0;
  boot1param->boot1_checksum = 0;	/* skip the actual checksum by substracting 0 */
  while (ptr < (unsigned short *)(file + instboot_info->bootend_adr))
      tmp_checksum -= *ptr++;
  boot1param->boot1_checksum = tmp_checksum;

  return 0;
#endif
  }

/**
 ** Patching:
 **/
static inline unsigned
check_read_disk_instructions (unsigned char *file, unsigned patch3adr,
		unsigned partition_relative_read_disk, unsigned read_disk)
  {
  unsigned short read_disk_adr, movaxdx_instr;
  unsigned char ret_instr, loadsw_instr;

  read_disk_adr = *(unsigned short *)((unsigned long)file + patch3adr);
  read_disk_adr += patch3adr + 2; /* relative addr */

  if (read_disk != read_disk_adr) {
      EXTRA ("instboot_info: read_disk function at 0x%X, "
		"and the callw calls address 0x%X, abort.\n",
		read_disk, read_disk_adr);
      return 1;
      }
  ret_instr = *(unsigned char *)((unsigned long)file + partition_relative_read_disk - 1);
  loadsw_instr = *(unsigned char *)((unsigned long)file + read_disk_adr);
  movaxdx_instr = *(unsigned short *)((unsigned long)file + read_disk_adr + 1);
  if (ret_instr != 0xC3) {
      EXTRA ("instboot_info: instruction before partition_relative_read_disk at 0x%X"
		" is not \"retw\" but (0x%X), abort.\n",
		partition_relative_read_disk, ret_instr);
      return 1;
      }
  if (loadsw_instr != 0xAD) {
      EXTRA ("instboot_info: read_disk function at 0x%X, "
		"first instruction is not \"lodsw\" (0x%X), abort.\n",
		read_disk_adr, loadsw_instr);
      return 1;
      }
  if (movaxdx_instr != 0xC289) {
      EXTRA ("instboot_info: read_disk function at 0x%X, "
		"second instruction is not \"movw %%ax,%%dx\" (0x%X), abort.\n",
		read_disk_adr, movaxdx_instr);
      return 1;
      }
  return 0;
  }

static inline unsigned
check_patch (unsigned char *file, instboot_info_t *instboot_info)
  {
  if (*(unsigned short *)(file+instboot_info->patch2adr) != 0x17C0 + instboot_info->deltaseg) {
      EXTRA ("Unrecognised patch pattern 2, abort.\n");
      return 1;
      }
  if (check_read_disk_instructions (file, instboot_info->patch3adr,
				instboot_info->partition_relative_read_disk_adr,
				instboot_info->read_disk_adr) != 0)
      return 1;
  if (*(unsigned *)(file + instboot_info->patch4adr) != 0x18CD19CD) {
      EXTRA ("Unrecognised pattern 4 for patch: 0x%X, abort.\n",
		*(unsigned *)(file + instboot_info->patch4adr));
      return 1;
      }
  return 0;
  }

static inline void
do_patch (unsigned char *file, instboot_info_t *instboot_info, enum gujin2_load_t gujin2_load,
	  unsigned do_auto_BIOSdlDISK_patch, unsigned do_partition_relative_patch,
	  unsigned do_single_sector_load_patch, unsigned do_read_retry_patch)
  {
  if (gujin2_load == gujin2_load_dos_com) /* %ds = %cs - 0x10 */
      *(unsigned short *)(file+instboot_info->patch2adr) = 0x0010 + instboot_info->deltaseg;
    else if (gujin2_load == gujin2_load_dos_exe || gujin2_load == gujin2_load_PIC)
      *(unsigned short *)(file+instboot_info->patch2adr) = instboot_info->deltaseg;

  if (gujin2_load == gujin2_load_dos_com || gujin2_load == gujin2_load_dos_exe
					 || gujin2_load == gujin2_load_PIC) {
      *(unsigned short *)(file+instboot_info->patch3adr) -= instboot_info->read_disk_adr;
      *(unsigned short *)(file+instboot_info->patch3adr) += instboot_info->partition_relative_read_disk_adr;
      *(unsigned short *)(file+instboot_info->patch3adr) -= 1;	/* will call a ret */
      EXTRA ("Extra: loader no-call patch done\n");
      }
    else if (gujin2_load == gujin2_load_CDROM_noemul) {
      unsigned short read_disk_adr = *(unsigned short *)((unsigned long)file + instboot_info->patch3adr);
      read_disk_adr += instboot_info->patch3adr + 2; /* relative addr for call */

      *(unsigned short *)(file+instboot_info->patch3adr) -= read_disk_adr;
      *(unsigned short *)(file+instboot_info->patch3adr) += instboot_info->read_disk_1st_adr;
      EXTRA ("Extra: CDROM initial intercept patch done\n");
      }
    else if (do_partition_relative_patch) {
      *(unsigned short *)(file+instboot_info->patch3adr) -= instboot_info->read_disk_adr;
      *(unsigned short *)(file+instboot_info->patch3adr) += instboot_info->partition_relative_read_disk_adr;
      EXTRA ("Extra: partition relative access patch done\n");
      }

  if (do_auto_BIOSdlDISK_patch) {
      unsigned short *movaxdx_ptr = (unsigned short *)((unsigned long)file + instboot_info->read_disk_adr + 1);
      EXTRA ("Extra: Auto disk (disk got at boot in DL register) patch (replace 0xC289==0x%X by 0xE688) done\n",
		*movaxdx_ptr);
      *movaxdx_ptr = 0xE688; /* mov %ah,%dh */
      }

  if (gujin2_load == gujin2_load_dos_com || gujin2_load == gujin2_load_dos_exe) {
      *(unsigned *)(file+instboot_info->patch4adr) = 0x21CD4CB4;
      EXTRA ("Extra: DOS exit patch done\n");
      }
    else if (do_read_retry_patch) {
      *(unsigned *)(file+instboot_info->patch4adr) = 0xCF1F0761;
      EXTRA ("Extra: read-retry patch done\n");
      }

  if (do_single_sector_load_patch) { /* This is incompatible with IDE or EBIOS processing, just for boot1 to load boot2 */
	/* Those defines are produced by a "xxd | diff" of boot.bin compiled with or without INT13SINGLE, see boot.c */
#define INIT_PATTERN_1	"\x6e\x5a\xe2\xf3\x6e\xec\xec\xa8\x81\xe0\xfb\xe3\x38\xa8\x08\x74\xf5\x8b\x4c\x02\x52\x83\xea\x07\xf3\x6d\x5a\xff\x0c\x75\xe6"
#define REPL_PATTERN_1	"\x2e\xfe\x0e\x0a\x00\x74\x0b\xfe\xc7\xfe\xc7\xfe\xc1\xb8\x01\x02\xeb\x52\x8b\x5c\x04\xc3\x89\x5c\x04\x2e\xa2\x0a\x00\xb0\x01"
#define INIT_PATTERN_2	"\x89\x5c\x04\xcd\x13\x73\x05"
#define REPL_PATTERN_2	"\xe8\xb2\xff\xcd\x13\x73\x98"
      if (memcmp (file + instboot_info->singlesectorload_patch1, INIT_PATTERN_1, sizeof(INIT_PATTERN_1) - 1))
	  ERROR ("Cannot apply single_sector_load patch, pattern 1 not recognised at %u.\n", instboot_info->singlesectorload_patch1);
	else if (memcmp (file + instboot_info->singlesectorload_patch2, INIT_PATTERN_2, sizeof(INIT_PATTERN_2) - 1))
	  ERROR ("Cannot apply single_sector_load patch, pattern 2 not recognised at %u.\n", instboot_info->singlesectorload_patch2);
	else {
	  memcpy (file + instboot_info->singlesectorload_patch1, REPL_PATTERN_1, sizeof(INIT_PATTERN_1) - 1);
	  memcpy (file + instboot_info->singlesectorload_patch2, REPL_PATTERN_2, sizeof(INIT_PATTERN_2) - 1);
	  EXTRA ("Extra: single_sector_load patch applied successfully\n");
	  }
      }
  }

/**
 ** File operations:
 **/

/*
 * Input:
 */
#ifdef WITH_INPUT_FILES
/* the "gujin" executable is exactly the same as the "instboot" executable,
    but already has the usual input files as constant inside the ELF file: */
/* They are in section .data because may be need to be modified */
asm (
"	.section	.data					\n"
"start_boot: .incbin \"gujin_images/boot.bin\"			\n"
"sizeof_boot = . - start_boot					\n"
"start_tinystd: .incbin \"gujin_images/tinystd.bin\"		\n"
"sizeof_tinystd = . - start_tinystd				\n"
"start_tinyusb: .incbin \"gujin_images/tinyusb.bin\"		\n"
"sizeof_tinyusb = . - start_tinyusb				\n"
"start_tinyext4: .incbin \"gujin_images/tinyext4.bin\"		\n"
"sizeof_tinyext4 = . - start_tinyext4				\n"
"start_tinycdrom: .incbin \"gujin_images/tinycdrom.bin\"	\n"
"sizeof_tinycdrom = . - start_tinycdrom				\n"
"start_tinyexe: .incbin \"gujin_images/tinyexe.bin\"		\n"
"sizeof_tinyexe = . - start_tinyexe				\n"
"	.previous						\n"
);
extern unsigned char start_boot[], sizeof_boot[];
extern unsigned char start_tinystd[], sizeof_tinystd[];
extern unsigned char start_tinyusb[], sizeof_tinyusb[];
extern unsigned char start_tinyext4[], sizeof_tinyext4[];
extern unsigned char start_tinycdrom[], sizeof_tinycdrom[];
extern unsigned char start_tinyexe[], sizeof_tinyexe[];
#endif

static inline unsigned char *
allocate_and_read (const char *fin_name, unsigned *filesize)
  {
  FILE *fin = 0;
  unsigned char *file, tmp;
  struct stat statbuf;
  /* We still do not know install.wholedisk.BytePerSector anyways,
     so we will correct it later if it is 2048 like CDROM by a realloc(): */
  unsigned sectorsize = 512;

#ifdef WITH_INPUT_FILES
  EXTRA ("looking for input file: \"%s\" in the internal file list\n", fin_name);
  if (!strcmp (fin_name, "boot.bin")) {
      statbuf.st_size = (unsigned long)sizeof_boot;
      fin = (FILE *)start_boot;
      }
  else if (!strcmp (fin_name, "tinystd.bin")) {
      statbuf.st_size = (unsigned long)sizeof_tinystd;
      fin = (FILE *)start_tinystd;
      }
  else if (!strcmp (fin_name, "tinyusb.bin")) {
      statbuf.st_size = (unsigned long)sizeof_tinyusb;
      fin = (FILE *)start_tinyusb;
      }
  else if (!strcmp (fin_name, "tinyext4.bin")) {
      statbuf.st_size = (unsigned long)sizeof_tinyext4;
      fin = (FILE *)start_tinyext4;
      }
  else if (!strcmp (fin_name, "tinycdrom.bin")) {
      statbuf.st_size = (unsigned long)sizeof_tinycdrom;
      fin = (FILE *)start_tinycdrom;
      }
  else if (!strcmp (fin_name, "tinyexe.bin")) {
      statbuf.st_size = (unsigned long)sizeof_tinyexe;
      fin = (FILE *)start_tinyexe;
      }
  else
#endif
  if (stat(fin_name, &statbuf) != 0) {
      EXTRA ("cannot stat input file: \"%s\"\n", fin_name);
      return 0;
      }
  /* Maximum limit due to 80x86 architecture: */
  if (statbuf.st_size <= sectorsize || statbuf.st_size >= 6*64*1024) {
      EXTRA ("input file %s invalid size: %d\n",
		 fin_name, (unsigned)statbuf.st_size);
      return 0;
      }
  *filesize = DIVROUNDUP (statbuf.st_size, sectorsize) * sectorsize;
  if ((file = malloc (*filesize)) == NULL) {
      EXTRA ("cannot malloc memory for buffer.\n");
      return 0;
      }
  memset (file, 0, *filesize);
#ifdef WITH_INPUT_FILES
  if (fin != 0) {
      memcpy (file, fin, statbuf.st_size);
      return file;
      }
#endif
  if ((fin = fopen (fin_name, "r")) == NULL) {
      EXTRA ("cannot open input file: \"%s\"\n", fin_name);
      free (file);
      return 0;
      }

  if (fread(file, 1, statbuf.st_size, fin) != statbuf.st_size) {
      EXTRA ("cannot read the complete array (hardware error ?)\n");
      free (file);
      if (fclose (fin) == EOF)
	  EXTRA ("cannot close input file: %s\n", fin_name);
      return 0;
      }
/*  if (!feof(fin)) { */
  if (fread(&tmp, 1, 1, fin) != 0) {
      EXTRA ("file not at end after reading array (concurrent access ?)\n");
      free (file);
      if (fclose (fin) == EOF) {
	  EXTRA ("cannot close input file: %s\n", fin_name);
	  return 0;
	  }
      return 0;
      }
  if (fclose (fin) == EOF) {
      EXTRA ("cannot close input file: %s\n", fin_name);
      free (file);
      return 0;
      }
  return file;
  }

/**
 ** Output:
 **/
#if 0 /* buffered version and fseek()/ftell() or fseeko()/ftello() interface. */
/*
 * Comment about _FILE_OFFSET_BITS:
 * look at glibc 2.1+ documentation: off_t is then 64 bits, and I use
 * it to probe the size of deviceswith fseek()/ftell().
 * It works without being defined but you may be limited to partition
 * size and/or disk size in _bytes_ below 2^31 (2 Gb).
 * Because I just create FAT12 or FAT16, I do not care so much, but
 * when the total size of the disk is needed to manage sectors with
 * a size not equal to 512 or to place and manage the BEER sectors & partition.
 *
 * To know the version of glib you are using, try "glib-config --version"
 * I should maybe include "/usr/lib/glib/include/glibconfig.h"
 * and test GLIB_MAJOR_VERSION / GLIB_MINOR_VERSION
 * Update:
 * It does not work:
 * For me it is a bug - if off_t is 64 bits, maybe the system cannot
 * manage fseeko()/ftello() of more than 32 bits - but the 64 bits
 * version of fseeko()/ftello() should _at_least_ return an error
 * when (pos >> 32 != 0) ... See FSEEKO_DOUBLECHECK.
 * Fortunately this hard disk was empty.
 */

#define xFILE	FILE *
#define xoff_t	off_t
#define xfileno(xF)	fileno(xF)

static int xwrite (xFILE fout, const void *buf, unsigned size)
  {
  return fwrite (buf, 1, size, fout) != size;
  }

static int xread (xFILE fout, void *buf, unsigned size)
  {
  return fread (buf, 1, size, fout) != size;
  }

static int xfopen (const char *filename, xFILE *xfd,
		int (*xfct) (xFILE fin, const void *buf, unsigned size))
  {
  if (sizeof(xoff_t)) <= 4)
      return EINVAL;	/* will not work anyway */
  return (*xfd = fopen (filename, (xfct == xwrite)? "w" : "r")) == NULL;
  }

static int xfopenmodify (const char *filename, xFILE *xfd,
		int (*xfct) (xFILE fin, const void *buf, unsigned size))
  {
  return (*xfd = fopen (filename, (xfct == xwrite)? "w+" : "r")) == NULL;
  }

static unsigned xclose (xFILE fout)
  {
  return (fclose (fout) == EOF);
  }

static unsigned xset_location (xFILE fout, long long fulloffset, long long position, unsigned BytePerSector)
  {
  long long pos = position * BytePerSector + fulloffset;
  if (pos >= 0x80000000LL)
      return 1;
  LOCATION ("Location: will fseek: %llu * %u + %llu * %u, i.e. %llu\n",
	position, BytePerSector, fulloffset / BytePerSector, BytePerSector, pos);
  return (fseek (fout, pos, SEEK_SET) != 0);
  }

/* Note: if you use /dev/null as a target, those ftell() give bogus values: */
static void xcheck_location (xFILE fout, long long fulloffset, long long pos, unsigned BytePerSector, const char *msg)
  {
  LOCATION ("Location: ftell: %ld sector, should be %lld sector, %s\n",
	ftell (fout) / BytePerSector, fulloffset / BytePerSector + pos, msg);
  }

#define DISPLAY_WARNING_32BIT_OFF_T() \
  if (sizeof (off_t) <= 4) {					\
      WARNING ("Not using 64 bits for 'off_t' type will limit the size of disks "	\
		"managed to %u Mbytes.\n", 0x7FFFFFFF / 1024 / 1024);			\
      }

#else /*************** default: unbuffered version and lseek64() interface. */

#define xFILE	int
#define xoff_t	long long
#define xfileno(xF)	xF

static int xwrite (xFILE fout, const void *buf, unsigned size)
  {
  return write (fout, buf, size) != size;
  }

static int xread (xFILE fout, void *buf, unsigned size)
  {
  return read (fout, buf, size) != size;
  }

static int xfopen (const char *filename, xFILE *xp,
		int (*xfct) (xFILE fin, const void *buf, unsigned size))
  {
  return (*xp = open (filename, (xfct == xwrite)? (O_CREAT | O_WRONLY | O_TRUNC | O_SYNC)
					 : O_RDONLY, S_IRUSR | S_IWUSR)) == -1;
  }

static int xfopenmodify (const char *filename, xFILE *xp,
		int (*xfct) (xFILE fin, const void *buf, unsigned size))
  {
  return (*xp = open (filename, (xfct == xwrite)? (O_CREAT | O_WRONLY | O_SYNC)
					 : O_RDONLY, S_IRUSR | S_IWUSR)) == -1;
  }

static unsigned xclose (xFILE fout)
  {
  return (close (fout) == -1);
  }

static unsigned xset_location (xFILE fout, long long fulloffset, long long position, unsigned BytePerSector)
  {
  long long pos = position * BytePerSector + fulloffset;

  LOCATION ("Location: will lseek64: %llu * %u + %llu * %u, i.e. %llu\n",
	position, BytePerSector, fulloffset / BytePerSector, BytePerSector, pos);
  return lseek64 (fout, pos, SEEK_SET) != pos;
  }

/* Note: if you use /dev/null as a target, those seek() give bogus values: */
static void xcheck_location (xFILE fout, long long fulloffset, long long pos, unsigned BytePerSector, const char *msg)
  {
  LOCATION ("Location: lseek64(0, SEEK_CUR): %lld sector, should be %lld sector, %s\n",
	(unsigned long long)lseek64 (fout, 0, SEEK_CUR) / BytePerSector, fulloffset / BytePerSector + pos, msg);
  }

#define DISPLAY_WARNING_32BIT_OFF_T() /* */

#endif /* buffered / unbuffered read/write/position */

static unsigned xcheck_cluster = 0;
static int xcheck (xFILE fin, const void *_buf, unsigned size)
  {
  static unsigned long long compare_offset = 0;
  unsigned char data[size];
  unsigned cpt;
  const unsigned char *buf = (const unsigned char *)_buf;

  if (xread (fin, data, size))
      return 1;
  if (memcmp(_buf, data, size) == 0)
      compare_offset += size;
    else {
      if (xcheck_cluster) {
	  PRINTF ("\rCOMPARISSON ERROR on cluster %u                   \n", xcheck_cluster);
	  compare_offset += size;
	  return 1;
	  }
	else for (cpt = 0; cpt < size; cpt++) {
	  if (data[cpt] != buf[cpt]) {
	      ERROR ("COMPARISSON ERROR (bad media or still not sync'ed?)\n"
		 "  difference detected at offset %llu, i.e. 0x%llX,\n"
		 "  wrote: 0x%02X 0x%02X 0x%02X 0x%02X; reading back: 0x%02X 0x%02X 0x%02X 0x%02X\n",
		 compare_offset, compare_offset,
		 buf[cpt], buf[cpt+1], buf[cpt+2], buf[cpt+3],
		 data[cpt], data[cpt+1], data[cpt+2], data[cpt+3]);
	      return 1;
	      }
	  compare_offset++;
	  }
      }
  return 0;
  }

#ifdef BLKGETSIZE64
#define GET_FD_LENGTH(fd) ({ \
	long long total_length;					\
	if (ioctl (fd, BLKGETSIZE64, &total_length) != 0)	\
	    total_length = -1LL;				\
	/* return */ total_length;				\
      })
#elif 1
#define GET_FD_LENGTH(fd) ({ \
	/* return */ lseek64 (fd, 0, SEEK_END);			\
	})
#else
#define GET_FD_LENGTH(fd) ({ \
	/* return */  lseek (fd, 0, SEEK_END);			\
	})
#endif

/*
 * Creation of a FAT12/FAT16/FAT32 filesystem:
 */
static unsigned
write_fat_12 (xFILE fout,
		unsigned char	dsk_mediadescriptor,
		unsigned short	nb_cluster_in_file,
		unsigned	nb_cluster_on_filesystem,
		unsigned	cluster_offset,
		unsigned	sizeof_fat,
		unsigned char *const buffer)
  {
  unsigned char *bufptr = buffer;

  if (nb_cluster_on_filesystem + 2 > sizeof_fat) {
      FILESYS ("Fs: %s: FAT parameters invalid: nb_cluster_on_filesystem %u, sizeof_fat %u\n",
		__FUNCTION__, nb_cluster_on_filesystem, sizeof_fat);
      return 1;
      }

  *bufptr++ = dsk_mediadescriptor;
  *bufptr++ = 0xFF;
  *bufptr++ = 0xFF;

  if (nb_cluster_in_file != 0) { /* else no end marker */
      unsigned short value;
      unsigned cpt;

      /* Check size before filling: */
      if (cluster_offset + nb_cluster_in_file > nb_cluster_on_filesystem) {
	  FILESYS ("Fs: %s: cluster_offset %u, nb_cluster_in_file %u, nb_cluster_on_filesystem %u, FAT overflow!\n",
			__FUNCTION__, cluster_offset, nb_cluster_in_file, nb_cluster_on_filesystem);
	  return 1;
	  }
      for (cpt = 0; cpt < 3 * cluster_offset / 2; cpt++)
	  *bufptr++ = 0;

      if (nb_cluster_in_file == 1)
	  value = 0x0FFF;
	else {
	  value = 0;
	  for (cpt = 0; cpt < nb_cluster_in_file - 1; cpt++) {
	      if (((cpt + cluster_offset) & 0x01) == 0) {
		  value = 2 + cluster_offset + cpt + 1; /* chain next cluster */
		  *bufptr++ = value & 0xFF;
		  value >>= 8;
		  /* setup the end marker if we exit here: */
		  value |= 0xFFF0;
		  }
		else {
		  value &= 0x0F;
		  value |= (2 + cluster_offset + cpt + 1) << 4;
		  *bufptr++ = value & 0xFF;
		  *bufptr++ = value >> 8;
		  /* setup the end marker if we exit here: */
		  value = 0x0FFF;
		  }
	      }
	  }
      /* write the end marker: This may erase the 4 msb bits of the
	 next cluster, but we do not care here... */
      *bufptr++ = value & 0xFF;
      *bufptr++ = value >> 8;
      }

  /* Fill end of FAT: */
  while (bufptr < &buffer[DIVROUNDUP((nb_cluster_on_filesystem + 2) * 3, 2)])
      *bufptr++ = 0;

  FILESYS ("Fs: %s: FAT uses %u bytes and contains %u bytes, %u bytes emptied\n",
		__FUNCTION__, DIVROUNDUP((nb_cluster_on_filesystem + 2) * 3, 2),
		sizeof_fat, sizeof_fat - DIVROUNDUP((nb_cluster_on_filesystem + 2) * 3, 2));

  /* Fill unused part of FAT: */
  while (bufptr < &buffer[sizeof_fat / sizeof(buffer[0])])
      *bufptr++ = 0xFF;
  return 0;
  }

static unsigned
write_fat_16 (xFILE fout,
		unsigned char	dsk_mediadescriptor,
		unsigned short	nb_cluster_in_file,
		unsigned	nb_cluster_on_filesystem,
		unsigned	cluster_offset,
		unsigned	sizeof_fat,
		unsigned char *const fat_buffer)
  {
//  unsigned short buffer[sizeof_fat / sizeof(buffer[0])], *bufptr = buffer;
  unsigned short *buffer = (unsigned short *)fat_buffer, *bufptr = buffer;

  if ((nb_cluster_on_filesystem + 2) * sizeof(buffer[0]) > sizeof_fat) {
      FILESYS ("Fs: %s: FAT parameters invalid: nb_cluster_on_filesystem %u, sizeof_fat %u\n",
		__FUNCTION__, nb_cluster_on_filesystem, sizeof_fat);
      return 1;
      }

  *bufptr++ = 0xFF00 | dsk_mediadescriptor; /* Little Endian */
  *bufptr++ = 0xFFFF;

  if (nb_cluster_in_file != 0) { /* else no end marker */
      unsigned cpt;

      /* Check size before filling: */
      if (cluster_offset + nb_cluster_in_file > nb_cluster_on_filesystem) {
	  FILESYS ("Fs: %s: cluster_offset %u, nb_cluster_in_file %u, nb_cluster_on_filesystem %u, FAT overflow!\n",
			__FUNCTION__, cluster_offset, nb_cluster_in_file, nb_cluster_on_filesystem);
	  return 1;
	  }
      for (cpt = 0; cpt < cluster_offset; cpt++)
	  *bufptr++ = 0;
      for (cpt = 0; cpt < nb_cluster_in_file - 1; cpt++)
	  *bufptr++ = 2 + cluster_offset + cpt + 1; /* chain next cluster */
      *bufptr++ = 0xFFFF; /* end marker */
      }

  /* Fill end of FAT: */
  while (bufptr < &buffer[nb_cluster_on_filesystem + 2])
      *bufptr++ = 0;

  FILESYS ("Fs: %s: FAT uses %u bytes and contains %u bytes, %u bytes emptied\n",
		__FUNCTION__, (nb_cluster_on_filesystem + 2) * (unsigned)sizeof(buffer[0]),
		sizeof_fat, sizeof_fat - (nb_cluster_on_filesystem + 2) * (unsigned)sizeof(buffer[0]));

  /*
   * I am far to be sure of the minimum difference, and which DOS versions uses
   * the number of sector per FAT (instead of recalculating it).
   * Windows (which version?) uses the number of sector per FAT in the FAT
   * header block, so shall read everything... but the FAT which have been
   * fixed by scandisk/chkdsk (does not update the "wrong" sectors/FAT).
   * Take care of not booting DOS to use FDISK.EXE: to delete a partition
   * you need to type its name, and it will get random characters on its name,
   * and those characters are not on the keyboard...
   * The real problem is that, for DOS, some clusters at the end are reserved
   * and not mapped on the FAT, so it does not need as many sector on the FAT,
   * so the second FAT is not at the right place, and  the root directory is
   * two sectors after what DOS thinks. Note that I hope to have the right
   * number of sectors/clusters for this FAT filesystem size, the one in the
   * FAT header is not used neither...
   *
   * To detect the sector/cluster limits, use checkdisk (without /f) with a lot
   * of filesystem size: it will display the recalculated number of bytes per clusters.
   * To detect the FAT size limits, use scandisk: it will say the media descriptor
   * is not found for the second FAT. Ask it to correct, and do a hexdump of
   * the partition to know where it has put it. So you can say me how many
   * clusters are reserved at end... another way may be to use the --position=
   * parameter of instboot and reduce one by one from endfs value to see at
   * which point scandisk is no more complainning.
   * Note that you should not use FORMAT of DOS because it can generate
   * FAT filesystems with 0 (yes zero) sectors/FAT... they cannot be checked
   * by chkdsk neither...
   */
  if (&buffer[sizeof_fat / sizeof(buffer[0])] - bufptr <= 11)
      WARNING ("generating a FAT16 which is probably not readable on DOS 6 or before\n");

  /* Fill unused part of FAT: */
  while (bufptr < &buffer[sizeof_fat / sizeof(buffer[0])])
      *bufptr++ = 0xFFFF;
  return 0;
  }

static unsigned
write_fat_32 (xFILE fout,
		unsigned char	dsk_mediadescriptor,
		unsigned short	nb_cluster_in_file,
		unsigned	nb_cluster_on_filesystem,
		unsigned	cluster_offset,
		unsigned	sizeof_fat,
		unsigned char *const fat_buffer)
  {
  unsigned *buffer = (unsigned *)fat_buffer, *bufptr = buffer;

  if ((nb_cluster_on_filesystem + 1) * sizeof(buffer[0]) > sizeof_fat) {
      FILESYS ("Fs: %s: FAT parameters invalid: nb_cluster_on_filesystem %u, sizeof_fat %u\n",
		__FUNCTION__, nb_cluster_on_filesystem, sizeof_fat);
      return 1;
      }

  *bufptr++ = 0x0FFFFF00 | dsk_mediadescriptor; /* Little Endian */
#if 0
  *bufptr++ = 0xFFFFFFFF;
  /* first cluster which contains the root directory, only one cluster: */
  *bufptr++ = 0x0FFFFFFF;
#else /* follow what mkdosfs writes there... */
  *bufptr++ = 0x0FFFFFFF;
  /* first cluster which contains the root directory, only one cluster: */
  *bufptr++ = 0x0FFFFFF8;
#endif

  if (nb_cluster_in_file != 0) { /* else no end marker */
      /* Check size before filling: */
      if (cluster_offset + nb_cluster_in_file > nb_cluster_on_filesystem) {
	  FILESYS ("Fs: %s: cluster_offset %u, nb_cluster_in_file %u, nb_cluster_on_filesystem %u, FAT overflow!\n",
			__FUNCTION__, cluster_offset, nb_cluster_in_file, nb_cluster_on_filesystem);
	  return 1;
	  }
      unsigned cpt;
      for (cpt = 1; cpt < cluster_offset; cpt++)	/* root cluster written already */
	  *bufptr++ = 0;
      for (cpt = 0; cpt < nb_cluster_in_file - 1; cpt++)
	  *bufptr++ = 2 + cluster_offset + cpt + 1; /* chain next cluster */
      *bufptr++ = 0x0FFFFFFF; /* end marker */
      }

  /* Fill end of FAT: */
  while (bufptr < &buffer[nb_cluster_on_filesystem + 3])
      *bufptr++ = 0;

  FILESYS ("Fs: %s: FAT uses %u bytes and contains %u bytes, %u bytes emptied\n",
		__FUNCTION__, (nb_cluster_on_filesystem + 1) * (unsigned)sizeof(buffer[0]),
		sizeof_fat, sizeof_fat - (nb_cluster_on_filesystem + 1) * (unsigned)sizeof(buffer[0]));

  /* Fill unused part of FAT: */
  while (bufptr < &buffer[sizeof_fat / sizeof(buffer[0])])
      *bufptr++ = 0xFFFFFFFF;
  return 0;
  }

static inline unsigned
write_fat_root (xFILE fout, unsigned filesize,
		unsigned nb_cluster_on_filesystem, unsigned nb_cluster_in_file, unsigned NbRootDirEntry,
		unsigned fatbit, unsigned nbFAT,
		unsigned char Mediadescriptor, unsigned NbSectorperFAT,
		const union fourbyte_u endnamestr, unsigned cluster_offset,
		unsigned Bytepersector,
		int (*xfct) (xFILE fin, const void *buf, unsigned size),
		unsigned char **fat_buffer)
  {
  /* TODO: check if we could have FAT12/FAT16 with FAT32 system for growing directory */
  directory_t rootdir[NbRootDirEntry];
//  directory_t rootdir[] = { [0 ... NbRootDirEntry-1] = {}};
  const directory_t
     entry1 = {
	.name = {'E', 'L', '-', 'B', 'O', 'O', 'T', '-'},
	.extension = {'V', '2', '8'},  /*VERSION*/
	.attributes = {
	    .readonly		= 0,
	    .hidden		= 0,
	    .system		= 0,
	    .volumelabel	= 1,
	    .subdirectory	= 0,
	    .archive		= 1,
	    .unused		= 0
	    },
	.reserved = {0, 0, 0, 0, 0, 0, 0, 0},
	.cluster_msb = 0,
	.date = {
	    .seconddiv2	= 0,
	    .minute	= 30,
	    .hour	= 8,
	    .day	= 1,	/* 1..31 */
	    .month	= 3,	/* 1..12 */
	    .year	= 19	/* 1980 + x */
	    },
	.cluster = 0,
	.size = 0
	},

     entry2 = {
	.name = {'B', 'O', 'O', 'T', '-',
		endnamestr.array[0],
		endnamestr.array[1],
		endnamestr.array[2]},
	.extension = {'S', 'Y', 'S'},
	.attributes = {
	    .readonly		= 1,
	    .hidden		= 1,
	    .system		= 1,
	    .volumelabel	= 0,
	    .subdirectory	= 0,
	    .archive		= 0,
	    .unused		= 0
	    },
	.reserved = {0, 0, 0, 0, 0, 0, 0, 0},
	.cluster_msb = 0,
	.date = {
	    .seconddiv2 = 25,
	    .minute     = 34,
	    .hour	= 12,
	    .day	= 15,	/* 1..31 */
	    .month      = 3,	/* 1..12 */
	    .year	= 19	/* 1980 + x */
	    },
	.cluster = 2,
	.size = filesize	/* include copy of gujin boot sector at beginning, add MBR copy */
	};
  unsigned short nbfat = nbFAT;

  if (!*fat_buffer) {
      *fat_buffer = malloc (NbSectorperFAT * Bytepersector);
      if (!*fat_buffer) {
	  FILESYS ("Fs: cannot alloc memory of %u bytes for the FAT buffer\n", NbSectorperFAT * Bytepersector);
	  return 1;
	  }
      if ((*((fatbit == 12)? write_fat_12 : ((fatbit == 16) ? write_fat_16 : write_fat_32)))
		(fout, Mediadescriptor, nb_cluster_in_file, nb_cluster_on_filesystem,
		cluster_offset, NbSectorperFAT * Bytepersector, *fat_buffer) != 0) {
	  FILESYS ("Fs: cannot generate the FAT\n");
	  return 1;
	  }
      }
  while (nbfat--)
      if ((*xfct) (fout, *fat_buffer, NbSectorperFAT * Bytepersector)) {
	  FILESYS ("Fs: cannot %s the FAT number %u (drive full?)\n", (xfct == xwrite) ? "write" : "check", nbFAT - nbfat);
	  return 1;
	  }

  memset (rootdir, 0, sizeof (rootdir)); /* variable size array cannot be inited by "= {}" */
  rootdir[0] = entry1;
  if (nb_cluster_in_file != 0) {	/* beforefs/afterfs */
      rootdir[1] = entry2;
      rootdir[1].cluster += cluster_offset;
      }
  if ((*xfct) (fout, rootdir, sizeof(rootdir))) {
      FILESYS ("Fs: cannot write the ROOT directory (drive full?)\n");
      return 1;
      }

  return 0;
  }

/**
 ** Write the output to the file/device:
 **/
static unsigned
write_full_file (xFILE fout, const unsigned char *file, unsigned filesize, unsigned BytePerSector,
		unsigned do_not_report_location,
		unsigned long long fulloffset, unsigned long long baseoffset, unsigned nbsectorinfullfile,
		const char *old_mbr, const char *new_mbr,
		int (*xfct) (xFILE fin, const void *buf, unsigned size), const char *actionstr)
  {
  if (new_mbr == 0) {
      /* This write the complete file _including_ the boot sector as a copy */
      EXTRA ("Extra: %s complete file\n", actionstr);
      if ((*xfct) (fout, file, filesize)) {
	  EXTRA ("%s error for the boot file%s\n",
			actionstr, (xfct == xwrite)? " (drive full or read only?)" : "");
	  if (!do_not_report_location)
	      xcheck_location (fout, fulloffset, baseoffset + filesize / BytePerSector,
				BytePerSector, "position of the error");
	  if (errno) perror("");
	  return 1;
	  }
      }
    else {
      /* The first sector is a copy of the MBR as described in the BEER sector: */
      EXTRA ("Extra: %s copy of MBR.\n", actionstr);
      if ((*xfct) (fout, new_mbr, BytePerSector)) {
	  EXTRA ("Extra: %s error for the first sector%s\n",
			actionstr, (xfct == xwrite)? " (drive full or read only?)" : "");
	  if (errno) perror("");
	  return 1;
	  }
      EXTRA ("Extra: %s total remaining after (file+BytePerSector).\n", actionstr);
      if ((*xfct) (fout, file+BytePerSector, filesize-BytePerSector)) {
	  EXTRA ("%s error for the boot file%s\n",
			actionstr, (xfct == xwrite)? " (drive full?)" : "");
	  if (!do_not_report_location)
	      xcheck_location (fout, fulloffset, baseoffset + filesize / BytePerSector,
				BytePerSector, "position of the error");
	  return 1;
	  }
      }
  if (!do_not_report_location)
      xcheck_location (fout, fulloffset, baseoffset + filesize / BytePerSector,
				BytePerSector, "after the complete file");

  if (old_mbr) {
      EXTRA ("Extra: %s the uninstall boot record\n", actionstr);
      /* write it in this safe place - it is not chained nor loaded until used. */
      if ((*xfct) (fout, old_mbr, BytePerSector)) {
	  ERROR ("%s error for the backup MBR%s\n.",
			actionstr, (xfct == xwrite)? " (drive full?)" : "");
	  return 1;
	  }
      }
    else if (nbsectorinfullfile > filesize / BytePerSector) { /* clear that space in case --full not given as parameter */
      unsigned char blanksector[BytePerSector];
      memset (blanksector, 0, sizeof (blanksector)); /* variable size array cannot be inited by "= {}" */
      if ((*xfct) (fout, blanksector, BytePerSector)) {
	  ERROR ("%s error for the blank backup MBR%s\n.",
			actionstr, (xfct == xwrite)? " (drive full?)" : "");
	  return 1;
	  }
      }
  if (!do_not_report_location)
      xcheck_location (fout, fulloffset, baseoffset + nbsectorinfullfile,
			BytePerSector, "after the uninstall boot record");
  return 0;
  }

static unsigned
write_empty_sectors (xFILE fout, unsigned nb, unsigned BytePerSector, unsigned char filler,
			int (*xfct) (xFILE fin, const void *buf, unsigned size), const char *actionstr)
  {
//#define MULTIPLE_SECTOR		(2 * 18  * 1024 / BytePerSector)
#define MULTIPLE_SECTOR		(32 * 1024 / BytePerSector)
  unsigned char empty_sector[MULTIPLE_SECTOR * BytePerSector];
  unsigned cpt;

  if (nb == 0)
      return 0;
  for (cpt = 0; cpt < sizeof(empty_sector); cpt++)
      empty_sector[cpt] = filler;
  cpt = nb;
  while (cpt >= MULTIPLE_SECTOR) {
      if ((*xfct) (fout, empty_sector, MULTIPLE_SECTOR * BytePerSector)) {
	  EXTRA ("Extra: cannot %s the %u sector (of %u) of this empty area%s\n",
		actionstr, nb - cpt, nb, (xfct == xwrite) ? " (drive full?)" : "");
	  perror ("");
	  return 1;
	  }
      cpt -= MULTIPLE_SECTOR;
      PRINTF ("\rstill has to %s %u sectors ", (xfct == xwrite) ? "write" : "check", cpt);
      fflush (stdout);
      }
  while (cpt--) {
      if ((*xfct) (fout, empty_sector, BytePerSector)) {
	  EXTRA ("Extra: cannot %s the %u sector (of %u) of this empty area%s\n",
		actionstr, nb - cpt, nb, (xfct == xwrite) ? " (drive full?)" : "");
	  perror ("");
	  return 1;
	  }
      PRINTF ("\rstill has to %s %u sectors ", (xfct == xwrite) ? "write" : "check", cpt);
      fflush (stdout);
      }
	   /*still has to check 32 sectors */
  PRINTF ("\r                              \r");
  fflush (stdout);
  return 0;
  }

static unsigned
write_empty_cluster (xFILE fout, unsigned start, unsigned nb, unsigned BytePerSector, unsigned char filler,
			int (*xfct) (xFILE fin, const void *buf, unsigned size), const char *actionstr, long long fulloffset,
			const struct fs_s *fs, unsigned char *const fat_buffer, unsigned *nb_new_bad_cluster)
  {
  const unsigned BytePerCluster = fs->NbSectorPerCluster * BytePerSector;
  unsigned char empty_clusters[(BytePerCluster > 512 && BytePerCluster <= 4096) ? (64 * BytePerCluster) : (8 * BytePerCluster)];
  unsigned cpt, cpt_single = 0;
  unsigned long long clusteroffset = 1ULL * (fs->NbReservedSector + fs->NbFat * fs->NbFat32Sector + fs->NbRootSector) * BytePerSector;

  for (cpt = 0; cpt < sizeof(empty_clusters); cpt++)
      empty_clusters[cpt] = filler;
  xcheck_location (fout, fulloffset, (clusteroffset + start * BytePerCluster)/BytePerSector, BytePerSector, "!! ERROR if different !! write_empty_cluster()");
  if (xset_location (fout, fulloffset, clusteroffset/BytePerSector + start * fs->NbSectorPerCluster, BytePerSector)) {
      ERROR ("Did not acheive to position at beginning of clusters: %llu\n", clusteroffset);
      xclose (fout);
      return 1;
      }
  if (nb)
      PRINTF ("Start %s-check %u clusters at index %u at offset %llu sectors\n", actionstr, nb, start, clusteroffset/BytePerCluster + start);

  for (cpt = 0; cpt < nb; cpt++) {
      unsigned long long oldpos = lseek64(fout, 0, SEEK_CUR);

      xcheck_cluster = start + cpt;
//#define TEST_BADCLUSTER
#ifndef TEST_BADCLUSTER
      if (cpt_single != 0)
	  cpt_single --;
      else if (cpt + sizeof(empty_clusters) / BytePerCluster < nb) {
	  if ((*xfct) (fout, empty_clusters, sizeof(empty_clusters)) == 0) {
	      cpt += sizeof(empty_clusters) / BytePerCluster;
	      PRINTF ("\rstill has to %s %u clusters of %u bytes  ", (xfct == xwrite) ? "write" : "read", nb - cpt, BytePerCluster);
	      continue;
	      }
	    else {
	      if (lseek64(fout, oldpos, SEEK_SET) != oldpos) {
		  ERROR ("Did not acheive to re-position at clusters failure: %llu\n", oldpos);
		  xclose (fout);
		  return 1;
		  }
	      cpt_single = sizeof(empty_clusters) / BytePerCluster - 1;
	      }
	  }
#endif

      if ((*xfct) (fout, empty_clusters, BytePerCluster)) {
//      if ((*xfct) (fout, empty_cluster, sizeof(empty_cluster)) || cpt == 5) { // test invalidate cluster start+5
//      if ((*xfct) (fout, empty_cluster, sizeof(empty_cluster)) || cpt == 6) { // test invalidate cluster start+6
//      if ((*xfct) (fout, empty_cluster, sizeof(empty_cluster)) || cpt == 5 || cpt == 6) { // test invalidate cluster start+5 and start+6
/* Check with TEST_BADCLUSTER defined:
./gujin boot.144 -w -f --position=endfs
/sbin/fsck.vfat boot.144
su -c "mount -t vfat -o loop ~etienne/gujin-2.7/boot.144 /mnt/disk/; yes > /mnt/disk/test ; umount /mnt/disk/"
xxd ~etienne/gujin-2.7/boot.144 | less
look for the unfilled cluster and compare with the gujin install line:
Extra: cannot writing the 0+5 cluster (byte offset 0x4C00..0x4E00) of this empty area, mark cluster bad in FAT
Same for FAT16/32:
./gujin -f --disk=EBIOS:0x80 -g=65536,0 /tmp/disk.img --position=endfs -w
su -c "mount -t vfat -o loop /tmp/disk.img ; yes > /mnt/disk/test ; umount /mnt/disk/"
xxd /tmp/disk.img | less

./gujin -f -d=BIOS:0x00 -g=525264 --fs=FAT:525264,8,0 /tmp/disk.img --position=endfs -w
su -c "mount -t vfat -o loop /tmp/disk.img ; yes > /mnt/disk/test ; umount /mnt/disk/"
xxd /tmp/disk.img | less
*/
	  EXTRA ("Extra: cannot %s the %u+%u cluster (byte offset 0x%llX..0x%llX) of this empty area, mark cluster bad in FAT\n", actionstr, start, cpt, oldpos, (unsigned long long)lseek64(fout, 0, SEEK_CUR));
//	  void *ptr = fat_buffer [((start + cpt + 2) * fs->FatSize) / 8];
	  void *ptr = &fat_buffer[((start + cpt + 2) * (fs->FatSize/4)) / 2];
	  if (fs->FatSize == 32) {
	      unsigned *uptr = (unsigned *)ptr;
	      if (*uptr == 0)
		  *nb_new_bad_cluster += 1;
		else if (*uptr != 0x0FFFFFF7) {
		  ERROR ("reading cluster value 0x%X\n", *uptr);
		  xclose (fout);
		  return 1;
		  }
	      *uptr = 0x0FFFFFF7;
	      }
	  else if (fs->FatSize == 16) {
	      unsigned short *sptr = (unsigned short *)ptr;
	      if (*sptr == 0)
		  *nb_new_bad_cluster += 1;
		else if (*sptr != 0xFFF7) {
		  ERROR ("reading cluster value 0x%X\n", *sptr);
		  xclose (fout);
		  return 1;
		  }
	      *sptr = 0xFFF7;
	      }
	  else if (cpt & 1) {
	      unsigned short *sptr = (unsigned short *)ptr;
	      if ((*sptr >> 4) == 0)
		  *nb_new_bad_cluster += 1;
		else if ((*sptr >> 4) != 0xFF7) {
		  ERROR ("reading cluster value 0x%X\n", (*sptr >> 4));
		  xclose (fout);
		  return 1;
		  }
	      *sptr = (*sptr & 0xF) | 0xFF70;
	      }
	  else {
	      unsigned short *sptr = (unsigned short *)ptr;
	      if ((*sptr & 0x0FFF) == 0)
		  *nb_new_bad_cluster += 1;
		else if ((*sptr & 0x0FFF) != 0xFF7) {
		  ERROR ("reading cluster value 0x%X\n", (*sptr & 0x0FFF));
		  xclose (fout);
		  return 1;
		  }
	      *sptr = (*sptr & 0xF000) | 0x0FF7;
	      }
	  if (xset_location (fout, fulloffset, clusteroffset + (start + cpt + 1ULL) * BytePerCluster, 1)) {
	      ERROR ("Did not acheive to reposition after modifying FAT bad cluster\n");
	      xclose (fout);
	      return 1;
	      }
	  }
      if (cpt % 100 == 0 || nb - cpt < 30 || cpt_single) {
	  PRINTF ("\rstill has to %s %u clusters of %u bytes", (xfct == xwrite) ? "write" : "read", nb - cpt, BytePerCluster);
	  fflush (stdout);
	  }
      }
  xcheck_cluster = 0;
	   /*still has to write 1000000 clusters of 2048 bytes*/
  PRINTF ("\r                                                  \r");
  fflush (stdout);
  return 0;
  }

static unsigned strlen_beer_device(const char *name) /* 0 if not beer device */
  {
  if (*(unsigned *)name != *(unsigned *)"/dev" || name[4] != '/')
      return 0;
  if (   (name[5] == 'h' || name[5] == 's')
      && name[6] == 'd'
      && (name[7] >= 'a' || name[7] <= 'z')
      && name[8] == '0'
      && name[9] == '\0')
      return strlen ("/dev/sda0");
  if (   !strncmp(name, "/dev/mmcblk", strlen("/dev/mmcblk"))
      && name[11] >= '0' && name[11] <= '9' && name[12] == 'p' && name[13] == '0' && name[14] == '\0')
      return strlen ("/dev/mmcblk0p0");
  return 0;
  }

static unsigned
write_output (const char *fout_name, const unsigned char *file, unsigned filesize, unsigned mbr_device_used,
		unsigned deltaseg, unsigned full_image,
		const char *old_mbr, const char *new_mbr,
		unsigned BytePerSector, unsigned confirm,
		enum gujin2_load_t gujin2_load, unsigned char sig0x29touse,
		const union fourbyte_u endnamestr,
		const struct fs_s *fs, const struct misc_s *misc,
		int (*xfct) (xFILE fin, const void *buf, unsigned size))
  {
  xFILE fout;
  const char *actionstr = (xfct == xwrite) ? "writing" : "checking";
  unsigned is_dev_null = 0, is_beer = 0;
  unsigned long long fulloffset;
  unsigned blen = strlen_beer_device (fout_name);
  static unsigned char *iso_mbr = 0, *fat_buffer = 0;
  unsigned nb_new_bad_cluster = 0;
  struct stat fout_stat;

  xcheck_cluster = 0;
  if (blen) {
      static char fout_name_buffer[80];
      fout_name_buffer[--blen] = '\0';
      if (!strncmp (fout_name, "/dev/mmcblk", strlen ("/dev/mmcblk") - 1))
	  fout_name_buffer[--blen] = '\0';
      while (blen--)
	  fout_name_buffer[blen] = fout_name[blen];
      EXTRA ("Extra: recognised BEER device '%s', will write to real device '%s'\n",
		fout_name, fout_name_buffer);
      fout_name = fout_name_buffer;
      is_beer = 1;
      if (misc->output_partition_offset == 0) {
	  EXTRA ("Extra: misc->output_partition_offset == 0, cannot be BEER, "
		"abort before writing anything to %s.\n", fout_name);
	  return 1;
	  }
      }

#if 0 /* target CDall or CDsmallHD does that... */
  if (udev_create_partition) {
      struct stat statbuf;
      if (stat(fout_name, &statbuf) != 0 || !S_ISBLK(statbuf.st_mode)) {
	  ERROR("The output '%s' should be a partition and either do not exists or is a regular file\n", fout_name);
	  ERROR("To automatically create a single partition on a device, the target shall be the whole device, and set --mbr-device=\n");
	  return 1;
	  }
      }
#endif

  if (xfct == xwrite && (misc->infs_file || misc->iso_file)) {
      fout = open (fout_name, O_RDWR);	/* else file might move on the disk */
      if (fout == -1) {
	  EXTRA ("Extra: cannot open O_RDWR output file: \"%s\"\n", fout_name);
	  return 1;
	  }
	}
  else if (xfopen (fout_name, &fout, xfct)) {
      EXTRA ("Extra: cannot open output file: \"%s\"\n", fout_name);
      return 1;
      }

  if (gujin2_load == gujin2_load_dos_com) {
      if (xfct == xwrite && filesize >= 64*1024) /* print only once , i.e. when xfct == xwrite */
	  WARNING ("Generating DOS gujin.com file bigger than 64 Kbytes!\n");
      fulloffset = 0;
      }
    else if (gujin2_load == gujin2_load_PIC || gujin2_load == gujin2_load_CDROM_noemul || misc->infs_file) {
      fulloffset = 0;
      }
    else if (gujin2_load == gujin2_load_dos_exe) { /* no header for .com files */
      struct dos_exe_str {
	  unsigned char  exe_signature[2];
	  unsigned short image_length_mod_512;
	  unsigned short image_length_divroundup_512;
	  unsigned short relocation_nb;
	  unsigned short header_size_div_16;
	  unsigned short min_added_memory_div_16;
	  unsigned short max_added_memory_div_16;
	  unsigned short stack_seg_offset;
	  unsigned short init_sp;
	  unsigned short checksum; /* ~SIGMA(words16bit) */
	  unsigned short init_ip;
	  unsigned short code_seg_offset;
	  unsigned short relocation_offset;
	  unsigned short overlay_number; /* 28 bytes up to (including) there */
//#define LONG_EXE_HEADER /* some OS do not load it else ? */
#ifdef LONG_EXE_HEADER
	  unsigned	 relocation_table[(512 - 28) / 4];
#else
	  unsigned	 relocation_table[1];  /* to have a total of 32 bytes */
#endif
	  } __attribute__ ((packed)) dos_exe = {
	.exe_signature = {'M', 'Z'},
	/* The file created is rounded up to 512 bytes due to checksums: */
	.image_length_mod_512 = 0,
	/* I do not understand _why_ I need +1 here on a real DOS6.20: */
	.image_length_divroundup_512 = filesize/512 + 1,
	.relocation_nb = 0,
	.header_size_div_16 = sizeof (struct dos_exe_str) / 16,
	/*
	 * min_added_memory_div_16: FIXME if BIG_MALLOC
	 * or ZLIB_BIG_MALLOC are defined, now set to 128K total
	 * It should be, without BIG_MALLOC nor ZLIB_BIG_MALLOC:
	 * (64*1024 - instboot_info->sbss) / 16, / * size of BSS * /
	 * Note also that VCPI switch uses up to 2 * 4096 + 4095 bytes
	 * over the stack.
	 */
	.min_added_memory_div_16 = 128 * 1024 / 16,
//	.max_added_memory_div_16 = 0xFFFF,
	.max_added_memory_div_16 = 5 * 64 * 1024 / 16, // to enable DOS_exec_pgm()
	.stack_seg_offset = deltaseg,
	.init_sp = 0,
	.checksum = 0, /* no need to set it */
	.init_ip = 0, /* instboot_info->code_start_adr, */
	.code_seg_offset = 0, /* %cs = program load address */
	.relocation_offset = offsetof (struct dos_exe_str, relocation_table),
	.overlay_number = 0
	};

      fulloffset = 0;
      if ((*xfct) (fout, &dos_exe, sizeof (dos_exe))) {
	  EXTRA ("%s error for the EXE descriptor block (drive full or Read Only?)\n",
			actionstr);
	  xclose (fout);
	  return 1;
	  }
      }
    else {
      unsigned long long start_sector;

      /* Check if the output is a disk or a partition, i.e. shall we offset
	   everything by install->misc.output_partition_offset: */
      if (fstat (xfileno(fout), &fout_stat) != 0) {
	  if (errno == ENOENT) {
	      EXTRA ("Extra: target file '%s' still does not exist, do not confirm,"
			"offset by %llu sectors of %u bytes.\n",
			fout_name, misc->output_partition_offset, BytePerSector);
	      confirm = 0;
	      fulloffset = misc->output_partition_offset * BytePerSector;
	      }
	    else {
	      EXTRA ("Extra: cannot stat existing target file/device '%s', abort\n", fout_name);
	      xclose (fout);
	      return 1;
	      }
	  }
#if defined (MAJOR) && defined (MINOR) /* FIXME: how to recognise /dev/null on Fedora 3? */
	else if ( MAJOR(fout_stat.st_rdev) == 1 && MINOR(fout_stat.st_rdev) == 3 && S_ISCHR(fout_stat.st_mode)) {
	  WARNING ("output device is '/dev/null', do not print ftell() results, do not confirm before %s, "
			"do not offset by misc->output_partition_offset.\n", actionstr);
	  fulloffset = 0;
	  is_dev_null = 1;
	  confirm = 0;
	  if (xfct != xwrite)
	      WARNING ("output device is '/dev/null', maybe you should add '--check=writeonly'.\n");
	  }
#endif
	else if (S_ISREG(fout_stat.st_mode)) {
	  EXTRA ("Extra: output '%s' is a regular file, do not confirm, "
			"offset by %llu sectors of %u bytes.\n",
			fout_name, misc->output_partition_offset, BytePerSector);
	  fulloffset = misc->output_partition_offset * BytePerSector;
	  confirm = 0;	/* floppyimage or gujin.com or gujin.exe */
	  }
	else if (strncmp (fout_name, "/dev/fd0", sizeof ("/dev/fd0") - 1) == 0) {
	  EXTRA ("Extra: do not confirm for the target '/dev/fd0*' (string compare), "
			"do not offset by misc->output_partition_offset.\n");
	  fulloffset = 0;
	  confirm = 0;
	  }
	/* NOTE: Do not use geo.total_sector here, it can be invalid for C/H/S only drives.
			Just use the start marker: */
	else if (get_device_start_sector (xfileno(fout), &start_sector, 0) != 0) {
	  ERROR ("error getting geometry of existing device %s, abort.\n", fout_name);
	  xclose (fout);
	  return 1;
	  }
	else if (start_sector != 0 && start_sector == misc->output_partition_offset) {
	  fulloffset = 0;
	  EXTRA ("Extra: output device is the right partition, do not offset every writes.\n");
	  }
	else if (start_sector == 0) {
	  fulloffset = misc->output_partition_offset * BytePerSector;
	  EXTRA ("Extra: output device is the whole disk, do offset every writes by "
			"misc->output_partition_offset %llu * %u = %llu.\n",
			misc->output_partition_offset, BytePerSector, fulloffset);
	  /* TIME TO CHECK IF PARTITION TABLE IS EMPTY: */
	  if (fulloffset == 0 && xfct == xwrite && confirm) {
	      FILE *checkfile;
	      bootafter_t bootafter;
	      if ((checkfile = fopen (fout_name, "r")) == NULL) {
		  EXTRA ("Extra: cannot open output file to read the possible partition table on '%s'\n", fout_name);
		  xclose (fout);
		  return 1;
		  }
	      if (fseeko(checkfile, 512 - sizeof(bootafter), SEEK_SET) != 0
		|| ftello (checkfile) != 512 - sizeof(bootafter)) {
		  EXTRA ("Extra: cannot position to read the possible partition table on '%s'\n", fout_name);
		  fclose (checkfile);
		  xclose (fout);
		  return 1;
		  }
	      if (fread (&bootafter, 1, sizeof(bootafter), checkfile) != sizeof(bootafter)) {
		  EXTRA ("Extra: cannot read the possible partition table of the device '%s'\n", fout_name);
		  fclose (checkfile);
		  xclose (fout);
		  return 1;
		  }
	      if (fseeko(checkfile, 0, SEEK_END) != 0) {
		  EXTRA ("Extra: cannot position to end to read the possible partition table on '%s'\n", fout_name);
		  fclose (checkfile);
		  xclose (fout);
		  return 1;
		  }
	      off_t nbtotalsector = ftello(checkfile) / BytePerSector;
	      if (fclose (checkfile) == EOF) {
		  EXTRA ("Extra: cannot close the device after reading the possible partition table: '%s'\n", fout_name);
		  xclose (fout);
		  return 1;
		  }
	      unsigned nb_valid_partition = 0, cpt_partition;
	      for (cpt_partition = 0; cpt_partition < 4; cpt_partition++)
		  if (   (bootafter.bootsect_partition[cpt_partition].indicator & 0x7F) == 0
		      && bootafter.bootsect_partition[cpt_partition].nb_sector_before < nbtotalsector
		      && bootafter.bootsect_partition[cpt_partition].nb_sector < nbtotalsector
		      && bootafter.bootsect_partition[cpt_partition].nb_sector_before
			 + bootafter.bootsect_partition[cpt_partition].nb_sector <= nbtotalsector
		      && bootafter.bootsect_partition[cpt_partition].nb_sector != 0)
		      nb_valid_partition++;
	      if (bootafter.Signature0xAA55 == 0xAA55 && nb_valid_partition > 0) {
		  ERROR("You seem to have a valid and non empty partition table on %s, as a\n"
			"safety measure the complete disk will not be transformed into a\n"
			"superfloppy, use --force flag to overwrite, delete all partitions\n"
			"with your usual partition table editor, or use:\n"
			"dd if=/dev/zero of=%s bs=512 count=1\n\n", fout_name, fout_name);
		  xclose (fout);
		  return 1;
		  }
	      }
	  }
	else {
	  EXTRA ("Extra: output device is not the whole disk nor the right partition (start at %llu)?.\n",
			misc->output_partition_offset);
	  xclose (fout);
	  return 1;
	  }

      if (fulloffset != 0) {
	  DISPLAY_WARNING_32BIT_OFF_T()
	  if (xset_location (fout, fulloffset, 0, BytePerSector)) { /* not depending on full_image flag */
	      xcheck_location (fout, fulloffset, 0, BytePerSector, "error on first positionement");
	      ERROR ("Did not acheive to set location to fulloffset = %llu\n", fulloffset);
	      xclose (fout);
	      return 1;
	      }
#ifdef FSEEKO_DOUBLECHECK
	  /* For me it is a serious bug - if off_t is 64 bits, maybe the system cannot manage fseeko()/ftello()
	   * of more than 32 bits - but the 64 bits version of fseeko()/ftello() should _at_least_
	   * return an error if (pos >> 32 != 0) ... Fortunately this hard disk was empty.
	   * exact output message:
Extra: output device is the whole disk, do offset every writes by misc->output_partition_offset 12529416 * 512.
Location: will fseeko: 0 * 512 + 12529416 * 512, i.e. 6415060992
Error: After setting successfully position to 6415060992 i.e. 0x17E5E1000 with fseeko(), I read position 2120093696 i.e. 0x7E5E1000 with ftello()
 I will not write anything on this disk...
Error: writing the output.
	   */
	  {
	  long long readback = ftello (fout);
	  if (!is_dev_null && readback != fulloffset) {
	      ERROR ("After setting successfully position to %lld i.e. 0x%llX with fseeko(), "
			"I read position %lld i.e. 0x%llX with ftello()\n"
			" I will not write anything on this disk...\n",
			fulloffset, fulloffset, readback, readback);
	      xclose (fout);
	      return 1;
	      }
	  }
#endif
	  }

      if (xfct == xwrite)
	  PRINTF ("Clusters start at sector 0x%llX + 0x%X = 0x%llX (alignement)\n",
		misc->output_partition_offset, misc->firstDataSectorInPartition,
		misc->output_partition_offset + misc->firstDataSectorInPartition);
      if (confirm) {
	  unsigned long long fssize = fs->NbFsSector * BytePerSector / 1024;
	  char unit[3] = "Kb";
	  if (fssize > 10 * 1024 * 1024) {
	      fssize /= 1024 * 1024;
	      unit[0] = 'G';
	      }
	  else if (fssize > 10 * 1024) {
	      fssize /= 1024;
	      unit[0] = 'M';
	      }
	  PRINTF ("You asked to create a new FAT%u filesystem of %llu %s on %s'%s',\n"
		      "and so ERASE completely this %s, confirm [NO/yes]: ",
		      fs->FatSize, fssize, unit, is_beer? (primary_GPT_sector? "StartBeer device of " : "B.E.E.R. device of "): "", fout_name,
		      (start_sector || fulloffset)? "partition" : "disk");
	  if (getchar() != 'y' || getchar() != 'e' || getchar() != 's') {
	      PRINTF ("aborted.\n");
	      xclose (fout);
	      return 1;
	      }
	  }

      if (xfct == xwrite)
	  EXTRA ("Extra: Begin writing to the device.\n");
	else
	  EXTRA ("Extra: Begin extensive verification of what has been written.\n");

      if (misc->iso_file) {	/* modifying ISO file by FAT with big initial GAP */
	  if (xset_location (fout, 0, 0, BytePerSector)) {
	      ERROR ("Did not acheive to set location to beginning of file\n");
	      xclose (fout);
	      return 1;
	      }
	  if (iso_mbr == 0 && xfct == xwrite) { /* checkonly will not work */
	      iso_mbr = malloc (512);
	      if (iso_mbr == 0 || xread (fout, iso_mbr, 512)) {
		  ERROR ("Did not acheive to read the 'MBR' of the ISO\n");
		  xclose (fout);
		  return 1;
		  }
	      if (xset_location (fout, 0, 0, BytePerSector)) {
		  ERROR ("Did not acheive to reset location to beginning of file\n");
		  xclose (fout);
		  return 1;
		  }
	      }
	  /* For that sector, replace "FAT32", "FAT16" or "FAT12" by "FAT  ": */
	  union {
	      bootbefore_t isofat16;
	      bootbefore_FAT32_t isofat32;
	      unsigned char iso[BytePerSector];
	      } isombr;
	  memcpy (isombr.iso, file, BytePerSector);
	  if (isombr.isofat16.part2.FileSysName[0] == 'F' && isombr.isofat16.part2.FileSysName[1] == 'A' && isombr.isofat16.part2.FileSysName[2] == 'T') {
	      isombr.isofat16.part2.FileSysName[3] = isombr.isofat16.part2.FileSysName[4] = ' ';
	      isombr.isofat16.part2.Signaturebyte0x29 = sig0x29touse; /* Some PC do not boot with 0x29 */
	      }
	  else if (isombr.isofat32.part2.FileSysName[0] == 'F' && isombr.isofat32.part2.FileSysName[1] == 'A' && isombr.isofat32.part2.FileSysName[2] == 'T') {
	      isombr.isofat32.part2.FileSysName[3] = isombr.isofat32.part2.FileSysName[4] = ' ';
	      isombr.isofat32.part2.Signaturebyte0x29 = sig0x29touse; /* Some PC do not boot with 0x29 */
	      }
	  /* and set the size of the partition/superfloppy: */
	  isombr.isofat16.part1.NbTotalsector2 = (isombr.isofat16.part1.NbTotalsector?: isombr.isofat16.part1.NbTotalsector2) + fulloffset/BytePerSector;
	  isombr.isofat16.part1.NbTotalsector = 0;
	  if ((*xfct) (fout, isombr.iso, BytePerSector)) {	/* add other sectors for FAT32 ? */
	      EXTRA ("Extra: %s error for the first ISO sector%s\n",
			actionstr, (xfct == xwrite)? " (ISO image file read only?)" : "");
	      if (errno) perror("");
	      xclose (fout);
	      return 1;
	      }
	  if (xset_location (fout, fulloffset, 0, BytePerSector)) {
	      xcheck_location (fout, fulloffset, 0, BytePerSector, "error on first re-positionement");
	      ERROR ("Did not acheive to set location to fulloffset = %llu\n", fulloffset);
	      xclose (fout);
	      return 1;
	      }
	  }

      EXTRA ("Extra: %s boot record as first sector.\n", actionstr);
      if ((*xfct) (fout, file, BytePerSector)) {
	  EXTRA ("Extra: %s error for the first sector%s\n",
			actionstr, (xfct == xwrite)? " (drive full or read only?)" : "");
	  if (errno) perror("");
	  xclose (fout);
	  return 1;
	  }

      if (misc->begin_file_sector_nr == 0) { /* positioncmd == "beforefs" */
	  if (write_full_file (fout, file, filesize, BytePerSector,
			is_dev_null || gujin2_loaded_externally (gujin2_load),
			fulloffset,
			misc->begin_file_sector_nr + 1, /* i.e. 1, after the real MBR just written, FAT32? */
			misc->nbsectorinfullfile,
			old_mbr, new_mbr, xfct, actionstr) != 0) {
	      xclose (fout);
	      return 1;
	      }
	  }

      if (fs->FatSize == 32) {
	  /* we need to put the 2 extra sectors, at 1 and at 6 */
	  struct {
		unsigned fsinfo_signature_0x41615252;
		unsigned reserved1[480/4];
		unsigned fsinfo_signature_0x61417272;
		unsigned nb_free_cluster;	/* -1 if unknown */
		unsigned most_recently_allocated_cluster;	/* -1 if unknown */
		unsigned char reserved2[14];
		unsigned short signature_0xAA55;
		//unsigned char padding[BytePerSector - 512];
		unsigned char padding[4096 - 512];
		} __attribute__((packed)) FSinfo = {
		fsinfo_signature_0x41615252: 0x41615252,
		fsinfo_signature_0x61417272: 0x61417272,
		nb_free_cluster: misc->nb_cluster_on_filesystem - misc->nb_cluster_in_file - 1,
		most_recently_allocated_cluster: misc->nb_cluster_at_begin + misc->nb_cluster_in_file + 2,
		signature_0xAA55: 0xAA55
		};
	  EXTRA ("Extra: %s FAT32 FS info sector.\n", actionstr);
	  if ((*xfct) (fout, (char *)&FSinfo, BytePerSector)) {
	      EXTRA ("Extra: %s error for FAT32 FS info sector%s\n",
			actionstr, (xfct == xwrite)? " (drive full or read only?)" : "");
	      if (errno) perror("");
	      xclose (fout);
	      return 1;
	      }
	  /* We try not to have holes here to be able to rename the file as *.iso and use it in /boot
	  if (!S_ISREG (fout_stat.st_mode) && full_image) { / * Do not bother, create device image file with holes */
	  if (full_image) {
	      EXTRA ("Extra: %s %u sectors in between FS info and MBR copy\n", actionstr, 6 - 2);
	      if (write_empty_sectors (fout, 6 - 2, BytePerSector, 0, xfct, actionstr)) {
		  EXTRA ("Extra: %s error for the 6 - 2 sectors (drive full?)\n", actionstr);
		  xclose (fout);
		  return 1;
		  }
	      }
	    else {
	      EXTRA ("Extra: seeking to 6 sectors i.e. MBR copy\n");
	      if (xset_location (fout, fulloffset, 6, BytePerSector)) {
		  ERROR ("Did not acheive to set location to write MBR copy\n");
		  xclose (fout);
		  return 1;
		  }
	      }
	  EXTRA ("Extra: %s FAT32 MBR copy sector.\n", actionstr);
	  if ((*xfct) (fout, file, BytePerSector)) {
	      EXTRA ("Extra: %s error for the FAT32 MBR copy sector%s\n",
			actionstr, (xfct == xwrite)? " (image file read only?)" : "");
	      if (errno) perror("");
	      xclose (fout);
	      return 1;
	      }
	  /* We try not to have holes here to be able to rename the file as *.iso and use it in /boot
	  if (!S_ISREG (fout_stat.st_mode) && full_image) { / * Do not bother, create device image file with holes */
	  if ((!S_ISREG (fout_stat.st_mode)  || fs->NbReservedSector < 256) && full_image) {
	      unsigned nb = fs->NbReservedSector - 6;

	      if (nb >= 1) {
		  EXTRA ("Extra: %s %u sectors as NbReservedSector\n", actionstr, nb - 1);
		  if (write_empty_sectors (fout, nb - 1, BytePerSector, 0, xfct, actionstr)) {
		      EXTRA ("Extra: %s error for the %u reserved sector(s) (drive full?)\n", actionstr, fs->NbReservedSector);
//		      xclose (fout);
//		      return 1;
		      PRINTF ("WARNING: some of the %u reserved sectors cannot be %s!\n", fs->NbReservedSector, actionstr);
		      if (xset_location (fout, fulloffset, fs->NbReservedSector, BytePerSector)) {
			  ERROR ("Did not acheive to set location to NbReservedSector\n");
			  xclose (fout);
			  return 1;
			  }
		      }
		  }
	      }
	    else {
	      EXTRA ("Extra: seeking to %u sectors i.e. NbReservedSector\n", fs->NbReservedSector);
	      if (xset_location (fout, fulloffset, fs->NbReservedSector, BytePerSector)) {
		  ERROR ("Did not acheive to set location to NbReservedSector\n");
		  xclose (fout);
		  return 1;
		  }
	      }
	  }
	/* We try not to have holes here to be able to rename the file as *.iso and use it in /boot
	else if (!S_ISREG (fout_stat.st_mode) && full_image) {  / * Do not bother, create device image file with holes */
	else if ((!S_ISREG (fout_stat.st_mode)  || fs->NbReservedSector < 256) && full_image) {
	  unsigned nb = fs->NbReservedSector;
	  if (misc->begin_file_sector_nr == 0)
	      nb -= filesize / BytePerSector;	/* positioncmd == "beforefs" */

	  if (nb >= 1) {
	      EXTRA ("Extra: %s %u sectors as NbReservedSector\n", actionstr, nb - 1);
	      if (write_empty_sectors (fout, nb - 1, BytePerSector, 0, xfct, actionstr)) {
		  EXTRA ("Extra: %s error for the %u reserved sector(s) (drive full?)\n", actionstr, fs->NbReservedSector);
//		  xclose (fout);
//		  return 1;
		  PRINTF ("WARNING: some of the %u reserved sectors cannot be %s!\n", fs->NbReservedSector, actionstr);
		  if (xset_location (fout, fulloffset, fs->NbReservedSector, BytePerSector)) {
		      ERROR ("Did not acheive to set location to NbReservedSector\n");
		      xclose (fout);
		      return 1;
		      }
		  }
	      }
	  }
	else if (is_dev_null) {
	  EXTRA ("Extra: should seek to (%llu + %u) * %u, but do not do it for /dev/null\n",
			fulloffset/BytePerSector, fs->NbReservedSector, BytePerSector);
	  }
	else { /* if (fs->NbReservedSector != 1) */
	  if (xset_location (fout, fulloffset, fs->NbReservedSector, BytePerSector)) {
	      ERROR ("Did not acheive to set location to NbReservedSector\n");
	      xclose (fout);
	      return 1;
	      }
	  }

      if (!is_dev_null)
	  xcheck_location (fout, fulloffset, fs->NbReservedSector, BytePerSector, "after FAT Reserved Sector");

      if (write_fat_root (fout, filesize + (mbr_device_used? BytePerSector : 0),
			misc->nb_cluster_on_filesystem, misc->nb_cluster_in_file,
			((fs->NbRootSector ?: fs->NbSectorPerCluster) * BytePerSector) / sizeof (directory_t),
			fs->FatSize, fs->NbFat,
			fs->MediaDescriptor, fs->NbFat32Sector,
			endnamestr, misc->nb_cluster_at_begin,
			BytePerSector, xfct, &fat_buffer) != 0) {
	  xclose (fout);
	  return 1;
	  }

      if (!is_dev_null)
	  xcheck_location (fout, fulloffset, misc->firstDataSectorInPartition,
				BytePerSector, "after FAT(s) + root directory");

      if (
#ifndef TEST_BADCLUSTER
		!S_ISREG (fout_stat.st_mode) &&
#endif
		full_image && fs->FatSize == 32) { /* Do not bother, create device image file with holes */
	  /* do not overwrite the first cluster being root directory: */
	  EXTRA ("Extra: Seeking over the root directory cluster\n");
	  if (xset_location (fout, fulloffset, misc->firstDataSectorInPartition + 1 * fs->NbSectorPerCluster, BytePerSector)) {
	      xclose (fout);
	      return 1;
	      }
	  if (write_empty_cluster (fout, 1, misc->nb_cluster_at_begin - 1, BytePerSector, 0, xfct, actionstr, fulloffset, fs, fat_buffer, &nb_new_bad_cluster)) {
	      EXTRA ("Extra: cannot %s the %u clusters at begin, and cannot disable them in FAT\n", actionstr, misc->nb_cluster_at_begin - 1);
	      xclose (fout);
	      return 1;
	      }
	  }
	else if (
#ifndef TEST_BADCLUSTER
		!S_ISREG (fout_stat.st_mode) &&
#endif
		full_image) { /* Do not bother, create device image file with holes */
	  if (write_empty_cluster (fout, 0, misc->nb_cluster_at_begin, BytePerSector, 0, xfct, actionstr, fulloffset, fs, fat_buffer, &nb_new_bad_cluster)) {
	      EXTRA ("Extra: cannot %s the %u clusters at begin, and cannot disable them in FAT\n", actionstr, misc->nb_cluster_at_begin);
	      xclose (fout);
	      return 1;
	      }
	  }
	else if (is_dev_null) {
	  EXTRA ("Extra: should seek to (%llu + %u) * %u, but do not do it for /dev/null\n",
			fulloffset/BytePerSector, misc->firstDataSectorInPartition
			  + misc->nb_cluster_at_begin * fs->NbSectorPerCluster, BytePerSector);
	  }
	else {
	  if (xset_location (fout, fulloffset, misc->firstDataSectorInPartition
			  + misc->nb_cluster_at_begin * fs->NbSectorPerCluster, BytePerSector)) {
	      xclose (fout);
	      return 1;
	      }
	  }
      }

  if (   gujin2_loaded_externally (gujin2_load) || gujin2_load == gujin2_load_CDROM_noemul || misc->infs_file
	/* positioncmd != "beforefs"      &&    positioncmd != "afterfs" */
      || (misc->begin_file_sector_nr != 0 && misc->begin_file_sector_nr < fs->NbFsSector)) {
      if (misc->infs_file) {
	  if (xset_location (fout, fulloffset, misc->begin_file_sector_nr, BytePerSector)) {
	      xclose (fout);
	      return 1;
	      }
	  }
      if (!gujin2_loaded_by_dos (gujin2_load) && !is_dev_null && (misc->nb_cluster_at_begin != 0 || misc->holy_filesystem))
	  xcheck_location (fout, fulloffset, misc->begin_file_sector_nr, BytePerSector,
			"after skiping/[writing or checking] (--full) beginning to offset file (--position)");
      if (write_full_file (fout, file, filesize, BytePerSector,
			is_dev_null || gujin2_loaded_externally (gujin2_load),
			fulloffset, misc->begin_file_sector_nr, misc->nbsectorinfullfile,
			old_mbr, new_mbr, xfct, actionstr) != 0) {
	  xclose (fout);
	  return 1;
	  }
      }

  if (!gujin2_loaded_externally (gujin2_load) && fs->NbFsSector != 0) {
      if (write_empty_sectors (fout, misc->nb_empty_sector_in_cluster, BytePerSector, 0, xfct, actionstr)) {
	  EXTRA ("%s error for the ending %u sectors of end-of-cluster filler (drive full?)\n",
			actionstr, misc->nb_empty_sector_in_cluster);
	  xclose (fout);
	  return 1;
	  }

      if (misc->nb_empty_sector_in_cluster != 0 && !is_dev_null) {
	  xcheck_location (fout, fulloffset,
		misc->begin_file_sector_nr + misc->nb_cluster_in_file * fs->NbSectorPerCluster, BytePerSector,
		"after end-of-cluster filler just after file");
	  }

#ifndef TEST_BADCLUSTER
      if (full_image && S_ISREG (fout_stat.st_mode)) { /* Do not bother, create device image file with holes */
	  if (xfct == xwrite && ftruncate (fout, fulloffset + fs->NbFsSector * BytePerSector) != 0) {
	      perror ("ftruncate");
	      ERROR ("Cannot increase the file to its final size\n");
	      xclose (fout);
	      return 1;
	      }
	  xset_location (fout, fulloffset, fs->NbFsSector, BytePerSector);
	  xcheck_location (fout, fulloffset, fs->NbFsSector, BytePerSector, "after the end of filesystem hole");
	  }
	else
#endif
	if (full_image) {
	  if (write_empty_cluster (fout, misc->nb_cluster_at_begin + misc->nb_cluster_in_file,
					misc->nb_cluster_at_end, BytePerSector, 0, xfct, actionstr, fulloffset, fs, fat_buffer, &nb_new_bad_cluster)) {
	      EXTRA ("Extra: cannot %s the %u clusters at end, and cannot disable them in FAT\n", actionstr, misc->nb_cluster_at_end);
	      xclose (fout);
	      return 1;
	      }

	  if (misc->nb_sector_at_end != 0) {
	      if (!is_dev_null)
		  xcheck_location (fout, fulloffset,
			misc->firstDataSectorInPartition + misc->nb_cluster_on_filesystem * fs->NbSectorPerCluster,
			BytePerSector, "after empty clusters to fill the filesystem (--position)");
	      EXTRA ("Extra: %u sectors not accessible for filesystem at end\n", misc->nb_sector_at_end);
	      if (write_empty_sectors (fout, misc->nb_sector_at_end, BytePerSector, 0xFF, xfct, actionstr)) {
		  EXTRA ("Extra: %s error for the %u ending filesystem sector filler (drive full?)\n", actionstr, misc->nb_sector_at_end);
//		  xclose (fout);
//		  return 1;
		  PRINTF ("WARNING: some of the %u sectors at end of device (after clusters) cannot be %s!\n", misc->nb_sector_at_end, actionstr);
		  }
	      if (!is_dev_null)
		  xcheck_location (fout, fulloffset, fs->NbFsSector, BytePerSector,
			"after sectors after the clusters to fill the filesystem");
	      }
	  }
	else
	  /* TODO: Still read all clusters, and invalidate the bad clusters in the FAT ? */;

      if (misc->begin_file_sector_nr >= fs->NbFsSector) { /* positioncmd == "afterfs" */
	  if (is_dev_null) {
	      EXTRA ("Extra: should seek to (%llu + %llu) * %u, but do not do it for /dev/null\n",
			fulloffset/BytePerSector, fs->NbFsSector, BytePerSector);
	      }
	    else if (xset_location (fout, fulloffset, fs->NbFsSector, BytePerSector)) {
	      xclose (fout);
	      return 1;
	      }
	  if (write_full_file (fout, file, filesize, BytePerSector, is_dev_null,
			fulloffset, misc->begin_file_sector_nr, misc->nbsectorinfullfile,
			old_mbr, new_mbr, xfct, actionstr) != 0) {
	      xclose (fout);
	      return 1;
	      }
	  }

      if (iso_mbr) {
	  /* We would better have the iso file size be a multiple of 2048 bytes, but we need
		that uninstall bootsector as the last 512 bytes whatsoever */
	  unsigned curnbsect = lseek64 (fout, 0, SEEK_CUR) / 512;
	  unsigned wantednbsect = (curnbsect + 4) & ~3;
	  unsigned hastoadd = wantednbsect - curnbsect - 1;
	  EXTRA ("ISO treat: current filesize nb512sect: %u, wants %u, adds %u\n", curnbsect, wantednbsect, hastoadd);
	  if (hastoadd != 0) {
	      if (write_empty_sectors (fout, hastoadd, BytePerSector, 0, xfct, actionstr)) {
		  EXTRA ("%s error for the ending %u sectors of end-of-cluster ISO filler (drive full?)\n",
				actionstr, hastoadd);
		  xclose (fout);
		  return 1;
		  }
	      }
	  if (xfct (fout, iso_mbr, 512)) {
	      EXTRA ("Extra: cannot write ISO 'MBR' (drive full?)\n");
	      free (iso_mbr);
	      iso_mbr = 0;
	      xclose (fout);
	      return 1;
	      }
	  }

      if (nb_new_bad_cluster) {
	  unsigned short nbfat = fs->NbFat;
	  int xfout = fout;
	  if (xfct != xwrite && (xfout = open (fout_name, O_WRONLY | O_SYNC)) == -1) {
	      ERROR ("Cannot re-open file %s for writing FAT with invalid clusters\n", fout_name);
	      xclose (fout);
	      return 1;
	      }
	  PRINTF ("nb_new_bad_cluster = %u, updating FATs\n", nb_new_bad_cluster);
	  xset_location (xfout, fulloffset, fs->NbReservedSector, BytePerSector);
	  while (nbfat--)
	      if (xwrite (xfout, fat_buffer, fs->NbFat32Sector * BytePerSector)) {
		  EXTRA ("Extra: cannot rewrite the FAT number %u\n", fs->NbFat - nbfat);
		  return 1;
		  }
	  if (xfct != xwrite && close (xfout) == -1) {
	      ERROR ("Cannot re-close file %s for writing FAT with invalid clusters\n", fout_name);
	      xclose (fout);
	      return 1;
	      }
	  }

      if (xfct == xwrite)
	  PRINTF ("Finishing write (flushing buffers) and then checking...\n");
      }

  if (iso_mbr && xfct != xwrite) {
      free (iso_mbr); /* if writeonly or error returned, memory leak */
      iso_mbr = 0;
      }
  if (fat_buffer && xfct != xwrite) {
      free (fat_buffer); /* if writeonly or error returned, memory leak */
      fat_buffer = 0;
      }

  if (xclose (fout)) {
      EXTRA ("cannot close output file: %s\n", fout_name);
      return 1;
      }

  if (misc->infs_file && xfct != xwrite) {
      int fout_desc;
      unsigned long block_index, start_hole = 0, current_block = 0;
      const unsigned nb_block = misc->holy_filesystem + (filesize + misc->holy_unit_size - 1) / misc->holy_unit_size;

      /* warning: ignoring return value of "chown" */
      if (chown (fout_name, 0, 0)); /* set owned by root for E2/3/4FS, NFTS, ... using special mount options */
      errno = 0;
      if (chmod (fout_name, S_ISVTX) != 0) { /* sticky bit, no read/write/execute by anybody, Gujin do not check */
	  /* that fails on FAT, treat that case now: */
	  unsigned attr;
	  errno = 0;
	  if ((fout_desc = open(fout_name, O_RDONLY)) != -1 && ioctl(fout_desc, FAT_IOCTL_GET_ATTRIBUTES, &attr) != -1) {
	      attr = ATTR_RO | ATTR_HIDDEN | ATTR_SYS;
	      if (ioctl(fout_desc, FAT_IOCTL_SET_ATTRIBUTES, &attr) == -1)
		  perror ("Cannot set POSIX nor FAT boot file attributes, IGNORED.");
	      }
	    else {
	      perror ("Warning cannot set boot file attributes");
	      if (fout_desc != -1)
		  chmod (fout_name, S_IRUSR | S_IRGRP | S_IROTH | S_IXUSR | S_IXGRP | S_IXOTH); /* at least disable write */
	      }
	  errno = 0;
	  if (fout_desc != -1)
	      if (close(fout_desc) == -1)
		  perror ("Closing boot file after setting its attributes");
	  }
      if ((fout_desc = open (fout_name, O_RDONLY |O_EXCL)) == -1) {	// We are root, we can open it whatever permissions.
	  perror (fout_name);
	  ERROR("Cannot re-open file '%s' after complete treatment to check it has not moved.\n", fout_name);
	  return 1;
	  }
      for (block_index = 0; block_index < nb_block; block_index++) {
	  unsigned long block = block_index;
	  if (ioctl (fout_desc, FIBMAP, &block)) {
	      perror ("FIBMAP");
	      }
	  if (block == 0) {
	      start_hole++;
	      continue;
	      }
	  if (start_hole != misc->holy_filesystem || (current_block != 0 && block != current_block + 1)) {
	      ERROR ("File '%s' (after complete treatment) has moved on disk (start_hole %lu, current_block %lu)\n", fout_name, start_hole, current_block);
	      close (fout_desc);
	      return 1;
	      }
	  if (current_block == 0 && block != misc->infs_position) {
	      ERROR("File '%s' (after complete treatment) has changed position on disk, start block %lu instead of %u\n", fout_name, block, misc->infs_position);
	      close (fout_desc);
	      return 1;
	      }
	  current_block = block;
	  }
      close (fout_desc);
      EXTRA ("Extra: File '%s' (after complete treatment) has not moved on disk: start hole of %lu blocks, start block at %u\n", fout_name, start_hole, misc->infs_position);
      }
  return 0;
  }

static inline unsigned
read_mbr (const char *mbr_device, char *mbr_buffer, unsigned Bytepersector)
  {
  FILE *fin;
  unsigned long long start_sector;

  if ((fin = fopen (mbr_device, "r")) == NULL) {
      EXTRA ("Extra: cannot open output file to read the MBR/partition table on '%s'\n", mbr_device);
      return 1;
      }
  if (fread (mbr_buffer, 1, Bytepersector, fin) != Bytepersector) {
      EXTRA ("Extra: cannot read the MBR of the MBR device '%s'\n", mbr_device);
      fclose (fin);
      return 1;
      }
  if (get_device_start_sector (fileno(fin), &start_sector, 0) != 0) {
      EXTRA ("Extra: Cannot get disk geometry for MBR copy.\n");
      fclose (fin);
      return 1;
      }
  if (start_sector != 0) {
      ERROR ("Extra: the MBR sector is not at zero, --mbr-device= shall be followed "
			"by a disk and not a partition\n");
      fclose (fin);
      return 1;
      }
  if (fclose (fin) == EOF) {
      EXTRA ("Extra: cannot close the MBR device after reading: '%s'\n", mbr_device);
      return 1;
      }
  return 0;
  }

static unsigned
write_mbr (const char *mbr_device, const char *new_mbr, unsigned Bytepersector,
		int (*xfct) (xFILE fin, const void *buf, unsigned size), unsigned reprobe_partition)
  {
  xFILE fout;
  int fout_desc;

  /* xfopenmodify because it is maybe a device image file and we do not want to trunk it: */
  if (xfopenmodify (mbr_device, &fout, xfct)) {
      EXTRA ("Extra: cannot open output device/file to insert the MBR on \"%s\"\n", mbr_device);
      return 1;
      }
  if ((*xfct) (fout, new_mbr, Bytepersector)) {
      EXTRA ("Extra: cannot write the MBR on '%s'\n", mbr_device);
      xclose (fout);
      return 1;
      }
  fout_desc = dup(xfileno(fout));
  if (xclose (fout)) {
      EXTRA ("Extra: cannot close the device/file after inserting its MBR on %s\n", mbr_device);
      return 1;
      }
  if (fout_desc < 0)
      EXTRA ("Extra: failed getting file descriptor of %s\n", mbr_device);
    else if (!reprobe_partition)
      EXTRA ("Extra: do not reprobe partition table of %s (infs)\n", mbr_device);
    else if (ioctl (fout_desc, BLKRRPART, 0) >= 0)
      EXTRA ("Extra: partition table of %s re-read successfully\n", mbr_device);
    else {
      EXTRA ("Extra: failed re-reading partition table of %s\n", mbr_device);
//      perror (mbr_device);	// /dev/hda: Device or resource busy
      }
  close (fout_desc);
  return 0;
  }

static inline bootafter_t create_partition_table (unsigned index,
	unsigned char bootable_flag, unsigned char system_indicator,
	unsigned newpart_nb_sector_before, unsigned newpart_nb_sector,
	unsigned init_heads, unsigned init_sectpertrack)
  {
  bootafter_t bootafter = {};
  unsigned cste = init_heads * init_sectpertrack, cylinder, remaind;

  bootafter.Signature0xAA55 = 0xAA55;
  bootafter.bootsect_partition[index].indicator = bootable_flag;
  bootafter.bootsect_partition[index].system_indicator = system_indicator;
  bootafter.bootsect_partition[index].nb_sector_before = newpart_nb_sector_before;
  bootafter.bootsect_partition[index].nb_sector = newpart_nb_sector;

  cylinder = newpart_nb_sector_before / cste;
  remaind = newpart_nb_sector_before % cste;
  bootafter.bootsect_partition[index].begin_head = remaind / init_sectpertrack;
  bootafter.bootsect_partition[index].begin_sector = remaind % init_sectpertrack + 1;
  if (cylinder >= 1024) {
      cylinder = 0x3FF;
//    bootafter.bootsect_partition[index].begin_head = init_heads;
//    bootafter.bootsect_partition[index].begin_sector = init_sectpertrack;
      bootafter.bootsect_partition[index].begin_head = 0;
      bootafter.bootsect_partition[index].begin_sector = 1;
      }
  bootafter.bootsect_partition[index].begin_cylinder_lsb = cylinder & 0xFF;
  bootafter.bootsect_partition[index].begin_cylinder_msb = cylinder >> 8;

  cylinder = (newpart_nb_sector_before + newpart_nb_sector - 1) / cste,
  remaind = (newpart_nb_sector_before + newpart_nb_sector - 1) % cste;
  bootafter.bootsect_partition[index].end_head = remaind / init_sectpertrack;
  bootafter.bootsect_partition[index].end_sector = remaind % init_sectpertrack + 1;
  if (cylinder >= 1024) {
      cylinder = 0x3FF;
      bootafter.bootsect_partition[index].end_head = init_heads - 1;
      bootafter.bootsect_partition[index].end_sector = init_sectpertrack;
      }
  bootafter.bootsect_partition[index].end_cylinder_lsb = cylinder & 0xFF;
  bootafter.bootsect_partition[index].end_cylinder_msb = cylinder >> 8;

  return bootafter;
  }

/*
 * PhysicaldriveNb_ifnew: 0xFF; unknown 0xFF?, best guess 0x80, to be adjusted by BIOS software
 * For simulated HD on CDROM, shall read 0xFF if the BIOS number is autodetected at run time,
 * else it really has to be the BIOS number
 */
static unsigned
read_and_update_mbr (char *mbr_buffer, unsigned char *file, const char *mbr_device,
		unsigned Bytepersector, const char **new_mbr, const char **old_mbr,
		unsigned init_heads, unsigned init_sectpertrack, enum gujin2_load_t gujin2_load,
		unsigned long long init_total_sector, unsigned char sig0x29touse,
		unsigned newpart_nb_sector_before, unsigned newpart_nb_sector,
		unsigned char partition_type, const char *partition_index,
		unsigned char PhysicaldriveNb_ifnew, unsigned char Mediadescriptor_ifnew,
		instboot_info_t *instboot_info, unsigned char mbr_codesig_bug)
  {
  struct install_bootsector_s {
      union {
	  struct {
	      unsigned char	jmpcode[2];
	      unsigned char	EBIOSaccess;
	      unsigned char	String[8];	/* "Gujin1" blank padded ! */
	      unsigned short	Bytepersector;
	      unsigned char	reserved1;	/* Sectorpercluster */
	      unsigned short	Reservedsector;	/* nb sectors before FAT */
	      unsigned char	reserved3;	/* NbFAT */
	      unsigned short	reserved4;	/* NbRootdirentry = 0 for FAT32 */
	      unsigned short	NbTotalsector;	/* = 0 if >= 0x10000 or FAT32 */
	      unsigned char	Mediadescriptor;/* If 0xBB, that is a MBR, total sector = (NbTotalsector << 32) + NbTotalsector2 */
	      unsigned short	reserved5;	/* NbSectorperFAT = 0 for FAT32 */
	      /* DOS 3+ : */
	      unsigned short	NbSectorpertrack;
	      unsigned short	NbHead;
	      unsigned		NbHiddensector;	/* nb of sectors before this partition */
	      /* DOS 4+ : */
	      unsigned		NbTotalsector2;

	      unsigned char	PhysicaldriveNb;
	      unsigned char	reserved6;	/* FATreserved */
	      unsigned char	Signaturebyte0x29;
	      unsigned		VolumeserialNb;
	      unsigned char	Volumelabel[11];	/* "Gujin2" blank padded ! */
	      unsigned char	FileSysName[8];		/* blank padded ! */
	      } __attribute__ ((packed)) field_FAT16;
	  struct {
	      unsigned char	jmpcode[2];
	      unsigned char	EBIOSaccess;
	      unsigned char	String[8];
	      unsigned short	Bytepersector;
	      unsigned char	reserved1;
	      unsigned short	Reservedsector;
	      unsigned char	reserved3;
	      unsigned short	reserved4;
	      unsigned short	NbTotalsector;
	      unsigned char	Mediadescriptor;
	      unsigned short	reserved5;
	      unsigned short	NbSectorpertrack;
	      unsigned short	NbHead;
	      unsigned		NbHiddensector;
	      unsigned		NbTotalsector2;
	      unsigned		NbSectorperFAT2;
	      struct {
		  unsigned short	active_fat	: 4;
		  unsigned short	reserved1	: 3;
		  unsigned short	one_fat_active	: 1;
		  unsigned short	reserved2	: 8;
		  } __attribute__ ((packed)) flags;
	      unsigned short	version;
	      unsigned		root_cluster;
	      unsigned short	FSinfosector;
	      unsigned short	save_bootrecord;
	      unsigned char	reserved[12];
	      unsigned char	PhysicaldriveNb;
	      unsigned char	reserved6;
	      unsigned char	Signaturebyte0x29; /* : this structure just written for this offset! */
	      unsigned		VolumeserialNb;
	      unsigned char	Volumelabel[11];
	      unsigned char	FileSysName[8];
	      } __attribute__ ((packed)) field_FAT32;
	  unsigned char all[0x1B6];
	  } area;
      unsigned short to_0x1B8;
      bootafter_t    bootafter;
      } __attribute__ ((packed)) *install_bootsector = (struct install_bootsector_s *)mbr_buffer;
  unsigned cpt;
  struct stat statbuf;
  /* The area we keep constant when re-installing Gujin. Sure the partition table also */
  unsigned short Bytepersect;
  unsigned short Reservedsector;
  unsigned short NbTotalsector;
  unsigned char	 Mediadescriptor;
  unsigned short NbSectorpertrack;
  unsigned short NbHead;
  unsigned	 NbHiddensector;
  unsigned	 NbTotalsector2;
  unsigned char	 PhysicaldriveNb;
  unsigned	 VolumeserialNb;

//  if (sizeof(struct install_bootsector_s) != Bytepersector) {
  if (sizeof(struct install_bootsector_s) != 512) {
      EXTRA ("ERROR: size of install MBR sector: %u\n", (unsigned)sizeof(struct install_bootsector_s));
      return 1;
      }

  cpt = 0xFFFFEEEE;
  if (stat (mbr_device, &statbuf) != 0 || S_ISREG(statbuf.st_mode)
	|| (   (cpt = read_mbr (mbr_device, mbr_buffer, Bytepersector)) == 0
	   &&  install_bootsector->bootafter.Signature0xAA55 != 0xAA55)) {
      if (cpt == 0xFFFFEEEE)
	  EXTRA ("Extra: Read MBR of a simple file: create a dummy partition table and disable uninstall\n");
	else /* Maybe completely blank disk, i.e. full of 0 - shall still install safely */
	  EXTRA ("Extra: the old MBR sector read is not bootable, 0xAA55 signature: 0x%X, no uninstall MBR\n",
			    ((unsigned short *)mbr_buffer)[255]);
      *old_mbr = 0;
      *install_bootsector = (struct install_bootsector_s) {};
      if (  newpart_nb_sector_before != 0
	 && newpart_nb_sector <= init_total_sector
	 && init_heads != 0 && init_sectpertrack != 0
	 && partition_index != 0 && *partition_index >= '1' && *partition_index <= '4') {
	  EXTRA ("Extra: CREATING partition table for partition %u, start %u length %u (%u heads, %u sect/track)\n",
		*partition_index - '0', newpart_nb_sector_before, newpart_nb_sector, init_heads, init_sectpertrack);
	  install_bootsector->bootafter = create_partition_table (*partition_index - '1', bootable,
			partition_type,	newpart_nb_sector_before, newpart_nb_sector, init_heads, init_sectpertrack);
	  udev_create_partition = &install_bootsector->bootafter;
	  }
      }
    else {
      if (cpt != 0) {
	  EXTRA ("ERROR: reading MBR sector\n");
	  return 1;
	  }
      if (partition_index != 0) {
	  unsigned nb_valid_partition = 0, cpt_partition;
	  for (cpt_partition = 0; cpt_partition < 4; cpt_partition++)
	      if (   (install_bootsector->bootafter.bootsect_partition[cpt_partition].indicator & 0x7F) == 0
		&& install_bootsector->bootafter.bootsect_partition[cpt_partition].nb_sector_before < init_total_sector
		&& install_bootsector->bootafter.bootsect_partition[cpt_partition].nb_sector < init_total_sector
		&& install_bootsector->bootafter.bootsect_partition[cpt_partition].nb_sector_before
			+ install_bootsector->bootafter.bootsect_partition[cpt_partition].nb_sector <= init_total_sector
		&& install_bootsector->bootafter.bootsect_partition[cpt_partition].nb_sector != 0)
		  nb_valid_partition++;

	  if (partition_index != 0 && nb_valid_partition > 0) {
	      ERROR("You seem to have a valid and non empty partition table on %s, as a\n"
		"safety measure this installer will not overwrite the old partition table by itself,\n"
		"check that '%s' is the right target, backup any data you want to keep and then\n"
		"you have to remove all partitions manually with your usual partition table editor, or use:\n"
		"   dd if=/dev/zero of=%s bs=%u count=%u\n\n",
		mbr_device, mbr_device, mbr_device, Bytepersector, init_sectpertrack);
	      puts ("If you wanted to use the current partition table, use a partition as\n"
		" the target device, i.e. 2nd parameter finish by a digit different of zero.");
	      puts ("If you wanted to erase and ignore the whole partition to create\n"
		" a floppy kind of image, remove the --mbr or --mbr-device= argument.");
	      return 1;
	      }
	  if (newpart_nb_sector_before != 0
		 && newpart_nb_sector <= init_total_sector
		 && init_heads != 0 && init_sectpertrack != 0
		 && partition_index != 0 && *partition_index >= '1' && *partition_index <= '4') {
	      EXTRA ("Extra: CREATING partition table for partition %u, start %u length %u (%u heads, %u sect/track)\n",
		*partition_index - '0', newpart_nb_sector_before, newpart_nb_sector, init_heads, init_sectpertrack);
	      install_bootsector->bootafter = create_partition_table (*partition_index - '1', bootable,
			partition_type,	newpart_nb_sector_before, newpart_nb_sector, init_heads, init_sectpertrack);
	      udev_create_partition = &install_bootsector->bootafter;
	      }
	  }
      *old_mbr = mbr_buffer + Bytepersector;
      }

  /* Want to install infs on a superfloppy? we must then keep most of the format for the filesystem header! */
  if (partition_index == 0 && newpart_nb_sector_before == 0) {
      unsigned long startbyte, endbyte;

      if (install_bootsector->area.field_FAT16.Signaturebyte0x29 == 0x29 || install_bootsector->area.field_FAT16.Signaturebyte0x29 == 0x28) {
	  startbyte = (unsigned long)&install_bootsector->area.field_FAT16.Bytepersector - (unsigned long)install_bootsector;
	  endbyte = (unsigned long)&install_bootsector->area.field_FAT16.Volumelabel[0] - (unsigned long)install_bootsector;
	  }
	else if (install_bootsector->area.field_FAT32.Signaturebyte0x29 == 0x29) {
	  startbyte = (unsigned long)&install_bootsector->area.field_FAT32.Bytepersector - (unsigned long)install_bootsector;
	  endbyte = (unsigned long)&install_bootsector->area.field_FAT32.Volumelabel[0] - (unsigned long)install_bootsector;
	  }
	else {
	  MBR ("Mbr: InFs on superfloppy: can do FAT16/FAT32, could do Ext*fs - cannot find FAT 0x29 marker.\n");
	  return 1;
	  }
      if (install_bootsector->area.field_FAT32.Signaturebyte0x29 == 0x29
	&& gujin2_load != gujin2_load_ebios && gujin2_load != gujin2_load_bios) {
	  MBR ("Mbr: InFs on superfloppy FAT32 can only be done for BIOS or EBIOS bootchain.\n");
	  return 1;
	  }
      *new_mbr = mbr_buffer;
      memcpy (mbr_buffer + Bytepersector, mbr_buffer, Bytepersector); /* initialise the uninstall MBR if *old_mbr != 0 */
      if (Bytepersector > 512)
	  memcpy (mbr_buffer + 512, &file[512], Bytepersector - 512);
      MBR ("Mbr: InFs on FAT superfloppy, copying FAT header from %lu to %lu, Bytepersector %u.\n", startbyte, endbyte, install_bootsector->area.field_FAT16.Bytepersector);
      for (cpt = 0; cpt < startbyte; cpt++)
	  install_bootsector->area.all[cpt] = file[cpt];
      for (cpt = endbyte; cpt < install_bootsector->area.field_FAT16.Bytepersector; cpt++)
	  install_bootsector->area.all[cpt] = file[cpt];
      if (install_bootsector->area.field_FAT32.Signaturebyte0x29 == 0x29) {
	  /* Adjust boot.asm " 13a: be 2b 00 $boot2_msg,%si" : */
	  *(unsigned short *)&install_bootsector->area.all[0x13B] += sizeof(FAT32_bootsect_t);
	  /* Copy "Gujin2" out of the way at 0x47: Volume label for FAT32 header */
	  for (cpt = 0; cpt < sizeof (install_bootsector->area.field_FAT32.Volumelabel); cpt++)
	      install_bootsector->area.field_FAT32.Volumelabel[cpt] = ((struct install_bootsector_s *)file)->area.field_FAT16.Volumelabel[cpt];
	  }
      if (gujin2_load == gujin2_load_ebios
	&& install_bootsector->area.field_FAT16.jmpcode[0] == 0xEB
	&& install_bootsector->area.field_FAT16.EBIOSaccess == 0x90)
	  install_bootsector->area.field_FAT16.EBIOSaccess = 0xE; /* i.e. EBIOS OK */
      checksum_boot1 ((unsigned char *)mbr_buffer, instboot_info, (boot1param_t *)(mbr_buffer + instboot_info->boot1param_adr));
// TODO: also copy the MBR to the MBR copy for FAT32
      return 0;
      }

  /*
   * We want to copy the current Partition Boot Record to the Master Boot Record.
   * This Master Boot Record may not be on the same Hard Drive as the Partition
   * Boot Record - it is an unusual case where the MBR of one disk will load
   * Gujin from another disk.
   * So we need to adjust the C/H/S description (we check the 0x29 signature
   * just in case: it has to be there because this is always Gujin bootrecord).
   * We should not touch those values if the 0x29 signature is already found,
   * because the user probably wants to keep the programmed values.
   * Abort if we see the 'Signaturebyte0x29' for FAT32 because I do not
   * know how you can get it... something is probably wrong, and Gujin cannot
   * be installed on such a boot sector (not enough space) and you do not
   * usually get a FAT32 on an un-partitioned device like a floppy.
   */
  if (   install_bootsector->area.field_FAT32.Signaturebyte0x29 == 0x29
      && install_bootsector->area.field_FAT16.Signaturebyte0x29 != 0x29
      && install_bootsector->area.field_FAT16.Signaturebyte0x29 != 0x28) {
      MBR ("Mbr: The FAT32 0x29 signature has been found on the MBR to use: '%s', abort.\n", mbr_device);
      return 1;
      }

  *new_mbr = mbr_buffer;
  memcpy (mbr_buffer + Bytepersector, mbr_buffer, Bytepersector);
  if (Bytepersector > 512)
      memcpy (mbr_buffer + 512, &file[512], Bytepersector - 512);

  /* We need to keep the old VolumeserialNb because some Internet rumor say that WinXP and Win2000
     want to find the old one or zero to re-initialise it. */
//  VolumeserialNb = ((struct install_bootsector_s *)file)->area.field_FAT16.VolumeserialNb;
    VolumeserialNb = install_bootsector->area.field_FAT16.VolumeserialNb;
//  if (VolumeserialNb != 0)
      MBR ("Mbr: keep old VolumeserialNb = 0x%X\n", VolumeserialNb);
//    else {
//      VolumeserialNb = rand();
//    MBR ("Mbr: old bootsector VolumeserialNb was null, set to a random one = 0x%X\n", VolumeserialNb);
//    }

  if ((install_bootsector->area.field_FAT16.Signaturebyte0x29 == 0x29 || install_bootsector->area.field_FAT16.Signaturebyte0x29 == 0x28)
	&& install_bootsector->area.field_FAT16.Bytepersector == 1 << (ffs(install_bootsector->area.field_FAT16.Bytepersector) - 1)) {
      MBR ("Mbr: FAT16 0x29 signature found on '%s', keep default C/H/S info block.\n", mbr_device);
      Bytepersect       = install_bootsector->area.field_FAT16.Bytepersector;
      Reservedsector    = install_bootsector->area.field_FAT16.Reservedsector;
      NbTotalsector     = install_bootsector->area.field_FAT16.NbTotalsector;
      Mediadescriptor   = install_bootsector->area.field_FAT16.Mediadescriptor;
      NbSectorpertrack  = install_bootsector->area.field_FAT16.NbSectorpertrack;
      NbHead            = install_bootsector->area.field_FAT16.NbHead;
      NbHiddensector    = install_bootsector->area.field_FAT16.NbHiddensector;
      NbTotalsector2    = install_bootsector->area.field_FAT16.NbTotalsector2;
      PhysicaldriveNb   = install_bootsector->area.field_FAT16.PhysicaldriveNb;
      MBR ("Mbr: first bytes: 0x%X, 0x%X, 0x%X\n"
	   "Mbr: InfoBlock: %u bytes/sector, %u short total sectors, %u total sector, "
		"media descriptor: 0x%X, physical drive nb 0x%X\n"
		"InfoBlock: %u sectors/track, %u heads, %u hidden sectors, volume serial nb 0x%X.\n",
		file[0], file[1], file[2],
		Bytepersect, NbTotalsector, NbTotalsector2, Mediadescriptor, PhysicaldriveNb,
		NbSectorpertrack, NbHead, NbHiddensector, VolumeserialNb);
      }
    else {
      /* FIXME: short NbHead would be 0 if LBA possible */
      NbHead = init_heads;
      /* FIXME: short NbSectorpertrack would be the max sector requests on IDE if LBA */
      NbSectorpertrack = init_sectpertrack;

      MBR ("Mbr: FAT16 0x29 signature NOT found on '%s', initialise info block.\n", mbr_device);
      Bytepersect       = Bytepersector;
      NbHiddensector  = 0;
//      Reservedsector = 1 + ((newpart_nb_sector_before > 0xFFFFU)? 0 : newpart_nb_sector_before); /* points to start of FAT */
      Reservedsector = 0;
      if (Bytepersector != 512 || init_total_sector > 0xFFFFFFLLU) { /* HD more than 32 Gb get 0xBB */
	  Mediadescriptor = 0xBB; /* If 0xBB (BootBlock), that is a MBR,
					total sector = (NbTotalsector << 32) + NbTotalsector2 */
	  NbTotalsector   = (unsigned short)(init_total_sector >> 32);
	  NbTotalsector2  = (unsigned)init_total_sector;
	  /* NbHiddensector  =	 non null for SOARB IDE feature - to be implemented */
	  }
	else if (init_total_sector <= 0xFFFF) {
	  NbTotalsector   = (unsigned short)init_total_sector;
	  NbTotalsector2  = 0;
	  Mediadescriptor = 0xF0; /* 1.44Mb, 1.68Mb, 2.88Mb with partitions */
	  }
	else {
	  NbTotalsector   = 0;
	  NbTotalsector2  = (unsigned)init_total_sector;
	  Mediadescriptor = Mediadescriptor_ifnew;
	  }
      PhysicaldriveNb   = PhysicaldriveNb_ifnew;
      if (file[0] == 0xEB)
	  file[2] = (gujin2_load == gujin2_load_ebios)? 0x0E : 0x90;
      MBR ("Mbr: first bytes: 0x%X, 0x%X, 0x%X\n"
	   "Mbr: InfoBlock: %u bytes/sector, Reservedsector %u, %llu sectors, heads %u, sect/track %u\n"
	   "Mbr: media descriptor: 0x%X, physical drive nb 0x%X, volume serial nb 0x%X\n",
		file[0], file[1], file[2],
		Bytepersect, Reservedsector, init_total_sector, init_heads, init_sectpertrack,
		Mediadescriptor, PhysicaldriveNb, VolumeserialNb);
      }

  for (cpt = 0; cpt < sizeof (install_bootsector->area.all); cpt++)
      install_bootsector->area.all[cpt] = file[cpt];

/* Windows Disk Signature bytes: The four bytes from offsets 0x1B8 through 0x1BB in the
   MBR are called the Windows 2000/XP Disk Signature or NT Drive Serial Number */

  MBR ("Mbr: has initialised new MBR up to 0x%X; WinNT/XP marker 0x%X kept.\n",
	(unsigned)sizeof (install_bootsector->area.all), install_bootsector->bootafter.WindowsNTmarker);
  for (cpt = 0; cpt < sizeof (install_bootsector->area.field_FAT16.FileSysName); cpt++)
      install_bootsector->area.field_FAT16.FileSysName[cpt] = "MBR     "[cpt];
  install_bootsector->area.field_FAT16.reserved1 = 0;
  install_bootsector->area.field_FAT16.reserved3 = 0;
  install_bootsector->area.field_FAT16.reserved4 = 1; /* NbRootdirentry SHALL NOT BE NULL! */
  install_bootsector->area.field_FAT16.reserved5 = 1; /* NbSectorperFAT SHALL NOT BE NULL! */
  install_bootsector->area.field_FAT16.reserved6 = 0;
  install_bootsector->bootafter.Signature0xAA55 = 0xAA55;

  install_bootsector->area.field_FAT16.Bytepersector = Bytepersect;
  install_bootsector->area.field_FAT16.Reservedsector = Reservedsector;
  install_bootsector->area.field_FAT16.NbTotalsector = NbTotalsector;
  install_bootsector->area.field_FAT16.Mediadescriptor = Mediadescriptor;
  install_bootsector->area.field_FAT16.NbSectorpertrack = NbSectorpertrack;
  install_bootsector->area.field_FAT16.NbHead = NbHead;
  install_bootsector->area.field_FAT16.NbHiddensector = NbHiddensector;
  install_bootsector->area.field_FAT16.NbTotalsector2 = NbTotalsector2;
  install_bootsector->area.field_FAT16.PhysicaldriveNb = PhysicaldriveNb;
//  install_bootsector->area.field_FAT16.Signaturebyte0x29 = 0x29;
/* This computer will not boot if Signaturebyte0x29 == 0x29
Proprieta'		Valore
Venditore del BIOS	Phoenix Technologies, LTD
Versione del BIOS	6.00 PG
Versione del Firmware	111.101
Data del BIOS		10/09/2003
Dimensioni del BIOS	256 KB
Segmentoin avvio dal BIOS	E000h
Versione DMI		2.3
   But anyway Gujin do not work perfectly with USB BIOS emulation neither
   WinNT is said to accept 0x28 as synonym for 0x29
Unfortunately my no-name dual-core portable PC do not like to see
something else than 0x29 there - it seems it cannot read the USB disk...
*/
  install_bootsector->area.field_FAT16.Signaturebyte0x29 = sig0x29touse;
  install_bootsector->area.field_FAT16.VolumeserialNb = VolumeserialNb;

  if (gujin2_load == gujin2_load_ebios
	&& install_bootsector->area.field_FAT16.jmpcode[0] == 0xEB
	&& install_bootsector->area.field_FAT16.EBIOSaccess == 0x90)
      install_bootsector->area.field_FAT16.EBIOSaccess = 0xE; /* i.e. EBIOS OK */

  /* Search 0xBEFC1F03 in disk.c for info */
  if (mbr_codesig_bug == 3) {
      EXTRA("Extra: Inserting HPCompaq bugfix destroying MBR structure\n");
      *(unsigned *)&install_bootsector->area.all[0x0A] = 0xBEFC1F03;
      }
  return 0;
  }

static unsigned
write_beer (struct beersect_s *beersect, const char *beer_device, unsigned Bytepersector,
		struct misc_s *misc, unsigned lba_support, unsigned device_index,
		int (*xfct) (xFILE fin, const void *buf, unsigned size))
  {
  xFILE fout;
  union {
	unsigned char  bytes[Bytepersector];
	unsigned short words[Bytepersector/2];
	struct beer_s data;
	} beer;
  unsigned char empty_sector[Bytepersector];
  unsigned cpt;

  /* Note that to come here, the output device has to be the complete disk and
   * not a partition, because the BEER is out of all partitions.
   * So we can use safely output_partition_offset.
   * CORRECTION: If we have a problem with 32 bits ftell()/fpos(), it may be
   * interresting to create a temporary partition starting at output_partition_offset
   * and finishing at the disk end - then we can still install a BEER system, and
   * remove the temporary partition after install.
   */

  for (cpt = 0; cpt < sizeof (beer.bytes); cpt++)
      beer.bytes[cpt] = 0;
  for (cpt = 0; cpt < sizeof(empty_sector); cpt++)
      empty_sector[cpt] = 0;

/* Now, misc->reported_disk_size   is misc->formatted_disk_size     rounded down to a number of pages,
	misc->restricted_disk_size is misc->output_partition_offset rounded down to a number of pages. */
  beer.data = (struct beer_s) {
	.signature_0xBEEF	= 0xBEEF,
	.beer_size		= sizeof (struct beer_s),
	.attr = {
		.reported_geometry_valid	= 0,
		.formated_geometry_valid	= 0,
		.service_dir_present		= 0,
		.device_support_lba		= lba_support,
		.config_time_stamp_valid	= 0, // FIXME
		.use_RABCA			= 1,
		.generated_record		= 0,
		.read_only			= 0,
		.reserved			= 0
		},
	.reported = {
		.cylinder		= 0,
		.head			= 0,
		.sector			= 0,
		.bytepersector		= 0,
		.sectorperdevice	= misc->reported_disk_size,
		},
	.formatted = {
		.cylinder		= 0,
		.head			= 0,
		.sector			= 0,
		.bytepersector		= 0,
		.sectorperdevice	= misc->formatted_disk_size,
		},
	.bcd_year			= 0, // FIXME
	.linear_day			= 0,
	.config_time_stamp		= 0,
	.reserved_1			= 0,
	.device_index			= device_index,
	.HostProtectedArea_start	= misc->restricted_disk_size - 1,
	.ReservedAreaBootCodeAddress	= misc->output_partition_offset + misc->begin_file_sector_nr,
	.BEERdir_nb_entry		= 0,
	.BEERdir_length			= 0,
	.reserved_2			= 0,
	.revision_3			= 3, /* revision 3 document used */
	.device_name			= "BEER device",
	.checksum			= 0,
//	.struct beer_directory_s	directory[0];
	};

  /* AM I COMPLETELY HOOSED HERE OR THERE IS A MAJOR COMPILER BUG ????
   * GCC: 2.95.3, Binutils: GNU ld version 2.13.1, beer.data.checksum always zero */
#if 0
  for (beer.data.checksum = 0, cpt = 0; cpt < sizeof(struct beer_s) / 2; cpt++)
      beer.data.checksum -= beer.words[cpt];
printf ("BEER size %u, checksum: 0x%X, cpt %u\n", sizeof(struct beer_s), beer.data.checksum, cpt);
#else
  {
  unsigned short checksum = 0, i = sizeof(beer.data) / 2;
  while (i--)
      checksum -= beer.words[i];
  beer.data.checksum = checksum;
//printf ("BEER size %u, checksum: 0x%X, cpt %u\n", sizeof(struct beer_s), beer.data.checksum, cpt);
  }
#endif

  if (beersect->BeerNbSector == 0 || beersect->BeerNbSector > misc->formatted_disk_size / 10) {
      EXTRA ("Extra: beersect->BeerNbSector seriously out of range: %u\n", beersect->BeerNbSector);
      return 1;
      }
  if (beersect->BeerPosition >= beersect->BeerNbSector) {
      EXTRA ("Extra: beersect->{BeerPosition,BeerNbSector}={%u, %u} invalid combination.\n",
		beersect->BeerPosition, beersect->BeerNbSector);
      return 1;
      }
  if (xfopen (beer_device, &fout, xfct)) {
      EXTRA ("Extra: cannot open output file to write the BEER on \"%s\"\n", beer_device);
      return 1;
      }
#if 0 // FIXME: does not work if reported hdgeo.total_sector is invalid !! ALSO NOT FOR primary_GPT_sector
  {
  struct get_hard_disk_geometry_s hdgeo;
  if (get_hard_disk_geometry (xfileno(fout), &hdgeo) != 0) { // take care forced_Heads, forced_SectPerTrack!
      xclose (fout);
      EXTRA ("Extra: Cannot get disk geometry for BEER.\n");
      return 1;
      }
  if (hdgeo.start_sector == 0) {
      EXTRA ("Extra: output device for BEER is the complete disk\n");
      }
    else {
      long long len;

      /* We will SEEK_END, so the two device shall finnish at the same place */
      if ((len = GET_FD_LENGTH(xfileno(fout))) == -1LL) {
	  EXTRA ("Extra: cannot get length for the BEER partition output\n");
	  return 1;
	  }
	else if (len / Bytepersector != hdgeo.total_sector - hdgeo.start_sector) {
	  EXTRA ("Extra: BEER output is a partition not finishing on the same sector as the whole disk, "
			"hdgeo.start_sector + len / Bytepersector = %llu, hdgeo.total_sector = %llu\n",
			hdgeo.start_sector + len / Bytepersector, hdgeo.total_sector);
	  return 1;
	  }
      EXTRA ("Extra: BEER output is a partition finishing on the same sector as the whole disk, i.e. %llu\n",
			hdgeo.total_sector);
      }
    /* fseek (fout, -Bytepersector * beersect->BeerNbSector, SEEK_END) */
  if (xset_location (fout, 0, hdgeo.total_sector - hdgeo.start_sector - beersect->BeerNbSector, Bytepersector)) {
      xcheck_location (fout, 0, hdgeo.total_sector - hdgeo.start_sector - beersect->BeerNbSector,
		      Bytepersector, "first BEER positionement");
      ERROR ("Did not acheive to set location to fulloffset = %llu sectors\n",
		hdgeo.total_sector - hdgeo.start_sector - beersect->BeerNbSector);
      xclose (fout);
      return 1;
      }
  }
#else
  {
  unsigned long long start_sector;
  if (get_device_start_sector (xfileno(fout), &start_sector, 0) != 0) {
      ERROR ("the BEER device target has to be a complete disk, not a partition, for now\n");
      return 1;
      }
  if (xset_location (fout, 0, misc->beer_sector_offset, Bytepersector)) {
      xcheck_location (fout, 0, misc->beer_sector_offset, Bytepersector,
				"first BEER positionement");
      ERROR ("Did not acheive to set location to fulloffset = %llu sectors\n",
		misc->beer_sector_offset);
      xclose (fout);
      return 1;
      }
  }
#endif
  for (cpt = 0; cpt < beersect->BeerPosition; cpt++) {
      if ((*xfct) (fout, empty_sector, sizeof (empty_sector))) {
	  EXTRA ("Extra: cannot write first sectors before BEER on '%s'\n", beer_device);
	  xclose (fout);
	  return 1;
	  }
      }
  if ((*xfct) (fout, beer.bytes, sizeof (beer.bytes))) {
      EXTRA ("Extra: cannot write the BEER sector on '%s'\n", beer_device);
      xclose (fout);
      return 1;
      }
  while (++cpt < beersect->BeerNbSector) {
      if ((*xfct) (fout, empty_sector, sizeof (empty_sector))) {
	  EXTRA ("Extra: cannot write last sectors after BEER on '%s'\n", beer_device);
	  xclose (fout);
	  return 1;
	  }
      }
  if (xclose (fout)) {
      EXTRA ("Extra: cannot close the device after writing its BEER on %s\n", beer_device);
      return 1;
      }
  return 0;
  }

/* Just a dummy scan: check if someone would have installed a BEER sector
   when the HPA was active, and warn^Wbug him... */
static unsigned
check_beer (const char *beer_device, unsigned long long output_partition_offset,
		unsigned Bytepersector)
  {
  xFILE fout;
  union {
	unsigned short words[Bytepersector/2];
	struct beer_s data;
	} beer;
  unsigned nbsectortoscan = 32;
  unsigned cpt;

  if (xfopen (beer_device, &fout, (int (*) (xFILE, const void *, unsigned))xread)) {
      EXTRA ("Extra: cannot open file to check superflous BEER sectors on \"%s\"\n", beer_device);
      return 1;
      }
  if (xset_location (fout, 0, output_partition_offset - nbsectortoscan, Bytepersector)) {
      xcheck_location (fout, 0, output_partition_offset - nbsectortoscan,
		      Bytepersector, "superflous BEER positionement");
      ERROR ("Did not acheive to set location to %llu sectors\n",
		output_partition_offset - nbsectortoscan);
      xclose (fout);
      return 1;
      }
  for (cpt = 0; cpt < nbsectortoscan; cpt++) {
      if (xread (fout, beer.words, sizeof (beer.words)) != 0) {
	  ERROR ("Did not acheive to read a setor at %llu\n",
		output_partition_offset - nbsectortoscan + cpt);
	  xclose (fout);
	  return 1;
	  }
      if (beer.data.signature_0xBEEF == 0xBEEF) {
	  unsigned short checksum = 0, i = sizeof(beer.data) / 2;
	  while (i--)
	      checksum -= beer.words[i];
	  WARNING ("!! BEER signature (with %svalid checksum) found at %llu, i.e. %u below HPA!\n",
		(checksum == beer.data.checksum) ? "" : "in",
		output_partition_offset - nbsectortoscan + cpt,
		nbsectortoscan + cpt);
	  WARNING ("!! It usually means someone did a Gujin install after Gujin has booted the "
		   "system and set the HPA.\n");
	  WARNING ("!! Always COLD boot your PC, uninstall previous Gujin and then boot Linux "
		   "_before_ running './gujin /dev/hda0'\n");
	  WARNING ("!! It is absolutely not a problem for now, just means that some cleaning of "
		   "the unallocated sectors of the disk (partition wise) is needed.\n");
	  }
      }
  if (xclose (fout)) {
      EXTRA ("Extra: cannot close the device after writing its BEER on %s\n", beer_device);
      return 1;
      }
  return 0;
  }

/**
 ** Helpers for main:
 **/

static const char *
pattern_match (const char *argv, const char *pat1, const char *pat2)
  {
  for (;;) {
      char curr = *argv++;

      if (curr == 0)
	  return 0;
      if (curr >= 'A' && curr <= 'Z')
	  curr += 'a' - 'A';
      if (pat1 != 0) {
	  char pat = *pat1++;

	  if (pat >= 'A' && pat <= 'Z')
	      pat += 'a' - 'A';
	  if (curr != pat)
	      pat1 = 0;
	    else if (*pat1 == 0)
	      return argv;
	  }
      if (pat2 != 0) {
	  char pat = *pat2++;

	  if (pat >= 'A' && pat <= 'Z')
	      pat += 'a' - 'A';
	  if (curr != pat)
	      pat2 = 0;
	    else if (*pat2 == 0)
	      return argv;
	  }
      if (pat1 == 0 && pat2 == 0)
	  return 0;
      }
  }

static inline unsigned getdec (const char **ptr)
  {
  unsigned nb = 0;
  while (**ptr >= '0' && **ptr <= '9') {
      nb = nb * 10 + **ptr - '0';
      *ptr += 1;
      }
  return nb;
  }

static inline unsigned gethex (const char **ptr)
  {
  unsigned nb = 0;
  for (;;) {
      int delta;

      if (**ptr >= '0' && **ptr <= '9')
	  delta = '0';
	else if  (**ptr >= 'a' && **ptr <= 'f')
	  delta = 'a' - 10;
	else if  (**ptr >= 'A' && **ptr <= 'F')
	  delta = 'A' - 10;
	else
	  return nb;
      nb = (nb << 4) + **ptr - delta;
      *ptr += 1;
      }
  }

static /* inline */ unsigned getnb (const char **ptr)
  {
  if ((*ptr)[0] != '0' || (*ptr)[1] != 'x')
      return getdec (ptr);
  *ptr += 2;
  return gethex (ptr);
  }

static unsigned
is_param (const char *argv, const char *pat1, const char *pat2, unsigned *val)
  {
  const char *scan = pattern_match (argv, pat1, pat2);
  unsigned negative = 0;
  if (scan == 0)
      return 0;
  if (*scan == 0) {
      PARAMERR ("option '%s' to short, abort\n", argv);
      exit (1);
      }
  if (*scan == '-') {
      negative = 1;
      scan++;
      }
  *val = getnb(&scan);
  if (*scan != '\0') {
      PARAMERR ("argument of option '%s' not a digit, abort\n", argv);
      exit (1);
      }
  if (negative)
      *val = -*val;
  return 1;
  }

static inline unsigned long long getldec (const char **ptr)
  {
  unsigned long long nb = 0;
  while (**ptr >= '0' && **ptr <= '9') {
      nb = nb * 10 + **ptr - '0';
      *ptr += 1;
      }
  return nb;
  }

static inline unsigned long long getlhex (const char **ptr)
  {
  unsigned long long nb = 0;
  for (;;) {
      int delta;

      if (**ptr >= '0' && **ptr <= '9')
	  delta = '0';
	else if  (**ptr >= 'a' && **ptr <= 'f')
	  delta = 'a' - 10;
	else if  (**ptr >= 'A' && **ptr <= 'F')
	  delta = 'A' - 10;
	else
	  return nb;
      nb = (nb << 4) + **ptr - delta;
      *ptr += 1;
      }
  }

static /* inline */ unsigned long long getlnb (const char **ptr)
  {
  if ((*ptr)[0] != '0' || (*ptr)[1] != 'x')
      return getldec (ptr);
  *ptr += 2;
  return getlhex (ptr);
  }

/*
 * Disk stuff:
 */
static unsigned
get_geometry (const char **cmd, struct geometry_s *geo)
  {
  /* "Psize,Pstart,Head,Sector/Track,SectorSize" */
  if (**cmd < '0' || **cmd > '9') {
      PARAMERR ("got '%c' instead of a digit when trying to get total_sector\n",
			**cmd);
      return 1;
      }
  geo->total_sector = getlnb (cmd);
  if (**cmd != 0 && **cmd != ',') {
      PARAMERR ("got '%c' instead of a comma after reading total_sector=%llu\n",
			**cmd, geo->total_sector);
      return 1;
      }
    else if (**cmd != 0) {
      *cmd += 1;
      }
  if (geo->total_sector == 0) {
      PARAMERR ("total_sector is zero\n");
      return 1;
      }
  /* If we have a total sector, set those defaults instead of the
	content of the boot record in case of disk image: */
//  geo->start_sector = 0;
//  geo->BytePerSector = 512;
  geo->diskindex = 0x80;
  if (geo->total_sector <= 2 * 36 * 80)
      ; /* keep MBR defaults in boot.c */
  else if (geo->total_sector <   2 * 32 * 0x200) /* 16 Mbytes */
      geo->heads = 2, geo->sectpertrack = 32;
  else if (geo->total_sector < 128 * 32 * 0x10000 /* 128 Gbytes */
	|| geo->total_sector >= 255 * 63 * 0x10000) /* 502 Gbytes */
      geo->heads = 128, geo->sectpertrack = 32;
  else
      geo->heads = 255, geo->sectpertrack = 63;

  if (**cmd != 0) {
      if (**cmd < '0' || **cmd > '9') {
	  PARAMERR ("got '%c' instead of a digit when trying to get start_sector\n",
			**cmd);
	  return 1;
	  }
      geo->start_sector = getlnb (cmd);
      if (**cmd != 0 && **cmd != ',') {
	  PARAMERR ("got '%c' instead of a comma after reading start_sector=%llu\n",
			**cmd, geo->start_sector);
	  return 1;
	  }
	else if (**cmd != 0) {
	  *cmd += 1;
	  }
      }

  if (**cmd != 0) {
      if (**cmd < '0' || **cmd > '9') {
	  PARAMERR ("got '%c' instead of a digit when trying to get heads\n",
			**cmd);
	  return 1;
	  }
      geo->heads = getnb (cmd);
      if (**cmd != 0 && **cmd != ',') {
	  PARAMERR ("got '%c' instead of a comma after reading heads=%u\n",
			**cmd, geo->heads);
	  return 1;
	  }
	else if (**cmd != 0) {
	  *cmd += 1;
	  }
      }

  if (**cmd != 0) {
      if (**cmd < '0' || **cmd > '9') {
	  PARAMERR ("got '%c' instead of a digit when trying to get sectpertrack\n",
			**cmd);
	  return 1;
	  }
      geo->sectpertrack = getnb (cmd);
      if (**cmd != 0 && **cmd != ',') {
	  PARAMERR ("got '%c' instead of a comma after reading sectpertrack=%u\n",
			**cmd, geo->sectpertrack);
	  return 1;
	  }
	else if (**cmd != 0) {
	  *cmd += 1;
	  }
      }

  if (geo->heads * geo->sectpertrack != 0) {
      geo->cylinders = DIVROUNDUP (geo->total_sector, geo->heads * geo->sectpertrack);
      }

  if (**cmd != 0) {
      if (**cmd < '0' || **cmd > '9') {
	  PARAMERR ("got '%c' instead of a digit when trying to get BytePerSector\n",
			**cmd);
	  return 1;
	  }
      geo->BytePerSector = getnb (cmd);
      if (**cmd != 0 && **cmd != ',') {
	  PARAMERR ("got '%c' instead of a comma after reading BytePerSector=%u\n",
			**cmd, geo->BytePerSector);
	  return 1;
	  }
	else if (**cmd != 0) {
	  *cmd += 1;
	  }
      }
  return 0;
  }

static inline unsigned
setup_beer (const char *beercmd, struct beersect_s *beersect)
  {
  *beersect = (struct beersect_s) {
	.BeerNbSector = 8, /* Nb sector at real end of disk for the single BEER sector = 8 */
	.BeerPosition = 4, /* half the previous value: that is 4 */
	.BeerPartition = {
		.minSector = 360 * 2,		/* equivalent of 360 K floppy */
		.maxSector = 32 * 1024 * 2,	/* 32 Mbytes if blank disk */
		}
	};

  if (!beercmd)
      return 0;

  if (*beercmd == '\0')
      return 0;
    else if (*beercmd >= '0' && *beercmd <= '9')
      beersect->BeerNbSector = getnb (&beercmd);
    else {
      EXTRA ("Extra: did not understand '--beer=' first parameter in '%s'\n", beercmd);
      return 1;
      }

  if (*beercmd == '\0')
      return 0;
    else if (*beercmd != ',') {
      EXTRA ("Extra: did not found the ',' but '%c' after first parameter.\n", *beercmd);
      return 1;
      }
  beercmd ++;

  if (*beercmd >= '0' && *beercmd <= '9')
      beersect->BeerPosition = getnb (&beercmd);
    else {
      EXTRA ("Extra: did not understand '--beer=' second parameter in '%s'\n", beercmd);
      return 1;
      }

  if (*beercmd == '\0')
      return 0;
    else if (*beercmd != ',') {
      EXTRA ("Extra: did not found the ',' but '%c' after second parameter.\n", *beercmd);
      return 1;
      }
  beercmd ++;

  if (*beercmd >= '0' && *beercmd <= '9')
      beersect->BeerPartition.minSector = getnb (&beercmd);
    else {
      EXTRA ("Extra: did not understand '--beer=' third parameter in '%s'\n", beercmd);
      return 1;
      }

  if (*beercmd == '\0')
      return 0;
    else if (*beercmd != ',') {
      EXTRA ("Extra: did not found the ',' but '%c' after third parameter.\n", *beercmd);
      return 1;
      }
  beercmd ++;

  if (*beercmd >= '0' && *beercmd <= '9')
      beersect->BeerPartition.maxSector = getnb (&beercmd);
    else {
      EXTRA ("Extra: did not understand '--beer=' fourth parameter in '%s'\n", beercmd);
      return 1;
      }

  if (*beercmd != '\0') {
      EXTRA ("Extra: found trailer char '%c' after fourth parameter.\n", *beercmd);
      return 1;
      }
  return 0;
  }

static inline unsigned
setup_geometry_beer (struct install_s *install, char *dn, const char * devname,
		unsigned BytePerSector, unsigned pagesize, unsigned long long hdgeo_total_sector)
  {
  int cpt, fd;
  long long max = 0, min = -1LL, partlen;

  if (primary_GPT_sector == 0)
      EXTRA ("Extra: will create a BEER partition at end of disk %s\n", dn);
    else
      EXTRA ("Extra: will create a floppy image at begining of disk %s (GPT partition scheme)\n", dn);

  for (cpt = 1; cpt < 64; cpt++) { /* max /dev/hda64, 64 partition on the disk */
      unsigned long long start_sector;
      char syspart[] = "/sys/block/hd?/hd?00/size_or_start", syspartstart[sizeof(syspart)];

      syspart[11] = syspart[15] = dn[5];	/* h or s */
      syspart[13] = syspart[17] = dn[7];	/* a..z */

      if (cpt < 10) {
	  syspart[18] = dn[8] = '0' + cpt;
	  syspart[19] = dn[9] = '\0';
	  }
	else {
	  syspart[18] = dn[8] = '0' + cpt / 10;
	  syspart[19] = dn[9] = '0' + cpt % 10;
	  syspart[20] = dn[10] = '\0';
	  }
      strcpy (syspartstart, syspart);
      strcat (syspartstart, "/start");
      strcat (syspart, "/size");
      if ((fd = open (syspart, O_RDONLY)) != -1) {
	  char str[64], *ptr;
	  int sizeread;

	  if ((sizeread = read (fd, str, sizeof(str) - 1)) <= 0) {
	      ERROR ("Extra: file '%s' read error\n", syspart);
	      return 1;
	      }
	  str[sizeread] = 0;
	  close (fd);
	  for (partlen = 0, ptr = str; *ptr != 0; ptr++) {
	      if (*ptr >= '0' && *ptr <= '9')
		  partlen = partlen * 10 + *ptr - '0';
		else if (*ptr != '\n') {
		  ERROR ("Extra: file '%s' contains unknown chars\n", syspart);
		  return 1;
		  }
	      }
	  if ((fd = open (syspartstart, O_RDONLY)) == -1) {
	      ERROR ("can read file '%s' but not file '%s' \n", syspart, syspartstart);
	      return 1;
	      }
	  if ((sizeread = read (fd, str, sizeof(str) - 1)) <= 0) {
	      ERROR ("Extra: file '%s' read error\n", syspartstart);
	      close (fd);
	      return 1;
	      }
	  close (fd);
	  str[sizeread] = 0;
	  for (start_sector = 0, ptr = str; *ptr != 0; ptr++) {
	      if (*ptr >= '0' && *ptr <= '9')
		  start_sector = start_sector * 10 + *ptr - '0';
		else if (*ptr != '\n') {
		  ERROR ("Extra: file '%s' contains unknown chars\n", syspartstart);
		  return 1;
		  }
	      }
	  EXTRA ("Extra: (/sys/ partition length) %s start %llu length %llu, ", dn, start_sector, (long long)partlen);
	  }
	else if ((fd = open (dn, O_RDONLY)) == -1) {
	  if (errno == EACCES) {
	      ERROR ("Extra: cannot open '%s' or '%s' to get partition mapping, you probably need to be root\n", dn, syspart);
	      return 1;
	      }
	  continue; /* do not assume /dev/hda6 does not exists when no /dev/hda3 */
	  }
	else {
	  if ((partlen = GET_FD_LENGTH(fd)) == 0 || partlen == -1LL) {
	      close (fd);
	      continue;
	      }
	  if (partlen == -1) {
	      close (fd);
	      ERROR ("%s failure to get last used sector, aborting for safety\n", __FUNCTION__);
	      return 1;
	      }
	  partlen /= BytePerSector;
	  if (get_device_start_sector (fd, &start_sector, 0) != 0) {
	      close (fd);
	      ERROR ("%s failure to get start of partition, aborting for safety\n", __FUNCTION__);
	      return 1;
	      }
	  close (fd);
	  EXTRA ("Extra: (direct partition length) %s start %llu length %llu, ", dn, start_sector, (long long)partlen);
	  }
      if (start_sector < min) {
	  EXTRA ("is minimum %llu, ", start_sector);
	  min = start_sector;
	  }
      if (partlen + start_sector < max)
	  EXTRA ("not a maximum %llu\n", (long long)partlen + start_sector);
	else {
	  EXTRA ("is maximum %llu\n", (long long)partlen + start_sector);
	  max = partlen + start_sector;
	  }
      }
  dn[8] = '\0';
  EXTRA ("Extra: last used sector of disk %s is %llu (first used %llu) on a total of %llu, %llu free sectors\n",
	dn, max, min, hdgeo_total_sector, hdgeo_total_sector - max);

  if (primary_GPT_sector != 0 && min < 256) {
      ERROR ("Do not have any free space to install Gujin at begining of the disk %s\n", devname);
      return 1;
      }
    else if (!primary_GPT_sector && hdgeo_total_sector < max) {
      ERROR ("Not enough free space at end of %s to fit its last used sector at %llu?\n", devname, max);
      return 1;
      }
    else if (!primary_GPT_sector && hdgeo_total_sector < (max / pagesize + 1) * pagesize) {
      ERROR ("Not enough free space at end of %s %s!\n",
		devname, (hdgeo_total_sector % pagesize) ? "for even rounding its size" : "to insert a BEER partition");
      return 1;
      }
    else { /* decide what will be the BEER partition: */
      unsigned long long space_available;

      /* wholedisk.total_sector is checked by chs_chain(): */
      install->wholedisk.total_sector = hdgeo_total_sector;
      install->wholedisk.start_sector = 0;
      /* Try to always report a hard disk with a size which is a multiple of 4 Kb: */
      install->misc.formatted_disk_size = install->wholedisk.total_sector;
      install->misc.reported_disk_size = (install->wholedisk.total_sector / pagesize) * pagesize;
      /* I am using misc->formatted_disk_size here, is that right? */
      install->misc.beer_sector_offset = install->misc.formatted_disk_size - install->beersect.BeerNbSector;
      install->misc.restricted_disk_size = (max / pagesize) * pagesize;

      if (   install->misc.restricted_disk_size < max /* i.e. not equal */
	  && install->misc.restricted_disk_size + pagesize <= hdgeo_total_sector) {
	  EXTRA ("Extra: minimum disk length has been rounded from %llu to %llu sectors\n",
		max, install->misc.restricted_disk_size + pagesize);
	  install->misc.restricted_disk_size += pagesize;
	  }
	else
	  EXTRA ("Extra: minimum disk length do not need to be rounded, it has %llu sectors\n",
		install->misc.restricted_disk_size);
      if (primary_GPT_sector) {
	  if (start_BEER_if_GPT + install->beersect.BeerNbSector >= min) {
	      ERROR ("Not enough free space at begining of %s for semi-BEER sectors\n", devname);
	      return 1;
	      }
	  space_available = min - start_BEER_if_GPT - install->beersect.BeerNbSector;
	  }
	else {
	  if (install->misc.restricted_disk_size + install->beersect.BeerNbSector >= hdgeo_total_sector) {
	      ERROR ("Not enough free space at end of %s for BEER sectors\n", devname);
	      return 1;
	      }
	  space_available = hdgeo_total_sector - (install->misc.restricted_disk_size + install->beersect.BeerNbSector);
	  }
      /* Refuse if there is not enough space, i.e. less than a 360 Kbytes floppy */
      if (space_available < install->beersect.BeerPartition.minSector) {
	  ERROR ("Not enough free space at end of %s for BEER partition, have %llu sectors, min %u\n",
		devname, space_available, install->beersect.BeerPartition.minSector);
	  return 1;
	  }
      /* Give some kind of maximum for a completely new HD, i.e. blank HD with 250 Gb free: */
      if (!primary_GPT_sector && space_available > install->beersect.BeerPartition.maxSector) {
	  unsigned long long tmp;
	  EXTRA ("Extra: limiting size of BEER partition for %s from %llu to %u sectors",
		devname, space_available, install->beersect.BeerPartition.maxSector);
	  install->partdisk.total_sector = install->beersect.BeerPartition.maxSector;
	  install->partdisk.start_sector = install->wholedisk.total_sector
			- install->beersect.BeerNbSector - install->partdisk.total_sector;

	  tmp = (install->partdisk.start_sector / pagesize) * pagesize;
	  if (tmp >= install->misc.restricted_disk_size) {
	      EXTRA (", and increasing disk len to %llu\n", tmp);
	      install->misc.restricted_disk_size = tmp;
	      }
	    else {
	      EXTRA ("\nERROR:ALGORITHM ERROR tmp=%llu, install->misc.restricted_disk_size=%llu\n",
			tmp, install->misc.restricted_disk_size);
	      return 1;
	      }
	  }
	else if (primary_GPT_sector) {
	  if (start_BEER_if_GPT + install->beersect.BeerNbSector + space_available > 1024 * 1024 / BytePerSector) {
	      EXTRA ("Extra: limiting size of semi-BEER partition for %s from %llu finish before 1Mb", devname, space_available);
	      space_available = 1024 * 1024 / BytePerSector - start_BEER_if_GPT - install->beersect.BeerNbSector;
	      }
	  EXTRA ("Extra: space_available for semi-BEER sector: %u, for semi-BEER partition: %llu sectors\n",
			install->beersect.BeerNbSector, space_available);
	  install->partdisk.total_sector = space_available;
	  install->partdisk.start_sector = start_BEER_if_GPT;
	  /* Overwrite beer_sector_offset: */
	  install->misc.beer_sector_offset = start_BEER_if_GPT + space_available;
	  }
	else {
	  EXTRA ("Extra: space_available for BEER sector: %u, for BEER partition: %llu sectors\n",
			install->beersect.BeerNbSector, space_available);
	  install->partdisk.total_sector = space_available;
	  install->partdisk.start_sector = install->wholedisk.total_sector
		- install->beersect.BeerNbSector - install->partdisk.total_sector;
	  }
      EXTRA ("Extra: Disk physically has %llu pages of 4 Kbytes, report %llu pages.\n",
	  install->misc.reported_disk_size / pagesize,
	  install->misc.restricted_disk_size / pagesize);
      }
  return 0;
  }

static inline unsigned
setup_geometry_hd (const char *geocmd, struct install_s *install,
			const char *devname, long long length,
			unsigned long long partdisk_start_sector)
{
  struct get_hard_disk_geometry_s hdgeo = {};	/* warning inited b4 used */
  unsigned device_name_len = strlen(geocmd), BytePerSector;
  char dn[device_name_len + 1];
  int fd;
  long long len;

  strcpy (dn, geocmd);

  /* remove the partition digits, if they exist: */
  if (!strncmp (dn, "/dev/mmcblk", strlen ("/dev/mmcblk"))) {
      /* partition named like /dev/mmcblk0p1 */
      dn[12] ='\0';
      }
  else if (strncmp (dn, "/dev/md", strlen ("/dev/md")) && strncmp (dn, "/dev/sr", strlen ("/dev/sr"))) {
      while (--device_name_len > 0 && dn[device_name_len] >= '0' && dn[device_name_len] <= '9')
	  dn[device_name_len] = '\0';
      }

  if ((fd = open (dn, O_RDONLY)) == -1) {
      /* O_EXCL on Linux means fail if already mounted. at least when supermount is not used... */
      ERROR ("cannot open device '%s' to get sector length & check for GPT\n", dn);
      perror (dn);
      return 1;
      }
    else if ((len = GET_FD_LENGTH(fd)) == -1LL) {
      if (errno)
	  perror("GET_FD_LENGTH");
      close (fd);
      ERROR ("cannot get length of device %s\n", dn);
      return 1;
      }
    else if ((BytePerSector = GET_PHYSICAL_BLOCKSIZE(fd)) < 512) {
      ERROR ("BytePerSector reported as %u on device %s\n", BytePerSector, dn);
      return 1;
      }
    else if (get_hard_disk_geometry (fd, &hdgeo) != 0) {
      if (errno)
	  perror("get_hard_disk_geometry");
      unsigned long long start_sector, nbsectors = 0;
      if (get_device_start_sector (fd, &start_sector, &nbsectors) == 0 && start_sector == 0 && nbsectors != 0) { /* DVD-RAM */
	  hdgeo.total_sector = nbsectors;
	  hdgeo.start_sector = start_sector;
	  hdgeo.cylinders = 0;
	  hdgeo.heads = 0;
	  hdgeo.sectpertrack = 0;
	  EXTRA ("Extra: Generate a disk geometry for DVD-RAM: %llu+%llu, hopes C/H/S unused!\n", hdgeo.start_sector, hdgeo.total_sector);
	  }
	else {
	  close (fd);
	  EXTRA ("Extra: removeable device? LVM? get_hard_disk_geometry '%s' failed.\n", dn);
	  return 1;
	  }
      }
//  unsigned long DeviceSize = GET_PHYSICAL_SIZE(fd);
  unsigned long DeviceSize = GET_PHYSICAL_SIZE(fd) / (BytePerSector/512);
  EXTRA ("Extra: device is %lu blocks of %u bytes, i.e. %llu bytes\n", DeviceSize, BytePerSector, (unsigned long long)DeviceSize * BytePerSector);

  /* CHECK for GPT (if whole disk), should not overwrite GTP (http://en.wikipedia.org/wiki/GUID_Partition_Table) by B.E.E.R. */
  if (hdgeo.start_sector == 0) { /* TAKE CARE, hdgeo.start_sector is in 512 bytes/sector whatever device */
      unsigned long last_sector = DeviceSize;
      while (last_sector-- >= DeviceSize - 8) {
//      unsigned long last_sector;
//      for (last_sector = DeviceSize - 8; last_sector < DeviceSize; last_sector++) {
	  EXTRA ("Extra: looking for possible secondary GPT at %lu\r\n", last_sector);
	  if (lseek64 (fd, (long long)last_sector * BytePerSector, SEEK_SET) != (long long)last_sector * BytePerSector) {
	      ERROR ("cannot lseek64 to %lu * %u to check for GPT, disk size %lu sectors\n", last_sector, BytePerSector, DeviceSize);
	      perror (dn);
	      break;
	      }
	    else {
	      char GPT_header_sig[8];
	      if (read (fd, GPT_header_sig, sizeof(GPT_header_sig)) != sizeof(GPT_header_sig)) {
		  ERROR ("cannot read 8 bytes from device to check for GPT at lba %lu, disk size %lu sectors\n", last_sector, DeviceSize);
		  perror (dn);
		  continue;
		  }
	      if (*(long long *)GPT_header_sig == *(long long *)"EFI PART") {
		  EXTRA ("Extra: Device %s has the GPT signature at lba %lu, disk size %lu sectors\n", dn, last_sector, DeviceSize);
		  /* FIXME: it would still be possible to move & update the GTP, so that the B.E.E.R. will set the HPA
			and keep the secondary GTP accessible, but the uninstall procedure may then be complex */
		  /* Instead we try to write the BEER is the first Mbyte if free space */
		  secondary_GPT_sector = last_sector;
		  break;
		  }
	      }
	  }

      if (lseek64 (fd, BytePerSector, SEEK_SET) != BytePerSector) {
	  ERROR ("cannot lseek64 to %u to check for GPT\n", BytePerSector);
	  perror (dn);
	  return 1;
	  }
	else {
	  char GPT_header_sig[8];
	  if (read (fd, GPT_header_sig, sizeof(GPT_header_sig)) != sizeof(GPT_header_sig)) {
	      ERROR ("cannot read 8 bytes from device to check for GPT at lba 1\n");
	      perror (dn);
	      return 1;
	      }
	  if (*(long long *)GPT_header_sig == *(long long *)"EFI PART") {
	      EXTRA ("Extra: Device %s has the GPT signature at lba 1\n", dn);
		/* FIXME: it would still be possible to move & update the GTP, so that the B.E.E.R. will set the HPA
			and keep the secondary GTP accessible, but the uninstall procedure may then be complex */
		/* Instead we try to write the BEER is the first Mbyte if free space */
	      primary_GPT_sector = last_sector;
	      }
	  }
      if (primary_GPT_sector && !secondary_GPT_sector) {
	  WARNING ("Device %s has the primary GPT signature at lba 1, but secondary GPT signature not found (device size %lu)!\n", dn, DeviceSize);
	  }
	else if (!primary_GPT_sector && secondary_GPT_sector) {
	  WARNING ("Device %s has a secondary GPT signature BUT NO primary GPT signature at lba 1, assuming no GPT!\n", dn);
	  }

      if (primary_GPT_sector) {
	  /* Just in case, display a warning if the two GPT are not identical... */
	  union {
	    unsigned char fullsector[BytePerSector];
	    struct GPT_header_s  GPT_header;
	    struct GPT_entry_s  GPT_entry[0];
	    } primary, secondary;
	  if (lseek64 (fd, BytePerSector, SEEK_SET) == BytePerSector
		&& read (fd, &primary, sizeof(primary)) == sizeof(primary)
		&& lseek64 (fd, (long long)secondary_GPT_sector * BytePerSector, SEEK_SET) == (long long)secondary_GPT_sector * BytePerSector
		&& read (fd, &secondary, sizeof(secondary)) == sizeof(secondary)) {
	      if (primary.GPT_header.backup_lba != secondary_GPT_sector) {
		  WARNING ("Extra: Device %s primary.GPT_header.backup_lba %llu != secondary_GPT_sector found at %llu\n", dn,
			primary.GPT_header.backup_lba, secondary_GPT_sector);
		  if (secondary_GPT_sector == 0
			&& lseek64 (fd, (long long)primary.GPT_header.backup_lba * BytePerSector, SEEK_SET) == (long long)primary.GPT_header.backup_lba * BytePerSector
			&& read (fd, &secondary, sizeof(secondary)) == sizeof(secondary)) {
		      WARNING ("Extra: Device %s assuming secondary_GPT_sector at %llu\n", dn, primary.GPT_header.backup_lba);
		      secondary_GPT_sector = primary.GPT_header.backup_lba;
		      }
		  }
	      unsigned long long primary_partition_entry_start_lba, secondary_partition_entry_start_lba;
	      primary_partition_entry_start_lba = primary.GPT_header.partition_entry_start_lba;
	      secondary_partition_entry_start_lba = secondary.GPT_header.partition_entry_start_lba;
	      EXTRA ("Extra: Device %s primary partition_entry_start_lba %llu, secondary partition_entry_start_lba %llu\n", dn,
		primary_partition_entry_start_lba, secondary_partition_entry_start_lba);
	      if (primary.GPT_header.revision != secondary.GPT_header.revision)
		  WARNING ("Device %s has different GPT header revision (primary 0x%X / secondary 0x%X)!\n", dn, primary.GPT_header.revision, secondary.GPT_header.revision);
	      if (primary.GPT_header.header_size != secondary.GPT_header.header_size)
		  WARNING ("Device %s has different GPT header header_size (primary %d / secondary %d)!\n", dn, primary.GPT_header.header_size, secondary.GPT_header.header_size);
//	      if (primary.GPT_header.crc32_header != secondary.GPT_header.crc32_header)
//		  WARNING ("Device %s has different GPT header crc32_header (primary 0x%X / secondary 0x%X)!\n", dn, primary.GPT_header.crc32_header, secondary.GPT_header.crc32_header);
	      if (primary.GPT_header.reserved != secondary.GPT_header.reserved)
		  WARNING ("Device %s has different GPT header reserved (primary 0x%X / secondary 0x%X)!\n", dn, primary.GPT_header.reserved, secondary.GPT_header.reserved);
	      if (primary.GPT_header.current_lba != secondary.GPT_header.backup_lba) // secondary.GPT_header.current_lba)
		  WARNING ("Device %s has different GPT header (primary current_lba %llu / secondary backup_lba %llu)!\n", dn, primary.GPT_header.current_lba, secondary.GPT_header.backup_lba);
	      if (primary.GPT_header.backup_lba != secondary.GPT_header.current_lba) // secondary.GPT_header.backup_lba)
		  WARNING ("Device %s has different GPT header (primary backup_lba %llu / secondary current_lba %llu)!\n", dn, primary.GPT_header.backup_lba, secondary.GPT_header.current_lba);
	      if (primary.GPT_header.first_useable_lba != secondary.GPT_header.first_useable_lba)
		  WARNING ("Device %s has different GPT header first_useable_lba (primary %llu / secondary %llu)!\n", dn, primary.GPT_header.first_useable_lba, secondary.GPT_header.first_useable_lba);
	      if (primary.GPT_header.last_useable_lba != secondary.GPT_header.last_useable_lba)
		  WARNING ("Device %s has different GPT header last_useable_lba (primary %llu / secondary %llu)!\n", dn, primary.GPT_header.last_useable_lba, secondary.GPT_header.last_useable_lba);
	      if (memcmp(primary.GPT_header.disk_guid, secondary.GPT_header.disk_guid, sizeof (primary.GPT_header.disk_guid)))
		   WARNING ("Device %s has different GPT header disk_guid (primary/secondary)!\n", dn);
//	      if (primary.GPT_header.partition_entry_start_lba != secondary.GPT_header.partition_entry_start_lba)
//		  WARNING ("Device %s has different GPT header partition_entry_start_lba (primary 0x%X/secondary 0x%X)!\n", dn, primary.GPT_header.partition_entry_start_lba, secondary.GPT_header.partition_entry_start_lba);
	      if (primary.GPT_header.number_of_partition_entry != secondary.GPT_header.number_of_partition_entry)
		  WARNING ("Device %s has different GPT header number_of_partition_entry (primary %u / secondary %u)!\n", dn, primary.GPT_header.number_of_partition_entry, secondary.GPT_header.number_of_partition_entry);
	      if (primary.GPT_header.sizeof_partition_entry != secondary.GPT_header.sizeof_partition_entry)
		  WARNING ("Device %s has different GPT header sizeof_partition_entry (primary %u / secondary %u)!\n", dn, primary.GPT_header.sizeof_partition_entry, secondary.GPT_header.sizeof_partition_entry);
	      if (primary.GPT_header.crc32_partition_array != secondary.GPT_header.crc32_partition_array)
		  WARNING ("Device %s has different GPT header crc32_partition_array (primary 0x%X/secondary 0x%X)!\n", dn, primary.GPT_header.crc32_partition_array, secondary.GPT_header.crc32_partition_array);
	      if (primary_partition_entry_start_lba != secondary_partition_entry_start_lba) {
		  unsigned nb_gpt_sect = primary.GPT_header.number_of_partition_entry * primary.GPT_header.sizeof_partition_entry / BytePerSector;
		  while (nb_gpt_sect--) {
		      if (lseek64 (fd, (long long)primary_partition_entry_start_lba * BytePerSector, SEEK_SET) == (long long)primary_partition_entry_start_lba * BytePerSector
			&& read (fd, &primary, sizeof(primary)) == sizeof(primary)
			&& lseek64 (fd, (long long)secondary_partition_entry_start_lba * BytePerSector, SEEK_SET) == (long long)secondary_partition_entry_start_lba * BytePerSector
			&& read (fd, &secondary, sizeof(secondary)) == sizeof(secondary)) {
			  if (memcmp(&primary, &secondary, sizeof(primary)))
			      WARNING ("Device %s has different GPT sectors (primary %llu / secondary %llu)!\n", dn, primary_partition_entry_start_lba, secondary_partition_entry_start_lba);
			  }
			else
			  WARNING ("Device %s cannot seek or read GPT sectors (primary %llu / secondary %llu)!\n", dn, primary_partition_entry_start_lba, secondary_partition_entry_start_lba);
		      primary_partition_entry_start_lba++;
		      secondary_partition_entry_start_lba++;
		      }
		  }
	      }
	    else
		WARNING ("Device %s could not read either primary or secondary GPT header!\n", dn);
	  }
      }

  if (close (fd) != 0) {
      EXTRA ("Extra: cannot close %s\n", dn);
      return 1;
      }
    else if (hdgeo.total_sector == 0) {
      EXTRA ("Extra: got hdgeo.total_sector == 0\n");
      return 1;
      }

  if (install->misc.forced_Heads) {
      EXTRA ("Extra: Overwriting number of heads from %u to %u.\n",
		hdgeo.heads, install->misc.forced_Heads);
      hdgeo.heads = install->misc.forced_Heads;
      }
  if (install->misc.forced_SectPerTrack) {
      EXTRA ("Extra: Overwriting number of sector/track from %u to %u.\n",
		hdgeo.sectpertrack, install->misc.forced_SectPerTrack);
      hdgeo.sectpertrack = install->misc.forced_SectPerTrack;
      }

  if (hdgeo.total_sector * BytePerSector != len) {
      unsigned long long tmp;
  /* MAYDAY: This C/H/S only disk (no lba) has its geometry 665/16/63, so logical is 332/32/63.
   * Its size is either 665*16*63 = 670320 or 332*32*63 = 669312 ....
Extra:  HDIO_GETGEO_BIG ioctl: total_sector: 669312, start_sector: 0, cylinders: 332, heads: 32, sectpertrack: 63.
Extra: device name /dev/hdb, len 343203840, total_sector 669312, so sector size 512, pages of 8 sectors
....
Extra: Reporting wholedisk: start_sector = 0, total_sector 669312, partdisk: start_sector = 603768, total_sector 65536
Extra: Reporting install->misc.output_partition_offset 603768
Extra: exit setup_geometry with: --geometry=65536,603768,32,63,512,669312,0,32,63,512 (cylinder: 332,332)
  Note the disk len of 343203840 bytes, i.e. 343203840/512=670320...
  Note also: 343203840 / 669312 = 512.771 could be rounded to 512 !
  */
      EXTRA ("Extra: something not perfect: total sector * %u = %llu, total len %llu\n",
		BytePerSector, hdgeo.total_sector * BytePerSector, len);
      if (   dn[0] == '/' && dn[1] == 'd' && dn[2] == 'e' && dn[3] == 'v' && dn[4] == '/'
		&& (dn[5] == 'h' || dn[5] == 's') && dn[6] == 'd' && (dn[7] >= 'a' && dn[7] <= 'z')
	  && (tmp = search_capacity_for_disk (dn[5], dn[7])) != 0) {
	  WARNING ("correcting HDIO_GETGEO_BIG / HDIO_GETGEO total size from %llu to %llu sectors\n",
			hdgeo.total_sector, tmp);
	  if (geocmd[8] == '0'
		&& hdgeo.total_sector != 16383*16*63
		&& hdgeo.total_sector != 65535*16*63
		&& hdgeo.total_sector != hdgeo.heads*hdgeo.sectpertrack*hdgeo.cylinders) {
	      ERROR (
		"This case (%u sectors forgotten in between the HD geometry and the device\n"
		"byte size) happen when HPA is already set - very high probability Gujin has\n"
		"started this PC and protected the B.E.E.R. partition. Because this area is\n"
		"hidden now this installer cannot exactly diagnose what is the content\n"
		"of the end of the disk. Power down and boot WITHOUT Gujin or \n"
		"only with Gujin \"gujin.exe\" on DOS, and then redo this install.\n",
		(unsigned)(tmp - hdgeo.total_sector));
	      return 1;
	      }
	  hdgeo.total_sector = tmp;
	  }
	else if (!strncmp(dn, "/dev/mmcblk", strlen ("/dev/mmcblk"))
		 && dn[11] >= '0' && dn[11] <= '9'
		 && (tmp = search_capacity_for_disk ('\0', dn[11])) != 0) {
	  WARNING ("correcting HDIO_GETGEO_BIG / HDIO_GETGEO total size (mmcblk) from %llu to %llu sectors\n",
			hdgeo.total_sector, tmp);
	  hdgeo.total_sector = tmp;
	  }
      if (len % 512) {
	  ERROR ("BytePerSector: length %llu is not a multiple of 512, abort.\n", len);
	  return 1;
	  }
      }

#if 0 /* len is the length of the partition, not the disk! */
  if (BytePerSector > 512 && hdgeo.total_sector % (BytePerSector/512) == 0) {
      EXTRA ("Extra: Correct Linux view of the device: hdgeo.total_sector from %llu to %llu\n", hdgeo.total_sector, hdgeo.total_sector / (BytePerSector/512));
      hdgeo.total_sector /= (BytePerSector/512);
      }
    else if ((len / hdgeo.total_sector) % 512) {
      BytePerSector = 512;
      WARNING ("Considering that geometry, the total_sector field of HDIO_GETGEO: %llu seems to be wrong, "
		"assume %u bytes/sectors.\n", hdgeo.total_sector, BytePerSector);
      }
    else
      BytePerSector = len / hdgeo.total_sector;
#endif

  unsigned pagesize = 4 * 1024 / BytePerSector;
  EXTRA ("Extra: device name %s, len %llu, total_sector %llu, so sector size %u, pages of %u sectors\n",
		dn, len, hdgeo.total_sector, BytePerSector, pagesize);

  if (strlen_beer_device(geocmd)) { /* only for BEER partition */
      unsigned ret = setup_geometry_beer (install, dn, devname, BytePerSector, pagesize, hdgeo.total_sector);
      if (ret)
	  return ret;
      }
    else /* standard install on a partition */ {
      /* Use the possible corrected values: */
      install->wholedisk.total_sector = hdgeo.total_sector;
      install->partdisk.total_sector = length / BytePerSector;
      install->partdisk.start_sector = partdisk_start_sector;
      }
  install->wholedisk.start_sector = 0;
  install->wholedisk.heads = install->partdisk.heads = hdgeo.heads;
  install->wholedisk.sectpertrack = install->partdisk.sectpertrack = hdgeo.sectpertrack;
  install->wholedisk.cylinders = install->partdisk.cylinders = hdgeo.cylinders;
  install->wholedisk.BytePerSector = install->partdisk.BytePerSector = BytePerSector;
  EXTRA ("Extra: Reporting wholedisk: start_sector = %llu, total_sector %llu, "
		"partdisk: start_sector = %llu, total_sector %llu\n",
	install->wholedisk.start_sector, install->wholedisk.total_sector,
	install->partdisk.start_sector, install->partdisk.total_sector);
  return 0;
  }

static void get_mbrname_from_devicename (char *mbrname, const char *devicename)
{
  while ((*mbrname++ = *devicename++) != '\0')
      continue;
  mbrname--;
  do mbrname --; while (*mbrname >= '0' && *mbrname <= '9');
  /* /dev/mmcblk0 is main device, /dev/mmcblk0p1 is partition... unlike /dev/sda0 */
  if (*mbrname == 'p' && *(unsigned *)(mbrname - 5) == *(unsigned *)"cblk")
      mbrname--;
    else if (mbrname[-1] == 's' && mbrname[0] == 'r') /* /dev/sr0 */
      mbrname++;
    else if (*(unsigned *)(mbrname - 3) == *(unsigned *)"cblk")
      mbrname++;
  *++mbrname = '\0';
}

static inline unsigned
setup_geometry (const char *geocmd, struct install_s *install,
		const char **diskcmd, const char **fscmd, const char **positioncmd,
		const char **mbr_device, const char **beer_device,
		const char *fout_name, const char **partition_index_ptr)
  {
  const char *devname = 0;

  install->misc.infs_file = 0;
  if (geocmd == 0) {
      /* create a false geocmd if geocmd==0 and we have a known extension: */
      const char *ptr = fout_name, *lastdot = 0;

      EXTRA ("Extra: '--geometry=' not given, start .com/.exe/.pic/.bcd/.num/.bios/.iso extension recognition:\n");

      while (*ptr) {
	  if (*ptr == '.')
	      lastdot = ptr;
	  ptr++;
	  }
      if (lastdot) {
	  static const struct {
	      const char *extension, *geocmd;
	      } dflt[] = {
	      { ".360", "floppy:360" }, { ".120", "floppy:120" },
	      { ".720", "floppy:720" }, { ".144", "floppy:144" },
	      { ".168", "floppy:168" }, { ".288", "floppy:288" },
	      { ".com", "dos:com" },
	      { ".exe", "dos:exe" },
	      { ".pic", "PIC" },
	      { ".bcd", "BCD" },	/* KEEP BCD INDEX 9 */

	      { ".bios", "BIOS:0x80,auto" },
	      { ".ebios", "EBIOS:0x80,auto" },
	      { ".idechs", "IDE:chs" },
	      { ".idelba", "IDE:lba" },
	      { ".idel48", "IDE:lba48" },
	      { ".iso", "ISO" }
	      };
	  unsigned i = sizeof (dflt) / sizeof (dflt[0]);
	  while (i--) {
	      unsigned j = 0;
	      while (   dflt[i].extension[j] != '\0'
		     && (   dflt[i].extension[j] == lastdot[j]
			 || (dflt[i].extension[j] == (lastdot[j] -'A' + 'a')
				&& dflt[i].extension[j] >= 'a' && dflt[i].extension[j] <= 'z')))
		  j++;
	      if (dflt[i].extension[j] == '\0' && lastdot[j] == '\0') {
		  if (i < 9) { /* KEEP BCD INDEX 9 */
		      geocmd = dflt[i].geocmd;
		      EXTRA ("set '--geometry=%s'.\n", geocmd);
		      }
		    else if (i == 9) { /* KEEP BCD INDEX 9 */
		      if (*diskcmd == 0) {
			  /* 16 sector loaded max, each 2 Kb so 32 kb loaded (and checksumed) per gujin1 star;
				10000000 (partition size) not used but shall be big enough to not trigger
				an error in instboot. DVD = 8Gb = 4194304 sectors of 2048 bytes */
			  *diskcmd = dflt[i].geocmd;
			  geocmd = "10000000,0,0,16,2048";
			  EXTRA ("\nExtra: set '--geometry=%s' and '--disk=%s' (standard CD/DVD El Torito boot)\n", geocmd, *diskcmd);
			  }
			else {
			  geocmd = dflt[i].geocmd;
			  EXTRA ("\nExtra: set '--geometry=%s' and keep '--disk=%s'.\n", geocmd, *diskcmd);
			  }
		      }
		    else if (i == 15) { /* *.iso */
		      geocmd = dflt[i].geocmd;
		      EXTRA ("set '--geometry=%s'.\n", geocmd);
		      if (*diskcmd == 0) {
			  *diskcmd = "BIOS:0x00,auto";
			  EXTRA ("Extra: set previously empty '--disk=%s'.\n", *diskcmd);
			  }
		      install->misc.iso_file = 1;
		      }
		    else if (i < sizeof(dflt)/sizeof(dflt[0])) {
		      if (fout_name[0] == '/') { /* absolute pathname */
			  unsigned long long start_chain, end_chain;
			  unsigned statbuf_st_dev, unit_size;

			  EXTRA ("infs file extension\n");
			  EXTRA ("Extra: target absolute pathname, test creating boot file whith hole as '%s'.\n", fout_name);
			  struct stat sb;
			  if (stat(fout_name, &sb) == 0) {
			      ERROR ("Cannot check if FIBMAP works, file '%s' exists (size %llu, %llu blocks allocated i.e %llu bytes allocated).\n", fout_name, (unsigned long long)sb.st_size, (unsigned long long)sb.st_blocks, (unsigned long long)sb.st_blocks * 512);
			      return 1;
			      }
			  if (infs_filemap (fout_name, 1, &unit_size, &start_chain, &end_chain, &statbuf_st_dev, 12) == 0) {
			      static char buffname[256];
			      int tst_FIBMAP;
			      char data;

			      install->misc.holy_filesystem = (start_chain == end_chain)? 12 : 0; /* i.e. only one block used, E2FS needs 12 */
			      install->misc.holy_unit_size = unit_size;
			      install->misc.infs_file = 1;
				  /* FIXME: install->wholedisk.BytePerSector still zero */
			      install->misc.begin_file_sector_nr = install->misc.holy_filesystem * (install->misc.holy_unit_size / 512);

			      sync();
			      unsigned long long start_partition;
			      if (get_from_statbuf_st_dev (statbuf_st_dev, buffname, &start_partition, 0) == 0)
				  geocmd = buffname;
			      if ((tst_FIBMAP = open (buffname, O_RDONLY /*| O_EXCL*/)) == -1) {
				  ERROR ("Checking if FIBMAP works on '%s': cannot open device, abort.\n", buffname);
				  perror (buffname);
				  unlink (fout_name);
				  return 1;
				  }
			      if (lseek64 (tst_FIBMAP, end_chain * unit_size, SEEK_SET) != end_chain * unit_size) {
				  ERROR ("Checking if FIBMAP works on '%s': cannot lseek64 to %llu, abort.\n", buffname, end_chain * unit_size);
				  perror (buffname);
				  unlink (fout_name);
				  return 1;
				  }
			      if (read (tst_FIBMAP, &data, 1) != 1) {
				  ERROR ("Checking if FIBMAP works on '%s': cannot read one byte, abort.\n", buffname);
				  perror (buffname);
				  unlink (fout_name);
				  return 1;
				  }
			      if (data != 0x5F) {
				  ERROR ("Checking if FIBMAP works on '%s': byte read is not infs_filemap() filler at %llu*%u: 0x5F but 0x%X, abort.\n", buffname, end_chain, unit_size, data);
				  ERROR ("Try a umount / remount cycle of %s to flush caches.\n", buffname);
				  perror (buffname);
				  unlink (fout_name);
				  return 1;
				  }
			      if (close (tst_FIBMAP) != 0) {
				  ERROR ("Checking if FIBMAP works on '%s': cannot close device, abort.\n", buffname);
				  perror (buffname);
				  unlink (fout_name);
				  return 1;
				  }
			      EXTRA ("Extra: filesystem on %s %s holes (unit size %u), and FIBMAP works.\n", buffname, install->misc.holy_filesystem? "supports": "doesn't support", unit_size);
			      if (*diskcmd == 0)
				  *diskcmd = dflt[i].geocmd;
			      unlink (fout_name);
			      if (*positioncmd == 0)
				  *positioncmd = "infs";
			      EXTRA ("Extra: set '--geometry=%s', with '--disk=%s'.\n", geocmd, *diskcmd);
			      if (!strncmp (buffname, "/dev/md", strlen ("/dev/md"))) {
				  ERROR("Installing on RAID %s still not supported\n", buffname); return 1;
				  return 1;
				  }
				else if (*mbr_device == 0) {
				  static char buffnamedevice[256];
				  get_mbrname_from_devicename (buffnamedevice, buffname);
				  EXTRA ("Extra: set previously empty '--mbr-device=%s'.\n", buffnamedevice);
				  *mbr_device = buffnamedevice;
				  }
			      if (start_partition == 0)
				*partition_index_ptr = 0;
			      }
			    else {
			      ERROR ("test file creation '%s' error, abort.\n", fout_name);
			      return 1;
			      }
			  }
			else
			  EXTRA ("\nExtra: infs file extension but not absolute pathname: ");
		      }
		  break;
		  }
	      }
	  if (geocmd == 0)
	      EXTRA ("failed.\n");
	  }
      }

#if defined (HDIO_GETGEO_BIG) || defined (HDIO_GETGEO)
  if (geocmd == 0) {
      struct stat fout_stat;

      EXTRA ("Extra: '--geometry=' not given, look if '%s' is a device: ", fout_name);
      if (stat (fout_name, &fout_stat) != 0) {
	  EXTRA ("NO, cannot stat\n");
	  }
	else if (S_ISREG(fout_stat.st_mode)) {
	  EXTRA ("is not a device\n");
	  }
#if defined (MAJOR) && defined (MINOR) /* FIXME: how to recognise /dev/null on Fedora 3? */
	else if (MAJOR(fout_stat.st_rdev) == 1 && MINOR(fout_stat.st_rdev) == 3 && S_ISCHR(fout_stat.st_mode)) {
	  EXTRA ("recognised /dev/null, do not probe its length\n");
	  }
#endif
	else {
	  const char *ptr = fout_name;
	  while (*ptr) ptr++;
	  ptr --;
	  if (*ptr >= '0' && *ptr <= '9') {
	      geocmd = fout_name;
	      EXTRA ("Yes and finish by a digit, set '--geometry=%s'\n", geocmd);
	      }
	    else {
	      EXTRA ("Yes but does not finish by a digit (it is not a partition)\n");
//	      EXTRA ("Hint: you may want to add a \"--geometry=%s\" to generate a floppy image (no partition);\n", fout_name);
//	      EXTRA ("Hint: and add \"--mbr-device=%s\" to generate a disk image (only first partition defined).\n", fout_name);
	      geocmd = fout_name;
	      if (*partition_index_ptr == 0 && *mbr_device)
		  *partition_index_ptr = "4";
	      }
	  }
      }
#endif

  if (geocmd == 0) {
      EXTRA ("Extra: '--geometry=' not given, start /dev/[hs]d[a-z]0 or /dev/mmcblk[0-9]p0 recognition: ");
      unsigned blen = strlen_beer_device(fout_name);
      if (blen != 0) {
	  static char devname[32];

	  devname[--blen] = '\0';
	  if (!strncmp (fout_name, "/dev/mmcblk", strlen ("/dev/mmcblk") - 1))
	      devname[--blen] = '\0';
	  while (blen--)
	      devname[blen] = fout_name[blen];
	  geocmd = fout_name;
	  EXTRA ("YES, the target device is a BEER partition description, set '--geometry=%s'\n", geocmd);
	  if (*mbr_device == 0) {
	      *mbr_device = devname;
	      EXTRA ("Extra: mbr_device not given, set it to %s\n", *mbr_device);
	      }
	  if (*beer_device == 0) {
	      *beer_device = devname;
	      EXTRA ("Extra: beer_device not given, set it to %s\n", *beer_device);
	      }
	  if (*diskcmd == 0 && fout_name[5] == 'h') { /* not for SCSI nor SD cards */
	      static char diskdefault[] = "ide:hd?";

	      diskdefault[6] = fout_name[7];
	      *diskcmd = diskdefault;
	      EXTRA ("Extra: beer_device not given and target /dev/hdx, set disk to --disk=%s\n", *diskcmd);
	      }
	  }
	else
	  EXTRA ("NO, not a BEER partition description.\n");
      }


  if (geocmd && *(unsigned *)geocmd == *(unsigned *)"/dev") {
      unsigned blen = strlen_beer_device(geocmd);

      if (blen) { /* BEER partition */
	  /* /dev/mmcblk0 is main device, /dev/mmcblk0p1 is partition... checking /dev/mmcblk?p0 */
	  /* copy '/dev/hda' up to disk letter */
	  static char basedevname[32];
	  basedevname[--blen] = '\0';
	  if (!strncmp (geocmd, "/dev/mmcblk", strlen ("/dev/mmcblk") - 1))
	      basedevname[--blen] = '\0';
	  while (blen--)
	      basedevname[blen] = geocmd[blen];
	  devname = basedevname;
	  }
	else
	  devname = geocmd;
      EXTRA ("Extra: set devname to '%s' to get partition size & offset and disk parameters.\n",
		devname);
      }

  if (devname) {
      int fout_desc;
      long long length;
#ifdef FDGETPRM
      struct floppy_struct fd_param;
#endif

      EXTRA ("Extra: geocmd beginning with '/', begin device geometry analysis.\n");
      if ((fout_desc = open (devname, O_RDONLY /*| O_EXCL*/)) == -1) {
	      /* O_EXCL on Linux means fail if already mounted.
		 at least when supermount is not used... */
	  ERROR ("cannot open device '%s'\n", devname);
	  perror (devname);
	  return 1;
	  }
	else if ((length = GET_FD_LENGTH(fout_desc)) == -1LL) {
	  ERROR ("cannot get length of device %s\n", devname);
	  close (fout_desc);
	  return 1;
	  }
#ifdef FDGETPRM
	else if (ioctl (fout_desc, FDGETPRM, &fd_param) >= 0) {
	  static const struct {
	      unsigned cylinders;
	      unsigned short heads, sectpertrack;
	      char *geocmd;
	      } stdfloppy[] = {
	      { 40, 2,  9, "floppy:360" }, { 80, 2, 15, "floppy:120" },
	      { 80, 2,  9, "floppy:720" }, { 80, 2, 18, "floppy:144" },
	      { 80, 2, 21, "floppy:168" }, { 80, 2, 36, "floppy:288" },
	      { 10, 2, 32, "floppy:320" } /* made up format, load quickly, bootchain[] not too many */
	      };
	  unsigned i = sizeof (stdfloppy) / sizeof (stdfloppy[0]);

	  EXTRA ("Extra: got FLOPPY geometry: %u/%u/%u total %u (sector size %llu bytes)\n",
			fd_param.track, fd_param.head, fd_param.sect,
			fd_param.size, (long long)length / fd_param.size);
	  while (i--) {
	      unsigned total_sector = stdfloppy[i].cylinders * stdfloppy[i].heads
						* stdfloppy[i].sectpertrack;
	      if (   fd_param.size  == total_sector
		  && fd_param.track == stdfloppy[i].cylinders
		  && fd_param.head  == stdfloppy[i].heads
		  && fd_param.sect  == stdfloppy[i].sectpertrack) {
		  geocmd = stdfloppy[i].geocmd;
		  install->partdisk.BytePerSector = length / total_sector;
		  EXTRA ("Extra: set geocmd to '%s' (default sector size: %u).\n",
				geocmd, install->partdisk.BytePerSector);
		  break;
		  }
	      }
	  if (geocmd == 0)
	      EXTRA ("unknown floppy format.\n");
	  if (*diskcmd == 0) {
	      *diskcmd = "BIOS:0x00,auto";
	      EXTRA ("Extra: '--disk=' not given, set to default: '--disk=%s'\n", *diskcmd);
	      }
	  }
#endif
	else {
	  unsigned long long partdisk_start_sector = 0;

	  if (strlen_beer_device(geocmd) == 0) { /* NOT for BEER partition */
	      if (get_device_start_sector (fout_desc, &partdisk_start_sector, 0) != 0) {
		  EXTRA ("Extra: cannot get device geometry (ioctl) for '%s' (length %llu * %u)\n",
				geocmd, (long long)length / 512, 512);
		  ERROR ("'--geometry=' not followed by a recognised word "
				"(dos or floppy or hd[a-z] or device) but by '%s'\n", geocmd);
		  close (fout_desc);
		  return 1;
		  }
	      }

	  if (setup_geometry_hd (geocmd, install, devname, length, partdisk_start_sector)) {
	      close (fout_desc);
	      return 1;
	      }
	  geocmd = 0;
	  if (*diskcmd == 0) {
	      if (partdisk_start_sector || *beer_device)
		  *diskcmd = "EBIOS:0x80,auto";
		else
		  *diskcmd = "EBIOS:0x00,auto";
	      EXTRA ("Extra: %s: '--disk=' not given, set to default: '--disk=%s'\n", __FUNCTION__, *diskcmd);
	      }
	  }
      if (fout_desc != -1)
	  close (fout_desc);
      }

  if (!geocmd) {
      if (!devname) {
	  ERROR ("not enought information to get geometry of the target\n");
	  return 1;
	  }
      }
    else if (geocmd[0] < '0' || geocmd[0] > '9') {
      /* geocmd does not start with a digit, try dos: or floppy: or ISO recognition: */
      if (   !strncasecmp (geocmd, "dos:", 4) || !strcasecmp (geocmd, "dos")
	  || !strcasecmp (geocmd, "PIC")  || !strcasecmp (geocmd, "BCD")) {
	   if (   geocmd[3] == '\0'
	       || geocmd[4] == '\0' /* still accept '--geometry=dos:' for '--geometry=dos:exe' */
	       || !strcasecmp (&geocmd[4], "exe")
	       || !strcasecmp (&geocmd[4], "EXE")
	       || !strcasecmp (&geocmd[4], "com")
	       || !strcasecmp (&geocmd[4], "COM")) {
	      if (*diskcmd == 0) {
		  *diskcmd = geocmd;
		  EXTRA ("Extra: recognised '--geometry=%s', '--disk=' not given, "
			 "set to: '--disk=%s'\n", geocmd, *diskcmd);
		  }
		else if (!strcmp (*diskcmd, geocmd)) {
		  EXTRA ("Extra: recognised '--geometry=%s', '--disk=%s' identical\n",
			 geocmd, *diskcmd);
		  }
		else {
		  ERROR ("'--geometry=%s' != '--disk=%s', different DOS access\n", geocmd, *diskcmd);
		  return 1;
		  }
	      }
	    else {
	      ERROR ("'--geometry=dos' not followed by ':com' nor ':exe' but by '%s'\n", &geocmd[4]);
	      return 1;
	      }
	  geocmd = 0;
	  }
	else if (!strncasecmp (geocmd, "floppy:", 7) || !strcasecmp (geocmd, "floppy")) {
	  const char *ptr = &geocmd[7], *dptr, *fptr;
	  unsigned tmp;

	  if (geocmd[6] == '\0')
	      tmp = 144;
	    else if (*ptr < '0' || *ptr > '9') {
	      ERROR ("'--geometry=floppy:' not followed by a digit\n");
	      return 1;
	      }
	    else
	      tmp = getdec (&ptr);
	  switch (tmp) {
	      /* Note: floppies with 2 sectors/track would have an odd number
	       * of root directory sectors to not waste one sector at the end
	       * of the floppy (one reserved sector at their beginning and
	       * two FAT so even number of sector for FATs).
	       * FIXME: what about 2.88 Mb floppies? */
	      /* Note: FDGETPRM may has set another bytePerSector as default,
	       * do not overwrite it here (do not set the last field of dptr)
	       */
	      case 360: dptr = "0720,0,2,09"; fptr = "FAT:0720,2,07,1,2,12,0x00,0xFD"; break;
	      case 120: dptr = "2400,0,2,15"; fptr = "FAT:2400,1,14,1,2,12,0x00,0xF9"; break;
	      case 720: dptr = "1440,0,2,09"; fptr = "FAT:1440,2,07,1,2,12,0x00,0xF9"; break;
	      /* WARNING: if you change sector/cluster on 1.44Mb floppy,
	       * or if --position=afterfs on a floppy (scandisk)
	       * you will also need to change (scandisk.exe) :-V-------------------V
						     "FAT:2880,2,14,1,2,12,0x00,0xF8" */
	      case 144: dptr = "2880,0,2,18"; fptr = "FAT:2880,1,14,1,2,12,0x00,0xF0"; break;
	      case 168: dptr = "3360,0,2,21"; fptr = "FAT:3360,1,14,1,2,12,0x00,0xF0"; break;
	      case 288: dptr = "5760,0,2,36"; fptr = "FAT:5760,2,28,1,2,12,0x00,0xF0"; break;
	      default:
		  ERROR ("'--geometry=floppy:' not followed by a recognised number: %u\n", tmp);
		  return 1;
	      case 320: dptr = "0640,0,2,32"; fptr = "FAT:0640,8,06,1,1,12,0x00,0xF0"; break;
	      }
	  EXTRA ("Extra: recognised '--geometry=floppy:%u', i.e. '--geometry=%s'\n", tmp, dptr);
	  geocmd = dptr;
	  if (*diskcmd == 0) {
	      *diskcmd = "BIOS:0x00";
	      EXTRA ("Extra: '--disk=' not given, set to: '--disk=%s'\n", *diskcmd);
	      }
	  if (*fscmd == 0) {
	      *fscmd = fptr;
	      EXTRA ("Extra: '--fs' not given, set to: '--fs=%s'\n", *fscmd);
	      }
	  }
	else if (*(unsigned *)geocmd == *(unsigned *)"ISO" || *(unsigned *)geocmd == *(unsigned *)"iso") {
	  struct stat fout_stat;
	  static char cmd[80];
	  if (stat (fout_name, &fout_stat) != 0) {
	      ERROR("Cannot stat file %s\n", fout_name);
	      return 1;
	      }
	  if (*fscmd == 0) {
	      *fscmd = "FAT:2400,1,14,1,2,12,0x00,0xF9";
	      EXTRA ("Extra: '--fs' not given, set to: '--fs=%s'\n", *fscmd);
	      }
	  unsigned nbsector = 720;
	  sscanf (*fscmd, "FAT:%u,", &nbsector);
	  EXTRA("Extra: ISO file %s size %llu 2048 bytes sectors, %llu 512 bytes sectors, increases by %u 512 bytes sectors\n",
		fout_name, (unsigned long long)fout_stat.st_size/2048, (unsigned long long)fout_stat.st_size/512, nbsector);
	  sprintf (cmd, "%llu,%llu,%u,%u,512,%llu,0",
		(unsigned long long)(fout_stat.st_size + nbsector*512)/512, (unsigned long long)fout_stat.st_size/512,
		install->misc.forced_Heads ?: 128, install->misc.forced_SectPerTrack ?: 32,
		(unsigned long long)(fout_stat.st_size + nbsector*512)/512);
	  geocmd = cmd;
	  }
	else {
	  PARAMERR ("'--geometry=' did not understand '%s', try '--geometry=floppy:144' ?\n", geocmd);
	  return 1;
	  }
     }

  if (geocmd) {
      if (geocmd[0] >= '0' && geocmd[0] <= '9') {
	  if (get_geometry (&geocmd, &install->partdisk) != 0) {
	      PARAMERR ("'--geometry=' did not understand first part, stop at '%c'\n", *geocmd);
	      return 1;
	      }
	  install->wholedisk = install->partdisk; /* as default */
	  if (*mbr_device && *geocmd == '\0' && install->wholedisk.start_sector == 0) {
	      install->partdisk.start_sector = install->wholedisk.sectpertrack;
	      install->partdisk.total_sector = install->wholedisk.total_sector - install->partdisk.start_sector;
	      }
	  if (*partition_index_ptr != 0) {
	      install->wholedisk.total_sector = install->partdisk.total_sector + install->partdisk.start_sector;
	      install->wholedisk.start_sector = 0;
	      }
	  if (*geocmd != '\0') {
	      /* "[,Dsize,Dstart,Head,Sector/Track,SectorSize] \n" */
	      if (get_geometry (&geocmd, &install->wholedisk) != 0) {
		  PARAMERR ("'--geometry=' did not understand second part, stop at '%c'\n", *geocmd);
		  return 1;
		  }
	      }
	  }
      if (*geocmd != '\0') {
	  PARAMERR ("'--geometry=' got trailing char '%c'\n", *geocmd);
	  return 1;
	  }
      }

/*
Extra:  HDIO_GETGEO ioctl: total_sector: 122040, start_sector: 0, cylinders: 1017, heads: 2, sectpertrack: 60.
Extra: something not perfect: total sector * 512 = 62484480, total len 62521344
Extra: device name /dev/sda, len 62521344, total_sector 122040, so sector size 512, pages of 8 sectors
Extra:  HDIO_GETGEO ioctl: start_sector: 0
Extra: Reporting wholedisk: start_sector = 0, total_sector 122040, partdisk: start_sector = 0, total_sector 122112
Extra: exit setup_geometry with: --geometry=122112,0,2,60,512,122040,0,2,60,512 (cylinder: 1017,1017)
 */
  if (*mbr_device != 0
//	&& install->partdisk.total_sector == install->wholedisk.total_sector
	&& install->partdisk.total_sector >= install->wholedisk.total_sector - install->wholedisk.sectpertrack
	&& install->partdisk.heads == install->wholedisk.heads
	&& install->partdisk.sectpertrack == install->wholedisk.sectpertrack
	&& install->partdisk.heads * install->partdisk.sectpertrack != 0
	&& install->partdisk.start_sector == 0 && install->wholedisk.start_sector == 0) {
      install->partdisk.total_sector = install->wholedisk.total_sector;
      EXTRA ("Extra: Want a MBR device on a whole disk, ");
      if (*partition_index_ptr != 0) {
	  install->partdisk.total_sector -= install->partdisk.total_sector % (install->partdisk.heads * install->partdisk.sectpertrack);
	  install->partdisk.total_sector -= install->partdisk.sectpertrack;
	  install->partdisk.start_sector = install->partdisk.sectpertrack;
	  EXTRA ("create partition %llu+%llu inside wholedisk %llu+%llu\n",
		install->partdisk.start_sector, install->partdisk.total_sector,
		install->wholedisk.start_sector, install->wholedisk.total_sector);
	  }
	else
	  EXTRA ("Do not create partition for infs\n");
      }

  install->misc.output_partition_offset = install->partdisk.start_sector - install->wholedisk.start_sector;
  if (install->misc.output_partition_offset != 0)
      EXTRA ("Extra: Reporting install->misc.output_partition_offset %llu\n",
		install->misc.output_partition_offset);
  EXTRA ("Extra: exit %s with: --geometry=%llu,%llu,%u,%u,%u,%llu,%llu,%u,%u,%u (cylinder: %u,%u)\n",
		__FUNCTION__,
		install->partdisk.total_sector, install->partdisk.start_sector,
		install->partdisk.heads, install->partdisk.sectpertrack,
		install->partdisk.BytePerSector,
		install->wholedisk.total_sector, install->wholedisk.start_sector,
		install->wholedisk.heads, install->wholedisk.sectpertrack,
		install->wholedisk.BytePerSector,
		install->partdisk.cylinders, /* cylinder used by partition */
		install->wholedisk.cylinders /* cylinder of the whole disk */
		);
  if (install->wholedisk.heads == 0 && install->wholedisk.sectpertrack != 0) {
      EXTRA ("Extra: in install->wholedisk, heads == 0 && sectpertrack != 0 so %s access and max %u sector/request\n",
		(install->wholedisk.sectpertrack & 0x80)? "absolute" : "partition relative",
		1 + (install->wholedisk.sectpertrack & 0x7F));
      }

 /* Do not reprint now because detecting IDE non LBA disks will need to overwrite
  * logical C/H/S by physical C/H/S:
  if (install->wholedisk.total_sector != 0) // else DOS loader
      REPRINTF("Reprint:    --geometry=%llu,%llu,%u,%u,%u,%llu,%llu,%u,%u,%u \\\n",
		install->partdisk.total_sector, install->partdisk.start_sector,
		install->partdisk.heads, install->partdisk.sectpertrack,
		install->partdisk.BytePerSector,
		install->wholedisk.total_sector, install->wholedisk.start_sector,
		install->wholedisk.heads, install->wholedisk.sectpertrack,
		install->wholedisk.BytePerSector);
  */
  return 0;
  }

static inline unsigned
get_ide (const char **ptrptr, struct ide_s *ide, unsigned short *diskindex, char *access_str)
  {
  const char *ptr = *ptrptr;

  if (ptr[0] == 'i' && ptr[1] == 'd' && ptr[2] == 'e' && IS_IDE_CHAR (ptr[3])) {
      /* -d=IDE:lba,ide0 */
      if (search_ide_io_addr (ptr[3] - '0', &ide->data_adr, &ide->dcr_adr) != 0) {
	  ERROR ("'-d=IDE:%s,ide%u' did not find I/O addresses in /proc/ioports\n",
			access_str, ptr[3] - '0');
	  return 1;
	  }
      ptr += 4;
      }
    else if (ptr[0] != '0' || ptr[1] != 'x') {
      PARAMERR ("'-d=IDE:%s,' did not find '0x' for the I/O address of IDE\n", access_str);
      return 1;
      }
    else {
      /* -d=IDE:lba,0x1F0,0x3F8 forced in hexadecimal */
      ide->data_adr = getnb (&ptr);
      if (ptr[0] != ',' || ptr[1] != '0' || ptr[2] != 'x') {
	  PARAMERR ("'-d=IDE:%s,0x%X' did not get ',0x' following\n",
			access_str, ide->data_adr);
	  return 1;
	  }
      if (ide->data_adr == 0) {
	  PARAMERR ("'-d=IDE:%s,0x0' the I/O address of IDE cannot be null\n",
			access_str);
	  return 1;
	  }
      ptr ++; /* skip ',' but keep it forced in hexa */
      ide->dcr_adr = getnb (&ptr);
      if (ptr[0] != ',') {
	  PARAMERR ("'-d=IDE:%s,0x%X,0x%X' did not get ',' following but '%c'\n",
			access_str, ide->data_adr, ide->dcr_adr, ptr[0]);
	  return 1;
	  }
      if (ide->dcr_adr == 0) {
	  PARAMERR ("'-d=IDE:%s,0x%X,0x0' the I/O address of IDE cannot be null\n",
			access_str, ide->data_adr);
	  return 1;
	  }
      }
  if (*ptr == ',') {
      ptr++;
      /* -d=IDE:lba,ide0,master/slave/0/1/2/3 */
      if (ptr[0] == 'm' && ptr[1] == 'a' && ptr[2] == 's' && ptr[3] == 't' && ptr[4] == 'e' && ptr[5] == 'r') {
	  *diskindex = 0;
	  ptr += sizeof ("master") - 1;
	  }
	else if (ptr[0] == 's' && ptr[1] == 'l' && ptr[2] == 'a' && ptr[3] == 'v' && ptr[4] == 'e') {
	  *diskindex = 1;
	  ptr += sizeof ("slave") - 1;
	  }
	else if (ptr[0] >= '0' || ptr[0] <= '3') {
	  *diskindex = ptr[0] - '0';
	  ptr += 1;
	  }
	else {
	  PARAMERR ("'-d=IDE:%s,0x%X,0x%X,' the master/slave has to be in between '0' and '3'\n",
			access_str, ide->data_adr, ide->dcr_adr);
	  return 1;
	  }
      }
  *ptrptr = ptr;
  return 0;
  }

static unsigned
setup_output (const char *diskcmd, enum gujin2_load_t *gujin2_load,
		struct geometry_s *geo, struct geometry_s *partgeo, struct ide_s *ide, struct fs_s *fs)
  {
  const char *ptr;

  if (diskcmd == 0) {
      PARAMERR ("no geometry or no disk description; unrecognised extension, not a partition, not a floppy, abort.\n");
      HINT ("for {auto|super}mount: Try on a command line: \"umount /mnt/floppy\"\n");
      HINT ("you may need to be root to umount and/or read some files in /proc\n");
      HINT ("If you want to force treating the HD as a floppy, set --geometry= to your target device\n");
      return 1;
      }
    else if ((ptr = pattern_match (diskcmd, "DOS", 0)) != 0) {
      if (   ptr[0] == '\0'
	  || (ptr[0] == ':' && ptr[1] == 'e' && ptr[2] == 'x' && ptr[3] == 'e' && ptr[4] == '\0')
	  || (ptr[0] == ':' && ptr[1] == 'E' && ptr[2] == 'X' && ptr[3] == 'E' && ptr[4] == '\0')) {
	  *gujin2_load = gujin2_load_dos_exe;
	  CHAIN ("Chain: set gujin2_load to gujin2_load_dos_exe\n");
	  REPRINTF("Reprint:    --disk=dos:exe\n");
	  return 0;
	  }
	else if ( (ptr[0] == ':' && ptr[1] == 'c' && ptr[2] == 'o' && ptr[3] == 'm' && ptr[4] == '\0')
	  || (ptr[0] == ':' && ptr[1] == 'C' && ptr[2] == 'O' && ptr[3] == 'M' && ptr[4] == '\0')) {
	  *gujin2_load = gujin2_load_dos_com;
	  CHAIN ("Chain: set gujin2_load to gujin2_load_dos_com\n");
	  REPRINTF("Reprint:    --disk=dos:com\n");
	  return 0;
	  }
	else {
	  PARAMERR ("did not recognise '--disk=dos[:exe|:com]' in --disk='%s'\n", diskcmd);
	  return 1;
	  }
      return 0;
      }
    else if ((ptr = pattern_match (diskcmd, "PIC", 0)) != 0) {
      CHAIN ("Chain: set gujin2_load to gujin2_load_PIC\n");
      REPRINTF("Reprint:    --disk=PIC\n");
      *gujin2_load = gujin2_load_PIC;
      return 0;
      }
    else if ((ptr = pattern_match (diskcmd, "BCD", 0)) != 0) { /* Boot CD */
      CHAIN ("Chain: set gujin2_load to gujin2_load_CDROM_noemul\n");
      *gujin2_load = gujin2_load_CDROM_noemul;
      geo->diskindex = 0x7F; /* needed for bootable CD */
      partgeo->diskindex  = 0x9F; /* just to not have 0x00 */
      return 0;
      }
    else if (   (ptr = pattern_match (diskcmd, "EBIOS", 0)) != 0
	     || (ptr = pattern_match (diskcmd, "BIOS", 0)) != 0) {
      if (ptr[0] != ':' || ptr[1] != '0' || ptr[2] != 'x') {
	  PARAMERR ("'-d=(E)BIOS' not followed by ':0x' in --disk='%s'\n", diskcmd);
	  return 1;
	  }
      ptr ++; /* skip ':' but keep it forced in hexa */
      partgeo->diskindex = getnb (&ptr);
      /* test -d=(E)BIOS:0x80,auto or -d=(E)BIOS:0x0,auto or -d=(E)BIOS:0x0,0x0 */
      if (ptr[0] == ',' && ptr[1] == 'a' && ptr[2] == 'u' && ptr[3] == 't' && ptr[4] == 'o') {
	  geo->diskindex = 0x7F;
	  }
	else if (ptr[0] == ',' && ptr[1] == '0' && ptr[2] == 'x') {
	  ptr ++; /* skip ',' but keep it forced in hexa */
	  geo->diskindex = getnb (&ptr);
	  }
	else if (*ptr == '\0') {
	  geo->diskindex = partgeo->diskindex;
	  }
	else {
	  PARAMERR ("'-d=(E)BIOS:0x%X' not at end nor followed by ',', followed by '%s' "
			"in --disk='%s'\n",
			geo->diskindex, ptr, diskcmd);
	  return 1;
	  }
     if (partgeo->diskindex > 0xFF || geo->diskindex > 0xFF) {
	  PARAMERR ("'-d=(E)BIOS:0x' too big BIOS number 0x%X,0x%X in --disk='%s'\n",
			partgeo->diskindex, geo->diskindex, diskcmd);
	  return 1;
	  }
      if (pattern_match (diskcmd, "BIOS", 0) != 0) {
	  *gujin2_load = gujin2_load_bios;
	  CHAIN ("Chain: --disk=BIOS:0x%X,0x%X\n", partgeo->diskindex, geo->diskindex);
	  }
	else {
	  *gujin2_load = gujin2_load_ebios;
	  CHAIN ("Chain: --disk=EBIOS:0x%X,0x%X\n", partgeo->diskindex, geo->diskindex);
	  }
      return 0;
      }
    else if ((ptr = pattern_match (diskcmd, "IDE", 0)) != 0) {
      if (ptr[0] != ':') {
	      // FIXME: not needed if geometry given...
	  PARAMERR ("'-d=IDE' not followed by ':' in --disk='%s'\n", diskcmd);
	  return 1;
	  }
      if (fs->FatSize == 32) {
	  PARAMERR ("'-d=IDE' not possible on FAT32 boot record, available space limited\n");
	  return 1;
	  }
      ptr++;
      /* TODO: support --disk=IDE:sda if possible */
      if (ptr[0] == 'h' && ptr[1] == 'd' && ptr[2] >= 'a' && ptr[2] <= 'z') {
	  int idenb;

	  EXTRA ("Extra: disk geometry to load Gujin2: %u/%u/%u (end of --geometry= line) may be changed soon...\n",
			geo->cylinders, geo->heads, geo->sectpertrack);
	  if (search_geometry_for_disk (ptr[2], geo, 0) != 0) { /* init complete geo field */
	      WARNING (" '-d=IDE:hd%c' search_geometry_for_disk failure\n", ptr[2]);
	      WARNING ("Extra: keeping end of second part of --geometry= line: %u,%u,512 (cylinder %u)\n",
			geo->heads, geo->sectpertrack, geo->cylinders);
/*	      return 1;	file does not exists in Debian. FIXME: When does this file exist??? */
	      }
	    else
	      EXTRA ("Extra: corrected end of second part of --geometry= line: %u,%u,512 (cylinder %u)\n",
			geo->heads, geo->sectpertrack, geo->cylinders);
	  if ((idenb = search_ide_for_disk (ptr[2])) < 0) {
	      ERROR ("'-d=IDE:hd%c' did not find this disk in /proc/ide.\n", ptr[2]);
	      return 1;
	      }
	  if (search_ide_io_addr (idenb, &ide->data_adr, &ide->dcr_adr) != 0) {
	      ERROR ("'-d=IDE:hd%c' did not find ide%d address in /proc/ioports.\n", ptr[2], idenb);
	      return 1;
	      }
	  if (search_masterslave (ptr[2], &geo->diskindex) != 0) {
	      ERROR ("'-d=IDE:hd%c' did not find master/slave status.\n", ptr[2]);
	      return 1;
	      }
	  if ((geo->total_sector = search_capacity_for_disk (ptr[0], ptr[2])) == 0) {
	      ERROR ("-d=IDE:hd%c' search_capacity_for_disk failure\n", ptr[2]);
	      return 1;
	      }
	  {
	  struct geometry_s g = {};
	  struct ide_capability_s cap_supported, cap_enabled;
	  /* This is at least needed to know c/h/s, lba or lba48 access: */
	  *gujin2_load = search_access (ptr[2], &g, &cap_supported, &cap_enabled);
	  EXTRA ("Extra: %s: got direct IDE geometry C/H/S = %u/%u/%u, total_sector %llu\n",
		__FUNCTION__, g.cylinders, g.heads, g.sectpertrack, g.total_sector);
	  if (geo->total_sector != g.total_sector) {
	      ERROR ("-d=IDE:hd%c' different capacity in /proc/.../capacity: %llu and /proc/.../identify: %llu\n",
			ptr[2], geo->total_sector, g.total_sector);
	      return 1;
	      }
	  if (   (g.cylinders != 0 && g.heads != 0 && g.sectpertrack != 0)
	      && (geo->cylinders != g.cylinders || geo->heads != g.heads || geo->sectpertrack != g.sectpertrack)) {
	      ERROR ("-d=IDE:hd%c' different PHYSICAL C/H/S in /proc/.../geometry: %u/%u/%u "
			"and /proc/.../identify: %u/%u/%u\n",
			ptr[2],
			geo->cylinders, geo->heads, geo->sectpertrack,
			g.cylinders, g.heads, g.sectpertrack);
	      return 1;
	      }
	  if (cap_supported.host_protected_area && cap_enabled.host_protected_area) {
	      EXTRA ("Extra: HPA is available and active, Gujin probably^Wmaybe already installed on this disk!\n");
#if 0
	  /* Life is not easy sometimes, this disk (or this BIOS) has
	   * "host_protected_area supported and active"
	   * even if Gujin did not set it... would be nice if inactive.
	   */
	      return 1;
#endif
	      }
	  if (cap_supported.DevConfOverlay && cap_enabled.DevConfOverlay) {
	      EXTRA ("Extra: DevConfOverlay is available and active, Gujin probably already installed on this disk!\n");
#if 0
	  /* Life is not easy sometimes, this disk (or this BIOS) has
	   * "DevConfOverlay is available and active"
	   * even if Gujin did not set it... would be nice if inactive.
	   */
	      return 1;
#endif
	      }
#if 0
	  if (cap_supported.SAOReservedAreaBoot && cap_enabled.SAOReservedAreaBoot) {
	      EXTRA ("Extra: SAORAB is available and active, do not know how to treat this disk, abort.\n");
	      return 1;
	      }
#endif
	  }
	  if (*gujin2_load == gujin2_load_error) {
	      ERROR ("'-d=IDE:hd%c' did not acheive autodetect in --disk='%s'\n", ptr[2], diskcmd);
	      return 1;
	      }
	  CHAIN ("Chain: -d=IDE:hd%c recognised: ", ptr[2]);
	  }
	else {
	  char *access_str;

	  if (ptr[0] == 'c' && ptr[1] == 'h' && ptr[2] == 's') {
	      *gujin2_load = gujin2_load_ide_chs;
	      access_str = "chs";
	      ptr = &ptr[3];
	      }
	    else if (ptr[0] == 'l' && ptr[1] == 'b' && ptr[2] == 'a' && ptr[3] == '4' && ptr[4] == '8') {
	      *gujin2_load = gujin2_load_ide_lba48;
	      access_str = "lba48";
	      ptr = &ptr[5];
	      }
	    else if (ptr[0] == 'l' && ptr[1] == 'b' && ptr[2] == 'a' && ptr[3] == '2' && ptr[4] == '8') {
	      *gujin2_load = gujin2_load_ide_lba;
	      access_str = "lba28";
	      ptr = &ptr[5];
	      }
	    else if (ptr[0] == 'l' && ptr[1] == 'b' && ptr[2] == 'a') {
	      *gujin2_load = gujin2_load_ide_lba;
	      access_str = "lba";
	      ptr = &ptr[3];
	      }
	    else {
	      PARAMERR ("'-d=IDE:' can only be followed by 'chs', 'lba', 'lba28', 'lba48' or a hd[a-z]\n");
	      return 1;
	      }
	  if (*ptr != ',') {
	      PARAMERR ("'-d=IDE:%s' not followed by ','\n", access_str);
	      return 1;
	      }
	  ptr += 1;

	  if (get_ide (&ptr, ide, &geo->diskindex, access_str) != 0)
	      return 1;

	  if (*ptr != '\0') {
	      PARAMERR ("'-d=IDE:%s,0x%X,0x%X,%u', not at end, found '%c' in --disk='%s'\n",
			access_str, ide->data_adr, ide->dcr_adr, geo->diskindex,
			*ptr, diskcmd);
	      return 1;
	      }

	  CHAIN ("Chain: ");
	  return 0;
	  }
      CHAIN ("--disk=IDE:%s,0x%X,0x%X,%u\n",
		(*gujin2_load == gujin2_load_ide_chs)
			? "chs"
			: ((*gujin2_load == gujin2_load_ide_lba48)? "lba48" : "lba28"),
		ide->data_adr,
		ide->dcr_adr,
		geo->diskindex);
      return 0;
      }
    else {
      PARAMERR ("'-d=' or '--disk=' has to be followed by either, "
		"'IDE', 'BIOS', 'EBIOS', or 'DOS', I got: --disk='%s'\n", diskcmd);
      return 1;
      }
  }

/*
 * FAT filesystem creation.
 * We know that:
 * nbTotalSector == nbReservedSector + nbFat * nbSectorInFat + nbSectorInRoot + nbDataSector
 * nbSectorInFat == (nbDataSector / SectorPerCluster) * aClusterSizeInBytes / SectorSize
 * Where aClusterSizeInBytes = 3/2 for FAT12, 2 for FAT16, 4 for FAT32.
 * We introduce:
 * nbDATandFATsector = nbTotalSector - nbReservedSector - nbSectorInRoot
 * NbClusterPerCluster = SectorPerCluster * SectorSize / aClusterSizeInBytes
 * So:
 * nbDATandFATsector == nbFat * nbSectorInFat + nbDataSector
 * nbSectorInFat = nbDataSector * aClusterSizeInBytes / (SectorPerCluster * SectorSize)
 * nbSectorInFat = (nbDATandFATsector - nbFat * nbSectorInFat) * aClusterSizeInBytes / (SectorPerCluster * SectorSize)
 * nbSectorInFat * (SectorPerCluster * SectorSize) / aClusterSizeInBytes = nbDATandFATsector - nbFat * nbSectorInFat
 * nbSectorInFat * (SectorPerCluster * SectorSize) / aClusterSizeInBytes + nbFat * nbSectorInFat = nbDATandFATsector
 * nbSectorInFat * (SectorPerCluster * SectorSize) + nbFat * nbSectorInFat * aClusterSizeInBytes = nbDATandFATsector * aClusterSizeInBytes
 * nbSectorInFat * (SectorPerCluster * SectorSize + nbFat * aClusterSizeInBytes) = nbDATandFATsector * aClusterSizeInBytes
 * nbSectorInFat = nbDATandFATsector * aClusterSizeInBytes / (SectorPerCluster * SectorSize + nbFat * aClusterSizeInBytes)
 * nbSectorInFat = nbDATandFATsector * 1 / (NbClusterPerCluster + nbFat)
 * nbSectorInFat = nbDATandFATsector / (NbClusterPerCluster + nbFat)
 *
 * Micro$oft got it wrong, but we try to stay compatible to get scandisk/chkdsk working
 *  (those tools do not use bootbefore->part1.NbSectorperFAT !!!):
 * They said they got: MSnbSectorInFat = nbDATandFATsector / (SectorPerCluster * 256 + nbFat)
 *
 * I tried to generate a filesystem which can be checked successfully with chkdsk/scandisk
 * of at least DOS 6.0 - but I am far to be sure of the following limits.
 * Note that it is quite easy for DOS FORMAT command to generate a filesystem which
 * _cannot_ be checked by scandisk (small partition sizes) - and it is quite easy
 * for scandisk to crash a perfectly valid FAT16 filesystem!
 * Look if mtool works, i.e. look if "mdir -a a:" or "mdir -a c:" see the files...
 *
 * Note that I am not so happy about those defaults, the FAT are
 * a lot too big for the intended usage... set it manually.
 *
 * Confirmed: MSDOS6- do not use NbSectorPerCluster neither, as described in the boot record,
 * they get it only from the size of the filesystem. I do not know if MSDOS6 compatibility
 * is really needed...
 */

static inline unsigned
default_NbSectorPerCluster (unsigned NbFsSector)
  {
  unsigned char NbSectorPerCluster;

  if (NbFsSector == 0) {
      FILESYS ("Error: %s: NbFsSector == 0\n", __FUNCTION__);
      return 0;
      }
#if 1 /* That is compatible with MSDOS 6 - */
    else if (NbFsSector <= 8 * 4085) /* limit to be checked, > 30960 */
      NbSectorPerCluster = 8; /* FAT12 else MSDOS 6 scandisk crash badly the FS */
    else {
      unsigned nbshr18 = NbFsSector >> 18;
      unsigned char nbshift = 0;

      while (nbshr18) {
	  nbshr18 >>= 1;
	  nbshift++;
	  }

      NbSectorPerCluster = 4 << nbshift;
      if (NbSectorPerCluster == 0) {
	  FILESYS ("Error: %s: probably too big a partition for a FAT16 filesystem.\n",
				__FUNCTION__);
	  NbSectorPerCluster = 128;
	  }
      }
#else /* That is no more compatible with MSDOS 6, but is resonable: */
    else if (NbFsSector <   0x8000)
      NbSectorPerCluster = 2; /* FAT12/FAT16 */
    else if (NbFsSector <  0x40000)
      NbSectorPerCluster = 8; /* FAT16 */
    else if (NbFsSector < 0x100000)
      NbSectorPerCluster = 32;
    else
      NbSectorPerCluster = 128;
#endif
  return NbSectorPerCluster;
  }

/* I know what you are looking for: a bug... scandisk/chkdsk report an error
 * or has corrected the filesystem ...
 * For instance, you can easily create (force it with --fs=362880 on a bigger partition):
 * Extra: entering setup_fs with fs->NbFsSector == 362880 and '--fs='
 * and Gujin calculates:
 * Extra: exiting setup_fs with '--fs=FAT:362880,8,11,1,2,16,0x80,0xF8'
 * We have:
 * Fs: NbFsSector 362880, NbReservedSector 1, NbRootSector 8, nbDATandFATsector 362871
 * Fs: if FatSize 16 and NbSectorPerCluster 8, then NbClusterPerCluster 2048, NbFatSector 178
 * So 178 sector per FAT, and scandisk/chkdsk of MSDOS 6 will change it to reflect 177 sector per FAT.
 * Note that scandisk/chkdsk do _NOT_ update the field bootbefore->NbSectorperFAT !!!
 * So 177 sectors per FAT enable us to store 177 * 512 / 2 = 45312 clusters
 *    178                                    178 * 512 / 2 = 45568 clusters
 * And we have:
 * Fs: so FAT16, nbSectorPerFAT 178, nbDATcluster 45314
 * So I think I am right, you need 178 sectors to allocate cluster number 45313.
 *
 * Note that I have seen scandisk/chkdsk doing error when the last block is allocated,
 * with --position=endfs, so maybe MSDOS never write there...
 * Confirmed: a number of clusters are un-writable at end of DOS 6.0- (chkdsk & scandisk
 * complains) but are OK on Win*. These clusters are not even mapped on the FAT, so the
 * FAT number of sector do not need to be as big as I calculate. The problem is that
 * DOS6- do not use the bootbefore->nbSectorPerFAT so cannot read this FAT filesystem...
 * This does not appear on a floppy - you need a real HD (or maybe descriptor 0xF8 and not 0xF0).
 * Oh well... people can do more study with --position=<starting cluster number> if needed.
 *
 * Note that if you clear the partition with --full, you can then do:
 * hexdump /dev/hdb3 > hdb3.hex
 * before and after, then you get an easy to read dump like:
$ diff hdb3.hex hdb3fixed.hex
 36c36
 < 0000240 0021 0022 0023 ffff 0000 0000 0000 0000
 ---
 > 0000240 0021 0022 0023 fff8 0000 0000 0000 0000
 39,40c39,44
 < 0016400 0000 0000 0000 0000 ffff ffff ffff ffff
 < 0016410 ffff ffff ffff ffff ffff ffff ffff ffff
 ---
 > 0016400 fff8 ffff 0003 0004 0005 0006 0007 0008
 > 0016410 0009 000a 000b 000c 000d 000e 000f 0010
 > 0016420 0011 0012 0013 0014 0015 0016 0017 0018
 > 0016430 0019 001a 001b 001c 001d 001e 001f 0020
 > 0016440 0021 0022 0023 fff8 0000 0000 0000 0000
 > 0016450 0000 0000 0000 0000 0000 0000 0000 0000
 42,47c46,48
 < 0016600 fff8 ffff 0003 0004 0005 0006 0007 0008
 < 0016610 0009 000a 000b 000c 000d 000e 000f 0010
 < 0016620 0011 0012 0013 0014 0015 0016 0017 0018
 < 0016630 0019 001a 001b 001c 001d 001e 001f 0020
 < 0016640 0021 0022 0023 ffff 0000 0000 0000 0000
 < 0016650 0000 0000 0000 0000 0000 0000 0000 0000
 ---
 > 002c600 4946 454c 3030 3130 4843 004b 0000 0000
 > 002c610 0000 0000 0000 9757 2e53 0002 2000 0002
 > 002c620 0000 0000 0000 0000 0000 0000 0000 0000
 * So bootbefore->nbSectorPerFAT is not changed by scandisk/chkdsk "recovery"...
 */
static /* inline */ unsigned
get_NbFatSector (struct fs_s *fs, unsigned nbDATandFATsector,
			unsigned FatSize, unsigned Bytepersector)
  {
  unsigned NbClusterPerCluster = (fs->NbSectorPerCluster * Bytepersector) * 8 / FatSize;
  unsigned NbFatSector = DIVROUNDUP (nbDATandFATsector, NbClusterPerCluster + fs->NbFat);
  static unsigned printed_once = 0;

  FILESYS ("Fs: if FatSize %u and NbSectorPerCluster %u, then NbClusterPerCluster %u, NbFatSector %u\n",
		FatSize, fs->NbSectorPerCluster, NbClusterPerCluster, NbFatSector);

  if (fs->nb_sector_FAT_wasted == 0xFFFFFFFF) {
      /* i.e. uninitialised, try to align clusters to cluster size by having FATs size % cluster size = 0,
		but it assumes the partition is itself aligned, and fs->NbReservedSector % cluster size = 0,
		and fs->NbRootSector % cluster size = 0, so manual control has to be possible */
      unsigned max_align = fs->NbSectorPerCluster;
      if (max_align > 4 * 4096)
	 max_align = 4 * 4096;
      unsigned towaste = max_align - (NbFatSector * fs->NbFat) % max_align;
      NbFatSector += towaste / fs->NbFat;
      FILESYS ("Fs: nb_sector_FAT_wasted parameter not used, try to align clusters, "
		"so use --nb_sector_FAT_wasted=%u\n", towaste / fs->NbFat);
      if (towaste && !printed_once++)
	  REPRINTF("--nb_sector_FAT_wasted=%u ", towaste / fs->NbFat);
      }
    else if (fs->nb_sector_FAT_wasted) {
      NbFatSector += fs->nb_sector_FAT_wasted;
      FILESYS ("Fs: Add nb_sector_FAT_wasted = %u sector so NbFatSector %u\n",
		fs->nb_sector_FAT_wasted, NbFatSector);
      if (!printed_once++)
	  REPRINTF("--nb_sector_FAT_wasted=%u ", fs->nb_sector_FAT_wasted);
      }
  return NbFatSector;
  }

static inline unsigned
get_total_sector (const char *name, unsigned long long *size, unsigned BytePerSector)
  {
#if 1

  int fd = open (name, O_RDONLY | O_EXCL);
  long long total_length;

  /* O_EXCL on Linux means fail if already mounted.
     at least when supermount is not used... */
  if (fd == -1) {
      PARAMERR ("%s: cannot open '%s'\n", __FUNCTION__, name);
      return 1;
      }
    else if ((total_length = GET_FD_LENGTH(fd)) == -1LL) {
      EXTRA ("disk length error, device too big?\n");
      close (fd);
      return 1;
      }
    else if (total_length == 0) {
      EXTRA ("disk length is null, device exists?\n");
      close (fd);
      return 1;
      }
  *size = total_length / BytePerSector;
  close (fd);
  return 0;

#else

  struct stat tmpstat;

  if (stat (name, &tmpstat) != 0) {
      EXTRA ("Extra: '--fs=FAT:%s': cannot stat '%s'\n", name, name);
      return 1;
      }
  EXTRA ("Extra: stat %s: st_size %llu, st_blksize %llu, st_blocks %llu\n",
	name,
	(unsigned long long)tmpstat.st_size,
	(unsigned long long)tmpstat.st_blksize,
	(unsigned long long)tmpstat.st_blocks);
// Is that a bug?
//Extra: stat /dev/hde1: st_size 0, st_blksize 4096, st_blocks 0
//Extra: stat /dev/hde: st_size 0, st_blksize 4096, st_blocks 0
//Extra: stat /dev/fd0: st_size 0, st_blksize 4096, st_blocks 0
  if (tmpstat.st_size == 0) {
      EXTRA ("Extra: '--fs=FAT:%s': stat size for '%s' is zero\n",
		name, name);
      return 1;
      }
  *size = tmpstat.st_size;
  return 0;

#endif
  }

static inline unsigned
setup_fs (const char *cmd, struct fs_s *fs,
	  const char *positioncmd, unsigned size_in_sector,
	  unsigned BytePerSector, unsigned short install_partdisk_diskindex,
	  unsigned long long output_partition_offset)
  {
  unsigned defautl;

  EXTRA ("Extra: entering %s with fs->NbFsSector == %llu and '--fs=%s'\n",
		__FUNCTION__, fs->NbFsSector, cmd);

  if (*cmd != '\0') {
      if (cmd[0] != 'F' || cmd[1] != 'A' || cmd[2] != 'T' || cmd[3] != ':') {
	  EXTRA ("Extra: '--fs=' not followed with 'FAT:' but with '%s'.\n", cmd);
	  return 1;
	  }
      cmd = &cmd[4];
      if (*cmd >= '0' && *cmd <= '9') {
	  unsigned long long tmp = getlnb (&cmd);
	  if (tmp > fs->NbFsSector && fs->NbFsSector != 0) {
	      ERROR ("'--fs=FAT:%llu' when fs->NbFsSector = %llu, "
			"refuse to create filesystem bigger than partition\n",
			tmp, fs->NbFsSector);
	      return 1;
	      }
	  fs->NbFsSector = tmp;
	  if (fs->NbFsSector == 0) {
	      EXTRA ("Extra: '--fs=FAT:0': NbFsSector cannot be zero\n");
	      return 1;
	      }
	  }
	else {
	  char name[strlen(cmd) + 1], *ptr = name;

	  for (;;) {
	      *ptr = *cmd;
	      if (*ptr == '\0')
		  break;
	      if (*cmd == ',') {
		  *ptr = '\0';
		  break;
		  }
	      ptr ++;
	      cmd ++;
	      }
	  if (get_total_sector (name, &fs->NbFsSector, BytePerSector) != 0)
	      return 1;
	  }
      }

#define GETFSCMD(cmd, field, field_default, zerovalid, strformat, strfield...) { \
  if (*cmd != '\0') {								\
      unsigned tmp;								\
										\
      if (*cmd != ',') {							\
	  EXTRA ("Extra: '" #strformat "' not followed by ',' but "		\
			"by '%c'\n", ## strfield, *cmd);			\
	  return 1;								\
	  }									\
      cmd ++;									\
      if (*cmd < '0' || *cmd > '9') {						\
	  EXTRA ("Extra: '" #strformat ",' not followed by a digit (for "	\
				#field ") but by '%c'\n", ## strfield, *cmd);	\
	  return 1;								\
	  }									\
      tmp = getnb (&cmd);							\
      if (!zerovalid && tmp == 0) {						\
	  EXTRA ("Extra: '" #strformat ",0': " #field " cannot "		\
			"be zero\n", ## strfield);				\
	  return 1;								\
	  }									\
      if (tmp > 256 * sizeof (fs->field)) {					\
	  EXTRA ("Extra: '" #strformat ",%u': " #field " too big, max "		\
			"%u\n", ## strfield, tmp, 256 * (unsigned)sizeof (fs->field));	\
	  return 1;								\
	  }									\
      fs->field = tmp;								\
      }										\
    else									\
      fs->field = field_default;						\
  }

 /* FAT16 supports a maximum of 65,524 clusters per volume.
     FAT32 volume must have a minimum of 65,527 clusters.
     Default FAT32 NbSectorPerCluster = 4KB only in between 257 MB and 8 GB */
  const unsigned NbMinSectorForDefaultFAT32 =  2 * 1024 * (1024 * 1024/BytePerSector);
  if (fs->NbFsSector > 4 * NbMinSectorForDefaultFAT32)
      defautl = 4 * 4096 / BytePerSector; /* 16 Kbytes cluster for FAT32 if > 8 Gb */
  else if (fs->NbFsSector >= NbMinSectorForDefaultFAT32)
      defautl = 4096 / BytePerSector; /* 4 Kbytes cluster for FAT32 by default */
  else if ((defautl = default_NbSectorPerCluster (fs->NbFsSector)) == 0)
      return 1;
  GETFSCMD (cmd, NbSectorPerCluster, defautl, 0, "--fs=FAT:%llu",
	fs->NbFsSector)

  unsigned default_NbRootSector;
  if (fs->NbFsSector < NbMinSectorForDefaultFAT32 && fs->NbFsSector / fs->NbSectorPerCluster < 65525) /* 65525 = 0xFFF5 */
      default_NbRootSector = 14;
    else
      default_NbRootSector = 0;
  GETFSCMD (cmd, NbRootSector, default_NbRootSector, 1, "--fs=FAT:%llu,%u",
	fs->NbFsSector, fs->NbSectorPerCluster)

  /* We have automatic "--nb_sector_FAT_wasted=" to align clusters, try to align start of the partition: */
  unsigned default_NbReservedSector;
  const unsigned roundsector = 4096 / BytePerSector;
  if (fs->NbRootSector != 0)
      default_NbReservedSector = roundsector - (fs->NbRootSector + fs->NbFsSector * fs->NbSectorPerCluster + output_partition_offset) % roundsector;
    else if (output_partition_offset % roundsector)
      default_NbReservedSector = 32 - output_partition_offset % roundsector;
    else
      default_NbReservedSector = 32;
  GETFSCMD (cmd, NbReservedSector, default_NbReservedSector, 0, "--fs=FAT:%llu,%u,%u",
	fs->NbFsSector, fs->NbSectorPerCluster, fs->NbRootSector)

  if (positioncmd) {
      EXTRA ("Extra: %s test positioncmd:'%s': ", __FUNCTION__, positioncmd);

      /* --position=[beforefs|afterfs], <startfs>|clusternb|endfs treated elsewhere */
      if (!strcasecmp (positioncmd, "beforefs")) {
	  /* Note that the boot sector may be written twice consecutively,
	   * but the second one may be the copy of the MBR.
	   * (fs->NbReservedSector is one by default).
	   */
	  if (size_in_sector + fs->NbReservedSector >= 0x10000) {
	      ERROR ("beforefs: bootloader size %u sectors shall fit in 16 bit number\n", size_in_sector);
	      return 1;
	      }
	  if (fs->FatSize == 32) {
	      ERROR ("'--position=beforefs' not supported on FAT32 multi-sector boot record\n");
	      return 1;
	      }
	  fs->NbReservedSector += size_in_sector;
	  EXTRA ("beforefs so increase fs->NbReservedSector by %u\n", size_in_sector);
	  }
	else if (!strcasecmp (positioncmd, "afterfs")) {
	  const char *pcmd = cmd;
	  static char buffer[80], *pbuf = buffer;

	  fs->NbFsSector -= size_in_sector;
	  EXTRA ("afterfs so decrease NbFsSector by %u", size_in_sector);
	  // FIXME: shall I do, to remove mtools warning:
	  // "Total number of sectors not a multiple of sectors per track!"
	  // fs->NbFsSector -= fs->NbFsSector % install->partdisk.sectpertrack;
	  do {
	      *pbuf++ = *pcmd++;
	      } while (*pcmd);
	  pbuf -= 4;
	  if (pbuf[0] == '0' && pbuf[1] == 'x' && pbuf[2] == 'F' && pbuf[3] == '0' && pbuf[4] == '\0') {
	      EXTRA (", and replace MediaDescriptor 0xF0 by 0xF8");
	      pbuf[3] = '8';
	      cmd = buffer;
	      }
	  EXTRA ("\n");
	  }
	else
	  EXTRA ("do nothing now\n");
      }

  GETFSCMD (cmd, NbFat, 2, 0, "--fs=FAT:%llu,%u,%u,%u",
	fs->NbFsSector, fs->NbSectorPerCluster, fs->NbRootSector, fs->NbReservedSector)

  {
  unsigned nbDATandFATsector = fs->NbFsSector - fs->NbReservedSector - fs->NbRootSector;
  unsigned NbFatSector = get_NbFatSector (fs, nbDATandFATsector, 12, BytePerSector);
  unsigned nbDATsector = nbDATandFATsector - fs->NbFat * NbFatSector;
  unsigned nbDATcluster = nbDATsector / fs->NbSectorPerCluster;
//  unsigned nbTrailingSector = nbDATsector - nbDATcluster * fs->NbSectorPerCluster;
  unsigned nbTrailingSector = nbDATsector % fs->NbSectorPerCluster;

  FILESYS ("Fs: NbFsSector %llu, NbReservedSector %u, NbRootSector %u, nbDATandFATsector %u\n",
		fs->NbFsSector, fs->NbReservedSector, fs->NbRootSector, nbDATandFATsector);

  if (nbDATcluster < 0xFF5) {
      FILESYS ("Fs: nbDATcluster %u < 4085 == 0xFF5, OK for FAT12\n", nbDATcluster);
      defautl = 12;
      }
    else {
      FILESYS ("Fs: nbDATcluster %u >= 4085 == 0xFF5 too big for FAT12\n", nbDATcluster);
      NbFatSector = get_NbFatSector (fs, nbDATandFATsector, 16, BytePerSector);
      nbDATsector = nbDATandFATsector - fs->NbFat * NbFatSector;
      nbDATcluster = nbDATsector / fs->NbSectorPerCluster;
      nbTrailingSector = nbDATsector % fs->NbSectorPerCluster;
      FILESYS ("Fs: if FAT16, nbSectorPerFAT %u, nbDATcluster %u (max 65525)\n",
		NbFatSector, nbDATcluster);
      if (nbDATcluster < 65525 && fs->NbRootSector != 0) { /* 65525 = 0xFFF5 */
	  defautl = 16;
	  }
	else if (fs->NbReservedSector < 8 || fs->NbRootSector != 0) {
	  ERROR ("Should be FAT32, but either NbReservedSector=%u != 32 or NbRootSector=%u != 0\n", fs->NbReservedSector, fs->NbRootSector);
	  return 1;
	  }
	else {
	  NbFatSector = get_NbFatSector (fs, nbDATandFATsector, 32, BytePerSector);
	  nbDATsector = nbDATandFATsector - fs->NbFat * NbFatSector;
	  nbDATcluster = nbDATsector / fs->NbSectorPerCluster;
	  nbTrailingSector = nbDATsector % fs->NbSectorPerCluster;
	  FILESYS ("Fs: so FAT32, nbSectorPerFAT %u, nbDATcluster %u\n",
		NbFatSector, nbDATcluster);
	  defautl = 32;
	  }
      }

  GETFSCMD (cmd, FatSize, defautl, 0, "--fs=FAT:%llu,%u,%u,%u,%u",
	fs->NbFsSector, fs->NbSectorPerCluster, fs->NbRootSector, fs->NbReservedSector,
	fs->FatSize)

  if (fs->FatSize != 12 && fs->FatSize != 16 && fs->FatSize != 32) {
      EXTRA ("Extra: FatSize can only be 12, 16 or 32 not %u!\n", fs->FatSize);
      return 1;
      }
  if (fs->FatSize == defautl) {
      fs->NbFat32Sector = NbFatSector;
      FILESYS ("Fs: FAT%u selected\n", fs->FatSize);
      }
    else {
      fs->NbFat32Sector = get_NbFatSector (fs, nbDATandFATsector, fs->FatSize, BytePerSector);
      if (fs->FatSize != 32 && fs->NbFat32Sector >= 0x10000) {
	  ERROR("Only FAT32 header can handle 32 bits number of FAT sectors (value 0x%X)", fs->NbFat32Sector);
	  return 1;
	  }
      FILESYS ("Fs: non default FAT%u selected, NbFatSector %u\n", fs->FatSize, fs->NbFat32Sector);
      }

  if (nbTrailingSector) {
      if (fs->FatSize != 32)
	  FILESYS ("Fs: FAT12/16 config leaves %u sectors, you should increase the number of root sectors by this amount (%u->%u)\n",
			nbTrailingSector, fs->NbRootSector, fs->NbRootSector + nbTrailingSector);
	else
	  FILESYS ("Fs: NbRootSector_default_used not set or FAT32, leave %u trailing sector\n", nbTrailingSector);
      }
  }

  GETFSCMD (cmd, PhysicalDriveNb, install_partdisk_diskindex, 1, "--fs=FAT:%llu,%u,%u,%u,%u,%u",
	fs->NbFsSector, fs->NbSectorPerCluster, fs->NbRootSector, fs->NbReservedSector,
	fs->FatSize, fs->NbFat)

  // FIXME: testing FAT size to set MediaDescriptor is not clean, but anyways floppies are disappering...
  // (fs->FatSize == 12)? 0xF0 : 0xF8
  // Some people use 0xF8 for non removeable, 0xF0 for removeable...
  GETFSCMD (cmd, MediaDescriptor, 0xF0 | ((fs->PhysicalDriveNb & 0x80)>>4), 1, "--fs=FAT:%llu,%u,%u,%u,%u,%u,0x%X",
	fs->NbFsSector, fs->NbSectorPerCluster, fs->NbRootSector, fs->NbReservedSector,
	fs->FatSize, fs->NbFat, fs->PhysicalDriveNb)

#undef GETFSCMD
  EXTRA ("Extra: exiting %s with '--fs=FAT:%llu,%u,%u,%u,%u,%u,0x%X,0x%X'\n",
	__FUNCTION__,
	fs->NbFsSector, fs->NbSectorPerCluster, fs->NbRootSector, fs->NbReservedSector,
	fs->NbFat, fs->FatSize, fs->PhysicalDriveNb, fs->MediaDescriptor);
  REPRINTF("--fs=FAT:%llu,%u,%u,%u,%u,%u,0x%X,0x%X \\\n",
	fs->NbFsSector, fs->NbSectorPerCluster, fs->NbRootSector, fs->NbReservedSector,
	fs->NbFat, fs->FatSize, fs->PhysicalDriveNb, fs->MediaDescriptor);

  return 0;
  }

static inline unsigned
setup_misc (struct misc_s *misc, const struct fs_s *fs, unsigned BytePerSector,
		unsigned uninstall_mbr_used, enum gujin2_load_t gujin2_load,
		unsigned filesize, const char *positioncmd,
		unsigned long long lba_start_partition)
  {
  unsigned cluster_position; /* not offseted by the first two invalid ones */

  misc->firstDataSectorInPartition = fs->NbReservedSector
		+ fs->NbFat * fs->NbFat32Sector + fs->NbRootSector;
  misc->nb_cluster_on_filesystem = (fs->NbFsSector - misc->firstDataSectorInPartition)
					/ fs->NbSectorPerCluster;

  misc->nbsectorinfullfile = DIVROUNDUP(filesize, BytePerSector);
  if (uninstall_mbr_used)
      misc->nbsectorinfullfile += 1;
  misc->nb_cluster_in_file = DIVROUNDUP (misc->nbsectorinfullfile, fs->NbSectorPerCluster);
  if (misc->nb_cluster_in_file >= misc->nb_cluster_on_filesystem) {
      EXTRA ("Extra: More cluster in file (%u) than in filesystem(%u)\n",
		misc->nb_cluster_in_file, misc->nb_cluster_on_filesystem);
      return 1;
      }
  /* Clear anyway the end of the cluster: */
  misc->nb_empty_sector_in_cluster = misc->nb_cluster_in_file * fs->NbSectorPerCluster
					- DIVROUNDUP (filesize, BytePerSector);
  if (uninstall_mbr_used) {
      if (misc->nb_empty_sector_in_cluster == 0) {
	  ERROR ("Should have planned space for uninstall_mbr, algorithm error!\n");
	  return 1;
	  }
      misc->nb_empty_sector_in_cluster --;
      }

  misc->nb_sector_at_end = fs->NbFsSector - misc->firstDataSectorInPartition
				- misc->nb_cluster_on_filesystem * fs->NbSectorPerCluster;

  if (positioncmd && *positioncmd != '\0') {
      if (!strcasecmp (positioncmd, "startfs"))
	  cluster_position = (fs->FatSize == 32) ? 1 : 0;
	else if (!strcasecmp (positioncmd, "endfs"))
	  cluster_position = misc->nb_cluster_on_filesystem - misc->nb_cluster_in_file;
	else if (positioncmd[0] >= '0' && positioncmd[0] <= '9') {
	  cluster_position = getnb (&positioncmd);
	  if (*positioncmd != '\0') {
	      ERROR ("'--position=' followed by number %u but found trailing char '%c'\n",
			cluster_position, *positioncmd);
	      return 1;
	      }
	  if (fs->FatSize == 32 && cluster_position == 0) {
	      ERROR ("'--position=%u on FAT32 where cluster 0 if the root directory\n", cluster_position);
	      return 1;
	      }
	  }
	else if (!strcasecmp (positioncmd, "beforefs")) {
	  misc->begin_file_sector_nr = 0;
	  misc->nb_empty_sector_in_cluster = 0;
	  /* do not chain the copy of the start sector: */
	  misc->lba_start_chain = lba_start_partition + 1;
	  /* -1 for the start sector (the uninstall_mbr is never chained): */
	  misc->lba_end_chain = misc->lba_start_chain + DIVROUNDUP (filesize, BytePerSector);
	  /* If BytePerSector > 512, chain the first sector to calculate the end of the sector checksum: */
	  if (BytePerSector == 512) misc->lba_start_chain ++;
	  EXTRA ("Extra: beforefs: lba_start_chain %llu, lba_end_chain %llu, nb_sector_at_end %u\n",
			misc->lba_start_chain, misc->lba_end_chain, misc->nb_sector_at_end);

	  /* note: if nb_cluster_in_file == 0, cluster_offset not used */
	  misc->nb_cluster_in_file = 0; /* i.e. not in FS */
	  misc->nb_cluster_at_end = misc->nb_cluster_on_filesystem - misc->nb_cluster_in_file;
	  REPRINTF("Reprint:    --position=beforefs \\\n");
	  return 0;
	  }
	else if (!strcasecmp (positioncmd, "afterfs")) {
	  misc->begin_file_sector_nr = fs->NbFsSector;
	  /* do not chain the copy of the start sector: */
	  misc->lba_start_chain = lba_start_partition + fs->NbFsSector - misc->nb_sector_at_end;
	  /* misc->nb_sector_at_end -= DIVROUNDUP (filesize, BytePerSector); */
	  /* -1 for the start sector (the uninstall_mbr is never chained): */
	  misc->lba_end_chain = misc->lba_start_chain + DIVROUNDUP (filesize, BytePerSector);
	  /* If BytePerSector > 512, chain the first sector to calculate the end of the sector checksum: */
	  if (BytePerSector == 512) misc->lba_start_chain ++;
	  EXTRA ("Extra: afterfs: lba_start_chain %llu, lba_end_chain %llu, nb_sector_at_end %u\n",
			misc->lba_start_chain, misc->lba_end_chain, misc->nb_sector_at_end);

	  /* note: if nb_cluster_in_file == 0, cluster_offset not used */
	  misc->nb_cluster_in_file = 0; /* i.e. not in FS */
	  misc->nb_cluster_at_end = misc->nb_cluster_on_filesystem - misc->nb_cluster_in_file;
	  REPRINTF("Reprint:    --position=afterfs \\\n");
	  return 0;
	  }
	else {
	  ERROR ("'--position=' unrecognised parameter '%s'\n", positioncmd);
	  return 1;
	  }
      }
    else
      cluster_position = (fs->FatSize == 32) ? 1 : 0;

  misc->nb_cluster_at_begin = cluster_position;
  misc->nb_cluster_at_end = misc->nb_cluster_on_filesystem - misc->nb_cluster_in_file;
//  if (fs->FatSize == 32)
//      misc->nb_cluster_at_end -= 1; /* root cluster */
  if (misc->nb_cluster_at_end >= misc->nb_cluster_at_begin) {
      misc->nb_cluster_at_end -= misc->nb_cluster_at_begin;
      EXTRA ("Extra: --position=%u, so nb_cluster: at_begin %u, in_file %u, at_end %u.\n",
			cluster_position, misc->nb_cluster_at_begin,
			misc->nb_cluster_in_file, misc->nb_cluster_at_end);
      }
    else {
      EXTRA ("Extra: --position=%u, limiting nb_cluster_at_begin from %u to %u, nb_cluster_at_end = 0.\n",
			cluster_position, misc->nb_cluster_at_begin, misc->nb_cluster_at_end);
      misc->nb_cluster_at_begin = misc->nb_cluster_at_end;
      misc->nb_cluster_at_end = 0;
      }

  if (misc->nb_cluster_at_end == 0)
      REPRINTF("Reprint:    --position=endfs \\\n");
    else if (cluster_position != ((fs->FatSize == 32) ? 1 : 0))
      REPRINTF("Reprint:    --position=%u \\\n", cluster_position);
  misc->begin_file_sector_nr = misc->firstDataSectorInPartition
				+ misc->nb_cluster_at_begin * fs->NbSectorPerCluster;

  /* do not chain the copy of the start sector: */
  misc->lba_start_chain = lba_start_partition + misc->begin_file_sector_nr;
  /* -1 for the start sector (the uninstall_mbr is never chained): */
  misc->lba_end_chain = misc->lba_start_chain + DIVROUNDUP (filesize, BytePerSector);
  /* If BytePerSector > 512, chain the first sector to calculate the end of the sector checksum: */
  if (BytePerSector == 512) misc->lba_start_chain ++;
  return 0;
  }

/*
 * Video/serial stuff:
 */
static inline unsigned
force_serialconf (const char *serialcom, serialconf_t *serialconf, unsigned short *serialport)
  {
  unsigned port, speed, nbbit, nbstop;
  char parity;
  const char *cmd = serialcom;

  if (*cmd < '0' || *cmd > '9') {
      PARAMERR ("reading serial parameter '%s', cannot get COM port number, "
		      "abort.\n", serialcom);
      return 1;
      }
  port = getnb (&cmd);
  if (port < 1 || port > 4) {
      PARAMERR ("no such BIOS COM%d serial port, abort.\n", port);
      return 1;
      }
    else {
      *serialconf = (serialconf_t) {
		.eightbit =	1,
		.constant_1 =	1,
		.doublestop =	0,
		.parity =	0,
		.even =		1,
		.speed =	B9600,
		.constant_0 =	0,
		};
      *serialport = port;	/* 1 based, 0 means screen */
      }
  if (*cmd != '\0') {
      if (*cmd != ',') {
	  PARAMERR ("no comma but '%c' in '%s', abort.\n", *cmd, serialcom);
	  return 1;
	  }
      cmd++;
      }

  if (*cmd != '\0') {
      if (*cmd < '0' || *cmd > '9') {
	  PARAMERR ("not a digit: '%c', abort.\n", *cmd);
	  return 1;
	  }
      speed = getnb (&cmd);
      switch (speed) {
	  case  110: serialconf->speed = B110; break;
	  case  150: serialconf->speed = B150; break;
	  case  300: serialconf->speed = B300; break;
	  case  600: serialconf->speed = B600; break;
	  case 1200: serialconf->speed = B1200; break;
	  case 2400: serialconf->speed = B2400; break;
	  case 4800: serialconf->speed = B4800; break;
	  case 9600: serialconf->speed = B9600; break;
	  default:
	      PARAMERR ("no such BIOS baud rate: %d, abort.\n", speed);
	      return 1;
	  }
      if (*cmd != '\0') {
	  if (*cmd != ',') {
	      PARAMERR ("no comma but '%c' in '%s', abort.\n", *cmd, serialcom);
	      return 1;
	      }
	  cmd++;
	  }
      }

  if (*cmd != '\0') {
      parity = *cmd++;
      if (parity != 'n' && parity != 'e' && parity != 'o') {
	  PARAMERR ("no such BIOS parity: %c, abort.\n", parity);
	  return 1;
	  }
	else {
	  if (parity == 'n')
	      serialconf->parity = 0;
	    else {
	      serialconf->parity = 1;
	      if (parity == 'e')
		  serialconf->even = 1;
		else
		  serialconf->even = 0;
	      }
	  }
      if (*cmd != '\0') {
	  if (*cmd != ',') {
	      PARAMERR ("no comma but '%c' in '%s', abort.\n", *cmd, serialcom);
	      return 1;
	      }
	  cmd++;
	  }
      }

  if (*cmd != '\0') {
      if (*cmd < '0' || *cmd > '9') {
	  PARAMERR ("not a digit: '%c', abort.\n", *cmd);
	  return 1;
	  }
      nbbit = getnb (&cmd);
      if (nbbit != 8 && nbbit != 7) {
	  PARAMERR ("no such BIOS bit/char: %d, abort.\n", nbbit);
	  return 1;
	  }
	else {
	  if (nbbit == 8)
	      serialconf->eightbit = 1;
	    else
	      serialconf->eightbit = 0;
	  }
      if (*cmd != '\0') {
	  if (*cmd != ',') {
	      PARAMERR ("no comma but '%c' in '%s', abort.\n", *cmd, serialcom);
	      return 1;
	      }
	  cmd++;
	  }
      }

  if (*cmd != '\0') {
      if (*cmd < '0' || *cmd > '9') {
	  PARAMERR ("not a digit: '%c', abort.\n", *cmd);
	  return 1;
	  }
      nbstop = getnb (&cmd);
      if (nbstop != 1 && nbstop != 2) {
	  PARAMERR ("no such BIOS stop length: %d, abort.\n", nbstop);
	  return 1;
	  }
	else {
	  if (nbstop == 2)
	      serialconf->doublestop = 1;
	    else
	      serialconf->doublestop = 0;
	  }
      if (*cmd != '\0') {
	  PARAMERR ("not at end after nb stop bits: found '%c', abort.\n", *cmd);
	  return 1;
	  }
      }

  EXTRA ("Extra: configuring for serial interface on COM%u, %u,%c,%u,%u.\n",
	*serialport,
	((unsigned short []){110,150,300,600,1200,2400,4800,9600})
			[serialconf->speed],
	serialconf->parity ? (serialconf->even ? 'e' : 'o') : 'n',
	serialconf->eightbit? 8 : 7,
	serialconf->doublestop? 2 : 1
	);
  REPRINTF("Reprint:    --serial=COM%u,%u,%c,%u,%u \\\n",
	*serialport,
	((unsigned short []){110,150,300,600,1200,2400,4800,9600})
			[serialconf->speed],
	serialconf->parity ? (serialconf->even ? 'e' : 'o') : 'n',
	serialconf->eightbit? 8 : 7,
	serialconf->doublestop? 2 : 1
	);
  return 0;
  }

inline unsigned
VIDEO_mode_is_valid (unsigned char VGA_valid_mode[16], unsigned char mode)
  {
  return !!(VGA_valid_mode[mode / 8] & (1 << (mode % 8)));
  }

inline unsigned VIDEO_mode_max (void)
  {
   return 0x7F;
  }

static inline void
listmode (unsigned char VGA_valid_mode[16])
  {
  unsigned short start = 0, cpt, firsttime = 1,
		 valid = VIDEO_mode_is_valid(VGA_valid_mode, 0);

  for (cpt = 1; cpt <= VIDEO_mode_max(); cpt++) {
      if (   cpt == VIDEO_mode_max()
	  || VIDEO_mode_is_valid(VGA_valid_mode, cpt) != valid) {
	  if (valid == 0)
	      start = cpt;
	    else {
	      const char *format;
	      if (!firsttime)
		  EXTRA (", ");
	      firsttime = 0;
	      if (start <= cpt-2) {
		  if (start == cpt-2)
		      format = "0x%X, 0x%X";
		    else
		      format = "0x%X..0x%X";
		  }
		else
		  format = "0x%X";
	      EXTRA (format, start, (cpt == VIDEO_mode_max())? cpt : cpt-1);
	      }
	  valid = !valid;
	  }
      }
  if (firsttime)
      EXTRA("NONE!");
  EXTRA (".\n");
  }

static inline unsigned
init_valid_video_mode (unsigned char VGA_valid_mode[16],
			const char *str, const char *fullparam)
  {
  unsigned val;

  if (*str < '0' || *str > '9') {
      PARAMERR ("reading initial number in %s, abort.\n", fullparam);
      return 1;
      }
  val = getnb (&str);
  if (val >= 0x80) {
      PARAMERR ("reading parameter %s, out of range: "
		"0x%X, abort.\n", fullparam, val);
      return 1;
      }
  VGA_valid_mode[val/8] &= ~(1 << ((val) % 8));
  while (*str) {
      unsigned char tmp = *str++;
      unsigned other;
      if (tmp != ',' && tmp != '-') {
	  PARAMERR ("Error reading parameter %s, not comma or minus"
		  " separator but '%c', abort.\n", fullparam, tmp);
	  return 1;
	  }
      if (*str < '0' || *str > '9') {
	  PARAMERR ("reading parameter %s after '%c', abort.\n",
			fullparam, tmp);
	  return 1;
	  }
      other = getnb (&str);
      if (other >= 0x80) {
	  PARAMERR ("reading parameter %s after '%c', out of range: 0x%X, abort.\n",
			fullparam, tmp, other);
	  return 1;
	  }
      if (tmp == ',')
	  VGA_valid_mode[other/8] &= ~(1 << ((other) % 8));
	else {
	  if (other <= val) {
	      PARAMERR ("reading parameter %s after '%c', invalid range, abort.\n",
			fullparam, tmp);
	      return 1;
	      }
	  while (val <= other) {
	      VGA_valid_mode[val/8] &= ~(1 << ((val) % 8));
	      val ++;
	      }
	  if (*str == '-') {
	      WARNING ("double minus in '%s'!\n", fullparam);
	      }
	  }
      }
  EXTRA ("Extra: '--exclude_video_mode=' gives up to now: ");
  listmode (VGA_valid_mode);
  return 0;
  }

/*
 * gujin_param stuff:
 */
static inline void set_default_cmdline (gujin_param_t *gujin_param, unsigned no_gujin_cmd_flag)
  {
  FILE *file_cmdline;
  if ((file_cmdline = fopen ("/proc/cmdline", "r")) == NULL) {
      PROC ("Proc: cannot open '/proc/cmdline' for reading!");
      return;
      }

#define COMMANDFILE
#ifdef COMMANDFILE
  char gujin_cmd_buffer[1024], *gujin_cmd_ptr = gujin_cmd_buffer;
  char gujin_cmd_boot_image[128], *gujin_cmd_boot_image_ptr = gujin_cmd_boot_image;
  *gujin_cmd_ptr = '\0';
  *gujin_cmd_boot_image_ptr = '\0';
#endif

  char line[1024], *ptr = line, *ptrend, *dst = gujin_param->extra_cmdline;
  int size = fread (line, 1, sizeof (line) - 1, file_cmdline);

  if (line[size-1] == '\n')
      size--;
  line[size] = '\0';
  fclose (file_cmdline);

  PROC ("Proc: initialise cmdline from /proc/cmdline: \"%s\"\n", line);
  /* try to save space (else overflow) and do not re-write twice parameters if
	Gujin has already loaded this kernel... */
  while (*ptr) {
      while (*ptr == ' ')
	  ptr++;
      ptrend = ptr;
      while (*ptrend != '\0' && *ptrend > ' ')
	  ptrend++;

      /* First forget all parameters automatically generated by Gujin: */
      if (!strncmp (ptr, "video=", sizeof ("video=") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      if (!strncmp (ptr, "keyboard=", sizeof ("keyboard=") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      if (!strncmp (ptr, "NumLock=", sizeof ("NumLock=") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      if (!strncmp (ptr, "mouse=", sizeof ("mouse=") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      if (!strncmp (ptr, "COLS=", sizeof ("COLS=") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      if (!strncmp (ptr, "TZ=", sizeof ("TZ=") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      if (!strncmp (ptr, "LINES=", sizeof ("LINES=") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      if (!strncmp (ptr, "lang=", sizeof ("lang=") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      /* removes automatic "root=/dev/hda3" but not "root=304" and not "root=/dev/mapper/": */
      if (!strncmp (ptr, "root=/dev/", sizeof ("root=/dev/") - 1)
		&& strncmp (ptr, "root=/dev/mapper/", sizeof ("root=/dev/mapper/") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
#ifdef COMMANDFILE
      /* Goes into /boot/gujin.cmd */
      if (!strncmp (ptr, "root=/dev/mapper/", sizeof ("root=/dev/mapper/") - 1)) {
	  while (ptr < ptrend)
	      *gujin_cmd_ptr++ = *ptr++;
	  *gujin_cmd_ptr++ = ' ';
	  continue;
	  }
#endif
      if (!strncmp (ptr, "ide", sizeof ("ide") - 1) && ptr[3] >= '0' && ptr[3] <= '9' && ptr[4] == '=') {
	  ptr = ptrend;
	  continue;
	  }
      if (!strncmp (ptr, "BOOT_IMAGE", sizeof ("BOOT_IMAGE") - 1)) {
#ifdef COMMANDFILE
	  if (*gujin_cmd_boot_image_ptr == '\0') { /* only the first BOOT_IMAGE= if multiple */
	      char *last_slash = ptrend;
	      while (*last_slash != '/' && last_slash > ptr)
		  last_slash--;
	      if (last_slash <= ptr)
		  last_slash = ptr + sizeof ("BOOT_IMAGE=") - 1;
		else
		  last_slash++;
	      /* We stop also after seeing a minus and a dot, complete with star:
	      while (last_slash < ptrend)
		  *gujin_cmd_boot_image_ptr++ = *last_slash++; */
	      unsigned char dot_seen = 0, minus_seen = 0;
	      while (last_slash < ptrend && !(dot_seen && minus_seen)) {
		  if (*last_slash == '.')
		      dot_seen = 1;
		  if (*last_slash == '-')
		      minus_seen = 1;
		  *gujin_cmd_boot_image_ptr++ = *last_slash++;
		  }
	      if (dot_seen && minus_seen)
		  *gujin_cmd_boot_image_ptr++ = '*';
	      *gujin_cmd_boot_image_ptr++ = '\0';
	      *gujin_cmd_boot_image_ptr = 'X'; /* only the first BOOT_IMAGE= if multiple */
	      }
#endif
	  ptr = ptrend;
	  continue;
	  }
      if (!strncasecmp (ptr, "vga=", sizeof ("vga=") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      /* name of the kernel? */
      if (!strncasecmp (ptr, "bzImage", sizeof ("bzImage") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      if (!strncasecmp (ptr, "vmlinu", sizeof ("vmlinu") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
      if (!strncasecmp (ptrend - 4, ".kgz", sizeof (".kgz") - 1)) {
	  ptr = ptrend;
	  continue;
	  }
#ifdef COMMANDFILE
      /* These parameters go to the command line editable while Gujin is running: */
      if (!strncmp (ptr, "quiet", sizeof ("quiet") - 1)) {
	  while (ptr < ptrend && (dst - gujin_param->extra_cmdline < sizeof (gujin_param->extra_cmdline) - 1))
	      *dst++ = *ptr++;
	  *dst++ = ' ';
	  ptr = ptrend;
	  continue;
	  }
      if (!strncmp (ptr, "debug", sizeof ("debug") - 1)) {
	  while (ptr < ptrend && (dst - gujin_param->extra_cmdline < sizeof (gujin_param->extra_cmdline) - 1))
	      *dst++ = *ptr++;
	  *dst++ = ' ';
	  ptr = ptrend;
	  continue;
	  }
      /* Other parameters go into /boot/gujin.cmd */
      while (ptr < ptrend)
	  *gujin_cmd_ptr++ = *ptr++;
      *gujin_cmd_ptr++ = ' ';
      *gujin_cmd_ptr = '\0';
#else
      /* so this parameter is probably needed for this PC... */
      while (ptr < ptrend && (dst - gujin_param->extra_cmdline < sizeof (gujin_param->extra_cmdline) - 1))
	  *dst++ = *ptr++;
      if (*ptr != ' ')
	  break;
      *dst++ = ' ';
      *dst = '\0';
#endif
      }
  *dst = '\0';
  PROC ("Proc: kept '--cmdline=\"%s\"'\n", gujin_param->extra_cmdline);
#ifdef COMMANDFILE
  if (gujin_cmd_boot_image[0] == '\0') {
      strcpy (gujin_cmd_boot_image, "vmlinuz-");
      /*
       * $ cat /proc/version
       * Linux version 2.6.31.12-174.2.3.fc12.i686.PAE (mockbuild@x86-04.phx2.fedoraproject.org) \
       *    (gcc version 4.4.2 20091222 (Red Hat 4.4.2-20) (GCC) ) #1 SMP Mon Jan 18 20:06:44 UTC 2010
       * for kernel vmlinuz-2.6.31.12-174.2.3.fc12.i686.PAE
       */
      FILE *proc_version = fopen ("/proc/version", "r");
      if (fscanf (proc_version, "Linux version %s", &gujin_cmd_boot_image[strlen("vmlinuz-")]) != 1) {
	  gujin_cmd_boot_image[strlen("vmlinuz-")] = '*';
	  gujin_cmd_boot_image[strlen("vmlinuz-")+1] = '\0';
	  }
      fclose (proc_version);
      }
  PROC ("Proc: has to add to /boot/gujin.cmd for BOOT_IMAGE='%s': \"%s\"\n", gujin_cmd_boot_image, gujin_cmd_buffer);
  if (gujin_cmd_buffer[0] == '\0')
      PROC ("Proc: no need to create /boot/gujin.cmd\n");
    else if (no_gujin_cmd_flag)
      PROC ("Proc: should create /boot/gujin.cmd but disabled by --no_gujin_cmd\n");
    else {
      PROC ("Proc: searching label of /boot in /etc/fstab:\n");

      char gujin_cmd_label[128];
      FILE *file_fstab;
      if ((file_fstab = fopen ("/etc/fstab", "r")) != NULL) {
	  char buffer[1024], dummy;
	  gujin_cmd_label[0] = '\0';
	  while (fgets(buffer, sizeof(buffer)-2, file_fstab)) {
	      if (sscanf(buffer, "%s /boot %c", gujin_cmd_label, &dummy) == 2)
		  break;
	      gujin_cmd_label[0] = '\0';
	      }
	  if (gujin_cmd_label[0] == '\0') {
	      rewind(file_fstab);
	      PROC("Proc: Assuming /boot is not a mounted partition, looking for the root device\n");
	      while (fgets(buffer, sizeof(buffer)-2, file_fstab)) {
		  char *cptr = buffer;
		  while (*cptr == ' ' || *cptr == '\t')
		      cptr++;
		  if (*cptr == '#')
		      continue;
		  if (sscanf(buffer, "%s /%c", gujin_cmd_label, &dummy) == 2 && (dummy == ' ' || dummy == '\t'))
		      break;
		  gujin_cmd_label[0] = '\0';
		  }
	      }
	  fclose (file_fstab);
	  if (gujin_cmd_label[0] != '\0') {
	      if (!strncmp("LABEL=", gujin_cmd_label, strlen("LABEL="))) {
		  strcpy (gujin_cmd_label, &gujin_cmd_label[6]);
		  PROC ("Proc: found LABEL='%s' for /boot (or /)\n", gujin_cmd_label);
		  }
		else {
		  DIR *directory;
		  PROC ("Proc: only found '%s' for /boot (or /)\n", gujin_cmd_label);
		  /* search /dev/disk/by-label/ for the readlink finishing by ../../sda1, its filename is the label */
		  if (!strncmp (gujin_cmd_label, "UUID=", strlen("UUID="))) {
		      char uuidname[256] = "/dev/disk/by-uuid/";
		      strcat (uuidname, gujin_cmd_label+5);
		      char buffer[256] = {};
		      ssize_t buflen;
		      if ((buflen = readlink(uuidname, buffer, sizeof(buffer)) != -1)) {
			  PROC ("Proc: found '%s' for UUID %s\n", buffer, gujin_cmd_label+5);
			  strcpy (gujin_cmd_label, "/dev");
			  strcat (gujin_cmd_label, buffer+5);
			  }
			else
			  PROC ("Proc: cannot read %s!\n", uuidname);
		      }
		  if (!strncmp (gujin_cmd_label, "/dev/disk/by-id", strlen("/dev/disk/by-id"))) {
		      char buffer[256] = {};
		      ssize_t buflen;
		      if ((buflen = readlink(gujin_cmd_label, buffer, sizeof(buffer)) != -1)) {
			  PROC ("Proc: found '%s' for link %s\n", buffer, gujin_cmd_label);
			  strcpy (gujin_cmd_label, "/dev");
			  strcat (gujin_cmd_label, buffer+5);
			  }
			else
			  PROC ("Proc: cannot read link %s!\n", gujin_cmd_label);
		      }
		  directory = opendir ("/dev/disk/by-label/");
		  if (directory == 0) {
		      PROC("Proc: Cannot open \"/dev/disk/by-label/\", cannot find the label of /boot (or /)\n");
		      gujin_cmd_label[0] = '\0';
		      }
		    else {
		      struct dirent *entry;
		      while ((entry = readdir(directory)) != 0) {
			  ssize_t buflen;
			  char buffer[256] = {}, filepath[256] = "/dev/disk/by-label/";
			  strcat (filepath, entry->d_name);
			  PROC ("Proc: looking where points '%s' ... ", filepath);
			  if ((buflen = readlink(filepath, buffer, sizeof(buffer)) != -1)) {
			      //buffer[buflen] = '\0';
			      PROC ("points to '%s'\n", buffer);
			      const char *ptr1 = strrchr(gujin_cmd_label, '/'), *ptr2 = strrchr(buffer, '/');
			      if (ptr1 && ptr2 && !strcmp(ptr1, ptr2)) {
				char *label = strrchr (filepath, '/') + 1;
				PROC("Proc: Found label '%s' for '%s'\n", label, gujin_cmd_label);
				strcpy (gujin_cmd_label, label);
				break;
				}
			      }
			    else
			      PROC ("failed\n");
			  }
		      if (entry == 0) {
			  PROC ("Proc: Did not find any label for /boot (or /), you should really define one!\n");
			  gujin_cmd_label[0] = '\0';
			  }
		      closedir (directory);
		      }
		  }
	      }
	  }
	else
	  PROC ("Proc: cannot open '/etc/fstab' for reading!\n");
      FILE *gujin_cmd;
      char gujin_string[4096];
      sprintf (gujin_string, ":%s:%s:::%s:%s\n", gujin_cmd_label, gujin_cmd_boot_image, "standard boot (*)", gujin_cmd_buffer);
      if ((gujin_cmd = fopen("/boot/gujin.cmd", "r")) != 0) {
	  char line[4096];
	  while (fgets(line, sizeof(line), gujin_cmd) != 0)
	      if (!strcmp (line, gujin_string)) {
		  gujin_string[0] = '\0';
		  break;
		  }
	  fclose (gujin_cmd);
	  }
      if (gujin_string[0] == '\0')
	  PROC ("Proc: line already present in /boot/gujin.cmd, no update needed.\n");
	else if ((gujin_cmd = fopen("/boot/gujin.cmd", "a")) == 0)
	  WARNING ("Cannot open /boot/gujin.cmd for update!\n");
	else {
	  PROC ("Proc: creating/appending /boot/gujin.cmd, please review it!\n");
	  if (ftell(gujin_cmd) == 0)
	      fputs ("# gujin.cmd: the distributed Linux kernel description\n\n"
		"# Format:\n"
		"#:filesystemlabel:kernelname:initrdname:timeout:menuname:commandline\n"
		"# First char: separator for that line, one of \"!$%&+/:=?@|\\t\"\n"
		"# First 3 fields can end with a star to allow every ending.\n"
		"# The 3 last fields will be applied to a boot solution corresponding\n"
		"# to the first 3 fields. Multiple identical first 3 fields show multiple\n"
		"# boot selection in Gujin menu, but identical lines are stripped.\n"
		"# Timeout is only used if Gujin do not have its internal timeout running.\n"
		"# If filesystemlabel is empty (line begin with \"::\"), the label of the\n"
		"# filesystem containing gujin.cmd is used (unlike \":*:\"), so that\n"
		"# filesystem must have a label!\n"
		"# If the last char of the line is the separator, the command line is\n"
		"# used unmodified (bootfrom= is not generated automagically).\n"
		"# Usually leave initrdname empty to use Gujin autoselection.\n\n",
		gujin_cmd);
	    else
	      fputs ("\n", gujin_cmd);
	  fputs (gujin_string, gujin_cmd);
	  fclose (gujin_cmd);
	  /* warning: ignoring return value of "chown" */
	  if (chown ("/boot/gujin.cmd", 0, 0)); /* set owned by root for E2/3/4FS, NFTS, ... using special mount options */
	  errno = 0;
	  }
      }

#endif
  }

static inline unsigned
force_gujin_param (gujin_param_t *gujin_param, const char *argv, unsigned *found)
  {
  unsigned val;
  const char *ptr;

  if ((ptr = pattern_match (argv, "--cmdline=", 0)) != 0) {
      if (found)
	  *found += 1;
      if (gujin_param) {
	  if (strlen (ptr) > sizeof (gujin_param->extra_cmdline) - 1) {
	      PARAMERR ("reading parameter %s, too long cmdline, abort.\n", argv);
	      return 1;
	      }
	  if (*ptr == '\0') {
	      EXTRA ("Extra: reading parameter '%s', empty cmdline, clear default.\n", argv);
	      gujin_param->extra_cmdline[0] = '\0';
	      }
	    else {
	      unsigned cpt;
	      for (cpt = 0; cpt < sizeof (gujin_param->extra_cmdline) - 1; cpt++)
		  if ((gujin_param->extra_cmdline[cpt] = ptr[cpt]) == '\0')
		      break;
	      gujin_param->extra_cmdline[cpt] = '\0';
	      EXTRA ("Extra: add to editable command line, for all kernel: \"%s\"\n", gujin_param->extra_cmdline);
	      }
	  }
      }

  if ((ptr = pattern_match (argv, "--bootdir=", 0)) != 0) {
      if (found)
	  *found += 1;
      if (gujin_param) {
	  if (strlen (ptr) > sizeof (gujin_param->scanpath) - 1) {
	      PARAMERR ("reading parameter %s, too long 'boot' directory name, abort.\n", argv);
	      return 1;
	      }
	  if (*ptr == '\0') {
	      EXTRA ("Extra: reading parameter '%s', empty cmdline, clear default.\n", argv);
	      gujin_param->scanpath[0] = 'b';
	      gujin_param->scanpath[1] = 'o';
	      gujin_param->scanpath[2] = 'o';
	      gujin_param->scanpath[3] = 't';
	      gujin_param->scanpath[4] = '\0';
	      }
	    else {
	      unsigned cpt;
	      for (cpt = 0; cpt < sizeof (gujin_param->scanpath) - 1; cpt++) {
		  if (ptr[cpt] == '/' || ptr[cpt] == '\\') {
		      PARAMERR ("reading parameter %s, 'boot' directory name cannot contain / or \\, abort.\n", argv);
		      return 1;
		      }
		  if ((gujin_param->scanpath[cpt] = ptr[cpt]) == '\0')
		      break;
		  }
	      gujin_param->scanpath[cpt] = '\0';
	      EXTRA ("Extra: replace search directory name 'boot' by \"%s\"\n", gujin_param->scanpath);
	      }
	  }
      }

  if ((ptr = pattern_match (argv, "--lang=", 0)) != 0) {
      if (found)
	  *found += 1;
      char *listlang[] = { LANG_CMDLINE };
      unsigned cptlang;
      for (cptlang = 0; cptlang < sizeof(listlang)/sizeof(listlang[0]); cptlang++)
	  if (!strcmp(listlang[cptlang], ptr - (sizeof("--lang=") - 3)))
	      break;
      if (cptlang >= sizeof(listlang)/sizeof(listlang[0])) {
	  PARAMERR ("reading parameter %s, %s is not in the LANG_CMDLINE array.\n", argv, ptr - (sizeof("--lang=") - 3));
	  return 1;
	  }
      if (gujin_param) {
	  gujin_param->MltStrLanguage = cptlang;
	  EXTRA ("Extra: default language: %s.\n", listlang[gujin_param->MltStrLanguage]);
	  }
      }

  if (is_param (argv, "--default_video_mode=", 0, &val)) {
      if (found)
	  *found += 1;
      if (gujin_param) {
	  gujin_param->default_video_mode = val;
	  EXTRA ("Extra: default video mode: 0x%X.\n", gujin_param->default_video_mode);
	  }
      }

  if (is_param (argv, "--min_nb_initrd=", 0, &val)) {
      if (found)
	  *found += 1;
      if (gujin_param) {
	  gujin_param->min_nb_initrd = val;
	  EXTRA ("Extra: minimum number of char to associate an initrd: %u.\n",
		gujin_param->min_nb_initrd);
	  }
      }

  if (is_param (argv, "--time_hour_offset=", 0, &val)) {
      if (found)
	  *found += 1;
      if (gujin_param) {
	  gujin_param->time_hour_offset = val;
	  EXTRA ("Extra: the timezone is GMT + %d.\n",
		(int)gujin_param->time_hour_offset);
	  }
      }

  if (is_param (argv, "--time_minute_offset=", 0, &val)) {
      if (found)
	  *found += 1;
      if (gujin_param) {
	  gujin_param->time_minute_offset = val;
	  EXTRA ("Extra: the timezone is adjusted by %d minutes.\n",
		(int)gujin_param->time_minute_offset);
	  }
      }

  if (is_param (argv, "--stop_emulation=", 0, &val)) {
      if (found)
	  *found += 1;
      if (gujin_param) {
	  gujin_param->stop_emulation = val;
	  EXTRA ("Extra: Stop CDROM HD/FD emulation: %u i.e. %s.\n",
		gujin_param->stop_emulation,
		(gujin_param->stop_emulation == 0) ? "never"
			 : ((gujin_param->stop_emulation == 2) ? "late" : "early"));
	  }
      }

  if (is_param (argv, "--quickboot=", 0, &val)) {
      if (found)
	  *found += 1;
      if (gujin_param) {
	  gujin_param->timeout_autoload = val + 1;
	  gujin_param->autoload.last_loaded = 0;
	  gujin_param->autoload.total_loadable = 1;
	  gujin_param->autoload.init_page = 0;
	  gujin_param->autoload.total_page = 1;
	  gujin_param->attrib.verbose = 0;
	  gujin_param->attrib.use_gujin_embedded_font = 0; /* Saves time, just display copyright */
	  gujin_param->attrib.disk_write_enable = 0;
	  gujin_param->attrib.search_disk_mbr = 0;
	  gujin_param->attrib.search_part_mbr = 1;
	  gujin_param->attrib.keep_all_part_mbr = 0;
	  gujin_param->attrib.probe_bios_floppy_disk = 0;
	  gujin_param->attrib.probe_bios_hard_disk = 1;
	  gujin_param->attrib.probe_ide_disk = 1;	/* can be reseted later */
	  gujin_param->attrib.probe_cdrom = 0;
	  gujin_param->attrib.probe_dos_disk = 0;
	  gujin_param->default_video_mode = 3; /* message if EDID abscent */
	  EXTRA ("Extra: Will quickly boot if only one bootsector found after %u second.\n",
		gujin_param->timeout_autoload - 1);
	  }
      }

  if ((ptr = pattern_match (argv, "--exclude_video_mode=", 0)) != 0) {
      if (found)
	  *found += 1;
      if (gujin_param) {
	  if (!gujin_param->compilation_attrib.vga_support) {
	      WARNING ("ingnoring parameter %s, VGA_SUPPORT not compiled in.\n", argv);
	      }
	    else {
	      if (init_valid_video_mode (gujin_param->VGA_valid_mode, ptr, argv) != 0) {
		  PARAMERR ("interpreting '%s' in parameter '%s', abort.\n", ptr, argv);
		  return 1;
		  }
	      }
	  }
      }

  if (is_param (argv, "--probe_bdi_file=", 0, &val)) {
      if (found)
	  *found += 1;
      if (gujin_param) {
	  if (val != 0 && val != 1) {
	      PARAMERR ("reading parameter %s, abort.\n", argv);
	      return 1;
	      }
	  gujin_param->attrib.probe_dos_disk = val;
	  EXTRA ("Extra: set probe_bdi_file/probe_dos_disk to %u.\n", gujin_param->attrib.probe_dos_disk);
	  }
      }

#define Apply(field, deflt)	\
    if (is_param (argv, "--" # field "=", 0, &val)) {			\
	if (found)							\
	    *found += 1;						\
	if (gujin_param) {						\
	    if (val != 0 && val != 1) {					\
		PARAMERR ("reading parameter %s, abort.\n", argv);	\
		return 1;						\
		}							\
	    gujin_param->attrib.field = val;				\
	    }								\
	}

    APPLY_ATTRIBUTE_ORDER(Apply)

#undef Apply

  return 0;
  }

static inline void
force_install_param (struct install_s *install, const char *argv, unsigned *found)
  {
  unsigned val;
  const char *ptr;

  if ((ptr = pattern_match (argv, "--default_ide_password=", 0)) != 0) {
      if (found)
	  *found += 1;
      if (1) { /* no real validity test can be done here */
	  if (*ptr == '\0') {
	      EXTRA ("Extra: Disabling default IDE password.\n");
	      install->misc.default_ide_password[0] = '\0';
	      }
	    else if (strlen (ptr) > sizeof(install->misc.default_ide_password)  - 1) {
	      PARAMERR ("reading parameter %s, too long IDE password, ignore whole parameter.\n", argv);
	      }
	    else {
	      memset (install->misc.default_ide_password,'\0', sizeof (install->misc.default_ide_password));
	      strcpy (install->misc.default_ide_password, ptr);
	      EXTRA ("Extra: Default IDE password set to '%s'.\n", install->misc.default_ide_password);
	      }
	  }
      }

  if (is_param (argv, "--nb_sector_FAT_wasted=", 0, &val)) {
      if (found)
	  *found += 1;
      install->fs.nb_sector_FAT_wasted = val;
      EXTRA ("Extra: set nb_sector_FAT_wasted to %u.\n", install->fs.nb_sector_FAT_wasted);
      }

  if (is_param (argv, "--head=", 0, &val)) {
      if (found)
	  *found += 1;
      install->misc.forced_Heads = val;
      EXTRA ("Extra: set forced head to %u.\n", install->misc.forced_Heads);
      }

  if (is_param (argv, "--sectorpertrack=", 0, &val)) {
      if (found)
	  *found += 1;
      install->misc.forced_SectPerTrack = val;
      EXTRA ("Extra: set forced sectorpertrack to %u.\n", install->misc.forced_SectPerTrack);
      }
  }

/*
 * Configuration stuff:
 */
static inline instboot_info_t *
init_and_check_instboot_info (unsigned char *file, unsigned filesize)
  {
  unsigned long instboot_info_ptr_ptr = (unsigned long)file + 0x200;
  unsigned instboot_info_ptr = *(unsigned *)instboot_info_ptr_ptr;
  instboot_info_t *instboot_info = (instboot_info_t *)(file + instboot_info_ptr);

  if (file[0] != 0xEB && file[0] != 0xE9) {
      EXTRA ("instboot_info: file do not begin with a byte jump but 0x%X,"
		" will not work, abort.\n", ((unsigned char *)file)[0]);
      return 0;
      }
  if (((unsigned short *)file)[255] != 0xAA55) {
      int i;
      EXTRA ("instboot_info: file do not have the 0xAA55 signature, abort.\n");
      for (i = 450; i < 600; i++) {
	  if ((*(unsigned short *)(file+i)) == 0xAA55) {
	      if (i < 510)
		  EXTRA ("instboot_info: BOOT1 is too small by %u bytes (add nops)\n", 510 - i);
		else
		  EXTRA ("instboot_info: BOOT1 is too big by %u bytes (reduce it)\n", i - 510);
	      }
	  }
      return 0;
      }
  if (   instboot_info_ptr == 0
      || instboot_info_ptr >= filesize - sizeof(instboot_info_t)) {
      EXTRA ("instboot_info offset = 0x%X, probably not a gujin file,"
		" abort.\n", instboot_info_ptr);
      return 0;
      }

  if (   instboot_info->bootend_adr != 0x1B6
      || instboot_info->checksum_start_adr != 0x3E
      || instboot_info->boot1param_adr !=  0x19A) {
      if (instboot_info->boot1param_adr == 0x196)
	  EXTRA ("Is that a Gujin-0.6 and previous format?\n");
      EXTRA ("instboot_info at 0x%X: checksum_start_adr = 0x%X, bootend_adr = 0x%X, "
		"boot1param_adr = 0x%X,\nprobably not a gujin file, abort.\n",
		instboot_info_ptr,
		instboot_info->checksum_start_adr,
		instboot_info->bootend_adr,
		instboot_info->boot1param_adr);
      return 0;
      }
  if (   instboot_info->code_start_adr >= 0x200
      || instboot_info->patch2adr >= 0x200
      || instboot_info->patch3adr >= 0x200
      || instboot_info->patch4adr >= 0x200) {
      EXTRA ("instboot_info: code_start_adr = 0x%X, patch2adr = 0x%X, "
	      "patch3adr = 0x%X, patch4adr = 0x%X\n"
	      "probably not a gujin file, abort.\n",
	      instboot_info->code_start_adr,
	      instboot_info->patch2adr,
	      instboot_info->patch3adr,
	      instboot_info->patch4adr);
      return 0;
      }
  if (((unsigned char *)file)[1] != instboot_info->code_start_adr - 2) {
      EXTRA ("instboot_info: file begin with a byte jump but target of jump"
		" is 0x%X+2 instead of 0x%X+2, will not work, abort.\n",
		 ((unsigned char *)file)[1], instboot_info->code_start_adr - 2);
      return 0;
      }
  if (instboot_info->gujin_param_adr >= 0x400) {
      EXTRA ("instboot_info: gujin_param_adr = 0x%X, will not work, abort.\n",
	      instboot_info->gujin_param_adr);
      return 0;
      }
  if (instboot_info->password_serial_adr >= 0x200) {
      EXTRA ("instboot_info: password_serial_adr = 0x%X, will not work, abort.\n",
	      instboot_info->password_serial_adr);
      return 0;
      }
  if (instboot_info->serialconf_port_plus_1_adr >= 0x200) {
      EXTRA ("instboot_info: serialconf_port_plus_1_adr = 0x%X, will not work, abort.\n",
	      instboot_info->serialconf_port_plus_1_adr);
      return 0;
      }

  if (check_patch (file, instboot_info) != 0)
      return 0;

  return instboot_info;
  }

static inline boot1param_t *
init_and_check_boot1param (unsigned char *file, instboot_info_t *instboot_info)
  {
  boot1param_t *boot1param = (boot1param_t *)(file
					+ instboot_info->boot1param_adr);

  check_initial_boot1_checksum (file, instboot_info, boot1param);

  return boot1param;
  }

static inline bootloader2_t *
init_and_check_bootchain (unsigned char *file, instboot_info_t *instboot_info)
  {
  bootloader2_t *bootchain = (bootloader2_t *)(file
					+ instboot_info->bootchain_adr);

  if (   instboot_info->bootchain_adr <= 0x200
      || instboot_info->bootchain_adr >= 0x400 - sizeof (bootloader2_t)) {
      EXTRA ("instboot_info: bootchain at address 0x%X, will not work, abort.\n",
		instboot_info->bootchain_adr);
      return 0;
      }
  if (   instboot_info->bootchain_nbblock < 2
      || instboot_info->bootchain_nbblock > 100) {
      EXTRA ("instboot_info: bootchain.nbblock = %d, unrealistic, abort.\n",
		instboot_info->bootchain_nbblock);
      return 0;
      }
  return bootchain;
  }

static inline bootloader2_t *
init_and_check_uninstall_mbr (unsigned char *file, instboot_info_t *instboot_info)
  {
  bootloader2_t *uninstall_mbr = (bootloader2_t *)(file
					+ instboot_info->uninstall_mbr_adr);
  /* Not sure about the upper limit which can work here: */
  if (   instboot_info->uninstall_mbr_adr <= 0x200
      || instboot_info->uninstall_mbr_adr >= 0x10000 - sizeof (bootloader2_t)) {
      EXTRA ("instboot_info: uninstall_mbr_adr at address 0x%X, unrealistic, abort.\n",
		instboot_info->uninstall_mbr_adr);
      return 0;
      }
  return uninstall_mbr;
  }

static inline gujin_param_t *
init_and_check_gujin_param (unsigned char *file, instboot_info_t *instboot_info)
  {
  gujin_param_t *gujin_param = (gujin_param_t *)(file
					+ instboot_info->gujin_param_adr);
  if (gujin_param->magic != 16980327) {
      EXTRA ("gujin_param magic value not found, abort.\n");
      return 0;
      }
  if (gujin_param->version != 0x0208) { /*VERSION*/
      EXTRA ("gujin_param: incompatible version, abort.\n");
      return 0;
      }

  return gujin_param;
  }

/*
 * bootsector stuff
 */
static inline void
setup_install_from_bootbefore (struct install_s *install,
				const bootbefore_t *bootbefore,
				const instboot_info_t *instboot_info)
  {
  unsigned NbTotalsector;

  if (bootbefore->part1.NbTotalsector != 0)
      NbTotalsector = bootbefore->part1.NbTotalsector;
    else
      NbTotalsector = bootbefore->part1.NbTotalsector2;

  /* Setup constants from boot file: */
  install->bootchain.adr = instboot_info->bootchain_adr;
  install->bootchain.nbblock = instboot_info->bootchain_nbblock;

  /* Setup defaults from boot file in the file to be written: */
  install->partdisk.start_sector = bootbefore->part1.NbHiddensector;
  install->partdisk.total_sector = NbTotalsector;
  install->partdisk.BytePerSector = bootbefore->part1.Bytepersector;
  install->partdisk.heads = bootbefore->part1.NbHead;
  install->partdisk.sectpertrack = bootbefore->part1.NbSectorpertrack;
  install->partdisk.cylinders = DIVROUNDUP (NbTotalsector,
					bootbefore->part1.NbSectorpertrack
					* bootbefore->part1.NbHead);
  if (bootbefore->part2.Signaturebyte0x29 == 0x29 || bootbefore->part2.Signaturebyte0x29 == 0x28) {
      install->fs.PhysicalDriveNb = bootbefore->part2.PhysicaldriveNb;
      install->fs.FatSize = bootbefore->part2.FileSysName[4] == '2'  ? 12
			: ((bootbefore->part2.FileSysName[4] == '6') ? 16 : 0);
      install->fs.NbFat32Sector = bootbefore->part1.NbSectorperFAT;
      }
    else {
      bootbefore_FAT32_t *bb32 = (bootbefore_FAT32_t *)bootbefore;
      if (bb32->part2.Signaturebyte0x29 == 0x29 || bb32->part2.Signaturebyte0x29 == 0x28) {
	  install->fs.PhysicalDriveNb = bb32->part2.PhysicaldriveNb;
	  install->fs.FatSize = 32;
	  }
      install->fs.NbFat32Sector = bb32->partFAT32.NbSectorperFAT2;
      }

  /* The FS can have a shorter size than the partition, but they are equal
     by default: */
  install->fs.NbFsSector = NbTotalsector;
  install->fs.NbFat = bootbefore->part1.NbFAT;
  install->fs.NbRootSector = bootbefore->part1.NbRootdirentry * sizeof (directory_t)
					/ bootbefore->part1.Bytepersector;
  install->fs.NbReservedSector = bootbefore->part1.Reservedsector;
  install->fs.MediaDescriptor = bootbefore->part1.Mediadescriptor;
  install->fs.NbSectorPerCluster = bootbefore->part1.Sectorpercluster;
  }

static inline void
update_bootbefore_from_install (bootbefore_t *bootbefore, const struct install_s *install)
  {
  unsigned NbTotalsector = install->fs.NbFsSector;
  unsigned NbRootdirentry = install->fs.NbRootSector
				* install->wholedisk.BytePerSector
				/ sizeof (directory_t);

  bootbefore->part1.NbHead           = install->partdisk.heads;
  bootbefore->part1.NbSectorpertrack = install->partdisk.sectpertrack;
  if (NbTotalsector < 0x10000) {
      bootbefore->part1.NbTotalsector  = NbTotalsector;
      bootbefore->part1.NbTotalsector2 = 0;
      }
    else {
      bootbefore->part1.NbTotalsector  = 0;
      bootbefore->part1.NbTotalsector2 = NbTotalsector;
      }

#if 0
  /* To be recognised by DOS as a floppy, for USB thumb drives,
	the first byte of the MBR has to be 0xE9: */
  if (   install->gujin2_load == gujin2_load_bios
//    && install->fs.PhysicalDriveNb < 0x80  // CHECKME
      && bootbefore->part1.jmpinstruction[0] == 0xEB
      && bootbefore->part1.EBIOSaccess == 0x90) {
      /* absolute jump + 16 bit addr with MSB = 0 */
      bootbefore->part1.jmpinstruction[0] = 0xE9;
      bootbefore->part1.EBIOSaccess = 0x00;
      }
  EXTRA ("Extra: boot record: 0x%X, 0x%X, 0x%X, i.e. %s jump\n",
	bootbefore->part1.jmpinstruction[0],
	bootbefore->part1.jmpinstruction[1],
	bootbefore->part1.EBIOSaccess,
	(bootbefore->part1.jmpinstruction[0] == 0xEB)? "relative"
		: ((bootbefore->part1.jmpinstruction[0] == 0xE9)? "absolute" : "unknown")
	);
#endif

  bootbefore->part1.Bytepersector = install->wholedisk.BytePerSector;
  /* bootbefore->part1.NbHiddensector == 0xFFFFFFFF if overflow */
  if (install->partdisk.start_sector >> 32)
      bootbefore->part1.NbHiddensector = 0xFFFFFFFF;
    else
      bootbefore->part1.NbHiddensector = install->partdisk.start_sector;

  bootbefore->part1.Sectorpercluster = install->fs.NbSectorPerCluster;
  if (NbRootdirentry < 0x10000)
      bootbefore->part1.NbRootdirentry   = NbRootdirentry;
    else {
      /* 4096 sector of root max, usually enough -:) */
      EXTRA ("Extra: install->fs.NbRootSector = %u, NbRootdirentry %u, too big!\n",
		install->fs.NbRootSector, NbRootdirentry);
      }
  bootbefore->part1.Mediadescriptor  = install->fs.MediaDescriptor;
  bootbefore->part1.Reservedsector = install->fs.NbReservedSector;

  if (install->fs.FatSize == 12) {
      unsigned cpt;
      for (cpt = 0; cpt < sizeof (bootbefore->part2.FileSysName); cpt++)
	  bootbefore->part2.FileSysName[cpt] = "FAT12   "[cpt];
      bootbefore->part2.PhysicaldriveNb = install->fs.PhysicalDriveNb;
      bootbefore->part1.NbSectorperFAT = install->fs.NbFat32Sector;
      }
    else if (install->fs.FatSize == 16) {
      unsigned cpt;
      for (cpt = 0; cpt < sizeof (bootbefore->part2.FileSysName); cpt++)
	  bootbefore->part2.FileSysName[cpt] = "FAT16   "[cpt];
      bootbefore->part2.PhysicaldriveNb = install->fs.PhysicalDriveNb;
      bootbefore->part1.NbSectorperFAT = install->fs.NbFat32Sector;
      }
    else {
      bootbefore_FAT32_t *bb32 = (bootbefore_FAT32_t *)bootbefore;
      bootbefore->part1.NbSectorperFAT = 0;
      bb32->partFAT32 = (FAT32_bootsect_t) {
	    NbSectorperFAT2: install->fs.NbFat32Sector,
	    //mkdosfs does not set the "one_fat_active" bit, i.e. enables FAT Mirroring:
	    //flags: { active_fat: 0, reserved1: 0, one_fat_active: 1, reserved2: 0 },
	    flags: { active_fat: 0, reserved1: 0, one_fat_active: 0, reserved2: 0 },
	    version: 0,
	    root_cluster: 2,
	    FSinfosector: 1,
	    save_bootrecord: 6,
	    reserved: { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
	    };
      bb32->part2 = (FAT_bootsect2_t) {
	    PhysicaldriveNb: install->fs.PhysicalDriveNb,
	    dirty_flag : 0,	/* need checkdisk at next boot */
	    surface_scan : 0,	/* checkdisk shall also do surface scan */
	    FATreserved : 0,	/* was current head, head for the boot sector */
	    Signaturebyte0x29: 0x29,	/* can also be 0x28 for Window NT */
	    VolumeserialNb: 0x39304C45,
	    Volumelabel: "gujin2\0\0\0\0\0",
	    FileSysName: "FAT32   "
	    };
      /* Adjust boot.asm " 13a: be 2b 00 $boot2_msg,%si" : */
      ((unsigned char *)bootbefore)[0x13B] += sizeof(FAT32_bootsect_t);
      /* here the gujin1 IDE code is overwritten, shall no more be used! */
      }
  /* FIXME: We should set values of partition 4 to be a copy of the
     parameter of the current partition instead of keeping it at zero
   */
  }

static inline void
report_bootsector_content (bootbefore_t *bootbefore)
  {
  char fatname[sizeof (bootbefore->part2.FileSysName) + 1];
  char label[sizeof (bootbefore->part2.Volumelabel) + 1];
  unsigned cpt;
  unsigned long long size_in_kb;
  unsigned size_in_mb, nb_cylinder;
  unsigned NbTotalsector = bootbefore->part1.NbTotalsector;

  if (NbTotalsector == 0)
      NbTotalsector = bootbefore->part1.NbTotalsector2;

  if (bootbefore->part1.NbSectorpertrack != 0 && bootbefore->part1.NbHead != 0) {
      nb_cylinder = NbTotalsector / bootbefore->part1.NbSectorpertrack;
      nb_cylinder /= bootbefore->part1.NbHead;
      }
    else
      nb_cylinder = 0;

  size_in_kb = (NbTotalsector * (bootbefore->part1.Bytepersector/512)) / (1024/512);
  size_in_mb = size_in_kb / 1024;

  MBR ("Mbr: jmpinstruction: 0x%X,0x%X, EBIOSaccess 0x%X i.e. %s\n",
		bootbefore->part1.jmpinstruction[0],
		bootbefore->part1.jmpinstruction[1],
		bootbefore->part1.EBIOSaccess,
		(bootbefore->part1.EBIOSaccess == 0x90)? "BIOS only"
			: ((bootbefore->part1.EBIOSaccess == 0x0E) ? "EBIOS"
			: "unknown")
		);

  MBR ("Mbr: NbHiddensector %u, Head %u, Sector/track %u, ",
		bootbefore->part1.NbHiddensector,
		bootbefore->part1.NbHead,
		bootbefore->part1.NbSectorpertrack);

  if (bootbefore->part1.NbHiddensector == 0)
      MBR ("i.e. floppy (%u cylinders) ", nb_cylinder);
    else
      MBR ("i.e. partition ");
  if (size_in_mb < 64)
      MBR ("size %u Kb.\n", (unsigned)size_in_kb);
    else
      MBR ("size %u Mb.\n", size_in_mb);

  MBR ("Mbr: Bytepersector %u, Sectorpercluster %u, Reservedsector %u, NbRootdirentry %u.\n",
		bootbefore->part1.Bytepersector,
		bootbefore->part1.Sectorpercluster,
		bootbefore->part1.Reservedsector,
		bootbefore->part1.NbRootdirentry);
  MBR ("Mbr: NbFAT %u, NbSectorperFAT %u, NbTotalsector %u, NbTotalsector2 %u, Mediadescriptor 0x%X.\n",
		bootbefore->part1.NbFAT,
		bootbefore->part1.NbSectorperFAT,
		bootbefore->part1.NbTotalsector,
		bootbefore->part1.NbTotalsector2,
		bootbefore->part1.Mediadescriptor);
  bootbefore_FAT32_t *bb32 = (bootbefore_FAT32_t *)bootbefore;
  if (bootbefore->part2.Signaturebyte0x29 == 0x29 || bootbefore->part2.Signaturebyte0x29 == 0x28) {
      for (cpt = 0; cpt < sizeof (bootbefore->part2.FileSysName); cpt++)
	  fatname[cpt] = bootbefore->part2.FileSysName[cpt];
      fatname[cpt] = '\0';
      for (cpt = 0; cpt < sizeof (bootbefore->part2.Volumelabel); cpt++)
	  label[cpt] = bootbefore->part2.Volumelabel[cpt];
      label[cpt] = '\0';
      MBR ("Mbr: PhysicaldriveNb 0x%X, FATreserved %u, Signaturebyte0x29 0x%X, FileSysName '%s', Volumelabel '%s'.\n",
		bootbefore->part2.PhysicaldriveNb, bootbefore->part2.FATreserved, bootbefore->part2.Signaturebyte0x29, fatname, label);
      MBR ("Mbr: dirty_flag %u, surface_scan %u, FATreserved %u, VolumeserialNb 0x%X\n",
		bootbefore->part2.dirty_flag, bootbefore->part2.surface_scan, bootbefore->part2.FATreserved, bootbefore->part2.VolumeserialNb);
      }
    else if (bb32->part2.Signaturebyte0x29 == 0x29 || bb32->part2.Signaturebyte0x29 == 0x28) {
      for (cpt = 0; cpt < sizeof (bb32->part2.FileSysName); cpt++)
	  fatname[cpt] = bb32->part2.FileSysName[cpt];
      fatname[cpt] = '\0';
      for (cpt = 0; cpt < sizeof (bb32->part2.Volumelabel); cpt++)
	  label[cpt] = bb32->part2.Volumelabel[cpt];
      label[cpt] = '\0';
      MBR ("Mbr: PhysicaldriveNb 0x%X, FATreserved %u, Signaturebyte0x29 0x%X, FileSysName '%s', Volumelabel '%s'.\n",
		bb32->part2.PhysicaldriveNb, bb32->part2.FATreserved, bb32->part2.Signaturebyte0x29, fatname, label);
      MBR ("Mbr: dirty_flag %u, surface_scan %u, FATreserved %u, VolumeserialNb 0x%X\n",
		bb32->part2.dirty_flag, bb32->part2.surface_scan, bb32->part2.FATreserved, bb32->part2.VolumeserialNb);
      MBR ("Mbr: NbSectorperFAT2 %u, active_fat %u, one_fat_active %u, version %u, root_cluster %u, FSinfosector %u, save_bootrecord %u\n",
		bb32->partFAT32.NbSectorperFAT2, bb32->partFAT32.flags.active_fat, bb32->partFAT32.flags.one_fat_active,
		bb32->partFAT32.version, bb32->partFAT32.root_cluster, bb32->partFAT32.FSinfosector, bb32->partFAT32.save_bootrecord);
      }
    else {
      MBR ("Mbr: not right signature: FAT16 Signaturebyte0x29 0x%X, FAT32 Signaturebyte0x29 0x%X\n",
		bootbefore->part2.Signaturebyte0x29, bb32->part2.Signaturebyte0x29);
      }
  }

static inline void
report_size (instboot_info_t *instboot_info, gujin_param_t *gujin_param)
  {
  /* look in boot.ld for the following: */
  if (instboot_info->diskcodeseg != 0) {
      SIZE ("Size: this configuration leaves %d bytes for code,\n",
		0x10000 - (instboot_info->diskcodeseg << 4));
      SIZE ("Size: and leaves %d bytes for diskcode, %d bytes for fscode,\n",
		0x10000 - ((instboot_info->fscodeseg - instboot_info->diskcodeseg) << 4),
		0x10000 - ((instboot_info->loadcodeseg - instboot_info->fscodeseg) << 4));
      SIZE ("Size: and leaves %d bytes for loadcode, %d bytes for usercode,\n",
		0x10000 - ((instboot_info->usercodeseg - instboot_info->loadcodeseg) << 4),
		0x10000 - ((instboot_info->xdataseg - instboot_info->usercodeseg) << 4));
      if (instboot_info->xstring1seg && instboot_info->xstring2seg)
	  SIZE ("Size: and leaves %d bytes for xdata, %d bytes for xstring1, %d bytes for xstring2.\n",
		0x10000 - ((instboot_info->xstring1seg - instboot_info->xdataseg) << 4),
		0x10000 - ((instboot_info->xstring2seg - instboot_info->xstring1seg) << 4),
		0x10000 - ((instboot_info->deltaseg - instboot_info->xstring2seg) << 4));
	else if (instboot_info->xstring1seg)
	  SIZE ("Size: and leaves %d bytes for xdata, %d bytes for xstring.\n",
		0x10000 - ((instboot_info->xstring1seg - instboot_info->xdataseg) << 4),
		0x10000 - ((instboot_info->deltaseg - instboot_info->xstring1seg) << 4));
	else if (instboot_info->xdataseg)
	  SIZE ("Size: and leaves %d bytes for xdata.\n",
		0x10000 - ((instboot_info->deltaseg - instboot_info->xdataseg) << 4));
	else
	  SIZE ("Size: and leaves %d bytes for usercode.\n",
		0x10000 - ((instboot_info->deltaseg - instboot_info->usercodeseg) << 4));
      }
    else if (instboot_info->deltaseg != 0)
      SIZE ("Size: this configuration leaves %d bytes for code.\n",
	0x10000 - (instboot_info->deltaseg << 4));
  SIZE ("Size: this configuration leaves %d bytes for stack%s.\n",
		0x10000 - instboot_info->edata,
		gujin_param->compilation_attrib.big_malloc? "" : " + heap");
  }

static inline void
print_version (const char *progname)
  {
  PRINTF ("%s version 2.8.7 -- see file gpl.txt for the licence\n", progname); /*VERSION*/
  }

#ifdef WITH_INPUT_FILES
static inline void
print_short_help (const char *progname)
  {
  PRINTF (
  "'%s' creates bootable disk/partition or DOS executable for the\n"
  "Gujin bootloader. It can generate aligned FAT12/16/32 filesystems.\n", progname);
  PRINTF (
  "'%s' also creates file /boot/gujin.cmd from /proc/cmdline to describe\n"
  "Linux kernel command parameters for this distribution.\n", progname);
  PRINTF (
  "It is a simplified version of 'instboot' executable, to do as 'root':\n"
  " - create a bootloader in a file, like Grub or Lilo:\n"
  "    ./gujin /boot/gujin.ebios   or   ./gujin /media/usbdisk/gujin.ebios\n"
  " - Same but with the default language as french:\n"
  "    ./gujin --lang=fr /boot/gujin.ebios \n"
  " - create a bootable FAT filesystem as superfloppy on a USB stick:\n"
  "    THIS WILL ERASE ALL FILES, backup now if needed!\n"
  "    FIRST double check /dev/sdg is your USB stick, and it is not mounted!\n"
  "    umount /dev/sdg* \n"
  "    ./gujin /dev/sdg              # (--disk=EBIOS:0x0,auto is the default)\n"
  "    Some PC will not boot USB stick with EBIOS, then you can try: \n"
  "    ./gujin --disk=BIOS:0x0,auto /dev/sdg \n"
  "    Some PC consider the SD card as the new floppy, accessed like: \n"
  "    ./gujin --disk=BIOS:0x0,auto /dev/mmcblk0 \n"
  "    The old floppy (BIOS access implicit) is initialised by: \n"
  "    ./gujin /dev/fd0 \n"
  "    To create a FAT filesystem on a 4.7Gb DVD-RAM (2 Kb sectors): \n"
  "    ./gujin /dev/cdrw1 --geometry=2236704,0,128,32,2048 \\\n"
  "                       --disk=EBIOS:0xE0,auto -c=writeonly \n"
  "      or simply: ./gujin /dev/sr0 \n"
  "    To create a read-only FAT filesystem on a 650 Mb CD-RW (512 b sectors): \n"
  "    ./gujin /tmp/cdrom.iso --geometry=1331200,0,128,32,512\\\n"
  "       --disk=EBIOS:0xE0,auto && wodim /tmp/cdrom.iso \n"
  " - create a FAT filesystem as partition'ed disk on a USB stick:\n"
  "    FIRST double check /dev/sdg1 is your USB stick, and it is not mounted!\n"
  "    umount /dev/sdg* ; fdisk /dev/sdg # and delete all partitions,\n"
  "       or type: dd bs=512 count=63 if=/dev/zero of=/dev/sdg \n"
  "    ./gujin --mbr /dev/sdg       # (--disk=EBIOS:0x80,auto is the default)\n"
  "    If your PC do not support the Extended BIOS on USB, try:\n"
  "    ./gujin --mbr --disk=BIOS:0x80,auto /dev/sdg \n"
  " - transform a bootable ISO file and write it to a USB stick:\n"
  "    ./gujin eeebuntu-2.0-standard.iso \\\n"
  "            && cat eeebuntu-2.0-standard.iso > /dev/sdg && sync \n"
  " - create a bootloader with menu for DOS:\n"
  "    ./gujin gujin.exe \n"
  " - create a bootloader without menu for DOS:\n"
  "    ./gujin --tiny mingujin.exe \n"
  " - create a bootloader for DVD/CDROM and burn it:\n"
  "    mkdir /tmp/cdrom && ./gujin /tmp/cdrom/gujin.bcd \\ \n"
  "        && mkisoimage -no-emul-boot -boot-load-size 4 -b gujin.bcd /tmp/cdrom \\\n"
  "             | wodim -tao - \n"
  " - create a bootloader outside of any partition, at end of disk:\n"
  "    ./gujin /dev/sdg0 \n"
  " - display information about the current bootloader of a disk/partition: \n"
  "    ./gujin --report /dev/sda \n"
  " - remove the Gujin bootloader and replace with the previous bootloader: \n"
  "    ./gujin --remove /boot/gujin.ebios or: ./gujin --remove /dev/sda \n"
  "A lot more options are available, --tiny can be applied to most previous\n"
  "examples to totally remove menus, also read --help parameter.\n"
  );
  }
#else
static inline void
print_short_help (const char *progname)
  {
  PRINTF (
  "%s transform a Gujin binary \"boot.bin\" file into a bootable disk/partition\n"
  " or a DOS executable for the Gujin bootloader.\n", progname);
  PRINTF (
  "'%s' also creates file /boot/gujin.cmd from /proc/cmdline to describe\n"
  "Linux kernel command parameters for this distribution.\n", progname);
  PRINTF ("Type \"%s -h\" or \"%s --help\" for help.\n",
		progname, progname);
  }
#endif

static inline void
print_help (const char *progname, const char *helpcmd)
  {
  PRINTF (
  "%s transform a Gujin binary \"boot.bin\" file into a bootable disk/partition\n"
  " or a DOS executable, or insert that file as a contiguous file inside a\n"
  " filesystem, taking care of the MBR if needed.\n"
  "'%s' also creates file /boot/gujin.cmd from /proc/cmdline to describe\n"
  "Linux kernel command parameters for this distribution.\n"
  "USAGE: %s [params] infile out[file|device]    (case insensitive params)\n"
  " where [params] can be:\n",
	progname, progname, progname);

  if (*helpcmd == '=')
      helpcmd++;

  PRINTF (
  " -v or --version to display version\n"
  " -h or --help  to display this help\n"
  " More help: \n"
  " -ha or -h=a or --help=a for everything, to be printed\n"
  " -hb for basic parameters, like verbosity, MBR and BEER devices...\n"
  " -hg for help on the --geometry=/-g= parameter controlling the disk description\n"
  " -hd for help on the --disk=/-d= parameter controlling the access method\n"
  " -hf for help on the --fs= parameter controlling the filesystem created\n"
  " -hl for the list of default attributes (default setup screen when in Gujin)\n"
  " -hx for exemples on how to use Gujin installer\n\n"
#ifndef WITH_INPUT_FILES
  "      In short, being root for all those commands:\n"
  " Install Gujin bootloader in a filesystem (RAID or LVM not supported):\n"
  " ./instboot boot.bin /boot/gujin.ebios\n"
  " Remove Gujin bootloader from Linux (but preferred way through Gujin menu):\n"
  " ./instboot --remove=/boot/gujin.ebios or: ./instboot --remove=/dev/sda\n"
  " Get information about which bootloader is installed on a device:\n"
  " ./instboot --report=/dev/sda or ./instboot --report=/boot/gujin.ebios\n"
  " Install and fully format a USB key in /dev/sdg to boot Gujin:\n"
  " umount /dev/sdg* ; dd if=/dev/zero of=/dev/sdg bs=512 count=63\n"
  " ./instboot boot.bin /dev/sdg --mbr --disk=EBIOS:0x80,auto --cmdline=\"\" \n"
  " A lot of PC only accept --disk=BIOS:0x80,auto (no EBIOS for USB).\n"
  " Modify a bootable ISO image to superfloppy and copy to a USB disk or SD card:\n"
  " ./instboot boot.bin easypeasy-1.1.iso\n"
  " umount /dev/sdg && cat easypeasy-1.1.iso > /dev/sdg && sync\n"
  " umount /dev/mmcblk0 && cat easypeasy-1.1.iso > /dev/mmcblk0 && fsync\n"
#endif
	);

  if (*helpcmd == 'b' || *helpcmd == 'a') {
      PRINTF ( "\n"
  " -w or --verbose[+-=[scpflmer]] to be more or less verbose about something\n"
  " -f or --full to fill completely the end of the image or device\n"
  " --force to do without question about creating filesystem on device\n"
  " --cmdline=\"append to command line for all kernels\" (default to /proc/cmdline)\n"
  " --lang=XX to define the default language to XX, one of: en,ru,sp,fr,it,pt,nl,de\n"
  " --bootdir=boot to replace the name of the directory searched (no capital letter!).\n"
  " --no_gujin_cmd forbid the installer to create /boot/gujin.cmd even if it is needed\n"
  " --default_video_mode=<>, --exclude_video_mode=<>[,<>][,<>-<>] intuitive.\n"
  " --min_nb_initrd=<0> minimum number of char to associate an initrd to a kernel.\n"
  " --time_hour_offset= and --time_minute_offset= for GMT offset for the clock.\n"
  " --stop_emulation=<1> stop CDROM emulation 0:never, 1:early, 2:late.\n"
  "   WARNING: Stopping BIOS emulation may create problems when BIOS disk number\n"
  "     swapping or *.BDI Gujin emulation on a stopped BIOS emulation...\n"
  " --quickboot=<1>  (seconds) to quickly boot if only one bootsector is found.\n"
  " -COM[1-4][,<9600>[,<n>[,<8>[,<1>]]]]] or --serial=COM[1-4]...\n"
  "   to setup a serial interface (VT100+) instead of the PC screen.\n"
  " --refuse_8x14font=[0,1] : mostly modes 0xF and 0x10 (640x350), if BIOS buggy.\n"
  "   same pattern \"--xxx=[0,1]\" for all other attributes, see -hl for a list.\n"
  " -c=[writeonly|checkonly] or --check=[] to suppress write/check of data written\n"
  " --default_ide_password=<> set a password to try first when a disk is locked.\n"
  " --mbr-device= to set up the Master Boot Sector of this device to load gujin,\n"
  "   for instance --mbr-device=/dev/sda or --mbr-device=/dev/hdb.\n"
  " --mbr to auto-guess '--mbr-device=' from the output file/device name.\n"
  " --beer-device= to set up the disk to conform to the B.E.E.R. specification,\n"
  "   that is creating an extra partition (few Mb containning a small FAT) at end\n"
  "   of the device, referenced only on the BEER structure located in the last\n"
  "   sectors, not in the partition table of the MBR. This area can be protected \n"
  "   with the IDE interface depending on the drive capabilities.\n"
  "   for instance --beer-device=/dev/sda or --beer-device=/dev/hdb.\n"
  " --extra_size=<nb_bytes> artificially increase bootloader size to plan for\n"
  "   upgrade without modifying FAT in partition or B.E.E.R.\n"
  " --sig0x29=<val> change the default 0x29 signature in the MBR to 0x28\n"
  "   if your PC do not boot at all.\n"
  " --report <device> If Gujin is installed on this device (MBR AND second stage)\n"
  "   display info of the currently installed Gujin.\n"
  " --remove <device|file> If Gujin is installed on this device (MBR and second\n"
  "   stage) AND gujin is not installed in HPA, uninstall Gujin by replacing\n"
  "   the disk MBR (and remove the second stage file if param is a file).\n"
  " --extract-mbr=<newfile> </boot/gujin.ebios> Extract the saved MBR excluding\n"
  "   the partition table in newfile for emergency recovery situations.\n"
  " --report-signature If this (ISO) simulation was started by Gujin, it left\n"
  "   a signature describing the file location and name of the simulated drive,\n"
  "   so this option display that signature and try to resolve it\n"
	);
      }

  if (*helpcmd == 'g' || *helpcmd == 'a') {
      PRINTF ( "\n"
  " The geometry of the device is written in the boot record of the FAT filesystem\n"
  " created and may be used to load and calculate checksum of Gujin2.             \n"
  " It is set with '-g=' or '--geometry=':                                        \n"
  " -g=size,start,head,sector/track,sectorsize[,overwrite same fields for loading]\n"
  " The 'size,start' are those of the boot record and if 'start' is null, the disk\n"
  " is a floppy and has no partition; else you overwrite 'size,start' in the      \n"
  " second part with the disk parameters (total size, and usually total start=0). \n"
  " The ending 'Head,Sector/Track' is only used to load Gujin2 at boot, it        \n"
  " defaults to the first values for '-d=BIOS' (logical C/H/S) but shall usually  \n"
  " be different for '-d=IDE:chs' (physical C/H/S).                               \n"
  " The ending 'Head,Sector/Track' are not used for LBA access (EBIOS, CDROM, IDE \n"
  " lba28 or lba48) so they are usually null. If only 'Sector/Track' is not null, \n"
  " it is used to encode something else: the maximum size in sector of a single   \n"
  " request, and if the partition offset (encoded in the partition header) is used\n"
  " or not. If the partition offset is used, most tools which relocate partition  \n"
  " on the disk will no more break the boot process, the two drawback are that it \n"
  " is only possible for LBA and this partition offset is not checksumed by Gujin.\n"
  " So when 'Head' is null, the most significant bit of 'Sector/Track' (i.e. 0x80)\n"
  " force absolute LBA when set (partition relocation will then not work) and the \n"
  " lower significant bits (mask with 0x7F) describe the maximum request size in  \n"
  " sector (starting at 1 i.e. 2 to 128 sectors of 512 or 2048 bytes).            \n"
  " Autoprobe: -g=/dev/sda4 or --geometry=/dev/fd0 or none (outfile is a device)  \n"
  "  When gujin is told to create a partition, you can select which one will be   \n"
  " created (by default number 4) by the --partition_index=[1..4] option.         \n"
  " -m=<0> or --maxsectload=<0> will load boot2 with max <n> sectors at a time    \n"
/* Extra note:
 * It is possible but not tested to have the start of the disk != 0 (see SAORAB)
 * If the partition offset is used it is used everywhere, and so even the MBR may have
 *  a non null "partition offset" field because Gujin MBR contains a partition header.
 * If the partition offset is used a bad fdisk tool overwriting this field will
 *  make the disk unbootable. The windows partition would also become unbootable.
 *  I did not find any buggy fdisk tool which rewrites the partition offset in the MBR.
 * The bootable CDROM patch (to load the end of the "no-emul" image) need to use
 * this partition offset.
 * Take care: even uninstall_mbr is relative in case user moves the partition before
 * dis-installing...
 */
	);
      }

  if (*helpcmd == 'd' || *helpcmd == 'a') {
      PRINTF ( "\n"
  " The boot time access method is deduced from 'outfile' if it is a device, or   \n"
  " has a '.com', '.exe', '.bcd' or '.pic' extension, or has a recognised number  \n"
  " extension ('.144' for 1.44 Mb, -d=floppy:[]), else if it is a file / you want \n"
  " to overwrite, use the '--disk=' or '-d=' parameter:                           \n"
  "  --disk=dos:[<exe>|com]       (--disk=dos gives by default a gujin.exe file)   \n"
  "  --disk=BCD  (for CDROM noemul, you may add -boot-info-table to mkisoimage)\n"
  "  --disk=PIC  (for position independant real mode code image, boot EPROM?)     \n"
  "  -d=floppy:[360|120|720|<144>|168|288]  (outfile: image of the floppy, see -f)\n"
  "  -d=IDE:hd[a-z] does a full autodetect from /proc files    (see verbose: -w=p)\n"
  "     You can do manually:  -d=IDE:lba,ide0,master  or  -d=IDE:lba,0x1F0,0x3F8,0\n"
  "     or -d=IDE:lba48,ide0,1   (48 bits LBA) (1 = slave, slave 2 & 3 not tested)\n"
  "     or -d=IDE:lba28,ide2,master (lba == lba28)     or    -d=IDE:chs,ide3,slave\n"
  "  -d=EBIOS:0x80 on a new PC with extended BIOS, for disk 0x80 at boot time     \n"
  "  -d=BIOS:0x80  on a old PC, this needs a valid logical C/H/S description      \n"
  "     For BIOS and EBIOS, floppies start at 0 (A:) and hard disks start at 0x80.\n"
  "     For BIOS and EBIOS, you can add another parameter after a comma, it is    \n"
  "     the disk number to load Gujin2 if different from the one written in the   \n"
  "     initial boot sector or MBR. It can also be 'auto' or 0x7F to load from    \n"
  "     the disk number passed in DL register by the BIOS - so then the disk      \n"
  "     can boot whatever its BIOS number on all conforming BIOS (all of them?).  \n"
  "     For instance: -d=EBIOS:0x80,auto or -d=BIOS:0x0,auto or -d=BIOS:0x0,0x0   \n"
  "  Option --read_retry can be used to force retrying after read error.          \n"
  "  For -d=BIOS only, option --single_sector_load can be used to load Gujin2     \n"
  "    from Gujin1, to be used on really broken BIOSes, it is slower.             \n"
  "  You can use --head=16 and --sectorpertrack=32 to force geometry if some      \n"
  "    of your setup needs it (mostly for use with USB drives).                   \n"
	);
      }

  if (*helpcmd == 'f' || *helpcmd == 'a') {
  /* Note: for FS, it is easier to use the number of root _sectors_ and not the
     number of root _entries_, each of them being 32 bytes wide, because
     the sector size appear a lot less in calculus... default: 14 * 512 / 32 = 224
     */
      PRINTF ( "\n"
  " By default the size of the filesystem created is the size of the device, and  \n"
  " other FAT parameters are deduced from it, but you can overwrite with '--fs=': \n"
  " --fs=FAT:nbSectorInFS,Sector/Cluster,NRS,RES,NF,FS,PDN,MD                     \n"
  "   Abbreviations: NRS: nb Root Sectors <14>   (default is for 224 root entries)\n"
  "      RES: nb FAT reserved sectors <1>, NF: nbFat <2>, FS: Fat size <12|16|32>,\n"
  "      PDN: physical drive nb <0x00|0x80>, MD: media descriptor <0xF0>          \n"
//  "   <nbSectorInFS> can be a device to get its size, i.e. --fs=FAT:/dev/sda4,8,3 \n"
// only if the device is already created and not modified...
  "   <PDN> is the disk number written, '-d=BIOS:0x..' is the disk number used.   \n"
  "   For FAT32, NRS has to be zero, and RES has to be >= 8, usually 32.          \n"
  "   For instance, if you have --geometry=2007746..., type --fs=FAT:2007746,32,14\n"
  "     for FAT16 and --fs=FAT:2007746,8,0 for FAT32 (leave RES autodetection).   \n"
//  " Autoprobe: --fs=/dev/sda4 reads the first sector to deduce FAT parameters.    \n"
  " --nb_sector_FAT_wasted=<> : add unused sectors at end of each FAT.            \n"
  " By default, RES and the FAT size are adjusted to start clusters at a round    \n"
  " address to optimise access speed on some devices.                             \n"
  " To place the file in the FAT filesystem, use '-p=' or '--position=<0>' like:  \n"
  " --position=[beforefs|<startfs>|clusternb|endfs|afterfs], clusternb is a number.\n"
	);
      }

  if (*helpcmd == 'l' || *helpcmd == 'a') {
      unsigned cpt = 0;
      PRINTF ( "\n"
  "It accept turning ON and OFF each of those options by --option=1 or --option=0.\n"
  "Note that --probe_bdi_file= is the same as --probe_dos_disk=, the selection is \n"
  "done at run time (running on DOS or on BIOS subsystem).\n");
#define Apply(field, deflt) \
    PRINTF ("%s,%s", #field, (++cpt % 3 == 0)? "\n" : " ");
      DISPLAY_APPLY_ATTRIBUTE_ORDER(Apply)
      MISC_APPLY_ATTRIBUTE_ORDER(Apply)
      USER_APPLY_ATTRIBUTE_ORDER(Apply)
      SEARCH_APPLY_ATTRIBUTE_ORDER(Apply)
      PROBE_APPLY_ATTRIBUTE_ORDER(Apply)
      PRINTF ("\n");
#undef Apply
      }

  if (*helpcmd == 'x' || *helpcmd == 'a') {
      PRINTF ( "\n"
  " Examples: create a DOS executable:                                            \n"
  "./gujin gujin.exe                                                               \n"
  " install Gujin on a floppy (size autodetect), being very verbose:              \n"
  "./gujin /dev/fd0 --verbose           # --verbose is the same as -w             \n"
  " On a floppy, install Gujin before the FAT filesystem (in FAT reserved area):  \n"
  "./gujin /dev/fd0 --position=beforefs -w    # --position same as -p             \n"
  " On a floppy, install Gujin at end of the FAT filesystem, filling empty space: \n"
  "./gujin /dev/fd0 -p=endfs --full               # --full same as -f             \n"
  " Create a floppy image of a 2.88 Mb or a 1.44 Mb disk (CDROM boot, DOSEMU...): \n"
  "./gujin -f boot.288                  # recognise 3-digit extension             \n"
  "./gujin -f fdimg --geometry=floppy:144     # --geometry same as -g             \n"
  " On a small dedicated FAT partition, update also the MBR; use EBIOS interface: \n"
  "./gujin --mbr /dev/sda4     # Will ERASE /dev/sda4                             \n"
  " Check what will be done, ask to use BIOS instead of the default EBIOS for HD: \n"
  "./gujin /dev/null -g=/dev/sda4 --disk=BIOS:0x80 --verbose                      \n"
  "   (note here the three interresting target: /dev/null, /dev/sda or /dev/sda4) \n"
  " Create a floppy disk image (without partition table) of the size of the device\n"
  "   with a standard FAT filesystem (like a USB Thumb drive in USB-FDD mode):    \n"
  "./gujin /dev/sdg --disk=BIOS:0x00,auto                                         \n"
  " Create a hard disk image (with last partition formated to FAT and covering    \n"
  "   all the space, containning file boot.bio) for CompactFlash in IDE adapter:  \n"
  " umount /dev/sdg1   # The MBR cannot be rewritten if a partition is mounted    \n"
  " dd if=/dev/zero of=/dev/sdg bs=512 count=1 # erase MBR/0xAA55 yourself        \n"
  "   and then type:                                                              \n"
  "./gujin /dev/sdg --disk=BIOS:0x80,auto --cmdline=\"\" \\                       \n"
  "                       --mbr-device=/dev/sdg --partition_index=4 --read_retry  \n"
  " You may have to force --disk=BIOS:0x80 or --disk=BIOS:0x0.                    \n"
  " The previous commands just need read/write access to the target partitions,   \n"
  " and r/w acces to the whole disk for --mbr-device, because they are using the  \n"
  " (E)BIOS interface. You need to be root to probe IDE stuff in /proc and /sys.  \n"
  " Ask to use the best ide interface, probe for disk capabilities:               \n"
  "./gujin /dev/hdb4 --disk=ide:hdb               # --disk same as -d             \n"
  " You want to overwrite the Master/Slave because you will physically swap the   \n"
  " hardware disks after setting them up, /dev/hdb4 boot sector will be loaded by \n"
  " the standard DOS MBR because this partition is marked active:                 \n"
  "./gujin /dev/hdb -g=/dev/hdb4 --disk=ide:lba28,ide0,master                     \n"
  " Loading with EBIOS, use the disk number passed in DL register at boot:        \n"
  "./gujin /dev/hdb4 --mbr-device=/dev/hdb --disk=EBIOS:0x80,auto                 \n"
  " If the target device name is a BEER partition, i.e. its partition number is 0,\n"
  " like in /dev/sda0 (exactly '/dev/[hs]d[a-z]0'), the --geometry, --disk,       \n"
  " --mbr-device and --beer-device are automatically generated (if not given).    \n"
  " In the standard case, get root and type:                                      \n"
  "./gujin /boot/gujin.ebios                                                      \n"
	);
      }
  }

static inline unsigned
treat_verbose (struct verbose_s *verbose, const char *tmp)
  {
  if (*tmp == '\0')
      *verbose = (struct verbose_s) {
	.warning = 1,
#define TREAT(letter, deflt, field)	.field = !deflt,
	VERBOSE_INSTANCIATE(TREAT)
#undef TREAT
	};
    else if (*tmp == '+' || *tmp == '=') {
      if (*tmp == '=')
	  *verbose = (struct verbose_s) {
#define TREAT(letter, deflt, field)	.field = 0,
		VERBOSE_INSTANCIATE(TREAT)
#undef TREAT
		};
      while (*++tmp) {
	  switch (*tmp) {
#define TREAT(letter, deflt, field) \
    case letter: case letter - 'a' + 'A': verbose->field = 1; break;
	VERBOSE_INSTANCIATE(TREAT)
#undef TREAT
	      default:
		PARAMERR ("unrecognised letter '%c' for -w or --verbose\n", *tmp);
		PRINTF ("List of valid letters:\n");
#define TREAT(letter, deflt, field)	PRINTF (#letter " : " #field "\n");
		VERBOSE_INSTANCIATE(TREAT)
#undef TREAT
		return 1;
	      }
	  }
      }
    else if (*tmp == '-') {
      while (*++tmp) {
	  switch (*tmp) {
#define TREAT(letter, deflt, field) \
    case letter: case letter - 'a' + 'A': verbose->field = 0; break;
	VERBOSE_INSTANCIATE(TREAT)
#undef TREAT
	      default:
		PARAMERR ("unrecognised letter '%c' for -w or --verbose\n", *tmp);
		PRINTF ("List of valid letters:\n");
#define TREAT(letter, deflt, field)	PRINTF (#letter " : " #field "\n");
		VERBOSE_INSTANCIATE(TREAT)
#undef TREAT
		return 1;
	      }
	  }
      }
    else {
      PARAMERR ("'-w' or '--verbose' followed by something but not '=' nor '+' nor '-'.\n");
      return 1;
      }
  return 0;
  }

static inline union fourbyte_u
endname (struct geometry_s *geo, enum gujin2_load_t gujin2_load)
  {
  union fourbyte_u str = {};

  if (gujin2_load == gujin2_load_bios) {
      str.array[0] = 'B'; str.array[1] = 'I'; str.array[2] = 'O';
      if (geo->heads == 2 && geo->BytePerSector == 512) {
	  if (geo->cylinders == 80) {
	      if (geo->sectpertrack == 36) {
		  str.array[0] = '2'; str.array[1] = '8'; str.array[2] = '8';
		  }
		else if (geo->sectpertrack == 21) {
		  str.array[0] = '1'; str.array[1] = '6'; str.array[2] = '8';
		  }
		else if (geo->sectpertrack == 15) {
		  str.array[0] = '1'; str.array[1] = '2'; str.array[2] = '0';
		  }
		else if (geo->sectpertrack == 18) {
		  str.array[0] = '1'; str.array[1] = '4'; str.array[2] = '4';
		  }
		else if (geo->sectpertrack == 9) {
		  str.array[0] = '7'; str.array[1] = '2'; str.array[2] = '0';
		  }
	      }
	    else if (   geo->cylinders == 40
		     && geo->sectpertrack == 9) {
	      str.array[0] = '3'; str.array[1] = '6'; str.array[2] = '0';
	      }
	  }
      }
    else if (gujin2_load == gujin2_load_ebios) {
      str.array[0] = 'E'; str.array[1] = 'B'; str.array[2] = 'I';
      }
    else if (gujin2_load == gujin2_load_ide_chs) {
      str.array[0] = 'C'; str.array[1] = 'H'; str.array[2] = 'S';
      }
    else if (gujin2_load == gujin2_load_PIC) {
      str.array[0] = 'P'; str.array[1] = 'I'; str.array[2] = 'C';
      }
    else if (gujin2_load == gujin2_load_CDROM_noemul) {
      str.array[0] = 'B'; str.array[1] = 'C'; str.array[2] = 'D';
      }
    else if (gujin2_load == gujin2_load_ide_lba) {
      str.array[0] = 'L'; str.array[1] = 'B'; str.array[2] = 'A';
      }
    else if (gujin2_load == gujin2_load_ide_lba48) {
      str.array[0] = 'L'; str.array[1] = '4'; str.array[2] = '8';
      }
    else {
      str.array[0] = 'H'; str.array[1] = 'D'; str.array[2] = ' ';
      }
  return str;
  }

/**
 ** Independant stuff for gujin_report(), just packed in to have a single executable:
 **/
static unsigned char fourKbuffer[4096];

int gujin_report_signature (void)
  {
  farptr irq13;
  struct {
      unsigned char  jmp_instruction[3]; /* jmp 0x8c (E9 89 00) for Gujin-2.8.2- */
      unsigned char  int13sf_sig[8];
      unsigned char  gujin_sig[8];	/* Officially "Vendor ID string" */
      unsigned       oldintaddr;
      unsigned       flags;
      unsigned char  int13biosdisk;
      unsigned char  int13filename[64];
      unsigned char  host_bios;
      unsigned char  ide_mask;
      unsigned short ide_base;
      unsigned char  ebios_bustype[4];
      unsigned char  ebios_Interface[8];
      unsigned char  ebios_bus[8];
      unsigned char  ebios_device[16];
      unsigned long long  partition_start;
      // Gujin-2.8.3+
      unsigned long long  partition_length;
      unsigned char  partition_type; /* 1 for ISO9660, 2 for Ext4fs, 3 for vfat */
      } __attribute__((packed)) signature;

  int fmem = open("/dev/mem", O_RDONLY);
  if (fmem == -1)
      perror ("Cannot open /dev/mem");
    else if (lseek64(fmem, 4 * 0x13, SEEK_SET) != 4 * 0x13)
      perror ("Cannot seek /dev/mem to see IRQ13 content");
    else if (read(fmem, &irq13, sizeof(irq13)) != sizeof(irq13))
      perror ("Cannot read /dev/mem to get IRQ13 content");
    else {
      unsigned linaddr = ((irq13 >> 16) << 4) + (irq13 & 0xFFFF);
      if (lseek64(fmem, linaddr, SEEK_SET) != linaddr)
	  perror ("Cannot seek /dev/mem to see IRQ13 TSR content");
	else if (read(fmem, &signature, sizeof(signature)) != sizeof(signature))
	  perror ("Cannot read /dev/mem to get IRQ13 TSR content");
	else if (memcmp(signature.int13sf_sig, "$INT13SF", sizeof(signature.int13sf_sig)))
	  fprintf(stderr, "No '$INT13SF' signature associated with DISK BIOS IRQ\n");
	else if (memcmp(signature.gujin_sig, "Gujin   ", sizeof(signature.gujin_sig)))
	  fprintf(stderr, "'$INT13SF' signature found but no Gujin signature with DISK BIOS IRQ\n");
	else {
	  const char *partition_mount_type = "auto";
	  printf ("# Automated scripts should keep the last line by 'gujin --report-signature | tail -1'\n");
	  printf ("previous_irq13=0x%X\n", signature.oldintaddr);
	  printf ("flags_irq13=0x%X\n", signature.flags);
	  /* following fields not defined by standard: */
	  printf ("biosdisk=0x%X\n", signature.int13biosdisk);
	  printf ("filename=%s\n", signature.int13filename);
	  printf ("host_bios=0x%X\n", signature.host_bios);
	  printf ("ide_base=0x%X\n", signature.ide_base);
	  printf ("ide_mask=0x%X\n", signature.ide_mask);
	  //printf ("ebios_bustype=%4s\n", signature.ebios_bustype); // displays "ebios_bustype=PCI USB"
	  {
	  char xtmp[sizeof(signature.ebios_bustype)+1] = {0};
	  *(int *)xtmp = *(int *)signature.ebios_bustype;
	  printf ("ebios_bustype=%s\n", xtmp);
	  }
	  //printf ("ebios_interface=%8s\n", signature.ebios_Interface);
	  {
	  char xtmp[sizeof(signature.ebios_Interface)+1] = {0};
	  *(long long *)xtmp = *(long long *)signature.ebios_Interface;
	  printf ("ebios_interface=%s\n", xtmp);
	  }
	  printf ("ebios_device=%02X,%02X,%02X,%02X,%02X,%02X,%02X,%02X,%02X,%02X,%02X,%02X,%02X,%02X,%02X,%02X\n",
		 signature.ebios_device[0],signature.ebios_device[1],signature.ebios_device[2],signature.ebios_device[3],
		 signature.ebios_device[4],signature.ebios_device[5],signature.ebios_device[6],signature.ebios_device[7],
		 signature.ebios_device[8],signature.ebios_device[9],signature.ebios_device[10],signature.ebios_device[11],
		 signature.ebios_device[12],signature.ebios_device[13],signature.ebios_device[14],signature.ebios_device[15]);
	  char readlink1_buff[80], *readlink1_ptr = 0, readlink2_buff[80], *readlink2_ptr = 0;
	  unsigned char slave;
	  if (*(long long *)&signature.ebios_Interface == *(long long *)"ATA     "
		|| *(long long *)&signature.ebios_Interface == *(long long *)"SATA    "
		|| *(long long *)&signature.ebios_Interface == *(long long *)"ATAPI   ")
	      slave = signature.ebios_device[0];
	    else
	      slave = !!(signature.ide_mask & 0x10);
	  if ((int *)&signature.ebios_bustype == (int *)"ISA ") {
	      printf ("isa_address=0x%X", *(unsigned short *)&signature.ebios_bus);
	      char disk_by_path[256];
	      sprintf (disk_by_path, "/dev/disk/by-path/pci-0000:00:1f.1-scsi-0:0:%u:0", slave);
	      int nbbytes = readlink(disk_by_path, readlink1_buff, sizeof(readlink1_buff));
	      if (nbbytes != -1) {
		  readlink1_ptr = readlink1_buff;
		  while (*readlink1_ptr) readlink1_ptr++;
		  while (*readlink1_ptr != '/' && readlink1_ptr >= readlink1_buff)
		      readlink1_ptr--;
		  if (*readlink1_ptr == '/')
		      readlink1_ptr++;
		  if (*readlink1_ptr)
		      printf ("resolved_disk_by_path=%s\n", readlink1_ptr);
		    else
		      readlink1_ptr = 0;
		  }
		else
		  printf ("#disk_by_path=%s do not exists\n", disk_by_path);
	      }
	    else {
	      printf ("PCI_bus=%u\n", signature.ebios_bus[0]);
	      printf ("PCI_device=%u\n", signature.ebios_bus[1]);
	      printf ("PCI_function=%u\n", signature.ebios_bus[2]);
		// /dev/disk/by-path/pci-0000:00:1d.7-usb-0:5:1.0-scsi-0:0:0:0 -> ../../sdb
		// /dev/disk/by-path/pci-0000:00:1f.1-scsi-0:0:0:0 -> ../../sr0
		// /dev/disk/by-path/pci-0000:00:1f.2-scsi-0:0:0:0 -> ../../sda
		// /dev/disk/by-path/pci-0000:00:1f.2-scsi-0:0:0:0-part1 -> ../../sda1
		// /dev/disk/by-path/pci-0000:00:1f.2-scsi-0:0:0:0-part2 -> ../../sda2
		// /dev/disk/by-path/pci-0000:00:1f.2-scsi-0:0:0:0-part3 -> ../../sda3
		// /dev/disk/by-path/pci-0000:00:1f.2-scsi-0:0:0:0-part4 -> ../../sda4
		// /dev/disk/by-path/pci-0000:00:1f.2-scsi-0:0:0:0-part5 -> ../../sda5
		// /dev/disk/by-path/pci-0000:00:1f.2-scsi-0:0:0:0-part6 -> ../../sda6
		// /dev/disk/by-path/pci-0000:00:1f.2-scsi-0:0:1:0 -> slave
		// /dev/disk/by-path/pci-0000:06:04.2 -> ../../mmcblk0
		// /dev/disk/by-path/pci-0000:06:04.2-part1 -> ../../mmcblk0p1
		// /dev/disk/by-path/pci-0000:00:1d.7-usb-0:1:1.0-scsi-0:0:0:0 -> ../../sdb
		// On a USB HUB:
		// /dev/disk/by-path/pci-0000:00:12.1-usb-0:X.Y:1.0-scsi-0:0:0:0 -> ../../sdb
	      char disk_by_path[256];
	      int cpt, nbbytes = nbbytes; /* inited b4 used */
	      if (*(int *)signature.ebios_Interface != *(int *)"USB ") {
		  sprintf (disk_by_path, "/dev/disk/by-path/pci-%04x:%02x:%02x.%x-scsi-0:0:%u:0",
			0, signature.ebios_bus[0], signature.ebios_bus[1], signature.ebios_bus[2], slave);
		  nbbytes = readlink(disk_by_path, readlink1_buff, sizeof(readlink1_buff));
		  }
		else {
		  unsigned cptmatch = 0, cpt2;
		  unsigned char PCI_function = signature.ebios_bus[2];
retry_match:
		  for (cpt = 1; cpt <= 8; cpt++) {
		      for (cpt2 = 0; cpt2 <= 4; cpt2++) {
			  if (cpt2 == 0) // no USB HUB
			      sprintf (disk_by_path, "/dev/disk/by-path/pci-%04x:%02x:%02x.%x-usb-0:%u:1.0-scsi-0:0:%u:0",
				0, signature.ebios_bus[0], signature.ebios_bus[1], PCI_function, cpt, slave);
			    else // go past 1 level USB HUB (USBs have 4 ports):
			      sprintf (disk_by_path, "/dev/disk/by-path/pci-%04x:%02x:%02x.%x-usb-0:%u.%u:1.0-scsi-0:0:%u:0",
				0, signature.ebios_bus[0], signature.ebios_bus[1], PCI_function, cpt, cpt2, slave);
			  int tmpnbbytes = readlink(disk_by_path, readlink1_buff, sizeof(readlink1_buff));
			  if (tmpnbbytes != -1) {
			      nbbytes = tmpnbbytes;
			      readlink1_buff[tmpnbbytes] = '\0';
			      if (cptmatch++ == 0)
				  printf ("#disk_by_path=%s links to %s\n", disk_by_path, readlink1_buff);
				else {
				  printf ("#disk_by_path=%s links to %s, two disk on USB, abort\n", disk_by_path, readlink1_buff);
				  break;
				  }
			      }
			  }
		      }
		  if (cptmatch == 0 && PCI_function == 2) {
		      // happens when loaded with OHCI and Linux runs EHCI (USB disk on USB HUB):
		      // lspci: 00:12.1 USB Controller: ATI Technologies Inc SB700 USB OHCI1 Controller
		      //        00:12.2 USB Controller: ATI Technologies Inc SB700/SB800 USB EHCI Controller
		      PCI_function = 1;
		      goto retry_match;
		      }
		  if (cptmatch != 1)
		      nbbytes = -1;
		  }
	      if (nbbytes != -1) {
		  readlink1_ptr = readlink1_buff;
		  while (*readlink1_ptr) readlink1_ptr++;
		  while (*readlink1_ptr != '/' && readlink1_ptr >= readlink1_buff)
		      readlink1_ptr--;
		  if (*readlink1_ptr == '/')
		      readlink1_ptr++;
		  if (*readlink1_ptr)
		      printf ("resolved_disk_by_path=%s\n", readlink1_ptr);
		    else
		      readlink1_ptr = 0;
		  }
		else
		  printf ("#disk_by_path not found\n");
	      }
	  if (*(long long *)&signature.ebios_Interface == *(long long *)"ATA     "
		|| *(long long *)&signature.ebios_Interface == *(long long *)"SATA    ")
	      printf ("ATA_device=%u\n", signature.ebios_device[0]);
	    else if (*(long long *)&signature.ebios_Interface == *(long long *)"ATAPI   ") {
	      printf ("ATAPI_device=%u\n", signature.ebios_device[0]);
	      printf ("ATAPI_LUN=%u\n", signature.ebios_device[1]);
	      }
	    else if (*(long long *)&signature.ebios_Interface == *(long long *)"SCSI    ") {
	      printf ("SCSI_PUN=%u\n", *(unsigned short *)signature.ebios_device);
	      printf ("SCSI_LUN=%u\n", *(unsigned *)&signature.ebios_device[2]);
	      }
	    else if (*(long long *)&signature.ebios_Interface == *(long long *)"USB     "
		  || *(long long *)&signature.ebios_Interface == *(long long *)"1394    "
		  || *(long long *)&signature.ebios_Interface == *(long long *)"FIBRE   "
		  || *(long long *)&signature.ebios_Interface == *(long long *)"I2O     ") {
	      printf ("serial_number=0x%llX, 0x%llX\n", ((long long *)signature.ebios_device)[0], ((long long *)signature.ebios_device)[1]);
	      // /dev/disk/by-id/usb-_USB_DISK_2.0_07740DB01FF3-0:0 -> ../../sdb
	      // /dev/disk/by-id/wwn-0x500000e04370e76a -> ../../sda
	      // /dev/disk/by-id/ata-FUJITSU_MJA2400BH_G2_K90CT9326VAN -> ../../sda
	      char pattern[32];
	      sprintf (pattern, "_%02X%02X%02X%02X%02X%02X-0:0",
		signature.ebios_device[9], signature.ebios_device[8], signature.ebios_device[7],
		signature.ebios_device[6], signature.ebios_device[5], signature.ebios_device[4]);
	      unsigned pattern_len = strlen(pattern);
	      struct dirent **namelist;
	      int nbbytes = -1, nb = scandir ("/dev/disk/by-id", &namelist, 0, alphasort);
	      if (nb >= 0) {
		  char first_usb[64] = "", single_usb = 0;
		  unsigned nbmatch = 0;
		  while (nb--) {
		      int len = strlen(namelist[nb]->d_name); // or: int len = _D_EXACT_NAMLEN(namelist[nb]);
		      if (*(int*)namelist[nb]->d_name == *(int*)"usb-" && *(int*)&namelist[nb]->d_name[len-4] == *(int*)"-0:0") {
			  if (first_usb[0] == '\0') {
			      strcpy (first_usb, namelist[nb]->d_name);
			      single_usb = 1;
			      }
			    else
			      single_usb = 0;
			  if (!strcmp(&namelist[nb]->d_name[len-pattern_len], pattern)) {
			      char tmpusb[128] = "/dev/disk/by-id/";
			      strcat (tmpusb, namelist[nb]->d_name);
			      int tmpnbbytes = readlink(tmpusb, readlink2_buff, sizeof(readlink2_buff));
			      if (tmpnbbytes != -1) {
				  readlink2_buff[tmpnbbytes] = '\0';
				  if (nbbytes != -1)
				      printf ("# WARNING: another match of serial number: %s, will abort\n", namelist[nb]->d_name);
				    else
				      printf ("# match of '/dev/disk/by-id/usb-*%s': %s\n", pattern, namelist[nb]->d_name);
				  nbbytes = tmpnbbytes;
				  nbmatch++;
				  }
			      }
			  }
		      free (namelist[nb]);
		      }
		  free (namelist);
		  if (nbmatch > 1)
		      nbbytes = -1;
		    else if (nbmatch == 0 && single_usb) {
		      char tmpusb[128] = "/dev/disk/by-id/";
		      strcat (tmpusb, first_usb);
		      int tmpnbbytes = readlink(tmpusb, readlink2_buff, sizeof(readlink2_buff));
		      if (tmpnbbytes != -1) {
			  readlink2_buff[tmpnbbytes] = '\0';
			  printf ("# single USB /dev/disk/by-id/usb-*: %s\n", first_usb);
			  nbbytes = tmpnbbytes;
			  pattern[0] = '\0';
			  }
		      }
		  }
	      if (nbbytes != -1) {
		  printf ("#disk_by_id=usb-*%s links to %s\n", pattern, readlink2_buff);
		  readlink2_ptr = readlink2_buff;
		  while (*readlink2_ptr) readlink2_ptr++;
		  while (*readlink2_ptr != '/' && readlink2_ptr >= readlink1_buff)
		      readlink2_ptr--;
		  if (*readlink2_ptr == '/')
		      readlink2_ptr++;
		  if (*readlink2_ptr)
		      printf ("resolved_disk_by_id=%s\n", readlink2_ptr);
		    else
		      readlink2_ptr = 0;
		  }
		else
		  printf ("#disk_by_id=usb-*%s do not exists\n", pattern);
	      }
	    else if (*(long long *)&signature.ebios_Interface == *(long long *)"RAID    ")
	      printf ("RAID_array_number=%u\n", *(unsigned *)signature.ebios_device);
	  printf ("partition_start=%llu\n", signature.partition_start);
	  /* Gujin <= v2.8.2 did not write fields PARTITION_LENGTH and PARTITION_TYPE, but easily recognised: */
	  if (signature.partition_length != 0xE0FA809C) {
	      printf ("partition_length=%llu\n", signature.partition_length);
	      if (signature.partition_type == 1)
		  partition_mount_type = "iso9660";
		else if (signature.partition_type == 2)
		  partition_mount_type = "ext4fs";
		else if (signature.partition_type == 3)
		  partition_mount_type = "vfat";
	      printf ("partition_type=%s\n", partition_mount_type);
	      }
	    else
	      signature.partition_type = signature.partition_length = 0;
	  char partition_buffer[128], disk[128] = "", header = 1;
	  FILE *fdpart = fopen ("/proc/partitions", "r");
	  // TODO: also probe devices in /sys/block which are not in /proc/partitions like sr0 but not loop and fd
	  // Let's do that TODO in a dirty way, only looking at /sys/block/sr[0-9]:
	  char *sys_block_sr = "/sys/block/sr0"; struct stat statbuf;
	  sys_block_sr[13]--;
	  if (fdpart) while (fgets(partition_buffer, sizeof(partition_buffer), fdpart)
				|| (sys_block_sr[13]++ <= '9' && stat(sys_block_sr, &statbuf) == 0)) {
	      if (sys_block_sr[13] >= '0') {
		  strcpy (partition_buffer, &sys_block_sr[11]);
		  }
	      char *part = partition_buffer;
	      while (*part) part++;
	      while ((part[-1] == '\n' || part[-1] == ' ' || part[-1] == '\t') && part > partition_buffer)
		  *--part = '\0';
	      if (part == partition_buffer) {
		  header = 0;
		  continue;
		  }
	      if (header)
		  continue;
	      char *part_lastchar = --part, *src, *dst;
	      while (*part != ' ' && *part != '\t' && part >= partition_buffer) part--;
	      part++; /* part = "sda" || part = "sda2" || part = "mmcblk0" || part = "mmcblk0p1" */
	      char isdisk = (*part_lastchar < '0' || *part_lastchar > '9') || sys_block_sr[13] >= '0';
	      if (!strncmp(part, "mmcblk", strlen ("mmcblk"))) { /* mmcblk0 vs mmcblk0p1 */
		  while (*part_lastchar >= '0' && *part_lastchar <= '9' && part_lastchar >= partition_buffer)
		      part_lastchar--;
		  isdisk = (*part_lastchar != 'p');
		  }
	      unsigned long long partition_start = 0, partition_length = 0;
	      if (isdisk) {
		  for (src = part, dst = disk; *src; src++)
		      *dst++ = *src;
		  *dst = '\0';
		  char sysbuffer[128];
		  sprintf (sysbuffer, "/sys/block/%s/size", disk);
		  FILE *fdsys = fopen (sysbuffer, "r");
		  if (fdsys) {
		      if (fscanf (fdsys, " %lld ", &partition_length) != 1)
			  partition_length = 0;
		      fclose (fdsys);
		      }
		  }
		else {
		  char sysbuffer[128];
		  partition_length = 0;
		  sprintf (sysbuffer, "/sys/block/%s/%s/start", disk, part);
		  FILE *fdsys = fopen (sysbuffer, "r");
		  if (fdsys) {
		      if (fscanf (fdsys, " %lld ", &partition_start) != 1)
			  partition_start = 0;
		      fclose (fdsys);
		      }
		  sprintf (sysbuffer, "/sys/block/%s/%s/size", disk, part);
		  fdsys = fopen (sysbuffer, "r");
		  if (fdsys) {
		      if (fscanf (fdsys, " %lld ", &partition_length) != 1)
			  partition_length = 0;
		      fclose (fdsys);
		      }
		  }
//printf ("Disk: '%s', partition '%s', start %llu\n", disk, part, partition_start);
	      if (partition_start != signature.partition_start) continue;
	      if (partition_length) {
		  if (strcmp(disk, part)) { /* exact length when partition */
		      if (partition_length != signature.partition_length)
			  continue;
		      }
		    else { /* One USB disk give BIOS size 2007808 and linux size 2009088 */
		      // TODO: allow a delta only when there is no exact match...
		      int delta = partition_length - signature.partition_length;
		      if (delta < 0) delta = -delta;
		      if (delta > 2048)
			  continue;
		      }
		  }
	      if (readlink1_ptr && strncmp (readlink1_ptr, disk, strlen(readlink1_ptr))) continue;
	      if (readlink2_ptr && strncmp (readlink2_ptr, disk, strlen(readlink2_ptr))) continue;
	      if (*(int *)part == *(int *)"loop") continue; /* loop* */
	      printf ("boot_disk=%s\n", disk);
	      if (strcmp(disk, part))
		  printf ("boot_partition=%s\n", part);
	      FILE *fdmount = fopen ("/proc/mounts", "r");
	      char mount_buffer[512], fromiso_displayed = 0;
	      if (fdmount) while (fgets(mount_buffer, sizeof(mount_buffer), fdmount)) {
		  char *diskptr = mount_buffer, *mountptr, *xptr;
		  while (*diskptr && *diskptr != ' ' && *diskptr != '\t')
		      diskptr++;
		  mountptr = diskptr;
		  while (*mountptr == ' ' || *mountptr == '\t')
		      mountptr++;
		  xptr = mountptr;
		  while (*xptr && *xptr != ' ' && *xptr != '\t')
		      xptr++;
		  *xptr = '\0';
		  *diskptr = '\0';
		  while (*diskptr != '/' && diskptr >= mount_buffer)
		      diskptr--;
		  if (*diskptr == '/')
		      diskptr++;
		  if (!strcmp(diskptr, part)) {
		      printf ("fromiso=%s/%s\n", mountptr, signature.int13filename);
		      fromiso_displayed++;
		      }
		  }
	      if (!fromiso_displayed) /* Debian live name this directory /isofrom */
		  printf ("mount -t %s /dev/%s /isofrom\n", partition_mount_type, part);
	      if (fdmount)
		  fclose (fdmount);
	      }
	  if (fdpart)
	      fclose (fdpart);
	  close (fmem);
	  return 0;
	  }
      }
  if (fmem != -1)
      close (fmem);

  return -1;
  }

#define STRBEGINEQL(str, cst_str) !memcmp (str, cst_str, sizeof (cst_str) - 1)

static const char *const recognise_MBR_string[] = {
	"noAA55", "unknown", "nullname", "Gujin", "Grub",
	"zLILO", "lbaLILO", "LILO", "oldLILO", "GAG", "XoslLoad", "Windows",
	"EBIOS MBR", "BIOS MBR", "SYSLINUX", "ISOLINUX"
	};

typedef struct {
    bootbefore_t	before;
    unsigned char	code_part[512 - sizeof(bootbefore_t)
				- sizeof(boot1param_t)
				- sizeof (serialconf_t)
				- sizeof(bootafter_t) ];
    boot1param_t	param;
    serialconf_t	serial_configuration;
    bootafter_t		after;
    } __attribute__ ((packed)) bootsector_t;

extern inline unsigned long long
chs2lba (unsigned cylinder, unsigned head, unsigned sector,
	 unsigned short nbhead, unsigned short nbsectorpertrack)
  {
  return (cylinder * nbhead + head) * nbsectorpertrack + sector - 1;
  }

static unsigned recognise_MBR (const bootsector_t *bootsector)
  {
  unsigned *ptr = (unsigned *)((char *)bootsector + 0x170);

  while (ptr < (unsigned *)((char *)bootsector + 0x1A0)) {
      if (*ptr == *(unsigned *)"GRUB")
	  break;
      ptr = (unsigned *)((char *)ptr + 1);
      }

  if (bootsector->after.Signature0xAA55 != 0xAA55 || bootsector->before.part1.jmpinstruction[0] == 0)
      return 0;
    else if (*ptr == *(unsigned *)"GRUB")
      return 4;	/* also String is kept to previous value (maybe null) */
    else if (STRBEGINEQL ((char *)bootsector + 0x1D, "\x88\x16\x00\x08\xB4\x08\xCD\x13")
	  || STRBEGINEQL ((char *)bootsector + 0x1E7, "Boot error\r\n"))
      return 14;
    else if (*(unsigned long long *)bootsector->before.part1.String == 0)
      return 2;
    else if (STRBEGINEQL (bootsector->before.part1.String, "Gujin"))
      return 3;
    else if (STRBEGINEQL (&bootsector->before.part1.String[2], "zLILO"))
      return 5;
    else if (STRBEGINEQL (bootsector->before.part1.String, "lbaLILO"))
      return 6;
    else if (STRBEGINEQL (&bootsector->before.part1.String[3], "LILO"))
      return 7;
    else if (STRBEGINEQL ((char *)bootsector + 2, "LILO"))
      return 8;
    else if (STRBEGINEQL ((char *)bootsector + 0x55, "GAG: "))
      return 9;
    else if (STRBEGINEQL ((char *)bootsector + 0x181, "XOSLLOAD"))
      return 10;
    else if (STRBEGINEQL (bootsector->before.part1.String, "NTFS    "))
      return 11;
    else if (STRBEGINEQL ((char *)bootsector + 0x113, "\xB4\x42\x8B\xF4\xCD\x13\x61\x61"))
      return 12;
    else if (STRBEGINEQL ((char *)bootsector + 0x63, "\xB8\x01\x02\x57\xCD\x13\x5F"))
      return 13;
    else if (STRBEGINEQL ((char *)bootsector + 0x94, "isolinux.bin"))
      return 15;
    else
      return 1;
  }

int gujin_extract_mbr(const char *mbr_file, const char *input_file)
  {
  int returned = -1;
  struct stat buf;
  if (stat(mbr_file, &buf) == 0)
      printf ("ERROR: Cannot write recovery MBR in file '%s', file already exists\n", mbr_file);
    else if (access (mbr_file, W_OK) == 0)
      printf ("ERROR: Cannot write recovery MBR in file '%s', no access right\n", mbr_file);
    else {
      FILE *mbrfd = fopen (mbr_file, "w");
      unsigned char mbrbuf[512];
      if (mbrfd == 0)
	  printf ("ERROR: Cannot write recovery MBR in file '%s', cannot open\n", mbr_file);
	else {
	  FILE *inputfd = fopen (input_file, "r");
	  if (!inputfd)
	      printf ("ERROR: Cannot open input file '%s'\n", input_file);
	    else if (fseek(inputfd, -512, SEEK_END) == -1)
	      printf ("ERROR: Cannot seek input file '%s' to (end-512 bytes)\n", input_file);
	    else if (fread(mbrbuf, sizeof(mbrbuf), 1, inputfd) != 1)
	      printf ("ERROR: Cannot read 512 bytes from input file '%s'\n", input_file);
	    else if (fclose(inputfd))
	      printf ("ERROR: Cannot close input file '%s'\n", input_file);
	    else if (inputfd = 0, fwrite(mbrbuf, offsetof(bootsector_t, after.bootsect_partition), 1, mbrfd) != 1)
	      printf ("ERROR: Cannot write recovery MBR in file '%s', write failed\n", mbr_file);
	    else if (fclose(mbrfd))
	      printf ("ERROR: Cannot write recovery MBR in file '%s', close failed\n", mbr_file);
	    else
	      mbrfd = 0, returned = 0;
	  if (inputfd && fclose(inputfd))
	      printf ("ERROR: Cannot close input file '%s'\n", input_file);
	  }
      if (mbrfd) {
	  fclose(mbrfd);
	  printf ("ERROR: Cannot write recovery MBR in file '%s', close failed\n", mbr_file);
	  }
	else {
	  printf ("The saved MBR from file '%s' has been saved in '%s', partition table not included\n", input_file, mbr_file);
	  printf ("That MBR seems to be: %s\n", recognise_MBR_string[recognise_MBR ((const bootsector_t *)mbrbuf)]);
	  printf ("If it really was the Gujin file you were using, you can restore the MBR by:\n");
	  printf ("cat %s > /dev/sdX  (replace X with the right drive)\n", mbr_file);
	  }
      }
  return returned;
  }

enum access_e { access_ERROR, read_BIOS, write_BIOS, read_EBIOS, write_EBIOS,
    read_IDEchs, write_IDEchs, read_IDElba, write_IDElba, read_IDElba48, write_IDElba48, statusHD };
static const char *const access_string[] = {
    "access_ERROR", "read_BIOS", "write_BIOS", "read_EBIOS", "write EBIOS",
    "read_IDEchs", "write_IDEchs", "read_IDElba", "write_IDElba", "read_IDElba48", "write_IDElba48",
    "statusHD"
    };
static enum access_e analyse_bootloader2_cmd (int dp, const union diskcmd *cmd,
	unsigned long long *lba, unsigned *nb, const bootbefore_t *bootbefore,
	unsigned log)
  {
  if ((cmd->int1302.reg_ax >> 8) == 0x42 || (cmd->int1302.reg_ax >> 8) == 0x43){
      *lba = cmd->int1342.lba;
      *nb = cmd->int1342.nb_block;
      if (log) printf ("read EBIOS %u sectors at lba %llu\n", *nb, *lba);
      return (cmd->int1302.reg_ax >> 8) == 0x42 ? read_EBIOS : write_EBIOS;
      }
    else if ((cmd->int1302.reg_ax >> 8) == 0x02 || (cmd->int1302.reg_ax >> 8) == 0x03) {
      unsigned C = 4 * (cmd->int1302.reg_cx & 0xC0) + (cmd->int1302.reg_cx >> 8),
		H = cmd->int1302.reg_dx >> 8,
		S = cmd->int1302.reg_cx & 0x3F;
      *lba = chs2lba (C, H, S, bootbefore->part1.NbHead, bootbefore->part1.NbSectorpertrack);
      *nb = cmd->int1302.reg_ax & 0xFF;
      if (log) printf ("read BIOS %u sectors at C/H/S %u/%u/%u, H0/C0 %u/%u, i.e. lba %llu\n",
		*nb, C, H, S, bootbefore->part1.NbHead, bootbefore->part1.NbSectorpertrack, *lba);
      return (cmd->int1302.reg_ax >> 8) == 0x02 ? read_BIOS : write_BIOS;
      }
    else if ((cmd->int1302.reg_ax >> 8) == 0x01) {
      if (log) printf ("first 'status of last HD operation' on drive dx=0x%X (trick for >512 bytes/sectors)\n", cmd->int1302.reg_dx);
      *lba = 0;
      *nb = 0;
      return statusHD;
      }
    else if ((cmd->int1302.reg_ax >> 8) != 0xF0
		|| (   cmd->hardide.ide_command != 0x20 && cmd->hardide.ide_command != 0x24
		    && cmd->hardide.ide_command != 0x30 && cmd->hardide.ide_command != 0x34)) {
      *lba = 0;
      *nb = 0;
      if (log) {
	  if (cmd->int1302.reg_ax >> 8 != 0 || cmd->hardide.ide_command != 0)
	      printf ("NOT an IDE command to read (0x%X,0x%X) at addr 0x%lX?\n", cmd->int1302.reg_ax >> 8, cmd->hardide.ide_command, (unsigned long)cmd - (unsigned long)fourKbuffer);
	    else
	      printf ("End load chain.\n");
	  }
      return access_ERROR;
      }
    else if ((cmd->hardide.ide_command == 0x20 || cmd->hardide.ide_command == 0x30) && (cmd->hardide.lba_head & 0x40) == 0) { /* LBA bit */
      unsigned C = 256 * cmd->hardide.cylinder_high + cmd->hardide.cylinder_low,
		H = cmd->hardide.lba_head & 0x0F,
		S = cmd->hardide.sector;
// How to get the IDE nbhead/nbsectorpertrack under Linux? - this is probably false:
//      *lba = chs2lba (C, H, S, dp->nbhead, dp->nbsectorpertrack);
      *lba = chs2lba (C, H, S, bootbefore->part1.NbHead, bootbefore->part1.NbSectorpertrack);
      *nb = cmd->hardide.nb_sect;
      if (log) printf ("read IDECHS %u sectors at C/H/S %u/%u/%u, H0/C0 %u/%u, i.e. lba %llu\n",
//		*nb, C, H, S, dp->nbhead, dp->nbsectorpertrack, *lba);
		*nb, C, H, S, bootbefore->part1.NbHead, bootbefore->part1.NbSectorpertrack, *lba);
      return cmd->hardide.ide_command == 0x20 ? read_IDEchs : write_IDEchs;
      }
    else {
      if (cmd->hardide.ide_command == 0x24 || cmd->hardide.ide_command == 0x34) {
	  *nb = 256 * cmd->hardide.lba48_nb_sect + cmd->hardide.sector;
	  *lba |= (unsigned long long)cmd->hardide.lba48_cylinder_high << 40;
	  *lba |= (unsigned long long)cmd->hardide.lba48_cylinder_low << 32;
	  *lba |= (unsigned)cmd->hardide.lba48_sector << 24;
	  *lba |= (unsigned)cmd->hardide.cylinder_high << 16;
	  *lba |= (unsigned)cmd->hardide.cylinder_low << 8;
	  *lba |= (unsigned)cmd->hardide.sector;
	  if (log) printf ("read IDELBA48 %u sectors at lba %llu\n", *nb, *lba);
	  return cmd->hardide.ide_command == 0x24 ? read_IDElba48 : write_IDElba48;
	  }
	else {
	  *nb = cmd->hardide.sector;
	  *lba |= (unsigned)(cmd->hardide.lba_head & 0x0F) << 24;
	  *lba |= (unsigned)cmd->hardide.cylinder_high << 16;
	  *lba |= (unsigned)cmd->hardide.cylinder_low << 8;
	  *lba |= (unsigned)cmd->hardide.sector;
	  if (log) printf ("read IDELBA %u sectors at lba %llu\n", *nb, *lba);
	  return cmd->hardide.ide_command != 0x20 ? read_IDElba : write_IDElba;
	  }
      }
  }

unsigned char second_sector_in_file[4096];	/* max sector size */
unsigned char second_sector_in_file_valid = 0;
unsigned char last_sector_in_file[4096];	/* max sector size */
unsigned char last_sector_in_file_valid = 0;

static unsigned load_gujin_sector (int dp, unsigned sector, unsigned char *buffer_fptr, unsigned nbsectors,
	const bootloader2_t *bootchain, const boot1param_t *boot1param, const bootbefore_t *bootbefore,
	unsigned log)
{
  unsigned nb, gujin_sector = 1; /* MBR already there */
  const bootloader2_t *bootchain_ptr = &boot1param->bootloader2;
  unsigned long long lba;
  unsigned cpt_bootchain = 0;

  do {
      analyse_bootloader2_cmd (dp, &bootchain_ptr->bootloader2_cmd, &lba, &nb, bootbefore, log);
      if (nbsectors == 0) { /* 2048 bytes/sectors */
	  if (lba == 0 && nb == 0 && cpt_bootchain != 0)
	      return 0;
	  }
	else if (sector >= gujin_sector && sector < gujin_sector + nb) {
	  unsigned long long searched_lba = lba + sector - gujin_sector;
	  unsigned nb_available = gujin_sector + nb - sector,
			nb_loaded = (nbsectors <= nb_available)? nbsectors : nb_available;
	  /* If we are inside a partition, that is values from the start of the disk which are accessed: */
	  struct stat sb;
	  unsigned long long start_sector = 0;
	  if (fstat(dp, &sb) == 0 && S_ISBLK(sb.st_mode) && get_device_start_sector (dp, &start_sector, 0) == 0)
	      searched_lba -= start_sector;
	  if (log) printf ("Read %u sectors at %llu (adjusted %llu): ", nb_loaded, searched_lba, start_sector);
	  if (lseek64 (dp, searched_lba * bootbefore->part1.Bytepersector, SEEK_SET) != searched_lba * bootbefore->part1.Bytepersector) {
	      if (log) puts ("lseek64 failed!");
	      perror("device seek (installed in B.E.E.R. inaccessible area?)");
	      return -1;
	      }
	  if (read (dp, buffer_fptr, nb_loaded * bootbefore->part1.Bytepersector) != nb_loaded * bootbefore->part1.Bytepersector) {
	      if (log) puts ("read failed!");
	      perror("device read (installed in B.E.E.R. inaccessible area?)");
	      return -1;
	      }
	  if (gujin_sector == 1 && second_sector_in_file_valid && memcmp (buffer_fptr, second_sector_in_file, bootbefore->part1.Bytepersector)) {
	      printf ("ERROR: sector read at %llu do not contain the second sector read from the file, trying to correct...\n", searched_lba);
	      unsigned idxmem;
	      for (idxmem = 0; idxmem < bootbefore->part1.Bytepersector - 4; idxmem++)
		  if (second_sector_in_file[idxmem] != buffer_fptr[idxmem])
		       break;
	      printf ("ERROR: at offset 0x%X, second_sector_in_file %02X %02X %02X %02X, Buffer %02X %02X %02X %02X\n",
		idxmem,
		second_sector_in_file[idxmem+0], second_sector_in_file[idxmem+1], second_sector_in_file[idxmem+2], second_sector_in_file[idxmem+3],
		buffer_fptr[idxmem+0], buffer_fptr[idxmem+1], buffer_fptr[idxmem+2], buffer_fptr[idxmem+3]);
	      memcpy (buffer_fptr, second_sector_in_file, bootbefore->part1.Bytepersector);
	      static unsigned display_only_once = 1;
	      if (display_only_once) {
		  unsigned long long dpos = lseek64 (dp, 0, SEEK_CUR);
		  printf ("ERROR: device seek position after reading 2nd sector reported as %llu i.e. %llu * %u\n", dpos, dpos / bootbefore->part1.Bytepersector, bootbefore->part1.Bytepersector);
		  printf ("ERROR: use the 'showmap' tool (make showmap) on the second stage file to see where it should be on disk!\n");
		  printf ("ERROR: correct the showmap value by multiplying by filesystem block size and add start of partition.\n");
		  printf ("ERROR: If you arrive here, you have probably used the wrong file to disinstall, having\n"
		      "ERROR: installed Gujin another time elsewhere; if you are sure that is not the case,\n"
		      "ERROR: try to do a umount/mount cycle to flush that filesystem internal cache, else maybe\n"
		      "ERROR: use the '--extract-mbr=' option to extract the recovery MBR stored in that file.\n"
		      "ERROR: This will also happen if the filesystem has buggy support for FIBMAP/FIGETBSZ ioctl,\n"
		      "ERROR: in that case Gujin cannot be installed on this filesystem\n");
		  display_only_once = 0;
		  }
	      }
	  if (log) puts ("OK");
	  nbsectors -= nb_loaded;
	  if (nbsectors == 0)
	      return 0;
	  sector += nb_loaded;
	  buffer_fptr += nb_loaded * bootbefore->part1.Bytepersector;
	  }
      gujin_sector += nb;
      bootchain_ptr = (const bootloader2_t *)(fourKbuffer + bootchain_ptr->header.next);
      if (bootchain_ptr != &bootchain[cpt_bootchain])
	  if (log) printf ("Unusual link: bootchain_ptr = 0x%lX, &bootchain[%u] = 0x%lX\n",
		(unsigned long)bootchain_ptr - (unsigned long)fourKbuffer, cpt_bootchain, (unsigned long)&bootchain[cpt_bootchain] - (unsigned long)fourKbuffer);
      cpt_bootchain++;
      } while (nb != 0 || cpt_bootchain == 1);
  return -2;
}

static int nb_fragment_file (int file_desc, unsigned long long *last_lba)
  {
  unsigned nb_block, block_index, current_block = 0, block_count = 0;
  unsigned bmap_block_size, fragment_cpt = 1;
  unsigned long block;

  unsigned filesize = lseek64 (file_desc, 0, SEEK_END);
  if (filesize == 0 || filesize >= 640*1024) {
      perror ("filesize invalid");
      return -1;
      }

  if (ioctl (file_desc, FIGETBSZ, &bmap_block_size) || bmap_block_size == 0) {
      perror ("FIGETBSZ");
      return -2;
      }

  nb_block = (filesize + bmap_block_size - 1) / bmap_block_size;
  for (block_index = 0; block_index < nb_block; block_index++) {
      if (lseek64 (file_desc, block_index * bmap_block_size, SEEK_SET) != block_index * bmap_block_size) {
	  perror ("lseek64");
	  return -3;
	  }
      block = block_index;
      if (ioctl (file_desc, FIBMAP, &block)) {
	  perror ("FIBMAP");
	  return -4;
	  }
      if (last_lba && block != 0)
	  *last_lba = block * (bmap_block_size / 512) + filesize/512;
      filesize -= bmap_block_size;
      if (block_index == nb_block - 1) {
	  if (block_index == block_count && current_block == 0)
	      fragment_cpt = 0;
	    else {
	      if (block == current_block + block_count)
		  block_count++;
		else
		  fragment_cpt++;
	      }
	  }
	else if (block == 0) /* we are in a hole */
	  block_count++;
	else if (block_index == 0) {
	  current_block = block;
	  block_count = 1;
	  }
	else if (block_index == block_count && current_block == 0) {
	  current_block = block;
	  block_count = 1;
	  }
	else if (block == current_block + block_count)
	  block_count++;
	else {
	  current_block = block;
	  block_count = 1;
	  }
	}
  return fragment_cpt;
  }

unsigned long long last_sector_contigous = 0, partition_start_contigous = 0;
int nb_fragment_contigous = 0;

static int get_remote_uninstall_lba (int dp, const bootbefore_t *bootbefore, unsigned log, unsigned confirm, unsigned char *second_sector_loaded)
  {
  unsigned long long lba = 0;
  unsigned nb, cpt_bootchain;
  // Misdesign: boot1param address is described in instboot_info, but we do not have ((unsigned long)fourKbuffer + 0x200) here...
  const boot1param_t *boot1param = (boot1param_t *)&fourKbuffer[0x19A];
  const bootloader2_t *bootchain = (bootloader2_t *)(fourKbuffer + boot1param->bootloader2.header.next);
  unsigned display = (bootbefore->part2.Signaturebyte0x29 == 0x29) || (bootbefore->part2.Signaturebyte0x29 == 0x28)
	|| ((const bootbefore_FAT32_t *)bootbefore)->part2.Signaturebyte0x29 == 0x29
	|| ((const bootbefore_FAT32_t *)bootbefore)->part2.Signaturebyte0x29 == 0x28
	|| !confirm;


  if (display && log) printf ("jmpinstruction 0x%X, 0x%X, EBIOSaccess 0x%X, Bytepersector %u, Reservedsector %u\n",
	bootbefore->part1.jmpinstruction[0], bootbefore->part1.jmpinstruction[1], bootbefore->part1.EBIOSaccess,
	bootbefore->part1.Bytepersector, bootbefore->part1.Reservedsector);
  if (display && log) printf ("Sectorpercluster %u, NbFAT %u, NbRootdirentry %u, Mediadescriptor 0x%X, NbSectorperFAT %u\n",
	bootbefore->part1.Sectorpercluster, bootbefore->part1.NbFAT, bootbefore->part1.NbRootdirentry,
	bootbefore->part1.Mediadescriptor, bootbefore->part1.NbSectorperFAT);
  if (display && log) printf ("Signaturebyte0x29: FAT16 0x%X / FAT32 0x%X, NbHead %u, NbSectorpertrack %u, NbHiddensector: %u,\n",
	bootbefore->part2.Signaturebyte0x29, ((const bootbefore_FAT32_t *)bootbefore)->part2.Signaturebyte0x29, bootbefore->part1.NbHead, bootbefore->part1.NbSectorpertrack, bootbefore->part1.NbHiddensector);
  if (display && log) printf ("NbTotalsector: %u, NbTotalsector2: %u, String '%s', MBRsig 0x%X, bootchain at 0x%lX\n",
	bootbefore->part1.NbTotalsector, bootbefore->part1.NbTotalsector2, bootbefore->part1.String,
	((unsigned short *)fourKbuffer)[255], (unsigned long)bootchain - (unsigned long)fourKbuffer);

  if (log) printf ("MBR seems to be: %s\n", recognise_MBR_string[recognise_MBR ((const bootsector_t *)bootbefore)]);
  if (display && log) printf ("boot1_checksum 0x%X, next 0x%X checksum 0x%X nbword %u, reg_dx/ide_dcr_address 0x%X reg_ax 0x%X\n",
	boot1param->boot1_checksum,
	boot1param->bootloader2.header.next, boot1param->bootloader2.header.checksum, boot1param->bootloader2.header.nbword,
	boot1param->bootloader2.bootloader2_cmd.int1302.reg_dx, boot1param->bootloader2.bootloader2_cmd.int1302.reg_ax);

  if (bootbefore->part1.Bytepersector & 0xFF || bootbefore->part1.Bytepersector > 16 * 1024 || bootbefore->part1.Bytepersector == 0) {
      printf ("Error bootbefore->part1.Bytepersector=%u (Not a Gujin bootloader?)\n", bootbefore->part1.Bytepersector);
      return 0;
      }

  if ((void *)bootchain < (void *)fourKbuffer || ((char *)bootchain - (char *)fourKbuffer) > 16 * 1024) {
      printf ("Error bootchain 0x%lX not nearby fourKbuffer 0x%lX, delta %ld (Not a Gujin bootloader?)\n",
	(unsigned long)bootchain, (unsigned long)fourKbuffer, (unsigned long)bootchain - (unsigned long)fourKbuffer);
      return 0;
      }

//{ const unsigned char *p = fourKbuffer, *e = &fourKbuffer[1024];
//while (p < e) printf ("0x%p: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X\n",
//	p, p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9], p[10], p[11], p[12], p[13], p[14], p[15]), p+=16;
//}

  if (bootbefore->part1.Bytepersector < sizeof (fourKbuffer)) {
      /* bootchain starts on second sector for 512 bytes/sectors, but continues on 3rd and maybe 4th sector */
      unsigned more_sector = (sizeof (fourKbuffer) - bootbefore->part1.Bytepersector) / bootbefore->part1.Bytepersector;
      /* We need to do it on fourKbuffer + bootbefore->part1.Bytepersector to fill bootchain content: */
      if (load_gujin_sector (dp, 1, fourKbuffer + bootbefore->part1.Bytepersector, more_sector, bootchain, boot1param, bootbefore, 0)) {
	  puts ("Error loading bootchain sectors (Gujin bootloader not installed?)");
	  return 0;
	  }
      }

  if (second_sector_loaded
	&& load_gujin_sector (dp, 1, second_sector_loaded, 1, bootchain, boot1param, bootbefore, 0)) {
      puts ("Error loading second_sector (Gujin bootloader not installed?)");
      return 0;
      }

  analyse_bootloader2_cmd (dp, &boot1param->bootloader2.bootloader2_cmd, &lba, &nb, bootbefore, log);
  for (cpt_bootchain = 0, nb = 1; nb != 0; cpt_bootchain++)
      analyse_bootloader2_cmd (dp, &bootchain[cpt_bootchain].bootloader2_cmd, &lba, &nb, bootbefore, log);

  const instboot_info_t **instboot_info_ptr_ptr = (const instboot_info_t **)(fourKbuffer + 0x200);
  unsigned instboot_info_ptr = *(unsigned *)instboot_info_ptr_ptr;
  unsigned char instboot_info_buffer[4096];
  unsigned nbsect = bootbefore->part1.Bytepersector <= 512 ? 2 : 1;

  if (load_gujin_sector (dp, instboot_info_ptr / bootbefore->part1.Bytepersector, instboot_info_buffer, nbsect, bootchain, boot1param, bootbefore, 0)) {
      puts ("Error loading instboot_info sectors (Gujin bootloader not installed?)");
      return 0;
      }
  instboot_info_t *instboot_info = (instboot_info_t *)(instboot_info_buffer + instboot_info_ptr % bootbefore->part1.Bytepersector);

  if (log) printf ("instboot_info at 0x%X, checksum_start_adr 0x%X, boot1param_adr 0x%X, bootend_adr 0x%X, uninstall_mbr_adr 0x%X\n",
	instboot_info_ptr, instboot_info->checksum_start_adr, instboot_info->boot1param_adr, instboot_info->bootend_adr, instboot_info->uninstall_mbr_adr);
  if (log) printf ("bootchain_adr 0x%X, bootchain_nbblock %u, gujin_param_adr 0x%X, code_start_adr 0x%X\n",
	instboot_info->bootchain_adr, instboot_info->bootchain_nbblock, instboot_info->gujin_param_adr, instboot_info->code_start_adr);
  if (log) printf ("patch2adr 0x%X, patch3adr 0x%X, patch4adr 0x%X, sbss 0x%X\n",
	instboot_info->patch2adr, instboot_info->patch3adr, instboot_info->patch4adr, instboot_info->sbss);
  if (log) printf ("edata 0x%X, deltaseg 0x%X, diskcodeseg 0x%X, fscodeseg 0x%X, loadcodeseg 0x%X, usercodeseg 0x%X, xdataseg 0x%X\n",
	instboot_info->edata, instboot_info->deltaseg, instboot_info->diskcodeseg, instboot_info->fscodeseg, instboot_info->loadcodeseg, instboot_info->usercodeseg, instboot_info->xdataseg);
  if (log) printf ("password_serial_adr 0x%X, serialconf_port_plus_1_adr 0x%X, read_disk_adr 0x%X, partition_relative_read_disk_adr 0x%X\n",
	instboot_info->password_serial_adr, instboot_info->serialconf_port_plus_1_adr, instboot_info->read_disk_adr, instboot_info->partition_relative_read_disk_adr);
  if (log) printf ("read_disk_1st_adr 0x%X, usbhdd_patch1 0x%X, usbhdd_patch2 0x%X, singlesectorload_patch1 0x%X, singlesectorload_patch2 0x%X\n",
	instboot_info->read_disk_1st_adr, instboot_info->usbhdd_patch1, instboot_info->usbhdd_patch2, instboot_info->singlesectorload_patch1, instboot_info->singlesectorload_patch2);

  if (instboot_info->gujin_param_adr < 0x200 || instboot_info->gujin_param_adr + sizeof(gujin_param_t) > 0x400) {
      printf ("Error instboot_info->gujin_param_adr = 0x%X! (Gujin bootloader not installed?)\n", instboot_info->gujin_param_adr);
      return 0;
      }

  if (log) {
      /* should be copied in second_sector_512bytes */
      gujin_param_t *gujin_param = (gujin_param_t *)(fourKbuffer + instboot_info->gujin_param_adr);
      if (gujin_param->magic != 16980327)
	  printf ("gujin_param bad magic %u\n", gujin_param->magic);
	else {
	  printf ("Gujin version %X installed, command line: \"%s\"\n", gujin_param->version, gujin_param->extra_cmdline);
	  printf ("That Gujin compiled with: ");
	  if (gujin_param->compilation_attrib.vga_support) printf ("vga_support, ");
	  if (gujin_param->compilation_attrib.big_malloc) printf ("big_malloc, ");
	  if (gujin_param->compilation_attrib.bios_only) printf ("bios_only, ");
	  if (gujin_param->compilation_attrib.initrd_loader) printf ("initrd_loader, ");
	  if (gujin_param->compilation_attrib.vmlinuz_loader) printf ("vmlinuz_loader, ");
	  if (gujin_param->compilation_attrib.elf_loader) printf ("elf_loader, ");
	  if (gujin_param->compilation_attrib.multiboot_loader) printf ("multiboot_loader, ");
	  if (gujin_param->compilation_attrib.multilingual) printf ("multilingual, ");
	  if (gujin_param->compilation_attrib.unicode) printf ("unicode, ");
	  printf ("\nMltStrLanguage: %u, scanpath \"%s\", default_video_mode 0x%X, default_text_video_mode 0x%X, default_graphic_video_mode 0x%X\n",
		gujin_param->MltStrLanguage, gujin_param->scanpath, gujin_param->default_video_mode, gujin_param->default_text_video_mode, gujin_param->default_graphic_video_mode);
	  printf ("min_nb_initrd: %u, stop_emulation: %u, timeout_autoload %u, autoload: last_loaded %u, total_loadable %u, init_page %u, total_page %u\n",
		gujin_param->min_nb_initrd, gujin_param->stop_emulation, gujin_param->timeout_autoload,
		gujin_param->autoload.last_loaded, gujin_param->autoload.total_loadable, gujin_param->autoload.init_page, gujin_param->autoload.total_page);
	  printf ("Configuration:\n");
#define Apply(field, deflt)     printf (#field " : %u, ", gujin_param->attrib.field);
	  APPLY_ATTRIBUTE_ORDER(Apply)
#undef Apply
	  printf ("kbdmap %s, time_hour_offset %d, time_minute_offset %d\n",
		kbdname (gujin_param->kbdmap) ?: "undefined", gujin_param->time_hour_offset, gujin_param->time_minute_offset);
	  printf ("Video cardname: \"%s\", valid VGA modes: ", gujin_param->cardname);
	  unsigned cpt;
	  for (cpt = 0; cpt < sizeof(gujin_param->VGA_valid_mode); cpt++)
	      printf ("0x%X, ", gujin_param->VGA_valid_mode[cpt]);
	  printf ("\nvga_mode description:\n");
	  for (cpt = 0; cpt < sizeof(gujin_param->vga_mode) / sizeof(gujin_param->vga_mode[0]); cpt++)
	      if (gujin_param->vga_mode[cpt].number != 0 && gujin_param->vga_mode[cpt].height != 0)
	      printf ("  %u: isgraphic %u, reserved %u, number 0x%X, bpp %u, height %u, width %u\n",
		cpt,
		gujin_param->vga_mode[cpt].isgraphic,
		gujin_param->vga_mode[cpt].reserved,
		gujin_param->vga_mode[cpt].number,
		gujin_param->vga_mode[cpt].bpp,
		gujin_param->vga_mode[cpt].height,
		gujin_param->vga_mode[cpt].width_div8 * 8);
	  printf ("partial_checksum 0x%X\n", gujin_param->partial_checksum);
	  }
      }

  unsigned char uninstall_mbr_buffer[4096];
  if (instboot_info->uninstall_mbr_adr / bootbefore->part1.Bytepersector == 0)
      memcpy (uninstall_mbr_buffer, bootbefore, bootbefore->part1.Bytepersector);
    else if (load_gujin_sector (dp, instboot_info->uninstall_mbr_adr / bootbefore->part1.Bytepersector, uninstall_mbr_buffer, 1, bootchain, boot1param, bootbefore, log)) {
      puts ("Error loading uninstall_mbr sector");
      return 0;
      }
  {
  const bootloader2_t *uninstall_mbr = (const bootloader2_t *)(uninstall_mbr_buffer + instboot_info->uninstall_mbr_adr % bootbefore->part1.Bytepersector);
  unsigned long long another_lba;

  if (log) puts ("Uninstall MBR (where to read, where to read current partition table, where to write):");
  analyse_bootloader2_cmd (dp, &uninstall_mbr[0].bootloader2_cmd, &lba, &nb, bootbefore, log);
  analyse_bootloader2_cmd (dp, &uninstall_mbr[1].bootloader2_cmd, &another_lba, &nb, bootbefore, log);
  analyse_bootloader2_cmd (dp, &uninstall_mbr[2].bootloader2_cmd, &another_lba, &nb, bootbefore, log);

  /* Recovery if /boot/gujin.ebios is not a contigous file: */
  if (nb_fragment_contigous > 1 && last_sector_contigous) {
      printf ("nb_fragment_contigous %d, last_sector_contigous %llu + partition_start_contigous %llu = %llu\n", nb_fragment_contigous, last_sector_contigous, partition_start_contigous, last_sector_contigous + partition_start_contigous);
      lba = last_sector_contigous + partition_start_contigous - 1;
      }

  /* If we are inside a partition, that is values from the start of the disk which are accessed: */
  struct stat sb;
  unsigned long long start_sector = 0;
  unsigned char OldMbrBuffer[4096], NewMbrBuffer[4096];
  if (fstat(dp, &sb) == 0 && S_ISBLK(sb.st_mode))
      get_device_start_sector (dp, &start_sector, 0);

  if (uninstall_mbr[0].header.nbword == 0)
      puts (" No uninstall MBR present.");
    else if (lseek64 (dp, (lba - start_sector) * bootbefore->part1.Bytepersector, SEEK_SET) != (lba - start_sector) * bootbefore->part1.Bytepersector
		|| read (dp, OldMbrBuffer, bootbefore->part1.Bytepersector) != bootbefore->part1.Bytepersector)
      printf ("Failed reading old MBR lba at %llu!\n", (lba - start_sector));
    else if (last_sector_in_file_valid && memcmp (last_sector_in_file, OldMbrBuffer, bootbefore->part1.Bytepersector)) {
      printf ("ERROR: sector read at %llu*%u do not contain the last sector read from the file\n", lba - start_sector, bootbefore->part1.Bytepersector);
      unsigned idxmem;
      for (idxmem = 0; idxmem < bootbefore->part1.Bytepersector - 4; idxmem++)
	  if (last_sector_in_file[idxmem] != OldMbrBuffer[idxmem])
	       break;
      printf ("ERROR: at offset 0x%X, last_sector_in_file %02X %02X %02X %02X, OldMbrBuffer %02X %02X %02X %02X\n",
		idxmem,
		last_sector_in_file[idxmem+0], last_sector_in_file[idxmem+1], last_sector_in_file[idxmem+2], last_sector_in_file[idxmem+3],
		OldMbrBuffer[idxmem+0], OldMbrBuffer[idxmem+1], OldMbrBuffer[idxmem+2], OldMbrBuffer[idxmem+3]);
      printf ("ERROR: This case has only ever appeared when FIBMAP/FIGETBSZ ioctl support is buggy on the target filesystem\n"
		"ERROR: use the '--extract-mbr=' option to extract the recovery MBR stored in that file.\n");
      }
    else {
      bootsector_t *bootsect = (bootsector_t *)OldMbrBuffer, *mbrsect = (bootsector_t *)NewMbrBuffer;
      unsigned cpt, mbrtype = recognise_MBR(bootsect);
      printf ("Old MBR (bootbefore->part1.Bytepersector %u) at lba %llu is %s\n", bootbefore->part1.Bytepersector, lba, recognise_MBR_string[mbrtype]);
      if (lseek64 (dp, 0, SEEK_SET) == 0
		&& read (dp, (unsigned char *)mbrsect, bootbefore->part1.Bytepersector) == bootbefore->part1.Bytepersector) {
	  if (bootsect->after.WindowsNTmarker == mbrsect->after.WindowsNTmarker)
	      printf ("Current and to-be-restored MBR have the same WindowsNTmarker: 0x%X\n", bootsect->after.WindowsNTmarker);
	    else
	      printf ("Current and to-be-restored MBR have different WindowsNTmarker: 0x%X, 0x%X\n", mbrsect->after.WindowsNTmarker, bootsect->after.WindowsNTmarker);
	  if (bootsect->after.Unknown == mbrsect->after.Unknown)
	      printf ("Current and to-be-restored MBR have the same signature at 0x1BC: 0x%X\n", bootsect->after.Unknown);
	    else
	      printf ("Current and to-be-restored MBR have different signature at 0x1BC: 0x%X, 0x%X\n", mbrsect->after.Unknown, bootsect->after.Unknown);
	  for (cpt = 0; cpt < 4; cpt++) {
	      if (bootsect->after.bootsect_partition[cpt].nb_sector_before == mbrsect->after.bootsect_partition[cpt].nb_sector_before
		  && bootsect->after.bootsect_partition[cpt].nb_sector == mbrsect->after.bootsect_partition[cpt].nb_sector) {
		  if (!memcmp (&bootsect->after.bootsect_partition[cpt], &mbrsect->after.bootsect_partition[cpt], sizeof(mbrsect->after.bootsect_partition[cpt])))
		      printf ("Current and to-be-restored MBR have the same partition %u: %u + %u\n", cpt, bootsect->after.bootsect_partition[cpt].nb_sector_before, bootsect->after.bootsect_partition[cpt].nb_sector);
		    else
		      printf ("Current and to-be-restored MBR have the same partition %u: %u + %u (but C/H/S values are different)\n", cpt, bootsect->after.bootsect_partition[cpt].nb_sector_before, bootsect->after.bootsect_partition[cpt].nb_sector);
		  }
		else
		  printf ("Current and to-be-restored MBR have different partition %u: %u+%u != %u+%u\n", cpt,
			mbrsect->after.bootsect_partition[cpt].nb_sector_before, mbrsect->after.bootsect_partition[cpt].nb_sector,
			bootsect->after.bootsect_partition[cpt].nb_sector_before, bootsect->after.bootsect_partition[cpt].nb_sector);
	      }
	  }
      if (mbrtype == 0 && !confirm)
	  printf ("Will not disinstall: saved MBR not bootable!\n");
      else if (lseek64 (dp, 0, SEEK_SET) != 0)
	  printf ("Will not disinstall: cannot seek to beginning of drive!\n");
      else if (log) {
	  printf ("Can disinstall, using --remove <device> (NOT a partition if installed on the main MBR!) or --remove <filename>\n");
	  return 1;
	  }
      else {
	  int ch;
	  if (mbrtype == 0)
	      printf ("WARNING: disinstalling current bootable disk by writing a non bootable MBR!\n");
	  printf ("If you really want to restore the MBR, keeping the old partitions, type YES: ");
	  if (!confirm || (((ch = getchar()) == 'y' || ch == 'Y') && ((ch = getchar()) == 'e' || ch == 'E') && ((ch = getchar()) == 's' || ch == 'S'))) {
	      bootsect->after = mbrsect->after;
	      int result = write (dp, bootsect, bootbefore->part1.Bytepersector);
// TODO: also copy the uninstalled MBR to the MBR copy for FAT32 INFS superfloppy
	      if (result == bootbefore->part1.Bytepersector) {
		  printf ("MBR successfully restored.\n");
		  return 1;
		  }
		else {
		  printf ("MBR restore FAILED, write() returns %d\n", result);
		  perror ("MBR");
		  }
	      }
	    else
	      printf ("MBR restoration cancelled\n");
	  }
      }
  }

  return 0;
  }

const char *gujin_device_from_file (const char *device_or_file)
  {
  struct stat statbuf;

  if (stat (device_or_file, &statbuf) == -1) {
      perror (device_or_file);
      return 0;
      }
  if (statbuf.st_size >= 640 * 1024 || S_ISLNK(statbuf.st_mode) || S_ISBLK(statbuf.st_mode))
      return device_or_file;

  if (!S_ISREG(statbuf.st_mode))  {
      printf ("ERROR: filename '%s' is not a regular file nor a device\n", device_or_file);
      return 0;
      }

  FILE *file = fopen ("/proc/diskstats", "r");
  if (file == NULL) {
      perror ("/proc/diskstats");
      return 0;
      }
  /* GOD PROMISES ETERNAL LIFE. WE CAN DELIVER IT. */
  static char device[64];
  int ret, major, minor;
  strcpy (device, "/dev/");
  while ((ret = fscanf (file, " %d %d %s %*[^\n]", &major, &minor, &device[5])) != EOF)
      if (ret == 3 && 256 * major + minor == statbuf.st_dev)
	  break;
  if (fclose(file)) {
      perror ("/proc/diskstats");
      return 0;
      }
  if (ret == EOF) {
      printf ("ERROR: did not find the device containing file '%s'\n", device_or_file);
      return 0;
      }
  /* We want the disk, not the partition, in most case: */
  char partition[64];
  strcpy (partition, device);
  if ((device[5] =='h' || device[5] =='s') && device[6] =='d' && device[7] >= 'a' && device[7] <= 'z')
      device[8] = '\0';
    else if (*(unsigned long long *)&device[3] == *(unsigned long long *)"v/mmcblk")
      device[12] = '\0'; // "/dev/mmcblk0p1"
  int fd = open (partition, O_RDONLY);
  if (fd != -1) {
      get_device_start_sector (fd, &partition_start_contigous, 0);
      close(fd);
      }
  printf ("Info: file '%s' in on device '%s' on disk '%s' (partition start %llu).\n", device_or_file, partition, device, partition_start_contigous);

  return device;
  }

  /* TODO: gujin --report/--remove=eltorito.iso ,  NbHiddensector: 1409316, NbTotalsector2: 1409956 */
int gujin_report (const char *device_or_file, unsigned confirm)
  {
  int fd;
  unsigned Bytepersector = 0; /* only set if strcmp (device, device_or_file) */
  unsigned char second_sector_loaded[4096];	/* max sector size */

  const char *device = gujin_device_from_file (device_or_file);
  if (device != 0 && strcmp (device, device_or_file)) {
      if ((fd = open (device_or_file, O_RDONLY)) == -1) {
	  perror (device_or_file);
	  return -1;
	  }
      for (;;) { /* skip the possible file hole at beginning of file, hole multiple of 4Kbytes */
	  if (read (fd, fourKbuffer, sizeof(fourKbuffer)) != sizeof(fourKbuffer)) {
	      perror (device_or_file);
	      return -2;
	      }
	  unsigned char *ptr = fourKbuffer;
	  while (*ptr == 0 && ptr < &fourKbuffer[sizeof(fourKbuffer)])
	      ptr++;
	  if (ptr < &fourKbuffer[sizeof(fourKbuffer)])
	      break;
	  }
      /* Search 0xBEFC1F03 in disk.c for info */
      if (*(unsigned short *)fourKbuffer == 0x6FEB && *(unsigned *)&fourKbuffer[0x0A] == 0xBEFC1F03) {
	  *(unsigned *)&fourKbuffer[0x0A] = 0x00020000;
	  printf ("Removing --hpcompaqbug patch at beginning of file\n");
	  }
      /* For gujin.bcd, restore part overwritten by -boot-info-table
	 See comment: "do a spare copy of the part overwritten by -boot-info-table" */
      const unsigned bootchain_offset = *(unsigned short *)&fourKbuffer[0x19A+2];
      if (bootchain_offset + 20*26 < sizeof(fourKbuffer)) {
	  struct boot_info_table_patch {	// 0x5e3
	      unsigned short signature_FEED;
	      unsigned short dest;
	      unsigned short nb;
	      unsigned short data[0];
	      } __attribute__((packed)) *boot_info_table_patch = (struct boot_info_table_patch *)&fourKbuffer[bootchain_offset + 20*26];
	  if (boot_info_table_patch->signature_FEED == 0xFEED && bootchain_offset + 20*26 + 2*boot_info_table_patch->nb < sizeof(fourKbuffer)) {
	      printf ("Removing boot_info_table_patch from *.bcd file: dest 0x%X nb %d\n", boot_info_table_patch->dest, boot_info_table_patch->nb);
	      unsigned short *dst = (unsigned short *)&fourKbuffer[boot_info_table_patch->dest];
	      unsigned cpt;
	      for (cpt = 0; cpt < boot_info_table_patch->nb; cpt++)
		  *dst++ = boot_info_table_patch->data[cpt];
	      }
	  }
      const bootbefore_t *bootbefore = (const bootbefore_t *)fourKbuffer;
      Bytepersector = bootbefore->part1.Bytepersector;
      if (Bytepersector == 0 || Bytepersector & 0xFF || Bytepersector > 16 * 1024) {
	  printf ("For file %s, Bytepersector would be %u at offset %u, not a Gujin file?\n", device_or_file, Bytepersector, (unsigned)lseek(fd, 0, SEEK_CUR));
	  return -2;
	  }
      if (Bytepersector == 4096) {
	  if (read (fd, second_sector_in_file, Bytepersector) != Bytepersector) {
	      perror (device_or_file);
	      return -2;
	      }
	  }
	else
	  memcpy (second_sector_in_file, fourKbuffer + Bytepersector, Bytepersector);
      second_sector_in_file_valid = 1;
      if (lseek64(fd, -(off64_t)Bytepersector, SEEK_END) != (off_t) -1)
	  if (read (fd, last_sector_in_file, Bytepersector) == Bytepersector)
	      last_sector_in_file_valid = 1;
      /* Recovery if /boot/gujin.ebios is not a contigous file: */
      if (Bytepersector == 512) {
	  nb_fragment_contigous = nb_fragment_file (fd, &last_sector_contigous);
	  if (nb_fragment_contigous < 0)
	      printf ("Cannot get nb fragment of file '%s', error %d!\n", device_or_file, nb_fragment_contigous);
	  else if (nb_fragment_contigous != 1)
	      printf ("File '%s' has %d fragment, last_sector %llu\n", device_or_file, nb_fragment_contigous, last_sector_contigous);
	  }
      close (fd);
      }

  if (!device || (fd = open (device, O_RDONLY)) == -1) {
      perror (device);
      return -1;
      }
  printf ("Reading beginning of %s to analyse its MBR:\n", device);
  if (read (fd, fourKbuffer, sizeof(fourKbuffer)) != sizeof(fourKbuffer)) {
      perror (device);
      return -2;
      }
  /* Search 0xBEFC1F03 in disk.c for info */
  if (*(unsigned short *)fourKbuffer == 0x6FEB && *(unsigned *)&fourKbuffer[0x0A] == 0xBEFC1F03) {
      *(unsigned *)&fourKbuffer[0x0A] = 0x00020000;
      printf ("Removing --hpcompaqbug patch at beginning of device\n");
      }
  int can_disinstall = get_remote_uninstall_lba (fd, (const bootbefore_t *)fourKbuffer, 1, confirm, second_sector_loaded);
  if (can_disinstall && second_sector_in_file_valid && memcmp (second_sector_in_file, second_sector_loaded, Bytepersector))
      printf ("WARNING: Gujin on device '%s' can be disinstalled, but it is not related to file '%s' !\n", device, device_or_file);

  unsigned extension = *(unsigned *)&device_or_file[strlen(device_or_file)-4];
  if (extension == *(const unsigned *)".iso" || extension == *(const unsigned *)".ISO") {
      off_t pos = lseek64 (fd, -512, SEEK_END);
      unsigned short array[512/2];
      const bootbefore_t *bootbefore = (const bootbefore_t *)fourKbuffer;
      if (bootbefore->part1.Bytepersector && *(unsigned *)bootbefore->part1.String == *(unsigned *)"Guji") {
	  if ((bootbefore->part1.NbTotalsector2 - bootbefore->part1.NbHiddensector) * bootbefore->part1.Bytepersector / 1024 < 720)
	      printf ("The FAT filesystem at end of the ISO is too small to be mounted, use --fs=FAT:<nb512bytessector> at creation time\n");
	    else
	      printf ("You can mount the %u Kbytes FAT filesystem by \"mount -t vfat -o loop,offset=%u %s /mnt\", see --full option to modify that FS\n",
		(bootbefore->part1.NbTotalsector2 - bootbefore->part1.NbHiddensector) * bootbefore->part1.Bytepersector / 1024,
		bootbefore->part1.NbHiddensector * bootbefore->part1.Bytepersector, device_or_file);
	  }
      if (pos != (off_t) -1 && (pos + 512) % 2048 == 0
	 && *(const unsigned long long *)bootbefore->part1.String == *(const unsigned long long *)"Gujin1\0\0"
	 &&  bootbefore->part1.NbHiddensector > 50
	 && read (fd, array, 512) == 512) {
	  printf ("Can uninstall ISO+Superfloppy: MBR is %s, initial ISO size %u\n", recognise_MBR_string[recognise_MBR ((const bootsector_t *)array)], bootbefore->part1.NbHiddensector * 512);
	  can_disinstall = 1;
	  }
	else
	  printf ("Cannot read ISO+Superfloppy last sector\n");
      }
  if (close(fd) != 0) {
      perror (device_or_file);
      return -3;
      }
  return can_disinstall? 0 : 1;
  }

/**
 ** Independant stuff for gujin_uninstall(), just packed in to have a single executable:
 **/

int gujin_uninstall (const char *device_or_file, unsigned confirm)
  {
  int fd;
  unsigned Bytepersector = 0; /* only set if strcmp (device, device_or_file) */
  unsigned char second_sector_loaded[4096];	/* max sector size */

  const char *device = gujin_device_from_file (device_or_file);
  if (device != 0 && strcmp (device, device_or_file)) {
      if ((fd = open (device_or_file, O_RDONLY)) == -1) {
	  perror (device_or_file);
	  return -1;
	  }
      for (;;) { /* skip the possible file hole at beginning of file, hole multiple of 4Kbytes */
	  if (read (fd, fourKbuffer, sizeof(fourKbuffer)) != sizeof(fourKbuffer)) {
	      perror (device_or_file);
	      return -2;
	      }
	  unsigned char *ptr = fourKbuffer;
	  while (*ptr == 0 && ptr < &fourKbuffer[sizeof(fourKbuffer)])
	      ptr++;
	  if (ptr < &fourKbuffer[sizeof(fourKbuffer)])
	      break;
	  }
      /* Search 0xBEFC1F03 in disk.c for info */
      if (*(unsigned short *)fourKbuffer == 0x6FEB && *(unsigned *)&fourKbuffer[0x0A] == 0xBEFC1F03) {
	  *(unsigned *)&fourKbuffer[0x0A] = 0x00020000;
	  printf ("Removing --hpcompaqbug patch at beginning of file\n");
	  }
      const bootbefore_t *bootbefore = (const bootbefore_t *)fourKbuffer;
      Bytepersector = bootbefore->part1.Bytepersector;
      if (Bytepersector == 0 || Bytepersector & 0xFF || Bytepersector > 16 * 1024) {
	  printf ("For file %s, Bytepersector would be %u, not a Gujin file?\n", device_or_file, Bytepersector);
	  return -2;
	  }
      if (Bytepersector == 4096) {
	  if (read (fd, second_sector_in_file, Bytepersector) != Bytepersector) {
	      perror (device_or_file);
	      return -2;
	      }
	  }
	else
	  memcpy (second_sector_in_file, fourKbuffer + Bytepersector, Bytepersector);
      second_sector_in_file_valid = 1;
      if (lseek64(fd, -(off64_t)Bytepersector, SEEK_END) != (off_t) -1)
	  if (read (fd, last_sector_in_file, Bytepersector) == Bytepersector)
	      last_sector_in_file_valid = 1;
      /* Recovery if /boot/gujin.ebios is not a contigous file: */
      if (Bytepersector == 512) {
	  nb_fragment_contigous = nb_fragment_file (fd, &last_sector_contigous);
	  if (nb_fragment_contigous < 0)
	      printf ("Cannot get nb fragment of file '%s', error %d!\n", device_or_file, nb_fragment_contigous);
	  else if (nb_fragment_contigous != 1)
	      printf ("File '%s' has %d fragment, last_sector %llu\n", device_or_file, nb_fragment_contigous, last_sector_contigous);
	  }
      close (fd);
      }

  if (!device || (fd = open (device, O_RDWR | O_SYNC)) == -1) {
      perror (device);
      return -1;
      }
  if (read (fd, fourKbuffer, sizeof(fourKbuffer)) != sizeof(fourKbuffer)) {
      perror (device);
      return -2;
      }
  /* Search 0xBEFC1F03 in disk.c for info */
  if (*(unsigned short *)fourKbuffer == 0x6FEB && *(unsigned *)&fourKbuffer[0x0A] == 0xBEFC1F03) {
      *(unsigned *)&fourKbuffer[0x0A] = 0x00020000;
      printf ("Removing --hpcompaqbug patch at beginning of device\n");
      }
  int disinstalled = get_remote_uninstall_lba (fd, (const bootbefore_t *)fourKbuffer, 0, confirm, second_sector_loaded);
  unsigned extension = *(unsigned *)&device_or_file[strlen(device_or_file)-4];
  if (extension == *(const unsigned *)".iso" || extension == *(const unsigned *)".ISO") {
      off_t pos = lseek64 (fd, -512, SEEK_END);
      unsigned short array[512/2];
      const bootbefore_t *bootbefore = (const bootbefore_t *)fourKbuffer;
      if (pos != (off_t) -1 && (pos + 512) % 2048 == 0
	 && *(const unsigned long long *)bootbefore->part1.String == *(const unsigned long long *)"Gujin1\0\0"
	 &&  bootbefore->part1.NbHiddensector > 50
	 && read (fd, array, 512) == 512) {
	  printf ("Uninstalling ISO+Superfloppy: MBR is %s, initial ISO size %u\n", recognise_MBR_string[recognise_MBR ((const bootsector_t *)array)], bootbefore->part1.NbHiddensector * 512);
	  if (lseek64 (fd, 0, SEEK_SET) == 0 && write (fd, array, 512) == 512 && ftruncate (fd, bootbefore->part1.NbHiddensector * 512) == 0)
	      disinstalled = 1;
	  }
	else
	  printf ("Cannot read ISO+Superfloppy last sector\n");
      }

  if (close(fd) != 0) {
      perror (device_or_file);
      return -3;
      }
  if (second_sector_in_file_valid && disinstalled) {
      if (memcmp (second_sector_in_file, second_sector_loaded, Bytepersector)) {
	  unsigned idxmem;
	  for (idxmem = 0; idxmem < Bytepersector - 7; idxmem++)
	      if (second_sector_in_file[idxmem] != second_sector_loaded[idxmem])
		  break;
	  printf ("second_sector_in_file[0x%X]: 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X\n", idxmem, second_sector_in_file[idxmem+0], second_sector_in_file[idxmem+1], second_sector_in_file[idxmem+2], second_sector_in_file[idxmem+3], second_sector_in_file[idxmem+4], second_sector_in_file[idxmem+5], second_sector_in_file[idxmem+6], second_sector_in_file[idxmem+7]);
	  printf ("second_sector_loaded[0x%X] : 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X\n", idxmem, second_sector_loaded[idxmem+0], second_sector_loaded[idxmem+1], second_sector_loaded[idxmem+2], second_sector_loaded[idxmem+3], second_sector_loaded[idxmem+4], second_sector_loaded[idxmem+5], second_sector_loaded[idxmem+6], second_sector_loaded[idxmem+7]);
	  printf ("WARNING: Gujin on device '%s' has been disinstalled, but it is not related to file '%s' -- keeping that file!\n", device, device_or_file);
	  }
      else if (unlink (device_or_file) == -1) {
	  printf ("ERROR: cannot unlink file '%s'.\n", device_or_file);
	  perror (device_or_file);
	  }
      }
  return disinstalled? 0 : 1;
  }

void setup_gujin_param_from_environment (gujin_param_t *gujin_param)
  {
  time_t atime = time(0);
  struct tm utc = *gmtime(&atime);
  struct tm local = *localtime(&atime);
  EXTRA ("Extra: local time %u:%u daylight %u, UTC time %u:%u daylight %u\n", local.tm_hour, local.tm_min, local.tm_isdst, utc.tm_hour, utc.tm_min, utc.tm_isdst);
  if (gujin_param->time_hour_offset || gujin_param->time_minute_offset)
      EXTRA ("Extra: keep user supplied values for gujin_param->time_hour_offset %d, time_minute_offset %u\n", gujin_param->time_hour_offset, gujin_param->time_minute_offset);
    else if (local.tm_hour == utc.tm_hour + (local.tm_isdst == 1) ? 1 : 0 && local.tm_min == utc.tm_min)
      EXTRA ("Extra: no need to set a timezone, local time == UTC\n");
    else {
      gujin_param->time_hour_offset = local.tm_hour - (utc.tm_hour + ((local.tm_isdst == 1) ? 1 : 0));
      gujin_param->time_minute_offset = local.tm_min - utc.tm_min;
      EXTRA ("Extra: has set timezone gujin_param->time_hour_offset %d, time_minute_offset %d\n", gujin_param->time_hour_offset, gujin_param->time_minute_offset);
      }

  char *env_LANG = getenv ("LANG");
  if (env_LANG == 0)
      env_LANG = getenv ("lang");
  if (gujin_param->MltStrLanguage == 0 && env_LANG != 0) {
      /* char *listlang[] = { LANG_CMDLINE }; */
      EXTRA ("Extra: Trying to autodetect language with LANG='%s'\n", env_LANG);
      if (env_LANG[0] == 'e' && env_LANG[1] == 'n')
	  gujin_param->MltStrLanguage = 0;
	else if (env_LANG[0] == 'f' && env_LANG[1] == 'r')
	  gujin_param->MltStrLanguage = 1;
	else if (env_LANG[0] == 'r' && env_LANG[1] == 'u')
	  gujin_param->MltStrLanguage = 2;
	else if (env_LANG[0] == 's' && env_LANG[1] == 'p')
	  gujin_param->MltStrLanguage = 3;
	else if (env_LANG[0] == 'i' && env_LANG[1] == 't')
	  gujin_param->MltStrLanguage = 4;
	else if (env_LANG[0] == 'p' && env_LANG[1] == 't')
	  gujin_param->MltStrLanguage = 5;
	else if (env_LANG[0] == 'd' && env_LANG[1] == 'e')
	  gujin_param->MltStrLanguage = 6;
	else if (env_LANG[0] == 'n' && env_LANG[1] == 'l')
	  gujin_param->MltStrLanguage = 7;
	else
	  EXTRA ("Extra: failed, do not start by en/fr/ru/sp/it/pt/de/nl: %c%c\n", env_LANG[0], env_LANG[1]);
      EXTRA ("Extra: gujin_param->MltStrLanguage %u\n", gujin_param->MltStrLanguage);
      }
  if (gujin_param->MltStrLanguage == 0 && access ("/etc/default/locale", R_OK) == 0) {
      EXTRA ("Extra: try to detect the language with '/etc/default/locale'\n");
      FILE *flang = fopen ("/etc/default/locale", "r");
      char buffer[1024];
      while (flang && fgets(buffer, sizeof(buffer), flang) != 0) {
	  char *ptr = buffer;
	  while (*ptr == ' ' || *ptr == '\t')
	      ptr++;
	  if (!strncmp (ptr, "LANG=", strlen("LANG="))) {
	      EXTRA ("Extra: got line: %s", buffer);
	      ptr += strlen ("LANG=");
	      if (*ptr == '"')
		  ptr++;
	      if (ptr[0] == 'e' && ptr[1] == 'n')
		  gujin_param->MltStrLanguage = 0;
		else if (ptr[0] == 'f' && ptr[1] == 'r')
		  gujin_param->MltStrLanguage = 1;
		else if (ptr[0] == 'r' && ptr[1] == 'u')
		  gujin_param->MltStrLanguage = 2;
		else if (ptr[0] == 's' && ptr[1] == 'p')
		  gujin_param->MltStrLanguage = 3;
		else if (ptr[0] == 'i' && ptr[1] == 't')
		  gujin_param->MltStrLanguage = 4;
		else if (ptr[0] == 'p' && ptr[1] == 't')
		  gujin_param->MltStrLanguage = 5;
		else if (ptr[0] == 'd' && ptr[1] == 'e')
		  gujin_param->MltStrLanguage = 6;
		else if (ptr[0] == 'n' && ptr[1] == 'l')
		  gujin_param->MltStrLanguage = 7;
		else
		  EXTRA ("Extra: failed, do not start by en/fr/ru/sp/it/pt/de/nl: %s\n", ptr);
	      }
	  }
      if (flang)
	  fclose (flang);
      EXTRA ("Extra: gujin_param->MltStrLanguage %u\n", gujin_param->MltStrLanguage);
      }

  if (gujin_param->kbdmap == kbd_unknown && (access ("/etc/default/keyboard", R_OK) == 0 || access ("/etc/sysconfig/keyboard", R_OK) == 0)) {
      EXTRA ("Extra: try to detect the keyboard with '/etc/default/keyboard' or '/etc/sysconfig/keyboard'\n");
      FILE *fkbd = fopen ("/etc/default/keyboard", "r");
      if (!fkbd)
	  fkbd = fopen ("/etc/sysconfig/keyboard", "r");
      char buffer[1024];
      while (fkbd && fgets(buffer, sizeof(buffer), fkbd) != 0) {
	  char *ptr = buffer;
	  while (*ptr == ' ' || *ptr == '\t')
	      ptr++;
	  if (!strncmp (ptr, "XKBLAYOUT=", strlen("XKBLAYOUT=")))
	      ptr += strlen ("XKBLAYOUT=");
	  else if (!strncmp (ptr, "KEYTABLE=", strlen("KEYTABLE=")))
	      ptr += strlen ("KEYTABLE=");
	  else
	      ptr = 0;
	  if (ptr) {
	      EXTRA ("Extra: got line: %s", buffer);
	      if (*ptr == '"')
		  ptr++;
	      if (ptr[0] == 'u' && ptr[1] == 's')
		  gujin_param->kbdmap = kbd_us;
		else if (ptr[0] == 'c' && ptr[1] == 'a')
		  gujin_param->kbdmap = kbd_ca;
		else if (ptr[0] == 'b' && ptr[1] == 'r')
		  gujin_param->kbdmap = kbd_br;
		else if (ptr[0] == 'g' && ptr[1] == 'b')
		  gujin_param->kbdmap = kbd_uk;
		else if (ptr[0] == 'u' && ptr[1] == 'k')
		  gujin_param->kbdmap = kbd_uk;
		else if (ptr[0] == 'n' && ptr[1] == 'o')
		  gujin_param->kbdmap = kbd_no;
		else if (ptr[0] == 'f' && ptr[1] == 'i')
		  gujin_param->kbdmap = kbd_fi;
		else if (ptr[0] == 'd' && ptr[1] == 'k')
		  gujin_param->kbdmap = kbd_dk;
		else if (ptr[0] == 'n' && ptr[1] == 'l')
		  gujin_param->kbdmap = kbd_nl;
		else if (ptr[0] == 's' && ptr[1] == 'p')
		  gujin_param->kbdmap = kbd_sp;
		else if (ptr[0] == 'p' && ptr[1] == 't')
		  gujin_param->kbdmap = kbd_pt;
		else if (ptr[0] == 'i' && ptr[1] == 't')
		  gujin_param->kbdmap = kbd_it;
		else if (ptr[0] == 'f' && ptr[1] == 'r')
		  gujin_param->kbdmap = kbd_fr;
		else if (ptr[0] == 'b' && ptr[1] == 'e')
		  gujin_param->kbdmap = kbd_be;
		else if (ptr[0] == 'd' && ptr[1] == 'e')
		  gujin_param->kbdmap = kbd_de;
		else if (ptr[0] == 'c' && ptr[1] == 'h')
		  gujin_param->kbdmap = kbd_ch;
		else
		  EXTRA("Extra: do not know what letters %c%c means for layout\n", ptr[0], ptr[1]);
	      }
	  }
      if (fkbd)
	  fclose (fkbd);
      EXTRA("Extra: kbdmap %s\n", kbdname (gujin_param->kbdmap) ?: "undefined");
      }
  }

/**
 ** The complete stuff:
 **/
int main (int argc, char **argv)
  {
  const char *fin_name = 0, *fout_name = 0, *command;
  const char *geocmd = 0, *diskcmd = 0, *fscmd = 0, *helpcmd = 0;
  const char *positioncmd = 0, *checkcmd = 0, *serialcmd = 0;
  const char *mbr_device = 0, *beer_device = 0, *beercmd = 0;
  const char *partition_index = 0, *extra_size = 0;
  const char *report = 0, *remove = 0, *extract_mbr = 0, *sig0x29 = 0, *maxsectload = 0;
  unsigned char *file, cpt;
  unsigned filesize, full_image = 0, confirm = 1, mbr_guess = 0;
  unsigned report_flag = 0, remove_flag = 0, no_gujin_cmd_flag = 0, report_signature = 0;
  unsigned do_single_sector_load_patch = 0, do_USB_HDD_patch = 0;
  unsigned do_read_retry_patch = 0; /* DOS INT0x21 patch is done anyway when DOS executable */
  unsigned char sig0x29touse = 0x29;
  struct install_s install = {};
  install.fs.nb_sector_FAT_wasted = 0xFFFFFFFFU; /* i.e. try to optimise to have cluster aligned */
#ifdef WITH_INPUT_FILES
  unsigned tiny = 0, tiny_eltorito = 0;
#endif

  /* typed pointers pointing inside "file": */
  instboot_info_t *instboot_info;
  bootloader2_t *bootchain, *uninstall_mbr;
  bootbefore_t *bootbefore;
  boot1param_t *boot1param;
  gujin_param_t *gujin_param;

  command = strrchr (argv[0], '/');
  if (command == 0)
      command = argv[0];
    else
      command++; /* skip '/' */

#ifndef WITH_INPUT_FILES
  puts ("Please note: instboot is being replaced by the \"gujin\" executable.");
#endif

  for (cpt = 1; cpt < argc; cpt++) {
      unsigned found = 0, i;
      const char *tmp;
      const struct {
	  const char *shortopt, *longopt, **storeopt;
	  } std_option[] = {
	  { "-h",	"--help",		&helpcmd		},
	  { "-d=",	"--disk=",		&diskcmd		},
	  { "-COM",	"--serial=COM",		&serialcmd		},
	  { "-g=",	"--geometry=",		&geocmd			},
	  { "-p=",	"--position=",		&positioncmd		},
	  { "-c=",	"--check=",		&checkcmd		},
	  { 0,		"--fs=",		&fscmd			},
	  { 0,		"--beer=",		&beercmd		},
	  { 0,		"--mbr-device=",	&mbr_device		},
	  { 0,		"--beer-device=",	&beer_device		},
	  { 0,		"--partition_index=",	&partition_index	},
	  { 0,		"--extra_size=",	&extra_size		},
	  { 0,		"--report=",		&report			},
	  { 0,		"--remove=",		&remove			},
	  { 0,		"--extract-mbr=",	&extract_mbr		},
	  { 0,		"--sig0x29=",		&sig0x29		},
	  { "-m=",	"--maxsectload=",	&maxsectload		},
	  };

      if (argv[cpt][0] != '-') {
	  if (fin_name == 0)
	      fin_name = argv[cpt];
	    else if (fout_name == 0)
	      fout_name = argv[cpt];
	    else {
	      PARAMERR ("too many input/output files: '%s', '%s', '%s', "
			"abort.\n", fin_name, fout_name, argv[cpt]);
	      PRINTF ("Type \"%s -h\" or \"%s --help\" for help.\n",
			command, command);
	      return 1;
	      }
	  found++;
	  }

      for (i = 0; i < sizeof (std_option) / sizeof (std_option[0]); i++) {
	  const char *c = pattern_match (argv[cpt], std_option[i].shortopt, std_option[i].longopt);
	  if (c) {
	      if (*std_option[i].storeopt) {
		  PARAMERR ("too many '%s'/'%s' parameters: %s'%s' and '%s', abort.\n",
				std_option[i].longopt, std_option[i].shortopt,
				std_option[i].shortopt, *std_option[i].storeopt,
				argv[cpt]);
		  PRINTF ("Type \"%s -h\" or \"%s --help\" for help.\n",
			command, command);
		  return 1;
		  }
	      *std_option[i].storeopt = c;
	      found++;
	      }
	  }

      if (force_gujin_param (0, argv[cpt], &found) != 0)
	  return 1;

      force_install_param (&install, argv[cpt], &found);

      if ((tmp = pattern_match (argv[cpt], "-f", "--full")) != 0 && *tmp == 0) {
	  if (*tmp != 0) {
	      PARAMERR ("reading parameter \"%s\", abort.\n", argv[cpt]);
	      return 1;
	      }
	  full_image = 1;
	  found++;
	  }

      if ((tmp = pattern_match (argv[cpt], "--report", "")) != 0 && *tmp == 0) {
	  report_flag = 1;
	  found++;
	  }

      if ((tmp = pattern_match (argv[cpt], "--no_gujin_cmd", "")) != 0 && *tmp == 0) {
	  no_gujin_cmd_flag = 1;
	  found++;
	  }

      if ((tmp = pattern_match (argv[cpt], "--remove", "")) != 0 && *tmp == 0) {
	  remove_flag = 1;
	  found++;
	  }

      if ((tmp = pattern_match (argv[cpt], "--report-signature", "")) != 0 && *tmp == 0) {
	  report_signature = 1;
	  found++;
	  }

      if ((tmp = pattern_match (argv[cpt], "--mbr", "")) != 0 && *tmp == 0) {
	  if (*tmp != 0) {
	      PARAMERR ("reading parameter \"%s\", abort.\n", argv[cpt]);
	      return 1;
	      }
	  mbr_guess = 1;
	  found++;
	  }

      if ((tmp = pattern_match (argv[cpt], "--single_sector_load", "")) != 0 && *tmp == 0) {
	  if (*tmp != 0) {
	      PARAMERR ("reading parameter \"%s\", abort.\n", argv[cpt]);
	      return 1;
	      }
	  do_single_sector_load_patch = 1;
	  found++;
	  }

      if ((tmp = pattern_match (argv[cpt], "--read_retry", "")) != 0 && *tmp == 0) {
	  if (*tmp != 0) {
	      PARAMERR ("reading parameter \"%s\", abort.\n", argv[cpt]);
	      return 1;
	      }
	  do_read_retry_patch = 1;
	  found++;
	  }

      if ((tmp = pattern_match (argv[cpt], "--usb_hdd", "")) != 0 && *tmp == 0) {
	  if (*tmp != 0) {
	      PARAMERR ("reading parameter \"%s\", abort.\n", argv[cpt]);
	      return 1;
	      }
	  do_USB_HDD_patch = 1;
	  found++;
	  }

      if ((tmp = pattern_match (argv[cpt], "--force", 0)) != 0 && *tmp == 0) { /* force_textmode, force_probe_root */
	  if (*tmp != '\0') {
	      PARAMERR ("reading parameter \"%s\", abort.\n", argv[cpt]);
	      return 1;
	      }
	  confirm = 0;
	  found++;
	  }

#ifdef WITH_INPUT_FILES
      if ((tmp = pattern_match (argv[cpt], "--tiny", "-t")) != 0 && *tmp == 0) {
	  if (*tmp != '\0') {
	      PARAMERR ("reading parameter \"%s\", abort.\n", argv[cpt]);
	      return 1;
	      }
	  tiny = 1;
	  found++;
	  }
#endif

      if ((tmp = pattern_match (argv[cpt], "-v", "--version")) != 0 && *tmp == 0) {
	  if (*tmp != '\0') {
	      PARAMERR ("reading parameter \"%s\", abort.\n", argv[cpt]);
	      return 1;
	      }
	  print_version (command);
	  return 0;
	  }

      if ((tmp = pattern_match (argv[cpt], "-w", "--verbose")) != 0
		&& argv[cpt][10] != '0' &&  argv[cpt][10] != '1') { /* not gujin_attr.version */
	  found++;
	  if (treat_verbose (&verbose, tmp) != 0)
	      return 1;
	  verbose.inited = 1;
	  }

      if ((tmp = pattern_match (argv[cpt], "--hpcompaqbug", "")) != 0 && *tmp == 0) {
	  install.misc.mbr_codesig_bug = 3;
	  found++;
	  }

      if (found != 1) {
	  PARAMERR ("reading parameter \"%s\", abort.\n", argv[cpt]);
	  return 1;
	  }
      }

  if (helpcmd) {
      print_help (command, helpcmd);
      return 0;
      }

  if (report_flag)
      report = fin_name;
  if (report)
      return gujin_report(report, confirm);

  if (remove_flag)
      remove = fin_name;
  if (remove)
      return gujin_uninstall(remove, confirm);

  if (report_signature)
      return gujin_report_signature();

  if (extract_mbr)
      return gujin_extract_mbr(extract_mbr, fin_name);

  if (partition_index != 0 && (*partition_index < '1' || *partition_index > '4')) {
      PARAMERR ("reading parameter partition_index: shall be 1, 2, 3 or 4 - read \"%s\", abort.\n", partition_index);
      return 1;
      }

  { /* Search 0xBEFC1F03 in disk.c for info */
  char HPCompaqstr[sizeof("HP Compaq 8000 Elite")];
  int fmem = open("/dev/mem", O_RDONLY);
  if (fmem == -1)
      EXTRA ("Extra: Cannot open /dev/mem to detect HPCompaq bug\n");
    else if (lseek64(fmem, 0xFD477, SEEK_SET) != 0xFD477)
      perror ("Cannot seek /dev/mem to detect HPCompaq bug");
    else if (read(fmem, HPCompaqstr, sizeof(HPCompaqstr)) != sizeof(HPCompaqstr))
      perror ("Cannot read /dev/mem to detect HPCompaq bug");
    else if (!strncmp (HPCompaqstr, "HP Compaq 8000 Elite", strlen ("HP Compaq 8000 Elite"))) {
      EXTRA ("Extra: \"HP Compaq 8000 Elite\" detected, forcing --hpcompaqbug parameter\n");
      install.misc.mbr_codesig_bug = 3;
      }
  if (fmem != -1)
      close (fmem);
  }

  if (sig0x29) {
      sig0x29touse = strtol (sig0x29, 0, 0);
      EXTRA("Extra: Using the byte signature 0x%X instead of 0x29 in the MBR\n", sig0x29touse);
      }

  if (maxsectload) {
      install.misc.maxsectloadtouse = strtol (maxsectload, 0, 0);
      EXTRA("Extra: Will load boot2 with max %u sectors at a time\n", install.misc.maxsectloadtouse);
      }

  if (fin_name == 0 || fout_name == 0) {
      if (fin_name == 0) {
	  /* also no command line parameter at all */
	  print_short_help (command);
	  return 1;
	  }
#ifdef WITH_INPUT_FILES
	else if (fin_name != 0 && fout_name == 0) {
	  fout_name = fin_name;
	  const char *extension = strrchr (fout_name, '.');

	  if (tiny == 0)
	      fin_name = "boot.bin";
	  else if (extension && !strcmp(extension, ".exe"))
	      fin_name = "tinyexe.bin";
	  else if (extension && !strcmp(extension, ".bcd")) {
	      fin_name = "tinycdrom.bin";
	      tiny_eltorito = 1;
	      }
	  else {
	      PARAMERR ("Input file not defined for tiny, only target '%s', abort.\n", fin_name);
	      fprintf (stdout, "Please explicitely select (as first parameter) one internal input file, choice:\n"
		"  tinystd.bin  only for floppies, no partitions analysed, only FAT support\n"
		"  tinyusb.bin  for USB thumb drives, only analyse FAT partitions\n"
		"  tinyext4.bin for single Linux distribution PC, only analyse ext*fs partitions\n");
	      fprintf (stdout, "It will load the newest (by date) vmlinux/initrd found,\n"
		"unless Control-Break is pressed while loading, then it loads the next one.\n"
		"Linux parameters can be provided now with the --cmdline= parameter,\n"
		"or later by adding in file /boot/gujin.cmd the line '::::::commandline',\n"
		"in the later case the filesystem must have a label.\n");
	      return 1;
	      }
	  if (!verbose.inited)
	     verbose.reprint = verbose.warning = 0;
	  EXTRA("Extra: select internal input %s\n", fin_name);
	  }
#endif
	else {
	  PARAMERR ("Input file '%s', no output file/device, abort.\n", fin_name);
	  return 1;
	  }
      }

  if (mbr_guess && !mbr_device) {
      static char mbrguess[256];
      get_mbrname_from_devicename (mbrguess, fout_name);
      EXTRA("Extra: '--mbr' given, has to guess mbr device from '%s', guessing '%s'\n", fout_name, mbrguess);
      mbr_device = mbrguess;
      }

  struct stat unused;
  if (mbr_device && stat(mbr_device, &unused) == 0) { /* i.e. target exists */
      xFILE fout;

      /* xfopenmodify because it is maybe a device image file and we do not want to trunk it: */
      if (xfopenmodify (mbr_device, &fout, 0)) {
	  ERROR ("Cannot open output device/file to synchronise partitions from the MBR on \"%s\"\n", mbr_device);
	  return 1;
	  }
      if (fout >= 0) {
	  if (ioctl (xfileno(fout), BLKRRPART, 0) >= 0)
	      EXTRA ("Extra: partition table of %s synchronised successfully\n", mbr_device);
	    else
	      EXTRA ("Extra: partition table of %s synchronisation failed\n", mbr_device);
	  }
      }

  struct stat statbuf;
  if (fout_name && access ("/usr/bin/devkit-disks", X_OK) == 0 && access ("/bin/grep", X_OK) == 0
	&& stat(fout_name, &statbuf) == 0 && S_ISBLK(statbuf.st_mode)) {
      char command[255], dev[80];
      unsigned cpt = 0;
      do
	  dev[cpt] = fout_name[cpt];
	  while ( (   (fout_name[cpt] >= '/' && fout_name[cpt] <= '9')
		|| (fout_name[cpt] >= 'a' && fout_name[cpt] <= 'z')
		|| fout_name[cpt] == '_' || fout_name[cpt] == '~' || fout_name[cpt] == '.'
		|| fout_name[cpt] == '-' || fout_name[cpt] == '+'
		|| (fout_name[cpt] >= 'A' && fout_name[cpt] <= 'Z'))
		&& ++cpt < sizeof(dev)-1);
      dev[cpt] = '\0';
      sprintf (command, "/usr/bin/devkit-disks --show-info %s | /bin/grep -e interface -e model -e vendor", dev);
      printf ("Check target: `%s`:\n", command);
      /* warning: ignoring return value of "system" */
      if (system (command));
      }

  {
  FILE *file_mount;
  if ((file_mount = fopen ("/proc/mounts", "r")) == NULL)
      PROC ("Proc: cannot open '/proc/mounts' for reading!");
    else {
      int ch, ch2, skip_to_eol = 0, pos = 0, patlen = strlen(fout_name);
      while ((ch = fgetc(file_mount)) != EOF) {
	  if (skip_to_eol) {
	      if (ch == '\n')
		  skip_to_eol = 0;
	      continue;
	      }
	  if (ch != fout_name[pos++]) {
	      pos = 0;
	      skip_to_eol = 1;
	      continue;
	      }
	  if (pos >= patlen) {
	      ch = fgetc(file_mount);
	      if (ch == ' ' || ch == '\t') {
		  ERROR ("Partition/device %s already mounted!\n", fout_name);
		  return 1;
		  }
		else if (ch >= '1' && ch <= '9' && ((ch2 = fgetc(file_mount)) == ' ' || ch2 == '\t' || (ch2 >= '1' && ch2 <= '9'))) {
		  ERROR ("Partition/device %s%c already mounted!\n", fout_name, ch);
		  return 1;
		  }
	      }
	  }
      fclose (file_mount);
      }
  }

  file = allocate_and_read (fin_name, &filesize);
  if (file == 0) {
      ERROR ("malloc or read input file.\n");
      return 1;
      }

  bootbefore = (bootbefore_t *)file;

  /* Some init and checks from the current image file: */
  instboot_info = init_and_check_instboot_info (file, filesize);
  if (instboot_info == 0) {
      ERROR ("recognising Gujin file (instboot_info).\n");
      return 1;
      }

  if (   bootbefore->part2.Signaturebyte0x29 != 0x29
      && ((const bootbefore_FAT32_t *)bootbefore)->part2.Signaturebyte0x29 != 0x29) {
      ERROR ("input file do not have the boot sector 0x29 signature, abort.\n");
      return 1;
      }

  boot1param = init_and_check_boot1param (file, instboot_info);
  if (boot1param == 0) {
      ERROR ("recognising Gujin file (boot1param).\n");
      return 1;
      }

  bootchain = init_and_check_bootchain (file, instboot_info);
  if (bootchain == 0) {
      ERROR ("recognising Gujin file (bootchain).\n");
      return 1;
      }

  uninstall_mbr = init_and_check_uninstall_mbr (file, instboot_info);
  if (uninstall_mbr == 0) {
      ERROR ("recognising Gujin file (uninstall_mbr).\n");
      return 1;
      }

  gujin_param = init_and_check_gujin_param (file, instboot_info);
  if (gujin_param == 0) {
      ERROR ("recognising Gujin file (gujin_param).\n");
      return 1;
      }

#ifdef WITH_INPUT_FILES
  if (tiny_eltorito && gujin_param->stop_emulation == 1) {
      gujin_param->stop_emulation = 2;
      EXTRA ("Extra: with minigujin.bcd, force --stop_emulation=2\n");
      }
#endif

  set_default_cmdline(gujin_param, no_gujin_cmd_flag);

  for (cpt = 1; cpt < argc; cpt++) {
      if (force_gujin_param (gujin_param, argv[cpt], 0) != 0) {
	  ERROR ("treating parameter '%s'.\n", argv[cpt]);
	  return 1;
	  }
      }
  setup_gujin_param_from_environment (gujin_param);

  report_size (instboot_info, gujin_param);

  REPRINTF("Reprint: %s %s %s %s \\\n", command, fin_name, fout_name, full_image? "--full " : "");
  REPRINTF("Reprint:    --cmdline=\"%s\" \\\n", gujin_param->extra_cmdline);

  if (serialcmd) {
      if (force_serialconf (serialcmd,
		(serialconf_t *)(file + instboot_info->password_serial_adr),
		(unsigned short *)(file + instboot_info->serialconf_port_plus_1_adr)
		) != 0)
	  return 1;
      }

  /*
   * param treatments:
   */
  /* report_install_content (&install); */
  /* report_bootsector_content (bootbefore); */
  setup_install_from_bootbefore (&install, bootbefore, instboot_info);
//  printf ("AFTER bootbefore: "); report_install_content (&install);

  if (setup_beer (beercmd, &install.beersect) != 0)
      return 1;
  if (   install.beersect.BeerNbSector != 8
      || install.beersect.BeerPosition != 4
      || install.beersect.BeerPartition.minSector != 720
      || install.beersect.BeerPartition.maxSector != 65536)
      REPRINTF("Reprint:    --beer=%u,%u,%u,%u \\\n",
		install.beersect.BeerNbSector,
		install.beersect.BeerPosition,
		install.beersect.BeerPartition.minSector,
		install.beersect.BeerPartition.maxSector);

  if (setup_geometry (geocmd, &install, &diskcmd, &fscmd, &positioncmd,
	&mbr_device, &beer_device, fout_name, &partition_index) != 0) {
      return 1;
      }
  if (install.misc.iso_file == 1) {
      /* we want those parameter as default if --disk=[E]BIOS:0x00,auto:
	 --quickboot=1 --search_el_torito=1 --search_disk_mbr=0 --search_part_mbr=0 --search_topdir_files=0
	 --search_subdir_files=0 --probe_bios_floppy_disk=1 --probe_bios_hard_disk=0 --probe_ide_disk=0 --probe_cdrom=0
	  we want those parameter as default if --disk=[E]BIOS:0x80,auto:
	 --quickboot=1 --search_el_torito=1 --search_disk_mbr=0 --search_part_mbr=0 --search_topdir_files=0
	 --search_subdir_files=0 --probe_bios_floppy_disk=1 --probe_bios_hard_disk=1 --probe_ide_disk=0 --probe_cdrom=0
	 */
      gujin_param->timeout_autoload = 1;
      gujin_param->autoload.last_loaded = 0;
      gujin_param->autoload.total_loadable = 1;
      gujin_param->autoload.init_page = 0;
      gujin_param->autoload.total_page = 1;
      gujin_param->attrib.verbose = 0;
      gujin_param->attrib.use_gujin_embedded_font = 0; /* Saves time, just display copyright */
      gujin_param->attrib.disk_write_enable = 0;
      gujin_param->default_video_mode = 3; /* message if EDID abscent */
      EXTRA ("Extra: set default setup for fast ISO booting on superfloppy.\n");
      gujin_param->attrib.search_el_torito = 1;
      gujin_param->attrib.search_disk_mbr = 0;
      gujin_param->attrib.search_part_mbr = 0;
      gujin_param->attrib.search_topdir_files = 0;
      gujin_param->attrib.search_subdir_files = 0;
      gujin_param->attrib.probe_bios_floppy_disk = 1;
      if (!strncmp(diskcmd, "BIOS:0x8", strlen("BIOS:0x8")) || !strncmp(diskcmd, "EBIOS:0x8", strlen("EBIOS:0x8")))
	  gujin_param->attrib.probe_bios_hard_disk = 1;
	else
	  gujin_param->attrib.probe_bios_hard_disk = 0;
      gujin_param->attrib.probe_ide_disk = 0;
      gujin_param->attrib.probe_cdrom = 0;
      gujin_param->attrib.probe_dos_disk = 0;
      }

//  printf ("AFTER geocmd: "); report_install_content (&install);

  if (setup_output (diskcmd, &install.gujin2_load,
			&install.wholedisk, &install.partdisk, &install.ide, &install.fs) != 0) {
      if (geocmd == 0 && diskcmd != 0 && geteuid() == 0)
	  HINT ("Have a '--disk=' parameter but no '--geometry', didn't you invert them?\n");
      return 1;
      }

  /* If install.partdisk.BytePerSector != 512 (but 2048 like CDROM), one can have: */
  /* Also the --extra_size= parameter, to plan to upgrade to a bigger Gujin installed
	in a partition/BEER without reformating the partition (reserved space in FAT big enough) */
  if ((filesize % install.partdisk.BytePerSector) != 0 || extra_size) {
      unsigned newsize = DIVROUNDUP (filesize, install.partdisk.BytePerSector) * install.partdisk.BytePerSector;
      unsigned char *oldfile = file;

      if (filesize % install.partdisk.BytePerSector)
	  EXTRA ("Extra: Size of sector not 512 but %u bytes, round up filesize from %u*512 to %u*512\n",
		install.partdisk.BytePerSector, filesize/512, newsize/512);
      if (extra_size) {
	  unsigned xsize = 0, xsect;
	  while (*extra_size >= '0' && *extra_size <= '9')
	      xsize = 10 * xsize + *extra_size++ - '0';
	  if (xsize == 0 || (*extra_size != '\0' && *extra_size != ' ' && *extra_size != '\t')) {
	      ERROR ("did not understand '--extra_size=' parameter at '%s'", extra_size);
	      return 1;
	      }
	    else
	      REPRINTF("Reprint:    --extra_size=%u \\\n", xsize);
	  xsect = DIVROUNDUP (xsize, install.partdisk.BytePerSector) * install.partdisk.BytePerSector;
	  EXTRA ("Extra: Increase the size (%u bytes) by %u bytes i.e. %u sectors\n", newsize, xsize, xsect);
	  newsize += xsect;
	  }
      unsigned char *newptr = realloc (file, newsize);
      if (newptr == NULL) {
	  EXTRA ("\ncannot realloc memory for buffer.\n");
	  free (file);
	  return 1;
	  }
      file = newptr;
      memset (file + filesize, 0, newsize - filesize);

      /* Well, do not forget to re-align pointers into "file" area: */
      instboot_info = (instboot_info_t *)((unsigned long)instboot_info + file - oldfile);
      bootchain =     (bootloader2_t *)  ((unsigned long)bootchain     + file - oldfile);
      uninstall_mbr = (bootloader2_t *)  ((unsigned long)uninstall_mbr + file - oldfile);
      bootbefore =    (bootbefore_t *)   ((unsigned long)bootbefore    + file - oldfile);
      boot1param =    (boot1param_t *)   ((unsigned long)boot1param    + file - oldfile);
      gujin_param =   (gujin_param_t *)  ((unsigned long)gujin_param   + file - oldfile);
      filesize = newsize;
      }

  if (install.wholedisk.BytePerSector == 0)
      install.wholedisk.BytePerSector = install.partdisk.BytePerSector; /* used to get OutOfFsSector */

  /* setup_output() may have overwrited logical C/H/S by physical one... */
  if (!gujin2_loaded_externally(install.gujin2_load)) {
      REPRINTF("Reprint:    --geometry=%llu,%llu,%u,%u,%u,%llu,%llu,%u,%u,%u \\\n",
		install.partdisk.total_sector, install.partdisk.start_sector,
		install.partdisk.heads, install.partdisk.sectpertrack,
		install.partdisk.BytePerSector,
		install.wholedisk.total_sector, install.wholedisk.start_sector,
		install.wholedisk.heads, install.wholedisk.sectpertrack,
		install.wholedisk.BytePerSector);
      if (install.gujin2_load == gujin2_load_bios) {
	  if (install.wholedisk.diskindex == 0x7F)
	      REPRINTF("Reprint:    --disk=BIOS:0x%X,auto ", install.partdisk.diskindex);
	    else if (install.partdisk.diskindex == install.wholedisk.diskindex)
	      REPRINTF("Reprint:    --disk=BIOS:0x%X ", install.wholedisk.diskindex);
	    else
	      REPRINTF("Reprint:    --disk=BIOS:0x%X,0x%X ", install.partdisk.diskindex, install.wholedisk.diskindex);
	  }
	else if (install.gujin2_load == gujin2_load_ebios) {
	  if (install.wholedisk.diskindex == 0x7F)
	      REPRINTF("Reprint:    --disk=EBIOS:0x%X,auto ", install.partdisk.diskindex);
	    else if (install.partdisk.diskindex == install.wholedisk.diskindex)
	      REPRINTF("Reprint:    --disk=EBIOS:0x%X ", install.wholedisk.diskindex);
	    else
	      REPRINTF("Reprint:    --disk=EBIOS:0x%X,0x%X ", install.partdisk.diskindex, install.wholedisk.diskindex);
	  }
	else if (install.gujin2_load == gujin2_load_CDROM_noemul)
	  REPRINTF("Reprint:    --disk=BCD,auto\n");
	else
	  REPRINTF("Reprint:    --disk=IDE:%s,0x%X,0x%X,%u ",
		(install.gujin2_load == gujin2_load_ide_chs)
			? "chs"
			: ((install.gujin2_load == gujin2_load_ide_lba48)? "lba48" : "lba28"),
		install.ide.data_adr,
		install.ide.dcr_adr,
		install.wholedisk.diskindex);
      }

  if (gujin2_loaded_externally (install.gujin2_load) || install.gujin2_load == gujin2_load_CDROM_noemul) {
      EXTRA ("Extra: dos/PIC/BCD bootloader, no FS setup, copy "
		"partdisk.{total_sector,BytePerSector} to wholedisk\n");
      install.wholedisk.total_sector = install.partdisk.total_sector;
      if (positioncmd)
	  EXTRA ("Extra: ignoring '--position=' set to '%s' for DOS/PIC bootloader\n",
			positioncmd);
      positioncmd = 0;
      install.fs.PhysicalDriveNb = install.partdisk.diskindex;
      }
    else if (positioncmd != 0 && !strcasecmp (positioncmd, "infs")) {
      install.fs.NbSectorPerCluster = 4096/512; /* FIXME: use an ioctl */
      EXTRA ("Extra: --position=infs, ignore fscmd: '%s', note that NbSectorPerCluster=%u\n",
		fscmd? fscmd : "", install.fs.NbSectorPerCluster);
      }
    else {
      /* OutOfFsSector used only if positioncmd say to do so: */
      unsigned OutOfFsSector = DIVROUNDUP (filesize, install.wholedisk.BytePerSector);
      if (mbr_device)
	  OutOfFsSector += 1;

      install.fs.NbFsSector = install.partdisk.total_sector; /* setup default */
      if (setup_fs (fscmd? fscmd : "", &install.fs,
			positioncmd, OutOfFsSector,
			install.wholedisk.BytePerSector,
			install.partdisk.diskindex,
			install.misc.output_partition_offset) != 0) {
	  ERROR ("did not understand '--fs=%s' parameter\n", fscmd? fscmd : "");
	  return 1;
	  }
      }

  if (gujin2_loaded_by_dos (install.gujin2_load)) {
      if (gujin_param->compilation_attrib.bios_only) {
	  ERROR ("compilation options includes (SETUP & BIOS_ONLY)\n"
		  " so cannot produce a DOS executable. Recompile Gujin.\n");
	  return 1;
	  }
      /* set the '+' as the display char (could be Sigma: 228 for checksum, but codepage pb) */
      gujin_param->dot_msg[0] = '+';
      gujin_param->attrib.disk_write_enable = 0; /* Why not in FORCED_WHEN_DOS() ? */

      if (mbr_device != 0) {
	  ERROR ("cannot have a '--mbr-device=' parameter for DOS .com or .exe files.\n");
	  return 1;
	  }
      FORCED_WHEN_DOS (1, gujin_param->attrib);
      }
    else {
      if (install.gujin2_load == gujin2_load_CDROM_noemul && instboot_info->read_disk_1st_adr == 0) {
	  ERROR ("compilation options excludes (DISK_SUPPORT & CDBIOS_SUPPORT)\n"
		" so cannot produce a noemul bootable CDROM. Recompile Gujin.\n");
	  return 1;
	  }
      FORCED_WHEN_DOS (0, gujin_param->attrib);

      if (do_single_sector_load_patch && install.gujin2_load != gujin2_load_bios) {
	  ERROR ("Cannot process single_sector_load patch when Gujin2 loaded without standard BIOS\n");
	  do_single_sector_load_patch = 0;
	  }
      if (do_single_sector_load_patch || do_read_retry_patch)
	  REPRINTF("Reprint:    %s %s \\\n",
		do_single_sector_load_patch? "--single_sector_load": "", do_read_retry_patch? "--read_retry" : "");
      }

  if (install.misc.forced_Heads || install.misc.forced_SectPerTrack)
      REPRINTF("Reprint:    %s%u %s%u \\\n",
	install.misc.forced_Heads? "--head=": "", install.misc.forced_Heads,
	install.misc.forced_SectPerTrack? "--sectorpertrack=" : "", install.misc.forced_SectPerTrack);

  do_patch (file, instboot_info, install.gujin2_load,
	    install.wholedisk.diskindex == 0x7F,
	    install.wholedisk.heads == 0 && install.wholedisk.sectpertrack != 0	&& (install.wholedisk.sectpertrack & 0x80) == 0,
	    do_single_sector_load_patch, do_read_retry_patch);

  update_bootbefore_from_install (bootbefore, &install);

  /* We check but cannot proceed this one here because only the PBR has to be patched, not the MBR: */
  if (do_USB_HDD_patch && install.gujin2_load != gujin2_load_bios) {
      ERROR ("Cannot process USB_HDD patch when Gujin2 loaded without standard BIOS\n");
      do_USB_HDD_patch = 0;
      }
    else if (do_USB_HDD_patch && ((FAT_bootsect1_t *)file)->NbHiddensector != ((FAT_bootsect1_t *)file)->NbSectorpertrack) {
      ERROR ("Cannot process USB_HDD patch when partition doesn't have NbHiddensector(%u) == NbSectorpertrack(%u)\n",
		((FAT_bootsect1_t *)file)->NbHiddensector, ((FAT_bootsect1_t *)file)->NbSectorpertrack);
      do_USB_HDD_patch = 0;
      }
    else if (do_USB_HDD_patch)
      REPRINTF("Reprint:    --usb_hdd  \\\n");

  /* --extra-partition=<disksize|hpa|none|SAORAB>,PartitionBootRecordLba>,<BeerSectorLba>
	last disk sector - 8 < BeerSectorLba <= last disk sector, default 4 sectors, 2 Kbytes.
		refuse to erase an existing Beer Sector if attr.service_dir_present or attr.read_only !
	(last disk sector - PartitionBootRecordLba) short enough for FAT16 and > 8 Mbytes
	disksize best if you know your disk can DevConfOverlay (letter 'C' in startup screen)
	hpa not that bad (letter 'H' in startup screen), it cannot be locked on HPA first generation
	none does not protect this extra partition at all, you should never create a standard
		partition here using fdisk/cfdisk/automatic partition detection from Linux installers...
	SAORAB to be implemented (how to force the BIOS to reprobe IDE disk IDE_IDENTIFY_DEVICE
				to update its functions?)
	*/

  if (positioncmd != 0 && !strcasecmp (positioncmd, "infs")) {
      unsigned unit_size;
      struct stat sb;
      if (stat(fout_name, &sb) == 0) {
	  ERROR ("Cannot install, file '%s' already exists (size %llu, %llu blocks allocated i.e %llu bytes allocated).\n", fout_name, (unsigned long long)sb.st_size, (unsigned long long)sb.st_blocks, (unsigned long long)sb.st_blocks * 512);
	  return 1;
	  }
      /* BUG FIX: we need to create a contigous file containing the uninstall MBR, increase filesize and adjust lba_end_chain: */
      if (infs_filemap (fout_name, filesize + install.wholedisk.BytePerSector, &unit_size,
		&install.misc.lba_start_chain, &install.misc.lba_end_chain, 0, install.misc.holy_filesystem) != 0)
	  return 1;
      install.misc.lba_end_chain -= 1;
      install.misc.infs_position = install.misc.lba_start_chain;
      install.misc.lba_start_chain = install.misc.lba_start_chain * (unit_size / install.wholedisk.BytePerSector);
//      install.misc.lba_end_chain = install.misc.lba_end_chain * (unit_size / install.wholedisk.BytePerSector); /* rounded up to FS block size */
      install.misc.lba_end_chain = install.misc.lba_start_chain + DIVROUNDUP (filesize, install.wholedisk.BytePerSector);	/* contiguous file */

      EXTRA ("Extra: temporary contiguous file created and mapped at %llu for %llu sectors, offset it by %llu sectors\n",
		install.misc.lba_start_chain, install.misc.lba_end_chain - install.misc.lba_start_chain,
		install.partdisk.start_sector + install.wholedisk.start_sector);
      install.misc.lba_start_chain += install.partdisk.start_sector + install.wholedisk.start_sector;
      install.misc.lba_end_chain += install.partdisk.start_sector + install.wholedisk.start_sector;
      if (install.wholedisk.BytePerSector == 512)
	  install.misc.lba_start_chain += 1;
      REPRINTF(" \\\nReprint:    --position=infs \\\n");
      }
    else if (!gujin2_loaded_externally (install.gujin2_load) && (install.gujin2_load != gujin2_load_CDROM_noemul)) {
      report_bootsector_content (bootbefore);
      /*
       * the last parameter, lba_start_partition, is the partition offset, i.e.
       * install.partdisk.start_sector (equal to bootbefore->part1.NbHiddensector):
       * While we are here, add also the disk offset (number of sectors before MBR) if
       * SAORAB used - not checked at all !
       */
      if (setup_misc (&install.misc, &install.fs, install.wholedisk.BytePerSector,
		mbr_device != 0, install.gujin2_load, filesize, positioncmd,
		install.partdisk.start_sector + install.wholedisk.start_sector) != 0) {
	  ERROR ("in setup_misc\n");
	  return 1;
	  }
      }
    else { /* still simple chain for the checksum calculus: */
      /* do not chain the copy of the start sector, but skip for CD noemul: */
      install.misc.lba_start_chain = install.partdisk.start_sector;
      /* -1 for the start sector (the uninstall_mbr is never chained): */
      install.misc.lba_end_chain = install.misc.lba_start_chain + DIVROUNDUP (filesize, install.wholedisk.BytePerSector);
      /* If BytePerSector > 512, chain the first sector to calculate the end of the sector checksum: */
      if (install.wholedisk.BytePerSector == 512)
	  install.misc.lba_start_chain ++;
      }

  /* Chain blocks and checksum : */
  if (chs_chain (&install, &boot1param->bootloader2, bootchain, mbr_device? uninstall_mbr : 0, install.partdisk.start_sector) != 0) {
      ERROR ("chaining the loader.\n");
      return 1;
      }

  if (install.gujin2_load == gujin2_load_CDROM_noemul) {
      if (instboot_info->bootchain_nbblock <= 20 + 3 || bootchain[19].header.next != 0) {
	  ERROR ("late patching bootchain for CDROM noemul -boot-info-table.\n");
	  return 1;
	  }
	else {
	  /* do a spare copy of the part overwritten by -boot-info-table */
	  /* sizeof (bootloader2_t) == 26, we need 3 blocks */
	  unsigned short *ptr = (unsigned short *)&bootchain[20];
	  EXTRA ("Extra: -boot-info-table spare copy at file offset %lu\n", (unsigned long)ptr - (unsigned long)file);
	  *ptr++ = 0xFEED;
	  *ptr++ = 0x0008;
	  *ptr++ = (64 - 8)/2;
	  memcpy (ptr, file + 8, 64 - 8);
	  memset (file + 8, 0, 64 - 8); /* so that checksum1 success with or without -boot-info-table */
	  memcpy (file + 3, "CD", sizeof("CD")); /* "Gujin1" is too long */
	  }
      }
  if (install.misc.default_ide_password[0] != '\0') {
      if (bootchain[instboot_info->bootchain_nbblock - 4].header.next != 0) {
	  ERROR ("late patching bootchain for default_ide_password.\n");
	  return 1;
	  }
	else {
	  /* I am a bomb technician - If you see me running, try to catch up! */
	  unsigned short *ptr = (unsigned short *)&bootchain[instboot_info->bootchain_nbblock - 3];
	  *ptr++ = 0xBEAF;
	  *ptr++ = (32)/2;
	  memcpy (ptr, install.misc.default_ide_password, sizeof(install.misc.default_ide_password) - 1);
	  REPRINTF("Reprint:    --default_ide_password=%s \\\n", install.misc.default_ide_password);
	  }
      }

  if (install.gujin2_load == gujin2_load_ebios
	&& !(mbr_device && *mbr_device)
	&& bootbefore->part1.jmpinstruction[0] == 0xEB
	&& bootbefore->part1.EBIOSaccess == 0x90)
      bootbefore->part1.EBIOSaccess = 0xE; /* i.e. EBIOS OK */

  if (calculate_checksum (file, filesize, instboot_info, boot1param,
				bootchain, install.wholedisk.BytePerSector) != 0) {
      ERROR ("checksum-ing the loader.\n");
      return 1;
      }

  {
  const char *new_mbr = 0, *old_mbr = 0;
  char mbr_buffer[2 * install.wholedisk.BytePerSector];

  if (!gujin2_loaded_externally (install.gujin2_load)) {
      if (beer_device && *beer_device)
	  REPRINTF("Reprint:    --beer-device=%s ", beer_device);
	else if (mbr_device && *mbr_device)
	  REPRINTF("Reprint:    ");
      if (mbr_device && *mbr_device) {
	  unsigned char partition_type;

	  if (install.fs.FatSize == 12)
	      partition_type = DOS_FAT12;
	    else if (install.fs.FatSize == 16) {
	      if (install.wholedisk.heads * install.wholedisk.sectpertrack == 0)
		  partition_type = (install.gujin2_load == gujin2_load_ebios) ? Win95_FAT16LBA : BIG_DOS_FAT16;
		else
		  partition_type = (install.wholedisk.total_sector / (install.wholedisk.heads * install.wholedisk.sectpertrack) > 1024) ? Win95_FAT16LBA : BIG_DOS_FAT16;
	      }
	    else if (install.fs.FatSize == 32) {
	      if (install.wholedisk.heads * install.wholedisk.sectpertrack == 0)
		  partition_type = Win95_FAT32LBA;
		else
		  partition_type = Win95_FAT32;
	      }
	    else {
	      ERROR ("The partiton created is not FAT12, FAT16 nor FAT32.\n");
	      return 1;
	      }
	  REPRINTF("--mbr-device=%s ", mbr_device);
	  if (read_and_update_mbr (mbr_buffer, file, mbr_device,
			install.wholedisk.BytePerSector, &new_mbr, &old_mbr,
			/* If initialised, give BIOS defaults for C/H/S, not physical, and BIOS/EBIOS access */
			install.partdisk.heads, install.partdisk.sectpertrack, install.gujin2_load,
			/* but the total size: */
			install.wholedisk.total_sector, sig0x29touse,
			/* if create a partition table (target device: file, USB disk), say its size and acces type and index: */
			install.partdisk.start_sector, install.partdisk.total_sector, partition_type,
			((beer_device && *beer_device) || install.misc.infs_file) ? 0 : partition_index,	// no partition created in MBR (when no 0xAA55) if BEER or infs
			/* Only required for BIOS HD emulation (with MBR), but then really required: */
			install.wholedisk.diskindex == 0x7F ? 0xFF : install.partdisk.diskindex,
			install.fs.MediaDescriptor, instboot_info, install.misc.mbr_codesig_bug) != 0) {
//			0xF8) != 0) {
	      if (install.misc.infs_file) {
		  unlink (fout_name);
		  ERROR ("Early reading the MBR for the output (nothing written, contiguous file %s removed).\n", fout_name);
		  }
		else
		  ERROR ("Early reading the MBR for the output (nothing written).\n");
	      return 1;
	      }

#define ALTMBRNAME "BOOTSECT.DOS"	// in make.h, not included here
#ifdef ALTMBRNAME
	  if (positioncmd != 0 && !strcasecmp (positioncmd, "infs") && old_mbr && install.partdisk.start_sector == 0 && fout_name[0] == '/') {
	      const char *srcptr = fout_name;
	      char nbuff[256], *dstptr = nbuff;
	      while (*srcptr)
		  *dstptr++ = *srcptr++;
	      while (*--dstptr != '/') continue;
	      srcptr = ALTMBRNAME;	/* "bootsect.dos" */
	      while (*srcptr)
		  *++dstptr = *srcptr++;
	      *++dstptr = '\0';
	      struct stat statbuff;
	      if (stat (nbuff, &statbuff) == 0) {
		  if (!S_ISREG(statbuff.st_mode)) {
		      EXTRA ("Extra: do not create \"%s\" bootable file, file exists and is not regular\n", nbuff);
		      nbuff[0] = '\0';
		      }
		    else if (statbuff.st_size != install.wholedisk.BytePerSector) {
		      EXTRA ("Extra: do not create \"%s\" bootable file, file exists and its size is %u\n", nbuff, (unsigned)statbuff.st_size);
		      nbuff[0] = '\0';
		      }
		    else if (statbuff.st_uid != 0) {
		      EXTRA ("Extra: do not create \"%s\" bootable file, file exists and is not own by root but %u\n", nbuff, statbuff.st_uid);
		      nbuff[0] = '\0';
		      }
		    else
		      EXTRA ("Extra: replacing \"%s\" bootable file containing initial MBR, take care of its partition table...\n", nbuff);
		  }
		else
		      EXTRA ("Extra: creating \"%s\" bootable file containing initial MBR, take care of its partition table...\n", nbuff);
	      if (nbuff[0] != '\0') {
		  FILE *fbootsect = fopen(nbuff, "w");
		  if (fbootsect == 0)
		      perror(nbuff);
		    else {
		      if (fwrite(old_mbr, 1, install.wholedisk.BytePerSector, fbootsect) != install.wholedisk.BytePerSector) {
			  EXTRA ("Extra: problem writing \"%s\" bootable file containing initial MBR\n", nbuff);
			  if (unlink(nbuff) != 0)
			      perror(nbuff);
			  }
		      if (fclose(fbootsect) != 0) {
			  EXTRA ("Extra: problem closing \"%s\" bootable file containing initial MBR\n", nbuff);
			  perror(nbuff);
			  if (unlink(nbuff) != 0)
			      perror(nbuff);
			  }
		      }
		  }
	      }
#endif
	  if (partition_index)
	      REPRINTF(" --partition_index=%c", *partition_index);
	  }
      REPRINTF("\n");
      }

  if (do_USB_HDD_patch) {
      if (memcmp (file + instboot_info->usbhdd_patch1, "\x89\xdf\xee\xad\x89\xc2\x31\xc9\xec\xa8\x80\xe0\xfb\xe3\x05\x4a\x6e\x42\xe2\xa8\xf9",
			sizeof("\x89\xdf\xee\xad\x89\xc2\x31\xc9\xec\xa8\x80\xe0\xfb\xe3\x05\x4a\x6e\x42\xe2\xa8\xf9") - 1)) {
	  ERROR ("first pattern to patch for --usb_hdd at 0x%X not found!\n", instboot_info->usbhdd_patch1);
	  return 1;
	  }
      if (memcmp (file + instboot_info->usbhdd_patch2, "\x8c\x44\x06", sizeof ("\x8c\x44\x06") - 1)) {
	  ERROR ("second pattern to patch for --usb_hdd at 0x%X not found!\n", instboot_info->usbhdd_patch2);
	  return 1;
	  }
      memcpy (file + instboot_info->usbhdd_patch1, "\x90\x08\xf6\x75\x0e\x2e\x8a\x36\x1a\x00\x08\xed\x75\x03\x80\xe9\x40\xfe\xcd\xfe\xce",
		    sizeof("\x90\x08\xf6\x75\x0e\x2e\x8a\x36\x1a\x00\x08\xed\x75\x03\x80\xe9\x40\xfe\xcd\xfe\xce") - 1);
      memcpy (file + instboot_info->usbhdd_patch2, "\xe8\xd0\xff", sizeof("\xe8\xd0\xff") - 1);
      *(unsigned short *)(file + 0x19A) += 0xAC4A; /* change checksum - may not work if alignment differ */
      /* Checksum Gujin1 does not cover NbHiddensector: */
      ((FAT_bootsect1_t *)file)->NbHiddensector = 0;	/* forced by the BIOS for MSDOS to boot, no need to put another value */
      }
  {
  union fourbyte_u end_name = endname (&install.wholedisk, install.gujin2_load);

  if (   (checkcmd == 0 || !!strcasecmp (checkcmd, "checkonly"))
      && write_output (fout_name, file, filesize, mbr_device != 0,
		instboot_info->deltaseg, full_image, old_mbr, new_mbr,
		install.wholedisk.BytePerSector, confirm, install.gujin2_load, sig0x29touse,
		end_name, &install.fs, &install.misc, xwrite) != 0) {
      ERROR ("writing the output.\n");
      return 1;
      }
  sync();

  if (   (checkcmd == 0 || !!strcasecmp (checkcmd, "writeonly"))
      && write_output (fout_name, file, filesize, mbr_device != 0,
		instboot_info->deltaseg, full_image, old_mbr, new_mbr,
		install.wholedisk.BytePerSector, 0, install.gujin2_load, sig0x29touse,
		end_name, &install.fs, &install.misc, xcheck) != 0) {
      ERROR ("checking what has been written in the output.\n");
      return 1;
      }
  }
//  if (do_USB_HDD_patch) { /* need to restore i.e. unpatch "file" but it is no more used anyway */  }

  /*
   * has worked up to now, now treat the BEER sector:
   */
  if (   !gujin2_loaded_externally (install.gujin2_load)
      && beer_device && *beer_device) {
      if (!strcmp (fout_name, "/dev/null")) {
	      /* would still work on mkdev /tmp/dev/null, string compare: volumtary bug */
	  ERROR ("refuse to write the BEER sectors when output is '/dev/null'.\n");
	  return 0;
	  }
	else if (strncmp (beer_device, fout_name, strlen (beer_device)) != 0) {
	  ERROR ("refuse to write the BEER sectors on %s when output is %s, they should match.\n",
			beer_device, fout_name);
	  return 0;
	  }
	else
	  EXTRA ("Extra: will write and/or check the BEER sectors on '%s'\n", beer_device);
      if (   (checkcmd == 0 || !!strcasecmp (checkcmd, "checkonly"))
	  && write_beer (&install.beersect, beer_device, install.wholedisk.BytePerSector,
			&install.misc,     (install.gujin2_load == gujin2_load_ide_lba)
					|| (install.gujin2_load == gujin2_load_ide_lba48),
			gujin2_loaded_by_bios (install.gujin2_load) ? install.fs.PhysicalDriveNb : 0xFF,
			xwrite) != 0) {
	  REPRINTF("\n");
	  ERROR ("writing the BEER.\n");
	  return 1;
	  }
      sync();
      if (   (checkcmd == 0 || !!strcasecmp (checkcmd, "writeonly"))
	  && write_beer (&install.beersect, beer_device, install.wholedisk.BytePerSector,
			&install.misc,     (install.gujin2_load == gujin2_load_ide_lba)
					|| (install.gujin2_load == gujin2_load_ide_lba48),
			gujin2_loaded_by_bios (install.gujin2_load) ? install.fs.PhysicalDriveNb : 0xFF,
			xcheck) != 0) {
	  REPRINTF("\n");
	  ERROR ("checking what has been written in the BEER.\n");
	  return 1;
	  }
      /* Just a dummy scan: check if someone would have installed a BEER sector
	 when the HPA was active, and warn^Wbug him... */
      check_beer (beer_device, install.misc.output_partition_offset, install.wholedisk.BytePerSector);
      }

  /*
   * Everything worked, the boot file is written and closed, the MBR is saved
   * in the boot file. It is time for this:
   */
  if (   !gujin2_loaded_externally (install.gujin2_load)
      && mbr_device && *mbr_device) {
      if (!strcmp (fout_name, "/dev/null")) {
	      /* would still work on mkdev /tmp/dev/null, string compare: volumtary bug */
	  REPRINTF("\n");
	  ERROR ("refuse to write the MBR when output is '/dev/null'.\n");
	  return 0;
	  }
	else
	  EXTRA ("Extra: will write and/or check the MBR on '%s'\n", mbr_device);
      if (new_mbr == 0) {
	  EXTRA ("Extra: ALGORITHM ERROR\n");
	  return 1;
	  }
      if (   (checkcmd == 0 || !!strcasecmp (checkcmd, "checkonly"))
	  && write_mbr (mbr_device, new_mbr, install.wholedisk.BytePerSector, xwrite, !install.misc.infs_file) != 0) {
	  ERROR ("writing the MBR.\n");
	  return 1;
	  }
      sync();
      if (   (checkcmd == 0 || !!strcasecmp (checkcmd, "writeonly"))
	  && write_mbr (mbr_device, new_mbr, install.wholedisk.BytePerSector, xcheck, !install.misc.infs_file) != 0) {
	  ERROR ("checking what has been written in the MBR.\n");
	  return 1;
	  }
      }
    else
      REPRINTF("\n");
  } /* new_mbr */

  return 0;
  }
