/* ide.h */
#ifndef IDE_H
#define IDE_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * Those are in microsecond, if the processor has the rdtsc instruction -
 * and it is available in this mode (virtual86 not guarantied).
 * If the processor changes frequencies after initial disk probing,
 * then I have a problem - but it has always being a bad thing to
 * change the processor frequency; the other solution is better and
 * more efficient (not a software solution).
 * I am not sure: shall I add "rep nop" in the loop?
 * Take care, max 4.294 second max here else you overflow integer.
 */
/* If the 386/486 has no TSC, I estimate the duration of one loop
   in function _IDE_wait_altstatus():
   Also used if IDE_SIMPLE_1us_per_inb is defined.
   Because that is a old hardware - 1 inb() is 1 microsecond, so: */
#define NOTSC_IDE_LOOP_DURATION	1000 /* nanosecond */

/* Time to finish the previous unknown command started by BIOS? */
/* Also timeout when there is no IDE at this address. */
#define TIMEOUT_LOOP_INIT	500
/* Also timeout when there is no drive on this IDE channel: */
#define TIMEOUT_LOOP_DRV_READY	100
/* Same timeout when there is no ATAPI drive on this IDE channel: */
#define TIMEOUT_LOOP_ATAPI_READY	500
/* Lets say 1 second for SMART execute offline immediate,
   even this command usually does not take more than 200 ms
   It is also where DRQ is waited in between the IDE PACKET command
   and the DRQ to write the packet: 50 us or 3 ms (for CDROMs)
 */
#define TIMEOUT_LOOP_EXEC_CMD	1000
/* Lets say 4 seconds for the CD/DVD-ROM to open/close the door
   and eject/insert the CDROM, some code has been added to
   be able to use a value over 4294, crappy ultra-speed DVD-RAM
   We need to wait longer with newer ultra speed DVD-RAM readers,
   here up to 8 seconds:
 */
#define TIMEOUT_LOOP_EXEC_ATAPICMD 8000
/* If DRQ is not immediately set after command when BUSY clear,
   we wait on this one - so keep it high (ATA1 or ATA 2 drives...) */
/* Ultra speed DVD-RAM are a lot slower than CDROM drives... */
#define TIMEOUT_LOOP_FIRST_SECT 3000	// 1000
/* One sector has been read/writen, time for next?
   200 not enought for old drives */
#define TIMEOUT_LOOP_NEXT_SECT	500
/* To know the result of the command: Longest seems to be to set
   a password or to remove it, with 110 ms on my 60 Gb disk */
#define TIMEOUT_LOOP_FINISH_CMD	400

struct IDE_control_str {
    unsigned char na0    : 1;	/* always write 0 */
    unsigned char notIEN : 1;
    unsigned char SWrst  : 1;
    unsigned char na1    : 1;	/* always write 1 */
    unsigned char na2    : 3;	/* always write 0 */
    unsigned char HOB    : 1;	/* High Order Bits for LBA48 */
    } __attribute__ ((packed));

struct IDE_error_str {
    unsigned char AMNF : 1;	/* gereral error (really obsolete) */
    unsigned char na1  : 1;
    unsigned char ABRT : 1;
    unsigned char na3  : 1;
    unsigned char IDNF : 1;
    unsigned char na5  : 1;
    unsigned char UNC  : 1;     /* UNCorrectable error  (really obsolete) */
    unsigned char BBK  : 1;     /* Bad BlocK (really obsolete) */
    } __attribute__ ((packed));

struct IDE_status_str {
    unsigned char ERR  : 1;	/* Error */
    unsigned char IDX  : 1;	/* Index (really really obsolete) */
    unsigned char CORR : 1;	/* Corrected Data (really obsolete) */
    unsigned char DRQ  : 1;	/* Data request */
    unsigned char DSC  : 1;	/* Device Seek Complete (obsolete) */
    unsigned char DF   : 1;	/* Device fault */
    unsigned char DRDY : 1;	/* Device ready */
    unsigned char BSY  : 1;	/* Busy */
    } __attribute__ ((packed));

struct ATAPI_interrupt_reason_str {
    unsigned char command_data	: 1;	/* C/D */
    unsigned char input_output	: 1;	/* I/O */
    unsigned char REL		: 1;	/* BUS release */
    unsigned char TAG		: 5;	/* TAGged commands */
    } __attribute__ ((packed));

struct IDE_feature_to_cmd_str {
    unsigned char feature;
    unsigned char sector_count;
    union {
	struct {
	    unsigned char sector;
	    unsigned short cylinder;
	    unsigned char drivehead;
	    } __attribute__ ((packed)) chs;
	struct {
	    unsigned lba : 28;
	    unsigned drv : 4;
	    } __attribute__ ((packed)) lba;
	} __attribute__ ((packed)) chs_lba;
    unsigned char command;
    unsigned char pad_to_8_bytes;
    } __attribute__ ((packed));

struct IDE_feature_to_cmd_addr48_str {
    unsigned short feature;
    unsigned short sector_count;
    union {
	struct {
	    unsigned       lba_low32;
	    unsigned short lba_high16;
	    unsigned short lba_unused;
	    } __attribute__ ((packed)) fields;
	unsigned long long lba;
	} __attribute__ ((packed)) lba;
    unsigned char drivehead;
    unsigned char command;
    } __attribute__ ((packed));

#define IDE_BLOCK28_INPUT	((struct IDE_feature_to_cmd_addr48_str *)0)
#define IDE_BLOCK48_INPUT	((struct IDE_feature_to_cmd_str *)0)
#define IDE_BLOCK28_OUTPUT	((struct IDE_feature_to_cmd_addr48_str *)1)
#define IDE_BLOCK48_OUTPUT	((struct IDE_feature_to_cmd_str *)1)

enum IDE_cmd {
    IDE_NOP				= 0x00,
    IDE_CFA_REQUEST_EXTENDED_ERROR	= 0x03,
    IDE_DEVICE_RESET			= 0x08,
	IDE_CALIBRATE_DRIVE		= 0x10, /* obsolete in ATA6 */
    IDE_READ_SECTOR			= 0x20,	/* with retry */
    IDE_READ_SECTOR_EXT			= 0x24,
    IDE_READ_DMA_EXT			= 0x25,
    IDE_READ_DMA_QUEUED_EXT		= 0x26,
    IDE_READ_NATIVE_MAX_ADDRESS_EXT	= 0x27,
    IDE_READ_MULTIPLE_EXT		= 0x29,
    IDE_READ_LOG_EXT			= 0x2F,
    IDE_WRITE_SECTOR			= 0x30,
    IDE_WRITE_SECTOR_EXT		= 0x34,
    IDE_WRITE_DMA_EXT			= 0x35,
    IDE_WRITE_DMA_QUEUED_EXT		= 0x36,
    IDE_SET_MAX_ADDRESS_EXT		= 0x37,
    IDE_CFA_WRITE_SECTORS_WITHOUT_ERASE	= 0x38,
    IDE_WRITE_MULTIPLE_EXT		= 0x39,
    IDE_WRITE_LOG_EXT			= 0x3F,
    IDE_READ_VERIFY_SECTOR		= 0x40,
    IDE_READ_VERIFY_SECTOR_EXT		= 0x42,
    IDE_SEEK				= 0x70,
    IDE_CFA_TRANSLATE_SECTOR		= 0x87,
    IDE_EXECUTE_DEVICE_DIAGNOSTIC	= 0x90,
	IDE_INITIALISE_DRIVE		= 0x91, /* obsolete in ATA6 */
    IDE_DOWNLOAD_MICROCODE		= 0x92,
    IDE_PACKET				= 0xA0,
    IDE_IDENTIFY_PACKET_DEVICE		= 0xA1,
    IDE_SERVICE				= 0xA2,
    IDE_SMART				= 0xB0,
    IDE_DEVICE_CONFIGURATION		= 0xB1,
    IDE_CFA_ERASE_SECTORS		= 0xC0,
    IDE_READ_MULTIPLE			= 0xC4,
    IDE_WRITE_MULTIPLE			= 0xC5,
    IDE_SET_MULTIPLE_MODE		= 0xC6,
    IDE_READ_DMA_QUEUED			= 0xC7,
    IDE_READ_DMA			= 0xC8,
    IDE_WRITE_DMA			= 0xCA,
    IDE_WRITE_DMA_QUEUED		= 0xCC,
    IDE_CFA_WRITE_MULTIPLE_WITHOUT_ERASE= 0xCD,
    IDE_CHECK_MEDIA_CARD_TYPE		= 0xD1,
    IDE_GET_MEDIA_STATUS		= 0xDA,
    IDE_MEDIA_LOCK			= 0xDE,
    IDE_MEDIA_UNLOCK			= 0xDF,
    IDE_STANDBY_IMMEDIATE		= 0xE0,
    IDE_IDLE_IMMEDIATE			= 0xE1,
    IDE_STANDBY				= 0xE2,
    IDE_IDLE				= 0xE3,
    IDE_READ_BUFFER			= 0xE4,
    IDE_CHECK_POWER_MODE		= 0xE5,
    IDE_SLEEP				= 0xE6,
    IDE_FLUSH_CACHE			= 0xE7,
    IDE_WRITE_BUFFER			= 0xE8,
    IDE_FLUSH_CACHE_EXT			= 0xEA,
    IDE_IDENTIFY_DEVICE			= 0xEC,
    IDE_MEDIA_EJECT			= 0xED,
    IDE_SET_FEATURES			= 0xEF,
    IDE_SECURITY_SET_PASSWORD		= 0xF1,
    IDE_SECURITY_UNLOCK			= 0xF2,
    IDE_SECURITY_ERASE_PREPARE		= 0xF3,
    IDE_SECURITY_ERASE_UNIT		= 0xF4,
    IDE_SECURITY_FREEZE_LOCK		= 0xF5,
    IDE_SECURITY_DISABLE_PASSWORD	= 0xF6,
    IDE_READ_NATIVE_MAX_ADDRESS		= 0xF8,
    IDE_SET_MAX_ADDRESS			= 0xF9,
    };

typedef struct {
    unsigned short	revision;	/* 1 */
    /* See disk.h for those: */
    struct config_multiword_dma_s	multiword_dma;
    struct config_ultra_dma_s		ultra_dma;
     unsigned long long max_lba;
    struct config_feature_s		feature;
    unsigned short	reserved[254 - 8 + 1];
    struct {
	unsigned char signature;	/* 0xA5 */
	unsigned char checksum;	/* total = 0 */
	} __attribute__ ((packed)) integrity;
    } __attribute__ ((packed)) ata_config_t;

/*
 * Just a hint:
 */
extern inline unsigned short
_IDE_ISA_guess_altadr (unsigned short ideIOadr)
  {
  /*
   * ide0=0x01F0,0x03F6,14
   * ide1=0x0170,0x0376,15
   * ide2=0x01e8,0x03ee,11
   * ide3=0x0168,0x036e,10
   * ide4=0x01E0,0x03E6
   * ide5=0x0160,0x0366
   */
  return ideIOadr + 0x206;
  }

/*
 * simple registers access (volatile):
 */
extern inline unsigned char
_IDE_get_altstatus (struct diskparam_str *dp)
  {
  return inb (dp->ideIOctrladr);
  }

#if defined (DEBUG) && (DEBUG & DEBUG_DISK)
extern inline unsigned char
_IDE_get_error (struct diskparam_str *dp)
  {
  return inb (dp->ideIOadr + 1);
  }
#endif

/* Same as _IDE_get_altstatus(), but returning bitfields: */
// used for short delay, 400 ns
extern inline struct IDE_status_str
_IDE_get_altstatus_bits (struct diskparam_str *dp)
  {
  struct IDE_status_str returned;

  asm ASMREAD (
"	inb	%%dx,%%al	# _IDE_get_altstatus_bits \n"
	: "=a" (returned)
	: "d" (dp->ideIOctrladr)
	);
  return returned;
  }

/* Same as _IDE_get_error(), but returning bitfields: */
// For DEBUG, and I also force a read after status.ERR == 1
extern inline struct IDE_error_str
_IDE_get_error_bits (struct diskparam_str *dp)
  {
  struct IDE_error_str returned;

  asm volatile (
"	inb	%%dx,%%al	# _IDE_get_error_bits \n"
	: "=a" (returned)
	: "d" (dp->ideIOadr + 1)
	);
  return returned;
  }

extern inline void
_IDE_set_device_control_register (struct diskparam_str *dp, struct IDE_control_str val)
  {
  asm volatile (
"	outb	%%al,%%dx	# _IDE_set_device_control_register \n"
	: /* no output */
	: "a" (val), "d" (dp->ideIOctrladr)
	);
  }

extern inline unsigned char
_IDE_get_sector_count (struct diskparam_str *dp)
  {
  return inb (dp->ideIOadr + 2);
  }

extern inline struct ATAPI_interrupt_reason_str
_ATAPI_get_interrupt_reason_bits (struct diskparam_str *dp)
  {
  struct ATAPI_interrupt_reason_str returned;

  asm volatile (
"	  inb	  %%dx,%%al	  # _IDE_get_error_bits \n"
	: "=a" (returned)
	: "d" (dp->ideIOadr + 2)
	);
  return returned;
  }

extern inline void
_IDE_set_drivehead (struct diskparam_str *dp, unsigned char val)
  {
  outb (dp->ideIOadr + 6, val);
  }

extern inline unsigned char
_IDE_get_drivehead (struct diskparam_str *dp)
  {
  return inb (dp->ideIOadr + 6);
  }

/* Same as _IDE_get_status(), but returning bitfields: */
// used to clear interrupt
extern inline struct IDE_status_str
_IDE_get_status_bits (struct diskparam_str *dp)
  {
  struct IDE_status_str returned;

  asm ASMREAD (
"	inb	%%dx,%%al	# _IDE_get_status_bits \n"
	: "=a" (returned)
	: "d" (dp->ideIOadr + 7)
	);
  return returned;
  }

extern inline unsigned short
_IDE_get_cylinder_low (struct diskparam_str *dp)
  {
  return inb (dp->ideIOadr + 4);
  }

extern inline unsigned short
_IDE_get_cylinder_high (struct diskparam_str *dp)
  {
  return inb (dp->ideIOadr + 5);
  }

extern inline unsigned short
_IDE_get_cylinder (struct diskparam_str *dp)
  {
// No word access: UDMA 100/133 controller are no more 16 bits compatibles,
// just 8 bits compatibles to go slower:
//  return inw (dp->ideIOadr + 4);
  return inb (dp->ideIOadr + 4) + 256 * inb (dp->ideIOadr + 5);
  }

extern inline void
_IDE_start_command (struct diskparam_str *dp, unsigned char val)
  {
  outb (dp->ideIOadr + 7, val);
  }

/*
 * These are working but not used:
 */
#if 0
extern inline unsigned char
_IDE_get_card_addr_reg (struct diskparam_str *dp)
  {
  return inb (dp->ideIOctrladr + 1);
  }

extern inline void
_IDE_set_feature (struct diskparam_str *dp, unsigned char val)
  {
  outb (dp->ideIOadr + 1, val);
  }

extern inline void
_IDE_set_sector_count (struct diskparam_str *dp, unsigned char val)
  {
  outb (dp->ideIOadr + 2, val);
  }

extern inline void
_IDE_set_sector (struct diskparam_str *dp, unsigned char val)
  {
  outb (dp->ideIOadr + 3, val);
  }

extern inline unsigned char
_IDE_get_sector (struct diskparam_str *dp)
  {
  return inb (dp->ideIOadr + 3);
  }

// No word access: UDMA 100/133 controller are no more 16 bits compatibles,
// just 8 bits compatibles...
//extern inline void
//_IDE_set_cylinder (struct diskparam_str *dp, unsigned short val)
//  {
//  outw (dp->ideIOadr + 4, val);
//  }

extern inline unsigned char
_IDE_get_status (struct diskparam_str *dp)
  {
  return inb (dp->ideIOadr + 7);
  }
#endif

/*
 * A bit more complex stuff, one waiting interface:
 * It loops anyway if BSY is set, if BSY is cleared
 * it break the loop anyway if ERR is set, else
 * it does a XOR with xor and a TEST, if the result
 * of the TEST (i.e. AND) is non-zero, it loops.
 * It always returns altstatus read.
 */
extern inline struct IDE_status_str
_IDE_wait_altstatus (struct diskparam_str *dp,
		     struct IDE_status_str testabort, /* BSY SET */
		     struct IDE_status_str xor,
		     struct IDE_status_str test,
		     unsigned *timeout)
  {
  struct IDE_status_str returned;
  struct {
      struct IDE_status_str	xor;
      struct IDE_status_str	test;
      unsigned short		ideIOctrladr;
      } __attribute__ ((packed)) edx = {
      .xor = xor,
      .test = test,
      .ideIOctrladr = dp->ideIOctrladr
      };

  asm volatile (
"	movb	%%al,%%ah	\n"
"	.p2align 2		\n"
"	1:			\n"
"	roll	$16,%%edx	\n"
"	inb	%%dx,%%al	# _IDE_wait_altstatus	\n"
"	roll	$16,%%edx	\n"
"	dec	%%ecx		\n"
"	jz	2f		\n"
"	testb	%%ah,%%al	\n"	/* usually %ah=$0x81 */
"	js	1b		\n"
"	jnz	2f		\n"
"	xorb	%%dl,%%al	\n"
"	testb	%%dh,%%al	\n"
"	jnz	1b		\n"
"	xorb	%%dl,%%al	\n"	/* restore the status read */
"	2:			\n"
	: "+c" (*timeout), "=a" (returned)
	: "d" (edx), "a" (testabort)
	);
  return returned;
  }

/* NOTE: %di should not rollback to zero here... */
extern inline void
//_IDE_get_buffer (struct diskparam_str *dp, farptr buffer, unsigned nbword)
_IDE_get_buffer (struct diskparam_str *dp, farptr *buffer, unsigned nbword)
  {
  asm volatile (
"	cld				\n"
"	pushl	%%es			\n" // insw cannot use %fs prefix
"	pushl	%%edi			\n"
"	popw	%%di			\n"
"	popw	%%es			\n"
"	rep insw %%dx,%%es:(%%di)	# _IDE_get_buffer	\n"
"	popl	%%es			\n"
//	: "+c" (nbword), "+D" (buffer)
	: "+c" (nbword), "+D" (*buffer)
	: "d" (dp->ideIOadr)
	: "memory"	// not sufficient (GCC-3.4)
	);
  }

extern inline void
_IDE_flush_buffer (struct diskparam_str *dp, unsigned nbword)
  {
  unsigned short dummy;
  asm volatile (
"	1:			\n"
"	inw	%%dx,%%ax	# _IDE_flush_buffer	\n"
"	loopl	1b		\n"
	: "+c" (nbword), "=a" (dummy)
	: "d" (dp->ideIOadr)
	);
  }

/* NOTE: %si should not rollback to zero here... */
extern inline void
//_IDE_put_buffer (struct diskparam_str *dp, farptr buffer, unsigned nbword)
_IDE_put_buffer (struct diskparam_str *dp, farptr *buffer, unsigned nbword)
  {
  asm volatile (
"	cld				\n"
"	pushl	%%esi			\n"
"	popw	%%si			\n"
"	popw	%%fs			\n"
"	rep outsw %%fs:(%%si),%%dx	# _IDE_put_buffer	\n"
//	: "+c" (nbword), "+S" (buffer)
	: "+c" (nbword), "+S" (*buffer)
	: "d" (dp->ideIOadr)
	: "memory"	// not sufficient (GCC-3.4)
	);
  }

extern inline void
_IDE_put_packetcmd (struct diskparam_str *dp, void *buffer, unsigned nbchar)
  {
  nbchar /= 2;
  asm volatile (
"	cld				\n"
"	rep outsw %%ds:(%%si),%%dx	# _IDE_put_packetcmd	\n"
	: "+c" (nbchar), "+S" (buffer)
	: "d" (dp->ideIOadr)
	);
  }

extern inline void
_IDE_set_feature_to_cmd (struct diskparam_str *dp,
			 const struct IDE_feature_to_cmd_str *block)
  {
  unsigned short dummy;

  asm volatile (
"	cld						\n"
"	outsb	%%ds:(%%si),%%dx	# feature	\n"
"	inc	%%dx					\n"
"	outsb	%%ds:(%%si),%%dx	# sector_count	\n"
"	inc	%%dx					\n"
"	outsb	%%ds:(%%si),%%dx	# sector	\n"
"	inc	%%dx					\n"
// I hate that: newer IDE controller are not even able to
// handle 16 bits writes - you have to write bytes per bytes,
// and considering the time taken by an outb/w ...
//"	outsw	%%ds:(%%si),%%dx	# cylinder	\n"
//"	add	$2,%%dx					\n"
"	outsb	%%ds:(%%si),%%dx	# cylinder low	\n"
"	inc	%%dx					\n"
"	outsb	%%ds:(%%si),%%dx	# cylinder high	\n"
"	inc	%%dx					\n"
"	outsb	%%ds:(%%si),%%dx	# drivehead	\n"
"	inc	%%dx					\n"
"	outsb	%%ds:(%%si),%%dx	# command	\n"
	: "+S" (block), "=d" (dummy)
	: "d" (dp->ideIOadr + 1), ASM_STRUCT_INPUT (*block)
	);
  }

/* Seems that some IDE refuses reads by 16 bits units (inw),
   even using some _really_ new hardware... */
extern inline unsigned
_IDE_readback_lba28 (struct diskparam_str *dp)
  {
  unsigned short dummy;
  unsigned returned;

  asm (
"	inb	%%dx,%%al		# sector	\n"
"	rorl	$8,%%eax				\n"
"	inc	%%dx					\n"
"	inb	%%dx,%%al		# cylinder_low	\n"
"	rorl	$8,%%eax				\n"
"	inc	%%dx					\n"
"	inb	%%dx,%%al		# cylinder_high	\n"
"	rorl	$8,%%eax				\n"
"	inc	%%dx					\n"
"	inb	%%dx,%%al		# drivehead	\n"
"	roll	$24,%%eax				\n"
	: "=a" (returned), "=d" (dummy)
	: "d" (dp->ideIOadr + 3)
	);
  return returned & 0x0FFFFFFF;
  }

extern inline void
_IDE_set_feature_to_cmd_addr48 (struct diskparam_str *dp,
				const struct IDE_feature_to_cmd_addr48_str *block)
  {
  unsigned short dummy;

  asm volatile (
"	cld							\n"
"	lodsb	%%ds:(%%si),%%al	# feature lsb		\n"
"	outsb	%%ds:(%%si),%%dx	# feature msb		\n"
"	outb	%%al,%%dx		# feature lsb		\n"
"	inc	%%dx						\n"
"	lodsb	%%ds:(%%si),%%al	# sector_count lsb	\n"
"	outsb	%%ds:(%%si),%%dx	# sector_count msb	\n"
"	outb	%%al,%%dx		# sector_count lsb	\n"
"	inc	%%dx						\n"
"	lodsl	%%ds:(%%si),%%eax	# lba 0-32		\n"
"	roll	$8,%%eax					\n"
"	outb	%%al,%%dx		# lba 24-31		\n"
"	inc	%%dx						\n"
"	outsb	%%ds:(%%si),%%dx	# lba 32-39		\n"
"	inc	%%dx						\n"
"	outsb	%%ds:(%%si),%%dx	# lba 40-47		\n"
"	subw	$2,%%dx						\n"
"	shrl	$8,%%eax		# lba 0-23		\n"
"	outb	%%al,%%dx		# lba 0-7		\n"
"	shrl	$8,%%eax					\n"
"	inc	%%dx						\n"
"	outb	%%al,%%dx		# lba 8-15		\n"
"	inc	%%dx						\n"
"	shrl	$8,%%eax					\n"
"	outb	%%al,%%dx		# lba 16-23		\n"
"	inc	%%dx						\n"
"	lodsw	%%ds:(%%si),%%ax	# lba_unused		\n"
"	outsb	%%ds:(%%si),%%dx	# drivehead		\n"
"	inc	%%dx						\n"
"	outsb	%%ds:(%%si),%%dx	# command		\n"
	: "+S" (block), "=d" (dummy)
	: "d" (dp->ideIOadr + 1), ASM_STRUCT_INPUT (*block)
	);
  }

extern inline unsigned long long
_IDE_readback_lba48 (struct diskparam_str *dp)
  {
  unsigned dummy;
  unsigned long long returned;
  struct {
      unsigned short IOctrladr;
      unsigned short IOadr;
      } __attribute__ ((packed)) edx = {
      dp->ideIOctrladr,
      dp->ideIOadr + 3
      };
  struct {
      struct IDE_control_str cl, ch;
      } __attribute__ ((packed)) cx = {
      .cl = { .HOB = 1, .notIEN = 1, .na1 = 1 },
      .ch = { .HOB = 0, .notIEN = 1, .na1 = 1 }
      };

  asm (
"	mov	%%cl,%%al				\n"
"	outb	%%al,%%dx		# HOB set	\n"
"	rorl	$16,%%edx		# IOadr+3	\n"
"	inb	%%dx,%%al		# lba 24-31	\n"
"	rorl	$8,%%eax				\n"
"	inc	%%dx			# +4		\n"
"	inb	%%dx,%%al		# lba 32-39	\n"
"	rorl	$8,%%eax				\n"
"	inc	%%dx			# +5		\n"
"	inb	%%dx,%%al		# lba 40-47	\n"
"	rorl	$8,%%eax				\n"
"	roll	$16,%%edx		# ideIOctrladr	\n"
"	mov	%%ch,%%al				\n"
"	outb	%%al,%%dx		# HOB clear	\n"
"	mov	%%eax,%%ecx				\n"
"	shrl	$16,%%ecx		# lba 32-47	\n"
"	roll	$16,%%edx		# IOadr+5	\n"
"	inb	%%dx,%%al		# lba 16-23	\n"
"	roll	$8,%%eax				\n"
"	dec	%%dx			# +4		\n"
"	inb	%%dx,%%al		# lba 8-15	\n"
"	roll	$8,%%eax				\n"
"	dec	%%dx			# +3		\n"
"	inb	%%dx,%%al		# lba 0-7	\n"
"	movl	%%ecx,%%edx				\n"
	: "=A" (returned), "=c" (dummy)
	: "d" (edx), "c" (cx)
	);
  return returned;
  }

extern inline unsigned short
_IDE_get_sector_count48 (struct diskparam_str *dp)
  {
  unsigned short returned;
  struct {
      unsigned short IOctrladr;
      unsigned short IOadr;
      } __attribute__ ((packed)) edx = {
      dp->ideIOctrladr,
      dp->ideIOadr + 2
      };
  struct {
      struct IDE_control_str cl, ch;
      } __attribute__ ((packed)) cx = {
      .cl = { .HOB = 1, .notIEN = 1, .na1 = 1 },
      .ch = { .HOB = 0, .notIEN = 1, .na1 = 1 }
      };

  asm (
"	mov	%%cl,%%al				\n"
"	outb	%%al,%%dx		# HOB set	\n"
"	rorl	$16,%%edx		# IOadr+2	\n"
"	inb	%%dx,%%al		# msb count	\n"
"	movb	%%al,%%ah				\n"
"	roll	$16,%%edx		# ideIOctrladr	\n"
"	mov	%%ch,%%al				\n"
"	outb	%%al,%%dx		# HOB clear	\n"
"	roll	$16,%%edx		# IOadr+2	\n"
"	inb	%%dx,%%al		# lsb count	\n"
	: "=a" (returned)
	: "d" (edx), "c" (cx)
	);
  return returned;
  }

/*
 * Following is not used, not even tested
 */
extern inline unsigned char
_IDE_reset (unsigned short ideIOadr, unsigned short ideIOctrladr,
		unsigned *timeout)
  {
  unsigned char status;
  unsigned twoadr = ideIOadr + 7;

  twoadr = (twoadr << 16) | ideIOctrladr;
  /*
   * to be called with *timeout = 0xF0000,
   * returns a "timeout" if *timeout = 0
   */
  asm volatile (
"	pushl	%%ecx						\n"
"	outb	%%al,%%dx					\n"
"loop_waitreset:						\n"
"	inb	%%dx,%%al	# wait a little bit		\n"
"	loopl	loop_waitreset					\n"
"	mov	%%ah,%%al					\n"
"	outb	%%al,%%dx					\n"
"	shrl	$16,%%edx					\n"
"	popl	%%ecx						\n"
"loop_waitbusy:							\n"
"	inb	%%dx,%%al					\n"
"	test	$0x80,%%al					\n"
"	loopnzl loop_waitbusy					\n"
	: "+c" (*timeout), "=a" (status), "+d" (twoadr)
	: "a" (0x0A0E)
	);

  return status;
  }
#endif /* IDE_H */
