/* hello_gpl.c : demo of a KGZ application.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef __ELF__
#error Gujin can only load ELF files, and this compiler is not producing ELF files
#endif

/* Constant entry block, max 64 KB code+data+stack, do not touch: */
asm (
"	.code16gcc		\n"
"	.section .init,\"ax\",@progbits	\n"
"	pushw	%ds		\n"
"	pushw	%es		\n"
"	pushw	%fs		\n"
"	pushw	%gs		\n"
"	pushfl			\n"
"	pushal			\n"
"	mov	%ss,%si		\n"
"	shl	$16,%esi	\n"
"	mov	%sp,%si		\n"
"	mov	%ss:56(%si),%si	\n"	// esi 4Kb common buffer
"	mov	%esi,%cs:farptr_realmodeparams	\n"
"	movw	%ss,%cs:1f+1	\n"
"	movw	%sp,%cs:2f+2	\n"
"	movw	%cs,%ax		\n"
"	movw	%ax,%ds		\n"
"	movw	%ax,%es		\n"
"	movw	%ax,%fs		\n"
"	movw	%ax,%gs		\n"
"	movw	%ax,%ss		\n"
"	movl	$0,%esp		\n"
"	cld			\n"
"	calll	main16		\n"
"	mov	%ax,%fs		\n"
"	roll	$16,%eax	\n"
"	mov	%ax,%gs		\n"
"1:	movw	$0,%ax		\n"
"	movw	%ax,%ss		\n"
"2:	movl	$0,%esp		\n"
"	popal			\n"
"	mov	%gs,%ax		\n"
"	roll	$16,%eax	\n"
"	mov	%fs,%ax		\n"
"	popfl			\n"
"	popw	%gs		\n"
"	popw	%fs		\n"
"	popw	%es		\n"
"	popw	%ds		\n"
"	lretw			\n"
"	.previous		\n"
/* some real-mode assembly we need after calling main32(),
   in .text/.data section, to return to Gujin, a kind of longjmp(): */
"realsw = 250 * 4		\n"	// use end of IRQs as temp
"	.text			\n"
"start_exitblock:		\n"
"	xor	%ax,%ax		\n"
"	mov	%ax,%ds		\n"
"	mov	%esi,realsw+0	\n"
"	mov	%edi,realsw+4	\n"
"	mov	%ebp,realsw+8	\n"
"	mov	%ebx,realsw+12	\n"
"	movw	%cs,%ax		\n"
"	movw	%ax,%ss		\n"
"	popl	%esi		\n"
"	popl	%edi		\n"
"	popl	%ebp		\n"
"	popl	%ebx		\n"
"	popw	%gs		\n"
"	popw	%fs		\n"
"	popw	%es		\n"
"	popw	%ds		\n"
"	popw	%dx		\n"
"	popl	%eax		\n"
"	lss	(%esp),%sp	\n"
"	pushw	%dx		\n"
"	pushl	%eax		\n"
"	iretw			\n"
"end_exitblock:			\n"
/* some real-mode assembly to return to real mode: */
"switch:			\n"
"	mov	%cr0,%eax	\n"	// 3 bytes
"	and	$0xfe,%al	\n"	// 3 bytes
"	mov	%eax,%cr0	\n"	// 3 bytes
"switch_off = . + 1		\n"
"switch_seg = . + 3		\n"
"	ljmpw	$0,$0		\n"	// 5 bytes
"	.previous		\n"
// Do not go back to .code32, following functions have to be .code16gcc
//"	.code32			\n"
);

#define INITSECTION	__attribute__ ((section (".init"), noinline))

enum standard_color {
	black,		blue,		green,		cyan,
	red,		magenta,	brown,		lightgray,
	darkgray,	lightblue,	lightgreen,	lightcyan,
	lightred,	lightmagenta,	yellow,		white
	};

/* We should delare that static inline, but inlining and changing
  .code16gcc/.code32 in the same file do not work reliably,
  even with -fno-toplevel-reorder : */
INITSECTION void
VGA_writestring (const char *string, unsigned char row, unsigned char col)
{
	unsigned short page_attribute = 16 * blue + lightgray, strlen_string;
	const char *endstring = string;

	/* We cannot rely on libc to provide strlen() in real mode: */
	while (*endstring)
		endstring++;
	strlen_string = endstring - string;

	asm (
"	xchgl	%4,%%ebp	\n"
"	int	$0x10		\n"
"	xchgl	%4,%%ebp	\n"
	: : "a" (0x1301), "b" (page_attribute),
	"c" (strlen_string), "d" (((unsigned short)row << 8) | col),
	"r" (string)	/* in fact %es:%bp, but "B" (string) doesn't work */
	);
}

INITSECTION void BIOS_wait (unsigned nb_microsecond)
{
	unsigned short status;

	/* Use volatile when there is at least one result, but even if the
		result is not used, the code has to be inserted (here
		status is the result): */
	asm volatile (
"	int	$0x15	"
	: "=a" (status)
	: "a" (0x8600), "d" (nb_microsecond & 0xFFFF), "c" (nb_microsecond >> 16)
	);
}

INITSECTION void pokel(unsigned farptr, unsigned val)
{
	asm (
"	pushl	%1		\n"
"	popw	%w1		\n"
"	popw	%%fs		\n"
"	movl	%0,%%fs:(%w1)	\n"
	: : "ri" (val), "bSD" (farptr));
}

INITSECTION unsigned long long rm_rdtsc (void)
  {
  unsigned long long returned;

  asm volatile (" rdtsc " : "=A" (returned) ); /* real volatile! */
  return returned;
  }

/* data in code section result in: "error: msg_bios causes a section type conflict", for:
static const char msg_bios[] __attribute__ ((section (".init") )) = "Hello BIOS world!\r\n";
  The clean way to handle it is to modify the linker script /usr/lib/ldscripts/elf_i386.x
  and replace "KEEP (*(.init))" by "KEEP (*(.init*))", and declare:
static const char msg_bios[] __attribute__ ((section (".init.1") )) = "Hello BIOS world!\r\n";
  But for this quick single-file demo we define it in assembler and declare it in C: */
extern const char msg_bios[];
extern unsigned farptr_realmodeparams;
asm (
"	.section .init,\"ax\",@progbits			\n"
"msg_bios: .asciz \"Hello BIOS GPL world!\\r\\n\"	\n"
"farptr_realmodeparams: .long 0				\n"
"	.previous					\n"
);

INITSECTION int main16 (void)
{
	/* Write a sample string: */
	VGA_writestring (msg_bios, 1, 0);
	/* We could also display the proposed command line at farptr_realmodeparams + 2 Kbytes... */
	/* measure the number of cycles to wait 3 seconds (needs Pentium or better): */
	unsigned long long cycles = rm_rdtsc ();
	/* 3 seconds before returning to Gujin: */
	BIOS_wait (3 * 1000 * 1000);
	/* send number of cycle as parameter to protected mode software: */
	cycles = rm_rdtsc () - cycles;
	pokel (farptr_realmodeparams + 0, cycles);
	pokel (farptr_realmodeparams + 4, cycles >> 32); /* 4 Kbytes max */
	return 0; /* continue and execute protected mode part if null */
}

/* SEPARATOR: before that comment, all code is in real mode, after that
  comment, all code is in protected mode (using ".code16gcc" / ".code32").
  All code in section .init and will be executed before _start().
  No data can be simply shared in between these two modes, use
  the 4 Kbytes buffer provided in %esi as segment:offset in real mode,
  its linear address is passed in %esi to protected mode.*/

/* Declare the size of the protected mode stack: */
#define STACKSIZE	64 * 1024	// in bytes

/* Some preprocessor magic to put the stack size in a string
   to access it in an asm() string: */
#define string(x)	#x
#define STRING(x)	string(x)
#define STACKSIZESTRING	STRING(STACKSIZE)

/* Define the GDT, lots of lines just to define an array
   of few 64 bits integer and its pointer/length:
   (That GDT is used to re-initialise the protected mode
    segments - because %ss is still not initialised -
    and to reset segments as real mode to return to Gujin.
 */
static const union gdt_u {
	struct {
		unsigned	limit		: 16;
		unsigned	base		: 24;
		unsigned	accessed	: 1;	/* if 1 */
		unsigned	writable	: 1;	/* if 1 */
		unsigned	expansion_direction	: 1;	/* down if 1 */
		unsigned	cste_10		: 2;
		unsigned	dpl		: 2;
		unsigned	present		: 1;
		unsigned	limit_msb	: 4;
		unsigned	available	: 1;
		unsigned	bit64		: 1;
		unsigned	big		: 1;
		unsigned	granularity	: 1;
		unsigned	base_msb	: 8;
		} __attribute__ ((packed)) dataseg;
	struct {
		unsigned	limit		: 16;
		unsigned	base		: 24;
		unsigned	accessed	: 1;
		unsigned	readable	: 1;
		unsigned	conforming	: 1;
		unsigned	cste_11		: 2;
		unsigned	dpl		: 2;
		unsigned	present		: 1;
		unsigned	limit_msb	: 4;
		unsigned	available	: 1;
		unsigned	bit64		: 1;
		unsigned	deflt		: 1;
		unsigned	granularity	: 1;
		unsigned	base_msb	: 8;
		} __attribute__ ((packed)) codeseg;
	} __attribute__ ((packed, aligned(8))) gdt[] = {
	{}, /* segment 0 is invalid */
	{ codeseg: { /* segment 0x08: real-mode code */
		limit : 0xFFFF,
		base : 0x80,	/* as real mode */
		accessed : 1,
		readable : 1,
		conforming : 0, /* ? */
		cste_11 : 3,
		present : 1,
		limit_msb : 0,
		deflt : 0,
		granularity : 0,
		} },
	{ codeseg: { /* segment 0x10 : protected-mode code */
		limit : 0xFFFF,
		accessed : 1,
		readable : 1,
		cste_11 : 3,
		present : 1,
		limit_msb : 0xF,
		deflt : 1,
		granularity : 1
		} },
	{ dataseg: { /* segment 0x18: protected-mode data */
		limit : 0xFFFF,
		accessed : 1,
		writable : 1,
		expansion_direction : 0,
		cste_10 : 2,
		present : 1,
		limit_msb : 0xF,
		big : 0,
		granularity : 1,
		} },
	{ dataseg: { /* segment 0x20: real-mode data */
		limit : 0xFFFF,
		accessed : 1,
		writable : 1,
		expansion_direction : 0,
		cste_10 : 2,
		present : 1,
		limit_msb : 0xF,
		granularity : 1,
		} },
	{ dataseg: { /* segment 0x28: real-mode stack */
		limit : 0xFFFF,
		accessed : 1,
		writable : 1,
		expansion_direction : 0,
		cste_10 : 2,
		present : 1,
		limit_msb : 0,
		granularity : 0,
		} },
	};

static const struct { /* not in stack to have a big alignment */
	unsigned short	limit;
	const union gdt_u *gdt;
	} __attribute__ ((packed, aligned(16))) gdtptr = {
	limit	: sizeof (gdt) - 1,	/* -1 : see Intel doc */
	gdt	: gdt
	};

/* Looks like a complex structure, but we just define a pointer to it: */
struct gpl_compliant_str {
	unsigned signature;
	unsigned version;
	unsigned feature;
	unsigned size;

	struct {
		unsigned esi, edi, ebp, ebx;
		unsigned short gs, fs, es, ds;
		unsigned short flags, ip, cs, sp, ss;
		} __attribute__ ((packed)) jmpbuf;

	/* those are linear address: */
	unsigned filename_array, gdt, regs, fourKbuffer;
	unsigned LOADER, STATE, UI, MOUSE, DI, UTIL, BOOTWAY;

	/* Those are real mode seg:off addresses: */
	unsigned font8x8_farptr, font8x14_farptr, font8x16_farptr,
		font10x20_farptr, font12x24_farptr, reserved_farptr[3];

	struct real_mode_fct { /* Those are for real-mode *.kgz use */
		unsigned reserved[19+6];
		} fct;

	void *current_param; /* some other information, not needed here */
	struct mouse_color_str *mouse_colors;
	void *reserved_ptr[6];
	} *forgpl;

/* Simple protected mode entry code, setting descriptors,
   the stack and saving EBP and ESI register */
asm (
"	.text					\n"
"	.code32					\n"
"	.global _start				\n"
"_start:					\n"
"	lgdt	gdtptr				\n"
"	jmp	$0x10,$1f			\n"
"	1:					\n"
"	mov	$0x18,%eax			\n"
"	mov	%eax,%ds			\n"
"	mov	%eax,%es			\n"
"	mov	%eax,%fs			\n"
"	mov	%eax,%gs			\n"
"	mov	%eax,%ss			\n"
"	mov	stack+" STACKSIZESTRING ",%esp	\n"
"	mov	%ds:0,%al			\n"
"	mov	%es:0,%al			\n"
"	mov	%fs:0,%al			\n"
"	mov	%gs:0,%al			\n"
"	mov	%ss:0,%al			\n"
"	mov	%ebp,forgpl			\n"
"	mov	%esi,realmodeparams		\n"
"	jmp	main32				\n"
"	.previous				\n"
);

void return_to_gujin (void)
{
	/* We need some assembly instructions in real mode, below 1 Mb,
		overwrite some uninteresting fields of forgpl, it has the
		advantage to be in Gujin %ds and %ss segment: */
	extern char start_exitblock[], end_exitblock[];
	char *dst = (char *)&forgpl->fct, *src;

	for (src = start_exitblock; src < end_exitblock; src++)
		*dst++ = *src;

	/* We need to jump to a segment with identical mapping to return
		to real mode, save/use/restore a part of the interrupt table: */
	extern unsigned short switch_off[], switch_seg[];
	*switch_off = (unsigned long)&forgpl->fct & 0xFFFF;
	*switch_seg = ((unsigned long)&forgpl->fct >> 16) << 12;

	/* return to real mode and branch to longjump(&forgpl->jmpbuf) */
	asm volatile (
"	cli				\n"
"	movzwl	%%ax,%%esp		\n"
"	mov	switch+0,%%esi		\n"
"	xchgl	%%esi,realsw+0		\n"
"	mov	switch+4,%%edi		\n"
"	xchgl	%%edi,realsw+4		\n"
"	mov	switch+8,%%ebp		\n"
"	xchgl	%%ebp,realsw+8		\n"
"	mov	switch+12,%%ebx		\n"
"	xchgl	%%ebx,realsw+12		\n"
"	lgdt	%0			\n"
"	mov	$0x20,%%eax		\n"	// dataseg
"	mov	%%eax,%%ds		\n"
"	mov	%%eax,%%es		\n"
"	mov	%%eax,%%fs		\n"
"	mov	%%eax,%%gs		\n"
"	mov	%%ds:0,%%eax		\n"
"	mov	%%es:0,%%eax		\n"
"	mov	%%fs:0,%%eax		\n"
"	mov	%%gs:0,%%eax		\n"
"	mov	$0x28,%%eax		\n"	// dataseg
"	mov	%%eax,%%ss		\n"
"	mov	%%ss:0,%%eax		\n"
"	ljmp	$0x08,$realsw - 0x80	\n"	// codeseg
	: : "m" (gdtptr), "a" (&forgpl->jmpbuf));
}

inline unsigned long long rdtsc (void)
  {
  unsigned long long returned;

  asm volatile (" rdtsc " : "=A" (returned) ); /* real volatile! */
  return returned;
  }

/* In protected mode, constants and variables can be used without restrictions: */
const char msg1[] = "Hello protected-mode text GPL world! ";	// .rodata
char msg2[] = "(ELF32) wait a bit...";	// .data
void *realmodeparams;	/* filled by your real mode function main16() */

/* Declare the stack we are using in protected mode, in BSS to be
	able later to measure its used part (initialised to 0): */
static unsigned stack[STACKSIZE / 4] __attribute__ ((aligned(32)));

void main32 (void)
{
	/* We are flat non-paged memory and interrupt disabled */
	/* Put a marker at the bottom of the stack to detect overflow,
		also use stack[] in C to stop "unused" warning: */
	stack[0] = 0xDEADBEEF;

	unsigned cpt1, cpt2;
	/* We want blue background color and lightgray foreground: */
	const unsigned short color = 16 * blue + lightgray;
	/* This work only in text modes (assumed mode 3), video memory
	at real mode address 0xB800:0000
	At this address, we shall write two bytes per char, one containing
	the foreground and background color, the other containing the letter */
	volatile unsigned short *video_array = (volatile unsigned short *)0xB8000;
	video_array += 2 * 80;	/* 2 lines down */
	for (cpt1 = 0; cpt1 < sizeof(msg1) - 1; cpt1++)
		video_array[cpt1] = (color << 8) + msg1[cpt1];
	for (cpt2 = 0; cpt2 < sizeof(msg2) - 1; cpt2++)
		video_array[cpt1++] = (color << 8) + msg2[cpt2];
	/* Clear few lines: */
	while (cpt1 < 10 * 80)
		video_array[cpt1++] = (color << 8) + ' ';
	if (forgpl == 0) {
#define ERRMSG "bad GZIP comment license, cannot return to Gujin - please reboot"
		for (cpt2 = 0; cpt2 < sizeof(ERRMSG) - 1; cpt2++)
			video_array[cpt1++] = (color << 8) + ERRMSG[cpt2];
		for (;;)
			continue;
		}

	/* display parameter address and first 64 bits parameter: */
	cpt1 -= 80 * 2; /* previous line */
	for (cpt2 = 0; cpt2 < sizeof("parameters at 0x") - 1; cpt2++)
		video_array[cpt1++] = (color << 8) + "parameters at 0x"[cpt2];
	for (cpt2 = 0; cpt2 < 8; cpt2++)
		video_array[cpt1++] = 0x1700 + "0123456789ABCDEF"[((unsigned long)realmodeparams >> (28 - 4*cpt2)) & 0x0F];
	video_array[cpt1++] = 0x1700 + ':';
	video_array[cpt1++] = 0x1700 + ' ';
	video_array[cpt1++] = 0x1700 + '0';
	video_array[cpt1++] = 0x1700 + 'x';

	/* The only parameter set by real-mode main16() is a 64 bits value: */
	unsigned long long cycle = *(unsigned long long *)realmodeparams; /* max 4 Kbytes */
	for (cpt2 = 0; cpt2 < 16; cpt2++)
		video_array[cpt1++] = 0x1700 + "0123456789ABCDEF"[(cycle >> (60 - 4*cpt2)) & 0x0F];

	/* Wait the same amount of time as in real mode, to see the message: */
	cycle += rdtsc();
	while (rdtsc() < cycle)
		continue;

	/* If you have modified MMU/PIC/interrupts, you need to
		restore them before calling: */
	return_to_gujin();
}
