/* gzlib.c */

/*
  The original source is the zlib library - but it has been heavily
  modified to save code space (begin with gcc -E), bugs are mine.

 *********  My changes are covered by the GPL license  *********
 ***************************************************************
 * Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 ***************************************************************

  zlib.h -- interface of the 'zlib' general purpose compression library
  version 1.1.3, July 9th, 1998

  Copyright (C) 1995-1998 Jean-loup Gailly and Mark Adler

  This software is provided 'as-is', without any express or implied
  warranty.  In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

  1. The origin of this software must not be misrepresented; you must not
     claim that you wrote the original software. If you use this software
     in a product, an acknowledgment in the product documentation would be
     appreciated but is not required.
  2. Altered source versions must be plainly marked as such, and must not be
     misrepresented as being the original software.
  3. This notice may not be removed or altered from any source distribution.

  Jean-loup Gailly        Mark Adler
  jloup@gzip.org          madler@alumni.caltech.edu


  The data format used by the zlib library is described by RFCs (Request for
  Comments) 1950 to 1952 in the files ftp://ds.internic.net/rfc/rfc1950.txt
  (zlib format), rfc1951.txt (deflate format) and rfc1952.txt (gzip format).
*/

#include "make.h"
#if SETUP & XCODE_SEGMENT		/* before other includes */
#define CALLING_FROM_XCODESEG
#endif

#include "instboot.h"
#include "library.h"
#include "debug.h"
#include "boot.h"	/* BOOT1_DOS_running(), struct registers {} */
#include "gzlib.h"	/* z_stream for vmlinuz_header_treat() */
#ifndef GZLIB_ALSO_TEST
#include "bios.h"	/* struct e820map_info for vmlinuz.h */
#else
struct e820map_info {
    long long addr;	/* start of memory segment */
    long long size;	/* size of memory segment */
    long type;		/* type of memory segment */
    } __attribute__ ((packed));
#endif
#include "user.h"	/* UI */
#include "vesabios.h"   /* struct vesa_mode_attribute_str */
#include "vmlinuz.h"	/* GZLIB_TREAT_{NAME,COMMENT} and LOADER.curfileload->KBalloced */
#include "messages.h"

static const char inflate_copyright[] __attribute__ ((unused)) =
	" inflate 1.1.3 Copyright 1995-1998 Mark Adler,"
	" modified by Etienne Lorrain, GPL ONLY ";

#ifdef GZLIB_USE_INTERMEDIATE_WINDOW

#ifdef GZLIB_TEST_MEASURE
enum crc32state_e crc32state;
static unsigned crc32list[2][16 * 32];	/* 32 * INTERMEDIATE_WINDOW size = 16 * 64 K = 1 Mbyte */
static unsigned crc32lastsize[2];
#endif /* GZLIB_TEST_MEASURE */

#ifndef GZLIB_BIG_MALLOC
/*
 * For the TINY configuration, we have enought space.
 * The stack is limited to 32Kb to keep check_stack()
 * working (signed bound limits), so we have:
 */
GZIP_FCT_PREFIX(memcpy_data2window) static inline void
memcpy_data2window (unsigned char *dest,
		    const unsigned char *src,
		    unsigned short size)
  {
  _memcpy (dest, src, size);
  }

GZIP_FCT_PREFIX(memcpy_window2window) static inline void
memcpy_window2window (unsigned char *dest,
		      const unsigned char *src,
		      unsigned short size)
  {
  _memcpy (dest, src, size);
  }

#define OUTBYTE(q, a)	{*q=(unsigned char)(a);}
#define CRCBYTE(q)	(*q++)

unsigned char       gzlib_window[1U<<MAX_WBITS];
#define ZWINALLOC(w)	gzlib_window /* malloc (w) */
#define ZWINFREE(w)	w = 0 /* free (w) */
#define ZWINFARPTR(win)	data_adr(win)

#else /* GZLIB_BIG_MALLOC */

/*
 * Argggg... gzlib needs a 32 Kbytes buffer to decompress,
 * (and even more to compress) and if BIG_MALLOC is not
 * defined we will not have enough malloc memory to load
 * all that e2fs requires and that gzlib buffer... We just
 * have approx 45..50 Kbytes free without BIG_MALLOC.
 *
 * Requesting BIG_MALLOC means we cannot run as a real
 * DOS/other-OS bootloader.
 *
 * Fortunately, because (now) gzlib is only accessing that
 * big buffer using macro OUTBYTE() or with zmemcpy() -
 * we can provide it a "farptr" and new OUTBYTE()/zmemcpy()
 * which handle such a kind of pointer. Whatever calculus
 * gzlib does (inside a 64K segment) on the address, it will
 * behave the same with a farptr. If we ever forget one
 * read or write to this big buffer (content of a farptr),
 * we will get a "General Protection Exception": easy to debug!
 */
GZIP_FCT_PREFIX(memcpy_data2window) static inline void
memcpy_data2window (unsigned char *dest,
		    const unsigned char *src,
		    unsigned short size)
  {
  smemcpy ((farptr)dest, src, size);
  }

GZIP_FCT_PREFIX(memcpy_window2window) static inline void
memcpy_window2window (unsigned char *dest,
		      const unsigned char *src,
		      unsigned short size)
  {
  asm volatile (
"	pushw	%%es						\n"
"	popw	%%fs						\n"
"	pushl	%%edi						\n"
"	popw	%%di						\n"
"	popw	%%es						\n"
"	cld							\n"
"	rep movsb %%es:(%%si),%%es:(%%di)	# no macro	\n"
"	nop							\n"
"	pushw	%%fs						\n"
"	popw	%%es						\n"
	: "+S" (src), "+D" (dest), "+c" (size)
	: : "memory", "cc"
	);
  }

/* Here we _assume_ that "q" is a farptr ! */
#define OUTBYTE(q, a)	pokeb(((farptr)q), a)
/* GCC-3.4: warning: use of cast expressions as lvalues is deprecated
#define CRCBYTE(q)	peekb(((farptr)q)++)
   q is a (char *)
 */
#define CRCBYTE(q)      peekb(((farptr)q++))

/*
 * The 32 Kb window malloc returns the address just following the top
 * of the stack, exactly %ss + 64 Kbytes because %sp has been cleared
 * at startup.
 * There is no re-entrancy check or malloc/free order check, take care!
 */
#define ZWINALLOC(w)	((void *)((0x1000U + getss()) << 16))
#define ZWINFREE(w)	w = 0
#define ZWINFARPTR(win)	((unsigned)win)

#endif /* GZLIB_BIG_MALLOC */
/*
 * If we were using an intermediate window when ASSEMBLY_TYPE == ASSEMBLY_DSES
 * we would have:
 * #if ASSEMBLY_TYPE != ASSEMBLY_DSES
 *  ...
 * #else
 *
 * GZIP_FCT_PREFIX(memcpy_data2window) static inline void
 * memcpy_data2window (unsigned char *dest,
 *	    const unsigned char *src,
 *	    unsigned short size)
 *   {
 *   memcpy_door_open (dest, src, size);
 *   }
 *
 * GZIP_FCT_PREFIX(memcpy_window2window) static inline void
 * memcpy_window2window (unsigned char *dest,
 *	      const unsigned char *src,
 *	      unsigned short size)
 *   {
 *   memcpy_door_open (dest, src, size);
 *   }
 *
 * #define OUTBYTE(q, a)   {*q=(unsigned char)(a);}
 * #define CRCBYTE(q)      (*q++)
 *
 * #define ZWINALLOC(w)    ((void *)0x10000U)
 * #define ZWINFREE(w)     w = 0
 * #define ZWINFARPTR(win) ((unsigned)win)
 *
 * GZIP_FCT_PREFIX(treat_data) static inline unsigned short
 * treat_data (void *out, void *buffer, unsigned len)
 *   {
 *   memcpy_door_open (out, buffer, len);
 *   return 0;
 *   }
 * #endif
 */

#ifndef treat_data
/*
 * The treat_data() function which consumes the decompressed data:
 */
#include "xms.h"
#include "bios.h"	/* CDROMbootspec */
#include "util.h"	/* UTIL.HIMEM_entrypoint */

/* Those two memory copier functions assume size != 0 */
GZIP_FCT_PREFIX(XMScopy) static inline unsigned
XMScopy (unsigned *KBalloced,
	 unsigned short *handle,
	 unsigned offset,
	 farptr   windowfarptr,
	 unsigned size)
  {
  unsigned char error;
  XMS_EMM emm_struct = {
      .size       = size + (size & 1),	/* even number of bytes */
      .src_handle = 0,		/* real mode address */
      .src_offset = windowfarptr,
      .dst_handle = *handle,
      .dst_offset = offset
      };

//  asm ("" : : : "memory");	// GCC-3.4.5 && DOSEMU ?
  if (1024U * *KBalloced < emm_struct.dst_offset + emm_struct.size) {
      /* The initrd will be relocated at end of memory, no more any trick needed */
      if (*KBalloced == 0) {
	  *KBalloced = 256;
	  ZDBG (("_XMS_alloc_memory(%u Kb) ", *KBalloced));
	  *handle = _XMS_allocate_memory (UTIL.HIMEM_entrypoint, *KBalloced, &error, UTIL.XMS_version);
	  if (error != 0 || *handle == 0) {
	      *handle = 0; /* see the cleaning by upper layers */
	      ZDBG (("error 0x%X!\r\n ", error));
//printf ("_XMS_alloc_memory error: %d\r\n", error);
	      return 1;
	      }
	  emm_struct.dst_handle = *handle;
	  ZDBG (("OK, using handle %u\r\n", *handle));
	  }
	else {
	  *KBalloced += 256; /* 256 Kbytes increment */
	  ZDBG (("_XMS_realloc_memory(%u Kb) ", *KBalloced));
	  if (_XMS_realloc_memory (UTIL.HIMEM_entrypoint, *handle, *KBalloced, &error, UTIL.XMS_version)) {
	      ZDBG (("error 0x%X, will need to free handle %u!\r\n", error, *handle));
//printf ("_XMS_realloc_memory error: %d\r\n", error);
#if 0 /* not needed, returning an error will ultimately free the memory block */
	      if (_XMS_free_memory(UTIL.HIMEM_entrypoint, *handle, &error))
		  ZDBG ((" [FAILED free handle 0x%X: 0x%X] ",
				*handle, error));
		else
		  *handle = 0;
#endif
	      return 2;
	      }
	  ZDBG (("OK\r\n"));
	  }
      }

//printf ("_XMS_move_memory %u bytes: ", emm_struct.size);
  if (_XMS_move_memory (UTIL.HIMEM_entrypoint, &emm_struct, &error) != 0) {
//printf ("error: %d\r\n", error);
      ZDBG (("_XMS_move_memory error 0x%X!\r\n", error));
      return 3;
      }
//puts ("OK");

  return 0;
  }

#if SETUP & USE_INT1587
#include "bios.h"
#include "util.h"	/* segment_door () */
#endif /* USE_INT1587 */

GZIP_FCT_PREFIX(treat_data) static inline unsigned short
treat_data (void *out, void *buffer, unsigned len)
  {
  if ((unsigned)out == LOADER.fileload[0].load_address && LOADER.uncompressed_signature == 0)
      LOADER.uncompressed_signature = peekl (ZWINFARPTR(buffer));

  if (BOOT1_DOS_running() && UTIL.HIMEM_entrypoint) {
//    printf ("XMScopy(%u) 0x%X->(handle 0x%X) 0x%X ", len, ZWINFARPTR(buffer), LOADER.curfileload->handle, (unsigned)out);
      return XMScopy (&LOADER.curfileload->KBalloced,
		      &LOADER.curfileload->handle,
		      (unsigned)out,
		      ZWINFARPTR(buffer),
		      len);
      }
    else {
#if SETUP & USE_INT1587
#if USER_SUPPORT & VESA_SUPPORT
      if (UI.parameter.winsize != VESA2_MARKER)
#endif
	{ /* else _BIOS_copy_extended_memory() may close segments doors */
	  unsigned short returned;

//	printf ("_BIOS_copy_extended_memory(%u) 0x%X->0x%X ", len, ZWINFARPTR(buffer), (unsigned)out); _BIOS_wait (3 * 1000 * 1000);
//		_BIOS_copy_extended_memory(6928) 0x4B1A0000->0x100000 failed error 0x301
	  returned = _BIOS_copy_extended_memory ((unsigned)out, farptr2linear(ZWINFARPTR(buffer)), (len+1)/2);
	  if (returned == 0) {
//	puts ("OK"); _BIOS_wait (3 * 1000 * 1000);
	      return 0;
	      }
	  ZDBG ((" [_BIOS_copy_extended_memory returns 0x%X] ", returned));
//	printf ("failed error 0x%X \r\n", returned); _BIOS_wait (3 * 1000 * 1000); /* DOSEMU: Failed error 0x301: address line 20 gating failed */
	  if ((unsigned)out == LOADER.curfileload->load_address) {
	      ZDBG ((" [trying segment_door] "));
	      if (segment_door (1) != 0) { /* DOSEMU booting Gujin floppy loading cleandisk.kgz does not like this lgdt... */
		  ZDBG ((" [cannot open doors, aborting] "));
//	printf ("failed segment_door() ");
		  return 0xF1;
		  }
		else
		  ZDBG ((" [doors opened] "));
	      }
	}
#endif /* USE_INT1587 */

//    printf ("memcpy_door_open(%u) 0x%X->0x%X(0x%X) ", len, ZWINFARPTR(buffer), (unsigned)out, linear2dsrelative((unsigned)out));
      memcpy_door_open ((void *)linear2dsrelative((unsigned)out), (void *)farptr2dsrelative(ZWINFARPTR(buffer)), len);
      return 0;
      }
  }
#endif /* treat_data */

#else /* GZLIB_USE_INTERMEDIATE_WINDOW */
#define CRCBYTE(q)	(*q++)
#endif /* GZLIB_USE_INTERMEDIATE_WINDOW */

/*
 * This is derived from the crc32() function in zlib, but to save
 * space I removed the precompiled array - so it is slower.
 * See zlib documentation for copyright of its own functions;
 * the following rewritten functions (added work) are GPL ONLY.
 * -----------------------------------------------------------
 */
#ifdef GZLIB_USE_COMPILED_CRC_ARRAY

GZIP_FCT_PREFIX(crc32) static unsigned
crc32 (unsigned crc, const unsigned char *buf, unsigned len)
  {
  static unsigned crc_table[256];

  if (buf == 0) {
      unsigned short n;
      unsigned poly = 0xedb88320L;	/* polynomial exclusive-or pattern */

      for (n = 0; n < 256; n++) {
	  unsigned char k;
	  unsigned c = (unsigned) n;
	  for (k = 0; k < 8; k++)
	      c = c & 1 ? poly ^ (c >> 1) : c >> 1;
	  crc_table[n] = c;
	  }
      return 0L;
      }

  crc = crc ^ 0xffffffffL;
  if (len != 0) {
      unsigned nb = ((len - 1) / 8) + 1;
      unsigned char databyte = CRCBYTE(buf);

      switch (len % 8) { do {
	  case 0: crc = crc_table[(crc ^ databyte) & 0xff] ^ (crc >> 8);
	  case 7: crc = crc_table[(crc ^ databyte) & 0xff] ^ (crc >> 8);
	  case 6: crc = crc_table[(crc ^ databyte) & 0xff] ^ (crc >> 8);
	  case 5: crc = crc_table[(crc ^ databyte) & 0xff] ^ (crc >> 8);
	  case 4: crc = crc_table[(crc ^ databyte) & 0xff] ^ (crc >> 8);
	  case 3: crc = crc_table[(crc ^ databyte) & 0xff] ^ (crc >> 8);
	  case 2: crc = crc_table[(crc ^ databyte) & 0xff] ^ (crc >> 8);
	  case 1: crc = crc_table[(crc ^ databyte) & 0xff] ^ (crc >> 8);
	  } while (--nb); }
      }
  return crc ^ 0xffffffffL;
  }

#elif defined (GZLIB_USE_PRECOMPILED_CRC_ARRAY)

#if 0
//#include "stdio.h"
int main (void)
  {
  static unsigned crc_table[256];
  unsigned short n;
  unsigned poly = 0xedb88320L;  /* polynomial exclusive-or pattern */

  for (n = 0; n < 256; n++) {
      unsigned char k;
      unsigned c = (unsigned) n;
      for (k = 0; k < 8; k++)
	  c = c & 1 ? poly ^ (c >> 1) : c >> 1;
      crc_table[n] = c;
      }
  printf ("static const unsigned crc_table[256] = {\n");
  for (n = 0; n < 256/8; n++)
      printf ("\t0x%8.8X, 0x%8.8X, 0x%8.8X, 0x%8.8X, 0x%8.8X, 0x%8.8X, 0x%8.8X, 0x%8.8X,\n",
	crc_table[8*n+0], crc_table[8*n+1], crc_table[8*n+2], crc_table[8*n+3],
	crc_table[8*n+4], crc_table[8*n+5], crc_table[8*n+6], crc_table[8*n+7]);
  printf ("\t};\n");
  return 0;
  }
#endif

GZIP_FCT_PREFIX(crc32) static unsigned
crc32 (unsigned crc, const unsigned char *buf, unsigned len)
  {
  const unsigned char *end = buf + len;

#if (SETUP & XDATA_SEGMENT)
  __attribute__ ((section (".xdata")))
#endif
static const unsigned crc_table[256] = {
	0x00000000, 0x77073096, 0xEE0E612C, 0x990951BA, 0x076DC419, 0x706AF48F, 0xE963A535, 0x9E6495A3,
	0x0EDB8832, 0x79DCB8A4, 0xE0D5E91E, 0x97D2D988, 0x09B64C2B, 0x7EB17CBD, 0xE7B82D07, 0x90BF1D91,
	0x1DB71064, 0x6AB020F2, 0xF3B97148, 0x84BE41DE, 0x1ADAD47D, 0x6DDDE4EB, 0xF4D4B551, 0x83D385C7,
	0x136C9856, 0x646BA8C0, 0xFD62F97A, 0x8A65C9EC, 0x14015C4F, 0x63066CD9, 0xFA0F3D63, 0x8D080DF5,
	0x3B6E20C8, 0x4C69105E, 0xD56041E4, 0xA2677172, 0x3C03E4D1, 0x4B04D447, 0xD20D85FD, 0xA50AB56B,
	0x35B5A8FA, 0x42B2986C, 0xDBBBC9D6, 0xACBCF940, 0x32D86CE3, 0x45DF5C75, 0xDCD60DCF, 0xABD13D59,
	0x26D930AC, 0x51DE003A, 0xC8D75180, 0xBFD06116, 0x21B4F4B5, 0x56B3C423, 0xCFBA9599, 0xB8BDA50F,
	0x2802B89E, 0x5F058808, 0xC60CD9B2, 0xB10BE924, 0x2F6F7C87, 0x58684C11, 0xC1611DAB, 0xB6662D3D,
	0x76DC4190, 0x01DB7106, 0x98D220BC, 0xEFD5102A, 0x71B18589, 0x06B6B51F, 0x9FBFE4A5, 0xE8B8D433,
	0x7807C9A2, 0x0F00F934, 0x9609A88E, 0xE10E9818, 0x7F6A0DBB, 0x086D3D2D, 0x91646C97, 0xE6635C01,
	0x6B6B51F4, 0x1C6C6162, 0x856530D8, 0xF262004E, 0x6C0695ED, 0x1B01A57B, 0x8208F4C1, 0xF50FC457,
	0x65B0D9C6, 0x12B7E950, 0x8BBEB8EA, 0xFCB9887C, 0x62DD1DDF, 0x15DA2D49, 0x8CD37CF3, 0xFBD44C65,
	0x4DB26158, 0x3AB551CE, 0xA3BC0074, 0xD4BB30E2, 0x4ADFA541, 0x3DD895D7, 0xA4D1C46D, 0xD3D6F4FB,
	0x4369E96A, 0x346ED9FC, 0xAD678846, 0xDA60B8D0, 0x44042D73, 0x33031DE5, 0xAA0A4C5F, 0xDD0D7CC9,
	0x5005713C, 0x270241AA, 0xBE0B1010, 0xC90C2086, 0x5768B525, 0x206F85B3, 0xB966D409, 0xCE61E49F,
	0x5EDEF90E, 0x29D9C998, 0xB0D09822, 0xC7D7A8B4, 0x59B33D17, 0x2EB40D81, 0xB7BD5C3B, 0xC0BA6CAD,
	0xEDB88320, 0x9ABFB3B6, 0x03B6E20C, 0x74B1D29A, 0xEAD54739, 0x9DD277AF, 0x04DB2615, 0x73DC1683,
	0xE3630B12, 0x94643B84, 0x0D6D6A3E, 0x7A6A5AA8, 0xE40ECF0B, 0x9309FF9D, 0x0A00AE27, 0x7D079EB1,
	0xF00F9344, 0x8708A3D2, 0x1E01F268, 0x6906C2FE, 0xF762575D, 0x806567CB, 0x196C3671, 0x6E6B06E7,
	0xFED41B76, 0x89D32BE0, 0x10DA7A5A, 0x67DD4ACC, 0xF9B9DF6F, 0x8EBEEFF9, 0x17B7BE43, 0x60B08ED5,
	0xD6D6A3E8, 0xA1D1937E, 0x38D8C2C4, 0x4FDFF252, 0xD1BB67F1, 0xA6BC5767, 0x3FB506DD, 0x48B2364B,
	0xD80D2BDA, 0xAF0A1B4C, 0x36034AF6, 0x41047A60, 0xDF60EFC3, 0xA867DF55, 0x316E8EEF, 0x4669BE79,
	0xCB61B38C, 0xBC66831A, 0x256FD2A0, 0x5268E236, 0xCC0C7795, 0xBB0B4703, 0x220216B9, 0x5505262F,
	0xC5BA3BBE, 0xB2BD0B28, 0x2BB45A92, 0x5CB36A04, 0xC2D7FFA7, 0xB5D0CF31, 0x2CD99E8B, 0x5BDEAE1D,
	0x9B64C2B0, 0xEC63F226, 0x756AA39C, 0x026D930A, 0x9C0906A9, 0xEB0E363F, 0x72076785, 0x05005713,
	0x95BF4A82, 0xE2B87A14, 0x7BB12BAE, 0x0CB61B38, 0x92D28E9B, 0xE5D5BE0D, 0x7CDCEFB7, 0x0BDBDF21,
	0x86D3D2D4, 0xF1D4E242, 0x68DDB3F8, 0x1FDA836E, 0x81BE16CD, 0xF6B9265B, 0x6FB077E1, 0x18B74777,
	0x88085AE6, 0xFF0F6A70, 0x66063BCA, 0x11010B5C, 0x8F659EFF, 0xF862AE69, 0x616BFFD3, 0x166CCF45,
	0xA00AE278, 0xD70DD2EE, 0x4E048354, 0x3903B3C2, 0xA7672661, 0xD06016F7, 0x4969474D, 0x3E6E77DB,
	0xAED16A4A, 0xD9D65ADC, 0x40DF0B66, 0x37D83BF0, 0xA9BCAE53, 0xDEBB9EC5, 0x47B2CF7F, 0x30B5FFE9,
	0xBDBDF21C, 0xCABAC28A, 0x53B39330, 0x24B4A3A6, 0xBAD03605, 0xCDD70693, 0x54DE5729, 0x23D967BF,
	0xB3667A2E, 0xC4614AB8, 0x5D681B02, 0x2A6F2B94, 0xB40BBE37, 0xC30C8EA1, 0x5A05DF1B, 0x2D02EF8D,
	};

  if (buf == 0)
	return 0L;

  crc = ~crc;
  while (buf < end) {
      unsigned c = (crc ^ CRCBYTE(buf)) & 0xff;
#if (SETUP & XDATA_SEGMENT)
      c = peekl (xdata_adr(crc_table) + c * sizeof(unsigned));
#else
      c = crc_table[c];
#endif
      crc = c ^ (crc >> 8);
      }
  return ~crc;
  }

#else /* !GZLIB_USE_COMPILED_CRC_ARRAY and !GZLIB_USE_PRECOMPILED_CRC_ARRAY */

GZIP_FCT_PREFIX(crc32) static unsigned
crc32 (unsigned crc, const unsigned char *buf, unsigned len)
  {
  const unsigned char *end = buf + len;

  if (buf == 0)
	return 0L;

  crc = ~crc;
  while (buf < end) {
      unsigned c = (crc ^ CRCBYTE(buf)) & 0xff;
      unsigned short k;
#if ASSEMBLY_TYPE == ASSEMBLY_NONE
#define POLY	0xedb88320L /* polynomial exclusive-or pattern */
      for (k = 8; k != 0; k--)
	  c = c & 1 ? (c >> 1) ^ POLY : c >> 1;
#elif 0	// Optimised in assembler - 1st step
      asm (
"	movw $8,%%cx		\n"
"	.align 4		\n"
"	1:			\n"
"	clc			\n"
"	rcrl %0			\n"
"	jnc 2f			\n"
"	xorl $0xedb88320,%0	\n"
"	2:			\n"
"	loopw 1b		\n"
	: "+r" (c), "=c" (k));
#else	// I think I cannot do faster without precompiled array
      unsigned short tmp;
      asm (
"	movw $8,%w2		\n"
"	.align 4		\n"
"	1:			\n"
"	bsfw %w0,%%cx		\n"
"	jz 2f			\n"
"	incw %%cx		\n"
"	subw %%cx,%w2		\n"
"	jc 3f			\n"
"	shrl %%cl,%0		\n"
"	xorl $0xedb88320,%0	\n"
"	jmp 1b			\n"
"	2:			\n"
"	xorw %%cx,%%cx		\n"
"	3:			\n"
"	addw %w2,%%cx		\n"
"	shrl %%cl,%0		\n"
	: "+r" (c), "=c" (k), "=r" (tmp));
#endif
      crc = c ^ (crc >> 8);
      }
  return ~crc;
  }
#endif /* GZLIB_USE_COMPILED_CRC_ARRAY */

#ifndef GZLIB_ALSO_TEST
GZIP_FCT_PREFIX(crc32check)
unsigned crc32check (unsigned inhibit)
  {
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
  unsigned filecpt = 0;
  void *window_buffer32K = ZWINALLOC(1U<<MAX_WBITS);

  ZDBG (("%s: 32K window at 0x%X, farptr 0x%X, stackseg = 0x%X\r\n",
		__FUNCTION__, window_buffer32K, ZWINFARPTR(window_buffer32K), getss()));
  while (filecpt < 2 && LOADER.fileload[filecpt].uncompressed_size != 0
	 // not if the file was not compressed, i.e. compressed_size is size rounded up:
	 && LOADER.fileload[filecpt].uncompressed_size > LOADER.fileload[filecpt].compressed_size) {
#if DEBUG & DEBUG_LOADER
#define CRC_TIMMING
#endif
#ifdef CRC_TIMMING
      unsigned long long tmp_rdtsc;
      unsigned memcpy_total = 0, crc32_total = 0;
#define RDTSC()	(UTIL.processor.calibrate_rdtsc? rdtsc() : 0)	/* Take care: no rdtsc => should not divide by 0 */
#endif
      unsigned crc32val = crc32 (0, 0, 0), filesize, loadaddress = 0;
      LOADER.curfileload = &LOADER.fileload[filecpt];

      if (inhibit & 1) {
	  ZDBG (("CRC32 recheck inhibited.\r\n"));
	  filecpt++;
	  continue;
	  }
      inhibit >>= 1;

      filesize = LOADER.curfileload->uncompressed_size;
      ZDBG (("checking %s crc32 %u bytes at 0x%X: ", filecpt? "initrd" : "kernel", filesize, LOADER.curfileload->load_address));
      PRINT (MSG_CHECKING_CRC32);
      while (filesize > 0) {
	  unsigned sizeused = (filesize > 0x8000)? 0x8000 : filesize;

#ifdef CRC_TIMMING
	  tmp_rdtsc = RDTSC();
#endif
	  if (BOOT1_DOS_running() && UTIL.HIMEM_entrypoint) {
	      unsigned char error;
	      XMS_EMM emm_struct = {
		  .size       = sizeused + (sizeused & 1),  /* even number of bytes */
		  .src_handle = LOADER.curfileload->handle,
		  .src_offset = loadaddress,
		  .dst_handle = 0,          /* real mode address */
		  .dst_offset = ZWINFARPTR(window_buffer32K),
		  };
	      /* When HIMEM is used, _BIOS_copy_extended_memory does not work (PC hard lock) */
	      if (_XMS_move_memory (UTIL.HIMEM_entrypoint, &emm_struct, &error) != 0) {
		  ZDBG (("_XMS_move_memory(%u words from handle %u offset 0x%X to 0x%X) error!\r\n",
			(sizeused+1)/2, LOADER.curfileload->handle, loadaddress, ZWINFARPTR(window_buffer32K)));
		  return 8 + filecpt;
		  }
	      }
	    else {
	      if (
#if USER_SUPPORT & VESA_SUPPORT
		  UI.parameter.winsize == VESA2_MARKER ||
#endif
		  _BIOS_copy_extended_memory (farptr2linear(ZWINFARPTR(window_buffer32K)), LOADER.curfileload->load_address + loadaddress, (sizeused+1)/2) != 0) {
#if USER_SUPPORT & VESA_SUPPORT
		  if (UI.parameter.winsize != VESA2_MARKER)
#endif
		      ZDBG (("_BIOS_copy_extended_memory(%u words from 0x%X to 0x%X) error!\r\n",
				(sizeused+1)/2, LOADER.curfileload->load_address + loadaddress, ZWINFARPTR(window_buffer32K)));
		  memcpy_door_open ((void *)farptr2dsrelative(ZWINFARPTR(window_buffer32K)), (void *)linear2dsrelative(LOADER.curfileload->load_address + loadaddress), sizeused);
		  }
	      }
#ifdef CRC_TIMMING
	  memcpy_total += RDTSC() - tmp_rdtsc;
	  tmp_rdtsc = RDTSC();
#endif
	  crc32val = crc32 (crc32val, window_buffer32K, sizeused);
#ifdef CRC_TIMMING
	  crc32_total += RDTSC() - tmp_rdtsc;
#endif
	  filesize -= sizeused;
	  loadaddress += sizeused;
	  }
      if (crc32val != LOADER.curfileload->max_size_or_crc32) {
	  ZDBG (("CRC32 failure: is 0x%X should be 0x%X!\r\n", crc32val, LOADER.curfileload->max_size_or_crc32));
	  return 8 + filecpt; /* ERROR_RECHECK_KERNEL_CRC32, ERROR_RECHECK_INITRD_CRC32 */
	  }
#ifdef CRC_TIMMING
	else if (UTIL.processor.calibrate_rdtsc > 100)
	  ZDBG (("done: 0x%X (memcpy %u ms, crc32 %u ms)\r\n", crc32val,
		10 * memcpy_total / UTIL.processor.calibrate_rdtsc,
		10 * crc32_total / UTIL.processor.calibrate_rdtsc));
#endif
	else
	  ZDBG (("done: 0x%X\r\n", crc32val));
      filecpt++;
      }
  ZWINFREE (window_buffer32K);
  return 0;
#else
  unsigned filecpt = 0;

  while (filecpt < 2 && LOADER.fileload[filecpt].uncompressed_size != 0) {
      unsigned crc32val = crc32 (0, 0, 0);
      LOADER.curfileload = &LOADER.fileload[filecpt];

      if (inhibit & 1) {
	  ZDBG (("CRC32 recheck inhibited.\r\n"));
	  filecpt++;
	  continue;
	  }
      inhibit >>= 1;
      ZDBG (("checking %s crc32 %u bytes at 0x%X: ", filecpt? "initrd" : "kernel", LOADER.curfileload->uncompressed_size, LOADER.curfileload->load_address));
#if ASSEMBLY_TYPE != ASSEMBLY_DSES
      crc32val = crc32 (crc32val, (unsigned char *)LOADER.curfileload->load_address, LOADER.curfileload->uncompressed_size);
#else
      crc32val = crc32 (crc32val, (unsigned char *)linear2dsrelative(LOADER.curfileload->load_address), LOADER.curfileload->uncompressed_size);
#endif
      if (crc32val != LOADER.curfileload->max_size_or_crc32) {
	  ZDBG (("CRC32 failure: is 0x%X should be 0x%X!\r\n", crc32val, LOADER.curfileload->max_size_or_crc32));
	  return 8 + filecpt; /* ERROR_RECHECK_KERNEL_CRC32, ERROR_RECHECK_INITRD_CRC32 */
	  }
	else
	  ZDBG (("done"));
      filecpt++;
      }
  return 0;
#endif
  }
#endif /* GZLIB_ALSO_TEST */

/*
 * These are helpers for decompression:
 */
GZIP_FCT_PREFIX(readbits) static inline unsigned
readbits (z_stream *z, unsigned short nb)
  {
  while (z->bitk < nb) {
      if (UNUSUAL(z->avail_in == 0))
	  return 1;
      z->bitb |= ((unsigned) (z->avail_in--, *z->next_in++)) << z->bitk;
      z->bitk += 8;
      }
  return 0;
  }

GZIP_FCT_PREFIX(maskbits) static inline unsigned short
maskbits (z_stream *z, unsigned short nb)
  {
  return z->bitb & ~(0xFFFFFFFFU << nb);
  }

GZIP_FCT_PREFIX(flushbits) static inline void
flushbits (z_stream *z, unsigned short nb)
  {
  z->bitb >>= nb;
  z->bitk -= nb;
  }

/*
   Huffman code decoding is performed using a multi-level table lookup.
   The fastest way to decode is to simply build a lookup table whose
   size is determined by the longest code.  However, the time it takes
   to build this table can also be a factor if the data being decoded
   is not very long.  The most common codes are necessarily the
   shortest codes, so those codes dominate the decoding time, and hence
   the speed.  The idea is you can have a shorter table that decodes the
   shorter, more probable codes, and then point to subsidiary tables for
   the longer codes.  The time it costs to decode the longer codes is
   then traded against the time it takes to make longer tables.

   This results of this trade are in the variables lbits and dbits
   below.  lbits is the number of bits the first level table for literal/
   length codes can decode in one step, and dbits is the same thing for
   the distance codes.  Subsequent tables are also less than or equal to
   those sizes.  These values may be adjusted either when all of the
   codes are shorter than that, in which case the longest code length in
   bits is used, or when the shortest code is *longer* than the requested
   table size, in which case the length of the shortest code in bits is
   used.

   There are two different values for the two tables, since they code a
   different number of possibilities each.  The literal/length table
   codes 286 possible values, or in a flat code, a little over eight
   bits.  The distance table codes 30 possible values, or a little less
   than five bits, flat.  The optimum values for speed end up being
   about one bit more than those, so lbits is 8+1 and dbits is 5+1.
   The optimum values may differ though from machine to machine, and
   possibly even between compilers.  Your mileage may vary.
 */

/* Given a list of code lengths and a maximum table size, make a set of
   tables to decode that set of codes.
   Return Z_OK on success, Z_BUF_ERROR if the given code set is incomplete,
   Z_DATA_ERROR if the input is invalid (an over-subscribed set of lengths),
   or Z_MEM_ERROR if not enough memory. */
GZIP_FCT_PREFIX(huft_build) static int huft_build (
	unsigned short *b,	/* code lengths in bits (all assumed <= BMAX) */
	unsigned short n,	/* number of codes (assumed <= 288) */
	unsigned short s,	/* number of simple-valued codes (0..s-1) */
	const unsigned short *d,/* list of base values for non-simple codes */
	const unsigned char *e,	/* list of extra bits for non-simple codes */
	inflate_huft_size *result,	/* starting table +
					  maximum lookup bits, returns actual */
	inflate_huft *hp,	/* space for trees */
	unsigned short *hn	/* hufts used in space */
	)
{
  /* work area, either 19 or 288: */
  unsigned short v[288] = {[0 ... 287] = 0 };
  unsigned short bit_length_count[BMAX + 1] = {[0 ... BMAX] = 0 };
  unsigned short maximum_code_length, current_code_length;
  int returned;

  unsigned short x[BMAX + 1];	/* bit offsets, then code stack */

  /* Generate counts for each bit length */
  {
  unsigned short i = n;
  unsigned short *p = b;

  do {
      bit_length_count[*p++]++;	/* assume all entries <= BMAX */
      } while (--i);
  }

  /* Test for null input--all zero length codes */
  if (bit_length_count[0] == n) {
      result->tree = (inflate_huft *) 0;
      result->need = 0;
      return Z_OK;
      }

  /* Find minimum and maximum length, bound "result->need" by those */
  for (current_code_length = 1;
       current_code_length <= BMAX;
       current_code_length++)
      if (bit_length_count[current_code_length])
	  break;
  if (result->need < current_code_length)	/* minimum code length */
      result->need = current_code_length;

  for (maximum_code_length = BMAX;
       maximum_code_length != 0;
       maximum_code_length--)
      if (bit_length_count[maximum_code_length])
	  break;
  if (result->need > maximum_code_length)
      result->need = maximum_code_length;

  /* Adjust last length count to fill out codes, if needed */
  {
  unsigned short j = current_code_length;
  int number_of_dummy_codes_added = 1 << current_code_length;

  for (;;) {
      number_of_dummy_codes_added -= bit_length_count[j];
      if (number_of_dummy_codes_added < 0)
	  return Z_DATA_ERROR;
      if (j <= maximum_code_length) {
	  if (j == maximum_code_length)
	      break;
	  number_of_dummy_codes_added <<= 1;
	  }
      j++;
      }

  /* Return Z_BUF_ERROR if we were given an incomplete table */
  if (number_of_dummy_codes_added != 0 && maximum_code_length != 1) {
      /* inflate_trees_fixed() will finish with this error (ignored),
	 but wait the table (distance table) to be filled in! */
      returned = Z_BUF_ERROR;
      }
    else
      returned = Z_OK;

  bit_length_count[maximum_code_length] += number_of_dummy_codes_added;
  }

  /* Generate starting offsets into the value table for each length */
  {
  unsigned short i = maximum_code_length + 1;
  unsigned short *p = &bit_length_count[1];
  unsigned short *xp = &x[1];
  unsigned short accumulate = 0;

  while (--i) {
      *xp++ = accumulate;
      accumulate += *p++;
      }
  }

  /* Make a table of values in order of bit lengths */
  {
  unsigned short i = 0;

  for (i = 0; i < n; ++i) {
      if (b[i] != 0) {
	  v[x[b[i]]] = i;
	  x[b[i]]++;
	  }
      }
  }

  /* Generate the Huffman codes and for each, make the table entries */
  {
  int table_level;
  int bits_before_this_table;	/* == (result->need * table_level) */
  inflate_huft *table_stack[BMAX];
  /* the Huffman code of length current_code_length bits for value *p: */
  unsigned short current_code;
  unsigned short *p;		/* pointer into v[] */
  unsigned short nb_entry_current_table;
  unsigned short *end_of_v = &v[x[maximum_code_length]];

  /* just to keep compilers happy: */
  table_stack[0] = (inflate_huft *) 0;
  nb_entry_current_table = 0;

  /* go through the bit lengths
    (current_code_length already is bits in shortest code) */
  for (x[0] = current_code = 0,	/* first Huffman code is zero */
       p = v,			/* grab values in bit order */
       table_level = -1,	/* no tables yet--level -1 */
       bits_before_this_table = -result->need;	/* bits decoded */
       current_code_length <= maximum_code_length;
       current_code_length++) {
      /* counter for codes of length current_code_length: */
      unsigned short a = bit_length_count[current_code_length];

      /* make tables up to required level */
      while (a--) {
	  while (current_code_length > bits_before_this_table + result->need) {
	      unsigned short bit_in_this_table;

	      table_level++;
	      /* previous table always "result->need" bits: */
	      bits_before_this_table += result->need;

	      /* compute minimum size table less than
		 or equal to "result->need" bits: */
	      nb_entry_current_table = maximum_code_length
					- bits_before_this_table;
	      /* table size upper limit: */
	      if (nb_entry_current_table > result->need)
		  nb_entry_current_table = result->need;

	      bit_in_this_table = current_code_length - bits_before_this_table;

	      if (1 << bit_in_this_table > a + 1) {
		  /* too few codes for k-w bit table */
		  /* deduct codes from patterns left */
		  unsigned short f = (1 << bit_in_this_table) - a + 1;
		  unsigned short *xp = &bit_length_count[current_code_length];

		  if (bit_in_this_table < nb_entry_current_table) {
		      /* try smaller tables upto nb_entry_current_table bits: */
		      while (++bit_in_this_table < nb_entry_current_table) {
			  f <<= 1;
			  xp++;
			  if (f <= *xp)
			      break; /* enough codes to use up
					bit_in_this_table bits */
			  f -= *xp;	 /* else deduct codes from patterns */
			  }
		      }
		  }
	      nb_entry_current_table = 1 << bit_in_this_table;

	      /* allocate new table */
	      table_stack[table_level] = hp + *hn;
	      *hn += nb_entry_current_table;
	      if (*hn > MANY)
		  return Z_MEM_ERROR;	/* not enough memory */

	      /* connect to last table, if there is one */
	      if (!table_level) {
		  /* first table is returned result: */
		  result->tree = table_stack[table_level];
		  }
		else {
		  unsigned short j = current_code >> (bits_before_this_table
							- result->need);
		  /* connect to last table: */
		  inflate_huft *ptr = &table_stack[table_level - 1][j];

		  /* save pattern for backing up: */
		  x[table_level] = current_code;
		  ptr->exop = bit_in_this_table;
		  ptr->bits = result->need; /* bits to dump before this table */
		  ptr->base = (table_stack[table_level]
				 - table_stack[table_level - 1]
				 - j);	    /* offset to this table */
		  }
	      }

	  {
	  /* set up table entry in tmp */
	  inflate_huft tmp;

	  tmp.bits = current_code_length - bits_before_this_table;
	  if (p >= end_of_v) {
	      /* out of values--invalid code: */
	      tmp.exop = 128 + 64;
	      tmp.base = 0;
	      }
	    else if (*p < s) {
	      /* 256 is end-of-block: */
	      tmp.exop = *p < 256 ? 0 : 32 + 64;
	      /* simple code is just the value: */
	      tmp.base = *p;
	      p++;
	      }
	    else {
	      /* non-simple--look up in lists: */
	      tmp.exop = e[*p - s] + 16 + 64;
	      tmp.base = d[*p - s];
	      p++;
	      }

	  /* fill code-like entries with tmp */
	  {
	  /* current_code repeats in table every (1 << tmp.bits) entries: */
	  unsigned short j;

	  for (j = current_code >> bits_before_this_table;
	       j < nb_entry_current_table;
	       j += (1 << tmp.bits))
	      table_stack[table_level][j] = tmp;
	  }
	  }

	  /* backwards increment the k-bit code current_code */
	  {
	  unsigned short j = 1 << current_code_length;
	  for (;;) {
	      unsigned short old_code = current_code;

	      j >>= 1;
	      current_code ^= j;
	      if ((old_code & j) == 0)
		  break;
	      }
	  }

	  /* backup over finished tables */
	  while ((current_code & ((1U << bits_before_this_table) - 1))
		 != x[table_level]) {
	      table_level--;
	      bits_before_this_table -= result->need;
	      }
	  }
      }
  }
  return returned;
  }

/*
 * At first entry, z->mode == INIT == 0 and all other
 * fields have to be null, but z->extra.
 * To abort, set "z->mode = BAD" and call inflate().
 */
GZIP_FCT_PREFIX(inflate) zret_t
inflate (z_stream *z)
  {
  /* it would take too much code to generate cplext[] and cpdext[] on line */
  static const unsigned short cplens[31] = {
      3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31,
      35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0
      };
  static const unsigned char cplext[31] = {
      0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2,
      3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0, 112, 112
      };

  static const unsigned short cpdist[30] = {
      1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193,
      257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145,
      8193, 12289, 16385, 24577
      };
  static const unsigned char cpdext[30] = {
      0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6,
      7, 7, 8, 8, 9, 9, 10, 10, 11, 11,
      12, 12, 13, 13
      };
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
  unsigned short m;
#endif

  if (z == 0)
      return Z_STREAM_ERROR;

#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
  m = (1U << MAX_WBITS) - z->write;
#endif

  for (;;) {
      unsigned short utmp = utmp; /* inited B4 used */
      inflate_huft *htmp = htmp; /* inited B4 used */
      static const signed char nbbitarray[] = {
	[INIT]		= 0,
	[SIGNATURE]	= 16,
	[METHOD]	= 8,
	[FLAGS]		= 8,
	[MODIF_TIME1]	= 8,
	[MODIF_TIME2]	= 8,
	[MODIF_TIME3]	= 8,
	[MODIF_TIME4]	= 8,
	[EXTRAFLAGS]	= 8,
	[OPERATING_SYSTEM]	= 8,
	[EXTRAFIELDLEN]	= 16,
	[EXTRAFIELD]	= 8,
	[ORIGNAME]	= 8,
	[COMMENT]	= 8,
	[HEADERCRC]	= 16,
	[TYPE]		= 3,
	[STORED_LENS]	= 16,
	[STORED_NOTLENS]= 16,
	[STORED_COPY]	= 0,
	[TABLE]		= 14,
	[BTREE]		= 3,
	[DTREE1]	= -1,
	[DTREE2]		= -2,
	[DTREE3]	= 0,
	[CODES_START]	= 0,
	[CODES_LEN]	= -1,
	[CODES_LENEXT]		= -2,
	[CODES_DIST]	= -1,
	[CODES_DISTEXT]		= -2,
	[CODES_COPY]	= 0,
	[DONE]		= 0,
	[CRCCHK]	= 16,
	[BAD]		= 0
	  };

      {
      unsigned short nb;
      unsigned short read;
#if 0
      const char *statename[] = {
	  "INIT", "SIGNATURE", "METHOD", "FLAGS",
	  "MODIF_TIME1", "MODIF_TIME2", "MODIF_TIME3", "MODIF_TIME4",
	  "EXTRAFLAGS", "OPERATING_SYSTEM",
	  "EXTRAFIELDLEN", "EXTRAFIELD", "ORIGNAME", "COMMENT", "HEADERCRC",
	  "TYPE",
	  "STORED_LENS", "STORED_NOTLENS", "STORED_COPY",
	  "TABLE", "BTREE", "DTREE1", "DTREE2", "DTREE3",
	  "CODES_START", "CODES_LEN", "CODES_LENEXT", "CODES_DIST",
	  "CODES_DISTEXT", "CODES_COPY",
	  "DONE", "CRCCHK",
	  "BAD"
	  };
#define PRINTSTATE
#endif

      if (nbbitarray[z->mode] == -1)
	  nb = z->sub.cdtr.curhuft.need;
	else if (nbbitarray[z->mode] == -2)
	  nb = z->sub.cdtr.sub.copy.get; /* or sub.trees.nbbit */
	else
	  nb = nbbitarray[z->mode];

      if (readbits (z, nb))
	  return Z_OK;
      read = maskbits (z, nb);
      if (nbbitarray[z->mode] == -1) {
	  htmp = z->sub.cdtr.curhuft.tree + read;
	  nb = htmp->bits;
#ifdef PRINTSTATE
	  printf ("mode %s, htmp->bits %u, htmp->exop = 0x%X\n",
			statename[z->mode], htmp->bits, htmp->exop);
#endif
	  }
	else {
	  utmp = read;
#ifdef PRINTSTATE
	  printf ("mode %s, utmp = 0x%X\n", statename[z->mode], utmp);
#endif
	  }

      flushbits (z, nb);
      }

      switch (z->mode) {
	  case INIT:
#if defined (GZLIB_USE_INTERMEDIATE_WINDOW)
	      if ((z->window = ZWINALLOC (1U << MAX_WBITS)) == 0)
		  return Z_MEM_ERROR;
#endif
#if defined (GZLIB_USE_INTERMEDIATE_WINDOW) \
 || defined (GZLIB_USE_COMPILED_CRC_ARRAY)
	      z->crc32 = crc32 (z->crc32, 0, 0);
#endif
	      z->mode++;	/* = SIGNATURE */
	      continue;
	  case SIGNATURE:
	      if (utmp != 0x8B1F
#ifndef treat_data
		  || (LOADER.accept_uncompressed_filesize & 0x80000000)
#endif
		  ) {
		  if (utmp != 0x8B1F)
		      ZDBG (("inflate(): invalid Gzip signature, "));
		    else
		      ZDBG (("inflate(): forbid initrd decompression, "));
		  z->mode = BAD;
#ifndef treat_data
		  /* patch to load an uncompressed file */
		  if (LOADER.accept_uncompressed_filesize) {
		      ZDBG (("will load a file of %u bytes without decompressing\r\n", LOADER.accept_uncompressed_filesize & 0x7FFFFFFF));
		      z->sub.stored.left = LOADER.accept_uncompressed_filesize & 0x7FFFFFFF;
		      z->sub.stored.load_uncompressed = 1;
		      z->mode = STORED_COPY;
		      z->avail_in += 2;
		      z->next_in -= 2;
		      }
#endif
		  continue;
		  }
	      z->mode++;	/* = METHOD */
	      continue;
	  case METHOD:
	      if (utmp != Z_DEFLATED) {
		  ZDBG (("unknown method"));
		  z->mode = BAD;
		  continue;
		  }
	      z->mode++;	/* = FLAGS */
	      continue;
	  case FLAGS:
	      z->sub.gzip_header.sub.data = utmp;
	      if (z->sub.gzip_header.sub.flags.reserved) {
		  ZDBG (("unknown flags"));
		  z->mode = BAD;
		  continue;
		  }
	      z->mode++;	/* = MODIF_TIME1 */
	      continue;
	  case MODIF_TIME1:
	  case MODIF_TIME2:
	  case MODIF_TIME3:
	  case MODIF_TIME4:
	  case EXTRAFLAGS:
	      z->mode++;	/* = next */
	      continue;
	  case OPERATING_SYSTEM:
	      z->sub.gzip_header.len = 0;	/* for ORIGNAME or COMMENT */
	      if (z->sub.gzip_header.sub.flags.extra_field)
		  z->mode++;	/* = EXTRAFIELDLEN */
		else if (z->sub.gzip_header.sub.flags.orig_name)
		  z->mode = ORIGNAME;
		else if (z->sub.gzip_header.sub.flags.comment)
		  z->mode = COMMENT;
		else if (z->sub.gzip_header.sub.flags.header_crc)
		  z->mode = HEADERCRC;
		else
		  z->mode = TYPE;
	      continue;
	  case EXTRAFIELDLEN:
	      z->sub.gzip_header.len = utmp;
	      if (z->sub.gzip_header.len) /* Is that needed? */
		  z->mode++;	/* = EXTRAFIELD */
		else if (z->sub.gzip_header.sub.flags.orig_name)
		  z->mode = ORIGNAME;
		else if (z->sub.gzip_header.sub.flags.comment)
		  z->mode = COMMENT;
		else if (z->sub.gzip_header.sub.flags.header_crc)
		  z->mode = HEADERCRC;
		else
		  z->mode = TYPE;
	      continue;
	  case EXTRAFIELD:
	      if (-- z->sub.gzip_header.len)
		  continue;
	      z->sub.gzip_header.len = 0;	/* for ORIGNAME or COMMENT */
	      if (z->sub.gzip_header.sub.flags.orig_name)
		  z->mode++;	/* = ORIGNAME */
		else if (z->sub.gzip_header.sub.flags.comment)
		  z->mode = COMMENT;
		else if (z->sub.gzip_header.sub.flags.header_crc)
		  z->mode = HEADERCRC;
		else
		  z->mode = TYPE;
	      continue;
	  case ORIGNAME:
	      if (z->sub.gzip_header.len <= GZLIB_MAXBUF) {
		  if (z->sub.gzip_header.len == GZLIB_MAXBUF)
		      z->sub.gzip_header.buffer[GZLIB_MAXBUF] = 0;
		    else
		      z->sub.gzip_header.buffer[z->sub.gzip_header.len]	= utmp;
		  }
	      z->sub.gzip_header.len++;
	      if (utmp != 0)
		  continue;
	      ZDBG (("%s: gzip name: %s\r\n",
			__FUNCTION__, z->sub.gzip_header.buffer));
#ifdef GZLIB_TREAT_NAME
	      GZLIB_TREAT_NAME (z->sub.gzip_header.buffer);
#endif
	      z->sub.gzip_header.len = 0;	/* for COMMENT */
	      if (z->sub.gzip_header.sub.flags.comment)
		  z->mode++; /* = COMMENT */
		else if (z->sub.gzip_header.sub.flags.header_crc)
		  z->mode = HEADERCRC;
		else
		  z->mode = TYPE;
	      continue;
	  case COMMENT:
	      if (z->sub.gzip_header.len <= GZLIB_MAXBUF) {
		  if (z->sub.gzip_header.len == GZLIB_MAXBUF)
		      z->sub.gzip_header.buffer[GZLIB_MAXBUF] = 0;
		    else
		      z->sub.gzip_header.buffer[z->sub.gzip_header.len]	= utmp;
		  }
	      z->sub.gzip_header.len++;
	      if (utmp != 0)
		  continue;
	      ZDBG (("%s: gzip comment: %s\r\n",
			__FUNCTION__, z->sub.gzip_header.buffer));
#ifdef GZLIB_TREAT_COMMENT
	      GZLIB_TREAT_COMMENT (z->sub.gzip_header.buffer);
#endif
	      if (z->sub.gzip_header.sub.flags.header_crc)
		  z->mode++; /* = HEADERCRC */
		else
		  z->mode = TYPE;
	      continue;
	  case HEADERCRC:
	      z->mode++;	/* = TYPE */
	      continue;
	  case TYPE:
	      z->last = utmp & 1;
	      utmp >>= 1;
	      if (utmp == 3) {
		  ZDBG (("invalid block type"));
		  z->mode = BAD;
		  }
		else if (utmp == 2) {
		  z->mode = TABLE;
		  }
		else if (utmp == 1) { /* inflate_trees_fixed() */
		  int k;	/* temporary variable */
		  /* number of hufts used in fixed_mem: */
		  unsigned short f = 0;
		  /* length list for huft_build: */
		  unsigned short c[288];

		  /* literal table */
		  for (k = 0; k < 144; k++)
		      c[k] = 8;
		  for (; k < 256; k++)
		      c[k] = 9;
		  for (; k < 280; k++)
		      c[k] = 7;
		  for (; k < 288; k++)
		      c[k] = 8;
		  z->sub.cdtr.lbitstree.need = 9;
		  huft_build (c,
			      288,
			      257,
			      cplens,
			      cplext,
			      &z->sub.cdtr.lbitstree,
			      z->hufts,
			      &f);
#if 0
for (k = 0; k < 512; k++) {
    static const inflate_huft fixed_tl[512] = {
	{96,7,256},	{0,8,80},	{0,8,16},	{84,8,115},
	{82,7,31},	{0,8,112},	{0,8,48},	{0,9,192},
	{80,7,10},	{0,8,96},	{0,8,32},	{0,9,160},
	{0,8,0},	{0,8,128},	{0,8,64},	{0,9,224},
	{80,7,6},	{0,8,88},	{0,8,24},	{0,9,144},
	{83,7,59},	{0,8,120},	{0,8,56},	{0,9,208},
	{81,7,17},	{0,8,104},	{0,8,40},	{0,9,176},
	{0,8,8},	{0,8,136},	{0,8,72},	{0,9,240},
	{80,7,4},	{0,8,84},	{0,8,20},	{85,8,227},
	{83,7,43},	{0,8,116},	{0,8,52},	{0,9,200},
	{81,7,13},	{0,8,100},	{0,8,36},	{0,9,168},
	{0,8,4},	{0,8,132},	{0,8,68},	{0,9,232},
	{80,7,8},	{0,8,92},	{0,8,28},	{0,9,152},
	{84,7,83},	{0,8,124},	{0,8,60},	{0,9,216},
	{82,7,23},	{0,8,108},	{0,8,44},	{0,9,184},
	{0,8,12},	{0,8,140},	{0,8,76},	{0,9,248},
	{80,7,3},	{0,8,82},	{0,8,18},	{85,8,163},
	{83,7,35},	{0,8,114},	{0,8,50},	{0,9,196},
	{81,7,11},	{0,8,98},	{0,8,34},	{0,9,164},
	{0,8,2},	{0,8,130},	{0,8,66},	{0,9,228},
	{80,7,7},	{0,8,90},	{0,8,26},	{0,9,148},
	{84,7,67},	{0,8,122},	{0,8,58},	{0,9,212},
	{82,7,19},	{0,8,106},	{0,8,42},	{0,9,180},
	{0,8,10},	{0,8,138},	{0,8,74},	{0,9,244},
	{80,7,5},	{0,8,86},	{0,8,22},	{192,8,0},
	{83,7,51},	{0,8,118},	{0,8,54},	{0,9,204},
	{81,7,15},	{0,8,102},	{0,8,38},	{0,9,172},
	{0,8,6},	{0,8,134},	{0,8,70},	{0,9,236},
	{80,7,9},	{0,8,94},	{0,8,30},	{0,9,156},
	{84,7,99},	{0,8,126},	{0,8,62},	{0,9,220},
	{82,7,27},	{0,8,110},	{0,8,46},	{0,9,188},
	{0,8,14},	{0,8,142},	{0,8,78},	{0,9,252},
	{96,7,256},	{0,8,81},	{0,8,17},	{85,8,131},
	{82,7,31},	{0,8,113},	{0,8,49},	{0,9,194},
	{80,7,10},	{0,8,97},	{0,8,33},	{0,9,162},
	{0,8,1},	{0,8,129},	{0,8,65},	{0,9,226},
	{80,7,6},	{0,8,89},	{0,8,25},	{0,9,146},
	{83,7,59},	{0,8,121},	{0,8,57},	{0,9,210},
	{81,7,17},	{0,8,105},	{0,8,41},	{0,9,178},
	{0,8,9},	{0,8,137},	{0,8,73},	{0,9,242},
	{80,7,4},	{0,8,85},	{0,8,21},	{80,8,258},
	{83,7,43},	{0,8,117},	{0,8,53},	{0,9,202},
	{81,7,13},	{0,8,101},	{0,8,37},	{0,9,170},
	{0,8,5},	{0,8,133},	{0,8,69},	{0,9,234},
	{80,7,8},	{0,8,93},	{0,8,29},	{0,9,154},
	{84,7,83},	{0,8,125},	{0,8,61},	{0,9,218},
	{82,7,23},	{0,8,109},	{0,8,45},	{0,9,186},
	{0,8,13},	{0,8,141},	{0,8,77},	{0,9,250},
	{80,7,3},	{0,8,83},	{0,8,19},	{85,8,195},
	{83,7,35},	{0,8,115},	{0,8,51},	{0,9,198},
	{81,7,11},	{0,8,99},	{0,8,35},	{0,9,166},
	{0,8,3},	{0,8,131},	{0,8,67},	{0,9,230},
	{80,7,7},	{0,8,91},	{0,8,27},	{0,9,150},
	{84,7,67},	{0,8,123},	{0,8,59},	{0,9,214},
	{82,7,19},	{0,8,107},	{0,8,43},	{0,9,182},
	{0,8,11},	{0,8,139},	{0,8,75},	{0,9,246},
	{80,7,5},	{0,8,87},	{0,8,23},	{192,8,0},
	{83,7,51},	{0,8,119},	{0,8,55},	{0,9,206},
	{81,7,15},	{0,8,103},	{0,8,39},	{0,9,174},
	{0,8,7},	{0,8,135},	{0,8,71},	{0,9,238},
	{80,7,9},	{0,8,95},	{0,8,31},	{0,9,158},
	{84,7,99},	{0,8,127},	{0,8,63},	{0,9,222},
	{82,7,27},	{0,8,111},	{0,8,47},	{0,9,190},
	{0,8,15},	{0,8,143},	{0,8,79},	{0,9,254},
	{96,7,256},	{0,8,80},	{0,8,16},	{84,8,115},
	{82,7,31},	{0,8,112},	{0,8,48},	{0,9,193},
	{80,7,10},	{0,8,96},	{0,8,32},	{0,9,161},
	{0,8,0},	{0,8,128},	{0,8,64},	{0,9,225},
	{80,7,6},	{0,8,88},	{0,8,24},	{0,9,145},
	{83,7,59},	{0,8,120},	{0,8,56},	{0,9,209},
	{81,7,17},	{0,8,104},	{0,8,40},	{0,9,177},
	{0,8,8},	{0,8,136},	{0,8,72},	{0,9,241},
	{80,7,4},	{0,8,84},	{0,8,20},	{85,8,227},
	{83,7,43},	{0,8,116},	{0,8,52},	{0,9,201},
	{81,7,13},	{0,8,100},	{0,8,36},	{0,9,169},
	{0,8,4},	{0,8,132},	{0,8,68},	{0,9,233},
	{80,7,8},	{0,8,92},	{0,8,28},	{0,9,153},
	{84,7,83},	{0,8,124},	{0,8,60},	{0,9,217},
	{82,7,23},	{0,8,108},	{0,8,44},	{0,9,185},
	{0,8,12},	{0,8,140},	{0,8,76},	{0,9,249},
	{80,7,3},	{0,8,82},	{0,8,18},	{85,8,163},
	{83,7,35},	{0,8,114},	{0,8,50},	{0,9,197},
	{81,7,11},	{0,8,98},	{0,8,34},	{0,9,165},
	{0,8,2},	{0,8,130},	{0,8,66},	{0,9,229},
	{80,7,7},	{0,8,90},	{0,8,26},	{0,9,149},
	{84,7,67},	{0,8,122},	{0,8,58},	{0,9,213},
	{82,7,19},	{0,8,106},	{0,8,42},	{0,9,181},
	{0,8,10},	{0,8,138},	{0,8,74},	{0,9,245},
	{80,7,5},	{0,8,86},	{0,8,22},	{192,8,0},
	{83,7,51},	{0,8,118},	{0,8,54},	{0,9,205},
	{81,7,15},	{0,8,102},	{0,8,38},	{0,9,173},
	{0,8,6},	{0,8,134},	{0,8,70},	{0,9,237},
	{80,7,9},	{0,8,94},	{0,8,30},	{0,9,157},
	{84,7,99},	{0,8,126},	{0,8,62},	{0,9,221},
	{82,7,27},	{0,8,110},	{0,8,46},	{0,9,189},
	{0,8,14},	{0,8,142},	{0,8,78},	{0,9,253},
	{96,7,256},	{0,8,81},	{0,8,17},	{85,8,131},
	{82,7,31},	{0,8,113},	{0,8,49},	{0,9,195},
	{80,7,10},	{0,8,97},	{0,8,33},	{0,9,163},
	{0,8,1},	{0,8,129},	{0,8,65},	{0,9,227},
	{80,7,6},	{0,8,89},	{0,8,25},	{0,9,147},
	{83,7,59},	{0,8,121},	{0,8,57},	{0,9,211},
	{81,7,17},	{0,8,105},	{0,8,41},	{0,9,179},
	{0,8,9},	{0,8,137},	{0,8,73},	{0,9,243},
	{80,7,4},	{0,8,85},	{0,8,21},	{80,8,258},
	{83,7,43},	{0,8,117},	{0,8,53},	{0,9,203},
	{81,7,13},	{0,8,101},	{0,8,37},	{0,9,171},
	{0,8,5},	{0,8,133},	{0,8,69},	{0,9,235},
	{80,7,8},	{0,8,93},	{0,8,29},	{0,9,155},
	{84,7,83},	{0,8,125},	{0,8,61},	{0,9,219},
	{82,7,23},	{0,8,109},	{0,8,45},	{0,9,187},
	{0,8,13},	{0,8,141},	{0,8,77},	{0,9,251},
	{80,7,3},	{0,8,83},	{0,8,19},	{85,8,195},
	{83,7,35},	{0,8,115},	{0,8,51},	{0,9,199},
	{81,7,11},	{0,8,99},	{0,8,35},	{0,9,167},
	{0,8,3},	{0,8,131},	{0,8,67},	{0,9,231},
	{80,7,7},	{0,8,91},	{0,8,27},	{0,9,151},
	{84,7,67},	{0,8,123},	{0,8,59},	{0,9,215},
	{82,7,19},	{0,8,107},	{0,8,43},	{0,9,183},
	{0,8,11},	{0,8,139},	{0,8,75},	{0,9,247},
	{80,7,5},	{0,8,87},	{0,8,23},	{192,8,0},
	{83,7,51},	{0,8,119},	{0,8,55},	{0,9,207},
	{81,7,15},	{0,8,103},	{0,8,39},	{0,9,175},
	{0,8,7},	{0,8,135},	{0,8,71},	{0,9,239},
	{80,7,9},	{0,8,95},	{0,8,31},	{0,9,159},
	{84,7,99},	{0,8,127},	{0,8,63},	{0,9,223},
	{82,7,27},	{0,8,111},	{0,8,47},	{0,9,191},
	{0,8,15},	{0,8,143},	{0,8,79},	{0,9,255}
	};
    inflate_huft *hufts = z->sub.cdtr.lbitstree.tree;
    if (z->sub.cdtr.lbitstree.need != 9)
	printf ("z->sub.cdtr.lbitstree.need = %u != 9\n",
		 z->sub.cdtr.lbitstree.need);
    if (   hufts[k].exop != fixed_tl[k].exop
	|| hufts[k].bits != fixed_tl[k].bits
	|| hufts[k].base != fixed_tl[k].base) {
	printf ("hufts[%u] = {%u, %u, %u} != fixed_tl[%u] = {%u, %u, %u}\n",
		k, hufts[k].exop, hufts[k].bits, hufts[k].base,
		k, fixed_tl[k].exop, fixed_tl[k].bits, fixed_tl[k].base);
	}
    }
#endif

		  /* distance table */
		  for (k = 0; k < 30; k++)
		      c[k] = 5;
		  z->sub.cdtr.dbitstree.need = 5;
		  huft_build (c,
			      30,
			      0,
			      cpdist,
			      cpdext,
			      &z->sub.cdtr.dbitstree,
			      z->hufts,
			      &f);

#if 0
for (k = 0; k < 32; k++) {
    static const inflate_huft fixed_td[32] = {
	{80,5,1},	{87,5,257},	{83,5,17},	{91,5,4097},
	{81,5,5},	{89,5,1025},	{85,5,65},	{93,5,16385},
	{80,5,3},	{88,5,513},	{84,5,33},	{92,5,8193},
	{82,5,9},	{90,5,2049},	{86,5,129},	{192,5,24577},
	{80,5,2},	{87,5,385},	{83,5,25},	{91,5,6145},
	{81,5,7},	{89,5,1537},	{85,5,97},	{93,5,24577},
	{80,5,4},	{88,5,769},	{84,5,49},	{92,5,12289},
	{82,5,13},	{90,5,3073},	{86,5,193},	{192,5,24577}
	};
    inflate_huft *hufts = z->sub.cdtr.dbitstree.tree;
    if (z->sub.cdtr.dbitstree.need != 5)
	printf ("z->sub.cdtr.dbitstree.need = %u != 9\n",
	 z->sub.cdtr.dbitstree.need);
    if (   hufts[k].exop != fixed_td[k].exop
	|| hufts[k].bits != fixed_td[k].bits
	|| hufts[k].base != fixed_td[k].base) {
	printf ("hufts[%u] = {%u, %u, %u} != fixed_td[%u] = {%u, %u, %u}\n",
		k, hufts[k].exop, hufts[k].bits, hufts[k].base,
		k, fixed_td[k].exop, fixed_td[k].bits, fixed_td[k].base);
	}
    hufts[k] = fixed_td[k];
    }
#endif

		  z->mode = CODES_START;
		  }
		else /* if (utmp == 0) */ {
		  flushbits (z, z->bitk & 7);
		  z->mode++;	/* = STORED_LENS */
		  }
	      continue;
	  case STORED_LENS:
	      z->sub.stored.left = utmp;
	      z->sub.stored.load_uncompressed = 0;
	      z->mode++;	/* = STORED_NOTLENS */
	      continue;
	  case STORED_NOTLENS:
	      if (z->sub.stored.left != (utmp ^ 0xFFFFU)) {
		  ZDBG (("invalid stored block lengths"));
		  z->mode = BAD;
		  continue;
		  }
	      z->bitb = 0;
	      z->bitk = 0;
	      if (z->sub.stored.left == 0)
		  z->mode = z->last ? DONE : TYPE;
		else
		  z->mode++;	/* = STORED_COPY */
	      continue;
	  case STORED_COPY:
	      if (z->avail_in == 0)
		  return Z_OK;
	      if (z->sub.stored.left > z->avail_in)
		  utmp = z->avail_in;
		else
		  utmp = z->sub.stored.left;
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
	      if (utmp > m)
		    utmp = m;
	      memcpy_data2window (&z->window[z->write], z->next_in, utmp);
	      z->write += utmp;
	      m -= utmp;
#else
	      if (z->avail_out < utmp) {
		  ZDBG (("%s: not enought memory to load kernel!\r\n", __FUNCTION__));
		  return Z_OUPUT_FULL;
		  }
	      memcpy_door_open (z->next_out, z->next_in, utmp);
	      z->next_out += utmp;
	      z->avail_out -= utmp;
	      z->total_out += utmp;
#endif
	      z->next_in += utmp;
	      z->avail_in -= utmp;
	      z->sub.stored.left -= utmp;

	      /* patch to load an uncompressed file */
	      if (z->sub.stored.left == 0 && z->sub.stored.load_uncompressed) {
#ifndef treat_data
		  ZDBG (("inflate(): finished loading an uncompressed file of %u bytes\r\n",
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
			z->total_out + z->write
#else
			z->total_out
#endif
			));
#endif
		  z->mode = ALLDONE;
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
		  m = 0; /* force inflate_flush() */
		  break;
#else
		  continue;
#endif
		  }
	      if (z->sub.stored.left == 0)
		  z->mode = z->last ? DONE : TYPE;
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
	      break;
#else
	      continue;
#endif


	  case TABLE:
	      z->sub.cdtr.sub.trees.table = utmp;

#ifndef PKZIP_BUG_WORKAROUND
	      if (   (z->sub.cdtr.sub.trees.table & 0x1f) > 29
		  || ((z->sub.cdtr.sub.trees.table >> 5) & 0x1f) > 29) {
		  ZDBG (("too many length or distance symbols"));
		  z->mode = BAD;
		  continue;
		  }
#endif

	      z->sub.cdtr.len = 258 + (z->sub.cdtr.sub.trees.table & 0x1f)
				+ ((z->sub.cdtr.sub.trees.table >> 5) & 0x1f);
	      z->sub.cdtr.sub.trees.index = 0;
	      z->mode++;	/* = BTREE */
	      continue;
	  case BTREE: {
	      static const unsigned char border[] = {
		  16, 17, 18,  0,  8,  7,  9,  6, 10,  5,
		  11,  4, 12,  3, 13,  2, 14,  1, 15, 32,
		  71, 80, 76, 32, 69, 46, 76, 79, 82, 82,
		  65, 73, 78, 46,  0,  1,  3, 67,  0
		  };

	      z->sub.cdtr.blens[border[z->sub.cdtr.sub.trees.index++]] = utmp;
	      if (z->sub.cdtr.sub.trees.index < 4 +
					(z->sub.cdtr.sub.trees.table >> 10))
		  continue;
	      while (z->sub.cdtr.sub.trees.index < sizeof (border) - 20)
		  z->sub.cdtr.blens[border[z->sub.cdtr.sub.trees.index++]] = 0;
	      z->sub.cdtr.curhuft.need = 7;


	      { /* inflate_trees_bits() */
	      unsigned short hn = 0;
	      int r;

	      r = huft_build (z->sub.cdtr.blens,
			      sizeof (border) - 20,
			      sizeof (border) - 20,
			      (unsigned short *) 0,
			      (unsigned char *) 0,
			      &z->sub.cdtr.curhuft,
			      z->hufts,
			      &hn);
	      if (r == Z_DATA_ERROR)
		  ZDBG (("oversubscribed dynamic bit lengths tree"));
		else if (r == Z_BUF_ERROR || z->sub.cdtr.curhuft.need == 0) {
		  ZDBG (("incomplete dynamic bit lengths tree"));
		  r = Z_DATA_ERROR;
		  }

	      if (r != Z_OK) {
		  z->mode = BAD;
		  continue;
		  }
	      }

	      z->sub.cdtr.sub.trees.index = 0;
	      z->mode++;	/* = DTREE1 */
	      }
	      continue;

	  case DTREE1:
	      if (htmp->base < 16) {
		  z->sub.cdtr.blens[z->sub.cdtr.sub.trees.index++] = htmp->base;
		  if (z->sub.cdtr.sub.trees.index >= z->sub.cdtr.len)
		      z->mode = DTREE3;
		  }
		else /* htmp->base == 16..18 */ {
		  if (htmp->base == 18)
		      z->sub.cdtr.sub.trees.nbbit = 7;
		    else
		      z->sub.cdtr.sub.trees.nbbit = htmp->base - 14;
		  z->mode++; /* = DTREE2 */
		  }
	      continue;
	  case DTREE2:
	      if (z->sub.cdtr.sub.trees.nbbit == 7)
		  utmp += 11;
		else
		  utmp += 3;

	      if (   z->sub.cdtr.sub.trees.index + utmp > z->sub.cdtr.len
		  || (   z->sub.cdtr.sub.trees.nbbit == 2
		      && z->sub.cdtr.sub.trees.index < 1)) {
		  ZDBG (("invalid bit length repeat"));
		  z->mode = BAD;
		  continue;
		  }
	      {
	      unsigned short tmp;

	      if (z->sub.cdtr.sub.trees.nbbit == 2)
		  tmp = z->sub.cdtr.blens[z->sub.cdtr.sub.trees.index - 1];
		else
		  tmp = 0;

	      do {
		  z->sub.cdtr.blens[z->sub.cdtr.sub.trees.index++] = tmp;
		  } while (--utmp);
	      }

	      if (z->sub.cdtr.sub.trees.index >= z->sub.cdtr.len)
		  z->mode++; /* = DTREE3 */
		else {
		  z->mode = DTREE1;
		  continue;
		  }
	  case DTREE3:
	      z->sub.cdtr.curhuft.need = 0;

	      /* must be <= 9 for lookahead assumptions: */
	      z->sub.cdtr.lbitstree.need = 9;
	      z->sub.cdtr.dbitstree.need = 6;

	      { /* inflate_trees_dynamic() */
	      unsigned short hn = 0;
	      unsigned short nl = 257 + (z->sub.cdtr.sub.trees.table & 0x1f);
	      int r;

	      /* build literal/length tree */
	      r = huft_build (z->sub.cdtr.blens,
			      nl,
			      257,
			      cplens,
			      cplext,
			      &z->sub.cdtr.lbitstree,
			      z->hufts,
			      &hn);

	      if (r != Z_OK || z->sub.cdtr.lbitstree.need == 0) {
		  if (r == Z_DATA_ERROR)
		      ZDBG (("oversubscribed literal/length tree"));
		    else if (r != Z_MEM_ERROR) {
		      ZDBG (("incomplete literal/length tree"));
		      r = Z_DATA_ERROR;
		      }
		  }

	      if (r == Z_OK) {
		  /* build distance tree */
		  r = huft_build (z->sub.cdtr.blens + nl,
				1 + ((z->sub.cdtr.sub.trees.table >> 5) & 0x1f),
				0,
				cpdist,
				cpdext,
				&z->sub.cdtr.dbitstree,
				z->hufts,
				&hn);
		  if (   r != 0
		      || (   z->sub.cdtr.dbitstree.need == 0
			  && (z->sub.cdtr.sub.trees.table & 0x1f) != 0)) {
		      if (r == Z_DATA_ERROR)
			  ZDBG (("oversubscribed distance tree"));
			else if (r == Z_BUF_ERROR) {
#ifdef PKZIP_BUG_WORKAROUND
			  r = Z_OK;
			  }
#else
			  ZDBG (("incomplete distance tree"));
			  r = Z_DATA_ERROR;
			  }
			else if (r != Z_MEM_ERROR) {
			  ZDBG (("empty distance tree with lengths"));
			  r = Z_DATA_ERROR;
			  }
#endif
		      }

		  }

	      if (r != Z_OK) {
		  z->mode = BAD;
		  continue;
		  }
	      }
	      z->mode++;	/* = CODES_START */
	      continue;

	  case CODES_START:
	      z->sub.cdtr.curhuft = z->sub.cdtr.lbitstree;
	      z->mode++;	/* = CODES_LEN */
	      continue;
	  case CODES_LEN:
	      if (htmp->exop == 0) {
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
		  /* CODES_LIT: (needs m != 0) */
		  OUTBYTE (&z->window[z->write], htmp->base);
		  z->write += 1;
		  m--;
		  z->mode = CODES_START;
		  break;
#else
		  if (z->avail_out == 0) {
		      ZDBG (("%s: not enought memory to load kernel!\r\n",
				__FUNCTION__));
		      return Z_OUPUT_FULL;
		      }
		  *(z->next_out++) = htmp->base;
		  z->avail_out --;
		  z->total_out ++;
		  z->mode = CODES_START;
#endif
		  }
		else if (htmp->exop & 16) {
		  z->sub.cdtr.sub.copy.get = htmp->exop & 15;
		  z->sub.cdtr.len = htmp->base;
		  z->mode++;	/* = CODES_LENEXT */
		  }
		else if ((htmp->exop & 64) == 0) {
		  z->sub.cdtr.curhuft.need = htmp->exop;
		  z->sub.cdtr.curhuft.tree = htmp + htmp->base;
		  }
		else if (htmp->exop & 32) {
		  /* CODES_END: */
		  if (z->bitk > 7) {
		      z->bitk -= 8;
		      z->avail_in++;
		      z->next_in--;
		      }
		  /* usual way to finish */
		  z->mode = z->last ? DONE : TYPE;
		  }
		else {
		  ZDBG (("invalid literal/length code: 0x%X\r\n", htmp->exop));
		  z->mode = BAD;
		  }
	      continue;
	  case CODES_LENEXT:
	      z->sub.cdtr.len += utmp;
	      z->sub.cdtr.curhuft = z->sub.cdtr.dbitstree;
	      z->mode++;	/* = CODES_DIST */
	      continue;
	  case CODES_DIST:
	      if (htmp->exop & 16) {
		  z->sub.cdtr.sub.copy.get = htmp->exop & 15;
		  z->sub.cdtr.sub.copy.dist = htmp->base;
		  z->mode++;	/* = CODES_DISTEXT */
		  }
		else if ((htmp->exop & 64) == 0) {
		  z->sub.cdtr.curhuft.need = htmp->exop;
		  z->sub.cdtr.curhuft.tree = htmp + htmp->base;
		  }
		else {
		  ZDBG (("invalid distance code"));
		  z->mode = BAD;
		  }
	      continue;
	  case CODES_DISTEXT:
	      z->sub.cdtr.sub.copy.dist += utmp;
	      z->mode++;	/* = CODES_COPY */
	      continue;
	  case CODES_COPY:
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
	      {
		inline void copy_codes (z_stream *z,
					unsigned short index,
					unsigned short *nb)
		  {
		  unsigned short tmp = ((1U << MAX_WBITS) + z->write - index)
					% (1U << MAX_WBITS);
		  if (*nb > (1U << MAX_WBITS) - tmp)
		      *nb = (1U << MAX_WBITS) - tmp;
		  memcpy_window2window (&z->window[z->write],
					&z->window[tmp],
					*nb);
		  }

	      if (z->sub.cdtr.len > m)
		  utmp = m;
		else
		  utmp = z->sub.cdtr.len;
	      copy_codes (z, z->sub.cdtr.sub.copy.dist, &utmp);
	      z->write += utmp;
	      m -= utmp;
	      z->sub.cdtr.len -= utmp;

	      if (z->sub.cdtr.len == 0)
		  z->mode = CODES_START;
	      }
	      break;
#else
	      if (z->avail_out < z->sub.cdtr.len) {
		  ZDBG (("%s: not enought memory to load kernel!\r\n",
			__FUNCTION__));
		  return Z_OUPUT_FULL;
		  }
	      memcpy_door_open (z->next_out,
				z->next_out - z->sub.cdtr.sub.copy.dist,
				z->sub.cdtr.len);
	      z->next_out += z->sub.cdtr.len;
	      z->avail_out -= z->sub.cdtr.len;
	      z->total_out += z->sub.cdtr.len;
	      z->mode = CODES_START;
	      continue;
#endif


	  case DONE:
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
	      if (z->write != 0) {
		  m = 0; /* force inflate_flush() */
		  break;
		  }
#else
	      /* This has the advantage to do a basic test of the RAM
		 and checks that A20 is open... but need access to
		 all of the memory at once */
	      z->crc32 = crc32 (crc32 (0, 0, 0),
				z->next_out - z->total_out,
				z->total_out);
	      LOADER.uncompressed_signature = *(unsigned *)(z->next_out - z->total_out);
#endif
	      z->sub.crcchk.arrayindex = 0;
	      flushbits (z, z->bitk & 7);
	      z->mode++;	/* = CRCCHK */
	      continue;
	  case CRCCHK:
	      z->sub.crcchk.sub.array[z->sub.crcchk.arrayindex++] = utmp;
	      if (z->sub.crcchk.arrayindex < 4)
		  continue;
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
	      ZWINFREE (z->window);
#endif
//printf ("data length: %u (should be %u), crc32 0x%X (should be 0x%X)\r\n", z->total_out, z->sub.crcchk.sub.data.length, z->crc32, z->sub.crcchk.sub.data.crc32);
	      if (z->sub.crcchk.sub.data.length != z->total_out) {
		  ZDBG (("%s: incorrect uncompressed length: %u != %u\r\n",
			__FUNCTION__,
			z->sub.crcchk.sub.data.length, z->total_out));
		  return Z_BAD_LENGTH;
		  }
	      if (z->sub.crcchk.sub.data.crc32 != z->crc32) {
		  ZDBG (("%s: incorrect crc32: 0x%X != 0x%X\r\n",
			__FUNCTION__,
			z->sub.crcchk.sub.data.crc32, z->crc32));
		  return Z_BAD_CRC;
		  }
#ifndef GZLIB_ALSO_TEST
	      /* For crc32check(), late re-check of complete CRC32: */
	      LOADER.curfileload->max_size_or_crc32 = z->crc32;
#endif
	      ZDBG (("%s: uncompress finished successfully.\r\n",
			__FUNCTION__));
//printf ("data length & CRC32 OK\r\n");
	      z->mode++;	/* = ALLDONE */
	  case ALLDONE:
#ifdef GZLIB_TEST_MEASURE
	      crc32state++;
#endif /* GZLIB_TEST_MEASURE */
	      return Z_STREAM_END;
	  case BAD:
#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
	      ZWINFREE (z->window);
#endif
	      ZDBG (("\r\n%s: state BAD !\r\n", __FUNCTION__));
	      return Z_DATA_ERROR;
	}

#ifdef GZLIB_USE_INTERMEDIATE_WINDOW
      if (m == 0) {
	  /* always flush the complete window, but last time: */
	  if (z->write > z->avail_out) {
	      ZDBG (("%s: not enought memory to load kernel!\r\n",
			__FUNCTION__));
	      ZWINFREE (z->window);
	      return Z_OUPUT_FULL;
	      }

	  utmp = treat_data (z->next_out, z->window, z->write);
	  if (utmp) {
	      ZDBG (("%s: memcpy error!\r\n", __FUNCTION__));
	      ZWINFREE (z->window);
	      return Z_BASE_MEMCOPY_ERROR + utmp;
	      }

	  z->crc32 = crc32 (z->crc32, z->window, z->write);
#ifdef GZLIB_TEST_MEASURE
	  {
	  unsigned index;

	  if (crc32state >= load_restart) {
	      for (index = 0; index < nbof (crc32list[0]); index++)
		  crc32list[0][index] = crc32list[1][index] = 0;
	      crc32lastsize[0] = crc32lastsize[1] = 0;
	      crc32state = load_kernel;
	      }
	  /* fill the array, assumes CRC32 are never null */
	  for (index = 0; index < nbof (crc32list[0]); index++)
	      if (crc32list[crc32state][index] == 0) {
		  crc32list[crc32state][index] = crc32 (0, z->window, z->write);
		  break;
		  }
	  if (z->write != 0x8000) /* That is the last window */
	      crc32lastsize[crc32state] = z->write;
//printf ("CRC32 total: 0x%X ; local[%d][%d] 0x%X (z->write %u)\r\n", z->crc32, crc32state, index, crc32list[crc32state][index], z->write);
//printf ("", index);	// GCC-3.4.5 + DOSEMU
	  }
#endif /* GZLIB_TEST_MEASURE */
	  z->next_out += z->write;
	  z->avail_out -= z->write;
	  z->total_out += z->write;
	  z->write = 0;
	  m = 1U << MAX_WBITS;
	  }
#endif
      }
  }

#ifdef GZLIB_TEST_MEASURE
GZIP_FCT_PREFIX(crc32multicheck) void
crc32multicheck (void)
  {
  unsigned filecpt = 0;
// Also can be used when linuxparam is in its window...
//  void *window_buffer32K = ZWINALLOC(1U<<MAX_WBITS);
  void *window_buffer32K = ((void *)((0x2000U + getss()) << 16));

//  printf ("crc32multicheck: 32K window at 0x%X, farptr 0x%X, stackseg = 0x%X\r\n", window_buffer32K, ZWINFARPTR(window_buffer32K), getss());
  while (filecpt < 2 && LOADER.fileload[filecpt].uncompressed_size != 0) {
      unsigned diff_rdtsc = 0, index = 0;
      LOADER.curfileload = &LOADER.fileload[filecpt];

      printf ("checking crc32 %u bytes at 0x%X: ", LOADER.curfileload->uncompressed_size, LOADER.curfileload->load_address);
      while (crc32list[filecpt][index] != 0) {
	  unsigned long long rdtsc_start = rdtsc_start; /* inited b4 used */
	  unsigned crc32val, sizeused = 0x8000; /* = 0x8000 up to last page, last page variable size */

	  if (crc32list[filecpt][index + 1] == 0 && crc32lastsize[filecpt] != 0)
	      sizeused = crc32lastsize[filecpt];

#if 1 /* When HIMEM is used, _BIOS_copy_extended_memory does not work (PC hard lock) */
	  if (BOOT1_DOS_running() && UTIL.HIMEM_entrypoint) {
	      unsigned char error;
	      XMS_EMM emm_struct = {
		  .size       = sizeused + (sizeused & 1),  /* even number of bytes */
		  .src_handle = LOADER.curfileload->handle,
		  .src_offset = 0x8000 * index,
		  .dst_handle = 0,          /* real mode address */
		  .dst_offset = ZWINFARPTR(window_buffer32K),
		  };
	      if (_XMS_move_memory (UTIL.HIMEM_entrypoint, &emm_struct, &error) != 0) {
		  printf ("_XMS_move_memory(%u words from handle %u offset 0x%X to 0x%X) error!\r\n",
			(sizeused+1)/2, LOADER.curfileload->handle, 0x8000 * index, ZWINFARPTR(window_buffer32K));
		  return;
		  }
	      }
	    else
#endif
	      {
	      if (
#if USER_SUPPORT & VESA_SUPPORT
		  UI.parameter.winsize == VESA2_MARKER ||
#endif
		  _BIOS_copy_extended_memory (farptr2linear(ZWINFARPTR(window_buffer32K)), LOADER.curfileload->load_address + 0x8000 * index, (sizeused+1)/2) != 0) {
#if USER_SUPPORT & VESA_SUPPORT
		  if (UI.parameter.winsize != VESA2_MARKER)
#endif
		      printf ("_BIOS_copy_extended_memory(%u words from 0x%X to 0x%X) error!\r\n",
				(sizeused+1)/2, LOADER.curfileload->load_address + 0x8000 * index, ZWINFARPTR(window_buffer32K));
		  memcpy_door_open ((void *)farptr2dsrelative(ZWINFARPTR(window_buffer32K)), (void *)linear2dsrelative(LOADER.curfileload->load_address + 0x8000 * index), sizeused);
		  }
	      }
	  if (UTIL.processor.calibrate_rdtsc)
	      rdtsc_start = rdtsc();
	  crc32val = crc32 (0, window_buffer32K, sizeused);
	  if (UTIL.processor.calibrate_rdtsc)
	      diff_rdtsc += rdtsc() - rdtsc_start;
	  if (crc32val != crc32list[filecpt][index]) {
	      printf ("CRC32[%s][%u] failure (%u bytes at 0x%X): is 0x%X should be 0x%X!\r\n",
			filecpt? "initrd" : "kernel", index,
			sizeused, LOADER.curfileload->load_address + 0x8000 * index,
			crc32val, crc32list[filecpt][index]);
	      }
//	    else printf ("CRC32[%u][%u] OK, ", filecpt, index);
	  index++;
	  }
      if (UTIL.processor.calibrate_rdtsc > 10)
	  printf ("(CRC32 computation in %u ms) ", diff_rdtsc / (UTIL.processor.calibrate_rdtsc / 10));
      puts ("done");
      filecpt++;
      }
  ZWINFREE (window_buffer32K);
  _BIOS_wait (5 * 1000 * 1000);
  }
#endif /* GZLIB_TEST_MEASURE */

