/* gzcopy.c */

/*
 * gzcopy: utility to copy (only) GZip files, possibly modifying its header.
 * Copyright (C) 2010 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* Changelog:
 * 15/9/2001: Etienne Lorrain. Initial release.
 * 23/8/2002: Etienne Lorrain. '-s' -> '-c', only one input/output file
 *
 */

#include <stdio.h>
#include <malloc.h>
#include <errno.h>
#include <sys/stat.h>	/* stat() */
#include <unistd.h>	/* stat() */

#if 0
#include <string.h>
#else
static inline unsigned my_strlen (const char *str)
  {
  const char *ptr = str;
  while (*ptr)
      ptr++;
  return ptr - str;
  }
#define strlen my_strlen

static inline void my_strcpy (char *dst, const char *src)
  {
  while (*src)
      *dst++ = *src++;
  *dst = *src;	/* copy the zero */
  }
#define strcpy my_strcpy

static inline void my_strcat (char *dst, const char *src)
  {
  dst += strlen (dst);
  strcpy (dst, src);
  }
#define strcat my_strcat

static inline unsigned strcmp (const char *s1, const char *s2)
  {
  while (*s1 && *s1 == *s2) {
      s1++;
      s2++;
      }
  return (*s1 != *s2);
  }
#endif

/*
 * To test the algorithm used in gzlib.h/gzlib.c, we can
 * define GZLIB_ALSO_TEST:
 * This is the main reason of the GPL license of this file gzcopy.c;
 * if someone wants to modify and use a less restrictive licence compatible
 * with gzip/gunzip, I would not oppose as long as GZLIB_ALSO_TEST is
 * undefined.
 */
#define GZLIB_ALSO_TEST

#define STR static const char

STR usage[] =
"gzcopy: utility to copy (only) GZip files, possibly modifying its header.\n"
"Copyright 2003 Etienne Lorrain. Available under the GPL license version 2.\n"
"USAGE:\n"
"    gzcopy -h              -> display this help\n"
"    gzcopy -V              -> display the version\n"
"    gzcopy -v              -> be verbose\n"
"    gzcopy -c infile.gz    -> show the comment. (-c0 for raw form)\n"	/* was -s */
"    gzcopy -c='' infile.gz outfile.gz    -> set (clear if '') the comment\n"
"    gzcopy -a=\"<a comment>\" in.gz out.gz -> add/append a comment\n"
"    gzcopy -p='<a comment>' in.gz out.gz -> add/prepend a comment.\n"
"    gzcopy -k infile.gz    -> show the header CRC. (-k0 for raw form)\n"
"    gzcopy -k={0,1} infile.gz outfile.gz  -> (de)activate the header CRC\n"
"    gzcopy -b infile.gz    -> show the text/binary flag\n"
"    gzcopy -b={0,1} infile.gz outfile.gz  -> set the flag: binary=0/text=1\n"
"    gzcopy -n infile.gz    -> show the original name. (-n0 for raw form)\n"
"    gzcopy -n=filename in.gz out.gz       -> set (clear if '') original name\n"
#ifdef GZLIB_ALSO_TEST
"    gzcopy -t infile.gz    -> test the content of the file (-t0 for quite).\n"
#endif
"NOTE:\n"
" Use multiple command: gzcopy -c= -a='com' -a='m' -pl='ent' in.gz out.gz .\n"
" Use option -a{s,l,t} or -p{s,l,t} to check/force a {space,linefeed,tab}\n"
"  in between comment parts. Use filename \"-\" for stdin and/or stdout.\n"
" Use the -f option to force overwriting output file, or if you want\n"
" the output in the input file - you then need a lot of virtual memory.\n";

STR version[] = "gzcopy version 1.1 (C) 2003 Etienne Lorrain, GPL v2.";
STR show_textflags_text_msg[]   = "\"%s\" is probably a text file\n";
STR show_textflags_binary_msg[] = "\"%s\" is probably a binary file\n";
STR show_header_crc_msg[]    = "\"%s\" header CRC16 is 0x%X.\n";
STR no_CRC_to_show_msg[]     = "\"%s\" does not contain any header CRC.\n";
STR show_name_msg[]          = "\"%s\" original name: \"%s\"\n";
STR no_name_to_show_msg[]    = "\"%s\" does not contain any original name.\n";
STR show_comment_msg[]       = "\"%s\" comment: \"%s\"\n";
STR no_comment_to_show_msg[] = "\"%s\" does not contain any comment.\n";

#define PRINTERR(format, args...) do { \
	if (errno) perror ("gzcopy");		\
	fprintf (stderr, "gzcopy: ");		\
	fprintf (stderr, format , ## args);	\
	fprintf (stderr, "\n");			\
	} while (0)

#define ENDHLP	", try: gzcopy -h"

STR unknown_action[]           = "unrecognised option -%c" ENDHLP;
STR malformed_crc_param[]      = "header crc option is neither "
				"enable (-k=1) nor disable (-k=0)" ENDHLP;
STR malformed_param[]	       = "no parameter after -%c= option" ENDHLP;
STR too_many_param[]           = "too many parameters: \"%s\"" ENDHLP;
STR unknown_comment[]          = "no comment to pre/append" ENDHLP;
STR no_input_file[]            = "no input file" ENDHLP;
STR no_output_file[]           = "no output file" ENDHLP;
STR cannot_open_input_file[]   = "cannot open input file \"%s\".";
STR not_gzip_file[]            = "input file \"%s\" is not a gzip file, "
					"signature: 0x%X.";
STR bad_header_crc[]           = "bad header crc32: is 0x%X, "
					"should be 0x%X.";
STR not_enough_memory[]        = "not enough memory.";
STR cannot_open_output_file[]  = "cannot open input file \"%s\".";
STR outfile_already_exists[]   = "output file \"%s\" already exists and "
				"option -f not specified.\n";
STR cannot_write_to_file[]     = "cannot write to file \"%s\".";
STR read_input_file_error[]    = "error reading from file \"%s\".";
STR cannot_close_input_file[]  = "cannot close input file \"%s\".";
STR cannot_close_output_file[] = "cannot close output file \"%s\".";

#define VERBOSE(format, args...) do { \
	if (verbose)				\
	    fprintf (stderr, format , ## args);	\
	} while (0)

STR treating_intput_file[] = "treating \"%s\" as input file.\n";
STR standart_input[]	   = "standart input";
STR standart_output[]	   = "standart output";
STR extra_field_present[]  = "extra field of %u bytes present.\n";
STR original_name[]	   = "original name: \"%s\".\n";
STR comment_present[]	   = "comment: \"%s\".\n";
STR header_crc_present[]   = "header CRC16 present: 0x%X.\n";
STR treating_output_file[] = "treating \"%s\" as output file.\n";
STR enabling_header_crc[]  = "enabling header CRC.\n";
STR disabling_header_crc[] = "disabling header CRC.\n";
STR removing_comment[]	   = "removing comment.\n";
STR replacing_comment[]    = "replacing comment with \"%s\".\n";
STR adding_comment[]	   = "adding comment \"%s\".\n";
STR appending_comment[]	   = "appending comment \"%s\".\n";
STR prepending_comment[]   = "prepending comment \"%s\".\n";
STR new_header_crc[]	   = "new header CRC16: 0x%X.\n";
STR removing_name[]        = "removing the original name field.\n";
STR changing_name[]        = "changing the original name field to \"%s\".\n";
STR set_probablytext_flag[]   = "set the probably ASCII/text flags.\n";
STR clear_probablytext_flag[] = "clear the probably ASCII/text flags.\n";

#undef STR

enum main_returned_enum {
    warn_empty_comment		= 1,
    warn_no_comment		= 2,
    err_none			= 0,
    err_too_many_param		= -1,
    err_unknown_comment		= -2,
    err_unknown_action		= -3,
    err_malformed_param		= -4,
    err_malformed_crc_param	= -5,
    err_no_input_file		= -6,
    err_no_output_file		= -7,
    err_cannot_open_input_file	= -8,
    err_not_gzip_file		= -9,
    err_bad_header_crc		= -10,
    err_not_enough_memory	= -11,
    err_cannot_open_output_file = -12,
    err_outfile_already_exists  = -13,
    err_cannot_write_to_file    = -14,
    err_read_input_file_error   = -15,
    err_cannot_close_input_file	= -16,
    err_cannot_close_output_file = -17
    };

/* The Gzip header (with unsigned char flags): */
struct gzheader_str {
    unsigned short      signature_0x8B1F;
    unsigned char       compression_method; /* =Z_DEFLATED */
    unsigned char	flags;
#define FLAGS_ascii		0x01	/* file probably ascii text */
#define FLAGS_header_crc	0x02	/* header CRC present */
#define FLAGS_extra_field	0x04	/* extra field present */
#define FLAGS_orig_name		0x08	/* original file name present */
#define FLAGS_comment		0x10	/* file comment present */
#define FLAGS_unknown		0xE0	/* reserved */
    unsigned            modif_time;	/* nb second since 1/1/1970 */
    unsigned char	extraflags;
#define EXTRAFLAGS_max_compress	2
#define EXTRAFLAGS_fast_algo	4
    unsigned char       operating_system;
    } __attribute__ ((packed));

/**
 ** These are extracts of gzlib.h/gzlib.c, simplified to compile
 ** on Unix, and are also there for testing purposes...
 **/

#ifndef GZLIB_ALSO_TEST

/*
 * The minimum needed, do not need gzlib.[ch] to generate:
 */

/* Just if you want to waste data space: */
/*#define GZLIB_USE_PRECOMPILED_CRC_ARRAY*/

/* The only compression method recognised by Gzip: */
#define Z_DEFLATED	8

/*
 * This is derived from the crc32() function in zlib, but to save
 * space I removed the precompiled array - so it is slower.
 * See zlib documentation for copyright of its own functions;
 * the following rewritten functions (added work) are GPL ONLY.
 * -----------------------------------------------------------
 */
#ifdef GZLIB_USE_PRECOMPILED_CRC_ARRAY
static unsigned
crc32 (unsigned crc, const unsigned char *buf, unsigned len)
  {
  static unsigned crc_table[256];

  if (buf == 0) {
      unsigned short n;
      unsigned poly = 0xedb88320L;	/* polynomial exclusive-or pattern */

      for (n = 0; n < 256; n++) {
	  unsigned char k;
	  unsigned c = (unsigned) n;
	  for (k = 0; k < 8; k++)
	      c = c & 1 ? poly ^ (c >> 1) : c >> 1;
	  crc_table[n] = c;
	  }
      return 0L;
      }

  crc = crc ^ 0xffffffffL;
  if (len != 0) {
      unsigned nb = ((len - 1) / 8) + 1;

      switch (len % 8) { do {
	  case 0: crc = crc_table[(crc ^ *buf++) & 0xff] ^ (crc >> 8);
	  case 7: crc = crc_table[(crc ^ *buf++) & 0xff] ^ (crc >> 8);
	  case 6: crc = crc_table[(crc ^ *buf++) & 0xff] ^ (crc >> 8);
	  case 5: crc = crc_table[(crc ^ *buf++) & 0xff] ^ (crc >> 8);
	  case 4: crc = crc_table[(crc ^ *buf++) & 0xff] ^ (crc >> 8);
	  case 3: crc = crc_table[(crc ^ *buf++) & 0xff] ^ (crc >> 8);
	  case 2: crc = crc_table[(crc ^ *buf++) & 0xff] ^ (crc >> 8);
	  case 1: crc = crc_table[(crc ^ *buf++) & 0xff] ^ (crc >> 8);
	  } while (--nb); }
      }
  return crc ^ 0xffffffffL;
  }
#else /* GZLIB_USE_PRECOMPILED_CRC_ARRAY */
static inline unsigned
crc32 (unsigned crc, const unsigned char *buf, unsigned len)
  {
  const unsigned char *end = buf + len;

  if (buf == 0)
	return 0L;

  crc = ~crc;
  while (buf < end) {
      unsigned c = (crc ^ *buf++) & 0xff;
      unsigned short k;
#if 0
#define POLY	0xedb88320L /* polynomial exclusive-or pattern */
      for (k = 8; k != 0; k--)
	  c = c & 1 ? (c >> 1) ^ POLY : c >> 1;
#else
      unsigned short tmp;
      asm (
"	movw $8,%w2		\n"
"	1:			\n"
"	bsfw %w0,%%cx		\n"
"	jz 2f			\n"
"	incw %%cx		\n"
"	subw %%cx,%w2		\n"
"	jc 3f			\n"
"	shrl %%cl,%0		\n"
"	xorl $0xedb88320,%0	\n"
"	jmp 1b			\n"
"	2:			\n"
"	xorw %%cx,%%cx		\n"
"	3:			\n"
"	addw %w2,%%cx		\n"
"	shrl %%cl,%0		\n"
	: "+r" (c), "=c" (k), "=r" (tmp));
#endif
      crc = c ^ (crc >> 8);
      }
  return ~crc;
  }
#endif /* GZLIB_USE_PRECOMPILED_CRC_ARRAY */

#else /* GZLIB_ALSO_TEST */

/*
 * That is mainly to test the gzlib.[ch] files:
 */
  /* This helps the compile: */
#define MAKE_H		/* gzlib.c: do not include make.h */
#define ASSEMBLY_DSES	0x000003
#define ASSEMBLY_TYPE	1	/* use a window */

#define XCODE_SEGMENT	0x000020
#define SETUP		0	/* no xcode segment */
#define awk_farcall(fct)	/* */

#undef GZLIB_BIG_MALLOC	/* 32 Kb window declared as variable */
#define GZIP_FCT_PREFIX(X)	/* */
#define UNUSUAL(X) X

#define LIBRARY_H	/* gzlib.c: do not include library.h */
#define VESABIOS_H	/* gzlib.c: do not include vesabios.h */
#define VMLINUZ_H	/* gzlib.c: do not include vmlinuz.h */
#define DEBUG_H		/* gzlib.c: do not include debug.h */

#define ZDBG(X)	/* */
/*#define ZDBG(X) printf X*/

#if 1
#define treat_data(out, buffer, len) 0 /* no error, just check checksum! */
#else
#define treat_data(out, buffer, len) ({ \
	static FILE *fptr = 0;					\
	if (fptr == 0) fptr = fopen ("gzcopy.out", "w");	\
	fwrite (buffer, 1, len, fptr);				\
	/* return */ 0; })
#endif

/* inside the window: */
static inline void _memcpy (unsigned char *dst, const unsigned char *src, unsigned size)
  {
  size++;
  while (--size)
      *dst++ = *src++;
  }

#define BOOT_H
#include "gzlib.c"	/* includes gzlib.h */

#endif /* GZLIB_ALSO_TEST */

/**
 ** end of extracts
 **/

static unsigned initheader_crc32 (struct gzheader_str *hdr)
  {
  unsigned crc = crc32 (0, 0, 0);
  return crc32 (crc, (unsigned char *)hdr, sizeof (struct gzheader_str));
  }

static unsigned addbyte_crc32 (unsigned crc, int val)
  {
  unsigned char tmp = val;

  return crc32 (crc, &tmp, 1);
  }

int main (int argc, char **argv)
  {
  char /* *progname = *argv,*/
       help_option[] = "-h",
       *help_option_ptr = help_option;

  enum main_returned_enum returned = err_none;

  char *infilename = 0, *outfilename = 0;
  FILE *infile = NULL, *outfile = NULL;

  int verbose = 0, show_name = 0, show_comment = 0, overwrite = 0,
      show_header_crc = 0, show_text_flag = 0,
      set_text_flag = -1, activate_header_crc = -1;

  struct gzheader_str header;
  unsigned header_crc = 0, calculated_header_crc;
  unsigned short extrafield_len = 0;
  char *extrafield = 0;
  char *name = 0, *comment = 0;

#ifdef GZLIB_ALSO_TEST
  unsigned also_test = 0;
#endif

  char *newcomment = 0, *newprecomment = 0, *newpostcomment = 0, *newname = 0;
  char checkprespc = 0, checkpostspc = 0;
  char *ptr;
  int cpt;

  if (argc == 1) {
      argc = 2;
      argv = &help_option_ptr;
      }
    else {
      argc--;
      argv++;
      }

  for (cpt = 0; cpt < argc; cpt++) {
      if (argv[cpt][0] == '-' && argv[cpt][1] != '\0') {
	  switch (argv[cpt][1]) {
	      case 'V':
		  printf ("%s\n", version);
		  return err_none;
	      case 'h':
		  printf ("%s\n", usage);
		  return err_none;
	      case 'v':
		  verbose = 1;
		  break;
	      case 'f':
		  if (argv[cpt][2] == '\0')
		      overwrite = 1;
		  break;
	      case 'b':
		  if (argv[cpt][2] == '\0') {
		      show_text_flag = 1;
		      break;
		      }
		  if (argv[cpt][2] == '0') {
		      show_text_flag = 2; /* raw form */
		      break;
		      }
		  if (argv[cpt][2] != '='
			|| (argv[cpt][3] != '0' && argv[cpt][3] != '1')) {
		      PRINTERR (malformed_param, argv[cpt][1]);
		      return err_malformed_param;
		      }
		  set_text_flag = (argv[cpt][3] == '1');
		  break;
	      case 'k':
		  if (argv[cpt][2] == '\0') {
		      show_header_crc = 1;
		      break;
		      }
		  if (argv[cpt][2] == '0') {
		      show_header_crc = 2; /* raw form */
		      break;
		      }
		  if (argv[cpt][2] != '='
			|| (argv[cpt][3] != '0' && argv[cpt][3] != '1')) {
		      PRINTERR (malformed_crc_param);
		      return err_malformed_crc_param;
		      }
		  activate_header_crc = (argv[cpt][3] == '1');
		  break;
	      case 'n':
		  if (argv[cpt][2] == '\0') {
		      show_name = 1;
		      break;
		      }
		  if (argv[cpt][2] == '0') {
		      show_name = 2;
		      break;
		      }
		  if (argv[cpt][2] != '=') {
		      PRINTERR (malformed_param, argv[cpt][1]);
		      return err_malformed_param;
		      }
		  newname = &argv[cpt][3];
		  break;
	      case 'c':
		  if (argv[cpt][2] == '\0') {
		      show_comment = 1;
		      break;
		      }
		  if (argv[cpt][2] == '0') {
		      show_comment = 2;
		      break;
		      }
		  if (argv[cpt][2] != '=') {
		      PRINTERR (malformed_param, argv[cpt][1]);
		      return err_malformed_param;
		      }
		  newcomment = &argv[cpt][3];
		  break;
	      case 'p': {
		  unsigned len;
		  unsigned char checkspc;
		  char *start;

		  len = 2;
		  switch (argv[cpt][len]) {
		      case 's':
			  checkspc = ' ';
			  len++;
			  break;
		      case 'l':
			  checkspc = '\n';
			  len++;
			  break;
		      case 't':
			  checkspc = '\t';
			  len++;
			  break;
		      default:
		      case '=':
			  checkspc = '\0';
			  break;
		      }

		  if (argv[cpt][len++] != '=') {
		      PRINTERR (malformed_param, argv[cpt][1]);
		      return err_malformed_param;
		      }
		  start = &argv[cpt][len];
		  len = strlen (start) + 1;
		  if (*start == 0) {
		      if (newprecomment == 0)
			  checkprespc = checkspc;
		      }
		    else if (newprecomment == 0) {
		      newprecomment = malloc (len);
		      if (newprecomment == 0)
			  goto label_not_enough_memory;
		      checkprespc = checkspc;
		      strcpy (newprecomment, start);
		      }
		    else {
		      char *tmp = malloc (strlen(newprecomment) + len + 1);
		      if (tmp == 0)
			  goto label_not_enough_memory;
		      strcpy (tmp, start);
		      /* last char: */
		      ptr = &tmp[len - 2];
		      if (checkspc != '\0' && *ptr != checkspc
			  && newprecomment[0] != checkspc)
			  *++ptr = checkspc;
		      ptr ++;
		      strcpy (ptr, newprecomment);
		      free (newprecomment);
		      newprecomment = tmp;
		      }
		  }
		  break;
	      case 'a': {
		  unsigned len;
		  unsigned char checkspc;
		  char *start;

		  len = 2;
		  switch (argv[cpt][len]) {
		      case 's':
			  checkspc = ' ';
			  len++;
			  break;
		      case 'l':
			  checkspc = '\n';
			  len++;
			  break;
		      case 't':
			  checkspc = '\t';
			  len++;
			  break;
		      case '=':
		      default:
			  checkspc = '\0';
			  break;
		      }

		  if (argv[cpt][len++] != '=') {
		      PRINTERR (malformed_param, argv[cpt][1]);
		      return err_malformed_param;
		      }
		  start = &argv[cpt][len];
		  len = strlen (start) + 1;
		  if (*start == 0) {
		      if (newpostcomment == 0)
			  checkpostspc = checkspc;
		      }
		    else if (newpostcomment == 0) {
		      newpostcomment = malloc (len);
		      if (newpostcomment == 0)
			  goto label_not_enough_memory;
		      checkpostspc = checkspc;
		      strcpy (newpostcomment, start);
		      }
		    else {
		      ptr = realloc (newpostcomment, strlen(newpostcomment) + len + 1);
		      if (ptr == 0)
			  goto label_not_enough_memory;
		      newpostcomment = ptr;
		      /* last char: */
		      ptr = &newpostcomment[strlen(newpostcomment) - 1];
		      if (checkspc != '\0' && *ptr != checkspc
			  && *start != checkspc)
			  *++ptr = checkspc;
		      ptr ++;
		      strcpy (ptr, start);
		      }
		  }
		  break;
#ifdef GZLIB_ALSO_TEST
	      case 't':
		  if (argv[cpt][2] != '0')
		      also_test = 1;
		    else
		      also_test = 2;
		  break;
#endif
	      default:
		  PRINTERR (unknown_action, argv[cpt][1]);
		  return err_unknown_action;
	      }
	  }
	else if (infilename == 0) {
	  infilename = argv[cpt];
	  }
	else if (outfilename == 0) {
	  outfilename = argv[cpt];
	  }
	else {
	  PRINTERR (too_many_param, argv[cpt]);
	  return err_too_many_param;
	  }
      }

  /*
   * Check option validity:
   */
  if (infilename == 0) {
      PRINTERR (no_input_file);
      return err_no_input_file;
      }

  if (newname || newcomment || newpostcomment || newprecomment
	|| activate_header_crc != -1 || set_text_flag != -1) {
      if (outfilename == 0) {
	  if (overwrite) {
	      outfilename = infilename;
	      }
	    else {
	      PRINTERR (no_output_file);
	      return err_no_output_file;
	      }
	  }
      }

  /*
   * Read the input file:
   */
  if (!strcmp (infilename, "-")) {
      infile = stdin;
      VERBOSE (treating_intput_file, standart_input);
      }
    else {
      infile = fopen (infilename, "r");
      if (infile == NULL) {
	  PRINTERR (cannot_open_input_file, infilename);
	  return err_cannot_open_input_file;
	  }
      VERBOSE (treating_intput_file, infilename);
      }

  ptr = (char *)&header;
  for (cpt = 0; cpt < sizeof (header); cpt++) {
      int tmp = fgetc (infile);
      if (tmp == EOF) {
	  if (!ferror (infile))
	      goto label_not_gzip_file;
	    else
	      goto label_read_input_file_error;
	  }
      *ptr++ = tmp;
      }

  calculated_header_crc = initheader_crc32 (&header);

  if (   header.signature_0x8B1F != 0x8B1F
      || header.compression_method != Z_DEFLATED
      || header.flags & FLAGS_unknown)
      goto label_not_gzip_file;

  if (header.flags & FLAGS_extra_field) {
      int tmp = fgetc (infile);
      if (tmp == EOF) {
	  if (!ferror (infile))
	      goto label_not_gzip_file;
	    else
	      goto label_read_input_file_error;
	  }
      calculated_header_crc = addbyte_crc32 (calculated_header_crc, tmp);
      extrafield_len = tmp;
      tmp = fgetc (infile);
      if (tmp == EOF) {
	  if (!ferror (infile))
	      goto label_not_gzip_file;
	    else
	      goto label_read_input_file_error;
	  }
      calculated_header_crc = addbyte_crc32 (calculated_header_crc, tmp);
      extrafield_len |= tmp << 8;
      VERBOSE (extra_field_present, extrafield_len);

      if (extrafield_len != 0) {
	  if ((extrafield  = malloc (extrafield_len)) == 0)
	      goto label_not_enough_memory;
	  }
      for (cpt = 0; cpt < extrafield_len; cpt++) {
	  tmp = fgetc (infile);
	  if (tmp == EOF) {
	      if (!ferror (infile))
		  goto label_not_gzip_file;
		else
		  goto label_read_input_file_error;
	      }
	  calculated_header_crc = addbyte_crc32 (calculated_header_crc, tmp);
	  extrafield[cpt] = tmp;
	  }
      }

  if (header.flags & FLAGS_orig_name) {
      unsigned name_len = 0, name_alloced = 0;
      int tmp;

      do {
	  tmp = fgetc (infile);
	  if (tmp == EOF) {
	      if (!ferror (infile))
		  goto label_not_gzip_file;
		else
		  goto label_read_input_file_error;
	      }
	  calculated_header_crc = addbyte_crc32 (calculated_header_crc, tmp);
	  if (name_len >= name_alloced) {
	      if (name != 0) {
		  if ((ptr = realloc (name, name_alloced += 1024)) == 0)
		      goto label_not_enough_memory;
		  name = ptr;
		  }
		else {
		  if ((name = malloc (name_alloced = 1024)) == 0)
		      goto label_not_enough_memory;
		  }
	      }
	  name[name_len++] = tmp;
	  } while (tmp != '\0');
      VERBOSE (original_name, name);
      }

  if (header.flags & FLAGS_comment) {
      unsigned comment_len = 0, comment_alloced = 0;
      int tmp;

      do {
	  tmp = fgetc (infile);
	  if (tmp == EOF) {
	      if (!ferror (infile))
		  goto label_not_gzip_file;
		else
		  goto label_read_input_file_error;
	      }
	  calculated_header_crc = addbyte_crc32 (calculated_header_crc, tmp);
	  if (comment_len >= comment_alloced) {
	      if (comment != 0) {
		  if ((ptr = realloc (comment, comment_alloced += 1024)) == 0)
		      goto label_not_enough_memory;
		  comment = ptr;
		  }
		else {
		  if ((comment = malloc (comment_alloced = 1024)) == 0)
		      goto label_not_enough_memory;
		  }
	      }
	  comment[comment_len++] = tmp;
	  } while (tmp != '\0');
      VERBOSE (comment_present, comment);
      }

  if (header.flags & FLAGS_header_crc) {
      int tmp = fgetc (infile);
      if (tmp == EOF) {
	  if (!ferror (infile))
	      goto label_not_gzip_file;
	    else
	      goto label_read_input_file_error;
	  }
      header_crc = tmp;
      tmp = fgetc (infile);
      if (tmp == EOF) {
	  if (!ferror (infile))
	      goto label_not_gzip_file;
	    else
	      goto label_read_input_file_error;
	  }
      header_crc |= tmp << 8;
      calculated_header_crc &= 0xFFFF;
      if (show_header_crc == 2)
	  printf ("0x%X\n", header_crc);
	else if (show_header_crc != 0)
	  printf (show_header_crc_msg, infilename, header_crc);
      VERBOSE (header_crc_present, header_crc);
      if (calculated_header_crc != header_crc) {
	  PRINTERR (bad_header_crc, header_crc, calculated_header_crc);
	  returned = err_bad_header_crc;
	  goto label_end;
	  }
      }
    else {
      if (show_header_crc != 0 && show_header_crc != 2)
	  printf (no_CRC_to_show_msg, infilename);
      }

  /*
   * Treat commands:
   */

  if (show_name == 2) {
      if (name != 0)
	  printf ("%s\n", name);
      }
    else if (show_name) {
      if (name != 0)
	  printf (show_name_msg, infilename, name);
	else
	  printf (no_name_to_show_msg, infilename);
      }

  if (show_comment == 2) {
      if (comment != 0)
	  printf ("%s\n", comment);
      }
    else if (show_comment) {
      if (comment != 0)
	  printf (show_comment_msg, infilename, comment);
	else
	  printf (no_comment_to_show_msg, infilename);
      }

  if (show_text_flag == 2) {
      printf ("%u\n", header.flags & FLAGS_ascii);
      }
    else if (show_text_flag) {
      if (header.flags & FLAGS_ascii)
	  printf (show_textflags_text_msg, infilename);
	else
	  printf (show_textflags_binary_msg, infilename);
      }

  if (set_text_flag != -1) {
      if (set_text_flag == 0) {
	  header.flags &= ~FLAGS_ascii;
	  VERBOSE (clear_probablytext_flag);
	  }
	else {
	  header.flags |= FLAGS_ascii;
	  VERBOSE (set_probablytext_flag);
	  }
      }

  if (activate_header_crc != -1) {
      if (activate_header_crc) {
	  header.flags |= FLAGS_header_crc;
	  VERBOSE (enabling_header_crc);
	  }
	else {
	  header.flags &= ~FLAGS_header_crc;
	  VERBOSE (disabling_header_crc);
	  }
      }

  if (newname) {
      if (*newname == 0) {
	  header.flags &= ~FLAGS_orig_name;
	  VERBOSE (removing_name);
	  }
	else {
	  header.flags |= FLAGS_orig_name;
	  VERBOSE (changing_name, newname);
	  if (name)
	      free (name);
	  name = malloc (strlen (newname) + 1);
	  if (name == 0)
	      goto label_not_enough_memory;
	  strcpy (name, newname);
	  }
      }

  if (newcomment) {
      if (*newcomment == '\0') {
	  VERBOSE (removing_comment);
	  if (comment)
	      free (comment);
	  comment = 0;
	  header.flags &= ~FLAGS_comment;
	  }
	else {
	  VERBOSE (replacing_comment, newcomment);
	  if (comment)
	      free (comment);
	  comment = malloc (strlen (newcomment) + 1);
	  if (comment == 0)
	      goto label_not_enough_memory;
	  strcpy (comment, newcomment);
	  header.flags |= FLAGS_comment;
	  }
      }

  if (newprecomment) {
      if ((header.flags & FLAGS_comment) == 0) {
	  VERBOSE (adding_comment, newprecomment);
	  comment = malloc (strlen (newprecomment) + 1);
	  if (comment == 0)
	      goto label_not_enough_memory;
	  strcpy (comment, newprecomment);
	  /* last char: */
	  ptr = &comment[strlen (comment)];
	  if (checkprespc != '\0' && *ptr != checkprespc) {
	      *++ptr = checkprespc;
	      *++ptr = '\0';
	      }
	  free (newprecomment);
	  newprecomment = 0;
	  }
	else {
	  VERBOSE (prepending_comment, newprecomment);
	  ptr = realloc (newprecomment, strlen (newprecomment)
					+ 1 + strlen (comment) + 1);
	  if (ptr == 0)
	      goto label_not_enough_memory;
	  newprecomment = ptr;
	  /* last char: */
	  ptr = &newprecomment[strlen (newprecomment)];
	  if (checkprespc != '\0' && *ptr != checkprespc && comment[0] != checkprespc)
	      *++ptr = checkprespc;
	  ptr ++;
	  strcpy (ptr, comment);
	  free (comment);
	  comment = newprecomment;
	  newprecomment = 0;
	  }
      header.flags |= FLAGS_comment;
      }

  if (newpostcomment) {
      if ((header.flags & FLAGS_comment) == 0) {
	  VERBOSE (adding_comment, newpostcomment);
	  comment = malloc (strlen (newpostcomment) + 1);
	  if (comment == 0)
	      goto label_not_enough_memory;
	  strcpy (comment, newpostcomment);
	  /* last char: */
	  ptr = &comment[strlen (comment)];
	  if (checkpostspc != '\0' && *ptr != checkpostspc) {
	      *++ptr = checkpostspc;
	      *++ptr = '\0';
	      }
	  free (newpostcomment);
	  newpostcomment = 0;
	  }
	else {
	  VERBOSE (appending_comment, newpostcomment);
	  ptr = realloc (comment, strlen (comment)
				+ 1 + strlen (newpostcomment) + 1);
	  if (ptr == 0)
	      goto label_not_enough_memory;
	  comment = ptr;
	  /* last char: */
	  ptr = &comment[strlen (comment) - 1];
	  if (   checkpostspc != '\0'
	      && *ptr != checkpostspc
	      && newpostcomment[0] != checkpostspc)
	      *++ptr = checkpostspc;
	  ptr ++;
	  strcpy (ptr, newpostcomment);
	  free (newpostcomment);
	  newpostcomment = 0;
	  }
      header.flags |= FLAGS_comment;
      }

#ifdef GZLIB_ALSO_TEST
  if (also_test) {
      long filepos, filelen;
      unsigned char *buffer;
      unsigned tmp;
      z_stream stream = { .mode = INIT, .avail_out = 0xFFFFFFFU};

      filepos = ftell (infile);
      if (fseek (infile, 0, SEEK_END) != 0)
	  PRINTERR ("fseek(end) error!\n");
      filelen = ftell (infile);
      buffer = malloc (filelen);
      if (buffer == 0)
	  goto label_not_enough_memory;
      if (fseek (infile, 0, SEEK_SET) != 0)
	  PRINTERR ("fseek(start) error!\n");
      if (fread (buffer, 1, filelen, infile) != filelen)
	  PRINTERR ("fread() error!\n");

      stream.next_in = buffer;
      stream.avail_in = filelen;

      tmp = inflate (&stream);
      if (tmp == Z_OK)
	  PRINTERR ("Unexpected end of file in %s!\n", infilename);
	else if (tmp != Z_STREAM_END)
	  PRINTERR ("Unzipping error %u in %s!\n", tmp, infilename);
	else if (also_test == 1)
	  printf ("test content/CRC of %s OK\n", infilename);

#if 0
      printf ("stream: avail_in = %u, len: %u, stored len: %u\n"
		"        crc32: 0x%X, stored crc32: 0x%X.\n",
		stream.avail_in,
		stream.total_out, stream.sub.crcchk.sub.data.length,
		stream.crc32, stream.sub.crcchk.sub.data.crc32);
#endif

      if (fseek (infile, filepos, SEEK_SET) != 0)
	  PRINTERR ("fseek(init) error!\n");
      }
#endif

  /*
   * Write the file back:
   */

  if (outfilename != 0) {
      struct stat inbuf;
      char *full_content = 0;
      unsigned full_length = 0;

      if (fstat (fileno(infile), &inbuf) != -1) {
	  full_content = malloc (inbuf.st_size); /* bit too much but... */
	  if (full_content != 0) {
	      char *ptr = full_content;
	      for (;;) {
		  int tmp = fgetc (infile);

		  if (tmp == EOF) {
		      if (ferror(infile)) {
			  free (full_content);
			  goto label_read_input_file_error;
			  }
		      break;
		      }
		  full_length++;
		  if (full_length >= inbuf.st_size) {
		      free (full_content);
		      goto label_read_input_file_error;
		      }
		  *ptr++ = tmp;
		  }
	      }
	    else if (outfilename == infilename)
	      goto label_read_input_file_error;
	  }

      if (!strcmp (outfilename, "-")) {
	  outfile = stdout;
	  VERBOSE (treating_output_file, standart_output);
	  }
	else {
	  if (overwrite == 0) {
	      struct stat buf;
	      if (stat (outfilename, &buf) != -1 && errno != ENOENT) {
		  errno = 0;
		  PRINTERR (outfile_already_exists, outfilename);
		  returned = err_outfile_already_exists;
		  goto label_end;
		  }
	      errno = 0;
	      }
	  outfile = fopen (outfilename, "w");
	  if (outfile == NULL) {
	      PRINTERR (cannot_open_output_file, outfilename);
	      returned = err_cannot_open_output_file;
	      goto label_end;
	      }
	  VERBOSE (treating_output_file, outfilename);
	  }

      ptr = (char *)&header;
      for (cpt = 0; cpt < sizeof (header); cpt++) {
	  if (fputc (*ptr++, outfile) == EOF)
	      goto label_cannot_write_to_file;
	  }

      calculated_header_crc = initheader_crc32 (&header);

      if (header.flags & FLAGS_extra_field) {
	  ptr = extrafield;
	  if (   fputc (extrafield_len, outfile) == EOF
	      || fputc (extrafield_len >> 8, outfile) == EOF)
	      goto label_cannot_write_to_file;
	  calculated_header_crc = addbyte_crc32 (calculated_header_crc,
							extrafield_len);
	  calculated_header_crc = addbyte_crc32 (calculated_header_crc,
							extrafield_len >> 8);

	  for (cpt = 0; cpt < extrafield_len; cpt++) {
	      if (fputc (*ptr, outfile) == EOF)
		  goto label_cannot_write_to_file;
	      calculated_header_crc = addbyte_crc32 (calculated_header_crc,
							*ptr);
	      ptr++;
	      }
	  }
      if (header.flags & FLAGS_orig_name) {
	  unsigned name_len = strlen (name) + 1;

	  ptr = name;
	  for (cpt = 0; cpt < name_len; cpt++) {
	      if (fputc (*ptr, outfile) == EOF)
		  goto label_cannot_write_to_file;
	      calculated_header_crc = addbyte_crc32 (calculated_header_crc,
							*ptr);
	      ptr++;
	      }
	  }
      if (header.flags & FLAGS_comment) {
	  unsigned comment_len = strlen (comment) + 1;

	  ptr = comment;
	  for (cpt = 0; cpt < comment_len; cpt++) {
	      if (fputc (*ptr, outfile) == EOF)
		  goto label_cannot_write_to_file;
	      calculated_header_crc = addbyte_crc32 (calculated_header_crc,
							*ptr);
	      ptr++;
	      }
	  }
      if (header.flags & FLAGS_header_crc) {
	  calculated_header_crc &= 0xFFFF;
	  if (   fputc (calculated_header_crc, outfile) == EOF
	      || fputc (calculated_header_crc >> 8, outfile) == EOF)
	      goto label_cannot_write_to_file;
	  if (show_header_crc == 2)
	      printf ("0x%X\n", header_crc);
	    else if (show_header_crc != 0)
	      printf (show_header_crc_msg, infilename, header_crc);
	  VERBOSE (new_header_crc, calculated_header_crc);
	  }

      if (full_content == 0) {
	  for (;;) {
	      int tmp = fgetc (infile);

	      if (tmp == EOF) {
		  if (ferror(infile))
		      goto label_read_input_file_error;
		  break;
		  }
	      if (fputc (tmp, outfile) == EOF)
		  goto label_cannot_write_to_file;
	      }
	  }
	else {
	  if (fwrite (full_content, 1, full_length, outfile) != full_length)
	      goto label_cannot_write_to_file;
	  }
      }

label_end:
  if (extrafield != 0)
      free (extrafield);
  if (name != 0)
      free (name);
  if (comment != 0)
      free (comment);

  if (infile != NULL) {
      if (fclose (infile) == EOF) {
	  PRINTERR (cannot_close_input_file, infilename);
	  returned = err_cannot_close_input_file;
	  }
      }

  if (outfile != NULL) {
      if (fclose (outfile) == EOF) {
	  PRINTERR (cannot_close_output_file, outfilename);
	  returned = err_cannot_close_output_file;
	  }
      }
  return returned;

label_not_gzip_file:
  PRINTERR (not_gzip_file, infilename, header.signature_0x8B1F);
  returned = err_not_gzip_file;
  goto label_end;

label_not_enough_memory:
  PRINTERR (not_enough_memory);
  returned = err_not_enough_memory;
  goto label_end;

label_read_input_file_error:
  PRINTERR (read_input_file_error, infilename);
  returned = err_read_input_file_error;
  goto label_end;

label_cannot_write_to_file:
  PRINTERR (cannot_write_to_file, outfilename);
  returned = err_cannot_write_to_file;
  goto label_end;
  }
