/* fs.h */
#ifndef FS_H
#define FS_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"

/*
 * path + filename max len:
 * Defined quite low to have malloced blocks of 96 (i.e. 128) bytes,
 * so better memory usage.
 */
//#define NAME_LENGTH	64 // pre-2.8
#define NAME_LENGTH	96

/*
 * max nb of way to boot system:
 * The structure is malloced, but this is just defined
 * as security - for instance is there were a loop.
 * Also used for the mingujin configuration.
 */
#if (DISK_SUPPORT == DOS_SUPPORT)
#define NB_DESC_ARRAY	5
#else
#define NB_DESC_ARRAY	60 /* too high may create stack overflow */
#endif

/*
  * max 15 ISO files per partition, supported by all but for tiny_exe
  */
#if (DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|E2FS_PROBE|ISOFS_PROBE))
#ifdef ISOEXT
#define NB_ISO	15
#endif
#endif

/*
 * max number of fragment for a FAT/E2FS file - if the file
 * is not fragmented, it has one fragment whatever its size.
 * Undefine it to have malloc/realloc/free system - so the
 * limit is the available memory in data/stack segment.
 * Need to be bigger than 2 for the end marker.
 * Keep it quite high in case of stripped RAID disk?
 * Max number of fragments for iso files is not that define.
 */
#ifndef FS_USE_MALLOC
#define FILE_MAX_FRAGMENT	50
#endif

/*
 * Enable this to debug Long File Name on FAT12/16/32
 * when DEBUG_FS is defined:
 */
//#define DEBUG_LFN

#ifdef DEBUG_LFN
#define LFNDBG(x)	FDBG(x)
#else
#define LFNDBG(x)	/* */
#endif

/*
 * Enable this to debug the gzlib inflate algorithm
 * in byte per byte mode:
 */
//#define TEST_BYTE_PER_BYTE_INFLATE

/*
 * The structure where we store all the bootable systems we have found:
 */
extern struct BOOTWAY_str {
    unsigned short nb, nb_initrd, sizeof_desc_str, nb_bdi;
    unsigned char max_name_length, max_desc_array;
    unsigned short nb_iso; /* before there was a gap */
    struct desc_str {
	unsigned inode;
	unsigned filesize;
	unsigned last_modification_date; /* in seconds since Jan 1st, 2001 (i.e. after Feb 29th, 2000) */
	unsigned char variant_number;
	unsigned char searchpath; /* =0 for copy_gujin_param.scanpath, =1 for install.386, =2 for install.amd, =3 for isolinux, =4 for casper, =5 for live */
	unsigned char variant_timeout;
	unsigned char unused;
	unsigned iso_filesize;	/* if inIsoRoot or inIsoSlashBoot */
	unsigned iso_inode;	/* if inIsoRoot or inIsoSlashBoot */
	char *iso_fsname;
	unsigned char disk;
	unsigned char partition;
	unsigned char name_offset;	/* to after the pattern */
	unsigned char inRoot : 1;
	unsigned char inISO : 1;
	enum boottype_e { is_MBR, is_PBR, is_linux, is_elf,
		is_initrd, is_bdi_file, is_el_torito, is_initramfs, is_multiboot } boottype : 4;
	unsigned char ReadOnly	 : 1;	/* mostly for BDI files */
	char filename[NAME_LENGTH];
	} __attribute__ ((packed)) *desc;
#ifdef NB_ISO
    /* Filesystem in *.ISO matter, only filed/exists(previous version) if nb_iso != 0 */
    struct {
	unsigned char max_iso, max_fragment, max_filename, sizeofremap;
	/* the remap part translate LBA from *.iso to partition relative when nb_remap != 0 */
	/* On ext2, a 5 Gb file (10485760 sectors) will mostly have 8192 sectors of data followed
	by 8 sectors of metadata, so 1280 segments minimum... that is remap of 15360 bytes.
	With 1024 bytes/blocks, 256 * 1024-blocks (512 sectors) will be followed by 1 K (2 sectors).
	With 2048 bytes/blocks, 512 * 2048-blocks (2048 sectors) will be followed by 2 K (4 sectors).
	With 4096 bytes/blocks, 1024 * 4096-blocks (8192 sectors) will be followed by 4 K (8 sectors).
	With 8192 bytes/blocks, 2048 * 8192-blocks (32768 sectors) will be followed by 8 K (16 sectors).
	With 16384 bytes/blocks, 4096 * 16384-blocks (131072 sectors) will be followed by 16 K (32 sectors).
	So if the MSB of remap->lba is set, one adds blocksize/sectorsize every ((blocksize/1024)^2)*sectorsize
	With 2048 bytes/sectors:
	With 2048 bytes/blocks, 512 * 2048-blocks (512 sectors) will be followed by 2 K (1 sectors).
	With 4096 bytes/blocks, 1024 * 4096-blocks (2048 sectors) will be followed by 4 K (2 sectors).
	With 8192 bytes/blocks, 2048 * 8192-blocks (8192 sectors) will be followed by 8 K (4 sectors).
	So the formula is every ((blocksize/(2 * sectorsize))^2)*sectorsize
	And to be applied in integers: (((2 * blocksize)/(2 * sectorsize))^2)*sectorsize/4
	That is: (blocksize/sectorsize)*(blocksize/sectorsize)*sectorsize/4
	i.e. : (blocksize/sectorsize)*blocksize/4
	*/
	unsigned short nb_remap;
	unsigned char gap_sectors_shift; /* sector_size << gap_sectors_shift */
	unsigned char intergap_shift;	/* 1 << intergap_shift */
	struct sector_chain_str {
	    unsigned long long lba;
	    unsigned nb;	/* end marker = 0 */
	    } remap[128];
	/* the following is per filesystem, it is just filed while analysing the filesystem and valid until another FS is open */
	unsigned short nb_found, current_used;
	struct {
	    unsigned inode;
	    unsigned size;	/* FIXME: unsigned long long, anyway only used on FS with 4Gb file limits */
	    char filename[56];
	    } found[NB_ISO];
	/* Following present if sizeofremap != 0: */
	unsigned short previous_nb_remap;
	unsigned short external_fs; /* 1 for ISO9660, 2 for Ext4fs, 3 for vfat */
	unsigned unused[15];
	} iso;
#endif /* NB_ISO */
    } BOOTWAY;

/*
 * Prototypes and intersegment stuff:
 */
void disk_analyse (void);
awk_farcall (disk_analyse);

unsigned system_file_load (struct desc_str *elem);
awk_farcall (system_file_load);

#if DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|E2FS_PROBE|ISOFS_PROBE)
unsigned file_get_chain (struct desc_str *elem, unsigned *nb_fragment, struct sector_chain_str fragment_array[0]);
awk_farcall (file_get_chain);
#endif

#endif /* FS_H */
