/* fs.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"
#if SETUP & XCODE_SEGMENT	/* before other includes */
#define CALLING_FROM_XCODESEG
#endif

#include "instboot.h"
#include "library.h"
#include "boot.h"	/* bootsector_t */
#include "debug.h"
#include "bios.h"	/* _BIOSDISK_reset_alldisks() */
#include "disk.h"
#include "gzlib.h"	/* z_stream for vmlinuz_header_treat() */
#include "fs.h"
#include "util.h"	/* UTIL.HIMEM_entrypoint to init gzlib.next_out */
#include "messages.h"	/* DISK_S_PARTITION_U_FILESYS_BIGGER_PARTITION */

#if USER_SUPPORT != 0
#include "user.h"
#endif
#include "vesabios.h"   /* struct vesa_capabilities_str */
#include "vmlinuz.h"

#if SETUP & XCODE_SEGMENT /* all but for tiny_img */
#define DISK_READSECTOR	DI.readsector
#else
extern unsigned char DISK_readsector (struct diskparam_str *dp, int partition,
			unsigned long long lba, unsigned number, farptr buffer);
#define DISK_READSECTOR	DISK_readsector
#endif

#ifdef COMMANDFILE
#ifndef FS_USE_MALLOC
static char cmdfile_base_buffer[512];
#endif
#endif

/*
 * Stuff used all arround:
 */
struct treat_directory_str {
    unsigned		nb; /* already present in array */
    struct desc_str	*array;
    unsigned char inRoot : 1;
    unsigned char inISO : 1;
    unsigned char which_dir :6;	/* /<boot> = 0, /install.386 = 1, /install.amd = 2, /isolinux = 3, /casper = 4, /live = 5 */
    unsigned char currentNBeltorito, newNBeltorito; /* temporaries increasing while analysing FS and FS iso images */
    unsigned char unused[5];
    union filesystem_union {
	struct common_filesystem {
	    unsigned disk, part;
	    unsigned sector_per_block;
	    unsigned byte_per_block;
	    unsigned first_data_block;
	    unsigned long long blocks_count;
	    unsigned inodes_count;
	    unsigned SectSize; /* different of DI.param[fs->common.disk].bytepersector in emulated disks */
	    struct slashdir { /* all accessed sequencially as an array */
		unsigned inode, size;
		} slash, slashboot, slashinstall386, slashinstallamd64, slashisolinux, slashcasper, slashlive;
	    int (*treatdir) (void *, unsigned char *, unsigned);
	    struct sector_chain_str * (*get_sector_chain) (
		union filesystem_union *fs,
		unsigned inode, unsigned filesize,
		struct sector_chain_str **sector_chain_buffer_ptr);
	    char fsname[64]; /* >= MAX_DISKNAMESIZE */
	    } common;
#if DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE)
	struct {
	    struct common_filesystem common;
	    unsigned       start_fat;
	    unsigned       fat_size;
	    unsigned       start_dir;
	    unsigned short nb_fat;
	    unsigned short fatbit;
	    struct {
		char		name[256];
		unsigned char	sequence;
		unsigned char	chksum;
		} lfn;
	    } fat;
#endif
#if DISK_SUPPORT & E2FS_PROBE
	struct {
	    struct common_filesystem common;
	    unsigned inode_size;	/* usually = 256 or 128 bytes */
	    unsigned inodes_per_group;
	    unsigned sizeof_ext_group_desc;
	    unsigned char use_hi_fields, use_extents;
	    } e2fs;
#endif
#if DISK_SUPPORT & ISOFS_PROBE
	struct {
	    struct common_filesystem common;
	    unsigned volume_sequence_number;
	    unsigned boot_catalog_lba; /* zero if none or reading error */
	    } iso9660;
#endif
	} filesystem;
    };

static const unsigned char dayinmonth[] = { 0, 31,28,31,30,31,30,31,31,30,31,30,31 };

#if DISK_SUPPORT != DOS_SUPPORT
/*
* Filesystem scanning in *.iso file:
*/
FS_FCT_PREFIX(FS_readsector)
static unsigned char FS_readsector (union filesystem_union *fs, unsigned long long lba, unsigned number, farptr buffer)
  {
  signed char deltashift = __builtin_ffs(fs->common.SectSize) - __builtin_ffs(DI.param[fs->common.disk].bytepersector);
  if (fs->common.SectSize == 0)
      /* first call, read in hardware sector */;
    else if (deltashift > 0) {
      lba <<= deltashift;
      number <<= deltashift;
      }
    else if (deltashift != 0) {
      deltashift = -deltashift;
      unsigned char deltamask = (1U << deltashift) - 1;
      if (lba & deltamask) {
	  DBG (("Cannot satisfy reading at LBA %llU\r\n", lba));
	  return 0xF8;
	  }
      if (number & deltamask) {
	  DBG (("Cannot satisfy reading %u sectors\r\n", number));
	  return 0xF8;
	  }
      lba >>= deltashift;
      number >>= deltashift;
      }
#ifndef NB_ISO
  return DISK_READSECTOR(&DI.param[fs->common.disk], fs->common.part, lba, number, buffer);
#else
  if (BOOTWAY.iso.nb_remap == 0)
      return DISK_READSECTOR(&DI.param[fs->common.disk], fs->common.part, lba, number, buffer);
  FDBG (("Remapping read ISO %u sectors from LBA %llu... ", number, lba));

  unsigned long long cur_lba = lba;
  while (number != 0) {
      unsigned segment = 0;
      while (segment < BOOTWAY.iso.nb_remap && cur_lba >= BOOTWAY.iso.remap[segment].nb)
	  cur_lba -= BOOTWAY.iso.remap[segment++].nb;
      if (segment >= BOOTWAY.iso.nb_remap) {
	  FDBG (("remapped fillhole %u sectors at end\r\n", number));
	  fmemset (buffer, 0, number * DI.param[fs->common.disk].bytepersector);
	  return 0;
	  }
      unsigned nbsect = BOOTWAY.iso.remap[segment].nb - cur_lba;
      if (nbsect > number)
	  nbsect = number;
      if (BOOTWAY.iso.remap[segment].lba == 0 && BOOTWAY.iso.nb_remap != 1) { /* CDnoemul: no hole */
	  FDBG (("remapped fillhole %u sectors inside *.iso\r\n", nbsect));
	  fmemset (buffer, 0, nbsect * DI.param[fs->common.disk].bytepersector);
	  }
	else {
	  unsigned long long reallba = BOOTWAY.iso.remap[segment].lba;
	  if (reallba & (1ULL << 63)) {
	      reallba &= ~(1ULL << 63);
	      reallba += ((cur_lba >> BOOTWAY.iso.intergap_shift) << BOOTWAY.iso.gap_sectors_shift);
	      if (cur_lba >> BOOTWAY.iso.intergap_shift != (cur_lba + nbsect) >> BOOTWAY.iso.intergap_shift)
		  nbsect -= (cur_lba + nbsect) & ((1 << BOOTWAY.iso.intergap_shift) - 1);
	      }
	  reallba += cur_lba;
	  FDBG (("remapped read %u sectors at %llu\r\n", nbsect, reallba));
	  unsigned char err = DISK_READSECTOR(&DI.param[fs->common.disk], fs->common.part, reallba, nbsect, buffer);
	  if (err) {
	      FDBG (("failed reading remapped ISO: %u\r\n", err));
	      return err;
	      }
	  }
      buffer += nbsect * DI.param[fs->common.disk].bytepersector;	/* hopes no overflow */
      cur_lba = lba + nbsect;
      number -= nbsect;
      }
  return 0;
#endif
  }
#endif /* DISK_SUPPORT != DOS_SUPPORT */

/*
 * GZLIB stuff & drawing:
 */
FS_FCT_PREFIX(gzlib_extract) static int
gzlib_extract (z_stream *gzlib, unsigned char *buffer, unsigned buflen)
  {bound_stack();{
  static unsigned char cpt;
  char array[3] = { '\010', '#', '\0' };
  int err;

  gzlib->next_in = buffer;
  gzlib->avail_in = buflen;

  if (gzlib->mode == ALLDONE)
      return 0; /* finish the file (DOSFS big block size > 4 Kb) */

  LOADER.gzip_byte_treated += buflen;

  if (gzlib->mode == INIT) {
      cpt = 0;
      print (array);
#ifdef GRUB_MULTIBOOT
      /* Only accept a multiboot header in the first 4Kb (fourKbuffer) instead of "8192 - sizeof(multiboot)", and only if FS block size is 4096 */
#define MULTIBOOT_HEADER_MAGIC	0x1BADB002
      unsigned multiboot_offset, max_multiboot_offset = (gzlib->avail_in >= 4096)? 1024 : (gzlib->avail_in/4);
      for (multiboot_offset = 0; multiboot_offset < max_multiboot_offset; multiboot_offset++)
	  if (((unsigned*)gzlib->next_in)[multiboot_offset] == MULTIBOOT_HEADER_MAGIC && (((unsigned*)gzlib->next_in)[multiboot_offset] + ((unsigned*)gzlib->next_in)[multiboot_offset+1] + ((unsigned*)gzlib->next_in)[multiboot_offset+2]) == 0)
	      break;	/* Magic and checksum OK */
      if (multiboot_offset < max_multiboot_offset) {
	  BOOTWAY.desc[LOADER.curfileload->BOOTWAY_desc_index].boottype = is_multiboot;
	  LOADER.accept_uncompressed_filesize = BOOTWAY.desc[LOADER.curfileload->BOOTWAY_desc_index].filesize;
	  DBG ((" [Change type of file loaded to multiboot and set LOADER.accept_uncompressed_filesize to %u] ", BOOTWAY.desc[LOADER.curfileload->BOOTWAY_desc_index].filesize));
	} else
#endif
#if defined (LINUZNAME1) || defined (LINUZNAME2) || defined (LINUZNAME3)
      if (gzlib->crc32 && *(unsigned short *)gzlib->next_in != 0x8B1F) { /* tiny_exe */
	  if ((err = vmlinuz_header_treat ((struct linux_param *)gzlib->next_in, gzlib->avail_in)) != 0)
	      return 0xB0 | err;
	  }
#endif
      }

#ifndef TEST_BYTE_PER_BYTE_INFLATE
  array[1] = "|/-\\"[cpt++ % (sizeof ("|/-\\") - 1)];
  print (array);

  err = inflate (gzlib);
  if (err == Z_STREAM_END) {
      DBG ((" [inflate returns Z_STREAM_END, gzlib->avail_in = %u] ",
			gzlib->avail_in));
      LOADER.gzip_byte_treated -= gzlib->avail_in;
      if (gzlib->avail_in >= 2) {
	  DBG ((" [unsigned short after compressed data: 0x%X] ",
			*(unsigned short *)gzlib->next_in));
	  /* FIXME: if we are at end of a 4 Kb block... */
	  if (*(unsigned short *)gzlib->next_in == 0x8B1F)
	      LOADER.initrd_index = -1;
	  }
#if USER_SUPPORT != 0 /* not tiny_img nor tiny_exe */
      if (LOADER.license_compliant == 1 || LOADER.license_compliant == 2)
	  print ("    ");
	else
#endif
	  print ("\010, ");
      return 0;
      }
    else if (err != Z_OK) {
      FDBG (("inflate returns %d!\r\n", err));
      array[1] = '!';
      print (array);
      return 0x90 | err;
      }
    else if (gzlib->avail_in != 0) {
      FDBG (("inflate returns with non empty input!\r\n"));
      array[1] = '?';
      print (array);
      return 0x9F;
      }
  return 0;
#else /* TEST_BYTE_PER_BYTE_INFLATE */
  {
  unsigned avail_in = gzlib->avail_in;

  gzlib->avail_in = 1;

  for (;;) {
      array[1] = "|/-\\"[cpt++ % (sizeof ("|/-\\") - 1)];
      print (array);

      err = inflate (gzlib);
      if (err == Z_STREAM_END) {
	  DBG ((" [inflate returns Z_STREAM_END, gzlib->avail_in = %u] ",
			gzlib->avail_in));
	  if (gzlib->avail_in >= 2) {
	      DBG ((" [unsigned short after compressed data: 0x%X] ",
			*(unsigned short *)gzlib->next_in));
	      if (*(unsigned short *)gzlib->next_in == 0x8B1F)
		  LOADER.initrd_index = -1;
	      }
	  print ("\010, ");
	  return 0;
	  }
	else if (err != Z_OK) {
	  FDBG (("inflate returns %d!\r\n", err));
	  array[1] = '!';
	  print (array);
	  return 0x90 | err;
	  }

      if (gzlib->avail_in == 0) {
	  if (--avail_in == 0)
	      return 0;
	  gzlib->avail_in = 1;
	  }
      }
  }
#endif /* TEST_BYTE_PER_BYTE_INFLATE */
  }}

#if DISK_SUPPORT & (E2FS_PROBE|FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|ISOFS_PROBE)
extern inline unsigned extension_equal (const char *test, const char *ext_lowercase, const char *ext_uppercase)
  {
  if (*(const unsigned int *)test == *(const unsigned int *)ext_lowercase)
      return 1;
  if (*(const unsigned int *)test == *(const unsigned int *)ext_uppercase)
      return 1;
  return 0;
  }

FS_FCT_PREFIX(file_check_aa55_sig) static unsigned
file_check_aa55_sig (union filesystem_union *fs, unsigned inode);

#ifdef COMMANDFILE
FS_FCT_PREFIX(file_treat_commandfile) static unsigned
file_treat_commandfile (union filesystem_union *fs, unsigned inode, unsigned filesize);
#endif

static const char *const SearchPathArray[5] = { ALT1SEARCHPATH, ALT2SEARCHPATH, ALT3SEARCHPATH, ALT4SEARCHPATH, ALT5SEARCHPATH };

FS_FCT_PREFIX(tst_elem) static enum {
    tst_elem_found,
    tst_elem_slashboot,
    tst_elem_slashinstall386,
    tst_elem_slashinstallamd64,
    tst_elem_slashisolinux,
    tst_elem_slashcasper,
    tst_elem_slashlive,
    tst_elem_iso,
    tst_elem_ignore,
    tst_elem_maybe,
    tst_elem_commandfile
    }
tst_elem (struct treat_directory_str *opaque,
	  const char *name, /*: maybe not zero terminated */
	  unsigned name_len)
  {
  struct desc_str *elem = &opaque->array[opaque->nb];
  char *end = elem->filename, *semicolon = 0;

  *elem = (struct desc_str) {0};
  elem->name_offset = 0;
  elem->disk = opaque->filesystem.common.disk;
  elem->partition = opaque->filesystem.common.part;
  elem->inRoot = opaque->inRoot;
  elem->inISO = opaque->inISO;
  elem->searchpath = opaque->which_dir;

#ifdef NB_ISO
  if (opaque->inISO) {
      extern char int13filename[64];
      while (elem->name_offset <= sizeof(int13filename))
	  if ((*end++ = BOOTWAY.iso.found[BOOTWAY.iso.current_used].filename[elem->name_offset++]) == '\0')
	      break;
      end[-1] = ':';
      }
#endif

  *end++ = '/';
  elem->name_offset++;
  if (!opaque->inRoot) {
      const char *path = opaque->which_dir? SearchPathArray[opaque->which_dir - 1] : copy_gujin_param.scanpath;
      while ((*end++ = *path++) != 0)
	  elem->name_offset++;
      end[-1] = '/';
      elem->name_offset++;
      }
  while (name_len-- && end < elem->filename + sizeof (elem->filename) - 2) {
      if (*name == ';')
	  semicolon = end;
      else if (*name < '0' || *name > '9')
	  semicolon = 0;
      *end++ = *name++;
      }
  *end = '\0';
  name = elem->filename + elem->name_offset; /* now zero ended */
  if (semicolon) {
	/* Remove the version part of the filename (after and including semicolon)
		whatever the filesystem - really needed only for ISO9660 */
	/* remove the last '.' of a filename if it finnishes by a dot: */
      if (*name != '.' && *(semicolon - 1) == '.')
	  semicolon--;
      FDBG ((" [suppress '%s' from '%s'] ", semicolon, name));
      end = semicolon;
      *end = '\0';
      }
    else if (*name != '.' && *(end - 1) == '.')
      *--end = '\0';

  if (opaque->inRoot) {
      /* Following test is _not_ case sensitive for FAT filesystems;
	* on E2FS, if you have directories "/boot" and later "/Boot",
	* it will keep the last one... not sure that is really usefull. */
      if (copy_gujin_param.attrib.search_subdir_files) {
	  unsigned cpt;
	  const char *compared = copy_gujin_param.scanpath;

	  /* assert (tst_elem_slashlive == tst_elem_slashboot + sizeof(SearchPathArray)/sizeof(SearchPathArray[0]) + 1); */
	  for (cpt = tst_elem_slashboot; cpt <= tst_elem_slashlive; cpt++) {
	      const char *nameptr = name;
	      for (;;) {
		  char pathchar = *compared++, test = *nameptr++;
		  if (pathchar >= 'A' && pathchar <= 'Z')
		      pathchar +=  'a' - 'A';
		  if (test >= 'A' && test <= 'Z')
		      test +=  'a' - 'A';
		  if (pathchar != test)
		      break;
		  if (pathchar == '\0') {
		      FDBG ((" [found sub directory '%s'] ", elem->filename));
		      return cpt;
		      }
		  }
	      compared = SearchPathArray[cpt - tst_elem_slashboot];
	      }
	  }
      if (!copy_gujin_param.attrib.search_topdir_files) {
	  /* Just search TheScanPath directory, ignore all files */
	  FDBG ((" [ignoring topdir file/directory '%s'] ", elem->filename));
	  return tst_elem_ignore;
	  }
      }

#ifdef COMMANDFILE
  if (!stricmp (COMMANDFILE, name))
      return tst_elem_commandfile;
#endif

#ifdef NB_ISO
  elem->iso_filesize = elem->iso_inode = 0;
  elem->iso_fsname = 0;
  if (opaque->inISO) {
      elem->iso_filesize = BOOTWAY.iso.found[BOOTWAY.iso.current_used].size;
      elem->iso_inode = BOOTWAY.iso.found[BOOTWAY.iso.current_used].inode;
      } /* Do not probe ISO files in ISO images: */
    else if ((copy_gujin_param.attrib.probe_file_in_iso_image || copy_gujin_param.attrib.search_el_torito)
	&& extension_equal (end - 4, ISOEXT, ISOEXT_UPPERCASE)) {
      FDBG ((" [is ISO disk image: '%s'] ", elem->filename));
      return tst_elem_iso;
      }
#endif

#ifdef KERNELEXT
  if (extension_equal (end - 4, KERNELEXT, KERNELEXT_UPPERCASE)) {
      while (elem->filename[elem->name_offset] != '\0' && elem->filename[elem->name_offset] != '-'
		&& elem->filename[elem->name_offset] != '+')
	  elem->name_offset++;
      if (elem->filename[elem->name_offset] != '-')
	  elem->name_offset = 0;
      elem->boottype = is_elf;
      FDBG ((" [is ELF: '%s'] ", elem->filename));
      return tst_elem_found;
      }
#endif

#ifdef ALTMBRNAME
  if (!BOOT1_DOS_running() && !stricmp (ALTMBRNAME, name)) {
      elem->boottype = is_PBR;
      elem->name_offset = 1; /* if >= 1 it is MBR/PBR in a file */
      FDBG ((" [is PBR: '%s'] ", elem->filename));
      return tst_elem_found;
      }
#endif

#ifdef INITRDNAME
  if (!strnicmp (INITRDNAME, name, strlen(INITRDNAME))) {
      elem->name_offset += strlen(INITRDNAME);
      elem->boottype = is_initrd;
      FDBG ((" [is initrd: '%s'] ", elem->filename));
      return tst_elem_found;
      }
#endif

#ifdef INITRDNAME2
  if (!strnicmp (INITRDNAME2, name, strlen(INITRDNAME2))) {
      elem->name_offset += strlen(INITRDNAME2);
      elem->boottype = is_initrd;
      FDBG ((" [is initrd: '%s'] ", elem->filename));
      return tst_elem_found;
      }
#endif

#ifdef INITRAMFSNAME
  if (!strnicmp (INITRAMFSNAME, name, strlen(INITRAMFSNAME))) {
      elem->name_offset += strlen(INITRAMFSNAME);
      elem->boottype = is_initramfs;
      FDBG ((" [is initramfs: '%s'] ", elem->filename));
      return tst_elem_found;
      }
#endif

#ifdef LINUZNAME1
  if (!strnicmp (LINUZNAME1, name, strlen(LINUZNAME1))) {
      elem->name_offset += strlen(LINUZNAME1);
      elem->boottype = is_linux;
      FDBG ((" [is %s: '%s'] ", LINUZNAME1, elem->filename));
      return tst_elem_found;
      }
#endif

#ifdef LINUZNAME2
  if (!strnicmp (LINUZNAME2, name, strlen(LINUZNAME2))) {
      elem->name_offset += strlen(LINUZNAME2);
      elem->boottype = is_linux;
      FDBG ((" [is %s: '%s'] ", LINUZNAME2, elem->filename));
      return tst_elem_found;
      }
#endif

#ifdef LINUZNAME3 /* LinuxMint-8.iso is not a kernel */
  if (!strnicmp (LINUZNAME3, name, strlen(LINUZNAME3))
	&& !extension_equal (end - 4, ISOEXT, ISOEXT_UPPERCASE)) {
      elem->name_offset += strlen(LINUZNAME3);
      elem->boottype = is_linux;
      FDBG ((" [is %s: '%s'] ", LINUZNAME3, elem->filename));
      return tst_elem_found;
      }
#endif

#ifdef BOOTDEVICEIMAGE /* take care of initrd.img */
  if (!BOOT1_DOS_running() && copy_gujin_param.attrib.probe_dos_disk) {	/* is probe_bdi_file in BIOS environment */
      if (extension_equal (end - 4, BOOTDEVICEIMAGE, BOOTDEVICEIMAGE_UPPERCASE)) {
	  elem->boottype = is_bdi_file;
	  elem->name_offset = 0; /* if >= 1 it is el-torito */
	  FDBG ((" [is BDI: '%s'] ", elem->filename));
	  return tst_elem_found;
	  }
      if (extension_equal (end - 4, BOOTDEVICEIMAGE_MAYBE1, BOOTDEVICEIMAGE_MAYBE1_UPPERCASE)
	|| extension_equal (end - 4, BOOTDEVICEIMAGE_MAYBE2, BOOTDEVICEIMAGE_MAYBE2_UPPERCASE)) {
	  elem->boottype = is_bdi_file;
	  elem->name_offset = 0; /* if >= 1 it is el-torito */
	  FDBG ((" [is maybe BDI: '%s'] ", elem->filename));
	  return tst_elem_maybe;
	  }
      }
#endif

  FDBG ((" [ignore: '%s' i.e. '%s'] ", elem->filename, name));
  return tst_elem_ignore;
  }
#endif

/**
 ** E2FS filesystem:
 **/

#if DISK_SUPPORT & E2FS_PROBE
#include "e2fs.h"

FS_FCT_PREFIX(E2FS_get_parameter) static inline unsigned
E2FS_get_parameter (union filesystem_union *fs, const unsigned char *first4Kbyte, unsigned long long device_size, unsigned dp_bytepersector)
  {
  /* FIXME: Do we want to interpret SUPERBLOCK_OFFSET as 1024 bytes or 2 sectors for CDROMs?
	i.e. do we want 1024/512 and 1024/2048 or only 1024 bytes offset? */
  const struct ext_super_block *super_block = (struct ext_super_block *)(first4Kbyte + SUPERBLOCK_OFFSET);

  if (sizeof (*super_block) != SUPERBLOCK_SIZE)
      __ERROR();
  if (SUPERBLOCK_OFFSET + SUPERBLOCK_SIZE > DEFAULT_BUFFER_SIZE) {
      FDBG (("[E2FS DEFAULT_BUFFER_SIZE=%u too small] ", DEFAULT_BUFFER_SIZE));
      return 0x0E;
      }

  if (super_block->s_magic != EXT2_SUPER_MAGIC) {
      /* Wrong Ident-i-Eeze */
      FDBG (("%s: error superblock bad magic: is 0x%X, should be 0x%X.\r\n",
		__FUNCTION__, super_block->s_magic, EXT2_SUPER_MAGIC));
      return 0x1;
      }
  if (super_block->s_rev_level > EXT2_MAX_SUPP_REV) {
      FDBG (("%s: error too high superblock revision level: is 0x%X, max 0x%X.\r\n",
		__FUNCTION__, super_block->s_rev_level, EXT2_MAX_SUPP_REV));
      return 0x2;
      }
  if (super_block->s_feature_incompat.reserved || super_block->s_feature_incompat.unknown) {
      FDBG (("%s: error incompatible field 'reserved' 0x%X.\r\n", __FUNCTION__, *CAST(unsigned *, &super_block->s_feature_incompat)));
      return 0x3;
      }
// I do handle bigger block size:
//  if ((EXT2_MIN_BLOCK_SIZE << super_block->s_log_block_size) > EXT2_MAX_BLOCK_SIZE) {
  if ((EXT2_MIN_BLOCK_SIZE << super_block->s_log_block_size) > 64 * 1024) {
      FDBG (("%s: error invalid super_block->s_log_block_size = %u.\r\n",
		__FUNCTION__, super_block->s_log_block_size));
      return 0x4;
      }
  if (super_block->s_blocks_per_group == 0) {
      FDBG (("%s: error super_block->s_blocks_per_group == 0.\r\n", __FUNCTION__));
      return 0x5;
      }

  if (super_block->s_rev_level == EXT2_GOOD_OLD_REV)
      fs->e2fs.inode_size = EXT2_GOOD_OLD_INODE_SIZE;
    else
      fs->e2fs.inode_size = super_block->s_inode_size;
  if (fs->e2fs.inode_size < offsetof (struct ext_inode, osd2.linux2.l_i_blocks_high)) {
      FDBG (("%s: inode size %u too small.\r\n", __FUNCTION__, super_block->s_inode_size));
      return 0x6;
      }

//  FDBG ((" (s_feature_compat: 0x%X, s_feature_incompat: 0x%X, s_feature_ro_compat 0x%X) .\r\n",
//		*(unsigned *)&super_block->s_feature_compat,		// = 0
//		*(unsigned *)&super_block->s_feature_incompat,		// = 2
//		*(unsigned *)&super_block->s_feature_ro_compat));	// = 1

// I do not use the filetype: struct ext_dir_entry->file_type
//  if (super_block->s_feature_incompat.filetype) {
//      FDBG (("%s: error incompatible field 'filetype'.\r\n", __FUNCTION__));
//      return 0x;
//      }

// Being read-only, I do not care if I cannot access the journal device:
//  if (super_block->s_feature_incompat.journal_dev) {
//      FDBG (("%s: error incompatible field 'journal_dev'.\r\n", __FUNCTION__));
//      return 0x8;
//      }

// Even if a filesystem check is needed, I try to find my files:
//  if (super_block->s_feature_incompat.recover) { /* Needs recovery */
//      FDBG (("%s: error incompatible field 'recover'.\r\n", __FUNCTION__));
//      return 0x9;
//      }

// I do not really know about this one:
//  if (super_block->s_feature_incompat.compression) {
//      FDBG (("%s: error incompatible field 'compression'.\r\n", __FUNCTION__));
//      return 0xA;
//      }

// If I correctly understand, I do not care where are the unused inodes
//  if (super_block->s_feature_incompat.meta_bg) {
//      FDBG (("%s: error incompatible field 'meta_bg'.\r\n", __FUNCTION__));
//      return 0xB;
//      }

// I do support that:
//  if (super_block->s_feature_incompat.extents) {
//      FDBG (("%s: error incompatible field 'extents'.\r\n", __FUNCTION__));
//      return 0xC;
//      }

// I do support that:
//  if (super_block->s_feature_incompat.bit64) {
//      FDBG (("%s: error incompatible field 'bit64'.\r\n", __FUNCTION__));
//      return 0xD;
//      }
// but not:
  if (super_block->s_feature_incompat.bit64 && (super_block->s_min_extra_isize || super_block->s_want_extra_isize)) {
      FDBG (("%s: error incompatible fields s_min_extra_isize %u, s_want_extra_isize %u, not supported (why would you want it?).\r\n",
		__FUNCTION__, super_block->s_min_extra_isize, super_block->s_want_extra_isize));
      return 0xD;
      }

  if (super_block->s_feature_incompat.mmp) {
      FDBG (("%s: error incompatible field 'mmp'.\r\n", __FUNCTION__));
      return 0xE;
      }

// If I correctly understand, I do not care where are the unused inodes
//  if (super_block->s_feature_incompat.flex_bg) {
//      FDBG (("%s: error incompatible field 'flex_bg'.\r\n", __FUNCTION__));
//      return 0xF;
//      }

  fs->common.byte_per_block = EXT2_MIN_BLOCK_SIZE << super_block->s_log_block_size;
  fs->e2fs.use_hi_fields = super_block->s_feature_incompat.bit64;
  fs->e2fs.use_extents = super_block->s_feature_incompat.extents;
  fs->common.blocks_count = super_block->s_blocks_count_lo;
  if (fs->e2fs.use_hi_fields)
      fs->common.blocks_count += 0x100000000ULL * super_block->s_blocks_count_hi;
  if (fs->common.blocks_count < super_block->s_first_data_block) {
      FDBG (("%s: error super_block->s_blocks_count = %llU, super_block->s_first_data_block = %u.\r\n",
		__FUNCTION__, fs->common.blocks_count, super_block->s_first_data_block));
      return 0x7;
      }
#if 0 /* For e2fs on Linux, the sector size deduced from the e2fs superblock is always 512
	even if we create an e2fs on a DVD-RAM */
  /* try to get the SectSize based on device_size OR size of filesystem image: */
  unsigned remainder, nb_device_blocks = ull_div_ul (device_size, fs->common.byte_per_block, &remainder);
  unsigned long long try_blocks_count = fs->common.blocks_count + (fs->common.blocks_count >> 8);
  unsigned relation = ull_div_ul (try_blocks_count, nb_device_blocks, &remainder);
  if (relation == 2)
      fs->common.SectSize = 1024;
    else if (relation == 4)
      fs->common.SectSize = 2048;
    else if (relation == 8)
      fs->common.SectSize = 4096;
    else
      fs->common.SectSize = 512;
  FDBG(("%s: considering device_size 0x%llX blocks_count 0x%llX, relation %u so SectSize %u\r\n", __FUNCTION__, device_size, fs->common.blocks_count, relation, fs->common.SectSize));
  if (fs->common.SectSize == 512 && BOOTWAY.iso.nb_remap == 0 && dp_bytepersector != 512) {
      FDBG(("%s: handling case of an e2fs generated on a whole DVD-RAM\r\n", __FUNCTION__));
      fs->common.SectSize = dp_bytepersector;
      }
#else
  /* We want the real sector size: */
  if (BOOTWAY.iso.nb_remap == 0)
      fs->common.SectSize = dp_bytepersector;
    else
      fs->common.SectSize = 512;
#endif
  fs->common.sector_per_block = fs->common.byte_per_block / fs->common.SectSize;

  fs->common.first_data_block = super_block->s_first_data_block;
  fs->common.inodes_count = super_block->s_inodes_count;
  fs->common.slash.inode= EXT2_ROOT_INO;
  fs->common.slash.size = 0; /* unused */
  fs->e2fs.sizeof_ext_group_desc = sizeof(struct {
    __u32	bg_block_bitmap;	/* Blocks bitmap block */
    __u32	bg_inode_bitmap;	/* Inodes bitmap block */
    __u32	bg_inode_table;		/* Inodes table block */
    __u16	bg_free_blocks_count;	/* Free blocks count */
    __u16	bg_free_inodes_count;	/* Free inodes count */
    __u16	bg_used_dirs_count;	/* Directories count */
    __u16	bg_pad;
    __u32	bg_reserved[3];
    });
  if (super_block->s_desc_size > fs->e2fs.sizeof_ext_group_desc)
      fs->e2fs.sizeof_ext_group_desc = super_block->s_desc_size;

  fs->e2fs.inodes_per_group = super_block->s_inodes_per_group;

  if (super_block->s_volume_name[0] != '\0') {
      char *dst = fs->common.fsname;
      const char *src = super_block->s_volume_name;
      while (*src && dst < &fs->common.fsname[lastof(fs->common.fsname)])
	  *dst++ = (char)*src++;
      *dst = '\0';
      FDBG (("%s: Filesystem name: '%s' ", __FUNCTION__, fs->common.fsname));
      }
    else
      fs->common.fsname[0] = '\0';
  /* fs->super->s_last_mounted not usually initialised */

  /* FIXME: Check for super_block->s_min_extra_isize &&  super_block->s_want_extra_isize ? */
  FDBG (("Filesystem opened (inode size %u, inodes_per_group %u).\r\n", fs->e2fs.inode_size, fs->e2fs.inodes_per_group));
  return 0;
  }

FS_FCT_PREFIX(E2FS_get_group_desc) static inline struct ext_group_desc *
E2FS_get_group_desc (union filesystem_union *fs, unsigned char shift_bps,
		unsigned sector_per_block, unsigned first_data_block, unsigned sizeof_ext_group_desc,
		unsigned group, unsigned char *buffer)
  {
  unsigned long long tmp = group * sizeof_ext_group_desc;
  unsigned long long sector_nr = tmp >> shift_bps;
  unsigned index = tmp & ((1U << shift_bps) - 1);

  /* I HATE this +1, shall be "DIVROUNDUP(SUPERBLOCK_OFFSET+SUPERBLOCK_SIZE, byte_per_block)" */
  sector_nr += (first_data_block + 1) * sector_per_block;

  if (sector_nr >= DI.param[fs->common.disk].partition[fs->common.part].length) {
      FDBG ((" [group block over partition limit: %llu, max %llu] ", sector_nr, DI.param[fs->common.disk].partition[fs->common.part].length));
      /* return 0; will be detected by FS_readsector() */
      }
  /* if crossing limit (impossible with 32), need to read 2: */
  unsigned nb_sector_to_read = (index + sizeof_ext_group_desc > (1U << shift_bps))? 2 : 1;
  unsigned char err = FS_readsector (fs, sector_nr, nb_sector_to_read, stack_adr (buffer));
  if (err) {
      FDBG ((" [error 0x%X reading group block] ", err));
      return 0;
      }

  return (struct ext_group_desc *)&buffer[index];
  }

FS_FCT_PREFIX(E2FS_get_inode_ptr) static inline struct ext_inode *
E2FS_get_inode_ptr (union filesystem_union *fs, unsigned char shift_bps,
		unsigned sector_per_block, unsigned inode_size, unsigned long long bg_inode_table,
		unsigned block, unsigned char *buffer)
  {
  /* FIXME: if variable inode size, tmp will be incorrect here: */
  unsigned long long tmp = block * inode_size;
  unsigned long long sector_nr = tmp >> shift_bps;
  unsigned index = tmp & ((1U << shift_bps) - 1);

  sector_nr += bg_inode_table * sector_per_block;

  if (sector_nr >= DI.param[fs->common.disk].partition[fs->common.part].length) {
      FDBG ((" [inode over partition limit: %llu, max %llu] ", sector_nr, DI.param[fs->common.disk].partition[fs->common.part].length));
      /* return 0; will be detected by FS_readsector() */
      }

  /* if crossing limit (impossible with 256 or 128), need to read 2: */
  unsigned nb_sector_to_read = (index + inode_size > (1U << shift_bps))? 2 : 1;
  unsigned char err = FS_readsector (fs, sector_nr, nb_sector_to_read, stack_adr (buffer));
  if (err) {
      FDBG ((" [error 0x%X reading inode block] ", err));
      return 0;
      }
  /* FIXME: Check for ((struct ext_inode *)&buffer[index]).i_extra_isize ? */
  return (struct ext_inode *)&buffer[index];
  }

/*
 * This function is also called when scanning a directory,
 * so we cannot use the "fourKbuffer" which contains the
 * directory content.
 */
FS_FCT_PREFIX(E2FS_read_inode) static unsigned
E2FS_read_inode (union filesystem_union *fs, unsigned inode, struct ext_inode *struct_inode)
  {bound_stack();{
  unsigned char tmpbuffer[2U * fs->common.SectSize];
  unsigned char ffs_sectsize = __builtin_ffs (fs->common.SectSize) - 1;
  unsigned group = (inode - 1) / fs->e2fs.inodes_per_group;
  unsigned block = (inode - 1) % fs->e2fs.inodes_per_group;

  FDBG ((" [%s: inode %u, group %u, block %u] ", __FUNCTION__, inode, group, block));

  struct ext_group_desc *group_desc = E2FS_get_group_desc (fs, ffs_sectsize,
		fs->common.sector_per_block, fs->common.first_data_block, fs->e2fs.sizeof_ext_group_desc,
		group, tmpbuffer);

  if (!group_desc) {
      FDBG ((" [error getting group descriptor] "));
      return 0x1;
      }
//  FDBG (("[bg_inode_table_hi = %u, bg_inode_table_lo = %u] ", group_desc->bg_inode_table_hi, group_desc->bg_inode_table_lo));
  unsigned long long bg_inode_table = group_desc->bg_inode_table_lo;
  if (fs->e2fs.use_hi_fields && fs->e2fs.sizeof_ext_group_desc > offsetof (struct ext_group_desc, bg_inode_table_hi))
      bg_inode_table += 0x100000000ULL * group_desc->bg_inode_table_hi;

  if (bg_inode_table == 0) {
      FDBG ((" [invalid block descriptor] "));
      return 0x2;
      }

  struct ext_inode *inode_ptr = E2FS_get_inode_ptr (fs, ffs_sectsize,
		fs->common.sector_per_block, fs->e2fs.inode_size, bg_inode_table, block, tmpbuffer);

  if (!inode_ptr) {
      FDBG ((" [error getting inode] "));
      return 0x3;
      }

  *struct_inode = *inode_ptr;
  return 0;
  }}

FS_FCT_PREFIX(E2FS_add_sector_chain) static inline unsigned
E2FS_add_sector_chain (unsigned *start, unsigned *end,
			struct sector_chain_str **sector_chain_buffer_ptr,
			unsigned *current,
			const unsigned long long blocks_count,
			const unsigned Sectorperblock)
  {
  struct sector_chain_str *sector_chain_buffer = *sector_chain_buffer_ptr;

  while (start < end) {
      unsigned long long lba;

      if (*start >= blocks_count) {
	  FDBG (("%s: block %u too high, max %llU!\r\n", __FUNCTION__, *start, blocks_count));
	  return 0x1;
	  }

      lba = *start * Sectorperblock;
      if (*current == 0xFFFFFFFE) {
	  if (lba == 0)
	      FDBG (("[create , start with hole] "));
	    else
	      FDBG (("[create new segment, lba %llU] ", lba));
	  *current = 0;
	  sector_chain_buffer[*current].lba = lba;
	  sector_chain_buffer[*current].nb = Sectorperblock;
//	  sector_chain_buffer[*current + 1].lba = 0;
//	  sector_chain_buffer[*current + 1].nb = 0;
	  }
	else if (lba == 0 && sector_chain_buffer[*current].lba == 0) {
	  sector_chain_buffer[*current].nb += Sectorperblock;
	  FDBG (("[++hole, %u blocks] ", sector_chain_buffer[*current].nb / Sectorperblock));
	  }
#ifdef NB_ISO
	else if ((sector_chain_buffer[*current].lba & (1ULL << 63))
		&& lba == (sector_chain_buffer[*current].lba & ~ (1ULL << 63)) + sector_chain_buffer[*current].nb
			+ ((sector_chain_buffer[*current].nb >> BOOTWAY.iso.intergap_shift) << BOOTWAY.iso.gap_sectors_shift)) {
//	  FDBG ((" [add LBA %llu] ", lba & ~ (1ULL << 63)));	//	too many messages for ISO
	  sector_chain_buffer[*current].nb += Sectorperblock;
	  }
	else if (!(sector_chain_buffer[*current].lba & (1ULL << 63))
		&& (sector_chain_buffer[*current].nb >> BOOTWAY.iso.intergap_shift) != 0
		&& lba == sector_chain_buffer[*current].lba + sector_chain_buffer[*current].nb
			+ ((sector_chain_buffer[*current].nb >> BOOTWAY.iso.intergap_shift) << BOOTWAY.iso.gap_sectors_shift)) {
	  FDBG ((" [initiate taged LBA %llu] ", lba));
	  sector_chain_buffer[*current].lba |=  (1ULL << 63);
	  sector_chain_buffer[*current].nb += Sectorperblock;
	  }
	else if (!(sector_chain_buffer[*current].lba &  (1ULL << 63))
		&& lba == sector_chain_buffer[*current].lba + sector_chain_buffer[*current].nb) {
#else
	else if (lba == sector_chain_buffer[*current].lba + sector_chain_buffer[*current].nb) {
#endif
//	  FDBG ((" [add lba %llu] ", lba));	//	too many messages for ISO
	  sector_chain_buffer[*current].nb += Sectorperblock;
	  }
	else {
#ifdef FS_USE_MALLOC
	  sector_chain_buffer = REALLOC (sector_chain_buffer, (*current+2+1) * sizeof (struct sector_chain_str), "chainbuf");
	  if (sector_chain_buffer == 0) {
	      FDBG (("%s: realloc failed!\r\n", __FUNCTION__));
	      return 0x3;
	      }
	  (*sector_chain_buffer_ptr) = sector_chain_buffer;
#else
	  if (*current >= FILE_MAX_FRAGMENT) {
	      FDBG (("%s: file has too many fragments!\r\n", __FUNCTION__));
	      return 0x4;
	      }
#endif
	  if (lba != 0) { /* accept for multiple hole in the same file */
	      struct sector_chain_str *ptr;
	      for (ptr = sector_chain_buffer;
		   ptr <= &sector_chain_buffer[*current];
		   ptr++) {
		  if (ptr->lba == lba) {
		      FDBG (("%s: loop detected!\r\n", __FUNCTION__));
		      return 0x5;
		      }
		  }
	      }
	  if (lba == 0)
	      FDBG ((" [append hole of %u sectors] ", Sectorperblock));
	    else
	      FDBG ((" [append for lba %llu] ", lba));
	  *current += 1;
	  sector_chain_buffer[*current].lba = lba;
	  sector_chain_buffer[*current].nb = Sectorperblock;
//	  sector_chain_buffer[*current + 1].lba = 0;
//	  sector_chain_buffer[*current + 1].nb = 0;
	  }
      start++;
      }
  if (*current == 0xFFFFFFFE)
      FDBG ((" [no block here for this file] "));
  return 0;
  }

union blocknr_inode_u {
    struct ext_inode *inode;
    unsigned blocknr; /* only used with read_analyse_chain(), not with analyse_extent_array(), so 32 bits sufficient */
    };

FS_FCT_PREFIX(read_analyse_chain) static unsigned
read_analyse_chain (union filesystem_union *fs, unsigned level,
			union blocknr_inode_u blocknr_inode, unsigned *nb_file_block,
			struct sector_chain_str **sector_chain_buffer_ptr, unsigned *current)
  {bound_stack();{
  unsigned blockbuffer[(level == 0) ? 0 : fs->common.byte_per_block / sizeof(unsigned)];
//  unsigned blockbuffer[/*(level == 0) ? 0 :*/ fs->common.byte_per_block / sizeof(unsigned)]; // saves 32 bytes of code
  unsigned char *buffer, *end_buffer;
  unsigned buflen, cpt, nbtimes = (level <= 1)? 1 : nbof (blockbuffer);

  if (level == 0) {
      buffer = (unsigned char *)blocknr_inode.inode->blocks.std_block.i_direct_block;
      buflen = sizeof (blocknr_inode.inode->blocks.std_block.i_direct_block);
      }
    else {
      buffer = (unsigned char *)blockbuffer;
      buflen = sizeof (blockbuffer);

      if (blocknr_inode.blocknr == 0) {
	  FDBG ((" [%s: indirect blocknr == 0, level %u] ",
		__FUNCTION__, level));
	  return 0;
	  }
      if (blocknr_inode.blocknr >= fs->common.blocks_count) {
	  FDBG (("%s: indirect blocknr %u too high, max %llU!\r\n",
		__FUNCTION__, blocknr_inode.blocknr, fs->common.blocks_count));
	  return 0xD;
	  }
      unsigned nbsect = fs->common.sector_per_block;
      unsigned long long iblock_lba = blocknr_inode.blocknr * nbsect;
      if (iblock_lba >= DI.param[fs->common.disk].partition[fs->common.part].length) {
	  FDBG (("%s: indirect blocknr lba %llu out of partition, max %llu!\r\n",
		__FUNCTION__, iblock_lba, DI.param[fs->common.disk].partition[fs->common.part].length));
	  /* return 0xE; will be detected by FS_readsector() */
	  }
      FDBG ((" [%s: reading indirect block at lba %llu of part %u] ", __FUNCTION__, iblock_lba, fs->common.part));
      unsigned char err = FS_readsector (fs, iblock_lba, 1 * nbsect, stack_adr (blockbuffer));

      if (err) {
	  FDBG (("%s: error 0x%X reading indirect block.\r\n",
		__FUNCTION__, err));
	  return 0xF;
	  }
      }

  for (cpt = 0; cpt < nbtimes; cpt++) {
      unsigned tmp;

      if (level > 1) {
	  union blocknr_inode_u nr = { .blocknr = blockbuffer[cpt] };
	  tmp = read_analyse_chain (fs, level - 1, nr, nb_file_block, sector_chain_buffer_ptr, current);
	  *nb_file_block -= 1;
	  }
	else {
	  if (*nb_file_block >= buflen / sizeof (blocknr_inode.inode->blocks.std_block.i_direct_block[0])) {
	      *nb_file_block -= buflen / sizeof (blocknr_inode.inode->blocks.std_block.i_direct_block[0]);
	      end_buffer = &buffer[buflen];
	      }
	    else {
	      end_buffer = &buffer[*nb_file_block * sizeof (blocknr_inode.inode->blocks.std_block.i_direct_block[0])];
	      *nb_file_block = 0;
	      }
	  tmp = E2FS_add_sector_chain ((unsigned *)buffer, (unsigned *)end_buffer,
		sector_chain_buffer_ptr, current, fs->common.blocks_count, fs->common.sector_per_block);
	  }

      if (tmp) {
	  FDBG (("%s: [level=%u, cpt=%u, nb_file_block %u] error 0x%X.\r\n",
		__FUNCTION__, level, cpt, *nb_file_block, tmp));
	  return tmp;
	  }
      }
  return 0;
  }}

FS_FCT_PREFIX(analyse_extent_array) static unsigned
analyse_extent_array (union filesystem_union *fs, struct extent_block_s *extent_block,
			struct sector_chain_str **sector_chain_buffer_ptr, unsigned *current)
  {bound_stack();{
  if (extent_block->header.eh_magic != EXT4_EXT_MAGIC) {
      FDBG (("ERROR extent with wrong eh_magic= 0x%X != 0x%X\r\n", extent_block->header.eh_magic, EXT4_EXT_MAGIC));
      return 1;
      }
  FDBG (("\r\nEXTENT: eh_magic 0x%X eh_entries %u eh_max %u eh_depth %u eh_generation %u\r\n",
	extent_block->header.eh_magic, extent_block->header.eh_entries, extent_block->header.eh_max,
	extent_block->header.eh_depth, extent_block->header.eh_generation));
  if (extent_block->header.eh_entries > extent_block->header.eh_max) {
      FDBG (("ERROR extent eh_entries %u > eh_max %u\r\n", extent_block->header.eh_entries, extent_block->header.eh_max));
      return 2;
      }
  if (extent_block->header.eh_depth > 4) {
      FDBG (("ERROR eh_depth = %u over maximum\r\n", extent_block->header.eh_depth));
      return 3;
      }

//#define MAX_EXTEND_SIZE 4096 /* in case fs->common.byte_per_block > 4096... */
#ifdef MAX_EXTEND_SIZE
  if (extent_block->header.eh_entries >= (MAX_EXTEND_SIZE - sizeof(struct ext4_extent_header)) / sizeof(struct ext4_extent)) {
      FDBG (("%s: eh_entries %u over Gujin limit\r\n", __FUNCTION__, extent_block->header.eh_entries));
      return 8;
      }
#else
  if (extent_block->header.eh_entries >= (fs->common.byte_per_block - sizeof(struct ext4_extent_header)) / sizeof(struct ext4_extent)) {
      FDBG (("%s: eh_entries %u over block limit\r\n", __FUNCTION__, extent_block->header.eh_entries));
      return 8;
      }
#endif

  unsigned cpt_entry;
  for (cpt_entry = 0; cpt_entry < extent_block->header.eh_entries; cpt_entry++) {
      if (extent_block->header.eh_depth) {
#ifndef MAX_EXTEND_SIZE
	  unsigned blockbuffer[fs->common.byte_per_block/4];
#else
	  unsigned blockbuffer[MAX_EXTEND_SIZE/4];
#endif
	  unsigned long long ei_leaf = 0x100000000ULL * extent_block->data.array_idx[cpt_entry].ei_leaf_hi + extent_block->data.array_idx[cpt_entry].ei_leaf_lo;
	  FDBG (("  EXTENT_IDX[%u]: ei_block %u ei_unused %u ei_leaf %llu\r\n",
		cpt_entry, extent_block->data.array_idx[cpt_entry].ei_block, extent_block->data.array_idx[cpt_entry].ei_unused, ei_leaf));
	  if (ei_leaf > fs->common.blocks_count) {
	      FDBG (("%s: block %llU to high (max %llU)\r\n", __FUNCTION__, ei_leaf, fs->common.blocks_count));
	      return 0xE;
	      }
	  unsigned long long iblock_lba = extent_block->data.array_idx[cpt_entry].ei_leaf_hi * fs->common.sector_per_block;
	  iblock_lba = (iblock_lba << 32) + extent_block->data.array_idx[cpt_entry].ei_leaf_lo * fs->common.sector_per_block;
	  if (iblock_lba >= DI.param[fs->common.disk].partition[fs->common.part].length) {
	      FDBG (("%s: indirect block lba %llu out of partition, max %llu!\r\n", __FUNCTION__, iblock_lba, DI.param[fs->common.disk].partition[fs->common.part].length));
	      /* return 0xE; will be detected by FS_readsector() */
	      }
	  FDBG ((" [%s: reading indirect fragment at lba %llu of part %u] ", __FUNCTION__, iblock_lba, fs->common.part));
	  /* FIXME: extent crossing block boundary? */
	  unsigned char err = FS_readsector (fs, iblock_lba, sizeof(blockbuffer) / fs->common.SectSize, stack_adr (blockbuffer));
	  if (err) {
	      FDBG (("%s: error 0x%X reading indirect fragment.\r\n", __FUNCTION__, err));
	      return 0xF;
	      }
//printf ("extent_block->data.array_idx[%u].ei_block = %u\r\n", cpt_entry, extent_block->data.array_idx[cpt_entry].ei_block); _BIOS_getkey();
	  unsigned ret = analyse_extent_array (fs, (struct extent_block_s *)((char *)blockbuffer + extent_block->data.array_idx[cpt_entry].ei_block), sector_chain_buffer_ptr, current);
	  if (ret != 0)
	      return 9;
	  }
	else {
	  unsigned long long ee_start = 0x100000000ULL * extent_block->data.array[cpt_entry].ee_start_hi + extent_block->data.array[cpt_entry].ee_start_lo;
	  FDBG (("  EXTENT[%u]: ee_block %u ee_len %u ee_start %llu\r\n",
		cpt_entry, extent_block->data.array[cpt_entry].ee_block, extent_block->data.array[cpt_entry].ee_len, ee_start));
	  if (ee_start + extent_block->data.array[cpt_entry].ee_len >= fs->common.blocks_count) {
	      FDBG (("%s: block ee_start=%llu + ee_len=%u too high, max %llU!\r\n", __FUNCTION__, ee_start, extent_block->data.array[cpt_entry].ee_len, fs->common.blocks_count));
	      return 4;
	      }

	  unsigned ee_nb = extent_block->data.array[cpt_entry].ee_len * fs->common.sector_per_block;
	  unsigned long long ee_lba = extent_block->data.array[cpt_entry].ee_start_hi * fs->common.sector_per_block;
	  ee_lba <<= 32;
	  ee_lba += extent_block->data.array[cpt_entry].ee_start_lo * fs->common.sector_per_block;

	  if (*current == 0xFFFFFFFE) {
	      *current = 0;
	      (*sector_chain_buffer_ptr)[*current].nb = ee_nb;
	      (*sector_chain_buffer_ptr)[*current].lba = ee_lba;
	      }
	    else if ((*sector_chain_buffer_ptr)[*current].lba + (*sector_chain_buffer_ptr)[*current].nb == ee_lba)
	      (*sector_chain_buffer_ptr)[*current].nb += ee_nb;
	    else {
#ifdef FS_USE_MALLOC
	      struct sector_chain_str *sector_chain_buffer = REALLOC (*sector_chain_buffer_ptr, (*current+2+1) * sizeof (struct sector_chain_str), "chainbu2");
	      if (sector_chain_buffer == 0) {
		  FDBG (("%s: realloc failed!\r\n", __FUNCTION__));
		  return 5;
		  }
	      (*sector_chain_buffer_ptr) = sector_chain_buffer;
#else
	      if (*current >= FILE_MAX_FRAGMENT) {
		  FDBG (("%s: file has too many fragments!\r\n", __FUNCTION__));
		  return 6;
		  }
#endif
	      *current += 1;
	      (*sector_chain_buffer_ptr)[*current].nb = ee_nb;
	      (*sector_chain_buffer_ptr)[*current].lba = ee_lba;
	      }
	  }
      }
  return 0;
  }}

FS_FCT_PREFIX(E2FS_get_sector_chain) static struct sector_chain_str *
E2FS_get_sector_chain (union filesystem_union *fs,
			unsigned inode, unsigned filesize,
			 struct sector_chain_str **sector_chain_buffer_ptr)
  {bound_stack();{
  struct ext_inode the_inode;
  unsigned tmp, level, nb_file_block;
  union blocknr_inode_u blocknr_inode;
  unsigned current = 0xFFFFFFFE;
  /* do not use filesize, manage hole at end of file */
  filesize = filesize;

  FDBG (("%s (inode %u):", __FUNCTION__, inode));

  if (inode == 0 || inode > fs->common.inodes_count) {
      FDBG (("error inode either null or over max=%u.\r\n", fs->common.inodes_count));
      return 0;
      }

  tmp = E2FS_read_inode (fs, inode, &the_inode);
  if (tmp) {
      FDBG (("error 0x%X E2FS_read_inode.\r\n", tmp));
      return 0;
      }
    else if (fs->e2fs.use_hi_fields && (the_inode.i_size_high || the_inode.osd2.linux2.l_i_blocks_high)) {
      FDBG (("E2FS_read_inode returns valid inode but size (%u << 32) or nb_blocks (%u << 32) too high.\r\n", the_inode.i_size_high, the_inode.osd2.linux2.l_i_blocks_high));
      return 0;
      }
    else
      FDBG (("i_blocks_lo %u i_size_lo %u: ", the_inode.i_blocks_lo, the_inode.i_size_lo));
/* Note about i_blocks: in linux/fs/stat.c::inode_add_bytes() manages that field
   in 512 bytes (hardcoded) sectors */

  blocknr_inode.inode = &the_inode;
  nb_file_block = the_inode.i_blocks_lo / (fs->common.byte_per_block / 512);

  if (fs->e2fs.use_extents && the_inode.i_flags.EXT4_EXTENTS_FL) {
      tmp = analyse_extent_array(fs, &the_inode.blocks.extent_block, sector_chain_buffer_ptr, &current);
      if (tmp) {
	  FDBG (("analyse_extent_array error 0x%X.\r\n", tmp));
	  return 0;
	  }
      }
    else for (level = 0; level <= 3; level++) {
      tmp = read_analyse_chain (fs, level, blocknr_inode, &nb_file_block, sector_chain_buffer_ptr, &current);
      if (tmp) {
	  FDBG (("read_analyse_chain [level %u] error 0x%X.\r\n", level, tmp));
	  return 0;
	  }
      /* WARNING: we use the fact that
	  __u32	i_simple_indirect_block;
	  __u32	i_double_indirect_block;
	  __u32	i_triple_indirect_block;
	 are contiguous: */
      blocknr_inode.blocknr = (&the_inode.blocks.std_block.i_simple_indirect_block)[level];
      }

  if (current == 0xFFFFFFFE) {
      FDBG (("no block found\r\n"));
      return 0;
      }
    else {
      struct sector_chain_str *buffer = *sector_chain_buffer_ptr, *ptr;
      unsigned cptsector = 0;

      current++;
      buffer[current].nb = 0;
      buffer[current].lba = 0; /* safety */

      for (ptr = buffer; ptr < &buffer[current]; ptr++) {
	  if (ptr->lba & (1ULL << 63))
	      FDBG (("%u sectors (with discontinuity) at %llu, ", ptr->nb, ptr->lba & ~(1ULL << 63)));
	    else if (ptr->lba != 0)
	      FDBG (("%u sectors at %llu, ", ptr->nb, ptr->lba));
	    else
	      FDBG (("hole of %u sectors, ", ptr->nb));
	  cptsector += ptr->nb;
	  }
      ptr --;
      if (cptsector == the_inode.i_blocks_lo) /* linux/fs/stat.c::inode_add_bytes() unit 512 bytes */
	  FDBG (("OK\r\n"));
	else
	  FDBG (("\r\nERROR nb sectors %u (without last hole: %u), should be %u\r\n",
		cptsector, (cptsector - ((ptr->lba == 0)? ptr->nb : 0)), the_inode.i_blocks_lo));

      if (ptr->lba == 0 && ptr->nb != 0) {
	  FDBG (("---> replace ptr->nb from %u to 0\r\n", ptr->nb));
	  ptr->nb = 0;
	  }

#if 0   /* manage hole at end of file */
      if (DIVROUNDUP (filesize, (fs->common.byte_per_block / 512)) > cptsector) {
	  buffer[current].nb = DIVROUNDUP (filesize, (fs->common.byte_per_block / 512)) - cptsector;
	  buffer[current].lba = 0; /* hole */
	  FDBG (("---> correcting missing hole at end of %u blocks.\r\n", buffer[current].nb));
	  current++;
	  buffer[current].nb = 0;
	  buffer[current].lba = 0; /* safety */
	  }
#endif

      return buffer;
      }
  }}

/*
 * On my test filesystems, there isn't any entry crossing
 * a block boundary.
 */

FS_FCT_PREFIX(E2FS_treat_directory) static int
E2FS_treat_directory (struct treat_directory_str *opaque,
		      unsigned char *buffer, unsigned len)
  {bound_stack();{
  struct ext_dir_entry *entry = (struct ext_dir_entry *)buffer,
	*last = (struct ext_dir_entry *)(buffer + len);

  while (entry < last) {
#if DEBUG & DEBUG_FS
      char name[EXT2_NAME_LEN];
      unsigned cpt;
      for (cpt = 0; cpt < sizeof(name)-1 && cpt < entry->name_len; cpt++)
	  if ((name[cpt] = entry->name[cpt]) == '\0')
	      break;
      name[cpt] = '\0';
#endif

      if (entry->rec_len == 0) {
	  FDBG ((" [%s: entry length=0, (pointers: entry 0x%X last 0x%X) end processing this directory] ",
		__FUNCTION__, (unsigned)entry, (unsigned)last));
	  return 1; /* can stop processing of this directory */
	  }
      /* test to see if we are reading a directory block: */
      if (entry->rec_len < 8) { /* entry->name at least one char */
	  FDBG ((" [%s: rec_len = %u <8, not a directory sector] ",
		__FUNCTION__, entry->rec_len));
	  return -1;
	  }
      if (entry->rec_len % 4) { /* entry->inode is aligned */
	  FDBG ((" [%s: rec_len = %u %% 4!=0, not a directory sector] ",
		__FUNCTION__, entry->rec_len));
	  return -1;
	  }
      if (entry->rec_len < (unsigned short)(entry->name_len) - 8) {
	  FDBG ((" [%s: rec_len = %u, name_len %u, not a directory sector] ",
		__FUNCTION__, entry->rec_len, entry->name_len));
	  return -1;
	  }

      if (entry->name_len == 0) {
	  FDBG ((" [%s: name_len = 0] ", __FUNCTION__));
	  /* Is that reserved? for delete/undelete? */
	  }

//      FDBG ((" [entry 0x%X: rec_len %u, name_len %u, name '%s'] ",
//		entry, entry->rec_len, entry->name_len, name));
      if (opaque->nb >= NB_DESC_ARRAY - 1) {
	  FDBG ((" [%s: no more space in NB_DESC_ARRAY] ", __FUNCTION__));
	  return 0;
	  }

      struct ext_inode the_inode;
      unsigned result;
      struct slashdir *whichdir;
      if ((result = tst_elem (opaque, entry->name, entry->name_len)) != tst_elem_ignore
	&& E2FS_read_inode (&opaque->filesystem, entry->inode, &the_inode) == 0)
	  switch (result) {
	      case tst_elem_commandfile:
#ifdef COMMANDFILE
		  if (the_inode.i_mode.fmt == fmt_reg)
		      file_treat_commandfile (&opaque->filesystem, entry->inode, the_inode.i_size_lo);
#endif
		  break;
	      case tst_elem_maybe:
		  if (the_inode.i_mode.fmt == fmt_reg && !file_check_aa55_sig (&opaque->filesystem, entry->inode)) {
		      FDBG ((" [%s: maybe file without 0xAA55 signature] ", __FUNCTION__));
		      break;
		      }
		  result = tst_elem_found;
	      case tst_elem_found:
	      case tst_elem_iso:
		  switch (the_inode.i_mode.fmt) {
		      case fmt_dir:
			  /* for instance "/initrd" directory */
			  FDBG ((" [%s: file '%s' is a directory] ", __FUNCTION__, name));
			  break;
		      case fmt_lnk:
			  /* shall we test the_inode.i_blocks == 0 ? */
			  FDBG ((" [%s: file '%s' is a symbolic link] ", __FUNCTION__, name));
			  break;
		      case fmt_reg:
			  if (opaque->filesystem.e2fs.use_hi_fields && the_inode.i_size_high) {
			      FDBG ((" [%s: NOT adding file '%s', size too big > %u<<32] ", __FUNCTION__, name, the_inode.i_size_high));
			      break;
			      }
#ifdef NB_ISO
			  if (BOOTWAY.iso.nb_found < NB_ISO - 1 && result == tst_elem_iso) {
			      BOOTWAY.iso.found[BOOTWAY.iso.nb_found].inode = entry->inode;
			      BOOTWAY.iso.found[BOOTWAY.iso.nb_found].size = the_inode.i_size_lo; /* ISOFS files < 32 bits sectors */
			      unsigned i = 0, j;
			      if (!opaque->inRoot) {
				  const char *path = opaque->which_dir? SearchPathArray[opaque->which_dir - 1] : copy_gujin_param.scanpath;
				  while ((BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i] = path[i]) != '\0')
				      i++;
				  BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i++] = '/';
				  }
			      for (j = 0; i < sizeof (BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename) - 1 && j < entry->name_len; i++)
				  if ((BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i] = entry->name[j++]) == '\0')
				      break;
			      BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i] = '\0';
			      BOOTWAY.iso.nb_found++;
			      break;
			      }
#endif
			  if (opaque->nb >= NB_DESC_ARRAY - 1) {
			      FDBG ((" [%s: no more space in NB_DESC_ARRAY] ", __FUNCTION__));
			      break;
			      }
#ifdef MIN_KERNEL_INITRD_SIZE
			  if (the_inode.i_size_high == 0 && the_inode.i_size_lo < MIN_KERNEL_INITRD_SIZE
				&& IS_KERNEL_OR_INITRD (&opaque->array[opaque->nb])) {
			      FDBG ((" [%s: filesize too small: %u] ", __FUNCTION__, the_inode.i_size_lo));
			      break;
			      }
#endif

			  // Shall we use i_crtime or i_ctime ?
			  unsigned delta_seconds = (2001 - 1970) * 365 * 24 * 60 * 60, last_modification_date;
			  delta_seconds += 8 * 24 * 60 * 60; /* 8 years of 366 days starting in 1972 in between */
			  if (the_inode.i_mtime < delta_seconds)
			      last_modification_date = 0;
			    else
			      last_modification_date = the_inode.i_mtime - delta_seconds;
			  FDBG ((" [%s: adding file '%s' size %u bytes date %u sec after Jan 1st, 2001] ", __FUNCTION__, name, the_inode.i_size_lo, last_modification_date));
			  opaque->array[opaque->nb].inode = entry->inode;
			  opaque->array[opaque->nb].filesize = the_inode.i_size_lo;
			  opaque->array[opaque->nb].last_modification_date = last_modification_date;
			  opaque->array[opaque->nb].ReadOnly = !the_inode.i_mode.write_user;	/* mostly for BDI files */
			  if (opaque->filesystem.common.fsname[0] != '\0') {
			      opaque->array[opaque->nb].iso_fsname = MALLOC(strlen(opaque->filesystem.common.fsname)+1, "isofsna");
			      if (opaque->array[opaque->nb].iso_fsname)
				  strcpy (opaque->array[opaque->nb].iso_fsname, opaque->filesystem.common.fsname);
			      }
			  opaque->nb++;
			  break;
		      default:
			  FDBG ((" [%s: file '%s' is not a regular file] ", __FUNCTION__, name));
			  break;
		      }
		  break;
	      case tst_elem_slashboot:
		  whichdir = &opaque->filesystem.common.slashboot;
		  goto E2FS_tst_elem_slashcommon;
	      case tst_elem_slashinstall386:
		  whichdir = &opaque->filesystem.common.slashinstall386;
		  goto E2FS_tst_elem_slashcommon;
	      case tst_elem_slashinstallamd64:
		  whichdir = &opaque->filesystem.common.slashinstallamd64;
		  goto E2FS_tst_elem_slashcommon;
	      case tst_elem_slashisolinux:
		  whichdir = &opaque->filesystem.common.slashisolinux;
		  goto E2FS_tst_elem_slashcommon;
	      case tst_elem_slashcasper:
		  whichdir = &opaque->filesystem.common.slashcasper;
		  goto E2FS_tst_elem_slashcommon;
	      case tst_elem_slashlive:
		  whichdir = &opaque->filesystem.common.slashlive;
		  goto E2FS_tst_elem_slashcommon;
E2FS_tst_elem_slashcommon:
		  if (the_inode.i_mode.fmt != fmt_dir)
		      FDBG ((" [%s: file '%s' not a directory] ", __FUNCTION__, name));
		    else if (!opaque->inRoot)
		      FDBG ((" [%s: directory '%s' but already in a subdirectory] ", __FUNCTION__, name));
		    else if (opaque->filesystem.e2fs.use_hi_fields && the_inode.i_size_high)
		      FDBG ((" [%s: NOT storing '%s' directory, size too big > %u<<32] ", __FUNCTION__, name, the_inode.i_size_high));
		    else {
		      FDBG ((" [%s: storing '%s' subdirectory at inode %u size %u] ", __FUNCTION__, name, entry->inode, the_inode.i_size_lo));
		      whichdir->inode = entry->inode;
		      whichdir->size = the_inode.i_size_lo;
		      }
		  break;
	      }
      entry = (struct ext_dir_entry *)((char *)entry + entry->rec_len);
      }
  if (entry != last)
      FDBG ((" [%s: entry 0x%X last 0x%X] ", __FUNCTION__, (unsigned)entry, (unsigned)last));

  FDBG ((" [%s: need to read more sectors] ", __FUNCTION__));
  return 0;
  }}
#endif /* E2FS_PROBE */

/**
 ** FAT* filesystem:
 **/
#if DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE)

FS_FCT_PREFIX(FAT_get_parameter) static inline unsigned
FAT_get_parameter (struct common_filesystem	*common,
		   const unsigned char		*first4Kbyte,
		   unsigned			*start_fat,
		   unsigned short		*nb_fat,
		   unsigned			*fat_size,
		   unsigned			*start_dir,
		   unsigned short		*fatbit)
  {
  struct diskparam_str *const dp = &DI.param[common->disk];
  struct partition_str *const dpp = &dp->partition[common->part];
  const bootsector_t *bootsector = (bootsector_t *)first4Kbyte;
  unsigned char err, Mediadescriptor;
  unsigned nbrootsector, sector_count;

  FDBG (("%s: ", __FUNCTION__));

  if (DEFAULT_BUFFER_SIZE < sizeof (bootsector_t)) {
      FDBG (("[FAT DEFAULT_BUFFER_SIZE=%u too small] ", DEFAULT_BUFFER_SIZE));
      return 0x0E;
      }

  if (bootsector->after.Signature0xAA55 != 0xAA55) {
      FDBG (("did not find 0xAA55 signature but 0x%X (partition not formated?).\r\n",
		bootsector->after.Signature0xAA55));
      return 1;
      }

  if (   bootsector->before.part2.Signaturebyte0x29 != 0x29 && bootsector->before.part2.Signaturebyte0x29 != 0x28
      && ((const bootbefore_FAT32_t *)&bootsector->before)->part2.Signaturebyte0x29 != 0x29) {
      FDBG (("did not find 0x29 signature.\r\n"));
      return 2;
      }

  if (bootsector->before.part1.Bytepersector == 0 || bootsector->before.part1.Bytepersector & 1) {
      FDBG (("Boot sector Bytepersector = %u invalid.\r\n", bootsector->before.part1.Bytepersector));
      return 3;
      }
  if (bootsector->before.part1.Sectorpercluster == 0) {
      FDBG (("Boot sector Sectorpercluster = %u invalid.\r\n", bootsector->before.part1.Sectorpercluster));
      return 4;
      }

  if (bootsector->before.part1.Bytepersector != 512)
      DBG (("{strange: Bytepersector = %u} ", bootsector->before.part1.Bytepersector));
  if (dp->access == bios_chs || dp->access == ebios_lba) {
      unsigned char drvnb = (bootsector->before.part2.Signaturebyte0x29 == 0x29 || bootsector->before.part2.Signaturebyte0x29 == 0x28)
		? bootsector->before.part2.PhysicaldriveNb
		: ((const bootbefore_FAT32_t *)&bootsector->before)->part2.PhysicaldriveNb;

      if (drvnb != dp->disknb)
	  DBG (("{strange: PhysicaldriveNb = 0x%X instead of 0x%X} ", drvnb, dp->disknb));
      }
  if (bootsector->before.part1.NbHead != dp->bios_maxhead +1)
      DBG (("{strange: NbHead %u != DI.param[disk].bios_maxhead %u +1} ",
		bootsector->before.part1.NbHead, dp->bios_maxhead));
  if (bootsector->before.part1.NbSectorpertrack != dp->bios_nbsectorpertrack)
      DBG (("{strange: NbSectorpertrack %u != DI.param[disk].bios_nbsectorpertrack %u} ",
		bootsector->before.part1.NbSectorpertrack, dp->bios_nbsectorpertrack));
  if (bootsector->before.part1.NbHiddensector != dpp->start)
      DBG (("{strange: NbHiddensector %u != DI.param[disk].partition[part].start %llu} ",
		bootsector->before.part1.NbHiddensector, dpp->start));
#if 0 /* If Mediadescriptor == 0xBB, that is a MBR, total sector = (NbTotalsector << 32) + NbTotalsector2 */
  if (bootsector->before.part1.Mediadescriptor == 0xBB)
      sector_count = ((unsigned long long)bootsector->before.part1.NbTotalsector << 32) + bootsector->before.part1.NbTotalsector2;
    else
#endif
      sector_count = (bootsector->before.part1.NbTotalsector != 0)
		? bootsector->before.part1.NbTotalsector
		: bootsector->before.part1.NbTotalsector2;
  common->sector_per_block = bootsector->before.part1.Sectorpercluster;
  common->blocks_count = sector_count / common->sector_per_block;
  common->SectSize = bootsector->before.part1.Bytepersector;

  nbrootsector = DIVROUNDUP (bootsector->before.part1.NbRootdirentry * sizeof (directory_t), common->SectSize);

  /* Warning: a lot of / all versions of the other operating system
   * do not use the following field but recalculate it more or less
   * correctly (always correct for FAT16, byte/sector 512).
   * I should maybe recalculate it myself - but not tonight -:)
   */
  if (bootsector->before.part1.NbSectorperFAT != 0)
      *fat_size = bootsector->before.part1.NbSectorperFAT; /* so FAT16 */
    else if (((bootbefore_FAT32_t *)&bootsector->before)->partFAT32.NbSectorperFAT2 != 0)
      *fat_size = ((bootbefore_FAT32_t *)&bootsector->before)->partFAT32.NbSectorperFAT2;
    else  {
      FDBG (("Boot sector NbSectorperFAT and NbSectorperFAT2 null together, invalid.\r\n"));
      return 4;
      }

  /* inodes_count is in fact nbcluster */
  common->inodes_count = (sector_count
		- bootsector->before.part1.Reservedsector
		- bootsector->before.part1.NbFAT * *fat_size
		- nbrootsector) / common->sector_per_block;

  /*
   * If the compiler is good, and there is only _one_ of FAT12_PROBE
   * FAT16_PROBE or FAT32_PROBE, the constant "*fatbit" will be propagated
   * _very_ deeply...
   */
  if (common->inodes_count < 4085) { /* 4085 == 0xFF5 */
      if (!partition_is_FAT12(dpp->type))
	  DBG ((" {FAT12 on disk %u part %u with type 0x%X!} ", common->disk, common->part, dpp->type));
#if DISK_SUPPORT & FAT12_PROBE
      *fatbit = 12;
#else
      return 5;
#endif
      }
//    else if (common->inodes_count < 65525) { /* 65525 == 0xFFF5 */
	/* Shall we handle FAT16 with illimited root directory? FAT32 with reduced MBR header? */
    else if (bootsector->before.part1.NbSectorperFAT != 0 && bootsector->before.part1.NbRootdirentry != 0) {
      if (!partition_is_FAT16(dpp->type))
	  DBG ((" {FAT16 on disk %u part %u with type 0x%X!} ", common->disk, common->part, dpp->type));
#if DISK_SUPPORT & FAT16_PROBE
      *fatbit = 16;
#else
      return 6;
#endif
      }
    else {
      if (!partition_is_FAT32(dpp->type))
	  DBG ((" {FAT32 on disk %u part %u with type 0x%X!} ", common->disk, common->part, dpp->type));
#if DISK_SUPPORT & FAT32_PROBE
      *fatbit = 28;
#else
      return 7;
#endif
      }

  *start_fat = bootsector->before.part1.Reservedsector;
  *nb_fat = bootsector->before.part1.NbFAT;
  *start_dir = *start_fat + *nb_fat * *fat_size; /* start_fat is the start of the first FAT here */

  /* Use FAT 1 if not FAT32: */
  if (bootsector->before.part2.Signaturebyte0x29 != 0x29 && bootsector->before.part2.Signaturebyte0x29 != 0x28) {
      unsigned i = i;
      bootbefore_FAT32_t *ptr = (bootbefore_FAT32_t *)&bootsector->before;

      FDBG (("FAT32 version 0x%X, NbSectorperFAT2 %u (NbSectorperFAT %u), root_cluster %u, FSinfosector %u, save_bootrecord %u, reserved: ",
		ptr->partFAT32.version, ptr->partFAT32.NbSectorperFAT2, bootsector->before.part1.NbSectorperFAT,
		ptr->partFAT32.root_cluster, ptr->partFAT32.FSinfosector, ptr->partFAT32.save_bootrecord));
      for (i = 0; i < sizeof (ptr->partFAT32.reserved) / sizeof (ptr->partFAT32.reserved[0]); i++)
	  FDBG (("0x%X, ", ptr->partFAT32.reserved[i]));
      FDBG (("\r\n"));
      if (   (ptr->part2.Signaturebyte0x29 == 0x29 || ptr->part2.Signaturebyte0x29 == 0x28)
	  && ptr->partFAT32.flags.one_fat_active
	  && ptr->partFAT32.flags.active_fat < *nb_fat) {
	  *start_fat += ptr->partFAT32.flags.active_fat * *fat_size; /* start_fat is the start of the used FAT for now on */
	  FDBG (("FAT32 select active FAT %u of %u FATs\r\n\t", ptr->partFAT32.flags.active_fat, *nb_fat));
	  }
	else
	  FDBG (("WARNING: not understood FAT32 flags value 0x%X, using FAT1\r\n\t", *CAST(unsigned short *, &ptr->partFAT32.flags)));
      common->slash.inode = ptr->partFAT32.root_cluster;
      }
    else
      common->slash.inode = 1; /* locally means root directory for FAT12/16 */

  common->first_data_block = *start_dir + nbrootsector;
  common->byte_per_block = common->sector_per_block * common->SectSize;
  common->slash.size = nbrootsector * common->SectSize;
  /* TODO: warn if FAT filesystem needs a scandisk... */

  Mediadescriptor = bootsector->before.part1.Mediadescriptor;
  FDBG (("Filesystem opened (nbrootsector %u, nb_fat %u, fat_size %u, nbcluster %u => fatbit %u, "
	"start_fat %u, start_dir %u, Mediadescriptor 0x%X, SectSize %u)\r\n\t",
	nbrootsector, *nb_fat, *fat_size, common->inodes_count, *fatbit,
	*start_fat, *start_dir, Mediadescriptor, common->SectSize));

  /* Handle dp->bytepersector > common->SectSize, for instance dp->bytepersector = 2048 and common->SectSize = 512
      FS_readsector() will do its checks because common->SectSize is initialised. */
  unsigned disk_sector_fat = *start_fat;
  const unsigned char *FATsignature = fourKbuffer;
  if (dp->bytepersector > common->SectSize) {
      unsigned short nb_fs_sector_per_disk_sector = dp->bytepersector / common->SectSize;
      FATsignature += (disk_sector_fat % nb_fs_sector_per_disk_sector) * common->SectSize;
      disk_sector_fat = (disk_sector_fat / nb_fs_sector_per_disk_sector) * nb_fs_sector_per_disk_sector;
      }

  if (disk_sector_fat != 0) {
      err = FS_readsector ((union filesystem_union *)common, disk_sector_fat, sizeof(fourKbuffer)/common->SectSize, data_adr(fourKbuffer));
      if (err != 0) {
	  FDBG (("error 0x%X read first FAT sector.\r\n", err));
	  return 8;
	  }
      }
  /* TODO: warn if FAT not equals...
  for (unsigned cpt = 1; cpt <= bootsector->before.part1.NbFAT; cpt++) {
      err = FS_readsector (dp, common->part, *start_fat + cpt * *fat_size, 1, data_adr(fourKbuffer) + common->SectSize);
      if (err != 0) {
	  FDBG (("error 0x%X read FAT sector on FAT number %u.\r\n", err, cpt));
	  return 8;
	  }
      if (!memeql (fourKbuffer, fourKbuffer + common->SectSize, common->SectSize)) {
	  FDBG (("error FAT number %u different from FAT number 1\r\n", cpt);
	  return 8;
	  }
      }
 */
  /* To prevent a floppy with only the MBR valid and everything else at zero to exit without error: */
  if (FATsignature[0] != Mediadescriptor) {
      FDBG (("error first (or active) FAT sector byte not 0x%X (i.e. Media Descriptor) but 0x%X.\r\n", Mediadescriptor, FATsignature[0]));
      if ((FATsignature[0] & 0xF8) != 0xF8) /* still work if Mediadescriptor is 0xF8, 0xF9, 0xFD */
	  return 9;
      }
  if (nbrootsector == 0 && *fatbit == 12) {
      if ((FATsignature[1] & 0x0F) != 0x0F) {
	  FDBG (("error first FAT12 sector & no rootdir first 12 bits are 0x%X, 0x%X, last 4 bits not 0x0F.\r\n", FATsignature[0], FATsignature[1]));
	  return 0xA;
	  }
      common->slash.inode = ((FATsignature[1] & 0xF0) << 4) | FATsignature[2];
      }
    else {
      unsigned nb0xFFbytes;
      if (*fatbit == 12)
	  nb0xFFbytes = 2; /* we know here nbrootsector != 0 */
	else if (*fatbit == 16)  {
	  if (nbrootsector == 0) { /* root dir is a cluster chain starting at cluster 1 */
	      nb0xFFbytes = 1;
	      common->slash.inode = *(unsigned short *)&FATsignature[2];
	      }
	    else
	      nb0xFFbytes = 3;
	  }
	else /* if (*fatbit == 28) */ {
	  if (nbrootsector != 0) {
	      FDBG (("error FAT32 with nbrootsector = %u != 0 not supported.\r\n", nbrootsector));
	      return 0xB;
	      }
	  if (FATsignature[3] != 0x0F) {
	      FDBG (("error first FAT sector byte index 3 not 0x0F but 0x%X.\r\n", FATsignature[3]));
	      if (FATsignature[3] != 0xFF)
		  return 0xC;
	      }
	  nb0xFFbytes = 2;
	  }
      do {
	  if (FATsignature[nb0xFFbytes] != 0xFF) {
	      FDBG (("error first FAT sector byte index %u not 0xFF but 0x%X.\r\n", nb0xFFbytes, FATsignature[nb0xFFbytes]));
	      return 0xD;
	      }
	  } while (--nb0xFFbytes);
      }

  return 0;
  }

FS_FCT_PREFIX(FAT_get_next_inode) static inline unsigned
FAT_get_next_inode (union filesystem_union *fs, unsigned inode, unsigned char fatbit_div_4, unsigned mask,
	unsigned *lba_cached, unsigned nb_sector_cached, unsigned char *sector_cache)
  {
  unsigned quartet_offset = inode * fatbit_div_4;
  unsigned start_quartet_cached = 2 * (*lba_cached - fs->fat.start_fat) * fs->common.SectSize;
  unsigned end_quartet_cached = start_quartet_cached + 2 * nb_sector_cached * fs->common.SectSize;
  unsigned next_inode;
  unsigned old_lba_cached = *lba_cached;

  if (   *lba_cached < fs->fat.start_fat
      || quartet_offset < start_quartet_cached
      || quartet_offset + fatbit_div_4 > end_quartet_cached) {
      static const unsigned ret[] = {
	  0xFFFFFFFF, 0x204C5047, 0x2E442E45,
	  0x72726F4C, 0x006E6961, 0xFFFFFFFE
	  };
      unsigned char err;
      *lba_cached = quartet_offset/2 / fs->common.SectSize;
      if (fs->common.SectSize < DI.param[fs->common.disk].bytepersector)
	  *lba_cached -= *lba_cached % (DI.param[fs->common.disk].bytepersector / fs->common.SectSize);
      start_quartet_cached = 2 * *lba_cached * fs->common.SectSize;
      *lba_cached += fs->fat.start_fat;

      if (*lba_cached >= fs->fat.start_fat + fs->fat.fat_size) {
	  FDBG ((" [%s: error calculating FAT: %u > %u]\r\n", __FUNCTION__, *lba_cached, fs->fat.start_fat + fs->fat.fat_size));
	  return ret[5];
	  }

      if (old_lba_cached == *lba_cached + 1 && nb_sector_cached > 1
	  && fs->common.SectSize >= DI.param[fs->common.disk].bytepersector) { /* very usual case, try to read just one sector */
	  memcpy (sector_cache, sector_cache + fs->common.SectSize, (nb_sector_cached - 1) * fs->common.SectSize);
	  FDBG ((" [%s: reading FAT: %u sectors at %u] ", __FUNCTION__, nb_sector_cached - 1, *lba_cached + 1));
	  err = FS_readsector (fs, *lba_cached + 1, nb_sector_cached - 1, data_adr(sector_cache) + fs->common.SectSize);
	  }
	else {
	  FDBG ((" [%s: reading FAT: %u sectors at %u] ", __FUNCTION__, nb_sector_cached, *lba_cached));
	  err = FS_readsector (fs, *lba_cached, nb_sector_cached, data_adr(sector_cache));
	  }
      if (err != 0) {
	  FDBG ((" [%s: error 0x%X reading FAT at sector %u]\r\n", __FUNCTION__, err, *lba_cached));
	  return ret[0];
	  }
      }

  next_inode = *(unsigned *)(sector_cache + ((quartet_offset - start_quartet_cached)/2));

  if ((quartet_offset - start_quartet_cached) & 1)
      next_inode >>= 4;
  return next_inode & mask;
  }

FS_FCT_PREFIX(FAT_get_sector_chain) static struct sector_chain_str *
FAT_get_sector_chain (union filesystem_union *fs,
			unsigned inode, unsigned filesize,
			struct sector_chain_str **sector_chain_buffer_ptr)
  {bound_stack();{
  struct sector_chain_str *sector_chain_buffer = *sector_chain_buffer_ptr;
  /* used because FAT is too big to fit completely in memory: */
  unsigned short fat_buffer_nb_sector = 2;
  if (fs->common.SectSize < DI.param[fs->common.disk].bytepersector)
      fat_buffer_nb_sector = DI.param[fs->common.disk].bytepersector / fs->common.SectSize;
  unsigned char fat_buffer[fat_buffer_nb_sector * fs->common.SectSize];
  unsigned fat_buffer_sector = 0;
  unsigned short current = 0;
  unsigned mask = (1U << fs->fat.fatbit) - 1;
  unsigned char fatelem_size_quartet;
  /* do not use filesize, manage uncomplete clusters at end of file */
  filesize = filesize;

#if DISK_SUPPORT & FAT32_PROBE
  if (fs->fat.fatbit/4 == 7) /* FAT32 unit is 32 bits */
      fatelem_size_quartet = 8;
    else
#endif /* FAT32_PROBE */
      fatelem_size_quartet = fs->fat.fatbit/4;

  if (inode == 0 || inode > fs->common.blocks_count) {
      FDBG (("error cluster=%u, either null or over max=%llU.\r\n", inode, fs->common.blocks_count));
      return 0;
      }
//    else if (inode >= 2 || (inode == 1 && fs->fat.start_dir == fs->common.first_data_block)) {
    else if (inode >= 2) {
      sector_chain_buffer[0].lba = fs->common.first_data_block
			+ (inode - 2) * fs->common.sector_per_block;
      sector_chain_buffer[0].nb = fs->common.sector_per_block;
      }
    else { /* == 1, locally means root directory for FAT12/16 */
      sector_chain_buffer[0].lba = fs->fat.start_dir;
      sector_chain_buffer[0].nb = fs->common.first_data_block - sector_chain_buffer[0].lba;
      FDBG (("%s returns [[%llu, +%u][0,0]]\r\n", __FUNCTION__, sector_chain_buffer[0].lba, sector_chain_buffer[0].nb));
      sector_chain_buffer[1].lba = 0; /* safety */
      sector_chain_buffer[1].nb = 0;
      return sector_chain_buffer;
      }

  for (;;) {
      unsigned lba; /* FAT is max 32 bits sectors */

      inode = FAT_get_next_inode (fs, inode, fatelem_size_quartet, mask,
				&fat_buffer_sector, fat_buffer_nb_sector, fat_buffer);

      if (inode & ~mask) {
	  FDBG (("%s: error in FAT cache!\r\n", __FUNCTION__));
	  return 0;
	  }
      if (inode == 0) {
	  FDBG (("%s: FAT chain contains free blocks!\r\n", __FUNCTION__));
	  return 0;
	  }
      if (inode > (mask & ~0x000F)) {
	  if (inode < (mask & ~0x0007)) {
	      FDBG (("%s: FAT chain contains bad blocks!\r\n", __FUNCTION__));
	      return 0;
	      }
	    else {
	      struct sector_chain_str *ptr;
	      FDBG ((" [%s: ", __FUNCTION__));
	      current++;
	      for (ptr = sector_chain_buffer;
		   ptr < &sector_chain_buffer[current];
		   ptr++)
		  FDBG (("%u sectors at %llu, ", ptr->nb, ptr->lba));
	      FDBG (("end with 0x%X]\r\n", inode));
	      sector_chain_buffer[current].nb = 0;
	      sector_chain_buffer[current].lba = 0; /* safety */
	      return sector_chain_buffer;
	      }
	  }
      if (inode > fs->common.inodes_count + 2) { /* the two reserved */
	  FDBG (("%s: FAT chain contains inode %u, max %u!\r\n",
			__FUNCTION__, inode, fs->common.inodes_count));
	  return 0;
	  }

      lba = fs->common.first_data_block + (inode - 2) * fs->common.sector_per_block;

      if (lba == sector_chain_buffer[current].lba + sector_chain_buffer[current].nb) {
	  sector_chain_buffer[current].nb += fs->common.sector_per_block;
	  }
	else {
#ifdef FS_USE_MALLOC
	  sector_chain_buffer = REALLOC (sector_chain_buffer, (current+2+1) * sizeof (struct sector_chain_str), "chainbu3");
	  if (sector_chain_buffer == 0) {
	      FDBG (("%s: realloc failed!\r\n", __FUNCTION__));
	      return 0;
	      }
	  (*sector_chain_buffer_ptr) = sector_chain_buffer;
#else
	  if (current >= FILE_MAX_FRAGMENT) {
	      FDBG (("%s: file has too many fragments!\r\n", __FUNCTION__));
	      return 0;
	      }
#endif
	  {
	  struct sector_chain_str *ptr;
	  for (ptr = sector_chain_buffer;
	       ptr <= &sector_chain_buffer[current];
	       ptr++) {
	      if (ptr->lba == lba) {
		  FDBG (("%s: FAT loop detected!\r\n", __FUNCTION__));
		  return 0;
		  }
	      }
	  }
	  current ++;
	  sector_chain_buffer[current].lba = lba;
	  sector_chain_buffer[current].nb = fs->common.sector_per_block;
	  }
      }
  }}

FS_FCT_PREFIX(FAT_get_filename) static inline unsigned
FAT_get_filename (struct treat_directory_str *opaque,
		unsigned char *filename, const directory_t *dir)
  {
  /* inline */ const struct file_attr_str lfn_attr = {
	.readonly	= 1,	.hidden		= 1,
	.system		= 1,	.volumelabel	= 1,
	.subdirectory	= 0,	.archive	= 0,
	.unused		= 0
	};

  if (FAT_attribute_equal (dir->attributes, lfn_attr)) {
      long_filename_t *lfn = (long_filename_t *)dir;
      char tmpname[nbof(lfn->name1) + nbof(lfn->name2) + nbof(lfn->name3) + 1];
      char *ptr, *src, *dst;
      unsigned short i;

      LFNDBG ((" <long filename sequence: deleted: %u, last: %u, unused: %u, "
		"entry: %u; cluster 0x%X; reserved: 0x%X => ",
		lfn->sequence.deleted, lfn->sequence.last,
		lfn->sequence.unused, lfn->sequence.entry,
		lfn->cluster, lfn->reserved));
      if (lfn->sequence.deleted) {
	  LFNDBG (("deleted, ignore> "));
	  opaque->filesystem.fat.lfn.sequence = 0xFF;
	  return 1;
	  }
      if (lfn->sequence.last) {
	  LFNDBG (("start recording: "));
	  opaque->filesystem.fat.lfn.sequence = lfn->sequence.entry;
	  opaque->filesystem.fat.lfn.name[0] = '\0';
	  }
	else if (lfn->sequence.entry != opaque->filesystem.fat.lfn.sequence) {
	  LFNDBG (("out of order (wait %u, have %u), ignore> ",
			opaque->filesystem.fat.lfn.sequence, lfn->sequence.entry));
	  opaque->filesystem.fat.lfn.sequence = 0xFF;
	  return 1;
	  }
      opaque->filesystem.fat.lfn.sequence --;

      /* Convert UTF16 -> UTF8 here:*/
      ptr = tmpname;
      for (i = 0; i < nbof(lfn->name1); i++) {
	  if (lfn->name1[i] >> 8 != 0 || lfn->name1[i] == 0)
	      LFNDBG (("'0x%X' ", lfn->name1[i]));
	    else
	      LFNDBG (("'%c' ", lfn->name1[i]));
	  if (lfn->name1[i] < 0x80)
	      *ptr++ = lfn->name1[i];
	    else if (lfn->name1[i] < 0x800) {
	      *ptr++ = 0xC0 + (lfn->name1[i] >> 6);
	      *ptr++ = 0x80 + (lfn->name1[i] & 0x3F);
	      }
	    else
	      *ptr++ = '?'; /* ignore >= 3 bytes chars, can't display them anyway */
	}
      for (i = 0; i < nbof(lfn->name2); i++) {
	  if (lfn->name2[i] >> 8 != 0 || lfn->name2[i] == 0)
	      LFNDBG (("'0x%X' ", lfn->name2[i]));
	    else
	      LFNDBG (("'%c' ", lfn->name2[i]));
	  if (lfn->name2[i] < 0x80)
	      *ptr++ = lfn->name2[i];
	    else if (lfn->name2[i] < 0x800) {
	      *ptr++ = 0xC0 + (lfn->name2[i] >> 6);
	      *ptr++ = 0x80 + (lfn->name2[i] & 0x3F);
	      }
	    else
	      *ptr++ = '?'; /* ignore >= 3 bytes chars, can't display them anyway */
	  }
      for (i = 0; i < nbof(lfn->name3); i++) {
	  if (lfn->name3[i] >> 8 != 0 || lfn->name3[i] == 0)
	      LFNDBG (("'0x%X' ", lfn->name3[i]));
	    else
	      LFNDBG (("'%c' ", lfn->name3[i]));
	  if (lfn->name3[i] < 0x80)
	      *ptr++ = lfn->name3[i];
	    else if (lfn->name3[i] < 0x800) {
	      *ptr++ = 0xC0 + (lfn->name3[i] >> 6);
	      *ptr++ = 0x80 + (lfn->name3[i] & 0x3F);
	      }
	    else
	      *ptr++ = '?'; /* ignore >= 3 bytes chars, can't display them anyway */
	  }
      *ptr = '\0';

      /* Search the first null in previous string: */
      for (ptr = tmpname; *ptr != 0; ptr++)
	  /* nothing */;

      /* Make room for beginning of name: */
      src = opaque->filesystem.fat.lfn.name;
      while (*src != '\0')
	  src++;
      dst = src + (ptr - tmpname);
      if (dst > &opaque->filesystem.fat.lfn.name[lastof(opaque->filesystem.fat.lfn.name)]) {
	  dst = &opaque->filesystem.fat.lfn.name[lastof(opaque->filesystem.fat.lfn.name)];
	  *dst-- = '\0';
	  src = dst - (ptr - tmpname);
	  }
      while (src >= opaque->filesystem.fat.lfn.name)
	  *dst-- = *src--;

      /* Copy beginning of name: */
      while (ptr > tmpname)
	  *dst-- = *--ptr;

      LFNDBG (("=> gives up to now: '%s' ", opaque->filesystem.fat.lfn.name));
      opaque->filesystem.fat.lfn.chksum = lfn->checksum;
      LFNDBG (("checksum 0x%X> ", lfn->checksum));
      return 1;
      }
    else {
      unsigned char i, sum = 0;
      unsigned char *name = filename;
      const char *ptr;

      ptr = dir->name;
      for (i = 0; i < sizeof (dir->name); i++)
	  sum = ((sum << 7) | (sum >> 1)) + *ptr++;
      ptr = dir->extension;
      for (i = 0; i < sizeof (dir->extension); i++)
	  sum = ((sum << 7) | (sum >> 1)) + *ptr++;

      if (sum == opaque->filesystem.fat.lfn.chksum && opaque->filesystem.fat.lfn.sequence == 0) {
	  unsigned short cpt;

	  for (cpt = 0; cpt < NAME_LENGTH - 1; cpt++) {
	      *name = opaque->filesystem.fat.lfn.name[cpt];
	      if (*name == 0)
		  break;
	      name++;
	      }
	  *name = '\0';
	  FDBG (("[lfn '%s' chksum OK] ", filename));
	  }
	else {
	  ptr = dir->name;
	  for (i = sizeof (dir->name); --i  && ptr[i] == ' ';)
	      /* nothing */;
	  for (i += 2; --i;)
	      *name++ = *ptr++;

	  ptr = dir->extension;
	  for (i = sizeof (dir->extension); --i && ptr[i] == ' ';)
	      /* nothing */;
	  if (i != 0 || ptr[i] != ' ') {
	      if (!dir->attributes.volumelabel)
		  *name++ = '.';
	      for (i += 2; --i;)
		  *name++ = *ptr++;
	      }
	  *name = '\0';
	  if (!dir->attributes.volumelabel)
	      strlwr ((char *)filename);
	  }

      opaque->filesystem.fat.lfn.sequence = 0xFF;
      return 0;
      }
  }

FS_FCT_PREFIX(FAT_treat_directory) static int
FAT_treat_directory (struct treat_directory_str *opaque,
		      const directory_t *dir, unsigned len)
  {bound_stack();{
  struct partition_str *const dpp = &DI.param[opaque->filesystem.common.disk].partition[opaque->filesystem.common.part];
  const directory_t *lastdir = dir + (len / sizeof(directory_t));

  while (dir < lastdir) {
      unsigned char name[NAME_LENGTH];

      if (dir->attributes.unused) {
	  FDBG ((" [%s: not a directory sector] ", __FUNCTION__));
	  return -1;
	  }
      if (dir->name[0] == (char)0xE5) {
	  //FDBG ((" [erased file] "));
	  dir ++;
	  continue;
	  }
      if (dir->name[0] == 0) {
	  FDBG ((" [%s: normal end of directory] ", __FUNCTION__));
	  return 1; /* can stop processing of this directory */
	  }
      if (FAT_get_filename (opaque, name, dir)) {
	  FDBG ((" [part of long filename] "));
	  dir ++;
	  continue;
	  }

      if (opaque->inRoot) {
	  /* inline */ const struct file_attr_str volumeattr = {
		.readonly	= 0,	.hidden		= 0,
		.system		= 0,	.volumelabel	= 1,
		.subdirectory	= 0,	.archive	= 1,
		.unused		= 0
		};
	  /* when told to remove a label, dosfslabel clears it to all spaces... */
//	  if (FAT_attribute_equal (dir->attributes, volumeattr) && *(unsigned *)name != *(unsigned *)"    ") {
	  if (FAT_attribute_equal (dir->attributes, volumeattr) && name[0] != '\0') { /* spaces stripped by FAT_get_filename() */
	      char *dst = opaque->filesystem.common.fsname;
	      const unsigned char *src = name;
	      while (*src && dst < &opaque->filesystem.common.fsname[lastof(opaque->filesystem.common.fsname)])
		  *dst++ = (char)*src++;
	      *dst = '\0';
	      FDBG ((" [volumename: '%s'] ", opaque->filesystem.common.fsname));
#ifdef NB_ISO
	      if (BOOTWAY.iso.nb_remap == 0)
#endif
		  {
		  dst = (char *)dpp->name;
		  src = name;
		  while (*src && dst < (char *)&dpp->name[lastof(dpp->name)])
		      *dst++ = (char)*src++;
		  *dst = '\0';
		  }
	      dir ++;
	      continue;
	      }
	  }
      if (dir->attributes.volumelabel) {
	  FDBG ((" [ignoring volumelabel '%s'] ", name));
	  dir ++;
	  continue;
	  }

#if defined (DOSNAME1) || defined (DOSNAME2)
#if defined (DOSNAME1) && defined (DOSNAME2)
#define IS_DOSNAME(name) \
	(!stricmp ((char *)name, DOSNAME1) || !stricmp ((char *)name, DOSNAME2))
#elif defined (DOSNAME1)
#define IS_DOSNAME(name)	(!stricmp (name, DOSNAME1))
#elif defined (DOSNAME2)
#define IS_DOSNAME(name)	(!stricmp (name, DOSNAME2))
#else
#define IS_DOSNAME(name)	0
#endif

      if (IS_DOSNAME(name)) {
	  if (opaque->inRoot) {
	      FDBG ((" [found file '%s', FAT partition is bootable] ", name));
	      dpp->misc.fat_bootable = 1;
	      }
	  }
#endif /* defined (DOSNAME1) || defined (DOSNAME2) */
      if (opaque->nb >= NB_DESC_ARRAY - 1) {
	  FDBG ((" [%s: no more space in NB_DESC_ARRAY] ", __FUNCTION__));
	  return -2;
	  }

      struct slashdir *whichdir;
      switch (tst_elem (opaque, (char *)name, strlen((char *)name))) {
	  case tst_elem_iso:
#ifdef NB_ISO
	      if (!dir->attributes.subdirectory && BOOTWAY.iso.nb_found < NB_ISO - 1) {
		  BOOTWAY.iso.found[BOOTWAY.iso.nb_found].inode = ((unsigned)dir->cluster_msb << 16) + dir->cluster;
		  BOOTWAY.iso.found[BOOTWAY.iso.nb_found].size = dir->size;
		  unsigned i = 0, j;
		  if (!opaque->inRoot) {
		      const char *path = opaque->which_dir? SearchPathArray[opaque->which_dir - 1] : copy_gujin_param.scanpath;
		      while ((BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i] = path[i]) != '\0')
			  i++;
		      BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i++] = '/';
		      }
		  for (j = 0; i < sizeof (BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename) - 1; i++)
		      if ((BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i] = ((char *)name)[j++]) == '\0')
			  break;
		  BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i] = '\0';
		  BOOTWAY.iso.nb_found++;
		  }
#endif
	      break;
	  case tst_elem_commandfile:
#ifdef COMMANDFILE
	      if (!dir->attributes.subdirectory)
		  file_treat_commandfile (&opaque->filesystem, ((unsigned)dir->cluster_msb << 16) + dir->cluster, dir->size);
#endif
	      break;
	  case tst_elem_maybe:
	      if (dir->attributes.subdirectory || !file_check_aa55_sig (&opaque->filesystem, ((unsigned)dir->cluster_msb << 16) + dir->cluster)) {
		  FDBG ((" [%s: maybe file without 0xAA55 signature] ", __FUNCTION__));
		  break;
		  }
	  case tst_elem_found:
#ifdef MIN_KERNEL_INITRD_SIZE
	      if (dir->size < MIN_KERNEL_INITRD_SIZE && IS_KERNEL_OR_INITRD (&opaque->array[opaque->nb])) {
		  FDBG ((" [%s: filesize too small: %u] ", __FUNCTION__, dir->size));
		  break;
		  }
#endif
	      if (opaque->nb >= NB_DESC_ARRAY - 1)
		  FDBG ((" [%s: no more space in NB_DESC_ARRAY to treat %s] ", __FUNCTION__, name));
		else if (!dir->attributes.subdirectory) {
		  unsigned last_modification_date, nbdays = 0, cptmonth;
		  for (cptmonth = 1; cptmonth < dir->date.month; cptmonth++)
		      nbdays += dayinmonth[cptmonth];
		  nbdays += dir->date.day - 1;
		  nbdays += (dir->date.year + 1980 - 2001) / 4; /* in 2005 add 1 leap day */
		  if (dir->date.month >= 3 && (dir->date.year + 1980) % 4 == 0)
		      nbdays ++; /* add 1 day after Feb 29th on leap year */
		  if (dir->date.year + 1980 < 2001)
		      last_modification_date = 0;
		    else
		      last_modification_date = (dir->date.year + 1980 - 2001) * 365 * 24 * 60 * 60
					+ nbdays * 24 * 60 * 60 + dir->date.hour * 60 * 60 + dir->date.minute * 60 + dir->date.seconddiv2 * 2;
		  FDBG ((" [%s: adding file '%s' size %u bytes cluster 0x%X date %u sec after Jan 1st, 2001] ", __FUNCTION__, name, dir->size, dir->cluster, last_modification_date));
		  opaque->array[opaque->nb].filesize = dir->size;
		  opaque->array[opaque->nb].inode = ((unsigned)dir->cluster_msb << 16) + dir->cluster;
		  opaque->array[opaque->nb].last_modification_date = last_modification_date;
		  opaque->array[opaque->nb].ReadOnly = dir->attributes.readonly;	/* mostly for BDI files */
		  if (opaque->filesystem.common.fsname[0] != '\0') {
		      opaque->array[opaque->nb].iso_fsname = MALLOC(strlen(opaque->filesystem.common.fsname)+1, "isofsna");
		      if (opaque->array[opaque->nb].iso_fsname)
			  strcpy (opaque->array[opaque->nb].iso_fsname, opaque->filesystem.common.fsname);
		      }
		  opaque->nb++;
		  }
		else {
		  /* for instance "/initrd" directory */
		  FDBG ((" [%s: file '%s' is a directory] ", __FUNCTION__, name));
		  }
	      break;
	  case tst_elem_slashboot:
	      whichdir = &opaque->filesystem.common.slashboot;
	      goto FAT_tst_elem_slashcommon;
	  case tst_elem_slashinstall386:
	      whichdir = &opaque->filesystem.common.slashinstall386;
	      goto FAT_tst_elem_slashcommon;
	  case tst_elem_slashinstallamd64:
	      whichdir = &opaque->filesystem.common.slashinstallamd64;
	      goto FAT_tst_elem_slashcommon;
	  case tst_elem_slashisolinux:
	      whichdir = &opaque->filesystem.common.slashisolinux;
	      goto FAT_tst_elem_slashcommon;
	  case tst_elem_slashcasper:
	      whichdir = &opaque->filesystem.common.slashcasper;
	      goto FAT_tst_elem_slashcommon;
	  case tst_elem_slashlive:
	      whichdir = &opaque->filesystem.common.slashlive;
	      goto FAT_tst_elem_slashcommon;
FAT_tst_elem_slashcommon:
	      if (!dir->attributes.subdirectory)
		  FDBG ((" [%s: file '%s' not a directory] ", __FUNCTION__, name));
		else if (!opaque->inRoot)
		  FDBG ((" [%s: directory '%s' but already in a subdirectory] ", __FUNCTION__, name));
		else {
		  FDBG ((" [subdirectory '%s' size %u at %u] ", name, dir->size, dir->cluster));
		  whichdir->inode = dir->cluster;
//		  whichdir->size = dir->size;	// always null...
		  whichdir->size = 0;
#ifdef FS_USE_MALLOC
		  /* alloc also for the end marker: */
		  struct sector_chain_str *sector_chain_buffer = MALLOC (2 * sizeof (struct sector_chain_str), "diretrea");
		  if (sector_chain_buffer == 0) {
		      FDBG (("%s: malloc failed!\r\n", __FUNCTION__));
		      break;
		      }
#else
		  struct sector_chain_str bigarray[FILE_MAX_FRAGMENT];
		  struct sector_chain_str *sector_chain_buffer = bigarray;
#endif
		  struct sector_chain_str *sector_chain = FAT_get_sector_chain (&opaque->filesystem, dir->cluster, 0, &sector_chain_buffer);
		  if (sector_chain != 0) {
		      while (sector_chain->nb != 0) {
			  whichdir->size += sector_chain->nb;
			  sector_chain++;
			  }
		      whichdir->size *= opaque->filesystem.common.SectSize;
		      }
#ifdef FS_USE_MALLOC
		  /* alloc also for the end marker: */
		  FREE (sector_chain_buffer);
#endif
		  }
	      break;
	  case tst_elem_ignore:
	      break;
	  }
      dir ++;
      }
  FDBG ((" [%s: need to read more sectors] ", __FUNCTION__));
  return 0;
  }}

#endif /* FAT12_PROBE|FAT16_PROBE|FAT32_PROBE */

/**
 ** ISO9660 / CDROM or DVD filesystem:
 **/
#if DISK_SUPPORT & ISOFS_PROBE
FS_FCT_PREFIX(ISO_get_boot_catalog_lba) static inline unsigned
ISO_get_boot_catalog_lba (struct diskparam_str *const dp, struct common_filesystem *common, const unsigned char *buff4Kbyte)
  {
  const ISO9660_Boot_record_volume_descriptor *BRVD;
  unsigned cpt = 0;

  if (dp->bytepersector > DEFAULT_BUFFER_SIZE / 2) { /* i.e. dp->bytepersector == 2352 */
      unsigned char err = FS_readsector ((union filesystem_union *)common, 0x11, 1, data_adr(buff4Kbyte));
      if (err != 0) {
	  FDBG (("error 0x%X reading 4Kb at LBA = 0x11.\r\n", err));
	  return 0;
	  }
      BRVD = (const ISO9660_Boot_record_volume_descriptor *)buff4Kbyte;
      }
    else
      BRVD = (const ISO9660_Boot_record_volume_descriptor *)(buff4Kbyte + common->byte_per_block);

  FDBG (("1st volume_descriptor: header_id_0x00: 0x%X, version_0x01: 0x%X, boot_catalog_lba at %u, bootsys_name: '%s', ident_name: %c%c%c%c%c\r\n",
	BRVD->header_id_0x00, BRVD->version_0x01, BRVD->boot_catalog_lba, BRVD->bootsys_name,
	BRVD->ident_name[0], BRVD->ident_name[1], BRVD->ident_name[2], BRVD->ident_name[3], BRVD->ident_name[4]));

  for (;;) {
      static const struct {
	unsigned char  header_id_0x00;
	unsigned char  ident_name[5]; /* 'CD001' */
	unsigned char  version_0x01;
	unsigned char  bootsys_name[0x26 - 0x07 +1]; /* 'EL TORITO SPECIFICATION\0\0\0\0\0\0\0\0\0' */
	} __attribute__ ((packed)) torito_pattern = {
	.header_id_0x00 = 0x00,
	.ident_name = "CD001",
	.version_0x01 = 0x01,
	.bootsys_name = "EL TORITO SPECIFICATION"
	}, Volume_Descriptor_Set_Terminator = {
	.header_id_0x00 = 0xFF,
	.ident_name = "CD001",
	.version_0x01 = 0x01,
	.bootsys_name = ""
	};
      unsigned char err;
      /* Do not forget to put this warning nearby your computer:
	"This machine has no brain - Use your own." */

      if (memeql (BRVD, &Volume_Descriptor_Set_Terminator, sizeof (Volume_Descriptor_Set_Terminator))) {
	  FDBG (("Volume_Descriptor_Set_Terminator at sector 0x%X.\r\n", 0x11+cpt));
	  return 0;
	  }
      if (memeql (BRVD, &torito_pattern, sizeof (torito_pattern))) {
	  FDBG (("torito_pattern recognised at sector 0x%X, ", 0x11+cpt));
	  break;
	  }
      if (++cpt >= 10) {
	  FDBG (("Stop searching for torito pattern after sector 0x%X.\r\n", 0x11+cpt));
	  return 0;
	  }
// handle "cat cdrom.iso > /dev/hda8":
//      err = FS_readsector (dp, common->part, 0x11+cpt, 1, data_adr(buff4Kbyte));
      err = FS_readsector ((union filesystem_union *)common, (0x11+cpt) * common->sector_per_block, common->sector_per_block, data_adr(buff4Kbyte));
      if (err != 0) {
	  FDBG (("error 0x%X reading sector at LBA = 0x%X.\r\n", err, 0x11+cpt));
	  return 0;
	  }
      BRVD = (const ISO9660_Boot_record_volume_descriptor *)buff4Kbyte;
      FDBG (("volume_descriptor: header_id_0x00: 0x%X, version_0x01: 0x%X, boot_catalog_lba at %u, bootsys_name: '%s', ident_name: %c%c%c%c%c\r\n",
		BRVD->header_id_0x00, BRVD->version_0x01, BRVD->boot_catalog_lba, BRVD->bootsys_name,
		BRVD->ident_name[0], BRVD->ident_name[1], BRVD->ident_name[2], BRVD->ident_name[3], BRVD->ident_name[4]));
      }
  for (cpt = 0; cpt < sizeof (BRVD->unused1) / sizeof (BRVD->unused1[0]); cpt++)
      if (BRVD->unused1[cpt] != 0)
	  break;
  if (cpt == sizeof (BRVD->unused1) / sizeof (BRVD->unused1[0]))
      FDBG (("torito unused1 empty as specs, "));
    else
      FDBG (("torito unused1 nonzero at %u, i.e. byte offset %d, ", cpt, (unsigned)&BRVD->unused1[cpt] - (unsigned)BRVD));
  for (cpt = 0; cpt < sizeof (BRVD->unused2) / sizeof (BRVD->unused2[0]); cpt++)
      if (BRVD->unused2[cpt] != 0)
	  break;
  if (cpt == sizeof (BRVD->unused2) / sizeof (BRVD->unused2[0]))
      FDBG (("torito unused2 empty as specs.\r\n"));
    else
      FDBG (("torito unused2 nonzero at %u, i.e. byte offset %d.\r\n", cpt, (unsigned)&BRVD->unused2[cpt] - (unsigned)BRVD));
  return BRVD->boot_catalog_lba;
  }

#if DEBUG & DEBUG_FS
FS_FCT_PREFIX(display_iso9660_directory_record) static unsigned char *
display_iso9660_directory_record (const struct CD_directory_record_str *rdr)
  {
  static unsigned char name[NAME_LENGTH];
  unsigned char len = sizeof (name) - 1;

  /* NOTE: quite a few CDROM have got something wrong, adding 1900 to the wrong field for the date */
  FDBG (("\r\ndirectory_record: length %u, date %u/%u/%u time %u:%u:%u offset %d * 15 min; ",
	rdr->length,
	rdr->date_time.day_1_31,
	rdr->date_time.month_1_12,
	rdr->date_time.years_since_1900 + 1900,
	rdr->date_time.hour_0_23,
	rdr->date_time.minute_0_59,
	rdr->date_time.second_0_59,
	rdr->date_time.nb_15minute_GMT_offset));
  if (ENDIAN32(rdr->data_length_be) != rdr->data_length_le)
      FDBG(("[bi-endian32 fail: data_length_be: 0x%X data_length_le: 0x%X] ",
		rdr->data_length_be, rdr->data_length_le));
  if (ENDIAN32(rdr->first_logical_block_number_be) != rdr->first_logical_block_number_le)
      FDBG(("[bi-endian32 fail: first_logical_block_number_be: 0x%X first_logical_block_number_le: 0x%X] ",
		rdr->first_logical_block_number_be, rdr->first_logical_block_number_le));
  if (ENDIAN16(rdr->volume_sequence_number_be) != rdr->volume_sequence_number_le)
      FDBG(("[bi-endian16 fail: volume_sequence_number_be: 0x%X volume_sequence_number_le: 0x%X] ",
		rdr->volume_sequence_number_be, rdr->volume_sequence_number_le));
  if (rdr->extended_attribute_record_length != 0 || rdr->file_unit_size != 0
	|| rdr->interleave_gap_size != 0 || rdr->volume_sequence_number_le != 1)
      FDBG (("[UNUSUAL extended_attribute_record_length 0x%X, file_unit_size %u, "
	     "interleave_gap_size %u, volume_sequence_number %u] ",
	 rdr->extended_attribute_record_length, rdr->file_unit_size,
	 rdr->interleave_gap_size, rdr->volume_sequence_number_le));

  if (rdr->file_identifier_length < len)
      len = rdr->file_identifier_length;
  name[len] = '\0';
  while (len--)
      name[len] = rdr->file_identifier[len];
  FDBG (("first_logical_block %u, data_length %u, attribute: %s%s%s%s%s%s%s%s; "
	 "file_identifier_length %u, file_identifier '%s'",
	rdr->first_logical_block_number_le,
	rdr->data_length_le,
	rdr->attributes.hidden?		"hidden, " : "",
	rdr->attributes.directory?	"directory, " : "",
	rdr->attributes.associated_file?	"associated_file, " : "",
	rdr->attributes.extended_attr?	"extended_attr, " : "",
	rdr->attributes.extended_info?	"extended_info, " : "",
	rdr->attributes.reserved1?	"reserved1, " : "",
	rdr->attributes.reserved2?	"reserved2, " : "",
	rdr->attributes.has_hard_link?	"has_hard_link, " : "",
	rdr->file_identifier_length,
//	rdr->file_identifier	// not zero terminated
	name
	));
  return name;
  }
#endif /* DEBUG & DEBUG_FS */


FS_FCT_PREFIX(ISO_get_parameter) static inline unsigned
ISO_get_parameter (struct common_filesystem *common,
			const unsigned char *buff4Kbyte,
			unsigned *volume_sequence_number,
			unsigned *boot_catalog_lba,
			struct treat_directory_str *opaque)
  {
  struct diskparam_str *const dp = &DI.param[common->disk];
  struct partition_str *const dpp = &dp->partition[common->part];
  const ISO9660_Primary_volume_descriptor *pvd = (const ISO9660_Primary_volume_descriptor *)buff4Kbyte;
  const struct CD_directory_record_str *dir = &pvd->root_directory_record;

  if (sizeof (ISO9660_Primary_volume_descriptor) != 2048)
      __ERROR();	/* the linker will produce an error here */

  if (DEFAULT_BUFFER_SIZE < sizeof (ISO9660_Primary_volume_descriptor)) {
      FDBG (("ISO DEFAULT_BUFFER_SIZE=%u too small\r\n", DEFAULT_BUFFER_SIZE));
      return 0x0E;
      }

  if (   pvd->volume_descriptor_type != 1
//      || pvd->standard_identifier[0] != 'C'
//      || pvd->standard_identifier[1] != 'D'
//      || pvd->standard_identifier[2] != '0'
//      || pvd->standard_identifier[3] != '0'
      || *CAST (const unsigned *, pvd->standard_identifier) != *CAST (const unsigned *, "CD00")
      || pvd->standard_identifier[4] != '1'
      || pvd->volume_descriptor_version != 1) {
      FDBG (("%s: not ISOFS: volume_descriptor_type: 0x%X, standard_identifier: %c%c%c%c%c, "
	     "volume_descriptor_version 0x%X\r\n",
		__FUNCTION__,
		pvd->volume_descriptor_type,
		pvd->standard_identifier[0], pvd->standard_identifier[1], /* FIXME: could be '\0' char ... */
		pvd->standard_identifier[2], pvd->standard_identifier[3],
		pvd->standard_identifier[4], pvd->volume_descriptor_version));
      return 2;
      }

  FDBG (("%s: ", __FUNCTION__));
  if (ENDIAN32(pvd->volume_space_size_be) != pvd->volume_space_size_le) {
      FDBG (("bi-endian32 fail: volume_space_size_be: 0x%X volume_space_size_le: 0x%X\r\n",
		pvd->volume_space_size_be, pvd->volume_space_size_le));
      return 3;
      }
  if (ENDIAN16(pvd->volume_sequence_number_be) != pvd->volume_sequence_number_le) {
      FDBG (("bi-endian16 fail: volume_sequence_number_be: 0x%X volume_sequence_number_le: 0x%X\r\n",
		pvd->volume_sequence_number_be, pvd->volume_sequence_number_le));
      return 4;
      }
  if (ENDIAN16(pvd->volume_set_size_be) != pvd->volume_set_size_le) {
      FDBG (("bi-endian16 fail: volume_set_size_be: 0x%X volume_set_size_le: 0x%X\r\n",
		pvd->volume_set_size_be, pvd->volume_set_size_le));
      return 5;
      }
  if (pvd->volume_sequence_number_le > pvd->volume_set_size_le) {
      FDBG (("volume_sequence_number %u is bigger than volume_set_size %u\r\n",
		pvd->volume_sequence_number_le, pvd->volume_set_size_le));
      return 6;
      }
  if (ENDIAN16(pvd->logical_block_size_be) != pvd->logical_block_size_le) {
      FDBG (("bi-endian16 fail: logical_block_size_be: 0x%X logical_block_size_le: 0x%X\r\n",
		pvd->logical_block_size_be, pvd->logical_block_size_le));
      return 7;
      }
  if (pvd->logical_block_size_le == 0 || pvd->logical_block_size_le % dp->bytepersector != 0)
      FDBG (("    WARNING: pvd->logical_block_size_le = %u while dp->bytepersector = %u!\r\n    ",
		pvd->logical_block_size_le, dp->bytepersector));
  if (!dir->attributes.directory) {
      FDBG(("root directory not a directory...\r\n"));
      return 8;
      }
  if (ENDIAN32(dir->first_logical_block_number_be) != dir->first_logical_block_number_le) {
      FDBG(("bi-endian32 fail: first_logical_block_number_be: 0x%X first_logical_block_number_le: 0x%X, not a ISO directory sector\r\n",
		dir->first_logical_block_number_be, dir->first_logical_block_number_le));
      return 9;
      }
  if (ENDIAN32(dir->data_length_be) != dir->data_length_le) {
      FDBG(("bi-endian32 fail: data_length_be: 0x%X data_length_le: 0x%X, not a ISO directory sector\r\n",
		dir->data_length_be, dir->data_length_le));
      return 0xB;
      }
  /* NOTE: because of FAT+ISO filesystem with ISO higher priority, we shall not modify
	anything if its is not ISO but is still FAT: we are now sure it is ISO9660 */
  common->inodes_count = 0;
  common->blocks_count = pvd->volume_space_size_le;
  *volume_sequence_number = pvd->volume_sequence_number_le;
  /* sectorsize is not DI.param[fs->common.disk].bytepersector in ISO images inside another FS: */
  common->SectSize = 2048;
  common->byte_per_block = pvd->logical_block_size_le ?: common->SectSize;
  common->sector_per_block = common->byte_per_block / common->SectSize;
  if (common->sector_per_block == 0) /* WARNING: 2048 / 2352 = 0 !!! */
      common->sector_per_block = 1; /* used in "filesystem bigger than partition" */

  FDBG(("path_table_size_le %u at path_table_location_le %u (optional at %u), "
	"path_table_size_be %u at path_table_location_be %u (optional at %u)\r\n",
	pvd->path_table_size_le, pvd->path_table_location_le, pvd->optional_path_table_location_le,
	ENDIAN32(pvd->path_table_size_be), ENDIAN32(pvd->path_table_location_be), ENDIAN32(pvd->optional_path_table_location_be)));

#if DEBUG & DEBUG_FS
  display_iso9660_directory_record (dir);
#endif

  common->slash.inode = dir->first_logical_block_number_le;
  if (dpp->start != 0 && common->slash.inode > dpp->length) {
      FDBG(("Absolute LBA used in this session: root at %u in partition start %llu length %llu\r\n",
		common->slash.inode, dpp->start, dpp->length));
      common->first_data_block = (unsigned)dpp->start;
      common->blocks_count -= common->first_data_block;
      }
    else
      common->first_data_block = 0;
  common->slash.size = dir->data_length_le;

  const char *end = &pvd->volume_identifier[sizeof(pvd->volume_identifier) - 1];
  while (*end == ' ' && end > pvd->volume_identifier)
      end--;
  /* Store somewhere "NimbleX" to use "from=" instead of "fromiso=/" */
  {
  char *dst = common->fsname;
  const char *src = pvd->volume_identifier;
  while (src <= end && dst < &common->fsname[lastof(common->fsname)])
      *dst++ = (char)*src++;
  *dst = '\0';
  }

#if DEBUG & DEBUG_FS
  {
  char system_identifier[sizeof (pvd->system_identifier)];

  end = &pvd->system_identifier[sizeof(pvd->system_identifier) - 1];
  while (*end == ' ' && end > pvd->system_identifier)
      end--;
  system_identifier[end - pvd->system_identifier + 1] = '\0';
  while (end >= pvd->system_identifier) {
      system_identifier[end - pvd->system_identifier] = *end;
      end--;
      }
  FDBG (("%s: slash.inode %u, slash.size %u, system_identifier '%s', volume_identifier: '%s', volume_sequence_number %u valid.\r\n",
	__FUNCTION__, common->slash.inode, common->slash.size, system_identifier, common->fsname, *volume_sequence_number));
  }
#endif

  if (boot_catalog_lba) {	/* i.e. search_el_torito set */
      unsigned char err;

// handle "cat cdrom.iso > /dev/hda8":
      *boot_catalog_lba = ISO_get_boot_catalog_lba (dp, common, buff4Kbyte) * common->sector_per_block;
      if (*boot_catalog_lba == 0)
	  FDBG (("Boot catalog not found.\r\n"));
// handle "cat cdrom.iso > /dev/hda8":
//	else if ((err = FS_readsector (dp, common->part, *boot_catalog_lba,
//		DEFAULT_BUFFER_SIZE / common->byte_per_block , data_adr(buff4Kbyte))) != 0) {
	else if ((err = FS_readsector ((union filesystem_union *)common, *boot_catalog_lba,
		DEFAULT_BUFFER_SIZE / common->byte_per_block * common->sector_per_block,
		data_adr(buff4Kbyte))) != 0) {
	  FDBG (("error 0x%X reading %u sectors at LBA = %u.\r\n",
// handle "cat cdrom.iso > /dev/hda8":
//		err, DEFAULT_BUFFER_SIZE / common->byte_per_block, *boot_catalog_lba));
		err, DEFAULT_BUFFER_SIZE / common->byte_per_block * common->sector_per_block, *boot_catalog_lba));
	  *boot_catalog_lba = 0;
	  }
	else {
	  const unsigned short *ptr = (const unsigned short *)buff4Kbyte;
	  unsigned short checksum = 0, cpt = sizeof (ISO9660_Validation_entry) / sizeof(checksum) + 1;
	  const ISO9660_Validation_entry *entry = (const ISO9660_Validation_entry *)buff4Kbyte;

	  while (--cpt)
	      checksum += *ptr++;
	  FDBG (("ISO9660_Validation_entry: signature1_0x55 = 0x%X, signature2_0xAA = 0x%X, calculated checksum 0x%X, checksum field: 0x%X\r\n",
		entry->signature1_0x55, entry->signature2_0xAA, checksum, entry->checksum));
	  FDBG (("  header_id_0x01 = 0x%X, platform 0x%X (0==8086), reserved 0x%X, manufacturer '%s'\r\n",
		entry->header_id_0x01, entry->platform_id, entry->reserved, entry->manufacturer));
	  if (checksum != 0 || entry->signature1_0x55 != 0x55 || entry->signature2_0xAA != 0xAA) {
	      FDBG (("Boot catalog not recognised.\r\n"));
	      }
	    else {
	      const ISO9660_Default_entry *catalog = (ISO9660_Default_entry *)buff4Kbyte;
	      while (++catalog < (ISO9660_Default_entry *)(&buff4Kbyte[DEFAULT_BUFFER_SIZE])) {
		  FDBG (("entry[%u]: boot_indicator = 0x%X (0x88=bootable), load segment 0x%X, sector_count %u, system_type 0x%X, unused 0x%X\r\n",
			catalog - (ISO9660_Default_entry *)buff4Kbyte, catalog->boot_indicator, catalog->load_segment,
			catalog->sector_count, catalog->system_type, catalog->unused));
		  if (catalog->boot_media_type == 0)
		      FDBG (("  no emulation, "));
		    else if (catalog->boot_media_type == 1)
		      FDBG (("  emulate 1.2 Mb, "));
		    else if (catalog->boot_media_type == 2)
		      FDBG (("  emulate 1.44 Mb, "));
		    else if (catalog->boot_media_type == 3)
		      FDBG (("  emulate 2.88 Mb, "));
		    else if (catalog->boot_media_type == 4)
		      FDBG (("  emulate HD, "));
		    else
		      FDBG (("  emulate unknown 0x%X, ", catalog->boot_media_type));
		  FDBG (("sector_count %u, sector_lba %U, reserved: ", catalog->sector_count, catalog->sector_lba));
		  for (cpt = 0; cpt < sizeof (catalog->reserved); cpt++)
		      FDBG (("0x%X, ", catalog->reserved[cpt]));
		  FDBG (("\r\n"));
		  if (catalog->sector_count == 0) {
		      catalog++; /* also malloc for end marker */
		      break;
		      }
		  }
#ifdef DISK_USE_MALLOC
	      unsigned newsize = (unsigned)catalog - (unsigned)buff4Kbyte;
	      allocmem *newptr = REALLOC (dpp->ElToritoCatalog, opaque->currentNBeltorito * sizeof(ISO9660_Default_entry) + newsize, "ElTorito");
	      if (newptr != 0) {
		  dpp->ElToritoCatalog = (ISO9660_Default_entry *)newptr;
		  opaque->newNBeltorito = newsize / sizeof (ISO9660_Default_entry);
		  memcpy (&dpp->ElToritoCatalog[opaque->currentNBeltorito], buff4Kbyte, newsize);
		  }
#else
	      FDBG (("no DISK_USE_MALLOC, do not store El-Torito boot catalog descriptors\r\n"));
#endif
	      }
	  }
      }

  return 0;
  }

FS_FCT_PREFIX(ISO_get_sector_chain) static struct sector_chain_str *
ISO_get_sector_chain (union filesystem_union *fs,
			unsigned inode, unsigned filesize,
			struct sector_chain_str **sector_chain_buffer_ptr)
  {bound_stack();{
  struct sector_chain_str *sector_chain_buffer = *sector_chain_buffer_ptr;

  inode -= fs->common.first_data_block; /* dpp->start re-added in read disk */
  if (inode >= fs->common.blocks_count) {
      FDBG (("error inode %u too high, max %llU.\r\n", inode, fs->common.blocks_count));
      return 0;
      }

// handle "cat cdrom.iso > /dev/hda8":
//  sector_chain_buffer[0].lba = inode; /* root treated the same as other files */
  sector_chain_buffer[0].lba = inode * fs->common.sector_per_block; /* root treated the same as other files */
  sector_chain_buffer[0].nb = DIVROUNDUP (filesize, fs->common.SectSize);
  sector_chain_buffer[1].lba = 0; /* safety */
  sector_chain_buffer[1].nb = 0;
  FDBG (("%s returns [[%llu, +%u][0,0]]\r\n", __FUNCTION__, sector_chain_buffer[0].lba, sector_chain_buffer[0].nb));
  return sector_chain_buffer;
  }}

FS_FCT_PREFIX(ISO_treat_directory) static int
ISO_treat_directory (struct treat_directory_str *opaque,
		      const struct CD_directory_record_str *dir, unsigned len)
  {bound_stack();{
  const struct CD_directory_record_str *lastdir = (const struct CD_directory_record_str *)((unsigned)dir + len),
		*startdir = dir;

  while (dir < lastdir) {
#if DEBUG & DEBUG_FS
      unsigned char *name = display_iso9660_directory_record (dir);
#endif
      if (dir->length == 0) {
	  FDBG (("[%s: dir entry length=0, Shall we end processing this directory (dir 0x%X, lastdir 0x%X)] ",
		__FUNCTION__, dir, lastdir));
	  /* Treat possible realignment at 2048 (max block read 4096 bytes), seen in slax-6.1.2.iso */
	  if ((unsigned)lastdir - (unsigned)dir >= 2048)
	      dir = (struct CD_directory_record_str *)((unsigned)startdir + 2048);
	    else {
//	      return 1; /* can stop processing of this directory */
	      break; /* There is a big amount of crap in between the last entry and the next directory sector...
			for instance in eeebuntu-3.0-standard is file /boot/id.tr valid??? */
	      }
	  }
      if (dir->length < sizeof (struct CD_directory_record_str)) {
	  FDBG (("[%s: dir entry length=%u is < %u, not a ISO directory sector] ",
		__FUNCTION__, dir->length, sizeof (struct CD_directory_record_str)));
	  return -1; /* abort processing of this directory */
	  }
      if (ENDIAN32(dir->data_length_be) != dir->data_length_le) {
	  FDBG (("[%s: bi-endian32 fail: data_length_be: 0x%X data_length_le: 0x%X, not a ISO directory sector] ",
		__FUNCTION__, dir->data_length_be, dir->data_length_le));
	  return -1; /* abort processing of this directory */
	  }
      if (ENDIAN32(dir->first_logical_block_number_be) != dir->first_logical_block_number_le) {
	  FDBG (("[%s: bi-endian32 fail: first_logical_block_number_be: 0x%X first_logical_block_number_le: 0x%X, not a ISO directory sector] ",
		__FUNCTION__, dir->first_logical_block_number_be, dir->first_logical_block_number_le));
	  return -1; /* abort processing of this directory */
	  }
      if (ENDIAN16(dir->volume_sequence_number_be) != dir->volume_sequence_number_le) {
	  FDBG (("[%s: bi-endian16 fail: volume_sequence_number_be: 0x%X volume_sequence_number_le: 0x%X, not a ISO directory sector] ",
		__FUNCTION__, dir->volume_sequence_number_be, dir->volume_sequence_number_le));
	  return -1; /* abort processing of this directory */
	  }
      const char *file_identifier = dir->file_identifier;
      unsigned char file_identifier_length = dir->file_identifier_length;
      unsigned char ReadOnly = 1;
      if (dir->length > sizeof (struct CD_directory_record_str) + file_identifier_length + 11) {
	  /* See http://www.osdever.net/downloads/docs/rrip112.zip */
	  const unsigned RockRidgeSig = *(unsigned *)
		"RR"	/* Rock Ridge signature */
		"\005"	/* Rock Ridge signature length */
		"\001";	/* Rock Ridge version */
	  const char *endisoname = file_identifier + file_identifier_length;
	  if ((unsigned)endisoname & 1)
	      endisoname++;	/* skip padding */
	  /* We ignore the byte Rock Ridge signature data (length = 5) seen as 0x81, 0x89, ... */
	  /* The byte Rock Ridge NaMe flags just after "NM" do not seem present */
	  /* We assume the "NM" entry is the first one if present, before "PX" */
	  if (*(unsigned *)endisoname == RockRidgeSig && *(unsigned short*)&endisoname[5] == *(unsigned short*)"NM") {
//char name[32]; unsigned cpt;
//for (cpt = 0; cpt < file_identifier_length; cpt++) name[cpt] = file_identifier[cpt]; name[cpt] = '\0';
//printf ("RR (%u) replace '%s' by ", dir->extended_attribute_record_length, name);
	      file_identifier = endisoname + 10;	/* NOT zero terminated */
	      file_identifier_length = endisoname[7] - 5;
//for (cpt = 0; cpt < file_identifier_length; cpt++) name[cpt] = file_identifier[cpt]; name[cpt] = '\0';
//printf ("'%s' len %u\n", name, file_identifier_length); _BIOS_getkey();
	      if (*(unsigned short *)&file_identifier[file_identifier_length] == *(unsigned short*)"PX") {
		  ReadOnly = !(file_identifier[file_identifier_length + 4] & 0x80); /* to check: octal 0000200 S_IWUSR write permission (owner) */
//if (*(const int *)file_identifier == *(const int *)"msdo") {printf ("%s: PX len %u (==044?) vers %u (==1?) val le: 0x%X /be: 0x%X  ReadOnly %u\r\n", file_identifier, file_identifier[file_identifier_length + 2], file_identifier[file_identifier_length + 3], *(int *)&file_identifier[file_identifier_length + 4], *(int *)&file_identifier[file_identifier_length + 8], ReadOnly); _BIOS_getkey();}
//I get: PX len 36 (==044?) vers 1 (==1?) val le 0x816D /be: 0x6D810000 ReadOnly 1 whatever file attribute (genisoimage), 0x816D == 0100555 , 0100000 = S_IFREG*/
		  }
	      }
	    else if (*(unsigned *)endisoname == RockRidgeSig && *(unsigned short*)&endisoname[5] == *(unsigned short*)"PX") {
	      ReadOnly = !(endisoname[9] & 0x80); /* to check: octal 0000200 S_IWUSR write permission (owner) */
//if (*(const int *)file_identifier == *(const int *)"msdo") {printf ("%s: PX len %u (==044?) vers %u (==1?) val le: 0x%X /be: 0x%X ReadOnly %u\r\n", file_identifier, endisoname[7], endisoname[8], *(int *)&endisoname[9], *(int *)&endisoname[13], ReadOnly); _BIOS_getkey();}
	      }
//else { printf ("noRR: 0x%X 0x%X 0x%X 0x%X 0x%X 0x%X 0x%X 0x%X\n", endisoname[0], endisoname[1], endisoname[2], endisoname[3], endisoname[4], endisoname[5], endisoname[6], endisoname[7]);  _BIOS_getkey(); }
	  }
      if (dir->file_unit_size != 0 || dir->interleave_gap_size != 0) {
	  FDBG (("[%s: dir entry is for interleaved file(unit %u, gap %u), ignored] ",
		__FUNCTION__, dir->file_unit_size, dir->interleave_gap_size));
	  }
	else if (dir->volume_sequence_number_le != opaque->filesystem.iso9660.volume_sequence_number) {
	  FDBG (("[%s: dir entry is for another volume_sequence_number (%u != %u), ignored] ",
		__FUNCTION__, dir->volume_sequence_number_le, opaque->filesystem.iso9660.volume_sequence_number));
	  }
	else if (opaque->nb >= NB_DESC_ARRAY - 1) {
	  FDBG ((" [%s: no more space in NB_DESC_ARRAY] ", __FUNCTION__));
	  return -2;
	  }
	else {
	  /* NOTE: the file version (any digit after and including semicolon) is removed by tst_elem()! */
	  struct slashdir *whichdir;
	  switch (tst_elem (opaque, file_identifier, file_identifier_length)) {
	      case tst_elem_iso:
#ifdef NB_ISO
		  if (!dir->attributes.directory && BOOTWAY.iso.nb_found < NB_ISO - 1) {
		      BOOTWAY.iso.found[BOOTWAY.iso.nb_found].inode = dir->first_logical_block_number_le;
		      BOOTWAY.iso.found[BOOTWAY.iso.nb_found].size = dir->data_length_le;
		      unsigned i = 0, j;
		      if (!opaque->inRoot) {
			  const char *path = opaque->which_dir? SearchPathArray[opaque->which_dir - 1] : copy_gujin_param.scanpath;
			  while ((BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i] = path[i]) != '\0')
			      i++;
			  BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i++] = '/';
			  }
		      for (j = 0; i < sizeof (BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename) - 1 && j < file_identifier_length; i++)
			  if ((BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i] = file_identifier[j++]) == '\0')
			      break;
		      BOOTWAY.iso.found[BOOTWAY.iso.nb_found].filename[i] = '\0';
		      BOOTWAY.iso.nb_found++;
		      }
#endif
		  break;
	      case tst_elem_commandfile:
#ifdef COMMANDFILE
		  if (!dir->attributes.directory)
		      file_treat_commandfile (&opaque->filesystem, dir->first_logical_block_number_le, dir->data_length_le);
#endif
		  break;
	      case tst_elem_maybe:
		  if (dir->attributes.directory || !file_check_aa55_sig (&opaque->filesystem, dir->first_logical_block_number_le)) {
		      FDBG ((" [%s: maybe file without 0xAA55 signature] ", __FUNCTION__));
		      break;
		      }
	      case tst_elem_found:
#ifdef MIN_KERNEL_INITRD_SIZE
		  if (dir->data_length_le < MIN_KERNEL_INITRD_SIZE && IS_KERNEL_OR_INITRD (&opaque->array[opaque->nb])) {
		      FDBG ((" [%s: filesize too small: %u] ", __FUNCTION__, dir->data_length_le));
		      break;
		      }
#endif
		  if (opaque->nb >= NB_DESC_ARRAY - 1)
		      FDBG ((" [%s: no more space in NB_DESC_ARRAY to treat %s] ", __FUNCTION__, name));
		    else if (!dir->attributes.directory) {
		      unsigned last_modification_date, nbdays = 0, cptmonth;
		      for (cptmonth = 1; cptmonth < dir->date_time.month_1_12; cptmonth++)
			  nbdays += dayinmonth[cptmonth];
		      nbdays += dir->date_time.day_1_31 - 1;
		      nbdays += (dir->date_time.years_since_1900 + 1900 - 2001) / 4; /* in 2005 add 1 leap day */
		      if (dir->date_time.month_1_12 >= 3 && (dir->date_time.years_since_1900 + 1900) % 4 == 0)
			  nbdays ++; /* add 1 day after Feb 29th on leap year */
		      if (dir->date_time.years_since_1900 + 1900 < 2001)
			  last_modification_date = 0;
			else
			  last_modification_date = (dir->date_time.years_since_1900 + 1900 - 2001) * 365 * 24 * 60 * 60
					+ nbdays * 24 * 60 * 60 + dir->date_time.hour_0_23 * 60 * 60 + dir->date_time.minute_0_59 * 60 + dir->date_time.second_0_59
					- dir->date_time.nb_15minute_GMT_offset * 15 * 60;
		      FDBG ((" [%s: adding file '%s' size %u bytes date %u sec after Jan 1st, 2001] ", __FUNCTION__, name, dir->data_length_le, last_modification_date));
		      opaque->array[opaque->nb].filesize = dir->data_length_le;
		      opaque->array[opaque->nb].inode = dir->first_logical_block_number_le;
		      opaque->array[opaque->nb].last_modification_date = last_modification_date;
		      opaque->array[opaque->nb].ReadOnly = ReadOnly;	/* mostly for BDI files */
		      if (opaque->filesystem.common.fsname[0] != '\0') {
			  opaque->array[opaque->nb].iso_fsname = MALLOC(strlen(opaque->filesystem.common.fsname)+1, "isofsna");
			  if (opaque->array[opaque->nb].iso_fsname)
			      strcpy (opaque->array[opaque->nb].iso_fsname, opaque->filesystem.common.fsname);
			  }
		      opaque->nb++;
		      }
		    else {
		      /* for instance "/initrd" directory */
		      FDBG ((" [%s: file '%s' is a directory] ", __FUNCTION__, name));
		      }
		  break;
	      case tst_elem_slashboot:
		  whichdir = &opaque->filesystem.common.slashboot;
		  goto ISOFS_tst_elem_slashcommon;
	      case tst_elem_slashinstall386:
		  whichdir = &opaque->filesystem.common.slashinstall386;
		  goto ISOFS_tst_elem_slashcommon;
	      case tst_elem_slashinstallamd64:
		  whichdir = &opaque->filesystem.common.slashinstallamd64;
		  goto ISOFS_tst_elem_slashcommon;
	      case tst_elem_slashisolinux:
		  whichdir = &opaque->filesystem.common.slashisolinux;
		  goto ISOFS_tst_elem_slashcommon;
	      case tst_elem_slashcasper:
		  whichdir = &opaque->filesystem.common.slashcasper;
		  goto ISOFS_tst_elem_slashcommon;
	      case tst_elem_slashlive:
		  whichdir = &opaque->filesystem.common.slashlive;
		  goto ISOFS_tst_elem_slashcommon;
ISOFS_tst_elem_slashcommon:
		  if (!dir->attributes.directory)
		      FDBG ((" [%s: file '%s' not a directory] ", __FUNCTION__, name));
		    else if (!opaque->inRoot)
		      FDBG ((" [%s: directory '%s' but already in a subdirectory] ", __FUNCTION__, name));
		    else {
		      whichdir->inode = dir->first_logical_block_number_le;
		      whichdir->size = dir->data_length_le;
		      FDBG ((" [subdirectory '%s' at %u size %u] ", name, whichdir->inode, whichdir->size));
		      }
		  break;
	      case tst_elem_ignore:
		  break;
	      }
	  }
      dir = (const struct CD_directory_record_str *) ((unsigned)dir + dir->length);
      }
  FDBG ((" [%s: need to read more sectors] ", __FUNCTION__));
  return 0;
  }}
#endif /* DISK_SUPPORT & ISOFS_PROBE */

#if DISK_SUPPORT & (E2FS_PROBE|FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|ISOFS_PROBE)

/* struct treat_directory_str *opaque only needed if search_boot_catalog != 0 */
FS_FCT_PREFIX(open_filesystem) static unsigned
open_filesystem (union filesystem_union *fs, unsigned search_boot_catalog, struct treat_directory_str *opaque)
  {bound_stack();{
  struct diskparam_str *const dp = &DI.param[fs->common.disk];
  struct partition_str *const dpp = &dp->partition[fs->common.part];
  unsigned tmp;
  unsigned char err;

  FDBG (("## %s Disk %u part %u type 0x%X, read first 4 Kbytes: ",
	__FUNCTION__, fs->common.disk, fs->common.part, dpp->type));

  fs->common.SectSize = 0;
  fs->common.fsname[0] = '\0';
  err = FS_readsector (fs, 0, DEFAULT_BUFFER_SIZE / dp->bytepersector, data_adr(fourKbuffer));
  if (err != 0) {
      FDBG (("error 0x%X reading 4Kb at LBA = 0.\r\n", err));
      }
    else if (   memeql (fourKbuffer + 4096 - 10, "SWAP_SPACE", sizeof ("SWAP_SPACE") - 1)
	     || memeql (fourKbuffer + 4096 - 10, "SWAPSPACE2", sizeof ("SWAPSPACE2") - 1)
	     || memeql (fourKbuffer + 4096 - 10, "S1SUSPEND", sizeof ("S1SUSPEND") - 1)
	     || memeql (fourKbuffer + 4096 - 10, "S2SUSPEND", sizeof ("S2SUSPEND") - 1)) {
      char name[11];
      memcpy (name, (char *)fourKbuffer + 4096 - 10, 10);
      name[10] ='\0';
      FDBG (("SWAP partition signature: '%s'.\r\n", name));
      dpp->misc.swap_partition = 1;
      return 0xE;
      }
    else if ((dpp->name[0] == '\0' || *(unsigned *)dpp->name == *(unsigned *)"flop") /* i.e. "floppy" */
#ifdef NB_ISO
		&& BOOTWAY.iso.nb_remap == 0
#endif
		) {
      const bootsector_t	*fat16 = (const bootsector_t *)fourKbuffer;
      const bootsector_FAT32_t	*fat32 = (const bootsector_FAT32_t *)fourKbuffer;
      FDBG (("initialise the partition name: "));
      if (((unsigned short *)fourKbuffer)[255] != 0xAA55)
	  FDBG (("no 0xAA55 signature.\r\n"));
	else if (fat16->before.part2.Signaturebyte0x29 == 0x29 || fat32->before.part2.Signaturebyte0x29 == 0x29
		|| fat16->before.part2.Signaturebyte0x29 == 0x28) {
	  const unsigned char *Volumelabel = (fat16->before.part2.Signaturebyte0x29 == 0x29 || fat16->before.part2.Signaturebyte0x29 == 0x28)?
		fat16->before.part2.Volumelabel : fat32->before.part2.Volumelabel;
	  if (Volumelabel[0] != ' ') {
	      unsigned char *dst = dpp->name;
	      const unsigned char *src = Volumelabel;
	      for (;;) {
		  if ((*dst++ = *src++) == '\0')
		      break;
		  if (src == &Volumelabel[lastof(fat16->before.part2.Volumelabel)]) {
		      *dst = '\0';
		      break;
		      }
		  }
	      FDBG (("FAT%s 0x29 signature, volume label '%s'.\r\n",
			(Volumelabel == fat16->before.part2.Volumelabel)?"12/16" : "32", dpp->name));
	      }
	    else
	      FDBG (("FAT%s 0x29 signature, but volume label not initialised.\r\n",
			(Volumelabel == fat16->before.part2.Volumelabel)?"12/16" : "32"));

	  }
	else
	  FDBG (("no FAT12/16 nor FAT32 0x29 signature.\r\n"));
      }
    else
      FDBG (("name already set to '%s' or in ISO\r\n", dpp->name));

  const char *srcfsname = 0;
#if DISK_SUPPORT & E2FS_PROBE
  unsigned long long device_size = dpp->length * dp->bytepersector;
#ifdef NB_ISO
  if (BOOTWAY.iso.nb_remap != 0)
      device_size = BOOTWAY.iso.found[BOOTWAY.iso.current_used].size;
#endif

  /* TODO: we force 512 bytes/sectors even on E2FS images in CDROM, we should take the size of the file / number of sector in the FS */
  if (err == 0 && (tmp = E2FS_get_parameter (fs, fourKbuffer, device_size, dp->bytepersector)) == 0) {
      char *s_last_mounted = ((struct ext_super_block *)(fourKbuffer + SUPERBLOCK_OFFSET))->s_last_mounted;
      while (*s_last_mounted && s_last_mounted - ((struct ext_super_block *)(fourKbuffer + SUPERBLOCK_OFFSET))->s_last_mounted < 20)
		s_last_mounted++;
      if (*s_last_mounted == 0)
	  strcpy (s_last_mounted+1, " GPL v2, Copyright 2003 Etienne Lorrain.");
      fs->common.treatdir = (typeof(fs->common.treatdir))E2FS_treat_directory;
      fs->common.get_sector_chain = E2FS_get_sector_chain;
      if (fs->common.fsname[0] != '\0') /* if unnamed, keep Gujin default "part 3" with a space so not used as root=LABEL= Linux cmdline */
	  srcfsname = fs->common.fsname;
      }
    else
#endif

#if DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE)
  /* WARNING: FAT_get_parameter can modify fourKbuffer - has to be last FS probed at LBA 0: */
	if (err == 0 && (tmp = FAT_get_parameter (&fs->common, fourKbuffer,
			&fs->fat.start_fat, &fs->fat.nb_fat, &fs->fat.fat_size,
			&fs->fat.start_dir, &fs->fat.fatbit)) == 0) {
      fs->common.treatdir = (typeof(fs->common.treatdir))FAT_treat_directory;
      fs->common.get_sector_chain = FAT_get_sector_chain;
      }
    else
#endif

#if DISK_SUPPORT & ISOFS_PROBE
// handle "cat cdrom.iso > /dev/hda8" (even with partition type = 0x83 i.e. Linux)
	if ((err = FS_readsector (fs, 0x10 * 2048 / dp->bytepersector, DEFAULT_BUFFER_SIZE / dp->bytepersector, data_adr(fourKbuffer))) != 0) {
      FDBG (("error 0x%X reading 4Kb at LBA = 0x%X.\r\n", err, 0x10 * fs->common.sector_per_block));
      return 0xF;
      } /* We have sector 0x10 and 0x11 in the buffer if 2048 bytes/sectors, not for 2352 bytes/sectors*/
    else if ((tmp = ISO_get_parameter (&fs->common, fourKbuffer, &fs->iso9660.volume_sequence_number,
		search_boot_catalog? &fs->iso9660.boot_catalog_lba : 0, opaque)) == 0) {
      fs->common.treatdir = (typeof(fs->common.treatdir))ISO_treat_directory;
      fs->common.get_sector_chain = ISO_get_sector_chain;
      /* Do not set the name as the default "CDROM", it is even less informative than current "session 3" */
      if (fs->common.fsname[0] != '\0' && fs->common.fsname[0] != ' '
	&& *CAST (const unsigned long long *, fs->common.fsname) != *CAST (const unsigned long long *, "CDROM   "))
	  srcfsname = fs->common.fsname;
      }
    else
#endif
      return 1;

  if (BOOTWAY.iso.nb_remap == 0 && srcfsname) {
      char *dst = (char *)dpp->name;
      while (*srcfsname && dst < (char *)&dpp->name[lastof(dpp->name)])
	  *dst++ = *srcfsname++;
      *dst = '\0';
      }

  FDBG (("FSname '%s': byte_per_block %u bytes, sector_per_block %u, first_data_block %u, inodes_count %u, blocks_count %llU.\r\n",
      dpp->name ?: (unsigned char *)"NULL", fs->common.byte_per_block, fs->common.sector_per_block, fs->common.first_data_block, fs->common.inodes_count, fs->common.blocks_count));

#ifdef NB_ISO
  if (BOOTWAY.iso.nb_remap == 0) {
      BOOTWAY.iso.gap_sectors_shift = __builtin_ffs (fs->common.sector_per_block) - 1;
      BOOTWAY.iso.intergap_shift = __builtin_ffs (fs->common.sector_per_block * fs->common.byte_per_block / 4) - 1;
      /* With 4096 bytes/blocks, 1024 * 4096-blocks (8192 sectors) will be followed by 4 K (8 sectors). */
//printf ("intergap %u, gap_sectors %u\r\n", 1 << BOOTWAY.iso.intergap_shift, 1 << BOOTWAY.iso.gap_sectors_shift);
#if DISK_SUPPORT & ISOFS_PROBE
      if (fs->common.get_sector_chain == ISO_get_sector_chain)
	  BOOTWAY.iso.external_fs = 1;
	else
#endif
#if DISK_SUPPORT & E2FS_PROBE
	     if (fs->common.get_sector_chain == E2FS_get_sector_chain)
	  BOOTWAY.iso.external_fs = 2;
	else
#endif
#if DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE)
	     if (fs->common.get_sector_chain == FAT_get_sector_chain)
	  BOOTWAY.iso.external_fs = 3;
	else
#endif
	  BOOTWAY.iso.external_fs = 0;
      }
#endif

#if DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE)
  if ((fs->common.get_sector_chain != FAT_get_sector_chain || dpp->start != 0) /* FAT floppy partition with BEER sector  */
#ifdef NB_ISO
		&& BOOTWAY.iso.nb_remap == 0
#endif
	)
#endif
      {
      unsigned long long fs_blocks_count;
      unsigned char spb = __builtin_ffs (fs->common.sector_per_block) - 1;

      if (fs->common.sector_per_block == (1U << spb))
	  fs_blocks_count = fs->common.blocks_count << spb;
	else /* then there is a unhandled limit ... */
	  fs_blocks_count = ul_mul_ul((unsigned)fs->common.blocks_count, fs->common.sector_per_block);

      if (fs_blocks_count > dpp->length) {
	  printf (DISK_S_PARTITION_U_FILESYS_BIGGER_PARTITION, dp->diskname, dpp->OSnumber);
	  DBG (("[ERROR %s: filesystem bigger (%llu) than partition (%llu)!] ",
		__FUNCTION__, fs_blocks_count, dpp->length));
//printf ("fs_blocks_count %llU, dpp->length %llU, dp->nbtotalsector %llU\r\n", fs_blocks_count, dpp->length, dp->nbtotalsector);
	  dpp->misc.fsanalyse_toobig = 1;
	  }
#if DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|E2FS_PROBE) /* do not complain for tinycdrom */
	else if (fs_blocks_count != dpp->length) {
	  unsigned long long wasted = dpp->length - fs_blocks_count;
	  FDBG (("[filesystem wastes %llu sectors of the partition (partition: %llU, filesystem: %llU * %u)] ",
		wasted, dpp->length, fs->common.blocks_count, fs->common.sector_per_block));
	  if (wasted > fs_blocks_count / 8 && wasted >= 256
#if DISK_SUPPORT & ISOFS_PROBE
		&& fs->common.get_sector_chain != ISO_get_sector_chain
#endif
		) {
	      printf (DISK_S_PARTITION_U_WASTES_U_SECTOR, dp->diskname, dpp->OSnumber, (unsigned)wasted);
	      }
	  }
#endif
      }

  return 0;
  }}

#ifdef COMMANDFILE
FS_FCT_PREFIX(treat_cmdline) inline void
treat_cmdline(const char *cptr, const char *end, const unsigned char *fsname,
		unsigned char *in_comment, const char **start_non_whitespace)
  {
  static unsigned initial_cmdfile_len;

  if (*start_non_whitespace != (const char *)-1)
      *start_non_whitespace = cptr;
    else
      initial_cmdfile_len = UTIL.cmdfile_len;
  while (cptr < end) {
      if (*in_comment && *cptr++ != '\n')
	  continue;
      if (*cptr == '#') {
	  *in_comment = 1;
	  if (*start_non_whitespace != (const char *)-1)
	      goto store_line;
	  cptr++;
	  continue;
	  }
      *in_comment = 0;
      if (*start_non_whitespace == (const char *)-1) {
	  if (*cptr != ' ' && *cptr != '\t' && *cptr != '\r' && *cptr != '\n')
	      *start_non_whitespace = cptr;
	  cptr++;
	  continue;
	  }
      if (*cptr == '\n' || cptr == end - 1) {
store_line:	;
	  unsigned len = cptr - *start_non_whitespace, strlen_fsname = 0;
	  if (len == 0)
	      break;
	  if (fsname)
	      strlen_fsname = strlen ((const char *)fsname);
#ifdef FS_USE_MALLOC
	  allocmem *tmpptr = REALLOC (UTIL.cmdfile_base, UTIL.cmdfile_len + len + 2 + strlen_fsname, "cmdline");
#else
	  allocmem *tmpptr;
	  if (UTIL.cmdfile_len + len + 1 + strlen_fsname > sizeof(cmdfile_base_buffer))
	      tmpptr = 0;
	    else
	      tmpptr = cmdfile_base_buffer;
#endif
	  if (tmpptr) {
	      UTIL.cmdfile_base = tmpptr;
	      if ((*start_non_whitespace)[0] == (*start_non_whitespace)[1] && strlen_fsname) {
		  UTIL.cmdfile_base[UTIL.cmdfile_len++] = (*start_non_whitespace)[0];
		  unsigned short xcpt;
		  for (xcpt = 0; xcpt < strlen_fsname; xcpt++)
		      UTIL.cmdfile_base[UTIL.cmdfile_len++] = fsname[xcpt];
		  UTIL.cmdfile_base[UTIL.cmdfile_len++] = (*start_non_whitespace)[1];
		  (*start_non_whitespace) += 2;
		  len -= 2;
		  }
	      while (len--) {
		  /* strip '\r' of "\r\n" and strip "\\ *\n[ \t]*" and "\\ *\r\n[ \t]*" */
		  if (**start_non_whitespace == '\\') {
		      const char *search_nonspace = *start_non_whitespace + 1;
		      while (*search_nonspace == ' ' || *search_nonspace == '\t' || *search_nonspace == '\r')
			  search_nonspace++;
		      if (*search_nonspace == '\n') {
			  *start_non_whitespace = (const char *)-1;
			  break;
			  }
			else
			  UTIL.cmdfile_base[UTIL.cmdfile_len++] = *(*start_non_whitespace)++;
		      }
		    else if (**start_non_whitespace != '\r')
		      UTIL.cmdfile_base[UTIL.cmdfile_len++] = *(*start_non_whitespace)++;
		    else
		      (*start_non_whitespace)++;
		  }
	      if (cptr == end - 1)
		  UTIL.cmdfile_base[UTIL.cmdfile_len++] = *(*start_non_whitespace)++;
		else if (*start_non_whitespace != (const char *)-1) {
		  /* Strip identical lines: */
		  const char *const end_added_line = &UTIL.cmdfile_base[UTIL.cmdfile_len];
		  UTIL.cmdfile_base[UTIL.cmdfile_len++] = '\n';
		  const char *scan = end_added_line - 1;
		  while (scan > UTIL.cmdfile_base && *scan != '\n')
		      scan--;
		  /* assert(*end_added_line == '\n'); */
		  while (scan > UTIL.cmdfile_base) { /* one more line before, we know (*scan == '\n') */
		      const char *added = end_added_line;
		      while (scan > UTIL.cmdfile_base && *--added == *--scan)
			  if (*scan == '\n')
			      break;
		      if (scan == UTIL.cmdfile_base || *scan == '\n') { /* found an identical line */
			  UTIL.cmdfile_len = initial_cmdfile_len;
			  break;
			  }
		      while (scan > UTIL.cmdfile_base && *--scan != '\n')
			  continue;
		      }
		  initial_cmdfile_len = UTIL.cmdfile_len;
		  }
	      UTIL.cmdfile_base[UTIL.cmdfile_len] = '\0';
	      }
	  *start_non_whitespace = (const char *)-1;
	  }
      cptr++;
      }
  }
#endif

FS_FCT_PREFIX(file_treat) static unsigned
file_treat (union filesystem_union *fs,
		unsigned inode, unsigned *filesize, unsigned skipoffset,
		int (*treat) (void *, unsigned char *, unsigned), void *opaque)
  {bound_stack();{
  struct partition_str *const dpp = &DI.param[fs->common.disk].partition[fs->common.part];
  struct sector_chain_str *current, *chain;

#ifdef FS_USE_MALLOC
  /* alloc also for the end marker: */
  struct sector_chain_str *sector_chain_buffer = MALLOC (2 * sizeof (struct sector_chain_str), "filetrea");

  if (sector_chain_buffer == 0) {
      *filesize = 0;
      FDBG (("%s: malloc failed!\r\n", __FUNCTION__));
      return 0x600;
      }
#else
  struct sector_chain_str bigarray[FILE_MAX_FRAGMENT];
  struct sector_chain_str *sector_chain_buffer = bigarray;
#endif

  chain = fs->common.get_sector_chain (fs, inode, *filesize, &sector_chain_buffer);

  if (chain == 0) {
      *filesize = 0;
      FDBG (("%s: cannot get LBA chain!\r\n", __FUNCTION__));
      dpp->misc.fsanalyse_error = 1;
#ifdef FS_USE_MALLOC
      FREE (sector_chain_buffer);
#endif
      return 0x200;
      }

  {
  unsigned nbsect = 0;
  for (current = chain; current->nb != 0; current++) {
      nbsect += current->nb;
      }
  if (nbsect < (*filesize + fs->common.SectSize - 1) / fs->common.SectSize)
      FDBG (("Warning: number of sectors: %d too small to map the file of size: %u.\r\n", nbsect, *filesize));
  }

  if (treat == 0) {
      unsigned i, *max_nb_elem = (unsigned *)skipoffset;
      struct sector_chain_str *array = opaque;
      FDBG (("%s: request LBA chain only success (max block request %u): ", __FUNCTION__, *max_nb_elem));
      for (i = 0; i < *max_nb_elem; i++) {
	  array[i] = chain[i];
	  if (array[i].nb == 0)
	      break;
	  FDBG (("%u: %u sectors at 0x%llX, ", i, array[i].nb, array[i].lba));
	  }
#ifdef FS_USE_MALLOC
      FREE (sector_chain_buffer);
#endif
      if (array[i].nb != 0) {
	  FDBG (("too many fragments, sorry.\r\n"));
	  return 0x900;
	  }
	else {
	  FDBG (("normal end, %u fragments\r\n", i));
	  *max_nb_elem = i;
	  return 0;
	  }
      }
  if ((unsigned)treat <= 4) {
      /* (unsigned)treat == 1 : return true if the file contains 0xAA55 signature at offset 510 */
      /* (unsigned)treat == 3 : treat the commandline */
      /* We need to save/restore fourKbuffer which is already in use */
#if SETUP & BIG_MALLOC
#error Sorry unimplemented,
#else
      farptr overstackbuffer = (0x1000U + getss()) << 16;
      smemcpy (overstackbuffer, fourKbuffer, sizeof(fourKbuffer));
#endif
      }

#ifdef COMMANDFILE
  unsigned char in_comment = 0;
  const char *start_non_whitespace = (const char *)-1;
#endif

  unsigned returned = 0, sizeread = 0;
  current = chain;
  FDBG (("BOOTWAY.iso.intergap_shift %u, BOOTWAY.iso.gap_sectors_shift %u\r\n", BOOTWAY.iso.intergap_shift, BOOTWAY.iso.gap_sectors_shift));

  unsigned long long base_lba = current->lba;
  for (;;) {
      unsigned short nbread = 0;
      int tmp;

      while (nbread < DEFAULT_BUFFER_SIZE) {
	  unsigned short nb = MIN ((DEFAULT_BUFFER_SIZE-nbread) / fs->common.SectSize, current->nb);
	  if (current->lba & (1ULL << 63)) {
	      if ((current->lba - base_lba) >> BOOTWAY.iso.intergap_shift) {
		  current->lba += 1 << BOOTWAY.iso.gap_sectors_shift;
		  base_lba = current->lba;
		  }
	      }
	  if (nb == 0) {
	      if (nbread == 0) {
		  *filesize = sizeread;
		  FDBG ((" [buffer smaller than hardware sector?] ", __FUNCTION__));
		  dpp->misc.fsanalyse_error = 1;
		  returned = 0x700;
		  break;
		  }
	      if (current->nb != 0)
		  FDBG ((" [cannot read another complete sector in the buffer] ", __FUNCTION__));
	      break;
	      }
	  if ((current->lba & ~(1ULL << 63)) + nb >= dpp->length) {
	      FDBG (("%s: lba over partition limit!\r\n", __FUNCTION__));
	      /* will be detected by FS_readsector()
	      returned = 0x300;
	      break;
	      */
	      }

	  if (skipoffset >= (unsigned)(nb * fs->common.SectSize)) {
	      skipoffset -= nb * fs->common.SectSize;
	      sizeread += nb * fs->common.SectSize;
	      }
	    else {
	      if ((current->lba & ~(1ULL << 63)) != 0) {
		  unsigned char err = FS_readsector (fs, current->lba & ~(1ULL << 63), nb, data_adr(fourKbuffer + nbread));
		  if (_BIOS_control_break_pressed()) /* error will be treated later */
		      break;
		  if (err != 0) {
		      *filesize = sizeread;
		      FDBG (("%s: error 0x%X reading file at LBA=%llu!\r\n", __FUNCTION__, err, current->lba & ~(1ULL << 63)));
		      returned = 0x400 | err;
		      break;
		      }
		  }
		else /* file holes are full of zeros: */
		  _memset (fourKbuffer + nbread, 0, nb * fs->common.SectSize);
	      nbread += nb * fs->common.SectSize;
	      }

	  current->lba += nb;
	  current->nb -= nb;
	  if (current->nb == 0) {
	      current++;
	      base_lba = current->lba;
	      }
	  }
      if (returned)
	  break;
      if (_BIOS_control_break_pressed()) { /* error will be treated later */
	  printf ("  %s\r\n", ERROR_CONTROL_BREAK);
	  break;
	  }

      sizeread += nbread;
      if (*filesize != 0 && sizeread > *filesize + fs->common.byte_per_block) {
	  FDBG (("[%s: we shall read %u bytes and have read %u, byte_per_block %u, problem]",
		 __FUNCTION__, *filesize, sizeread, fs->common.byte_per_block));
	  *filesize = sizeread;
	  returned = 0x800;
	  break;
	  }

      if ((unsigned)treat == 1) { /* return true if the file contains 0xAA55 signature at offset 510 */
	  unsigned short signature = *(unsigned short *)(fourKbuffer + skipoffset + 510);
	  returned = (signature == 0xAA55);
	  break;
	  }
#ifdef COMMANDFILE
	else if ((unsigned)treat == 3) {
	  const char *end = (const char *)fourKbuffer + skipoffset + nbread;
	  if (sizeread > *filesize)
	      end -= sizeread - *filesize;
	  treat_cmdline((const char *)fourKbuffer + skipoffset, end,
		(fs->common.fsname[0] == '\0') ? dpp->name : (unsigned char *)fs->common.fsname,
		&in_comment, &start_non_whitespace);
	  if (sizeread >= *filesize) {
	      FDBG (("[%s: treat_cmdline() done, read %u bytes]\r\n", __FUNCTION__, *filesize));
	      returned = 0;
	      break;
	      }
	  tmp = 0;
	  }
#endif
	else
	  tmp = (*treat) (opaque, fourKbuffer + skipoffset, nbread - skipoffset);
      skipoffset = 0;

      if (tmp != 0) {
	  *filesize = sizeread;
	  FDBG (("[%s: treat function returns %d, so stop]\r\n", __FUNCTION__, tmp));
	  returned = 0x500 | tmp;
	  break;
	  }
      if (current->nb == 0) {
	  *filesize = sizeread;
	  FDBG (("[%s: done, read %u bytes]\r\n", __FUNCTION__, *filesize));
	  returned = 0;
	  break;
	  }
      }
#ifdef FS_USE_MALLOC
  FREE (sector_chain_buffer);
#endif
  if ((unsigned)treat <= 4) {
#if SETUP & BIG_MALLOC
#error Sorry unimplemented,
#else
      lmemcpy (fourKbuffer, (0x1000U + getss()) << 16, sizeof(fourKbuffer));
#endif
      }
  return returned;
  }}

FS_FCT_PREFIX(file_check_aa55_sig) static unsigned
file_check_aa55_sig (union filesystem_union *fs, unsigned inode)
  {
  unsigned filesize = 4096;
  return file_treat (fs, inode, &filesize, 0, (void *)1, 0);
  }

#ifdef COMMANDFILE
FS_FCT_PREFIX(file_treat_commandfile) static unsigned
file_treat_commandfile (union filesystem_union *fs, unsigned inode, unsigned filesize)
  {
  return file_treat (fs, inode, &filesize, 0, (void *)3, 0);
  }
#endif

/* noinline: take care of the size of the stack... */
FS_FCT_PREFIX(partition_analyse) static void __attribute__ ((noinline))
partition_analyse (struct treat_directory_str *opaque)
  {bound_stack();{
  unsigned open_result;
  struct diskparam_str *const dp = &DI.param[opaque->filesystem.common.disk];
  struct partition_str *const dpp = &dp->partition[opaque->filesystem.common.part];

  dpp->misc.fat_bootable = 0;

  if (dp->access == hardide_atapi && dpp->type == CD_DA_sector_type) {
      FDBG (("## Do not analyse CD_DA_sector_type (audio)\r\n"));
      }
    else if (dpp->misc.active == part_extended) {
      FDBG (("## Do not analyse EXTENDED partition\r\n"));
      return;
      }
    else if (dpp->type == EMPTY) {
      FDBG (("## Do not analyse EMPTY partition\r\n"));
      return;
      }

//  dpp->name[0] = '\0';
  open_result = open_filesystem (&opaque->filesystem, copy_gujin_param.attrib.search_el_torito, opaque);
#ifdef NB_ISO
  if (BOOTWAY.iso.nb_remap && !copy_gujin_param.attrib.probe_file_in_iso_image) {
      FDBG (("open_filesystem() returns %d, but configured not to probe files in ISO\r\n", open_result));
      }
    else
#endif
  if (open_result == 0) {
      FDBG (("open_filesystem() success, Scan root directory: "));
#ifdef NB_ISO
      opaque->inISO = (BOOTWAY.iso.nb_remap != 0);
#endif
      opaque->inRoot = 1;
      opaque->which_dir = 0;
      struct slashdir *dir_scanned;
#if DEBUG & DEBUG_FS
      const char *scan_name = "root";
#endif
      for (dir_scanned = &opaque->filesystem.common.slashboot; dir_scanned <= &opaque->filesystem.common.slashlive; dir_scanned++)
	  dir_scanned->inode = 0;
      for (dir_scanned = &opaque->filesystem.common.slash; dir_scanned <= &opaque->filesystem.common.slashlive; dir_scanned++) {
	  if (opaque->inRoot || (dir_scanned->inode != 0 && dir_scanned->size != 0)) {
	      FDBG (("\r\nscanning '%s' (inode %u size %u): ", scan_name, dir_scanned->inode, dir_scanned->size));
	      file_treat (&opaque->filesystem, dir_scanned->inode, &dir_scanned->size, 0, opaque->filesystem.common.treatdir, opaque);
	      FDBG (("done, dir size %u\r\n", dir_scanned->size));
	      }

#if DEBUG & DEBUG_FS
	  scan_name = opaque->inRoot? copy_gujin_param.scanpath : SearchPathArray[dir_scanned - &opaque->filesystem.common.slashboot];
#endif
	  if (!opaque->inRoot)
	      opaque->which_dir++;
	  opaque->inRoot = 0;
	  }
      }
    else {
      FDBG (("%s: error open_filesystem 0x%X.\r\n\r\n", __FUNCTION__, open_result));
      }

  /* write the default partition name here and not in main.c */
  if (dpp->name[0] == '\0'
#ifdef NB_ISO
      && BOOTWAY.iso.nb_remap == 0
#endif
	) {
      STRINIT (dpp->name, "part 00");  /* max 99 - 4 partitions */
      if (dpp->OSnumber >= 10) {
	  dpp->name[5] += dpp->OSnumber / 10;
	  dpp->name[6] += dpp->OSnumber % 10;
	  }
	else {
	  dpp->name[5] += dpp->OSnumber;
	  dpp->name[6] = '\0';
	  }
      }

#if DISK_SUPPORT & BOOTSECT_PROBE
  /* NOTE: cannot load at 0x7C00 and exit to DOS... */
#if !(DEBUG & DEBUG_FS)
  if (BOOT1_DOS_running()) {
      FDBG (("skip bootsector / El-Torito / BDI because running on DOS.\r\n"));
      return;
      }
#endif

#if DISK_SUPPORT & ISOFS_PROBE
  if (opaque->newNBeltorito != 0) {
      static const char *const name_array[] = {"noemul", "fd1.20", "fd1.44", "fd2.88", "hd", "unknown"};
      unsigned cpt = opaque->currentNBeltorito + 1; /* skip header */

      FDBG (("Search EL-TORITO bootsector(s): newNBeltorito = %u, currentNBeltorito %u.\r\n", opaque->newNBeltorito, opaque->currentNBeltorito));
      while (   opaque->nb < NB_DESC_ARRAY - 1 && dpp->ElToritoCatalog[cpt].sector_lba != 0
	     && cpt < opaque->currentNBeltorito + opaque->newNBeltorito
	     && dpp->ElToritoCatalog[cpt].boot_indicator == 0x88	/* is bootable */) {
	  /* Re-order to have the el-torito before possible kernel files in this CD session: */
	  unsigned index = opaque->nb;
	  while (index > 0
#ifdef NB_ISO
		&& !BOOTWAY.iso.nb_remap
#endif
		&& opaque->array[index-1].disk == opaque->filesystem.common.disk
		&& opaque->array[index-1].partition == opaque->filesystem.common.part) {
	      opaque->array[index] = opaque->array[index-1];
	      index--;
	      }
	  opaque->array[index] = (struct desc_str) {
		.disk = opaque->filesystem.common.disk,
		.partition = opaque->filesystem.common.part,
		.inode = dpp->ElToritoCatalog[cpt].sector_lba,	/* sector to load */
		.boottype = is_el_torito,
		.name_offset = cpt, /* DIRTY: is the index inside boot catalog, start at 1 */
		.filesize = dpp->ElToritoCatalog[cpt].sector_count,
		.filename = ""
		};
	  char *dst = opaque->array[index].filename;
#ifdef NB_ISO
	  if (BOOTWAY.iso.nb_remap) {
	      opaque->array[index].iso_inode = BOOTWAY.iso.found[BOOTWAY.iso.current_used].inode;
	      opaque->array[index].iso_filesize = BOOTWAY.iso.found[BOOTWAY.iso.current_used].size;

	      unsigned i = 0;
	      while (BOOTWAY.iso.found[BOOTWAY.iso.current_used].filename[i] != '\0')
		  *dst++ = BOOTWAY.iso.found[BOOTWAY.iso.current_used].filename[i++];
	      *dst++ = ':';
	      }
#endif
	  strcpy (dst, name_array[(dpp->ElToritoCatalog[cpt].boot_media_type <= 4) ? dpp->ElToritoCatalog[cpt].boot_media_type : 5]);
	  opaque->nb++;
	  cpt++;
	  }
      opaque->currentNBeltorito += opaque->newNBeltorito;
      opaque->newNBeltorito = 0;
      }
#endif

  if (!partition_has_bootsector (dpp->type)) { /* ISO as BDI is in partition with bootsectors */
      FDBG (("skip bootsector, partition type has none.\r\n"));
      }
#ifdef NB_ISO
    else if (BOOTWAY.iso.nb_remap == 0 && !copy_gujin_param.attrib.search_part_mbr) {
      FDBG (("skip bootsector, search_part_mbr not set.\r\n"));
      }
    else if (BOOTWAY.iso.nb_remap != 0 && !copy_gujin_param.attrib.search_disk_mbr) {
      // Detect the MBR of ISO images only when copy_gujin_param.attrib.search_disk_mbr is set:
      FDBG (("skip bootsector in ISO image, search_disk_mbr not set.\r\n"));
      }
#else
    else if (!copy_gujin_param.attrib.search_part_mbr) {
      FDBG (("skip bootsector, search_part_mbr not set.\r\n"));
      }
#endif
      /* We have to do that after analysing the files because for FAT,
	we add the bootsector if the root dir contains io.sys/ibmbio.com: */
    else if (copy_gujin_param.attrib.keep_all_part_mbr == 0 && open_result == 0 && !dpp->misc.fat_bootable) {
      FDBG (("skip this bootsector, no 'io.sys' on an supported filesystem.\r\n"));
      }
    else if (opaque->nb < NB_DESC_ARRAY - 1) {
#if 1
      union {
	  bootsector_t bootsector;
	  unsigned char all[opaque->filesystem.common.SectSize];
	  unsigned char all_nocrash[dp->bytepersector];
	  } buffer;
      bootsector_t *bootsect = &buffer.bootsector;
      farptr buffer_farptr = stack_adr (bootsect);
#else
      bootsector_t *bootsect = (bootsector_t *)fourKbuffer;
      farptr buffer_farptr = data_adr (bootsect);
#endif
      unsigned char err = FS_readsector (&opaque->filesystem, 0, 1, buffer_farptr);
      unsigned MBRtype;

      if (err != 0) {
	  FDBG (("%s: error 0x%X reading MBR\r\n", __FUNCTION__, err));
	  }
	else if ((MBRtype = recognise_MBR (bootsect)) == 0) {
	  FDBG (("%s: MBR signature is 0x%X, first byte null?\r\n", __FUNCTION__, bootsect->after.Signature0xAA55));
	  }
	else {
	  unsigned index;
	  char *dst;
	  const unsigned char *str = bootsect->before.part1.String;
	  /* We already know that if we have a FAT partition and we are
	     here, there is an IO.SYS/IBMBIO.COM, so DOSEMU, mkdosfs
	     will not appear. Do a basic renaming. */
	  if (MBRtype == 1) { /* "unknown" */
	      while (*str > ' ' && *str <= '~')
		  str++;
	      if (str - bootsect->before.part1.String >= 4)
		  str = bootsect->before.part1.String;
		else
		  str = (const unsigned char *) "noname";
	      }
	    else
	      str = (const unsigned char *) recognise_MBR_string[MBRtype];

	  FDBG (("%s: add bootsector '%s'\r\n", __FUNCTION__, str));

	  /* Re-order to have the PBR (LILO, GRUB) before possible kernel files in this partition,
		but after possible MBR: */
	  index = opaque->nb;
	  while ( index > 0
#ifdef NB_ISO
		&& !BOOTWAY.iso.nb_remap
#endif
		&& opaque->array[index-1].disk == opaque->filesystem.common.disk
		&& opaque->array[index-1].partition == opaque->filesystem.common.part
		&& opaque->array[index-1].boottype != is_MBR) {
	      opaque->array[index] = opaque->array[index-1];
	      index--;
	      }

	  opaque->array[index] = (struct desc_str) {
	      .disk = opaque->filesystem.common.disk,
	      .partition = opaque->filesystem.common.part,
	      .inode = dpp->start,
	      .boottype = is_PBR,
	      .name_offset = 0,	/* if >= 1 it is MBR/PBR in a file */
	      .filesize = 0,
	      .filename = ""
	      };

	  dst = opaque->array[index].filename;
#ifdef NB_ISO
	  if (BOOTWAY.iso.nb_remap) {
	      opaque->array[index].boottype = is_bdi_file;
	      opaque->array[index].inode = BOOTWAY.iso.found[BOOTWAY.iso.current_used].inode;
	      opaque->array[index].filesize = BOOTWAY.iso.found[BOOTWAY.iso.current_used].size;
	      //opaque->array[index].name_offset = 0; /* if >= 1 it is el-torito */

	      unsigned i = 0;
	      while (BOOTWAY.iso.found[BOOTWAY.iso.current_used].filename[i] != '\0')
		  *dst++ = BOOTWAY.iso.found[BOOTWAY.iso.current_used].filename[i++];
	      *dst++ = ':';
	      }
#endif
	  for (; dst < &opaque->array[index].filename[lastof (opaque->array[index].filename)]; dst++) {
	      *dst = *str;
	      if (*str <= ' ' || *str > '~')
		  break;
	      str++;
	      }
	  *dst = '\0';
	  opaque->nb++;
	  }
      }
#endif /* BOOTSECT_PROBE */

  FDBG ((", end scan.\r\n\r\n"));
  }}
#endif /* E2FS_PROBE|FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|ISOFS_PROBE */

/**
 ** DOS managed filesystem:
 **/
#if DISK_SUPPORT & DOS_SUPPORT
#include "dos.h"

//#define DOS_LONG_FILENAME /* is that usefull (open a file on Win95+)? */

FS_FCT_PREFIX(DOS_analyse) static inline void
DOS_analyse (struct treat_directory_str *opaque)
  {
  unsigned short i, error;
  char pattern[NAME_LENGTH] = "A:\\*.*";
  struct DOS_DTA_str DosDta;
  /* inline */ const struct DOS_attribute_str attr_allowable = {
	.read_only    = 1,  .hidden    = 1, .system  = 1,
	.volume_label = 0,  .directory = 0, .archive = 1
	};

#ifdef DOS_LONG_FILENAME
  struct LfnFindData_str FindData;
  unsigned short handle;
  /* inline */ const struct DOS_attribute_str attr_required = {
	.read_only    = 0,  .hidden    = 0, .system  = 0,
	.volume_label = 0,  .directory = 0, .archive = 0
	};
  /* inline */ const struct DOS_attribute_str vol_attr_allowable = {
	.read_only    = 0,  .hidden    = 1, .system  = 1,
	.volume_label = 1,  .directory = 0, .archive = 1
	};
#endif

  FDBG ((" [%s: disk %u, name %c: ", __FUNCTION__,
		 opaque->filesystem.common.disk, DI.param[opaque->filesystem.common.disk].disknb));
  DOS_SetDTA (&DosDta);

  {
 /*
  * Dirty if (access == dos_part), the "partition" pointer is
  * used as the DOS name pointer because nbpartition == 0.
  */
  /* inline */ const struct DOS_attribute_str vol_label = {
	.read_only    = 0,  .hidden    = 0, .system  = 0,
	.volume_label = 1,  .directory = 0, .archive = 0
	};
  FDBG (("label: "));
  pattern[0] = DI.param[opaque->filesystem.common.disk].disknb; /* the letter 'C' */
  DosDta.name_ext[0] = 0;
  if (
#ifdef DOS_LONG_FILENAME
      DOS_LFN_FindFirst (pattern, vol_attr_allowable, vol_label, &FindData, &handle) == 0 ||
#endif
      DOS_FindFirst (pattern, vol_label, &error) == 0) {
      char *name_ext = DosDta.name_ext;

#ifdef DOS_LONG_FILENAME
      if (*name_ext == 0) /* DOS_LFN_FindFirst() worked */
	  name_ext = FindData.full_filename;
#endif

#ifdef DISK_USE_MALLOC
      DI.param[opaque->filesystem.common.disk].partition = MALLOC (MAX_DISKNAMESIZE, "DOSname");
#endif
      if (DI.param[opaque->filesystem.common.disk].partition != 0) {
	  unsigned cpt;
	  char *dest = (char *)DI.param[opaque->filesystem.common.disk].partition;
	  for (cpt = 0; cpt < MAX_DISKNAMESIZE-1; cpt++)
	      if ((dest[cpt] = name_ext[cpt]) == '\0')
		  break;
	  dest[cpt] = '\0';
	  /* remove the '.' in 'EL-BOOT-.27': */
	  if (dest[8] == '.')
	      *(unsigned *)&dest[8] = *(unsigned *)&dest[9];
	  FDBG (("'%s']\r\n", dest));
	  }
	else
	  FDBG (("nowhere to copy '%s']\r\n", name_ext));
      }
    else {
#ifdef DISK_USE_MALLOC
      DI.param[opaque->filesystem.common.disk].partition = 0;
#else
      if (DI.param[opaque->filesystem.common.disk].partition != 0)
	  DI.param[opaque->filesystem.common.disk].partition[0].start = 0;
#endif
      FDBG (("not found]\r\n"));
      }
  }

  static const char *const filename_array[] = {
#ifdef INITRDNAME
      INITRDNAME,		/* initrd filename at first index */
#endif
#ifdef INITRDNAME2
      INITRDNAME2,		/* initrd filename at two first index */
#endif
#ifdef INITRAMFSNAME
      INITRAMFSNAME,	/* initramfs filename at next index, an initrd without decompression */
#endif
#ifdef LINUZNAME1
      LINUZNAME1,
#endif
#ifdef LINUZNAME2
      LINUZNAME2,
#endif
#ifdef LINUZNAME3
      LINUZNAME3,
#endif
      };

#ifdef KERNELEXT
#ifdef COMMANDFILE
  for (i = 0; i < 2 * (nbof(filename_array) + 2); i++) {
#else
  for (i = 0; i < 2 * (nbof(filename_array) + 1); i++) {
#endif
#else
  for (i = 0; i < 2 * nbof(filename_array); i++) {
#endif
      unsigned name_offset = 0;
      char *pat = pattern;
      *pat++ = DI.param[opaque->filesystem.common.disk].disknb; /* the letter 'C' */
      *pat++ = ':';
      *pat++ = '\\';
      if (i & 1) {
	  unsigned cpt = 0;
	  if (!copy_gujin_param.attrib.search_subdir_files)
	      continue;
	  while ((*pat = copy_gujin_param.scanpath[cpt++]) != 0)
	      pat++;
	  *pat++ = '\\';
	  opaque->inRoot = 0;
	  }
	else {
	  if (!copy_gujin_param.attrib.search_topdir_files)
	      continue;
	  opaque->inRoot = 1;
	  }

#ifdef KERNELEXT
#ifdef COMMANDFILE
      if ((i >> 1) == nbof(filename_array) + 1) {
	  STRINIT (pat, COMMANDFILE);
	  }
	else
#endif
      if ((i >> 1) == nbof(filename_array)) {
	  STRINIT (pat, "*");
	  STRINIT (&pat[1], KERNELEXT);
	  }
	else
#endif
	  {
	  const char *src = filename_array[i >> 1];
	  while (*src) {
	      *pat++ = *src++;
	      name_offset++;
	      }
#ifndef DOS_LONG_FILENAME
	  if (src - filename_array[i >> 1] == 7)	/* "vmlinuz" -> "vmlinu~1", "bzImage"  -> "bzImag~1" */
	      pat--, name_offset--;
#endif
	  *(unsigned *)pat = *(unsigned *)"*.*";
	  }

      FDBG ((" [searching \"%s\": ", pattern));
      DosDta.name_ext[0] = 0;
      if (
#ifdef DOS_LONG_FILENAME
	  DOS_LFN_FindFirst (pattern, attr_allowable, attr_required, &FindData, &handle) == 0 ||
#endif
	  DOS_FindFirst (pattern, attr_allowable, &error) != 0)
	  FDBG (("failed DOS_FindFirst, error 0x%X ", error));
	else {
	  for (;;) {
	      unsigned filesize, last_modification_date;
	      char *filename;
	      struct desc_str *elem = &opaque->array[opaque->nb];
	      *elem = (struct desc_str) {0};

#ifdef DOS_LONG_FILENAME
	      if (DosDta.name_ext[0] == 0) {
		  filesize = FindData.lsb_file_size;
		  filename = FindData.full_filename;
		  }
		else
#endif
		  {
		  unsigned nbdays = 0, cptmonth;
		  for (cptmonth = 1; cptmonth < DosDta.date.month; cptmonth++)
		      nbdays += dayinmonth[cptmonth];
		  nbdays += DosDta.date.day - 1;
		  nbdays += (DosDta.date.year + 1980 - 2001) / 4; /* in 2005 add 1 leap day */
		  if (DosDta.date.month >= 3 && (DosDta.date.year + 1980) % 4 == 0)
		      nbdays ++; /* add 1 day after Feb 29th on leap year */
		  if (DosDta.date.year + 1980 < 2001)
		      last_modification_date = 0;
		    else
		      last_modification_date = (DosDta.date.year + 1980 - 2001) * 365 * 24 * 60 * 60
					+ nbdays * 24 * 60 * 60 + DosDta.date.hour * 60 * 60 + DosDta.date.minute * 60 + DosDta.date.seconddiv2 * 2;
		  FDBG ((" [%s: adding file '%s' size %u bytes date %u sec after Jan 1st, 2001] ", __FUNCTION__, DosDta.name_ext, DosDta.size, last_modification_date));
		  filesize = DosDta.size;
		  filename = DosDta.name_ext;
		  }

	      if (opaque->nb >= NB_DESC_ARRAY - 1) {
		  FDBG (("search finnished early, %s not included]\r\n", filename));
#ifdef DOS_LONG_FILENAME
		  if (DosDta.name_ext[0] == 0)
		      DOS_LFN_FindClose (handle, &error);
#endif
		  return;
		  }
	      elem->filesize = filesize;
	      elem->inRoot = !(i & 1);
	      elem->ReadOnly = DosDta.attribute.readonly;	/* mostly for BDI files */
	      elem->last_modification_date = last_modification_date;
#if defined (INITRDNAME) && defined (INITRDNAME2)
	      if ((unsigned)(i >> 1) == 0)	/* initrd for i = 0 and 1 */
		  elem->boottype = is_initrd;
		else
#define INITRAMFSNAMEINDEX 2
#elif defined (INITRDNAME) || defined (INITRDNAME2)
	      if ((unsigned)(i >> 1) <= 1)	/* initrd for i = 0 to 3 */
		  elem->boottype = is_initrd;
		else
#define INITRAMFSNAMEINDEX 1
#else
#define INITRAMFSNAMEINDEX 0
#endif
#ifdef INITRAMFSNAME
		     if ((unsigned)(i >> 1) == INITRAMFSNAMEINDEX)
		  elem->boottype = is_initramfs;
		else
#endif
		  elem->boottype = is_linux;
#ifdef KERNELEXT
	      if ((i >> 1) == nbof(filename_array)) {
		  elem->boottype = is_elf;
		  /* filename is a kind of "linux-2.kgz" */
		  name_offset = 0;
		  while (filename[name_offset] != '\0' && filename[name_offset] != '-'
			&& filename[name_offset] != '+')
		      name_offset++;
		  if (filename[name_offset] != '-')
		      name_offset = 0;
		  }
#endif
#ifdef COMMANDFILE
	      if ((i >> 1) == nbof(filename_array) + 1) {
		  unsigned short gujin_cmd_handle, nbread_errorcode;
		  unsigned char cread;
		  if (DOS_OpenFile (filename, DOS_open_read_only, &gujin_cmd_handle) == 0) {
		      unsigned char in_comment = 0;
		      unsigned start_non_whitespace = (unsigned)-1, nb_total_read = 0, initial_cmdfile_len = UTIL.cmdfile_len;
//		      while (nb_total_read < filesize && DOS_ReadFile (gujin_cmd_handle, &cread, 1, &nbread_errorcode) == 0 && nbread_errorcode == 1) {
		      while (DOS_ReadFile (gujin_cmd_handle, &cread, 1, &nbread_errorcode) == 0 && nbread_errorcode == 1) {
			  nb_total_read++;
			  if (in_comment && cread != '\n')
			      continue;
			  if (cread == '#') {
			      in_comment = 1;
			      if (start_non_whitespace != (unsigned)-1)
				  goto store_line_dos;
			      continue;
			      }
			  in_comment = 0;
			  if (start_non_whitespace == (unsigned)-1) {
			      if (cread != ' ' && cread != '\t' && cread != '\r' && cread != '\n')
				  start_non_whitespace = nb_total_read - 1;
			      continue;
			      }
			  if (cread == '\n' || nb_total_read == filesize) {
store_line_dos:	  ;
			      const char *fsname = (const char *)DI.param[opaque->filesystem.common.disk].partition;
			      unsigned strlen_fsname = 0;
			      if (fsname)
				  strlen_fsname = strlen (fsname);
			      unsigned len = nb_total_read - start_non_whitespace - 1;
#ifdef FS_USE_MALLOC
			      allocmem *tmpptr = REALLOC (UTIL.cmdfile_base, UTIL.cmdfile_len + len + 1 + strlen_fsname, "cmdline2");
#else
			      allocmem *tmpptr;
			      if (UTIL.cmdfile_len + len + 1 + strlen_fsname > sizeof(cmdfile_base_buffer))
				  tmpptr = 0;
				else
				  tmpptr = cmdfile_base_buffer;
#endif
			      if (tmpptr) {
				  UTIL.cmdfile_base = tmpptr;
				  unsigned newpos, generalpos;
				  /* FIXME: treat DOS errors there: */
				  DOS_SeekFile (gujin_cmd_handle, DOS_SEEK_CUR, 0, &generalpos, &nbread_errorcode);
				  DOS_SeekFile (gujin_cmd_handle, DOS_SEEK_SET, start_non_whitespace, &newpos, &nbread_errorcode);

				  char cread1, cread2;
				  if (strlen_fsname
					&& DOS_ReadFile (gujin_cmd_handle, &cread1, 1, &nbread_errorcode) == 0
					&& nbread_errorcode == 1
					&& DOS_ReadFile (gujin_cmd_handle, &cread2, 1, &nbread_errorcode) == 0
					&& nbread_errorcode == 1
					&& cread1 == cread2) {
					  len -= 2;
					  UTIL.cmdfile_base[UTIL.cmdfile_len++] = cread1;
					  unsigned short xcpt;
					  for (xcpt = 0; xcpt < strlen_fsname; xcpt++)
					      UTIL.cmdfile_base[UTIL.cmdfile_len++] = fsname[xcpt];
					  UTIL.cmdfile_base[UTIL.cmdfile_len++] = cread2;
					  }
				    else
					DOS_SeekFile (gujin_cmd_handle, DOS_SEEK_SET, start_non_whitespace, &newpos, &nbread_errorcode);

				  while (len--) {
				      DOS_ReadFile (gujin_cmd_handle, &cread, 1, &nbread_errorcode);
				      /* strip '\r' of "\r\n" and strip "\\ *\n[ \t]*" and "\\ *\r\n[ \t]*" */
				      if (cread == '\\') {
					  DOS_SeekFile (gujin_cmd_handle, DOS_SEEK_CUR, 0, &newpos, &nbread_errorcode);
					  while (DOS_ReadFile (gujin_cmd_handle, &cread, 1, &nbread_errorcode) == 0
						&& (cread == ' ' || cread == '\t' || cread == '\r'))
					      continue;
					  if (cread == '\n') {
					      start_non_whitespace = (unsigned)-1;
					      break;
					      }
					    else {
					      UTIL.cmdfile_base[UTIL.cmdfile_len++] = '\\';
					      DOS_SeekFile (gujin_cmd_handle, DOS_SEEK_SET, newpos, &newpos, &nbread_errorcode);
					      }
					  }
					else if (cread != '\r')
					  UTIL.cmdfile_base[UTIL.cmdfile_len++] = cread;
				      }
				  DOS_SeekFile (gujin_cmd_handle, DOS_SEEK_SET, generalpos, &generalpos, &nbread_errorcode);

				  if (start_non_whitespace != (unsigned)-1 || nb_total_read == filesize) {
				      UTIL.cmdfile_base[UTIL.cmdfile_len++] = '\n';
				      /* Strip identical lines: */
				      const char *const end = &UTIL.cmdfile_base[UTIL.cmdfile_len-1]; /* of the added line */
				      const char *scan = end - 1; /* before '\n' */
				      while (scan > UTIL.cmdfile_base && *scan != '\n')
					  scan--;
				      /* const char *const start = scan + 1; / * of the added line, if we have (*scan == '\n') */
				      while (scan > UTIL.cmdfile_base) { /* one more line before, we know (*scan == '\n') */
					  const char *added = end;
					  while (scan > UTIL.cmdfile_base && *--added == *--scan)
					      if (*scan == '\n')
						  break;
					  if (scan == UTIL.cmdfile_base || *scan == '\n') { /* found an identical line */
					      UTIL.cmdfile_len = initial_cmdfile_len;
					      break;
					      }
					    else while (*--scan != '\n' && scan > UTIL.cmdfile_base)
					      continue;
					    }
				      initial_cmdfile_len = UTIL.cmdfile_len;
				      }
				  UTIL.cmdfile_base[UTIL.cmdfile_len] = '\0';
				  }
			      start_non_whitespace = (unsigned)-1;
			      }
			  }
		      DOS_CloseFile (gujin_cmd_handle, &nbread_errorcode);
		      }
		  }
		else
#endif

#ifdef MIN_KERNEL_INITRD_SIZE
	      if (filesize < MIN_KERNEL_INITRD_SIZE && IS_KERNEL_OR_INITRD (elem)) {
		  FDBG ((" [%s: filesize too small: %u] ", __FUNCTION__, filesize));
		  }
		else
#endif
		  {
		  char *ptr = elem->filename;

		  elem->disk = opaque->filesystem.common.disk;
		  elem->partition = opaque->filesystem.common.part;
		  elem->inRoot = opaque->inRoot;
		  elem->inISO = opaque->inISO;
		  elem->searchpath = opaque->which_dir;
		  *ptr++ = '/';
		  name_offset++;
		  if (!opaque->inRoot) {
		      unsigned cpt = 0;
		      while ((*ptr = copy_gujin_param.scanpath[cpt++]) != 0)
			  ptr++;
		      *ptr++ = '/';
		      name_offset += cpt + 1;
		      }
		  while (ptr < &elem->filename[sizeof(elem->filename)-1])
		      if (*filename >= 'A' && *filename <= 'Z')
			  *ptr++ = *filename++ - 'A' + 'a';
		      else if (*filename != '\0')
			  *ptr++ = *filename++;
		      else
			  break;
		  *ptr ='\0';
		  elem->name_offset = name_offset;
		  FDBG (("'%s', ", elem->filename));
		  opaque->nb++;
		  }
	      if (
#ifdef DOS_LONG_FILENAME
		  (DosDta.name_ext[0] == 0)?
			DOS_LFN_FindNext (handle, &FindData, &error) :
#endif
			DOS_FindNext (&error)) {
		  FDBG (("FindNext: no more "));
		  break;
		  }
	      }
	  }
      FDBG (("] "));
      }
#ifdef DOS_LONG_FILENAME
  if (DosDta.name_ext[0] == 0) /* DOS_LFN_FindFirst() in use */
      DOS_LFN_FindClose (handle, &error);
#endif
  FDBG (("[%s end search on this disk]\r\n\r\n", __FUNCTION__));
  }

FS_FCT_PREFIX(DOS_system_file_load) static inline unsigned
DOS_system_file_load (unsigned disk, char *filename, unsigned *filesize,
			 unsigned skipoffset, z_stream *gzlibptr)
  {
  unsigned short handle, nbread_errorcode;
  int returned;

#if DISK_SUPPORT == DOS_SUPPORT /* full path given for tiny_exe: */
  char *fullfilename = filename;
#else
  /* declare that in %ss, i.e. in %ds < 64 K, remember SETUP & BIG_MALLOC ! */
  char fullfilename[NAME_LENGTH];

  fullfilename[0] = DI.param[disk].disknb; /* the letter, i.e. 'C' */
  fullfilename[1] = ':';
  strcpy (&fullfilename[2], filename);
#endif

  if (DOS_OpenFile (fullfilename, DOS_open_read_only, &handle) != 0) {
      FDBG ((" [DOS_OpenFile %s failed, error 0x%X] ", fullfilename, handle));
      return 0x100;
      }

  if (DOS_SeekFile (handle, DOS_SEEK_SET, skipoffset, filesize, &nbread_errorcode) == 0)
      FDBG ((" [seeked to skipoffset=%u successfully, now offset %u bytes] ", skipoffset, *filesize));
    else {
      FDBG ((" [PANIC cannot seek to %u, errorcode 0x%X] ", skipoffset, nbread_errorcode));
      return 0x500;
      }

  /* we cannot have fourKbuffer malloced if using BIG_MALLOC here: */
  while (DOS_ReadFile (handle, fourKbuffer, DEFAULT_BUFFER_SIZE, &nbread_errorcode) == 0) {
      FDBG ((" [read %u bytes] ", nbread_errorcode));
      *filesize += nbread_errorcode;
      /* do something */
      returned = gzlib_extract (gzlibptr, fourKbuffer, nbread_errorcode);

      if (returned) {
	  FDBG ((" [DOS_ReadFile treatfct error 0x%X] ", returned));
	  return 0x200 | returned;
	  }

      if (nbread_errorcode != DEFAULT_BUFFER_SIZE) {
	  FDBG ((" [DOS_ReadFile loop normal exit, read %u bytes "
		 "when %u requested, total read: %u] ",
		nbread_errorcode, DEFAULT_BUFFER_SIZE, *filesize));
	  nbread_errorcode = DEFAULT_BUFFER_SIZE; /* used as a marker */
	  break;
	  }
      }

  if (nbread_errorcode != DEFAULT_BUFFER_SIZE) {
      FDBG ((" [DOS_ReadFile failed, error 0x%X] ", nbread_errorcode));
      if (DOS_CloseFile (handle, &nbread_errorcode) != 0)
	  FDBG ((" [DOS_CloseFile failed after error, error 0x%X] ", nbread_errorcode));
      return 0x300;
      }

  if (DOS_CloseFile (handle, &nbread_errorcode) != 0) {
      FDBG ((" [DOS_CloseFile failed, error 0x%X] ", nbread_errorcode));
      return 0x400;
      }
  return 0;
  }
#endif /* DOS_SUPPORT */

#if 0
FS_FCT_PREFIX(print_desc_str) void
print_desc_str (void)
  {
  struct desc_str *desc;
  for (desc = BOOTWAY.desc; desc < &BOOTWAY.desc[BOOTWAY.nb]; desc++) {
      printf ("%u: inode %u filesize %u last_modification_date 0x%X variant_number %u variant_timeout %u\r\n",
	desc - BOOTWAY.desc, desc->inode, desc->filesize, desc->last_modification_date, desc->variant_number, desc->variant_timeout);
      static const char *const array_searchpath[] = {"/boot", "/install.386", "/install.amd", "/isolinux", "/casper", "/live"};
      printf ("  %s, iso_filesize %u, iso_inode %u, disk %u, partition %u, name_offset %u\r\n",
	array_searchpath[desc->searchpath], desc->iso_filesize, desc->iso_inode, desc->disk, desc->partition, desc->name_offset);
      static const char *const array_boottype[] = {"MBR", "PBR", "linux", "elf", "initrd", "bdi_file", "el_torito", "initramfs", "multiboot" };
      printf ("  inRoot %u, inISO %u, ReadOnly %u, boottype %s, iso_fsname %s, filename %s\r\n",
	desc->inRoot, desc->inISO, desc->ReadOnly, array_boottype[desc->boottype], desc->iso_fsname, desc->filename);
      if (desc - BOOTWAY.desc > 8)
	  _BIOS_wait(500000);
      }
  _BIOS_getkey();
  }
#endif

/*
 * External functions:
 */
FS_FCT_PREFIX(disk_analyse) void
disk_analyse (void)
  {bound_stack();{
  struct treat_directory_str opaque = {};

#ifdef COMMANDFILE
#ifdef FS_USE_MALLOC
  if (UTIL.cmdfile_base && UTIL.cmdfile_len)
      FREE (UTIL.cmdfile_base);
#endif
  UTIL.cmdfile_base = 0;
  UTIL.cmdfile_len = 0;
#endif
#ifdef FS_USE_MALLOC
  if (BOOTWAY.desc != 0) {
      unsigned cptdesc;
      for (cptdesc = 0; cptdesc < BOOTWAY.nb + BOOTWAY.nb_initrd; cptdesc++)
	  if (BOOTWAY.desc[cptdesc].iso_fsname && BOOTWAY.desc[cptdesc].variant_number == 0)
	      FREE (BOOTWAY.desc[cptdesc].iso_fsname);
      FREE (BOOTWAY.desc); /* fragmented malloc memory: free() and then malloc() clears holes */
      }
  /* TODO: allocate it one at a time each time when opaque->nb++ or opaque.nb++
	but then it may not work better, by creating a lot of holes in malloc memory */
  BOOTWAY.desc = MALLOC ((NB_DESC_ARRAY+1) * sizeof (struct desc_str), "BOOTdesc"); /* +1: see sorting at end */
  if (BOOTWAY.desc == 0) {
      FDBG (("%s: malloc failed!\r\n", __FUNCTION__));
      return;
      }
#else
  static struct desc_str desc_array[NB_DESC_ARRAY+1]; /* +1: see sorting at end */
  BOOTWAY.desc = desc_array;
#endif

  opaque.array = BOOTWAY.desc;
  BOOTWAY.nb = 0;
  BOOTWAY.nb_initrd = 0;
  BOOTWAY.nb_bdi = 0;
  BOOTWAY.max_name_length = NAME_LENGTH;
  BOOTWAY.max_desc_array = NB_DESC_ARRAY;
  BOOTWAY.sizeof_desc_str = sizeof (struct desc_str);
  DBG (("sizeof struct desc_str: %u, sizeof struct BOOTWAY_str: %u\r\n",
	sizeof (struct desc_str), sizeof (struct BOOTWAY_str)));
  BOOTWAY.nb_iso = 0;
#ifdef NB_ISO
  BOOTWAY.iso.nb_remap = BOOTWAY.iso.nb_found = 0;
  BOOTWAY.iso.current_used = -1;
  BOOTWAY.iso.max_iso = NB_ISO;
  BOOTWAY.iso.max_fragment = sizeof (BOOTWAY.iso.remap) / sizeof (BOOTWAY.iso.remap[0]);
  BOOTWAY.iso.max_filename = sizeof (BOOTWAY.iso.found[0].filename) / sizeof (BOOTWAY.iso.found[0].filename[0]);
  BOOTWAY.iso.sizeofremap = sizeof (BOOTWAY.iso.remap[0]);
#endif

  PRINT (ANALYSE_FILESYSTEM);

  for (opaque.filesystem.common.disk = 0;
       opaque.filesystem.common.disk < DI.nbdisk && opaque.nb < NB_DESC_ARRAY;
       opaque.filesystem.common.disk++)
      {
      PRINTF ("%s...\r", DI.param[opaque.filesystem.common.disk].diskname);

      FDBG (("\r\n#### %s disk %u i.e. %s: (nb found = %u):\r\n",
		__FUNCTION__, opaque.filesystem.common.disk,
		DI.param[opaque.filesystem.common.disk].diskname, opaque.nb));

      if (DI.param[opaque.filesystem.common.disk].bytepersector == 0)
	  FDBG (("## Do not analyse disk without media: dp->bytepersector == 0\r\n"));
	else
#if DISK_SUPPORT & DOS_SUPPORT
	  if (DI.param[opaque.filesystem.common.disk].access == dos_part) {
	  opaque.filesystem.common.part = 0;
	  DOS_analyse (&opaque);
	  }
	else
#endif /* DOS_SUPPORT */
	     if (DI.param[opaque.filesystem.common.disk].partition == 0) /* GCC-3.4.3 bug? */
	  FDBG (("## Do not analyse disk without partition (malloc failure?): dp->partition == 0\r\n"));
	else
	  {
#if DISK_SUPPORT & (E2FS_PROBE|FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|ISOFS_PROBE)
#if DISK_SUPPORT & BOOTSECT_PROBE
	  if (   !BOOT1_DOS_running()	/* : cannot load at 0x7C00 and exit to DOS... */
		/* floppy no-part trick: put it (the PBR, not MBR) only once, after finding IO.SYS: */
	      && !(  DI.param[opaque.filesystem.common.disk].nbpartition > 4 /* maybe floppy + BEER */
		   && DI.param[opaque.filesystem.common.disk].partition[4].start == 0)
	      && !(  DI.param[opaque.filesystem.common.disk].nbpartition == 1
		   && DI.param[opaque.filesystem.common.disk].partition[0].start == 0)
	      && DI.param[opaque.filesystem.common.disk].bootable /* i.e. MBR readable, 0xAA55 signature and not null first byte */
	      && copy_gujin_param.attrib.search_disk_mbr
	      && opaque.nb < NB_DESC_ARRAY - 1) {
#if 0 /* Always keep the MBRs, because autoboot will not work if the MBR disapear... */
	      /* First, if it is a disk MBR, it has to have at least one partition
		 with the bootable flag in its primaries: */
	      if (   bootsect->after.bootsect_partition[0].indicator != bootable
		  && bootsect->after.bootsect_partition[1].indicator != bootable
		  && bootsect->after.bootsect_partition[2].indicator != bootable
		  && bootsect->after.bootsect_partition[3].indicator != bootable) {
		  FDBG (("%s: MBR without bootable primaries\r\n", __FUNCTION__));
		  }
		else
#endif

	      opaque.array[opaque.nb] = (struct desc_str) {
		  .disk = opaque.filesystem.common.disk,
		  .partition = 0,
		  .inode = 0,		/* sector to load */
		  .boottype = is_MBR,
		  .name_offset = 0,
		  .filesize = 0,
//		  .filename = "unknown"
//		  .filename = recognise_MBR_string[DI.param[opaque.filesystem.common.disk].bootable];
		  };
	      char *dst = opaque.array[opaque.nb].filename;
	      const char *src = recognise_MBR_string[DI.param[opaque.filesystem.common.disk].bootable];
	      for (;;) {
		  *dst = *src;
		  if (*dst == '\0')
		      break;
		  if (dst == &opaque.array[opaque.nb].filename[sizeof (opaque.array[opaque.nb].filename) - 1]) {
		      *dst = '\0';
		      break;
		      }
		  dst++; src++;
		  }
	      opaque.nb++;
	      }
#endif /* BOOTSECT_PROBE */

	 SET_DRIVE_ACTIVE (&DI.param[opaque.filesystem.common.disk]);

	  for (opaque.filesystem.common.part = 0;
	       opaque.filesystem.common.part < DI.param[opaque.filesystem.common.disk].nbpartition;
	       opaque.filesystem.common.part++) {
	      PRINTF ("%s:%u...\r", DI.param[opaque.filesystem.common.disk].diskname, opaque.filesystem.common.part);

#ifdef NB_ISO
	      opaque.currentNBeltorito = opaque.newNBeltorito = 0;
	      BOOTWAY.iso.nb_found = 0;
#endif
	      partition_analyse (&opaque);
#ifdef NB_ISO
	      BOOTWAY.nb_iso += BOOTWAY.iso.nb_found;
	      union filesystem_union fs_of_iso = opaque.filesystem;
	      for (BOOTWAY.iso.current_used = 0; BOOTWAY.iso.current_used < BOOTWAY.iso.nb_found; BOOTWAY.iso.current_used++) {
		  PRINTF ("%s:%s...\r", DI.param[opaque.filesystem.common.disk].diskname, BOOTWAY.iso.found[BOOTWAY.iso.current_used].filename);

		  FDBG (("\r\nScan ISO file %s size %u inode %u: ", BOOTWAY.iso.found[BOOTWAY.iso.current_used].filename, BOOTWAY.iso.found[BOOTWAY.iso.current_used].size, BOOTWAY.iso.found[BOOTWAY.iso.current_used].inode));
		  if (UTIL.malloc_failure_size_requested)
		      FDBG (("[clearing malloc_failure_size_requested=%u] ", UTIL.malloc_failure_size_requested));
		  UTIL.malloc_failure_size_requested = 0;
		  unsigned filesize = BOOTWAY.iso.found[BOOTWAY.iso.current_used].size, nb_segment = sizeof(BOOTWAY.iso.remap)/sizeof(BOOTWAY.iso.remap[0]);
		  unsigned returned = file_treat (&fs_of_iso, BOOTWAY.iso.found[BOOTWAY.iso.current_used].inode, &filesize, (unsigned)&nb_segment, 0, &BOOTWAY.iso.remap);
		  if (returned) {
		      if (returned == 0x900 || UTIL.malloc_failure_size_requested)
			  printf (FILE_S_TOO_MANY_FRAGMENTS, BOOTWAY.iso.found[BOOTWAY.iso.current_used].filename);
		      if (UTIL.malloc_failure_size_requested)
			  FDBG (("[clearing malloc_failure_size_requested=%u] ", UTIL.malloc_failure_size_requested));
		      UTIL.malloc_failure_size_requested = 0;
		      FDBG (("%s: error file_treat 0x%X to get ISO mapping.\r\n", __FUNCTION__, returned));
		      continue;
		      }
		  FDBG (("Got chain of %u elements (real filesize %u, filesize %u, holes?):\r\n", nb_segment, BOOTWAY.iso.found[BOOTWAY.iso.current_used].size, filesize));
		  BOOTWAY.iso.nb_remap = nb_segment;
		  partition_analyse (&opaque);
		  BOOTWAY.iso.nb_remap = 0;
		  }
#endif
	      }

	  SET_DRIVE_IDLE (&DI.param[opaque.filesystem.common.disk]);

#endif /* E2FS_PROBE|FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|ISOFS_PROBE */
	  }
      }
  if (opaque.nb == 0) {
      FDBG (("%s end adding, none found.\r\n", __FUNCTION__));
      return;
      }

  {
  unsigned dsk1, dsk2, part1, part2;
  for (dsk1 = 0; dsk1 < DI.nbdisk; dsk1++)
      for (part1 = 0; part1 < DI.param[dsk1].nbpartition; part1++)
	  for (dsk2 = dsk1; dsk2 < DI.nbdisk; dsk2++)
	      for (part2 = 0; part2 < DI.param[dsk2].nbpartition; part2++) {
		  if (dsk1 == dsk2 && part1 == part2)
		      continue;
		  if ((DI.param[dsk2].access == bios_chs || DI.param[dsk2].access == ebios_lba)
			&& DI.param[dsk1].access >= hardide_chs && DI.param[dsk1].access <= hardide_lba48)
		      continue;
		  if (DI.param[dsk1].partition[part1].name[0] == '\0')
		      continue;
		  if (DI.param[dsk1].partition[part1].misc.beer_partition || DI.param[dsk2].partition[part2].misc.beer_partition)
		      continue;
		  if (   part1 <= part2 /* report only once, but if same partition number of 2 disks */
		      && !strcmp((char *)DI.param[dsk1].partition[part1].name, (char *)DI.param[dsk2].partition[part2].name)) {
		      if (*(unsigned *)DI.param[dsk1].partition[part1].name != *(unsigned *)"part" &&
			  *(unsigned *)DI.param[dsk1].partition[part1].name != *(unsigned *)"CDRO") /* no report for default name */
			  printf (WARNING_TWO_PARTITION_WITH_SAME_LABEL,
				part1, DI.param[dsk1].diskname, part2, DI.param[dsk2].diskname, DI.param[dsk1].partition[part1].name);
		      DI.param[dsk1].partition[part1].misc.conflicting_fsname = 1;
		      DI.param[dsk2].partition[part2].misc.conflicting_fsname = 1;
		      }
		  }
  }
  FDBG (("%s end adding (nb found = %u), ", __FUNCTION__, opaque.nb));

 /*
  * We need to keep the order of MBR & kernels and BDI, the order of initrd
  * is not important but they have to be at end to not display them as clic-able.
  */
  BOOTWAY.nb = 0;
  BOOTWAY.nb_initrd = 0;
  BOOTWAY.nb_bdi = 0;
  unsigned idx;
  for (idx = 0; idx < opaque.nb; idx++) {
      /* Are you coming from ZZ9 Plural Z Alpha? */
      if (BOOTWAY.desc[idx].boottype == is_initrd || BOOTWAY.desc[idx].boottype == is_initramfs) {
	  BOOTWAY.nb_initrd++;
	  }
	else {
	  if (BOOTWAY.desc[idx].boottype == is_bdi_file)
	      BOOTWAY.nb_bdi++;
	  BOOTWAY.nb++; /* images/MBR AND bdi */
	  }
      }
  idx = 0;
  while (idx < BOOTWAY.nb) {
      if (BOOTWAY.desc[idx].boottype == is_initrd || BOOTWAY.desc[idx].boottype == is_initramfs) {
	  unsigned j;

	  BOOTWAY.desc[opaque.nb] = BOOTWAY.desc[idx];  /* +1: see sorting at end */
	  for (j = idx; j < opaque.nb; j++)
	      BOOTWAY.desc[j] = BOOTWAY.desc[j + 1];
	  }
	else
	  idx++;
      }

//printf ("Before sort:\r\n"); print_desc_str();

  for (idx = 0; idx < BOOTWAY.nb; idx++) {
      if (BOOTWAY.desc[idx].iso_inode != 0)
	  continue;
      if (BOOTWAY.desc[idx].boottype != is_bdi_file && BOOTWAY.desc[idx].boottype != is_linux)
	  continue;
      struct desc_str *const cur = &BOOTWAY.desc[opaque.nb],	/* +1: see sorting at end */
		*prev = &BOOTWAY.desc[idx-1];

      *cur = BOOTWAY.desc[idx];
	  /* Sort to have *.bdi in between kernels and iso: */
	  /* Sort kernel by date, before *.BDI */
      while (prev >= &BOOTWAY.desc[0] && prev->disk == cur->disk && prev->partition == cur->partition
	&& (prev->iso_inode < cur->iso_inode || (cur->boottype == is_linux && (prev->boottype == is_bdi_file
		|| (prev->boottype == is_linux && prev->last_modification_date < cur->last_modification_date))))) {
	  prev[1] = *prev;
	  prev--;
	  }
      prev[1] = *cur;
      }

//printf ("Before variant:\r\n"); print_desc_str();

#ifdef COMMANDFILE
  /* Treat variant by command line here to be sure to have read every "gujin.cmd" before counting variants */
  unsigned cpt4variant;
  for (cpt4variant = 0; cpt4variant < BOOTWAY.nb && opaque.nb < NB_DESC_ARRAY - 1; cpt4variant++) {
      if (BOOTWAY.desc[cpt4variant].boottype != is_linux)
	  continue;
      unsigned initrd_index = initrd_file (cpt4variant); /* need to be done after counting & sorting initrds */
      const char *kernel_filename = BOOTWAY.desc[cpt4variant].filename;
      const char *initrd_filename = initrd_index? BOOTWAY.desc[initrd_index].filename : 0;
      const char *fsname = BOOTWAY.desc[cpt4variant].iso_fsname ?: (char *)DI.param[BOOTWAY.desc[cpt4variant].disk].partition[BOOTWAY.desc[cpt4variant].partition].name;
      unsigned cpt = 0, nbvar = commandfile_get (fsname, kernel_filename, initrd_filename, cmdf_getnb, 0, 0, 0);
      FDBG ((" [%s found %u variant for %s:{%s,%s}]", __FUNCTION__, nbvar, fsname, kernel_filename, initrd_filename));
//printf (" [%s found %u variant for %s:{%s,%s}]", __FUNCTION__, nbvar, fsname, kernel_filename, initrd_filename); _BIOS_getkey();
      if (nbvar == 0)
	  BOOTWAY.desc[cpt4variant].variant_timeout = BOOTWAY.desc[cpt4variant].variant_number = 0;
	else for (;;) {
	  BOOTWAY.desc[cpt4variant].variant_timeout = commandfile_get (fsname, kernel_filename, initrd_filename, cmdf_gettimeout, cpt, 0, 0);
	  FDBG ((" [%u timeout %u]", cpt, BOOTWAY.desc[cpt4variant].variant_timeout));
//printf (" [%u timeout %u]", cpt, BOOTWAY.desc[cpt4variant].variant_timeout); _BIOS_getkey();
	  if (++cpt >= nbvar || opaque.nb >= NB_DESC_ARRAY - 1)
	      break;
	  unsigned last = opaque.nb++;
	  while (last-- > cpt4variant)
	      BOOTWAY.desc[last+1] = BOOTWAY.desc[last];
	  BOOTWAY.nb++;
	  cpt4variant++;
	  BOOTWAY.desc[cpt4variant].variant_number++;
	  }
      }
#endif

//printf ("After variant:\r\n"); print_desc_str();

  FDBG (("%u images/MBR (including %u bdi), %u initrd.\r\n", BOOTWAY.nb, BOOTWAY.nb_bdi, BOOTWAY.nb_initrd));
  }}

/*
 * start_address is only input.
 * size is input (maximal size) and output (real _uncompressed_ size).
 */
FS_FCT_PREFIX(system_file_load) unsigned
system_file_load (struct desc_str *elem)
  {bound_stack();{
  z_stream gzlib = { .mode = INIT }; /* everything is null */
  unsigned returned;

#if defined (LINUZNAME1) || defined (LINUZNAME2) || defined (LINUZNAME3)
  /* We use gzlib.crc32 as a marker for gzlib_extract()
     to say if it has to check/skip the vmlinuz header,
     or not even try to check. Mingujin.exe just set this bit depending on ".kgz" extension,
     so have also to start with something different than 0x8B1F to be is_linux.
     Concat'ed vmlinux+initrd will have bit has_vmlinuz_header set. */
  gzlib.crc32 = (elem->boottype == is_linux) && (LOADER.curfileload == LOADER.fileload);
#endif

  gzlib.avail_out = LOADER.curfileload->max_size_or_crc32;
  if (   BOOT1_DOS_running()
#if !(SETUP & BIOS_ONLY)
      && UTIL.HIMEM_entrypoint
#endif
      ) {
//      gzlib.next_out = 0; // done anyway
      }
    else {
#if ASSEMBLY_TYPE == ASSEMBLY_DSES
      gzlib.next_out = (void *)linear2dsrelative(LOADER.curfileload->load_address);
#else
      gzlib.next_out = (void *)LOADER.curfileload->load_address;
#endif
      }

#if DISK_SUPPORT & DOS_SUPPORT
  if (DI.param[elem->disk].access == dos_part) {
      FDBG ((" [Will load DOS file '%s' size %u bytes with initial offset: %u bytes] ",
		elem->filename, elem->filesize, LOADER.gzip_byte_treated));
      returned = DOS_system_file_load (elem->disk, elem->filename,
				  &LOADER.curfileload->compressed_size,
				  LOADER.gzip_byte_treated, &gzlib);
      if (returned)
	  returned += 0x2000;
      }
    else
#endif
      {
#if DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|E2FS_PROBE|ISOFS_PROBE)
      union filesystem_union fs = { .common = { .disk = elem->disk, .part = elem->partition }};

      SET_DRIVE_ACTIVE (&DI.param[elem->disk]);

      returned = open_filesystem (&fs, 0, 0);
      if (returned)
	  FDBG (("%s: error open_filesystem 0x%X.\r\n", __FUNCTION__, returned));
	else {
	  typedef int (*fctptr) (void *, unsigned char *, unsigned);
#ifdef NB_ISO
	  if (elem->inISO) {
	      FDBG (("\r\nKernel ISO file inode %u: ", elem->iso_inode));
	      unsigned filesize = elem->iso_filesize, nb_segment = sizeof(BOOTWAY.iso.remap)/sizeof(*BOOTWAY.iso.remap);
	      if (BOOTWAY.iso.current_used == (unsigned short)-1) {
		  unsigned returned = file_treat (&fs, elem->iso_inode, &filesize, (unsigned)&nb_segment, 0, &BOOTWAY.iso.remap);
		  if (returned) {
		      FDBG (("%s: error file_treat 0x%X to get ISO mapping.\r\n", __FUNCTION__, returned));
		      return 0xF000;
		      }
		  FDBG (("Got chain of %u elements (real filesize %u, filesize %u, holes?):\r\n", nb_segment, elem->iso_filesize, filesize));
		  BOOTWAY.iso.previous_nb_remap = BOOTWAY.iso.nb_remap = nb_segment;
		  /* Dirty: we need the right BOOTWAY.iso.current_used for next open_filesystem() to set "device_size": */
		  for (BOOTWAY.iso.current_used = 0; BOOTWAY.iso.current_used < BOOTWAY.iso.nb_found; BOOTWAY.iso.current_used++)
		      if (BOOTWAY.iso.found[BOOTWAY.iso.current_used].size == filesize)
			  break;
		  }
		else
		  BOOTWAY.iso.nb_remap = BOOTWAY.iso.previous_nb_remap;
	      returned = open_filesystem (&fs, 0, 0);
	      if (returned)
		  FDBG (("%s: error open_filesystem inside ISO 0x%X.\r\n", __FUNCTION__, returned));
	      }
	  if (returned == 0)
#endif
	    {
	    LOADER.curfileload->compressed_size = elem->filesize;
	    FDBG ((" [Will load file '%s' max size %u bytes with initial offset: %u bytes] ",
		elem->filename, LOADER.curfileload->compressed_size, LOADER.gzip_byte_treated));
	    returned = file_treat (&fs, elem->inode, &LOADER.curfileload->compressed_size,
					LOADER.gzip_byte_treated, (fctptr) gzlib_extract, &gzlib);
	    }
#ifdef NB_ISO
	  BOOTWAY.iso.nb_remap = 0;
#endif
	  /* Warning: LOADER.curfileload->compressed_size is rounded up to the filesystem blocksize! */
	  if (returned)
	      FDBG (("%s: error file_treat 0x%X.\r\n", __FUNCTION__, returned));
	  }

      SET_DRIVE_IDLE (&DI.param[elem->disk]);

      if (returned)
	  returned += 0x3000;
#else
      returned = 0x3000;
      FDBG (("Only DOS_SUPPORT compiled in, cannot load!\r\n"));
#endif
      }
  LOADER.curfileload->uncompressed_size = gzlib.total_out;

  return returned;
  }}

#if DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|E2FS_PROBE|ISOFS_PROBE)
FS_FCT_PREFIX(file_get_chain) unsigned
file_get_chain (struct desc_str *elem, unsigned *nb_fragment, struct sector_chain_str fragment_array[0])
  {bound_stack();{
  union filesystem_union fs = { .common = { .disk = elem->disk, .part = elem->partition }};

  SET_DRIVE_AUTO_ACTIVE (&DI.param[elem->disk]);

  unsigned returned = open_filesystem (&fs, 0, 0);
  if (returned)
      FDBG (("%s: error open_filesystem 0x%X.\r\n", __FUNCTION__, returned));

//#define DEBUG_MERGE
#ifdef DEBUG_MERGE
  printf ("First open_filesystem: sector_per_block %u, byte_per_block %u, blocks_count %llu, SectSize %u\r\n", fs.common.sector_per_block, fs.common.byte_per_block, fs.common.blocks_count, fs.common.SectSize);
#endif
  unsigned external_SectSize = fs.common.SectSize;

#ifdef NB_ISO
  unsigned total_nb_fragment = *nb_fragment;
  if (returned == 0 && elem->inISO) {
      FDBG (("\r\nMAP in ISO file inode %u: ", elem->iso_inode));
      unsigned filesize = elem->iso_filesize, nb_segment = sizeof(BOOTWAY.iso.remap)/sizeof(*BOOTWAY.iso.remap);
      unsigned returned = file_treat (&fs, elem->iso_inode, &filesize, (unsigned)&nb_segment, 0, &BOOTWAY.iso.remap);
      if (returned) {
	  FDBG (("%s: error file_treat 0x%X to get ISO mapping.\r\n", __FUNCTION__, returned));
	  return 0xF000;
	  }
      FDBG (("Got chain of %u elements (real filesize %u, filesize %u, holes?):\r\n", nb_segment, elem->iso_filesize, filesize));
      BOOTWAY.iso.nb_remap = nb_segment;
      returned = open_filesystem (&fs, 0, 0);
      if (returned)
	  FDBG (("%s: error open_filesystem inside ISO 0x%X.\r\n", __FUNCTION__, returned));
#ifdef DEBUG_MERGE
      printf ("Second open_filesystem: sector_per_block %u, byte_per_block %u, blocks_count %llu, SectSize %u\r\n", fs.common.sector_per_block, fs.common.byte_per_block, fs.common.blocks_count, fs.common.SectSize);
#endif
      }
  if (returned == 0)
#endif	/* NB_ISO */
      returned = file_treat (&fs, (elem->boottype == is_el_torito)? elem->iso_inode : elem->inode,
			(elem->boottype == is_el_torito)? &elem->iso_filesize : &elem->filesize, (unsigned)nb_fragment, 0, fragment_array);
  if (returned)
      FDBG (("%s: error file_treat 0x%X.\r\n", __FUNCTION__, returned));

  if (external_SectSize < DI.param[elem->disk].bytepersector) {
      unsigned char shift = __builtin_ffs(DI.param[elem->disk].bytepersector) - __builtin_ffs (external_SectSize);
      struct sector_chain_str *fragptr = fragment_array;
      while (fragptr->nb) {
	  if (fragptr->nb % (1 << shift) || (unsigned)fragptr->lba % (1 << shift)) {
	      FDBG (("Conversion FSsectsize %u DISKsectsize %u failed for lba %llu/nb %u\r\n", external_SectSize, DI.param[elem->disk].bytepersector, fragptr->lba, fragptr->nb));
	      return 0xD000;
	      }
	  fragptr->nb >>= shift;
	  fragptr->lba >>= shift;
	  fragptr++;
	  }
      }
    else if (external_SectSize > DI.param[elem->disk].bytepersector) {
      unsigned char shift = __builtin_ffs (external_SectSize) - __builtin_ffs(DI.param[elem->disk].bytepersector);
      struct sector_chain_str *fragptr = fragment_array;
      while (fragptr->nb) {
	  fragptr->nb <<= shift;
	  fragptr->lba <<= shift;
	  fragptr++;
	  }
      }

#ifdef NB_ISO
  if (returned == 0 && elem->inISO) {
#ifdef DEBUG_MERGE
      unsigned cpt;
      printf ("merging image with segments:\r\n");
      for (cpt = 0; cpt < *nb_fragment; cpt++)
	  printf (" %u sectors at %llU,", fragment_array[cpt].nb, fragment_array[cpt].lba);
      printf ("\r\ninto image with segments:\r\n");
      for (cpt = 0; cpt < BOOTWAY.iso.nb_remap; cpt++)
	  printf (" %u sectors at %llU,", BOOTWAY.iso.remap[cpt].nb, BOOTWAY.iso.remap[cpt].lba);
      printf ("\r\ngives:\r\n");
#endif
      struct sector_chain_str merge_fragment_array[total_nb_fragment];
      unsigned char shift1 = 0, shift2 = 0;
      if (fs.common.SectSize > external_SectSize)
	  shift1 = __builtin_ffs(fs.common.SectSize) - __builtin_ffs (external_SectSize);
//	else if (external_SectSize > fs.common.SectSize)
//	  shift2 = __builtin_ffs (external_SectSize) - __builtin_ffs(fs.common.SectSize);
      unsigned cptmerge, cptfrag, mapped_nb = fragment_array[0].nb << shift1;
      unsigned long long mapped_lba = fragment_array[0].lba << shift1;

      for (cptmerge = cptfrag = 0; cptfrag < *nb_fragment && cptmerge < total_nb_fragment; cptmerge++) {
	  /* I re-used the wheel, these tractor wheels are the main feature of my motocycle... */
	  unsigned long long cur_lba = mapped_lba;
	  unsigned segment = 0, nbsect = mapped_nb;
	  while (cur_lba >= BOOTWAY.iso.remap[segment].nb << shift2)
	      cur_lba -= BOOTWAY.iso.remap[segment++].nb << shift2;
	  if (nbsect > (BOOTWAY.iso.remap[segment].nb << shift2) - cur_lba)
	      nbsect = (BOOTWAY.iso.remap[segment].nb << shift2) - cur_lba;
	  if (BOOTWAY.iso.remap[segment].lba == 0 || (mapped_lba == 0 && fragment_array[1].nb != 0))
	      merge_fragment_array[cptmerge].lba = 0; /* we have a hole, and sure not CDnoemul */
	    else {
	      merge_fragment_array[cptmerge].lba = (BOOTWAY.iso.remap[segment].lba << shift2) + cur_lba;
	      if (BOOTWAY.iso.remap[segment].lba >> 63)
		  merge_fragment_array[cptmerge].lba += (cur_lba >> BOOTWAY.iso.intergap_shift) << BOOTWAY.iso.gap_sectors_shift;
	      }
	  merge_fragment_array[cptmerge].nb = nbsect;

	  if (BOOTWAY.iso.remap[segment].lba >> 63
		&& cur_lba >> BOOTWAY.iso.intergap_shift != (cur_lba + nbsect) >> BOOTWAY.iso.intergap_shift
		&& (cur_lba & ((1 << BOOTWAY.iso.intergap_shift) - 1)) != 0)
	       nbsect = merge_fragment_array[cptmerge].nb
		= (1 << BOOTWAY.iso.intergap_shift) - (cur_lba & ((1 << BOOTWAY.iso.intergap_shift) - 1));
	  mapped_lba += nbsect;
	  mapped_nb -= nbsect;
	  if (mapped_nb == 0) {
	      mapped_lba = fragment_array[++cptfrag].lba << shift1;
	      mapped_nb = fragment_array[cptfrag].nb << shift1;
	      }
	  }
      if (cptmerge >= total_nb_fragment) {
	  FDBG (("too many fragments\r\n"));
	  return 0xE000;
	  }
      for (cptfrag = 0; cptfrag < cptmerge; cptfrag++) {
#ifdef DEBUG_MERGE
	  printf (" %u sectors at %llU,", merge_fragment_array[cptfrag].nb, merge_fragment_array[cptfrag].lba);
#endif
	  fragment_array[cptfrag] = merge_fragment_array[cptfrag];
	  }
      *nb_fragment = cptmerge;
      fragment_array[cptmerge].lba = 0;
      fragment_array[cptmerge].nb = 0;
#ifdef DEBUG_MERGE
      printf ("\r\n press a key to continue\r\n"); _BIOS_getkey();
#endif
      }
  BOOTWAY.iso.nb_remap = 0;
#endif	/* NB_ISO */

// If we plan to simulate a floppy from a CDROM, we'd better leave the CDROM ACTIVE...
  if (returned != 0)
      SET_DRIVE_IDLE (&DI.param[elem->disk]);

  return 0;
  }}
#endif	/* DISK_SUPPORT & (FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|E2FS_PROBE|ISOFS_PROBE) */

/*
 * The main structure of this file in .BSS
 */
struct BOOTWAY_str BOOTWAY;

