/* fntbin2h.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>

/* Advise on how to write a font:
 * Write the minuscule "o", then deduce from it:
 * p, q, d, b. g, c, u, then n, r, s, z, then x, y
 * a and e comes also from o
 * From b deduce h and then h deduce k and l then t and i, j
 * For capitals, the "vanillia ice" is the B for me.
 * (vanillia ice == starting point)
 */


int main (int argc, char **argv)
  {
  FILE *fin, *fout;
  struct stat statbuf;
  unsigned fntheigh;
  time_t srctime;

  if (argc != 3) {
      printf (  " Quick and dirty software to tranform a binary font file\n"
		" to a font include file for the Gujin bootloader.\n"
		" USAGE: fntbin2h <srcfile> <dstfile>\n");
      return 0;
      }

  if (stat(argv[1], &statbuf) != 0) {
      printf ("Error: cannot stat input file: \"%s\"\n", argv[1]);
      return 1;
      }
  srctime = statbuf.st_mtime;
  if (statbuf.st_size % 256 != 0) {
      printf ("Error: input \"%s\" file size: %ld, not multiple of 256\n",
		argv[1], statbuf.st_size);
      return 1;
      }
  fntheigh = statbuf.st_size / 256;
  if (fntheigh <= 4 || fntheigh > 64) {
      printf ("Error: input \"%s\" file size: %ld, font heigh %u, seriously out of range\n",
		argv[1], statbuf.st_size, fntheigh);
      return 1;
      }
  {
  unsigned char font[256][fntheigh];

  if ((fin = fopen (argv[1], "r")) == NULL) {
      printf ("Error: cannot open input file: \"%s\"\n", argv[1]);
      return 1;
      }
  if (fread(font, fntheigh, 256, fin) != 256) {
      printf ("Error: cannot completely read input file: %s\n", argv[1]);
      fclose (fin);
      return 1;
      }
  if (fclose (fin) == EOF) {
      printf ("Error: cannot close input file: %s\n", argv[1]);
      return 1;
      }

  if (stat(argv[2], &statbuf) == 0 && statbuf.st_mtime >= srctime) {
      printf ("Error: output file \"%s\" exists and is newer than input file \"%s\".\n",
		argv[2], argv[1]);
      return 1;
      }

  if ((fout = fopen (argv[2], "w")) == NULL) {
      printf ("Error: cannot open output file: \"%s\"\n", argv[2]);
      return 1;
      }

  fprintf (fout, "const unsigned char font8x%u[256][%u] = {\n", fntheigh, fntheigh);

  {
  unsigned font_index;
  const unsigned char font_order[256] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,

	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F,

	0x7F,  '',  '',  '',  '',  '',  '',  '%',
	 '',  '',  '',  '',  '.',  ':',  ',',  ';',
	'\'',  '"',  '`',  '^',  '~',  '',  '',  '',
	 '',  '',  '',  '',  '',  '&',  '',  '',
	 '[',  ']',  '{',  '}',  '(',  ')',  '<',  '>',
	 '#',  '@',  '?',  '',	 '!',  '',  '',  '|',
	 '+',  '-',  '',  '',  '/', '\\',  '_',  '',
	 '',  '*',  '=',  '',  '',  '$',  '',  '',
	 '0',  '1',  '2',  '3',  '4',  '5',  '6',  '7',
	 '8',  '9',  ' ', 0xA0,	 '',  '',  '',  '', /* NBSP, big/small islandic thorn */

	/* Accented letter order:
		grave, acute, circumflex, tilda, trema, round */
	 'A',  '',  '',  '',  '',  '',  '',  '',
	 'B',  'C',  '',  'D',  '',  'E',  '',  '',
	 '',  '',  'F',  'G',  'H',  'I',  '',  '',
	 '',  '',  'J',  'K',  'L',  'M',  'N',  '',
	 'O',  '',  '',  '',  '',  '',  '',  'P',
	 'Q',  'R',  'S',  '',  'T',  'U',  '',  '',
	 '',  '',  'V',  'W',  'X',  'Y',  '',  'Z',

	 'a',  '',  '',  '',  '',  '',  '',  '',
	 'b',  'c',  '',  'd',  '',  'e',  '',  '',
	 '',  '',  'f',  'g',  'h',  'i',  '',  '',
	 '',  '',  'j',  'k',	 'l',  'm',  'n',  '',
	 'o',  '',  '',  '',  '',  '',  '',  'p',
	 'q',  'r',  's',  't',	 'u',  '',  '',  '',
	 '',  'v',  'w',  'x',  'y',  '',  '',  'z'
	};

  for (font_index = 0; font_index < 256; font_index++) {
      unsigned cpt;
      if (   font_order[font_index] < 0x20
	  || (font_order[font_index] >= 0x7F && font_order[font_index] < 0xA0))
	  fprintf (fout, "    [0x%X] = {\n", font_order[font_index]);
	else if (font_order[font_index] == '\'' || font_order[font_index] == '\\')
	  fprintf (fout, "    ['\\%c'] = {\n", font_order[font_index]);
	else
	  fprintf (fout, "    ['%c'] = {\n", font_order[font_index]);
      for (cpt = 0; cpt < fntheigh; cpt++) {
	  unsigned tmp = font[font_order[font_index]][cpt];
	  fprintf (fout, "\t%c (%c,%c,%c,%c,%c,%c,%c,%c),\n",
//		(cpt == 0) ? 'T' : ((cpt == fntheigh - 1) ? 'B' : 'M'),
		'B',
		(tmp & 0x80)? 'X' : '_', (tmp & 0x40)? 'X' : '_',
		(tmp & 0x20)? 'X' : '_', (tmp & 0x10)? 'X' : '_',
		(tmp & 0x08)? 'X' : '_', (tmp & 0x04)? 'X' : '_',
		(tmp & 0x02)? 'X' : '_', (tmp & 0x01)? 'X' : '_');
	  }
      fprintf (fout, "\t},\n");
      }

  fprintf (fout, "    }; /* font8x%u */\n", fntheigh);

  } /* font_order */
  if (fclose (fout) == EOF) {
      printf ("Error: cannot close output file: %s\n", argv[2]);
      return 1;
      }

  } /* font */
  return 0;
  }
