/* ega.h */
#ifndef EGA_H
#define EGA_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

struct EGA_mode_str {
    enum { cpuORsetreset = 0, latch, cpuASsetreset,
	    cpuASsetrestANDbitmask /* VGA only */} write_source : 2;
    unsigned char enable_tristate : 1;      /* = 0 */
    unsigned char read_mode : 1;
    unsigned char enable_odd_even_addr : 1; /* = 1 */
    unsigned char CGA_shift_register : 1;   /* = 0 */
    unsigned char enable_256_color :1;
    unsigned char reserved : 1;
    };

struct EGA_setup_str {
    unsigned char graphic_mode : 1;
    unsigned char chain_maps : 1;
    enum { A0000_BFFFF = 0, A0000_AFFFF,
	    B0000_B7FFF, B8000_BFFFF } memory_map : 2;
    unsigned char reserved : 4;
    };

#if 1	/* Is that working everywhere ? */
extern inline void
OUTW (unsigned short addr, unsigned char off, unsigned char val)
  {
  outw (addr, (((unsigned short)val) << 8) | off);
  }
#else
extern inline void
OUTW (unsigned short addr, unsigned char off, unsigned char val)
  {
  outb (addr, off);
  outb (addr+1, val);
  }
#endif

#if (__GNUC__ > 3) || (__GNUC__ == 3 && __GNUC_MINOR__ >= 3)
union EGA_setup_mode_union {
    struct EGA_mode_str mode;
    struct EGA_setup_str setup;
    unsigned char all;
    } __attribute__ ((__transparent_union__));

extern inline void
_EGA_GCR_out (unsigned char offset, union EGA_setup_mode_union value)
  {
  OUTW (0x3CE, offset, value.all);
  }
#else
extern inline void
_EGA_GCR_out (unsigned char offset, unsigned char value)
  {
  OUTW (0x3CE, offset, value);
  }
#endif

extern inline void
_EGA_CRTR_out (unsigned char offset, unsigned char value)
  {
  OUTW (0x3D4, offset, value); /* 3B4 in mono mode ! */
  }

/* Some old VGA may not have more than offset 0xC to 0xF readable */
extern inline unsigned char
_EGA_CRTR_in (unsigned char offset)
  {
  outb (0x3D4, offset); /* 3B4 in mono mode ! */
  return inb (0x3D5); /* 3B5 in mono mode ! */
  }

extern inline void
_EGA_SEQ_out (unsigned char offset, unsigned char value)
  {
  OUTW (0x3C4, offset, value);
  }

extern inline void
_EGA_ATTR_out (unsigned char offset, unsigned char value)
  {
  disable_interrupt();
  inb (0x3DA); /* 0x3BA in monochrome */ /* reset flip/flop */
  outb (0x3C0, offset | 0x20);
  outb (0x3C0, value);
  enable_interrupt();
  }

extern inline unsigned char
_EGA_ATTR_in (unsigned char offset)
  {
  unsigned char returned;

  disable_interrupt();
  inb (0x3DA); /* 0x3BA in monochrome */ /* reset flip/flop */
  outb (0x3C0, offset | 0x20);
  returned = inb (0x3C1);
  enable_interrupt();
  return returned;
  }

/*
 * Interresting stuff:
 */

extern inline void
_EGA_blinking (unsigned char value)
  {
  if (value)
      _EGA_ATTR_out (0x10, _EGA_ATTR_in (0x10) | 0x08);
    else
      _EGA_ATTR_out (0x10, _EGA_ATTR_in (0x10) & ~0x08);
  }

extern inline void
_EGA_setcursor (unsigned short value)
  {
  _EGA_CRTR_out (0xF, value);
  _EGA_CRTR_out (0xE, value >> 8);
  }

extern inline void
_EGA_enable_plane (unsigned char value)
  {
  _EGA_SEQ_out (2, value);
  }

/* if bit b (0..3) set, write 0xFF in plane b: */
extern inline void
_EGA_set_reset (unsigned char mask)
  {
  _EGA_GCR_out (0, mask);
  }

/* if bit b (0..3) set, write comes from set/reset register: */
extern inline void
_EGA_set_reset_enable (unsigned char mask)
  {
  _EGA_GCR_out (1, mask);
  }

/* for read mode 1, color comparisson: */
extern inline void
_EGA_read_compare (unsigned char color)
  {
  _EGA_GCR_out (2, color);
  }

enum EGA_write_mode { EGA_overwride = 0, EGA_and, EGA_or, EGA_xor };

extern inline void
_EGA_write_setup (enum EGA_write_mode mode, unsigned char rotate)
  {
  _EGA_GCR_out (3, (unsigned char)(rotate | ((unsigned char)mode << 3)));
  }

/* for read mode 0: */
extern inline void
_EGA_select_read_plane (unsigned char plane)
  {
  _EGA_GCR_out (4, plane);
  }

extern inline void
_EGA_read_write_mode (struct EGA_mode_str mode)
  {
#if (__GNUC__ > 3) || (__GNUC__ == 3 && __GNUC_MINOR__ >= 3)
  _EGA_GCR_out (5, mode);
#else
  _EGA_GCR_out (5, *(unsigned char *)&mode);
#endif
  }

extern inline void
_EGA_setup (struct EGA_setup_str setup)
  {
#if (__GNUC__ > 3) || (__GNUC__ == 3 && __GNUC_MINOR__ >= 3)
  _EGA_GCR_out (6, setup);
#else
  _EGA_GCR_out (6, *(unsigned char *)&setup);
#endif
  }

/* for read mode 1, color comparisson: */
extern inline void
_EGA_select_read_ignore (unsigned char mask)
  {
  _EGA_GCR_out (7, mask);
  }

/* if bit set, pixel change allowed: */
extern inline void
_EGA_write_bitmask (unsigned char mask)
  {
  _EGA_GCR_out (8, mask);
  }

/*
 * DAC treatments:
 */
extern inline unsigned char
_EGA_get_retrace (void)
  {
  return inb (0x3C2) & 0x80;
  }

extern inline void wait_retrace (unsigned timeout)
  {
  while (--timeout && _EGA_get_retrace () != 0);
  }

extern inline void
_EGA_getDAC (unsigned short index,
	      unsigned char *_red, unsigned char *_green, unsigned char *_blue)
  {
  outb (0x3C7, index);
  *_red = inb (0x3C9);
  *_green = inb (0x3C9);
  *_blue = inb (0x3C9);
  inb (0x3C7);
  }

extern inline void
_EGA_setDAC (unsigned short index,
	      unsigned char _red, unsigned char _green, unsigned char _blue)
  {
  wait_retrace (0x10000);
  outb (0x3C8, index);
  outb (0x3C9, _red);
  outb (0x3C9, _green);
  outb (0x3C9, _blue);
  inb (0x3C8);
  }

extern inline void
_EGA_setDACarray (unsigned short startindex, unsigned short nb,
		   const struct DAC_str *palette)
  {
  unsigned /*short*/ cpt = 0;

  wait_retrace (0x10000);
  outb (0x3C8, startindex);
  while (cpt < nb) {
      outb (0x3C9, palette[cpt].red);
      outb (0x3C9, palette[cpt].green);
      outb (0x3C9, palette[cpt].blue);
      cpt ++;
      }
  inb (0x3C8);
  }

#endif /* EGA_H */
