/* dos.h */
#ifndef DOS_H
#define DOS_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * gcc stuff:
 */
typedef union {
    union {
	struct {
	    unsigned char xl, xh;
	    } __attribute__ ((packed)) b;
	unsigned short xw;
	} __attribute__ ((packed)) w;
    unsigned xl;
    } __attribute__ ((packed)) Qreg;


/*
 * DOS 1+:
 */
extern inline void
DOS_TerminateProgram (void)
  {
  Qreg ax;

  ax.w.b.xh = 0x00;
  /* %cs has to point to the PSP segment, deprecated */
  /* errorlevel is set to 0 */
  asm volatile (" int	$0x21 # DOS_TerminateProgram "
	: /* no output */
	: "a" (ax)
	);
  }

extern inline unsigned char
DOS_StdinInputWithEcho (void) /* check keyboard ^C/^Break */
  {
  Qreg ax;

  ax.w.b.xh = 0x01;

  asm volatile (" int	$0x21 # DOS_StdinInputWithEcho "
	: "+a" (ax)
	);
  return ax.w.b.xl;
  }

extern inline void
DOS_StdoutOutput (unsigned char character) /* check keyboard ^C/^Break */
  {
  Qreg ax, dx;

  ax.w.b.xh = 0x02;
  dx.w.b.xl = character;

  asm volatile (" int	$0x21 # DOS_StdoutOutput "
	: "+a" (ax)	// AL = last character output
	: "d" (dx)
	);
  }

extern inline unsigned char
DOS_StdauxInput (void) /* check keyboard ^C/^Break */
  {
  Qreg ax;

  ax.w.b.xh = 0x03;

  asm volatile (" int   $0x21 # DOS_StdauxInput "
	: "+a" (ax)
	);
  return ax.w.b.xl;
  }

extern inline void
DOS_StdauxOutput (unsigned char character) /* check keyboard ^C/^Break */
  {
  Qreg ax, dx;

  ax.w.b.xh = 0x04;
  dx.w.b.xl = character;
  asm volatile (" int	$0x21 # DOS_StdauxOutput "
	: /* no output */
	: "a" (ax), "d" (dx)
	);
  }

extern inline void
DOS_StdprnOutput (unsigned char character) /* check keyboard ^C/^Break */
  {
  Qreg ax, dx;

  ax.w.b.xh = 0x05;
  dx.w.b.xl = character;
  asm volatile (" int	$0x21 # DOS_StdprnOutput "
	: /* no output */
	: "a" (ax), "d" (dx)
	);
  }

extern inline unsigned char
DOS_ConsoleInput (char *character) /* no ^C/^Break */
  {
  Qreg ax, dx;
  unsigned char zero;

  ax.w.b.xh = 0x06;
  dx.w.b.xl = 0xFF;

  asm volatile (
"	int	$0x21 # DOS_ConsoleInput	\n"
"	setz	%0				\n"
	: "=qm" (zero), "+a" (ax)
	: "d" (dx)
	);
  *character = ax.w.b.xl;
  return zero; /* != 0 if no char present */
  }

extern inline void
DOS_ConsoleOutput (unsigned char character) /* no ^C/^Break */
  {
  Qreg ax, dx;

  ax.w.b.xh = 0x06;
  dx.w.b.xl = (character == 0xFF) ? 0 : character;

  asm volatile (
"	int	$0x21 # DOS_ConsoleOutput	\n"
	: "+a" (ax) // AL = character output
	: "d" (dx)
	);
  }

extern inline unsigned char
DOS_DirectInputWithoutEcho (void) /* no ^C/^Break */
  {
  Qreg ax;

  ax.w.b.xh = 0x07;

  asm volatile ("  int	$0x21 # DOS_DirectInputWithoutEcho "
	: "+a" (ax)
	);
  return ax.w.b.xl;
  }

extern inline unsigned char
DOS_StdinInput (void) /* check keyboard ^C/^Break */
  {
  Qreg ax;

  ax.w.b.xh = 0x08;

  asm volatile (" int   $0x21 # DOS_StdinInput "
	: "+a" (ax)
	);
  return ax.w.b.xl;
  }

/* check keyboard ^C/^Break: */
extern inline void
DOS_WriteStringStdout (const char *dollar_string)
  {
  Qreg ax, dx;

  ax.w.b.xh = 0x09;
  dx.w.xw = (unsigned short)(unsigned)dollar_string; /* %ds:%dx */

  asm volatile (" int	$0x21 # DOS_WriteStringStdout "
	: "+a" (ax)	// AL == '$'
	: "d" (dx), ASM_STRUCT_INPUT (*dollar_string)
	);
  }

extern inline void
DOS_IdleInterrupt (void)
  {
  asm (" int	$0x28 # DOS_IdleInterrupt " );
  }

extern inline void
DOS_ResetDisk (void)
  {
  Qreg ax;

  ax.w.b.xh = 0x0D;
  /* Flushes also files */
  asm volatile (" int	$0x21 # DOS_ResetDisk "
	: : "a" (ax) : "cc"
	);
  }

extern inline unsigned char
DOS_SelectDisk (unsigned char disk)
  {
  Qreg ax, dx;

  ax.w.b.xh = 0x0E;
  dx.w.b.xl = disk; /* 0 for A: */

  asm volatile (" int	$0x21 # DOS_SelectDisk "
	: "+a" (ax)
	: "d" (dx)
	);
  return ax.w.b.xl; /* number of potentially valid drive letters */
  }

extern inline unsigned char
DOS_GetCurrentDefaultDrive (void)
  {
  Qreg ax;

  ax.w.b.xh = 0x19;
  asm (" int	$0x21 # DOS_GetCurrentDefaultDrive "
	: "+a" (ax)
	);
  return ax.w.b.xl; /* 0 for A: */
  }

struct DOS_DTA_str {
    unsigned char		OSdepend[0x15];
    struct file_attr_str	attribute;
    struct {
	unsigned seconddiv2	: 5;
	unsigned minute		: 6;
	unsigned hour		: 5;
	unsigned day		: 5;	/* 1..31 */
	unsigned month		: 4;	/* 1..12 */
	unsigned year		: 7;	/* 1980 + x */
	} __attribute__ ((packed)) date;
    unsigned			size;
    char			name_ext[13]; /* ASCIZ */
    } __attribute__ ((packed));

extern inline void
DOS_SetDTA (struct DOS_DTA_str *dta_atr)
  {
  Qreg ax, dx;

  ax.w.b.xh = 0x1A;
  dx.w.xw = (unsigned short)(unsigned)dta_atr; /* %ds:%dx */

  asm volatile (" int	$0x21 # DOS_SetDTA "
	: : "a" (ax), "d" (dx)
	);
  }

extern inline unsigned char
DOS_GetAllocationTableInformation (unsigned char *SectorPerCluster,
				   unsigned short *BytesPerSector,
				   unsigned short *ClustersOnDisk)
  {
  Qreg ax, bx, cx, dx;

  ax.w.b.xh = 0x1B;

  asm (
"	pushl	%%ds						\n"
"	int	$0x21 # DOS_GetAllocationTableInformation	\n"
"	movb	%%ds:(%%bx),%bl	# get it directly		\n"
"	popl	%%ds						\n"
	: "+a" (ax), "=b" (bx), "=c" (cx), "=d" (dx)
	);
  *SectorPerCluster = ax.w.b.xl; /* buggy DOS 7.10 */
  *BytesPerSector = cx.w.xw;
  *ClustersOnDisk = dx.w.xw; /* buggy DOS 7.10 */
  return bx.w.b.xl; /* media ID byte */
  }

extern inline unsigned char
DOS_GetAllocationTableInformationForDrive (
		unsigned char drive, /* 0 for default, 1 for A: */
		unsigned char *SectorPerCluster,
		unsigned short *BytesPerSector,
		unsigned short *ClustersOnDisk)
  {
  Qreg ax, bx, cx, dx;

  ax.w.b.xh = 0x1C;
  dx.w.b.xl = drive; /* 0 : default */

  asm (
"	pushl	%%ds						\n"
"	int	$0x21 # DOS_GetAllocationTableInformation_drive	\n"
"	movb	%%ds:(%%bx),%%bl	# get it directly		\n"
"	popl	%%ds						\n"
	: "+a" (ax), "+d" (dx), "=b" (bx), "=c" (cx)
	);
  *SectorPerCluster = ax.w.b.xl; /* 0xFF if invalid drive */ /* buggy DOS 7.10 */
  *BytesPerSector = cx.w.xw;
  *ClustersOnDisk = dx.w.xw; /* buggy DOS 7.10 */
  return bx.w.b.xl; /* media ID byte */
  }

extern inline void
DOS_SetInterruptVector (unsigned char vector_number, farptr address)
  {
  Qreg ax, dx;

  ax.w.b.xh = 0x25;
  ax.w.b.xl = vector_number;
  dx.xl = address; /* %ds:%dx */

  asm volatile (
"	pushl	%%ds					\n"
"	pushl	%%edx					\n"
"	popw	%%dx					\n"
"	popw	%%ds					\n"
"	int	$0x21 # DOS_SetInterruptVector		\n"
"	popl	%%ds					\n"
	: : "a" (ax), "d" (dx)
	);
  }

/*
 * DOS 2+:
 */

extern inline farptr
DOS_GetDTA (void)
  {
  Qreg ax;
  farptr dta_adr;

  ax.w.b.xh = 0x2F;

  asm (
"	pushl	%%es			\n"
"	int	$0x21 # DOS_GetDTA	\n"
"	pushw	%%es			\n"
"	pushw	%%bx			\n"
"	popl	%%ebx			\n"
"	popl	%%es			\n"
	: "=b" (dta_adr)
	: "a" (ax)
	);
  return dta_adr;
  }

extern inline unsigned short DOS_old_version (void)
  {
  Qreg ax;

  ax.w.b.xh = 0x30;

  asm (" int $0x21 # DOS_old_version"
	: "+a" (ax)
	: : "bx", "cx");

  return ax.w.xw; /* %ah major, %al minor */
  }

/* DOS 4.0 + */
extern inline unsigned char
DOS_GetBootDrive (void)
  {
  Qreg ax, dx;

  ax.w.xw = 0x3305;

  asm (" int    $0x21 # DOS_GetBootDrive "
	: "=d" (dx)
	: "a" (ax)
	);
  return dx.w.b.xl; /* 1 for A: */
  }

/* DOS 5+ : */
extern inline unsigned short DOS_new_version (void)
  {
  Qreg ax, bx, dx;

  ax.w.xw = 0x3306;
  bx.w.xw = 0; /* version if failed: 0 */

  asm (" int $0x21 # DOS_new_version, DOS 5.0 +"
	: "+a" (ax), "+b" (bx), "=d" (dx)
	);
  if (ax.w.b.xl != 0xFF && bx.w.b.xl >= 5)
      return bx.w.xw; /* %bl: major */
    else
      return 0;
  }

extern inline unsigned short DOS_version (void)
  {
  unsigned short version;

  version = DOS_new_version ();
  if (version == 0)
      version = DOS_old_version ();
  /* return it in BCD to be simple, i.e. 0x301: */
  return (version >> 8) | (version << 8);
  }

extern inline farptr
DOS_GetInterruptVector (unsigned char vector_number)
  {
  Qreg ax;
  farptr returned;

  ax.w.b.xh = 0x35;
  ax.w.b.xl = vector_number;

  asm (
"	pushl	%%es					\n"
"	int	$0x21 # DOS_GetInterruptVector		\n"
"	pushw	%%es					\n"
"	pushw	%%bx					\n"
"	popl	%%ebx					\n"
"	popl	%%es					\n"
	: "=b" (returned)
	: "a" (ax)
	);
  return returned;
  }

extern inline unsigned
DOS_GetDiskFreeSpace (unsigned char drive,
			unsigned short *SectorPerCluster,
			unsigned short *BytesPerSector,
			unsigned short *ClustersPerDrive,
			unsigned short *NbAvailableCluster)
  {
  Qreg ax, bx, cx, dx;

  ax.w.b.xh = 0x36;
  dx.w.b.xl = drive; /* 0 for default, 1 for A: */

  asm ("  int	$0x21 # DOS_GetDiskFreeSpace "
	: "+a" (ax), "+d" (dx), "=c" (cx), "=b" (bx)
	);
  if (ax.w.xw == 0xFFFF)
      return 1;
  *ClustersPerDrive = dx.w.xw;
  *NbAvailableCluster = bx.w.xw;
  *SectorPerCluster = ax.w.xw;
  *BytesPerSector = cx.w.xw;
  return 0;
  }

/* All following are "Using Handle": */

struct DOS_attribute_str {
    unsigned short	read_only    : 1;
    unsigned short	hidden       : 1;
    unsigned short	system       : 1;
    unsigned short	volume_label : 1;
    unsigned short	directory    : 1;
    unsigned short	archive      : 1;
    unsigned short	reserved     : 10;
    } __attribute__ ((packed));

extern inline unsigned char
DOS_CreateFile (const char *name,
		struct DOS_attribute_str attribute,
		unsigned short *handle)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.b.xh = 0x3C;

  asm volatile (
"	int	$0x21 # DOS_CreateFile, truncate if exists	\n"
"	setc	%0						\n"
	: "=qm" (carry), "=a" (*handle)
	: "a" (ax), "c" (attribute),
	  "d" (name),	/* in fact %ds:%dx */
	  ASM_STRUCT_INPUT (*name)
	: "cc");
  return carry;
  }

enum DOS_open_access {
    DOS_open_read_only = 0,
    DOS_open_write_only,
    DOS_open_read_write
    };

extern inline unsigned char
DOS_OpenFile (const char *name,
	      unsigned char attribute,
	      unsigned short *handle)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.b.xh = 0x3D;
  ax.w.b.xl = attribute;

  asm volatile (
"	stc		# Error creation if DOS not running	\n"
"	int	$0x21	# DOS_OpenFile				\n"
"	setc	%0						\n"
	: "=qm" (carry), "=a" (*handle)
	: "a" (ax), "d" (name),	/* in fact %ds:%dx */
	  ASM_STRUCT_INPUT (*name)
	: "cc");
  return carry;
  }

extern inline unsigned char
DOS_CloseFile (unsigned short handle, unsigned short *errorcode)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.b.xh = 0x3E;

  asm volatile (
"	int	$0x21 # DOS_CloseFile		\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*errorcode)
	: "a" (ax), "b" (handle)
	: "cc");
  return carry;
  }

extern inline unsigned char
DOS_ReadFile (unsigned short handle, void *buffer, unsigned short nb,
	      unsigned short *nbread_errorcode)
  {
  unsigned char carry;
  Qreg ax;
  char **buf = (char **)&buffer;

  ax.w.b.xh = 0x3F;

  asm volatile (
"	int	$0x21 # DOS_ReadFile		\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*nbread_errorcode), ASM_STRUCT_OUTPUT (**buf)
	: "a" (ax), "b" (handle), "c" (nb),
		"d" (buffer) /* in fact %ds:%dx */
	: "cc");
  return carry;
  }

extern inline unsigned char
DOS_WriteFile (unsigned short handle, const char *buffer,
	       unsigned short nb, unsigned short *nbwrite_errorcode)
  {
  unsigned char carry;
  Qreg ax;
  /* Writing 0 bytes truncate the file to current position */

  ax.w.b.xh = 0x40;

  asm volatile (
"	int	$0x21 # DOS_WriteFile		\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*nbwrite_errorcode)
	: "a" (ax), "b" (handle), "c" (nb),
		"d" (buffer), /* in fact %ds:%dx */
	  ASM_STRUCT_INPUT (*buffer)
	: "cc");
  return carry;
  }

extern inline unsigned char
DOS_DeleteFile (const char *name, unsigned short *errorcode)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.b.xh = 0x41;

  asm volatile (
"	int	$0x21 # DOS_DeleteFile		\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*errorcode)
	: "a" (ax), "d" (name), /* %ds:%dx */
	  ASM_STRUCT_INPUT (*name)
	: "cc");
  return carry;
  }

enum DOS_seek { DOS_SEEK_SET = 0, DOS_SEEK_CUR, DOS_SEEK_END };

extern inline unsigned char
DOS_SeekFile (unsigned short handle,
	      enum DOS_seek  origin,
	      signed int     pos,
	      unsigned       *newpos,
	      unsigned short *errorcode)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.b.xh = 0x42;
  ax.w.b.xl = origin;

  asm volatile (
"	movw	%%cx,%%dx			\n"
"	sarl	$16,%%ecx			\n"
"	int	$0x21 # DOS_SeekFile		\n"
"	setc	%0				\n"
"	shll	$16,%%edx			\n"
"	mov	%%ax,%%dx			\n"
	: "=qm" (carry), "=a" (*errorcode), "=d" (*newpos)
	: "a" (ax), "b" (handle), "c" (pos)
	: "cc");
  return carry;
  }

extern inline unsigned char
DOS_GetAttributeFile (const char *name,
		      struct DOS_attribute_str *attribute,
		      unsigned short *errorcode)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.xw = 0x4300;

  asm (
"	int	$0x21 # DOS_GetAttributeFile	\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*errorcode), "=c" (*attribute)
	: "a" (ax), "d" (name),	/* in fact %ds:%dx */
	  ASM_STRUCT_INPUT (*name)
	: "cc");
  return carry;
  }

extern inline unsigned char
DOS_SetAttributeFile (const char *name,
		      struct DOS_attribute_str attribute,
		      unsigned short *errorcode)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.xw = 0x4301;

  asm volatile (
"	int	$0x21 # DOS_SetAttributeFile	\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*errorcode)
	: "a" (ax), "c" (attribute),"d" (name), /* in fact %ds:%dx */
	  ASM_STRUCT_INPUT (*name)
	: "cc");
  return carry;
  }

union DOS_device_info {
    struct DOS_char_device_info {
	unsigned short standard_input	: 1;
	unsigned short standard_output	: 1;
	unsigned short NUL_device	: 1;
	unsigned short clock_device	: 1;
	unsigned short device_special	: 1;	/* uses INT 29 */
	unsigned short raw_binary_mode	: 1;
	unsigned short EOF_on_input	: 1;
	unsigned short char_device	: 1;	/* set */
	unsigned short msdos_keyb	: 1;	/* ??? */
	unsigned short reserved1	: 2;
	unsigned short OPEN_CLOSE	: 1;
	unsigned short reserved2	: 1;
	unsigned short output_until_busy: 1;
	unsigned short IOCTL_requests	: 1;	/* %ax=0x4402 */
	unsigned short reserved3	: 1;
	} __attribute__ ((packed)) character;
    struct DOS_file_device_info {
	unsigned short drive_number	: 6;	/* 0 = A: */
	unsigned short file_has_not_been_written : 1;
	unsigned short char_device	: 1;	/* clear */
	unsigned short generate_INT_24	: 1;	/* if device full/read after EOF */
	unsigned short reserved1	: 2;
	unsigned short media_not_removable : 1;
	unsigned short reserved2	: 2;
	unsigned short don_t_set_file_date_time_on_closing : 1;	/* DOS 3.0+ */
	unsigned short file_is_remote	: 1;	/* DOS 3.0+ */
	} __attribute__ ((packed)) file;
    } __attribute__ ((packed));

extern inline unsigned char
DOS_GetDeviceInfo (unsigned short handle, union DOS_device_info *devinfo,
			unsigned short *errorcode)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.xw = 0x4400;

  asm (
"	int	$0x21 # DOS_GetDeviceInfo	\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*errorcode), "=d" (*devinfo)
	: "a" (ax), "b" (handle)
	: "cc");
  return carry;
  }

/* returns 0xFF if ready or 0 if not ready in "*status_errorcode" */
extern inline unsigned char
DOS_GetDeviceOutputStatus (unsigned short handle,
			unsigned short *status_errorcode)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.xw = 0x4407;

  asm (
"	int	$0x21 # DOS_GetDeviceInfo	\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*status_errorcode)
	: "a" (ax), "b" (handle)
	: "cc");
  if (!carry)
      *status_errorcode &= 0xFF;
  return carry;
  }

extern inline unsigned char
DOS_GetCwd (unsigned char disk, /* 0 for default, 1 for A: */
	    char name[64],	/* allocated by user */
	    unsigned short *errorcode)
  {
  unsigned char carry;
  Qreg ax, dx;

  ax.w.b.xh = 0x47;
  dx.w.b.xl = disk;

  asm (
"	int	$0x21 # DOS_GetCwd	\n"
"	setc	%0			\n"
	: "=qm" (carry), "=a" (*errorcode), ASM_STRUCT_OUTPUT (*name)
	: "a" (ax), "d" (dx),
	  "S" (name) /* in fact %ds:%si, ASCIIZ without first \ */
	: "cc");
  return carry;
  }

extern inline unsigned char
DOS_alloc_mem (unsigned short nbparagraph, unsigned short *error_seg)
  {
  unsigned char carry;
  unsigned short max_block_size; /* if carry and error_seg = 8 */
  Qreg ax;

  ax.w.b.xh = 0x48;

  asm volatile (
"	int	$0x21 # DOS_alloc_mem		\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*error_seg), "=b" (max_block_size)
	: "a" (ax), "b" (nbparagraph)
	: "memory", "cc");
  return carry;
  }

extern inline unsigned char
DOS_free_mem (unsigned short seg, unsigned short *error)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.b.xh = 0x49;

  asm volatile (
"	pushl	%%es				\n"
"	movw	%%dx,%%es			\n"
"	int	$0x21 # DOS_free_mem		\n"
"	popl	%%es				\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*error)
	: "a" (ax), "d" (seg)
	: "memory", "cc");
  return carry;
  }

/* if the following fails on some DOS (not enough memory),
   have to free the block: */
extern inline unsigned char
DOS_realloc_mem (unsigned short seg,
		 unsigned short new_nbparagraph,
		 unsigned short *error)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.b.xh = 0x4A;

  asm volatile (
"	pushl	%%es				\n"
"	movw	%%dx,%%es			\n"
"	int	$0x21 # DOS_realloc_mem		\n"
"	popl	%%es				\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*error), "+b" (new_nbparagraph)
	: "a" (ax), "d" (seg)
	: "memory", "cc");
  return carry;
  }

struct DOS_exec_str {
    unsigned short current_environment_segment; /* copy caller environment if 0 */
    farptr command_tail_pointer;
    farptr first_fcb;
    farptr second_fcb;
    farptr initial_ss_sp; /* subfunction 1 */
    farptr initial_cs_ip; /* subfunction 1 */
    };
extern inline unsigned char
DOS_exec_pgm (const char *filename, struct DOS_exec_str *env_to_set,
		 unsigned short *error)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.b.xh = 0x4B;
  ax.w.b.xl = 0;

  asm volatile (
"	pushl	%%gs				\n"
"	pushl	%%es				\n"
"	pushl	%%ds				\n"
"	pushal					\n"
"	int	$0x21 # DOS_exec_pgm		\n"
"	mov	%%ax,%%fs			\n"
"	popal					\n"
"	popl	%%ds				\n"
"	popl	%%es				\n"
"	popl	%%gs				\n"
"	mov	%%fs,%%ax			\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*error)
	: "a" (ax), "d" (filename), /* DS:DX */
	  "b" (env_to_set) /* ES:BX */
	: "memory", "cc");
  return carry;
  }

extern inline unsigned short
DOS_exec_cmd (const char *cmdline_CR)
  {
  unsigned short ax;
  asm volatile (
"	pushl	%%gs				\n"
"	pushl	%%es				\n"
"	pushl	%%ds				\n"
"	pushal					\n"
"	mov	%%ss,%%cs:savess		\n"
"	mov	%%sp,%%cs:savesp		\n"
"	int	$0x2E	# DOS_exec_cmd		\n"
"	mov	%%ax,%%fs			\n"
"	mov	$0x1234,%%ax			\n"
"savess = . - 2					\n"
"	mov	%%ax,%%ss			\n"
"	mov	$0x1234,%%sp			\n"
"savesp = . - 2					\n"
"	popal					\n"
"	popl	%%ds				\n"
"	popl	%%es				\n"
"	popl	%%gs				\n"
"	mov	%%fs,%%ax			\n"
	: "=a" (ax)
	: "S" (cmdline_CR) /* DS:SI */
	: "memory", "cc");
  return ax;
  }

/* Only use once per sub-process: */
extern inline unsigned short
DOS_get_return_code (void)
  {
  unsigned short ax;
  asm volatile ("  int	$0x21	# DOS_get_return_code " : "=a" (ax) : "a" (0x4D00));
  return ax;
  }

extern inline unsigned char
DOS_FindFirst (const char *pattern,
		struct DOS_attribute_str attribute,
		unsigned short *errorcode)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.b.xh = 0x4E;
  ax.w.b.xl = 1;	/* append change path */

  asm volatile (
"	int	$0x21 # DOS_FindFirst		\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*errorcode)
	: "a" (ax), "c" (attribute),
	  "d" (pattern),	/* in fact %ds:%dx */
	  ASM_STRUCT_INPUT (*pattern)
	: "memory", "cc");
  return carry;
  }

extern inline unsigned char
DOS_FindNext (unsigned short *errorcode)
  {
  unsigned char carry;
  Qreg ax;

  ax.w.b.xh = 0x4F;

  asm volatile (
"	int	$0x21 # DOS_FindNext		\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*errorcode)
	: "a" (ax)
	: "memory", "cc");
  return carry;
  }

extern inline unsigned short
DOS_GetCurrentPSPSegment (void)
  {
  Qreg ax, bx;

  ax.w.b.xh = 0x62;
  asm volatile (" int	$0x21 # DOS_GetCurrentPSPSegment "
	: "=b" (bx)
	: "a" (ax)
	);
  return bx.w.xw;
  }

/* CONTINUE with Qreg */



/*
 * Long Filename DOS Functions (Windows 95 minimum)
 * Functions available when IFSMgr is running,
 * not under bare MS-DOS 7
 */
struct LfnFindData_str {
    unsigned file_attribute;	/* LSB: std DOS */
    unsigned long long file_creation_time;
    unsigned long long file_access_time;
    unsigned long long file_modif_time;
    unsigned msb_file_size;
    unsigned lsb_file_size;
    unsigned reserved[2];
    unsigned char full_filename[260]; /* ASCIZ */
    unsigned char short_filename[14]; /* ASCIZ */
    } __attribute__ ((packed));

extern inline unsigned char
DOS_LFN_FindFirst (const char *pattern,
		struct DOS_attribute_str allowable,
		struct DOS_attribute_str required,
		struct LfnFindData_str *FindData,
		unsigned short *handle_errorcode)
  {
  unsigned char carry;
  unsigned short unicode_flags;
  struct {
      struct DOS_attribute_str allowable;
      struct DOS_attribute_str required;
      } __attribute__ ((packed)) attributes = {
      allowable: allowable,
      required: required
      };

  asm volatile (
"	stc					\n"
"	int	$0x21 # DOS_LFN_FindFirst	\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*handle_errorcode), "=c" (unicode_flags)
	: "a" ((unsigned short)0x714E),
	  "c" (attributes),
	  "S" (1), /* MSDOS date/time format */
	  "d" (pattern), /* in fact %ds:%dx, ASCIZ */
	  "D" (FindData) /* in fact %es:%di */
	: "memory", "cc");
  /* *handle_errorcode == 0x7100 if not supported */
  return carry;
  }

extern inline unsigned char
DOS_LFN_FindNext (unsigned short handle,
		struct LfnFindData_str *FindData,
		unsigned short *errorcode)
  {
  unsigned char carry;
  unsigned short unicode_flags;

  asm volatile (
"	stc					\n"
"	int	$0x21 # DOS_LFN_FindNext	\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*errorcode), "=c" (unicode_flags)
	: "a" ((unsigned short)0x714F), "b" (handle),
	  "S" (1), /* MSDOS date/time format */
	  "D" (FindData) /* in fact %es:%di */
	: "memory", "cc");
  return carry;
  }

extern inline unsigned char
DOS_LFN_FindClose (unsigned short handle, unsigned short *errorcode)
  {
  unsigned char carry;

  asm volatile (
"	stc					\n"
"	int	$0x21 # DOS_LFN_FindClose	\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*errorcode)
	: "a" ((unsigned short)0x71A1), "b" (handle)
	: "memory", "cc");
  return carry;
  }

/* DOS 5+ : ALLOCATE HMA SPACE only valid if DOS=HIGH */
extern inline farptr
DOS_allocate_hma_space (unsigned short nbbytes, unsigned short *total_alloc)
  {
  farptr addr;
  asm volatile (
"	pushl	%%es		\n"
"	int	$0x2F		\n"
"	pushw	%%es		\n"
"	pushw	%%di		\n"
"	popl	%%edi		\n"
"	popl	%%es		\n"
	: "=D" (addr), "=b" (*total_alloc)
	: "a" ((unsigned short)0x4A02), "b" (nbbytes)
	);
  return addr;
  }

extern inline unsigned DOS_loaded_HIGH (void)
  {
  unsigned short total_alloc;
  return DOS_allocate_hma_space (0, &total_alloc) != 0xFFFFFFFF
	 || total_alloc != 0;
  }

#if 0	// This has to be debugged - when it is not installed
struct SMARTDRV_status { /* 0xFF if drive do not exist */
    unsigned char	drive         : 6;
    unsigned char	write_through : 1; /* i.e. not write-cached */
    unsigned char	not_cached    : 1;
    unsigned char	abscent;	/* non zero if no cache present */
    } __attribute__ ((packed));

enum SMARTDRV_enum {
    SMARTDRV_get_info = 0,
    SMARTDRV_read_cache_on,
    SMARTDRV_read_cache_off,
    SMARTDRV_write_cache_on,
    SMARTDRV_write_cache_off
    };

extern inline struct SMARTDRV_status
SMARTDRV_control (unsigned short drive, enum SMARTDRV_enum command)
  {
  /* SMARTDRV v4.00+ and PC-Cache v8.0,
     drive = 0 for A: ... */
  unsigned short ax = 0x4A10;
  struct SMARTDRV_status returned;

  asm volatile (
"	pushl	%%ebp		\n"
"	movw	%%bx,%%bp	\n"
"	movw	$3,%%bx		\n"
"	int	$0x2F	# SMARTDRV_control	\n"
"	popl	%%ebp		\n"
	: "+a" (ax), "+b" (drive), "=d" (returned)
	: "d" (command)
	);
  if (ax != 0xBABE)
      return (struct SMARTDRV_status) { 0, 0, 0, 0xFF };
  return returned;
  }

extern inline void
SMARTDRV_flush_buffer (void)
  {
  /* SMARTDRV v4.00+ and PC-Cache v8.0, commit before flushing */
  asm volatile (" int	$0x2F	# SMARTDRV_flush_buffer "
	: : "a" ((unsigned short)0x4A10),
	    "b" ((unsigned short)1)
	);
  }
#endif

/* CD-ROM v2.00+ - DRIVE CHECK: */
extern inline unsigned char
_CDROM_iscdrom (unsigned short drive) /* drive = 0 for A: */
  {
  unsigned short is_mscdex, supported;

  asm volatile (
"	int	$0x2F	# _CDROM_iscdrom		\n"
	: "=a" (supported), "=b" (is_mscdex)
	: "a" ((unsigned short)0x150B), "c" (drive)
	);
  if (is_mscdex != 0xADAD)
      return 0;
  return supported; /* non-zero if supported */
  }

extern inline unsigned char
EMM_get_version (void)
  {
  unsigned short ax = 0x4600;

  asm ASMREAD (" int	$0x67 # EMM_get_version "
	: "+a" (ax)
	);
  if (ax >> 8 == 0)
      return ax & 0xFF;	/* BCD coded */
    else
      return 0;
  }

extern inline farptr
EMM386_check (void)
  {
  unsigned short ax = 0xFFA5, bx = 0, cx = 0;

  asm ASMREAD (" int	$0x67 # EMM386_check "
	: "+a" (ax), "+b" (bx), "+c" (cx)
	);
  if (ax == 0x845A || ax == 0x84A5)
      return ((unsigned)bx << 16) | cx;
    else
      return 0;
  }

/* bit0: active, bit1: auto, i.e.  00 : ON, 01 : OFF, 10 : AUTO */
extern inline unsigned char
EMM386_get_state (farptr entrypoint)
  {
  unsigned short ax = 0x0000;

#if 0
  asm ASMREAD (" lcallw *%1  # EMM386_entry_point, EMM386_get_state "
	: "+a" (ax)
	: "m" (entrypoint)
	);
#else
  asm ASMREAD (" lcallw *%a1  # EMM386_entry_point, EMM386_get_state "
	: "+a" (ax)
	: "pr" (&entrypoint)
	);
#endif
  return ax >> 8;
  }

extern inline unsigned char
EMM386_set_state (farptr entrypoint, unsigned char newstate)
  {
  unsigned short ax = 0x0100 | newstate;

  /* once disabled, never call the entry point again ! */
#if 0
  asm volatile (" lcallw *%1 # EMM386_entry_point, EMM386_set_state "
	: "+a" (ax)		/* modified? */
	: "m" (entrypoint)
	);
#else
  asm volatile (" lcallw *%a1  # EMM386_entry_point, EMM386_set_state "
	: "+a" (ax)
	: "pr" (&entrypoint)
	);
#endif
  return ax >> 8;
  }

/* returns major:minor or zero: */
/* MS Windows 3.00 is reported to "object violently" to this call */
extern inline unsigned short
_VCPI_installation_check (void)
  {
  unsigned short returned, status;

  asm volatile (
"	int	$0x67	# _VCPI_installation_check "
	: "=a" (status), "=b" (returned)
	: "a" (0xDE00)
	);
  if (status >> 8)
      return 0;
  return returned;
  }

extern inline unsigned
_VCPI_get_max_address (void)
  {
  unsigned short status;
  unsigned returned; /* physical address of highest 4K memory page */

  asm volatile (
"	int	$0x67	# _VCPI_get_max_address "
	: "=a" (status), "=d" (returned)
	: "a" (0xDE02)
	);
  if (status >> 8)
      return 0;
  return returned;
  }

extern inline unsigned
_VCPI_get_nb_free_pages (void)
  {
  unsigned short status;
  unsigned returned;

  asm volatile (
"	int	$0x67	# _VCPI_get_nb_free_pages "
	: "=a" (status), "=d" (returned)
	: "a" (0xDE03)
	);
  if (status >> 8)
      return 0;
  return returned;
  }

extern inline unsigned
_VCPI_alloc_4Kpage (void)
  {
  unsigned short status;
  unsigned returned;

  asm volatile (
"	int	$0x67	# _VCPI_alloc_4Kpage "
	: "=a" (status), "=d" (returned)
	: "a" (0xDE04)
	);
  if (status >> 8)
      return 0;
  return returned & ~0xFFF;
  }

extern inline unsigned
_VCPI_free_4Kpage (unsigned address)
  {
  unsigned short status;

  asm volatile (
"	int	$0x67	# _VCPI_free_4Kpage "
	: "=a" (status)
	: "a" (0xDE05), "d" (address & ~0xFFF)
	);
  return (status >> 8);
  }

/* page_number is logical_addr >> 12 */
extern inline unsigned
_VCPI_get_physical_address (unsigned page_number)
  {
  unsigned short status;
  unsigned returned;

  asm volatile (
"	int	$0x67	# _VCPI_get_physical_address "
	: "=a" (status), "=d" (returned)
	: "a" (0xDE06), "c" (page_number)
	);
  if (status >> 8)
      return 0;
  return returned;
  }

extern inline CR0_t
_VCPI_get_CR0 (void)
  {
  unsigned short status;
  CR0_t returned;

  asm volatile (
"	int	$0x67	# _VCPI_get_CR0 "
	: "=a" (status), "=b" (returned)
	: "a" (0xDE07)
	);
  if (status >> 8)
      return (CR0_t) {0};
  return returned;
  }

extern inline unsigned char
_VCPI_get_PM_interface (farptr page_table_4K,
			unsigned short *first_unused,
			seg_t descriptor[3],
			unsigned *entrypoint)
  {
  Qreg ax;

  ax.w.xw = 0xDE01;

  asm volatile (
"	pushl	%%es	\n"
"	pushl	%%edi	\n"
"	popw	%%di	\n"
"	popw	%%es	\n"
"	int	$0x67	# _VCPI_get_PM_interface \n"
"	popl	%%es	\n"
	: "+a" (ax), "=b" (*entrypoint), "=D" (*first_unused),
	  ASM_STRUCT_OUTPUT (descriptor[0])
	: "D" (page_table_4K),	/* %es:(%di) */
	  "S" (descriptor)	/* %ds:(%si) */
	);
  return ax.w.b.xh; /* zero if successful */
  }

struct _VCPI_switch_str {
    CR3_t		cr3;
    unsigned		gdtr_linadr;
    unsigned		idtr_linadr;
    unsigned short	ldtr;
    unsigned short	tr;
    unsigned		entry_adr;
    unsigned short	entry_seg;
    } __attribute__ ((packed));

/* You'd better call that interrupt disabled! */
extern inline void
_VCPI_switch_virtual_to_real (struct _VCPI_switch_str *vcpi_info)
  {
  extern char switch2realmode[];

  vcpi_info->entry_adr = (unsigned)switch2realmode;
  asm volatile (
"	pushal				\n"
"	movw	%%cs,%%bx		\n"
"	shll	$16,%%ebx		\n"
"	movw	%%gs,%%bx		\n"
"	movw	%%sp,%%bp		\n"
"	shll	$16,%%ebp		\n"
"	movw	%%ss,%%bp		\n"
"	int	$0x67			\n"
"	# _VCPI_switch_virtual_to_real	\n"
"	# not reached area		\n"
"switch2realmode:			\n"
"	mov	%%cr0,%%eax		\n"
"	and	$0x7FFFFFFF,%%eax	\n"
"	mov	%%eax,%%cr0		\n"
"	jmp 1f	\n 1:			\n"
"	xorl	%%edx,%%edx		\n"
"	mov	%%edx,%%cr3		\n"
"					\n"
"	movw	$0x18,%%dx		\n"
"	mov	%%dx,%%ds		\n"
"	mov	%%dx,%%es		\n"
"	mov	%%dx,%%fs		\n"
"	mov	%%dx,%%gs		\n"
"	mov	%%dx,%%ss		\n"
"	mov	%%cr0,%%eax		\n"
// should that be 0xFFFFFFFE here:
"	and	$0xFE,%%eax		\n"
"	mov	%%eax,%%cr0		\n"
"	jmp 1f	\n 1:			\n"
"					\n"
"	mov	%%bp,%%ss		\n"
"	mov	%%bp,%%ds		\n"
"	mov	%%bp,%%es		\n"
"	shrl	$16,%%ebp		\n"
"	movl	%%ebp,%%esp		\n"
"	movw	%%bx,%%gs		\n"
"	shrl    $16,%%ebx		\n"
"	pushl	%%ebx			\n"
"	pushl	$welcome_back		\n"
"	lretl				\n"
"welcome_back:				\n"
"	popal				\n"
	:
	: "a" (0xDE0C), "S" (farptr2linear (stack_adr(vcpi_info))), ASM_STRUCT_INPUT (*vcpi_info)
	);
  }

extern inline void
_VCPI_copypages (struct _VCPI_switch_str *vcpi_info, farptr entrypoint,
		farptr pagetable, farptr first_unused,
		unsigned dstlinadr, unsigned srclinadr, unsigned short nbpage)
  {
  extern char VCPI_copy16K[];
  pagetable_t apagetable = { .bit = {
      .present		= 1,
      .read_write	= 1,
      .user_super	= 1,
      .write_through	= 1,
      .cache_disabled	= 0,
      .accessed		= 0,
      .dirty		= 0,
      .reserved_0	= 0,
      .global_page	= 0,
      .available	= 0,
      .base_address	= dstlinadr >> 12
      }};
  unsigned page = (first_unused - pagetable) / sizeof (apagetable);
  unsigned cpt = nbpage;

  vcpi_info->entry_adr = (unsigned)VCPI_copy16K;

  while (cpt--) {
      pokel (first_unused, apagetable.all);
      first_unused += sizeof (apagetable);
      apagetable.bit.base_address++;
      }

  asm volatile (
"	cli					\n"
"	pushal					\n"
"	movw	%%cs,%%bx			\n"
"	shll	$16,%%ebx			\n"
"	movw	%%gs,%%bx			\n"
"	movw	%%sp,%%bp			\n"
"	shll	$16,%%ebp			\n"
"	movw	%%ss,%%bp			\n"
"	movw	$0xDE0C,%%ax			\n"
"	int	$0x67				\n"
"	# _VCPI_copy16Kandreturn		\n"
"	# not reached area			\n"
"VCPI_copy16K:					\n"
"	movw	$0x18,%%ax	# src & dst seg	\n"
"	mov	%%ax,%%es			\n"
"	mov	%%ax,%%ds	# unlimit at 0	\n"
"	movl	%%edi,%%esi			\n"
"	movl	%%ecx,%%edi			\n"
"	shrl	$16,%%edi	# page		\n"
"	shll	$12,%%edi	# *= 4096	\n"
"	movzwl	%%cx,%%ecx	# nbpage	\n"
"	shll	$12,%%ecx	# *= 4096	\n"
"	shrl	$2,%%ecx	# by 32 bits	\n"
"	cld					\n"
"	rep movsl %%ds:(%%esi),%%es:(%%edi)	\n"
"						\n"
"	movw	$0x20,%%ax			\n"
"	mov	%%ax,%%ss			\n"
"	movl	%%ebp,%%esp			\n"
"	shrl	$16,%%esp			\n"
"						\n"
"	movzwl	%%bx,%%eax			\n"
"	pushl	%%eax	# virtual86 gs		\n"
"	movzwl	%%bp,%%eax			\n"
"	pushl	%%eax	# virtual86 fs		\n"
"	pushl	%%eax	# virtual86 ds		\n"
"	pushl	%%eax	# virtual86 es		\n"
"	pushl	%%eax	# virtual86 ss		\n"
"	shrl	$16,%%ebp			\n"
"	pushl	%%ebp	# virtual86 esp		\n"
"	pushfl					\n"
"	shrl	$16,%%ebx			\n"
"	pushl	%%ebx	# virtual86 cs		\n"
"	pushl	$VCPI_copy16K_return		\n"
"	movw	$0xDE0C,%%ax			\n"
"	movl	$0x28,-4(%%esp)			\n"
"	movl	%%edx,-8(%%esp)			\n"
"	lcalll	*-8(%%esp)			\n"
"VCPI_copy16K_return:				\n"
"	popal					\n"
"	sti					\n"
	:
	: "S" (farptr2linear (stack_adr(vcpi_info))), "d" (entrypoint),
	  "D" (srclinadr), "c" ((page << 16) | nbpage), ASM_STRUCT_INPUT (*vcpi_info)
	);
  }

extern inline unsigned
get_window_version (unsigned short *window_version,
		    unsigned short *mode)
  {
  unsigned short command_error = 0x160A;

  asm volatile (
"	int	$0x2F	# get_window_version	\n"
	: "+a" (command_error), "=b" (*window_version),
	  "=c" (*mode)
	);

  return command_error == 0; /* true if no error */
  }

extern inline unsigned short
init_broadcast (unsigned short window_version, farptr *startup_info,
		farptr *virtual86_callback)
  {
  unsigned short error;

  asm volatile (
"	pushl	%%gs		\n"
"	pushl	%%ds		\n"
"	pushl	%%es		\n"
"	pushl	%%edx		\n"
"	pushl	%%eax		\n"	// safety
"	xorw	%%bx,%%bx	\n"
"	movw	%%bx,%%es	\n"
"	xorw	%%si,%%si	\n"
"	movw	%%si,%%ds	\n"
"	xorw	%%cx,%%cx	\n"
"	pushf			\n"
"	movw	$0,%%dx		\n"	// would be 1 for 286 DOS extender
"	int	$0x2F	# init_broadcast	\n"
"	popf			\n"
"	pushw	%%es		\n"
"	pushw	%%bx		\n"
"	popl	%%ebx		\n"
"	pushw	%%ds		\n"
"	pushw	%%si		\n"
"	popl	%%esi		\n"
"	popl	%%eax		\n"	// safety
"	popl	%%edx		\n"
"	popl	%%es		\n"
"	popl	%%ds		\n"
"	popl	%%gs		\n"
	: "=c" (error), "=b" (*startup_info), "=S" (*virtual86_callback)
	: "a" ((unsigned short)0x1605), "D" (window_version)
	);

  return error; /* 0 if OK to load */
  }

/* called interrupts disabled: */
extern inline char
virtual86 (farptr virtual86_callback, unsigned enableV86)
  {
  unsigned char carry;

  asm volatile (
"	pushl	%%ds	\n"
"	pushl	%%es	\n"
"	pushl	%%gs	\n"
"	pushal		\n"
"	lcallw	*%a2	# virtual86 \n"	// should not be 16(%esp) !
"	popal		\n"
"	popl	%%gs	\n"
"	popl	%%es	\n"
"	popl	%%ds	\n"
"	setc	%0	\n"
	: "=q" (carry)
	: "a" (enableV86), "r" (&virtual86_callback)
	);
  return carry;
  }

/* in real mode: */
extern inline void
exit_broadcast (void)
  {
  asm volatile (
"	pushl	%%ds		\n"
"	pushl	%%es		\n"
"	pushl	%%gs		\n"
"	pushal			\n"
"	pushf			\n"
"	movw	$0,%%dx		\n"	// would be 1 for 286 DOS extender
"	int	$0x2F	# exit_broadcast	\n"
"	popf			\n"
"	popal			\n"
"	popl	%%gs		\n"
"	popl	%%es		\n"
"	popl	%%ds		\n"
	: : "a" ((unsigned short)0x1606)
	);
  }

/* in protected mode: */
extern inline void
begin_exit_broadcast (void)
  {
  asm volatile (
"	int	$0x2F	# begin_exit_broadcast	\n"
	: : "a" ((unsigned short)0x1609)
	);
  }

/* Others: */
extern inline void
Win95_set_DOS_flag (unsigned on_off)
  {
  asm volatile (
"       int     $0x21   # Win95_set_DOS_flag(dl)  \n"
	: : "a" ((unsigned short)0x3307), "d" (!!on_off)
	);
  }

extern inline farptr
set_pgm_to_exec_on_exit (void)
  {
  unsigned short result, off;
  unsigned seg;

  asm volatile (
"	int	$0x2F	# set_pgm_to_exec_on_exit	\n"
	: "=a" (result), "=d" (seg), "=c" (off)
	: "a" ((unsigned short)0x4B20),
	  "d" ((unsigned short)0),
	  "c" ((unsigned short)0)
	);

  if (result == 0)
      return (seg << 16) + off;
  return 0; /* error */
  }

#endif /* DOS_H */
