/* disk.h */
#ifndef DISK_H
#define DISK_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"

/*
 * maximal number of disks, partition & freelist per disk (for boot2)
 * NB_DISK is set less or equal to 15 to not see the compaq bug
 *	(disk 0x80 = disk 0x90 = disk 0xA0 ...)
 * If you do not want the free list, just undefine NB_FREELIST.
 * Note that if NB_PARTITION is <= 4, extended partitions are not probed.
 * For mingujin.exe, we have DISK_SUPPORT == DOS_SUPPORT
 */
#if (DISK_SUPPORT & PROBE_ONLY_BIOSBOOTDEVICE) || (DISK_SUPPORT == DOS_SUPPORT)
#define NB_IDE			0
#define NB_DISK			1
#if DISK_SUPPORT & NO_PROBE_PARTITIONS
#define NB_PARTITION		1
#else
#define NB_PARTITION		4
#endif
#else
#define NB_IDE			10
#define NB_DISK			15
#define NB_PARTITION		64 /* 4 primary + 30 extended + 30 data */
#define NB_FREELIST		NB_PARTITION
#endif

/*
 * This is the number of sector which are reserved at the beginning of an
 * extended partition, for instance at the beginning of an extended MSDOS
 * partition, one sector is reserved for the partition description.
 * The "reserved" space is where we could put Gujin boot code.
 * Inside a DATA partition, like FATxx, some sectors are free but
 * it is variable for FAT32 (copy of boot block).
 */
#define RESERVEDSECTOR 1

/*
 * IDE only, to reduce the code size: (just saves 92 bytes)
 * We define that but keep in mind that the HPT 372 aditionnal IDE
 * breaks the 1 microsec/inb() on the secondary controller (only!);
 * I get:
Probing IDE 0xBC00,0xC002,10 master drive: calibrating calibrateIDEloop: 1000 loop
		executed in 1049793 cycles, i.e. 750 ns per loop
Probing IDE 0xBC00,0xC002,10 slave drive: calibrating calibrateIDEloop: 1000 loop
		executed in 1260138 cycles, i.e. 900 ns per loop
 * That is still more than 400ns, so it is OK for ATA drives.
 */
#define IDE_SIMPLE_400ns_WAIT

/*
 * IDE only, to reduce the code size: (just saves 140 bytes for me)
 * If you define that, the timings will be wrong on the second IDE channel
 * of the HPT 372 aditionnal IDE. I did not find (using following define)
 * another system where this is not at least true - but I did not test a
 * lot of system; please report if you have such a system, thanks!
 * NOTE that you are _writing_ to your disk with this code,
 * so that is _important_, more important than IDE_SIMPLE_400ns_WAIT !
 */
//#define IDE_SIMPLE_1us_per_inb

/*
 * If you want to debug the timings of the IDE subsystem:
 * (you need to have DEBUG_DISK to output the results, and
 * you may need to disable some support, like DOS support,
 * to fit in code size)
 */
#if DEBUG & DEBUG_DISK
//#define DEBUG_IDE_TIMING
#endif

/*
 * used in ide.h and the disk description struct disk_interface_str:
 */
struct config_multiword_dma_s {
    unsigned short  mode0    : 1;
    unsigned short  mode1    : 1; /* and below */
    unsigned short  mode2    : 1; /* and below */
    unsigned short  reserved : 13;
    } __attribute__ ((packed));

struct config_ultra_dma_s {
    unsigned short  mode0    : 1;
    unsigned short  mode1    : 1; /* and below */
    unsigned short  mode2    : 1; /* and below */
    unsigned short  mode3    : 1; /* and below */
    unsigned short  mode4    : 1; /* and below */
    unsigned short  mode5    : 1; /* and below */
    unsigned short  reserved : 10;
    } __attribute__ ((packed));

struct config_feature_s {
    unsigned short	smart_feature	: 1;
    unsigned short	smart_selftest	: 1;
    unsigned short	smart_errorlog	: 1;
    unsigned short	security	: 1;
    unsigned short	standby_powerup	: 1;
    unsigned short	RW_DMA_queued	: 1;
    unsigned short	acoustic	: 1;
    unsigned short	host_prot_area	: 1;
    unsigned short	lba48		: 1;
    unsigned short	reserved	: 7;
    } __attribute__ ((packed));

enum IDE_security_enum {
	IDE_security_unlock_user = 0,
	IDE_security_unlock_master = 1,
	IDE_security_set_pw_user = 2,
	IDE_security_set_pw_master = 3,
	IDE_security_disable_user = 4,
	IDE_security_disable_master = 5,
	IDE_security_erase_user = 6,
	IDE_security_erase_master = 7,
	};

enum sector_type_enum {	/* CDROM/DVD */
	all_sector_type			= 0, /* no check */
	CD_DA_sector_type		= 1, /* 2352 audio bytes/sectors */
	mode_1_sector_type		= 2, /* 2048 bytes/sectors */
	mode_2_formless_sector_type	= 3, /* 2336 bytes/sectors */
	mode_2_form_1_sector_type	= 4, /* 2048 bytes/sectors */
	mode_2_form_2_sector_type	= 5  /* 2324 + 4 bytes/sectors */
	};

/*
 * The disk subsystem for boot2 :
 *
 * Dirty if (access == dos_part), the "partition" pointer is used as
 * the DOS name pointer because nbpartition == 0 (then diskname is "DOS A:").
 */
struct diskparam_str;
extern struct disk_interface_str {
    unsigned		nbdisk;

    unsigned char	max_disk, max_partition, max_freelist, max_IDE_found;
    unsigned short	sizeof_diskparam_str,	sizeof_partition_str;
    unsigned short	sizeof_freelist_str,	sizeof_IDE_found_str;
    unsigned char	nb_bios_fd, nb_bios_hd, cannot_reset_floppy, nb_bios_blacklist;
    unsigned char	bios_blacklist[4];
    unsigned		reserved1[2];

    /* if (partition < 0), standard disk access based at lba == 0,
	else add partition base and check partition limit: */
    unsigned char	(*readsector) (struct diskparam_str *dp, int partition,
			unsigned long long lba, unsigned number, farptr buffer);
    unsigned char	(*writesector) (struct diskparam_str *dp, int partition,
			unsigned long long lba, unsigned number, farptr buffer);
    int			(*ideSecurity) (struct diskparam_str *dp, char password[32],
			enum IDE_security_enum action);
    unsigned long long	unused;

    struct diskparam_str {
	enum {	bios_chs, ebios_lba,
		hardide_chs, hardide_lba, hardide_lba48, hardide_atapi,
		dos_part
		} access : 8;
	unsigned char	disknb;	/* BIOS number or DOS letter
				   or 0=master, non zero=slave */
	unsigned char	biostype;	/* 1:floppy w/o change detect, 2: with, 3: HD */
	unsigned char	drivetype;	/* 1: 360K, 2: 1.2M, 3: 720K, 4: 1.44M, 6: 2.88M, 0x10: ATAPI */
	unsigned char	diskname[32];

	struct ide_attribute_str {
	    unsigned smart			: 1;
	    unsigned host_protected_area	: 1;
	    unsigned security			: 1;
	    unsigned lba48			: 1;
	    unsigned removable			: 1;
	    unsigned SAORAB			: 1;
	    unsigned config_overlay		: 1;
	    unsigned reserved			: 25;
	    } ide_attribute;
	struct error_log_str {
	    unsigned read_media			: 1;
	    unsigned write_media		: 1;
	    unsigned access_over_disk_tryed	: 1;
	    unsigned access_over_partition_tryed: 1;
	    unsigned no_ebios_fct		: 1;
	    unsigned ebios_size_zero		: 1;
	    unsigned chs_bios_part_mismatch	: 1;
	    unsigned chs_ajusted_bootsect	: 1;
	    unsigned diskheader_ignored		: 1;
	    unsigned disk_locked		: 1;
	    unsigned disk_was_pw_locked		: 1;
	    unsigned SMART_disabled		: 1;
	    unsigned SMART_failure		: 1;
	    unsigned analyse_partition_failed	: 1;
	    unsigned NB_PARTITION_exceded	: 1;
	    unsigned partition_overlap		: 1;
	    unsigned partition_over_limit	: 1;
	    unsigned beer_checksum_error	: 1;
	    unsigned beer_in_extended_partition	: 1;
	    unsigned read_media_retried		: 1;
	    unsigned ideReadConfig_failed	: 1;
	    unsigned reserved			: 11;
	    } error_log;
	unsigned short	bytepersector;
	unsigned short	nbhead;		  /* DOS: MediaDescriptorByte */
	unsigned	nbcylinder;       /* DOS: ClustersOnDisk */
	unsigned	BEER_size;
	unsigned long long	BEER_sector;
	unsigned long long	BEER_HostProtectedArea_start; /* or HPA found at init */
	unsigned long long	BEER_ReservedAreaBootCodeAddress;
	unsigned long long	BEER_disksize;
	  signed long long	fulldisk_offset;	/* Nb sectors before MBR */
	unsigned long long	fulldisk_size;		/* only if MBR is offseted */
	unsigned long long	nbtotalsector;
	unsigned long long	config_max_lba;
	unsigned short	nbsectorpertrack; /* DOS: SectorPerCluster */

	unsigned short	infobit;	/* Int13/48 extensions */
	unsigned short	ebios_version;	/* if present, used if access = ebios */
	ebios_bitmap_t	ebios_bitmap;
	unsigned short	reserved3;	/* ebios_bitmap is 16 bits */

	unsigned char  ebios_bustype[4];	/* exactly "ISA" or "PCI" */
	unsigned char  ebios_Interface[8];	/* exactly "ATA", "ATAPI", "SCSI", "USB"...*/
	union interface_path_u	ebios_bus;
	union device_path_u	ebios_device;

	struct security_status_str	security_status;
	unsigned short	ide_master_password_revision; /* if command_supported1.security */
	unsigned short	ideIOadr;	/* if known, else 0 */
	unsigned short	ideIOctrladr;	/* if known, else 0 */
	unsigned char	lba_slave_mask;	/* see Phenix Bios (head | mask) */
	unsigned char	irq;		/* if known, else 0 */
	unsigned char	multiplemode;	/* int 0x13/0x21,0x22,0x24 or hardide,
						set but not used. */
	unsigned char	BEER_device_index;
	struct config_multiword_dma_s	initial_multiDMA, maximum_multiDMA;
	struct config_ultra_dma_s	initial_ultraDMA, maximum_ultraDMA;
	struct config_feature_s		initial_feature, maximum_feature;
	unsigned	CalibrationIdeloop; /* nanoseconds (10^-9) per loop */

	/* Following three only for C/H/S partition, from standard BIOS: */
	unsigned short	bios_nbcylinder;	/* max 1024 */
	unsigned char	bios_maxhead;		/* max 255 */
	unsigned char	bios_nbsectorpertrack;	/* max 63 */

	unsigned char	bootable : 7;	/* nonzero: bootable, > 1 when special bootsector recognised */
	unsigned char	sigAA55  : 1;	/* has signature 0xAA55, can contain partition */
	unsigned char	OSdisknumber;	/* as written in the boot block */

	unsigned short	nbpartition;
	unsigned short	nbOSnumber;
	unsigned short	unused;
	/* Following value is current extended primary partition,
	   if there is more than one extended primary partition. */
	unsigned	first_ext_partition_start; /* NOT A LONG LONG */
	struct partition_str {
	    unsigned long long	start, length;
	    /* OSnumber should be equal to the # in /dev/hda#: */
	    unsigned char	type, OSnumber;
	    struct partition_misc_str {
		unsigned char order		: 6;
		unsigned char fsanalyse_toobig	: 1;
		unsigned char fsanalyse_error	: 1;
		unsigned char maybe_root	: 1;
		unsigned char fat_bootable	: 1;
		unsigned char beer_partition	: 1;
		unsigned char swap_partition	: 1;
		unsigned char conflicting_fsname	: 1;
		unsigned char reserved		: 1;
		enum part_state {
		    part_invalid = 0, part_active, part_inactive, part_extended
		    } active			: 2;
		} __attribute__ ((packed)) misc;
#define MAX_DISKNAMESIZE	(64-20-4) /* bigger than "floppy" */
	    unsigned char	name[MAX_DISKNAMESIZE];
#if KEEP_STRUCT_MAPPING || (DISK_SUPPORT & ISOFS_PROBE)
	    ISO9660_Default_entry *ElToritoCatalog; /* first one is ISO9660_Validation_entry */
#endif
	    } __attribute__ ((packed)) *partition;

#if KEEP_STRUCT_MAPPING || (defined (NB_FREELIST) && (NB_FREELIST != 0))
	unsigned	nbfreelist;
	struct freelist_str {
	    unsigned long long start, length;
	    } *freelist;
#endif

#if KEEP_STRUCT_MAPPING || (DISK_SUPPORT & IDE_SUPPORT)
	char set_max_password[32];
#endif
	} *param;

#if KEEP_STRUCT_MAPPING || (DISK_SUPPORT & (EBIOS_SUPPORT | IDE_SUPPORT))
    unsigned	nb_IDE_found;
    struct IDE_found_str {
	unsigned short	ideIOadr;
	unsigned short	ideIOctrladr;
	unsigned char	irq;
	unsigned char	bios_order;
	unsigned short	reserved;
	} *IDE_found;
#endif /* DISK_SUPPORT & (EBIOS_SUPPORT | IDE_SUPPORT) */
    } DI;

/*
 * Indirect functions:
 */
/* Do NOT call this function: */
__attribute__((weak)) void asm_dummy_disk (void)
  {
  awk_farfct (DI.readsector);
  awk_farfct (DI.writesector);
  awk_farfct (DI.ideSecurity);
  }

#ifdef IDE_ATAPI_POWER_MANAGEMENT
/* readsector called with lba = number = buffer == 0 manages the power state
	of the disk (some DVD-RAM needs manual management):
	partition == -2 -> set drive AUTO, if failed set drive to ACTIVE
	partition == -3 -> set drive ACTIVE
	partition == -4 -> set drive IDLE
	partition == -5 -> set drive STANDBY */
#define SET_DRIVE_AUTO_ACTIVE(dp)	DI.readsector(dp, -2, 0, 0, 0)
#define SET_DRIVE_ACTIVE(dp)		DI.readsector(dp, -3, 0, 0, 0)
#define SET_DRIVE_IDLE(dp)		DI.readsector(dp, -4, 0, 0, 0)
#define SET_DRIVE_STANDBY(dp)		DI.readsector(dp, -5, 0, 0, 0)
#else
#define SET_DRIVE_AUTO_ACTIVE(dp)	/* nothing */
#define SET_DRIVE_ACTIVE(dp)		/* nothing */
#define SET_DRIVE_IDLE(dp)		/* nothing */
#define SET_DRIVE_STANDBY(dp)		/* nothing */
#endif

/**
 ** The generic functions to translate CHS to/from LBA:
 **/
extern inline unsigned long long
chs2lba (unsigned cylinder, unsigned head, unsigned sector,
	 unsigned short nbhead, unsigned short nbsectorpertrack)
  {
  return (cylinder * nbhead + head) * nbsectorpertrack + sector - 1;
  }

extern inline void
lba2chs (unsigned long long lba,
	 unsigned *cylinder, unsigned *head, unsigned *sector,
	 unsigned short nbhead, unsigned short nbsectorpertrack)
  {
  unsigned cylinder_size = nbhead * nbsectorpertrack;

#if 0 /* called in dbg_check_chs2lba() with lba 32 bits and in DISK_RWsector() when lba < dp->nbtotalsector */
  unsigned char bcs = __builtin_ffs (cylinder_size) - 1, bst = __builtin_ffs (nbsectorpertrack) - 1;
  if (cylinder_size == (1 << bcs) && nbsectorpertrack == (1 << bst)) { /* disk size can be bigger is nbsectorpertrack = 32 and nb_head = 16 or nb_head = 256 */
      *cylinder = lba >> bcs;
      unsigned long long ltmp = lba & ((1 << bcs) - 1);
      *head = ltmp >> bst;
      *sector = (ltmp & ((1 << bst) - 1)) + 1;
      }
    else /* then there is a unhandled limit for the disk size... */
#endif
      {
      unsigned tmp;
      *cylinder = ull_div_ul (lba, cylinder_size, &tmp);
      *head = tmp / nbsectorpertrack;
      *sector = (tmp % nbsectorpertrack) + 1; /* 0 do not exists */
      }
  }

/*
 * The functions:
 */
void probedisk (void);
awk_farcall (probedisk);

#if DISK_SUPPORT & IDE_SUPPORT
struct freeze_IDE_s {
    unsigned HostProtectedArea_unset	: 1;
    unsigned disk_size_unset		: 1;
    unsigned unfrozen_password		: 1;
    unsigned unfrozen_config		: 1;
    unsigned unfrozen_hpa		: 1;
    unsigned unlocked_hpa		: 1;
    unsigned no_set_password_hpa	: 1;
    unsigned hidden_hpa			: 1;
    unsigned unhidden_password		: 1;
    unsigned reserved			: 23;
    } __attribute__ ((packed));

void freeze_IDE (struct freeze_IDE_s mask);
awk_farcall (freeze_IDE);

unsigned long long get_native_max_hpa (struct diskparam_str *dp);
awk_farcall (get_native_max_hpa);
#endif

#endif /* DISK_H */
