/* disk.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"
#if SETUP & XCODE_SEGMENT	/* before other includes */
#define CALLING_FROM_XCODESEG
#endif

#include "instboot.h"
#include "library.h"
#include "boot.h"
#include "debug.h"
#include "messages.h"
#include "bios.h"
#include "disk.h"
#include "util.h"	/* for UTIL.processor.calibrate_rdtsc */
#include "kbd.h"

//#define TINYDBG(format, ...) printf (format, ## __VA_ARGS__)
#define TINYDBG(format, ...) /* */

#if DISK_SUPPORT & DOS_SUPPORT
#include "dos.h"
#endif

#if DISK_SUPPORT & (IDE_SUPPORT|EBIOS_SUPPORT)
#include "ide.h"	/* _IDE_ISA_guess_altadr() */
#endif

#if USER_SUPPORT != 0
#include "user.h"
#endif

/*
 * To define when GCC-3.4.3 will stop doing multiple "move $0,300(%esp)"
 * to clear a structure in the stack when optimising for size...
 */
//#define GCC_OPTIMISED_MEMSET

/*
 * When I do not know which error to report:
 */
#define LOCALERROR 0x99

/*
 * When the error may have cleared next time:
 */
#define ERROR_PLEASE_RETRY	0xFA

/*
 * This is used to update DI.nb_bios_fd and DI.nb_bios_hd, it should
 * be defined even if we do not have BIOS_SUPPORT. We are not
 * expecting an exception because it calls the motherboard BIOS
 * (called with parameter 0 and 0x80) which is usually cleaner.
 */
DISK_FCT_PREFIX(probe_nbBIOS) static inline unsigned
probe_nbBIOS (unsigned BIOSdisk)
  {
  unsigned char CmosDriveType, status;
  unsigned short tmpcx, tmpdx;
  farptr DBTadr;

  /*
   * Unsafe because we ask to the manager of the _first_ disk
   * how many disk there is in the system - think about
   * removable disks with added drivers:
   */
  if (_BIOSDISK_getparam (BIOSdisk, &tmpcx, &tmpdx, &DBTadr, &CmosDriveType, &status))
      return 0;
    else
      return tmpdx & 0xFF;
  }

#if DISK_SUPPORT & (BIOS_SUPPORT|EBIOS_SUPPORT)
#if defined (DEBUG) && (DEBUG & DEBUG_DISK)
DISK_FCT_PREFIX(BIOSDISK_error) static void
BIOSDISK_error (unsigned char drive, unsigned char status)
  {
  if (status == 0) {
      if (_BIOSDISK_getstatus (drive, &status) && status == 0)
	  status = LOCALERROR;
      DDBG (("_BIOSDISK_getstatus on 0x%X returns 0x%X: ", drive, status));
      }

#define BIOSERRORCODELIST() \
      P (0x00,	"")					\
      P (0x01,	"invalid function or parameter")	\
      P (0x02,	"bad sector")				\
      P (0x03,	"write protect")			\
      P (0x04,	"sector not found")			\
      P (0x05,	"disk reset failed")			\
      P (0x06,	"disk changed or removed")		\
      P (0x07,	"bad disk parameter table")		\
      P (0x08,	"DMA overrun")				\
      P (0x09,	"DMA cross 64K boundary")		\
      P (0x0A,	"bad disk sector flag")			\
      P (0x0B,	"bad disk track")			\
      P (0x0C,	"unsupported track")			\
      P (0x0D,	"invalid sector nb")			\
      P (0x0E,	"address mark detected")		\
      P (0x0F,	"DMA arbitration problem")		\
      P (0x10,	"ECC/CRC error for read")		\
      P (0x11,	"ECC recovered data read")		\
      P (0x20,	"controller failure")			\
      P (0x31,	"no media in drive")			\
      P (0x32,	"invalid CMOS drive type")		\
      P (0x40,	"seek failure")				\
      P (0x80,	"timeout, drive not ready")		\
      P (0xAA,	"disk drive not ready")			\
      P (0xB0,	"volume not locked in drive")		\
      P (0xB1,	"volume locked in drive")		\
      P (0xB2,	"volume not removable")			\
      P (0xB3,	"volume in use")			\
      P (0xB4,	"volume lock count exceeded")		\
      P (0xB5,	"eject request failed")			\
      P (0xB6,	"volume present but Read Only")		\
      P (0xBB,	"disk undefined error")			\
      P (0xCC,	"disk write fault")			\
      P (0xE0,	"disk status error")			\
      P (0xFF,	"sense operation failed")		\
      P (LOCALERROR,	"local software error")

#if 1 // lets save space in .rodata

  {
  static const char array_val[] = {
#define P(val, msg)	val,
	BIOSERRORCODELIST()
#undef P
      };
  static const char *const array_msg[] = {
#define P(val, msg)	msg,
	BIOSERRORCODELIST()
#undef P
      };
  const char *ptr = "unknown error";
  unsigned i;

  for (i = 0; i < nbof (array_val); i++)
      if (array_val[i] == status) {
	  ptr = array_msg[i];
	  break;
	  }

  DDBGVAR ((ptr));
  DDBG (("\r\n"));
  }

#else

  {
  char *ptr = "unknown error";

  switch (status) {
#define P(val, msg)	case val: ptr = msg; break;
	BIOSERRORCODELIST()
#undef P
      }
  DDBG ((ptr));
  DDBG (("\r\n"));
  }

#endif

  }
#else
#define BIOSDISK_error(drive, status)	/* */
#endif /* DEBUG */
#endif /* BIOS_SUPPORT|EBIOS_SUPPORT */

/**
 ** The low level BIOS functions:
 **/
#if DISK_SUPPORT & BIOS_SUPPORT

DISK_FCT_PREFIX(BIOSDISK_gettype) static inline unsigned char
BIOSDISK_gettype (unsigned char disk, unsigned char *biostype, unsigned *nbsect)
  {
  unsigned char status;

  DDBG (("%s: ", __FUNCTION__));
#if defined (TREAT_EXCEPTION)
       /* Unfortunately, the Adaptec SCSI BIOS I have is seriously buggy:
	* it crash with an invalid assembler instruction on the BIOSDISK_gettype
	* BIOS call.
	* Reference: Adaptec AHA-2940U2W SCSI BIOS v2.20.0 (c) 1999 Adaptec, Inc
	* General Protection at address 0xCC000F79 invalid code: 0x58 0xC3 0xC8 0x02 0 0
	* INT 0x13 address in BIOS environment: 0xCC0012B0
	* 1 IDE disk (EBIOS) master on 1st IDE, 1 CDROM slave on 2nd IDE, 1 SCSI disk.
	* The owners of these card are strongly advised to upgrade
	* the BIOS of their card to the latest version, even if
	* they are using Windows and everything works fine (because
	* windows does not use the BIOS).
	*/
  if (shit_handler (1, 0) != 0) {
      DDBG ((" EXCEPTION RECEIVED!\r\n"));
      if (DI.nb_bios_blacklist < sizeof (DI.bios_blacklist) - 1)
	  DI.bios_blacklist[DI.nb_bios_blacklist++] = disk;
      return 0;	/* no disk here */
      }
#endif

  if (_BIOSDISK_gettype (disk, nbsect, &status)) {
      BIOSDISK_error (disk, status);
#if defined (TREAT_EXCEPTION)
      shit_handler (0, 0);
#endif
      return 0;	/* no disk here */
      }

#if defined (TREAT_EXCEPTION)
  shit_handler (0, 0);
#endif

  DDBG (("returns 0x%X, i.e. %s",
		status,
		(status == 1)? "floppy w/o change detect" :
		(status == 2)? "floppy with change detect" :
		(status == 3)? "hard disk" : "unknown\r\n"));
  if (status >= 1 && status <= 3) {
      *biostype = *nbsect;
      if (*nbsect != 0 && *nbsect != disk) /* but buggy Award 486 BIOS && HTP bios! */
	  DDBG ((", probably nb sector = 0x%X i.e. %U", *nbsect, *nbsect));
      DDBG (("\r\n"));
      }

  return status; /* return non-zero if present */
  }

DISK_FCT_PREFIX(getbytepersector) static inline unsigned short
getbytepersector (farptr DBTadr)
{
  if (DBTadr > 256*4) { /* check that it doesn't point in interrupt table ! */
      unsigned char bps = peekb (DBTadr + offsetof (struct DBT_str, bytepersector));
//      if (bps <= 3) {	// : that is the official limit.
      if (bps < 10) {	// : let's authorise it up to (128 << 9) = 64 Kb.
	  DDBG (("[%u bytes/sect (real)] ", 128 << bps));
	  return 128 << bps;
	  }
	else {
	  DDBG (("[512 bytes/sect (bps too big: %u)] ", bps));
	  return 512;
	  }
      }
    else {
      DDBG (("[512 bytes/sect (DBTadr invalid)] "));
      return 512;
      }
}

DISK_FCT_PREFIX(BIOSDISK_getparam) static inline unsigned char
BIOSDISK_getparam (unsigned char disk, struct diskparam_str *dp)
  {
  unsigned char status;
  unsigned short tmpcx, tmpdx;
  farptr DBTadr;

  DDBG (("%s: ", __FUNCTION__));

#if defined (TREAT_EXCEPTION)
  if (shit_handler (1, 0) != 0) {	/* never seen that, just in case */
      DDBG ((" EXCEPTION RECEIVED!\r\n"));
      if (DI.nb_bios_blacklist < sizeof (DI.bios_blacklist) - 1)
	  DI.bios_blacklist[DI.nb_bios_blacklist++] = disk;
      return 1;
      }
#endif

  if (_BIOSDISK_getparam (disk, &tmpcx, &tmpdx, &DBTadr, &dp->drivetype, &status)) {
      DDBG (("error. "));
#if defined (TREAT_EXCEPTION)
      shit_handler (0, 0);
#endif
      return 1;
      }

#if defined (TREAT_EXCEPTION)
  shit_handler (0, 0);
#endif

  if (BOOT1_DOS_running() && DBTadr == 0 && tmpcx == 0xC2C4 && tmpdx == 0x0101) {
	/* Windows in the emulation DOS box: C/H/S=963/2/4, DBT adr = 0
	   but cannot be accessed by BIOS, only DOS interface. */
      DDBG (("error, recognised window DOS box. "));
      return 1;
      }

  /* do not believe number of disks */
  dp->nbhead = (tmpdx >> 8) + 1;
  dp->nbsectorpertrack = tmpcx & 0x3F;
  dp->nbcylinder = (tmpcx >> 8) + ((tmpcx << 2) & 0x0300) + 1;
  dp->bytepersector = getbytepersector (DBTadr);

#if defined (DEBUG) && (DEBUG & DEBUG_DISK)
  DDBG (("reports nbdisk: %u, CMOS drive type: %u, DBT adr: 0x%X, DBT: { ",
	tmpdx & 0xFF, dp->drivetype, DBTadr));
  if (DBTadr) {
      unsigned i = sizeof (struct DBT_str);
      while (i--)
	  DDBG (("0x%X, ", peekb (DBTadr++)));
      }
  DDBG (("}, nbhead: %u, nbsectorpertrack: %u, nbcylinder: %u;\r\n",
	dp->nbhead, dp->nbsectorpertrack, dp->nbcylinder));
#endif

  return 0;
  }

#if !(DISK_SUPPORT & NO_FLOPPY_SUPPORT)
/* variable size array: not declared in inline function:
static inline */ DISK_FCT_PREFIX(biosFD_RWsector) static unsigned char
biosFD_RWsector (struct diskparam_str *dp,
		 unsigned char	drive,
		 unsigned short	cylinder,
		 unsigned char	head,
		 unsigned char	sector,
		 unsigned char	number,
		 farptr		buffer_fptr,
		 const unsigned	read)
  {
  /* This set segment bit 4..11 to zero, each page is 64Kb: */
  farptr buffer = linear2farptr (farptr2linear (buffer_fptr));
  DDBG ((" <FD BIOS %s %u sector starting at %u/%u/%u at 0x%X, adjusted address 0x%X> ",
	read? "reading" : "writing", number, cylinder, head, sector, buffer_fptr, buffer));

  while (number != 0) {
      /* the buffers may not be aligned on 512 bytes boundaries */
      /* one of "buffer" and "tmp_buffer" is not crossing a DMA fontier
	because the two fit in the 64 Kb stack: */
      unsigned char tmp_buffer[dp->bytepersector];

      /* Three tries, because if the floppy disk has been changed, first
	 function call may return "disk changed" (after resetting BIOS),
	 and the motor of the floppy drive may be slow to start. */
      unsigned max = dp->nbsectorpertrack - sector + 1;
      unsigned char maxtry = 3, ret, partial = number;

      if (partial > 18) {	/* 1.68M floppy */
	  partial = 18;
	  DDBG ((", splitting to %u due to 18 sectors frontier, ", partial));
	  }

      if (partial > max) {
	  partial = max;
	  DDBG ((", splitting to %u due to HEAD frontier", partial));
	  }

      max = (0x10000 - (buffer & 0xFFFF)) / dp->bytepersector;
      if (partial > max) {
	  partial = max;
	  DDBG ((", splitting to %u due to DMA frontier", partial));
	  }

      if (partial != 0)
	  DDBG ((", using direct: "));
	else { /* DMA inside the 512 bytes buffer */
	  DDBG ((", using indirect: "));
#ifdef WRITE_ENABLE
	  if (!read)
	      lmemcpy (tmp_buffer, smallestoffset (buffer),
			sizeof (tmp_buffer));
#endif /* WRITE_ENABLE */
	  }

      do {
	  unsigned char nbprocessed;
	  ret = _BIOSDISK_RWsector (drive, cylinder, head, sector,
	      partial? partial : 1, partial? buffer: stack_adr(tmp_buffer),
#ifdef WRITE_ENABLE
	      read,
#else /* WRITE_ENABLE */
	      1,
#endif /* WRITE_ENABLE */
	      &nbprocessed
	      );
	  if (nbprocessed != (partial? partial : 1))
	      DBG((" [nbsector successfully read/written %u] ", nbprocessed));
	  if (ret == 0x11) {
	      DDBG ((" <ECC recovered data read> "));
	      ret = 0;
	      }
	  } while (   ret != 0    /* successful completion */
		   && ret != 0x80 && ret != 0xAA
		   && ret != 0x03
		   && --maxtry);

      if (ret) {
	  DDBG (("error 0x%X] ", ret));
	  return ret;
	  }
	else
	  DDBG (("OK, "));

      if (partial == 0) {
#ifdef WRITE_ENABLE
	  if (read)
#endif /* WRITE_ENABLE */
	      smemcpy (smallestoffset (buffer), tmp_buffer,
			sizeof (tmp_buffer));
	  partial = 1;
	  }

      number -= partial;
      buffer += partial * dp->bytepersector;
      if (buffer & ~0xF000FFFFU) {
	  unsigned tmp = (buffer >> 16) & 0xFFF;
	  buffer &= 0xF000FFFFU;
	  buffer += tmp << 28;
	  }
      if ((unsigned short)sector + partial <= dp->nbsectorpertrack)
	  sector += partial;
	else {
	  sector = 1;
	  if (++head >= dp->nbhead) {
	      head = 0;
	      cylinder++;
	      }
	  }
      }
  return 0;
  }
#endif /* NO_FLOPPY_SUPPORT */

#endif /* BIOS_SUPPORT */

/**
 ** The low level extended BIOS functions:
 **/
#if DISK_SUPPORT & EBIOS_SUPPORT
DISK_FCT_PREFIX(EBIOS_analyse_phoenix) static inline void
EBIOS_analyse_phoenix (struct diskparam_str *dp, farptr configptr)
  {
  int i;
  ebiosPhoenix_t ebiosPhoenix;
  unsigned char checksum = 0, *ptr = (unsigned char *)&ebiosPhoenix;

  for (i = 0; i <= 14; i++)
      checksum -= *ptr++ = peekb(configptr + i);
  *ptr++ = peekb(configptr + 15);
  *ptr++ = peekb(configptr + 16);

  if (checksum == ebiosPhoenix.checksum) {
      DDBG ((" Phoenix Bios compatible, ideIOadr 0x%X,0x%X,%u, "
		"multiple %u, lba_slave_mask = 0x%X",
		ebiosPhoenix.dataIOadr,
		ebiosPhoenix.controlIOadr,
		ebiosPhoenix.irq,
		ebiosPhoenix.multisectorsize,
		ebiosPhoenix.lba_slave_mask));

      dp->ideIOadr = ebiosPhoenix.dataIOadr;
      dp->ideIOctrladr = ebiosPhoenix.controlIOadr;
      dp->irq = ebiosPhoenix.irq;
      dp->lba_slave_mask = ebiosPhoenix.lba_slave_mask;
      dp->multiplemode = ebiosPhoenix.multisectorsize;
      }
    else {
      DDBG ((" Non Phoenix Bios compatible."));
// done in diskcheck():
//      dp->ideIOadr = 0;
//      dp->ideIOctrladr = 0;
//      dp->irq = 0;
//      dp->lba_slave_mask = 0;
//      dp->multiplemode = 0;
      }
  }

DISK_FCT_PREFIX(EBIOS_analyse_EDD3) static inline void
EBIOS_analyse_EDD3 (struct diskparam_str *dp, ebiosinfo_t *ebiosinfo)
  {
  unsigned char checksum = 0, *ptr = (unsigned char *)&ebiosinfo->signature0xBEDD;

  while (ptr <= (unsigned char *)&ebiosinfo->bytechecksum)
      checksum += *ptr++;

  DDBG (("V3.0 check: signature0xBEDD = 0x%X, chksum 0x%X (==0?), length 0x%X (==0x24?)",
		ebiosinfo->signature0xBEDD, checksum, ebiosinfo->lendevicepath));
  if (   (ebiosinfo->signature0xBEDD == 0xBEDD || ebiosinfo->signature0xBEDD == 0xDDBE)
      && checksum == 0) {
      unsigned i;

      i = nbof(ebiosinfo->bustype);
      while (i--)
	  dp->ebios_bustype[i] = ebiosinfo->bustype[i];

      i = nbof(ebiosinfo->Interface);
      while (i--)
	  dp->ebios_Interface[i] = ebiosinfo->Interface[i];

      DDBG ((": info: %s disk on %s bus, ", dp->ebios_Interface, dp->ebios_bustype));
      dp->ebios_bus = ebiosinfo->bus;
      dp->ebios_device = ebiosinfo->device;

#if DEBUG & DEBUG_DISK /* Disable to save space when DEBUG_DISK? */
      if (STRBEGINEQL(ebiosinfo->bustype, "ISA"))
	  DDBG (("addr 0x%X, pad 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, ",
		ebiosinfo->bus.isa.addr,
		ebiosinfo->bus.isa.pad[0],
		ebiosinfo->bus.isa.pad[1],
		ebiosinfo->bus.isa.pad[2],
		ebiosinfo->bus.isa.pad[3],
		ebiosinfo->bus.isa.pad[4],
		ebiosinfo->bus.isa.pad[5]));
	else /* if (STRBEGINEQL(ebiosinfo->bustype, "PCI")) */
	  DDBG (("bus 0x%X, device 0x%X, function 0x%X, pad 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, ",
		ebiosinfo->bus.pci.busnr,
		ebiosinfo->bus.pci.devicenr,
		ebiosinfo->bus.pci.functionnr,
		ebiosinfo->bus.pci.pad[0],
		ebiosinfo->bus.pci.pad[1],
		ebiosinfo->bus.pci.pad[2],
		ebiosinfo->bus.pci.pad[3],
		ebiosinfo->bus.pci.pad[4]));
      if (STRBEGINEQL(ebiosinfo->Interface, "ATAPI"))
	  DDBG (("is_slave %u, logical_unit_nr 0x%X, ", ebiosinfo->device.atapi.slave, ebiosinfo->device.atapi.logical_unit_nr));
	else if (STRBEGINEQL(ebiosinfo->Interface, "ATA") || STRBEGINEQL(ebiosinfo->Interface, "SATA"))
	  DDBG (("is_slave %u, ", ebiosinfo->device.ata.slave));
      DDBG (("WorldWideNumber 0x%llX:%llX, ",
		dp->ebios_device.FibreChannel.WorldWideNumber, dp->ebios_device.FibreChannel.reserved));
#endif
      }
  }

DISK_FCT_PREFIX(EBIOSDISK_getparam) static inline unsigned char
EBIOSDISK_getparam (unsigned char disk, struct diskparam_str *dp)
  {
  ebiosinfo_t ebiosinfo;
  unsigned char status, nbtry = 1;

  for (;;) {
//      ebiosinfo.infobit = 0; /* Dell machines & PhoenixBIOS 4.0 Release 6.0 */
#ifdef GCC_OPTIMISED_MEMSET
      ebiosinfo = (ebiosinfo_t){}; // clears EDD 3.0
#else
      _memset (&ebiosinfo, 0, sizeof(ebiosinfo));
#endif

      ebiosinfo.buffersize = sizeof (ebiosinfo);

      /*
       * Note RedHat 5.2, DOSEMU, extended get geometry:
       * real extended BIOS returns C/H/S not translated for BIOS, i.e. head=16
       * DOSEMU extended BIOS returns C/H/S with head = 255
       * This geometry is anyway not used in this bootloader.
       */

      if (_EBIOSDISK_getparam (disk, &ebiosinfo, &status) || status != 0) {
	  DDBG (("EBIOS getparam error, status 0x%X.\r\n", status));
	  return LOCALERROR;
	  }
      if (ebiosinfo.buffersize == 0x1E && ebiosinfo.nbtotalsector == 0) {
	  /* A Compact Flash with a direct IDE -> Compact Flash adapter (no electronic compoment): */
	  /* I am using an ADA CompactFlash-IDE40 Rev 1.0 adapter from ES&S www.esskabel.de */
	  DDBG (("EBIOS (buffersize before 0x%X after 0x%X) and ebiosinfo.nbtotalsector == 0, "
		"set ebiosinfo.nbtotalsector = nbcylinder * nbhead * nbsectorpertrack = %u * %u * %u = %u\r\n",
		sizeof (ebiosinfo), ebiosinfo.buffersize,
		ebiosinfo.nbcylinder, ebiosinfo.nbhead, ebiosinfo.nbsectorpertrack,
		ebiosinfo.nbcylinder * ebiosinfo.nbhead * ebiosinfo.nbsectorpertrack));
	  ebiosinfo.nbtotalsector = ebiosinfo.nbcylinder * ebiosinfo.nbhead * ebiosinfo.nbsectorpertrack;
	  }
      if (ebiosinfo.nbtotalsector != 0 || --nbtry != 0)
	  break;
      /*	buggy EBIOS emulation of Win98 USB drives?, asking twice gives:
	Hard disk 0x82: BIOSDISK_gettype: returns 0x3, i.e. hard disk, probably nb sector = 0x79800 i.e. 497664
	EBIOS detected: version 0x1, extension 0x93, features(0x3): extended_fct removable
	EBIOS (buffersize before 0x4A after 0x4A) reports: nbhead: 0, nbsectorpertrack: 0, nbcylinder: 0, bytepersector: 0, nbtotalsector: 0X0 (i.e. 0), infobit 0x1D
	V3.0 check: signature0xBEDD = 0x0, chksum 0x0 (==0?), length 0x0 (==0x24?)
	EBIOS extension data at 0x0
	 [BIOS getparam for C/H/S partitions: 243/64/32] ; Invalid drive (C/H/S) and unknown drive size!
		Then:
	Hard disk 0x82: BIOSDISK_gettype: returns 0x3, i.e. hard disk, probably nb sector = 0x79800 i.e. 497664
	EBIOS detected: version 0x1, extension 0x93, features(0x3): extended_fct removable
	EBIOS (buffersize before 0x4A after 0x4A) reports: nbhead: 64, nbsectorpertrack: 32, nbcylinder: 243, bytepersector: 512, nbtotalsector: 0X79800 (i.e. 497664), infobit 0x1F
	V3.0 check: signature0xBEDD = 0x0, chksum 0x0 (==0?), length 0x0 (==0x24?)
	EBIOS extension data at 0x0
	 [BIOS getparam for C/H/S partitions: 243/64/32]  [use EBIOS nbtotalsector; usual EBIOS total == C*H*S; UNUSUAL BIOS total + H*S = 499712 != EBIOS total = 0X79800 i.e. 497664 (EBIOS_total - BIOS_total)/H/S = 0]
		So we re-ask...
	*/
      if (UTIL.CDROM_HDFD_emulation_state == 1 && UTIL.CDROM_HDFD_emulation_spec.bootdrive != 0
		&& UTIL.CDROM_HDFD_emulation_spec.bootdrive == disk) {
	  unsigned nb_cyl = UTIL.CDROM_HDFD_emulation_spec.lsbnbcyl + ((UTIL.CDROM_HDFD_emulation_spec.nbsector_msbnbcyl & 0xC0) << 2);
	  unsigned nb_heads = UTIL.CDROM_HDFD_emulation_spec.nbhead;
	  unsigned nb_sect = UTIL.CDROM_HDFD_emulation_spec.nbsector_msbnbcyl & 0x3F;
	  ebiosinfo.nbtotalsector = nb_cyl * nb_heads * nb_sect;
	  DDBG(("EBIOS size null, booted El-Torito with C/H/S= %u/%u/%u, so disk size %llu\r\n", nb_cyl, nb_heads, nb_sect, ebiosinfo.nbtotalsector));
	  if (ebiosinfo.nbtotalsector == 0) {
	      DDBG(("EBIOS size still null, assuming 8 Gb = 4194304 sectors\r\n"));
	      ebiosinfo.nbtotalsector = 4194304;
	      }
	  break;
	  }
      DDBG ((" [Retry _EBIOSDISK_getparam due to ebiosinfo.nbtotalsector==0] "));
      }

  DDBG (("EBIOS (buffersize before 0x%X after 0x%X) reports: nbhead: %u, nbsectorpertrack: %u"
	     ", nbcylinder: %u, bytepersector: %u, nbtotalsector: %llU, infobit 0x%X\r\n",
		sizeof (ebiosinfo), ebiosinfo.buffersize, ebiosinfo.nbhead, ebiosinfo.nbsectorpertrack,
		ebiosinfo.nbcylinder, ebiosinfo.bytepersector, ebiosinfo.nbtotalsector, ebiosinfo.infobit));
  /* HPT370/372 have a buggy BIOS... :
	EBIOS reports: nbhead: 12, nbsectorpertrack: 35, nbcylinder: 989, bytepersector: 512, nbtotalsector: 4294967295
	ideIdentify reports: C/H/S=989/12/35, RWmult=8, trans CHS invalid: C/H/S=0/0/0, total= 0, BigDisk total: 0
    Note that this isn't butyfull neither:
	EBIOS reports: nbhead: 15, nbsectorpertrack: 17, nbcylinder: 1023, bytepersector: 512, nbtotalsector: 4211146754
	ideIdentify reports: C/H/S=1023/15/17, RWmult=16, trans CHS valid: C/H/S=1023/15/17, total= 4211146755, BigDisk total: 0
		no lba on a > 1 Gb disk, swap size: 0xFB010003 -> 0x3FB01, i.e. capacity 260865 sectors
    Note that it does not seems to work very well neither with this, because
    this HTP bios removes 1 sector (for its own use?) but on the latter case,
    it removes 1 from the MSB so you have warnings like "partition over disk limit"...
    Then there is Intel Bios which does work correctly:
    EBIOS (buffersize before 0x4A after 0x1A) reports: nbhead: 0, nbsectorpertrack: 0, nbcylinder: 0, bytepersector: 512, nbtotalsector: 4,030,464, infobit 0x1
    WARNING: nbtotalsector = 0x3D8000, correcting it to 0x8001003D!!!
    */
  if ((unsigned)ebiosinfo.nbtotalsector == 0xFFFFFFFF) {
      unsigned tmp = ebiosinfo.nbcylinder;
      tmp *= ebiosinfo.nbhead;
      tmp *= ebiosinfo.nbsectorpertrack;
      DDBG (("WARNING: nbtotalsector = 0x%llX, correcting to %u!!!\r\n", ebiosinfo.nbtotalsector, tmp));
      ebiosinfo.nbtotalsector = tmp;
      }
    else if (   ebiosinfo.nbsectorpertrack && ebiosinfo.nbsectorpertrack < 63
	     && ebiosinfo.nbhead && ebiosinfo.nbhead <= 16
	     && ((unsigned)ebiosinfo.nbtotalsector & 0x0FE00000) != 0
	     && ((unsigned)ebiosinfo.nbtotalsector & 0x0000FFE0) == 0) {
      /* They also removed one to the wrong side, usually: */
      unsigned swapped = ebiosinfo.nbtotalsector & 0xFFFF;
      swapped += 1;
      swapped <<= 16;
      swapped |= (unsigned)ebiosinfo.nbtotalsector >> 16;
      DDBG (("WARNING: nbtotalsector = 0x%llX, correcting it to 0x%X!!!\r\n", ebiosinfo.nbtotalsector, swapped));
      ebiosinfo.nbtotalsector = swapped;
      }

  if (ebiosinfo.buffersize > offsetof (ebiosinfo_t, signature0xBEDD))
      EBIOS_analyse_EDD3 (dp, &ebiosinfo);
  /* Note: I have seen an EBIOS reserving space for itself at end of HD,
  once (IDE)16514064 - (EBIOS)16498755 = 15309 == 63 * 243 and on
  another drive 14931 = 237 * 63 sectors so BEER on these EBIOS will not
  be found - unless the real disk size if given in the MBR with 0x29 signature.

  ideIdentify irq 14:[input 28 1 sectors]  [28 cmd 0xEC drivehead 0xA0]  [readback= 0x0] ideIdentify reports: C/H/S=16383/16/63, RWmult=16, trans CHS valid: C/H/S=16383/16/63, total= 16514064, BigDisk total: 16514064 i.e. 0xFBFC10, BigBigLBA48total: 0X0
  Hard disk 0x80: BIOSDISK_gettype: returns 0x3, i.e. hard disk, probably nb sector = 0xFB8182 i.e. 16482690
  EBIOS detected: version 0x21, extension 0x99, features(0x5): extended_fct enhanced
  EBIOS (buffersize before 0x4A after 0x1E) reports: nbhead: 255, nbsectorpertrack: 63, nbcylinder: 1027, bytepersector: 512, nbtotalsector: 0XFBC043 (i.e. 16498755), infobit 0x2
  V3.0 check: signature0xBEDD = 0x0, chksum 0x0 (==0?), length 0x0 (==0x24?)
  EBIOS extension data at 0xF000A660 Phoenix Bios compatible, ideIOadr 0x1F0,0x3F6,14, multiple 16, lba_slave_mask= 0xE0
   [BIOS getparam for C/H/S partitions: 1024/255/63]  [use EBIOS nbtotalsector; usual EBIOS total == C*H*S; usual BIOS total + H*S == EBIOS total]

  ideIdentify irq 15:[input 28 1 sectors]  [28 cmd 0xEC drivehead 0xA0]  [readback= 0x0] ideIdentify reports: C/H/S=16383/16/63, RWmult=16, trans CHS valid: C/H/S=16383/16/63, total= 16514064, BigDisk total: 20015856 i.e. 0x1316AF0, BigBigLBA48total: 0X0
  Hard disk 0x81: BIOSDISK_gettype: returns 0x3, i.e. hard disk, probably nb sector = 0x130F1DC i.e. 19984860
  EBIOS detected: version 0x21, extension 0x99, features(0x5): extended_fct enhanced
  EBIOS (buffersize before 0x4A after 0x1E) reports: nbhead: 255, nbsectorpertrack: 63, nbcylinder: 1245, bytepersector: 512, nbtotalsector: 0X131309D (i.e. 20000925), infobit 0x2
  V3.0 check: signature0xBEDD = 0x0, chksum 0x0 (==0?), length 0x0 (==0x24?)
  EBIOS extension data at 0xF000A680 Phoenix Bios compatible, ideIOadr 0x170,0x376,15, multiple 16, lba_slave_mask= 0xE0
   [BIOS getparam for C/H/S partitions: 1024/255/63]  [use EBIOS nbtotalsector; ; usual BIOS total + H*S == EBIOS total]
  */

  DDBG (("EBIOS extension data at 0x%X", ebiosinfo.configptr));
  if (ebiosinfo.configptr && ebiosinfo.configptr != 0xFFFFFFFFU)
      EBIOS_analyse_phoenix (dp, ebiosinfo.configptr);
  DDBG (("\r\n"));

  dp->nbhead = ebiosinfo.nbhead;
  dp->nbsectorpertrack = ebiosinfo.nbsectorpertrack;
  dp->nbcylinder = ebiosinfo.nbcylinder;
  dp->bytepersector = ebiosinfo.bytepersector;
  dp->nbtotalsector = ebiosinfo.nbtotalsector;
  dp->infobit = ebiosinfo.infobit;

  return 0;
  }

DISK_FCT_PREFIX(EBIOSDISK_probe) static inline int
EBIOSDISK_probe (unsigned char disk, struct diskparam_str *dp)
  {
  /* never seen an exception here, so not treated (2nd call) */
  if (_EBIOSDISK_probe (disk, &dp->ebios_version, &dp->ebios_bitmap) == 0) {
      DDBG (("EBIOS detected: version 0x%X, features(0x%X): %s%s%s\r\n",
		dp->ebios_version,
		*CAST(unsigned short *, &dp->ebios_bitmap),
		dp->ebios_bitmap.extended_fct? "extended_fct " : "",
		dp->ebios_bitmap.removable? "removable " : "",
		dp->ebios_bitmap.enhanced? "enhanced " : ""
		));
      if (dp->ebios_bitmap.extended_fct)
	  dp->access = ebios_lba;
	else
	  dp->access = bios_chs;
      return 0;
      }
    else
      dp->access = bios_chs;
  return 1;
  }

#endif /* EBIOS_SUPPORT */

/**
 ** The low level IDE functions:
 **/
#if DISK_SUPPORT & IDE_SUPPORT

enum IDE_error {
    IDE_OK = 0,
    IDE_badparam,
    IDE_busy_timeout,
    IDE_DriveFault,
    IDE_busy_drq_timeout,
    IDE_driveselect_error,
    IDE_cmderror,
    IDE_devicefault,
    IDE_DRQ,
    IDE_data_busy_timeout,
    IDE_busy_notdrq_timeout,
    IDE_data_cmderror,
    IDE_data_devicefault,
#if DISK_SUPPORT & ATAPI_SUPPORT
    IDE_atapi_interrupt_reason,
#endif /* ATAPI_SUPPORT */
    IDE_undocumented
    };

#if DEBUG & DEBUG_DISK
static const char *const errmsg[] = {
    [IDE_OK]			= "IDE OK",
    [IDE_badparam]		= "IDE bad parameters",
    [IDE_busy_timeout]		= "IDE busy timeout",
    [IDE_DriveFault]		= "IDE DriveFault",
    [IDE_busy_drq_timeout]	= "IDE busy drq timeout",
    [IDE_driveselect_error]	= "IDE driveselect error",
    [IDE_cmderror]		= "IDE cmderror",
    [IDE_devicefault]		= "IDE devicefault",
    [IDE_DRQ]			= "IDE datarequest",
    [IDE_data_busy_timeout]	= "IDE data busy timeout",
    [IDE_busy_notdrq_timeout]	= "IDE data busy not drq timeout",
    [IDE_data_cmderror]		= "IDE data cmderror",
    [IDE_data_devicefault]	= "IDE data devicefault",
#if DISK_SUPPORT & ATAPI_SUPPORT
    [IDE_atapi_interrupt_reason]	= "IDE atapi interrupt reason",
#endif /* ATAPI_SUPPORT */
    [IDE_undocumented]		= "IDE undocumented"
    };

#define DBG_DISPLAY_ERROR(ide_error) do { \
      struct IDE_error_str error = ide_error;	\
      DDBG ((" [status.ERR set: "));		\
      if (error.AMNF)				\
	  DDBG (("AMNF, "));			\
      if (error.na1)				\
	  DDBG (("na1, "));			\
      if (error.ABRT)				\
	  DDBG (("ABRT, "));			\
      if (error.na3)				\
	  DDBG (("na3, "));			\
      if (error.IDNF)				\
	  DDBG (("IDNF, "));			\
      if (error.na5)				\
	  DDBG (("na5, "));			\
      if (error.UNC)				\
	  DDBG (("UNC, "));			\
      if (error.BBK)				\
	  DDBG (("BBK, "));			\
      DDBG (("] "));				\
      } while (0)

#ifdef DEBUG_IDE_TIMING
DISK_FCT_PREFIX(RDTSC_DISPLAY) static void
RDTSC_DISPLAY (const char *msg, unsigned dsk_rtdsc,
		unsigned init, unsigned left, unsigned altstatus)
  {
  unsigned nb = dsk_rtdsc;
  char unit;
  /* UTIL.processor.calibrate_rdtsc is per 1/100 seconds */

  if (UTIL.processor.calibrate_rdtsc < 100)
      return;

  if (nb > UTIL.processor.calibrate_rdtsc) {
      // nb = nb / (100 * UTIL.processor.calibrate_rdtsc) * 1000; /* ms */
      nb /= (UTIL.processor.calibrate_rdtsc/100); /* in 0.1 millisecond */
      unit = 'm';
      }
    else {
      nb *= 1000;
      if (nb >  UTIL.processor.calibrate_rdtsc) {
	  nb /= (UTIL.processor.calibrate_rdtsc/100);	/* in 0.1 us */
	  unit = 'u';
	  }
	else {
	  nb *= 1000;
	  nb /= (UTIL.processor.calibrate_rdtsc/100);	/* in 0.1 ns */
	  unit = 'n';
	  }
      }
  if (nb < 1000) {
      if (left == 0)
	  DDBG ((" [%s timeout(%u) in %u cycles, i.e. %u.%u %cs; altstatus = 0x%X] ",
		msg, init, dsk_rtdsc, nb / 10, nb % 10, unit, altstatus));
	else
	  DDBG ((" [%s passed(%u, %u left) in %u cycles, i.e. %u.%u %cs; altstatus = 0x%X] ",
		msg, init, left, dsk_rtdsc, nb / 10, nb % 10, unit, altstatus));
      }
    else {
      if (left == 0)
	  DDBG ((" [%s timeout(%u) in %u cycles, i.e. %u %cs; altstatus = 0x%X] ",
		msg, init, dsk_rtdsc, nb / 10, unit, altstatus));
	else
	  DDBG ((" [%s passed(%u, %u left) in %u cycles, i.e. %u %cs; altstatus = 0x%X] ",
		msg, init, left, dsk_rtdsc, nb / 10, unit, altstatus));
      }
  }
#else
#define RDTSC_DISPLAY(str, tsc, init, left, status) /* */
#endif /* DEBUG_IDE_TIMING */
#else
#define RDTSC_DISPLAY(str, tsc, init, left, status) /* */
#endif /* DEBUG_DISK */

DISK_FCT_PREFIX(calibrateIDEloop) static inline void
calibrateIDEloop (struct diskparam_str *dp)
  {
  /* UTIL.processor.calibrate_rdtsc is per 1/100 seconds */

#ifndef IDE_SIMPLE_1us_per_inb
//  if (UTIL.processor.calibrate_rdtsc != 0) {
  if (UTIL.processor.calibrate_rdtsc >= 10) {
      unsigned long long tsc;
      unsigned timeout = 1000, tmp;
      struct IDE_status_str read __attribute__ ((unused));

      DDBG (("%s: 1000 loop executed in ", __FUNCTION__));
      tsc = rdtsc();
      /* This shall always timeout, it should with
	 those parameters, else there is no IDE here: */
      read = _IDE_wait_altstatus (dp,
		(struct IDE_status_str) { /* TESTABORT */
			.BSY = 1, .DRDY = 0, .DF = 0, .DSC = 0,
			.DRQ = 0, .CORR = 0, .IDX = 0, .ERR = 0
			},
		(struct IDE_status_str) { /* XOR */
			.BSY = 1, .DRDY = 0, .DF = 1, .DSC = 0,
			.DRQ = 1, .CORR = 1, .IDX = 1, .ERR = 1
			},
		(struct IDE_status_str) { /* TEST */
			.BSY = 1, .DRDY = 1, .DF = 1, .DSC = 1,
			.DRQ = 1, .CORR = 1, .IDX = 1, .ERR = 1
			},
		&timeout);
      tsc = rdtsc() - tsc;
      if (timeout != 0)
	  DDBG ((" ERROR no timeout! (status: 0x%X) ", *(unsigned char *)&read));
      tmp = (unsigned) tsc;
      DDBG (("%u cycles, ", tmp));
      tmp *= 1000;
      tmp /= (UTIL.processor.calibrate_rdtsc / 10);
      DDBG (("i.e. %u ns per loop\r\n", tmp));
      dp->CalibrationIdeloop = tmp;
      }
    else
#endif /* IDE_SIMPLE_1us_per_inb */
      {
      DDBG (("%s: no useable TSC, using default %u ns/loop\r\n",
		__FUNCTION__,
		NOTSC_IDE_LOOP_DURATION));
      dp->CalibrationIdeloop = NOTSC_IDE_LOOP_DURATION;
      }
  }

enum IDE_wait_enum {
  IDE_initial_wait,
  IDE_driveselect_wait,
#if DISK_SUPPORT & ATAPI_SUPPORT
  IDE_atapiselect_wait, /* previous replacement for ATAPI */
#endif
  IDE_postcmd_wait,
#if DISK_SUPPORT & ATAPI_SUPPORT
  IDE_postatapicmd_wait, /* extra step after writing the ATAPI command packet */
#endif
  IDE_firstsector_wait,
  IDE_nextsector_wait,
  IDE_finnish_wait
  };

DISK_FCT_PREFIX(IDE_wait) static
#if !(DEBUG & DEBUG_DISK)
inline
#endif
struct IDE_status_str
IDE_wait (struct diskparam_str *dp, enum IDE_wait_enum state)
  {
  unsigned timeout;
  unsigned initial_timeout __attribute__ ((unused));
  struct IDE_status_str status;
#ifndef IDE_SIMPLE_1us_per_inb
  unsigned CalibrationIdeloop = NOTSC_IDE_LOOP_DURATION;
#else
  unsigned CalibrationIdeloop = dp->CalibrationIdeloop;
#endif /* IDE_SIMPLE_1us_per_inb */

#ifdef DEBUG_IDE_TIMING
  unsigned long long dsk_rtdsc;

  /* Take care: no rdtsc => should not divide by 0 */
#define RDTSC()		(UTIL.processor.calibrate_rdtsc? rdtsc() : 0)
#define RDTSC_START()	dsk_rtdsc = RDTSC()
#define RDTSC_STOP()	dsk_rtdsc = RDTSC() - dsk_rtdsc
#else
#define RDTSC_START()	/* */
#define RDTSC_STOP()	/* */
#endif

  switch (state) {
      case IDE_initial_wait:
	  initial_timeout = timeout = 1000000U * TIMEOUT_LOOP_INIT / CalibrationIdeloop;
	  RDTSC_START();
	  status = _IDE_wait_altstatus (dp,
		(struct IDE_status_str) { .BSY = 1, .ERR = 1 },
		(struct IDE_status_str) {},
		(struct IDE_status_str) { .DRQ = 1 },
		&timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY ("init not DRQ", dsk_rtdsc, initial_timeout,
			timeout, _IDE_get_altstatus (dp));
	  break;
      case IDE_driveselect_wait:
#ifdef IDE_SIMPLE_400ns_WAIT
	  RDTSC_START();
	  _IDE_get_altstatus_bits(dp);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("select short_delay: inb(altstatus)", dsk_rtdsc, 0,
			0, _IDE_get_altstatus (dp));
#else
	  /* 400 ns = 400 / 1,000,000,000 = 1/2500000, calibrate_loopl is nb loops/hundredth second: */
	  timeout = (UTIL.processor.calibrate_loopl? UTIL.processor.calibrate_loopl : 160000L) / 25000;
	  RDTSC_START();
	  short_delay (timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("select short_delay (400ns)", dsk_rtdsc, timeout,
			0, _IDE_get_altstatus (dp));
#endif /* IDE_SIMPLE_400ns_WAIT */

	  initial_timeout = timeout = 1000000U * TIMEOUT_LOOP_DRV_READY / CalibrationIdeloop;
	  RDTSC_START();
	  status = _IDE_wait_altstatus (dp,
		(struct IDE_status_str) { .BSY = 1, .ERR = 1 },
		(struct IDE_status_str) { .DRDY = 1 },
		(struct IDE_status_str) { .DRQ = 1, .DRDY = 1 },
		&timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("not DRQ yes DRDY", dsk_rtdsc, initial_timeout,
			timeout, _IDE_get_altstatus (dp));
	  break;
#if DISK_SUPPORT & ATAPI_SUPPORT
      case IDE_atapiselect_wait:
#ifdef IDE_SIMPLE_400ns_WAIT
	  RDTSC_START();
	  _IDE_get_altstatus_bits(dp);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("select short_delay: inb(altstatus)", dsk_rtdsc, 0,
			0, _IDE_get_altstatus (dp));
#else
	  /* 400 ns = 400 / 1,000,000,000 = 1/2500000, calibrate_loopl is nb loops/hundredth second: */
	  timeout = (UTIL.processor.calibrate_loopl? UTIL.processor.calibrate_loopl : 160000L) / 25000;
	  RDTSC_START();
	  short_delay (timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("select short_delay (400ns)", dsk_rtdsc, timeout,
			0, _IDE_get_altstatus (dp));
#endif /* IDE_SIMPLE_400ns_WAIT */

	  initial_timeout = timeout = 1000000U * TIMEOUT_LOOP_ATAPI_READY / CalibrationIdeloop;
	  RDTSC_START();
	  status = _IDE_wait_altstatus (dp,
		(struct IDE_status_str) { .BSY = 1, .ERR = 1 },
		(struct IDE_status_str) {},
		(struct IDE_status_str) { .DRQ = 1},
		&timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("not DRQ for ATAPI", dsk_rtdsc, initial_timeout,
			timeout, _IDE_get_altstatus (dp));
	  break;
#endif /* ATAPI_SUPPORT */
      case IDE_postcmd_wait:
#ifdef IDE_SIMPLE_400ns_WAIT
	  RDTSC_START();
	  _IDE_get_altstatus_bits(dp);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("postcmd short_delay: inb(altstatus)", dsk_rtdsc, 0,
			0, _IDE_get_altstatus (dp));
#else
	  timeout = (UTIL.processor.calibrate_loopl? UTIL.processor.calibrate_loopl : 160000L) / 25000;
	  RDTSC_START();
	  short_delay (timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("postcmd short_delay (400ns)", dsk_rtdsc, timeout,
			0, _IDE_get_altstatus (dp));
#endif /* IDE_SIMPLE_400ns_WAIT */

	  initial_timeout = timeout = 1000000U * TIMEOUT_LOOP_EXEC_CMD / CalibrationIdeloop;
	  RDTSC_START();
	  status = _IDE_wait_altstatus (dp,
		(struct IDE_status_str) { .BSY = 1, .ERR = 1 },
		(struct IDE_status_str) {},
		(struct IDE_status_str) {},
		&timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("not busy after cmd", dsk_rtdsc, initial_timeout,
			timeout, _IDE_get_altstatus (dp));
	  break;
#if DISK_SUPPORT & ATAPI_SUPPORT
      case IDE_postatapicmd_wait:
#ifdef IDE_SIMPLE_400ns_WAIT
	  RDTSC_START();
	  _IDE_get_altstatus_bits(dp);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("IDE_postatapicmd_wait: inb(altstatus)", dsk_rtdsc, 0,
				      0, _IDE_get_altstatus (dp));
#else
	  timeout = (UTIL.processor.calibrate_loopl? UTIL.processor.calibrate_loopl : 160000L) / 25000;
	  RDTSC_START();
	  short_delay (timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY("IDE_postatapicmd_wait (400ns)", dsk_rtdsc, timeout,
				      0, _IDE_get_altstatus (dp));
#endif /* IDE_SIMPLE_400ns_WAIT */

#if TIMEOUT_LOOP_EXEC_ATAPICMD > 4294
	  {
	  unsigned long_timeout = TIMEOUT_LOOP_EXEC_ATAPICMD;

	  while (long_timeout) {
	      unsigned partial_timeout = (long_timeout > 4000)? 4000 : long_timeout;
	      long_timeout -= partial_timeout;
#else
	  {
	  unsigned partial_timeout = TIMEOUT_LOOP_EXEC_ATAPICMD;
	      {
#endif
	      initial_timeout = timeout = 1000000U * partial_timeout / CalibrationIdeloop;
	      RDTSC_START();
	      status = _IDE_wait_altstatus (dp,
			(struct IDE_status_str) { .BSY = 1, .ERR = 1 },
			(struct IDE_status_str) {},
			(struct IDE_status_str) {},
			&timeout);
	      RDTSC_STOP();
	      RDTSC_DISPLAY("not busy after atapi cmd", dsk_rtdsc, initial_timeout,
			timeout, _IDE_get_altstatus (dp));
	      }
	  }
	  break;
#endif /* ATAPI_SUPPORT */
      case IDE_firstsector_wait:
	  initial_timeout = timeout = 1000000U * TIMEOUT_LOOP_FIRST_SECT / CalibrationIdeloop;
	  RDTSC_START();
	  status = _IDE_wait_altstatus (dp,
		(struct IDE_status_str) { .BSY = 1, .ERR = 1 },
		(struct IDE_status_str) { .DRQ = 1},
		(struct IDE_status_str) { .DRQ = 1},
		&timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY ("firstsector wait DRQ", dsk_rtdsc, initial_timeout,
			timeout, _IDE_get_altstatus (dp));
	  break;
      case IDE_nextsector_wait:
	  initial_timeout = timeout = 1000000U * TIMEOUT_LOOP_NEXT_SECT / CalibrationIdeloop;
	  RDTSC_START();
	  status = _IDE_wait_altstatus (dp,
		(struct IDE_status_str) { .BSY = 1, .ERR = 1 },
		(struct IDE_status_str) { .DRQ = 1},
		(struct IDE_status_str) { .DRQ = 1},
		&timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY ("nextsector wait DRQ", dsk_rtdsc, initial_timeout,
			timeout, _IDE_get_altstatus (dp));
	  break;
      case IDE_finnish_wait:
	  initial_timeout = timeout = 1000000U * TIMEOUT_LOOP_FINISH_CMD / CalibrationIdeloop;
	  RDTSC_START();
	  status = _IDE_wait_altstatus (dp,
		(struct IDE_status_str) { .BSY = 1, .ERR = 1 },
		(struct IDE_status_str) {},
		(struct IDE_status_str) {},
		&timeout);
	  RDTSC_STOP();
	  RDTSC_DISPLAY ("last not BUSY", dsk_rtdsc, initial_timeout,
			timeout, _IDE_get_altstatus (dp));
	  break;
      default:
	  status = _IDE_get_altstatus_bits(dp);
	  break;
      }
  return status;
  }

/*
 * One of block28 and block48 has to be <= 1, the other been an address.
 * If readback != 0 (has to be for IDE_PACKET):
 *    if block28->command == IDE_PACKET readback ATAPI nbbytes i.e. cylinder,
 *    if IDE_BLOCK*8_INPUT, readback lba,
 *    if IDE_BLOCK*8_OUTPUT, readback sector_count.
 */
DISK_FCT_PREFIX(IDE_non_data_command) static enum IDE_error
IDE_non_data_command (struct diskparam_str *dp,
		      const struct IDE_feature_to_cmd_str *block28,
		      const struct IDE_feature_to_cmd_addr48_str *block48,
		      void *readback)
  {
  struct IDE_status_str status;
  unsigned char drivehead;
#define is_lba28_cmd(block48)	(block48 == IDE_BLOCK28_INPUT || block48 == IDE_BLOCK28_OUTPUT)
#define is_lba48_cmd(block28)	(block28 == IDE_BLOCK48_INPUT || block28 == IDE_BLOCK48_OUTPUT)

#ifdef DEBUG_IDE_TIMING
  unsigned long long dsk_rtdsc;
#endif

  if (is_lba28_cmd(block48)) {
      if (is_lba48_cmd(block28)) {
	  DDBG (("%s: block28 = 0x%X, block48 = 0x%X: ABORT!\r\n", __FUNCTION__, (unsigned)block28, (unsigned)block48));
	  return IDE_badparam;
	  }
      drivehead = block28->chs_lba.chs.drivehead;
      DDBG ((" [28 cmd 0x%X drivehead 0x%X] ", block28->command, drivehead));
      }
    else if (is_lba48_cmd(block28)) {
      drivehead = block48->drivehead;
      DDBG ((" [48 cmd 0x%X drivehead 0x%X] ", block48->command, drivehead));
      }
    else {
      DDBG (("%s: block28 = 0x%X, block48 = 0x%X: ABORT!\r\n", __FUNCTION__, (unsigned)block28, (unsigned)block48));
      return IDE_badparam;
      }

  if ((status = IDE_wait(dp, IDE_initial_wait)).BSY) {
      /* Well, the only way would be to reset the IDE interface here... */
      DDBG ((" [ERROR timeout to get first BSY inactive, status 0x%X] ",
			*(unsigned char *)&status));
      return IDE_busy_timeout;
      }
    else if (!(status.DF || status.ERR) && status.DRQ) {
#if DISK_SUPPORT & ATAPI_SUPPORT
      if (is_lba28_cmd(block48) && block28->command == IDE_PACKET) {
	  struct ATAPI_interrupt_reason_str interrupt_reason = _ATAPI_get_interrupt_reason_bits (dp);
	  DDBG ((" [status 0x%X, first DRQ set when ERR and DF cleared, ", *(unsigned char *)&status));
	  if (interrupt_reason.command_data == 0 && interrupt_reason.input_output == 1) {
	      unsigned short nbbyte = _IDE_get_cylinder (dp);
	      DDBG (("ATAPI interrupt reason 0x%X, can only continue after read flush %u bytes] ",
			*(unsigned char *)&interrupt_reason, nbbyte));
	      _IDE_flush_buffer (dp, (nbbyte + 1)/2);
	      if ((status = IDE_wait(dp, IDE_initial_wait)).BSY) {
		  DDBG ((" [ERROR after flush timeout to get first BSY inactive, status 0x%X] ",
			*(unsigned char *)&status));
		  return IDE_busy_timeout;
		  }
		else if (!(status.DF || status.ERR) && status.DRQ) {
		  DDBG ((" [status 0x%X after flush, first DRQ set when ERR and DF cleared, cannot continue] ",
				*(unsigned char *)&status));
		  return IDE_busy_drq_timeout;
		  }
	      }
	    else {
	      DDBG (("ATAPI interrupt reason 0x%X, cannot continue] ", *(unsigned char *)&interrupt_reason));
	      return IDE_busy_drq_timeout;
	      }
	  }
	else
#endif
	  {
	  DDBG ((" [status 0x%X, first DRQ set when ERR and DF cleared, cannot continue] ",
			*(unsigned char *)&status));
	  return IDE_busy_drq_timeout;
	  }
      }

  disable_interrupt();
//PCI card: 1103:0004 00:0b.0 Mass storage controller: Triones Technologies, Inc. HPT366/368/370/370A/372/372N (rev 01)
// with only some IDE disks print the first "<" but not the second ">"
//print("<");
  _IDE_set_device_control_register (dp, (struct IDE_control_str) { .notIEN = 1, .na1 = 1 });
//print(">");

//  if (_IDE_get_drivehead(dp) & 0x10 != drivehead & 0x10)
      {
      _IDE_set_drivehead (dp, drivehead);

#if DISK_SUPPORT & ATAPI_SUPPORT
      /* With some IDE & CDROM combination, we will never get DRDY even
	 if we wait 1000 inb() cycles - so do not try to. */
      if (is_lba28_cmd(block48) && (block28->command == IDE_PACKET || block28->command == IDE_IDENTIFY_PACKET_DEVICE)) {
	  if (block28->command == IDE_PACKET && readback == 0) {
	      enable_interrupt();
	      DDBG (("%s: block28 PACKET command and null packet address, ABORT!\r\n", __FUNCTION__));
	      return IDE_badparam;
	      }
	  DDBG (("[IDE_atapiselect_wait, "));
	  status = IDE_wait (dp, IDE_atapiselect_wait);
	  if (status.DRDY)
	      DDBG (("status 0x%X] ", *(unsigned char *)&status));
	    else {
	      DDBG (("status 0x%X - forcing DRDY to 1 for ATAPI] ", *(unsigned char *)&status));
	      status.DRDY = 1;
	      }
	  }
	else
#endif /* ATAPI_SUPPORT */
	  {
	  DDBG ((" [IDE_driveselect_wait, "));
	  status = IDE_wait (dp, IDE_driveselect_wait);
	  DDBG (("status 0x%X]", *(unsigned char *)&status));
	  }
      if (status.BSY) {
	  _IDE_set_drivehead (dp, drivehead ^ 0x10);
	  enable_interrupt();
	  DDBG ((" [ERROR timeout to get second BSY inactive, selecting other IDE channel: 0x%X] ",
			drivehead ^ 0x10));
	  return IDE_busy_timeout;
	  }
/* FIXME: I have status.IDX with WinMe boot disk sometimes, what does that means??? */
      if (!status.DRDY) {
	  /* Some old ATAPI CDROM arrive here on IDE identify - read ATAPI reset signature: */
	  if (readback && block48 == IDE_BLOCK28_INPUT)
	      *(unsigned *)readback = _IDE_readback_lba28 (dp);
	  _IDE_set_drivehead (dp, drivehead ^ 0x10);
	  enable_interrupt();
	  DDBG ((" [ERROR no DriveRDY, selecting other IDE channel: 0x%X] ", drivehead ^ 0x10));
	  return IDE_driveselect_error;
	  }
      if (!(status.DF || status.ERR) && status.DRQ) {
	  _IDE_set_drivehead (dp, drivehead ^ 0x10);
	  enable_interrupt();
	  DDBG ((" [ERROR second DRQ set when ERR and DF cleared, "
		 "cannot continue, selecting other IDE channel: 0x%X] ", drivehead ^ 0x10));
	  return IDE_busy_drq_timeout;
	  }
      }

  RDTSC_START();
  if (is_lba28_cmd(block48))
      _IDE_set_feature_to_cmd (dp, block28);
    else
      _IDE_set_feature_to_cmd_addr48 (dp, block48);
  RDTSC_STOP();
  RDTSC_DISPLAY ("write cmd block", dsk_rtdsc, 0, 0, _IDE_get_altstatus (dp));

  if ((status = IDE_wait (dp, IDE_postcmd_wait)).BSY) {
      enable_interrupt();
      DDBG ((" [ERROR timeout to get after-cmd BSY inactive, alt-status 0x%X] ",
		*(unsigned char *)&status));
      return IDE_busy_timeout;
      }

  status = _IDE_get_status_bits (dp); /* clear the interrupt */

#if DISK_SUPPORT & ATAPI_SUPPORT
  if (is_lba28_cmd(block48) && block28->command == IDE_PACKET) {
      struct ATAPI_interrupt_reason_str interrupt_reason = _ATAPI_get_interrupt_reason_bits (dp);

      DDBG (("[status before writing packet: 0x%X, interrupt reason 0x%X] ",
		*(unsigned char *)&status, *(unsigned char *)&interrupt_reason));
      if (!interrupt_reason.command_data || interrupt_reason.input_output) {
	  enable_interrupt();
	  DDBG ((" [ATAPI: ERROR on interrupt reason] "));
	  return IDE_atapi_interrupt_reason;
	  }
#if 0
      {
      farptr readback_farptr = data_adr(readback);
      _IDE_put_buffer (dp, &readback_farptr, 12/6);
      }

//    _IDE_put_buffer (dp, data_adr(readback), 12/6);
#else
      _IDE_put_packetcmd (dp, readback, 12);
#endif

      if ((status = IDE_wait (dp, IDE_postatapicmd_wait)).BSY) {
	  enable_interrupt();
	  DDBG (("[ERROR timeout to get after-ATAPI-cmd BSY inactive, alt-status 0x%X] ",
		*(unsigned char *)&status));
	  return IDE_busy_timeout;
	  }
      status = _IDE_get_status_bits (dp); /* clear the possible interrupt */
      if (status.DRQ) {
	  struct ATAPI_interrupt_reason_str interrupt_reason = _ATAPI_get_interrupt_reason_bits (dp);
	  DDBG (("[status after writing packet: 0x%X, DRQ set, interrupt reason 0x%X] ",
			*(unsigned char *)&status, *(unsigned char *)&interrupt_reason));
	  if (interrupt_reason.command_data) {
	      enable_interrupt();
	      DDBG ((" [ATAPI: ERROR on interrupt reason: shall not have command_data bit] "));
	      return IDE_atapi_interrupt_reason;
	      }
	  /* And remember, that is a perfect black box, with the ratio of its sides,
		the quadratic sequence 1:4:9, to follow and go into! */
	  *(unsigned short *)readback = _IDE_get_cylinder (dp);
	  if (interrupt_reason.input_output && block48 == IDE_BLOCK28_OUTPUT) {
	      enable_interrupt();
	      DDBG ((" [ATAPI: ERROR on interrupt reason: disagree IDE_BLOCK28_OUTPUT and input_output set] "));
	      return IDE_atapi_interrupt_reason;
	      }
	    else if (!interrupt_reason.input_output && block48 == IDE_BLOCK28_INPUT) {
	      enable_interrupt();
	      DDBG ((" [ATAPI: ERROR on interrupt reason: disagree IDE_BLOCK28_INPUT and input_output clear] "));
	      return IDE_atapi_interrupt_reason;
	      }
	  DDBG ((" [ATAPI: exit %s with DRQ, need to %s at least %u bytes] ",
		__FUNCTION__, (block48 == IDE_BLOCK28_INPUT)? "read" : "write", *(unsigned short *)readback));
	  }
	else {
	  *(unsigned short *)readback = _IDE_get_cylinder (dp);
	  DDBG (("[status after writing packet: 0x%X, DRQ cleared, byte count 0x%X] ",
		*(unsigned char *)&status, *(unsigned short *)readback));
	  }
      }
    else
#endif /* ATAPI_SUPPORT */
	 if (readback) {
      if (block48 == IDE_BLOCK28_INPUT) {
	  *(unsigned *)readback = _IDE_readback_lba28 (dp);
	  DDBG (("[readback= 0x%X] ", *(unsigned *)readback));
	  }
	else if (block48 == IDE_BLOCK28_OUTPUT) {
	  *(unsigned char *)readback = _IDE_get_sector_count (dp);
	  DDBG (("[readback= 0x%X] ", *(unsigned char *)readback));
	  }
	else if (block28 == IDE_BLOCK48_INPUT) {
	  *(unsigned long long *)readback = _IDE_readback_lba48 (dp);
	  DDBG (("[readback= 0x%llX] ", *(unsigned long long *)readback));
	  }
	else { /* if (block28 == IDE_BLOCK48_OUTPUT) */
	  *(unsigned short *)readback = _IDE_get_sector_count48 (dp);
	  DDBG (("[readback= 0x%X] ", *(unsigned short *)readback));
	  }
      }

  enable_interrupt();

  if (status.ERR) {
#if DEBUG & DEBUG_DISK
      DBG_DISPLAY_ERROR (_IDE_get_error_bits (dp));
#else
      _IDE_get_error_bits (dp);
#endif
      return IDE_cmderror;
      }
  if (status.DF)
      return IDE_devicefault;
  if (status.DRQ)
      return IDE_DRQ;
  return IDE_OK;
  }

/* HPT370/372 have a buggy BIOS... : they forget to set the
   device control register before doing reads/writes,
   that is why they crash all the time. Well, no need to comment */
DISK_FCT_PREFIX(IDE_reset_DCR_for_HTP_BIOS) static inline void
IDE_reset_DCR_for_HTP_BIOS (struct diskparam_str *dp)
  {
  _IDE_set_device_control_register (dp,
			(struct IDE_control_str) { .notIEN = 0, .na1 = 1 });
  }

/*
 * One of block28 and block48 has to be an address, the other has to
 * be zero for input, 1 for output.
 * TODO: Old portable PC (486 based) have an 8 bit IDE interface,
 * DATA port is 8 bits instead of 16. It can be set by a "set IDE feature",
 * but maybe better to be able to handle 8 bits here to not crash a DOS
 * debug session running. (IDE_SET_FEATURES in ide.h, no GET)
 */
DISK_FCT_PREFIX(IDE_PIO_data) static enum IDE_error
IDE_PIO_data (struct diskparam_str *dp,
#if !(DISK_SUPPORT & ATAPI_SUPPORT)
	      const
#endif
		   struct IDE_feature_to_cmd_str *block28,
	      const struct IDE_feature_to_cmd_addr48_str *block48,
	      farptr buffer, void *readback)
  {
  struct IDE_status_str status;
  enum IDE_error err;
  unsigned sector_count, blocksize;

  if (buffer == 0) {
      DDBG (("%s: buffer = 0\r\n", __FUNCTION__));
      return IDE_badparam;
      }

  err = IDE_non_data_command (dp, block28, block48, readback);

  if (err == IDE_DRQ) {
      /* a great world that is */
      }
    else if (err == IDE_OK) {
      /* DRQ may not be immediately here, wait interrupt enabled: */
      if ((status = IDE_wait (dp, IDE_firstsector_wait)).BSY) {
	  DDBG ((" [ERROR timeout to get third BSY inactive, status 0x%X] ",
			*(unsigned char *)&status));
	  return IDE_data_busy_timeout;
	  }
	else if (status.DF || status.ERR) {
	  DDBG ((" [ERROR DF || ERR after cmd, status 0x%X] ",
			*(unsigned char *)&status));
	  return IDE_busy_notdrq_timeout;
	  }
	else if (!status.DRQ) {
	  DDBG ((" [ERROR no DRQ after cmd, status 0x%X] ",
			*(unsigned char *)&status));
	  return IDE_busy_notdrq_timeout;
	  }
      }
    else
      return err; /* for instance IDE_badparam */

  static const unsigned char IDE_cmd_512[] = {
      /* From ATA7: */
      IDE_DOWNLOAD_MICROCODE,
      IDE_IDENTIFY_DEVICE,
      IDE_READ_BUFFER,
      IDE_READ_LOG_EXT,
      IDE_SECURITY_DISABLE_PASSWORD,
      IDE_SECURITY_ERASE_UNIT,
      IDE_SECURITY_SET_PASSWORD,
      IDE_SECURITY_UNLOCK,
      IDE_SET_MAX_ADDRESS, /* i.e. SET MAX SET PASSWORD, SET MAX UNLOCK with PIO data out */
      IDE_SMART, /* i.e. SMART READ DATA, SMART READ LOG, SMART WRITE LOG with PIO data */
      IDE_WRITE_BUFFER,
      IDE_WRITE_LOG_EXT,
      /* We add this one too for Gujin: */
      IDE_IDENTIFY_PACKET_DEVICE
      };

  if (is_lba28_cmd(block48)) {
#if DISK_SUPPORT & ATAPI_SUPPORT
      if (block28->command == IDE_PACKET) {
	  blocksize = *(unsigned short *)readback;
	  if (blocksize > block28->chs_lba.chs.cylinder) {
	      DDBG ((" [limit max read size in block28->chs.cylinder %u, ATAPI readback %u, use %u] ",
			block28->chs_lba.chs.cylinder, blocksize, block28->chs_lba.chs.cylinder));
	      blocksize = block28->chs_lba.chs.cylinder;
	      }
	  sector_count = 0xFFFFFFFF; /* i.e. variable number of loops */
	  DDBG ((" [%s PACKET first %u of %u bytes] ", block48 == IDE_BLOCK28_INPUT ? "input" : "output", blocksize, block28->chs_lba.chs.cylinder));
	  }
	else
#endif /* ATAPI_SUPPORT */
	  {
	  unsigned short cpt;
	  for (cpt = 0; cpt < sizeof(IDE_cmd_512)/sizeof(IDE_cmd_512[0]); cpt++)
	      if (block28->command == IDE_cmd_512[cpt])
		  break;
	  if (cpt < sizeof(IDE_cmd_512)/sizeof(IDE_cmd_512[0]))
	      blocksize = 512;
	    else
	      blocksize = dp->bytepersector;
	  sector_count = (block28->sector_count == 0)? 256 : block28->sector_count;
	  DDBG ((" [%s IDE28 %u sectors of %u bytes] ", block48 == IDE_BLOCK28_INPUT ? "input" : "output", sector_count, blocksize));
	  }
      }
    else {
      sector_count = (block48->sector_count == 0)? 256 : block48->sector_count;
      blocksize = dp->bytepersector? dp->bytepersector : 512; /* IDE_READ_NATIVE_MAX_ADDRESS_EXT & IDE_SET_MAX_ADDRESS_EXT */
      DDBG ((" [%s IDE48 %u sectors of %u bytes] ", block28 == IDE_BLOCK48_INPUT ? "input" : "output", sector_count, blocksize));
      }

  while (sector_count != 0) {
#ifdef DEBUG_IDE_TIMING
      unsigned long long dsk_rtdsc;
#endif

/* FIXME: Something crasy I do not achieve to correct: compiler GCC_3.4 bug?
   normally _IDE_put_buffer()/_IDE_get_buffer() second parameter should
   simply be a farptr, not a farptr* . If I correct it and correct ide.h
   the IDE read no more works - and the correction is obvious
 */
      if (block48 == IDE_BLOCK28_OUTPUT || block28 == IDE_BLOCK48_OUTPUT) {
	  if ((buffer & 0xFFFF) + blocksize > 0x10000)
	      DBG ((" [%s: PANIC POINTER WILL ROLLBACK writing, sector count %u, blocksize %u, is_lba28_cmd %u, block28->command %u!] ", __FUNCTION__, sector_count, blocksize, is_lba28_cmd(block48), block28->command));
	  RDTSC_START();
	  disable_interrupt();
//	  _IDE_put_buffer (dp, buffer, blocksize/2);
	  _IDE_put_buffer (dp, &buffer, blocksize/2);
	  enable_interrupt();
	  RDTSC_STOP();
	  RDTSC_DISPLAY ("write one sector", dsk_rtdsc, 0, 0, _IDE_get_altstatus (dp));
	  }
	else {
	  if ((buffer & 0xFFFF) + blocksize > 0x10000)
	      DBG ((" [%s: PANIC POINTER WILL ROLLBACK reading, sector count %u, blocksize %u, is_lba28_cmd %u, block28->command %u!] ", __FUNCTION__, sector_count, blocksize, is_lba28_cmd(block48), block28->command));
	  RDTSC_START();
	  disable_interrupt();
//	  _IDE_get_buffer (dp, buffer, blocksize/2);
	  _IDE_get_buffer (dp, &buffer, blocksize/2);
	  enable_interrupt();
	  RDTSC_STOP();
	  RDTSC_DISPLAY ("read one sector", dsk_rtdsc, 0, 0, _IDE_get_altstatus (dp));
	  }

#if DISK_SUPPORT & ATAPI_SUPPORT
      if (sector_count == 0xFFFFFFFF) {
	  if (*(unsigned short *)readback > block28->chs_lba.chs.cylinder) {
	      /* [limit max read size in block28->chs.cylinder 2048, ATAPI readback 2352, use 2048]  */
	      unsigned flushsize = *(unsigned short *)readback - block28->chs_lba.chs.cylinder;
	      DDBG ((" [max read size limited so now flush %u bytes] ", flushsize));
	      _IDE_flush_buffer (dp, (flushsize + 1) / 2);
	      }
	  block28->chs_lba.chs.cylinder -= blocksize;
	  /* [input PACKET first 18 of 25 bytes] ideAtapiRequestSense
	     [input PACKET first 20 of 84 bytes] ideAtapiReadToc
	  if (block28->chs_lba.chs.cylinder == 0)
	   */
	  if (block28->chs_lba.chs.cylinder < 512)
	      sector_count = 0;
	    else
	      DDBG (("[ATAPI wait next packet] "));
	  }
	else
#endif
	  sector_count--;

      if (sector_count == 0) {
	  if ((status = IDE_wait (dp, IDE_finnish_wait)).BSY) {
	      DDBG ((" [ERROR timeout to get last BSY inactive, status 0x%X] ",
			*(unsigned char *)&status));
	      return IDE_data_busy_timeout;
	      }
#if DISK_SUPPORT & ATAPI_SUPPORT
	    else if (   is_lba28_cmd(block48) && block28->command == IDE_PACKET
		      && status.DRQ   && !(status.ERR || status.DF)) {
	      DDBG (("[ATAPI get next packet] "));
	      sector_count = 0xFFFFFFFF;
	      }
#endif
	    else if (status.DRQ || status.ERR || status.DF) {
	      DDBG ((" [ERROR on last status 0x%X] ", *(unsigned char *)&status));
	      }
	  }
	else {
	  if ((status = IDE_wait (dp, IDE_nextsector_wait)).BSY) {
	      DDBG ((" [ERROR timeout to get intermediate BSY inactive, status 0x%X] ",
			*(unsigned char *)&status));
	      return IDE_data_busy_timeout;
	      }
	    else if (status.DF || status.ERR) {
	      DDBG ((" [ERROR DF || ERR in intermediate, status 0x%X] ",
			*(unsigned char *)&status));
	      return IDE_busy_notdrq_timeout;
	      }
	    else if (!status.DRQ) {
	      DDBG ((" [ERROR no DRQ in intermediate, status 0x%X] ",
			*(unsigned char *)&status));
	      return IDE_busy_notdrq_timeout;
	      }
	  }
      status = _IDE_get_status_bits (dp); /* clear the interrupt */
      if (status.ERR) {
#if DEBUG & DEBUG_DISK
	  DBG_DISPLAY_ERROR (_IDE_get_error_bits (dp));
#else
	  _IDE_get_error_bits (dp);
#endif
	  return IDE_data_cmderror;
	  }
      if (status.DF)
	  return IDE_data_devicefault;
#if DISK_SUPPORT & ATAPI_SUPPORT
      if (sector_count == 0xFFFFFFFF) {
	  *(unsigned short *)readback = blocksize = _IDE_get_cylinder (dp);
	  if (blocksize <= block28->chs_lba.chs.cylinder)
	      DDBG (("[ATAPI next packet blocksize %u] ", blocksize));
	    else {
	      DDBG (("[ATAPI wish I/O of %u bytes, limit it to block28->chs_lba.chs.cylinder i.e. %u bytes] ",
			blocksize, block28->chs_lba.chs.cylinder));
	      blocksize = block28->chs_lba.chs.cylinder;
	      }
	  }
#endif
      }
  return IDE_OK;
  }

/*
 * If the compiler is perfect, this shall not reduce the code size:
 * (the problem is probably lack of write combinning optimisation)
 * Also pad to 8 bytes the structure IDE_feature_to_cmd_str and
 * call our own memcpy (inline function) instead of leaving the
 * compiler inserting an out of line memcpy for 7 bytes.
 */
#define INIT_CMDBLOCK28(block28, cmd, lba_slave_mask, cst_feature, cst_sector_count) \
	static const struct IDE_feature_to_cmd_str block28pattern = {	\
		.feature = cst_feature,				\
		.sector_count = cst_sector_count,		\
		.command = cmd					\
		};						\
	struct IDE_feature_to_cmd_str block28 = block28pattern;	\
	/*struct IDE_feature_to_cmd_str block28; memcpy (&block28, &block28pattern, sizeof(block28));*/ \
	block28.chs_lba.chs.drivehead = lba_slave_mask

#define INIT_CMDBLOCK48(block48, cmd, lba_slave_mask, cst_feature, cst_sector_count) \
	static const struct IDE_feature_to_cmd_addr48_str block48pattern = {	\
		.feature = cst_feature,						\
		.sector_count = cst_sector_count,				\
		.command = cmd							\
		};								\
	struct IDE_feature_to_cmd_addr48_str block48 = block48pattern;		\
	block48.drivehead = lba_slave_mask

#define INIT_PKTCMDBLOCK(block28, lba_slave_mask, cst_cylinder) \
	static const struct IDE_feature_to_cmd_str block28pattern = {	\
		.chs_lba = { .chs = { .cylinder = cst_cylinder }},	\
		.command = IDE_PACKET					\
		};							\
	/*struct IDE_feature_to_cmd_str block28 = block28pattern;*/		\
	struct IDE_feature_to_cmd_str block28; memcpy (&block28, &block28pattern, sizeof(block28)); \
	block28.chs_lba.chs.drivehead = lba_slave_mask

DISK_FCT_PREFIX(IDE_get_power_mode) static inline int
IDE_get_power_mode (struct diskparam_str *dp)
  {
  enum IDE_error ret;
  unsigned char val;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_CHECK_POWER_MODE, dp->lba_slave_mask, 0, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= 0,
	.chs_lba = { .chs = { .sector = 0, .cylinder = 0, .drivehead = dp->lba_slave_mask }} ,
	.command	= IDE_CHECK_POWER_MODE
	};
#endif

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_OUTPUT, &val);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }

  DDBG ((" [%s gives 0x%X, i.e. ", __FUNCTION__, val));
  if (val == 0xFF) {
      DDBG (("active or idle] "));
      return 0;
      }
    else if (val == 0x80) {
      DDBG (("idle] "));
      return 1;
      }
    else if (val == 0) {
      DDBG (("standby] "));
      return 2;
      }
    else {
      DDBG (("unknown] "));
      return -IDE_undocumented;
      }
  }

DISK_FCT_PREFIX(IDE_set_power_mode) static inline int
IDE_set_power_mode (struct diskparam_str *dp, unsigned idle_sleep_standby)
  {
  enum IDE_error ret;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_IDLE, dp->lba_slave_mask, 0, 0);
  block28.command = ((unsigned char []) {IDE_IDLE, IDE_SLEEP, IDE_STANDBY})[idle_sleep_standby];
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= 0,
	.chs_lba = { .chs = { .sector = 0, .cylinder = 0, .drivehead = dp->lba_slave_mask }} ,
	.command	= ((unsigned char []) {IDE_IDLE, IDE_SLEEP, IDE_STANDBY})[idle_sleep_standby]
	};
#endif

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_OUTPUT, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s (%s) returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, ((const char *[]) {"IDE_IDLE", "IDE_SLEEP", "IDE_STANDBY"})[idle_sleep_standby],
		 errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }

  return 0;
  }


DISK_FCT_PREFIX(IDE_get_native_max_address) static int
IDE_get_native_max_address (struct diskparam_str *dp, unsigned *max)
  {
  enum IDE_error ret;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_READ_NATIVE_MAX_ADDRESS, dp->lba_slave_mask, 0, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= 0,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_READ_NATIVE_MAX_ADDRESS
	};
#endif

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, max);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

/* The previous command had to be IDE_get_native_max_address() */
DISK_FCT_PREFIX(IDE_set_native_max_address) static int
IDE_set_native_max_address (struct diskparam_str *dp, unsigned max,
				unsigned preserve_over_reset)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= !!preserve_over_reset,
	{.lba = { .lba = max, .drv = (dp->lba_slave_mask | 0x40) >> 4 }},
	.command	= IDE_SET_MAX_ADDRESS
	};

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

/* The following set max related commands shall never be called
   just after a IDE_get_native_max_address() else their meaning
   changes... They are also usefull for LBA48. */
DISK_FCT_PREFIX(IDE_set_password_set_max) static inline int
IDE_set_password_set_max (struct diskparam_str	*dp,
				char		password[32])
  {
  enum IDE_error ret;
  struct {
	unsigned short	reserved1;
	char		password[32];
	unsigned short	reserved2[239];
	} __attribute__((packed)) buffer;
  unsigned tmp = 32;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_SET_MAX_ADDRESS, dp->lba_slave_mask, 0x01, 1);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0x01,
	.sector_count	= 1,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_SET_MAX_ADDRESS
	};
#endif

  while (tmp--)
      buffer.password[tmp] = password[tmp];

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_OUTPUT,
			stack_adr(&buffer), 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }


DISK_FCT_PREFIX(IDE_lock_set_max) static inline int
IDE_lock_set_max (struct diskparam_str *dp)
  {
  enum IDE_error ret;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_SET_MAX_ADDRESS, dp->lba_slave_mask, 0x02, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0x02,
	.sector_count	= 0,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_SET_MAX_ADDRESS
	};
#endif

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(IDE_unlock_set_max) static inline int
IDE_unlock_set_max (struct diskparam_str	*dp,
			char			password[32])
  {
  enum IDE_error ret;
  struct {
	unsigned short	reserved1;
	char		password[32];
	unsigned short	reserved2[239];
	} __attribute__((packed)) buffer;
  unsigned tmp = 32;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_SET_MAX_ADDRESS, dp->lba_slave_mask, 0x03, 1);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0x03,
	.sector_count	= 1,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_SET_MAX_ADDRESS
	};
#endif

  while (tmp--)
      buffer.password[tmp] = password[tmp];

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_OUTPUT,
			stack_adr(&buffer), 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

/* SET MAX SET PASSWORD shall have been done before.
   It disables SET MAX ADDRESS, SET MAX SET PASSWORD,
		SET MAX LOCK, SET MAX UNLOCK.
 */
DISK_FCT_PREFIX(IDE_freeze_set_max) static inline int
IDE_freeze_set_max (struct diskparam_str *dp)
  {
  enum IDE_error ret;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_SET_MAX_ADDRESS, dp->lba_slave_mask, 0x04, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0x04,
	.sector_count	= 0,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_SET_MAX_ADDRESS
	};
#endif

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(IDE_get_native48_max_address) static int
IDE_get_native48_max_address (struct diskparam_str *dp,
			       unsigned long long *max)
  {
  enum IDE_error ret;
#ifdef INIT_CMDBLOCK48
  INIT_CMDBLOCK48 (block48, IDE_READ_NATIVE_MAX_ADDRESS_EXT, dp->lba_slave_mask, 0, 0);
#else
  struct IDE_feature_to_cmd_addr48_str block48 = {
	.feature	= 0,
	.sector_count	= 0,
	.lba = { .lba = 0 },
	.drivehead	= dp->lba_slave_mask | 0x40, /* request in LBA */
	.command	= IDE_READ_NATIVE_MAX_ADDRESS_EXT
	};
#endif

  ret = IDE_non_data_command (dp, IDE_BLOCK48_INPUT, &block48, max);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

/* The previous command had to be IDE_get_native48_max_address() */
DISK_FCT_PREFIX(IDE_set_native48_max_address) static int
IDE_set_native48_max_address (struct diskparam_str *dp,
			      unsigned long long max,
			      unsigned preserve_over_reset)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_addr48_str block48 = {
	.feature	= 0,
	.sector_count	= !!preserve_over_reset,
	.lba = { .lba = max },
	.drivehead	= dp->lba_slave_mask | 0x40, /* request in LBA */
	.command	= IDE_SET_MAX_ADDRESS_EXT
	};

  ret = IDE_non_data_command (dp, IDE_BLOCK48_INPUT, &block48, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(IDE_get_SMART_status) static inline int
IDE_get_SMART_status (struct diskparam_str *dp)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0xDA,
	.sector_count	= 0,
	.chs_lba = {.chs = { .sector = 0, .cylinder = 0xC24F,
			.drivehead = dp->lba_slave_mask }},
	.command	= IDE_SMART
	};
  unsigned val;

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, &val);
  val = (val >> 8) & 0xFFFF; /* cylinder */

  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }

  if (val == 0xC24F)
      return 0;		/* in good working conditions */
    else if (val == 0x2CF4)
      return 1;		/* in bad conditions */
    else {
      DDBG ((" [%s reads 0x%X] ", __FUNCTION__, val));
      return -IDE_undocumented;
      }
  }

enum SMART_logging_subcmd_enum {
    SMART_logging_std_offline = 0,
    SMART_logging_short_offline = 1,
    SMART_logging_extended_offline = 2,
    /* 3-63 Reserved, 64-126 Vendor specific */
    SMART_logging_abort = 127,
    /* 128 Reserved */
    SMART_logging_short_captive = 129,
    SMART_logging_extended_captive = 130,
    /* 131-191 Reserved, 192-255 Vendor specific */
    };

DISK_FCT_PREFIX(IDE_execute_SMART_logging) static inline int
IDE_execute_SMART_logging (struct diskparam_str *dp,
			   enum SMART_logging_subcmd_enum subcmd)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0xD4,
	.sector_count	= 0,
	.chs_lba = {.chs = { .sector = subcmd, .cylinder = 0xC24F,
		.drivehead = dp->lba_slave_mask }},
	.command	= IDE_SMART
	};

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(IDE_calibratedrive) static inline int
IDE_calibratedrive (struct diskparam_str *dp)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= 0,
	.chs_lba = {.chs = { .sector = !((dp->lba_slave_mask >> 6) & 1),
		.cylinder = 0, .drivehead = dp->lba_slave_mask }},
	.command	= IDE_CALIBRATE_DRIVE
	};

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(IDE_Initialisedrive) static inline int
IDE_Initialisedrive (struct diskparam_str *dp)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= dp->nbsectorpertrack,
	.chs_lba = {.chs = { .sector = 0, .cylinder = 0,
		.drivehead = dp->lba_slave_mask | (dp->nbhead - 1) }},
	.command	= IDE_INITIALISE_DRIVE
	};

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(ideIdentify) static int
ideIdentify (struct diskparam_str *dp, farptr buffer, unsigned *signature)
  {
  enum IDE_error ret;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_IDENTIFY_DEVICE, dp->lba_slave_mask, 0, 1);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= 1,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_IDENTIFY_DEVICE
	};
#endif

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_INPUT, buffer, signature);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(idelba48RWsector) static inline int
idelba48RWsector (struct diskparam_str	*dp,
		unsigned long long	lba,
		unsigned short		number,
		farptr			buffer,
		const unsigned		read)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_addr48_str block48 = {
	.feature	= 0,
	.sector_count	= number,
	.lba = { .lba = lba },
	.drivehead	= dp->lba_slave_mask | 0x40, /* request in LBA */
	.command	= read? IDE_READ_SECTOR_EXT : IDE_WRITE_SECTOR_EXT
	};

  ret = IDE_PIO_data (dp, read? IDE_BLOCK48_INPUT : IDE_BLOCK48_OUTPUT,
			&block48, buffer, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(idelbaRWsector) static inline int
idelbaRWsector (struct diskparam_str	*dp,
		unsigned		lba,
		unsigned char		number,
		farptr			buffer,
		const unsigned		read)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= number,
	{.lba = { .lba = lba, .drv = (dp->lba_slave_mask | 0x40) >> 4 }},
	.command	= read? IDE_READ_SECTOR : IDE_WRITE_SECTOR
	};

  ret = IDE_PIO_data (dp, &block28,
			read? IDE_BLOCK28_INPUT : IDE_BLOCK28_OUTPUT,
			buffer, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(idechsRWsector) static inline int
idechsRWsector (struct diskparam_str	*dp,
		unsigned short		cylinder,
		unsigned char		head,
		unsigned char		sector,
		unsigned char		number,
		farptr			buffer,
		const unsigned		read)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= number,
	.chs_lba = {.chs = { .sector = sector, .cylinder = cylinder,
		.drivehead = (dp->lba_slave_mask & ~0x40) | (head & 0x0F) }},
	.command	= read? IDE_READ_SECTOR : IDE_WRITE_SECTOR
	};

  ret = IDE_PIO_data (dp, &block28,
			read? IDE_BLOCK28_INPUT : IDE_BLOCK28_OUTPUT,
			buffer, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(ideRWbuffer) static inline int
ideRWbuffer (struct diskparam_str	*dp,
		farptr			buffer,
		const unsigned		read)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= 1,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= read? IDE_READ_BUFFER : IDE_WRITE_BUFFER
	};

  ret = IDE_PIO_data (dp, &block28,
			read? IDE_BLOCK28_INPUT : IDE_BLOCK28_OUTPUT,
			buffer, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(IDE_freeze_password) static inline int
IDE_freeze_password (struct diskparam_str *dp)
  {
  enum IDE_error ret;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_SECURITY_FREEZE_LOCK, dp->lba_slave_mask, 0, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= 0,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_SECURITY_FREEZE_LOCK
	};
#endif

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(IDE_freeze_config) static inline int
IDE_freeze_config (struct diskparam_str *dp)
  {
  enum IDE_error ret;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_DEVICE_CONFIGURATION, dp->lba_slave_mask, 0xC1, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0xC1,
	.sector_count	= 0,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_DEVICE_CONFIGURATION
	};
#endif

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(IDE_restore_config) static inline int
IDE_restore_config (struct diskparam_str *dp)
  {
  enum IDE_error ret;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_DEVICE_CONFIGURATION, dp->lba_slave_mask, 0xC0, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0xC0,
	.sector_count	= 0,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_DEVICE_CONFIGURATION
	};
#endif

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, 0);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

/*
 * readback contains at exit, when writing the configuration,
 * the result of _IDE_get_sector_count(), and in case of error
 * it is the location of the error: word location in upper 16 bits,
 * bit location low in lowest byte, bit location high in bits 8-15.
 */
DISK_FCT_PREFIX(ideReadConfig) static int
ideReadConfig (struct diskparam_str *dp, farptr buffer)
  {
  enum IDE_error ret;
  unsigned readback;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_DEVICE_CONFIGURATION, dp->lba_slave_mask, 0xC2, 1);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0xC2,
	.sector_count	= 1,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_DEVICE_CONFIGURATION
	};
#endif

  if (sizeof (ata_config_t) != 512)
      __ERROR();	/* the linker will produce an error here */

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_INPUT, buffer, &readback);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X, "
		"bit location low 0x%X, bit location high 0x%X, word location %u] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp),
		readback & 0xFF, (readback >> 8) & 0xFF, readback >> 16));
      return -(int)ret;
      }
  return 0;
  }

DISK_FCT_PREFIX(ideWriteConfig) static inline int
ideWriteConfig (struct diskparam_str *dp, farptr buffer)
  {
  enum IDE_error ret;
  unsigned readback;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_DEVICE_CONFIGURATION, dp->lba_slave_mask, 0xC3, 1);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0xC3,
	.sector_count	= 1,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_DEVICE_CONFIGURATION
	};
#endif

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_OUTPUT, buffer, &readback);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X, "
		"bit location low 0x%X, bit location high 0x%X, word location %u] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp),
		readback & 0xFF, (readback >> 8) & 0xFF, readback >> 16));
      return -(int)ret;
      }
  return 0;
  }

/*
 * ATAPI stuffing:
 */
#if DISK_SUPPORT & ATAPI_SUPPORT
DISK_FCT_PREFIX(ideAtapiIdentify) static inline int
ideAtapiIdentify (struct diskparam_str *dp, farptr buffer, unsigned *signature)
  {
  enum IDE_error ret;
#ifdef INIT_CMDBLOCK28
  INIT_CMDBLOCK28 (block28, IDE_IDENTIFY_PACKET_DEVICE, dp->lba_slave_mask, 0, 1);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= 1,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command	= IDE_IDENTIFY_PACKET_DEVICE
	};
#endif

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_INPUT, buffer, signature);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }

enum CD_POWER {
    CD_START_VALID = 0,	CD_ACTIVE = 1,
    CD_IDLE = 2,	CD_STANDBY = 3,
    CD_reserved1 = 4,	CD_obscelete = 5,
    CD_reserved2 = 6,	CD_LU_CONTROL = 7,
    CD_reserved3 = 8,	CD_reserved4 = 9,
    CD_FORCE_IDLE_0 = 10,	CD_FORCE_STANDBY_0 = 11
    };

/* Use this one to open/close CDROM door: */
DISK_FCT_PREFIX(ideAtapiStartStopUnit) static inline int
ideAtapiStartStopUnit (struct diskparam_str *dp, enum CD_POWER power,
	unsigned start, unsigned loEj, unsigned Immed)
  {
  enum IDE_error ret;
  struct {
	unsigned operation_code_0x1B	: 8;
	unsigned Immed			: 1;
	unsigned reserved1		: 23;
	unsigned start		: 1;
	unsigned loEj		: 1;
	unsigned reserved3	: 2;
	enum CD_POWER PowerCondition	: 4;
	unsigned reserved4	: 8;

	unsigned filler12_1	: 16;
	unsigned filler12_2;
	} __attribute__((packed)) StartStopUnit_packet = {
	.operation_code_0x1B	= 0x1B,
	.Immed			= Immed,
	.start			= start,
	.loEj			= loEj
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = 0,
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (StartStopUnit_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, &StartStopUnit_packet);
  if (ret != IDE_OK) {
      DDBG (("%s returns '%s', status: 0x%X, error: 0x%X\r\n",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else
      DDBG (("%s done\r\n", __FUNCTION__));
  return 0;
  }

DISK_FCT_PREFIX(ideAtapiTestUnitReady) static int
ideAtapiTestUnitReady (struct diskparam_str *dp)
  {
  enum IDE_error ret;
  /* static const */ struct {
	unsigned operation_code_0x00	: 8;
	unsigned reserved1		: 24;
	unsigned reserved2;
	unsigned reserved3;
	} __attribute__((packed)) TestUnitReady_packet = {
	.operation_code_0x00 = 0x00,
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = 0,
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (TestUnitReady_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, &TestUnitReady_packet);
  if (ret != IDE_OK) {
      DDBG (("%s returns '%s', status: 0x%X, error: 0x%X\r\n",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else
      DDBG (("%s done\r\n", __FUNCTION__));
  return 0;
  }

DISK_FCT_PREFIX(ideAtapiPreventAllowMediumRemoval) static inline int
ideAtapiPreventAllowMediumRemoval (struct diskparam_str *dp, unsigned prevent_player, unsigned prevent_tray)
  {
  enum IDE_error ret;
  struct {
	unsigned char operation_code_0x1E	: 8;
	unsigned char reserved1[3];
	unsigned char prevent_player		: 1;
	unsigned char prevent_tray		: 1; /* only if RMB and Mchngr in INQUIRY */
	unsigned char reserved2			: 6;
	unsigned char reserved3; /* Control */
	unsigned char filler12[6];
	} __attribute__((packed)) PreventAllowMediumRemoval_packet = {
	.operation_code_0x1E = 0x1E,
	.prevent_player = prevent_player,
	.prevent_tray = prevent_tray
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = 0,
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (PreventAllowMediumRemoval_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, &PreventAllowMediumRemoval_packet);
  if (ret != IDE_OK) {
      DDBG (("%s returns '%s', status: 0x%X, error: 0x%X\r\n",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else
      DDBG (("%s done\r\n", __FUNCTION__));
  return 0;
  }

DISK_FCT_PREFIX(ideAtapiSetCdSpeed) static inline int
ideAtapiSetCdSpeed (struct diskparam_str *dp, unsigned read_speed, unsigned write_speed)
  {
  enum IDE_error ret;
  struct {
	unsigned char	operation_code_0xBB;
	unsigned char	reserved1;
	unsigned short	read_speed;	/* big endian, kByte/s, 0xFFFF: max speed */
	unsigned short	write_speed;	/* big endian, kByte/s, 0xFFFF: max speed */
	unsigned short	reserved2;
	unsigned	reserved3;
	} __attribute__((packed)) SetCdSpeed_packet = {
	.operation_code_0xBB = 0xBB,
	.read_speed = ENDIAN16(read_speed),
	.write_speed = ENDIAN16(write_speed),
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = 0,
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (SetCdSpeed_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, &SetCdSpeed_packet);
  if (ret != IDE_OK) {
      DDBG (("%s returns '%s', status: 0x%X, error: 0x%X\r\n",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else
      DDBG (("%s done\r\n", __FUNCTION__));
  return 0;
  }

struct cd_request_sense_str {
    unsigned char response_code	: 7; /* 0x70 or 0x71 */
    unsigned char valid		: 1; /* shall be set */
    unsigned char segment_number;
    unsigned char sense_key	: 4;
    unsigned char reserved	: 1;
    unsigned char ILI		: 1;
    unsigned char EOM		: 1;
    unsigned char filemark	: 1;
    unsigned information; /* big endian */
    unsigned char additional_sense_length;
    unsigned command_specific_information; /* big endian */
    unsigned char ASC;
    unsigned char ASCQ;
    unsigned char field_replaceable_unit_code;
    unsigned char sense_key_specific[3];
    unsigned char additionnal_sense_bytes[7];
    } __attribute__((packed));

//DISK_FCT_PREFIX(ideAtapiRequestSense) static inline int
DISK_FCT_PREFIX(ideAtapiRequestSense) static int
ideAtapiRequestSense (struct diskparam_str *dp, struct cd_request_sense_str *sense)
  {
  enum IDE_error ret;
  /* static const */ struct {
	unsigned char operation_code_0x03;
	unsigned char reserved1[3];
	unsigned char allocation_length;
	unsigned char reserved3; /* Control */
	unsigned char filler12[6];
	} __attribute__((packed)) RequestSense_packet = {
	.operation_code_0x03 = 0x03,
	.allocation_length = sizeof (struct cd_request_sense_str)
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, sizeof (struct cd_request_sense_str));
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = sizeof (struct cd_request_sense_str),
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (RequestSense_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_INPUT, data_adr(sense), &RequestSense_packet);
  if (ret != IDE_OK) {
      DDBG (("%s returns '%s', status: 0x%X, error: 0x%X\r\n",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else {
      sense->information = ENDIAN32(sense->information);
      sense->command_specific_information = ENDIAN32(sense->command_specific_information);
      DDBG (("%s done, sense_key=0x%X ASC=0x%X ASCQ=0x%X ", __FUNCTION__,
		sense->sense_key, sense->ASC, sense->ASCQ));
      if (sense->sense_key == 6 && sense->ASC == 0x29 && sense->ASCQ == 0)
	  DDBG (("(Power ON, Reset or bus device reset occurred)"));
	else if (sense->sense_key == 6 && sense->ASC == 0x28 && sense->ASCQ == 0)
	  DDBG (("(medium changed)"));
	else if (sense->sense_key == 5 && sense->ASC == 0x64 && sense->ASCQ == 0)
	  DDBG (("(illegal mode for this track)"));
	else if (sense->sense_key == 5 && sense->ASC == 0x30 && sense->ASCQ == 2)
	  DDBG (("(cannot read medium - incompatible format)"));
	else if (sense->sense_key == 5 && sense->ASC == 0x24 && sense->ASCQ == 0)
	  DDBG (("(invalid field in CDB)"));
	else if (sense->sense_key == 5 && sense->ASC == 0x21 && sense->ASCQ == 0)
	  DDBG (("(LBA out of range)"));
	else if (sense->sense_key == 3 && sense->ASC == 0x11 && sense->ASCQ == 0)
	  DDBG (("(unrecovered read error)"));
	else if (sense->sense_key == 2 && sense->ASC == 0x3A && sense->ASCQ <= 2)
	  DDBG (("(medium not present, tray %s)", ((const char *[3]){"unknown", "closed", "open"})[sense->ASCQ]));
	else if (sense->sense_key == 2 && sense->ASC == 0x04 && sense->ASCQ == 1)
	  DDBG (("(Logical unit is in process of becoming ready)"));

      DDBG (("\r\n"));
      }
  return 0;
  }

struct cd_inquiry_str {
    unsigned char peripheral_device_type: 5;
    unsigned char peripheral_qualifier	: 3;
    unsigned char reserved1		: 7;
    unsigned char RMB			: 1;
    unsigned char ANSI_version		: 3;
    unsigned char ECMA_version		: 3;
    unsigned char ISO_IEC_version	: 2;
    unsigned char response_data_format	: 4;
    unsigned char reserved2		: 1;
    unsigned char NormACA		: 1;
    unsigned char TmrTsk		: 1;
    unsigned char AERC			: 1;
    unsigned char additional_length;
    unsigned char reserved3;
    unsigned char Addr16		: 1;
    unsigned char Addr32		: 1;
    unsigned char ACKREQQ		: 1;
    unsigned char MChngr		: 1;
    unsigned char MultiP		: 1;
    unsigned char VS1			: 1;
    unsigned char EncServ		: 1;
    unsigned char reserved4		: 1;
    unsigned char VS2			: 1;
    unsigned char CmdQue		: 1;
    unsigned char TranDis		: 1;
    unsigned char Linked		: 1;
    unsigned char Sync			: 1;
    unsigned char WBus16		: 1;
    unsigned char WBus32		: 1;
    unsigned char RelAdr		: 1;
    unsigned char vendor_id[8];
    unsigned char product_id[8];
    unsigned product_revision_level; /* big endian */
    /* optional after offset 36:
    unsigned char vendor_specific[55-36 + 1];
    unsigned char reserved5[95-56 + 1];
    unsigned char vendor_specific_params[0];
     */
    } __attribute__((packed));

DISK_FCT_PREFIX(ideAtapiInquiry) static inline int
ideAtapiInquiry (struct diskparam_str *dp, struct cd_inquiry_str *inquiry)
  {
  enum IDE_error ret;
  /* static const */ struct {
	unsigned char operation_code_0x12;
	unsigned char EVPD		: 1; /* Enable Vital Product Data  = 0 else other cd_inquiry_str */
	unsigned char CmdDt		: 1; /* Command Data  = 0 else other cd_inquiry_str  */
	unsigned char reserved1	: 6;
	unsigned char page_or_operation_code; /* = 0 unless EVPD or CmdDt */
	unsigned char reserved2;
	unsigned char allocation_length;
	unsigned char reserved3; /* Control */
	unsigned char filler12[6];
	} __attribute__((packed)) Inquiry_packet = {
	.operation_code_0x12 = 0x12,
	.allocation_length = sizeof (struct cd_inquiry_str)
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, sizeof (struct cd_inquiry_str));
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = sizeof (struct cd_inquiry_str),
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (Inquiry_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_INPUT, data_adr(inquiry), &Inquiry_packet);
  if (ret != IDE_OK) {
      DDBG (("%s returns '%s', status: 0x%X, error: 0x%X\r\n",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else {
      inquiry->product_revision_level = ENDIAN32(inquiry->product_revision_level);
      /* do not display the strings here, they are not zero terminated...
      DDBG (("%s done, vendor_id='%s' product_id='%s'\r\n", __FUNCTION__,
		inquiry->vendor_id, inquiry->product_id)); */
      DDBG (("%s done\r\n", __FUNCTION__));
      }
  return 0;
  }

struct cd_capacity_str {
    unsigned	lba;
    unsigned	blocksize;
    } __attribute__((packed));

DISK_FCT_PREFIX(ideAtapiReadCdRecordedCapacity) static inline int
ideAtapiReadCdRecordedCapacity (struct diskparam_str *dp, struct cd_capacity_str *cap)
  {
  enum IDE_error ret;
  /* static const */ struct {
	unsigned operation_code_0x25	: 8;
	unsigned RELADR			: 1;
	unsigned reserved1		: 23;
	unsigned lba;	/* big endian */
	unsigned reserved2		: 16;
	unsigned PMI			: 1;
	unsigned reserved3		: 15;
	} __attribute__((packed)) ReadCdRecordedCapacity_packet = {
	.operation_code_0x25 = 0x25,
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, sizeof (struct cd_capacity_str));
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = sizeof (struct cd_capacity_str),
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (ReadCdRecordedCapacity_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_INPUT, data_adr(cap), &ReadCdRecordedCapacity_packet);
  if (ret != IDE_OK) {
      DDBG (("%s returns '%s', status: 0x%X, error: 0x%X\r\n",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else {
      /* BIG -> LITTLE endian */
      cap->lba = ENDIAN32(cap->lba) + 1;
      cap->blocksize = ENDIAN32(cap->blocksize);
      DDBG (("%s done, blocksize %u, lba %U (extra sector included)\r\n", __FUNCTION__, cap->blocksize, cap->lba));
      }
  return 0;
  }

struct cd_header_str {
    unsigned char cd_data_mode;
    unsigned char reserved[3];
    unsigned lba; /* big endian */
    } __attribute__((packed));

DISK_FCT_PREFIX(ideAtapiReadHeader) static inline int
ideAtapiReadHeader (struct diskparam_str *dp, unsigned lba, struct cd_header_str *lbaheader)
  {
  enum IDE_error ret;
  struct {
	unsigned char operation_code_0x44	: 8;
	unsigned char reserved1	: 1;
	unsigned char MSF		: 1;
	unsigned char reserved2	: 3;
	unsigned char reserved3	: 3;
	unsigned lba;	/* big endian */
	unsigned char reserved4;
	unsigned char allocation_length_msb;
	unsigned char allocation_length_lsb;
	unsigned char control;
	unsigned short	filler12;
	} __attribute__((packed)) ReadHeader_packet = {
	.operation_code_0x44 = 0x44,
	.lba = ENDIAN32(lba),
	.allocation_length_lsb = sizeof (struct cd_header_str),
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, sizeof (struct cd_header_str));
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = sizeof (struct cd_header_str),
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (ReadHeader_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_INPUT, data_adr(lbaheader), &ReadHeader_packet);
  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else {
      lbaheader->lba = ENDIAN32 (lbaheader->lba);
      DDBG ((" [%s done, cd_data_mode %u, lba %U] ", __FUNCTION__, lbaheader->cd_data_mode, lbaheader->lba));
      }
  return 0;
  }


struct cd_TOC_str {
    unsigned short	data_length;	/* all excliding itself */
    unsigned char	first_track_number;
    unsigned char	last_track_number;
    struct cd_track_str {
	unsigned char	reserved;
	unsigned char	control	: 4;
	unsigned char	adr	: 4;
	unsigned char	track_number;
	unsigned char	reserved2;
	unsigned	absolute_CDROM_address;
	} __attribute__ ((packed)) track[30]; /* variable size array */
    } __attribute__((packed));

DISK_FCT_PREFIX(ideAtapiReadToc) static inline int
ideAtapiReadToc (struct diskparam_str *dp, struct cd_TOC_str *toc, unsigned starting_track)
  {
  enum IDE_error ret;
  struct {
	unsigned char	operation_code_0x43;
	unsigned char	reserved1	: 1;
	unsigned char	MSF		: 1;
	unsigned char	reserved2	: 3;
	unsigned char	logical_unit_number : 3;
	unsigned	reserved3;
	unsigned char	starting_track;
	unsigned char	allocation_length_msb;
	unsigned char	allocation_length_lsb;
	unsigned char	control;
	unsigned short	filler12;
	} __attribute__((packed)) ReadToc_packet = {
	.operation_code_0x43 = 0x43,
	.starting_track = starting_track,
	.allocation_length_lsb = sizeof (struct cd_TOC_str)
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, sizeof (struct cd_TOC_str));
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = sizeof (struct cd_TOC_str),
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (ReadToc_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_INPUT, data_adr(toc), &ReadToc_packet);
  if (ret != IDE_OK) {
      DDBG (("%s returns '%s', status: 0x%X, error: 0x%X\r\n",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else {
      struct cd_track_str *ptr;

      toc->data_length = ENDIAN16(toc->data_length);
      DDBG (("%s done, data_length %u (max %u), first_track_number %u, last_track_number %u\r\n",
		__FUNCTION__, toc->data_length, sizeof (struct cd_TOC_str), toc->first_track_number, toc->last_track_number));
      if (toc->data_length > sizeof (struct cd_TOC_str))
	  toc->data_length = sizeof (struct cd_TOC_str);
      for (ptr = toc->track; ptr - toc->track < toc->data_length; ptr++)
	  ptr->absolute_CDROM_address = ENDIAN32 (ptr->absolute_CDROM_address);
      }
  return 0;
  }

#undef COMPATIBILITY_SCSI_READ	/* No DVD read if defined */

DISK_FCT_PREFIX(ideAtapiRWsector) static inline int
ideAtapiRWsector (struct diskparam_str	*dp,
		    unsigned			lba,
		    unsigned char		number,
		    farptr			buffer,
		    const unsigned char		sector_type,
		    const unsigned		read)
  {
  enum IDE_error ret;
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit0: no overlapped, bit1: no DMA */
	.sector_count	= 0, /* bit3..7: no tag */
	.chs_lba = {.chs = {
		.sector = 0,	/* n/a */
		.cylinder = 2048 * number, /* so read 31 sectors max! */
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
  union {
      struct read_cd_str {
#ifdef COMPATIBILITY_SCSI_READ
	  unsigned char operation_code_0xBE;
	  unsigned char RELADR			: 1;
	  unsigned char reserved1		: 1;
	  enum sector_type_enum	sector_type	: 3;
	  unsigned char reserved2		: 3;
	  unsigned lba; /* big endian */
	  unsigned char len16_23;	/* MSB of len */
	  unsigned short len; /* big endian */
	  unsigned char reserved3	: 1;
	  enum {
	      none_error	= 0,
	      C2only		= 1,
	      C2andBLOCK	= 2,
//	      reserved_error	= 3
	      } error_field		: 2;
	  unsigned char ECC_EDC		: 1;
	  unsigned char user_data	: 1;
	  enum {
	      header_none	= 0,
	      header_only	= 1,
	      subheader_only	= 2,
	      header_all	= 3
	      } header_code		: 2;
	  unsigned char SYNC		: 1;
	  enum {
	      no_sub_channel_data		= 0,
	      raw_sub_channel_data		= 1,
	      Q_sub_channel_data		= 2,
	      reserved_sub_channel_data	= 3,
	      PW_sub_channel_data		= 4
	      } sub_channel_data	: 3;
	  unsigned char reserved4	: 5;
#else /* COMPATIBILITY_SCSI_READ */
	  unsigned char operation_code_0x28;
	  unsigned char reserved1;
	  unsigned lba; /* big endian */
	  unsigned char reserved2;
	  unsigned short len; /* big endian */
#endif /* COMPATIBILITY_SCSI_READ */
	  struct atapi_control_str {
	      unsigned char link : 1;
	      unsigned char flag : 1; /* obscelete */
	      unsigned char NACA : 1; /* Normal Auto Contingent Allegiance: too much B.E.E.R. */
	      unsigned char reserved	  : 3;
	      unsigned char vendor_specific : 2;
	      } __attribute__((packed)) control;
#ifndef COMPATIBILITY_SCSI_READ
	  unsigned char filler12[2];
#endif
	  } __attribute__((packed)) ReadCd;
      struct write_cd_str {
	  unsigned char operation_code_0x2A;
	  unsigned char RELADR		: 1;
	  unsigned char reserved1	: 2;
	  unsigned char FUA		: 1; /* force unit access */
	  unsigned char DPO		: 1; /* Disable Page Out */
	  unsigned char reserved2	: 3;
	  unsigned lba; /* big endian */
	  unsigned char reserved3;
	  unsigned short len; /* big endian */
	  unsigned char control;
	  unsigned char filler12[2];
	  } __attribute__((packed)) WriteCd;
      } RWsector_packet;

  if (sizeof (RWsector_packet.ReadCd) != 12)
      __ERROR();        /* the linker will produce an error here */
  if (sizeof (RWsector_packet.WriteCd) != 12)
      __ERROR();        /* the linker will produce an error here */

  if (read) {
      RWsector_packet.ReadCd = (struct read_cd_str) {
#ifdef COMPATIBILITY_SCSI_READ
	  .operation_code_0xBE = 0xBE,
	  .RELADR = 0, /* it shall, what is the meaning of "shall" these days? */
	  .sector_type = sector_type,
	  .len16_23  = (number >> 16) & 0xFF,
	  .error_field = none_error,
	  .ECC_EDC = 0, /* include 8 bytes of pad data for mode 1 if set */
	  .user_data = 1,
	  .header_code = header_none,
	  .SYNC = 0,	  /* include the sync bit if set */
	  .sub_channel_data = no_sub_channel_data,
#else /* COMPATIBILITY_SCSI_READ */
	  .operation_code_0x28 = 0x28,
	  .reserved1 = 0,
	  .reserved2 = 0,
#endif /* COMPATIBILITY_SCSI_READ */
	  .lba = ENDIAN32(lba),
	  .len = ENDIAN16(number),
	  };
      }
    else
      RWsector_packet.WriteCd = (struct write_cd_str) {
	.operation_code_0x2A = 0x2A,
	.RELADR = 0,
	.FUA = 0,	/* do not physically write immediately if 0 */
	.DPO = 0,	/* standard caching system if 0 */
	.lba = ENDIAN32(lba),
	.len = ENDIAN16(number),
	};

  ret = IDE_PIO_data (dp, &block28, read? IDE_BLOCK28_INPUT : IDE_BLOCK28_OUTPUT, buffer, &RWsector_packet);

  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
//    else
//      DDBG (("%s done\r\n", __FUNCTION__));

  return 0;
  }

/* Never seen those two working - only for CD changer? */
DISK_FCT_PREFIX(ideAtapiLoadUnloadCd) static inline int
ideAtapiLoadUnloadCd (struct diskparam_str *dp, unsigned open)
  {
  enum IDE_error ret;
  struct {
	unsigned operation_code_0xA6	: 8;
	unsigned Immed			: 1;
	unsigned reserved1		: 23;

	unsigned start		: 1;
	unsigned LoUnlo		: 1;
	unsigned reserved2	: 30;

	unsigned slot		: 8;
	unsigned reserved3	: 24;
	} __attribute__((packed)) LoadUnloadCd_packet = {
	.operation_code_0xA6 = 0xA6,
	.Immed = 0, /* return before full completion of load/unload when set */
	.LoUnlo = 1, /* else abort previous load/unload if start also equal 0 */
	.start = !open, /* i.e. load (unload when 0, then slot ignored) */
	.slot = 0 /* not a CD changer */
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, 0);
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = 0,
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (LoadUnloadCd_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_non_data_command (dp, &block28, IDE_BLOCK28_INPUT, &LoadUnloadCd_packet);
  if (ret != IDE_OK) {
      DDBG (("%s returns '%s', status: 0x%X, error: 0x%X\r\n",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else
      DDBG (("%s done\r\n", __FUNCTION__));
  return 0;
  }

struct cd_mechanism_status_str {
    unsigned char	current_slot	: 5;
    enum { chgr_ready = 0,
	   chgr_load_in_progress = 1, chgr_unload_in_progress = 2,
	   chgr_initialising = 3
	   }		changer_state	: 2;
    unsigned char	fault		: 1;
    unsigned char	reserved1	: 4;
    unsigned char	door_open	: 1;
    enum { mech_idle = 0,
	   mech_playing = 1, mech_scanning = 2,
	   mech_initialising = 7	/* other: reserved */
	   }		cd_mech_state	: 3;
    unsigned char	current_lba_msb;
    unsigned char	current_lba;
    unsigned char	current_lba_lsb;
    unsigned char	nb_slot_avail	: 6;
    unsigned char	reserved2	: 2;
    unsigned char	length_slot_table_msb;
    unsigned char	length_slot_table_lsb;
    struct {
	unsigned char	disk_changed	: 1;
	unsigned char	reserved1	: 6;
	unsigned char	disk_present	: 1;
	unsigned char	reserved2[3];
	} __attribute__((packed)) slot_table[1]; /* variable length */
    } __attribute__((packed));

DISK_FCT_PREFIX(ideAtapiMechanismStatus) static inline int
ideAtapiMechanismStatus (struct diskparam_str *dp, struct cd_mechanism_status_str *mech)
  {
  enum IDE_error ret;
  /* static const */ struct {
	unsigned char	operation_code_0xBD;
	unsigned char	reserved1[7];
	unsigned char	allocation_length_msb;
	unsigned char	allocation_length_lsb;
	unsigned short	reserved2;
	} __attribute__((packed)) MechanismStatus_packet = {
	.operation_code_0xBD = 0xBD,
	.allocation_length_lsb = sizeof (struct cd_mechanism_status_str)
	};
#ifdef INIT_PKTCMDBLOCK
  INIT_PKTCMDBLOCK (block28, dp->lba_slave_mask, sizeof (struct cd_mechanism_status_str));
#else
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0, /* bit1: no overlapped, bit0: no DMA */
	.sector_count	= 0,
	.chs_lba = {.chs = {
		.sector = 0, /* bit3..7: no tag */
		.cylinder = sizeof (struct cd_mechanism_status_str),
		.drivehead = dp->lba_slave_mask
		}},
	.command	= IDE_PACKET
	};
#endif

  if (sizeof (MechanismStatus_packet) != 12)
      __ERROR();        /* the linker will produce an error here */

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_INPUT, data_adr(mech), &MechanismStatus_packet);
  if (ret != IDE_OK) {
      DDBG (("%s returns '%s', status: 0x%X, error: 0x%X\r\n",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(char)ret;
      }
    else {
      DDBG (("%s done, current_slot %u, changer_state %s, fault: %u, door_open: %u, cd_mech_state: %u,\r\n",
	      __FUNCTION__, mech->current_slot,
	      ((const char *const[4]) {"ready", "load_in_progress", "unload_in_progress", "initialising"})[mech->changer_state],
	      mech->fault, mech->door_open, mech->cd_mech_state));
      DDBG (("  current_lba %u, nb_slot_avail %u, length_slot_table %u, disk_changed[0] %u, disk_present[0] %u\r\n",
		0x10000 * mech->current_lba_msb + 0x100 * mech->current_lba + mech->current_lba_lsb,
		mech->nb_slot_avail, 0x100 * mech->length_slot_table_msb + mech->length_slot_table_lsb,
		mech->slot_table[0].disk_changed, mech->slot_table[0].disk_present));
      }
  return 0;
  }

/* Just suggar functions: */
DISK_FCT_PREFIX(ideAtapiOpen) static inline int
ideAtapiOpen (struct diskparam_str *dp)
  {
  return ideAtapiStartStopUnit (dp, CD_START_VALID, 0, 1, 0);
  }

DISK_FCT_PREFIX(ideAtapiClose) static inline int
ideAtapiClose (struct diskparam_str *dp)
  {
  return ideAtapiStartStopUnit (dp, CD_START_VALID, 1, 1, 0);
  }

/* 176 Kb/s is CDROM read "times 1", is that different for DVD? */
DISK_FCT_PREFIX(ideAtapiSpeed) static inline int
ideAtapiSpeed (struct diskparam_str *dp, unsigned read_times, unsigned write_times)
  {
  return ideAtapiSetCdSpeed (dp, 176 * read_times, 176 * write_times);
  }

DISK_FCT_PREFIX(ideAtapiMaxSpeed) static inline int
ideAtapiMaxSpeed (struct diskparam_str *dp)
  {
  return ideAtapiSetCdSpeed (dp, 0xFFFF, 0xFFFF);
  }

DISK_FCT_PREFIX(print_request_sense) static inline void
print_request_sense (struct diskparam_str *dp)
  {
  struct cd_request_sense_str sense = {};

  printf ("ideAtapiRequestSense returns %d", ideAtapiRequestSense (dp, &sense));
  printf (" i.e. valid %u segment_number 0x%X response_code 0x%X sense_key 0x%X ASC 0x%X ASCQ 0x%X\r\n",
	sense.valid, sense.segment_number, sense.response_code, sense.sense_key, sense.ASC, sense.ASCQ);
  printf ("  reserved %u, ILI %u, EOM %u, filemark %u information 0x%X additional_sense_length %u\r\n",
	sense.reserved, sense.ILI, sense.EOM, sense.filemark, sense.information, sense.additional_sense_length);
  printf ("  command_specific_information 0x%X, field_replaceable_unit_code 0x%X, sense_key_specific 0x%X 0x%X 0x%X\r\n",
	sense.command_specific_information, sense.field_replaceable_unit_code,
	sense.sense_key_specific[0], sense.sense_key_specific[1], sense.sense_key_specific[2]);
  if (sense.additional_sense_length > 10)
      printf ("  additionnal_sense_bytes: 0x%X 0x%X 0x%X 0x%X 0x%X 0x%X 0x%X\r\n",
	sense.additionnal_sense_bytes[0], sense.additionnal_sense_bytes[1], sense.additionnal_sense_bytes[2],
	sense.additionnal_sense_bytes[3], sense.additionnal_sense_bytes[4], sense.additionnal_sense_bytes[5],
	sense.additionnal_sense_bytes[6]);
  }
#endif /* ATAPI_SUPPORT */

#if (USER_SUPPORT != 0) && (DISK_SUPPORT & IDE_SUPPORT)
/* READ FIRST - BEFORE READING THIS FUNCTION -
	ideSecurity() use in main.c and the comment before its use. */
/* This only set security with level to high */
awk_farret (ideSecurity);
DISK_FCT_PREFIX(ideSecurity) int
ideSecurity (struct diskparam_str *dp, char password[32],
		enum IDE_security_enum action)
  {
  enum IDE_error ret;
  unsigned cpt;
  /* do not use fourKbuffer for buffer if called from unknown universe: */
  struct password_s {
      struct {
	  unsigned short identifier	: 1; /* 0: user password */
	  unsigned short reserved1	: 7;
	  unsigned short security	: 1; /* 0 high, 1 max */
	  unsigned short reserved2	: 7;
	  } __attribute__((packed)) control;
      unsigned char password[32];
      unsigned short master_password_revision;
      unsigned short reserved[(512 - 32 - 4)/2];
      } __attribute__((packed)) buffer = {};
  static const unsigned char cmdarray[] = {
      [IDE_security_unlock_user/2] = IDE_SECURITY_UNLOCK,
      [IDE_security_set_pw_user/2] = IDE_SECURITY_SET_PASSWORD,
      [IDE_security_disable_user/2] = IDE_SECURITY_DISABLE_PASSWORD,
      [IDE_security_erase_user/2] = IDE_SECURITY_ERASE_UNIT
      };
  struct IDE_feature_to_cmd_str block28 = {
	.feature	= 0,
	.sector_count	= 1,
	.chs_lba = { .chs = { .drivehead = dp->lba_slave_mask }},
	.command = cmdarray[action >> 1]
	};

  if (sizeof (buffer) != 512)
      __ERROR();	/* the linker will produce an error here */

  cpt = sizeof (buffer.password);
  while (cpt --) {
      buffer.password[cpt] = password[cpt];
      password[cpt] = 0;
      }
  if (   action == IDE_security_set_pw_master
      && dp->ide_master_password_revision != 0xFFFF)
      buffer.master_password_revision = dp->ide_master_password_revision;
  buffer.control.identifier = action & 1;

#if 0
  printf ("\r\nideSecurity cmd 0x%X called with complete buffer (hexadecimal):", block28.command);
  for (cpt = 0; cpt < /*512*/ 64; cpt++) {
      extern const char *const itoa_array;
      unsigned char tmp = ((char *)&buffer)[cpt];
      printf ((cpt % 16 == 0)? "\r\n %c%c": ", %c%c", itoa_array[tmp >> 4], itoa_array[tmp & 0x0F]);
      }
  printf ("\r\n");
#endif

  ret = IDE_PIO_data (dp, &block28, IDE_BLOCK28_OUTPUT,
			stack_adr(&buffer), 0);

  cpt = sizeof (struct password_s) / 4;
  while (cpt --)
      ((unsigned *)&buffer)[cpt] = 0;

  IDE_reset_DCR_for_HTP_BIOS (dp);

  if (ret != IDE_OK) {
      DDBG ((" [%s returns '%s', status: 0x%X, error: 0x%X] ",
		__FUNCTION__, errmsg[ret],
		_IDE_get_altstatus (dp), _IDE_get_error (dp)));
      return -(int)ret;
      }
  return 0;
  }
#endif

DISK_FCT_PREFIX(ide_abscent) static inline unsigned
ide_abscent (unsigned short IOadr)
  {
  unsigned char tmp, status;
  unsigned timeout = 0x10000;

  /* This is only called for ISA standard addresses, so we can do: */
  unsigned short ideIOctrladr = _IDE_ISA_guess_altadr (IOadr);

#if 0
  /* read 0 on some oldish IDE disks (220 Mbytes) if not BIOS initialised,
     and problem of the 4 disks per IDE channel: */
  if (((tmp = _IDE_get_drivehead (IOadr)) & 0xA0) != 0xA0) {
      /* two bit set permanently */
      DDBG (("no IDE present, drive/head reg = 0x%X", tmp));
      return 1;
      }
#endif

#if 0 && (DEBUG & DEBUG_DISK)
  {
  unsigned cpt = 8, ioscan = ideIOctrladr;

  DDBG (("IDE scanning: "));
  while (cpt--) {
      unsigned nbcycle, val;
      unsigned long long tsc = rdtsc();
      val = inb (ioscan);
      nbcycle = (unsigned)(rdtsc() - tsc);
      DDBG (("read 0x%X at 0x%X in %u cycle, ", val, ioscan, nbcycle));
      ioscan = IOadr + cpt;
      }
  DDBG (("\r\n"));
  }
#endif

  /* _IDE_get_altstatus() do not clear a possible interrupt pending
     unlike _IDE_get_status(), but they should be the same... */
  status = inb (IOadr + 7);
  tmp = inb (ideIOctrladr);
  if (tmp != status) {
      DDBG (("no IDE present, two status not equal (firsts IDE): status = 0x%X, altstatus = 0x%X ", status, tmp));
      return 1;
      }

  /* We test here the BSY bit on the alt-status register: */
  while (  ((tmp = inb (ideIOctrladr)) & 0x80)
	 && --timeout) ;

  if (timeout == 0) {
      DDBG (("no IDE present, altstatus register = 0x%X (status register = 0x%X) ", tmp, status));
      return 1;
      }

#if 0
  /* New UDMA disks/IDE do no more map this register in I/O space: */
  if (((tmp = _IDE_get_card_addr_reg (ideIOctrladr)) & 0x03) == 3) {
      /* at least 1 drive selected */
      DDBG (("no IDE present, Card Address Register = 0x%X", tmp));
      return 1;
      }
#endif

  DDBG (("seems present. "));
  return 0;
  }

#endif /* IDE_SUPPORT */

/**
 ** The generic function(s) to read or write disks:
 ** Keep the number of sectors low, below 256 - or even
 ** 127? IDE48 could handle 16 bits sectors.
 **/
DISK_FCT_PREFIX(DISK_RWsector) static
#ifndef WRITE_ENABLE
inline /* to get "read" parameter optimised away */
#endif
unsigned char
DISK_RWsector (struct diskparam_str *dp, int partition,
		unsigned long long lba,
		unsigned number, farptr buffer,
		const unsigned read)
  {
  unsigned char returned;

  DDBG (("%s %s disk %s %u sector at %llU ",
	__FUNCTION__, read? "read" : "write", dp->diskname, number, lba));
  if (partition >= 0)
      DDBG (("(of partition %d) ", partition));
  if (partition >= dp->nbpartition) {
      DDBG (("too high partition: max %u.\r\n", dp->nbpartition));
      return 2;
      }
  if (dp->fulldisk_offset)
      DDBG (("(MBR offset %llU) ", dp->fulldisk_offset));

  if (partition >= 0) {
      if (lba >= dp->partition[partition].length) {
	  DDBG ((": lba over partition limit (max %llU).\r\n", dp->partition[partition].length));
	  dp->error_log.access_over_partition_tryed = 1;
	  return 1;
	  }
      lba += dp->partition[partition].start;
      }

  if (lba >= dp->nbtotalsector) {
      if (partition >= 0)
	  DDBG ((": too high LBA %llU + %llU = %llU, max %llU.\r\n",
			lba, dp->partition[partition].start,
			lba + dp->partition[partition].start,
			dp->nbtotalsector));
	else
	  DDBG ((": too high LBA %llU (max %llU).\r\n", lba, dp->nbtotalsector));
      dp->error_log.access_over_disk_tryed = 1;
      return 2;
      }

  if (!(partition >= 0 && dp->partition[partition].misc.beer_partition)) {
      /* Add fulldisk_offset if valid for every access but on BEER partition */
      if (dp->fulldisk_size != 0 && lba + dp->fulldisk_offset >= dp->fulldisk_size) {
	  DDBG ((": too high FullLBA: %llU + %llU >= %llU.\r\n",
		lba, dp->fulldisk_offset, dp->fulldisk_size));
	  return 3;
	  }
	else
	  lba += dp->fulldisk_offset;
      }

#ifdef TREAT_EXCEPTION
  /* The HTP bug when not handled, and some power saving mode wakeup errors:
	Also new ultra-high speed DVD drive cannot read a sector of a CDROM in
	so short time as 5 seconds... */
  if (shit_handler (1, 5) != 0) {
      DDBG ((" EXCEPTION/timeout 5 second RECEIVED!\r\n"));
      if (dp->access == ebios_lba || dp->access == bios_chs) {
	  unsigned char status;

// There is another BUG in HTP BIOS, and I do not know how what it is exactly,
// it is running dbgdisk.exe without the IDE probe:
// {strange: NbHead 16 != DI.param[disk].bios_nbhead 64}
// DISK_RWsector read disk 0x85 8 sector at 303 (of partition 0):  EXCEPTION/timeout 5 second RECEIVED!
// Reseting drive 0x85: OK.
// DISK_RWsector read disk 0x85 8 sector at 303 (of partition 0): OK
// but after:
// Probing 0x82: EBIOSDISK_probe failed.
// Probing 0x83: EBIOSDISK_probe failed.
// Probing 0x84: EBIOSDISK_probe failed.
// Probing 0x85: EBIOSDISK_probe failed.
// That is with disk 0x80 protected by password (and not unlocked)
// Check if commenting that is better:
//	  DDBG (("Reseting drive 0x%X: ", dp->disknb));
//	  if (_BIOSDISK_resetHD (dp->disknb, &status)) {
//	      DDBG (("failed: 0x%X.\r\n", status));
	      DDBG (("Reseting all drives: "));
	      if (_BIOSDISK_reset_alldisks(&status))
		  DDBG (("failed: 0x%X.\r\n", status));
		else
		  DDBG (("OK\r\n"));
//	      }
//	    else
//	      DDBG (("OK.\r\n"));
	  }
#if DISK_SUPPORT & ATAPI_SUPPORT
	else if (dp->access == hardide_atapi) {
	  unsigned cpt = 20;
	  struct cd_request_sense_str sense;

	  DDBG (("[General timeout, send ATAPI device reset now to lba_slave_mask 0x%X] ", dp->lba_slave_mask));
	  _IDE_set_drivehead (dp, dp->lba_slave_mask);
	  _IDE_get_altstatus_bits(dp); /* or use "short_delay();" ? */
	  _IDE_start_command (dp, IDE_DEVICE_RESET);
	  _IDE_get_altstatus_bits(dp); /* or use "short_delay();" ? */
	  DDBG (("[after reset and altstatus, status 0x%X readback= 0x%X] ", _IDE_get_altstatus (dp), _IDE_readback_lba28 (dp)));
	  for (;;) {
	      if (--cpt == 0)   /* 100 ms, max wait for CDROM: 0.5 second */
		  break;
	      _BIOS_wait (100000);
	      if (ideAtapiTestUnitReady(dp) == 0)
		  break;
	      ideAtapiRequestSense (dp, &sense);
	      DDBG (("ideAtapiTestUnitReady failed, retry in 100 ms... "));
	      }
	  }
#endif
      return ERROR_PLEASE_RETRY;
      }
#endif

#if DISK_SUPPORT & EBIOS_SUPPORT
  if (dp->access == ebios_lba) {
      unsigned short nbsector_really_processed;
      returned = _EBIOSDISK_RWsector (dp->disknb, lba, number, buffer, read, &nbsector_really_processed);
      if (number != nbsector_really_processed)
	  DBG ((" [nbsector successfully read/written %u] ", nbsector_really_processed));
      if (returned == 0x11) {
	  DDBG ((" <ECC recovered data read> "));
	  returned = 0;
	  }
      if (returned) {
	  unsigned char status;
	  DDBG ((": error 0x%X: ", returned));
	  BIOSDISK_error (dp->disknb, returned);
#if 1
/*
DISK_RWsector read disk EBIOS 0x82 1 sector at 0 : error 0x80: timeout, drive not ready
analysebootsector: READ master boot sector FAILED 0x80 !
....
DISK_RWsector read disk EBIOS 0x83 1 sector at 0  EXCEPTION/timeout 5 second RECEIVED!
Reseting all drives: OK
DISK_RWsector read disk EBIOS 0x83 1 sector at 0 : OK.
*/
	  DDBG (("Reseting all drives: "));
	  if (_BIOSDISK_reset_alldisks(&status))
	      DDBG (("failed: 0x%X.\r\n", status));
	    else
	      DDBG (("OK\r\n"));
#endif
	  }
	else
	  DDBG ((": OK.\r\n"));
      }
    else
#endif /* EBIOS_SUPPORT */

#if DISK_SUPPORT & IDE_SUPPORT
	 if (dp->access == hardide_lba48) {
      returned = -idelba48RWsector (dp, lba, number, buffer, read);
      IDE_reset_DCR_for_HTP_BIOS (dp);
      if (returned)
	  DDBG ((": [48] error %s\r\n", errmsg[returned]));
	else
	  DDBG ((": OK.\r\n"));
      if (returned == IDE_busy_drq_timeout)
	  returned = ERROR_PLEASE_RETRY;
      }
    else if (dp->access == hardide_lba) {
      returned = -idelbaRWsector (dp, lba, number, buffer, read);
      IDE_reset_DCR_for_HTP_BIOS (dp);
      if (returned)
	  DDBG ((": [28] error %s\r\n", errmsg[returned]));
	else
	  DDBG ((": OK.\r\n"));
      if (returned == IDE_busy_drq_timeout)
	  returned = ERROR_PLEASE_RETRY;
      }
    else
#endif /* IDE_SUPPORT */
#if DISK_SUPPORT & ATAPI_SUPPORT
	 if (dp->access == hardide_atapi) {
      unsigned char sector_type = (partition >= 0)? dp->partition[partition].type : 0;
      struct cd_request_sense_str sense;

      if (dp->bytepersector == 512) { /* CDROM with MBR of 512 sectors */
	  if (lba & 3)
	      DDBG (("CDROM 512 bytes/sector, lba %llU non multiple of 4\r\n", lba));
	  if (number & 3)
	      DDBG (("CDROM 512 bytes/sector, nbsector %u non multiple of 4\r\n", number));
	  lba >>= 2;
	  number >>= 2;
	  }

#ifdef COMPATIBILITY_SCSI_READ
      DDBG ((" [read sector type 0x%X] ", sector_type));
#endif
      /* Greetings Professor Falken.
	 A strange game. The only winning move is not to play.
	 How about a nice game of chess? */
      for (;;) {
	/* Over an ATAPI interface, max 65535 bytes can be read at once, so max 31
		sectors of 2048 bytes... use max 16 sectors */
	  unsigned max_partial = dp->error_log.read_media_retried? 1 : 16;
	  unsigned partial = (number > max_partial)? max_partial : number;
	  returned = -ideAtapiRWsector (dp, lba, partial, buffer, sector_type, read);
	  if (returned || number == partial)
	      break;
	  number -= partial;
	  lba += partial;
	  buffer += (partial << 16) * (2048/16);
	  }

      if (returned) {
	  unsigned cpt = 20;

#ifdef TREAT_EXCEPTION
	  shit_handler (0, 5); /* following will take some time (can be called twice)... */
#endif
	  DDBG ((": [ATAPI] error %s\r\n", errmsg[returned]));
/*  [input PACKET first 2048 of 4096 bytes] [ATAPI wait next packet]
    [ERROR timeout to get intermediate BSY inactive, status 0xD0]
    [ideAtapiRWsector returns 'IDE data busy timeout', status: 0xD0, error: 0x0] :
	 [ATAPI] error IDE data busy timeout (when reading BEER sector at end of disk) */
	  if (returned != IDE_busy_timeout && returned != IDE_data_busy_timeout)
	      ideAtapiRequestSense (dp, &sense);
	  /* If a DVD-RAM answer randomly IDE_cmderror while reading a sector, we should retry - after reset ATAPI */

	  DDBG (("[IDE[_data]_busy_timeout, send ATAPI device reset now to lba_slave_mask 0x%X] ", dp->lba_slave_mask));
	  _IDE_set_drivehead (dp, dp->lba_slave_mask);
	  _IDE_get_altstatus_bits(dp); /* or use "short_delay();" ? */
	  _IDE_start_command (dp, IDE_DEVICE_RESET);
	  _IDE_get_altstatus_bits(dp); /* or use "short_delay();" ? */
	  DDBG (("[after reset and altstatus, status 0x%X readback= 0x%X] ", _IDE_get_altstatus (dp), _IDE_readback_lba28 (dp)));
	  for (;;) {
	      if (--cpt == 0)   /* 100 ms, max wait for CDROM: 2 second */
		  break;
	      _BIOS_wait (100000);
	      if (ideAtapiTestUnitReady(dp) == 0) {
		  returned = ERROR_PLEASE_RETRY;
		  break;
		  }
	      ideAtapiRequestSense (dp, &sense);
	      DDBG (("ideAtapiTestUnitReady failed, retry in 100 ms... "));
	      }
	  }
	else
	  DDBG ((": OK.\r\n"));
      IDE_reset_DCR_for_HTP_BIOS (dp);
      }
    else
#endif /* ATAPI_SUPPORT */

	if (dp->nbhead == 0 || dp->nbsectorpertrack == 0) {
      DDBG ((": cannot acces CHS disk with geometry nbhead: %u, nbsectorpertrack %u\r\n",
			dp->nbhead, dp->nbsectorpertrack));
      returned = 4;
      }
    else { /* start of C/H/S area */
      unsigned cylinder, head, sector;

      lba2chs (lba, &cylinder, &head, &sector,
		dp->nbhead, dp->nbsectorpertrack);

      if (cylinder >= dp->nbcylinder) {
	  DDBG ((": error too high cylinder: %u (max %u)\r\n",
			cylinder, dp->nbcylinder));
	  returned = 1;
	  }
	else
	  {
#if DISK_SUPPORT & IDE_SUPPORT
	  if (dp->access == hardide_chs) {
	      returned = -idechsRWsector (dp, cylinder, head, sector,
						number, buffer, read);
	      IDE_reset_DCR_for_HTP_BIOS (dp);
	      if (returned)
		  DDBG ((": error %s\r\n", errmsg[returned]));
		else
		  DDBG ((": OK.\r\n"));
	      if (returned == IDE_busy_drq_timeout)
		  returned = ERROR_PLEASE_RETRY;
	      }
	    else
#endif /* IDE_SUPPORT */

#if DISK_SUPPORT & BIOS_SUPPORT
#if !(DISK_SUPPORT & NO_FLOPPY_SUPPORT)
		  if (dp->disknb < 0x80) {
	      DDBG (("BIOS FD read: "));

	      if (cylinder >= 82 && dp->drivetype >= 1 && dp->drivetype <= 6) {
		  /* Floppy driver can physically break here... */
		  /* Take care USB FDD keys with C/H/S = 1017/2/60 */
		  DBG (("Aborting read/write of floppy disk with "
			"cylinder higher than 82 !\r\n"));
		  returned = 1;
		  }
		else
		  returned = biosFD_RWsector (dp, dp->disknb,
				cylinder, head, sector, number,
				buffer, read);
	      if (returned == 0x11) {
		  DDBG ((" <ECC recovered data read> "));
		  returned = 0;
		  }
	      if (returned != 0) {
		  DDBG (("error 0x%X: ", returned));
		  BIOSDISK_error (dp->disknb, returned);
		  }
		else
		  DDBG (("OK.\r\n"));
	      }
	    else
#endif /* NO_FLOPPY_SUPPORT */
	      {
	      DDBG (("BIOS HD read: "));

	      if ((cylinder & 0xC00) != 0 && dp->nbhead <= 16) {
		  /* old bios with 4096 sectors, elsewhere compatible */
		  head |= ((cylinder & 0x0C00) >> 4);
		  cylinder &= 0x3FF;
		  DDBG ((" [Adjusting cylinder/head for strange BIOS] "));
		  }

	      unsigned char nbprocessed;
	      returned = _BIOSDISK_RWsector (dp->disknb,
				cylinder, head, sector, number,
				buffer, read, &nbprocessed);
	      if (nbprocessed != number)
		  DBG((" [nbsector successfully read/written %u] ", nbprocessed));

	      if (returned == 0x11) {
		  DDBG ((" <ECC recovered data read> "));
		  returned = 0;
		  }
	      if (returned == 0x80 || returned == 0xAA) {
		  DDBG (("permanent failure: "));
		  BIOSDISK_error (dp->disknb, returned);
		  }
		else if (returned != 0) {
		  DDBG (("error: "));
		  BIOSDISK_error (dp->disknb, returned);
		  }
		else
		  DDBG (("OK.\r\n"));
	      }
#else /* BIOS_SUPPORT */
	      {
	      DDBG (("no BIOS disk compiled in.\r\n"));
	      returned = LOCALERROR;
	      }
#endif /* BIOS_SUPPORT */
	  } /* cylinder >= dp->nbcylinder */
      } /* end of C/H/S area */

#ifdef TREAT_EXCEPTION
  shit_handler (0, 5);
#endif

  return returned;
  }

awk_farret (DISK_readsector);
DISK_FCT_PREFIX(DISK_readsector) unsigned char
DISK_readsector (struct diskparam_str *dp, int partition,
		 unsigned long long lba,
		 unsigned number, farptr buffer)
  {bound_stack();{
  /* Because of my "strange" Quantum Fireball Plus KA 6.4Gb,
     with the [E]BIOS interface, I need to retry at least once.
     Also seen after Power Saving generated by DOS/BIOS */
  unsigned char cpt, ret = ret; /* inited B4 used */

#ifdef IDE_ATAPI_POWER_MANAGEMENT
  if (partition < -1 && lba == 0 && number == 0 && buffer == 0) {
#if DISK_SUPPORT & ATAPI_SUPPORT
      if (dp->access == hardide_atapi) {
//#define SET_DRIVE_AUTO_ACTIVE(dp)	DISK_readsector(dp, -2, 0, 0, 0)
//#define SET_DRIVE_ACTIVE(dp)	DISK_readsector(dp, -3, 0, 0, 0)
//#define SET_DRIVE_IDLE(dp)	DISK_readsector(dp, -4, 0, 0, 0)
//#define SET_DRIVE_STANDBY(dp)	DISK_readsector(dp, -5, 0, 0, 0)
	  if (partition == -2)
	      cpt = CD_LU_CONTROL;
	    else if (partition == -3)
	      cpt = CD_ACTIVE;
	    else if (partition == -4)
	      cpt = CD_IDLE;
	    else if (partition == -5)
	      cpt = CD_STANDBY;
	    else
	      return 0x97;
	  for (;;) {
	      ret = ideAtapiStartStopUnit (dp, cpt, 0, 0, 0);
	      DDBG (("ideAtapiStartStopUnit (dp, %u, 0, 0, 0) returns %d, ", cpt, ret));
	      if (ret == 0 || cpt != CD_LU_CONTROL)
		  break;
	      cpt = CD_ACTIVE;
	      }
	  return ret;
	  }
#endif
#if DISK_SUPPORT & IDE_SUPPORT
      if (dp->access == hardide_lba48 || dp->access == hardide_lba) {
	  // I do not know - does not seem to work
	  if (partition == -2)
	      cpt = 0;
	    else if (partition == -3)
	      cpt = 0;
	    else if (partition == -4)
	      cpt = 1;
	    else if (partition == -5)
	      cpt = 2;
	    else
	      return 0x97;
	  ret = IDE_set_power_mode (dp, cpt);
	  DDBG (("IDE_set_power_mode (dp, %u) returns %d, ", cpt, ret));
	  return ret;
	  }
#endif
      return 0x98; /* not handled */
      }
#endif /* IDE_ATAPI_POWER_MANAGEMENT */

  for (cpt = 2; cpt != 0; --cpt) {
      /*
       * last parameter is either null or not null, on a
       * non working RAM set all bits to not have one
       * toggled bit to crash the HD:
       */
      ret = DISK_RWsector (dp, partition, lba, number, buffer, 0xFFFFFFFF);
      if (ret != ERROR_PLEASE_RETRY)
	  break;
      dp->error_log.read_media_retried = 1;
      }
  if (ret)
      dp->error_log.read_media = 1;
  return ret;
  }}


#ifdef WRITE_ENABLE
awk_farret (DISK_writesector);
DISK_FCT_PREFIX(DISK_writesector) unsigned char
DISK_writesector (struct diskparam_str *dp, int partition,
		  unsigned long long lba,
		  unsigned number, farptr buffer)
  {bound_stack();{
  /* Because of my "strange" Quantum Fireball Plus KA 6.4Gb,
     with the [E]BIOS interface, I need to retry at least once.
     Also seen after Power Saving generated by DOS/BIOS */
  unsigned char cpt, ret = ret; /* inited B4 used */

  if (!copy_gujin_param.attrib.disk_write_enable) {
      DBG ((" [%s: WRITE DISK disabled in setup] ", __FUNCTION__));
      return 0xEE;
      }

  for (cpt = 2; cpt != 0; --cpt) {
      ret = DISK_RWsector (dp, partition, lba, number, buffer, 0);
      if (ret != ERROR_PLEASE_RETRY)
	  break;
      }
  if (ret)
      dp->error_log.write_media = 1;
  return ret;
  }}
#endif

/**
 ** The bit which understand partitions:
 **/
#if !(DISK_SUPPORT & NO_PROBE_PARTITIONS)
/* recursive pre-declaration: */
static unsigned
analyse_partition (struct diskparam_str *dp, unsigned long long lba,
		struct partition_str *partition, unsigned *nbpartition,
		farptr buffer_fptr);

DISK_FCT_PREFIX(analyse_extended_partition) static inline void
analyse_extended_partition (struct diskparam_str *dp,
		struct partition_str *partition, unsigned *nbpartition,
		farptr buffer_fptr)
  {
  unsigned short extended_partition_index = *nbpartition - 1;
  unsigned long long trylba = partition[extended_partition_index].start;

  int retval = analyse_partition (dp, trylba, partition, nbpartition, buffer_fptr);
  if (retval != 0) {
#if 0	/* has disappeared one day, the day everyone has a correct partition table... */
      DDBG (("\r\nWARNING: Extended partition block not found at LBA = %llU, ", trylba));
      printf (WARNING_EXTENDED_PARTITION_BLOCK_NOT_FOUND_AT_LLU, trylba);

      if (*nbpartition >= 2) {	/* there is a partition before the extended one */
	  /* EBIOS sometimes returns nbsectorpertrack = 0: */
	  unsigned nbsectorpertrack = (dp->nbsectorpertrack != 0)? dp->nbsectorpertrack : 63;

	  trylba = partition[*nbpartition - 2].start;
	  trylba += partition[*nbpartition - 2].length;
	  trylba += nbsectorpertrack - 1;

	  unsigned char bitpos = __builtin_ffs (nbsectorpertrack) - 1;
	  if (nbsectorpertrack == (1U << bitpos)) /* partition size can be bigger is nbsectorpertrack = 32 */
		trylba = (trylba >> bitpos) << bitpos;
	    else { /* then there is a unhandled limit for the partition size... */
	      unsigned tmp;
	      ull_div_ul (trylba, nbsectorpertrack, &tmp);
	      trylba -= tmp;
	     }
	  }

      if (trylba != partition[extended_partition_index].start) {
	  printf (MSG_TRYING_LBA, trylba);
	  DDBG ((" trying %llU...\r\n", trylba));
	  }
	else {
	  puts (MSG_FULL_STOP);
	  DDBG ((MSG_FULL_STOP));
	  dp->error_log.analyse_partition_failed = 1;
	  /* enter_read_only_mode(); */
	  copy_gujin_param.attrib.disk_write_enable = 0;
	  return;
	  }

      if (analyse_partition (dp, trylba, partition, nbpartition, buffer_fptr) == 0) {
	  printf (": %s.\r\n", WORD_OK);
	  DDBG ((": OK.\r\n"));
	  partition[extended_partition_index].start = trylba;
	  }
	else
#endif
	  {
	  puts (MSG_COLON_FAILED);
	  DDBG ((MSG_COLON_FAILED));
	  dp->error_log.analyse_partition_failed = 1;
	  /* enter_read_only_mode(); */
	  copy_gujin_param.attrib.disk_write_enable = 0;
	  return;
	  }
      }

#if DEBUG & DEBUG_DISK
  {
  unsigned long long extended_end = partition[extended_partition_index].start
			+ partition[extended_partition_index].length;
  /* take care: analyse_partition() has changed *nbpartition,
     this points to the last partition: */
  struct partition_str *pptr;
  unsigned long long lastlba;

  if (   partition[extended_partition_index].type != Linux_extended
      && partition[extended_partition_index].start != dp->first_ext_partition_start) {
      /* extended or LBA extended partition size are not cumulative,
	 the size does not include size of internal extended
	 partitions - non sense, look what I have to do:
	 (CHECKME: only for non primary extended partition,
		or only for DOS_extended, not Win95_extendedLBA ?) */
      unsigned short cpt = extended_partition_index + 1;

      while (   cpt < *nbpartition
	     && partition[cpt].type != DOS_extended
	     && partition[cpt].type != Win95_extendedLBA)
	  cpt++;
      pptr = &partition[cpt - 1];
      }
    else
      pptr = &partition[*nbpartition - 1];
  lastlba = pptr->start + pptr->length;

  if (extended_end != lastlba)
      DDBG (("\r\nInvalid length of extended partition, "
		"it is %llU and should be %llU, delta = %lld.",
		partition[extended_partition_index].length,
		lastlba - partition[extended_partition_index].start,
		extended_end - lastlba));
  }
#endif
  }

#if DEBUG & DEBUG_DISK
DISK_FCT_PREFIX(dbg_check_chs2lba) static inline void
dbg_check_chs2lba (const struct diskparam_str *dp,
		   unsigned lba, unsigned partno,
		   const struct bootsect_partition_str *buffpart)
  {
  unsigned disk_lba, check_lba, last_lba;
  unsigned bios_nbcylinder = (dp->bios_nbcylinder > 1024)
				? 1024
				: dp->bios_nbcylinder;
  unsigned short bios_nbsectorpertrack = (dp->bios_nbsectorpertrack > 64)
				? 63
				: dp->bios_nbsectorpertrack;
  unsigned short bios_nbhead = dp->bios_maxhead + 1;

  if (bios_nbsectorpertrack <= 1 || bios_nbhead == 0) {
      DDBG (("\r\nCannot check C/H/S partition with "
	     "%u sector/track and %u heads",
		bios_nbsectorpertrack, bios_nbhead));
      return;
      }

  last_lba = chs2lba (bios_nbcylinder - 1,	/* zero based */
		      bios_nbhead - 1,		/* zero based */
		      bios_nbsectorpertrack,	/* one based */
		      bios_nbhead,
		      bios_nbsectorpertrack);
  check_lba = chs2lba (begin_cylinder(buffpart),
		       buffpart->begin_head,
		       buffpart->begin_sector,
		       bios_nbhead,
		       bios_nbsectorpertrack);
  disk_lba = buffpart->nb_sector_before + lba;

  if (disk_lba >= last_lba) {
      /* If the starting C/H/S cannot be encoded (with the current nbhead and nbsector),
	 it has to be either maxcyl/0/1 or maxcyl/maxhead/maxsect or a combination: */
      if (   begin_cylinder(buffpart) != bios_nbcylinder - 1
	  || (buffpart->begin_head != 0 && buffpart->begin_head != bios_nbhead - 1)
	  || (buffpart->begin_sector != 1 && buffpart->begin_sector != bios_nbsectorpertrack)
	  ) {
	  DDBG (("\r\nInvalid 'saturated' start: C/H/S = %u/%u/%u "
		 "(lba = %u, last_lba = %u)\r\n",
			begin_cylinder(buffpart),
			buffpart->begin_head,
			buffpart->begin_sector,
			disk_lba, last_lba));
	  }
      }
    else if (check_lba != disk_lba) {
      unsigned should_cylinder, should_head, should_sector;

      DDBG (("\r\nInvalid start C/H/S [H0/S0]<->base+LBAstart "
		"in disk 0x%X, partition %u at LBA=%u:\r\n"
		"     %u/%u/%u [%u/%u] = %u <-> %u + %lu = %lu",
		dp->disknb, partno, lba,
		begin_cylinder(buffpart),
		buffpart->begin_head,
		buffpart->begin_sector,
		bios_nbhead,
		bios_nbsectorpertrack,
		check_lba,
		lba,
		buffpart->nb_sector_before,
		buffpart->nb_sector_before + lba ));
      lba2chs (disk_lba, &should_cylinder, &should_head, &should_sector,
		bios_nbhead, bios_nbsectorpertrack);
      DDBG (("; should be C/H/S %u/%u/%u.\r\n",
		should_cylinder, should_head, should_sector));
      }

  check_lba = chs2lba (end_cylinder(buffpart),
		       buffpart->end_head,
		       buffpart->end_sector,
		       bios_nbhead,
		       bios_nbsectorpertrack);
  disk_lba = buffpart->nb_sector_before + lba;
  disk_lba += buffpart->nb_sector - 1;

  if (disk_lba >= last_lba) {
      /* If the ending C/H/S cannot be encoded (with the current nbhead and nbsector),
	 it has to be maxcyl/maxhead/maxsect: */
      if (   end_cylinder(buffpart) != bios_nbcylinder - 1
	  || buffpart->end_head != bios_nbhead - 1
	  || buffpart->end_sector != bios_nbsectorpertrack) {
	  DDBG (("\r\nInvalid 'saturated' end: C/H/S = %u/%u/%u (lba = %u, last_lba = %u)\r\n",
			end_cylinder(buffpart),
			buffpart->end_head,
			buffpart->end_sector,
			disk_lba, last_lba));
	  }
      }
    else if (check_lba != disk_lba) {
      unsigned should_cylinder, should_head, should_sector;

      DDBG (("\r\nInvalid end C/H/S [H0/S0]<->base+LBAstart+length-1"
		" in disk 0x%X, partition %u at LBA=%u:\r\n"
		"  %u/%u/%u [%u/%u] = %u <-> %u + %lu + %lu - 1 = %lu",
		dp->disknb, partno, lba,
		end_cylinder(buffpart),
		buffpart->end_head,
		buffpart->end_sector,
		bios_nbhead,
		bios_nbsectorpertrack,
		check_lba,
		lba,
		buffpart->nb_sector_before,
		buffpart->nb_sector,
		lba + buffpart->nb_sector_before + buffpart->nb_sector - 1
		));
      lba2chs (disk_lba, &should_cylinder, &should_head, &should_sector,
		bios_nbhead, bios_nbsectorpertrack);
      DDBG (("; should be C/H/S %u/%u/%u.\r\n",
		should_cylinder, should_head, should_sector));
      }
  }
#endif

DISK_FCT_PREFIX(crc32) static unsigned
crc32 (unsigned crc, const unsigned char *buf, unsigned len)
  {
  const unsigned char *end = buf + len;

  if (buf == 0)
	return 0L;

  crc = ~crc;
  while (buf < end) {
      unsigned c = (crc ^ *buf++) & 0xff;
      unsigned short k;
#define POLY	0xedb88320L /* polynomial exclusive-or pattern */
      for (k = 8; k != 0; k--)
	  c = c & 1 ? (c >> 1) ^ POLY : c >> 1;
      crc = c ^ (crc >> 8);
      }
  return ~crc;
  }

/* Take care, recursive function: calls analyse_extended_partition()
   which call analyse_partition(): */
DISK_FCT_PREFIX(analyse_partition) static unsigned
analyse_partition (struct diskparam_str *dp, unsigned long long lba,
		struct partition_str *partition, unsigned *nbpartition,
		farptr buffer_fptr)
  {bound_stack();{
  unsigned short i;
  /* do not calloc too much to not overflow stack in this recursive function,
	do not calloc a sector (512 bytes - not even 2048 bytes) for each
	extended partition (my test disk has 7 levels) - just one bootafter_t each */
  bootafter_t bootafter;

  /* GCC-3.4: do not remove the following DDBG (()), catches stack overflow: */
  DDBG (("[entering %s with dp->bytepersector = %u (stack used = %U, total stack %U)]\r\n",
		__FUNCTION__, dp->bytepersector, 0x10000 - getesp(), totalstack()));

  if (DISK_readsector (dp, -1, lba, 1, buffer_fptr) != 0) {
      DDBG (("\r\n%s: READ lba %llU FAILED! ", __FUNCTION__, lba));
      return 1;
      }
  //  bootafter = buffer_fptr->bootsector.after;
  DDBG (("First bytes of sector 0x%llX: 0x%X, 0x%X, 0x%X; ", lba, peekb(buffer_fptr), peekb(buffer_fptr+1), peekb(buffer_fptr+2)));
  lmemcpy (&bootafter, buffer_fptr + offsetof(bootsector_t, after), sizeof (bootafter));

  DDBG (("signature: 0x%X\r\n", bootafter.Signature0xAA55));
  if (bootafter.Signature0xAA55 != 0xAA55) {
      if (bootafter.Signature0xAA55 == 0xF6F6) {
	  /*
	   * In between the DOS/Window fdisk and DOS/Window format,
	   * the boot record (all 512 bytes) are set to 0xF6.
	   * The Seagate partitionning software set the first 32 bits
	   * to the current lba: *(unsigned *)buffer_fptr == lba
	   * and the other bytes to 0xF6.
	   * To fall here, you need to have an unformated/uninitialised
	   * extended partition...
	   */
	  return 2;
	  }
	else
	  return 3;
      }

  for (i = 0; i < nbof(bootafter.bootsect_partition) && *nbpartition < NB_PARTITION; i++) {
      struct bootsect_partition_str *buffpart = &bootafter.bootsect_partition[i];
      struct partition_str *part = &partition[*nbpartition];

      part->misc = (struct partition_misc_str) {};
      switch (buffpart->indicator) {
	case bootable:
	   part->misc.active = part_active;
	   break;
	 case non_bootable:
	   part->misc.active = part_inactive;
	   break;
	 default:
	   DDBG (("\r\n Invalid partition marker: neither active nor "
		 "inactive: 0x%X", buffpart->indicator));
	   part->misc.active = part_invalid;
	   continue;
	 }

#if DEBUG & DEBUG_DISK
      if (   buffpart->begin_sector != 0
	  && buffpart->nb_sector_before != 0
	  && (lba >> 32) == 0)
	  dbg_check_chs2lba (dp, (unsigned)lba, i, buffpart);
#endif

      if (buffpart->system_indicator == EMPTY && lba != 0)
	  continue; /* completely ignore those ones */

      if (( (unsigned long long)buffpart->nb_sector_before + buffpart->nb_sector > 0xFFFFFFFFULL
	      || (unsigned long long)buffpart->nb_sector_before + buffpart->nb_sector >= dp->nbtotalsector)
		&& buffpart->system_indicator == GPT_scheme) {
	  union {
	      unsigned char fullsector[dp->bytepersector];
	      struct GPT_header_s  GPT_header;
	      struct GPT_entry_s  GPT_entry;
	      } gptbuf;
	  if (DISK_readsector (dp, -1, lba + 1, 1, stack_adr (&gptbuf)) != 0)
	      DDBG (("\r\n%s: READ lba %llU for GPT buffer FAILED! ", __FUNCTION__, lba + 1));
	    else if (*(unsigned long long *)gptbuf.GPT_header.signature != *(unsigned long long *)"EFI PART")
	      DDBG (("\r\n%s: GPT header signature invalid: '%s'! ", __FUNCTION__, gptbuf.GPT_header.signature)); /* followed by a zero in next field */
	    else if (gptbuf.GPT_header.current_lba != lba + 1)
	      DDBG (("\r\n%s: GPT header current_lba invalid: 0x%X, should be 0x%X! ", __FUNCTION__, gptbuf.GPT_header.current_lba, lba + 1));
	    else if (gptbuf.GPT_header.revision != 0x10000)
	      DDBG (("\r\n%s: GPT header revision invalid: 0x%X! ", __FUNCTION__, gptbuf.GPT_header.revision));
	    else if (gptbuf.GPT_header.header_size < offsetof(struct GPT_header_s, crc32_partition_array) || gptbuf.GPT_header.header_size >= dp->bytepersector)
	      DDBG (("\r\n%s: GPT header size invalid: %u! ", __FUNCTION__, gptbuf.GPT_header.header_size));
	    else if (gptbuf.GPT_header.first_useable_lba < 2 + gptbuf.GPT_header.number_of_partition_entry * gptbuf.GPT_header.sizeof_partition_entry / dp->bytepersector)
	      DDBG (("\r\n%s: GPT header first_useable_lba invalid: %llu, min: %llu! ", __FUNCTION__, gptbuf.GPT_header.first_useable_lba, 2 + gptbuf.GPT_header.number_of_partition_entry * gptbuf.GPT_header.sizeof_partition_entry / dp->bytepersector));
	    else if (gptbuf.GPT_header.partition_entry_start_lba >= gptbuf.GPT_header.first_useable_lba)
	      DDBG (("\r\n%s: GPT header partition_entry_start_lba invalid: %llu, max: %llu! ", __FUNCTION__, gptbuf.GPT_header.partition_entry_start_lba, gptbuf.GPT_header.first_useable_lba));
	    else if (gptbuf.GPT_header.last_useable_lba >= dp->nbtotalsector)
	      DDBG (("\r\n%s: GPT header last_useable_lba invalid: %llu, max: %llu! ", __FUNCTION__, gptbuf.GPT_header.last_useable_lba, dp->nbtotalsector));
	    else {
	      unsigned right_crc32 = gptbuf.GPT_header.crc32_header;
	      unsigned long long partition_entry_start_lba = gptbuf.GPT_header.partition_entry_start_lba;
	      unsigned long long first_useable_lba = gptbuf.GPT_header.first_useable_lba;
	      unsigned long long last_useable_lba = gptbuf.GPT_header.last_useable_lba;
	      unsigned number_of_partition_entry = gptbuf.GPT_header.number_of_partition_entry;
	      unsigned sizeof_partition_entry = gptbuf.GPT_header.sizeof_partition_entry;
	      DDBG (("%s: GPT header: signature '%s', revision 0x%X, header_size %u, crc32_header 0x%X, reserved %u\r\n",
			__FUNCTION__, gptbuf.GPT_header.signature, gptbuf.GPT_header.revision, gptbuf.GPT_header.header_size, gptbuf.GPT_header.crc32_header, gptbuf.GPT_header.reserved));
	      DDBG (("   current_lba %llu, backup_lba %llu, first_useable_lba %llu, last_useable_lba %llu\r\n",
			gptbuf.GPT_header.current_lba, gptbuf.GPT_header.backup_lba, gptbuf.GPT_header.first_useable_lba, gptbuf.GPT_header.last_useable_lba));
	      // DDBG((unsigned char disk_guid[16];))
	      DDBG (("   partition_entry_start_lba %llu, number_of_partition_entry %u, sizeof_partition_entry %u, crc32_partition_array 0x%X\r\n",
			gptbuf.GPT_header.partition_entry_start_lba, gptbuf.GPT_header.number_of_partition_entry, gptbuf.GPT_header.sizeof_partition_entry, gptbuf.GPT_header.crc32_partition_array));
	      gptbuf.GPT_header.crc32_header = 0;
	      DDBG (("\r\n Cancel MBR partition search, analyse GPT_scheme...\r\n"));
	      *nbpartition = 0;
	      dp->first_ext_partition_start = 0;
	      if (crc32 (0, (const unsigned char *)&gptbuf, gptbuf.GPT_header.header_size) != right_crc32)
		  DDBG (("\r\n%s: GPT crc32 invalid, should be 0x%X! ", __FUNCTION__, right_crc32));
		else {
		  struct GPT_entry_s *entryptr = 0;
		  right_crc32 = gptbuf.GPT_header.crc32_partition_array;
		  unsigned partial_crc32 = 0, cptpart = 0;
		  while (*nbpartition < NB_PARTITION && cptpart++ < number_of_partition_entry) {
		      if (entryptr == 0 || (unsigned)entryptr - (unsigned)&gptbuf >= dp->bytepersector) {
			  if (DISK_readsector (dp, -1, lba + partition_entry_start_lba, 1, stack_adr (&gptbuf)) != 0) {
			      DDBG (("\r\n%s: READ lba %llU for GPT buffer FAILED! ", __FUNCTION__, partition_entry_start_lba));
			      break;
			      }
			  partition_entry_start_lba++;
			  entryptr = (struct GPT_entry_s *)&gptbuf;
			  }
		      partial_crc32 = crc32 (partial_crc32, (const unsigned char *)entryptr, sizeof_partition_entry);
		      if (entryptr->first_lba < first_useable_lba || entryptr->last_lba > last_useable_lba) {
			  DDBG (("%s: partition %llu..%llu not in useable area %llu..%llu, ignoring\r\n", __FUNCTION__, entryptr->first_lba, entryptr->last_lba, first_useable_lba, last_useable_lba));
			  entryptr = (struct GPT_entry_s *)((unsigned)entryptr + sizeof_partition_entry);
			  continue;
			  }
		      part->misc = (struct partition_misc_str) {};
		      part->start = entryptr->first_lba + lba; /* SOARB if GPT ? */
		      part->length = entryptr->last_lba + 1 - part->start;

		      if (*(unsigned *)entryptr->partition_type_guid == 0xEBD0A0A2
				&& *(unsigned short *)&entryptr->partition_type_guid[4] == 0xB9E5 && *(unsigned short *)&entryptr->partition_type_guid[6] == 0x4433
				&& *(unsigned long long *)&entryptr->partition_type_guid[8] == *(unsigned long long *)"\x87\xC0\x68\xB6\xB7\x26\x99\xC7") {
			  part->type = Linux;
			  part->misc.active = part_active;
			  }
			else if (*(unsigned *)entryptr->partition_type_guid == 0xC12A7328 /* Fedora 12, why different? maybe if bootable */
				&& *(unsigned short *)&entryptr->partition_type_guid[4] == 0xF81F && *(unsigned short *)&entryptr->partition_type_guid[6] == 0x11D2
				&& *(unsigned long long *)&entryptr->partition_type_guid[8] == *(unsigned long long *)"\xBA\x4B\x00\xA0\xC9\x3E\xC9\x3B") {
			  part->type = Linux;
			  part->misc.active = part_active;
			  }
			else if (*(unsigned *)entryptr->partition_type_guid == 0x0657FD6D
				&& *(unsigned short *)&entryptr->partition_type_guid[4] == 0xA4AB && *(unsigned short *)&entryptr->partition_type_guid[6] == 0x43C4
				&& *(unsigned long long *)&entryptr->partition_type_guid[8] == *(unsigned long long *)"\x84\xE5\x09\x33\xC8\x4B\x4F\x4F") {
			  part->type = Linux_Swap;
			  part->misc.active = part_inactive;
			  part->misc.swap_partition = 1;
			  }
			else if (*(unsigned *)entryptr->partition_type_guid == 0xE3C9E316
				&& *(unsigned short *)&entryptr->partition_type_guid[4] == 0x0B5C && *(unsigned short *)&entryptr->partition_type_guid[6] == 0x4DB8
				&& *(unsigned long long *)&entryptr->partition_type_guid[8] == *(unsigned long long *)"\x81\x7D\xF9\x2D\xF0\x02\x15\xAE") {
			  part->type = Win95_FAT32LBA;
			  part->misc.active = part_active;
			  }
			else if (*(unsigned *)entryptr->partition_type_guid == 0xA19D880F
				&& *(unsigned short *)&entryptr->partition_type_guid[4] == 0x05FC && *(unsigned short *)&entryptr->partition_type_guid[6] == 0x4D3B
				&& *(unsigned long long *)&entryptr->partition_type_guid[8] == *(unsigned long long *)"\xA0\x06\x74\x3F\x0F\x84\x91\x1E") {
			  part->type = LinuxAutoRaid;
			  part->misc.active = part_active;
			  }
			else {
			  part->type = DOS_secondary; /* i.e. do not know... */
			  part->misc.active = part_active;
			  }
		      part->OSnumber = part->misc.order = *nbpartition;
		      unsigned cpt;
		      for (cpt = 0; cpt < nbof (entryptr->name_utf_16le) && cpt < nbof (part->name); cpt++)
			  if ((part->name[cpt] = entryptr->name_utf_16le[cpt]) == '\0')
			      break; /* just truncate UTF_16LE */
		      part->name[lastof(part->name)] = '\0';
#if DISK_SUPPORT & ISOFS_PROBE
		      part->ElToritoCatalog = 0;
#endif
		      DDBG (("found GPT partition name '%s' type 0x%X (OSnumber %u), start %llU, length %llU (ends at %llU)\r\n",
				part->name, part->type, part->OSnumber, part->start, part->length, part->start + part->length));
		      *nbpartition += 1;
		      part++;
		      entryptr = (struct GPT_entry_s *)((unsigned)entryptr + sizeof_partition_entry);
		      }
		  if (partial_crc32 != right_crc32)
		      DDBG (("\r\ncrc32_partition_array should be 0x%X and is 0x%X !!\r\n", right_crc32, partial_crc32));
		  }
	      break;
	      }
	  }

      if (  buffpart->nb_sector_before > dp->nbtotalsector
	  || buffpart->nb_sector > dp->nbtotalsector) {
	  DDBG (("\r\n Ignoring partition 0x%X+0x%X, end disk at 0x%X",
		buffpart->nb_sector_before, buffpart->nb_sector, dp->nbtotalsector));
	  continue;
	  }

      part->start = buffpart->nb_sector_before + lba;
      part->length = buffpart->nb_sector;
      part->type = buffpart->system_indicator;
      part->misc.order = *nbpartition;
      part->name[0] = '\0';
#if DISK_SUPPORT & ISOFS_PROBE
      part->ElToritoCatalog = 0;
#endif
      *nbpartition += 1;

      if (   buffpart->system_indicator == DOS_extended
	  || buffpart->system_indicator == Win95_extendedLBA) {
	  part->misc.active = part_extended;
	  /* Micro$oft could do logical thing times to times...
	   * For them, there is only one starting LBA for all extended
	   * partitions, that is the one of the primary extended
	   * partition. So the "nb_sector_before" of extended
	   * partitions is relative to the start of the
	   * _first_primary_extended_ partition.
	   * It means that an extended partition block is not like
	   * a complete disk in a disk, it is not self contained...
	   * I do exclude here the "Linux_extended", hoping that
	   * people who will use this partition type will not do
	   * the same error. See also the size checking of extended
	   * partitions.
	   */
	  part->start = buffpart->nb_sector_before; /* reset 'part->start' value */
	  if (lba == 0)
	      dp->first_ext_partition_start = buffpart->nb_sector_before;
	    else
	      part->start += dp->first_ext_partition_start;
	  }
	else if (buffpart->system_indicator == Linux_extended)
	  part->misc.active = part_extended;

      if (lba == 0)
	  part->OSnumber = i + 1;
	else if (part->misc.active != part_extended)
	  part->OSnumber = dp->nbOSnumber++;
	else
	  part->OSnumber = 0;

      DDBG (("\r\nfound %spartition type 0x%X (OSnumber %u), "
		"start %llU (%lu + %llu), length %lU (ends at %llU)"
		" [%u/%u/%u to %u/%u/%u]\r\n",
		part->OSnumber? "" : "extended ",
		buffpart->system_indicator,
		part->OSnumber,
		buffpart->nb_sector_before + lba,
		buffpart->nb_sector_before,
		lba,
		buffpart->nb_sector,
		buffpart->nb_sector_before + lba + buffpart->nb_sector,
		begin_cylinder(buffpart),
		buffpart->begin_head,
		buffpart->begin_sector,
		end_cylinder(buffpart),
		buffpart->end_head,
		buffpart->end_sector
		));
#if NB_PARTITION > 4
      if (part->misc.active == part_extended) {
	  DDBG (("\r\n\r\nAnalysing extended partition type 0x%X, "
			"start %llU (%lu + %u), length %lU...\r\n",
			buffpart->system_indicator,
			part->start,
			buffpart->nb_sector_before,
			(lba == 0)? (unsigned)lba : dp->first_ext_partition_start,
			buffpart->nb_sector));
#if 0
	  if (*nbpartition >= 2) {
	      DDBG (("*nbpartition = %u, "
			"partition[*nbpartition - 2].start = %llU,\r\n\t\t"
			"partition[*nbpartition - 2].length = %llU,\r\n\t\t"
			"prev->{start+length} = %llU\r\n",
			*nbpartition,
			partition[*nbpartition - 2].start,
			partition[*nbpartition - 2].length,
			partition[*nbpartition - 2].start + partition[*nbpartition - 2].length));
	      }
#endif
	  analyse_extended_partition (dp, partition, nbpartition, buffer_fptr);
	  }
#endif
      }

  if (*nbpartition == NB_PARTITION && i < nbof(bootafter.bootsect_partition)) {
      DDBG (("\r\n STOPPED analysing partition probably because compiled-in limit reached!.\r\n"));
      dp->error_log.NB_PARTITION_exceded = 1;
      }

  return 0;
  }}
#endif /* NO_PROBE_PARTITIONS */

#if DISK_SUPPORT & (BIOS_SUPPORT | EBIOS_SUPPORT | IDE_SUPPORT)

DISK_FCT_PREFIX(correct_CHS_from_disk_header) static inline void
correct_CHS_from_disk_header (struct diskparam_str *dp,
			      bootbefore_t *before)
  {
  unsigned nbtotalsector = (before->part1.NbTotalsector != 0)? before->part1.NbTotalsector : before->part1.NbTotalsector2;

  if (nbtotalsector == 0 || before->part1.Bytepersector == 0 || before->part1.NbHead == 0 || before->part1.NbSectorpertrack == 0) {
      DDBG (("[%s: disk header ignored because its nbtotalsector == %u, Bytepersector %u, NbHead = %u, NbSectorpertrack = %u] ",
		__FUNCTION__, nbtotalsector, before->part1.Bytepersector, before->part1.NbHead, before->part1.NbSectorpertrack));
      dp->error_log.diskheader_ignored = 1;
      return;
      }

#if !(DISK_SUPPORT & NO_PROBE_PARTITIONS)
  dp->bios_maxhead = before->part1.NbHead - 1;
  dp->bios_nbsectorpertrack = before->part1.NbSectorpertrack;
  /* 8 Mb USB disk with 15584 sectors: [overwrite for C/H/S partitions: 0/255/63]
     so round up. */
  dp->bios_nbcylinder = DIVROUNDUP (nbtotalsector,
			before->part1.NbSectorpertrack * before->part1.NbHead);
  DDBG (("[store for C/H/S partitions: %u/%u/%u from bootsector header] ",
	dp->bios_nbcylinder, dp->bios_maxhead + 1, dp->bios_nbsectorpertrack));
#endif

  if (   dp->nbsectorpertrack != before->part1.NbSectorpertrack
      || dp->nbhead != before->part1.NbHead
      || dp->nbtotalsector != nbtotalsector) {
      DDBG ((" [DP contains: nbsectorpertrack %u, nbhead %u, nbtotalsector %llU != "
		"bootsector contains: nbsectorpertrack %u, nbhead %u, nbtotalsector %U]\r\n",
		dp->nbsectorpertrack, dp->nbhead, dp->nbtotalsector,
		before->part1.NbSectorpertrack, before->part1.NbHead, nbtotalsector));
      if (dp->access != bios_chs) {
	  DDBG (("[ignore disk header adjustment because not accessing this disk with BIOS] "));
	  return;
	  }
      if (dp->nbsectorpertrack == 0 || dp->nbhead == 0) {
	  DDBG (("[ignore disk header adjustment because before->part1.NbHead %u, before->part1.NbSectorpertrack %u] ",
		before->part1.NbHead, before->part1.NbSectorpertrack));
	  return;
	  }
      if (dp->nbsectorpertrack == before->part1.NbSectorpertrack && dp->nbhead == before->part1.NbHead)
	  printf (INFO_ADJUSTING_SIZE_FROM_BOOTSECTOR_S, dp->diskname);
	else
	  printf (INFO_ADJUSTING_PARAMETERS_FROM_BOOTSECTOR_S, dp->diskname);
      dp->nbsectorpertrack = before->part1.NbSectorpertrack;
      dp->nbhead = before->part1.NbHead;
      dp->nbtotalsector = nbtotalsector;
      dp->nbcylinder = DIVROUNDUP (nbtotalsector, dp->nbsectorpertrack * dp->nbhead);
      dp->error_log.chs_ajusted_bootsect = 1;
      }
  }

DISK_FCT_PREFIX(correct_CHS_from_partition) static inline void
correct_CHS_from_partition (struct diskparam_str *dp,
			    struct bootsect_partition_str bootsect_partition[4])
  {
  unsigned char maxhead = 0, maxsect = 0, cpt;

  for (cpt = 0; cpt < 4; cpt++) {
      if ((bootsect_partition[cpt].indicator & 0x7F) != 0)
	  continue;
      if (bootsect_partition[cpt].system_indicator == EMPTY)
	  continue;
      if (bootsect_partition[cpt].end_head > maxhead)
	  maxhead = bootsect_partition[cpt].end_head;
      if (bootsect_partition[cpt].end_sector > maxsect)
	  maxsect = bootsect_partition[cpt].end_sector;
      }

  if (dp->access == bios_chs || dp->access == ebios_lba) {
      /* Some EBIOS say in lba_slave_mask that they access disk in LBA,
       * so this message is not important, others or simple BIOS may
       * have a problem (disk in BIOS setup does not match the one
       * when the partitions were created)
       * Also no message for blank disks nor simple floppies.
       */
      if ((dp->lba_slave_mask & 0x40) == 0 && maxhead != 0 && maxsect != 0) {
	  if (maxhead != dp->bios_maxhead || maxsect != dp->bios_nbsectorpertrack) {
#if DISK_SUPPORT == (BIOS_SUPPORT|EBIOS_SUPPORT|FAT12_PROBE|FAT16_PROBE|FAT32_PROBE|PROBE_ONLY_BIOSBOOTDEVICE) /* tinyusb */ \
	|| DISK_SUPPORT == (BIOS_SUPPORT|EBIOS_SUPPORT|E2FS_PROBE|PROBE_ONLY_BIOSBOOTDEVICE) /* tinyext4 */
	      if (dp->access == bios_chs)
#endif
	      printf (PARTITION_CHS_MISMATCH_BIOS_CHS,
			dp->diskname,
			maxhead + 1, maxsect,
			dp->bios_maxhead + 1,
			dp->bios_nbsectorpertrack);
	      dp->error_log.chs_bios_part_mismatch = 1;
	      /* FIXME: reconfigure it (and reprobe)? */
	      }
	  }
      }
#if !(DISK_SUPPORT & NO_PROBE_PARTITIONS)
    else if (dp->access == hardide_chs || dp->access == hardide_lba
				       || dp->access == hardide_lba48) {
      unsigned tmpcyl;

      /* Those are still at zero if no 0x29 signature, initialise them here: */
      dp->bios_maxhead = maxhead;
      dp->bios_nbsectorpertrack = maxsect;
      if (dp->bios_nbsectorpertrack) {
	  unsigned cylinder_size = (dp->bios_maxhead+1) * dp->bios_nbsectorpertrack;

	  unsigned char bitpos = __builtin_ffs (cylinder_size) - 1;
	  if (cylinder_size == (1U << bitpos)) /* partition size can be bigger is nbsectorpertrack = 32 and nb_head = 16 or nb_head = 256 */
	      tmpcyl = dp->nbtotalsector >> bitpos;
	    else { /* then there is a unhandled limit for the partition size... */
	      unsigned remain;
	      tmpcyl = ull_div_ul (dp->nbtotalsector, cylinder_size, &remain);
	      }
	  }
	else
	  tmpcyl = 0;
      if (tmpcyl > 1024)
	  dp->bios_nbcylinder = 1024;
	else
	  dp->bios_nbcylinder = tmpcyl;
      DDBG (("[hardide use for C/H/S partitions: %u/%u/%u] ",
		dp->bios_nbcylinder, dp->bios_maxhead + 1,
		dp->bios_nbsectorpertrack));
      }
#endif
  }

/**
 ** This sort the partition in the start increasing order,
 ** to have "empty" partitions (or length == 0) at
 ** end of the array. It is needed for freelist calculus.
 **/
DISK_FCT_PREFIX(sort_partition) static inline void
sort_partition (struct partition_str *part, unsigned nbpart)
  {
  unsigned i, j, min;

  checkptr (part, "sort_partition invalid parameter.");

  for (i = 0; i < nbpart; i++) {
      min = i;
      for (j = i+1; j < nbpart; j++) {
	  if (part[j].length == 0 || part[j].misc.active == part_invalid)
	      continue;
	  if (part[j].start < part[min].start)
	      min = j;
	  }
      if (min != i) {
	  struct partition_str tmp = part[i];
	  part[i] = part[min];
	  part[min] = tmp;
	  }
      }
  }

/**
 ** Just to calculate the list of sector not mapped in a partition.
 ** It also checks if there is overlap of partitions, or a partition
 ** over the limit of the disk.
 **/
#if defined (NB_FREELIST) && NB_FREELIST != 0
DISK_FCT_PREFIX(calculatefreelist) static inline void
calculatefreelist (struct diskparam_str *dp)
  {
  unsigned short	cpt_part;
  unsigned long long	curlba;
  unsigned		nbfreelist = 0;

#ifdef DISK_USE_MALLOC
#if 1
  struct freelist_str	freelist[NB_FREELIST];
#else
  struct freelist_str	*freelist = (struct freelist_str *)fourKbuffer;

  if (sizeof (fourKbuffer) < NB_FREELIST * sizeof (struct freelist_str))
      __ERROR();	/* the linker will produce an error here */
#endif
#else
  static struct freelist_str _disk_freelist[NB_DISK][NB_FREELIST];
  struct freelist_str	*freelist = dp->freelist = _disk_freelist[dp - DI.param];
#endif

  DDBG (("%s: ", __FUNCTION__));
  checkptr (dp, "calculatefreelist invalid parameter.");

  curlba = RESERVEDSECTOR; /* first sector of disk that can be in free list */
  for (cpt_part = 0;
       cpt_part < dp->nbpartition && nbfreelist < NB_FREELIST;
       cpt_part++) {
      struct partition_str *dpp = &dp->partition[cpt_part];

      DDBG (("."));
      if (dpp->type == EMPTY || dpp->length == 0)
	  continue;
      if (dpp->start == 0 && curlba == RESERVEDSECTOR) {
	  /* the dummy floppy partition, i.e. MBR in 1st partition: */
	  curlba = dpp->length;
	  continue;
	  }
      if (dpp->start < curlba) {
	  DDBG ((" [ERROR %s: partition overlap!] ", __FUNCTION__));
	  printf (DISK_S_PARTITION_U_OVERLAP_DETECTED, dp->diskname, dpp->OSnumber);
	  dp->error_log.partition_overlap = 1;
	  }
	else if (dpp->start > curlba) {
	  DDBG (("%llU sectors at lba %llU, ", dpp->start - curlba, curlba));
	  freelist[nbfreelist].start = curlba;
	  freelist[nbfreelist].length = dpp->start - curlba;
	  nbfreelist++;
	  }
#if (DISK_SUPPORT & BIOS_SUPPORT) && (DISK_SUPPORT & EBIOS_SUPPORT)
      if (   dp->access == bios_chs && (dpp->type == Win95_extendedLBA
	  || dpp->type == Win95_FAT16LBA || dpp->type == HiddenWin95_FAT16LBA
	  || dpp->type == Win95_FAT32LBA || dpp->type == HiddenWin95_FAT32LBA))
	  DDBG ((" [WARNING: partition type 0x%X on a disk without LBA access] ", dpp->type));
#endif

      if (dpp->misc.active == part_extended)
	  curlba = dpp->start + RESERVEDSECTOR;
	else
	  curlba = dpp->start + dpp->length;
      }

  DDBG (("*"));

  if (curlba > dp->nbtotalsector) {
      DDBG ((" [%s: partition extend over disk limit] ", __FUNCTION__));
      printf (S_S_LAST_PARTITION_OVER_DISK_LIMIT_DETECTED_U_KB,
		  (dp->access == bios_chs || dp->access == ebios_lba) ? MSG_WARNING : MSG_ERROR,
		  dp->diskname, (unsigned)(curlba - dp->nbtotalsector) / 2);
      dp->error_log.partition_over_limit = 1;
      }
    else if (curlba == dp->BEER_sector) {
      DDBG (("%llU sectors at end used for BEER sectors, lba %llU, ",
		      dp->nbtotalsector - curlba, curlba));
      }
    else if (curlba < dp->nbtotalsector && nbfreelist < NB_FREELIST) {
      /* space left at end of disk: */
      freelist[nbfreelist].start = curlba;
      freelist[nbfreelist].length = dp->nbtotalsector - curlba;
      DDBG (("%llU sectors at end, lba %llU, ", dp->nbtotalsector - curlba, curlba));
      nbfreelist++;
      }

  dp->nbfreelist = nbfreelist;
#ifdef DISK_USE_MALLOC
  if (nbfreelist == 0) {
      DDBG ((" no free block!\r\n"));
      dp->freelist = 0;
      return;
      }
  dp->freelist = MALLOC (nbfreelist * sizeof (struct freelist_str), "frelist");
  if (dp->freelist == 0) {
      DDBG (("%s: malloc for %u freelist failed!\r\n", __FUNCTION__, nbfreelist));
      dp->nbfreelist = 0;
      return;
      }
  while (nbfreelist--)
      dp->freelist[nbfreelist] = freelist[nbfreelist];
#endif /* DISK_USE_MALLOC */
  DDBG (("\r\nNote: %u free list found (max compiled-in %u).\r\n", dp->nbfreelist, NB_FREELIST));
  }
#endif /* NB_FREELIST */

/**
 ** (Cool, not cold) BEER stuff, look at ATA docs at
 ** http://www.t13.org/project/d1367r3.pdf
 *
 ** Note that I tried to find the (same) BEER sector with the (E)BIOS interface
 ** and with the IDE interface. Usually the EBIOS interface return the true size
 ** of the disk so it will work correctly - but here I have a PC with a disk
 ** and the disk size reported by EBIOS is too low:
 Hard disk 0x80: BIOSDISK_gettype: returns 0x3, i.e. hard disk, probably nb sector = 0xBFF04F i.e. 12578895
 EBIOS (...) reports: nbhead: 15, nbsectorpertrack: 63, nbcylinder: 13327, bytepersector: 512, nbtotalsector: 0XC02B5F (i.e. 12594015), infobit 0x3
 ** so 'Search BEER sector: ... BEER sector at 0XC02B5E (i.e. 12594014)' and IDE hard:
 ideIdentify reports: C/H/S=13328/15/63, RWmult=16, trans CHS valid: C/H/S=13328/15/63, total= 12594960, BigDisk total: 12594960 i.e. 0xC02F10, BigBigLBA48total: 0X0
 ** so 'Search BEER sector: ... BEER sector at 0XC02F0F (i.e. 12594959)'
 ** delta = 945 = 15 * 63 Maybe I should treat it, but this is not done, and EBIOS may
 ** forbid reading those sectors...
 **/
DISK_FCT_PREFIX(analyseBEERsector) static inline unsigned char
analyseBEERsector (struct diskparam_str *dp, unsigned long long last_used_sector)
  {
  unsigned char err;
  struct beer_s *beer = beer; /* inited b4 used */
  unsigned long long beer_sector = dp->nbtotalsector - 1;
  long long nbsector;
  unsigned cpt;
#if 1
#define BUFFERSIZE 4096
  unsigned buffer_size = BUFFERSIZE, nbsectorscan = BUFFERSIZE / dp->bytepersector;
  union { bootsector_t bootsector; unsigned char all[BUFFERSIZE]; } real_buffer, *buffer = &real_buffer;
  farptr buffer_farptr = stack_adr (buffer);
#undef BUFFERSIZE
#else
  unsigned buffer_size = sizeof (fourKbuffer), nbsectorscan = sizeof (fourKbuffer) / dp->bytepersector;
  void *buffer = (void *)fourKbuffer;
  farptr buffer_farptr = data_adr (buffer);
#endif

  if (sizeof (struct beer_directory_s) != 64)
      __ERROR();        /* the linker will produce an error here */
  if (sizeof (struct beer_s) != 128)
      __ERROR();	/* the linker will produce an error here */

  if (dp->bytepersector == 512 && dp->access == hardide_atapi) {
      DDBG (("%s: Do not search for BEER partition on CDROM/DVD with MBR declaring 512 bytes/sectors\r\n", __FUNCTION__));
//printf ("%s: Do not search for BEER partition on CDROM/DVD with MBR declaring 512 bytes/sectors\r\n", __FUNCTION__); _BIOS_getkey();
      return 7;
      }
  if (dp->bytepersector == 2048) {
      DDBG (("%s: Searching for BEER partition on CDROM/DVD: do not read last sector\r\n", __FUNCTION__));
      /* Newer DVDRW can write the last cylinder on CDRW, but cannot read it;
		and ATAPI reset cannot bring them back to life... */
      beer_sector -= 1;
      }
  nbsector = beer_sector - last_used_sector;

  if (nbsector <= 0) {
      DDBG (("%s: BEER sector at %llU would be inside the last partition!\r\n",
		__FUNCTION__, beer_sector));
      return 1;
      }
    else if ((long long)nbsectorscan > nbsector) {
      DDBG ((" [limiting BEER scanning from %u to %u sectors] ", nbsectorscan, (unsigned)nbsector));
      nbsectorscan = (unsigned)nbsector;
      }

  DDBG (("%s: BEER sector at %llU, scanning %u sectors: ",
	__FUNCTION__, beer_sector, nbsectorscan));

  if (dp->bytepersector > buffer_size) {
      DDBG (("%s: dp->bytepersector %u, buffer_size %u: FAILED!\r\n",
		__FUNCTION__, dp->bytepersector, buffer_size));
      return 4;
      }

#if 1
  cpt = buffer_size/sizeof(int);
  while (cpt --)
      ((int *)buffer)[cpt] = 0;

  err = DISK_readsector (dp, -1, beer_sector - nbsectorscan + 1, nbsectorscan, buffer_farptr);

  if (err != 0) {
      /* a setmax may have limited (BIOS) reads of the end of the disk,
	 and BIOS cannot be told to reprobe disks after boot. */
      DDBG (("READ FAILED 0x%X, still trying to look for BEER in the partial read.\r\n", err));
//      return 2;
      }
#else
  err = DISK_readsector (dp, -1, beer_sector - nbsectorscan + 1,
				nbsectorscan, buffer_farptr);

  if (err != 0) {
      DDBG (("READ FAILED 0x%X !\r\n", err));
      return 2;
      }
#endif

  dp->BEER_sector = 0;
  cpt = nbsectorscan;
  while (cpt-- != 0) {
      beer = (struct beer_s *)(((unsigned)buffer) + cpt * dp->bytepersector);
//if ((unsigned)beer >> 16) printf ("Error cpt %u, nbsectorscan %u, beer_sector %u\r\n", cpt, nbsectorscan, beer_sector);
      DDBG (("cpt %u, signature 0x%X == 0xBEEF ? ", cpt, beer->signature_0xBEEF));
      if (beer->signature_0xBEEF == 0xBEEF) {
	  unsigned short checksum = 0;
	  const unsigned short *ptr = (const unsigned short *)beer;

	  ptr += beer->beer_size / 2;
	  while (ptr > (const unsigned short *)beer)
	      checksum += *--ptr;
	  DDBG (("yes, BEER size %u == 128 ?, revision %u, "
		"checksum: 0x%X, total shall be 0 and is 0x%X, %svalid\r\n",
		beer->beer_size, beer->revision_3,
		beer->checksum, checksum, checksum? "in" : ""));
	  if (checksum == 0) {
	      dp->BEER_sector = beer_sector - nbsectorscan + cpt + 1;
	      DDBG (("Found BEER sector at %llU\r\n", dp->BEER_sector));
	      break;
	      }
	    else
	      dp->error_log.beer_checksum_error = 1;
	  }
      DDBG (("no, "));
      }

  if (dp->BEER_sector == 0) {
      DDBG (("so not beer compatible -:).\r\n"));
      return 3;
      }
#if KEEP_STRUCT_MAPPING || (DISK_SUPPORT & IDE_SUPPORT)
  cpt = 32;
  while (cpt --)
      dp->set_max_password[cpt] = beer->device_name[cpt];
#endif
  DDBG (("device_name: '%s', device_index 0x%X, reserved_1 0x%X, "
	 "reserved_2 0x%X, reported.sectorperdevice %llU, formatted.sectorperdevice %llU, ",
	 beer->device_name, beer->device_index,
	 beer->reserved_1, beer->reserved_2,
	 beer->reported.sectorperdevice, beer->formatted.sectorperdevice));
  dp->BEER_disksize = beer->reported.sectorperdevice; /* mandatory */
  dp->BEER_device_index = beer->device_index;
  dp->BEER_size = beer->beer_size;
  if (beer->attr.service_dir_present)
	dp->BEER_size += beer->BEERdir_nb_entry * beer->BEERdir_length;
  if (beer->attr.reported_geometry_valid)
      DDBG (("reported_geometry_valid: cylinder = %u, head = %u,"
	     " sector = %u, bytepersector = %u\r\n",
	     beer->reported.cylinder,
	     beer->reported.head,
	     beer->reported.sector,
	     beer->reported.bytepersector));
  if (beer->attr.formated_geometry_valid)
      DDBG (("formatted_geometry_valid: cylinder = %u, head = %u,"
	     " sector = %u, bytepersector = %u, sectorperdevice %llU\r\n",
	     beer->formatted.cylinder,
	     beer->formatted.head,
	     beer->formatted.sector,
	     beer->formatted.bytepersector,
	     beer->formatted.sectorperdevice));
  if (beer->attr.config_time_stamp_valid)
      DDBG (("config_time_stamp_valid: bcd_year 0x%X, linear_day %u, "
	     "config_time_stamp %u\r\n",
	    beer->bcd_year, beer->linear_day, beer->config_time_stamp));
  if (beer->attr.use_RABCA) {
      dp->BEER_HostProtectedArea_start = beer->HostProtectedArea_start;
      dp->BEER_ReservedAreaBootCodeAddress = beer->ReservedAreaBootCodeAddress;
      DDBG (("HostProtectedArea_start: %llU, ReservedAreaBootCodeAddress %llU\r\n",
	    beer->HostProtectedArea_start, beer->ReservedAreaBootCodeAddress));
      }
  if (beer->attr.service_dir_present)
      DDBG (("BEERdir_nb_entry: %u, BEERdir_length: %u\r\n",
	    beer->BEERdir_nb_entry, beer->BEERdir_length));
  return 0;
  }

extern inline unsigned
all_non_beer_partition_empty (unsigned nbpartition, const struct partition_str *partition)
  {
  while (nbpartition-- > 0)
      if (!partition[nbpartition].misc.beer_partition && partition[nbpartition].type != EMPTY)
	  return 0;
  return 1;
  }

DISK_FCT_PREFIX(beer_boot_sector_update) unsigned
beer_boot_sector_update (struct diskparam_str *dp, bootsector_t *bootsector)
  {
  bootsector_t backupsector;
  unsigned char err;
  unsigned returned = 0;

  if (   dp->BEER_ReservedAreaBootCodeAddress == 0
      || dp->BEER_ReservedAreaBootCodeAddress > dp->nbtotalsector) {
      DDBG (("\r\n%s: will not read MBR Backup sector at %llu (max %llu)! ",
		__FUNCTION__, dp->BEER_ReservedAreaBootCodeAddress, dp->nbtotalsector));
      return returned;
      }

  err = DISK_readsector (dp, -1, dp->BEER_ReservedAreaBootCodeAddress, 1, stack_adr(&backupsector));
  if (err != 0) {
      DDBG (("\r\n%s: READ MBR Backup sector FAILED 0x%X ! ", __FUNCTION__, err));
      return returned;
      }
  if (backupsector.after.Signature0xAA55 != 0xAA55) {
      DDBG (("\r\n%s: READ MBR Backup sector NO 0xAA55 SIGNATURE ! ", __FUNCTION__));
      return returned;
      }
#if USER_SUPPORT != 0
  if (!memeql (&backupsector, bootsector, sizeof(backupsector.before)
				 + sizeof(backupsector.code_part) // + sizeof(backupsector.param)
		)  ) {
      printf (DISK_BEERBOOT_DIFF_BACKUP, dp->diskname, DISK_BEERBOOT_CODE);
#ifdef WRITE_ENABLE
      if (copy_gujin_param.attrib.disk_write_enable) {
	  unsigned key, i;

	  key = UI.function.getkey (10 * TICKS_PER_SECOND) & 0xFF;
	  if (key == '\r' || key == '\n') {
	      printf (DISK_BEERBOOT_CONFIRM_UPDATE_S, DISK_BEERBOOT_CODE);
	      key = UI.function.getkey (10 * TICKS_PER_SECOND) & 0xFF;
	      if (key == '\r' || key == '\n') {
		  backupsector.before = bootsector->before;
		  i = sizeof (backupsector.code_part);
		  while (i--)
		      backupsector.code_part[i] = bootsector->code_part[i];
		  backupsector.param = bootsector->param;
		  err = DI.writesector (dp, -1, dp->BEER_ReservedAreaBootCodeAddress, 1, stack_adr(&backupsector));
		  }
		else
		  key = TIMEOUT_KEYCODE();
	      }
	    else if (key == ' ') {
	      printf (DISK_BEERBOOT_CONFIRM_RESTORE_S, DISK_BEERBOOT_CODE);
	      key = UI.function.getkey (50 * TICKS_PER_SECOND) & 0xFF;
	      if (key == '\r' || key == '\n') {
		  bootsector->before = backupsector.before;
		  i = sizeof (bootsector->code_part);
		  while (i--)
		      bootsector->code_part[i] = backupsector.code_part[i];
		  bootsector->param = backupsector.param;
		  err = DI.writesector (dp, -1, 0, 1, stack_adr(bootsector));
		  }
		else
		  key = TIMEOUT_KEYCODE();
	      }
	    else { /* (key == '\e') or timeout */
	      key = TIMEOUT_KEYCODE();
	      }
	  puts ((key == (unsigned)TIMEOUT_KEYCODE()) ? WORD_IGNORED : (err? WORD_ERROR : WORD_OK));
	  }
	else
#endif
	  puts (WORD_READONLY);
      }
#endif
#if 0
  /* We should not "restore" it if only few hidden/unhidden byte difference... */
  if (memeql (&backupsector.after, &bootsector->after, sizeof(backupsector.after)))
      return returned;
#else
  if (   backupsector.after.WindowsNTmarker == bootsector->after.WindowsNTmarker
      && backupsector.after.Unknown == bootsector->after.Unknown
      && backupsector.after.Signature0xAA55 == bootsector->after.Signature0xAA55) {
      struct bootsect_partition_str *bp = &bootsector->after.bootsect_partition[3];
      struct bootsect_partition_str *bbp = &backupsector.after.bootsect_partition[3];

      while (bp >= bootsector->after.bootsect_partition) {
	  if (bbp->nb_sector_before != bp->nb_sector_before)
	      break;
	  if (bbp->nb_sector != bp->nb_sector)
	      break;
	  if ((*(unsigned long long*)bbp ^ *(unsigned long long*)bp) & ~0x1000000080ULL)
	      break;
	  bp --;
	  bbp --;
	  }
      if (bp < bootsector->after.bootsect_partition)
	  return returned;
      }
#endif
  printf (DISK_BEERBOOT_DIFF_BACKUP, dp->diskname, DISK_BEERBOOT_PARTITION);
#ifdef WRITE_ENABLE
  if (copy_gujin_param.attrib.disk_write_enable) {
      unsigned key;

      key = UI.function.getkey (10 * TICKS_PER_SECOND) & 0xFF;
      if (key == '\r' || key == '\n') {
	  printf (DISK_BEERBOOT_CONFIRM_UPDATE_S, DISK_BEERBOOT_PARTITION);
	  key = UI.function.getkey (50 * TICKS_PER_SECOND) & 0xFF;
	  if (key == '\r' || key == '\n') {
	      backupsector.after = bootsector->after;
	      err = DI.writesector (dp, -1, dp->BEER_ReservedAreaBootCodeAddress, 1, stack_adr(&backupsector));
	      }
	    else
	      key = TIMEOUT_KEYCODE();
	  }
	else if (key == ' ') {
	  printf (DISK_BEERBOOT_CONFIRM_RESTORE_S, DISK_BEERBOOT_PARTITION);
	  key = UI.function.getkey (10 * TICKS_PER_SECOND) & 0xFF;
	  if (key == '\r' || key == '\n') {
	      bootsector->after = backupsector.after;
	      err = DI.writesector (dp, -1, 0, 1, stack_adr(bootsector));
	      returned = 1;
	      }
	    else
	      key = TIMEOUT_KEYCODE();
	  }
	else { /* (key == '\e') or timeout */
	  key = TIMEOUT_KEYCODE();
	  }
      puts ((key == (unsigned)TIMEOUT_KEYCODE()) ? WORD_IGNORED : (err? WORD_ERROR : WORD_OK));
      }
    else
#endif
      puts (WORD_READONLY);
  return returned;
  }

#if !(   (DISK_SUPPORT & (BIOS_SUPPORT | EBIOS_SUPPORT)) \
      && (DISK_SUPPORT & IDE_SUPPORT))
#define INLINE_analysebootsector
#endif

DISK_FCT_PREFIX(analysebootsector) static
#ifdef INLINE_analysebootsector
inline
#endif
int analysebootsector (struct diskparam_str *dp)
#ifdef INLINE_analysebootsector
  {{
#else
  {bound_stack();{
#endif
  unsigned beer_partition_restored = 0;
  unsigned nbpartition;
  unsigned char err;
  /* To not have "too high FullLBA" message when searching BEER partition, reading MBR copy...
	do not set dp->{fulldisk_offset,fulldisk_size} before this function ends. */
  long long final_fulldisk_offset = 0;
  unsigned long long final_fulldisk_size = 0;
#ifdef DISK_USE_MALLOC
  struct partition_str partition[NB_PARTITION];
//  struct partition_str *partition = (struct partition_str *)fourKbuffer; // but NB_PARTITION <= 30
#else
  static struct partition_str _disk_partition[NB_DISK][NB_PARTITION];
  struct partition_str *partition = dp->partition = _disk_partition[dp - DI.param];
#endif

// fourKbuffer may also be used in analyseBEERsector(), take care:
// We want "union { bootsector_t bootsector; unsigned char all[dp->bytepersector]; } buffer;"
//   but dp->bytepersector may change depending on the first MBR read.
#if 0
#define BUFFERSIZE 4096
  union { bootsector_t bootsector; unsigned char all[BUFFERSIZE]; } real_buffer, *buffer = &real_buffer;
  farptr buffer_fptr = stack_adr (buffer);
#undef BUFFERSIZE
#else
  union ubs { bootsector_t bootsector; } *buffer = (union ubs *)fourKbuffer;
  farptr buffer_fptr = data_adr (buffer);
#endif

  DDBG (("\r\n%s: ", __FUNCTION__));
  checkptr (dp, "analysebootsector invalid parameter.");

  do {
      nbpartition = 0;
      TINYDBG ("Will memset... ");
      _memset (buffer, 0xEE, sizeof (fourKbuffer));
      TINYDBG ("Will DISK_readsector(0x%X) for MBR... ", dp->disknb);
      if ((err = DISK_readsector (dp, -1, 0, 1, buffer_fptr)) != 0) {
	  DDBG (("READ master boot sector FAILED 0x%X !\r\n", err));
#if DISK_SUPPORT & ATAPI_SUPPORT
	  if (dp->access == hardide_atapi) {
	      DDBG (("ATAPI sector 0 not readable.\r\n"));
	      goto ATAPI_readTOC;
	      }
#endif
	  return 1; /* caller may retry (IDE password, recalibrate, ATAPI reset device) */
	  }
      TINYDBG ("DISK_readsector(0x%X) success\r\n", dp->disknb);
      /* HP Compaq 8000 Elite CMT PC with BIOS 786G7 v01.02 compares bytes offset 0x0B, 0x0C, 0x0D
	to see if they are identical to Microsoft MBR, and obviously crash if they are not.
	Try to keep format: at 0x0A is the number of bytes to overwrite with "standard" values */
      if (*(unsigned *)&fourKbuffer[0x0A] == 0xBEFC1F03)
	  *(unsigned *)&fourKbuffer[0x0A] = 0x00020000;

      if (dp->bytepersector <= 2048) /* save it for BEER comparisson & restore */
	  memcpy ((unsigned char *)buffer + 2048, buffer, 2048);

      DDBG (("first bytes are 0x%X, 0x%X, 0x%X, signature is 0x%X, ",
		buffer->bootsector.before.part1.jmpinstruction[0],
		buffer->bootsector.before.part1.jmpinstruction[1],
		buffer->bootsector.before.part1.EBIOSaccess,
		buffer->bootsector.after.Signature0xAA55));
      dp->sigAA55 = (buffer->bootsector.after.Signature0xAA55 == 0xAA55);
      dp->bootable = recognise_MBR (&buffer->bootsector);

      if (!dp->sigAA55) {
	  DDBG (("disk do not contains an 0x55AA partition scheme\r\n"));
// We want to check for BEER sector (to restore MBR) or treat CDROM session as partitions, so:
//	  dp->nbpartition = 0; /* dp is inited to 0 before calling */
//	  return 2; /* caller will not retry */
	  }
	else {
	  DDBG (("disk may contain partitions, "));
	  /* Handle superfloppy with FAT32, but to boot DVD-RAM we would need to handle ATAPI in first level bootloader */
	  if (buffer->bootsector.before.part2.Signaturebyte0x29 == 0x29 || buffer->bootsector.before.part2.Signaturebyte0x29 == 0x28
		|| ((bootbefore_FAT32_t *)&buffer->bootsector.before)->part2.Signaturebyte0x29 == 0x29) {
	      if (buffer->bootsector.before.part2.Signaturebyte0x29 == 0x29 || buffer->bootsector.before.part2.Signaturebyte0x29 == 0x28)
		  dp->OSdisknumber = buffer->bootsector.before.part2.PhysicaldriveNb;
		else
		  dp->OSdisknumber = ((bootbefore_FAT32_t *)&buffer->bootsector.before)->part2.PhysicaldriveNb;
	      TINYDBG ("disk may containing FAT partitions\r\n");

	      if (buffer->bootsector.before.part1.Mediadescriptor == 0xBB)
		  final_fulldisk_size = ((unsigned long long)buffer->bootsector.before.part1.NbTotalsector << 32)
					+ buffer->bootsector.before.part1.NbTotalsector2;
		else
		  final_fulldisk_size = (buffer->bootsector.before.part1.NbTotalsector != 0)
				    ? buffer->bootsector.before.part1.NbTotalsector
				    : buffer->bootsector.before.part1.NbTotalsector2;
	      final_fulldisk_offset = (long long)(signed int)buffer->bootsector.before.part1.NbHiddensector;
	      DDBG (("disk MBR content: sector size %u, Mediadescriptor 0x%X, SIGNED sector before %d (MBR offset), "
		     "SO final fulldisk offset %lld, final fulldisk size %llU\r\n",
			buffer->bootsector.before.part1.Bytepersector, buffer->bootsector.before.part1.Mediadescriptor,
			buffer->bootsector.before.part1.NbHiddensector, final_fulldisk_offset, final_fulldisk_size));

	      if (final_fulldisk_size <= (unsigned long long)((final_fulldisk_offset < 0 )? -final_fulldisk_offset : final_fulldisk_offset)
		  || buffer->bootsector.before.part1.Mediadescriptor != 0xBB
		  || buffer->bootsector.before.part1.Bytepersector == 0
		  || (buffer->bootsector.before.part1.Bytepersector & 0x80FF) != 0
		  || buffer->bootsector.before.part1.Bytepersector != 1 << (__builtin_ffs (buffer->bootsector.before.part1.Bytepersector) - 1)) { /* not even SAORAB */
		  DDBG (("fulldisk_size <= fulldisk_offset, Bytepersector %u or no 0xBB, reset fulldisk_offset = fulldisk_size = 0;\r\n", buffer->bootsector.before.part1.Bytepersector));
		  final_fulldisk_offset = final_fulldisk_size = 0;
		  }
		else {
		  dp->bytepersector = buffer->bootsector.before.part1.Bytepersector;
		  dp->nbtotalsector = (buffer->bootsector.before.part1.NbTotalsector != 0)
			    ? buffer->bootsector.before.part1.NbTotalsector
			    : buffer->bootsector.before.part1.NbTotalsector2;
		  DDBG (("fully trusting disk content, adjusting bytepersector %u, nbtotalsector %llu\r\n", dp->bytepersector, dp->nbtotalsector));
		  }
	      /* To get floppy with non conventionnal size working, or even CDROM? : */
	      correct_CHS_from_disk_header (dp, &buffer->bootsector.before);
	      }
	    else {
	      DDBG (("no FAT12/16/32 0x29 signature\r\n"));
	      dp->OSdisknumber = 0xFF;
	      final_fulldisk_offset = 0; /* really force it */
	      final_fulldisk_size = 0;
	      }

	  correct_CHS_from_partition (dp, buffer->bootsector.after.bootsect_partition);

#if (DEBUG & DEBUG_DISK) && ((DEBUG & DEBUG_OUTPUT) == SCREEN)
	  DDBG (("Press a key to continue..."));
	  _BIOS_getkey();
#endif
#if !(DISK_SUPPORT & NO_PROBE_PARTITIONS)
	  if (dp->bytepersector == 0) {
	      TINYDBG ("dp->bytepersector == 0\r\n");
	      DDBG (("dp->bytepersector == 0, no media, skip partition & BEER search\r\n"));
	      return 0;
	      }
	    else if (*(unsigned *)((char *)&buffer->bootsector + 0x1A2) == *(unsigned *)"TLDR") { /* WinNT loader: "NTLDR" at 0x1A1 */
	      TINYDBG ("NTLDR marker\r\n");
	      DDBG (("Found 'LDR' signature at 0x1A3, skip partition search\r\n"));
	      }
	    else {
	      dp->nbOSnumber = 5;
	      /*
	       * Here we will re-read sector at LBA = 0, but maybe
	       * buffer->bootsector.before.Bytepersector is higher than 512 and
	       * the space added may be defined to contains new partitions
	       * definitions, or maybe NbHiddensector was valid and not null:
	       */
	      TINYDBG ("Will analyse_partition(0x%X)... ", dp->disknb);
	      analyse_partition (dp, 0, partition, &nbpartition, buffer_fptr);
	      TINYDBG ("analyse_partition(0x%X) returned %u partitions\r\n", dp->disknb, nbpartition);
	      dp->nbOSnumber -= 5;
	      }
#endif
	  }

#if (DEBUG & DEBUG_DISK) && ((DEBUG & DEBUG_OUTPUT) == SCREEN)
      DDBG (("Press a key to continue..."));
      _BIOS_getkey();
#endif
      DDBG (("\r\nSearch BEER sector: "));
      unsigned long long last_used_sector = 0, last_used_sector_extended = 0;
      if (nbpartition != 0) {
	  unsigned last_partition_used __attribute__ ((unused)) = 0;
	  unsigned last_partition_used_extended __attribute__ ((unused)) = 0;
	  unsigned i = nbpartition;
	  while (i--) {
	      unsigned endpart = partition[i].start + partition[i].length;
	      if (partition[i].misc.active == part_extended) {
		  if (endpart > last_used_sector_extended) {
		      last_used_sector_extended = endpart - 1;
		      last_partition_used_extended = i;
		      }
		  }
		else {
		  if (endpart > last_used_sector) {
		      last_used_sector = endpart - 1;
		      last_partition_used = i;
		      }
		  }
	      }
	  DDBG (("last partition used %u, type 0x%X, last sector at %llU; ",
		last_partition_used, partition[last_partition_used].type, last_used_sector));
	  DDBG (("last extended partition used %u, type 0x%X, last sector at %llU\r\n",
		last_partition_used_extended, partition[last_partition_used_extended].type,
		last_used_sector_extended));
	  if (dp->BEER_HostProtectedArea_start && (last_used_sector == 0 || last_used_sector > dp->BEER_HostProtectedArea_start)) {
	      DDBG (("Initial HPA %llu of disk reseted because last_used_sector = %llu\r\n", dp->BEER_HostProtectedArea_start, last_used_sector));
	      dp->BEER_HostProtectedArea_start = 0;
	      }
	  }
	else {
	  DDBG (("no partition scheme on this disk, also reset disk clipping\r\n"));
	  dp->BEER_HostProtectedArea_start = 0;
	  }

      if (dp->error_log.read_media) {
	  DDBG (("Do not search for BEER sector when at least one disk read has failed\r\n"));
	  break;
	  }
      if (dp->nbtotalsector < 10) { /* just as safety, there was no pb with this 18 sectors floppy disk... */
	  DDBG (("Do not search for BEER sector in a disk of %llU sectors\r\n", dp->nbtotalsector));
	  break;
	  }

      TINYDBG ("Will analyseBEERsector(0x%X)... ", dp->disknb);
      err = analyseBEERsector (dp, last_used_sector);
      TINYDBG ("analyseBEERsector(0x%X) returned %u\r\n", dp->disknb, err);
      DDBG (("analyseBEERsector returned %u\r\n", err));
	  /* What am I doing?
	   * dp->BEER_ReservedAreaBootCodeAddress contains a copy of the current
	   * MBR, note that it is _not_ the uninstall copy, just a current copy.
	   * When Gujin analyses a hard disk and find a BEER sector, it checks that the
	   * code and the partition area of the MBR equals those of
	   * ReservedAreaBootCodeAddress. If not equal and write enable, it proposes
	   * to restore them, to update the backup or to do nothing.
	   * So if someone has overwritten your boot sector, just boot with another
	   * Gujin - like a floppy or a CDROM - and you can retore them.
	   * Note that BIOS manufacturer could load this ReservedAreaBootCodeAddress
	   * sector when it exists as an alternative of sector 0 (MBR) to boot (BEER spec).
	   *
	   * The sector at BEER_HostProtectedArea_start + 1 is the MBR of a
	   * virtual disk (virtual floppy disk?) usually of size >= 360K
	   * and <= 32 Mb. It is analysed as a MBR, that is it could contain
	   * a partition scheme or a simple partition like a floppy.
	   * The partition scheme will not cover the complete available area
	   * because you remember we have the BEER sector(s) at the end...
	   * If there isn't any partition scheme - then the filesystem shall
	   * not be as big as the partition.
	   *
	   * So the end of the disk is exactly like a disk in a disk, and
	   * it contains the real bootloader. It can be completely hidden
	   * from any application by (preferably) setting disk size and freeze
	   * (if available) else by set HPA (and password freeze if available).
	   * The real MBR of the disk can be overwritten but we have a copy,
	   * and some BIOS have also the "virus protection" of the MBR.
	   *
	   * Note that someone may say one day that he is controlling completely
	   * the "standard" partition scheme, rewriting it depending on the
	   * operating system to run, so hidden partitions are really hidden -
	   * and no more extended partitions.
	   * All the real partitions are then declared with "struct beer_directory_s",
	   * max number of partitions with the beer sector being the last scanned sector:
	   * (4096 - sizeof (struct beer_s)) / sizeof (struct beer_directory_s) = 62 partitions.
	   * min size of beer area (1 sector) :
	   * (512 - 128) / 64 = 6 partitions.
	   * I am not able right now to load directly such a "struct beer_directory_s".
	   */
      if (   last_used_sector_extended != 0 /* at least one extended partition */
	  && dp->BEER_HostProtectedArea_start > last_used_sector
	  && dp->BEER_HostProtectedArea_start <= last_used_sector_extended) {
	  /*
	   * I have had a problem with some tools to recover partition tables because
	   * they insist on having the (if present) extended partition to cover the
	   * complete free space at end of the disk, so the BEER partition was not analysed
	   * because not in "free" space - just in free space in the extended partition.
	   * So the check for this case and display a message to the user.
	   * The problem is when we will limit the size of the disk or set the HPA,
	   * part of the extended area will not be accessible. Hope that the user
	   * will restore the partition to the saved values - i.e. the backup sector
	   * in the BEER partition (so need to analyse it!).
	   */
	  puts (WARNING_BEER_PARTITION_IN_EXTENDED_SPACE);
	  dp->error_log.beer_in_extended_partition = 1;
	  }
      if (   dp->BEER_HostProtectedArea_start
	  && dp->BEER_HostProtectedArea_start <= last_used_sector)
	  puts (ERROR_BEER_PARTITION_IN_USED_SPACE);
      if (   dp->BEER_HostProtectedArea_start + 1 > last_used_sector
	  && dp->BEER_HostProtectedArea_start + 1 < dp->BEER_sector) {

	  if (dp->bytepersector <= 2048)
	      memcpy (buffer, (unsigned char *)buffer + 2048, 2048);
	    else {
	      /* re-read MBR in buffer_fptr because analyse_partition has loaded
			something else in buffer->bootsector... */
	      if (DISK_readsector (dp, -1, 0, 1, buffer_fptr) != 0)
		  DDBG (("re-read of MBR failed, abort.\r\n"));
	      }

	  if ((beer_partition_restored = beer_boot_sector_update (dp, &buffer->bootsector)) != 0)
	      DDBG (("re-start partition probing after beer MBR update\r\n"));
#if !(DISK_SUPPORT & NO_PROBE_PARTITIONS)
	    else {
	      unsigned oldnbpartition = nbpartition;
	      DDBG (("analyse partitions after BEER alternate boot sector: "));
	      analyse_partition (dp, dp->BEER_HostProtectedArea_start + 1, partition, &nbpartition, buffer_fptr);
	      if (oldnbpartition != nbpartition)
		  DDBG (("found %u partitions in BEER floppy\r\n", nbpartition - oldnbpartition));
		else if (nbpartition >= NB_PARTITION)
		  DDBG (("cannot add BEER partitions, maximum reached\r\n"));
		else {
		  DDBG (("No partitions scheme in BEER floppy, add dummy one of %llu sectors starting at %llU\r\n",
			dp->BEER_sector - (dp->BEER_HostProtectedArea_start + 1),
			dp->BEER_HostProtectedArea_start + 1));
		  partition[nbpartition++] = (struct partition_str) {
			.start = dp->BEER_HostProtectedArea_start + 1,
			.length = dp->BEER_sector - (dp->BEER_HostProtectedArea_start + 1),
			.misc = { .maybe_root = 1, .active = part_active, .beer_partition = 1 },
			.type = DOS_FAT16,
			.name = "B.E.E.R.",
			.OSnumber = 0,
			};
		  }
	      }
#endif
	  }
	else
	  DDBG (("Do not probe BEER partitions: last_used_sector = %llU, dp->BEER_sector = %llU, "
			"dp->BEER_HostProtectedArea_start + 1 = %llU not in between.\r\n\r\n",
			last_used_sector, dp->BEER_sector, dp->BEER_HostProtectedArea_start + 1));
      /* do not know if there can be a infinite loop there - forbid it... */
      } while (beer_partition_restored);

  /*
   * Check if we have a floppy without a partition scheme, whe
   * then add a dummy partition to probe the files.
   * Do it after BEER sector search: if someone erased the whole
   * MBR with 0, no partition are defined but the BEER backup may be used
   * to recover - unless like here we create the dummy floppy partition...
   * not under NO_PROBE_PARTITIONS compilation switch because tiny_img
   * still detect the virtual floppy partition
   * After that we will have a disk with 4 empty primary partitions
   * and a "special" partition, unless we had 'NTLDR' signature at 0x1A1
   * on the MBR of the floppy - so only the "special" partition.
   * Note that there exist CDROMs with an 0xAA55 signature and 4 empty partitions.
   */
  if (all_non_beer_partition_empty (nbpartition, partition)) {
#if DISK_SUPPORT & ATAPI_SUPPORT
ATAPI_readTOC:
      if (dp->access == hardide_atapi) {
	  struct cd_request_sense_str sense;
	  struct cd_TOC_str toc = {};

	  DDBG (("No partition, atapi, read TOC: \r\n"));
	  if (ideAtapiReadToc (dp, &toc, 0) == 0) {
	      unsigned last_partition = (toc.data_length - 2) / sizeof (struct cd_track_str);

	      if (last_partition > nbof (toc.track))
		  last_partition = nbof (toc.track);
	      DDBG (("toc.data_length = %u and nbof (toc.track) = %u so last_partition = %u\r\n",
			toc.data_length, nbof (toc.track), last_partition));
	      for (nbpartition = 0; nbpartition < last_partition; nbpartition++) {
#ifdef COMPATIBILITY_SCSI_READ
		  struct cd_header_str lbaheader = {};
#endif

		  partition[nbpartition].OSnumber = toc.track[nbpartition].track_number;
		  partition[nbpartition].start = toc.track[nbpartition].absolute_CDROM_address;

		  if (   toc.track[nbpartition].track_number < toc.first_track_number
		      || toc.track[nbpartition].track_number > toc.last_track_number) {
		      DDBG (("  track number %u i.e. 0x%X starting at %u not in between first_track_number %u and last_track_number %u\r\n",
				toc.track[nbpartition].track_number, toc.track[nbpartition].track_number,
				toc.track[nbpartition].absolute_CDROM_address,
				toc.first_track_number, toc.last_track_number));
		      partition[nbpartition].length = 0;
		      break;
		      }
		    else if (toc.track[nbpartition+1].absolute_CDROM_address <= partition[nbpartition].start) {
		      DDBG (("  toc.track[%u].absolute_CDROM_address = %u <= partition[nbpartition].start = %llu\r\n",
				nbpartition + 1, toc.track[nbpartition+1].absolute_CDROM_address, partition[nbpartition].start));
		      partition[nbpartition].length = 0;
		      break;
		      }
		    else if (toc.track[nbpartition+1].absolute_CDROM_address - 1 > dp->nbtotalsector) {
		      /* Seems to happen with "cdrecord -tao -multi cdrom.iso" */
		      DDBG (("  toc.track[%u].absolute_CDROM_address = %u - 1 > dp->nbtotalsector = %llu\r\n",
				nbpartition + 1, toc.track[nbpartition+1].absolute_CDROM_address, dp->nbtotalsector));
		      partition[nbpartition].length = dp->nbtotalsector - partition[nbpartition].start;
		      }
		    else {
		      partition[nbpartition].length = toc.track[nbpartition+1].absolute_CDROM_address - partition[nbpartition].start;
		      }
		  if (dp->bytepersector == 512) { /* MBR corrected */
		      partition[nbpartition].length <<= 2;
		      partition[nbpartition].start <<= 2;
		      }
		  STRINIT (partition[nbpartition].name, "session 00");
		  if (toc.track[nbpartition].track_number >= 10) {
		      partition[nbpartition].name[8] += toc.track[nbpartition].track_number / 10;
		      partition[nbpartition].name[9] += toc.track[nbpartition].track_number % 10;
		      }
		    else {
		      partition[nbpartition].name[8] += toc.track[nbpartition].track_number;
		      partition[nbpartition].name[9] = '\0';
		      }
		  /* ideAtapiReadCdRecordedCapacity done, blocksize 2048, lba 333,096
			[track number 170 i.e. 0xAA starting at 333097 not in between first_track_number 1 and last_track_number 1]
		     So first partition LBA 0..333095; Note that sector at LBA 333096 is readable on this disk.
		     BUT so many CDROM/DVD use the last sector as described in their filesystem - so the real bug is
			not to "format" this last sector (so 333097 sectors starting at 0 finishing at 333096):
			This last choice also requests to add one to the size given by ideAtapiReadCdRecordedCapacity()
		  if (toc.track[nbpartition+1].track_number == 0xAA)
		      partition[nbpartition].length -= 1;
		   */
		  partition[nbpartition].misc = (struct partition_misc_str) { .order = nbpartition };
		  DDBG (("  track %u at %llu length %llu (CTRL: 0x%X ADR: 0x%X RSVD1: 0x%X RSVD2: 0x%X) contains %s ",
			partition[nbpartition].OSnumber,
			partition[nbpartition].start,
			partition[nbpartition].length,
			toc.track[nbpartition].control,
			toc.track[nbpartition].adr,
			toc.track[nbpartition].reserved,
			toc.track[nbpartition].reserved2,
			(toc.track[nbpartition].control & 0x04) ? "data" : "music"));
		  /* Do not read header of a music track: it does not exists */
		  if ((toc.track[nbpartition].control & 0x04) == 0) {
		      partition[nbpartition].type = CD_DA_sector_type;
		      DDBG (("CD_DA_sector_type.\r\n"));
		      }
#ifdef COMPATIBILITY_SCSI_READ
		    else if (ideAtapiReadHeader (dp, partition[nbpartition].start, &lbaheader) == 0) {
		      if (lbaheader.cd_data_mode == 2) {
			  partition[nbpartition].type = mode_2_form_1_sector_type;	/* 2048 bytes/sectors */
			  DDBG (("mode_2_form_1_sector_type.\r\n"));
			  }
			else if (lbaheader.cd_data_mode == 1) {
			  partition[nbpartition].type = mode_1_sector_type;	/* 2048 bytes/sectors */
			  DDBG (("mode_1_sector_type.\r\n"));
			  }
			else {
			  partition[nbpartition].type = all_sector_type;
			  DDBG (("all_sector_type.\r\n"));
			  }
		      }
		    else {
// mode_2_formless_sector_type   = 3, /* 2336 bytes/sectors */
// mode_2_form_2_sector_type     = 5  /* 2324 + 4 bytes/sectors */
//		      partition[nbpartition].type = mode_2_formless_sector_type; /* DVD: no headers? */
//		      partition[nbpartition].type = mode_2_form_2_sector_type;	/* DVD: no headers? */
//		      DDBG (("mode_2_formless_sector_type.\r\n"));
		      ideAtapiRequestSense (dp, &sense);
		      partition[nbpartition].type = 0xFF;
		      DDBG (("none_sector_type, 0xFF.\r\n"));
		      }
#else
		    else {
		      partition[nbpartition].type = mode_1_sector_type;	/* do not set it to EMPTY */
		      DDBG (("mode_1_sector_type.\r\n"));
		      /* NOTE: mode_1_sector_type value does trigger warning "{FAT16 on disk 1 part 0 with type 0x2!}"
			when building FAT16 filesystem on CDROM but else works OK */
		      }
#endif
		  } /* for () */
	      } /* ideAtapiReadToc() == 0 */
	    else
	      ideAtapiRequestSense (dp, &sense);
	  } /* dp->access == hardide_atapi */
	else
#endif
	  {
	  DDBG (("Add floppy partition: \r\n"));
	  partition[nbpartition] = (struct partition_str) {
		.start = 0,
		.length = (dp->BEER_HostProtectedArea_start != 0 && dp->BEER_HostProtectedArea_start < dp->nbtotalsector)? (dp->BEER_HostProtectedArea_start+1) : dp->nbtotalsector,
		.misc = { .maybe_root = 1, .active = part_active, .order = nbpartition },
		.type = DOS_FAT12,
		.name = "floppy",
		.OSnumber = 1
		};
	  if (partition[nbpartition].length >= 3 * 1024 * 1024 / 512) {	/* more than 2.88 Mb */
	      if (dp->access == ebios_lba)
		  partition[nbpartition].type = Win95_FAT16LBA;
		else
		  partition[nbpartition].type = BIG_DOS_FAT16;
	      }
	  DDBG (("Special floppy partition of %llU sectors starting at %llu, type 0x%X\r\n",
			partition[nbpartition].length, partition[nbpartition].start, partition[nbpartition].type));
	  nbpartition++;
	  }
      } /* no partition */

  DDBG (("Sort partition: \r\n"));
  TINYDBG ("Will sort_partition(nbpartition %u)... ", nbpartition);
  sort_partition (partition, nbpartition);
  TINYDBG ("sort_partition(0x%X) returned\r\n", dp->disknb);

  dp->nbpartition = nbpartition;
#ifdef DISK_USE_MALLOC
  if (nbpartition != 0) {
      dp->partition = MALLOC (nbpartition * sizeof (struct partition_str), "partitio");
      if (dp->partition == 0) {
	  dp->nbpartition = 0;
	  DDBG (("%s: malloc for %u partitions failed!\r\n",
			__FUNCTION__, nbpartition));
	  }
	else {
	  while (nbpartition --)
	      dp->partition[nbpartition] = partition[nbpartition];
	  }
      }
    else
      dp->partition = 0;
#endif /* DISK_USE_MALLOC */
  DDBG (("%u partition found (max compiled-in %u).\r\n", dp->nbpartition, NB_PARTITION));

#if defined (NB_FREELIST) && NB_FREELIST != 0
  dp->nbfreelist = 0;
  if (dp->partition != 0	/* no freelist if malloc failed */
#if DISK_SUPPORT & DOS_SUPPORT
	&& dp->access != dos_part
#endif
	)
      calculatefreelist (dp);
#endif /* NB_FREELIST */

  dp->fulldisk_offset = final_fulldisk_offset;
  dp->fulldisk_size = final_fulldisk_size;
  return 0;
  }}
#endif /* BIOS_SUPPORT | EBIOS_SUPPORT | IDE_SUPPORT */

/**
 ** Probe all BIOS Hard Drives, and then BIOS Floppy drives:
 **/
#if DISK_SUPPORT & (BIOS_SUPPORT | EBIOS_SUPPORT)
DISK_FCT_PREFIX(diskcheck) static int
diskcheck (unsigned char disk, struct diskparam_str *dp)
  {
  unsigned bios_total = 0, chs_total;

#if DISK_SUPPORT & BIOS_SUPPORT
  /* On some PC, BIOSDISK_getparam returns values (probably
     the CMOS type of slave IDE) even for a non existant drive.
     EBIOS can also return 0 head, 0 sector, 0 track ... */

  if (BIOSDISK_gettype(disk, &dp->biostype, &bios_total) == 0) {
      TINYDBG ("BIOSDISK_gettype() abscent\r\n");
      DDBG ((" [abscent] "));
      return 1;
      }
#endif /* BIOS_SUPPORT */

#if DISK_SUPPORT & EBIOS_SUPPORT
  if (EBIOSDISK_probe (disk, dp) == 0 && EBIOSDISK_getparam (disk, dp) == 0) {
      TINYDBG ("EBIOSDISK_probe() & EBIOSDISK_getparam() passed\r\n");
      /* BIOS developpers cannot read the DOC for ebios_bitmap.extended_fct, check if really EBIOS is not supported: */
      if (dp->access == bios_chs) {
	  unsigned char a_sector[dp->bytepersector]; /* has been inited as ebiosinfo.bytepersector */
	  unsigned read = 1;
	  TINYDBG ("only BIOS, force check EBIOS... ");
	  /* try read twice if first error is "disk changed" */
	  if (   _EBIOSDISK_RWsector (dp->disknb, 0x10, 1, stack_adr(a_sector), read, 0) == 0
	      || _EBIOSDISK_RWsector (dp->disknb, 0x10, 1, stack_adr(a_sector), read, 0) == 0) { /* ISOFS sector */
	      printf (INFO_DISK_X_EXTENDED_BIOS_WITHOUT_EXTENDED_FCT_BUT_WORKS, disk);
	      dp->access = ebios_lba;
	      }
	  }
      if (dp->access == ebios_lba) {
	  /* Note that on my new motherboard, the EBIOS size is reported as
	   * zero if the drive is not LBA, only CHS (internal IDE). The drive clearly
	   * report correctly its number of sector for IDE identify in the
	   * two field "current_capacity_in_sector" and "lba_capacity", with
	   * the correct endianess. Another HD with wrong endianess report also 0.
	   * One more bug.
	   * I checked that with dp->nbtotalsector null the BIOS and the EBIOS
	   * interface return error even for reading the MBR.
	   * (error 0x4: sector not found)
	   */
#if !(DISK_SUPPORT & NO_PROBE_PARTITIONS)
	  unsigned char status;
	  unsigned short tmpcx, tmpdx;
	  farptr DBTadr;

	  TINYDBG ("EBIOS, check BIOS_getparam... ");
	  DDBG ((" [BIOS getparam for C/H/S partitions: "));
	  if (_BIOSDISK_getparam (disk, &tmpcx, &tmpdx, &DBTadr, &dp->drivetype, &status)) {
	      DDBG (("error _BIOSDISK_getparam]"));
	      }
	    else if ((tmpcx & 0x3F) == 0) {
	      DDBG (("_BIOSDISK_getparam tmpcx = 0x%X, tmpdx = 0x%X, unbelivable]", tmpcx, tmpdx));
	      }
	    else {
	      dp->bios_maxhead = tmpdx >> 8;
	      dp->bios_nbsectorpertrack = tmpcx & 0x3F;
	      dp->bios_nbcylinder = (tmpcx >> 8) + ((tmpcx << 2) & 0x0300) + 1;
	      DDBG (("%u/%u/%u, type %u] ", dp->bios_nbcylinder, dp->bios_maxhead + 1,
			dp->bios_nbsectorpertrack, dp->drivetype));
	      }
#endif
	  chs_total = dp->nbhead * dp->nbsectorpertrack * dp->nbcylinder;

	  if (dp->nbtotalsector == 0) {
	      if (chs_total == 0) {
		  DDBG (("; Invalid drive (C/H/S) and unknown EBIOS drive size!\r\n"));
		  return 1;
		  }
		else if (bios_total < 0x100) {
		  DDBG ((" BIOSDISK_gettype bios_total 0x%X, use Cylinder*Head*Sector = %u] ",
				bios_total, chs_total));
		  dp->nbtotalsector = chs_total;
		  }
		else {
		  dp->nbtotalsector = bios_total + (unsigned)dp->nbsectorpertrack * dp->nbhead;

		  if (dp->nbtotalsector == chs_total)
		      DDBG ((" usual BIOSDISK_gettype bios_total bug, use C*H*S == BIOSDISK_gettype() + H * S] "));
		    else if (chs_total == 16 * 63 * 1024 && bios_total >= chs_total)
		      DDBG ((" maximum C/H/S description, use BIOSDISK_gettype() + H * S] "));
		    else if ((((unsigned)dp->nbtotalsector * 10 / chs_total) - 9) <= 2)
		      DDBG ((" 10%% OK, use BIOSDISK_gettype() + H * S] "));
		    else {
		      DDBG ((" 10%% failed, use Cylinder*Head*Sector = %u] ", chs_total));
		      dp->nbtotalsector = chs_total;
		      }
		  }
	      }
	    else {
	      DDBG ((" [use EBIOS nbtotalsector; "));
	      if (dp->nbtotalsector < 16*63*16384 && chs_total != 0) {
		  if (chs_total == dp->nbtotalsector)
		      DDBG (("usual EBIOS total == C*H*S"));
		    else
		      DDBG (("UNUSUAL EBIOS total = %llU != C*H*S = %u, delta %lld",
				dp->nbtotalsector, chs_total,
				dp->nbtotalsector - chs_total));
		  }
#if DISK_SUPPORT & BIOS_SUPPORT
	      if (bios_total < 0x100)
		  DDBG (("; UNUSUAL BIOS total == 0x%X", bios_total));
		else if (bios_total + dp->nbhead * dp->nbsectorpertrack == dp->nbtotalsector)
		  DDBG (("; usual BIOS total + H*S == EBIOS total"));
		else {
		  DDBG (("; UNUSUAL BIOS total + H*S = %U != EBIOS total = %llU",
			bios_total + dp->nbhead * dp->nbsectorpertrack, dp->nbtotalsector));
		  if (dp->nbhead != 0 && dp->nbsectorpertrack != 0)
		      DDBG ((" (EBIOS_total - BIOS_total)/H/S = %u",
				((unsigned)dp->nbtotalsector - bios_total)
				/ dp->nbhead / dp->nbsectorpertrack));
		  }
#endif /* BIOS_SUPPORT */
	      DDBG (("]\r\n"));
	      }

	  STRINIT (dp->diskname, "EBIOS 0x00");
	  dp->diskname[8] += disk >> 4;
	  dp->diskname[9] += disk & 0xF;
	  return 0;
	  }
	else {
	  DDBG((" [no extended fct] "));
	  printf (INFO_DISK_X_EXTENDED_BIOS_WITHOUT_EXTENDED_FCT_USE_BIOS, disk);
	  dp->error_log.no_ebios_fct = 1;
	  /* FIXME: In this case (only bit "enhanced disk drive (EDD) functions
		(AH=48h,AH=4Eh) supported" set), should we still use the
		EBIOS total sector? (We shall _not_ use EBIOS C/H/S) */
	  }
      }
#endif /* EBIOS_SUPPORT */

#if DISK_SUPPORT & BIOS_SUPPORT
  if (BIOSDISK_getparam (disk, dp) == 0) {
      TINYDBG ("BIOSDISK_getparam() only passed\r\n");
      if ((chs_total = dp->nbhead * dp->nbsectorpertrack * dp->nbcylinder) == 0) {
	  /* at least one field (nbhead, nbsectorpertrack, or nbcylinder) equals 0: */
	  DDBG (("; Invalid drive (C/H/S)!\r\n"));
	  return 1;
	  }
      DDBG ((" [nbtotalsector: EBIOS: %llU, BIOSDISK_gettype(): %u, C*H*S: %u: ",
		dp->nbtotalsector, bios_total, chs_total));

      if (dp->nbtotalsector != 0) {
	  DDBG (("use EBIOS] "));
	  }
	else if (bios_total < 0x100) {
#if 0
	  if (dp->drivetype == 0x10) {	/* USB drive simulated as USB-FDD - but not all of them need the correction... */
	      DDBG ((" BIOSDISK_gettype bios_total too low, use Cylinder*Head*Sector + Head*Sector] "));
	      dp->nbtotalsector = chs_total + dp->nbhead * dp->nbsectorpertrack;
	      }
#endif
	  DDBG ((" BIOSDISK_gettype bios_total too low, use Cylinder*Head*Sector] "));
	  dp->nbtotalsector = chs_total;
	  }
	else /* dp->nbtotalsector == 0 */ {
	  dp->nbtotalsector = bios_total + (unsigned)dp->nbsectorpertrack * dp->nbhead;

	  if (dp->nbtotalsector == chs_total)
	      DDBG ((" usual BIOSDISK_gettype bios_total bug, use C*H*S == BIOSDISK_gettype() + H * S] "));
	    else if (chs_total == 16 * 63 * 1024 && bios_total >= chs_total)
	      DDBG ((" maximum C/H/S description, use BIOSDISK_gettype() + H * S] "));
	    else if ((((unsigned)dp->nbtotalsector * 10 / chs_total) - 9) <= 2) {
	      DDBG ((" 10%% OK, use BIOSDISK_gettype() + H * S and increase by one dp->nbcylinder] "));
	      /* else "analyseBEERsector: BEER sector at... error too high cylinder: 1010 (max 1010)" */
	      dp->nbcylinder += 1;
	      }
	    else {
	      if (bios_total == disk)
		  DDBG ((" BIOSDISK_gettype() nbtotalsector not set (still disk number), use C*H*S"));
		else
		  DDBG ((" 10%% failed, use Cylinder*Head*Sector"));
	      dp->nbtotalsector = chs_total;
	      }
	  }
#if !(DISK_SUPPORT & NO_PROBE_PARTITIONS)
      dp->bios_nbcylinder = dp->nbcylinder; /* also increase by one dp->nbcylinder */
      dp->bios_maxhead = dp->nbhead - 1;
      dp->bios_nbsectorpertrack = dp->nbsectorpertrack;
      DDBG (("\r\n  [use for C/H/S partitions: %u/%u/%u]\r\n  ",
			dp->bios_nbcylinder, dp->bios_maxhead + 1,
			dp->bios_nbsectorpertrack));
#endif
      STRINIT (dp->diskname, "BIOS 0x00");
      dp->diskname[7] += disk >> 4;
      dp->diskname[8] += disk & 0xF;
      return 0;
      }
  DDBG ((" Invalid drive (getparam).\r\n"));
#else /* BIOS_SUPPORT */
  DDBG (("; BIOS disk interface not compiled in!\r\n"));
#endif /* BIOS_SUPPORT */

  return 1;
  }

DISK_FCT_PREFIX(probeBIOSdisk) static inline unsigned
probeBIOSdisk (unsigned disk, struct diskparam_str *dp)
  {
  DDBG (("\r\n%s disk 0x%X: ", (disk >= 0x80)? "Hard" : "Floppy", disk));
  checkptr (dp, "probeBIOSdisk invalid parameter.");
#ifdef GCC_OPTIMISED_MEMSET
  *dp = (struct diskparam_str) {};
#else
  _memset (dp, 0, sizeof (struct diskparam_str));
#endif
  dp->disknb = disk;
  if (diskcheck (disk, dp) != 0) {
#if 0 // untested
      if (disk >= 0x99 && disk <= 0xF9) {
	  /* Probing BIOS disk 0x9F may fail altogether (late stop emulation) */
	  // dp->biostype =
	  // dp->ebios_version
	  // dp->ebios_bitmap
	  dp->access = ebios_lba;
	  dp->ebios_bitmap.extended_fct = 1;
	  // dp->nbhead = ebiosinfo.nbhead;
	  // dp->nbsectorpertrack = ebiosinfo.nbsectorpertrack;
	  // dp->nbcylinder = ebiosinfo.nbcylinder;
	  dp->bytepersector = 2048;
	  dp->nbtotalsector = 0x800000; /* 16 Gb */
	  // dp->infobit = ebiosinfo.infobit;
	  // dp->bios_maxhead = ;
	  // dp->bios_nbsectorpertrack = ;
	  // dp->bios_nbcylinder = ;
	  STRINIT (dp->diskname, "CDEMUL");
	  } else
#endif
      return 1;
      }
#if (DEBUG & DEBUG_DISK) && ((DEBUG & DEBUG_OUTPUT) == SCREEN)
      DDBG (("Press a key to continue..."));
      _BIOS_getkey();
#endif
  TINYDBG ("diskcheck(0x%x, 0x%X) passed, try analysebootsector\r\n", disk, dp);
//  dp->nbpartition = 0;	// done by memset()
//  dp->partition = 0;
  if (analysebootsector (dp) == 1)
      return 1;
  TINYDBG ("analysebootsector(0x%X) passed\r\n", dp);
  DDBG ((", %u partitions.\r\n\r\n", dp->nbpartition));
  return 0;
  }

DISK_FCT_PREFIX(probe_bios) static inline unsigned
probe_bios (unsigned nbdisk, struct diskparam_str *param)
  {
  signed short base_disk;

  for (base_disk = 0x80; base_disk >= 0; base_disk -= 0x80) {
      unsigned char nb_consecutive_fail = 0;
      unsigned char disk, max_disk, max_disk1;
      PRINTF ("BIOS 0x%X...\r", base_disk);

      if (base_disk) {
	  max_disk = DI.nb_bios_hd;
	  if (!copy_gujin_param.attrib.probe_bios_hard_disk) {
	      DDBG (("Skiping HD probe due to current attributes.\r\n"));
	      continue;
	      }
	  max_disk1 = _BIOSDISK_get_nb_harddisk();
	  if (max_disk1 > max_disk && max_disk1 < 0x80)
	      max_disk = max_disk1;
	  DDBG (("Probing BIOS Hard drives (total = %u):", max_disk));
	  }
	else {
	  max_disk = DI.nb_bios_fd;
	  if (!copy_gujin_param.attrib.probe_bios_floppy_disk) {
	      DDBG (("Skiping floppy probe due to current attributes.\r\n"));
	      continue;
	      }
	  max_disk1 = _BIOS_equipment_flags().nb_floppy_less_1 + 1;
	  if (max_disk1 > max_disk)
	      max_disk = max_disk1;
	  DDBG (("Probing "
#if !(DISK_SUPPORT & NO_FLOPPY_SUPPORT)
		"BIOS/EBIOS "
#else
		"EBIOS only "
#endif
		"floppy drives (total = %u):", max_disk));
	  }
#if (DEBUG & DEBUG_DISK) && ((DEBUG & DEBUG_OUTPUT) == SCREEN)
      DDBG (("Press a key to continue (nb_consecutive_fail %u)...", nb_consecutive_fail));
      _BIOS_getkey();
#endif

      if (max_disk == 0 || max_disk > NB_DISK)
	  max_disk = NB_DISK;	/* the BIOS value is invalid */

      for (disk = 0; disk < max_disk && nbdisk < NB_DISK && nb_consecutive_fail <= 3; disk++) {
	  if (probeBIOSdisk (base_disk + disk, &param[nbdisk]) == 0) {
	      nbdisk++;
	      nb_consecutive_fail = 0;
	      }
	    else
	      nb_consecutive_fail++;
#if (DEBUG & DEBUG_DISK) && ((DEBUG & DEBUG_OUTPUT) == SCREEN)
	  DDBG (("Press a key to continue (nb_consecutive_fail %u)...", nb_consecutive_fail));
	  _BIOS_getkey();
#endif
	  }
      DDBG ((" done probe_bios base_disk 0x%X\r\n\r\n", base_disk));
      }
  DDBG ((" %s end\r\n", __FUNCTION__));
  return nbdisk;
  }
#endif /* BIOS_SUPPORT | EBIOS_SUPPORT */

//#if DISK_SUPPORT & (EBIOS_SUPPORT | IDE_SUPPORT)
#if DISK_SUPPORT & IDE_SUPPORT
UTIL_FCT_PREFIX(reorder_IDE_for_linux) static void
reorder_IDE_for_linux (void)
  {
  /*
   * The only place where we modify the order of IDEs:
   * If the IDE interface exists, it is in the following order.
   * If the IDE address is not listed, relative initial order is kept,
   * and they are at end.
   */
  static const unsigned short idearray[] = {
	0x1F0, 0x170,
	0x1E8, 0x168,
	0x1E0, 0x160,
//	0xe820, 0xe828,
//	0xB400, 0xBC00, /* HighPoint HPT 372 IDE controller */
//	0xD400, 0xD800  /* HighPoint HPT366/HPT370 */
	};
  unsigned short order;

  ZDBG (("%s: %u IDE found\r\n", __FUNCTION__, DI.nb_IDE_found));
  for (order = 0; order < DI.nb_IDE_found; order++)
     ZDBG (("0x%X, ", DI.IDE_found[order].ideIOadr));
  ZDBG (("\r\n"));

  for (order = 0; order < nbof(idearray); order++) {
      unsigned short cpt;
      for (cpt = order + 1; cpt < DI.nb_IDE_found; cpt++)
	  if (DI.IDE_found[cpt].ideIOadr == idearray[order])
	      break;
      if (cpt < DI.nb_IDE_found) {
	  struct IDE_found_str save = DI.IDE_found[cpt];
	  while (cpt > order) {
	      DI.IDE_found[cpt] = DI.IDE_found[cpt-1];
	      cpt--;
	      }
	  DI.IDE_found[cpt] = save;
	  }
      }

  for (order = 0; order < DI.nb_IDE_found; order++)
     ZDBG (("0x%X, ", DI.IDE_found[order].ideIOadr));
  ZDBG (("\r\n"));
  }
#endif /* (EBIOS_SUPPORT | IDE_SUPPORT) */

/**
 ** Probe all IDE:
 **/
#if DISK_SUPPORT & IDE_SUPPORT
DISK_FCT_PREFIX(get_native_max_hpa) unsigned long long
get_native_max_hpa (struct diskparam_str *dp)
  {
  unsigned long long ide_max_native48_hpa;
  unsigned ide_max_native_hpa;

  if (dp->ide_attribute.lba48)
      if (IDE_get_native48_max_address (dp, &ide_max_native48_hpa) == 0)
	  return ide_max_native48_hpa;
  if (IDE_get_native_max_address (dp, &ide_max_native_hpa) == 0)
      return ide_max_native_hpa;
  return 0;
  }

DISK_FCT_PREFIX(probeIDEdisk_attribute) static struct ide_attribute_str
probeIDEdisk_attribute (struct diskparam_str *dp, const ata_identify_t *buff, farptr buffer_farptr, unsigned restore_config)
  {
  struct ide_attribute_str ide_attribute = {};

  DDBG ((" [ATA major_version mask = 0x%X] ", buff->major_version_number));
/* GCC-4.2: warning: dereferencing type-punned pointer will break strict-aliasing rules
  if (   (   *(unsigned short *)&buff->command_supported1 == 0
	  && *(unsigned short *)&buff->command_supported2 == 0
	  && *(unsigned short *)&buff->command_supported3 == 0)
      || (   *(unsigned short *)&buff->command_supported1 == 0xFFFF
	  && *(unsigned short *)&buff->command_supported2 == 0xFFFF
	  && *(unsigned short *)&buff->command_supported3 == 0xFFFF)) {
 */
  if (   memeql (&buff->command_supported1, "\0\0\0\0\0\0", 6)
     || memeql (&buff->command_supported1, "\xFF\xFF\xFF\xFF\xFF\xFF", 6)) {
      DDBG (("ATA1, ATA2 or simple ATA3 device.\r\n\t"));
      return ide_attribute;
      }

  if (buff->command_supported2.DevConfOverlay && restore_config) {
      ata_config_t config;
      unsigned restore_config_done = 0;
      DDBG (("DevConfOverlay %sabled.\r\n\t", buff->command_enabled2.DevConfOverlay? "en" : "dis"));

      DDBG (("Before IDE_restore_config(): "));
      for (;;) {
	  if (ideReadConfig (dp, stack_adr(&config)) != 0) {
	      dp->error_log.ideReadConfig_failed = 1;
	      DDBG (("ideReadConfig failed\r\n\t"));
	      break;
	      }
	    else {
	      unsigned char checksum = 0;
	      unsigned cpt = 512;

	      while (cpt --)
		  checksum += ((unsigned char *)&config)[cpt];

	      /* Version 2 adds UDMA6, streaming, Force unit access, smart selective, smart conveyance */
	      DDBG (("ideReadConfig success: revision %u (== 1/2?), signature 0x%X (== 0xA5?), "
		     "calculated checksum: 0x%X, mask multiword_dma 0x%X, "
		     "ultra_dma 0x%X, maxlba %llU, cmdset 0x%X i.e. ",
		     config.revision, config.integrity.signature,
		     checksum,
		     *CAST(unsigned short *, &config.multiword_dma),
		     *CAST(unsigned short *, &config.ultra_dma), config.max_lba,
		     *CAST(unsigned *, &config.feature)));
	      if (config.feature.smart_feature)
		  DDBG (("smart_feature, "));
	      if (config.feature.smart_selftest)
		  DDBG (("smart_selftest, "));
	      if (config.feature.smart_errorlog)
		  DDBG (("smart_errorlog, "));
	      if (config.feature.security)
		  DDBG (("security, "));
	      if (config.feature.standby_powerup)
		  DDBG (("standby_powerup, "));
	      if (config.feature.RW_DMA_queued)
		  DDBG (("RW_DMA_queued, "));
	      if (config.feature.acoustic)
		  DDBG (("acoustic, "));
	      if (config.feature.host_prot_area)
		  DDBG (("host_prot_area, "));
	      if (config.feature.lba48)
		  DDBG (("lba48, "));
	      }
	  IDE_restore_config (dp);
	  if (restore_config_done++) {
	      int tmp __attribute__ ((unused));
	      unsigned atapi_signature;

	      /* TODO: save UDMA ATA level for master and slave */
	      dp->maximum_multiDMA = config.multiword_dma;
	      dp->maximum_ultraDMA = config.ultra_dma;
	      dp->maximum_feature = config.feature;
	      dp->config_max_lba = config.max_lba;
	      DDBG (("\r\n\tRE-reading ideIdentify buffer: "));
	      tmp = ideIdentify (dp, buffer_farptr, &atapi_signature);
	      DDBG (("returns 0x%X, atapi_signature = 0x%X\r\n\t", tmp, atapi_signature));
	      break;
	      }
	    else {
	      dp->initial_multiDMA = config.multiword_dma;
	      dp->initial_ultraDMA = config.ultra_dma;
	      dp->initial_feature = config.feature;
	      }
	  DDBG (("\r\n\tAfter  IDE_restore_config(): "));
	  }
      ide_attribute.config_overlay = 1;
      }
    else if (buff->command_supported2.DevConfOverlay)
      ide_attribute.config_overlay = 1;

  if (buff->command_supported1.smart) {
      DDBG (("SMART supported and %sactive; ", buff->command_enabled1.smart? "" : "in"));
      ide_attribute.smart = 1;
      }

  if (buff->command_supported1.security) {
      DDBG (("SECURE supported and %sactive; ", buff->command_enabled1.security? "" : "in"));
      ide_attribute.security = 1;
      dp->ide_master_password_revision = buff->master_password_revision;
      dp->security_status = buff->security_status;
      }

  if (buff->command_supported1.removable) {
      DDBG (("REMOVEABLE supported and %sactive; ", buff->command_enabled1.removable? "" : "in"));
      ide_attribute.removable = 1;
      }

  if (buff->command_supported2.lba48) {
      DDBG (("lba48 %sabled, max lba %llU, i.e. %u Gb; ", buff->command_enabled2.lba48? "en" : "dis",
		buff->max_user_lba48, (unsigned)(buff->max_user_lba48 >> 21)));
      ide_attribute.lba48 = (buff->max_user_lba48 != 0);
      }

  if (buff->command_supported2.SAOReservedAreaBoot) {
      DDBG ((" SAOReservedAreaBoot supported and %sabled.", buff->command_enabled2.SAOReservedAreaBoot? "en" : "dis"));
      ide_attribute.SAORAB = 1;
      }

  if (buff->command_supported1.host_protected_area) {
      DDBG (("\r\n\tHOST protected area supported and %sactive; ", buff->command_enabled1.host_protected_area? "" : "in"));
#if DEBUG & DEBUG_DISK
      if (buff->command_supported1.removable)
	  DDBG ((" [removeable implemented, use of IDE_get_native*_max_address prohibited] "));
	else {
	  unsigned ide_max_native_hpa;
	  /* FIXME: Shall we use that length as the disk length so that the BEER sector will be scanned
		at the right place - even if those sectors are unreadable? */
	  if (ide_attribute.lba48) {
	      unsigned long long ide_max_native48_hpa;
	      if (IDE_get_native48_max_address (dp, &ide_max_native48_hpa) == 0) {
		  if (ide_max_native48_hpa < 10 * 0xFFFFFFFFULL)
		      DDBG (("max native lba48: %llU; ", ide_max_native48_hpa));
		    else
		      DDBG (("max native lba48: 0x%llX; ", ide_max_native48_hpa));
		  }
		else
		  DDBG (("max native lba48: failed; "));
	      }
	  if (IDE_get_native_max_address (dp, &ide_max_native_hpa) == 0)
	      DDBG (("max native lba: %U;\r\n\t", ide_max_native_hpa));
	    else {
	      DDBG (("max native lba unavailable;\r\n\t"));
	      }
      }
#endif /* DEBUG_DISK */
      ide_attribute.host_protected_area = 1;
      }
  return ide_attribute;
  }

/* By default, mask == 0 and everything shall be frozen: */
DISK_FCT_PREFIX(freeze_IDE) void
freeze_IDE (struct freeze_IDE_s mask)
  {
//#undef DBG
//#define DBG(X) printf X

  unsigned disk, disksizechanged = 0;

  for (disk = 0; disk < DI.nbdisk; disk++) {
      struct diskparam_str *dp = &DI.param[disk];
      struct ide_attribute_str ide_attribute = dp->ide_attribute;

      DBG (("\r\nDisk %s: ", dp->diskname[0]? (const char *)dp->diskname : "unnamed"));
      if (dp->access == hardide_atapi || dp->access == dos_part) {
	  DBG (("do not freeze CDROMs or DOS disks"));
	  continue;
	  }

      if (   dp->access != hardide_chs
	  && dp->access != hardide_lba
	  && dp->access != hardide_lba48) {
	  unsigned atapi_signature, cpt;
	  int tmp;
#if 1
	  ata_identify_t buffer, *buff = &buffer;
	  farptr buffer_farptr = stack_adr (buff);
#else
	  ata_identify_t *buff = (ata_identify_t *)fourKbuffer;
	  farptr buffer_farptr = data_adr (buff);
#endif
	  if (dp->access != ebios_lba || dp->ideIOadr == 0 || dp->ideIOctrladr == 0) {
	      DBG (("BIOS only or unknown ideIOadr/ideIOctrladr"));
	      continue;
	      }
	  /* IDE disks are listed before [E]BIOS disks: */
	  for (cpt = 0; cpt < disk; cpt++) {
	      if (   dp->ideIOadr == DI.param[cpt].ideIOadr
		  && dp->ideIOctrladr == DI.param[cpt].ideIOctrladr
		  && (dp->lba_slave_mask & 0x30) == (DI.param[cpt].lba_slave_mask & 0x30)) {
		  DBG (("EBIOS listed also as disk %s", DI.param[cpt].diskname));
		  break;
		  }
	      }
	  if (cpt < disk)
	      continue;

	  /* TODO: We have to freeze if starting a new kernel but not if returning to DOS */
#if !(DEBUG & (DEBUG_DISK | DEBUG_FS))
	  if (BOOT1_DOS_running())
	      continue;
#endif

	  DDBG (("EBIOS only disk, reading ideIdentify buffer: "));
	  tmp = ideIdentify (dp, buffer_farptr, &atapi_signature);
	  DDBG (("returns 0x%X, atapi_signature = 0x%X\r\n\t", tmp, atapi_signature));
	  if (tmp != 0)
	      continue;
	  ide_attribute = probeIDEdisk_attribute (dp, buff, buffer_farptr, 0);
	  }

      if (mask.unfrozen_password)
	  DBG (("do not freeze password, "));
	else if (ide_attribute.security == 0)
	  DBG (("do not have security feature, "));
	else {
	  int ret __attribute__ ((unused));

	  DBG (("freeze password system: "));
	  ret = IDE_freeze_password (dp);
	  DBG (("%d, ", ret));
	  }

      if (mask.disk_size_unset)
	  DBG (("do not set disk size, "));
	else if (!ide_attribute.config_overlay)
	  DBG (("do not have config overlay, "));
	else if (dp->BEER_disksize > dp->nbtotalsector)
	  DBG (("BEER_disksize (%llU) bigger than dp->nbtotalsector (%llU), ",
		dp->BEER_disksize, dp->nbtotalsector));
	else if (dp->BEER_disksize == 0)
	  DBG (("BEER_disksize null, "));
	else {
	  int ret;
#if 1
	  ata_config_t conf, *config = &conf;
	  farptr config_farptr = stack_adr (config);
#else
	  ata_config_t *config = (ata_config_t *)fourKbuffer;
	  farptr config_farptr = data_adr (config);

	  if (sizeof (fourKbuffer) < sizeof (ata_config_t))
	      __ERROR();	/* the linker will produce an error here */
#endif

	  DBG (("ideReadConfig: "));
	  ret = ideReadConfig (dp, config_farptr);
	  DBG (("return %d, ", ret));
	  if (ret == 0) {
	      unsigned char checksum = 0;
	      unsigned cpt;

	      for (cpt = 0; cpt < 512; cpt++)
		  checksum += ((unsigned char *)config)[cpt];
	      DDBG (("ide Config revision %u (== 1?), signature 0x%X (== 0xA5?), "
			"calculated checksum: 0x%X, maxlba %llU, ",
			config->revision, config->integrity.signature,
			checksum, config->max_lba));
	      if (checksum != 0)
		  DBG (("checksum invalid, "));
		else {
		  if (config->max_lba != dp->BEER_disksize) {
		      disksizechanged = 1;
		      config->max_lba = dp->BEER_disksize - 1;
		      DDBG (("set disksize/max_lba to BEER_disksize - 1, i.e. %llU, ",
				config->max_lba));
		      }
		  /* Do not change them by default: */
		  /* FIXME: ATA6 master and ATA5 slave => force ATA5 UDMA on both */
		  config->multiword_dma = dp->initial_multiDMA;
		  config->ultra_dma = dp->initial_ultraDMA;
		  config->feature = dp->initial_feature;
		  DDBG (("multiword_dma 0x%X, ultra_dma 0x%X, feature 0x%X, ",
			*CAST(unsigned short *, &config->multiword_dma),
			*CAST(unsigned short *, &config->ultra_dma),
			*CAST(unsigned short *, &config->feature)));

		  config->feature.security &= mask.unhidden_password;
		  if (!mask.unhidden_password)
		      DDBG (("hide password, "));
		  config->feature.host_prot_area &= !mask.hidden_hpa;
		  if (mask.hidden_hpa)
		      DDBG (("hide hpa, "));
		  checksum = 0;
		  for (cpt = 0; cpt < 512; cpt++)
		      checksum += ((unsigned char *)config)[cpt];
		  config->integrity.checksum -= checksum;
		  DBG (("ideWriteConfig: "));
		  ret = ideWriteConfig (dp, config_farptr);
		  DBG (("return %d, ", ret));
//		  if (ret == 0) {
//		      DDBG (("set BIOS disk size %llU, ", config->max_lba));
//			at EBDA + 0x3D for 0x80, EBDA + 0x4D for 0x81 ?
//			or at INT 0x41, INT 0x46
//		      }
		  }
	      }
	  }

      if (mask.HostProtectedArea_unset)
	  DBG (("do not set Host Protected Area, "));
	else if (mask.hidden_hpa)
	  DBG (("HPA has just been hidden, ")); /* cannot set before hidding on some HDs */
	else if (dp->BEER_HostProtectedArea_start == 0)
	  DBG (("BEER_HostProtectedArea_start not set, "));
	else if (dp->BEER_HostProtectedArea_start >= dp->nbtotalsector)
	  DBG (("BEER_HostProtectedArea_start %llU too high, max dp->nbtotalsector %llU, ",
		dp->BEER_HostProtectedArea_start, dp->nbtotalsector - 1));
	else if (ide_attribute.host_protected_area == 0)
	  DBG (("do not have Host Protected Area feature, "));
	else {
	  int ret = -1;

	  if (ide_attribute.lba48) {
	      unsigned long long max48;

	      DBG (("Get native 48 Host Protected Area "));
	      ret = IDE_get_native48_max_address (dp, &max48);
	      DBG (("returns %d i.e. %llU sectors, ", ret, max48));
	      if (ret == 0) {
		  if (dp->BEER_HostProtectedArea_start >= max48) {
		      if (dp->BEER_HostProtectedArea_start == max48)
			  DBG (("same BEER_HostProtectedArea_start and max48, "));
			else
			  DBG (("BEER_HostProtectedArea_start IS OVER max48, "));
		      }
		    else {
		      DBG (("Set native 48 Host Protected Area to %llU (preserve_over_reset: 0) ",
				dp->BEER_HostProtectedArea_start));
		      ret = IDE_set_native48_max_address (dp, dp->BEER_HostProtectedArea_start, 0);
		      disksizechanged = 1;
		      DBG (("returns %d, ", ret));
		      }
		  }
	      }
	  if (ret != 0) {
	      unsigned max28;

	      DBG (("Get native Host Protected Area "));
	      ret = IDE_get_native_max_address (dp, &max28);
	      DBG (("returns %d i.e. %u sectors, ", ret, max28));
	      if (ret == 0) {
		  if (dp->BEER_HostProtectedArea_start >= max28) {
		      if (dp->BEER_HostProtectedArea_start == max28)
			  DBG (("same BEER_HostProtectedArea_start and max28, "));
			else
			  DBG (("BEER_HostProtectedArea_start IS OVER max28, "));
		      }
		    else {
		      DBG (("Set native Host Protected Area to %llU (preserve_over_reset: 0): ",
				dp->BEER_HostProtectedArea_start));
		      ret = IDE_set_native_max_address (dp, dp->BEER_HostProtectedArea_start, 0);
		      disksizechanged = 1;
		      DBG (("returns %d, ", ret));
		      }
		  }
	      }
	  if (dp->BEER_sector) { /* not if we just set back the HPA to the value we found at init */
	      if (!mask.no_set_password_hpa) {
		  int tmp __attribute__ ((unused));

		  DBG (("IDE_set_password_set_max "));
		  tmp = IDE_set_password_set_max (dp, dp->set_max_password);
		  DBG (("returns %d, ", tmp));
		  }
	      if (!mask.unlocked_hpa) {
		  int tmp __attribute__ ((unused));

		  DBG (("IDE_lock_set_max "));
		  tmp = IDE_lock_set_max (dp);
		  DBG (("returns %d, ", tmp));
		  }
	      if (!mask.unfrozen_hpa) {
		  int tmp __attribute__ ((unused));

		  DBG (("IDE_freeze_set_max "));
		  tmp = IDE_freeze_set_max (dp);
		  DBG (("returns %d, ", tmp));
		  }
	      }
	  }

      if (mask.unfrozen_config)
	  DBG (("do not freeze config, "));
	else if (ide_attribute.config_overlay == 0)
	  DBG (("do not have config overlay feature, "));
	else {
	  int ret __attribute__ ((unused));

	  DBG (("freeze config system: "));
	  ret = IDE_freeze_config (dp);
	  DBG (("%d, ", ret));
	  }
      IDE_reset_DCR_for_HTP_BIOS (dp);

#if 0
      if (dp->BEER_device_index != 0xFF && disksizechanged) {
	  unsigned char status, ret;

	  /* FIXME: For me the hardware is not reprobed, the size
	   * of the disk is not updated... Is there a BIOS function
	   * available for that?
	   * Is the size of the disk is updated after a read failure?
	   */
	  DBG (("\r\nReset BIOS disks 0x%X to acknowledge changes: ",
			dp->BEER_device_index));
	  ret = _BIOSDISK_reinitHD (dp->BEER_device_index, &status);
	  ret = _BIOSDISK_resetHD (dp->BEER_device_index, &status);
//	  ret = _BIOSDISK_recalibrateHD (dp->BEER_device_index, &status);
	  DBG (("returns 0x%X, status 0x%X\r\n", ret, status));
	  disksizechanged = 0;
	  }
#endif
      }

#if 1
  if (disksizechanged) {
      unsigned char status __attribute__ ((unused));
      unsigned char ret __attribute__ ((unused));

      /* FIXME: This does not force a re-probe on my EBIOS, i.e. HPA is updated
	on the disk but not on the BIOS (max sector). Is there a better way? */
      DBG (("\r\nReset all BIOS disks for BIOS to acknowledge changes: "));
      ret = _BIOSDISK_reset_alldisks (&status);
//    ret = _BIOSDISK_reset_floppy (&status);
      DBG (("returns 0x%X, status 0x%X\r\n", ret, status));
      }
#endif

//    _BIOS_wait (3000000);
//    _BIOS_wait (3000000);
//    _BIOS_wait (3000000);
//#undef DBG
//#define DBG(X) /**/
  }

DISK_FCT_PREFIX(probeIDEdisk_setdiskname) static inline void
probeIDEdisk_setdiskname (struct diskparam_str *dp, const ata_identify_t *buff)
  {
  extern const char *const itoa_array;
  unsigned char *nameptr;
  unsigned char val;

#if DISK_SUPPORT & ATAPI_SUPPORT
  if (buff->general_config.ATAPI_device == 2) {
      nameptr = STRINIT (dp->diskname, "ATAPI/");
      if (buff->capabilities.lba_supported) {
	  nameptr = STRINIT (nameptr, "lba ");
	  dp->lba_slave_mask |= 0x40;
	  }
	else
	  nameptr = STRINIT (nameptr, "msf "); /* NEVER TESTED */
      dp->access = hardide_atapi;
      }
    else
#endif /* ATAPI_SUPPORT */
      {
      nameptr = STRINIT (dp->diskname, "IDE/");
      if (buff->capabilities.lba_supported) {
	  if (dp->ide_attribute.lba48 && buff->max_user_lba48 != 0) {
	      nameptr = STRINIT (nameptr, "lba48 ");
	      dp->access = hardide_lba48;
	      }
	    else {
	      nameptr = STRINIT (nameptr, "lba ");
	      dp->access = hardide_lba;
	      }
	  dp->lba_slave_mask |= 0x40;
	  }
	else {
	  nameptr = STRINIT (nameptr, "chs ");
	  dp->access = hardide_chs;
	  }
      }

  val = ((dp->lba_slave_mask >> 4) & 0x03) ^ 0x2;
  const char *master_slave = (val == 0)? "master" : "slave";
  while (*master_slave)
      *nameptr++ = *master_slave++;
  if (val > 1) /* unusual slave 2 and 3, untested */
      *nameptr++ = '0' + val;
  nameptr = STRINIT (nameptr, "@0x");
  val = 16;
  if (dp->ideIOadr >> 12 == 0)
      val -= 4;
  do {
      val -= 4;
      *nameptr++ = itoa_array[(dp->ideIOadr >> val) & 0x0F];
      } while (val != 0);

  nameptr = STRINIT (nameptr, ",0x");
  val = 16;
  if (dp->ideIOctrladr >> 12 == 0)
      val -= 4;
  do {
      val -= 4;
      *nameptr++ = itoa_array[(dp->ideIOctrladr >> val) & 0x0F];
      } while (val != 0);
  *nameptr++ = '\0';
  }

DISK_FCT_PREFIX(probeIDEdisk_smart) static inline void
probeIDEdisk_smart (struct diskparam_str *dp, unsigned smart_enabled)
  {
  int ret;

  if (!smart_enabled) {
      ret = 2;
      }
    else {
      DDBG ((" [collecting SMART info for %s: ", dp->diskname));
      ret = IDE_execute_SMART_logging (dp, SMART_logging_std_offline);
      DDBG (("logging_std_offline: %d, Get SMART status: ", ret));
      ret = IDE_get_SMART_status (dp);
      DDBG (("status: %d]\r\n\t", ret));
      if (ret != 1) /* ignore everything else */
	  ret = 0;
      }
  if (ret != 0) {
      print (SMART_REPORT);
      print ((char *)dp->diskname);
      if (ret == 1) {
	  puts (SMART_THRESHOLD_EXCEED);
	  dp->error_log.SMART_failure = 1;
	  }
	else {
	  puts (SMART_DISABLED);
	  dp->error_log.SMART_disabled = 1;
	  }
      }
  }

DISK_FCT_PREFIX(probeIDEdisk_get_size_geometry) static inline void
probeIDEdisk_get_size_geometry (struct diskparam_str *dp, const ata_identify_t *buff)
  {
  if (   buff->capabilities.lba_supported /* extra check */
      && dp->ide_attribute.lba48
      && buff->command_enabled2.lba48
      && buff->lba_capacity == 0x0FFFFFFF) {
      dp->nbtotalsector = buff->max_user_lba48;
      DDBG ((" [Setting disk size to LBA48 %llU sectors] ", dp->nbtotalsector));
      dp->nbhead = 0;
      dp->nbcylinder = 0;
      dp->nbsectorpertrack = 0;
      }
    else if (   buff->valid_description.translation_fields_valid
	     && buff->current_capacity_in_sector != 0) {
      dp->nbcylinder = buff->number_of_current_cylinder;
      dp->nbhead = buff->number_of_current_head;
      dp->nbsectorpertrack = buff->number_of_current_sector_per_track;
      DDBG ((" [valid translation, using %u/%u/%u] ",
		dp->nbcylinder, dp->nbhead, dp->nbsectorpertrack));
      if (   buff->number_of_current_head != 0
	  && buff->number_of_current_sector_per_track != 0
	  && buff->number_of_current_cylinder == 16383
	  && buff->number_of_current_head == 16
	  && buff->number_of_current_sector_per_track == 63
	  && buff->lba_capacity >= buff->current_capacity_in_sector) {
	  dp->nbtotalsector = buff->lba_capacity;
	  dp->nbcylinder = buff->lba_capacity / dp->nbhead / dp->nbsectorpertrack;
	  DDBG ((" [adjusting BIGDISK size %u sector, %u cylinders]",
		buff->lba_capacity, dp->nbcylinder));
	  }
#if 0 /* happens on 340 Mb Quantum Prodrive LPS, but also if HPA is set... */
	else if (buff->lba_capacity == (unsigned)
			  buff->number_of_current_cylinder
			* buff->number_of_current_head
			* buff->number_of_current_sector_per_track - 1) {
	  dp->nbtotalsector = buff->lba_capacity + 1;
	  DDBG ((" [disk reports MAXimum lba instead of total number, corrected to %u]",
			buff->lba_capacity + 1));
	  }
#endif
	else if (   buff->lba_capacity >= buff->current_capacity_in_sector
		 && buff->capabilities.lba_supported) {
	  dp->nbtotalsector = buff->lba_capacity;
	  DDBG ((" [capacity %U sectors in LBA, %U more than in CHS] ",
		buff->lba_capacity, buff->lba_capacity - buff->current_capacity_in_sector));
	  }
	else {
	  dp->nbtotalsector = buff->current_capacity_in_sector;
	  DDBG ((" [capacity %U sectors in CHS] ", buff->current_capacity_in_sector));
	  }
      }
    else {
      unsigned nbtotalsector;

      dp->nbcylinder = buff->number_of_cylinder;
      dp->nbhead = buff->number_of_head;
      dp->nbsectorpertrack = buff->number_of_sector_per_track;
      DDBG ((" [INvalid translation, using %u/%u/%u] ", dp->nbcylinder, dp->nbhead, dp->nbsectorpertrack));
      nbtotalsector = dp->nbcylinder * dp->nbhead * dp->nbsectorpertrack;
      if (nbtotalsector == 0) {
	  DDBG ((" [One of C/H/S null, use current_capacity_in_sector %U] ", buff->current_capacity_in_sector));
	  nbtotalsector = buff->current_capacity_in_sector;
	  }
	else if ((nbtotalsector + 5) / 10 == (buff->current_capacity_in_sector + 5) / 10) {
	  DDBG ((" [valid current_capacity_in_sector %U] ", buff->current_capacity_in_sector));
	  nbtotalsector = buff->current_capacity_in_sector;
	  }
	else {
	  DDBG ((" [INvalid current_capacity_in_sector = %u, using C*H*S = %u] ",
			buff->current_capacity_in_sector, nbtotalsector));
	  }
      dp->nbtotalsector = nbtotalsector;
      }
  }

DISK_FCT_PREFIX(probeIDEdisk_analysebootsector) static inline void
probeIDEdisk_analysebootsector (struct diskparam_str *dp, unsigned diskno, unsigned ask_password)
  {
  unsigned char cpt = 2;

#if DISK_SUPPORT & ATAPI_SUPPORT
  if (dp->access == hardide_atapi) {
      unsigned ret = ideAtapiStartStopUnit (dp, CD_ACTIVE, 0, 0, 1);
      ret = ret; /* ret used even if no debug */
      DDBG (("ideAtapiStartStopUnit(dp, CD_ACTIVE, 0, 0, 1) returns %d\r\n", ret));
      DDBG ((" [wait 100 ms] "));
      _BIOS_wait (100000);
//      unsigned ret = ideAtapiStartStopUnit (dp, CD_START_VALID, 1, 0, 1);
//      ret = ret; /* ret used even if no debug */
//      DDBG (("ideAtapiStartStopUnit(dp, CD_START_VALID, 1, 0, 1) returns %d\r\n", ret));
      }
#endif

  while (analysebootsector (dp) == 1) {
      int ret __attribute__ ((unused));

      if (ask_password) {
	  if (!BOOT1_DOS_running()) {
#if USER_SUPPORT != 0
	      char password[33];	/* get_line set a '\0', ideSecurity just needs an array */
	      unsigned ret;
	      int nbtry = 0;
	      extern void *default_ide_password_codeptr;

	      if (codeseg_peekl (default_ide_password_codeptr) == 0x0010BEAF) {
		  nbtry = -1; /* We have a default password set by instboot,
			so we just need to show the psychic paper to the IDE disk: */
		  lmemcpy (password, code_adr(default_ide_password_codeptr) + 4, 32);
		  }

	      do {
		  enum IDE_security_enum cmd = IDE_security_unlock_user;

		  if (nbtry == 0 && UTIL.keyboard != serialkbd && !kbdmap_is_precise(copy_gujin_param.kbdmap)) {
		      print (RECOGNIZE_KEYBOARD);
		      if (kbd_detect() != 0) {
			  printf (FAIL_DISK_IGNORED, 0);
			  dp->error_log.disk_locked = 1;
			  return;
			  }
		      puts ("");
		      kbd_setup (copy_gujin_param.kbdmap);
		      }

		  printf (DISK_US_LOCKED_PASSWORD, diskno, dp->diskname);
		  if (nbtry != -1) {
		      unsigned tmp = 0;
		      for (;;) {
			  print (TYPE_PASSWORD_OR_ESC);
			  password[0] = '\0';
			  ret = get_line (password, sizeof(password), '*', ' ', '~');
			  if (ret == 0 && STREQL(password, "MASTER PASSWORD") && tmp++ == 0) {
			      cmd = IDE_security_unlock_master;
			      printf (WARNING_MASTER_PASSWORD_X, dp->ide_master_password_revision);
			      continue;
			      }
			  tmp = 0;
			  while (password[tmp] != 0)
			      tmp++;
			  while (tmp < sizeof(password))
			      password[tmp++] = '\0';
			  break;
			  }
		      }
		    else {
		      print (DISK_TRY_DEFAULT_PASSWORD);
		      ret = 0;
		      }
		  if (ret == 0) {
		      ret = ideSecurity (dp, password, cmd);
		      if (ret == (unsigned)-IDE_data_cmderror)
			  puts (ERROR_PASSWORD_RETRY);
		      }
		  } while (nbtry++ < 15 && ret == (unsigned)-IDE_data_cmderror);
	      if (ret == 0) {
		  puts (OK_DISK_UNLOCKED);
		  dp->error_log.disk_was_pw_locked = 1;
		  continue;
		  }
		else {
		  printf (FAIL_DISK_IGNORED, ret);
		  dp->error_log.disk_locked = 1;
		  break;
		  }
#endif
	      }
	  printf (DISK_S_LOCKED_PASSWORD_IGNORED, dp->diskname);
	  dp->error_log.disk_locked = 1;
	  break;
	  }

#if DISK_SUPPORT & ATAPI_SUPPORT
      if (dp->access == hardide_atapi) {
	  if (--cpt > 0)
	      continue;	/* retry once the complete analysis */
	  break;
	  }
#endif /* ATAPI_SUPPORT */

      DDBG (("\r\n\tFAILED, trying calibrate drive: "));
      ret = IDE_calibratedrive (dp);
      DDBG (("IDE_calibratedrive returns %d, ", ret));

      if (--cpt == 0) {
	  DDBG (("giving up.\r\n"));
	  break;
	  }

      if (   (dp->lba_slave_mask & 0x40) == 0 /* access in C/H/S mode */
	  && dp->nbcylinder != 0
	  && dp->nbhead != 0 && dp->nbhead <= 16
	  && dp->nbsectorpertrack != 0 && dp->nbsectorpertrack <= 63) {
	  DDBG (("trying IDE_Initialisedrive with %u/%u/%u: ",
		dp->nbcylinder, dp->nbhead, dp->nbsectorpertrack));
	  ret = IDE_Initialisedrive (dp);
	  DDBG (("IDE_Initialisedrive returns %d, ", ret));
	  }
      DDBG ((" retrying analysebootsector() "));
      }
  }

DISK_FCT_PREFIX(probeIDEdisk) static unsigned
probeIDEdisk (struct diskparam_str *dp, unsigned nbdisk)
  {bound_stack();{
  int tmp, maxloop = 2;
  unsigned atapi_signature = 0;
//  unsigned ide_max_native_hpa = (unsigned)-1;

#if 0
  ata_identify_t buffer, *buff = &buffer;
  farptr buffer_farptr = stack_adr (buff);
#else // saves 256 bytes for diskcode
  ata_identify_t *buff = (ata_identify_t *)fourKbuffer;
  farptr buffer_farptr = data_adr (buff);

  if (sizeof (fourKbuffer) < sizeof (ata_identify_t))
      __ERROR();	/* the linker will produce an error here */
#endif

  if (sizeof (ata_identify_t) != 512)
      __ERROR();	/* the linker will produce an error here */
  if (sizeof (atapi_identify_t) != 512)
      __ERROR();	/* the linker will produce an error here */
  checkptr (dp, "probe IDE disk invalid parameter.");

  DDBG (("%s: calibrating ", __FUNCTION__));
//  DDBG (("(initial altstatus: 0x%X) ", _IDE_get_altstatus (dp))); // at zero on some add-on cards
  calibrateIDEloop (dp); /* before any other disk access! */
  /* dp->diskname is still not set here! */

  for (;;) {
      tmp = ideIdentify (dp, buffer_farptr, &atapi_signature);
      if (tmp != 0 || maxloop-- == 0)
	  break;
      DDBG (("\r\nideIdentify reports: RWmult=%u, C/H/S=%u/%u/%u, trans CHS %svalid: C/H/S=%u/%u/%u,\r\n\t",
		buff->RWmultiple_max_sector,
		buff->number_of_cylinder,
		buff->number_of_head,
		buff->number_of_sector_per_track,
		buff->valid_description.translation_fields_valid? "" : "in",
		buff->number_of_current_cylinder,
		buff->number_of_current_head,
		buff->number_of_current_sector_per_track
		));
      DDBG (("total= %U i.e. 0x%X, BigDisk total: %U i.e. 0x%X, BigBigLBA48total: %llU, %s supported, DMA%s supported;\r\n\t",
		buff->current_capacity_in_sector,
		buff->current_capacity_in_sector,
		buff->lba_capacity,
		buff->lba_capacity,
		buff->max_user_lba48,
		buff->capabilities.lba_supported? "LBA" : "CHS only",
		buff->capabilities.dma_supported? "" : " not"
		));
      DDBG (("Security: master_password_revision 0x%X, security_erase_unit_time %u min, enhanced_security_erase_time %u min\r\n\t",
		buff->master_password_revision, 2* buff->security_erase_unit_time, 2* buff->enhanced_security_erase_time));
      DDBG (("Security: supported %u, enabled %u, locked %u, frozen %u, count_expired %u, enhanced_erase %u, high/max level %u\r\n\t",
		buff->security_status.supported, buff->security_status.enabled, buff->security_status.locked,
		buff->security_status.frozen, buff->security_status.count_expired, buff->security_status.enhanced_erase,
		buff->security_status.high_level));

      if (   memeql (&buff->command_supported1, "\0\0\0\0\0\0", 6)
	 || memeql (&buff->command_supported1, "\xFF\xFF\xFF\xFF\xFF\xFF", 6)) {
	  DDBG (("ATA1, ATA2 or simple ATA3 device, cannot really probe LBA clipping.\r\n\t"));
	  break;
	  }
      /* Remove possible software or hardware clipping, did not acheive to remove 32Gb hardware clipping from Hitachi drives */
      if (buff->capabilities.lba_supported && buff->command_supported1.host_protected_area && buff->command_enabled1.host_protected_area) {
	  unsigned long long max48;
	  if (buff->command_supported2.lba48 && buff->command_enabled2.lba48 && IDE_get_native48_max_address (dp, &max48) == 0) {
	      if (max48 > buff->max_user_lba48) {
		  int ret = IDE_set_native48_max_address (dp, max48, 0);
		  DDBG (("Removed lba48 hardware clipping at %llu to %llu return %u\r\n", buff->max_user_lba48, max48, ret));
		  /* Save the initial HPA, that variable is not overwritten if no BEER is detected */
		  /* FIXME: if reprobe disk is done by ^R, that value will be the total disk size... */
		  if (dp->BEER_HostProtectedArea_start == 0) /* first pass */
		      dp->BEER_HostProtectedArea_start = buff->max_user_lba48;
		  if (ret == 0)
		      continue;
		  }
	      }
	    else {
	      unsigned max28;
	      if (IDE_get_native_max_address (dp, &max28) == 0 && max28 > (buff->max_user_lba48 ?: buff->lba_capacity)) {
		  int ret = IDE_set_native_max_address (dp, max28, 0);
		  DDBG (("Removed lba28 hardware clipping at %llu to %u return %u\r\n", buff->max_user_lba48, max28, ret));
		  /* Save the initial HPA, that variable is not overwritten if no BEER is detected */
		  /* FIXME: if reprobe disk is done by ^R, that value will be the total disk size... */
		  if (dp->BEER_HostProtectedArea_start == 0) /* first pass */
		      dp->BEER_HostProtectedArea_start = buff->max_user_lba48;
		  if (ret == 0)
		      continue;
		  }
	      }
	  }
      break;
      }

  if (tmp == 0) {
      if (   !buff->capabilities.lba_supported
	  && buff->current_capacity_in_sector != buff->lba_capacity
	  && (buff->current_capacity_in_sector & 0x0FE00000) != 0
	  && (buff->current_capacity_in_sector & 0x00000FE0) == 0) {
	  unsigned swapped =  (buff->current_capacity_in_sector << 16)
			| (buff->current_capacity_in_sector >> 16);
	  DDBG (("no lba on a > 1 Gb disk, swap size: 0x%X -> 0x%X, i.e. capacity %u sectors\r\n\t",
		buff->current_capacity_in_sector, swapped, swapped));
	  buff->current_capacity_in_sector = swapped;
	  }
      if (!buff->sector_size.shall_be_0 && buff->sector_size.shall_be_1 && !buff->sector_size.multiple_logical_per_physical
	&& buff->sector_size.logical_sector_longer_than_256) {
	  DDBG (("word 106 value: 0x%X, word_per_logical_sector %u\r\n\t", buff->sector_size, buff->word_per_logical_sector));
	  dp->bytepersector = 2 * buff->word_per_logical_sector;
	  }
	else
	  dp->bytepersector = 512;
      if (!buff->sector_size.shall_be_0 && buff->sector_size.shall_be_1 && buff->sector_size.multiple_logical_per_physical)
	  DDBG (("word 106 value: 0x%X, %u logical sectors per physical\r\n\t", buff->sector_size, 1 << buff->sector_size.logical_per_physical));
      dp->multiplemode = buff->RWmultiple_max_sector;
      if (!copy_gujin_param.attrib.probe_ide_disk) {
	  DDBG (("ideIdentify success, but just probe ATAPI: no gujin_param.probe_ide_disk.\r\n"));
	  return 2;
	  }
      tmp = IDE_get_power_mode (dp);
      /* if (tmp == 0)
	  dp->bytepersector = 0; / * HD powered up in standby mode */
      DDBG (("\r\n\t"));
      if (buff->general_config.ATAPI_device == 2) {
	  DDBG (("WARNING: clearing ideIdentify::general_config.ATAPI_device to 0 for IDE device!\r\n\t"));
	  buff->general_config.ATAPI_device = 0;
	  }
      }
    else if (((atapi_signature >> 8) & 0xFFFF) != 0xEB14) {
      DDBG (("ideIdentify failed (%d) and signature = 0x%X, nothing.\r\n", tmp, atapi_signature));
      return 2;
      }
    else if (!copy_gujin_param.attrib.probe_cdrom) {
      DDBG (("ideIdentify failed (%d) and ATAPI signature, but no gujin_param.probe_cdrom.\r\n", tmp));
      return 2;
      }
    else {
#if DISK_SUPPORT & ATAPI_SUPPORT
      atapi_identify_t *buffpi = (atapi_identify_t *)buff;

      DDBG (("ideIdentify failed (%d) and signature = 0x%X, ATAPI.\r\n", tmp, atapi_signature));
      tmp = ideAtapiIdentify (dp, buffer_farptr, &atapi_signature);
      DDBG (("ideAtapiIdentify returns %d, ATAPI_device shall be 2: %u, "
		"response %scomplete, packet %s bytes, DRQ_INTRQ_timings %u, command set %u (5:cdrom)\r\n",
		tmp,
		buffpi->general_config.ATAPI_device,
		buffpi->general_config.incomplete_response? "in" : "",
		((const char *[4]){ "12" ,"16", "??", "???"})[buffpi->general_config.cmd_packet_len],
		buffpi->general_config.DRQ_INTRQ_timings,
		buffpi->general_config.command_packet_set
		));
      if (   buffpi->general_config.ATAPI_device == 2
	  && buffpi->general_config.cmd_packet_len == 0
	  && buffpi->general_config.command_packet_set == 5) {
	  struct cd_capacity_str cap;
	  unsigned cpt = 11;	/* Shall wait more than 6 seconds for DVD-RAM ? Or reset ATAPI so no permanent wait */
	  int ret, medium_present = 1;

#if 0
	  DDBG (("ATAPI detected, start with ATAPI device reset: "));
	  _IDE_set_drivehead (dp, dp->lba_slave_mask);
	  _IDE_get_altstatus_bits(dp); /* or use "short_delay();" ? */
	  _IDE_start_command (dp, IDE_DEVICE_RESET);
	  _IDE_get_altstatus_bits(dp); /* or use "short_delay();" ? */
	  DDBG ((" status 0x%X readback= 0x%X", _IDE_get_altstatus (dp), _IDE_readback_lba28 (dp)));
	  _BIOS_wait (100000); DDBG ((", wait 100 ms"));
	  DDBG ((", status 0x%X readback= 0x%X\r\n", _IDE_get_altstatus (dp), _IDE_readback_lba28 (dp)));
#endif

	  while ((ret = ideAtapiTestUnitReady(dp)) != 0 && --cpt != 0) {
	      struct cd_request_sense_str sense;

	      DDBG (("ideAtapiTestUnitReady return %d, ", ret));
	      if ((ret = ideAtapiRequestSense (dp, &sense)) != 0) {
		  DDBG (("ideAtapiRequestSense return %d, assume no medium\r\n", ret));
		  medium_present = 0;
		  break;
		  }
	      if (sense.sense_key == 2 && sense.ASC == 0x3A && sense.ASCQ <= 2) {
		  medium_present = 0; /* no media in drive */
		  break;
		  }
	      if (sense.sense_key == 6 && sense.ASC == 0x28 && sense.ASCQ == 0) {
		  DDBG (("ask again without delay: "));
		  dp->multiplemode |= 1; /* just store that the drive can do medium changed */
		  continue;
		  }
	      if (sense.sense_key == 6 && sense.ASC == 0x29 && sense.ASCQ == 0) {
		  DDBG (("ask again without delay: "));
		  continue;
		  }
	      if (   !(sense.sense_key == 2 && sense.ASC == 0x04 && sense.ASCQ == 1)
		  // && !(sense.sense_key == 0 && sense.ASC == 0x00 && sense.ASCQ == 0)
		  && !(sense.sense_key == 6 && sense.ASC == 0x28 && sense.ASCQ == 0)) {
		  break;
		  }
	      DDBG ((" [wait 600 ms] "));
	      _BIOS_wait (600000);	/* 600 ms, max wait for CDROM: 6 seconds */
	      }

#if 0
	  {
	  struct cd_inquiry_str inquiry;
	  if (ideAtapiInquiry (dp, &inquiry) == 0) {
	      unsigned char *ptr, *last = nameptr;

	      *nameptr++ = ' ';
	      *nameptr++ = '(';
	      for (ptr = inquiry.vendor_id; ptr < &inquiry.vendor_id[nbof(inquiry.vendor_id)]; ptr++, nameptr++) {
		  *nameptr = *ptr;
		  if (*ptr > ' ' && *ptr <= '~')
		      last = nameptr;
		  }
	      if (*last != ' ') {   /* last has been set after initialisation */
		  nameptr = last + 1;
		  *nameptr++ = ')';
		  }
		else
		  nameptr = last;
	      /* dp->diskname overflow when also trying to display inquiry.product_id */
	      }
	  }
#endif

	  if (medium_present && ideAtapiReadCdRecordedCapacity (dp, &cap) == 0 && cap.lba != 0) {
	      dp->nbtotalsector = cap.lba;
	      dp->bytepersector = cap.blocksize;
//	      printf ("ideAtapiMaxSpeed returns %d; ", ideAtapiMaxSpeed (dp)); // do not work for DVD
	      /* FIXME: Here, some CD drive report ideAtapiReadCdRecordedCapacity() 2352 instead of
	       * 2048 (medium sector size), we could maybe adjust it at the first MBR read
	       * in probeIDEdisk_get_size_geometry() to 2048?
	       */
	      }
	    else {
	      if (medium_present) {
		  struct cd_request_sense_str sense;
		  ideAtapiRequestSense (dp, &sense);
		  }
	      dp->nbtotalsector = 0;
	      dp->bytepersector = 0; /* no disk / not accessible */
	      }
	  }
	else
#endif /* ATAPI_SUPPORT */
	  return 1;
      }

  dp->ide_attribute = probeIDEdisk_attribute (dp, buff, buffer_farptr, 1);
  /* diskname depends on disk attributes: */
  probeIDEdisk_setdiskname (dp, buff);
  /* diskname has to be set for probeIDEdisk_smart: */
  if (dp->ide_attribute.smart)
      probeIDEdisk_smart (dp, buff->command_enabled1.smart);
  /* size and geometry depend on attributes: */
  if (dp->access != hardide_atapi)
      probeIDEdisk_get_size_geometry (dp, buff);

//  dp->nbpartition = dp->partition = 0;	// done by memset()
  if (dp->bytepersector == 0)
      DDBG (("\r\n\tDo not probe partitions, no media present."));
    else if (dp->nbtotalsector == 0)
      DDBG (("\r\n\tDo not probe partitions, HD power up in standby?"));
    else {
#ifdef IDE_ATAPI_POWER_MANAGEMENT
      unsigned char tmp = SET_DRIVE_ACTIVE(dp);
      DDBG (("SET_DRIVE_ACTIVE(dp) returns %u", tmp));
#endif

      probeIDEdisk_analysebootsector (dp, nbdisk, dp->ide_attribute.security && buff->command_enabled1.security);
      DDBG ((", %u partitions.", dp->nbpartition));

#ifdef IDE_ATAPI_POWER_MANAGEMENT
      tmp = SET_DRIVE_IDLE(dp);
      DDBG (("SET_DRIVE_IDLE(dp) returns %u", tmp));
#endif
      }

  return 0;
  }}

DISK_FCT_PREFIX(probe_ide) static inline unsigned
probe_ide (unsigned nbdisk, struct diskparam_str *param)
  {
  unsigned short cpt;
  union PCI_hardware_characteristics_u pciaccess;
  unsigned short version_bcd;
  struct PCIv3_support PCI_support;
#ifndef DISK_USE_MALLOC
  static struct IDE_found_str _disk_idefound[NB_IDE];
#endif

  DDBG (("\r\n\n%s: Know from EBIOS the IDE interfaces:", __FUNCTION__));
  for (cpt = 0; cpt < DI.nb_IDE_found; cpt++)
      DDBG ((" 0x%X/0x%X/%u ", DI.IDE_found[cpt].ideIOadr, DI.IDE_found[cpt].ideIOctrladr, DI.IDE_found[cpt].irq));

  PRINTF ("PCI BIOS...\r");
  if (_PCIBIOS_install_check (&pciaccess, &version_bcd, &PCI_support) == 0 && version_bcd != 0) {
      DDBG (("\r\nBIOS PCI %X.%X with %u buses and pciaccess 0x%X/PCI_support 0x%X detected, no extended ISA probing.\r\n",
		version_bcd >> 8, version_bcd & 0xFF, PCI_support.last_pci_bus + 1, pciaccess.all, *(unsigned short *)&PCI_support));
      DDBG (("Probing other PCI IDE interfaces:\r\n"));
      /* see http://www.t13.org/Documents/UploadedDocuments/technical/e00149r0.pdf : */
      static const unsigned char pci_subclass_array[] = { 0x01, 0x04, 0x06, 128 };	/* 128: some add-on PCI/IDE cards */
      unsigned pci_subclass_cpt;

      for (pci_subclass_cpt = 0; pci_subclass_cpt < sizeof(pci_subclass_array)/sizeof(pci_subclass_array[0]); pci_subclass_cpt++) {
	  unsigned char programming_interface = 0;
	  while (programming_interface <= 0x8F) {
	      struct PCI_classcode classcode_ide = {.pci_class = 1, .pci_subclass = pci_subclass_array[pci_subclass_cpt] , .programming_interface = programming_interface};
	      unsigned short device_index = (unsigned short)-1;
	      struct PCI_location location = {};
	      static const struct { unsigned short vendid, devid; unsigned char revision; } blacklist[] = {
		  { 0x1103, 0x0004, 1 }
		  };
	      /* Let's excite the 'marsmat' with a bit of electricity... */
	      while (PCI_support.function_3_implemented && _PCIBIOS_find_classcode (classcode_ide, ++device_index, &location) == 0) {
		  unsigned char header_type, base_class, sub_class, irq, revision;
		  unsigned short vendid, devid, command;
		  unsigned cmdblock[2], ctrlblock[2], blockcpt, idecpt, blcpt;
#if DEBUG & DEBUG_DISK
		  unsigned char prog_interface;
		  unsigned busmaster, CardBusCis;

		  _PCIBIOS_read_config_byte(location, 0x09, &prog_interface);
		  _PCIBIOS_read_config_dword(location, 0x20, &busmaster);
		  _PCIBIOS_read_config_dword(location, 0x28, &CardBusCis);
#endif

		  if (_PCIBIOS_read_config_byte(location, 0x0E, &header_type) != 0 || (header_type & 0x7F) != 0) {
		      DDBG (("header_type read failure or unmanaged header: 0x%X\r\n", header_type));
		      continue;
		      }

		  DDBG (("IDE found at bus_number %u, device 0x%X, function %u, ", location.bus_number, location.device, location.function));
		  if (   _PCIBIOS_read_config_word(location, 0x00, &vendid) != 0
		      || _PCIBIOS_read_config_word(location, 0x02, &devid) != 0
		      || _PCIBIOS_read_config_byte(location, 0x08, &revision) != 0
		      || _PCIBIOS_read_config_byte(location, 0x0B, &base_class) != 0
		      || _PCIBIOS_read_config_byte(location, 0x0A, &sub_class) != 0
		      || _PCIBIOS_read_config_byte(location, 0x3C, &irq) != 0
		      || _PCIBIOS_read_config_word(location, 0x04, &command) != 0
		      || _PCIBIOS_read_config_dword(location, 0x10, &cmdblock[0]) != 0
		      || _PCIBIOS_read_config_dword(location, 0x14, &ctrlblock[0]) != 0
		      || _PCIBIOS_read_config_dword(location, 0x18, &cmdblock[1]) != 0
		      || _PCIBIOS_read_config_dword(location, 0x1C, &ctrlblock[1]) != 0) {
		      DDBG (("PCIBIOS do not work\r\n"));
		      continue;
		      }
		  DDBG (("header_type 0x%X, VendorID 0x%X DeviceID 0x%X Revision 0x%X base_class %u, sub_class %u, prog_interface 0x%X, CardBusCis 0x%X command 0x%X\r\n",
			header_type, vendid, devid, revision, base_class, sub_class, prog_interface, CardBusCis, command));
		  if (base_class != classcode_ide.pci_class || sub_class != classcode_ide.pci_subclass)
		      continue;
		  if (sub_class == 1 && (classcode_ide.programming_interface & 1) == 0) {
		      DDBG (("Force cmdblock/ctrlblock[0] to read 0x1F1/0x3F5 instead of 0x%X/0x%X\r\n", cmdblock[0], ctrlblock[0]));
		      cmdblock[0] = 0x1F1;
		      ctrlblock[0] = 0x3F5;
		      irq = 14;
		      }
		  if (sub_class == 1 && (classcode_ide.programming_interface & 4) == 0) {
		      DDBG (("Force cmdblock/ctrlblock[1] to read 0x171/0x375 instead of 0x%X/0x%X\r\n", cmdblock[1], ctrlblock[1]));
		      cmdblock[1] = 0x171;
		      ctrlblock[1] = 0x375;
		      irq = 15;
		      }
		  for (blcpt = 0; blcpt < sizeof(blacklist)/sizeof(blacklist[0]); blcpt++)
		      if (devid == blacklist[blcpt].devid && vendid == blacklist[blcpt].vendid && revision == blacklist[blcpt].revision)
			  break;
		  if (blcpt < sizeof(blacklist)/sizeof(blacklist[0]))
		      DDBG (("IDE pair at 0x%X/0x%X and at 0x%X/0x%X, busmaster 0x%X, irq %u is blacklisted.\r\n", cmdblock[0], ctrlblock[0], cmdblock[1], ctrlblock[1], busmaster, irq));
		    else
		      DDBG (("IDE pair at 0x%X/0x%X and at 0x%X/0x%X, busmaster 0x%X, irq %u is probably fine.\r\n", cmdblock[0], ctrlblock[0], cmdblock[1], ctrlblock[1], busmaster, irq));

		  for (blockcpt = 0; blockcpt < 2 && DI.nb_IDE_found < NB_IDE; blockcpt++) {
		      //if (cmdblock[blockcpt] == 0)
		      if ((cmdblock[blockcpt] & 0x03) != 0x01 || (ctrlblock[blockcpt] & 0x03) != 0x01) {
			  /* Also handle cmdblock[blockcpt]=0xFFFFFFFF or ctrlblock[blockcpt]=0xFFFFFFFF when IDE but no IDE plug on mainboard */
			  DDBG (("PCI IDE addresses 0x%X, 0x%X not in I/O space\r\n", cmdblock[blockcpt], ctrlblock[blockcpt]));
			  continue;
			  }
		      cmdblock[blockcpt] &= ~0x03;
		      ctrlblock[blockcpt] &= ~0x03;
		      if (cmdblock[blockcpt] < 0x100) {
			  DDBG (("PCI IDE addresses 0x%X too low\r\n", cmdblock[blockcpt]));
			  continue;
			  }
		      for (idecpt = 0; idecpt < DI.nb_IDE_found; idecpt++)
			  if (DI.IDE_found[idecpt].ideIOadr == cmdblock[blockcpt])
			      break;
		      if (idecpt < DI.nb_IDE_found) {
			  if (blcpt < sizeof(blacklist)/sizeof(blacklist[0])) {
			      DDBG (("Blacklisted IDE device, remove it from known IDE addresses.\r\n"));
			      DI.nb_IDE_found--;
			      while (idecpt < DI.nb_IDE_found) {
				  DI.IDE_found[idecpt] = DI.IDE_found[idecpt+1];
				  idecpt++;
				  }
			      continue;
			      }
			  if (DI.IDE_found[idecpt].ideIOctrladr == 0) {
			      DDBG (("Completing IDE definition, set ideIOctrladr to 0x%X\r\n", ctrlblock[blockcpt]));
			      DI.IDE_found[idecpt].ideIOctrladr = ctrlblock[blockcpt];
			      }
			    else if (DI.IDE_found[idecpt].ideIOctrladr != ctrlblock[blockcpt])
			      DDBG (("Conflicting IDE definition, ideIOctrladr 0x%X and ctrlblock 0x%X\r\n", DI.IDE_found[idecpt].ideIOctrladr, ctrlblock[blockcpt]));
				/* Note that:
				Conflicting IDE definition, ideIOctrladr 0xDC02 and ctrlblock 0xDC00
				IDE at 0xE000 already known
				*/
			  if (DI.IDE_found[idecpt].irq == 0 && irq <= 15) {
			      DDBG (("Completing IDE definition, set irq to %u\r\n", irq));
			      DI.IDE_found[idecpt].irq = irq;
			      }
			  DDBG (("IDE at 0x%X already known\r\n", cmdblock[blockcpt]));
			  continue;
			  }
			else if (blcpt < sizeof(blacklist)/sizeof(blacklist[0]))
			  continue;
		      /* Then, the Microvac said: */
		      {
		      if ((command & 1) == 0) /* I/O space control */
			  _PCIBIOS_write_config_word(location, 0x04, command | 1);
		      unsigned char status = inb (cmdblock[blockcpt] + 7),
				altstatus = inb (ctrlblock[blockcpt] + 2);
		      if (status == altstatus) {
			  DDBG (("IDE status and altstatus+2 equal to: 0x%X, adjusting\r\n", altstatus));
			  ctrlblock[blockcpt] += 2;
			  }
			else {
			  DDBG (("IDE status 0x%X different of altstatus+2 0x%X\r\n", status, altstatus));
			  /* Because the granularity of the PCI address is 4 bytes, search the 3 other bytes: */
			  if (status == (altstatus = inb (ctrlblock[blockcpt]))) {
			      DDBG (("Keeping ctrlblock at 0x%X\r\n", ctrlblock[blockcpt]));
			      }
			    else if (status == (altstatus = inb (ctrlblock[blockcpt] + 1))) {
			      DDBG (("Adjusting altstatus IOaddr +1 to read: 0x%X\r\n", altstatus));
			      ctrlblock[blockcpt] += 1;
			      }
			    else if (status == (altstatus = inb (ctrlblock[blockcpt] + 3))) {
			      DDBG (("Adjusting altstatus IOaddr +3 to read: 0x%X\r\n", altstatus));
			      ctrlblock[blockcpt] += 3;
			      }
			    else {
			      if ((command & 1) == 0) /* I/O space control */
				  _PCIBIOS_write_config_word(location, 0x04, command);
			      DDBG (("forget IDE, did not find a status/altstatus pair (+0: 0x%X, +2: 0x%X, +1: 0x%X, +3: 0x%X)\r\n",
				inb (ctrlblock[blockcpt]), inb (ctrlblock[blockcpt] + 2), inb (ctrlblock[blockcpt] + 1), inb (ctrlblock[blockcpt] + 3)));
			      continue;
			      }
			  }
		      }
#ifdef DISK_USE_MALLOC
		      struct IDE_found_str *tmp = REALLOC (DI.IDE_found, (DI.nb_IDE_found + 1) * sizeof (struct IDE_found_str), "IDEfnd");
		      if (tmp == 0) {
			  DDBG (("Cannot add 0x%X, realloc failed\r\n", cmdblock[blockcpt]));
			  continue;
			  }
		      DI.IDE_found = tmp;
#else
		      if (DI.IDE_found == 0)
			  DI.IDE_found = _disk_idefound;
#endif
		      DI.IDE_found[DI.nb_IDE_found].ideIOadr = cmdblock[blockcpt];
		      DI.IDE_found[DI.nb_IDE_found].ideIOctrladr = ctrlblock[blockcpt];
		      DI.IDE_found[DI.nb_IDE_found].irq = (irq <= 15)? irq : 0;
		      DI.nb_IDE_found++;
		      }
		  // device_index++; already incremented
		  }
	      if (programming_interface == 0)
		  programming_interface = 0x80;
		else if (programming_interface == 0x82)
		  programming_interface = 0x8E; /* speeds up VirtualBox */
		else
		  programming_interface++;
	      }
	  }
      DDBG (("Probing other PCI IDE finished\r\n"));
      }
    else {
      /*
	* When there isn't any EBIOS but some added ISA card
	* not supported by BIOS, or no hard disk on these IDE.
	* Its order is just the probing order.
	*/
      static const unsigned short ISA_IDE_array[] = {
		0x1F0, 0x170, 0x1E8, 0x168, 0x1E0, 0x160
		// cannot probe those because of _IDE_ISA_guess_altadr():
		//      , 0xe820, 0xe828, 0xB400, 0xBC00, 0xD400, 0xD800
	  };
      unsigned short index;
      DDBG (("Probing other ISA IDE interfaces:\r\n"));
      for (index = 0; index < nbof(ISA_IDE_array); index++) {
	  DDBG (("0x%X: ", ISA_IDE_array[index]));
	  for (cpt = 0; cpt < DI.nb_IDE_found; cpt++)
	      if (ISA_IDE_array[index] == DI.IDE_found[cpt].ideIOadr)
		  break;
	  if (cpt < DI.nb_IDE_found) {
	      DDBG (("known by EBIOS\r\n"));
	      continue;
	      }
	  if (ide_abscent (ISA_IDE_array[index])) {
	      DDBG (("IDE abscent\r\n"));
	      continue;
	      }
	  if (DI.nb_IDE_found >= NB_IDE) {
	      DDBG (("Cannot add 0x%X, array full\r\n", ISA_IDE_array[index]));
	      continue;
	      }
#ifdef DISK_USE_MALLOC
	  {
	  struct IDE_found_str *tmp = REALLOC (DI.IDE_found, (DI.nb_IDE_found + 1) * sizeof (struct IDE_found_str),"IDEfnd2");
	  if (tmp == 0) {
	      DDBG (("Cannot add 0x%X, realloc failed\r\n", ISA_IDE_array[index]));
	      continue;
	      }
	  DI.IDE_found = tmp;
	  }
#else
	  if (DI.IDE_found == 0)
	      DI.IDE_found = _disk_idefound;
#endif
	  DI.IDE_found[DI.nb_IDE_found].ideIOadr = ISA_IDE_array[index];
	  DI.IDE_found[DI.nb_IDE_found].ideIOctrladr = _IDE_ISA_guess_altadr(ISA_IDE_array[index]);
	  DI.IDE_found[DI.nb_IDE_found].irq = 0; /* Linux will guess it */
	  DI.nb_IDE_found++;
	  DDBG (("IDE detected.\r\n"));
	  }
      }

//  for (cpt = 0; cpt < DI.nb_IDE_found; cpt++)
//      printf ("IDE%u: 0x%X,0x%X,%u; ", cpt, DI.IDE_found[cpt].ideIOadr, DI.IDE_found[cpt].ideIOctrladr, DI.IDE_found[cpt].irq);
//  puts("");

  if (!copy_gujin_param.attrib.IDE_in_BIOS_order)
      reorder_IDE_for_linux();

//  for (cpt = 0; cpt < DI.nb_IDE_found; cpt++)
//      printf ("IDE%u: 0x%X,0x%X,%u; ", cpt, DI.IDE_found[cpt].ideIOadr, DI.IDE_found[cpt].ideIOctrladr, DI.IDE_found[cpt].irq);
//  puts("");

#if 0 // ATAPI signature not present is usual conditions
  if (copy_gujin_param.attrib.probe_cdrom) {
      for (cpt = 0; cpt < DI.nb_IDE_found; cpt++) {
	  for (index = 0; index < 2; index++) {
	      unsigned short tmp;
	      DDBG (("Probing IDE CDROM 0x%X,0x%X %s for early reset: ",
			DI.IDE_found[cpt].ideIOadr, DI.IDE_found[cpt].ideIOctrladr, (index)? "slave" : "master" ));
	      param->lba_slave_mask = (index)? 0xB0 : 0xA0;
	      param->ideIOadr = DI.IDE_found[cpt].ideIOadr;
	      param->ideIOctrladr = DI.IDE_found[cpt].ideIOctrladr;
	      _IDE_set_drivehead (param, param->lba_slave_mask);
	      _IDE_get_altstatus_bits(param); /* or use "short_delay();" ? */
	      tmp = _IDE_get_cylinder (param);
	      if (tmp == 0xEB14) {
		  DDBG (("initial 0xEB14 signature present, send ATAPI reset: "));
		  _IDE_get_altstatus_bits(param); /* or use "short_delay();" ? */
		  _IDE_start_command (param, IDE_DEVICE_RESET);
		  _IDE_get_altstatus_bits(param); /* or use "short_delay();" ? */
		  DDBG (("[after reset and altstatus, status 0x%X readback= 0x%X] ",
			_IDE_get_altstatus (param), _IDE_readback_lba28 (param)));
		  }
		else
		  DDBG (("initial 0xEB14 signature abscent (0x%X), ignore.\r\n", tmp));
	      }
	  }
      }
#endif

  for (cpt = 0; cpt < DI.nb_IDE_found && nbdisk < NB_DISK; cpt++) {
      unsigned index;
      for (index = 0; index < 2; index++) {
	  unsigned status;
	  PRINTF ("IDE 0x%X,0x%X,%u %s...\r", DI.IDE_found[cpt].ideIOadr, DI.IDE_found[cpt].ideIOctrladr, DI.IDE_found[cpt].irq, (index)? "slave" : "master" );

	  DDBG (("\r\nProbing IDE 0x%X,0x%X,%u %s drive: ",
			DI.IDE_found[cpt].ideIOadr,
			DI.IDE_found[cpt].ideIOctrladr,
			DI.IDE_found[cpt].irq,
			(index)? "slave" : "master" ));
#if 1	/* code space saving */
	  param[nbdisk] = (struct diskparam_str) {};
	  param[nbdisk].disknb = index;
	  param[nbdisk].lba_slave_mask = (index)? 0xB0 : 0xA0;
	  param[nbdisk].ideIOadr = DI.IDE_found[cpt].ideIOadr;
	  param[nbdisk].ideIOctrladr = DI.IDE_found[cpt].ideIOctrladr;
	  param[nbdisk].irq = DI.IDE_found[cpt].irq;
#else
	  param[nbdisk] = (struct diskparam_str) {
	      .disknb = index,
	      .lba_slave_mask = (index)? 0xB0 : 0xA0,
	      .ideIOadr = DI.IDE_found[cpt].ideIOadr,
	      .ideIOctrladr = DI.IDE_found[cpt].ideIOctrladr,
	      .irq = DI.IDE_found[cpt].irq
	      };
#endif
	  status = probeIDEdisk (&param[nbdisk], nbdisk);
	  IDE_reset_DCR_for_HTP_BIOS (&param[nbdisk]);
	  if (status == 0) {
	      nbdisk++;
	      DDBG ((" -> IDE disk/cdrom present\r\n"));
	      }
	    else if (status == 1)
	      DDBG ((" -> unmanaged ATAPI device present\r\n"));
	    else
	      DDBG ((" -> nothing detected\r\n"));
	  }
      }
  DDBG ((" %s end\r\n\r\n", __FUNCTION__));
  return nbdisk;
  }
#endif /* IDE_SUPPORT */

/**
 ** Probe all DOS:
 **/
#if DISK_SUPPORT & DOS_SUPPORT
DISK_FCT_PREFIX(probe_dos) static unsigned
probe_dos (unsigned nbdisk, struct diskparam_str *param)
  {
  unsigned short nbDOSdisk;

  /* This has to be in the current segment, to not have
     a cross segment linker error: */
  asm volatile (
"	jmp	1f					\n"
"fail_int_0x24_marker:					\n"
"	.byte	0					\n"
#if (DEBUG & DEBUG_DISK)
"fail_int_0x24_ah:							\n"
"	.byte	0							\n"
"fail_int_0x24_di:							\n"
"	.hword	0							\n"
"fail_int_0x24_device_header_ctrl_block:				\n"
"	.int	0							\n"
#endif
"fail_int_0x24:						\n"
"	incb	%cs:fail_int_0x24_marker		\n"
#if (DEBUG & DEBUG_DISK)
"	mov	%ah,%cs:fail_int_0x24_ah				\n"
"	mov	%di,%cs:fail_int_0x24_di				\n"
"	mov	%bp,%cs:2 + fail_int_0x24_device_header_ctrl_block	\n"
"	mov	%si,%cs:fail_int_0x24_device_header_ctrl_block		\n"
#endif
"	mov	$0x3,%ax # fail system call (DOS 3.x+)	\n"
"	iretw						\n"
"	1:						\n"
      );

  DDBG (("\r\nProbing DOS drives:"));
  nbDOSdisk = DOS_SelectDisk (DOS_GetCurrentDefaultDrive());
  DDBG ((" max drive %u", nbDOSdisk));
  if (nbDOSdisk > 'Z' - 'A' + 1) {
      DDBG ((" too high, I do not believe it (letter 'A' + %u)\r\n", nbDOSdisk));
      }
    else {
      unsigned short cptdisk, BytesPerSector, ClustersOnDisk;
      unsigned char SectorPerCluster, MediaDescriptorByte;

      DDBG ((" = letter %c\r\n", 'A' + nbDOSdisk - 1));
      for (cptdisk = 0; nbdisk < NB_DISK && cptdisk < nbDOSdisk; cptdisk++) {
	  if (   cptdisk == 1
	      && _BIOS_equipment_flags().nb_floppy_less_1 == 0) {
	      DDBG (("Do not probe 'B:' because only one floppy drive\r\n"));
	      continue;
	      }
	  if (!copy_gujin_param.attrib.probe_cdrom) {
	      unsigned short ret = _CDROM_iscdrom(cptdisk);
	      if (ret != 0) {
		  DDBG (("Do not probe '%c:' because it is a CDROM (ret=0x%X)\r\n",
			'A' + cptdisk, ret));
		  continue;
		  }
	      }
	  PRINTF ("DOS %c:...\r", 'A' + cptdisk);

	  DDBG (("Probing disk %c: ", 'A' + cptdisk));
	  /* DOS_GetAllocationTableInformationForDrive will call "INT 24
	     - Critical Error Handler" if the drive is not formated: */
	  {
	  extern char fail_int_0x24[], fail_int_0x24_marker[];
#if (DEBUG & DEBUG_DISK)
	  extern char fail_int_0x24_ah[];
	  extern short fail_int_0x24_di[];
	  extern farptr fail_int_0x24_device_header_ctrl_block[];
#endif
	  farptr int24 = DOS_GetInterruptVector (0x24);

	  /* That is code segment in the current segment, i.e. probably extraseg... */
	  cs_pokeb (fail_int_0x24_marker, 0);
	  DOS_SetInterruptVector (0x24, ((unsigned)getcs() << 16) + (unsigned)fail_int_0x24);
	  MediaDescriptorByte = DOS_GetAllocationTableInformationForDrive (
					cptdisk + 1,
					&SectorPerCluster,
					&BytesPerSector,
					&ClustersOnDisk);
	  DOS_SetInterruptVector (0x24, int24);
	  if (cs_peekb (fail_int_0x24_marker) != 0) {
#if (DEBUG & DEBUG_DISK)
	      unsigned short di = cs_peekw (fail_int_0x24_di);

	      DDBG (("INT0x24 called with AH=0x%X, DI=0x%X, BP:SI=0x%X: ",
			cs_peekb (fail_int_0x24_ah),
			di,
			cs_peekl (fail_int_0x24_device_header_ctrl_block)));
	      if (di > 0x0C)
		  DDBG (("unknown/invalid meaning of DI\r\n"));
		else
		  DDBGVAR (( ((const char *const []) {
			"write protect error\r\n",
			"unknown unit\r\n",
			"drive not ready\r\n",
			"unknown command\r\n",
			"data error (bad CRC)\r\n",
			"bad request structure length\r\n",
			"seek error\r\n",
			"unknown media type\r\n",
			"sector not found\r\n",
			"printer out of paper\r\n",
			"write fault\r\n",
			"read fault\r\n",
			"general failure\r\n"
			})[di]));
#endif
	      DOS_ResetDisk();
	      continue;
	      }
	  }
	  if (SectorPerCluster == 0xFF) {
	      DDBG (("cannot get allocation table, no drive there.\r\n"));
	      continue;
	      }
	    else
	      DDBG (("drive present (MediaDescriptorByte 0x%X, ClustersOnDisk %u, SectorPerCluster %u, BytesPerSector %u).\r\n",
		MediaDescriptorByte, ClustersOnDisk, SectorPerCluster, BytesPerSector));

#if 1	/* code space saving */
	  param[nbdisk] = (struct diskparam_str) {};
	  param[nbdisk].access = dos_part;
	  param[nbdisk].disknb = 'A' + cptdisk;
	  param[nbdisk].bytepersector = BytesPerSector;
	  param[nbdisk].nbtotalsector = SectorPerCluster * ClustersOnDisk;
	  param[nbdisk].nbsectorpertrack = SectorPerCluster;
	  param[nbdisk].nbcylinder = ClustersOnDisk;
	  param[nbdisk].nbhead = MediaDescriptorByte;
//	  param[nbdisk].nbpartition = 0;
//	  param[nbdisk].partition = 0;
#else
	  param[nbdisk] = (struct diskparam_str) {
		.access = dos_part,
		.disknb = 'A' + cptdisk,
		.bytepersector = BytesPerSector,
		.nbtotalsector = SectorPerCluster * ClustersOnDisk,
		.nbsectorpertrack = SectorPerCluster,
		.nbcylinder = ClustersOnDisk,
		.nbhead = MediaDescriptorByte,
		.nbpartition = 0,
		.partition = 0,
		};
#endif
	  STRINIT (param[nbdisk].diskname, "DOS A:");
	  param[nbdisk].diskname[4] += cptdisk;
	  nbdisk++;
	  }
      }
  DDBG ((" %s end\r\n", __FUNCTION__));
  return nbdisk;
  }
#endif /* DOS_SUPPORT */

/**
 ** We are nearly there, keep reading...
 **/

#if (DISK_SUPPORT & EBIOS_SUPPORT) && !(DISK_SUPPORT & PROBE_ONLY_BIOSBOOTDEVICE)
DISK_FCT_PREFIX(get_ebios_ide_addresses) static void
get_ebios_ide_addresses (void)
  {
  unsigned short disk, max_disk, max_disk1, max_disk2, cpt;
  unsigned nb_ide_found = 0;
#ifdef DISK_USE_MALLOC
  struct IDE_found_str	ide_found[NB_IDE];
#else
  struct IDE_found_str	*ide_found = DI.IDE_found = _disk_idefound;
#endif

  /* Boring: Win 2000 try to say that there is no [E]BIOS disk accessible
     by setting _BIOSDISK_get_nb_harddisk() i.e. "max_disk BIOS memory "
     to zero... but I cannot trigger on zero - too usual a value. */
  max_disk1 = DI.nb_bios_hd;
  max_disk2 = _BIOSDISK_get_nb_harddisk();
  if (max_disk2 > max_disk1 && max_disk2 < 0x80)
      max_disk = max_disk2;
    else
      max_disk = max_disk1;

  DDBG (("%s: probing IDE interfaces (max_disk BIOS call %u, "
	"max_disk BIOS memory %u - using %u):\r\n",
	__FUNCTION__, max_disk1, max_disk2, max_disk));

  for (disk = 0x80; disk < 0x80 + max_disk; disk++) {
      unsigned short	ebios_version;
      ebios_bitmap_t	ebios_bitmap;
      unsigned char	status;
      unsigned short	ideIOadr, ideIOctrladr;
      unsigned char	irq;
      ebiosinfo_t	ebiosinfo = {};

      ebiosinfo.buffersize = sizeof (ebiosinfo);
      DDBG (("Probing 0x%X: ", disk));
#if defined (TREAT_EXCEPTION)
      if (shit_handler (1, 0) != 0) { /* never seen that, just in case */
	  DDBG ((" EXCEPTION RECEIVED!\r\n"));
	  continue;
	  }
#endif
      if (_EBIOSDISK_probe (disk, &ebios_version, &ebios_bitmap) != 0) {
	  DDBG (("_EBIOSDISK_probe failed.\r\n"));
	  continue;
	  }
#if defined (TREAT_EXCEPTION)
      shit_handler (0, 0);
#endif
      if (ebios_version > 0x2100) {
	  DDBG (("[high enough version, enable prefetch: "));
	  unsigned char carry = _EBIOSDISK_set_hw_conf(disk, enable_prefetch, &status);
	  if (carry)
	      DDBG (("failed carry] "));
	    else if (status == 0)
	      DDBG (("all fine] "));
	    else if (status == 1)
	      DDBG (("all fine, other drive affected] "));
	    else
	      DDBG (("status 0x%X ] ", status));
#if 1
	  DDBG (("[high enough version, enable max PIO: "));
	  carry = _EBIOSDISK_set_hw_conf(disk, set_max_PIO, &status);
	  if (carry)
	      DDBG (("failed carry] "));
	    else if (status == 0)
	      DDBG (("all fine] "));
	    else if (status == 1)
	      DDBG (("all fine, other drive affected] "));
	    else
	      DDBG (("status 0x%X ] ", status));
#else
	  DDBG (("[high enough version, enable max DMA: "));
	  carry = _EBIOSDISK_set_hw_conf(disk, enable_INT_13h_DMA_max, &status);
	  if (carry)
	      DDBG (("failed carry] "));
	    else if (status == 0)
	      DDBG (("all fine] "));
	    else if (status == 1)
	      DDBG (("all fine, other drive affected] "));
	    else
	      DDBG (("status 0x%X ] ", status));
#endif
	  }
      DDBG (("[EBIOSDISK_probe version 0x%X bitmap 0x%X] ", ebios_version, ebios_bitmap));
      if (_EBIOSDISK_getparam (disk, &ebiosinfo, &status) != 0 || status != 0) {
	  DDBG (("_EBIOSDISK_getparam failed.\r\n"));
	  continue;
	  }

      if (ebiosinfo.configptr == 0 || ebiosinfo.configptr == 0xFFFFFFFFU) {
	  DDBG ((" Non Phoenix Bios compatible.\r\n"));
	  continue;
	  }
	else {
	  int i = 15;
	  unsigned char checksum = 0;

	  while (i--)
	      checksum -= peekb(ebiosinfo.configptr + i);

	  if (checksum != peekb (ebiosinfo.configptr + offsetof(ebiosPhoenix_t, checksum))) {
	      DDBG ((" Phoenix Bios bad checksum.\r\n"));
	      continue;
	      }
	  }
      ideIOadr = peekw (ebiosinfo.configptr + offsetof(ebiosPhoenix_t, dataIOadr));
      if (ideIOadr == 0) {
	  DDBG (("ideIOadr null.\r\n"));
	  continue;
	  }
      for (cpt = 0; cpt < nb_ide_found; cpt++)
	  if (ide_found[cpt].ideIOadr == ideIOadr)
	      break;
      if (cpt < nb_ide_found) {
	  DDBG (("param.ideIOadr 0x%X already known.\r\n", ideIOadr));
	  continue;
	  }
      ideIOctrladr = peekw (ebiosinfo.configptr + offsetof(ebiosPhoenix_t, controlIOadr));
      if (ideIOctrladr == 0) {
	  /* EBIOS 1.6 on Compaq Deskpro 590 */
	  ideIOctrladr = _IDE_ISA_guess_altadr (ideIOadr);
	  DDBG ((" [Correcting ideIOctrladr of 0x%X from 0 to 0x%X] ",
			ideIOadr, ideIOctrladr));
	  }
      if (inb (ideIOctrladr) == 0) {
	  /* That is for the f*****g add-on PCI IDE boards which do not
	     implement reading the status without clearing the interrupt,
	     on address ideIOctrladr. Is that so difficult to read the ATA
	     specifications and implement them? Are you waiting for your
	     costumer to bring back the boards and demamd total reimbursement
	     plus extra to make them believe they brought an ATA card because
	     it was written on the BOX? Do you think all of them clicked
	     the "not to sue" EULA and that you can prove it? */
	  DDBG (("Invalid IDE found: 0x%X,0x%X.\r\n", ideIOadr, ideIOctrladr));
	  printf (INVALID_IDE_INTERFACE_AT_X_X, ideIOadr, ideIOctrladr);
	  continue;
	  }
      irq = peekb (ebiosinfo.configptr + offsetof(ebiosPhoenix_t, irq));
      DDBG (("Adding IDE found: 0x%X,0x%X,%u.\r\n", ideIOadr, ideIOctrladr, irq));
      if (nb_ide_found >= NB_IDE) {
	  DDBG (("Cannot add, array full\r\n"));
	  continue;
	  }
      ide_found[nb_ide_found].ideIOadr = ideIOadr;
      ide_found[nb_ide_found].ideIOctrladr = ideIOctrladr;
      ide_found[nb_ide_found].irq = irq;
      ide_found[nb_ide_found].bios_order = disk - 0x80;
      nb_ide_found++;
      }

  DI.nb_IDE_found = nb_ide_found;
#ifdef DISK_USE_MALLOC
  if (nb_ide_found != 0) {
      DI.IDE_found = MALLOC (nb_ide_found * sizeof (struct IDE_found_str), "idefound");
      if (DI.IDE_found == 0) {
	  DI.nb_IDE_found = 0;
	  DDBG (("%s: malloc for %u IDE failed!\r\n", __FUNCTION__, nb_ide_found));
	  }
	else {
	  while (nb_ide_found--)
	      DI.IDE_found[nb_ide_found] = ide_found[nb_ide_found];
	  }
      }
    else
      DI.IDE_found = 0;
#endif /* DISK_USE_MALLOC */
  DDBG (("Note: %u IDE found (max compiled-in %u).\r\n", DI.nb_IDE_found, NB_IDE));
  }
#endif

/**
 ** The top-level function:
 **/
DISK_FCT_PREFIX(probedisk) void
probedisk (void)
  {bound_stack();{
  unsigned nbdisk = 0;
#ifdef DISK_USE_MALLOC
  struct diskparam_str	param[NB_DISK] = {};

  /* We need to free everything for the reprobe of the menu: */
  unsigned cptdisk;
  if (DI.nbdisk != 0 && DI.param == 0)
      DBG (("DI.nbdisk != 0 && DI.param == 0!\r\n"));

  for (cptdisk = 0; cptdisk < DI.nbdisk; cptdisk++) {
      struct diskparam_str *dp = &DI.param[cptdisk];
#if DISK_SUPPORT & ISOFS_PROBE
      unsigned cptpart;

      for (cptpart = 0; cptpart < dp->nbpartition; cptpart++)
	  if (dp->partition[cptpart].ElToritoCatalog)
	      FREE (dp->partition[cptpart].ElToritoCatalog);
#endif
      if (   dp->access != dos_part /* dirty DOS diskname trick */
	  && dp->nbpartition != 0 && dp->partition == 0)
	  DBG (("dp->nbpartition != 0 && dp->partition == 0!\r\n"));
      if (dp->partition != 0)
	  FREE (dp->partition);
#if defined (NB_FREELIST) && (NB_FREELIST != 0)
      if (dp->nbfreelist != 0 && dp->freelist == 0)
	  DBG (("dp->nbfreelist != 0 && dp->freelist == 0!\r\n"));
      if (dp->freelist != 0)
	  FREE (dp->freelist);
#endif
      }
  if (DI.param != 0)
      FREE (DI.param);
  DI.nbdisk = 0;
  if (DI.nb_IDE_found != 0 && DI.IDE_found == 0)
      DBG (("DI.nb_IDE_found != 0 && DI.IDE_found == 0!\r\n"));
  if (DI.IDE_found != 0)
      FREE (DI.IDE_found);
  DI.nb_IDE_found = 0;
  kbd_setup (copy_gujin_param.kbdmap);	/* move memory block down */
#if 0  // list memory block still allocated:
  {
  unsigned cpt;
  for (cpt = 0; cpt < MALLOC_ARRAY_SIZE; cpt++)
      if (UTIL.memalloc.malloc_array[cpt].size > 0)
	  DBG (("[malloced block of %d bytes still present at 0x%X]",
		UTIL.memalloc.malloc_array[cpt].size,
		UTIL.memalloc.malloc_array[cpt].adr));
  }
#endif
#else /* DISK_USE_MALLOC */
  static struct diskparam_str _disk_param[NB_DISK];
  struct diskparam_str	*param = DI.param = _disk_param;
#endif /* DISK_USE_MALLOC */

  if (sizeof (bootsector_t) != 512)
      __ERROR();	/* the linker will produce an error here */

#if SETUP & XCODE_SEGMENT
  extern unsigned char (*fptr_DISK_readsector) (struct diskparam_str *dp, int partition, unsigned long long lba, unsigned number, farptr buffer);
  DI.readsector = *fptr_DISK_readsector;
#ifdef WRITE_ENABLE
  extern unsigned char (*fptr_DISK_writesector) (struct diskparam_str *dp, int partition, unsigned long long lba, unsigned number, farptr buffer);
  DI.writesector = *fptr_DISK_writesector;
#endif
#if (USER_SUPPORT != 0) && (DISK_SUPPORT & IDE_SUPPORT)
  extern int (*fptr_ideSecurity) (struct diskparam_str *dp, char password[32], enum IDE_security_enum action);
  DI.ideSecurity = *fptr_ideSecurity;
#endif
#else /* !XCODE_SEGMENT */
  DI.readsector = (typeof(DI.readsector))(STATE.codeseg_adr | (unsigned)DISK_readsector);
#ifdef WRITE_ENABLE
  DI.writesector = (typeof(DI.writesector))(STATE.codeseg_adr | (unsigned)DISK_writesector);
#endif
#if (USER_SUPPORT != 0) && (DISK_SUPPORT & IDE_SUPPORT)
  DI.ideSecurity = (typeof(DI.ideSecurity))(STATE.codeseg_adr | (unsigned)ideSecurity);
#endif
#endif /* !XCODE_SEGMENT */

  DI.max_IDE_found = NB_IDE;
#ifdef NB_FREELIST
  DI.max_freelist = NB_FREELIST;
#endif
  DI.max_disk = NB_DISK;
  DI.max_partition = NB_PARTITION;
  DI.sizeof_diskparam_str = sizeof (struct diskparam_str);
  DI.sizeof_partition_str = sizeof (struct partition_str);
#if defined (NB_FREELIST) && (NB_FREELIST != 0)
  DI.sizeof_freelist_str = sizeof (struct freelist_str);
#endif
#if DISK_SUPPORT & (EBIOS_SUPPORT | IDE_SUPPORT)
  DI.sizeof_IDE_found_str = sizeof (struct IDE_found_str);
#endif
  DBG (("max_IDE_found %u, max_freelist %u, max_disk %u, max_partition %u\r\n",
	DI.max_IDE_found, DI.max_freelist, DI.max_disk, DI.max_partition));
  DBG (("sizeof struct diskparam_str %u, sizeof struct partition_str %u, "
	"sizeof struct freelist_str %u, sizeof struct IDE_found_str %u.\r\n",
	DI.sizeof_diskparam_str, DI.sizeof_partition_str,
	DI.sizeof_freelist_str, DI.sizeof_IDE_found_str));

  if (copy_gujin_param.attrib.probe_bios_floppy_disk && !DI.cannot_reset_floppy) {
      unsigned char status;

      DDBG (("\r\nInitialising BIOS floppy disk system: "));
      /* init floppy only, for Phoenix BIOS: */
#if defined (TREAT_EXCEPTION)
      if (shit_handler (1, 4) != 0) {
	  DDBG ((" EXCEPTION/timeout 4 second RECEIVED!\r\n"));
	  DI.cannot_reset_floppy = 1;
	  }
	else {
#endif
	  if (_BIOSDISK_reset_floppy (&status) != 0)
	      DDBG (("error 0x%X\r\n", status));
	    else
	      DDBG (("done (status 0x%X).\r\n", status));
#if defined (TREAT_EXCEPTION)
	  }
      shit_handler (0, 4);
#endif
      }

  DDBG (("\r\nStart probe disks:"));
  DI.nb_bios_fd = probe_nbBIOS (0x00);
  DI.nb_bios_hd = probe_nbBIOS (0x80);
  DDBG (("BIOS interrupts report %u HD, %u FD\r\n", DI.nb_bios_hd, DI.nb_bios_fd));
#if (DEBUG & DEBUG_DISK) && ((DEBUG & DEBUG_OUTPUT) == SCREEN)
  DDBG (("Press a key to continue..."));
  _BIOS_getkey();
#endif

#if DISK_SUPPORT & PROBE_ONLY_BIOSBOOTDEVICE
  {
  unsigned short cpt = 0xFFFF, max_disk = DI.nb_bios_hd, max_disk1;
  unsigned char test_disk = UTIL.bios_boot_dl;

  max_disk1 = _BIOSDISK_get_nb_harddisk();
  if (max_disk1 > max_disk && max_disk1 < 0x80)
      max_disk = max_disk1;

  TINYDBG ("NB_DISK %u, DI.nb_bios_hd 0x%X, _BIOSDISK_get_nb_harddisk() 0x%X, max_disk 0x%X, UTIL.bios_boot_dl 0x%X\r\n",
	NB_DISK, DI.nb_bios_hd, max_disk1, max_disk, UTIL.bios_boot_dl);

  nbdisk = 1;
  while (probeBIOSdisk (test_disk, &param[0])) {
      // if UTIL.bios_boot_dl invalid, try floppies then HD:
      if (cpt == 0xFFFF) {
	  if (copy_gujin_param.attrib.probe_bios_floppy_disk)
	      test_disk = cpt = 0;
	    else
	      test_disk = cpt = 0x80;
	  }
	else if (cpt > 0x80 + max_disk) {
	  nbdisk = 0;
	  TINYDBG ("probeBIOSdisk failed, cpt = 0x%X, abort\r\n", cpt);
	  break;
	  }
	else if (cpt != _BIOS_equipment_flags().nb_floppy_less_1 + 1)
	  test_disk = ++cpt;
	else
	  test_disk = cpt = 0x80;
      TINYDBG ("probeBIOSdisk failed, retry with 0x%X, ", test_disk);
      }
TINYDBG ("probeBIOSdisk done\r\n");
  }
#else /* PROBE_ONLY_BIOSBOOTDEVICE */

#if DISK_SUPPORT & EBIOS_SUPPORT
  get_ebios_ide_addresses ();
#endif

/* We have to probe IDE disks first because if some disks are
   protected by the IDE password system they have to be unlocked
   before the BIOS probing: */
#if DISK_SUPPORT & IDE_SUPPORT
#if !(DEBUG & (DEBUG_DISK|DEBUG_FS))	/* quick and dirty to get a log */
  if (BOOT1_DOS_running()) {
      DDBG (("Skiping IDE probe in DOS+ environment.\r\n"));
      }
    else
#endif
	 if (!copy_gujin_param.attrib.probe_ide_disk && !copy_gujin_param.attrib.probe_cdrom)
      DDBG (("Skiping IDE probe due to current attributes.\r\n\r\n"));
    else {
      unsigned char status __attribute__ ((unused));
      unsigned char ret __attribute__ ((unused));
      nbdisk = probe_ide (nbdisk, param);

      /* FIXME: This does not force a re-probe on my EBIOS, i.e. HPA is updated
	on the disk but not on the BIOS (max sector). Is there a better way? */
      DBG (("\r\nReset all BIOS disks for BIOS to acknowledge changes:"));
      ret = _BIOSDISK_reset_alldisks (&status);
      DBG (("returns 0x%X, status 0x%X\r\n", ret, status));
      }
#else /* DISK_SUPPORT & IDE_SUPPORT */
#if DISK_SUPPORT & EBIOS_SUPPORT
  if (!copy_gujin_param.attrib.IDE_in_BIOS_order)
      reorder_IDE_for_linux(); /* normally called in probe_ide(), but we do not have IDE_SUPPORT here */
#endif
  DDBG (("Probing IDE not compiled-in.\r\n"));
#endif /* DISK_SUPPORT & IDE_SUPPORT */

#if DISK_SUPPORT & (BIOS_SUPPORT | EBIOS_SUPPORT)
  nbdisk = probe_bios (nbdisk, param);
#else
  DDBG (("Probing BIOS HD/FD not compiled-in.\r\n"));
#endif /* DISK_SUPPORT & (BIOS_SUPPORT | EBIOS_SUPPORT) */


#if DISK_SUPPORT & DOS_SUPPORT
  if (!BOOT1_DOS_running())
      DDBG (("Skiping DOS probe in BIOS environment.\r\n"));
    else if (!copy_gujin_param.attrib.probe_dos_disk) /* is probe_bdi_file in BIOS environment */
      DDBG (("Skiping DOS probe due to current attributes.\r\n"));
    else
      nbdisk = probe_dos (nbdisk, param);
#else
  DDBG (("Probing DOS not compiled-in.\r\n"));
#endif /* DISK_SUPPORT & DOS_SUPPORT */

#endif /* PROBE_ONLY_BIOSBOOTDEVICE */

  DDBG (("\r\nEnd probe disks.\r\n"));

  DI.nbdisk = nbdisk;
#ifdef DISK_USE_MALLOC
  if (nbdisk != 0) {
      DI.param = MALLOC (nbdisk * sizeof (struct diskparam_str), "DIparam");
      if (DI.param == 0) {
	  DI.nbdisk = 0;
	  DDBG (("%s: malloc for %u disk failed!\r\n", __FUNCTION__, nbdisk));
	  }
	else {
	  while (nbdisk--)
	      DI.param[nbdisk] = param[nbdisk];
	  }
      }
    else
      DI.param = 0;
#endif /* DISK_USE_MALLOC */
  DDBG (("Note: %u disk found (max compiled-in %u).\r\n", DI.nbdisk, NB_DISK));
  }}

/**
 ** Main variable declaration:
 **/
struct disk_interface_str DI;
